//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// FDISK GPT specific Dialog and windowing functions
//
// Author: J. van Wijk
//
// JvW  20-09-2015 Initial version, derived from DFSDFDSK
//


#include <txlib.h>                              // TX library interface

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfsupart.h>                           // FDISK partition functions
#include <dfsafdsk.h>                           // FDISK display & analysis
#include <dfscfdsk.h>                           // FDISK callback and commands
#include <dfsdfgpt.h>                           // FDISK GPT dialog functions
#include <dfsufgpt.h>                           // FDISK GPT utility functions
#include <dfsufdsk.h>                           // FDISK utility functions
#include <dfsdgen.h>                            // Generic Dialog functions
#include <dfswin.h>                             // window and help definitions
#include <dfsver.h>                             // version and naming info

#if defined (USEWINDOWING)

// PCREATE dialog specific definitions

#define DFSD_WID_PCRDLG      800                // window id dialog frame
#define DFSD_WID_CBTMAN      802                // push  button Manual Edit
#define DFSD_WID_CEOFFS      803                // entryfield   Offset
#define DFSD_WID_CRBSTA      804                // radio button At-Start
#define DFSD_WID_CRBEND      805                // radio button At-End
#define DFSD_WID_CPNAME      807                // entry Partition Name
#define DFSD_WID_CATVAL      808                // entry Attribute Value
#define DFSD_WID_CBTATT      809                // push  button Specify Attr
#define DFSD_WID_CCBALN      810                // check-box    no Align
#define DFSD_WID_CRBREC      811                // radio-box    Recovery
#define DFSD_WID_CRBCLR      812                // radio-box    Clear BS
#define DFSD_WID_CPTEXT      814                // part type leader
#define DFSD_WID_CPDESC      815                // part type GUID string
#define DFSD_WID_CSPINV      816                // spin-value type list
#define DFSD_WID_CESIZE      817                // entryfield create-size
#define DFSD_WID_CCBMAX      718                // check-box    Use MAX size
#define DFSD_WID_CBUTTN      819                // buttons Ok and Cancel

#define DFSH_PG             (DFSH_FDISK + 560)
#define DFSH_PG_CBTMAN      (DFSH_PG + 2)
#define DFSH_PG_CEOFFS      (DFSH_PG + 3)
#define DFSH_PG_CRBSTA      (DFSH_PG + 4)
#define DFSH_PG_CRBEND      (DFSH_PG + 5)
#define DFSH_PG_CPNAME      (DFSH_PG + 7)
#define DFSH_PG_CATVAL      (DFSH_PG + 9)
#define DFSH_PG_CBTATT      (DFSH_PG + 9)
#define DFSH_PG_CRBREC      (DFSH_PG +10)
#define DFSH_PG_CRBCLR      (DFSH_PG +11)
#define DFSH_PG_CCBALN      (DFSH_PG +13)
#define DFSH_PG_CSPINV      (DFSH_PG +16)
#define DFSH_PG_CESIZE      (DFSH_PG +17)
#define DFSH_PG_CCBMAX      (DFSH_PG +18)
#define DFSH_PG_CBUTTN      (DFSH_PG +19)

#define DFSH_PG_FRAME       (DFSH_PG +20)       // shown help-on-help (2nd)


#define DFSD_PCREAT_WC          73              // horizontal size in columns
#define DFSD_PCREAT_WL          24              // vertical   size in lines


typedef struct dfs_pcgptdata                    // GPT partition-create data
{
   DFSPARTINFO     *fsp;                        // freespace info (geo, size)
   TXTT             tpsbuf;                     // titlefield create-size
   TXTT             vpsbuf;                     // entryfield create-size
   double           vpsize;                     // floating size
   double           fssize;                     // maximum size, from offset
   TXTT             vpobuf;                     // entryfield offset
   double           vpoffs;                     // floating offset
   TXTT             vpnbuf;                     // entryfield GPT partition name
   TXTT             vavbuf;                     // entryfield GPT attribute value
   TXTT             vidstr;                     // GUID-string value
   ULONG            vstype;                     // system-type value
   BOOL             vrbGpt;                     // radio button GPT partition
   BOOL             vrbSta;                     // radio button At-Start
   BOOL             vrbEnd;                     // radio button At-End
   BOOL             vrbClr;                     // radio button Clear BS (or Recover)
   BOOL             vcbMax;                     // check-box Use Maximum size
   BOOL             vcbAln;                     // check-box do NOT align offset/size
   TXSELIST        *gptList;                    // SELIST with GPT types
} DFS_PCGPTDATA;                                // end of struct "dfs_pcgptdata"


static char dfsdf_bt_ok[]       = " OK ";
static char dfsdf_bt_cancel[]   = "Cancel";

static char        *pcrGptHelpText[] =
{
   "",
   " This dialog allows specification of all the information needed to",
   " create a new GPT-style partition, once the freespace is selected.",
   "",
   TXHELPITEM(002, "'Manual edit type GUID' button")
   " This push button allows you to manually specify a hexadecimal",
   " value for the partition type to be created. You may need it",
   " for (new) GPT partition types that are not in the DFSee standard",
   " list of partition types.",
   "",
   " The edit dialog will start with the last value selected from",
   " the list, so you can use it as a starting point.",
   "",
   TXHELPITEM(003, "Offset in freespace")
   " For more help on the complete dialog, press <F1> again ...",
   "",
   " This allows specification of the offset from the beginning or",
   " end of the selected freespace area (depending on selection).",
   " It allows you to reserve some freespace there for future use.",
   "",
   " It will be interpreted as a decimal value in megabytes (MiB).",
   " As an alternative, you can use the mcs-number formats to",
   " specify the offset in gigabytes (,g), cylinders (,c) etc.",
   "",
   " Using the <Enter> key here will end the dialog and execute a ",
   " CREATE command based on all specified values.",
   "",
   TXHELPITEM(004, "'From start freespace' button")
   " When set, this button indicates that the new partition will",
   " be created at the START of the selected freespace area.",
   " (possibly leaving some reserved space, defined by 'offset')",
   "",
   " The radio-button is toggled using the <Space-bar> or the mouse",
   "",
   " Using the <Enter> key here will end the dialog and execute a ",
   " CREATE command based on all specified values.",
   "",
   TXHELPITEM(005, "'From end freespace' button")
   " When set, this button indicates that the new partition will",
   " be created at the END of the selected freespace area.",
   " (possibly leaving some reserved space, defined by 'offset')",
   "",
   " The radio-button is toggled using the <Space-bar> or the mouse",
   "",
   " Using the <Enter> key here will end the dialog and execute a ",
   " CREATE command based on all specified values.",
   "",
   TXHELPITEM(007, "Partion Name entryfield")
   " For more help on the complete dialog, press <F1> again ...",
   "",
   " This field allows specification of a descriptive NAME for the",
   " new partition. It is given as an ASCII string with a maximum",
   " length 36 characters (stored as UNICODE, in 72 bytes).",
   "",
   " When no user specified name is present, a default name based",
   " on the currently selected partition type-GUID is filled in.",
   " Make sure to make this value unique, at least per system!",
   "",
   " Using the <Enter> key here will end the dialog and execute a ",
   " CREATE command based on all specified values.",
   "",
   TXHELPITEM(008, "Attribute flag value entryfield")
   " For more help on the complete dialog, press <F1> again ...",
   "",
   " This field allows specification of hexadecimal value that",
   " represent the 64 possible flag-bits for the new partition.",
   "",
   " Using the <Enter> key here will end the dialog and execute a ",
   " CREATE command based on all specified values.",
   "",
   TXHELPITEM(009, "'Specify Attribute Flags' button")
   " For more help on the complete dialog, press <F1> again ...",
   "",
   " This will open a dialog where each of the DEFINED (for DFsee)",
   " flag-bits can be read out and specified individually.",
   "",
   TXHELPITEM(010, "Recovery mode, keep existing bootsector intact")
   " For more help on the complete dialog, press <F1> again ...",
   "",
   " When marked, this option will make sure the bootsector for the ",
   " newly created partition intact so it will be re-used instead of",
   " created new.",
   "",
   " This is MANDATORY if you want to RECOVER and existing filesystem",
   " without having to re-create that information again later.",
   "",
   " So, if you are RE-CREATING the partition for RECOVERY purposes,",
   " DO NOT check this since it will wipe out the bootsector with",
   " important size, format and label-information!",
   "",
   " The check-box is toggled using the <Space-bar> or the mouse",
   "",
   " Using the <Enter> key here will end the dialog and execute a ",
   " CREATE command based on all specified values.",
   "",
   TXHELPITEM(011, "Create NEW mode, clear existing bootsector")
   " For more help on the complete dialog, press <F1> again ...",
   "",
   " When marked, this option will cause the bootsector for the ",
   " newly created partition to be CLEARED using the 0xF6 pattern",
   " just like any standard FDISK tool would do.",
   "",
   " This can be useful to make sure any old formatting information",
   " does not interfere with the new desired usage of the partition.",
   "",
   " If you are RE-CREATING the partition for RECOVERY purposes,",
   " DO NOT check this since it will wipe out the bootsector",
   " with important size, format and label-information!",
   "",
   " The check-box is toggled using the <Space-bar> or the mouse",
   "",
   " Using the <Enter> key here will end the dialog and execute a ",
   " CREATE command based on all specified values.",
   "",
   TXHELPITEM(013, "Do NOT Align to MiB boundaries")
   " When checked, this will cause the values for offset and size to",
   " be UNALIGNED to the regular 1-Mib (0x800 sectors) boundaries.",
   "",
   " This may allow for slighter larger partitions at the start and",
   " end of the disk, at the expense of non standard alignment.",
   "",
   TXHELPITEM(016, "Partition system type")
   " This is the 'system-type' for the partition, an 128-bit unique",
   " value (partition type GUID) that represents the purpose of the",
   " partition, and relates to the filesystem or the operating system",
   " using this partition.",
   "",
   " The most used values can be selected from the list directly, and",
   " while navigating the list, a more verbose description of the type",
   " is shown at the status line at the bottom of the screen.",
   "",
   " The list entry has three parts separated by a vertical bar:",
   "",
   " - 4 hexadecimal digits 'shorthand' value, as used by Linux GDISK",
   "",
   " - A short description of the owner/manufacturer of the type",
   "",
   " - The DFSee short description of the partition type.",
   "",
   "",
   " Any other (custom) values can be specified as hexadecimal values",
   " by selecting the last entry 'Manually set custom GUID value'",
   " or using the [Manual edit type GUID] button.",
   "",
   TXHELPITEM(017, "Partition size field")
   " For more help on the complete dialog, press <F1> again ...",
   "",
   " This field allows specification of the desired size for the",
   " new partition. It can be given as a decimal value that is",
   " interpreted as megabytes (MiB) by default.",
   "",
   " As an alternative, you can use the mcs-number formats to",
   " specify the size in gigabytes (,g), cylinders (,c) etc.",
   "",
   " Using the <Enter> key here will end the dialog and execute a ",
   " CREATE command based on all specified values.",
   "",
   TXHELPITEM(018, "Use MAX size: xxxx.yy MiB")
   " When checked, this will force the size to be used to the MAXIMUM",
   " available size in the slected freespace area, considering the",
   " current offset value.",
   "",
   " The entryfield wil be disabled (readonly), showing either the",
   " exact size in sectors when not-aligned, or '*' when aligned.",
   "",
   " When unchecked, the size entryfield is enabled to allow input",
   " of a desired size in Mib or sectors.",
   "",
   TXHELPITEM(019, "OK and Cancel buttons")
   " When all values are set to the desired values, the [OK]",
   " button will end the dialog and execute a CREATE command.",
   "",
   " Actually, using <Enter> on ANY field in the dialog EXCEPT the",
   " spin-value control with partition-types will do the same.",
   "",
   " The [Cancel] button will end the dialog without creating",
   " a new partition, just like the 'Escape' key would.",
   "",
   TXHELPITEM(020, "GPT Partition Create Dialog window")
   " This dialog allows specification of all the information needed",
   " to create a new GPT partition, once the freespace is selected.",
   "",
   " Using the <Enter> key on ANY field in the dialog EXCEPT the",
   " spin-value control with partition-types will end the dialog",
   " and execute a CREATE command based on all specified values.",
   "",
   " There will be a final confirmation popup in the CREATE, so",
   " you will have another chance of checking all values and",
   " CANCEL the create if any wrong values have been used.",
   "",
   " The dialog ends with the following keys / actions:",
   "",
   "  - <ENTER> End dialog, and CREATE the partition as specified",
   "  - <Esc>   End dialog, do not create a partition",
   "",
   "",
   "",
   NULL
};



// Dialog window procedure, for the GPT-style CreatePart Dialog
static ULONG dfsGptCrPartDlgWinProc             // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    msg param 1
   TXWMPARAM           mp2                      // IN    msg param 2
);


// Dialog window procedure, for the CREATE-GPT partition dialog fields
static ULONG dfsGptCreateWinProc                // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    msg param 1
   TXWMPARAM           mp2                      // IN    msg param 2
);


// Set a new default partition name based on current type GUID, unless edited
static void dfsGptSetDefaultName
(
   DFS_PCGPTDATA      *pcd                      // IN    Create Partition data
);

// Calculate exact maximum partitionsize based on alignment and offset/position
static void dfsGptCpGetMaxSize
(
   DFS_PCGPTDATA      *pcd                      // IN    Part Create Data
);


/*****************************************************************************/
// Initialize help system for the Create GPT Dialog (accessible from startup)
/*****************************************************************************/
void dfsCreateGptInitHelp
(
   void
)
{
   ENTER();

   txwRegisterHelpText( DFSH_PG, "dfsgptcreate", "GPT-style Partition-Create", pcrGptHelpText);

   VRETURN ();
}                                               // end 'dfsCreateGptInitHelp'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Present a GPT-style Create-Partition Dialog, with freespace info as input
/*****************************************************************************/
BOOL txwCreateGptPartDialog
(
   DFSPARTINFO        *f,                       // IN    freespace information
   char               *create                   // OUT   Create command
)
{
   BOOL                rc = FALSE;              // function return
   ULONG               result = 0;

   ENTER();

   if (f != NULL)                               // MUST have a freespace area!
   {
      if (txwIsWindow( TXHWND_DESKTOP))         // is there a desktop ?
      {
         TXRECT        position;                // reference size/position
         TXWHANDLE     cframe;                  // file-dialog frame
         TXWHANDLE     centry;                  // entryfield create-size
         TXWHANDLE     centof;                  // entryfield offset
         TXWHANDLE     centpn;                  // entryfield partition name
         TXWHANDLE     ccbMax;                  // Checkbox Use-MAX-size
         TXWHANDLE     ccbAln;                  // Checkbox Do-NOT-align
         TXWHANDLE     chex0x;                  // Text OUTPUT field Hex 0x
         TXWHANDLE     centav;                  // entryfield attribute value
         TXWHANDLE     cspinv;                  // spin-value type list
         TXWHANDLE     cptext;                  // partition type text leader
         TXWHANDLE     cpdesc;                  // partition type GUID string
         TXWHANDLE     cbtok;                   // OK button
         TXWHANDLE     cbtcan;                  // CANCEL button
         TXWHANDLE     cbtMan;                  // push  button Manual edit
         TXWHANDLE     cbtAtt;                  // push  button Specify Attr
         TXWHANDLE     crbSta;                  // radio button At-Start
         TXWHANDLE     crbEnd;                  // radio button At-End
         TXWHANDLE     ccbRec;                  // radio-box    Recovery
         TXWHANDLE     ccbClr;                  // radio-box    Clear BS
         TXWINDOW      window;                  // setup window data
         ULONG         style;
         USHORT        phsize;                  // parent window width
         USHORT        pvsize;                  // parent window height
         short         ww;                      // dialog window width
         short         wc;                      // std column with
         DFS_PCGPTDATA cpp;                     // private create-part data
         static BOOL   recovery = FALSE;        // default NO recovery mode
         ULN64         sectors;                 // a size in sectors
         BYTE          unit;                    // unit for size (k,m,g,c,h,s)

         memset( &cpp, 0, sizeof(DFS_PCGPTDATA)); // start empty

         cpp.gptList = dfsGptTypeSelist( TRUE); // create type list, with MANUAL

         cpp.fsp = f;                           // attach fsp reference

         cpp.vstype = 0;                        // custom system-type value
         cpp.vrbGpt = TRUE;                     // radio button Primary
         cpp.vrbSta = TRUE;                     // radio button At-Start
         cpp.vrbEnd = FALSE;                    // radio button At-End
         cpp.vrbClr = !recovery;                // mutual exclusive value
                                                // but remember previous :-)

         strcpy( cpp.vidstr, cpp.gptList->items[cpp.gptList->selected]->desc);

         strcpy(  cpp.vpobuf, "0");             // offset value
         dfsGptCpGetMaxSize( &cpp);             // calculate max sizes
         TRACES(( "fsp-info:%8.8lx, id:%2.2hu size:%9.1lf\n", f, f->id, cpp.vpsize));

         dfsGptSetDefaultName( &cpp);           // prefered partition name
         strcpy(  cpp.vavbuf, "0");             // prefered attribute value

         txwQueryWindowPos( TXHWND_DESKTOP, FALSE, &position);
         phsize = position.right  - position.left;
         pvsize = position.bottom - position.top;

         ww = min( DFSD_PCREAT_WC,  (phsize - 5));
         wc = 32;

         position.left   = (phsize - ww) /2;    // always center the dialog
         position.right  = ww;
         position.top    = (pvsize < 26) ? 0 : 1;
         position.bottom = DFSD_PCREAT_WL;

         TRECTA( "pos/size", (&position));


         style = TXWS_DIALOG | TXWS_DISABLED | TXWS_MOVEABLE | TXCS_CLOSE_BUTTON;
         txwSetupWindowData(
            position.top, position.left, position.bottom, position.right,
            style | TXWS_CAST_SHADOW,           // window frame style
            DFSH_PG_FRAME,                      // dialog help
            ' ', ' ',                           // window & border char
            TXWSCHEME_COLORS,                   // std scheme colors
            " Specify GPT-style partition properties ",
            txwstd_footer,
            &window);
         window.st.buf      = NULL;             // NO artwork attached
         window.dlgFocusID  = DFSD_WID_CESIZE;
         cframe = txwCreateWindow( TXHWND_DESKTOP, TXW_CANVAS, 0, 0, &window, NULL);
         txwSetWindowUShort( cframe, TXQWS_ID, DFSD_WID_PCRDLG);

         //- Button triggers an action, should NOT dismiss the dialog -> DLGE
         style = TXWS_PBUTTON | TXBS_DLGE_BUTTON | TXWS_HCHILD2MOVE | TXWS_VCHILD_MOVE;
         txwSetupWindowData(
            1, 2, 3, 27,
            style, DFSH_PG_CBTMAN,              // style + help
            ' ', ' ',                           // window/border chars
            TXWSCHEME_COLORS,                   // std scheme colors
            "", "",                             // title & footer text
            &window);
         window.bu.text    = "Manual Edit type GUID";
         cbtMan = txwCreateWindow(   cframe, TXW_BUTTON, cframe, 0, &window, NULL);
         txwSetWindowUShort( cbtMan, TXQWS_ID, DFSD_WID_CBTMAN);
         txwSetWindowUShort( cbtMan, TXQWS_GROUP, 1);


         style = TXWS_DSPINBT     | TXLS_DROP_ENTER  | TXWS_LEFTJUSTIFY |
                 TXWS_HCHILD2SIZE | TXWS_HCHILD2MOVE;
         txwSetupWindowData(
            1, wc +2, 2, ww - wc -5,
            style, DFSH_PG_CSPINV,              // style and help
            ' ', ' ',                           // window/border chars
            TXWSCHEME_COLORS,                   // std scheme colors
            "Select GPT partition-type", "",
            &window);
         window.lb.list = cpp.gptList;          // spin-value type list
         window.lb.cpos = cpp.gptList->selected - cpp.gptList->top;
         cspinv = txwCreateWindow(  cframe, TXW_LISTBOX, cframe, 0, &window, dfsGptCreateWinProc);
         txwSetWindowUShort( cspinv, TXQWS_ID, DFSD_WID_CSPINV);


         //- Partition type leading text string
         style = TXWS_OUTPUT | TXWS_HCHILD2MOVE;
         txwSetupWindowData(
            5, 2, 1, 28,                        // UL corner + vert/hor size
            style, 0,                           // window frame style + help
            ' ', ' ',                           // window/border chars
            TXWSCHEME_COLORS,                   // std scheme colors
            "", "",
            &window);
         window.sl.buf     = "Current partition-type GUID:";
         cptext = txwCreateWindow(   cframe, TXW_STLINE, cframe, 0, &window, NULL);
         txwSetWindowUShort( cptext, TXQWS_ID, DFSD_WID_CPTEXT);

         //- Partition type GUID description string
         style = TXWS_OUTPUT | TXWS_HCHILD2MOVE;
         txwSetupWindowData(
            5, 31, 1, 38,                       // UL corner + vert/hor size
            style, 0,                           // window frame style + help
            ' ', ' ',                           // window/border chars
            TXWSCHEME_COLORS,                   // std scheme colors
            "", "",
            &window);
         window.sl.buf     = cpp.vidstr;
         cpdesc = txwCreateWindow(   cframe, TXW_STLINE, cframe, 0, &window, NULL);
         txwSetWindowUShort( cpdesc, TXQWS_ID, DFSD_WID_CPDESC);
         TRACES(("GUID string HWND: %8.8lx\n", cpdesc));


         style = TXWS_ENTRYBT | TXWS_VCHILD2MOVE; // entryfield with title, allow move
         txwSetupWindowData(
            7, 1, 2,  13,
            style, DFSH_PG_CEOFFS,              // style and help
            ' ', ' ',                           // window/border chars
            TXWSCHEME_COLORS,                   // std scheme colors
            "Offset", "",                       // title & footer text
            &window);
         window.ef.leftcol  = 0;
         window.ef.maxcol   = TXW_INVALID;
         window.ef.curpos   = 0;
         window.ef.markCol  = 0;
         window.ef.markSize = 0;
         window.ef.rsize    = TXMAXTT;
         window.ef.buf      = cpp.vpobuf;
         window.ef.history  = NULL;
         centof = txwCreateWindow(   cframe, TXW_ENTRYFIELD, cframe, 0, &window, dfsGptCreateWinProc);
         txwSetWindowUShort( centof, TXQWS_ID, DFSD_WID_CEOFFS);


         style = TXWS_RADIOB | TXWS_HCHILD2SIZE | TXWS_VCHILD2MOVE;
         txwSetupWindowData(
            7, 15, 1, wc -13,
            style, DFSH_PG_CRBSTA,              // style + help
            ' ', ' ',                           // window/border chars
            TXWSCHEME_COLORS,                   // std scheme colors
            "", "",                             // title & footer text
            &window);
         window.bu.text    = "From start frsp.";
         window.bu.checked = &cpp.vrbSta;
         crbSta = txwCreateWindow(   cframe, TXW_BUTTON, cframe, 0, &window, dfsGptCreateWinProc);
         txwSetWindowUShort( crbSta, TXQWS_ID, DFSD_WID_CRBSTA);
         txwSetWindowUShort( crbSta, TXQWS_GROUP, 2);


         style = TXWS_RADIOB | TXWS_HCHILD2SIZE | TXWS_VCHILD2MOVE;
         txwSetupWindowData(
            8, 15, 1, wc -13,
            style, DFSH_PG_CRBEND,              // style + help
            ' ', ' ',                           // window/border chars
            TXWSCHEME_COLORS,                   // std scheme colors
            "", "",                             // title & footer text
            &window);
         window.bu.text    = "From end freesp.";
         window.bu.checked = &cpp.vrbEnd;
         crbEnd = txwCreateWindow(   cframe, TXW_BUTTON, cframe, 0, &window, dfsGptCreateWinProc);
         txwSetWindowUShort( crbEnd, TXQWS_ID, DFSD_WID_CRBEND);
         txwSetWindowUShort( crbEnd, TXQWS_GROUP, 2);


         style = TXWS_AUTOCHK | TXWS_HCHILD2SIZE | TXWS_VCHILD_MOVE;
         txwSetupWindowData(
            7, wc +4, 1, ww - wc -7,
            style, DFSH_PG_CCBMAX,              // style + help
            ' ', ' ',                           // window/border chars
            TXWSCHEME_COLORS,                   // std scheme colors
            "", "",                             // title & footer text
            &window);
         window.bu.text    = cpp.tpsbuf;
         window.bu.checked = &cpp.vcbMax;
         ccbMax = txwCreateWindow(   cframe, TXW_BUTTON, cframe, 0, &window, dfsGptCreateWinProc);
         txwSetWindowUShort( ccbMax, TXQWS_ID, DFSD_WID_CCBMAX);


         style = TXWS_ENTRYB      |             // entryfield without title, allow size and move
                 TXWS_HCHILD2SIZE | TXWS_HCHILD2MOVE | TXWS_VCHILD2MOVE;
         txwSetupWindowData(
            8, wc +4, 1,  ww - wc -7,
            style, DFSH_PG_CESIZE,              // style and help
            ' ', ' ',                           // window/border chars
            TXWSCHEME_COLORS,                   // std scheme colors
            "", "",                             // title & footer text
            &window);
         window.ef.leftcol  = 0;
         window.ef.maxcol   = TXW_INVALID;
         window.ef.curpos   = 0;
         window.ef.markCol  = 0;
         window.ef.markSize = 0;
         window.ef.rsize    = TXMAXTT;
         window.ef.buf      = cpp.vpsbuf;
         window.ef.history  = NULL;
         centry = txwCreateWindow(   cframe, TXW_ENTRYFIELD, cframe, 0, &window, dfsGptCreateWinProc);
         txwSetWindowUShort( centry, TXQWS_ID, DFSD_WID_CESIZE);


         if (dfsa->expertui)
         {
            style = TXWS_AUTOCHK | TXWS_HCHILD2SIZE | TXWS_VCHILD_MOVE;
            txwSetupWindowData(
               10, 1, 1, ww,
               style, DFSH_PG_CCBALN,           // style + help
               ' ', ' ',                        // window/border chars
               TXWSCHEME_COLORS,                // std scheme colors
               "", "",                          // title & footer text
               &window);
            window.bu.text    = "Do NOT align the offset and size values to standard boundaries";
            window.bu.checked = &cpp.vcbAln;
            ccbAln = txwCreateWindow(   cframe, TXW_BUTTON, cframe, 0, &window, dfsGptCreateWinProc);
            txwSetWindowUShort( ccbAln, TXQWS_ID, DFSD_WID_CCBALN);
         }


         style = TXWS_AUTORAD | TXWS_HCHILD2SIZE | TXWS_VCHILD_MOVE;
         txwSetupWindowData(
            12, 1, 1, ww,
            style, DFSH_PG_CRBREC,              // style + help
            ' ', ' ',                           // window/border chars
            TXWSCHEME_COLORS,                   // std scheme colors
            "", "",                             // title & footer text
            &window);
         window.bu.text    = "Recovery mode, keep existing bootsector   (sector 0) intact";
         window.bu.checked = &recovery;
         ccbRec = txwCreateWindow(   cframe, TXW_BUTTON, cframe, 0, &window, dfsGptCreateWinProc);
         txwSetWindowUShort( ccbRec, TXQWS_ID, DFSD_WID_CRBREC);
         txwSetWindowUShort( ccbRec, TXQWS_GROUP, 4);


         style = TXWS_AUTORAD | TXWS_HCHILD2SIZE | TXWS_VCHILD_MOVE;
         txwSetupWindowData(
            13, 1, 1, ww,
            style, DFSH_PG_CRBCLR,              // style + help
            ' ', ' ',                           // window/border chars
            TXWSCHEME_COLORS,                   // std scheme colors
            "", "",                             // title & footer text
            &window);
         window.bu.text    = "Create NEW mode, clear existing bootsector with 0xF6 pattern";
         window.bu.checked = &cpp.vrbClr;
         ccbClr = txwCreateWindow(   cframe, TXW_BUTTON, cframe, 0, &window, dfsGptCreateWinProc);
         txwSetWindowUShort( ccbClr, TXQWS_ID, DFSD_WID_CRBCLR);
         txwSetWindowUShort( ccbClr, TXQWS_GROUP, 4);


         if (dfsa->expertui)
         {
            //- Attribute HEX indicator '0x', output only
            style = TXWS_OUTPUTT | TXWS_HCHILD_MOVE | TXWS_VCHILD_MOVE;
            txwSetupWindowData(
               15, 1, 2,  3,
               style, 0,                        // style and help
               ' ', ' ',                        // window/border chars
               TXWSCHEME_COLORS,                // scheme colors
               "Hex", "",
               &window);
            window.sl.buf = " 0x";
            chex0x = txwCreateWindow( cframe, TXW_STLINE, cframe, 0, &window, dfsGptCreateWinProc);


            style = TXWS_ENTRYBT | TXWS_HCHILD_MOVE | TXWS_VCHILD_MOVE;
            txwSetupWindowData(
               15, 4, 2,  19,
               style, DFSH_PG_CATVAL,           // style and help
               ' ', ' ',                        // window/border chars
               TXWSCHEME_COLORS,                // std scheme colors
               "Attribute flags ", "",          // title & footer text
               &window);
            window.ef.leftcol  = 0;
            window.ef.maxcol   = TXW_INVALID;
            window.ef.curpos   = 0;
            window.ef.markCol  = 0;
            window.ef.markSize = 0;
            window.ef.rsize    = 16;             // 16 hex digits
            window.ef.buf      = cpp.vavbuf;
            window.ef.history  = NULL;
            centav = txwCreateWindow(   cframe, TXW_ENTRYFIELD, cframe, 0, &window, dfsGptCreateWinProc);
            txwSetWindowUShort( centav, TXQWS_ID, DFSD_WID_CATVAL);
         }

         style = TXWS_ENTRYBT | TXWS_HCHILD_MOVE | TXWS_VCHILD_MOVE;
         txwSetupWindowData(
            15, wc -2, 2,  40,
            style, DFSH_PG_CPNAME,              // style and help
            ' ', ' ',                           // window/border chars
            TXWSCHEME_COLORS,                   // std scheme colors
            "Partition Name", "",               // title & footer text
            &window);
         window.ef.leftcol  = 0;
         window.ef.maxcol   = TXW_INVALID;
         window.ef.curpos   = 0;
         window.ef.markCol  = 0;
         window.ef.markSize = 0;
         window.ef.rsize    = GPT_PNAME_LEN + 1;
         window.ef.buf      = cpp.vpnbuf;
         window.ef.history  = NULL;
         centpn = txwCreateWindow(   cframe, TXW_ENTRYFIELD, cframe, 0, &window, dfsGptCreateWinProc);
         txwSetWindowUShort( centpn, TXQWS_ID, DFSD_WID_CPNAME);


         if (dfsa->expertui)
         {
            //- Button triggers an action, should NOT dismiss the dialog -> DLGE
            style = TXWS_PBUTTON | TXBS_DLGE_BUTTON | TXWS_HCHILD2MOVE | TXWS_VCHILD_MOVE;
            txwSetupWindowData(
               18, 2, 3, 27,
               style, DFSH_PG_CBTATT,           // style + help
               ' ', ' ',                        // window/border chars
               TXWSCHEME_COLORS,                // std scheme colors
               "", "",                          // title & footer text
               &window);
            window.bu.text    = "Specify Attribute Flags";
            cbtAtt = txwCreateWindow(   cframe, TXW_BUTTON, cframe, 0, &window, NULL);
            txwSetWindowUShort( cbtAtt, TXQWS_ID, DFSD_WID_CBTATT);
            txwSetWindowUShort( cbtAtt, TXQWS_GROUP, 1);
         }

         style = TXWS_PBUTTON | TXWS_VCHILD_MOVE;
         txwSetupWindowData(
            18, ww - 28,                        // position
            3, strlen( dfsdf_bt_ok) + 4,        // vertical/horizontal size
            style, DFSH_PG_CBUTTN,              // style and help
            ' ', ' ', TXWSCHEME_COLORS, "",  "",
            &window);
         window.bu.text = dfsdf_bt_ok;
         cbtok = txwCreateWindow(   cframe, TXW_BUTTON, cframe, 0, &window,
                                    txwDefWindowProc);
         txwSetWindowUShort( cbtok, TXQWS_ID, TXMBID_OK); // rc == ID


         style = TXWS_PBUTTON | TXWS_HCHILD2MOVE | TXWS_VCHILD_MOVE;
         txwSetupWindowData(
            18, ww - 14,                        // position
            3, strlen( dfsdf_bt_cancel) + 4,    // vertical/horizontal size
            style, DFSH_PG_CBUTTN,              // style and help
            ' ', ' ', TXWSCHEME_COLORS, "",  "",
            &window);
         window.bu.text = dfsdf_bt_cancel;
         cbtcan = txwCreateWindow(  cframe, TXW_BUTTON, cframe, 0, &window,
                                    txwDefWindowProc);
         txwSetWindowUShort( cbtcan, TXQWS_ID, TXMBID_CANCEL); // rc == ID

         result = txwDlgBox( TXHWND_DESKTOP, TXHWND_DESKTOP, dfsGptCrPartDlgWinProc, cframe, &cpp);
         if (result != TXDID_CANCEL)
         {
            if (create != NULL)
            {
               ULONG   ltype = cpp.gptList->items[cpp.gptList->selected]->value - TXDID_MAX;
               double  size  = 0.0;
               ULN64   attr  = 0;
               TXTM    optstr;                  // option string

               if (ltype == DFSGPT_CUSTOM) // custom specified type
               {
                  if (dfsUidStringIsValid( cpp.vidstr) == FALSE) // not specified yet
                  {
                     TXTT  s1 = {0};

                     if (dfsGuidUuidDialog( " Specify partition type as a GPT-style GUID ",
                                              NULL, 0, DFSH_PG_CSPINV, s1) != TXDID_CANCEL)
                     {
                        strcpy( cpp.vidstr, s1);
                     }
                  }
               }
               else
               {
                  strcpy( cpp.vidstr, cpp.gptList->items[cpp.gptList->selected]->desc);
               }
               sprintf( create, "cr gpt -f:%hu -name:'%s' -t:'%36.36s'",
                                     f->id, cpp.vpnbuf, cpp.vidstr + 1);

               if (!cpp.vcbMax) // Use MAX size, so do NOT specify!
               {
                  dfsGptCpGetMaxSize( &cpp);    // calculate max/derived sizes

                  sectors = TxaParseNumber( cpp.vpsbuf, DFSRADIX_SIZE, &unit);
                  sectors = dfsApplyNumberUnit( sectors, (unit == TXA_DFUNIT) ? 'm' : unit, DFSTORE);
                  if (sectors)
                  {
                     size = (double) (DFSECT2MIB(sectors, dfsGetSectorSize())) + 0.001;
                  }
                  if ((size <= cpp.fssize) && (size != 0)) // given size acceptable
                  {
                     cpp.vpsize = size;
                  }
                  TRACES(( "size: %lf  vpsize: %lf  ltype:%lu  type:'%s' offset:%s\n",
                            size,  cpp.vpsize,  ltype, cpp.vidstr, cpp.vpobuf));

                  if (unit == 's')              // size in sectors, use field 'as is'
                  {
                     sprintf( optstr, " -s:%s", cpp.vpsbuf);
                  }
                  else
                  {
                     sprintf( optstr, " -s:%.0lf", (double) (cpp.vpsize - 0.5));
                  }
                  strcat(  create, optstr);
               }

               sscanf( cpp.vavbuf, "%llX", &attr);
               if (attr != 0)
               {
                  sprintf( optstr, " -attrib:'%16.16llX'", attr);
                  strcat(  create, optstr);
               }

               if (cpp.vcbAln)                  // no alignment wanted
               {
                  strcat( create, " -align-");
               }

               sectors = TxaParseNumber( cpp.vpobuf, DFSRADIX_SIZE, &unit);
               sectors = dfsApplyNumberUnit( sectors, (unit == TXA_DFUNIT) ? 'm' : unit, DFSTORE);
               if (sectors == 0)                // empty or zero
               {
                  if (cpp.vrbEnd)
                  {
                     strcat( create, " -e");
                  }
               }
               else                             // valid, non-zero offset value
               {
                  if (cpp.vrbEnd)
                  {
                     strcat( create, " -e:");
                  }
                  else
                  {
                     strcat( create, " -r:");
                  }
                  strcat( create, cpp.vpobuf);
               }
               if (cpp.vrbClr)
               {
                  strcat( create, " -C");
               }
            }
            rc = TRUE;
         }
         txSelDestroy( &(cpp.gptList));         // remove GPT type list
      }
   }
   BRETURN (rc);
}                                               // end 'txwCreateGptPartDialog'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Dialog window procedure, for the GPT-style CreatePart Dialog
/*****************************************************************************/
static ULONG dfsGptCrPartDlgWinProc             // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    msg param 1
   TXWMPARAM           mp2                      // IN    msg param 2
)
{
   ULONG               rc    = NO_ERROR;

   ENTER();
   TRCMSG( hwnd, msg, mp1, mp2);
   if (hwnd != 0)
   {
      DFS_PCGPTDATA   *cpd  = txwQueryWindowPtr(  hwnd, TXQWP_USER);
      TXSELIST        *list = NULL;
      TXS_ITEM        *item = NULL;
      TXTT             s1   = {0};
      ULN64            attr =  0;
      TXWHANDLE        ctrl = NULL;             // handle for a child control window

      switch (msg)
      {
         case TXWM_COMMAND:
            switch ((ULONG) mp2)                // command source
            {
               case TXCMDSRC_RADIOBUTTON:
                  switch ((ULONG) mp1)
                  {
                     case DFSD_WID_CRBSTA:
                     case DFSD_WID_CRBEND:
                        cpd->vrbSta =  cpd->vrbEnd;
                        cpd->vrbEnd = !cpd->vrbEnd;
                        txwInvalidateWindow( hwnd, FALSE, TRUE); // dlg incl controls
                        break;

                     default:
                        break;
                  }
                  break;

               case TXCMDSRC_CHECKBOX:
                  switch ((ULONG) mp1)
                  {
                     case DFSD_WID_CCBMAX:
                     case DFSD_WID_CCBALN:
                        dfsGptCpGetMaxSize( cpd); // calculate max sizes
                        txwInvalidateWindow( hwnd, FALSE, TRUE); // dlg incl controls
                        break;

                     default:
                        break;
                  }
                  break;

               case TXCMDSRC_PUSHBUTTON:
                  switch ((ULONG) mp1)
                  {
                     case DFSD_WID_CBTMAN:
                        if (strlen( cpd->vidstr) != 0) // show existing value
                        {
                           strcpy( s1, cpd->vidstr);
                        }
                        if (dfsGuidUuidDialog( " Specify partition type as a GPT-style GUID ",
                                                 NULL, 0, DFSH_PG_CBTMAN, s1) != TXDID_CANCEL)
                        {
                           cpd->gptList->selected = cpd->gptList->count -1; // set to last (custom)
                           strcpy( cpd->vidstr, s1);
                        }
                        dfsGptSetDefaultName( cpd);
                        txwInvalidateWindow( hwnd, FALSE, TRUE); // dlg incl controls
                        break;

                     case DFSD_WID_CBTATT:
                        sscanf(  cpd->vavbuf, "%llX", &attr);
                        dfsGptAttributeDialog( &attr);
                        sprintf( cpd->vavbuf, "%16.16llx", attr);
                        txwInvalidateWindow( hwnd, FALSE, TRUE); // dlg incl controls
                        break;

                     default:
                        rc = txwDefDlgProc( hwnd, msg, mp1, mp2);
                        break;
                  }
                  break;

               default:
                  rc = txwDefDlgProc( hwnd, msg, mp1, mp2);
                  break;
            }
            break;

         case TXWM_CONTROL:
            if ((list = (TXSELIST *) mp2) != NULL)
            {
               item = list->items[list->selected];
            }
            TRACES(("WM_CONTROL list:%8.8lx item:%8.8lx, selected %lu = 0x%02hx\n",
                    list, item, list->selected, list->items[list->selected]->value));
            if (item != NULL)
            {
               if (TXSH2FROMMP(mp1) == TXLN_SELECT)
               {
                  if (dfsUidStringIsValid( item->desc)) // item is a GUID, so set it current
                  {
                     //- moving through the list, without opening it, but DO make it current!
                     strcpy( cpd->vidstr, item->desc);
                  }
                  txwPostMsg( TXHWND_DESKTOP, TXWM_SETFOOTER, (TXWMPARAM) item->desc, 0);
               }
               else if (TXSH2FROMMP(mp1) == TXLN_ENTER)
               {
                  if (item->value == (TXDID_MAX + DFSGPT_CUSTOM)) // Partition custom type value
                  {
                     if (strlen( cpd->vidstr) != 0)      // show existing value
                     {
                        strcpy( s1, cpd->vidstr);
                     }
                     if (dfsGuidUuidDialog( " Specify partition type as a GPT-style GUID ",
                                              NULL, 0, DFSH_PG_CSPINV, s1) != TXDID_CANCEL)
                     {
                        strcpy( cpd->vidstr, s1);
                     }
                  }
                  else                          // other list item, copy GUID string
                  {
                     strcpy( cpd->vidstr, item->desc);
                  }
               }
               dfsGptSetDefaultName( cpd);
               txwInvalidateWindow( hwnd, FALSE, TRUE); // dlg incl controls
            }
            break;

         case TXWM_CHAR:
            switch ((ULONG) mp2)
            {
               case TXk_F10:                    // popup (list) requested
                  ctrl = txwWindowFromID( hwnd, DFSD_WID_CSPINV);
                  txwPostMsg( ctrl, TXWM_CHAR, 0, (TXWMPARAM) TXk_ENTER); // ENTER on list, popup
                  break;

               default:
                  rc = txwDefDlgProc( hwnd, msg, mp1, mp2);
                  break;
            }
            break;

         default:
            rc = txwDefDlgProc( hwnd, msg, mp1, mp2);
            break;
      }
   }
   else
   {
      rc = TX_INVALID_HANDLE;
   }
   RETURN( rc);
}                                               // end 'dfsGptCrPartDlgWinProc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Dialog window procedure, for the CREATE-GPT partition dialog fields
/*****************************************************************************/
static ULONG dfsGptCreateWinProc               // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    msg param 1
   TXWMPARAM           mp2                      // IN    msg param 2
)
{
   ULONG               rc = NO_ERROR;

   ENTER();
   TRCMSG( hwnd, msg, mp1, mp2);

   if (hwnd != 0)
   {
      TXWHANDLE       owner  = txwQueryWindow( hwnd, TXQW_OWNER); // get dialog info
      DFS_PCGPTDATA  *cpd    = txwQueryWindowPtr(  owner, TXQWP_USER);
      USHORT          wid    = txwQueryWindowUShort( hwnd, TXQWS_ID);

      switch (msg)
      {
         case TXWM_SETFOCUS:
            if ((wid == DFSD_WID_CEOFFS) && (((BOOL) mp1) == FALSE))
            {
               TRACES(("Offset lost focus, value: '%s'\n", cpd->vpobuf));
               dfsGptCpGetMaxSize( cpd);        // calculate max sizes
               txwInvalidateWindow( owner, FALSE, TRUE); // dlg incl controls
            }
         default:                               // do standard processing too
            rc = txwDefWindowProc( hwnd, msg, mp1, mp2);
            break;
      }
   }
   else
   {
      rc = TX_INVALID_HANDLE;
   }
   RETURN( rc);
}                                               // end 'dfsGptCreateWinProc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Calculate exact maximum partitionsize based on alignment and offset/position
/*****************************************************************************/
static void dfsGptCpGetMaxSize
(
   DFS_PCGPTDATA      *pcd                      // IN    Part Create Data
)
{
   ULN64               offset   = 0;            // specified offset in sectors
   ULN64               slack    = 0;            // alignment slack, sectors
   ULN64               sectors  = 0;            // available size in sectors
   ULN64               size     = 0;            // field size in sectors
   BYTE                s_unit;                  // unit for size (k,m,g,c,h,s)
   BYTE                o_unit;                  // unit for offset value
   ULN64               alignSec;

   ENTER();

   alignSec = dfsAlignmentSectors( 'G', pcd->fsp); // get current alignment sectors

   //- Calculate offset value in SECTORS
   offset = TxaParseNumber( pcd->vpobuf, DFSRADIX_SIZE, &o_unit);
   offset = dfsApplyNumberUnit( offset, (o_unit == TXA_DFUNIT) ? 'm' : o_unit, DFSTORE);

   if ((o_unit == 's') || pcd->vcbAln)          // specified in sectors, or No-Align
   {                                            // force SIZE to maximum, in sectors
      if (pcd->fsp->sectors > offset)           // offset within freespace area
      {
         sectors = pcd->fsp->sectors - offset;  // raw size in sectors
      }
   }
   else                                         // no sectors, align as specified
   {
      if ((pcd->fsp->basePsn % alignSec) != 0)  // start of freespace NOT aligned
      {
         slack = alignSec - (pcd->fsp->basePsn % alignSec); // front-slack
      }
      //- calculate specified offset, aligned to 1-MiB multiples, consider front-slack
      offset = ((offset + alignSec -1) / alignSec) * alignSec;

      TRACES(("offset: 0x%llx  basePsn:0x%llx, slack:0x%llx\n", offset, pcd->fsp->basePsn, slack));

      if ((offset >= alignSec) && (slack != 0)) // correct perceived offset for alignment
      {
         TRACES(("Correcting offset by alignment\n"));
         offset -= alignSec;                    // by subtracting alignSec
         offset += slack;                       // (slack will be close to alignSec)
         sprintf( pcd->vpobuf, "0x%llx,s", offset); // new offset value, as sectors
      }
      else
      {
         offset += slack;
         sprintf( pcd->vpobuf, "%.0lf", (double) (offset / alignSec)); // new offset value, MiB
      }

      TRACES(("Aligned offset: 0x%llx sectors,  offset field: '%s'\n", offset, pcd->vpobuf));

      if (pcd->fsp->sectors > offset)           // offset within freespace area
      {
         sectors = pcd->fsp->sectors - offset;  // raw size in sectors
         slack   = sectors % alignSec;          // slack size at end
         sectors = sectors - slack;             // aligned (at end) maximum size
         TRACES(("aligned sectors:0x%llx  slack:0x%llx\n", sectors, slack));
      }
   }
   pcd->fssize = TXSMIB( sectors, pcd->fsp->bpsector);
   pcd->vpsize = pcd->fssize;                   // show max in MiB, 2 decimals not rounded!
   sprintf( pcd->tpsbuf, "Use MAX size: %.2lf MiB", pcd->vpsize);

   //- handle the specified size field contents, adjust to MAX when too large
   size = TxaParseNumber( pcd->vpsbuf, DFSRADIX_SIZE, &s_unit);
   size = dfsApplyNumberUnit( size, (s_unit == TXA_DFUNIT) ? 'm' : s_unit, DFSTORE);

   //- When not specified, or (close to the) MAX for offset alignment, redo the SIZE field
   if ((pcd->vcbMax) || (pcd->vpsbuf[0] == '*') ||
       (size == 0)   || ((size + max( 0x1000, (2*alignSec))) > sectors))
   {
      //- Use 's when specified and NOT at MAX size, or when no-alignment is used
      if (((s_unit == 's') && (size < sectors)) || pcd->vcbAln)
      {
         sprintf( pcd->vpsbuf, "0x%llx,s", sectors);   //- new size value, raw
      }
      else
      {
         if (pcd->vcbMax)
         {
            strcpy(  pcd->vpsbuf, "*");
         }
         else
         {
            if (pcd->vpsize > 0.5)
            {
               sprintf( pcd->vpsbuf, "%.0lf", pcd->vpsize - 0.5); // new size, MiB rounded down
            }
            else
            {
               strcpy(  pcd->vpsbuf, "0");
            }
         }
      }
   }
   VRETURN();
}                                               // end 'dfsGptCpGetMaxSize'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set a new default partition name based on current type GUID, unless edited
/*****************************************************************************/
static void dfsGptSetDefaultName
(
   DFS_PCGPTDATA      *pcd                      // IN    Create Partition data
)
{
   DFS_GUID            typeGuid = {0};          // Binary type GUID

   ENTER();
   TRACES(("Name IN : '%s'\n", pcd->vpnbuf));

   if ((strlen(  pcd->vpnbuf) == 0)       ||       //- empty
       (strncmp( pcd->vpnbuf, "New: ", 5) == 0))   //- or still default ?
   {
      dfsUidStringToBinary( pcd->vidstr, typeGuid, TRUE);
      sprintf( pcd->vpnbuf, "New: %s", dfsGptGuidDescription( typeGuid));
      TxStrip( pcd->vpnbuf, pcd->vpnbuf, 0, ' ');  //- strip trailing spaces
   }
   TRACES(("Name OUT: '%s'\n", pcd->vpnbuf));
   VRETURN();
}                                           // end 'dfsGptSetDefaultName'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Prompt for (new) GPT-style partition type using custom prompt string
/*****************************************************************************/
BOOL dfsPromptGptPartitionType
(
   char               *prompt,                  // IN    specific prompt string
   ULONG               helpid,                  // IN    specific help-id
   char               *gptype                   // INOUT GPT type string (38)
)
{
   BOOL                rc = TRUE;               // function return
   ULONG               listcode = TXDID_OK;
   TXRECT              where = {18,2,0,0};      // fixed position
   TXTT                s1    = {0};             // type string, empty

   TXSELIST           *gpTypeList = dfsGptTypeSelist( TRUE); // create type list, including MANUAL

   ENTER();

   if (gpTypeList != NULL)
   {
      listcode = txwListBox( TXHWND_DESKTOP, TXHWND_DESKTOP, &where,
                             prompt, "", helpid, TXLB_MOVEABLE,
                             cSchemeColor, cSchemeColor, // std list colors
                             gpTypeList);
      switch (listcode)
      {
         case TXDID_MAX + DFSGPT_CUSTOM:
            if (dfsGuidUuidDialog( " Specify partition type as a GPT-style GUID ",
                                     NULL, 0, helpid, s1) != TXDID_CANCEL)
            {
               strcpy( gptype, s1);
            }
            else
            {
               rc = FALSE;
            }
            break;

         case TXDID_CANCEL:
            rc = FALSE;
            break;

         default:
            strcpy( gptype, gpTypeList->items[gpTypeList->selected]->desc);
            break;
      }
      txSelDestroy( &(gpTypeList));             // remove GPT type list
   }
   BRETURN (rc);
}                                           // end 'dfsPromptGptPartitionType'
/*---------------------------------------------------------------------------*/


/*======================= GPT ATTRIBUTES ====================================*/
static BOOL        gptaEsp  = FALSE;            // EFI System Partition
static BOOL        gptaIgn  = FALSE;            // EFI to-be ignored, no read
static BOOL        gptaEbb  = FALSE;            // Bios Boot (aka active)

static BOOL        gptaCp0  = FALSE;            // Chrome boot-prio 0
static BOOL        gptaCp1  = FALSE;            // Chrome boot-prio 1
static BOOL        gptaCp2  = FALSE;            // Chrome boot-prio 2
static BOOL        gptaCp3  = FALSE;            // Chrome boot-prio 3
static BOOL        gptaCt0  = FALSE;            // Chrome rem-tries 0
static BOOL        gptaCt1  = FALSE;            // Chrome rem-tries 1
static BOOL        gptaCt2  = FALSE;            // Chrome rem-tries 2
static BOOL        gptaCt3  = FALSE;            // Chrome rem-tries 3
static BOOL        gptaCsb  = FALSE;            // Chrome successful boot

static BOOL        gptaMro  = FALSE;            // MS Readonly
static BOOL        gptaMhd  = FALSE;            // MS Hidden
static BOOL        gptaMnm  = FALSE;            // MS No auto-mount

/*

[ ] bit  0: EFI System Partition  (ESP)
[ ] bit  1: EFI To be Ignored, no read by EFI
[ ] bit  2: EFI Bios Bootable  (aka 'active')

[ ] bit 48: Chrome-OS Boot priority,   bit-0
[ ] bit 49: Apple boot volume / B-prio bit-1
[ ] bit 50: Chrome-OS Boot priority,   bit-2
[ ] bit 51: Chrome-OS Boot priority,   bit-3
[ ] bit 52: Chrome-OS Remaining tries, bit-0
[ ] bit 53: Chrome-OS Remaining tries, bit-1
[ ] bit 54: Chrome-OS Remaining tries, bit-2
[ ] bit 55: Chrome-OS Remaining tries, bit-3
[ ] bit 56: Chrome-OS Kernel boot successful

[ ] bit 60: Microsoft Read-Only filesystem
[ ] bit 62: Microsoft Hidden    filesystem
[ ] bit 63: Microsoft No automatic Mounting

*/

#define   DFSGPTATTRWIDGETS  15
static TXWIDGET  DfsGptAttrWidgets[DFSGPTATTRWIDGETS] =    // order determines TAB-order!
{
   { 1, 0,  1, 46, 0, 2, 0, TXWS_AUTOCHK, 0, TXStdButton( &gptaEsp, "bit  0: EFI System Partition  (ESP)"       )},
   { 2, 0,  1, 46, 0, 2, 0, TXWS_AUTOCHK, 0, TXStdButton( &gptaIgn, "bit  1: EFI To be Ignored, no read by EFI" )},
   { 3, 0,  1, 46, 0, 2, 0, TXWS_AUTOCHK, 0, TXStdButton( &gptaEbb, "bit  2: EFI Bios Bootable  (aka 'active')" )},

   { 5, 0,  1, 46, 0, 2, 0, TXWS_AUTOCHK, 0, TXStdButton( &gptaCp0, "bit 48: Chrome-OS Boot priority,   bit-0"  )},
   { 6, 0,  1, 46, 0, 2, 0, TXWS_AUTOCHK, 0, TXStdButton( &gptaCp1, "bit 49: Apple boot volume / B-prio bit-1"  )},
   { 7, 0,  1, 46, 0, 2, 0, TXWS_AUTOCHK, 0, TXStdButton( &gptaCp2, "bit 50: Chrome-OS Boot priority,   bit-2"  )},
   { 8, 0,  1, 46, 0, 2, 0, TXWS_AUTOCHK, 0, TXStdButton( &gptaCp3, "bit 51: Chrome-OS Boot priority,   bit-3"  )},
   { 9, 0,  1, 46, 0, 2, 0, TXWS_AUTOCHK, 0, TXStdButton( &gptaCt0, "bit 52: Chrome-OS Remaining tries, bit-0"  )},
   {10, 0,  1, 46, 0, 2, 0, TXWS_AUTOCHK, 0, TXStdButton( &gptaCt1, "bit 53: Chrome-OS Remaining tries, bit-1"  )},
   {11, 0,  1, 46, 0, 2, 0, TXWS_AUTOCHK, 0, TXStdButton( &gptaCt2, "bit 54: Chrome-OS Remaining tries, bit-2"  )},
   {12, 0,  1, 46, 0, 2, 0, TXWS_AUTOCHK, 0, TXStdButton( &gptaCt3, "bit 55: Chrome-OS Remaining tries, bit-3"  )},
   {13, 0,  1, 46, 0, 2, 0, TXWS_AUTOCHK, 0, TXStdButton( &gptaCsb, "bit 56: Chrome-OS Kernel boot successful"  )},

   {15, 0,  1, 46, 0, 2, 0, TXWS_AUTOCHK, 0, TXStdButton( &gptaMro, "bit 60: Microsoft Read-Only filesystem"    )},
   {16, 0,  1, 46, 0, 2, 0, TXWS_AUTOCHK, 0, TXStdButton( &gptaMhd, "bit 62: Microsoft Hidden    filesystem"    )},
   {17, 0,  1, 46, 0, 2, 0, TXWS_AUTOCHK, 0, TXStdButton( &gptaMnm, "bit 63: Microsoft No automatic mounting"   )}
};

static TXGW_DATA DfsGptAttrDlg =
{
   DFSGPTATTRWIDGETS,                           // number of widgets
   DFSH_PG_CBTATT,                              // help, widget overrules
   810,                                         // base window ID
   NULL,                                        // widget window procedure
   NULL,                                        // persistent position TXRECT
   DfsGptAttrWidgets                            // array of widgets
};


/*************************************************************************************************/
// Present GPT attribute display and update dialog and return the possibly modified value
/*************************************************************************************************/
void dfsGptAttributeDialog
(
   ULN64              *attrib                   // INOUT 64 GPT attribute bits
)
{
   ULN64               flags = *attrib;

   ENTER();
   TRACES(("IN:  %8.8lx\n", flags));

   gptaEsp = ((flags & GPT_ATTR_SYSTEM_PARTITION) != 0);
   gptaIgn = ((flags & GPT_ATTR_TO_BE_SW_IGNORED) != 0);
   gptaEbb = ((flags & GPT_ATTR_LEGACY_BIOS_BOOT) != 0);

   gptaCp0 = ((flags & GPT_ATTR_CHR_BOOT_PRIOR_0) != 0);
   gptaCp1 = ((flags & GPT_ATTR_CHR_BOOT_PRIOR_1) != 0);
   gptaCp2 = ((flags & GPT_ATTR_CHR_BOOT_PRIOR_2) != 0);
   gptaCp3 = ((flags & GPT_ATTR_CHR_BOOT_PRIOR_3) != 0);
   gptaCt0 = ((flags & GPT_ATTR_CHR_REMN_TRIES_0) != 0);
   gptaCt1 = ((flags & GPT_ATTR_CHR_REMN_TRIES_1) != 0);
   gptaCt2 = ((flags & GPT_ATTR_CHR_REMN_TRIES_2) != 0);
   gptaCt3 = ((flags & GPT_ATTR_CHR_REMN_TRIES_3) != 0);
   gptaCsb = ((flags & GPT_ATTR_CHR_SUCCESS_BOOT) != 0);

   gptaMro = ((flags & GPT_ATTR_MS_PART_READONLY) != 0);
   gptaMhd = ((flags & GPT_ATTR_MS_PART_ISHIDDEN) != 0);
   gptaMnm = ((flags & GPT_ATTR_MS_NO_AUTO_MOUNT) != 0);

   if (txwWidgetDialog( TXHWND_DESKTOP, TXHWND_DESKTOP,
           NULL, " Inspect/modify GPT attribute bits ",
           TXWD_MOVEABLE | TXWD_HCENTER | TXWD_VCENTER,
           0,   &DfsGptAttrDlg) != TXDID_CANCEL)
   {
      flags &= ~GPT_MASK_DEFINED_FLAGS;         // clear the defined ones

      if (gptaEsp) flags |= GPT_ATTR_SYSTEM_PARTITION;
      if (gptaIgn) flags |= GPT_ATTR_TO_BE_SW_IGNORED;
      if (gptaEbb) flags |= GPT_ATTR_LEGACY_BIOS_BOOT;

      if (gptaCp0) flags |= GPT_ATTR_CHR_BOOT_PRIOR_0;
      if (gptaCp1) flags |= GPT_ATTR_CHR_BOOT_PRIOR_1;
      if (gptaCp2) flags |= GPT_ATTR_CHR_BOOT_PRIOR_2;
      if (gptaCp3) flags |= GPT_ATTR_CHR_BOOT_PRIOR_3;
      if (gptaCt0) flags |= GPT_ATTR_CHR_REMN_TRIES_0;
      if (gptaCt1) flags |= GPT_ATTR_CHR_REMN_TRIES_1;
      if (gptaCt2) flags |= GPT_ATTR_CHR_REMN_TRIES_2;
      if (gptaCt3) flags |= GPT_ATTR_CHR_REMN_TRIES_3;
      if (gptaCsb) flags |= GPT_ATTR_CHR_SUCCESS_BOOT;

      if (gptaMro) flags |= GPT_ATTR_MS_PART_READONLY;
      if (gptaMhd) flags |= GPT_ATTR_MS_PART_ISHIDDEN;
      if (gptaMnm) flags |= GPT_ATTR_MS_NO_AUTO_MOUNT;

      *attrib = flags;                          // return combined value
   }
   TRACES(("OUT: %8.8lx\n", flags));
   VRETURN();
}                                               // end 'dfsGptAttributeDialog'
/*---------------------------------------------------------------------------*/


#endif
