//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// FDISK Dialog and windowing functions
//
// Author: J. van Wijk
//
// JvW  21-09-2015 Split up, seperate CR-MBR and CR-GPT modules
// JvW  18-11-2002 Initial version, derived from TXWSTDLG and DFSCFDSK
//


#include <txlib.h>                              // TX library interface

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfsupart.h>                           // FDISK partition functions
#include <dfsafdsk.h>                           // FDISK display & analysis
#include <dfscfdsk.h>                           // FDISK callback and commands
#include <dfsdfdsk.h>                           // FDISK dialog functions
#include <dfsufdsk.h>                           // FDISK utility functions
#include <dfswin.h>                             // window and help definitions
#include <dfsver.h>                             // version and naming info
#include <dfsosapi.h>                           // OS specific stuff

#if defined (USEWINDOWING)

// LVMSET dialog specific definitions

#define DFSD_WID_LVMDLG     690                 // window id dialog frame
#define DFSD_WID_TEXT       691                 // window id info text
#define DFSD_WID_EVOL       692                 // window id entry volume
#define DFSD_WID_EPAR       693                 // window id entry partition
#define DFSD_WID_ELET       694                 // window id entry driveletter
#define DFSD_WID_CMEN       695                 // window id checkbox BM menu
#define DFSD_WID_CINS       696                 // window id checkbox install
#define DFSD_WID_CSIZ       697                 // window id checkbox size sync
#define DFSD_WID_CSNC       698                 // window id checkbox sync LVM

#define DFSH_LS             (DFSH_FDISK + 690)
#define DFSH_LS_EVOL        (DFSH_LS + 1)
#define DFSH_LS_EPAR        (DFSH_LS + 1)
#define DFSH_LS_ELET        (DFSH_LS + 1)       // same help chapter for all
#define DFSH_LS_CMEN        (DFSH_LS + 1)
#define DFSH_LS_CINS        (DFSH_LS + 1)
#define DFSH_LS_CSIZ        (DFSH_LS + 1)
#define DFSH_LS_CSNC        (DFSH_LS + 1)

#define DFSH_LS_FRAME       (DFSH_LS + 9)       // shown help-on-help (2nd)

#define DFSD_LVMSET_WC          56              // horizontal size in columns
#define DFSD_LVMSET_WL          19              // vertical   size in lines

static char dfsdf_bt_ok[]       = " OK ";
static char dfsdf_bt_cancel[]   = "Cancel";
static char dfsdf_bt_prev[]     = "Prev";
static char dfsdf_bt_next[]     = "Next";

static char  dfsdf_footer[] = "F1=Help F2/5/8=HEX F4=OK PgDn=Next PgUp=Prev  F12=Minimize Alt+F10=Maximize";

static BOOL            lvmSetHelpInit   = TRUE; // needs init
static char           *lvmSetHelpText[] =
{
   "",
TXHELPITEM(001,"LVM Dialog fields and buttons")
   " This dialog allows verification and modification of the",
   " LVM 'Drive Letter Assignment Table' or DLAT used by LVM.",
   "",
   " To identify the involved partition, some information is",
   " displayed at the top of the dialog, this includes the label,",
   " filesystem, size of the partition, disk nr and name and the",
   " starting cylinder for the partition.",
   "",
   " LVM information is partition specific, defining:",
   "",
   "  - Volumename, when the partition is part of a volume",
   "    To delete the volume (but not the partition!) specify '-'",
   "",
   "  - Partition name, to identify this partition",
   "    To remove the name (make it anonymous) specify '-'",
   "",
   "  - Driveletter, when the volume is visible for OS/2",
   "    To hide the volume from OS/2 specify '-' or '--'",
   "    You can also specify '*' to let OS/2 assign the first available",
   "    driveletter on attaching the device (most useful on removables)",
   "",
#if defined (DEV32)
   "  - Sync driveletter assignment with live LVM engine",
   "    This will cause DFSee to attempt to communicate any changes",
   "    in driveletters, either assigning a new/different letter or",
   "    hiding the volume, to the LVM engine. This will allow a new",
   "    volume to be accessible without the need to reboot",
   "",
   "    DO NOT USE THIS OPTION WITH REMOVALE MEDIA (like USB disks).",
   "    Due to bugs in the LVM engine, it is quite likely that the",
   "    system will TRAP when doing that, or fail to assign the letter",
   "    It should work without problems with fixed disks.",
   "",
   "    Because of this the option is NOT RECOMMENDED!",
   "",
#endif
   "  - Sync LVM start/size, synchronize the start-sector and size of",
   "    the partition in the LVM info with the partition-tables.",
   "    (This choice is present only when there IS a difference)",
   "",
   "  - On-BM-menu, when the volume is on the boot menu",
   "    Check this to add the volume to the IBM BootManager menu",
   "",
   "  - Installable, volume is target for installation",
   "    Check this to make the volume a traget for OS/2 installation",
   "",
   "",
   " For the entry-fields, the CURRENT value is displayed in green,",
   " just below the field when it has the focus",
   "",
   "",
   "",
TXHELPITEM(009,"LVM Dialog window")
   "",
   " The dialog is presented when prompting is enabled (no -P- switch)",
   "",
   " It is used to show the current values of the information, and",
   " the possibly new values specified as options on the LVM command.",
   "",
   " When closing the dialog, the resulting changes will be displayed",
   " and a final confirmation to update the info is presented.",
   "",
   " The dialog ends with the following keys / actions:",
   "",
   "  - <ENTER> End dialog, if changes, ask confirmation and write back",
   "  - <PgDn>  Like ENTER, then display dialog for NEXT partition",
   "  - <PgUp>  Like ENTER, then display dialog for PREVIOUS partition",
   "  - <Esc>   End dialog, do not commit or write back any changes",
   "",
   "  - <F2>    HEX-edit DriveLetter Assignment Table, LVM-info   (DLAT)"
   "  - <F5>    HEX-edit First sector for partition, bootsector    (PBR)",
   "  - <F8>    HEX-edit Last  sector for partition, LVM-signature (BBR)",
   "",
   NULL
};


// PTEDIT dialog specific definitions (uses array[4] for the 4 entries)

#define DFSH_PT             (DFSH_FDISK + 600)

#define DFSH_PT_FRAME       (DFSH_PT + 1)       // shown help-on-help (2nd)
#define DFSH_PT_SMART       (DFSH_PT + 2)       // on smart update
#define DFSH_PT_DECIM       (DFSH_PT + 3)       // decimal pos/size
#define DFSH_PT_DECHS       (DFSH_PT + 4)       // decimal pos/size
#define DFSH_PT_BNAVI       (DFSH_PT + 5)       // navigational buttons
#define DFSH_PT_BSORT       (DFSH_PT + 6)       // table sort buttons
#define DFSH_PT_TABLE       (DFSH_PT + 7)       // on second entry-line

#define DFSM_UPDT_PDESC     (TXWM_USER  + 600)  // PTEdit partition descr update
#define DFSM_UPDT_FIELD     (TXWM_USER  + 601)  // PTEdit field update
#define DFSM_SHOW_FIELD     (TXWM_USER  + 602)  // PTEdit field display/smart
                                                // mp1 == value has changed
                                                // mp2 == originating field
#define DFSD_PT_C           (6 + (nr *3))
#define DFSD_PT_T           (7 + (nr *3))

#define DFSD_PTEDIT_WC          76              // horizontal size in columns
#define DFSD_PTEDIT_WL          24              // vertical   size in lines

#define DFSD_PENR                3              // partition entry field
#define DFSD_PACT                3              // active checkbutton
#define DFSD_PHID                3              // hidden checkbutton
#define DFSD_PTYP                4              // type entry field
#define DFSD_DESC                9              // big value field length
#define DFSD_BIGV               12              // big value field length
#define DFSD_CYLS                8              // cylinders field length
#define DFSD_HSEC                5              // head/sect field length

// window id's for first entry, others have index * 100 added

#define DFSD_WID_ENTR           600
#define DFSD_PT_WID(base,entry)  (DFSD_WID_ENTR + base + ((entry & 3) * 100))
#define DFSD_PT_BSE(wid)         (wid % 100)
#define DFSD_PT_ENT(wid)        ((wid / 100) -  6)
#define DFSD_PT_FLD(wid)        ((wid % 100) > 50)

#define DFSD_DES                61              // type description
#define DFSD_ABS                62              // absolute position
#define DFSD_SMB                63              // Size in Mb
#define DFSD_CBC                64              // calculated beg cylinder
#define DFSD_CBH                65              // calculated beg head
#define DFSD_CBS                66              // calculated beg sector
#define DFSD_CEC                67              // calculated end cylinder
#define DFSD_CEH                68              // calculated end head
#define DFSD_CES                69              // calculated end sector
#define DFSD_ENR                70              // ptable entry-number text
#define DFSD_ACT                71              // ptable active indicator
#define DFSD_HID                72              // ptable hidden indicator
#define DFSD_TYP                73              // ptable system type
#define DFSD_REL                74              // ptable relative position
#define DFSD_SEC                75              // ptable size in sectors
#define DFSD_TBC                84              // ptable beg cylinder
#define DFSD_TBH                85              // ptable beg head
#define DFSD_TBS                86              // ptable beg sector
#define DFSD_TEC                87              // ptable end cylinder
#define DFSD_TEH                88              // ptable end head
#define DFSD_TES                89              // ptable end sector


#define DFSD_WID_PTEDLG        600              // window id dialog frame
#define DFSD_WID_PSMA          601              // window id smart update
#define DFSD_WID_PDEC          602              // window id decimal pos/size
#define DFSD_WID_PCHS          603              // window id decimal chs/type
#define DFSD_WID_PTXT          604              // window id info text
#define DFSD_WID_PHDR          605              // window id table header
#define DFSD_WID_PDES          606              // window id info text
#define DFSD_WID_BSRT          607              // window id button sort-S
#define DFSD_WID_BCUS          608              // window id button sort-#

static char dfsdf_bt_ssort[]   = "Sort-StartSec";
static char dfsdf_bt_csort[]   = "Sort-#";

static  char *ptedit_headerS[] =
{
   "#   >",
   NULL
};

static  char *ptedit_headerL[] =
{
   "",
   "S  Act  Hide Type",
   NULL
};

static  char *ptedit_headerM[] =
{
   "    Start           End     ",
   "Cyl Head Sec    Cyl Head Sec",
   NULL
};

static  char *ptedit_headerR[] =
{
   " (Abs)       (MiB)",
   "Offset  Size, sect",
   NULL
};

typedef struct ptedit_entry                     // info for one entry
{                                               // FIRST ROW, display only
   TXWHANDLE           hDes;                    // type description
   TXTS                vDes;                    // value
   TXWHANDLE           hAbs;                    // absolute position
   TXTS                vAbs;
   ULONG               lAbs;
   TXWHANDLE           hSmb;                    // Size in Mb
   TXTS                vSmb;
   TXWHANDLE           hCbc;                    // calculated beg cylinder
   TXTS                vCbc;
   ULONG               uCbc;
   TXWHANDLE           hCbh;                    // calculated beg head
   TXTS                vCbh;
   ULONG               uCbh;
   TXWHANDLE           hCbs;                    // calculated beg sector
   TXTS                vCbs;
   ULONG               uCbs;
   TXWHANDLE           hCec;                    // calculated end cylinder
   TXTS                vCec;
   ULONG               uCec;
   TXWHANDLE           hCeh;                    // calculated end head
   TXTS                vCeh;
   ULONG               uCeh;
   TXWHANDLE           hCes;                    // calculated end sector
   TXTS                vCes;
   ULONG               uCes;
                                                // SECOND ROW, entryfields
   TXWHANDLE           hEnr;                    // ptable entry-number text
   TXTS                vEnr;                    // value
   TXWHANDLE           hAct;                    // ptable active indicator
   BOOL                vAct;
   TXWHANDLE           hHid;                    // ptable hidden indicator
   BOOL                vHid;
   TXWHANDLE           hTyp;                    // ptable system type
   TXTS                vTyp;                    // value
   USHORT              uTyp;
   TXWHANDLE           hRel;                    // ptable relative position
   TXTS                vRel;                    // value
   ULONG               lRel;
   TXWHANDLE           hSec;                    // ptable size in sectors
   TXTS                vSec;                    // value
   ULONG               lSec;
   TXWHANDLE           hTbc;                    // ptable end cylinder
   TXTS                vTbc;
   ULONG               uTbc;
   TXWHANDLE           hTbh;                    // ptable beg head
   TXTS                vTbh;
   ULONG               uTbh;
   TXWHANDLE           hTbs;                    // ptable beg sector
   TXTS                vTbs;
   ULONG               uTbs;
   TXWHANDLE           hTec;                    // ptable beg cylinder
   TXTS                vTec;
   ULONG               uTec;
   TXWHANDLE           hTeh;                    // ptable end head
   TXTS                vTeh;
   ULONG               uTeh;
   TXWHANDLE           hTes;                    // ptable end sector
   TXTS                vTes;
   ULONG               uTes;
} PTEDIT_ENTRY;                                 // end of struct "ptedit_entry"

typedef struct ptedit_info                      // info needed in window-proc
{
   TXWHANDLE           hSmart;                  // smart update
   TXWHANDLE           hHexSNS;                 // hexadecimal pos/size
   TXWHANDLE           hHexCHS;                 // hexadecimal CHS and type
   TXWHANDLE           hPartDesc;               // partition description
   USHORT              disknr;                  // partition description
   USHORT              eEntry;                  // edit entry number
   ULONG               eSect;                   // edit sector number
   ULONG               tableSn;                 // sector nr part-table itself
   ULONG               ebrBase;                 // offset 1st EBR sector
   PTEDIT_ENTRY        e[4];
} PTEDIT_INFO;                                  // end of struct "ptedit_info"

static BOOL            pteSmart      = TRUE;    // use smart update
static BOOL            pteHelpInit   = TRUE;    // needs init
static char           *pteHelpText[] =
{
   "",
   " This dialog allows verification and modification of the four",
   " entries in a partition-table as found in MBR and EBR sectors",
   "",
   " To identify the involved MBR/EBR sector, some information is",
   " displayed at the top of the dialog.",
   "",
   " A partition entry has the following components:",
   "",
   "  - Boot indicator, a single byte flag, values 0x00 or 0x80",
   "    Set for primary partitions that are 'active'",
   "",
   "  - Type indicator or system-type, a single byte value",
   "",
   "  - Begin-CHS, 3 values combined into 3 bytes that indicate the",
   "    start sector in Cylinder, Head and Sector format.",
   "    For cylinder values beyond 1023 special values will be used",
   "",
   "  - End-CHS, 3 values combined into 3 bytes that indicate the",
   "    last sector in Cylinder, Head and Sector format.",
   "    For cylinder values beyond 1023 special values will be used",
   "",
   "  - Offset-value, a 32-bit value that is the offset from the",
   "    start of the disk for a primary partition (start sector)",
   "    or the offset from the first EBR sector for logicals",
   "",
   "  - Size-value, a 32-bit number, partition size in sectors",
   "",
TXHELPITEM(001,"PTEDIT Dialog window")
   " The dialog is used to show the current values in the table, and",
   " any changes made to them while the dialog is up.",
   "",
   " When closing the dialog, the resulting changes will be displayed",
   " and a final confirmation to update the info is presented.",
   "",
   " The dialog ends with the following keys / actions:",
   "",
   "  - <ENTER> End dialog, if changes, ask confirmation and write back",
   "  - <PgDn>  Like ENTER, then display dialog for NEXT partition-table",
   "  - <PgUp>  Like ENTER, and then dialog for PREVIOUS partition-table",
   "  - <Esc>   End dialog, do not commit or write back any changes",
   "",
   "  - <F2>    HEX-edit this partition table sector at current entry",
   "  - <F5>    HEX-edit First sector for partition, bootsector    (PBR)",
   "  - <F8>    HEX-edit Last  sector for partition, LVM-signature (BBR)",
   "                     spare bootsector (NTFS) or anything else",
   "",
TXHELPITEM(002,"Smart updating")
   " For more help on the complete dialog, press <F1> again ...",
   "",
   " This property will cause an update to one field to propagate to",
   " other related or dependant fields. As an example, when updating",
   " any CHS related value, the 32-bit linear starting position and",
   " size fields might change as well. In the same way, when updating",
   " these linear values, the CHS fields will change accordingly.",
   "",
   " DFSee will try to keep the whole partition-table entry consistent",
   " in this way. Note however that no information from other entries",
   " or other partition tables are used so YOU are still responsible",
   " for overall consistency.",
   "",
   " The check-box is toggled using the <Space-bar> or the mouse",
   "",
TXHELPITEM(003,"Decimal position and size fields")
   " For more help on the complete dialog, press <F1> again ...",
   "",
   " This property will cause the relative and absolute position and ",
   " the size field to be displayed in the decimal numbersystem.",
   " For position and size (in sectors), hexadecimal is the default.",
   "",
   " Note: The size in megabytes is always in decimal notation.",
   "",
   " The check-box is toggled using the <Space-bar> or the mouse",
   "",
TXHELPITEM(004,"Decimal CHS fields")
   " For more help on the complete dialog, press <F1> again ...",
   "",
   " This property will cause the CHS fields to be displayed in the ",
   " decimal numbersystem as oposed to the hexadecimal system.",
   " For CHS decimal is the default."
   "",
   " The check-box is toggled using the <Space-bar> or the mouse",
   "",
TXHELPITEM(005,"Navigational buttons")
   "These buttons navigate through the chain of partition-tables",
   "on a disk (MBR/EBR) or end the editing session by writing the",
   "modified tables back to disk, or cancel updates for this table",
   "",
   " [Prev]   Will write back changes to the current tables if any,",
   " <PgUp>   and go to the previous table, that links to this one",
   "",
   " [Next]   Will write back changes to the current tables if any,",
   " <PgDn>   and go to the next EBR if present, or to the MBR",
   "",
   " [ OK ]   Will write back changes to the current tables if any,",
   " <Enter>  and then end the PTE edit session.",
   "",
   " [Cancel] Will end the PTE session without writing any changes.",
   "  <Esc>",
   "",
TXHELPITEM(006,"Table sort buttons")
   "These will sort the partition-tables, changing the order in which",
   "the represented partitions are stored in the table slots",
   "",
   "This affects partition numbering in environments like the Linux,",
   " 'UX' number in the '/dev/hdaX' style device-names, and Windows",
   " 'BI' numbering as used in the ARC paths (BOOT.INI file)",
   "",
   " [Sort-#]         Sort by the number in the first column (custom)",
   "",
   " [Sort-StartSec]  Sort by starting sector number for the partition",
   "",
   "",
   "            Note: You can use the SPACEBAR to push any",
   "                  button when no mouse is available.",
   "",
TXHELPITEM(007,"Editable fields for table entries")
   " For more help on the complete dialog, press <F1> again ...",
   "",
   " These are the one-to-one representations of each of the fields in",
   " an partition-table entry. They all can be updated to makes changes",
   " to the type, position or size of the partition in question.",
   "",
   " Directly above the fields is another row of display-only values",
   " that are directly related and are calculated from all available",
   " information and have no limitations like the 1023 cylinder limit",
   " For the starting position of logical partitions, this value is",
   " the ABSOLUTE position of the first sector as opposed to the RELATIVE",
   " offset that is used in the entry itself.",
   "",
   " Below is a more detailed description for each of the fields, this",
   " can also be called up directly using the <F1> when the field has",
   " the focus.",
   "",
TXHELPITEM(070,"Custom sorting number")
   " Initially set to the table slot-number 1..4, this field can",
   " be used to set any sort order by changing the contents of one",
   " or more fields in this column, and press the [Sort-#] button.",
   "",
TXHELPITEM(071,"Partition-table active indicator")
   " For more help on the complete dialog, press <F1> again ...",
   "",
   " This check-box indicates that the corresponding partition is an",
   " 'active' one, meaning it will be the one that is booted by the",
   " startup code in the master boot record (MBR).",
   "",
   " There should only be ONE active partition on a disk and it must be",
   " a PRIMARY defined in the first partition-table (MBR, sector 0).",
   "",
   " Depending on the capabilities of the BIOS and the MBR bootcode,",
   " the active partition might need to be BELOW cylinder 1024 to be",
   " able to boot succesfully. Only a full implementation of extended",
   " INT-13 support by all software involved allows booting from any",
   " position on the disk.",
   "",
   " The check-box is toggled using the <Space-bar> or the mouse",
   "", "", "", "", "", "", "", "", "", "", "", "",
   "",
TXHELPITEM(072,"Partition-table hidden indicator")
   " For more help on the complete dialog, press <F1> again ...",
   "",
   " This check-box indicates that the corresponding partition is a",
   " 'hidden' one, meaning it will be invisible to classic operating",
   " systems like DOS, OS/2 and Windows-9x.  It might be visible for",
   " other operating systems like Windows-2000, -XP, Linux and newer",
   " OS/2 versions like WsEB, MCP and eComStation.",
   "",
   " The hiding is based on adding a flag to the regular system-type,",
   " effectively the same as adding 0x10 to the value.",
   "",
   " Examples values are 0x06 for normal FAT and 0x16 for hidden FAT.",
   ""
   " Changing the value of this check-box will make the corresponding",
   " change to the system-type and vice-versa.",
   "",
   " Only the types 0x01 through 0x0e, except 0x05 are hidable.",
   "",
TXHELPITEM(073,"Partition-table system type")
   " For more help on the complete dialog, press <F1> again ...",
   "",
   " This is the 'system-type' for the partition, a single byte that",
   " often represents the kind of filesystem or the operating system",
   " using this partition.",
   "",
   " There are some special ones like 0x00 for UNUSED and 0x05/0x0f",
   " used for so-called 'EXTENDED-containers'.",
   "",
   " Some well-known types are:",
   "",
   " 01/04/06  16-bit    FAT            0a  OS/2 Bootmanager",
   " 0b/0c     32-bit    FAT32          35  Journalled FS (LVM)",
   " 07        OS/2 native filesystem   82  Linux SWAP partition",
   " 07        Windows-NT filesystem    83  Linux native filesystem",
   " 05        Standard EXTENDED type   0f  MS BIG-Extended type",
   "",
   " A full list of types recognized by DFSee can be obtained by using",
   " the command 'type'. A specific (hexadecimal) value can be shown",
   " using 'type [0x]nn',",
   "",
TXHELPITEM(074,"Partition-table relative position")
   " For more help on the complete dialog, press <F1> again ...",
   "",
   " This value, also called 'bootsector offset' is the number of",
   " sectors from the partition-table sector (MBR or EBR) to the",
   " bootsector for this partition.",
   "",
   " For primary partitions and the 1st extended-container in the MBR",
   " this will be equal to the absolute position of that bootsector.",
   "",
   " For logical partitions, defined in an EBR sector, it will usually",
   " be a fixed number equal to the track-size (sectors per track)",
   " because the EBR is at head-0 and the bootsector at head-1.",
   " This value is 63 (0x3f) for the most used geometry of 255/63.",
   "",
   " For the links to EBR sectors (all logicals except the first one)",
   " it is the number of sectors between the 1st-EBR and this one.",
   " This makes it a bit difficult to calculate manually, so updating",
   " is best done through the 'Begin-CHS' fields and smart-updating.",
   "",
   " When this field is modified, the absolute start sector (Abs) will",
   " be calculated and displayed just above the input-field.",
   "",
   " When 'Smart field updating' is enabled, all CHS fields will be",
   " recalculated and displayed again using the new values.",
   "",
TXHELPITEM(075,"Partition-table size in sectors")
   " For more help on the complete dialog, press <F1> again ...",
   "",
   " This is the size as a number of sectors used by by this partition.",
   "",
   " As a convenience, the size may also be entered as a floating-point",
   " number (with a DOT in the number) with the size in MEGABYTES.",
   " This size is converted to a number of sectors using the geometry",
   " currently in use. The size is also displayed in this format, just",
   " above the sectors input field.",
   "",
   " Specifying the size in megabytes makes it a bit difficult to end",
   " the partition exactly on a cylinder boundary, so fine-tuning is",
   " best done through the 'End-CHS' fields and smart-updating.",
   "",
   " When this field is modified, the size in megabytes (MiB) will",
   " be calculated and and displayed just above the input-field.",
   "",
   " When 'Smart field updating' is enabled, all CHS fields will be",
   " recalculated and displayed again using the new values.",
   "",
TXHELPITEM(084,"Partition-table begin cylinder")
   " For more help on the complete dialog, press <F1> again ...",
   "",
   " This is the starting cylinder for the partition",
   " The displayed value in the input field is limited to a maximum",
   " of 1023, just like the corresponding partition-table data.",
   " However, it IS possible to enter larger values here to specify",
   " a starting cylinder beyond 1023 cylinders, this larger value",
   " will be displayed just above the actual input-field and it will",
   " be used in any calculations used in 'Smart field updating'.",
   "",
   " When 'Smart field updating' is enabled, the start position and",
   " size for the partition will be recalculated and redisplayed.",
   " If the corresponding head and sector fields are zero, they will",
   " be set to the most likely values based on the current geometry.",
   "",
TXHELPITEM(085,"Partition-table begin head")
   " For more help on the complete dialog, press <F1> again ...",
   "",
   " This is the starting head for the partition",
   " The displayed value in the input field is limited to a maximum of",
   " 255 or to the geometry value if 'Smart field updating' is enabled.",
   "",
   " If the value is zero, it will be initialized to the most likely",
   " value based on the current geometry."
   "",
   " When 'Smart field updating' is enabled, the start position and",
   " size for the partition will be recalculated and redisplayed.",
   "",
TXHELPITEM(086,"Partition-table begin sector")
   " For more help on the complete dialog, press <F1> again ...",
   "",
   " This is the starting sector for the partition",
   " The displayed value in the input field is limited to a maximum of",
   " 63 or to the geometry value if 'Smart field updating' is enabled.",
   "",
   " If the value is zero, it will be initialized to the most likely",
   " value based on the current geometry."
   "",
   " When 'Smart field updating' is enabled, the start position and",
   " size for the partition will be recalculated and redisplayed.",
   "",
TXHELPITEM(087,"Partition-table end cylinder")
   " For more help on the complete dialog, press <F1> again ...",
   "",
   " This is the ending (last) cylinder for the partition",
   " The displayed value in the input field is limited to a maximum",
   " of 1023, just like the corresponding partition-table data.",
   " However, it IS possible to enter larger values here to specify",
   " a starting cylinder beyond 1023 cylinders, this larger value",
   " will be displayed just above the actual input-field and it will",
   " be used in any calculations used in 'Smart field updating'.",
   "",
   " When 'Smart field updating' is enabled, the start position and",
   " size for the partition will be recalculated and redisplayed.",
   " If the corresponding head and sector fields are zero, they will",
   " be set to the most likely values based on the current geometry.",
   "",
TXHELPITEM(088,"Partition-table end head")
   " For more help on the complete dialog, press <F1> again ...",
   "",
   " This is the ending (last) head for the partition",
   " The displayed value in the input field is limited to a maximum of",
   " 255 or to the geometry value if 'Smart field updating' is enabled.",
   "",
   " If the value is zero, it will be initialized to the most likely",
   " value based on the current geometry."
   "",
   " When 'Smart field updating' is enabled, the start position and",
   " size for the partition will be recalculated and redisplayed.",
   "",
TXHELPITEM(089,"Partition-table end sector")
   " For more help on the complete dialog, press <F1> again ...",
   "",
   " This is the ending (last) sector for the partition",
   " The displayed value in the input field is limited to a maximum of",
   " 63 or to the geometry value if 'Smart field updating' is enabled.",
   "",
   " If the value is zero, it will be initialized to the most likely",
   " value based on the current geometry."
   "",
   " When 'Smart field updating' is enabled, the start position and",
   " size for the partition will be recalculated and redisplayed.",
   "",
   NULL
};



// Dialog window procedure, for the PTEDIT dialog fields
static ULONG dfsFdskPteditWinProc               // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   ULONG               mp1,                     // IN    msg param 1
   ULONG               mp2                      // IN    msg param 2
);

// Update current CHS values from linear Ptable fields
static void dfsFdskPteLinear2Chs
(
   PTEDIT_ENTRY       *pi,                      // IN    PTE entry info
   ULONG               field                    // IN    originating field
);

// Update linear Ptable fields from current CHS values
static void dfsFdskPteChs2Linear
(
   PTEDIT_ENTRY       *pi,                      // IN    PTE entry info
   ULONG               field                    // IN    originating field
);

// Dialog window procedure, for the FDISK NEXT/PREV type Dialogs
static ULONG dfsFdskNpDlgWinProc                // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   ULONG               mp1,                     // IN    msg param 1
   ULONG               mp2                      // IN    msg param 2
);

// Compare PTEDIT_ENTRY elements for qsort, sort on start-sector except 0
int dfsComparePteEntry
(
   const void         *one,
   const void         *two
);


// Compare PTEDIT_ENTRY elements for qsort, sort on custom-value
int dfsComparePteCustom
(
   const void         *one,
   const void         *two
);

/*****************************************************************************/
// Display LVM dialog with default values and allow changes
/*****************************************************************************/
ULONG dfsFdskLvmSetDialog                       // RET   dialog result
(
   FDSK_CB_INFO       *cbp,                     // INOUT LVMSET parameters
   S_LVINF            *lvm,                     // LVM sector
   S_LVPART           *lp                       // LVM part info
)
{
   ULONG               rc = NO_ERROR;           // function return

   ENTER();

   if (txwIsWindow( TXHWND_DESKTOP))
   {
      TXRECT           deskpos;                 // desktop size/position
      static TXRECT    position = {0,0,0,0};    // reference size/position
      TXWHANDLE        dframe;                  // dialog frame
      TXWHANDLE        dtext;                   // informational text
      TXWHANDLE        drlet;                   //
      TXTM             drlet_v;                 // entryfield driveletter
      TXTM             drlet_o;                 // original driveletter
      TXWHANDLE        dvolu;                   //
      TXTM             dvolu_v;                 // entryfield volume name
      TXTM             dvolu_o;                 // original volume name
      TXWHANDLE        dpart;                   //
      TXTM             dpart_v;                 // entryfield partition name
      TXTM             dpart_o;                 // original partition name
      TXWHANDLE        dmenu;                   //
      BOOL             dmenu_v;                 // checkbox on-bm-menu
      BOOL             dmenu_o;                 // original on-bm-menu
      TXTM             dmenu_t;                 // checkbutton text
      TXWHANDLE        dinst;                   //
      BOOL             dinst_v;                 // checkbox installable
      BOOL             dinst_o;                 // original installable
      TXTM             dinst_t;                 // checkbutton text
      TXWHANDLE        dsize;                   //
      BOOL             dsize_v;                 // checkbox size sync
      #if defined (DEV32)
         TXWHANDLE     dsync;
      #endif
      TXWHANDLE        dbtok;                   // OK button
      TXWHANDLE        dbtcan;                  // CANCEL button
      TXWHANDLE        dbtprev;                 // PREV   button
      TXWHANDLE        dbtnext;                 // NEXT   button
      TXWINDOW         window;                  // setup window data
      ULONG            style;                   // window style
      USHORT           phsize;                  // parent window width
      USHORT           pvsize;                  // parent window height
      short            eflength = LVM_NAME_L-1; // length of entryfields (19!)
      TXLN             title;
      TXTS             mibsize;
      char           **mText;
      int              lines;                   // nr of lines
      int              ll = DFSD_LVMSET_WC - 4; // real max line length
      DFSPARTINFO     *p = (DFSPARTINFO *) cbp->more;
      ULONG            size = p->sectors;

      if (lvmSetHelpInit)                       // init helptext needed
      {
         txwRegisterHelpText( DFSH_LS, "dfslvmdialog", "LVMSET Dialog help", lvmSetHelpText);
         lvmSetHelpInit = FALSE;
      }

      strcpy(  mibsize, "");
      sprintf( title, "Drive: %2.2s = %8.8s %11.11s  Size : %s\n"
                      "Disk : %2hu = %-*.*s   Cyl : %5lu",
                       p->drive, p->fsform, p->plabel,
                       dfstrSizeXiB( mibsize, "", lp->sectors, p->bpsector, ""),
                       p->disknr, LVM_NAME_L, LVM_NAME_L, lvm->DiskName,
                       dfstPsn2Cyl( DFSTORE, lp->partPsn));

      mText = txString2Text( title, &ll, &lines);

      sprintf( title, " SET LVM information for partition: %2.2hu ", p->id);

      sprintf( drlet_o, "%c%c", (lp->letter != 0) ? lp->letter : '-',
                                (lp->letter != 0) ? ':'        : '-');
      sprintf( dvolu_o, "%-*.*s", LVM_NAME_L, LVM_NAME_L, lp->VoluName);
      TxStrip( dvolu_o, dvolu_o, 0, ' ');       // remove trailing spaces
      sprintf( dpart_o, "%-*.*s", LVM_NAME_L, LVM_NAME_L, lp->PartName);
      TxStrip( dpart_o, dpart_o, 0, ' ');       // remove trailing spaces

      strcpy( drlet_v, (strlen(cbp->bname) == 0) ? drlet_o :
                       (cbp->bname[0] != '-') ? cbp->bname : "- hidden -");

      strcpy( dvolu_v, (strlen(cbp->vname) == 0) ? dvolu_o :
                       (cbp->vname[0] != '-') ? cbp->vname : "- none -");
      if ((strlen(lp->VoluName)) && (cbp->vname[0] == '-'))
      {
         strcat( dvolu_v, " (delete)");
      }

      strcpy( dpart_v, (strlen(cbp->pname) == 0) ? dpart_o :
                       (cbp->pname[0] != '-') ? cbp->pname : "- none -");
      if ((strlen(lp->PartName)) && (cbp->pname[0] == '-'))
      {
         strcat( dpart_v, " (delete)");
      }

      dmenu_o = (lp->OnBmMenu != 0);
      dmenu_v = (cbp->ntype != 0xff) ? (cbp->ntype != 0)   : dmenu_o;
      sprintf( dmenu_t, "On BM menu   (%s)", (dmenu_o) ? "on"   : "off");

      dinst_o = (lp->Installable != 0);
      dinst_v = (cbp->stype != 0xff) ? (cbp->stype != 0)   : dinst_o;
      sprintf( dinst_t, "Installable  (%s)", (dinst_o) ? "on"   : "off");

      dsize_v = cbp->cbOption1;

      txwQueryWindowPos( TXHWND_DESKTOP, FALSE, &deskpos);
      phsize = deskpos.right  - deskpos.left;
      pvsize = deskpos.bottom - deskpos.top;

      if (position.right == 0)                  // not initialized yet ?
      {                                         // start with desktop position
         if (deskpos.left + DFSD_LVMSET_WC < phsize)
         {
            position.left   = (phsize - DFSD_LVMSET_WC) / 2;
            position.right  = DFSD_LVMSET_WC;
            if (position.left > 4)
            {
               position.left -= 1;              // a little left from center
            }
         }
         if (DFSD_LVMSET_WL + 4 < pvsize)
         {
            position.top += 2;                  // move dialog a bit down
         }
         if (position.top   + DFSD_LVMSET_WL < pvsize)
         {
            position.bottom = DFSD_LVMSET_WL;
         }
      }
      else
      {
         position.right  = DFSD_LVMSET_WC;      // fixed size of start Dlg
         position.bottom = DFSD_LVMSET_WL;      // (but keeps position)
      }
      TRECTA( "pos/size", (&position));

      style = TXWS_DFRAME | TXWS_CAST_SHADOW | TXCS_CLOSE_BUTTON;
      txwSetupWindowData(
         position.top,                          // upper left corner
         position.left,
         position.bottom,                       // vertical size
         position.right,                        // horizontal size
         style, DFSH_LS_FRAME,                  // style & help
         ' ', ' ',                              // window & border char
         TXWSCHEME_COLORS,                      // std scheme colors
         title, dfsdf_footer,
         &window);
      window.dlgFocusID = DFSD_WID_EVOL;        // volume gets focus
      window.st.buf     = NULL;                 // no artwork attached
      dframe = txwCreateWindow( TXHWND_DESKTOP, TXW_CANVAS, 0, 0, &window,
                                NULL);          // will be set by CreateDlg
      txwSetWindowUShort( dframe, TXQWS_ID, DFSD_WID_LVMDLG);
      txwSetWindowPtr(    dframe, TXQWP_DATA, &position); // for exit position

      //- add child windows to dframe ...

      txwSetupWindowData(
         position.bottom - 18,                  // UL corner, line   (relative)
         1, 2, ll,                              // UL column + vert/hor size
         TXWS_OUTPUT, 0,                        // window frame style + help
         ' ', ' ',                              // window/border chars
         TXWSCHEME_COLORS,                      // std scheme colors
         "", "",                                // title/footer text
         &window);
      window.tv.topline = 0;
      window.tv.leftcol = 0;
      window.tv.maxtop  = TXW_INVALID;
      window.tv.maxcol  = TXW_INVALID;
      window.tv.buf     = mText;
      dtext = txwCreateWindow( dframe, TXW_TEXTVIEW, dframe, 0, &window,
                               txwDefWindowProc);
      txwSetWindowUShort( dtext, TXQWS_ID, DFSD_WID_TEXT);


      style = TXWS_ENTRYBT | TXWS_HCHILD2SIZE;
      txwSetupWindowData(
         position.bottom - 15,                  // UL corner, line   (relative)
         1, 2, eflength +2,                     // UL column + vert/hor size
         style, DFSH_LS_EVOL,                   // style + help
         ' ', ' ',
         TXWSCHEME_COLORS,                      // std scheme colors
         "Volume name", "",                     // title and footer text
         &window);
      window.ef.leftcol = 0;
      window.ef.maxcol  = TXW_INVALID;
      window.ef.curpos  = 0;
      window.ef.rsize   = eflength;
      window.ef.buf     = dvolu_v;
      window.ef.history = NULL;
      dvolu = txwCreateWindow( dframe, TXW_ENTRYFIELD, dframe, 0, &window,
                               txwDefWindowProc);
      txwSetWindowUShort( dvolu, TXQWS_ID, DFSD_WID_EVOL);


      style = TXWS_ENTRYBT | TXWS_HCHILD2SIZE;
      txwSetupWindowData(
         position.bottom - 15,                  // UL corner, line   (relative)
         30, 2, eflength +2,                    // UL column + vert/hor size
         style, DFSH_LS_EPAR,                   // style + help
         ' ', ' ',
         TXWSCHEME_COLORS,                      // std scheme colors
         "Partition name", "",
         &window);
      window.ef.leftcol = 0;
      window.ef.maxcol  = TXW_INVALID;
      window.ef.curpos  = 0;
      window.ef.rsize   = eflength;
      window.ef.buf     = dpart_v;
      window.ef.history = NULL;
      dpart = txwCreateWindow( dframe, TXW_ENTRYFIELD, dframe, 0, &window,
                               txwDefWindowProc);
      txwSetWindowUShort( dpart, TXQWS_ID, DFSD_WID_EPAR);


      style = TXWS_ENTRYBT | TXWS_HCHILD2SIZE | TXWS_HCHILD2MOVE;
      txwSetupWindowData(
         position.bottom - 11,                  // UL corner, line   (relative)
         1, 2, 13,                              // UL column + vert/hor size
         style, DFSH_LS_ELET,                   // style + help
         ' ', ' ',
         TXWSCHEME_COLORS,                      // std scheme colors
         "Driveletter", "",
         &window);
      window.ef.leftcol = 0;
      window.ef.maxcol  = TXW_INVALID;
      window.ef.curpos  = 0;
      window.ef.rsize   = 3;
      window.ef.buf     = drlet_v;
      window.ef.history = NULL;
      drlet = txwCreateWindow( dframe, TXW_ENTRYFIELD, dframe, 0, &window,
                               txwDefWindowProc);
      txwSetWindowUShort( drlet, TXQWS_ID, DFSD_WID_ELET);

      if ((lp->sectors != size) || (lp->partPsn != p->basePsn))
      {                                         // only when start/size differs
         style = TXWS_AUTOCHK | TXWS_HCHILD2SIZE | TXWS_HCHILD2MOVE;
         txwSetupWindowData(
            position.bottom - 12,               // UL corner, line   (relative)
            30, 1, 25,                          // UL column + vert/hor size
            style, DFSH_LS_CSIZ,                // style + help
            ' ', ' ', TXWSCHEME_COLORS, "", "",
            &window);
         window.bu.text    = "Sync LVM size/start";
         window.bu.checked = &dsize_v;
         dsize = txwCreateWindow( dframe, TXW_BUTTON, dframe, 0, &window,
                                  txwDefWindowProc);
         txwSetWindowUShort( dsize, TXQWS_ID, DFSD_WID_CSIZ);
         txwSetWindowUShort( dsize, TXQWS_GROUP, 1);
      }

      if (dfsa->expertui)
      {
         style = TXWS_AUTOCHK | TXWS_HCHILD2SIZE | TXWS_HCHILD2MOVE;
         txwSetupWindowData(
            position.bottom - 11,               // UL corner, line   (relative)
            30, 1, 25,                          // UL column + vert/hor size
            style, DFSH_LS_CINS,                // style + help
            ' ', ' ', TXWSCHEME_COLORS, "", "",
            &window);
         window.bu.text    = dinst_t;
         window.bu.checked = &dinst_v;
         dinst = txwCreateWindow( dframe, TXW_BUTTON, dframe, 0, &window,
                                  txwDefWindowProc);
         txwSetWindowUShort( dinst, TXQWS_ID, DFSD_WID_CINS);
         txwSetWindowUShort( dinst, TXQWS_GROUP, 1);
      }

      style = TXWS_AUTOCHK | TXWS_HCHILD2SIZE | TXWS_HCHILD2MOVE;
      txwSetupWindowData(
         position.bottom - 10,                  // UL corner, line   (relative)
         30, 1, 25,                             // UL column + vert/hor size
         style, DFSH_LS_CMEN,                   // style + help
         ' ', ' ', TXWSCHEME_COLORS, "", "",
         &window);
      window.bu.text    = dmenu_t;
      window.bu.checked = &dmenu_v;
      dmenu = txwCreateWindow( dframe, TXW_BUTTON, dframe, 0, &window,
                               txwDefWindowProc);
      txwSetWindowUShort( dmenu, TXQWS_ID, DFSD_WID_CMEN);
      txwSetWindowUShort( dmenu, TXQWS_GROUP, 1);

      #if defined (DEV32)                       // OS2 feature only, LVM sync
         if ((dfsa->expertui) &&                // but only when in Expert mode
             (dfsLvmEnginePresent()))           // and when LVM Engine is there
         {
            style = TXWS_AUTOCHK | TXWS_HCHILD2SIZE | TXWS_HCHILD2MOVE;
            txwSetupWindowData(
               position.bottom - 8,             // UL corner, line   (relative)
               1,  1, 52,                       // UL column + vert/hor size
               style, DFSH_LS_CSNC,             // style + help
               ' ', ' ', TXWSCHEME_COLORS, "", "",
               &window);
            window.bu.text    = "Sync driveletter assignment with live LVM engine";
            window.bu.text    = "Sync driveletter to LVM engine, NOT RECOMMENDED!";
            window.bu.checked = &cbp->lvmSync;
            dsync = txwCreateWindow( dframe, TXW_BUTTON, dframe, 0, &window,
                                     txwDefWindowProc);
            txwSetWindowUShort( dsync, TXQWS_ID, DFSD_WID_CSNC);
         }
      #endif

      if (p->id > 1)                            // is there a previous ?
      {
         style = TXWS_PBUTTON | TXWS_VCHILD_MOVE;
         txwSetupWindowData(
            position.bottom - 6,                // UL corner, line   (relative)
            3,                                  // column
            3, strlen( dfsdf_bt_prev) + 4,      // vertical/horizontal size
            style, DFSH_LS,                     // style and help
            ' ', ' ', TXWSCHEME_COLORS, "",  "",
            &window);
         window.bu.text = dfsdf_bt_prev;
         dbtprev = txwCreateWindow(  dframe, TXW_BUTTON, dframe, 0, &window,
                                     txwDefWindowProc);
         txwSetWindowUShort( dbtprev, TXQWS_ID, DFSD_PREV); // rc == ID
      }

      if (p->id < dfsPartitions())              // is there a next ?
      {
         style = TXWS_PBUTTON | TXWS_VCHILD_MOVE;
         txwSetupWindowData(
            position.bottom - 6,                // UL corner, line   (relative)
            13,                                 // column
            3, strlen( dfsdf_bt_next) + 4,      // vertical/horizontal size
            style, DFSH_LS,                     // style and help
            ' ', ' ', TXWSCHEME_COLORS, "",  "",
            &window);
         window.bu.text = dfsdf_bt_next;
         dbtnext = txwCreateWindow(  dframe, TXW_BUTTON, dframe, 0, &window,
                                     txwDefWindowProc);
         txwSetWindowUShort( dbtnext, TXQWS_ID, DFSD_NEXT); // rc == ID
      }

      style = TXWS_PBUTTON | TXWS_VCHILD_MOVE | TXWS_HCHILD2MOVE;
      txwSetupWindowData(
         position.bottom - 6,                   // UL corner, line   (relative)
         position.right - 25,                   // column
         3, strlen( dfsdf_bt_ok) + 4,           // vertical/horizontal size
         style, DFSH_LS,                        // style and help
         ' ', ' ', TXWSCHEME_COLORS, "",  "",
         &window);
      window.bu.text = dfsdf_bt_ok;
      dbtok = txwCreateWindow(   dframe, TXW_BUTTON, dframe, 0, &window,
                                 txwDefWindowProc);
      txwSetWindowUShort( dbtok, TXQWS_ID, TXMBID_OK); // rc == ID


      style = TXWS_PBUTTON | TXWS_VCHILD_MOVE | TXWS_HCHILD2MOVE;
      txwSetupWindowData(
         position.bottom - 6,                   // UL corner, line   (relative)
         position.right - 15,                   // column
         3, strlen( dfsdf_bt_cancel) + 4,       // vertical/horizontal size
         style, DFSH_LS,                        // style and help
         ' ', ' ', TXWSCHEME_COLORS, "",  "",
         &window);
      window.bu.text = dfsdf_bt_cancel;
      dbtcan = txwCreateWindow(  dframe, TXW_BUTTON, dframe, 0, &window,
                                 txwDefWindowProc);
      txwSetWindowUShort( dbtcan, TXQWS_ID, TXMBID_CANCEL); // rc == ID

      rc = txwDlgBox( TXHWND_DESKTOP, TXHWND_DESKTOP,
                      dfsFdskNpDlgWinProc, dframe, NULL);

      if (rc != TXDID_CANCEL)                   // copy changed values back
      {
         if (toupper(drlet_v[0]) != drlet_o[0]) // driveletter change
         {
            cbp->bname[0] = (isalpha(drlet_v[0]) || (drlet_v[0] == '*'))
                                           ? toupper(drlet_v[0]) : '-';
            cbp->cbOptNum2++;
         }
         if (strcmp( dvolu_o, dvolu_v) != 0)    // volumename change
         {
            TxStrip( dvolu_v, dvolu_v, 0, ' '); // strip trailing spaces
            strcpy( cbp->vname, (isgraph(dvolu_v[0])) ? dvolu_v : "-");
            cbp->cbOptNum2++;
         }
         if (strcmp( dpart_o, dpart_v) != 0)    // partition name change
         {
            TxStrip( dpart_v, dpart_v, 0, ' '); // strip trailing spaces
            strcpy( cbp->pname, (isgraph(dpart_v[0])) ? dpart_v : "-");
            cbp->cbOptNum2++;
         }
         if (dmenu_o  !=  dmenu_v)              // on-menu change
         {
            cbp->ntype = (dmenu_v) ? 1 : 0;
            cbp->cbOptNum2++;
         }
         if (dinst_o  !=  dinst_v)              // installable change
         {
            cbp->stype = (dinst_v) ? 1 : 0;
            cbp->cbOptNum2++;
         }
         if ((cbp->cbOption1 = dsize_v) != FALSE) // size synchronize ?
         {
            cbp->cbOptNum2++;                   // count as change
         }
         switch (rc)                            // dialog result
         {
            case DFSD_NEXT:                     // request next
            case DFSD_PREV:                     // request prev
            case DFSD_REQ5:
            case DFSD_REQ6:
            case DFSD_REQ7:
            case DFSD_REQ8:
            case DFSD_EDIT:                     // request editor
               cbp->cbOptNum1 = rc;
               break;

            default:
               cbp->cbOptNum1 = DFSD_NONE;
               break;
         }
         rc = NO_ERROR;
      }
      else                                      // cancel / escape
      {
         cbp->cbOptNum1 = DFSD_NONE;            // stop partition iteration
      }
      txFreeText( mText);                       // free the text memory
   }
   RETURN (rc);
}                                               // end 'dfsFdskLvmSetDialog'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display PTEDIT dialog with default values and allow changes
/*****************************************************************************/
ULONG dfsFdskPteditDialog                       // RET   dialog result
(
   FDSK_CB_INFO       *cbp,                     // IN    operation info
   S_BOOTR            *br                       // INOUT sectors with PTABLE
)
{
   ULONG               rc = NO_ERROR;           // function return
   PTEDIT_INFO        *ptinfo   = TxAlloc( 1, sizeof(PTEDIT_INFO));
   static TXRECT       position = {0,0,0,0};    // size/position, initialized
                                                // once from desktop position
   ENTER();

   if ((ptinfo) && (txwIsWindow( TXHWND_DESKTOP)))
   {
      TXRECT           deskpos;                 // desktop size/position
      TXWHANDLE        hframe;                  // dialog frame
      TXWHANDLE        htext;                   // informational text
      TXWHANDLE        hheadS;                  // header-text Sort + Active
      TXWHANDLE        hheadL;                  // header-text left
      TXWHANDLE        hheadM;                  // header-text middle
      TXWHANDLE        hheadR;                  // header-text right
      TXWHANDLE        hbtok;                   // OK button
      TXWHANDLE        hbtcan;                  // CANCEL button
      TXWHANDLE        hbtprev;                 // PREV   button
      TXWHANDLE        hbtnext;                 // NEXT   button
      TXWHANDLE        hbtsort;                 // SORT-S button
      TXWHANDLE        hbtcust;                 // SORT-# button
      TXWINDOW         window;                  // setup window data
      ULONG            style;                   // window style
      USHORT           phsize;                  // parent window width
      USHORT           pvsize;                  // parent window height
      TXLN             title;
      TXTM             pdesc;
      char           **mText;
      int              lines;                   // nr of lines
      int              ll = 50;                 // real max line length
      int              nr;                      // entry number 0..3

      TRACES(( "disk: %hu sn:%8.8lx  cbp->string: '%s'\n",
                cbp->disknr, cbp->thisSn, cbp->string));

      ptinfo->disknr   = cbp->disknr;
      ptinfo->tableSn  = cbp->thisSn;
      ptinfo->ebrBase  = dfsEbrBase();
      ptinfo->eEntry   = cbp->stype;
      ptinfo->eSect    = cbp->cbOptNum2;
      if (cbp->cbOption1 == FALSE)              // reset if requested
      {
         pteSmart = FALSE;                      // no smart for now
         cbp->cbOption1 = TRUE;                 // but keep pteSmart value
      }                                         // in next PgDn/PgUp calls

      if (pteHelpInit)                          // init helptext needed
      {
         txwRegisterHelpText( DFSH_PT, "dfspteditor", "PTEDIT Dialog help", pteHelpText);
         pteHelpInit = FALSE;
      }
      mText = txString2Text( cbp->string, &ll, &lines);
      strcpy(  pdesc, "None");
      sprintf( title, " Partition-table edit disk %hu, sector:%8.8lx on Cyl:%6lu ",
                   cbp->disknr, cbp->thisSn, dfstPsn2Cyl( DFSTORE,  cbp->thisSn));

      for (nr = 0; nr < 4; nr++)                // handle all entries
      {
         PTEDIT_ENTRY *pi = &(ptinfo->e[nr]);
         DFSPARTENTRY *pe = &(br->PartitionTable[nr]);

         if (pe->PartitionType      || pe->Status            ||
             pe->FirstSector.Head   || pe->LastSector.Head   ||
             pe->FirstSector.SecCyl || pe->LastSector.SecCyl ||
             pe->BootSectorOffset   || pe->NumberOfSectors   )
         {
            pi->uTyp = (USHORT) pe->PartitionType;
            pi->lRel = pe->BootSectorOffset;
            pi->lSec = pe->NumberOfSectors;
            pi->lAbs = pi->lRel;
            if (ptinfo->tableSn != 0)           // not the MBR
            {
               if (dfsIsExtendedType(pi->uTyp))
               {
                  pi->lAbs += ptinfo->ebrBase;  // add offset to 1st ebr
               }
               else
               {
                  pi->lAbs += ptinfo->tableSn;  // add offset to ebr psn
               }
            }
            dfstPsn2Chs( DFSTORE, pi->lAbs,               &pi->uCbc, &pi->uCbh, &pi->uCbs);
            dfstPsn2Chs( DFSTORE, pi->lAbs + pi->lSec -1, &pi->uCec, &pi->uCeh, &pi->uCes);

            pi->vAct = ((pe->Status & 0x80) != 0);
            pi->uTbc = DFSC2CYLIND(   pe->FirstSector.SecCyl);
            pi->uTbh = (USHORT)   (   pe->FirstSector.Head);
            pi->uTbs = DFSC2SECTOR(   pe->FirstSector.SecCyl);
            pi->uTec = DFSC2CYLIND(   pe->LastSector.SecCyl);
            pi->uTeh = (USHORT)   (   pe->LastSector.Head);
            pi->uTes = DFSC2SECTOR(   pe->LastSector.SecCyl);

            pi->vHid = dfsPartTypeHidable( pi->uTyp) && (pi->uTyp & DFS_P_PHIDDEN);
            sprintf( pi->vTyp, "%2.2hX", pi->uTyp);
            dfsPartTypeDescription(   pi->uTyp, pi->vDes);

            txwDlgUl2Field( pi->hRel, pi->lRel, pi->vRel, 10, dfsa->hexSNS);
            txwDlgUl2Field( pi->hSec, pi->lSec, pi->vSec, 10, dfsa->hexSNS);
            txwDlgUl2Field( pi->hTbc, pi->uTbc, pi->vTbc,  6, dfsa->hexCHS);
            txwDlgUl2Field( pi->hTbh, pi->uTbh, pi->vTbh,  3, dfsa->hexCHS);
            txwDlgUl2Field( pi->hTbs, pi->uTbs, pi->vTbs,  3, dfsa->hexCHS);
            txwDlgUl2Field( pi->hTec, pi->uTec, pi->vTec,  6, dfsa->hexCHS);
            txwDlgUl2Field( pi->hTeh, pi->uTeh, pi->vTeh,  3, dfsa->hexCHS);
            txwDlgUl2Field( pi->hTes, pi->uTes, pi->vTes,  3, dfsa->hexCHS);

            txwDlgUl2Field( pi->hAbs, pi->lAbs, pi->vAbs, 10, dfsa->hexSNS);
            sprintf(        pi->vSmb, "%10.1lf",
                              TXSMIB( pe->NumberOfSectors, dfsGetSectorSize()));
            txwDlgUl2Field( pi->hCbc, pi->uCbc, pi->vCbc,  6, dfsa->hexCHS);
            txwDlgUl2Field( pi->hCbh, pi->uCbh, pi->vCbh,  3, dfsa->hexCHS);
            txwDlgUl2Field( pi->hCbs, pi->uCbs, pi->vCbs,  3, dfsa->hexCHS);
            txwDlgUl2Field( pi->hCec, pi->uCec, pi->vCec,  6, dfsa->hexCHS);
            txwDlgUl2Field( pi->hCeh, pi->uCeh, pi->vCeh,  3, dfsa->hexCHS);
            txwDlgUl2Field( pi->hCes, pi->uCes, pi->vCes,  3, dfsa->hexCHS);
         }
         else
         {
            strcpy(  pi->vDes, "Not used");
         }
      }

      txwQueryWindowPos( TXHWND_DESKTOP, FALSE, &deskpos);
      phsize = deskpos.right  - deskpos.left;
      pvsize = deskpos.bottom - deskpos.top;

      if (position.right == 0)                  // not initialized yet ?
      {                                         // start with desktop position
         position = deskpos;
         if (position.left  + DFSD_PTEDIT_WC < phsize)
         {
            position.left   = (phsize - DFSD_PTEDIT_WC) /2;
            position.right  = DFSD_PTEDIT_WC;
            if (position.left > 2)
            {
               position.left -= 1;
            }
         }
         if (DFSD_PTEDIT_WL + 4 < pvsize)
         {
            position.top += 2;                  // move dialog a bit down
         }
         if (position.top   + DFSD_PTEDIT_WL < pvsize)
         {
            position.bottom = DFSD_PTEDIT_WL;
         }
      }
      else
      {
         position.right  = DFSD_PTEDIT_WC;      // fixed size of start Dlg
         position.bottom = DFSD_PTEDIT_WL;      // (but keeps position)
      }

      TRECTA( "pos/size", (&position));

      txwSetupWindowData(
         position.top,                          // upper left corner
         position.left,
         position.bottom,                       // vertical size
         position.right,                        // horizontal size
         TXWS_DFRAME      |
         TXWS_CAST_SHADOW |
         TXCS_CLOSE_BUTTON,                     // style
         DFSH_PT_FRAME,                         // help
         ' ', ' ',                              // window & border chars
         TXWSCHEME_COLORS,                      // std scheme colors
         title, dfsdf_footer,
         &window);
      window.dlgFocusID = DFSD_PT_WID( DFSD_ACT, cbp->stype); // first FOCUS
      window.st.buf     = NULL;                 // no artwork attached
      hframe = txwCreateWindow( TXHWND_DESKTOP, TXW_CANVAS, 0, 0, &window,
                                NULL);          // will be set by CreateDlg
      txwSetWindowUShort( hframe, TXQWS_ID,   DFSD_WID_PTEDLG);
      txwSetWindowPtr(    hframe, TXQWP_DATA, &position); //- for exit position
      TRACES(( "Frame: %8.8lx with attached PTINFO: %8.8lx\n", hframe, ptinfo));

      //- add child windows to frame ...

      //- Smart updating checkbutton
      txwSetupWindowData(
         1, 1, 1, 20,                           // UL corner + vert/hor size
         TXWS_AUTOCHK, DFSH_PT_SMART,           // style & help
         ' ', ' ',                              // window/border chars
         TXWSCHEME_COLORS,                      // std scheme colors
         "", "",
         &window);
      window.bu.text    = "Smart fields";
      window.bu.checked = &(pteSmart);
      ptinfo->hSmart = txwCreateWindow(   hframe, TXW_BUTTON, hframe, 0, &window,
                                          dfsFdskPteditWinProc);
      txwSetWindowUShort( ptinfo->hSmart, TXQWS_ID, DFSD_WID_PSMA);
      txwSetWindowUShort( ptinfo->hSmart, TXQWS_GROUP, 1);


      //- Decimal postition and sizes checkbutton
      txwSetupWindowData(
         2, 1, 1, 20,                           // UL corner + vert/hor size
         TXWS_CHECKB, DFSH_PT_DECIM,            // style & help
         ' ', ' ',                              // window/border chars
         TXWSCHEME_COLORS,                      // std scheme colors
         "", "",
         &window);
      window.bu.text    = "Hex Pos + Size";
      window.bu.checked = &(dfsa->hexSNS);
      ptinfo->hHexSNS = txwCreateWindow(   hframe, TXW_BUTTON, hframe, 0, &window,
                                          dfsFdskPteditWinProc);
      txwSetWindowUShort( ptinfo->hHexSNS, TXQWS_ID, DFSD_WID_PDEC);
      txwSetWindowUShort( ptinfo->hHexSNS, TXQWS_GROUP, 1);

      //- Decimal postition and sizes checkbutton
      txwSetupWindowData(
         3, 1, 1, 20,                           // UL corner + vert/hor size
         TXWS_CHECKB, DFSH_PT_DECHS,            // style & help
         ' ', ' ',                              // window/border chars
         TXWSCHEME_COLORS,                      // std scheme colors
         "", "",
         &window);
      window.bu.text    = "Hex CHS values";
      window.bu.checked = &(dfsa->hexCHS);
      ptinfo->hHexCHS = txwCreateWindow(   hframe, TXW_BUTTON, hframe, 0, &window,
                                           dfsFdskPteditWinProc);
      txwSetWindowUShort( ptinfo->hHexCHS, TXQWS_ID, DFSD_WID_PCHS);
      txwSetWindowUShort( ptinfo->hHexCHS, TXQWS_GROUP, 1);

      //- Partition description string
      style = TXWS_OUTPUT | TXWS_HCHILD2MOVE;
      txwSetupWindowData(
         1, 23, 1, DFSD_PTEDIT_WC -25,          // UL corner + vert/hor size
         style, 0,                              // window frame style + help
         ' ', ' ',                              // window/border chars
         TXWSCHEME_COLORS,                      // std scheme colors
         "", "",
         &window);
      window.sl.buf     = pdesc;
      ptinfo->hPartDesc = txwCreateWindow(   hframe, TXW_STLINE, hframe, 0, &window,
                                             dfsFdskPteditWinProc);
      txwSetWindowUShort( ptinfo->hPartDesc, TXQWS_ID, DFSD_WID_PDES);
      TRACES(( "PDESC: windows handle is : %8.8lx\n", ptinfo->hPartDesc));

      //- Table identification string
      style = TXWS_OUTPUT | TXWS_HCHILD2MOVE;
      txwSetupWindowData(
         2, 23, 1, DFSD_PTEDIT_WC -25,          // UL corner + vert/hor size
         style, 0,                              // window frame style + help
         ' ', ' ', TXWSCHEME_COLORS, "", "",
         &window);
      window.tv.topline = 0; window.tv.maxtop  = TXW_INVALID;
      window.tv.leftcol = 0; window.tv.maxcol  = TXW_INVALID;
      window.tv.buf     = mText;
      htext = txwCreateWindow( hframe, TXW_TEXTVIEW, hframe, 0, &window,
                               txwDefWindowProc);
      txwSetWindowUShort(      htext, TXQWS_ID, DFSD_WID_PTXT);

      //- Table column header string, left
      style = TXWS_OUTPUT | TXWS_VCHILD2MOVE;
      txwSetupWindowData(
         4, 1, 2, 17,                           // UL corner + vert/hor size
         style, 0,                              // window frame style + help
         ' ', ' ', TXWSCHEME_COLORS, "", "",
         &window);
      window.tv.topline = 0; window.tv.maxtop  = TXW_INVALID;
      window.tv.leftcol = 0; window.tv.maxcol  = TXW_INVALID;
      window.tv.buf     = ptedit_headerL;       // static header line
      hheadL = txwCreateWindow( hframe, TXW_TEXTVIEW, hframe, 0, &window,
                                txwDefWindowProc);

      //- Table column header string, Sort + Active
      style = TXWS_OUTPUT | TXWS_VCHILD2MOVE;
      txwSetupWindowData(
         6, 1, 1, 5,                            // UL corner + vert/hor size
         style, 0,                              // window frame style + help
         ' ', ' ', TXWSCHEME_COLORS, "", "",
         &window);
      window.tv.topline = 0; window.tv.maxtop  = TXW_INVALID;
      window.tv.leftcol = 0; window.tv.maxcol  = TXW_INVALID;
      window.tv.buf     = ptedit_headerS;       // static header line
      hheadS = txwCreateWindow( hframe, TXW_TEXTVIEW, hframe, 0, &window,
                                txwDefWindowProc);

      //- Table column header string, middle
      style = TXWS_OUTPUT | TXWS_VCHILD2MOVE | TXWS_HCHILD2MOVE;
      txwSetupWindowData(
         4, 21, 2, 28,                          // UL corner + vert/hor size
         style, 0,                              // window frame style + help
         ' ', ' ', TXWSCHEME_COLORS, "", "",
         &window);
      window.tv.topline = 0; window.tv.maxtop  = TXW_INVALID;
      window.tv.leftcol = 0; window.tv.maxcol  = TXW_INVALID;
      window.tv.buf     = ptedit_headerM;       // static header line
      hheadM = txwCreateWindow( hframe, TXW_TEXTVIEW, hframe, 0, &window,
                                txwDefWindowProc);

      //- Table column header string, right
      style = TXWS_OUTPUT | TXWS_VCHILD2MOVE | TXWS_HCHILD_MOVE;
      txwSetupWindowData(
         4, 55, 2, 18,                          // UL corner + vert/hor size
         style, 0,                              // window frame style + help
         ' ', ' ', TXWSCHEME_COLORS, "", "",
         &window);
      window.tv.topline = 0; window.tv.maxtop  = TXW_INVALID;
      window.tv.leftcol = 0; window.tv.maxcol  = TXW_INVALID;
      window.tv.buf     = ptedit_headerR;       // static header line
      hheadR = txwCreateWindow( hframe, TXW_TEXTVIEW, hframe, 0, &window,
                                txwDefWindowProc);

      for (nr = 0; nr < 4; nr++)                // all fields for each entry
      {
         PTEDIT_ENTRY *pi = &(ptinfo->e[nr]);
         sprintf( pi->vEnr, "%u", nr +1);       // entry and custom-sort number

         //- Type description field, output only
         style = TXWS_OUTPUT | TXWS_VCHILD2MOVE;
         txwSetupWindowData(
            DFSD_PT_C,  9, 1, DFSD_DESC,        // UL corner + vert/hor size
            style,  0,                          // style & help
            ' ', ' ',                           // window/border chars
            cLineTextNr1, TXWSCHEME_COLOR5,     // line-1 + scheme colors
            "", "",
            &window);
         window.sl.buf = pi->vDes;
         pi->hDes = txwCreateWindow(   hframe, TXW_STLINE, hframe, 0, &window,
                                       dfsFdskPteditWinProc);
         txwSetWindowUShort( pi->hDes, TXQWS_ID, DFSD_PT_WID( DFSD_DES, nr));

         //- Calculated Begin Cyl  output only
         style = TXWS_OUTPUT | TXWS_VCHILD2MOVE | TXWS_HCHILD2MOVE;
         txwSetupWindowData(
            DFSD_PT_C, 19, 1, DFSD_CYLS -2,     // UL corner + vert/hor size
            style,  0,                          // style & help
            ' ', ' ',                           // window/border chars
            cLineTextNr1, TXWSCHEME_COLOR5,     // line-1 + scheme colors
            "", "",
            &window);
         window.sl.buf     = pi->vCbc;
         pi->hCbc = txwCreateWindow(   hframe, TXW_STLINE, hframe, 0, &window,
                                       dfsFdskPteditWinProc);
         txwSetWindowUShort( pi->hCbc, TXQWS_ID, DFSD_PT_WID( DFSD_CBC, nr));

         //- Calculated Begin Head output only
         style = TXWS_OUTPUT | TXWS_VCHILD2MOVE | TXWS_HCHILD2MOVE;
         txwSetupWindowData(
            DFSD_PT_C, 26, 1, DFSD_HSEC -2,     // UL corner + vert/hor size
            style,  0,                          // style & help
            ' ', ' ',                           // window/border chars
            cLineTextNr1, TXWSCHEME_COLOR5,     // line-1 + scheme colors
            "", "",
            &window);
         window.sl.buf     = pi->vCbh;
         pi->hCbh = txwCreateWindow(   hframe, TXW_STLINE, hframe, 0, &window,
                                       dfsFdskPteditWinProc);
         txwSetWindowUShort( pi->hCbh, TXQWS_ID, DFSD_PT_WID( DFSD_CBH, nr));

         //- Calculated Begin Sect output only
         style = TXWS_OUTPUT | TXWS_VCHILD2MOVE | TXWS_HCHILD2MOVE;
         txwSetupWindowData(
            DFSD_PT_C, 30, 1, DFSD_HSEC -2,     // UL corner + vert/hor size
            style,  0,                          // style & help
            ' ', ' ',                           // window/border chars
            cLineTextNr1, TXWSCHEME_COLOR5,     // line-1 + scheme colors
            "", "",
            &window);
         window.sl.buf     = pi->vCbs;
         pi->hCbs = txwCreateWindow(   hframe, TXW_STLINE, hframe, 0, &window,
                                       dfsFdskPteditWinProc);
         txwSetWindowUShort( pi->hCbs, TXQWS_ID, DFSD_PT_WID( DFSD_CBS, nr));

         //- Calculated End   Cyl  output only
         style = TXWS_OUTPUT | TXWS_VCHILD2MOVE | TXWS_HCHILD2MOVE;
         txwSetupWindowData(
            DFSD_PT_C, 35, 1, DFSD_CYLS -2,     // UL corner + vert/hor size
            style,  0,                          // style & help
            ' ', ' ',                           // window/border chars
            cLineTextNr1, TXWSCHEME_COLOR5,     // line-1 + scheme colors
            "", "",
            &window);
         window.sl.buf = pi->vCec;
         pi->hCec = txwCreateWindow(   hframe, TXW_STLINE, hframe, 0, &window,
                                       dfsFdskPteditWinProc);
         txwSetWindowUShort( pi->hCec, TXQWS_ID, DFSD_PT_WID( DFSD_CEC, nr));

         //- Calculated End   Head output only
         style = TXWS_OUTPUT | TXWS_VCHILD2MOVE | TXWS_HCHILD2MOVE;
         txwSetupWindowData(
            DFSD_PT_C, 42, 1, DFSD_HSEC -2,     // UL corner + vert/hor size
            style,  0,                          // style & help
            ' ', ' ',                           // window/border chars
            cLineTextNr1, TXWSCHEME_COLOR5,     // line-1 + scheme colors
            "", "",
            &window);
         window.sl.buf = pi->vCeh;
         pi->hCeh = txwCreateWindow(   hframe, TXW_STLINE, hframe, 0, &window,
                                       dfsFdskPteditWinProc);
         txwSetWindowUShort( pi->hCeh, TXQWS_ID, DFSD_PT_WID( DFSD_CEH, nr));

         //- Calculated End   Sect output only
         style = TXWS_OUTPUT | TXWS_VCHILD2MOVE | TXWS_HCHILD2MOVE;
         txwSetupWindowData(
            DFSD_PT_C, 46, 1, DFSD_HSEC -2,     // UL corner + vert/hor size
            style,  0,                          // style & help
            ' ', ' ',                           // window/border chars
            cLineTextNr1, TXWSCHEME_COLOR5,     // line-1 + scheme colors
            "", "",
            &window);
         window.sl.buf = pi->vCes;
         pi->hCes = txwCreateWindow(   hframe, TXW_STLINE, hframe, 0, &window,
                                       dfsFdskPteditWinProc);
         txwSetWindowUShort( pi->hCes, TXQWS_ID, DFSD_PT_WID( DFSD_CES, nr));

         //- Start position, absolute sector output only
         style = TXWS_OUTPUT | TXWS_VCHILD2MOVE | TXWS_HCHILD_MOVE;
         txwSetupWindowData(
            DFSD_PT_C, 51, 1, DFSD_BIGV -2,     // UL corner + vert/hor size
            style,  0,                          // style & help
            ' ', ' ',                           // window/border chars
            cLineTextNr1, TXWSCHEME_COLOR5,     // line-1 + scheme colors
            "", "",
            &window);
         window.sl.buf     = pi->vAbs;
         pi->hAbs = txwCreateWindow(   hframe, TXW_STLINE, hframe, 0, &window,
                                       dfsFdskPteditWinProc);
         txwSetWindowUShort( pi->hAbs, TXQWS_ID, DFSD_PT_WID( DFSD_ABS, nr));

         //- Size in megabytes output only
         style = TXWS_OUTPUT | TXWS_VCHILD2MOVE | TXWS_HCHILD_MOVE;
         txwSetupWindowData(
            DFSD_PT_C, 63, 1, DFSD_BIGV -2,     // UL corner + vert/hor size
            style,  0,                          // style & help
            ' ', ' ',                           // window/border chars
            cLineTextNr1, TXWSCHEME_COLOR5,     // line-1 + scheme colors
            "", "",
            &window);
         window.sl.buf     = pi->vSmb;
         pi->hSmb = txwCreateWindow(   hframe, TXW_STLINE, hframe, 0, &window,
                                       dfsFdskPteditWinProc);
         txwSetWindowUShort( pi->hSmb, TXQWS_ID, DFSD_PT_WID( DFSD_SMB, nr));


         //- Partition-entry number = custom sort-order entryfield
         style = TXWS_ENTRYB | TXWS_VCHILD2MOVE;
         txwSetupWindowData(
            DFSD_PT_T, 0, 1, DFSD_PENR,         // UL column + vert/hor size
            style, DFSH_PT + DFSD_ENR,          // style & help
            ' ', ' ',                           // window/border chars
            TXWSCHEME_COLORS,                   // std scheme colors
            "", "",
            &window);
         window.sl.buf = pi->vEnr;
         window.ef.leftcol = 0;
         window.ef.maxcol  = TXW_INVALID;
         window.ef.curpos  = 0;
         window.ef.rsize   = 1;
         window.ef.buf     = pi->vEnr;
         window.ef.history = NULL;
         pi->hEnr = txwCreateWindow(   hframe, TXW_ENTRYFIELD, hframe, 0, &window,
                                       dfsFdskPteditWinProc);
         txwSetWindowUShort( pi->hEnr, TXQWS_ID, DFSD_PT_WID( DFSD_ENR, nr));

         //- Active flag, checkbox (automatic)
         style = TXWS_AUTOCHK | TXWS_VCHILD2MOVE;
         txwSetupWindowData(
            DFSD_PT_T, 4, 1, DFSD_PACT,         // UL column + vert/hor size
            style, DFSH_PT + DFSD_ACT,          // style & help
            ' ', ' ', TXWSCHEME_COLORS, "", "",
            &window);
         window.bu.text    = "";
         window.bu.checked = &(pi->vAct);
         pi->hAct = txwCreateWindow(   hframe, TXW_BUTTON, hframe, 0, &window,
                                       dfsFdskPteditWinProc);
         txwSetWindowUShort( pi->hAct, TXQWS_ID, DFSD_PT_WID( DFSD_ACT, nr));

         //- Hidden flag, checkbox (non-auto!)
         style = TXWS_CHECKB | TXWS_VCHILD2MOVE;
         txwSetupWindowData(
            DFSD_PT_T,  9, 1, DFSD_PHID,        // UL column + vert/hor size
            style, DFSH_PT + DFSD_HID,          // style & help
            ' ', ' ', TXWSCHEME_COLORS, "", "",
            &window);
         window.bu.text    = "";
         window.bu.checked = &(pi->vHid);
         pi->hHid = txwCreateWindow(   hframe, TXW_BUTTON, hframe, 0, &window,
                                       dfsFdskPteditWinProc);
         txwSetWindowUShort( pi->hHid, TXQWS_ID, DFSD_PT_WID( DFSD_HID, nr));

         //- System type entryfield
         style = TXWS_ENTRYB | TXWS_VCHILD2MOVE;
         txwSetupWindowData(
            DFSD_PT_T, 14, 1, DFSD_PTYP,        // UL column + vert/hor size
            style, DFSH_PT + DFSD_TYP,          // style & help
            ' ', ' ', TXWSCHEME_COLORS, "", "",
            &window);
         window.ef.leftcol = 0;
         window.ef.maxcol  = TXW_INVALID;
         window.ef.curpos  = 0;
         window.ef.rsize   = 2;
         window.ef.buf     = pi->vTyp;
         window.ef.history = NULL;
         pi->hTyp = txwCreateWindow(   hframe, TXW_ENTRYFIELD, hframe, 0, &window,
                                       dfsFdskPteditWinProc);
         txwSetWindowUShort( pi->hTyp, TXQWS_ID, DFSD_PT_WID( DFSD_TYP, nr));

         //- Table Begin Cyl  entryfield
         style = TXWS_ENTRYB | TXWS_VCHILD2MOVE | TXWS_HCHILD2MOVE;
         txwSetupWindowData(
            DFSD_PT_T, 18, 1, DFSD_CYLS,        // UL column + vert/hor size
            style, DFSH_PT + DFSD_TBC,          // style & help
            ' ', ' ', TXWSCHEME_COLORS, "", "",
            &window);
         window.ef.leftcol = 0;
         window.ef.maxcol  = TXW_INVALID;
         window.ef.curpos  = 0;
         window.ef.rsize   = 6;
         window.ef.buf     = pi->vTbc;
         window.ef.history = NULL;
         pi->hTbc = txwCreateWindow(   hframe, TXW_ENTRYFIELD, hframe, 0, &window,
                                       dfsFdskPteditWinProc);
         txwSetWindowUShort( pi->hTbc, TXQWS_ID, DFSD_PT_WID( DFSD_TBC, nr));

         //- Table Begin Head entryfield
         style = TXWS_ENTRYB | TXWS_VCHILD2MOVE | TXWS_HCHILD2MOVE;
         txwSetupWindowData(
            DFSD_PT_T, 25, 1, DFSD_HSEC,        // UL column + vert/hor size
            style, DFSH_PT + DFSD_TBH,          // style & help
            ' ', ' ', TXWSCHEME_COLORS, "", "",
            &window);
         window.ef.leftcol = 0;
         window.ef.maxcol  = TXW_INVALID;
         window.ef.curpos  = 0;
         window.ef.rsize   = 3;
         window.ef.buf     = pi->vTbh;
         window.ef.history = NULL;
         pi->hTbh = txwCreateWindow(   hframe, TXW_ENTRYFIELD, hframe, 0, &window,
                                       dfsFdskPteditWinProc);
         txwSetWindowUShort( pi->hTbh, TXQWS_ID, DFSD_PT_WID( DFSD_TBH, nr));

         //- Table Begin Sect entryfield
         style = TXWS_ENTRYB | TXWS_VCHILD2MOVE | TXWS_HCHILD2MOVE;
         txwSetupWindowData(
            DFSD_PT_T, 29, 1, DFSD_HSEC,        // UL column + vert/hor size
            style, DFSH_PT + DFSD_TBS,          // style & help
            ' ', ' ', TXWSCHEME_COLORS, "", "",
            &window);
         window.ef.leftcol = 0;
         window.ef.maxcol  = TXW_INVALID;
         window.ef.curpos  = 0;
         window.ef.rsize   = 3;
         window.ef.buf     = pi->vTbs;
         window.ef.history = NULL;
         pi->hTbs = txwCreateWindow(   hframe, TXW_ENTRYFIELD, hframe, 0, &window,
                                       dfsFdskPteditWinProc);
         txwSetWindowUShort( pi->hTbs, TXQWS_ID, DFSD_PT_WID( DFSD_TBS, nr));

         //- Table End   Cyl  entryfield
         style = TXWS_ENTRYB | TXWS_VCHILD2MOVE | TXWS_HCHILD2MOVE;
         txwSetupWindowData(
            DFSD_PT_T, 34, 1, DFSD_CYLS,        // UL column + vert/hor size
            style, DFSH_PT + DFSD_TEC,          // style & help
            ' ', ' ', TXWSCHEME_COLORS, "", "",
            &window);
         window.ef.leftcol = 0;
         window.ef.maxcol  = TXW_INVALID;
         window.ef.curpos  = 0;
         window.ef.rsize   = 6;
         window.ef.buf     = pi->vTec;
         window.ef.history = NULL;
         pi->hTec = txwCreateWindow(   hframe, TXW_ENTRYFIELD, hframe, 0, &window,
                                       dfsFdskPteditWinProc);
         txwSetWindowUShort( pi->hTec, TXQWS_ID, DFSD_PT_WID( DFSD_TEC, nr));

         //- Table End   Head entryfield
         style = TXWS_ENTRYB | TXWS_VCHILD2MOVE | TXWS_HCHILD2MOVE;
         txwSetupWindowData(
            DFSD_PT_T, 41, 1, DFSD_HSEC,        // UL column + vert/hor size
            style, DFSH_PT + DFSD_TEH,          // style & help
            ' ', ' ', TXWSCHEME_COLORS, "", "",
            &window);
         window.ef.leftcol = 0;
         window.ef.maxcol  = TXW_INVALID;
         window.ef.curpos  = 0;
         window.ef.rsize   = 3;
         window.ef.buf     = pi->vTeh;
         window.ef.history = NULL;
         pi->hTeh = txwCreateWindow(   hframe, TXW_ENTRYFIELD, hframe, 0, &window,
                                       dfsFdskPteditWinProc);
         txwSetWindowUShort( pi->hTeh, TXQWS_ID, DFSD_PT_WID( DFSD_TEH, nr));

         //- Table End   Sect entryfield
         style = TXWS_ENTRYB | TXWS_VCHILD2MOVE | TXWS_HCHILD2MOVE;
         txwSetupWindowData(
            DFSD_PT_T, 45, 1, DFSD_HSEC,        // UL column + vert/hor size
            style, DFSH_PT + DFSD_TES,          // style & help
            ' ', ' ', TXWSCHEME_COLORS, "", "",
            &window);
         window.ef.leftcol = 0;
         window.ef.maxcol  = TXW_INVALID;
         window.ef.curpos  = 0;
         window.ef.rsize   = 3;
         window.ef.buf     = pi->vTes;
         window.ef.history = NULL;
         pi->hTes = txwCreateWindow(   hframe, TXW_ENTRYFIELD, hframe, 0, &window,
                                       dfsFdskPteditWinProc);
         txwSetWindowUShort( pi->hTes, TXQWS_ID, DFSD_PT_WID( DFSD_TES, nr));

         //- Start position, relative sector entryfield
         style = TXWS_ENTRYB | TXWS_VCHILD2MOVE | TXWS_HCHILD_MOVE;
         txwSetupWindowData(
            DFSD_PT_T, 50, 1, DFSD_BIGV,        // UL column + vert/hor size
            style, DFSH_PT + DFSD_REL,          // style & help
            ' ', ' ', TXWSCHEME_COLORS, "", "",
            &window);
         window.ef.leftcol = 0;
         window.ef.maxcol  = TXW_INVALID;
         window.ef.curpos  = 0;
         window.ef.rsize   = 10;
         window.ef.buf     = pi->vRel;
         window.ef.history = NULL;
         pi->hRel = txwCreateWindow(   hframe, TXW_ENTRYFIELD, hframe, 0, &window,
                                       dfsFdskPteditWinProc);
         txwSetWindowUShort( pi->hRel, TXQWS_ID, DFSD_PT_WID( DFSD_REL, nr));

         //- Size in sectors entryfield
         style = TXWS_ENTRYB | TXWS_VCHILD2MOVE | TXWS_HCHILD_MOVE;
         txwSetupWindowData(
            DFSD_PT_T, 62, 1, DFSD_BIGV,        // UL column + vert/hor size
            style, DFSH_PT + DFSD_SEC,          // style & help
            ' ', ' ', TXWSCHEME_COLORS, "", "",
            &window);
         window.ef.leftcol = 0;
         window.ef.maxcol  = TXW_INVALID;
         window.ef.curpos  = 0;
         window.ef.rsize   = 10;
         window.ef.buf     = pi->vSec;
         window.ef.history = NULL;
         pi->hSec = txwCreateWindow(   hframe, TXW_ENTRYFIELD, hframe, 0, &window,
                                       dfsFdskPteditWinProc);
         txwSetWindowUShort( pi->hSec, TXQWS_ID, DFSD_PT_WID( DFSD_SEC, nr));
      }

      style = TXWS_PBUTTON | TXBS_DLGE_BUTTON | TXWS_VCHILD_MOVE;
      txwSetupWindowData(
         position.bottom - 6,                   // UL corner, line   (relative)
         1,                                     // column
         3, strlen( dfsdf_bt_csort) + 4,        // vertical/horizontal size
         style, DFSH_PT_BSORT,                  // style and help
         ' ', ' ', TXWSCHEME_COLORS, "",  "",
         &window);
      window.bu.text = dfsdf_bt_csort;
      hbtcust = txwCreateWindow(  hframe, TXW_BUTTON, hframe, 0, &window,
                                  dfsFdskPteditWinProc);
      txwSetWindowUShort( hbtcust, TXQWS_ID, DFSD_WID_BCUS); // rc == ID

      if (cbp->thisSn != 0)
      {
         style = TXWS_PBUTTON | TXWS_VCHILD_MOVE | TXWS_HCHILD2MOVE;
         txwSetupWindowData(
            position.bottom - 6,                // UL corner, line   (relative)
            14,                                 // column
            3, strlen( dfsdf_bt_prev) + 4,      // vertical/horizontal size
            style, DFSH_PT_BNAVI,               // style and help
            ' ', ' ', TXWSCHEME_COLORS, "",  "",
            &window);
         window.bu.text = dfsdf_bt_prev;
         hbtprev = txwCreateWindow(  hframe, TXW_BUTTON, hframe, 0, &window,
                                     dfsFdskPteditWinProc);
         txwSetWindowUShort( hbtprev, TXQWS_ID, DFSD_PREV); // rc == ID
      }

      style = TXWS_PBUTTON | TXWS_VCHILD_MOVE | TXWS_HCHILD2MOVE;
      txwSetupWindowData(
         position.bottom - 6,                   // UL corner, line   (relative)
         23,                                    // column
         3, strlen( dfsdf_bt_next) + 4,         // vertical/horizontal size
         style, DFSH_PT_BNAVI,                  // style and help
         ' ', ' ', TXWSCHEME_COLORS, "",  "",
         &window);
      window.bu.text = dfsdf_bt_next;
      hbtnext = txwCreateWindow(  hframe, TXW_BUTTON, hframe, 0, &window,
                                  dfsFdskPteditWinProc);
      txwSetWindowUShort( hbtnext, TXQWS_ID, DFSD_NEXT); // rc == ID

      style = TXWS_PBUTTON | TXWS_VCHILD_MOVE | TXWS_HCHILD2MOVE;
      txwSetupWindowData(
         position.bottom - 6,                   // UL corner, line   (relative)
         34,                                    // column
         3, strlen( dfsdf_bt_ok) + 4,           // vertical/horizontal size
         style, DFSH_PT_BNAVI,                  // style and help
         ' ', ' ', TXWSCHEME_COLORS, "",  "",
         &window);
      window.bu.text = dfsdf_bt_ok;
      hbtok = txwCreateWindow(   hframe, TXW_BUTTON, hframe, 0, &window,
                                 dfsFdskPteditWinProc);
      txwSetWindowUShort( hbtok, TXQWS_ID, TXMBID_OK); // rc == ID

      style = TXWS_PBUTTON | TXWS_VCHILD_MOVE | TXWS_HCHILD2MOVE;
      txwSetupWindowData(
         position.bottom - 6,                   // UL corner, line   (relative)
         43,                                    // column
         3, strlen( dfsdf_bt_cancel) + 4,       // vertical/horizontal size
         style, DFSH_PT_BNAVI,                  // style and help
         ' ', ' ', TXWSCHEME_COLORS, "",  "",
         &window);
      window.bu.text = dfsdf_bt_cancel;
      hbtcan = txwCreateWindow(  hframe, TXW_BUTTON, hframe, 0, &window,
                                 dfsFdskPteditWinProc);
      txwSetWindowUShort( hbtcan, TXQWS_ID, TXMBID_CANCEL); // rc == ID

      style = TXWS_PBUTTON | TXBS_DLGE_BUTTON | TXWS_VCHILD_MOVE | TXWS_HCHILD_MOVE;
      txwSetupWindowData(
         position.bottom - 6,                   // UL corner, line   (relative)
         position.right - 20,                   // column
         3, strlen( dfsdf_bt_ssort) + 4,        // vertical/horizontal size
         style, DFSH_PT_BSORT,                  // style and help
         ' ', ' ', TXWSCHEME_COLORS, "",  "",
         &window);
      window.bu.text = dfsdf_bt_ssort;
      hbtsort = txwCreateWindow(  hframe, TXW_BUTTON, hframe, 0, &window,
                                  dfsFdskPteditWinProc);
      txwSetWindowUShort( hbtsort, TXQWS_ID, DFSD_WID_BSRT); // rc == ID

      rc = txwDlgBox( TXHWND_DESKTOP, TXHWND_DESKTOP,
                      dfsFdskNpDlgWinProc, hframe, ptinfo);

      if (rc != TXDID_CANCEL)                   // copy changed values back
      {
         cbp->cbOptNum2 = 0;                    // Back to MBR if no next
         for (nr = 0; nr < 4; nr++)             // handle all entries
         {
            PTEDIT_ENTRY *pi = &(ptinfo->e[nr]);
            DFSPARTENTRY *pe = &(br->PartitionTable[nr]);

            if (pi->vAct)
            {
               pe->Status         |= 0x80;
            }
            else
            {
               pe->Status         &= 0x7f;
            }
            pe->PartitionType      = (BYTE) pi->uTyp;
            pe->BootSectorOffset   =        pi->lRel;
            pe->NumberOfSectors    =        pi->lSec;

            pe->FirstSector.SecCyl = DFSCOMBINE( pi->uTbs, pi->uTbc);
            pe->FirstSector.Head   = (BYTE) pi->uTbh;

            pe->LastSector.SecCyl  = DFSCOMBINE( pi->uTes, pi->uTec);
            pe->LastSector.Head    = (BYTE) pi->uTeh;

            if (dfsIsExtendedType(pi->uTyp))
            {
               cbp->cbOptNum2 = pi->lAbs;       // PSN of next EBR
            }
         }

         switch (rc)                            // dialog result
         {
            case DFSD_REQ5:
            case DFSD_REQ6:
            case DFSD_REQ7:
            case DFSD_REQ8:
            case DFSD_EDIT:                     // request editor
               cbp->stype     = ptinfo->eEntry; // position info
               cbp->cbOptNum2 = ptinfo->eSect;  // position info
            case DFSD_NEXT:                     // request next
            case DFSD_PREV:                     // request prev
               cbp->cbOptNum1 = rc;
               break;

            default:
               cbp->cbOptNum1 = DFSD_NONE;
               break;
         }
         rc = NO_ERROR;
      }
      else                                      // cancel / escape
      {
         cbp->cbOptNum1 = DFSD_NONE;            // stop Ptable iteration
      }
      txFreeText( mText);                       // free the text memory
   }
   else if (ptinfo == NULL)
   {
      rc = DFS_ALLOC_ERROR;
   }
   TxFreeMem( ptinfo);                          // free when allocated
   RETURN (rc);
}                                               // end 'dfsFdskPteditDialog'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Dialog window procedure, for the PTEDIT dialog fields
/*****************************************************************************/
static ULONG dfsFdskPteditWinProc               // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   ULONG               mp1,                     // IN    msg param 1
   ULONG               mp2                      // IN    msg param 2
)
{
   ULONG               rc = NO_ERROR;

   ENTER();
   TRCMSG( hwnd, msg, mp1, mp2);

   if (hwnd != 0)
   {
      TXWINDOW     *win    = txwWindowData(  hwnd);
      TXWHANDLE     owner  = txwQueryWindow( hwnd, TXQW_OWNER); // get dialog info
      USHORT        wid    = txwQueryWindowUShort( hwnd, TXQWS_ID);
      USHORT        field  = DFSD_PT_BSE( wid);
      int           entry  = DFSD_PT_ENT( wid);
      PTEDIT_INFO  *ptinfo = (PTEDIT_INFO *) txwQueryWindowPtr( owner, TXQWP_USER);
      PTEDIT_ENTRY *pi     = &(ptinfo->e[entry]);
      USHORT        maxvalue;
      ULONG         refvalue;

      TRACES(( "own: %8.8lx with attached PTINFO: %8.8lx\n", owner, ptinfo));
      TRACES(( "wid: %3hu field: %3hu  entry: %hu\n", wid, field, entry));

      switch (msg)
      {
         case TXWM_CHAR:
            switch (mp2)
            {
               case TXk_UP:                     // to same field, next entry
                  if (DFSD_PT_FLD(wid))         // one of the PTentry fields
                  {
                     if ((field == DFSD_ENR) && (entry == 0))
                     {
                        txwSetFocus( txwWindowFromID( owner, DFSD_WID_PCHS));
                     }
                     else                       // previous table row
                     {
                        txwSetFocus( txwWindowFromID( owner, DFSD_PT_WID( field, ((entry +3) %4))));
                     }
                  }
                  else if (wid == DFSD_WID_PSMA) // to sort-# button
                  {
                     txwSetFocus( txwWindowFromID( owner, DFSD_WID_BCUS));
                  }
                  else if (wid == DFSD_WID_BCUS) // to last Active checkbox
                  {
                     txwSetFocus( txwWindowFromID( owner, DFSD_PT_WID(  DFSD_ENR, 3)));
                  }
                  else if (wid == DFSD_PREV)     // to type checkbox
                  {
                     txwSetFocus( txwWindowFromID( owner, DFSD_PT_WID(  DFSD_TYP, 3)));
                  }
                  else if (wid == DFSD_NEXT)     // to begin-head field
                  {
                     txwSetFocus( txwWindowFromID( owner, DFSD_PT_WID(  DFSD_TBH, 3)));
                  }
                  else if (wid == TXMBID_OK)     // to end-cyl field
                  {
                     txwSetFocus( txwWindowFromID( owner, DFSD_PT_WID(  DFSD_TEC, 3)));
                  }
                  else if (wid == TXMBID_CANCEL) // to end-cyl field
                  {
                     txwSetFocus( txwWindowFromID( owner, DFSD_PT_WID(  DFSD_TES, 3)));
                  }
                  else if (wid == DFSD_WID_BSRT) // to last Active checkbox
                  {
                     txwSetFocus( txwWindowFromID( owner, DFSD_PT_WID(  DFSD_REL, 3)));
                  }
                  else
                  {
                     rc = txwDefWindowProc( hwnd, msg, mp1, mp2);
                  }
                  break;

               case TXk_DOWN:
                  if (DFSD_PT_FLD(wid))         // one of the PTentry fields
                  {
                     if      (((field == DFSD_ENR)  ||
                               (field == DFSD_ACT)  ||
                               (field == DFSD_HID)) && (entry == 3))
                     {
                        txwSetFocus( txwWindowFromID( owner, DFSD_WID_BCUS));
                     }
                     else if ((field == DFSD_TYP) && (entry == 3))
                     {
                        if (ptinfo->tableSn != 0) // EBR
                        {
                           txwSetFocus( txwWindowFromID( owner, DFSD_PREV));
                        }
                        else
                        {
                           txwSetFocus( txwWindowFromID( owner, DFSD_NEXT));
                        }
                     }
                     else if (((field == DFSD_TBC)  ||
                               (field == DFSD_TBH)  ||
                               (field == DFSD_TBS)) && (entry == 3))
                     {
                        txwSetFocus( txwWindowFromID( owner, DFSD_NEXT));
                     }
                     else if ((field == DFSD_TEC)   && (entry == 3))
                     {
                        txwSetFocus( txwWindowFromID( owner, TXMBID_OK));
                     }
                     else if (((field == DFSD_TEH)  ||
                               (field == DFSD_TES)) && (entry == 3))
                     {
                        txwSetFocus( txwWindowFromID( owner, TXMBID_CANCEL));
                     }
                     else if (((field == DFSD_REL)  ||
                               (field == DFSD_SEC)) && (entry == 3))
                     {
                        txwSetFocus( txwWindowFromID( owner, DFSD_WID_BSRT));
                     }
                     else                       // next table row
                     {
                        txwSetFocus( txwWindowFromID( owner, DFSD_PT_WID( field, ((entry +1) %4))));
                     }
                  }
                  else
                  {
                     rc = txwDefWindowProc( hwnd, msg, mp1, mp2);
                  }
                  break;

               case DFSD_REQ5:
               case DFSD_REQ6:
               case DFSD_REQ7:
               case DFSD_REQ8:
               case DFSD_EDIT:                  // request Hex editor
                  ptinfo->eEntry = entry;
                  switch (mp2)
                  {
                     case DFSD_REQ5:
                     case DFSD_REQ6: ptinfo->eSect = pi->lAbs;               break;
                     case DFSD_EDIT: ptinfo->eSect = ptinfo->tableSn;        break;
                     default:        ptinfo->eSect = pi->lAbs + pi->lSec -1; break;
                  }
               case TXk_ENTER:                  // force update of field vars
               case TXk_PGUP:                   // before quiting the dialog
               case TXk_PGDN:                   // (just like focus loss)
               case TXk_F4:
                  txwSendMsg( hwnd, DFSM_UPDT_FIELD, FALSE, 0); // force updates.
               default:
                  rc = txwDefWindowProc( hwnd, msg, mp1, mp2);
                  break;
            }
            break;

         case DFSM_UPDT_FIELD:                  // detect (nr) field changes
         case TXWM_SETFOCUS:                    // on loosing focus too!
            if (DFSD_PT_FLD(wid))               // a PT field ?
            {
               static ULONG old;                // old value to remember
                      ULONG cur = 0;            // current field value

               switch (field)
               {
                  case DFSD_ACT:         cur = (ULONG) win->bu.checked;            break;
                  case DFSD_HID:         cur = (ULONG) win->bu.checked;            break;
                  case DFSD_TYP: txwDlgField2Ul( &cur, win->ef.buf, TRUE);         break;
                  case DFSD_TBC: txwDlgField2Ul( &cur, win->ef.buf, dfsa->hexCHS); break;
                  case DFSD_TBH: txwDlgField2Ul( &cur, win->ef.buf, dfsa->hexCHS); break;
                  case DFSD_TBS: txwDlgField2Ul( &cur, win->ef.buf, dfsa->hexCHS); break;
                  case DFSD_TEC: txwDlgField2Ul( &cur, win->ef.buf, dfsa->hexCHS); break;
                  case DFSD_TEH: txwDlgField2Ul( &cur, win->ef.buf, dfsa->hexCHS); break;
                  case DFSD_TES: txwDlgField2Ul( &cur, win->ef.buf, dfsa->hexCHS); break;
                  case DFSD_REL: txwDlgField2Ul( &cur, win->ef.buf, dfsa->hexSNS); break;
                  case DFSD_SEC:
                     if (strchr( win->ef.buf, '.') == NULL) // simple number ?
                     {
                        txwDlgField2Ul( &cur, win->ef.buf, dfsa->hexSNS);
                     }
                     else                       // decimal megabytes
                     {
                        double  mb = 0.0;
                        sscanf( win->ef.buf, "%lf", &mb);
                        cur = (ULONG) (mb * 2048.0);
                     }
                     break;
               }
               if (mp1)                         // getting Focus ?
               {
                  static int   previous_entry = -1;

                  TRACES(( "PDESC: entry: %d previous_entry: %d\n", entry, previous_entry));
                  if (entry != previous_entry)  // update partition description
                  {
                     txwPostMsg( ptinfo->hPartDesc, DFSM_UPDT_PDESC, (ULONG) entry, 0);
                  }
                  if (win->class == TXW_ENTRYFIELD) // cursor at 1st non-space
                  {
                     win->ef.curpos = (short) strspn( win->ef.buf, " ");
                     txwInvalidateWindow( hwnd, FALSE, FALSE);
                  }
                  old = cur;                    // remember value
               }
               else                             // Loosing Focus again
               {
                  switch (field)                // base field identification
                  {
                     case DFSD_TYP: pi->uTyp = (USHORT) cur; break;
                     case DFSD_TBC: pi->uTbc = (USHORT) cur; break;
                     case DFSD_TBH: pi->uTbh = (USHORT) cur; break;
                     case DFSD_TBS: pi->uTbs = (USHORT) cur; break;
                     case DFSD_TEC: pi->uTec = (USHORT) cur; break;
                     case DFSD_TEH: pi->uTeh = (USHORT) cur; break;
                     case DFSD_TES: pi->uTes = (USHORT) cur; break;
                     case DFSD_REL: pi->lRel =          cur; break;
                     case DFSD_SEC: pi->lSec =          cur; break;
                  }
                  txwPostMsg( hwnd, DFSM_SHOW_FIELD, (cur != old), field);
               }
            }
            if (msg != DFSM_UPDT_FIELD)         // not our private message ?
            {
               rc = txwDefWindowProc( hwnd, msg, mp1, mp2); // default handling too!
            }
            break;

         case DFSM_UPDT_PDESC:                  // update partition description
            refvalue = ptinfo->e[ mp1].lAbs;
            TRACES(( "PDESC: entry %lu lAbs:%8.8lx\n", mp1, refvalue));
            dfsGetPartDesc( dfsGetPartInfo(
                         dfsDiskPsn2PartId( ptinfo->disknr, refvalue)), win->sl.buf);
            txwInvalidateWindow( hwnd, TRUE, FALSE);
            break;

         case DFSM_SHOW_FIELD:                  // 'display' & smart update
            TRACES(( "SHOW_FIELD for %3hu\n", field));
            switch (field)                      // base field identification
            {
               case DFSD_TYP:                   // partition type
                  if ((pi->uTyp) || ((pi->lSec) && (pi->lRel)))
                  {
                     dfsPartTypeDescription( pi->uTyp, pi->vDes);
                  }
                  else                          // still, or again, not used
                  {
                     strcpy(  pi->vDes, "Not used");
                  }
                  txwInvalidateWindow( pi->hDes, TRUE, FALSE);
                  sprintf( pi->vTyp, "%2.2hX", pi->uTyp);
                  if ((mp2 == field) && dfsPartTypeHidable( pi->uTyp))
                  {
                     pi->vHid = ((pi->uTyp & DFS_P_PHIDDEN) != 0);
                     txwInvalidateWindow( pi->hHid, TRUE, FALSE);
                  }
                  txwInvalidateWindow( pi->hTyp, TRUE, FALSE);
                  break;

               case DFSD_REL:                   // bootsector offset (relative)
                  if (mp1)                      // value change ?
                  {
                     pi->lAbs = pi->lRel;
                     if (ptinfo->tableSn != 0)  // not the MBR
                     {
                        if (dfsIsExtendedType(pi->uTyp))
                        {
                           pi->lAbs += ptinfo->ebrBase; // add offset to 1st EBR
                        }
                        else
                        {
                           pi->lAbs += ptinfo->tableSn; // add offset to ebr PSN
                        }
                     }
                     txwDlgUl2Field( pi->hAbs, pi->lAbs, pi->vAbs, 10, dfsa->hexSNS);

                     if ((mp2 == field) && pteSmart && pi->uTyp) // smart update allowed
                     {
                        dfsFdskPteLinear2Chs( pi, field);
                     }
                  }
                  txwDlgUl2Field( pi->hRel, pi->lRel, pi->vRel, 10, dfsa->hexSNS);
                  break;

               case DFSD_ABS:                   // Absolute start sector
                  pi->lRel = pi->lAbs;          // (indirect update only)
                  if (ptinfo->tableSn != 0)     // not the MBR
                  {
                     if (dfsIsExtendedType(pi->uTyp))
                     {
                        refvalue = ptinfo->ebrBase;
                     }
                     else
                     {
                        refvalue = ptinfo->tableSn;
                     }
                     if (pi->lRel > refvalue)
                     {
                        pi->lRel -= refvalue;   // subtract offset
                     }
                     else
                     {
                        pi->lRel  = 0;
                     }
                  }
                  txwDlgUl2Field( pi->hRel, pi->lRel, pi->vRel, 10, dfsa->hexSNS);
                  txwDlgUl2Field( pi->hAbs, pi->lAbs, pi->vAbs, 10, dfsa->hexSNS);
                  break;

               case DFSD_SEC:                   // size in sectors
                  if (mp1)                      // value change ?
                  {
                     sprintf( pi->vSmb, "%10.1lf", TXSMIB(pi->lSec, dfsGetSectorSize()));
                     txwInvalidateWindow( pi->hSmb, TRUE, FALSE);
                  }
                  if ((mp2 == field) && pteSmart && pi->uTyp) // smart update allowed
                  {
                     dfsFdskPteLinear2Chs( pi, field);
                  }
                  txwDlgUl2Field( pi->hSec, pi->lSec, pi->vSec, 10, dfsa->hexSNS);
                  break;

               case DFSD_TBC:                   // Table begin Cyl
                  if ((mp1) || ((pi->uTbc == 0) && pteSmart && pi->uTyp))
                  {
                     if ((ptinfo->tableSn == 0) && (pi->uTbc == 0) &&
                          dfsIsExtendedType( pi->uTyp) && (pteSmart))
                     {
                        pi->uTbc = 1;           // 1st extended at cylinder 1!
                        if (pi->uTec == 0)
                        {
                           pi->uTec = 1;        // adjust end-cyl too
                           txwPostMsg( pi->hTec, DFSM_SHOW_FIELD, TRUE,
                                       DFSD_TEC); // allow smart-updates!
                        }
                     }
                     pi->uCbc = pi->uTbc;       // Keep un-clipped value here!
                     if (pi->uTbc > 1023)
                     {
                        pi->uTbc  = 1023;       // clip to 1024 cylinder limit
                     }
                     txwDlgUl2Field( pi->hCbc, pi->uCbc, pi->vCbc,  6, dfsa->hexCHS);

                     if ((mp2 == field) && pteSmart && pi->uTyp) // smart update allowed
                     {
                        if (pi->uTbh == 0)      // no head value yet
                        {
                           if (((ptinfo->tableSn == 0) && // in the MBR
                                (pi->uTbc == 0))       || // at 1st cylinder
                               ((ptinfo->tableSn != 0) && // or non-EXT in EBR
                                (!dfsIsExtendedType( pi->uTyp))))
                           {
                              pi->uTbh = 1;
                           }
                           txwPostMsg( pi->hTbh, DFSM_SHOW_FIELD, TRUE, field);
                        }
                        if (pi->uTbs == 0)      // no sector value yet
                        {
                           pi->uTbs = 1;
                           txwPostMsg( pi->hTbs, DFSM_SHOW_FIELD, TRUE, field);
                        }
                        dfsFdskPteChs2Linear( pi, field);
                     }
                  }
                  txwDlgUl2Field( pi->hTbc, pi->uTbc, pi->vTbc,  6, dfsa->hexCHS);
                  break;

               case DFSD_TBH:                   // Table begin Head
                  if ((mp1) || ((pi->uTbh == 0) && pteSmart && pi->uTyp))
                  {
                     maxvalue = (pteSmart) ? dfstGeoHeads(DFSTORE) -1 : 254;
                     if (pi->uTbh > maxvalue)
                     {
                        pi->uTbh  = maxvalue;
                     }
                     pi->uCbh = pi->uTbh;
                     txwDlgUl2Field( pi->hCbh, pi->uCbh, pi->vCbh,  3, dfsa->hexCHS);

                     if ((mp2 == field) && pteSmart && pi->uTyp) // smart update allowed
                     {
                        dfsFdskPteChs2Linear( pi, field);
                     }
                  }
                  txwDlgUl2Field( pi->hTbh, pi->uTbh, pi->vTbh,  3, dfsa->hexCHS);
                  break;

               case DFSD_TBS:                   // Table begin Sector
                  if ((mp1) || ((pi->uTbs == 0) && pteSmart && pi->uTyp))
                  {
                     maxvalue = (pteSmart) ? dfstGeoSectors(DFSTORE) : 63;
                     if (pi->uTbs > maxvalue)
                     {
                        pi->uTbs  = maxvalue;
                     }
                     else if ((pi->uTbs == 0) && pteSmart && pi->uTyp)
                     {
                        pi->uTbs  = 1;
                     }
                     pi->uCbs = pi->uTbs;
                     txwDlgUl2Field( pi->hCbs, pi->uCbs, pi->vCbs,  3, dfsa->hexCHS);

                     if ((mp2 == field) && pteSmart && pi->uTyp) // smart update allowed
                     {
                        dfsFdskPteChs2Linear( pi, field);
                     }
                  }
                  txwDlgUl2Field( pi->hTbs, pi->uTbs, pi->vTbs,  3, dfsa->hexCHS);
                  break;

               case DFSD_TEC:                   // Table end   Cyl
                  if ((mp1) || ((pi->uTec == 0) && pteSmart && pi->uTyp))
                  {
                     pi->uCec = pi->uTec;       // Keep un-clipped value here!
                     if (pi->uTec > 1023)
                     {
                        pi->uTec  = 1023;       // clip to 1024 cylinder limit
                     }
                     txwDlgUl2Field( pi->hCec, pi->uCec, pi->vCec,  6, dfsa->hexCHS);

                     if ((mp2 == field) && pteSmart && pi->uTyp) // smart update allowed
                     {
                        if (pi->uTeh == 0)      // no head value yet
                        {
                           pi->uTeh = 255;      // will be clipped to maximum
                           txwPostMsg( pi->hTeh, DFSM_SHOW_FIELD, TRUE, field);
                        }
                        if (pi->uTes == 0)      // no sector value yet
                        {
                           pi->uTes = 255;      // will be clipped to maximum
                           txwPostMsg( pi->hTes, DFSM_SHOW_FIELD, TRUE, field);
                        }
                        dfsFdskPteChs2Linear( pi, field);
                     }
                  }
                  txwDlgUl2Field( pi->hTec, pi->uTec, pi->vTec,  6, dfsa->hexCHS);
                  break;

               case DFSD_TEH:                   // Table end   Head
                  if ((mp1) || ((pi->uTeh == 0) && pteSmart && pi->uTyp))
                  {
                     maxvalue = (pteSmart) ? dfstGeoHeads(DFSTORE) -1 : 254;
                     if (pi->uTeh > maxvalue)
                     {
                        pi->uTeh  = maxvalue;
                     }
                     pi->uCeh = pi->uTeh;
                     txwDlgUl2Field( pi->hCeh, pi->uCeh, pi->vCeh,  3, dfsa->hexCHS);

                     if ((mp2 == field) && pteSmart && pi->uTyp) // smart update allowed
                     {
                        dfsFdskPteChs2Linear( pi, field);
                     }
                  }
                  txwDlgUl2Field( pi->hTeh, pi->uTeh, pi->vTeh,  3, dfsa->hexCHS);
                  break;

               case DFSD_TES:                   // Table end   Sector
                  if ((mp1) || ((pi->uTes == 0) && pteSmart && pi->uTyp))
                  {
                     maxvalue = (pteSmart) ? dfstGeoSectors(DFSTORE) : 63;
                     if ( (pi->uTes > maxvalue) ||
                         ((pi->uTes == 0) && pteSmart && pi->uTyp))
                     {
                        pi->uTes  = maxvalue;
                     }
                     pi->uCes = pi->uTes;
                     txwDlgUl2Field( pi->hCes, pi->uCes, pi->vCes,  3, dfsa->hexCHS);

                     if ((mp2 == field) && pteSmart && pi->uTyp) // smart update allowed
                     {
                        dfsFdskPteChs2Linear( pi, field);
                     }
                  }
                  txwDlgUl2Field( pi->hTes, pi->uTes, pi->vTes,  3, dfsa->hexCHS);
                  break;

               default:
                  break;
            }
            break;

         default:
            rc = txwDefWindowProc( hwnd, msg, mp1, mp2);
            break;
      }
   }
   else
   {
      rc = TX_INVALID_HANDLE;
   }
   RETURN( rc);
}                                               // end 'dfsFdskPteditWinProc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Update linear Ptable fields from current CHS values
/*****************************************************************************/
static void dfsFdskPteChs2Linear
(
   PTEDIT_ENTRY       *pi,                      // IN    PTE entry info
   ULONG               field                    // IN    originating field
)
{
   ULONG               snB;                     // begin SN
   ULONG               snE;                     // end   SN

   ENTER();

   snB = dfstChs2Psn( DFSTORE, pi->uCbc, pi->uCbh, pi->uCbs);
   snE = dfstChs2Psn( DFSTORE, pi->uCec, pi->uCeh, pi->uCes);

   TRACES(( "field: %3hu  snB: %8.8lx  snE: %8.8lx\n", field, snB, snE));

   if (snB != pi->lAbs)                         // start position change
   {
      pi->lAbs = snB;
      txwPostMsg( pi->hAbs, DFSM_SHOW_FIELD, TRUE, field);
   }
   if (snE >= snB)                              // positive size-value ?
   {
      ULONG            sec = snE - snB +1;

      if (sec != pi->lSec)
      {
         pi->lSec = sec;
         txwPostMsg( pi->hSec, DFSM_SHOW_FIELD, TRUE, field);
      }
   }
   VRETURN ();
}                                               // end 'dfsFdskPteChs2Linear'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Update current CHS values from linear Ptable fields
/*****************************************************************************/
static void dfsFdskPteLinear2Chs
(
   PTEDIT_ENTRY       *pi,                      // IN    PTE entry info
   ULONG               field                    // IN    originating field
)
{
   ENTER();

   dfstPsn2Chs( DFSTORE, pi->lAbs, &pi->uTbc, &pi->uTbh, &pi->uTbs);
   txwPostMsg( pi->hTbc, DFSM_SHOW_FIELD, TRUE, field);
   txwPostMsg( pi->hTbh, DFSM_SHOW_FIELD, TRUE, field);
   txwPostMsg( pi->hTbs, DFSM_SHOW_FIELD, TRUE, field);

   if (pi->lSec != 0)
   {
      dfstPsn2Chs( DFSTORE, pi->lAbs + pi->lSec -1, &pi->uTec, &pi->uTeh, &pi->uTes);
      txwPostMsg( pi->hTec, DFSM_SHOW_FIELD, TRUE, field);
      txwPostMsg( pi->hTeh, DFSM_SHOW_FIELD, TRUE, field);
      txwPostMsg( pi->hTes, DFSM_SHOW_FIELD, TRUE, field);
   }
   VRETURN ();
}                                               // end 'dfsFdskPteLinear2Chs'
/*---------------------------------------------------------------------------*/



/*****************************************************************************/
// Dialog window procedure, for the FDISK NEXT/PREV type Dialogs
// Shared between LvmSet and PTEdit, keep WID's unique!
/*****************************************************************************/
static ULONG dfsFdskNpDlgWinProc                // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   ULONG               mp1,                     // IN    msg param 1
   ULONG               mp2                      // IN    msg param 2
)
{
   ULONG               rc    = NO_ERROR;

   ENTER();
   TRCMSG( hwnd, msg, mp1, mp2);
   if (hwnd != 0)
   {
      PTEDIT_INFO  *inf = (PTEDIT_INFO *) txwQueryWindowPtr( hwnd, TXQWP_USER);

      switch (msg)
      {
         case TXWM_COMMAND:
            if ((mp1 == DFSD_WID_PDEC) ||
                (mp1 == DFSD_WID_PCHS) ||       // non-auto decimal checkboxes
                (DFSD_PT_BSE(mp1) == DFSD_HID)) // or the HIDDEN checkbox
            {
               USHORT        wid    = (USHORT) mp1;
               TXHANDLE      ctrl   = txwWindowFromID( hwnd, wid);
               int           entry  = DFSD_PT_ENT( wid);
               PTEDIT_ENTRY *pi = &(inf->e[entry]);
               int           i;

               TRACES(( "dlg: %8.8lx with attached PTINFO: %8.8lx\n", hwnd, inf));
               TRACES(( "wid: %3hu entry: %hu\n", wid, entry));

               if (wid == DFSD_WID_PDEC)        // decimal sector/size
               {
                  for (i = 0; i < 4; i++)
                  {
                     pi = &(inf->e[i]);

                     txwDlgField2Ul( &(pi->lRel), pi->vRel, dfsa->hexSNS);
                     txwDlgField2Ul( &(pi->lSec), pi->vSec, dfsa->hexSNS);
                     txwDlgField2Ul( &(pi->lAbs), pi->vAbs, dfsa->hexSNS);
                  }
                  dfsa->hexSNS = !(dfsa->hexSNS);
                  txwInvalidateWindow( ctrl, FALSE, FALSE);

                  for (i = 0; i < 4; i++)
                  {
                     pi = &(inf->e[i]);
                     if ((pi->uTyp == 0) && (pi->lSec == 0))
                     {
                        continue;               // skip unused fields
                     }
                     txwDlgUl2Field( pi->hRel, pi->lRel, pi->vRel, 10, dfsa->hexSNS);
                     txwDlgUl2Field( pi->hSec, pi->lSec, pi->vSec, 10, dfsa->hexSNS);
                     txwDlgUl2Field( pi->hAbs, pi->lAbs, pi->vAbs, 10, dfsa->hexSNS);
                  }
               }
               else if (wid == DFSD_WID_PCHS)   // decimal CHS values
               {
                  for (i = 0; i < 4; i++)
                  {
                     pi = &(inf->e[i]);

                     txwDlgField2Ul( &(pi->uTbc), pi->vTbc, dfsa->hexCHS);
                     txwDlgField2Ul( &(pi->uTbh), pi->vTbh, dfsa->hexCHS);
                     txwDlgField2Ul( &(pi->uTbs), pi->vTbs, dfsa->hexCHS);
                     txwDlgField2Ul( &(pi->uTec), pi->vTec, dfsa->hexCHS);
                     txwDlgField2Ul( &(pi->uTeh), pi->vTeh, dfsa->hexCHS);
                     txwDlgField2Ul( &(pi->uTes), pi->vTes, dfsa->hexCHS);
                     txwDlgField2Ul( &(pi->uCbc), pi->vCbc, dfsa->hexCHS);
                     txwDlgField2Ul( &(pi->uCbh), pi->vCbh, dfsa->hexCHS);
                     txwDlgField2Ul( &(pi->uCbs), pi->vCbs, dfsa->hexCHS);
                     txwDlgField2Ul( &(pi->uCec), pi->vCec, dfsa->hexCHS);
                     txwDlgField2Ul( &(pi->uCeh), pi->vCeh, dfsa->hexCHS);
                     txwDlgField2Ul( &(pi->uCes), pi->vCes, dfsa->hexCHS);
                  }
                  dfsa->hexCHS = !(dfsa->hexCHS);
                  txwInvalidateWindow( ctrl, FALSE, FALSE);

                  for (i = 0; i < 4; i++)
                  {
                     pi = &(inf->e[i]);
                     if ((pi->uTyp == 0) && (pi->lSec == 0))
                     {
                        continue;               // skip unused fields
                     }
                     txwDlgUl2Field( pi->hTbc, pi->uTbc, pi->vTbc,  6, dfsa->hexCHS);
                     txwDlgUl2Field( pi->hTbh, pi->uTbh, pi->vTbh,  3, dfsa->hexCHS);
                     txwDlgUl2Field( pi->hTbs, pi->uTbs, pi->vTbs,  3, dfsa->hexCHS);
                     txwDlgUl2Field( pi->hTec, pi->uTec, pi->vTec,  6, dfsa->hexCHS);
                     txwDlgUl2Field( pi->hTeh, pi->uTeh, pi->vTeh,  3, dfsa->hexCHS);
                     txwDlgUl2Field( pi->hTes, pi->uTes, pi->vTes,  3, dfsa->hexCHS);
                     txwDlgUl2Field( pi->hCbc, pi->uCbc, pi->vCbc,  6, dfsa->hexCHS);
                     txwDlgUl2Field( pi->hCbh, pi->uCbh, pi->vCbh,  3, dfsa->hexCHS);
                     txwDlgUl2Field( pi->hCbs, pi->uCbs, pi->vCbs,  3, dfsa->hexCHS);
                     txwDlgUl2Field( pi->hCec, pi->uCec, pi->vCec,  6, dfsa->hexCHS);
                     txwDlgUl2Field( pi->hCeh, pi->uCeh, pi->vCeh,  3, dfsa->hexCHS);
                     txwDlgUl2Field( pi->hCes, pi->uCes, pi->vCes,  3, dfsa->hexCHS);
                  }
               }
               else                             // must be the HIDDEN checkbox ...
               {
                  if (dfsPartTypeHidable( pi->uTyp))
                  {
                     pi->vHid = !(pi->vHid);    // toggle hidden status
                     txwInvalidateWindow( ctrl, FALSE, FALSE);
                     if (pi->vHid)
                     {
                        pi->uTyp |=  DFS_P_PHIDDEN;
                     }
                     else
                     {
                        pi->uTyp &= ~DFS_P_PHIDDEN;
                     }
                     txwPostMsg( pi->hTyp, DFSM_SHOW_FIELD, TRUE, DFSD_PT_FLD(wid));
                  }
                  else                          // type not hidable
                  {
                     TxMessage( TRUE, DFSH_PT + DFSD_HID,
                        "You can not hide a type %02.2hX partition!\n\n"
                        "Types 01-04 and 06-0E are hidable.", pi->uTyp);
                  }
               }
            }
            else if (mp1 == DFSD_WID_BSRT)      // SORT-S
            {
               PTEDIT_ENTRY *table = &(inf->e[0]);

               qsort( table, 4, sizeof(PTEDIT_ENTRY), dfsComparePteEntry);
               TxPrint("PTEdit %sBR at 0x%8.8lx, sorted on start-sector\n",
                       (inf->tableSn == 0) ? "M" : "E", inf->tableSn);

               txwInvalidateWindow( hwnd, FALSE, TRUE); // dlg incl controls
            }
            else if (mp1 == DFSD_WID_BCUS)      // SORT-#
            {
               PTEDIT_ENTRY *table = &(inf->e[0]);

               qsort( table, 4, sizeof(PTEDIT_ENTRY), dfsComparePteCustom);
                TxPrint("PTEdit %sBR at 0x%8.8lx, sorted on custom sequence#\n",
                       (inf->tableSn == 0) ? "M" : "E", inf->tableSn);

               txwInvalidateWindow( hwnd, FALSE, TRUE); // dlg incl controls
            }
            else                                // others ...
            {
               rc = txwDefDlgProc( hwnd, msg, mp1, mp2);
            }
            break;

         case TXWM_CHAR:
            switch (mp2)
            {
               case TXk_F3:
                  txwPostMsg( hwnd, TXWM_CLOSE, 0, 0);
                  break;

               case TXk_F4:
                  txwDismissDlg( hwnd, DFSD_NONE); // save, end dlg
                  break;

               case TXk_PGDN:
                  txwDismissDlg( hwnd, DFSD_NEXT); // save, to next table
                  break;

               case TXk_PGUP:
                  txwDismissDlg( hwnd, DFSD_PREV); // save, to prev table
                  break;

               case DFSD_EDIT:                  // allow several Fkeys for
               case DFSD_REQ5:                  // editing, higher level
               case DFSD_REQ6:                  // may edit at different
               case DFSD_REQ7:                  // sectors for each one
               case DFSD_REQ8:
                  txwDismissDlg( hwnd, mp2);    // save, start editor
                  break;

               default:
                  rc = txwDefDlgProc( hwnd, msg, mp1, mp2);
                  break;
            }
            break;

         default:
            rc = txwDefDlgProc( hwnd, msg, mp1, mp2);
            break;
      }
   }
   else
   {
      rc = TX_INVALID_HANDLE;
   }
   RETURN( rc);
}                                               // end 'dfsFdskNpDlgWinProc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Compare PTEDIT_ENTRY elements for qsort, sort on start-sector except 0
/*****************************************************************************/
int dfsComparePteEntry
(
   const void         *one,
   const void         *two
)
{
   PTEDIT_ENTRY       *p1  = (PTEDIT_ENTRY *) one;
   PTEDIT_ENTRY       *p2  = (PTEDIT_ENTRY *) two;
   ULONG               u1  = p1->lAbs -1;
   ULONG               u2  = p2->lAbs -1;
   int                 res = 0;

   ENTER();

   TRACES(( "u1: %8.8lx u2: %8.8lx\n", u1, u2));

   if (u1 != u2)
   {
      res = (int) ((u1 < u2) ? -1 : +1);
   }
   RETURN(res);
}                                               // end 'dfsComparePteEntry'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Compare PTEDIT_ENTRY elements for qsort, sort on custom-value
/*****************************************************************************/
int dfsComparePteCustom
(
   const void         *one,
   const void         *two
)
{
   PTEDIT_ENTRY       *p1  = (PTEDIT_ENTRY *) one;
   PTEDIT_ENTRY       *p2  = (PTEDIT_ENTRY *) two;
   int                 res = 0;

   ENTER();

   res = (int) (p1->vEnr[0] - p2->vEnr[0]);
   RETURN(res);
}                                               // end 'dfsComparePteCustom'
/*---------------------------------------------------------------------------*/

#endif                                          // USEWINDOWING
