//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// FAT Dialog related functions
//
// Author: J. van Wijk
//
// JvW  26-12-2005 Initial version
//

#include <txlib.h>                              // TX library interface

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsupart.h>                           // FS partition utilities
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfswin.h>                             // Window & menu definition
#include <dfstdlg.h>                            // generic dialogs definitions
#include <dfsffat.h>                            // FAT formatting and defs
#include <dfsdfat.h>                            // generic dialogs interface

#define DFSOS_BASE 200
#define DFSOSVALUE(os) ((os) - DFSOS_BASE)
#define DFSosItem(name,ltr,i,t,d) TXSitem(name, DFSOS_BASE + (ltr),TXWH_USE_OWNER_HELP,0,i,t,d)

//- SELIST with the FAT bootsector operating system selections
DFSosItem(fatI,'I',10,"IBM  or  PC-DOS"     ,"IBM or PC-DOS, using IBMBIO.COM + IBMDOS.COM  boot files");
DFSosItem(fatM,'M',10,"Classic  MS-DOS"     ,"MS-DOS from Microsoft, using IO.SYS/MSDOS.SYS boot files");
DFSosItem(fatO,'O', 1,"OS/2 or  eComStation","OS/2 style, using OS2KRNL, OS2LDR and OS2BOOT boot files");
DFSosItem(fatV,'V', 1,"Veit's   OS2LDR.SEK" ,"OS/2, boot to OS2LDR for bootable USB (Veit Kannegieser)");
DFSosItem(fatN,'N',10,"Windows  NT/2000/XP" ,"Windows-NT style, using NTLDR boot file  (on FAT16 only)");
DFSosItem(fatW,'W', 1,"Windows  95/98 FAT16","Windows-9x style, using IO.SYS/MSDOS.SYS (on FAT16 only)");
DFSosItem(fat9,'9',10,"Windows  98/ME FAT32","Windows-9x style, using WINBOOT.SYS      (on FAT32 only)");
DFSosItem(fat2,'2',12,"Copy FAT32 Spares"   ,"Copy from spare-bootsector at sector 6/7 (on FAT32 only)");

TXSitem(mmsp,0,0,TXSF_DISABLED | TXSF_SEPARATOR, 0,"","");

static TXS_ITEM *fatb16[] = {&fatI, &fatM, &fatO, &fatV, &fatN, &fatW, &mmsp, &fat9, &fat2};
static TXS_ITEM *fatb32[] = {&fat9};

TXSlist(fatBootCodeF16,6,6,fatb16);             // format, FAT16 types
TXSlist(fatBootCodeF32,1,1,fatb32);             // format, FAT32
TXSlist(fatFixBootCode,9,9,fatb16);             // fixboot, any type, incl spare


// Set radio buttons and entryfield visibility and select fat16/32 selist
static void dfsFatFmtSetControls
(
   FAT_FMT_INFO       *ffi                      // IN    FAT format information
);


// Dialog window procedure, for the FAT FORMAT dialog, set dependant fields
static ULONG dfsFatFmtWinProc                   // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    msg param 1
   TXWMPARAM           mp2                      // IN    msg param 2
);


/*========================== FORMAT =============================================================*/
static TXTM        fort1  = "";                 // object/store description

static BOOL        forr1  = FALSE;              // 512 bytes
static BOOL        forr2  = FALSE;              //   1 kiB
static BOOL        forr3  = FALSE;              //   2 kiB
static BOOL        forr4  = FALSE;              //   4 kiB
static BOOL        forr5  = FALSE;              //   8 kiB
static BOOL        forr6  = FALSE;              //  16 kiB
static BOOL        forr7  = FALSE;              //  32 kiB

static BOOL        forf1  = FALSE;              // FAT12
static BOOL        forf2  = FALSE;              // FAT16
static BOOL        forf3  = FALSE;              // FAT32

static TXTS        fore1  = "";                 // Volume Label
static TXTS        fore2  = "";                 // SerialNumber
static TXTS        fore3  = "";                 // Root entries
static TXTS        fore4  = "";                 // Fat-1 offset

static BOOL        forc1  = FALSE;              // LONG format

static BOOL        forOK  = FALSE;              // OK button, dummy variable
static BOOL        forCA  = FALSE;              // Cancel button dummy
/*
Cluster size       Use boot-code based on: [Windows 9x          ]
( ) 512 bytes
( )   1 KiB        ( ) 12-bit FAT
( )   2 KiB        ( ) 16-bit FAT                    Root entries
()   4 KiB        () 32-bit FAT (FAT32)           [512         ]
( )   8 KiB
( )  16 KiB        Volume Label     SerialNumber     Offset FAT-1
( )  32 KiB       [            ]   [            ]   [1           ]

[ ] Use LONG format, initialize all clusters to ZERO contents
*/
#define   DFSDGFFMTWIDGETS 22
#define   DFSDGFFMTLST     4                    // BootCode selist
#define   DFSDGFFMTC01     5                    // Cluster-size .5 KiB
#define   DFSDGFFMTC02     6                    // Cluster-size  1 KiB
#define   DFSDGFFMTC04     7                    // Cluster-size  2 KiB
#define   DFSDGFFMTC08     8                    // Cluster-size  4 KiB
#define   DFSDGFFMTC16     9                    // Cluster-size  8 KiB
#define   DFSDGFFMTC32    10                    // Cluster-size 16 KiB
#define   DFSDGFFMTC64    11                    // Cluster-size 32 KiB
#define   DFSDGFFMTF12    12                    // FAT-12
#define   DFSDGFFMTF16    13                    // FAT-16
#define   DFSDGFFMTF32    14                    // FAT-32
#define   DFSDGFFMTRSZ    15                    // Root-size
#define   DFSDGFFMTVOL    16                    // Volume label
#define   DFSDGFFMTSER    17                    // Serial number
#define   DFSDGFFMTOFF    18                    // Fat-1 offset

static TXWIDGET  dfsDgFfmtWidgets[DFSDGFFMTWIDGETS] = // order determines TAB-order!
{
   {0,  0, 1, 15, 0, 0, 0, TXWS_OUTPUT,  0, TXStdStline( "FORMAT object:")},
   {0, 17, 1, 50, 0, 0, 0, TXWS_OUTPUT,  0, TXStdStline( fort1)},

   {2,  0, 1, 14, 0, 0, 0, TXWS_OUTPUT,  0, TXStdStline( "Cluster size")},
   {2, 20, 1, 24, 0, 0, 0, TXWS_OUTPUT,  0, TXStdStline( "Use boot-code based on:")},
   {2, 44, 1, 23, 0, 0, 0, DFSDSLISTBOX, 0, TXStdLstBox( NULL, 0, NULL)},

   {4,  0, 1, 14, 0, 1, 0, TXWS_AUTORAD, 0, TXStdButton( &forr1, "512 bytes")},
   {5,  0, 1, 14, 0, 1, 0, TXWS_AUTORAD, 0, TXStdButton( &forr2, "  1 KiB  ")},
   {6,  0, 1, 14, 0, 1, 0, TXWS_AUTORAD, 0, TXStdButton( &forr3, "  2 KiB  ")},
   {7,  0, 1, 14, 0, 1, 0, TXWS_AUTORAD, 0, TXStdButton( &forr4, "  4 KiB  ")},
   {8,  0, 1, 14, 0, 1, 0, TXWS_AUTORAD, 0, TXStdButton( &forr5, "  8 KiB  ")},
   {9,  0, 1, 14, 0, 1, 0, TXWS_AUTORAD, 0, TXStdButton( &forr6, " 16 KiB  ")},
   {10, 0, 1, 14, 0, 1, 0, TXWS_AUTORAD, 0, TXStdButton( &forr7, " 32 KiB  ")},

   {4, 20, 1, 18, 0, 5, 0, TXWS_AUTORAD, 0, TXStdButton( &forf1, "12-bit FAT")},
   {5, 20, 1, 18, 0, 5, 0, TXWS_AUTORAD, 0, TXStdButton( &forf2, "16-bit FAT")},
   {6, 20, 1, 18, 0, 5, 0, TXWS_AUTORAD, 0, TXStdButton( &forf3, "32-bit FAT")},

   {5, 53, 2, 14, 0, 0, 0, TXWS_ENTRYBT, 0, TXStdEntryf( fore3, 8,  "Root entries")},

   {9, 19, 2, 14, 0, 0, 0, TXWS_ENTRYBT, 0, TXStdEntryf( fore1, 11, "Volume Label")},
   {9, 36, 2, 14, 0, 0, 0, TXWS_ENTRYBT, 0, TXStdEntryf( fore2, 8,  "SerialNumber")},
   {9, 53, 2, 14, 0, 0, 0, TXWS_ENTRYBT, 0, TXStdEntryf( fore4, 2,  "Fat-1 offset")},

   {12, 0, 1, 66, 0, 3, 0, TXWS_AUTOCHK, 0, TXStdButton( &forc1,
                          "Use LONG format, initialize all clusters to ZERO contents")},

   {14,19, 3,  8, 0, 0, TXDID_OK,     DFSDMPBUTTON, 0, TXStdButton( &forOK, " OK ")},
   {14,41, 3,  8, 0, 0, TXDID_CANCEL, DFSDMPBUTTON, 0, TXStdButton( &forCA, "Cancel")}
};

#define WID_FMTFAT           810
#define DFSFATFMT_RECREATE  (TXDID_MAX -1)      // recreate the dialog

static TXGW_DATA dfsDgFfmtDlg =
{
   DFSDGFFMTWIDGETS,                            // number of widgets
   DFSC_FMTFAT,                                 // help, widget overrules
   WID_FMTFAT,                                  // base window ID
   dfsFatFmtWinProc,                            // widget window procedure
   NULL,                                        // persistent position TXRECT
   dfsDgFfmtWidgets                             // array of widgets
};

/*************************************************************************************************/
// Present FAT-FORMAT options dialog and execute resulting format command
/*************************************************************************************************/
ULONG dfsFatFormatDialog
(
   FAT_FMT_INFO       *ffi                      // INOUT FAT format information
)
{
   ULONG               rc = NO_ERROR;           // function return
   TXLN                command;
   TXTM                escaped;                 // String with escaped ' characters
   TXTM                option;
   USHORT              focus = DFSDGFFMTVOL;    // field to get focus
   TXSELIST           *list;

   ENTER();

   dfsBEGINWORK();                              // signal work starting

   dfsDriveDescription( fort1);                 // original object description
                                                // ignoring 'fs fat' change

   dfsdShowWg( dfsDgFfmtWidgets[ DFSDGFFMTF12], (ffi->allowedBits & 12));
   dfsdShowWg( dfsDgFfmtWidgets[ DFSDGFFMTF16], (ffi->allowedBits & 16));
   dfsdShowWg( dfsDgFfmtWidgets[ DFSDGFFMTF32], (ffi->allowedBits & 32));

   forc1 = ffi->longFormat;
   forf1 = ffi->desiredBits & 12;
   forf2 = ffi->desiredBits & 16;
   forf3 = ffi->desiredBits & 32;

   dfsFatFmtSetControls(    ffi);               // initialize dialog controls
   strcpy(  fore1,          ffi->label);
   sprintf( fore2, "%8.8x", ffi->serial);
   sprintf( fore3, "%hu",   ffi->desiredRsize);
   sprintf( fore4, "%2.2x", ffi->reservedSect);

   TxSelCharSelect( dfsDgFfmtWidgets[ DFSDGFFMTLST].lb.list, ffi->os[0]);

   while ((rc = txwWidgetDialog( TXHWND_DESKTOP, TXHWND_DESKTOP,
           ffi, " Specify FAT/FAT32 formatting options ",
           TXWD_MOVEABLE | TXWD_HCENTER | TXWD_VCENTER, focus, &dfsDgFfmtDlg))
        != TXDID_CANCEL)
   {
      if (rc != DFSFATFMT_RECREATE)
      {
         strcpy(    command, "format");
         if (forc1)
         {
            strcat( command, " -l");            // long format
         }
         if (dfsa->verbosity >= TXAO_VERBOSE)
         {
            strcat( command, " -O:v");          // propagate verbose
         }
         strcat(    command, " -c:");
         if (forr1) strcat( command,  "1");
         if (forr2) strcat( command,  "2");
         if (forr3) strcat( command,  "4");
         if (forr4) strcat( command,  "8");
         if (forr5) strcat( command, "16");
         if (forr6) strcat( command, "32");
         if (forr7) strcat( command, "64");

         TxStrip( fore1, fore1, ' ', ' ');
         if (strlen(fore1))                     // Label, string
         {
            TxRepStr( fore1, '\'', "\\'", escaped, TXMAXTM); // escape single quotes
            sprintf( option, " -v:'%s'",  escaped);
            strcat(  command, option);
         }
         TxStrip( fore2, fore2, ' ', ' ');
         if (isxdigit( fore2[0]))               // hex serial specified
         {
            strcat(  command, " -s:0x");
            strcat(  command, fore2);
         }
         if (!forf3)                            // not for FAT32
         {
            TxStrip( fore3, fore3, ' ', ' ');
            if (isdigit( fore3[0]))             // RootSize value specified
            {
               strcat(  command, " -r:");
               strcat(  command, fore3);
            }
         }
         TxStrip( fore4, fore4, ' ', ' ');
         if (isxdigit( fore4[0]))               // hex serial specified
         {
            strcat(  command, " -a:0x");
            strcat(  command, fore4);
         }

         list = dfsDgFfmtWidgets[ DFSDGFFMTLST].lb.list;
         sprintf( option, " -o:'%c'", (char) list->items[ list->selected]->value - DFSOS_BASE);
         strcat(  command, option);

         if ( forf1)         strcat( command, " -f:12");
         if ( forf2)         strcat( command, " -f:16");
         if ( forf3)         strcat( command, " -f:32");

         dfsExecEnd( command);
         break;
      }
   }
   dfsENDWORK();                                // signal work done
   RETURN (rc);
}                                               // end 'dfsFatFormatDialog'
/*-----------------------------------------------------------------------------------------------*/


/*****************************************************************************/
// Set radio buttons and entryfield visibility and select fat16/32 selist
/*****************************************************************************/
static void dfsFatFmtSetControls
(
   FAT_FMT_INFO       *ffi                      // IN    FAT format information
)
{
   USHORT              minsize;                 // minimum cluster size
   USHORT              desired = ffi->desiredCsize;

   ENTER();

   if      (forf1)
   {
      minsize = ffi->minimumCs_12;
   }
   else if (forf2)
   {
      minsize = ffi->minimumCs_16;
   }
   else
   {
      minsize = ffi->minimumCs_32;
      if (ffi->defaultCsize < minsize)
      {
         ffi->defaultCsize = minsize;           // clip to real minimum (4096 BPS)
      }
   }

   dfsdShowWg( dfsDgFfmtWidgets[ DFSDGFFMTRSZ], (forf3 == FALSE));

   dfsdShowWg( dfsDgFfmtWidgets[ DFSDGFFMTC01], (minsize <=  1));
   dfsdShowWg( dfsDgFfmtWidgets[ DFSDGFFMTC02], (minsize <=  2));
   dfsdShowWg( dfsDgFfmtWidgets[ DFSDGFFMTC04], (minsize <=  4));
   dfsdShowWg( dfsDgFfmtWidgets[ DFSDGFFMTC08], (minsize <=  8));
   dfsdShowWg( dfsDgFfmtWidgets[ DFSDGFFMTC16], (minsize <= 16));
   dfsdShowWg( dfsDgFfmtWidgets[ DFSDGFFMTC32], (minsize <= 32));
   dfsdShowWg( dfsDgFfmtWidgets[ DFSDGFFMTC64], (minsize <= 64));

   if (desired == 0)
   {
      desired = (forf3) ? ffi->defaultCsize : minsize;
   }
   else if (desired < minsize)
   {
      desired = minsize;
   }
   forr1 = forr2 = forr3 = forr4 = forr5 = forr6 = forr7 = FALSE;

   if      (desired <=  1) forr1 = TRUE;
   else if (desired <=  2) forr2 = TRUE;
   else if (desired <=  4) forr3 = TRUE;
   else if (desired <=  8) forr4 = TRUE;
   else if (desired <= 16) forr5 = TRUE;
   else if (desired <= 32) forr6 = TRUE;
   else                    forr7 = TRUE;

   if (forf3)                                   // FAT32 selected
   {
      strcpy( fore4, "20");
      dfsDgFfmtWidgets[ DFSDGFFMTLST].lb.list = &fatBootCodeF32;
   }
   else
   {
      strcpy( fore4, "01");
      dfsDgFfmtWidgets[ DFSDGFFMTLST].lb.list = &fatBootCodeF16;
   }
   VRETURN ();
}                                               // end 'dfsFatFmtSetControls'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Dialog window procedure, for the FAT FORMAT dialog, set dependant fields
/*****************************************************************************/
static ULONG dfsFatFmtWinProc                   // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    msg param 1
   TXWMPARAM           mp2                      // IN    msg param 2
)
{
   ULONG               rc = NO_ERROR;

   ENTER();
   TRCMSG( hwnd, msg, mp1, mp2);

   if (hwnd != 0)
   {
      FAT_FMT_INFO *fi = (FAT_FMT_INFO *) txwQueryWindowPtr( hwnd, TXQWP_USER);

      switch (msg)
      {
         case TXWM_COMMAND:
            switch ((ULONG) mp2)                // command source
            {
               case TXCMDSRC_RADIOBUTTON:
                  switch ((ULONG) mp1)          // Window-ID
                  {
                     case WID_FMTFAT + DFSDGFFMTF12:
                     case WID_FMTFAT + DFSDGFFMTF16:
                     case WID_FMTFAT + DFSDGFFMTF32:
                        dfsFatFmtSetControls( fi); // update other fields
                        txwDismissDlg( hwnd, DFSFATFMT_RECREATE);
                        //- txwInvalidateWindow( hwnd, FALSE, TRUE); // whole dlg
                        break;

                     default:
                        break;
                  }
                  break;

               default:
                  rc = txwDefDlgProc( hwnd, msg, mp1, mp2);
                  break;
            }
            break;

         default:
            rc = txwDefDlgProc( hwnd, msg, mp1, mp2);
            break;
      }
   }
   else
   {
      rc = TX_INVALID_HANDLE;
   }
   RETURN( rc);
}                                               // end 'dfsFatFmtWinProc'
/*---------------------------------------------------------------------------*/


/*************************************************************************************************/
// Present FAT-FIXBOOT selection list and execute resulting fixboot command
/*************************************************************************************************/
ULONG dfsFatFixBootDialog                       // return result
(
   char                fixType                  // IN   default fixboot type
)
{
   ULONG               rc = NO_ERROR;           // function return
   ULONG               listcode;
   TXSELIST           *list = &fatFixBootCode;
   TXRECT              where  = {25,6,0,0};     // fixed position
   TXLN                command;

   ENTER();
   dfsBEGINWORK();                              // signal work starting

   TxSelCharSelect( list, fixType);             // set list to default

   listcode = txwListBox( TXHWND_DESKTOP, TXHWND_DESKTOP, &where,
                        " Select BOOTCODE type ", "", 5405,  TXLB_MOVEABLE,
                          cSchemeColor,   cSchemeColor,  list);

   if (listcode != TXDID_CANCEL)
   {
      sprintf(    command, "fixboot %c",
                    (char) (list->items[ list->selected]->value - DFSOS_BASE));
      dfsExecEnd( command);
   }
   dfsENDWORK();                                // signal work done
   RETURN (rc);
}                                               // end 'dfsFatFixBootDialog'
/*-----------------------------------------------------------------------------------------------*/

