//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// FDISK PTE GPT Dialog and windowing functions
//
// Author: J. van Wijk
//
// JvW  30-10-2015 Split up, separate edit-LVM in own module
// JvW  21-09-2015 Split up, separate CR-MBR and CR-GPT modules
// JvW  18-11-2002 Initial version, derived from TXWSTDLG and DFSCFDSK
//


#include <txlib.h>                              // TX library interface

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfsupart.h>                           // FDISK partition functions
#include <dfsafdsk.h>                           // FDISK display & analysis
#include <dfscfdsk.h>                           // FDISK callback and commands
#include <dfsdfgpt.h>                           // FDISK GPT dialog functions
#include <dfsufdsk.h>                           // FDISK utility functions
#include <dfsufgpt.h>                           // FDISK GPT utility functions
#include <dfsdegpt.h>                           // FDISK PTE GPT dialog functions
#include <dfsdgen.h>                            // Generic Dialog functions
#include <dfswin.h>                             // window and help definitions
#include <dfsver.h>                             // version and naming info

#if defined (USEWINDOWING)

// PTEDIT GPT dialog specific definitions

#define DFSH_EG             (DFSH_FDISK + 650)

#define DFSH_EG_FRAME       (DFSH_EG + 1)       // shown help-on-help (2nd)
#define DFSH_EG_DECIM       (DFSH_EG + 2)
#define DFSH_EG_EFIRST      (DFSH_EG + 3)
#define DFSH_EG_ELAST       (DFSH_EG + 4)
#define DFSH_EG_EBNAVI      (DFSH_EG + 5)       // navigational buttons
#define DFSH_EG_EBTMPG      (DFSH_EG + 6)
#define DFSH_EG_EPNAME      (DFSH_EG + 7)
#define DFSH_EG_EATVAL      (DFSH_EG + 8)
#define DFSH_EG_EBTATT      (DFSH_EG + 9)
#define DFSH_EG_EBTMAN      (DFSH_EG +15)
#define DFSH_EG_ESPINV      (DFSH_EG +16)


#define DFSD_PTEGPT_WC          73              // horizontal size in columns
#define DFSD_PTEGPT_WL          24              // vertical   size in lines

#define DFSD_PENR                3              // partition entry field
#define DFSD_PTYP                4              // type entry field
#define DFSD_DESC                9              // description field length

#define DFSD_WID_EDIALOG       850              // window id dialog frame
#define DFSD_WID_PDEC          852              // window id decimal sec/size
#define DFSD_WID_EGPTXT        854              // window id info text
#define DFSD_WID_EGPHDR        855              // window id table header
#define DFSD_WID_EGPDES        856              // window id info text
#define DFSD_WID_EFIRST        803              // entryfield   First sector
#define DFSD_WID_ELAST         804              // entryfield   Last  sector
#define DFSD_WID_EBTMPG        806              // push  button Manual Edit P-guid
#define DFSD_WID_EPNAME        807              // entry Partition Name
#define DFSD_WID_EATVAL        808              // entry Attribute Value
#define DFSD_WID_EBTATT        809              // push  button Specify Attr
#define DFSD_WID_ETSIZE        811              // size informational string
#define DFSD_WID_EPGUID        812              // partition GUID descr
#define DFSD_WID_ETGUID        813              // part type GUID descr
#define DFSD_WID_EPDESC        814              // part type GUID string
#define DFSD_WID_EBTMAN        815              // push  button Manual Edit T-guid
#define DFSD_WID_ESPINV        816              // spin-value type list


typedef struct ptedit_info                      // info needed in window-proc
{
   S_GPTENTRY         *gpte;                    // GPT entry contents
   USHORT              disknr;                  // disk number
   ULN64               eSect;                   // edit sector number
   TXWHANDLE           hFirst;                  // first sector of partition
   TXTS                vFirst;
   ULN64               lFirst;
   TXWHANDLE           hLast;                   // last  sector of partition
   TXTS                vLast;
   ULN64               lLast;
   TXWHANDLE           hSize;                   // partition size informational string
   TXTT                vSize;
   TXTT                vpnbuf;                  // entryfield GPT partition name
   TXTT                vavbuf;                  // entryfield GPT attribute value
   TXTT                vtgstr;                  // type GUID-string value
   TXTT                vpgstr;                  // part GUID-string value
   TXSELIST           *gptList;                 // SELIST with GPT types
} PTEGPT_INFO;                                  // end of struct "ptedit_info"

static char           *pteGptHelpText[] =
{
   "",
   " This dialog allows verification and modification of all the",
   " values in a GPT partition table entry, for a single partition",
   "",
   " A GPT partition entry has the following components:",
   "",
   "  - Type GUID, this is a unique 64-bit value that indicates the",
   "    intended use of the partition. Unique values are defined for",
   "    quite a few operating systems and/or filesystems.",
   "",
   "  - Part GUID, this is a unique 64-bit value that identifies this",
   "    specific partition. Operating systems use it to administer its",
   "    usage and mounting of the contents/filesystem. It is generated",
   "    when creating the partition, and should normally not be changed."
   "",
   "  - First PSN, a 64-bit sectornumber that is the offset from the",
   "    start of the disk to the start of the partition (bootsector).",
   "",
   "  - Last PSN, a 64-bit sectornumber that corresponds to the last",
   "    sector in the partition (partition size is difference + 1).",
   "",
   "  - Partition Name, upto 36 (Unicode) characters to add a human",
   "    readable name to the partition, in addition to the GUID.",
   "    It will be displayed in the various DFSee overviews and tables"
   "",
   "  - Attribute flags, 64 individual bits defining specific properties",
   "    for the partition. Some generic defined ones, some specific for",
   "    the actual type of the partition.",
   "",
TXHELPITEM(001,"PTEDIT GPT Dialog window")
   " The dialog is used to show the current values in the table, and",
   " any changes made to them while the dialog is up.",
   "",
   " When closing the dialog, the resulting changes will be displayed",
   " and a final confirmation to update the info is presented.",
   "",
   " The dialog ends with the following keys / actions:",
   "",
   "  - <ENTER> End dialog, if changes, ask confirmation and write back",
   "  - <PgDn>  Like ENTER, then display dialog for NEXT table-entry",
   "  - <PgUp>  Like ENTER, and then dialog for PREVIOUS table-entry",
   "  - <Esc>   End dialog, do not commit or write back any changes",
   "",
   "  - <F2>    HEX-edit the Partition Table Array at the current entry",
   "  - <F5>    HEX-edit First sector for partition, bootsector    (PBR)",
   "  - <F8>    HEX-edit Last  sector for partition, like a spare BR",
   "",
TXHELPITEM(002,"Use hexadecimal sector/size values")
   " For more help on the complete dialog, press <F1> again ...",
   "",
   " When checked, this will cause the 'first sector', 'last sector'",
   " and the partition-size as number of sectors to be displayed in",
   " hexadecimal format, as oposed to the decimal numbersystem.",
   "",
   " Note: The size in megabytes is always in decimal notation.",
   "",
   " The check-box is toggled using the <Space-bar> or the mouse",
   "",
   TXHELPITEM(003, "First sector")
   " For more help on the complete dialog, press <F1> again ...",
   "",
   " This is the exact starting position, or FIRST sector, of the",
   " defined partition on the disk. Together with the 'Last sector'",
   " it determines the location and the size of the partition.",
   "",
   " It can be shown and edited either as a hexadecimal, or as a"
   " decimal value, depending on the 'use hex ...' checkbox.",
   "",
   TXHELPITEM(004, "Last sector")
   " For more help on the complete dialog, press <F1> again ...",
   "",
   " This is the exact ending position, or LAST sector, of the",
   " defined partition on the disk. Together with 'First sector'",
   " it determines the location and the size of the partition.",
   "",
   " It can be shown and edited either as a hexadecimal, or as a"
   " decimal value, depending on the 'use hex ...' checkbox.",
   "",
TXHELPITEM(005,"Navigational buttons")
   " These buttons navigate through the array of partitions (PTA)",
   " allowing changes to each of the used table entries, and writing",
   " any modified entries back to disk, or cancel any updates",
   "",
   " [Prev]   Will write back changes to the primary/alternate PTA,",
   " <PgUp>   and go to the previous table entry (if there is one)",
   "",
   " [Next]   Will write back changes to the primary/alternate PTA,",
   " <PgDn>   and go to the next table entry (if there is one)",
   "",
   " [ OK ]   Will write back changes to the primary/alternate PTA,",
   " <Enter>  and then end the PTE edit session.",
   "",
   " [Cancel] Will end the PTE session without writing any changes.",
   "  <Esc>   for this entry",
   "",
   TXHELPITEM(006, "'Manual edit part GUID' button")
   " This push button allows you to manually specify a hexadecimal",
   " value for the partition GUID. You may need it to make the value",
   " unique after cloning or otherwise duplicating the partition.",
   "",
   TXHELPITEM(007, "Partion Name entryfield")
   " For more help on the complete dialog, press <F1> again ...",
   "",
   " This field allows specification of a descriptive NAME for the",
   " new partition. It is given as an ASCII string with a maximum",
   " length 36 characters (stored as UNICODE, in 72 bytes).",
   "",
   " At creation time, DFSee will generate a default name based",
   " on the currently selected partition type-GUID filled in.",
   " Make sure to make this value unique, at least per system!",
   "",
   TXHELPITEM(008, "Attribute flag value entryfield")
   " For more help on the complete dialog, press <F1> again ...",
   "",
   " This field allows specification of hexadecimal value that",
   " represent the 64 possible flag-bits for the new partition.",
   "",
   TXHELPITEM(009, "'Specify Attribute Flags' button")
   " For more help on the complete dialog, press <F1> again ...",
   "",
   " This will open a dialog where each of the DEFINED (for DFsee)",
   " flag-bits can be read out and specified individually.",
   "",
   TXHELPITEM(015, "'Manual edit type GUID' button")
   " This push button allows you to manually specify a hexadecimal",
   " value for the type of this partition. You may need it for (new)",
   " GPT partition types that are not in the DFSee standard list of",
   " partition types.",
   "",
   " The edit dialog will start with the last value selected from",
   " the list, so you can use it as a starting point.",
   "",
   TXHELPITEM(016, "Select GPT partition type")
   " This is the 'system-type' for the partition, an 128-bit unique",
   " value (partition type GUID) that represents the purpose of the",
   " partition, and relates to the filesystem or the operating system",
   " using this partition.",
   "",
   " The most used values can be selected from the list directly, and",
   " while navigating the list, a more verbose description of the type",
   " is shown at the status line at the bottom of the screen.",
   "",
   " The list entry has three parts separated by a vertical bar:",
   "",
   " - 4 hexadecimal digits 'shorthand' value, as used by Linux GDISK",
   "",
   " - A short description of the owner/manufacturer of the type",
   "",
   " - The DFSee short description of the partition type.",
   "",
   " Other (custom) values can be specified as hexadecimal values",
   " by selecting the last entry 'Manually set custom GUID value'",
   " or using the [Manual edit type GUID] button.",
   "",
   " The corresponding formatted GUID value is displayed in the status",
   " line at the bottom of the screen while navigating the list, and is",
   " displayed in an output field below the spinbox as well.",
   NULL
};


static char dfseg_bt_ok[]       = " OK ";
static char dfseg_bt_cancel[]   = "Cancel";
static char dfseg_bt_prev[]     = "Prev";
static char dfseg_bt_next[]     = "Next";

static char  dfseg_footer[] = "F1=Help F2/5/8=HEX F4=OK PgDn=Next PgUp=Prev  F12=Minimize Alt+F10=Maximize";


// Dialog window procedure, for the PTEDIT dialog fields
static ULONG dfsFdskPteGptWinProc               // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    msg param 1
   TXWMPARAM           mp2                      // IN    msg param 2
);

// Dialog window procedure, for the PTEDIT GPT NEXT/PREV type Dialogs
static ULONG dfsEditGptDlgWinProc               // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    msg param 1
   TXWMPARAM           mp2                      // IN    msg param 2
);

// Update descriptional size field from first/last PSN
static void dfsEditGptSetSize
(
   PTEGPT_INFO        *pti                      // IN    PTE entry info
);


/*****************************************************************************/
// Initialize help system for the PTEdit-GPT Dialog (accessible from startup)
/*****************************************************************************/
void dfsFdskPteGptInitHelp
(
   void
)
{
   ENTER();

   txwRegisterHelpText( DFSH_EG, "dfspteditgpt", "PTEdit GPT Dialog help", pteGptHelpText);

   VRETURN ();
}                                               // end 'dfsFdskPteGptInitHelp'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display PTEDIT dialog with default values and allow changes
/*****************************************************************************/
ULONG dfsFdskPteGptDialog                       // RET   dialog result
(
   FDSK_CB_INFO       *cbp,                     // IN    operation info
   S_GPTENTRY         *gpte                     // IN    GPT entry
)
{
   ULONG               rc = NO_ERROR;           // function return
   PTEGPT_INFO        *ptinfo   = TxAlloc( 1, sizeof(PTEGPT_INFO));
   static TXRECT       position = {0,0,0,0};    // size/position, initialized
                                                // once from desktop position
   ENTER();
   TRACES(("GPT entry ptr: %p\n", gpte));

   if ((ptinfo) && (txwIsWindow( TXHWND_DESKTOP)))
   {
      TXRECT           deskpos;                 // desktop size/position
      TXWHANDLE        cframe;                  // dialog frame
      TXWHANDLE        hHexSNS;                 // hexadecimal pos/size
      TXWHANDLE        cdescr;                  // partition description
      TXWHANDLE        cdname;                  // disk name string
      TXWHANDLE        centpn;                  // entryfield partition name
      TXWHANDLE        chex0x;                  // Text OUTPUT field Hex 0x
      TXWHANDLE        centav;                  // entryfield attribute value
      TXWHANDLE        cspinv;                  // spin-value type list
      TXWHANDLE        ctguid;                  // partition type GUID description
      TXWHANDLE        cpguid;                  // partition GUID description
      TXWHANDLE        cpdesc;                  // partition type GUID string
      TXWHANDLE        cbtMan;                  // push  button Manual edit T-guid
      TXWHANDLE        cbtMpg;                  // push  button Manual edit P-guid
      TXWHANDLE        cbtAtt;                  // push  button Specify Attr
      TXWHANDLE        hbtok;                   // OK button
      TXWHANDLE        hbtcan;                  // CANCEL button
      TXWHANDLE        hbtprev;                 // PREV   button
      TXWHANDLE        hbtnext;                 // NEXT   button
      TXWINDOW         window;                  // setup window data
      ULONG            style;                   // window style
      USHORT           phsize;                  // parent window width
      USHORT           pvsize;                  // parent window height
      short            ww;                      // dialog window width
      short            wc;                      // std column with
      TXLN             title;                   // Dynamic title text
      TXTM             pdesc;                   // partition description text
      static USHORT    desiredFocus;            // WID for desired focus field (last)

      TRACES(( "disk: %hu sn:%llx  cbp->string: '%s'\n",
                cbp->disknr, cbp->thisSn, cbp->string));

      strcpy( ptinfo->vpgstr, dfsFsUuidValueString( gpte->partGuid));
      strcpy( ptinfo->vtgstr, dfsFsUuidValueString( gpte->typeGuid));

      ptinfo->gptList = dfsGptTypeSelist( TRUE); // create type list, with MANUAL

      //- Set current selection to actual type-GUID
      ptinfo->gptList->selected = ptinfo->gptList->count -1; // when not found, use 'custom'
      TxSelDescriptionSelect( ptinfo->gptList,  ptinfo->vtgstr, strlen( ptinfo->vtgstr));

      sprintf( ptinfo->vavbuf, "%16.16llx", gpte->attrFlags);

      ptinfo->gpte     = gpte;
      ptinfo->disknr   = cbp->disknr;

      ptinfo->lFirst   = gpte->firstPSN;
      ptinfo->lLast    = gpte->lastPSN;

      txwDlgU642Field( ptinfo->hFirst, ptinfo->lFirst, ptinfo->vFirst, 12, dfsa->hexSNS);
      txwDlgU642Field( ptinfo->hLast,  ptinfo->lLast,  ptinfo->vLast,  12, dfsa->hexSNS);

      dfsEditGptSetSize( ptinfo);

      TxUnic2Ascii( gpte->partName, ptinfo->vpnbuf, GPT_PNAME_LEN);

      sprintf( title, " GPT Partition-table edit disk %hu, table entry %hu ", cbp->disknr, cbp->partnr);
      dfsGetPartDesc( dfsGetPartInfo( dfsDiskPsn2PartId( ptinfo->disknr, cbp->sn)), pdesc);

      txwQueryWindowPos( TXHWND_DESKTOP, FALSE, &deskpos);
      phsize = deskpos.right  - deskpos.left;
      pvsize = deskpos.bottom - deskpos.top;

      ww = min( DFSD_PTEGPT_WC,  (phsize - 5));
      wc = 32;

      if (position.right == 0)                  // not initialized yet ?
      {                                         // start with desktop position
         position = deskpos;
         if (position.left  + DFSD_PTEGPT_WC < phsize)
         {
            position.left   = (phsize - ww) /2;
            position.right  = ww;
         }
         if (DFSD_PTEGPT_WL + 4 < pvsize)
         {
            position.top += 2;                  // move dialog a bit down
         }
      }
      else
      {
         position.right  = ww;                  // fixed size of start Dlg
      }
      position.bottom = DFSD_PTEGPT_WL;

      TRECTA( "pos/size", (&position));

      if (cbp->cbOption4)
      {
         desiredFocus  = DFSD_WID_EPNAME;
         cbp->cbOption4 = FALSE;                // Initialized FOCUS field to default one
      }

      txwSetupWindowData(
         position.top, position.left, position.bottom, position.right,
         TXWS_DFRAME | TXWS_CAST_SHADOW | TXCS_CLOSE_BUTTON, // style
         DFSH_EG_FRAME,                         // help
         ' ', ' ',                              // window & border chars
         TXWSCHEME_COLORS,                      // std scheme colors
         title, dfseg_footer,
         &window);
      window.dlgFocusID = desiredFocus;         // first FOCUS
      window.st.buf     = NULL;                 // no artwork attached
      cframe = txwCreateWindow( TXHWND_DESKTOP, TXW_CANVAS, 0, 0, &window, NULL);
      txwSetWindowUShort( cframe, TXQWS_ID,   DFSD_WID_EDIALOG);
      txwSetWindowPtr(    cframe, TXQWP_DATA, &position); //- for exit position
      TRACES(( "Frame: %8.8lx with attached PTINFO: %8.8lx\n", cframe, ptinfo));

      //- add child windows to frame ...

      //- Disk name string
      style = TXWS_OUTPUT | TXWS_HCHILD2MOVE;
      txwSetupWindowData(
         1, 1, 1, 20,
         style, 0,                              // window frame style + help
         ' ', ' ', TXWSCHEME_COLORS, "", "",
         &window);
      window.sl.buf     = cbp->string;
      cdname = txwCreateWindow( cframe, TXW_STLINE, cframe, 0, &window, NULL);
      txwSetWindowUShort(       cdname, TXQWS_ID, DFSD_WID_EGPTXT);

      //- Partition description string
      style = TXWS_OUTPUT | TXWS_HCHILD2MOVE;
      txwSetupWindowData(
         1, 21, 1, DFSD_PTEGPT_WC -23,          // UL corner + vert/hor size
         style, 0,                              // window frame style + help
         ' ', ' ',                              // window/border chars
         TXWSCHEME_COLORS,                      // std scheme colors
         "", "",
         &window);
      window.sl.buf     = pdesc;
      cdescr = txwCreateWindow(   cframe, TXW_STLINE, cframe, 0, &window, dfsFdskPteGptWinProc);
      txwSetWindowUShort( cdescr, TXQWS_ID, DFSD_WID_EGPDES);


      style = TXWS_DSPINBT | TXLS_DROP_ENTER | TXWS_LEFTJUSTIFY | TXWS_HCHILD2SIZE | TXWS_HCHILD2MOVE;
      txwSetupWindowData(
         3, wc +2, 2, ww - wc -5,
         style, DFSH_EG_ESPINV,                 // style and help
         ' ', ' ',                              // window/border chars
         TXWSCHEME_COLORS,                      // std scheme colors
         "Select GPT partition-type", "",
         &window);
      window.lb.list = ptinfo->gptList;             // spin-value type list
      window.lb.cpos = ptinfo->gptList->selected - ptinfo->gptList->top;
      cspinv = txwCreateWindow(  cframe, TXW_LISTBOX, cframe, 0, &window, dfsFdskPteGptWinProc);
      txwSetWindowUShort( cspinv, TXQWS_ID, DFSD_WID_ESPINV);


      if (dfsa->expertui)
      {
         //- Button triggers an action, should NOT dismiss the dialog -> DLGE
         style = TXWS_PBUTTON | TXBS_DLGE_BUTTON | TXWS_HCHILD2MOVE | TXWS_VCHILD_MOVE;
         txwSetupWindowData(
            4, 1, 3, 27,
            style, DFSH_EG_EBTMAN,              // style + help
            ' ', ' ',                           // window/border chars
            TXWSCHEME_COLORS,                   // std scheme colors
            "", "",                             // title & footer text
            &window);
         window.bu.text    = "Manual Edit Type GUID >";
         cbtMan = txwCreateWindow(   cframe, TXW_BUTTON, cframe, 0, &window, NULL);
         txwSetWindowUShort( cbtMan, TXQWS_ID, DFSD_WID_EBTMAN);
         txwSetWindowUShort( cbtMan, TXQWS_GROUP, 1);
      }
      else
      {
         //- Partition type leading text string
         style = TXWS_OUTPUT | TXWS_HCHILD2MOVE;
         txwSetupWindowData(
            6, 2, 1, 28,                        // UL corner + vert/hor size
            style, 0,                           // window frame style + help
            ' ', ' ',                           // window/border chars
            TXWSCHEME_COLORS,                   // std scheme colors
            "", "",
            &window);
         window.sl.buf     = "Current partition-type GUID:";
         ctguid = txwCreateWindow(   cframe, TXW_STLINE, cframe, 0, &window, NULL);
         txwSetWindowUShort( ctguid, TXQWS_ID, DFSD_WID_ETGUID);
      }

      //- Partition type GUID description string
      style = TXWS_OUTPUT | TXWS_HCHILD2MOVE;
      txwSetupWindowData(
         6, 32, 1, 38,                          // UL corner + vert/hor size
         style, 0,                              // window frame style + help
         ' ', ' ',                              // window/border chars
         TXWSCHEME_COLORS,                      // std scheme colors
         "", "",
         &window);
      window.sl.buf     = ptinfo->vtgstr;
      cpdesc = txwCreateWindow(   cframe, TXW_STLINE, cframe, 0, &window, NULL);
      txwSetWindowUShort( cpdesc, TXQWS_ID, DFSD_WID_EPDESC);
      TRACES(("GUID string HWND: %8.8lx\n", cpdesc));


      if (dfsa->expertui)
      {
         //- Button triggers an action, should NOT dismiss the dialog -> DLGE
         style = TXWS_PBUTTON | TXBS_DLGE_BUTTON | TXWS_HCHILD2MOVE | TXWS_VCHILD_MOVE;
         txwSetupWindowData(
            8, 1, 3, 27,
            style, DFSH_EG_EBTMPG,              // style + help
            ' ', ' ',                           // window/border chars
            TXWSCHEME_COLORS,                   // std scheme colors
            "", "",                             // title & footer text
            &window);
         window.bu.text    = "Manual Edit Part GUID >";
         cbtMpg = txwCreateWindow(   cframe, TXW_BUTTON, cframe, 0, &window, NULL);
         txwSetWindowUShort( cbtMpg, TXQWS_ID, DFSD_WID_EBTMPG);
         txwSetWindowUShort( cbtMpg, TXQWS_GROUP, 1);
      }
      else
      {
         //- Partition type leading text string
         style = TXWS_OUTPUT | TXWS_HCHILD2MOVE;
         txwSetupWindowData(
            8, 2, 1, 28,                        // UL corner + vert/hor size
            style, 0,                           // window frame style + help
            ' ', ' ',                           // window/border chars
            TXWSCHEME_COLORS,                   // std scheme colors
            "", "",
            &window);
         window.sl.buf     = "Specific partition     GUID:";
         cpguid = txwCreateWindow(   cframe, TXW_STLINE, cframe, 0, &window, NULL);
         txwSetWindowUShort( cpguid, TXQWS_ID, DFSD_WID_EPGUID);
      }

      //- Partition GUID description string
      style = TXWS_OUTPUT | TXWS_HCHILD2MOVE;
      txwSetupWindowData(
         8, 32, 1, 38,                          // UL corner + vert/hor size
         style, 0,                              // window frame style + help
         ' ', ' ',                              // window/border chars
         TXWSCHEME_COLORS,                      // std scheme colors
         "", "",
         &window);
      window.sl.buf = ptinfo->vpgstr;
      cpdesc = txwCreateWindow(   cframe, TXW_STLINE, cframe, 0, &window, NULL);
      txwSetWindowUShort( cpdesc, TXQWS_ID, DFSD_WID_EPDESC);
      TRACES(("GUID string HWND: %8.8lx\n", cpdesc));


      //- Decimal postition and sizes checkbutton (updates global dfsa->hexSNS)
      txwSetupWindowData(
         10, wc, 1, 38,                         // UL corner + vert/hor size
         TXWS_CHECKB, DFSH_EG_DECIM,            // style & help
         ' ', ' ',                              // window/border chars
         TXWSCHEME_COLORS,                      // std scheme colors
         "", "",
         &window);
      window.bu.text    = "Use hexadecimal sector/size values";
      window.bu.checked = &(dfsa->hexSNS);
      hHexSNS = txwCreateWindow(   cframe, TXW_BUTTON, cframe, 0, &window, dfsFdskPteGptWinProc);
      txwSetWindowUShort( hHexSNS, TXQWS_ID, DFSD_WID_PDEC);
      txwSetWindowUShort( hHexSNS, TXQWS_GROUP, 1);


      style = TXWS_ENTRYBT | TXWS_VCHILD2MOVE;  // entryfield with title, allow move
      txwSetupWindowData(
         12, 1, 2,  14,
         style, DFSH_EG_EFIRST,                 // style and help
         ' ', ' ',                              // window/border chars
         TXWSCHEME_COLORS,                      // std scheme colors
         "First sector", "",                    // title & footer text
         &window);
      window.ef.leftcol  = 0;
      window.ef.maxcol   = TXW_INVALID;
      window.ef.curpos   = 0;
      window.ef.markCol  = 0;
      window.ef.markSize = 0;
      window.ef.rsize    = TXMAXTT;
      window.ef.buf      = ptinfo->vFirst;
      window.ef.history  = NULL;
      ptinfo->hFirst = txwCreateWindow(   cframe, TXW_ENTRYFIELD, cframe, 0, &window, dfsFdskPteGptWinProc);
      txwSetWindowUShort( ptinfo->hFirst, TXQWS_ID, DFSD_WID_EFIRST);

      style = TXWS_ENTRYBT     |                // entryfield without title, allow size and move
              TXWS_HCHILD2SIZE | TXWS_HCHILD2MOVE | TXWS_VCHILD2MOVE;
      txwSetupWindowData(
         12, 15, 2,  14,
         style, DFSH_EG_ELAST,                  // style and help
         ' ', ' ',                              // window/border chars
         TXWSCHEME_COLORS,                      // std scheme colors
         "Last sector", "",                     // title & footer text
         &window);
      window.ef.leftcol  = 0;
      window.ef.maxcol   = TXW_INVALID;
      window.ef.curpos   = 0;
      window.ef.markCol  = 0;
      window.ef.markSize = 0;
      window.ef.rsize    = TXMAXTT;
      window.ef.buf      = ptinfo->vLast;
      window.ef.history  = NULL;
      ptinfo->hLast = txwCreateWindow(   cframe, TXW_ENTRYFIELD, cframe, 0, &window, dfsFdskPteGptWinProc);
      txwSetWindowUShort( ptinfo->hLast, TXQWS_ID, DFSD_WID_ELAST);

      //- Partition #sectors and Size in MiB output field
      style = TXWS_OUTPUT | TXWS_HCHILD2MOVE;
      txwSetupWindowData(
         13, wc, 1, 38,                         // UL corner + vert/hor size
         style, 0,                              // window frame style + help
         ' ', ' ',                              // window/border chars
         TXWSCHEME_COLORS,                      // std scheme colors
         "", "",
         &window);
      window.sl.buf = ptinfo->vSize;
      ptinfo->hSize = txwCreateWindow(   cframe, TXW_STLINE, cframe, 0, &window, NULL);
      txwSetWindowUShort( ptinfo->hSize, TXQWS_ID, DFSD_WID_ETSIZE);

      if (dfsa->expertui)
      {
         //- Attribute HEX indicator '0x', output only
         style = TXWS_OUTPUTT | TXWS_HCHILD_MOVE | TXWS_VCHILD_MOVE;
         txwSetupWindowData(
            15, 1, 2,  3,
            style, 0,                           // style and help
            ' ', ' ',                           // window/border chars
            TXWSCHEME_COLORS,                   // scheme colors
            "Hex", "",
            &window);
         window.sl.buf = " 0x";
         chex0x = txwCreateWindow( cframe, TXW_STLINE, cframe, 0, &window, dfsFdskPteGptWinProc);


         style = TXWS_ENTRYBT | TXWS_HCHILD_MOVE | TXWS_VCHILD_MOVE;
         txwSetupWindowData(
            15, 4, 2,  19,
            style, DFSH_EG_EATVAL,              // style and help
            ' ', ' ',                           // window/border chars
            TXWSCHEME_COLORS,                   // std scheme colors
            "Attribute flags ", "",             // title & footer text
            &window);
         window.ef.leftcol  = 0;
         window.ef.maxcol   = TXW_INVALID;
         window.ef.curpos   = 0;
         window.ef.markCol  = 0;
         window.ef.markSize = 0;
         window.ef.rsize    = 16;                // 16 hex digits
         window.ef.buf      = ptinfo->vavbuf;
         window.ef.history  = NULL;
         centav = txwCreateWindow(   cframe, TXW_ENTRYFIELD, cframe, 0, &window, dfsFdskPteGptWinProc);
         txwSetWindowUShort( centav, TXQWS_ID, DFSD_WID_EATVAL);
      }

      style = TXWS_ENTRYBT | TXWS_HCHILD_MOVE | TXWS_VCHILD_MOVE;
      txwSetupWindowData(
         15, wc -2, 2,  40,
         style, DFSH_EG_EPNAME,                 // style and help
         ' ', ' ',                              // window/border chars
         TXWSCHEME_COLORS,                      // std scheme colors
         "Partition Name", "",                  // title & footer text
         &window);
      window.ef.leftcol  = 0;
      window.ef.maxcol   = TXW_INVALID;
      window.ef.curpos   = 0;
      window.ef.markCol  = 0;
      window.ef.markSize = 0;
      window.ef.rsize    = GPT_PNAME_LEN + 1;
      window.ef.buf      = ptinfo->vpnbuf;
      window.ef.history  = NULL;
      centpn = txwCreateWindow(   cframe, TXW_ENTRYFIELD, cframe, 0, &window, dfsFdskPteGptWinProc);
      txwSetWindowUShort( centpn, TXQWS_ID, DFSD_WID_EPNAME);

      if (dfsa->expertui)
      {
         //- Button triggers an action, should NOT dismiss the dialog -> DLGE
         style = TXWS_PBUTTON | TXBS_DLGE_BUTTON | TXWS_HCHILD2MOVE | TXWS_VCHILD_MOVE;
         txwSetupWindowData(
            18, 1, 3, 27,
            style, DFSH_EG_EBTATT,              // style + help
            ' ', ' ',                           // window/border chars
            TXWSCHEME_COLORS,                   // std scheme colors
            "", "",                             // title & footer text
            &window);
         window.bu.text    = "Specify Attribute Flags";
         cbtAtt = txwCreateWindow(   cframe, TXW_BUTTON, cframe, 0, &window, dfsFdskPteGptWinProc);
         txwSetWindowUShort( cbtAtt, TXQWS_ID, DFSD_WID_EBTATT);
      }

      style = TXWS_PBUTTON | TXWS_VCHILD_MOVE | TXWS_HCHILD2MOVE;
      txwSetupWindowData(
         18, wc -2, 3, 8,                       // vertical/horizontal size
         style, DFSH_EG_EBNAVI,                 // style and help
         ' ', ' ', TXWSCHEME_COLORS, "",  "",
         &window);
      window.bu.text = dfseg_bt_prev;
      hbtprev = txwCreateWindow(  cframe, TXW_BUTTON, cframe, 0, &window, dfsFdskPteGptWinProc);
      txwSetWindowUShort( hbtprev, TXQWS_ID, DFSDLG_PREV); // rc == ID

      style = TXWS_PBUTTON | TXWS_VCHILD_MOVE | TXWS_HCHILD2MOVE;
      txwSetupWindowData(
         18, wc +8, 3, 8,                       // vertical/horizontal size
         style, DFSH_EG_EBNAVI,                 // style and help
         ' ', ' ', TXWSCHEME_COLORS, "",  "",
         &window);
      window.bu.text = dfseg_bt_next;
      hbtnext = txwCreateWindow(  cframe, TXW_BUTTON, cframe, 0, &window, dfsFdskPteGptWinProc);
      txwSetWindowUShort( hbtnext, TXQWS_ID, DFSDLG_NEXT); // rc == ID

      style = TXWS_PBUTTON | TXWS_VCHILD_MOVE | TXWS_HCHILD2MOVE;
      txwSetupWindowData(
         18, wc +18, 3, 8,                      // vertical/horizontal size
         style, DFSH_EG_EBNAVI,                 // style and help
         ' ', ' ', TXWSCHEME_COLORS, "",  "",
         &window);
      window.bu.text = dfseg_bt_ok;
      hbtok = txwCreateWindow(   cframe, TXW_BUTTON, cframe, 0, &window, dfsFdskPteGptWinProc);
      txwSetWindowUShort( hbtok, TXQWS_ID, TXMBID_OK); // rc == ID

      style = TXWS_PBUTTON | TXWS_VCHILD_MOVE | TXWS_HCHILD2MOVE;
      txwSetupWindowData(
         18, wc +28, 3, 10,                     // vertical/horizontal size
         style, DFSH_EG_EBNAVI,                 // style and help
         ' ', ' ', TXWSCHEME_COLORS, "",  "",
         &window);
      window.bu.text = dfseg_bt_cancel;
      hbtcan = txwCreateWindow(  cframe, TXW_BUTTON, cframe, 0, &window, dfsFdskPteGptWinProc);
      txwSetWindowUShort( hbtcan, TXQWS_ID, TXMBID_CANCEL); // rc == ID

      rc = txwDlgBox( TXHWND_DESKTOP, TXHWND_DESKTOP, dfsEditGptDlgWinProc, cframe, ptinfo);
      if (rc != TXDID_CANCEL)                   // copy changed values back
      {
         dfsUidStringToBinary( ptinfo->vtgstr, gpte->typeGuid, TRUE);
         dfsUidStringToBinary( ptinfo->vpgstr, gpte->partGuid, TRUE);

         sscanf(  ptinfo->vavbuf, "%llX", &gpte->attrFlags);

         txwDlgField2U64( &(ptinfo->lFirst), ptinfo->vFirst, dfsa->hexSNS);
         txwDlgField2U64( &(ptinfo->lLast),  ptinfo->vLast,  dfsa->hexSNS);
         gpte->firstPSN   = ptinfo->lFirst;
         gpte->lastPSN    = ptinfo->lLast;

         TxAscii2Unic( ptinfo->vpnbuf, gpte->partName, GPT_PNAME_LEN);

         switch (rc)                            // dialog result
         {
            case DFSDLG_REQ5:
            case DFSDLG_REQ6:
            case DFSDLG_REQ7:
            case DFSDLG_REQ8:
            case DFSDLG_EDIT:                   // request editor
               cbp->cbOptNum2 = ptinfo->eSect;  // sector position info
            case DFSDLG_NEXT:                   // request next
            case DFSDLG_PREV:                   // request prev
               cbp->cbOptNum1 = rc;
               desiredFocus   = rc;             // same window-ID next time!
               break;

            default:
               cbp->cbOptNum1 = DFSDLG_NONE;
               break;
         }
         rc = NO_ERROR;
      }
      else                                      // cancel / escape
      {
         cbp->cbOptNum1 = DFSDLG_NONE;          // stop Ptable iteration
      }
   }
   else if (ptinfo == NULL)
   {
      rc = DFS_ALLOC_ERROR;
   }
   TxFreeMem( ptinfo);                          // free when allocated
   RETURN (rc);
}                                               // end 'dfsFdskPteGptDialog'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Dialog window procedure, for the PTEDIT dialog fields
/*****************************************************************************/
static ULONG dfsFdskPteGptWinProc               // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    msg param 1
   TXWMPARAM           mp2                      // IN    msg param 2
)
{
   ULONG               rc = NO_ERROR;

   ENTER();
   TRCMSG( hwnd, msg, mp1, mp2);

   if (hwnd != 0)
   {
      USHORT        wid   = txwQueryWindowUShort( hwnd, TXQWS_ID);
      TXWHANDLE     owner = txwQueryWindow( hwnd, TXQW_OWNER); // get dialog info
      PTEGPT_INFO  *pti   = (PTEGPT_INFO *) txwQueryWindowPtr( owner, TXQWP_USER);

      TRCMSG( hwnd, msg, mp1, mp2);
      TRACES(( "own: %8.8lx, pid:%hu  with attached PTINFO: %8.8lx\n", owner, wid, pti));

      switch (msg)
      {
         case TXWM_CHAR:
            switch ((ULONG) mp2)
            {
               case TXk_UP:
                  if     ((wid == DFSDLG_PREV)    ||
                          (wid == DFSDLG_NEXT)    ||
                          (wid == TXMBID_OK)      ||
                          (wid == TXMBID_CANCEL)   ) // to name field
                  {
                     txwSetFocus( txwWindowFromID( owner, DFSD_WID_EPNAME));
                  }
                  else if (wid == DFSD_WID_EBTATT) // to attribute value field
                  {
                     txwSetFocus( txwWindowFromID( owner, DFSD_WID_EATVAL));
                  }
                  else
                  {
                     rc = txwDefWindowProc( hwnd, msg, mp1, mp2);
                  }
                  break;

               case TXk_DOWN:
                  if     (wid == DFSD_WID_EPNAME)
                  {
                     txwSetFocus( txwWindowFromID( owner, DFSDLG_NEXT));
                  }
                  else if (wid == DFSD_WID_EATVAL) // to attribute edit button
                  {
                     txwSetFocus( txwWindowFromID( owner, DFSD_WID_EBTATT));
                  }
                  else
                  {
                     rc = txwDefWindowProc( hwnd, msg, mp1, mp2);
                  }
                  break;

               case DFSDLG_REQ5:
               case DFSDLG_REQ6:
               case DFSDLG_REQ7:
               case DFSDLG_REQ8:
               case DFSDLG_EDIT:                // request Hex editor
                  switch ((ULONG) mp2)
                  {
                     default:                   // for _EDIT, no eSect really needed!
                     case DFSDLG_REQ5:
                     case DFSDLG_REQ6: pti->eSect = pti->lFirst;  break;
                     case DFSDLG_REQ7:
                     case DFSDLG_REQ8: pti->eSect = pti->lLast;   break;
                  }
               case TXk_ENTER:                  // force update of field vars
               case TXk_PGUP:                   // before quiting the dialog
               case TXk_PGDN:                   // (just like focus loss)
               case TXk_F4:
               default:
                  rc = txwDefWindowProc( hwnd, msg, mp1, mp2);
                  break;
            }
            break;

         case TXWM_SETFOCUS:                    // on loosing focus
            if (((wid == DFSD_WID_EFIRST) ||
                 (wid == DFSD_WID_ELAST)) && (((BOOL) mp1) == FALSE))
            {
               txwDlgField2U64(   &(pti->lFirst), pti->vFirst, dfsa->hexSNS);
               txwDlgField2U64(   &(pti->lLast),  pti->vLast,  dfsa->hexSNS);
               dfsEditGptSetSize(   pti);
               txwInvalidateWindow( pti->hSize, FALSE, FALSE);
            }
            rc = txwDefWindowProc( hwnd, msg, mp1, mp2); // default handling too!
            break;

         default:
            rc = txwDefWindowProc( hwnd, msg, mp1, mp2);
            break;
      }
   }
   else
   {
      rc = TX_INVALID_HANDLE;
   }
   RETURN( rc);
}                                               // end 'dfsFdskPteGptWinProc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Dialog window procedure, for the PTEDIT GPT NEXT/PREV type Dialogs
/*****************************************************************************/
static ULONG dfsEditGptDlgWinProc               // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    msg param 1
   TXWMPARAM           mp2                      // IN    msg param 2
)
{
   ULONG               rc    = NO_ERROR;

   ENTER();
   TRCMSG( hwnd, msg, mp1, mp2);

   if (hwnd != 0)
   {
      USHORT           wid    = (USHORT) mp1;
      PTEGPT_INFO     *pti = (PTEGPT_INFO *) txwQueryWindowPtr( hwnd, TXQWP_USER);
      TXSELIST        *list   = NULL;
      TXS_ITEM        *item   = NULL;
      TXTT             s1     = {0};
      ULN64            attr   =  0;
      TXWHANDLE        ctrl   = NULL;           // handle for a child control window

      switch (msg)
      {
         case TXWM_COMMAND:
            switch ((ULONG) mp2)                // command source
            {
               case TXCMDSRC_CHECKBOX:
                  if (wid == DFSD_WID_PDEC)     // decimal sector/size
                  {
                     ctrl = txwWindowFromID( hwnd, wid);
                     TRACES(("ctrl ptr:%p  hwnd:%p  pti:%p\n", ctrl, hwnd, pti));

                     txwDlgField2U64( &(pti->lFirst), pti->vFirst, dfsa->hexSNS);
                     txwDlgField2U64( &(pti->lLast),  pti->vLast,  dfsa->hexSNS);

                     dfsa->hexSNS = !(dfsa->hexSNS);
                     txwInvalidateWindow( ctrl, FALSE, FALSE);

                     txwDlgU642Field( pti->hFirst, pti->lFirst, pti->vFirst, 12, dfsa->hexSNS);
                     txwDlgU642Field( pti->hLast,  pti->lLast,  pti->vLast,  12, dfsa->hexSNS);

                     dfsEditGptSetSize(   pti);
                     txwInvalidateWindow( pti->hSize, FALSE, FALSE);
                  }
                  break;

               case TXCMDSRC_PUSHBUTTON:
                  switch ((ULONG) mp1)
                  {
                     case DFSD_WID_EBTMAN:
                        if (strlen( pti->vtgstr) != 0) // show existing value
                        {
                           strcpy( s1, pti->vtgstr);
                        }
                        if (dfsGuidUuidDialog( " Specify partition type as a GPT-style GUID ",
                                                 NULL, 0, DFSH_EG_EBTMAN, s1) != TXDID_CANCEL)
                        {
                           pti->gptList->selected = pti->gptList->count -1; // set to last (custom)
                           strcpy( pti->vtgstr, s1);
                        }
                        txwInvalidateWindow( hwnd, FALSE, TRUE); // dlg incl controls
                        break;

                     case DFSD_WID_EBTMPG:
                        if (strlen( pti->vpgstr) != 0) // show existing value
                        {
                           strcpy( s1, pti->vpgstr);
                        }
                        if (dfsGuidUuidDialog( " Specify partition ID as a GPT-style GUID ",
                                                 NULL, 0, DFSH_EG_EBTMPG, s1) != TXDID_CANCEL)
                        {
                           strcpy( pti->vpgstr, s1);
                        }
                        txwInvalidateWindow( hwnd, FALSE, TRUE); // dlg incl controls
                        break;

                     case DFSD_WID_EBTATT:
                        sscanf(  pti->vavbuf, "%llX", &attr);
                        dfsGptAttributeDialog( &attr);
                        sprintf( pti->vavbuf, "%16.16llx", attr);
                        txwInvalidateWindow( hwnd, FALSE, TRUE); // dlg incl controls
                        break;

                     default:
                        rc = txwDefDlgProc( hwnd, msg, mp1, mp2);
                        break;
                  }
                  break;

               default:
                  break;
            }
            break;

         case TXWM_CONTROL:
            if ((list = (TXSELIST *) mp2) != NULL)
            {
               item = list->items[list->selected];
            }
            TRACES(("WM_CONTROL list:%8.8lx item:%8.8lx, selected %lu = 0x%02hx\n",
                    list, item, list->selected, list->items[list->selected]->value));
            if (item != NULL)
            {
               if (TXSH2FROMMP(mp1) == TXLN_SELECT)
               {
                  if (dfsUidStringIsValid( item->desc)) // item is a GUID, so set it current
                  {
                     //- moving through the list, without opening it, but DO make it current!
                     strcpy( pti->vtgstr, item->desc);
                  }
                  txwPostMsg( TXHWND_DESKTOP, TXWM_SETFOOTER, (TXWMPARAM) item->desc, 0);
               }
               else if (TXSH2FROMMP(mp1) == TXLN_ENTER)
               {
                  if (item->value == (TXDID_MAX + DFSGPT_CUSTOM)) // Partition custom type value
                  {
                     if (strlen( pti->vtgstr) != 0) // show existing value
                     {
                        strcpy( s1, pti->vtgstr);
                     }
                     if (dfsGuidUuidDialog( " Specify partition type as a GPT-style GUID ",
                                              NULL, 0, DFSD_WID_ESPINV, s1) != TXDID_CANCEL)
                     {
                        strcpy( pti->vtgstr, s1);
                     }
                  }
                  else                          // other list item, copy GUID string
                  {
                     strcpy( pti->vtgstr, item->desc);
                  }
               }
               txwInvalidateWindow( hwnd, FALSE, TRUE); // dlg incl controls
            }
            break;

         case TXWM_CHAR:
            switch ((ULONG) mp2)
            {
               case TXk_F3:
                  txwPostMsg( hwnd, TXWM_CLOSE, 0, 0);
                  break;

               case TXk_F4:
                  txwDismissDlg( hwnd, DFSDLG_NONE); // save, end dlg
                  break;

               case TXk_PGDN:
                  txwDismissDlg( hwnd, DFSDLG_NEXT); // save, to next table
                  break;

               case TXk_PGUP:
                  txwDismissDlg( hwnd, DFSDLG_PREV); // save, to prev table
                  break;

               case DFSDLG_EDIT:                  // allow several Fkeys for
               case DFSDLG_REQ5:                  // editing, higher level
               case DFSDLG_REQ6:                  // may edit at different
               case DFSDLG_REQ7:                  // sectors for each one
               case DFSDLG_REQ8:
                  txwDismissDlg( hwnd, (ULONG) mp2); // save, start editor
                  break;

               case TXk_F10:                    // popup (list) requested
                  ctrl = txwWindowFromID( hwnd, DFSD_WID_ESPINV);
                  txwPostMsg( ctrl, TXWM_CHAR, 0, (TXWMPARAM) TXk_ENTER); // ENTER on list, popup
                  break;

               default:
                  rc = txwDefDlgProc( hwnd, msg, mp1, mp2);
                  break;
            }
            break;

         default:
            rc = txwDefDlgProc( hwnd, msg, mp1, mp2);
            break;
      }
   }
   else
   {
      rc = TX_INVALID_HANDLE;
   }
   RETURN( rc);
}                                               // end 'dfsEditGptDlgWinProc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Update descriptional size field from first/last PSN
/*****************************************************************************/
static void dfsEditGptSetSize
(
   PTEGPT_INFO        *pti                      // IN    PTE entry info
)
{
   ULN64            psize;                      // partition size in sectors

   ENTER();

   psize = pti->lLast - pti->lFirst + 1;
   if (dfsa->hexSNS)
   {
      sprintf( pti->vSize, "Size #sect : 0x%12.12llx", psize);
   }
   else
   {
      sprintf( pti->vSize, "Size #sect : %14llu", psize);
   }
   dfstr64XiB( pti->vSize, " = ", psize * dfsGetSectorSize(), "");

   VRETURN ();
}                                               // end 'dfsEditGptSetSize'
/*---------------------------------------------------------------------------*/


#endif                                          // USEWINDOWING
