//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// DFSee major functions: imaging, cloning, wiping, scanning, searching ...
//
// Author: J. van Wijk
//
// JvW  27-07-2006   Initial version, split off from DFSMAJOR.C
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfshpfs.h>                            // HPFS structure defs
#include <dfstore.h>                            // Store and sector I/O
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfs.h>                                // DFS  navigation and defs
#include <dfsdgen.h>                            // DFS generic dialogs
#include <dfsulzw.h>                            // DFSee compression interface
#include <dfsclone.h>                           // cloning and moving
#include <dfsutil.h>                            // DFS  utility functions
#include <dfsspace.h>                           // DFS file-space interface
#include <dfsupart.h>                           // PART utility functions
#include <dfsufdsk.h>                           // FDSK utility functions
#include <dfsufgpt.h>                           // FDISK GPT utility functions
#include <dfsalist.h>                           // sector Area list handling


// disk-2-disk re-clone default script name when using -skipbads
#define DFSCL_D2D_BADS_RETRY    "d2dretry.dfs"

// first  part of script, parameter: 1 STRING, d2d options
static char d2dretry_header[] =
   ";Script generated by CLONE -skipbads, RETRY bad areas for disks: %s\n"
   "; ~ no-parameters when run with script-file-dialog prompting~\n"
   ";\n";


// Progress related definitions and variables
#define DFSCL_SMART_FAST_FACTOR   99            // smart (unused) areas 99 times faster
                                                // than regular used areas (cloning)

static  char       *cmd_clone[] =
{
   "",
   "Copy or Compare sectors from other to current store (clone)",
   "",
   " Usage: CLONE | COMP [at [size]] [-d:disk | -p:pid | -v:vol] [-f:sn] [-a:sn]",
   "",
   "   at           : start sector 'to' store   (default is sector 0  )",
   "   size         : size to handle in sectors (default is whole area)",
   "",
   "   -a:mcs       : start sector 'to' store   (default is 0 megabyte)",
   "   -b:sectors   : Number of sectors to buffer, the default is to",
   "                  use the current number of sectors per track",
   "                  (18 for diskettes, usually 63 for harddisks)",
   "                  The size is automatically limited to available buffer",
   "                  memory, and reduced to 1 when hitting bad sectors.",
   "   -d[:disk]    : disk number to clone FROM/compare with, 1 .. max",
   "   -D[:disk]    : TO disk, disk number to clone TO, 1 .. max",
   "   -E:i         : ignore errors, continue cloning on read errors",
   "   -E:q         : quit on any read errors, exit  (batch  default)",
   "   -E:c         : confirm by user on read errors (normal default)",
   "   -f:mcs       : start sector other store, (default is 0 megabyte)",
   "   -l           : add different sectors to sector list (in compare)",
   "   -L           : exclude LVM signature area, clone FS-area only",
   "   -M           : Do not write fixed 0xFE pattern on bad-sectors in",
   "                : the source, but keep the possibly correct contents",
   "   -name        : Update (LVM) diskname by postfixing '-YYMMDD' date",
   "   -name:'-str' : Update (LVM) diskname using specified postfix string",
   "   -name:'str'  : Update the complete (LVM) diskname on a clone disk",
   "   -p[:pid]     : partition to clone FROM/compare with, PID/driveletter",
   "   -P[:pid]     : TO partition, partition to clone TO, PID/driveletter",
   "   -R           : Reverse copy direction, from High to Low sector nrs",
   "   -s:mcs       : size to handle as mcs-number   (see DFSTERMS.TXT)",
   "   -skipbads:n  : Skip 'n' sectors on each bad-sector area detected",
   "                  Implies -E:i = 'ignore errors' and -M = 'merge'",
   "   -S           : Smart skip unused space in filesystem or disk",
   "   -v[:vol]     : volume to clone/compare, driveletter; default is C:",
   "   -V[:vol]     : TO volume, volume to clone TO, driveletter",
   "                : at the destination sector (CLONE merging).",
   "   -X           : Suppress REBOOT popup after successful completion",
   "",
   "   -! or -!-    : Force (-!), or skip (-!-) the generic CLONE dialog",
   "",
   " The mcs-number format is [0x|0t]nnnn[,g|m|k|c|h|s] HEX or decimal value,",
   " in GiB, MiB, KiB, Cylinders or Sectors. Default is decimal MiB.",
   "",
   NULL
};


static  char       *cmd_move[] =
{
   "",
   "MOVE -b|-e | -f  = Move towards Begin or End, or Move/Copy to freespace area",
   "",
   " Usage: MOVE -b[:offset] | -e[:offset [-f:fsp-id [-c [-l:x]]] [-a | -align-]",
   "",
   "   -a           : Align resulting partition START to a cylinder boundary",
   "   -align-      : Do NOT align KiB/MiB SIZE values to track/cyl multiples",
   "   -b           : Move to exact begin of freespace area BEFORE partition",
   "   -b:mcs-nr    : Move towards begin freespace, offset from current pos.",
   "   -c           : Copy, not Move, do NOT delete original.   (requires -f)",
   "   -e           : Move to exact end of freespace area AFTER the partition",
   "   -e:*         : Move to exact end of freespace area specified with  -f",
   "   -e -f:[id]   : Move to exact begin of freespace area selected with -f",
   "   -e:mcs-nr    : Move towards end fsp, offset from current (or start -f)",
   "   -f:fsp-id    : Move/Copy to fsp-area on same/other disk  (implies  -e)",
   "   -l:letter    : Set preferred driveletter for new volume  (requires -c)",
   "   -P           : Prompt for offset value to be used, in Mib or Cylinders",
   "",
   " The mcs-number format is [0x]nnnn[,g|m|k|c|t|s] a HEX or decimal value,",
   " in GiB, MiB, KIb, Cylinders, Tracks or Sectors. Default is decimal MiB.",
   "",
   NULL
};


static char cloneReboot[] =
   "The destination partition needs to be rediscovered by the operating-system "
   "before using it. The safest way to do that is (re)booting\n\n"
   "You can safely do multiple operations before booting "
   "as long as the destination partitions are NOT accessed.";

static char winDriveLetters[] =
   "Copying or moving partitions may cause Windows drive-letter changes! "
   "When using Windows, make sure to use the disk manager to check/assign "
   "drive letters as required before working with the system.";


// Delete a partition, read disk-info and (re) select disk; invalidates disk & part pointers !!!
static ULONG dfsDeleteAndSelectDisk
(
   USHORT              part,                    // IN    partition to delete
   USHORT              disk                     // IN    disk to (re) select
);


// Create a DFS script to re-clone the bad-areas in given area list
static void dfsMkAreaCloneScript
(
   S_AREALIST         *list,                    // IN    area list
   char               *d2dOption                // IN    disk-2-disk options
);

/*****************************************************************************/
// High level CLONE and COMP command logic
/*****************************************************************************/
ULONG dfsCloneCompCommand
(
   int                 cc,                      // IN    number of parameters
   char               *c1,                      // IN    parameter string 1
   char               *c2,                      // IN    parameter string 2
   BOOL                clone                    // IN    action is CLONE
)
{
   ULONG               rc = NO_ERROR;           // function return
   ULN64               sn;                      // at      sector number
   ULN64               sl;                      // from    sector number
   ULN64               sectors;
   TXLN                s0, s1;                  // temporary string space
   TXTM                dc;
   BOOL                ok = TRUE;
   DFSPARTINFO        *p  = NULL;
   DFST_HANDLE         tstore = DFSTORE;        // current store
   DFST_HANDLE         fstore = DFSTALT;        // alternate store
   DFSISTORE          *fsi = NULL;              // FROM store info struct
   DFSOBJECT           obj_to   = SINF->object; // defaults for batch mode
   DFSOBJECT           obj_from = SINF->object; // or -!- option used
   ULONG               flags    = DFSI_STANDARD;
   ULN64               skiparea = 0;            // skip size on bad-sectors
   ULN64               sizediff = 0;            // different store sizes
   BOOL                doFixpbr = FALSE;
   TXTT                oldDiskName = {0};       // existing LVM disk name
   TXTT                newDiskName = {0};       // updated  LVM disk name

   ENTER();

   if (TxaOption('?') || ((cc > 1) && (c1[0] == '?')))
   {
      TxShowTxt( cmd_clone);
      RETURN (rc);
   }

   if (TxaOption('S'))                          // smart-use
   {
      flags |= DFSI_SMARTUSE;
   }
   if (TxaOption('M'))                          // merge bad-sectors with
   {                                            // original data at dest
      flags |= DFSI_MERGEBAD;
   }
   if (TxaOption('R'))                          // Force reverse direction copy
   {                                            // from high to low sector nrs
      flags |= DFSI_HIGH2LOW;
   }
   if (TxaOption('o'))                          // Add 'Overlapped' warning and
   {                                            // confirmation on Aborting
      flags |= DFSI_OVERLAPD;                   // would loose partition ...
   }
   if (TxaOptSet( DFS_O_SKBADS))                // set size of area to skip on BAD sectors
   {
      skiparea = TxaOptNum( DFS_O_SKBADS, NULL, 2048);
   }
   sectors = 0;                                 // clone whole object, unless
                                                // specified otherwise

   if ((dfsa->batch == FALSE) && !TxaOptUnSet('!'))
   {
      ok &= (dfsGetObjectValue( tstore, "DPV", &obj_to  ));
      ok &= (dfsGetObjectValue( fstore, "dpv", &obj_from));
   }
   if (ok)
   {
      rc = dfsSetObjectCurrent( "DPVI", TRUE);  // open object when specified
      if (rc != DFS_ALLOC_ERROR)                // internal error, rest is 'ok'
      {
         if ((dfsGetObjectOpenCmd( "dpvi", s1) == NO_ERROR) && (strlen( s1) != 0)) // got valid open command
         {
            sprintf( s0, "store %u#%s", fstore, s1);
            if (dfsMultiCommand( s0, 0, FALSE, FALSE, FALSE) != NO_ERROR)
            {
               TxPrint( "\nERROR opening the 'FROM' location using '%s'!\n", s1);
               ok = FALSE;
            }
         }
         else
         {
            ok = FALSE;
         }
      }
      else
      {
         ok = FALSE;
      }
   }
   ok &= TxaOptMutEx((cc > 1), "a", "if 'start' is specified!", NULL);
   ok &= TxaOptMutEx((cc > 2), "s", "if 'size'  is specified!", NULL);
   ok &= dfsOptionSizeSectors( 's', tstore, 'm', 0, &sectors); //- size
   ok &= dfsOptionSizeSectors( 'a', tstore, 'm', 0, &sn);      //- at pos
   ok &= dfsOptionSizeSectors( 'f', fstore, 'm', 0, &sl);      //- from pos

   if (ok && !TxaOption('!'))
   {
      if (cc > 1)                               // regular parameters
      {
         sn = dfsGetSymbolicSN( c1, nav.this);  // default current LSN
         if (cc > 2)
         {
            sectors = dfsGetSymbolicSize( c2, sectors, fstore, sn);
         }
      }
      if (sectors == 0)                         // default copy size, minimum
      {                                         // of source and destination
         sectors  = dfstGetLogicalSize( fstore);
         if (dfstGetLogicalSize( tstore) < sectors)
         {
            if (clone)                          // issue warning later ...
            {
               sizediff = sectors - dfstGetLogicalSize( tstore);
            }
            sectors = dfstGetLogicalSize( tstore);
            if (sectors > sn)                   // if not from start ...
            {
               sectors -= sn;                   // from sn to end of area
            }
         }
         else
         {
            if (sectors > sl)                   // if not from start ...
            {
               sectors -= sl;                   // from sn to end of area
            }
         }
      }

      if (dfstQueryStoreType( fstore, &fsi, NULL) != DFST_UNUSED)
      {
         if (fsi != NULL)
         {
            p = fsi->p;
            if ((p != NULL)              &&     // it is a partition
                (TxaOption('L'))         &&     // exclude-LVM option
                (sectors > p->lvmReserved))     // valid reserved area
            {
               sectors  -= p->lvmReserved;      // exclude signature-area
               TRACES(( "sectors -L: 0x%llx\n", sectors));
            }
            else if ((obj_from == DFSO_DISK) && (obj_to == DFSO_DISK))
            {
               TRACES(("fsi->disknr:%hu\n", fsi->disknr));
               if (TxaOptSet( TXA_O_NAME))      // Update LVM diskname on clone
               {
                  DFSDISKINFO *d = dfsGetDiskInfo( fsi->disknr);

                  if ((d != NULL) && (d->lvmPartitions != 0))
                  {
                     strcpy( oldDiskName, d->DiskName);
                     TxaOptAsString( TXA_O_NAME, TXMAXLN, s1);
                     TRACES(("new name s1: '%s'\n", s1));
                     if (strlen( s1))
                     {
                        if ((s1[0] == '-') && (strlen( s1) < (LVM_NAME_L -1))) // valid postfix given
                        {
                           strcpy( newDiskName, d->DiskName);             //- start with current FROM name
                           newDiskName[ LVM_NAME_L - strlen( s1) -1] = 0; //- maximum prefix length
                           strcat( newDiskName, s1);                      //- append given postfix
                        }
                        else
                        {
                           strcpy( newDiskName, s1);
                        }
                     }
                     else                       // empty, create one
                     {
                        time_t    tt = time( &tt); // current date/time
                        TXTS      now;          // date string buffer

                        strftime( now, TXMAXTS, "%Y%m%d", localtime( &tt));
                        strcpy( newDiskName, d->DiskName);  //- start with current FROM name
                        newDiskName[ 11] = 0;               //- maximum prefix length
                        strcat( newDiskName, now);          //- append 8 character date
                     }
                  }
               }
               TRACES(("Change LVM diskname from '%s' to '%s'\n", oldDiskName, newDiskName));
            }
         }

         strcpy( dc, ""); dfstrSz64( dc, "", sectors, "");
         TxPrint( "\n%s\nAlternate %s area\n", (clone) ? "Clone  " : "Compare", dfstStoreDesc1(fstore));
         if (dfstStoreType( tstore) != DFST_UNUSED)
         {
            TXTM stpos = {0};                   // source and target position
            TXTM skipb = {0};                   // Skipbad area text

            if ((sn != 0) || (sl != 0))
            {
               sprintf( stpos, "Start position : %12.12llX, TO obj start position %12.12llx\n", sl, sn);
            }
            if (skiparea != 0)
            {
               dfstrSz64( skipb, "Size to skip on each BAD-sector area  : ", skiparea, "\n");
            }
            TxPrint( "TO object %s\n", dfstStoreDesc1(tstore));

            strcpy( dc, "");
            dfstrSz64( dc, "", sectors, "");
            TxPrint( "Sectors to handle : %s ", dc);
            dfsX10(  "to LSN: ", sn, "",    " ");
            dfsX10(  "from: ",   sl, "", "\n\n");

            if (sizediff != 0)
            {
               strcpy( s0, ""); dfstrSz64( s0, "Size difference : ", sizediff, "");
               TxNamedMessage( !dfsa->batch, 5010, " WARNING: Clone size mismatch ",
                          "WARNING: The 'TO' object is smaller than the 'FROM' object, "
                          "the clone will be truncated and might not be fully functional.\n\n%s", s0);
            }
            if ((clone == FALSE) || (dfsa->batch) || (TxaOptUnSet('C')) ||
                (TxConfirm( 5008,
               "Make sure the FROM disk/partition is clean (no open files)\n"
               "and the TO disk/partition is not in use when possible\n\n"
               "Size to clone  : %s%s\nFROM : %s\nTO   : %s\n%s%s\n"
               "Are you sure this is what you want to clone ? [Y/N] : ", dc,
               (flags & DFSI_HIGH2LOW) ? "  (HIGH to LOW sect-nrs)" : "",
                dfstStoreDescription1(fstore, s0),
                dfstStoreDescription1(tstore, s1), stpos, skipb)))
            {
               TRACES(("flags: 0x%8.8x, FsUnallocSmart: 0x0%llx\n", flags, dfsa->FsUnallocSmart));

               TxCancelAbort();                 // cancel any pending abort (alloc!)
               if ((flags & DFSI_SMARTUSE) && (dfsa->FsUnallocSmart == 0)) // no Smart alloc info available yet
               {
                  TxPrint( "\nGet Smart-ALLOC info. May be aborted with <Esc> if ETA accuracy is unimportant.\n");
                  DFSFNCALL( dfsa->FsAllocDisplay, 0, 0, "@", NULL); // single line ALLOC display / info gathering

                  if (TxAbort())
                  {
                     TxPrint( "Gathering Smart-ALLOC info was aborted. ETA calculations will be less accurate!\n");
                  }
                  TxCancelAbort();              // cancel any pending abort (alloc!)
               }

               if (clone)
               {
                  if (DFSTORE_WRITE_ALLOWED)
                  {
                     rc = dfsCloneStoreArea( sn, sectors, fstore, tstore, sl,
                                             flags, skiparea, dfsa->verbosity,
                                             oldDiskName, newDiskName);
                     if (rc == NO_ERROR)
                     {
                        if (p && p->primary && SINF->p) // primary part to part
                        {
                           switch (dfsa->FsModeId)
                           {
                              case DFS_FS_FAT:
                              case DFS_FS_HPFS:
                              case DFS_FS_NTFS:
                              case DFS_FS_JFS:
                                 doFixpbr = TRUE; // hiddensectors update
                                 break;

                              default:
                                 break;
                           }
                        }
                        dfsReadDiskInfo( FDSK_QUIET);  //- re-read all partition info
                        sprintf( s0, "CLONE completed successfully.\n");
                        TxPrint( "\n%s", s0);
                        if (!TxaOption('X'))           //- not a higher-level operation
                        {
                           sprintf( s1, "%s\n%s", s0, cloneReboot);
                           TxNamedMessage( !dfsa->batch, 5533, " INFO: Reboot recommended ", s1);
                        }
                     }
                  }
                  else
                  {
                     rc = DFS_READ_ONLY;
                  }
               }
               else
               {
                  rc = dfsCompareStoreArea(  sn, sectors, fstore, tstore,
                                             sl, flags, dfsa->verbosity,
                                             oldDiskName, newDiskName);
               }
            }
            else
            {
               rc = DFS_NO_CHANGE;
            }
            //- always switch back, even on Esc or fail!
            //- sprintf( s0, "store %u", tstore); // switch back to default!
            //- dfsMultiCommand( s0, 0, FALSE, FALSE, FALSE);
         }
         else                                   // no other store open
         {
            TxPrint( "\nThe TO store (object) for %s (%hu) is not linked "
                     "to any data source!\n",
                      (clone) ? "clone" : "compare", tstore);
            TxShowTxt( cmd_clone);
            rc = DFS_CMD_FAILED;
         }
      }
      else                                      // no other store open
      {
         TxPrint( "\nThe FROM store (object) for %s (%hu) is not linked "
                  "to any data source!\n",
                   (clone) ? "clone" : "compare", fstore);
         TxShowTxt( cmd_clone);
         rc = DFS_CMD_FAILED;
      }
   }
   if (DFSTORE != tstore)                       // still swapped stores ?
   {
      sprintf( s0, "store %u", tstore);        // switch back to default!
      dfsMultiCommand( s0, 0, FALSE, FALSE, FALSE);
   }
   if (doFixpbr)
   {
      dfsFixPbrHsGeoFields( TRUE, TRUE, TRUE, FALSE);
      TxPrint( "Note: When Bootsector Geo IS updated, comparing the cloned partitions will fail!\n");
   }
   if ((!ok && (rc == NO_ERROR)) || (TxaOption('!')))
   {
      #if defined (USEWINDOWING)
         if (dfsa->batch)
         {
            TxShowTxt( cmd_clone);
            rc = DFS_CMD_FAILED;
         }
         else
         {
            dfsCloneDialog( clone, obj_from, sl, obj_to, sn, sectors, 0);
         }
      #else
         TxShowTxt( cmd_clone);
         rc = DFS_CMD_FAILED;
      #endif
   }
   RETURN (rc);
}                                               // end 'dfsCloneCompCommand'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Clone area of sectors from the CURRENT store (FROM) to another one (TO)
/*****************************************************************************/
ULONG dfsCloneStoreArea
(
   ULN64               start,                   // IN    start sector
   ULN64               sectors,                 // IN    sectors to copy
   DFST_HANDLE         from,                    // IN    from store
   DFST_HANDLE         to,                      // IN    to   store
   ULN64               fstart,                  // IN    from start-sector
   ULONG               flags,                   // IN    cloning options
   ULN64               badskip,                 // IN    size to skip on bads
   BOOL                progress,                // IN    display progress
   char               *oldLvmName,              // IN    old LVM disk name
   char               *newLvmName               // IN    new LVM disk name
)
{
   ULONG               rc = NO_ERROR;           // function return
   USHORT              bps;                     // sectorsize to use
   BYTE               *cb;                      // clone sector buffer
   ULONG               size;                    // clone buffer, variable size
   ULN64               rsn;                     // relative sector number
   ULONG               bs = 0;                  // blocksize to copy
   ULN64               lastc;                   // last cloned
   DFSISTORE          *tsi;
   DFSISTORE          *fsi;                     // from store info
   DFSTORETYPE         tt;
   DFSTORETYPE         ft;                      // from store type
   BOOL                samedisk = FALSE;
   BOOL                ok       = TRUE;         // OK to clone ...
   BOOL                smartuse = (flags & DFSI_SMARTUSE); // skip unused
   BOOL                high2low = (flags & DFSI_HIGH2LOW); // reverse order
   ULN64               failures = 0;            // sector read failures
   ULONG               skipbads = 0;            // extra skip on bad areas
   TXLN                text;
   TXTS                d2dClone = {0};          // d2dClone TO / FROM option

   ENTER();

   if ((bps = dfstGetSectorSize( from)) == dfstGetSectorSize( to))
   {
      if ((tt = dfstQueryStoreType( to, &tsi, NULL)) == DFST_PHDISK)
      {
         if ((ft = dfstQueryStoreType( from, &fsi, NULL)) == DFST_PHDISK)
         {
            if (tsi->disknr == fsi->disknr)     // on same physical disk!
            {
               samedisk = TRUE;
            }
            else if ((tsi->partid == 0) && (tsi->partid == 0)) // disk-2-disk, not same
            {
               sprintf( d2dClone, " -d:%hu -D:%hu", fsi->disknr, tsi->disknr);
            }
         }
         if (((tsi->partid == 0) && (tt == DFST_PHDISK)) && // bare disk
             ((fsi->partid != 0) || (ft != DFST_PHDISK))  ) // not bare
         {
            if ((!dfsa->batch) &&               // forced or confirmed
                (!TxConfirm( 5009,
               "The TARGET for 'clone' is a (whole) physical disk "
               "while the SOURCE is not, this could be dangerous! "
               "The TARGET must be selected BEFORE using 'clone' "
               "by using a 'part', 'vol' or 'disk' command\n\n"
               "Are you sure you selected the right TARGET ? [Y/N] : ")))
            {
               TxPrint("\nClone operation aborted, no changes made\n");
               ok = FALSE;                      // abort the cloning
            }
         }
      }
      if (ok)                                   // passed checks ?
      {
         //- Use default SMART-size when smartuse, and optimal-for-OS otherwise
         size = dfsGetBufferSize( DFSOPTIMALBUF, DFSMAXBUFSIZE);
         if (samedisk && !TxaOption('b'))       // bigger buffer if same disk, faster
         {
            if (dfsa->FsModeId != DFS_FS_HPFS)  // HPFS fragmented by 8Mb bitmaps
            {                                   // so keep smallish buffer
               size = DFSMAXBUFSIZE;            // 64 MiB (for non-DOS only!)
            }
         }
         if (size <= RBUFBYTES / bps)           // fits in existing I/O buffer
         {
            cb = rbuf;                          // allocated buffer (no free!)
         }
         else
         {
            cb = TxAlloc( bps, size);           // larger buffer, non-DOS only!
         }
         if (cb != NULL)                        // buffer allocation OK
         {
            ULN64      csn = 0;                 // actual clone sector number
            ULONG      bads;                    // nr of bad sectors in buffer
            ULONG      j;
            ULONG      bufusage;                // sectors used in this buffer
            ULN64      smartSkipped = 0;        // total sectors smart skipped
            ULN64      tpsn  = dfstLSN2Psn( to,   start);
            ULN64      fpsn  = dfstLSN2Psn( from, fstart);
            TXTIMER    stime = TxTmrGetNanoSecFromStart();

            if (samedisk && (fpsn <= tpsn) && ((fpsn + sectors) > tpsn))
            {
               high2low = TRUE;                 // force reverse copying order
            }                                   // to avoid overwrite of data
            if (badskip != 0)
            {
               skipbads = (badskip > size) ? (badskip - size) : 1;

               dfsFreeAreaList( &dfsa->salist); // free existing, if any
               if ((dfsa->salist = dfsNewAreaList( "Skipped on CLONE  : ")) != NULL)
               {
                  strcat( dfsa->salist->descr, dfstStoreDesc1( DFSTORE) +10);
               }
            }

            TRACES(("samedisk: %u  tpsn:0x%llx  fpsn:0x%llx  sectors:0x%llx\n",
                     samedisk,     tpsn,        fpsn,        sectors));

            TxPrint("Copy direction SN : %s %s disk, %s-clone, buffer% 6u sect",
                     (high2low) ? "Backward," : "Forward, ",
                     (samedisk) ? " same"     : "other",
                     (smartuse) ? "Smart" : "Full ", size);
            dfsSize( " = ", size, "\n");
            if (progress)
            {
               csn = (high2low) ? fstart + sectors - 1 : fstart;
               dfsX10("First sector todo : ", csn, "", "\n");

               dfsProgressInit( 0, sectors, (smartuse) ? dfsa->FsUnallocSmart : 0,
                                  "Sector:", "cloned", DFSP_BARS, DFSCL_SMART_FAST_FACTOR);

               dfsProgressShow( 0, 0, 0, NULL); // 0 of 0 done (if first is bad :-)
            }
            for ( rsn = 0;
                 (rsn < sectors) && (rc == NO_ERROR);
                  rsn += size)
            {
               bs  = min(( sectors - rsn), size);
               csn = (high2low) ? sectors - bs - rsn : rsn;

               if (smartuse) //- For alloc info to be reliable, the FROM store
               {             //- MUST be current opened object / active store!
                  for (bufusage = 0, j = 0; j < bs; j++)
                  {
                     if ((DFSFNCALL( dfsa->FsLsnAllocated, fstart + csn + j, 0, NULL, NULL) != 0))
                     {
                        bufusage++;             // count number of sectors
                     }                          // that are really in-use
                  }
                  TRACES(("Sector: 0x%llx bufusage %u out of %u\n", csn, bufusage, bs));
               }
               else
               {
                  bufusage = bs;                // whole buffer, no smarts
               }

               if (bufusage != 0)               // any used buffer sectors
               {
                  TRACES(("Clone : %u sectors, csn: 0x%llx\n", bs, csn));

                  if (skipbads == 0)            // retry with single sectors
                  {
                     rc = dfstBadSectorRead( from, fstart + csn, bs, NULL, &bads, cb);
                     if (rc != NO_ERROR)        // some read error
                     {
                        failures += bads;       // keep count in all cases ...
                        sprintf( text, "Read  error on %5u sector(s) at sector 0x%16.16llx.", bads, fstart + csn);
                        switch (dfsa->eStrategy)
                        {
                           case TXAE_QUIT:      // no action, keep bad rc
                              TxPrint( "%s  Aborting ...\n", text);
                              break;

                           case TXAE_CONFIRM:
                              if (dfsa->batch)  // CONFIRM not possible, but do NOT ignore errors!
                              {
                                 TxPrint( "%s  Aborting (batch) ...\n", text);
                                 break;
                              }
                              else
                              {
                                 dfsProgressSuspend(); // allow regular screen output
                                 if (TxConfirm( 5221, "%s\nDo you want to %s to the clone destination "
                                                                "and continue ? [Y/N]: ", text,
                                    ((flags & DFSI_MERGEBAD)) ? "skip writing this sector(s)"
                                                              : "write a fixed 0xFE pattern"))
                                 {
                                    if (TxConfirm( 5333, "Ignore errors from now on ? [Y/N]: "))
                                    {
                                       dfsa->eStrategy = TXAE_IGNORE;
                                    }
                                    dfsProgressResume( "");
                                 }
                                 else           // keep the bad RC and quit
                                 {
                                    break;
                                 }
                              }
                           case TXAE_IGNORE:    // ignore the error
                           default:
                              TxPrint( "%s  Ignore, continue ...\n", text);
                              if ((flags & DFSI_MERGEBAD) == 0)
                              {
                                 bads = 0;      // signal 'NO MERGE' to Write
                              }
                              rc = NO_ERROR;
                              break;
                        }
                     }
                     if (rc == NO_ERROR)        // With or without BADs ...
                     {
                        if (newLvmName && (strlen( newLvmName)))
                        {
                           S_LVINF     *li;
                           S_LVSIG     *ls;

                           if ((li = (S_LVINF *) TxMemStr( cb, sg_lvinf, bs * bps)) != NULL)
                           {
                              if (strncmp( li->DiskName, oldLvmName, LVM_NAME_L) == 0)
                              {
                                 TxCopy(   li->DiskName, newLvmName, LVM_NAME_L);
                                 li->LvmCRC = 0; // recalculate CRC
                                 li->LvmCRC = TxCalculateLvmCrc((BYTE *) li, SECTORSIZE);
                                 TRACES(("LVM INFO update diskname => '%s'\n", newLvmName));
                              }
                           }
                           if ((ls = (S_LVSIG *) TxMemStr( cb, sg_lvsig, bs * bps)) != NULL)
                           {
                              if (strncmp( ls->DiskName, oldLvmName, LVM_NAME_L) == 0)
                              {
                                 TxCopy(   ls->DiskName, newLvmName, LVM_NAME_L);
                                 ls->LvmCRC = 0; // recalculate CRC
                                 ls->LvmCRC = TxCalculateLvmCrc((BYTE *) ls, SECTORSIZE);
                                 TRACES(("LVM SIGN update diskname => '%s'\n", newLvmName));
                              }
                           }
                        }
                        rc = dfstBadSectorWrite( to, start + csn, bs, bads, cb);
                        if (rc != NO_ERROR)
                        {
                           sprintf( text, "Write error on %5u sector(s) at sector 0x%16.16llx.", bs, start + csn);
                           switch (dfsa->eStrategy)
                           {
                              default:
                              case TXAE_QUIT:   // no action, keep bad rc
                                 TxPrint( "%s  Aborting ...\n", text);
                                 break;

                              case TXAE_CONFIRM:
                                 if (dfsa->batch) // CONFIRM not possible, but do NOT ignore errors!
                                 {
                                    TxPrint( "%s  Aborting (batch) ...\n", text);
                                 }
                                 else
                                 {
                                    dfsProgressSuspend(); // allow regular screen output
                                    if (TxConfirm( 5222, "%s\nContinue the clone ? [Y/N]: ", text))
                                    {
                                       if (TxConfirm( 5333, "Ignore errors from now on ? [Y/N]: "))
                                       {
                                          dfsa->eStrategy = TXAE_IGNORE;
                                          sprintf( text, "Ignoring further errors, continue cloning");
                                       }
                                       else
                                       {
                                          sprintf( text, "Continue cloning");
                                       }
                                       dfsProgressResume( text);
                                       rc = NO_ERROR;
                                    }
                                 }
                                 break;

                              case TXAE_IGNORE: // ignore the error
                                 TxPrint( "%s  Ignore, continue ...\n", text);
                                 rc = NO_ERROR;
                                 break;
                           }
                        }
                     }
                  }
                  else                          // just read whole buffer once
                  {
                     rc = dfstReadLsn( from, fstart + csn, bs, cb);
                     if (rc == NO_ERROR)
                     {
                        rc = dfstWriteLsn( to, start + csn, bs, cb);
                     }
                     else                       // skip the extra sectors ...
                     {
                        failures += (bs + skipbads);
                        dfsAddListArea( dfsa->salist, fstart + csn, bs + skipbads);

                        rsn += skipbads;        // extra advance to skip
                        rc = NO_ERROR;          // Ignore read-error, continue
                     }
                  }
               }
               else                             // handle progress for skipped
               {
                  TRACES(("Smart SKIP : %u sectors, csn: 0x%llx\n", bs, csn));

                  //- only count these completely unused buffers as 'fast smart' area
                  //- since these will not from disk, and not compressed either.

                  smartSkipped += bs;

               }
               if (progress)
               {
                  dfsProgressShow( rsn + bs, smartSkipped, failures, "Bad:");
               }
               if (TxAbort())                   // user abort/escape requested
               {
                  TXTM s0;                      // text buffer

                  sprintf(   s0, "Cloned sofar is : %4.1lf %%", ((double) 100.0 *
                                             (double) (rsn) / (double) (sectors)));
                  dfstrSz64( s0, " = ", rsn,  "\n\n");

                  dfsProgressSuspend();         // allow regular screen output
                  if ((dfsa->batch) || (TxConfirm( 5223,
                      "%sAborting a cloning operation will leave an incomplete"
                      " destination area, NOT a true clone of the original!\n\n"
                      "%sAre you sure you want to ABORT the cloning ? [Y/N] : ", s0,
                       (flags & DFSI_OVERLAPD) ? "This clone is part of a MOVE "
                        "operation with OVERLAPPING 'from' and 'to' locations. "
                        "Aborting may result in a DELETED original partition!\n\n": "")))
                  {
                     rc = DFS_USER_ABORT;
                  }
                  else
                  {
                     TxCancelAbort();
                     dfsProgressResume( "");
                  }
               }
            }
            rsn   = rsn -size +bs;              // calculate real last rsn
            lastc = fstart + (high2low) ? csn : (rsn -1);

            if (progress)
            {
               dfsProgressTerm();
            }
            if (rc == DFS_PSN_LIMIT)            // reached end of disk/vol/part
            {
               if (progress)
               {
                  TxPrint("\nEnd of disk/volume\n");
               }
               rc = NO_ERROR;
            }
            dfsX10( "Last sector done  : ", lastc, "", " ");
            dfsSz64("size ", rsn, "\n");

            if ((smartuse) && (smartSkipped > 0))
            {
               TxPrint( "SmartUse optimize : %4.1lf %%   ", ((double) 100.0 * (double) smartSkipped / (double) rsn));
               dfsSz64( "  skipped ", smartSkipped, "\n");
            }
            if (failures != 0)
            {
               TXTM    badstring;

               sprintf( badstring, " sectors, %s in resulting clone:\n",
                   ((flags & DFSI_MERGEBAD) || skipbads) ? "not touched" : "0xFE pattern");
               dfsSz64( "Read failures  on : ", failures, badstring);

               if (skipbads != 0)               // failed ones are in the arealist
               {
                  dfsDisplayAreaList( dfsa->salist);

                  TxPrint( "\nYou can retry cloning the affected bad areas, without a '-skipbads' option and\n"
                             "with the '-M' sector merge option (by running the script, generated just now)\n");

                  dfsMkAreaCloneScript( dfsa->salist, d2dClone);
               }
            }
            dfsDisplayThroughput( stime, rsn, bps);
            if (cb != rbuf)                     // when not existing buffer
            {
               TxFreeMem( cb);
            }
         }
         else
         {
            rc = DFS_ALLOC_ERROR;
         }
      }
   }
   else
   {
      TxPrint( "\nError: sectorsize of 'to' and 'from' store not equal!\n");
      rc = DFS_ST_MISMATCH;
   }
   RETURN (rc);
}                                               // end 'dfsCloneStoreArea'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create a DFS script to re-clone the bad-areas in given area list
/*****************************************************************************/
static void dfsMkAreaCloneScript
(
   S_AREALIST         *list,                    // IN    area list
   char               *d2dOption                // IN    disk-2-disk options
)
{
   S_AREAELEM         *area;
   TXLN                fspec;
   FILE               *fh;

   ENTER();

   if (list && d2dOption && strlen( d2dOption))
   {
      #if defined (USEWINDOWING)
         strcpy( fspec, "*.dfs");
         if (txwSaveAsFileDialog( fspec, NULL, DFSCL_D2D_BADS_RETRY, 0, NULL, NULL,
             " Specify SCRIPT filename to save bad-area RETRY-clone commands ", fspec))
      #else
         strcpy( fspec, "DFSCL_D2D_BADS_RETRY");
      #endif
      {
         TxRepl( fspec, FS_PALT_SEP, FS_PATH_SEP); // fixup ALT separators
         TxFnameExtension( fspec, "dfs");       // make sure there is an extension
         if ((fh = fopen(  fspec, "w")) == 0)
         {
            TxPrint("Error opening script : '%s' for write\n", fspec);
         }
         else
         {
            TxPrint("Generating script : '%s', for disk-to-disk area-retry\n", fspec);
            fprintf( fh, d2dretry_header, d2dOption);

            //- generate one command for each of the bad-sector areas
            for (area = list->head; area; area = area->next)
            {
               TRACES(("Clone command, start: 0x0%llx size: 0x%llx\n", area->start, area->size));
               fprintf( fh, "clone -B -E:i -M %s -a:0x%llx,s -f:0x%llx,s -s:0x%llx,s\n",
                        d2dOption, area->start, area->start, area->size);
            }
            TxNamedMessage( TRUE, 0, " INFO: script generated ", "Retry-CLONE script '%s' generated.\n\nIt can be executed "
            #if defined (USEWINDOWING)
                                "from the 'Scripts -> Run DFSee script' menu", fspec);
            #else
                                "from the DFSee commandline using 'run'", fspec);
            #endif
            fclose(  fh);
         }
      }
   }
   VRETURN();
}                                               // end 'dfsMkAreaCloneScript'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Compare area of sectors from (other) store to current
/*****************************************************************************/
ULONG dfsCompareStoreArea
(
   ULN64               start,                   // IN    start sector
   ULN64               sectors,                 // IN    sectors to copy
   DFST_HANDLE         from,                    // IN    from store
   DFST_HANDLE         to,                      // IN    to   store
   ULN64               fstart,                  // IN    from start-sector
   ULONG               flags,                   // IN    compare options
   BOOL                progress,                // IN    display progress
   char               *oldLvmName,              // IN    old LVM disk name
   char               *newLvmName               // IN    new LVM disk name
)
{
   ULONG               rc = NO_ERROR;           // function return
   USHORT              bps;                     // sectorsize to use
   BYTE               *cb;                      // compare 1st sector buffer
   BYTE               *c2;                      // compare 2nd sector buffer
   ULONG               size;                    // buffer, variable size
   ULN64               rsn;                     // relative sector number
   ULONG               bs = 0;                  // blocksize to compare
   ULN64               lastc;                   // last compared
   DFSISTORE          *tsi;
   DFSISTORE          *fsi;                     // 1st store info
   DFSTORETYPE         tt;
   DFSTORETYPE         ft;                      // 1st store type
   BOOL                samedisk = FALSE;
   ULN64               diffsect = 0;            // nr of different sectors
   BOOL                smartuse = (flags & DFSI_SMARTUSE); // skip unused

   ENTER();

   if ((bps = dfstGetSectorSize( from)) == dfstGetSectorSize( to))
   {
      if ((tt = dfstQueryStoreType( to, &tsi, NULL)) == DFST_PHDISK)
      {
         if ((ft = dfstQueryStoreType( from, &fsi, NULL)) == DFST_PHDISK)
         {
            if (tsi->disknr == fsi->disknr)     // on same physical disk!
            {
               samedisk = TRUE;
            }
         }
      }
      #if defined (DOS32)
         //- need TWO DPMI compatible buffers for compare, so split the largest buffer (rbuf) in half
         size = dfsGetBufferSize( DFSOPTIMALBUF, (RBUFBYTES / bps) / 2);
      #else
         size = dfsGetBufferSize( DFSOPTIMALBUF, DFSMAXBUFSIZE);
      #endif
      if (samedisk && !TxaOption('b'))          // bigger buffer if same disk, faster
      {
         #if !defined (DOS32)
            if (dfsa->FsModeId != DFS_FS_HPFS)  // HPFS fragmented by 8Mb bitmaps
            {                                   // so keep smallish buffer
               size = DFSMAXBUFSIZE;            // 64 MiB (for non-DOS only!)
            }
         #endif
      }
      if (size <= (RBUFBYTES / bps) / 2)        // both fit in existing I/O buffer
      {
         cb = rbuf;                             // allocated buffer (no free!)
         c2   = rbuf + (size * bps);            // c2 uses 2nd half of rbuf
      }
      else
      {
         cb = TxAlloc( bps, size);              // 1st read buffer
         c2 = TxAlloc( bps, size);              // 2nd read buffer
      }

      if ((cb != NULL) && (c2 != NULL))         // buffer allocation OK
      {
         BOOL    allsectors = TxaOption('l');   // compare ALL and add to list
         int     i          = 0;
         ULONG   j;
         ULONG   bufusage;                      // sectors used in this buffer
         ULN64   totaluse = 0;                  // sectors used in total
         TXTIMER stime = TxTmrGetNanoSecFromStart();

         TxPrint("Compare to area on: %s disk, %s-compare, buffer% 6u sect",
                  (samedisk) ? "same" : "other",
                  (smartuse) ? "Smart" : "Full", size);
         dfsSize( " = ", size, "\n");

         dfsInitList( 0, "-w", "-d");        // with optimal list options
         if (progress)
         {
            dfsX10( "First sector todo : ", fstart, "", "\n");
            dfsProgressInit( 0, sectors, 0, "Sector:", "compared", DFSP_BARS, 0);
         }
         for ( rsn = 0;
              (rsn < sectors) && (rc == NO_ERROR) && !TxAbort();
               rsn += size)
         {
            bs  = min(( sectors - rsn), size);

            //- For allocation info to be reliable, the FROM store
            //- MUST be the current opened object or active store!

            if (smartuse)
            {
               for (bufusage = 0, j = 0; j < bs; j++)
               {
                  if ((DFSFNCALL( dfsa->FsLsnAllocated, fstart + rsn + j, 0, NULL, NULL) != 0))
                  {
                     bufusage++;                // count number of sectors
                  }                             // that are really in-use
               }
               TRACES(("Sector: 0x%llx bufusage %u out of %u\n", rsn, bufusage, bs));
               totaluse += bufusage;
            }
            else
            {
               bufusage = bs;                   // whole buffer, no smarts
            }

            if (bufusage != 0)                  // any used buffer sectors
            {
               TRACES(("Compare 1st : %u sectors, rsn: 0x%llx\n", bs, rsn));
               if ((rc = dfstReadLsn(  from, fstart + rsn, bs, cb)) == NO_ERROR)
               {
                  TRACES(("Compare 2nd : 0x%llx  size: %u\n", start + rsn, bs));
                  rc = dfstReadLsn(  to, start + rsn, bs, c2);
                  if (rc == NO_ERROR)           // compare contents
                  {
                     for (i = 0; i < bs; i++)   // check per sector
                     {
                        //- to be refined, could fixup old/new LVM diskname here

                        if (memcmp( cb + i * bps, c2 + i * bps, bps) != 0)
                        {
                           if (allsectors)
                           {
                              dfsAdd2SectorList( start + rsn + i); // add to list
                           }
                           else                 // Quit on first mismatch
                           {
                              rc = DFS_ST_MISMATCH;
                           }
                           diffsect++;          // count mismatches
                        }
                     }
                  }
               }
            }
            if (rc == NO_ERROR)
            {
               if (progress)
               {
                  dfsProgressShow( rsn + bs, 0, dfsa->snlist[0], "Mismatches:");
               }
            }
         }
         rsn   = rsn - size  + i;               // calculate real last rsn
         lastc = fstart + rsn -1;

         if (progress)
         {
            dfsProgressTerm();
         }
         if (rc == DFS_PSN_LIMIT)               // reached end of disk/vol/part
         {
            if (progress)
            {
               TxPrint("End of disk/volume\n");
            }
            rc = NO_ERROR;
         }
         dfsX10( "Last sector done  : ", lastc, "", " ");
         dfsSz64("size ", rsn, "\n");

         if ((smartuse) && ((lastc - fstart) > totaluse))
         {
            TxPrint( "SmartUse optimize : %4.1lf %%   ", (double) 100.0 -
                     ((double) 100.0 * (double) (totaluse) / (double) (lastc - fstart)));
            dfsSz64( "  used ", totaluse, "\n");
         }
         dfsDisplayThroughput( stime, rsn, bps);

         if (diffsect != 0)
         {
            dfsSz64(  "\nDifferent sectors : ", diffsect, "\n");
            if (!allsectors)
            {
               dfsX10( "First difference  : ", lastc, "", "\n");
            }
            else
            {
               TxPrint( "\nThe sector list %s\n",
                         (dfsa->snlist[0] >= (dfsa->snsize -1)) ?
                         "is full, not all differences are listed!" :
                         "holds all differences, use the 'list' command to\n"
                         "show, or 'export' to save the list to a file\n");
            }
            rc = DFS_ST_MISMATCH;
         }
         TxPrint( "\nThe compared areas are %s%sidentical%s\n",
                  (diffsect) ?  CBR   :     CBG,
                  (diffsect) ? "NOT " : "", CNN );

         if (cb != rbuf)                        // when not existing buffer
         {
            TxFreeMem( cb);
            TxFreeMem( c2);
         }
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   else
   {
      TxPrint( "\nError: sectorsize of 'to' and 'from' store not equal!\n");
      rc = DFS_ST_MISMATCH;
   }
   RETURN (rc);
}                                               // end 'dfsCompareStoreArea'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Move/copy current partition to same/other FSP-area; update Ptables and LVM
/*****************************************************************************/
ULONG dfsMovePartition
(
   char               *param                    // IN    help param or NULL
)
{
   ULONG               rc = NO_ERROR;           // function return
   BOOL                ok = TRUE;               // arg/option combinations OK
   TXA_OPTION         *opt;                     // option pointer

   ENTER();

   if (((param) && (param[0] == '?')) || TxaOption('?'))
   {
      TxShowTxt( cmd_move);
   }
   else if (SINF->p != NULL)                    // partition selected ?
   {
      DFSPARTINFO *p    = SINF->p;
      DFSPARTINFO *fspB = SINF->p->fspChain;    // freespace begin
      DFSPARTINFO *fspE = SINF->p->fspAfter;    // freespace to end
      DFSDISKINFO *dest = dfsGetDiskInfo( p->disknr);
      DFSDISKINFO *srcd = dest;                 // source == destination
      BOOL     forward  = TxaOptSet('e');
      BOOL     otherFsp = TxaOption('f');
      BOOL     copy2Fsp = TxaOption('c');

      TRACES(("p:0x%p fspB:0x%p fsbE:%p dsk:0x%p forward:%s\n",
               p, fspB, fspE, dest, (forward) ? "YES" : "NO"));

      if (otherFsp && !forward)                 // just a freespace area set
      {
         TxaOptSetItem( "-e");                  // default at start freespace
         forward = TRUE;
      }
      if ((ok = TxaOptSet('b') || forward) == FALSE)
      {
         TxPrint( "\n* You MUST specify one of the '-f', '-e' or '-b' options!");
      }
      ok &= TxaOptMutEx(  FALSE, "bf", "", NULL);
      if (ok)
      {
         ok = TxaOptMutEx(  FALSE, "be", "", NULL);
      }
      ok &= TxaOptMutEx( !TxaOption('f'), "c",  // -c only allowed with -f
              "if no '-f' freespace-area is specified", NULL);

      if ((opt = TxaOptValue('f')) != NULL) // freespace
      {
         switch (opt->type)
         {
            case TXA_NO_VAL:
            case TXA_STRING:
               TxPrint( "\n* Need a NUMBER value for '-f' freespace option!");
               ok = FALSE;
               break;

            default:
               if ((fspB = dfsGetFreeInfo((USHORT) opt->value.number)) != NULL)
               {
                  fspE = fspB;                  // begin & end same fsp area
                  dest = dfsGetDiskInfo( fspB->disknr);
                  TxPrint( "\n%s to freespace area %hu on disk %hu\n",
                          (copy2Fsp) ? "Copy" : "Move", fspB->id, fspB->disknr);
               }
               else
               {
                  TxPrint("\n* Invalid freespace specification\n");
                  ok = FALSE;
                  rc = DFS_VALUE_ERROR;         // avoid usage to be shown
               }
               break;
         }
      }

      if ((ok) && (dest != NULL))               // parameters OK sofar
      {
         DFSPARTINFO  *fspD     = (fspE) ? fspE : fspB; // one MUST be a valid freespace
         char          dstKind  = (fspD->partent.PartitionType <= DFS_FSP_GPT_STYLE) ? 'G' :
                                  (p->primary) ? 'P' : 'L';
         ULN64         oldStart = (p->primary) ?  p->basePsn : p->partPsn;
         ULN64         partSize = (p->lastPsn - oldStart + 1);
         ULONG         mbrLimit;                // MBR area limit 1 track/cyl
         ULN64         baseMin;                 // minimal new basePsn
         ULN64         baseMax;                 // maximal new basePsn
         ULN64         lastMax;                 // maximal new lastPsn
         ULN64         defOffset = 0;           // default offset to move
         ULN64         maxOffset = 0;           // maximum offset possible
         ULN64         offset    = 0;           // move offset in sectors
         TXTM          pdescr;                  // partition description text
         TXTM          mdescr;                  // move/copy description text
         TXTM          offsStr;                 // offset string
         TXTS          offsVal;                 // specified offset
         TXLN          s0, s1;                  // large text/cmd buffer
         ULN64         newStart;
         static BOOL   smartClone = TRUE;       // smart cloning, default TRUE
         static BOOL   doNotAlign;              // no alignment,  default FALSE
         ULONG         dstAlign = dfsAlignmentSectors( dstKind, fspD); // alignment for DESTINATION

         doNotAlign = TxaOptUnSet( DFS_O_ALIGN);

         mbrLimit = (dstKind == 'G') ? 0 : (p->primary) ? dest->geoSecs : dest->cSC;
         baseMin  = max( mbrLimit, (fspB) ? fspB->basePsn : oldStart);
         lastMax  = (fspE) ? fspE->lastPsn : p->lastPsn;
         baseMax  =  lastMax +1  - partSize;

         TRACES(( "\nsrcP:%s dstKind:%c oldStart:0x%llX baseMin:0x%llX baseMax:0x%x lastMax:0x%llX partSize:0x%llx\n",
                        (p->primary) ? "Pri" : "Log", dstKind, oldStart, baseMin, baseMax, lastMax, partSize));

         if (otherFsp)                          // move/copy to other fsp area
         {
            if ((lastMax - baseMin +1) < partSize) // check size of destination
            {
               TxPrint( "\n* Selected freespace area is too small to hold the partition!\n");
               ok = FALSE;
               rc = DFS_VALUE_ERROR;
            }
            else if (!dfsFreeSpaceTypeMatch(  fspD->partent.PartitionType,
                                            ((fspD->disknr == p->disknr) &&
                                            !copy2Fsp), p->primary))
            {
               TxPrint( "\n* Selected freespace area is not suited for a %s partition!\n",
                                      (p->primary) ? "PRIMARY" : "LOGICAL");
               ok = FALSE;
               rc = DFS_VALUE_ERROR;
            }
            else
            {
               maxOffset = baseMax - fspD->basePsn; // maximum offset in the fsp
            }
         }
         else                                  // move within before/after fsp
         {
            if (forward)
            {
               maxOffset = baseMax - oldStart;
            }
            else
            {
               maxOffset = oldStart - baseMin;
            }
            defOffset = maxOffset;              // move to 'other' end of fsp
         }
         if (ok)                                // OK sofar, calculate offset
         {
            sprintf( pdescr, "Part-id: %2.2hu on %s-disk %hu, "
                     "type 0x%2.2hhx %2.2s %s %-6.6s %.1lf MiB", p->id,
                     (p->tablenr == GPT_STYLE) ? "GPT" : "MBR",
                      p->disknr, p->partent.PartitionType, p->drive,
                     (p->tablenr == GPT_STYLE) ? "GPT-style" : (p->primary) ? "Primary" : "Logical",
                                    SINF->afsys, TXSMIB( p->sectors, p->bpsector));
            if (otherFsp)
            {
               sprintf( mdescr, "%s to freespace %hu (%s), on disk %hu",
                       (copy2Fsp) ? "Copy" : "Move", fspB->id, fspB->descr, fspB->disknr);
            }
            else
            {
               sprintf( mdescr, "Move towards %s of adjacent freespace",
                                 (forward) ? "END" : "BEGIN");
            }
            opt = (forward) ? TxaOptValue('e') : TxaOptValue('b');
            if ((TxaOption('P')) &&             // Interactive, prompt offset
                (!dfsa->batch) )                // unless batchmode active
            {
               sprintf( offsVal, "%.0lf", (maxOffset < 977) ? 0.0 : TXSMIB(maxOffset, p->bpsector) - 0.5);
               sprintf( offsStr, "0x%llx sectors = %s MiB", maxOffset, offsVal);
               if (opt->type == TXA_NUMBER)     // keep specified value
               {
                  sprintf( offsVal, "0x%llX,%c", opt->value.number, opt->unit);
               }
               else                             // use default move offset
               {
                  if (defOffset < 977)          // avoid '0' rounding problems
                  {
                     strcpy(  offsVal, "0");
                  }
                  else
                  {
                     sprintf( offsVal, "%.0lf", TXSMIB( defOffset, p->bpsector) - 0.5);
                  }
               }

               if (otherFsp)
               {
                  sprintf( s1, "MAX offset from start: %s\n\n"
                               "Specify offset from start freespace", offsStr);
               }
               else
               {
                  smartClone = (maxOffset >= partSize);
                  sprintf( s1, "MAX move offset: %s (rounded down)\n\n"
                               "Specify offset to move partition", offsStr);
               }
               sprintf( s0, "%s\n%s\n\n%s, use '*' for maximum\n"
                            "Value in mcs-number format, default MiB\n",
                             pdescr, mdescr, s1);

               #if defined (USEWINDOWING)
                  if ((dfsPromptOptDialog( " Specify offset and options for MOVE/COPY ",
                                             s0, 5537, 15, offsVal,
                              &smartClone, "Skip unused areas in moving (smart), can be MUCH faster!",
                              &doNotAlign, "Do NOT align target partition to cylinder/Mib boundaries'",
                                             NULL, NULL, NULL, NULL, NULL, NULL)) == NO_ERROR)
               #else
                  if (TxPrompt( 5537, 15, offsVal, s0))
               #endif
               {
                  sprintf( offsStr, "-%c:%s", ((char) ((forward) ? 'e' : 'b')), offsVal);
                  TxaOptSetItem( offsStr);      // set with prompted value
               }
               else                             // aborted from Prompt dialog
               {
                  ok = FALSE;
                  rc = DFS_NO_CHANGE;
               }
            }
            opt = (forward) ? TxaOptValue('e') : TxaOptValue('b');
            switch (opt->type)
            {
               case TXA_NO_VAL:                 // option only, use default
                  offset = defOffset;
                  break;

               case TXA_STRING:                 // string, use 'MAX' value
                  dfsX10( "\nUsing MAXIMUM offset value: ", maxOffset, "", " = ");
                  TxPrint( "%.0lf MiB\n", TXSMIB( maxOffset, p->bpsector) - 0.5);
                  offset = maxOffset;
                  break;

               default:
                  switch (opt->unit)
                  {
                     case 's':                  // sectors, OK as is
                        offset = opt->value.number;
                        break;

                     case 'h': case 't':        // heads (or tracks :-)
                        offset = opt->value.number * p->geoSecs;
                        break;

                     case 'c':                  // cylinders
                        offset = opt->value.number * p->cSC;
                        break;

                     case 'k':                  // Kilobytes
                        offset = DFSKB2SECTS( opt->value.number, p->bpsector);
                        if (doNotAlign == FALSE)
                        {                       // unless -align-, track align
                           offset = (((offset -1) / p->geoSecs) +1) * p->geoSecs;
                        }
                        break;

                     default:                   // Megabytes or Gigabytes
                        offset = DFSMB2SECTS( opt->value.number, p->bpsector);
                        TRACES(("offset from MB2S: 0x%llx\n", offset));
                        if (opt->unit == 'g')
                        {
                           offset *= 1024;      // Giga correction
                        }
                        if ((doNotAlign == FALSE) && (offset != 0))
                        {                       // unless -align-, destination align
                           offset = (((offset -1) / dstAlign) +1) * dstAlign;
                        }
                        TRACES(("offset after Align: 0x%llx\n", offset));
                        break;
                  }
                  break;
            }
            if (ok)
            {
               TxPrint( "\n%s, ", mdescr);
            }
         }
         if (ok)                                // if OK sofar, calculate start
         {                                      // and prepare CLONE/CR command
            if (otherFsp)                       // move/copy to other fsp area
            {
               newStart    = baseMin  + offset;
            }
            else
            {
               if (forward)
               {
                  newStart = oldStart + offset;
               }
               else
               {
                  newStart = oldStart - offset;
               }
            }
            if ((TxaOption('a')) && (newStart > dest->geoSecs)) // Cyl align start ?
            {
               ULONG   gross = newStart;

               newStart = (((gross -1) / dstAlign) +1) * dstAlign;
               if (newStart != gross)
               {
                  if (forward)
                  {
                     offset += (newStart - gross);
                  }
                  else if (offset > (newStart - gross))
                  {
                     offset -= (newStart - gross);
                  }
                  else                          // alignment eliminates
                  {                             // specified move offset!
                     offset = 0;
                  }
                  if (newStart > baseMax)       // out of range now ?
                  {
                     TxPrint( "alignment (-a) invalidates move!\n");
                     ok = FALSE;
                     rc = DFS_NO_CHANGE;
                  }
               }
            }
            if ((offset == 0) && (!otherFsp)) // no real move
            {
               TxPrint( "offset 0, no move required!\n");
               ok = FALSE;
               rc = DFS_NO_CHANGE;
            }
         }
         if (ok)
         {
            TXTM       dc;
            ULN64      adjustedSize = partSize; // adjusted/aligned to LOG/PRI for CR only
            ULN64      adjustedNew  = newStart; // adjusted NEW start  for CR and CLONE

            sprintf(  offsStr, "0x%llX Sectors = %.3lf Cyl = %.0lf MiB",
                      offset, ((double) offset) / p->cSC,
                     (offset < 977) ? 0.0 : TXSMIB( offset,   p->bpsector) - 0.5);
            TxPrint( "offset %s\n", offsStr);

            if ((doNotAlign == FALSE) && (p->primary == FALSE)) // logical, not aligned
            {                                   // adjust for geometry difference
               adjustedSize = partSize - srcd->geoSecs + dest->geoSecs;
            }

            if (doNotAlign == FALSE)            // no align, size in sectors
            {
               sprintf( dc, "-a:0x%llX,s -s:0x%llX,s", newStart, adjustedSize);
            }
            else                                // align size (offset already done)
            {
               ULN64   alignedSize = (((adjustedSize - 1) / dstAlign) + 1) * dstAlign;

               sprintf( dc, "-a:0x%llX,s -s:%.0lf", newStart, TXSMIB(alignedSize, p->bpsector) + 0.5);
            }

            if (fspD->partent.PartitionType <= DFS_FSP_GPT_STYLE)
            {
               TXTT      partString;            // specific partition GUID
               TXTM      guidString = {0};      // guid option text
               TXTT      typeString;
               TXTT      nameString;
               TXTT      attrString = {0};

               if (p->tablenr == GPT_STYLE)
               {
                  if (!copy2Fsp)                // for MOVE, transfer the partition GUID
                  {
                     strcpy(  partString, dfsFsUuidValueString( p->partGuid));
                     sprintf( guidString, " -guid:'%36.36s'", partString +1);
                  }
                  strcpy( typeString, dfsFsUuidValueString( p->typeGuid));
                  if (strlen( p->lvm.VoluName) < (GPT_PNAME_LEN -2))
                  {
                     sprintf( nameString, "%s:%s", (copy2Fsp) ? "C" : "M", p->lvm.VoluName);
                  }
                  else                          // straight copy
                  {
                     TxCopy( nameString, p->lvm.VoluName, GPT_PNAME_LEN);
                  }
                  if (p->attrFlags)
                  {
                     sprintf( attrString, " -attrib:'%12.12llX'", p->attrFlags);
                  }
               }
               else                             // source partition was MBR style
               {
                  strcpy(  typeString, dfsFsUuidValueString( dfsMbrType2GptGuid( p->partent.PartitionType)));
                  sprintf( nameString, "%s of %s", (copy2Fsp) ? "Copy" : "Move", p->plabel);
               }
               sprintf( s1, "cr -B -I- gpt -t:'%36.36s' -name:'%s'%s%s %s -d:%hu -o",
                        typeString +1, nameString, attrString, guidString, dc, dest->disknr);
            }
            else                                // MBR style target freespace
            {
               BOOL    toPrimary = p->primary;  // default target style, same as part

               if ((p->tablenr == GPT_STYLE) && // source was a GPT, target LOG only
                   (fspD->partent.PartitionType == DFS_FSP_LOG))
               {
                  toPrimary = FALSE;            // force to a logical
               }

               //- target type on MBR can be taken from partition (set by AddPartInfo)
               sprintf( s1, "cr -B -M -I- %s 0x%hhx %s -d:%hu%s -o",
                        (toPrimary) ? "pri" : "log",
                         p->partent.PartitionType, dc, dest->disknr,
                        (p->partent.Status & 0x80) ? " -F" : "");
            }
            TRACES(("Prepared CREATE partition command:\n\n%s\n\n", s1));

            if (p->lvmPresent)
            {
               TXTT    vname;
               TXTT    pname;
               ULONG   snp    = p->lvm.PartitId;
               ULONG   snv    = p->lvm.VolumeId;
               ULONG   snd    = dest->lvmDiskId;
               ULONG   snb    = dest->lvmBootId;
               BYTE    letter = p->lvm.letter;

               TxCopy( vname, p->lvm.VoluName, LVM_NAME_L);
               TxCopy( pname, p->lvm.PartName, LVM_NAME_L);

               if (copy2Fsp)                    // Copy, mangle name/Id
               {
                  TXTS mangle;

                  sprintf( mangle, "C%2.2u", fspD->id);
                  vname[16] = '\0';
                  strcat( vname, "-");
                  strcat( vname, mangle);
                  pname[16] = '\0';
                  strcat( pname, "-");
                  strcat( pname, mangle);

                  snp = dfsLvmIdMangle( snp, mangle);
                  snv = dfsLvmIdMangle( snv, mangle);

                  if ((opt = TxaOptValue('l')) != NULL) // drive letter given
                  {
                     if ((opt->type == TXA_STRING)  &&
                         (strlen(opt->value.string) != 0))
                     {
                        letter = (char) toupper( opt->value.string[0]);
                     }
                     else if (opt->value.number == 0) // -l- or -l:0
                     {
                        letter = 0;             // remove drive letter
                     }
                  }
                  else
                  {
                     letter = 0;                // not same letter on copy
                     TxPrint( "Copied volume will be HIDDEN (no driveletter), "
                              "use LVM cmd/menu to assign one.\n");
                  }
               }
               sprintf( s0, " -Lvm:\"-D -s%s -p:'%s' -v:'%s'%s -l:%c "
                        "-lvmsnp:0x%x -lvmsnv:0x%x -lvmsnd:0x%x -lvmsnb:0x%x\"",
                       (p->partent.PartitionType == DFS_P_WARP_LVM) ? " -J" : "",
                        pname, vname, (p->lvm.OnBmMenu) ? " -m"  : "",
                       (letter) ? letter : '0', snp, snv, snd, snb);
               strcat(  s1, s0);
            }

            //- Build the corresponding partition CLONE  command string

            if (!p->primary)                    // it is a logical ...
            {
               if (fspD->partent.PartitionType > DFS_FSP_GPT_STYLE) // MBR freespace
               {
                  adjustedNew  = newStart + dest->geoSecs; // not for GPT destination!
               }

            }
            sprintf( s0, "clone -!- -X -C- -a:0x%llX,s -f:0x%llX,s -s:0x%llX,s -d:%hu -trace-",
                                      adjustedNew, p->basePsn, p->sectors, p->disknr);
            if (smartClone)
            {
               if ((offset > partSize) || (otherFsp)) // no overlap ?
               {
                  strcat( s0, " -S");           // use SMART cloning
               }
               else
               {
                  TxPrint( "'SMART' clone option is ignored since areas are overlapping!\n");

                  //- to be refined, may set up cached alloc info before delete
                  //- to allow smart cloning even if source part is deleted
               }
            }
            TRACES(("Prepared CLONE command:\n\n%s\n\n", s0));

            sprintf( dc, "disk -q -r- %hu", srcd->disknr); // select source disk
            rc = dfsMultiCommand( dc, 0, FALSE, FALSE, FALSE); // without re-read info!

            sprintf( dc, "part -d:%hu", srcd->disknr); // show partitions as table
            rc = dfsMultiCommand( dc, 0, FALSE, FALSE, TRUE);

            TxPrint(             "Old partition disk: %hu\n", srcd->disknr);
            dfsGeoDispTransPsn(  "Old start position:", srcd->geoHeads, srcd->geoSecs, p->basePsn);
            dfsGeoDispTransPsn(  "Old  end  position:", srcd->geoHeads, srcd->geoSecs, p->basePsn + p->sectors -1);

            sprintf( dc, "disk -q -r- %hu", dest->disknr); // select destination disk
            rc = dfsMultiCommand( dc, 0, FALSE, FALSE, FALSE); // without re-read info!
            TxPrint(           "\nNew partition disk: %hu\n", dest->disknr);
            dfsGeoDispTransPsn(  "New start position:", dest->geoHeads, dest->geoSecs, adjustedNew);
            dfsGeoDispTransPsn(  "New  end  position:", dest->geoHeads, dest->geoSecs, adjustedNew + p->sectors -1);

            if      (newStart < baseMin)
            {
               TxPrint( "\n* New start position is BEFORE allowed minimum,\n"
                          "  partition would start before begin freespace!\n");
               rc = DFS_VALUE_ERROR;
            }
            else if (newStart > baseMax)
            {
               TxPrint( "\n* New start position is AFTER allowed maximum,\n"
                          "  partition-end would extend beyond freespace!\n");
               rc = DFS_VALUE_ERROR;
            }
            else                                // move offset is OK
            {
               BOOL deleteBeforeClone = TRUE;   // overlapped move -b

               if ((forward) ||                 // move is forward/otherFsp
                   (offset > partSize) ||       // or non-overlapping backward
                   (copy2Fsp) )                 // or copy, no delete at all
               {
                  deleteBeforeClone = FALSE;    // delete AFTER cloning
               }
               sprintf( dc, " using offset %s", offsStr);
               TRACES(( "offstr:'%s' len %u dc:'%s'\n", offsStr, strlen(offsStr), dc));
               if ((dfsa->batch) || (TxConfirm( 5539,
                   "Estimated time to complete will be shown on the status/progress\n"
                   "line near the bottom of the screen after a few seconds (ETA) ...\n"
                   "%s%s\n\n%s%s ? [Y/N] : ", (partSize < DFSECT_2GIB) ? "\n" :
                   "Note: COPY/MOVE of a partition this large may take a long time,\n"
                   "(minutes, perhaps hours) depending on exact size and system speed ...\n\n",
                    pdescr, mdescr, (offset) ? dc : "")))
               {
                  USHORT destDisk = dest->disknr;  //- dest-> invalid after DELETE/CR!
                  USHORT oldPart  = p->id;         //- partition to be deleted

                  if (deleteBeforeClone)
                  {
                     rc = dfsDeleteAndSelectDisk( oldPart, destDisk);
                     if (rc == NO_ERROR)
                     {
                        strcat( s0, " -o");     // add OVERLAP abort warning
                     }                          // to CLONE abort confirmation
                  }
                  else
                  {
                     //- only here smartclone (-S) would be acceptable ...
                  }
                  if (rc == NO_ERROR)
                  {
                     if (!TxaOptUnSet('C'))     // unless -C- "no contents copy"
                     {
                        TxPrint( "\nCLONE the partition data to the new location, may take a long time ...");
                        rc = dfsMultiCommand( s0, 0, TRUE, FALSE, TRUE); // CLONE
                     }
                     if (rc == NO_ERROR)        // delete before create ==> same PID
                     {                          // and no temporary extra primaries
                        if (!TxaOption('c') && !deleteBeforeClone)
                        {
                           rc = dfsDeleteAndSelectDisk( oldPart, destDisk);
                        }
                     }
                     if (rc == NO_ERROR)
                     {
                        TxPrint( "\nRECREATE the partition with same properties in the new location ...");
                        rc = dfsMultiCommand( s1, 0, TRUE, FALSE, TRUE); // CR
                        if (rc == NO_ERROR)
                        {
                           sprintf( s0, "%s completed successfully.\n", (copy2Fsp) ? "COPY" : "MOVE");
                           sprintf( s1, "%s\n%s", s0, cloneReboot);
                           TxPrint( "\n%s", s0);
                           TxNamedMessage( !dfsa->batch, 5533, " INFO: copy/move completed ", s1);

                           #if   defined (WIN32)
                              // Always issue warning if booted system is windows
                           #else
                              TRACES(("WIN candidates: %hu\n", dfsa->winCandidates));
                              // Any primary FAT32/NTFS or primary FAT16 + FAT32/NTFS
                              if (dfsa->winCandidates > 1) // likely Windows installed
                           #endif
                              {
                                 TxNamedMessage( !dfsa->batch, 0, " INFO: Windows driveletters ", winDriveLetters);
                              }
                        }
                        else
                        {
                           TxPrint( "\nCREATE of partition in new location failed!\n");
                        }
                     }
                     else
                     {
                        TxPrint( "\nCLONE partition failed!\n");
                     }
                     sprintf( dc, "part -d:%hu", destDisk); // show partitions as table
                     rc = dfsMultiCommand( dc, 0, FALSE, FALSE, TRUE);
                  }
                  else if (rc != DFS_NO_CHANGE)
                  {
                     TxPrint( "\nDelete old partition failed!\n");
                  }
               }
               else
               {
                  rc = DFS_NO_CHANGE;
               }
            }
         }
      }
   }
   else
   {
      TxPrint( "\n* Moving requires a harddisk partition to be selected.\n"
               "  Use the 'part' command or menu to select a partition first!");
      ok = FALSE;
   }
   if (!ok && (rc != DFS_NO_CHANGE) && (rc != DFS_VALUE_ERROR))
   {
      TxShowTxt( cmd_move);                     // give usage
      rc = DFS_VALUE_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsMovePartition'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// Delete a partition, read disk-info and (re) select disk;
// invalidates disk & part pointers !!!
/*****************************************************************************/
static ULONG dfsDeleteAndSelectDisk
(
   USHORT              part,                    // IN    partition to delete
   USHORT              disk                     // IN    disk to (re) select
)
{
   ULONG               rc = NO_ERROR;           // function return
   TXTM                dc;

   ENTER();

   TxPrint( "\nDELETE the partition to be MOVEd from the original location ...");
   sprintf( dc, "delete -B %hu", part);
   rc = dfsMultiCommand( dc, 0, TRUE, FALSE, TRUE);
   if (rc == NO_ERROR)                          // reselect destination disk
   {
      sprintf( dc, "disk -q %hu", disk);
      rc = dfsMultiCommand( dc, 0, FALSE, FALSE, FALSE);
   }
   RETURN (rc);
}                                               // end 'dfsDeleteAndSelectDisk'
/*--------------------------------------------------------------------------------------------*/

