//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// FDISK callback-functions for LVM related updates and display
//
// Author: J. van Wijk
//
// JvW  30-01-2007 Initial version, split-off from DFSCFDSK
//

#include <txlib.h>                              // TX library interface

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfsupart.h>                           // FDISK partition functions
#include <dfsafdsk.h>                           // FDISK display & analysis
#include <dfscfdsk.h>                           // FDISK callback and commands
#include <dfsdflvm.h>                           // FDISK LVM dialog functions
#include <dfsufdsk.h>                           // FDISK utility functions
#include <dfswin.h>                             // windowing functions
#include <dfsver.h>                             // version and naming info
#include <dfsosapi.h>                           // OS specific stuff


// Set new disk-level information for LVM-info sector
static void dfsInitLvmDskInfo
(
   S_LVINF            *sd,                      // IN    LVM info sector
   DFSDISKINFO        *d                        // IN    Target disk info
);

// Initialize LVM-info sector from existing LVM-signature sector when found
static BOOL dfsInitInfoFromSig                  // RET   LVM-sig found & used
(
   S_LVINF            *sd,                      // IN    LVM info sector
   ULONG               index,                   // IN    index to be updated
   FDSK_CB_PHASE       phase,                   // IN    callback phase
   DFSPARTINFO        *p                        // IN    Partition info
);

// Set new disk-level information and init BBR area for LVM-signature sector
static ULONG dfsInitLvmSigArea
(
   S_LVSIG            *sg,                      // IN    LVM signature sector
   DFSDISKINFO        *d,                       // IN    Target disk info
   DFSPARTINFO        *p                        // IN    Partition info
);

// Create and write a fake EBR sector for the LVM reported partition size
static ULONG dfsMkFakeEbrSector
(
   DFSPARTINFO        *p,                       // IN    partition information
   ULONG               fakePsn,                 // IN    sector to write to
   ULONG               pStart,                  // IN    partition start PSN
   ULONG               pSize                    // IN    reported part size
);

// Create and write the sectors for sec/prim Bad Block Relocation feature/table
static ULONG dfsMkBadBlockFeature
(
   ULONG               featPsn,                 // IN    1st sector to write to
   ULONG               replLSN,                 // IN    1st replacement LSN
   ULONG               tableSize,               // IN    BBR replacement sectors
   ULONG               tableSect                // IN    BBR table sectors
);

// Create and write the four sectors for sec/prim Drive Linking feature/table
static ULONG dfsMkDriveLinkFeature
(
   ULONG               featPsn,                 // IN    1st sector to write to
   ULONG               diskId,                  // IN    LVM disk ID
   ULONG               partId                   // IN    LVM part ID
);

// Update lvmsnp and lvmsnd at first match in LVM Drive-linking sector at PSN
static ULONG dfsFdskLvmSetDriveLink
(
   ULONG               dLink,                   // IN    Drive-link sector PSN
   ULONG               newSnD,                  // IN    new LVM-this-disk-id
   ULONG               newSnP,                  // IN    new LVM-partition-id
   ULONG               oldSnP                   // IN    current partition ID
);


/*****************************************************************************/
// Implement the 'dfsFdskLvmWalk' callback display for MBR/EBR or LVM sectors
/*****************************************************************************/
ULONG dfsFdskLvmWalk                            // RET PENDING  => call again
(                                               //     NO_ERROR => all done
   BYTE               *sec,                     // INOUT sectors with BM info
   FDSK_CB_PHASE       phase,                   // IN    callback phase
   FDSK_CB_INFO       *cbp                      // IN    operation info
)
{
   ULONG               rc = NO_ERROR;           // function return
   BYTE                st;

   ENTER();
   TRACES(("Phase:%u Related:%u Confirmed:%u Option:%u Flag:%u  Disk:%hu thisSn:%llx\n",
            phase, cbp->related, cbp->confirmed, cbp->option, cbp->flag, cbp->disknr, cbp->thisSn));
   if ((sec != NULL) && (phase == FDSK_PERFORM))
   {
      switch (st = dfsIdentifySector( cbp->sn, 0, sec))
      {
         case ST_MASTR:
         case ST_EXTBR:
         case ST_LVINF:
         case ST_LVSIG:
         case ST_LVREM:
         case ST_LVDLF:
         case ST_LVDLT:
         case ST_LVBBF:
         case ST_LVBBT:
            st = 0;                             // use autodetect
            TxPrint("\n");
            rc = dfsReadAnDisplay( cbp->sn, 0, &st);
            dfsAdd2SectorList( cbp->sn);        // add to sector list
            break;

         default:
            break;
      }
   }
   RETURN (rc);
}                                               // end 'dfsFdskLvmWalk'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Implement the 'Update info' callback operation on LVM-info/signature sector
// Note: called for EVERY LVM sector on the target disk, not just for one part!
/*****************************************************************************/
ULONG dfsFdskLvmSetInfo                         // RET PENDING  => call again
(                                               //     NO_ERROR => all done
   BYTE               *sec,                     // INOUT a MBR/EBR/LVM/BR sect
   FDSK_CB_PHASE       phase,                   // IN    callback phase
   FDSK_CB_INFO       *cbp                      // IN    callback parameters
)
{
   ULONG               rc = NO_ERROR;           // function return
   S_LVINF            *sd = (S_LVINF *) sec;    // as LVM info sector
   S_LVSIG            *sg = (S_LVSIG *) sec;    // as LVM signature sector
   DFSPARTINFO        *p  = (DFSPARTINFO *) cbp->more; // could be NULL !!
   DFSDISKINFO        *d  = dfsGetDiskInfo( cbp->disknr);
   S_LVPART           *lp = NULL;               // LVM part info
   ULONG               pi;                      // partition index
   ULONG               pnew     = MAXLVINF;     // index of free slot
   BOOL                changes  = FALSE;
   DEVICE_STATE        v_screen = TxScreenState( DEVICE_TEST);
   BOOL                quiet    = FALSE;
   ULONG               size     = 0;
   BYTE                sectype;
   BOOL                matching_lvm_sector  = FALSE;
   BOOL                matching_sig_sector  = FALSE;
   ULONG               refId;
   ULONG               refI2;                   // second reference Id

   ENTER();
   TRACES(("Phase:%u Related:%u Confirmed:%u Option:%u Flag:%u  Disk:%hu thisSn:%llx\n",
            phase, cbp->related, cbp->confirmed, cbp->option, cbp->flag, cbp->disknr, cbp->thisSn));

   #if defined (DEV32)
   TRACES(("cbp->lvmSync now : %s  syncName:'%s' lvm.letter: %d\n",
           (cbp->lvmSync) ? "TRUE" : "FALSE", cbp->syncName, cbp->syncLetter));
   #endif

   sectype = dfsIdentifySector( cbp->sn, 0, sec);
   if (p != NULL)                               // specific partition update
   {
      size = p->sectors;                        // size from part-tables

      if ((cbp->sn == (p->lastPsn)) &&
          ((p->partent.PartitionType == DFS_P_WARP_LVM) || (cbp->doAllBBR)))
      {
         if (cbp->cbOption3)                    // Clear the BBR sector ?
         {
            changes    = TRUE;                  // make sure it is written
         }
         matching_sig_sector = TRUE;            // right LVM-sign sector found
         TRACES(( "LVM sign at: %llx, p-base: %llx\n", cbp->sn, p->basePsn));
      }
      else if (cbp->sn == (p->partPsn + p->geoSecs -1))
      {
         matching_lvm_sector = TRUE;            // right LVM-info sector found
         TRACES(( "LVM info at: %llx, p-base: %llx\n", cbp->sn, p->basePsn));
      }
   }
   else                                         // disk-level operation
   {
      if (cbp->sn == (d->geoSecs -1))           // MBR track LVM info sector
      {
         matching_lvm_sector = TRUE;            // right LVM-info sector found
         changes = TRUE;                        // make sure it is written back
         TRACES(( "LVM info at: %llx (MBR track)\n", cbp->sn));
      }
   }

   if (sectype == ST_LVINF)                     // existing LVM info sector
   {
      if (p && matching_lvm_sector)             // guard against p == NULL
      {
         if ((p->primary == FALSE) &&           // info for a logical, and
             (cbp->cbOption2))                  // -D option, default values
         {                                      // in fields to avoid garbage
            TRACES(( "Existing LVM-info for logical and '-D', reset to defaults ...\n"));
            memset( sd, 0, SECTORSIZE);
            memcpy( sd->Signature, sg_lvinf, SG_LVINF);
            dfsInitLvmDskInfo( sd, d);          // set disk-level info
            pnew = 0;                           // use first slot

            if ((cbp->cbOption3 == FALSE) &&    // unless -Clear, try SIG-recovery
                (dfsInitInfoFromSig(sd, 0, phase, p)))
            {
               changes    = TRUE;
               strcpy( cbp->cbOptStr1, "recover"); // propagate 'change' to PERFORM phase
            }
         }
         else                                   // find suitable entry
         {
            TRACES(( "Existing LVM-info sector, check/delete DLAT ...\n"));
            for ( pi = 0; pi < MAXLVINF; pi++)
            {
               if (sd->part[pi].sectors != 0)   // valid existing DLAT entry
               {
                  if ((p != NULL) && (p->basePsn == sd->part[pi].partPsn)) //- same PSN
                  {
                     TRACES(("Match '%*.*s' to part: %hu\n", LVM_NAME_L,
                              LVM_NAME_L, sd->part[pi].PartName, p->id));
                     lp = &( sd->part[pi]);     // LVPART to work on

                     if (cbp->cbOption2)        // Delete existing DLAT-entry
                     {                          // before setting new values
                        memset( lp, 0, sizeof( S_LVPART));
                        pnew = pi;
                        lp   = NULL;            // no info at this point
                        TRACES(("DLAT-entry %hu has been cleared\n", pi));

                        if ((cbp->cbOption3 == FALSE) && // unless -Clear, try SIG-recovery
                            (dfsInitInfoFromSig(sd, pi, phase, p)))
                        {
                           changes    = TRUE;
                           strcpy( cbp->cbOptStr1, "recover"); // propagate 'change' to PERFORM phase
                        }
                     }
                  }
               }
               else if (pnew == MAXLVINF)
               {
                  pnew = pi;                    // use empty slot for new
               }
            }
         }
      }
      TRACES(( "New entry index %u\n", pnew));
   }
   else if ((sectype == ST_LVSIG) ||             // existing LVM sign sector
            (sectype == ST_LVREM)  )             // or BBR V-deleted one
   {
      TRACES(( "Existing LVM-sign sector, no initialization ...\n"));

      //- to be refined, no further preparation needed ?
   }
   else if (matching_lvm_sector)                // not an LVM info sector (yet)
   {
      TRACES(( "Not an LVM sector, create new DLAT ...\n"));
      memset( sd, 0, SECTORSIZE);
      memcpy( sd->Signature, sg_lvinf, SG_LVINF);

      dfsInitLvmDskInfo( sd, d);                // set disk-level info

      pnew    = 0;                              // use first slot

      if ((p != NULL) && (!TxaOptUnSet('r')) && // unless -r-, try SIG-recovery
          (dfsInitInfoFromSig( sd, 0, phase, p)))
      {
         changes    = TRUE;
         strcpy( cbp->cbOptStr1, "recover");    // propagate 'change' to PERFORM phase
      }
      else                                      // allow compatibility only
      {
         //- set new HPFS/JFS partitions to installable by default!
         if ((p != NULL) && (p->partent.PartitionType == DFS_P_INST_FS))
         {
            sd->part[ pnew].Installable = (BYTE) TRUE;
         }
         TxPrint( "LVM-DLAT sector at: 0x0%llx (%cBR track) created new\n",
                   cbp->sn, (cbp->sn == (d->geoSecs -1)) ? 'M' : 'E');
      }
      sectype = ST_LVINF;
   }
   else if (matching_sig_sector)                // not an LVM sign sector (yet)
   {
      TRACES(( "Not an LVM BBR sector, create new and init BBR area ...\n"));
      memset( sg, 0, SECTORSIZE);
      memcpy( sg->Signature, sg_lvsig, SG_LVSIG);

      rc = dfsInitLvmSigArea( sg, d, p);        // set disk info and BBR area
      if (rc == NO_ERROR)
      {
         sectype = ST_LVSIG;
      }
   }

   if ((p != NULL) && (lp == NULL) && (matching_lvm_sector)) // no info, ONE part
   {
      if (pnew == MAXLVINF)                     // none yet, find obsolete one
      {
         for ( pi = 0; (pi < MAXLVINF) && (pnew == MAXLVINF); pi++)
         {
            if (dfsFdskDlatEntry2Part( &(sd->part[pi]), p->disknr) == NULL)
            {
               pnew = pi;                       // this is an obsolete entry
            }
         }                                      // will use entry-5 (invalid)
      }                                         // when no obsolete one found

      lp = &( sd->part[pnew]);                  // new LVPART to work on

      if ((strlen(cbp->pname)   == 0) &&        // no name yet, and no delete
          (strlen(lp->PartName) == 0) &&        // not initialized yet
          (TxaOption('V')  ||                   // and no default create or
          (cbp->cbOption2  == FALSE)  ))        // a non -D sig-recovery
      {                                         // Note: should fix P#933
         if (p->partent.PartitionType == DFS_P_BOOTMGR)
         {
            sprintf( cbp->pname, "BootManager - P%2.2hu", p->id);
         }
         else
         {
            sprintf( cbp->pname, "P%2.2hu - ", p->id);
            dfstrSizeXiB( cbp->pname, "", size, p->bpsector, "");
         }
         TRACES(( "New default cbp->pname: '%s'\n", cbp->pname));
      }
      if ((strlen(cbp->vname)   == 0) &&        // no volume name yet, and
          (strlen(lp->VoluName) == 0) &&        // not initialized yet and
          (TxaOption('V')))                     // default Vol-name wanted ?
      {
         if (p->partent.PartitionType == DFS_P_BOOTMGR)
         {
            strcpy( cbp->vname, "OS2");
         }
         else
         {
            TXTS          descr;                // partition type description

            dfsPartTypeDescription( p->partent.PartitionType, descr);
            TxStrip( descr, descr, 0, ' ');     // remove trailing spaces

            sprintf( cbp->vname, "V%2.2hu - %s", p->id, descr);
         }
      }

      lp->sectors = p->sectors;                 // Raw size, will be updated!
      lp->partPsn = p->basePsn;                 // Must be correct, used to
                                                // identify on PERFORM phase!
      cbp->cbOption1 = TRUE;                    // start/size change indicator
   }

   if ((rc == NO_ERROR) && (lp != NULL) && (matching_lvm_sector))
   {
      TRACES(( "Work on LVM-info sector ...\n"));

      cbp->result = TRUE;                       // signal LVM info found

      switch (phase)
      {
         case FDSK_PREPARE:
            nav.down = cbp->thisSn;             // 'd' to LVM-info
            nav.xtra = p->partPsn;              // 'x' to related MBR/EBR

            if (!dfsa->batch && !dfsDialogAppropriate())
            {
               TxPrint("\nLVMinfo on disk %u : %8.8x   partition: %2.2hu",
                                   cbp->disknr, (ULONG) cbp->thisSn, p->id);
            }
            if ((cbp->cbOption1 == FALSE)     &&
                ((lp->sectors   != size)      ||
                 (lp->partPsn   != p->basePsn)))
            {
               if ((dfsa->batch) || dfsDialogAppropriate() ||
                    TxConfirm( 5108,
                   "The size or startsector in the LVM information "
                   "do not match the information from the partition "
                   "tables. Synchronize the LVM values ? [Y/N] : "))
               {
                  cbp->cbOption1 = TRUE;
               }
            }
            #if defined (DEV32)
               p->lvmPresent = TRUE;            // will be, and must be for SyncDesired
               cbp->lvmSync  = dfsLvmSyncDesired( p);
               TRACES(("cbp->lvmSync set : %s\n", (cbp->lvmSync) ? "TRUE" : "FALSE"));
            #endif
            #if defined (USEWINDOWING)
               if (dfsDialogAppropriate())
               {
                  if (cbp->option && cbp->cbOption3 && cbp->doAllBBR &&
                      (p->partent.PartitionType != DFS_P_WARP_LVM))
                  {
                     cbp->cbOptNum2 = 1;        // Force one change (BBR erase)
                  }
                  // else (dialog needed, when creating new LVM info with defaults!)
                  {
                     rc = dfsFdskLvmSetDialog( cbp, sd, lp);
                  }
                  if (cbp->cbOptNum2 == 0)      // no changes ?
                  {
                     quiet = TRUE;
                     TxScreenState( DEVICE_OFF);
                  }
               }
            #endif                              // USEWINDOWING
            if ((rc != TXDID_CANCEL) &&         // dialog not canceled
                (!TxaOption('R')     ))         // not LVM -R read-only mode
            {
               if ((dfsa->batch    == FALSE) &&
                   (cbp->confirmed == FALSE) )
               {
                  TxPrint("\n Partition start  : 0x%8.8x "
                          "for partition %02u  => ", (ULONG) lp->partPsn, p->id);
                  if (lp->partPsn != p->basePsn)
                  {
                     if (cbp->cbOption1)        // change will be made
                     {
                        TxPrint("0x%s%8.8x%s", CBG, (ULONG) p->basePsn, CNN);
                        changes = TRUE;
                     }
                     else
                     {
                        TxPrint("0x%s%8.8x%s  NOT changed!", CBR, (ULONG) p->basePsn, CNN);
                     }
                  }
                  else
                  {
                     TxPrint( "no change");
                  }
                  TxPrint("\n Partition size   : 0x%8.8x = %9.1lf MiB   => ",
                      lp->sectors, TXSMIB( lp->sectors, p->bpsector));
                  if (lp->sectors != size)
                  {
                     if (cbp->cbOption1)        // change will be made
                     {
                        TxPrint("0x%s%8.8x%s  = %9.1lf MiB", CBG, size, CNN,
                                               TXSMIB( size, p->bpsector));
                        changes = TRUE;
                     }
                     else
                     {
                        TxPrint("0x%s%8.8x%s  = %9.1lf MiB!", CBR, size, CNN,
                                               TXSMIB( size, p->bpsector));
                     }
                  }
                  else
                  {
                     TxPrint( "no change");
                  }

                  TxPrint("\n Volume    name   : %-20.20s         => ", lp->VoluName);
                  if (strlen(cbp->vname) != 0)
                  {
                     if (cbp->vname[0] == '-')  // remove
                     {
                        TxPrint( "%snone %s%s",          CBG, (strlen(lp->VoluName)) ?
                                 "(delete volume)" : "", CNN);
                     }
                     else
                     {
                        TxPrint( "%s%s%s   %s",  CBY, cbp->vname, CNN,
                                (lp->VolumeId == 0) ? "(create new)" : "");
                     }
                     changes = TRUE;
                  }
                  else
                  {
                     if (lp->VoluName[LVM_NAME_L -1] != 0) // too long
                     {
                        strncpy( cbp->vname, lp->VoluName, LVM_NAME_L);
                        cbp->vname[LVM_NAME_L -1] = 0;
                        TxPrint( "%s%s%s   (shortened)",  CBY, cbp->vname, CNN);
                        changes = TRUE;
                     }
                     else
                     {
                        TxPrint( "no change");
                     }
                  }
                  TxPrint("\n Partition name   : %-20.20s         => ", lp->PartName);
                  if (strlen(cbp->pname) != 0)
                  {
                     if (cbp->pname[0] == '-')  // remove
                     {
                        TxPrint( "%snone %s%s",      CBG, (strlen(lp->PartName)) ?
                                 "(anonymous)" : "", CNN);
                     }
                     else
                     {
                        TxPrint( "%s%s%s   %s",  CBC, cbp->pname, CNN,
                                (lp->PartitId == 0) ? "(create new)" : "");
                     }
                     changes = TRUE;
                  }
                  else
                  {
                     if (lp->PartName[LVM_NAME_L -1] != 0) // too long
                     {
                        strncpy( cbp->pname, lp->PartName, LVM_NAME_L);
                        cbp->pname[LVM_NAME_L -1] = 0;
                        TxPrint( "%s%s%s   (shortened)",  CBY, cbp->pname, CNN);
                        changes = TRUE;
                     }
                     else
                     {
                        TxPrint( "no change");
                     }
                  }
                  TxPrint("\n Drive letter     : %c%c%27.27s=> ",
                                       (lp->letter != 0) ? lp->letter : '-',
                                       (lp->letter != 0) ? ':' : '-', "");
                  if (strlen(cbp->bname) != 0)
                  {
                     if      (cbp->bname[0] == '-')  // remove
                     {
                        TxPrint( "%snone (hidden)%s",   CBG, CNN);
                     }
                     else if (cbp->bname[0] == '*')  // dynamic
                     {
                        TxPrint( "%s* (auto assign)%s",   CBG, CNN);
                     }
                     else
                     {
                        TxPrint( "%s%s:%s", CBG, cbp->bname, CNN);
                     }
                     changes = TRUE;
                  }
                  else
                  {
                     TxPrint( "no change");
                  }
                  TxPrint("\n On BMGR menu     : %s%26.26s=> ",
                                           (lp->OnBmMenu) ? "Yes" : "No ", "");
                  if (cbp->ntype != 0xff)
                  {
                     TxPrint( "%s%s%s", CBM, (cbp->ntype) ? "Yes" : "No ", CNN);
                     changes = TRUE;
                  }
                  else
                  {
                     TxPrint( "no change");
                  }
                  TxPrint("\n Installable flag : %s%26.26s=> ",
                                        (lp->Installable) ? "Yes" : "No ", "");
                  if (cbp->stype != 0xff)
                  {
                     TxPrint( "%s%s%s", CBM, (cbp->stype) ? "Yes" : "No ", CNN);
                     changes = TRUE;
                  }
                  else
                  {
                     TxPrint( "no change");
                  }

                  TxPrint("\n Part/Vol/Disk Id : 0x%8.8x/%8.8x/%8.8x => ",
                          lp->PartitId, lp->VolumeId, sd->ThisDiskId);
                  if ((TxaOptSet( DFS_O_LVMSNP)) ||
                      (TxaOptSet( DFS_O_LVMSNV)) ||
                      (TxaOptSet( DFS_O_LVMSND)) ||
                      (TxaOptSet( DFS_O_LVMSNB))  )
                  {
                     TxPrint( "%sForced update%s\n", CBC, CNN);
                     changes = TRUE;
                  }
                  else
                  {
                     TxPrint( "no change\n");
                  }

                  if (quiet)                    // end of suppressed dialog
                  {                             // values when not changed.
                     TxScreenState( v_screen);  // reset explicit quiet
                  }

                  if (cbp->option && cbp->cbOption3 && cbp->doAllBBR &&
                      (p->partent.PartitionType != DFS_P_WARP_LVM))
                  {
                     TxPrint("\n LVM type Sig/BBR : Erase for non LVM type 0x%2.2hx  => Force %sCOMPATIBILITY%s Volume\n",
                               p->partent.PartitionType, CBC, CNN);
                     changes = TRUE;
                  }

                  if (changes)
                  {
                     TxPrint( "\n");
                     if (TxConfirm( 5109,
                         "Cmd: %s\n\nUpdate LVM values for partition %2.2hu "
                         "to displayed values ? [Y/N] : ", cbp->string, p->id))
                     {
                        cbp->confirmed = TRUE;  // propagate to LVSIG update
                     }
                     else                       // not confirmed
                     {
                        dfsa->explain = TRUE;
                        rc = DFS_CMD_FAILED;    // aborted (no update wanted)
                     }
                  }
                  else
                  {
                     //- TxPrint( "No changes to LVM information for partition %2.2hu.\n", p->id);
                     rc = DFS_CMD_FAILED;
                  }
               }
            }
            else                                // canceled or read-only
            {
               if (TxaOption('R'))
               {
                  TxPrint( "LVM Read-only mode, no changes made!\n");
                  rc = DFS_CMD_FAILED;
               }
               rc = DFS_CMD_FAILED;             // CANCEL is not-confirmed
            }
            if (quiet)                          // if not done yet ...
            {
               TxScreenState( v_screen);        // reset explicit quiet
               quiet = FALSE;
            }
            break;

         case FDSK_PERFORM:
            TxPrint("\nLVMinfo on disk %u : 0x%8.8x,  partition: %2.2hu\n",
                                cbp->disknr, (ULONG) cbp->thisSn, p->id);

            if (strcmp(cbp->cbOptStr1, "recover") == 0) // recovered info present
            {
               TxPrint( "Recovered info from signature sector applied ...\n");
               changes = TRUE;
            }
            if (lp->sectors != size)
            {
               TxPrint(" Partition size   : 0x%8.8x = %8.1lf MiB    => ",
                   lp->sectors, TXSMIB( lp->sectors, p->bpsector));
               if (cbp->cbOption1)              // change will be made
               {
                  TxPrint("0x%s%8.8x%s  = %8.1lf Mb\n", CBG, size, CNN,
                                         TXSMIB( size, p->bpsector));
                  lp->sectors = size;
                  changes = TRUE;
               }
               else
               {
                  TxPrint("0x%s%8.8x%s  = %8.1lf Mb!\n", CBR, size, CNN,
                                         TXSMIB( size, p->bpsector));
               }
            }
            if (lp->partPsn != p->basePsn)
            {
               TxPrint(" Partition start  : 0x%8.8x%19.19s=> ", (ULONG) lp->partPsn, "");
               if (cbp->cbOption1)              // change will be made
               {
                  TxPrint("0x%s%8.8x%s\n", CBG, (ULONG) p->basePsn, CNN);
                  lp->partPsn = p->basePsn;
                  changes = TRUE;
               }
               else
               {
                  TxPrint("0x%s%8.8x%s  NOT changed!\n", CBR, (ULONG) p->basePsn, CNN);
               }
            }
            if (TxaOptSet( DFS_O_LVMSNP))       // force specific Part-id
            {
               refId = lp->PartitId;
               lp->PartitId = TxaOptNum( DFS_O_LVMSNP, "Partition-S#", refId);
               TxPrint(" LVM partition Id : 0x%8.8x%19.19s=> 0x%8.8x\n", refId,
                       "", lp->PartitId);
               changes = TRUE;
            }
            if (TxaOptSet( DFS_O_LVMSNV))       // force specific Vol-id
            {
               refId = lp->VolumeId;
               lp->VolumeId = TxaOptNum( DFS_O_LVMSNV, "Volume-S#",    refId);
               TxPrint(" LVM volume    Id : 0x%8.8x%19.19s=> 0x%8.8x\n", refId,
                       "", lp->VolumeId);
               changes = TRUE;
            }
            if (TxaOptSet( DFS_O_LVMSND))       // force specific Disk-id
            {
               refId = sd->ThisDiskId;
               sd->ThisDiskId = TxaOptNum( DFS_O_LVMSND, "ThisDisk-S#", refId);
               TxPrint(" LVM this disk Id : 0x%8.8x%19.19s=> 0x%8.8x\n", refId,
                       "", sd->ThisDiskId);
               changes = TRUE;
            }
            if (TxaOptSet( DFS_O_LVMSNB))       // force specific Boot-id
            {
               refId = sd->BootDiskId;
               sd->BootDiskId = TxaOptNum( DFS_O_LVMSNB, "BootDisk-S#", refId);
               TxPrint(" LVM boot disk Id : 0x%8.8x%19.19s=> 0x%8.8x\n", refId,
                       "", sd->BootDiskId);
               changes = TRUE;
            }
            if (strlen(cbp->vname) != 0)
            {
               TxPrint(" Volume    name   : %-20.20s         => ", lp->VoluName);
               if (cbp->vname[0] == '-')        // remove
               {
                  TxPrint( "%snone %s%s\n",        CBG, (strlen(lp->VoluName)) ?
                           "(delete volume)" : "", CNN);

                  memset( lp->VoluName, 0,          LVM_NAME_L);
                  lp->VolumeId = 0;
               }
               else
               {
                  TxPrint( "%s%s%s\n",  CBY, cbp->vname, CNN);
                  memcpy( lp->VoluName, cbp->vname, LVM_NAME_L);

                  if (!TxaOptSet( DFS_O_LVMSNV)) // unless specific Vol-id set
                  {
                     lp->VolumeId = TxCalculateLvmCrc( (BYTE *) cbp->vname, LVM_NAME_L);
                  }
               }
               changes = TRUE;
            }
            if (strlen(cbp->pname) != 0)
            {
               TxPrint(" Partition name   : %-20.20s         => ", lp->PartName);
               if (cbp->pname[0] == '-')        // remove
               {
                  TxPrint( "%snone %s%s\n",    CBG, (strlen(lp->PartName)) ?
                           "(anonymous)" : "", CNN);

                  memset( lp->PartName, 0,          LVM_NAME_L);
               }
               else
               {
                  TxPrint( "%s%s%s\n",  CBC, cbp->pname, CNN);
                  memcpy( lp->PartName, cbp->pname, LVM_NAME_L);
                  if (lp->PartitId == 0)        // create new unique Id
                  {
                     lp->PartitId = p->basePsn + (((ULONG) p->disknr) << 28);
                  }
               }
               changes = TRUE;
            }
            if (strlen(cbp->bname) != 0)
            {
               BYTE    newletter;

               TxPrint(" Drive letter     : %c%c%27.27s=> ",
                                  (lp->letter != 0) ? lp->letter : '-',
                                  (lp->letter != 0) ? ':' : '-', "");
               if      (cbp->bname[0] == '-')        // remove
               {
                  TxPrint( "%snone (hidden)%s\n",   CBG, CNN);
                  newletter = (BYTE) 0;
               }
               else if (cbp->bname[0] == '*')   // dynamic
               {
                  TxPrint( "%s* (auto assign)%s",   CBG, CNN);
                  newletter = (BYTE) '*';
               }
               else
               {
                  TxPrint( "%s%s:%s\n", CBG, cbp->bname, CNN);
                  newletter = (BYTE) toupper( cbp->bname[0]);
               }
               changes = TRUE;

               #if defined (DEV32)
               if ((cbp->lvmSync) &&            // LVM sync wanted
                   (lp->VoluName[0] != 0))      // and have a volumename
               {
                  TxCopy( cbp->syncName, lp->VoluName, LVM_NAME_L);
                  cbp->syncLetter = newletter;  // Sync by caller, after WE
               }                                // have written our changes
               else
               {
                  lp->letter   = newletter;     // make the change ourselves
                  cbp->lvmSync = FALSE;         // and cancel engine request
               }
               #else
                  lp->letter   = newletter;     // make the change ourselves
               #endif
            }
            if (cbp->ntype != 0xff)
            {
               TxPrint(" On BMGR menu     : %s%26.26s=> ",
                                       (lp->OnBmMenu) ? "Yes" : "No ", "");
               TxPrint( "%s%s%s\n", CBM, (cbp->ntype) ? "Yes" : "No ", CNN);
               lp->OnBmMenu = (BYTE) cbp->ntype;
               changes = TRUE;
            }
            if (cbp->stype != 0xff)
            {
               TxPrint(" Installable flag : %s%26.26s=> ",
                                    (lp->Installable) ? "Yes" : "No ", "");
               TxPrint( "%s%s%s\n", CBM, (cbp->stype) ? "Yes" : "No ", CNN);
               lp->Installable = (BYTE) cbp->stype;
               changes = TRUE;
            }

            if (changes)
            {
               TRACES(( "LVM info sector changes ...\n"));
               sd->LvmCRC = 0;                  // recalculate CRC
               sd->LvmCRC = TxCalculateLvmCrc((BYTE *) sec, SECTORSIZE);

               p->lvm = *lp;                    // propagate to LVMSIG update

               rc = DFS_PENDING;                // request write back
            }
            else
            {
               TxPrint( "No changes to LVM information needed\n");
            }
            break;

         case FDSK_PENDING:
            break;

         default:
            rc = DFS_CMD_FAILED;
            break;
      }
   }
   else if ((rc == NO_ERROR) && (matching_sig_sector)) // existing LVM signature (BBR)
   {
      TRACES(( "Work on LVM-sign sector ...\n"));
      if ((phase == FDSK_PERFORM) && ((dfsa->batch) || (cbp->confirmed)))
      {
         TxPrint("\nLVMsign on disk %u : 0x%8.8x,  partition: %2.2hu\n",
                             cbp->disknr, (ULONG) cbp->thisSn, p->id);

         if (cbp->cbOption3)                    // Clear LVM signature forcing
         {                                      // volume to be Compatibility
            TxPrint( "Sector CLEARED to : 0xF6 pattern, "
                     "forcing a Compatibility Volume.\n");

            memset( sg, 0xf6, SECTORSIZE);      // Clear the BBR sector
            rc = DFS_PENDING;                   // request write back
         }
         else
         {
            if (sg->sectors != size)
            {
               TxPrint(" Partition size   : 0x%8.8x = %8.1lf MiB    => ",
                   sg->sectors, TXSMIB( sg->sectors, p->bpsector));
               if (cbp->cbOption1)              // change will be made
               {
                  TxPrint("0x%s%8.8x%s  = %8.1lf Mb\n", CBG, size, CNN,
                                         TXSMIB( size, p->bpsector));
                  sg->sectors = size;
                  changes = TRUE;
               }
               else
               {
                  TxPrint("0x%s%8.8x%s  = %8.1lf Mb!\n", CBR, size, CNN,
                                         TXSMIB( size, p->bpsector));
               }
            }
            if (sg->firstPsn != p->basePsn)
            {
               TxPrint(" Partition start  : 0x%8.8x%19.19s=> ", sg->firstPsn, "");
               if (cbp->cbOption1)              // change will be made
               {
                  TxPrint("0x%s%8.8x%s\n", CBG, (ULONG) p->basePsn, CNN);
                  sg->firstPsn = p->basePsn;
                  changes = TRUE;
               }
               else
               {
                  TxPrint("0x%s%8.8x%s  NOT changed!\n", CBR, (ULONG) p->basePsn, CNN);
               }
            }
            if (sg->lastPsn != p->lastPsn)
            {
               TxPrint(" Partition endPsn : 0x%8.8x%19.19s=> ", sg->lastPsn, "");
               if (cbp->cbOption1)              // change will be made
               {
                  long      delta = sg->lastPsn - p->lastPsn;
                  S_LVFEAT *lf;                 // LVM feature info
                  int       fi;                 // feature index

                  TxPrint("0x%s%8.8x%s\n",  CBG, (ULONG) p->lastPsn, CNN);
                  sg->lastPsn = p->lastPsn;
                  changes = TRUE;

                  if (sg->fakeEbrSet != 0)
                  {
                     TxPrint(" Fake EBR  at Psn : 0x%8.8x%19.19s=> 0x%8.8x\n",
                               sg->fakeEbrPsn, "", sg->fakeEbrPsn - delta);
                     sg->fakeEbrPsn -= delta;   // adjust to moved last-cyl
                  }
                  for (fi = 0; fi < LVM_FEATURES; fi++) // update feature PSNs
                  {
                     lf = &(sg->feature[fi]);
                     TRACES(( "Check PSNs for LVM feature %u (%lu), with %lu sectors\n",
                                                      fi, lf->FeatureId, lf->sectors));
                     if (lf->sectors != 0)      // feature in use
                     {
                        if (lf->FeatureId != LVMF_BAD_BLOCKR) // not relative ?
                        {
                           if (lf->priPsn != 0) // primary PSN used
                           {
                              TxPrint(" Feature %3u prim : 0x%8.8x%19.19s=> 0x%8.8x\n",
                                        fi, lf->priPsn, "", lf->priPsn - delta);
                              lf->priPsn -= delta; // adjust to moved last-cyl
                           }
                           if (lf->secPsn != 0) // secondary PSN used
                           {
                              TxPrint(" Feature %3u sec. : 0x%8.8x%19.19s=> 0x%8.8x\n",
                                        fi, lf->secPsn, "", lf->secPsn - delta);
                              lf->secPsn -= delta; // adjust to moved last-cyl
                           }
                        }
                     }
                  }
               }
               else
               {
                  TxPrint("0x%s%8.8x%s  NOT changed!\n", CBR, (ULONG) p->basePsn, CNN);
               }
            }
            if (TxaOptSet( DFS_O_LVMSNP) ||     // specific lvmSNP
                TxaOptSet( DFS_O_LVMSND)  )     // or lvmSND
            {
               S_LVFEAT *lf;                    // LVM feature info
               int       fi;                    // feature index

               for (fi = 0; fi < LVM_FEATURES; fi++) // iterate LVM features
               {
                  lf = &(sg->feature[fi]);
                  TRACES(( "LVM feature %u (%lu), with %lu sectors\n",
                                    fi, lf->FeatureId, lf->sectors));
                  if (lf->sectors != 0)         // feature in use
                  {
                     if ((lf->FeatureId == LVMF_DRIVE_LINK) &&
                         (TxaOptSet( DFS_O_LVMSNP) || // specific lvmSNP
                          TxaOptSet( DFS_O_LVMSND) )) // or lvmSND
                     {
                        refId = TxaOptNum( DFS_O_LVMSNP, NULL, sg->PartitId);
                        refI2 = TxaOptNum( DFS_O_LVMSND, NULL, d->lvmDiskId);

                        TRACES(( "DriveLink oldPid:%8.8x, newPid:%8.8x\n", sg->PartitId, refId));

                        if (refId != sg->PartitId) // was changed in Sign sector
                        {
                           TRACES(("Set DriveLink sectors PID and DID values\n"));
                           dfsFdskLvmSetDriveLink( lf->priPsn, refI2, refId, sg->PartitId);
                           dfsFdskLvmSetDriveLink( lf->secPsn, refI2, refId, sg->PartitId);
                        }
                        else
                        {
                           TRACES(("DriveLink sectors PID and DID, no update needed\n"));
                        }
                     }
                  }
               }
            }
            if (TxaOptSet( DFS_O_LVMSNP))       // force specific Part-id
            {
               refId = sg->PartitId;
               sg->PartitId = TxaOptNum( DFS_O_LVMSNP, "Partition-S#", refId);
               TxPrint(" LVM partition Id : 0x%8.8x%19.19s=> 0x%8.8x\n", refId,
                       "", sg->PartitId);
               changes = TRUE;
            }
            if (TxaOptSet( DFS_O_LVMSNV))       // force specific Vol-id
            {
               refId = sg->VolumeId;
               sg->VolumeId = TxaOptNum( DFS_O_LVMSNV, "Volume-S#",    refId);
               TxPrint(" LVM volume    Id : 0x%8.8x%19.19s=> 0x%8.8x\n", refId,
                       "", sg->VolumeId);
               changes = TRUE;
            }
            if (TxaOptSet( DFS_O_LVMSNB))       // force specific Boot-id
            {
               refId = sg->BootDiskId;
               sg->BootDiskId = TxaOptNum( DFS_O_LVMSNB, "BootDisk-S#", refId);
               TxPrint(" LVM boot disk Id : 0x%8.8x%19.19s=> 0x%8.8x\n", refId,
                       "", sg->BootDiskId);
               changes = TRUE;
            }
            if (strlen(cbp->vname) != 0)
            {
               TxPrint(" Volume    name   : %-29.29s=> ", sg->VoluName);
               if (cbp->vname[0] == '-')        // remove
               {
                  TxPrint( "%snone (delete volume)%s\n",   CBG, CNN);
                  memset( sg->VoluName, 0,          LVM_NAME_L);
                  sg->VolumeId = 0;

                  //- set signature to "LVM BBRemoved" for 'deleted-volume'
                  //- this will make the partition 'available' again in LVM
                  memcpy( sg->Signature, sg_lvrem, SG_LVSIG);
               }
               else
               {
                  TxPrint( "%s%s%s\n",  CBY, cbp->vname, CNN);
                  memcpy( sg->VoluName, cbp->vname, LVM_NAME_L);

                  sg->VolumeId = TxCalculateLvmCrc( (BYTE *) cbp->vname,
                                                             LVM_NAME_L);
               }
               changes = TRUE;
            }
            if (strlen(cbp->pname) != 0)
            {
               TxPrint(" Partition name   : %-29.29s=> ", sg->PartName);
               if (cbp->pname[0] == '-')        // remove
               {
                  TxPrint( "%snone (noname)%s\n",   CBG, CNN);
                  memset( sg->PartName, 0,          LVM_NAME_L);
               }
               else
               {
                  TxPrint( "%s%s%s\n",  CBC, cbp->pname, CNN);
                  memcpy( sg->PartName, cbp->pname, LVM_NAME_L);
                  if (sg->PartitId == 0)        // create new unique Id
                  {
                     sg->PartitId = p->basePsn + (((ULONG) p->disknr) << 28);
                  }
               }
               changes = TRUE;
            }
            if (strlen(cbp->bname) != 0)
            {
               BYTE    newletter;

               TxPrint(" Drive letter     : %c%c%27.27s=> ",
                                  (sg->letter != 0) ? sg->letter : '-',
                                  (sg->letter != 0) ? ':' : '-', "");
               if (cbp->bname[0] == '-')        // remove
               {
                  TxPrint( "%snone (hidden)%s\n",   CBG, CNN);
                  newletter = (BYTE) 0;
               }
               else
               {
                  TxPrint( "%s%s:%s\n", CBG, cbp->bname, CNN);
                  newletter = (BYTE) toupper( cbp->bname[0]);
               }
               #if defined (DEV32)
               if (cbp->lvmSync == FALSE)       // Unless dynamic LVM Engine
               #endif                           // letter update is pending
               {
                  sg->letter = newletter;       // make change ourselves ...
               }
               changes = TRUE;
            }

            if (changes)
            {
               TRACES(( "LVM signature sector changes ...\n"));
               sg->LvmCRC = 0;                  // recalculate CRC
               sg->LvmCRC = TxCalculateLvmCrc((BYTE *) sec, SECTORSIZE);

               rc = DFS_PENDING;                // request write back
            }
            else
            {
               TxPrint( "No changes to LVM signature-sector needed.\n");
            }
         }
         cbp->confirmed = FALSE;                // confirm next LVINF update
      }
   }
   else if ((rc == NO_ERROR) && (p == NULL))    // work on global LVM info/sign
   {                                            // whole disk (CRC/GEO/Name)
      if (cbp->doAllBBR)                        // Erase non-LVM BBR sectors
      {
         if ((sectype == ST_LVSIG) ||           // existing signature sector
             (sectype == ST_LVREM)  )           // or BBR V-deleted one
         {
            p = dfsGetPartInfo( dfsLvmSnP2PartId( cbp->disknr, sg->PartitId));
            if ((p != NULL) && (p->partent.PartitionType != DFS_P_WARP_LVM))
            {
               TRACES(("Signature sector found on non-0x35 type\n"));

               switch (phase)
               {
                  case FDSK_PREPARE:
                     if ((!TxaOption('R')) &&   // not read-only
                         (dfsa->batch || cbp->confirmed || TxConfirm( 5110,
                         "Erase all LVM BBR (signature) sectors for non LVM (0x35) "
                         "type partitions on disk %hu  ? [Y/N] : ", cbp->disknr)))
                     {
                        TxPrint("\nLVMsign on disk %u : 0x%8.8x,"
                                "  BBR sector on non LVM type 0x%2.2hx partition %hu.\n",
                              cbp->disknr, (ULONG) cbp->thisSn, p->partent.PartitionType, p->id);
                        cbp->confirmed = TRUE;     // just confirm once per disk
                     }
                     else
                     {
                        if (TxaOption('R'))
                        {
                           TxPrint( "LVM Read-only mode, no changes made!\n");
                        }
                        rc = DFS_CMD_FAILED;    // aborted (no update wanted)
                     }
                     break;

                  case FDSK_PERFORM:

                     TxPrint( "Sector CLEARED to : 0xF6 pattern, "
                              "forcing a Compatibility Volume.\n");

                     memset( sg, 0xf6, SECTORSIZE); // Clear the BBR sector
                     rc = DFS_PENDING;          // request write back
                     break;

                  case FDSK_PENDING:
                     break;

                  default:
                     rc = DFS_CMD_FAILED;
                     break;
               }
            }
            else
            {
               rc = DFS_CMD_FAILED;             // (no update needed)
            }
         }
         else if (phase != FDSK_PENDING)        // type will be ST_FDISK when erased!
         {
            rc = DFS_CMD_FAILED;                // (no update needed)
         }
      }
      else
      {
         ULONG            sectorCRC;            // original CRC
         char            *DiskName;             // DiskName field in sector

         if ((sectype == ST_LVSIG) ||           // existing signature sector
             (sectype == ST_LVREM)  )           // or BBR V-deleted one
         {
            DiskName = sg->DiskName;            // Set right field to update
         }                                      // Note: CRC fields are at
         else                                   //       same place in both!
         {
            DiskName = sd->DiskName;
         }

         switch (phase)
         {
            case FDSK_PREPARE:
               sectorCRC  = sd->LvmCRC;
               sd->LvmCRC = 0;                  // prepare for calculation
               sd->LvmCRC = TxCalculateLvmCrc((BYTE *) sec, SECTORSIZE);

               if ((sectype == ST_LVINF)   &&   // LVM info sector
                   (cbp->cbOptNum1 == 1)   &&   // GEO update requested
                   (strlen(cbp->vname) == 0))   // and no name change
               {
                  if ((!TxaOption('R')) &&      // not read-only
                      (dfsa->batch || cbp->confirmed || TxConfirm( 5103,
                      "Force LVM geometry for disk %hu to the "
                      "current DFSee logical geometry:\n\n"
                      "  L-Geo  Cyl :%6u H:%3u S:%-3u  ? [Y/N] : ",
                      cbp->disknr, d->geoCyls, d->geoHeads, d->geoSecs)))
                  {

                     TxPrint("LVM-DLAT   disk %u : 0x%8.8X Cyl:%6u H:%3u S:%-3u => ",
                        cbp->disknr, (ULONG) cbp->thisSn, sd->geoCyls, sd->geoHeads, sd->geoSecs);

                     if ((sd->geoCyls  != d->geoCyls)  ||
                         (sd->geoHeads != d->geoHeads) ||
                         (sd->geoSecs  != d->geoSecs)   )
                     {
                        TxPrint("Cyl:%6u H:%3u S:%-3u\n", d->geoCyls, d->geoHeads, d->geoSecs);
                     }
                     else                       // Geo was OK
                     {
                        TxPrint( "OK, no change needed.\n");
                        rc = DFS_CMD_FAILED;    // (no update needed)
                     }
                     cbp->confirmed = TRUE;     // just confirm once per disk
                  }
                  else
                  {
                     if (TxaOption('R'))
                     {
                        TxPrint( "LVM Read-only mode, no changes made!\n");
                     }
                     rc = DFS_CMD_FAILED;       // aborted (no update wanted)
                  }
               }

               if (strlen(cbp->vname))          // diskname specified,
               {                                // silent CRC update too
                  TRACES(("LVM disknr %hu prompt %llu %s %s\n",
                           cbp->disknr, cbp->cbOptNum2, cbp->pname, cbp->vname));

                  if ((strcmp(cbp->pname, "-") == 0) && // prompt requested
                      (cbp->cbOptNum2 != cbp->disknr) ) // first for this disk
                  {
                     TXTM s1;                   // diskname buffer

                     cbp->cbOptNum2 = cbp->disknr; // no more prompt this disk
                     strcpy( s1, d->DiskName);
                     if (!dfsa->batch && dfsDialogAppropriate())
                     {
                        TxPrompt( 5411, BMSYS_LONG -1, s1,
                                 "Specify a unique LVM disk name for disk %hu", cbp->disknr);
                     }
                     s1[ BMSYS_LONG -1] = 0;    // shorten to max length
                     strcpy( cbp->vname, s1);
                  }
                  if ((sectype == ST_LVINF) ||  // LVM info sector
                      (sectype == ST_LVSIG) ||  // LVM signature sector
                      (sectype == ST_LVREM)  )  // or BBR V-deleted one
                  {
                     TxPrint("LVM-%s disk %u : %-*.*s at 0x%8.8X  => %s\n",
                        (sectype == ST_LVINF) ? "DLAT  " : "BBR   ",
                        cbp->disknr, LVM_NAME_L, LVM_NAME_L, DiskName,
                        (ULONG) cbp->thisSn, cbp->vname);

                     if (changes || memcmp( DiskName, cbp->vname, LVM_NAME_L) != 0)
                     {
                        if ((!TxaOption('R')) && // not read-only
                            (dfsa->batch || cbp->confirmed || TxConfirm( 5104,
                            "Update LVM disk name for disk %hu to:\n\n  '%s'  ? [Y/N] : ",
                             cbp->disknr, cbp->vname)))
                        {
                           cbp->confirmed = TRUE; // just confirm once per disk
                        }
                        else
                        {
                           if (TxaOption('R'))
                           {
                              TxPrint( "LVM Read-only mode, no changes made!\n");
                           }
                           rc = DFS_CMD_FAILED; // aborted (no update wanted)
                        }
                     }
                     else
                     {
                        if (sectype == ST_LVINF) // Disk-ID sync needed ?
                        {
                           if ((sd->BootDiskId == d->lvmBootId) &&
                               (sd->ThisDiskId == d->lvmDiskId)  )
                           {
                              rc = DFS_CMD_FAILED; // (no update needed)
                           }
                        }
                        else
                        {
                           rc = DFS_CMD_FAILED; // (no update needed)
                        }
                     }
                  }
                  else
                  {
                     TxPrint("LVM-????   disk %u : No LVM sector yet!    0x%8.8X  "
                             "   Set name not possible\n", cbp->disknr, (ULONG) cbp->thisSn);
                     rc = DFS_CMD_FAILED;       // aborted (no update wanted)
                  }
               }

               if ((cbp->cbOptNum1 == 0)   &&   // no GEO update requested
                   (strlen(cbp->vname) == 0))   // and no name change
               {                                // remove obsolete DLAT entries
                  if (sectype == ST_LVINF)      // from the LVM info sector
                  {
                     for (pi = 0; pi < 4; pi++) // check each DLAT entry
                     {
                        lp = &(sd->part[pi]);
                        if (lp->sectors != 0)
                        {
                           if (dfsFdskDlatEntry2Part( lp, SINF->disknr) == NULL)
                           {
                              if (TxaOption('R')) // read-only, just display
                              {
                                 TxPrint("LVM-info   disk %u : obsolete DLAT entry"
                                         " %u (LVM 'corrupt' errors!)\n",
                                               cbp->disknr, pi+1);
                              }
                              else
                              {
                                 BYTE st = ST_LVINF;
                                 TXTM pdesc;

                                 TxPrint("\n");
                                 dfsReadAnDisplay( cbp->thisSn, 0, &st);

                                 sprintf(   pdesc, "Partition 1st PSN : 0x%8.8X = %-10u",
                                            lp->partPsn, lp->partPsn);
                                                // dfstrSiz8( pdesc, " size : ", lp->sectors, "");
                                 dfstrSizeBps( pdesc, " size : ", lp->sectors, d->bpsector, "");

                                 if ((dfsa->batch || cbp->confirmed || TxConfirm( 5105,
                                     "Obsolete entry %u found in LVM DLAT info sector "
                                     "0x%8.8X on disk %hu. This may cause 'corrupted "
                                     "partition table' messages from the LVM program!"
                                     "\n\n%s\n\nRemove this DLAT entry ? [Y/N] : ",
                                      pi, (ULONG) cbp->thisSn, cbp->disknr, pdesc)))
                                 {
                                    memset( lp, 0, sizeof( S_LVPART)); // remove entry
                                    sd->LvmCRC = TxCalculateLvmCrc((BYTE *) sec, SECTORSIZE);
                                    cbp->confirmed = TRUE; //- but no CRC confirm
                                    TRACES(("DLAT-entry %hu has been cleared\n", pi));

                                    TxPrint("LVM-info   disk %u : Removed obsolete "
                                            "entry from : 0x%8.8X\n",
                                             cbp->disknr, (ULONG) cbp->thisSn);
                                 }
                              }
                           }
                        }
                     }
                  }

                  if ((sectype == ST_LVINF) ||  // LVM info sector
                      (sectype == ST_LVSIG) ||  // LVM signature sector
                      (sectype == ST_LVREM)  )  // or BBR V-deleted one
                  {
                     TxPrint("LVM-%s disk %u : %-*.*s at 0x%8.8x, CRC: 0x%s%8.8x%s =",
                        (sectype == ST_LVINF) ? "DLAT  " : "BBR   ",
                        cbp->disknr, LVM_NAME_L, LVM_NAME_L, DiskName, (ULONG) cbp->thisSn,
                        (sd->LvmCRC == sectorCRC) ? CBG : CBR,  sectorCRC, CNN);

                     if (sd->LvmCRC == sectorCRC)
                     {
                        TxPrint( "  OK\n");
                        rc = DFS_CMD_FAILED;
                     }
                     else
                     {
                        TxPrint( "> 0x%8.8x\nCRC corrected now : ", sd->LvmCRC);
                        if ((!TxaOption('R')) && // not read-only
                            (dfsa->batch || cbp->confirmed || TxConfirm( 5106,
                            "Update CRC/sequence values for disk %hu ? [Y/N] : ", cbp->disknr)))
                        {
                           cbp->confirmed = TRUE; // just confirm once per disk
                        }
                        else
                        {
                           rc = DFS_CMD_FAILED; // aborted (no update wanted)
                        }
                     }
                     if (sectype == ST_LVSIG)   // check SEQ warning for this partition
                     {
                        p = dfsGetPartInfo( dfsLvmSnP2PartId( cbp->disknr, sg->PartitId));
                        if (p && (p->flag2 & DFS_F_LVMBADSEQN))
                        {
                           S_LVFEAT *lf = &(sg->feature[1]); // 2nd feature in array

                           TxPrint("LVM-DrLink disk %u : partition %02.2hu, "
                                   "Fix sequence number : ", cbp->disknr, p->id);

                           if ((dfsMkDriveLinkFeature( lf->priPsn, d->lvmDiskId, sg->PartitId) == NO_ERROR) &&
                               (dfsMkDriveLinkFeature( lf->secPsn, d->lvmDiskId, sg->PartitId) == NO_ERROR)  )
                           {
                              TxPrint( "%sOK%s\n", CBG, CNN);
                           }
                           else
                           {
                              TxPrint( "%sFailed!%s\n", CBR, CNN);
                           }
                           //- Keep CRC RC since BBR sector itself does NOT need to be written!
                        }
                     }
                  }
                  else
                  {
                     rc = DFS_CMD_FAILED;       // aborted (no update wanted)
                  }
               }
               break;

            case FDSK_PERFORM:
               if (strlen(cbp->vname))          // diskname specified,
               {
                  memset(    DiskName, 0,          LVM_NAME_L);
                  strncpy(   DiskName, cbp->vname, LVM_NAME_L);
                  strcpy( d->DiskName, cbp->vname);

                  if (sectype == ST_LVINF)      // Sync Disk-ID's for all
                  {
                     sd->BootDiskId  = (d->lvmBootId) ? d->lvmBootId
                                                      : 0xdf5eed00 + d->disknr;
                     sd->ThisDiskId  = (d->lvmDiskId) ? d->lvmDiskId
                                                      : 0xdf5eed00 + d->disknr;
                     d->lvmBootId = sd->BootDiskId;
                     d->lvmDiskId = sd->ThisDiskId;
                  }
               }
               if ((sectype == ST_LVINF) &&     // LVM info sector
                   (cbp->cbOptNum1 == 1)  )     // GEO update requested
               {
                  sd->geoCyls  = d->geoCyls;
                  sd->geoHeads = d->geoHeads;
                  sd->geoSecs  = d->geoSecs;
               }
               sd->LvmCRC = 0;                  // recalculate CRC
               sd->LvmCRC = TxCalculateLvmCrc((BYTE *) sec, SECTORSIZE);
               rc = DFS_PENDING;                // request write back
               break;

            case FDSK_PENDING:
               break;

            default:
               rc = DFS_CMD_FAILED;
               break;
         }
      }
   }
   #if defined (DEV32)
   TRACES(("cbp->lvmSync now : %s  syncName:'%s' lvm.letter: %d\n",
           (cbp->lvmSync) ? "TRUE" : "FALSE", cbp->syncName, cbp->syncLetter));
   #endif
   RETURN (rc);
}                                               // end 'dfsFdskLvmSetInfo'
/*-----------------------------------------------------------------------------------------------*/


/*************************************************************************************************/
// Set new disk-level information for LVM-info sector
/*************************************************************************************************/
static void dfsInitLvmDskInfo
(
   S_LVINF            *sd,                      // IN    LVM info sector
   DFSDISKINFO        *d                        // IN    Target disk info
)
{
   ENTER();

   if ((d->DiskName[2] == ' ') ||               // no valid name yet,
       (d->DiskName[0] == 0  ) ||
       (d->lvmDiskId   == 0  )  )               // or no LVM info set
   {
      if (TxaOptSet( 'n'))                      // default name specified ?
      {
         TxCopy(  d->DiskName, TxaOptStr( 'n', NULL, ""), LVM_NAME_L);
      }
      if (strlen( d->DiskName) == 0)            // if none set, make default
      {
         sprintf( d->DiskName, "%sDisk%hu - ", dfsMediaTypeDescr( dfsDid2DiskType(d->disknr)), d->disknr);
         dfstrSz64XiB( d->DiskName, "", d->sectors, d->bpsector, "");
      }
   }
   TxCopy( sd->DiskName, d->DiskName, LVM_NAME_L);
   sd->ThisDiskId  = (d->lvmDiskId)    ? d->lvmDiskId
                                       : 0xdf5eed00 + d->disknr;
   d->lvmDiskId    = sd->ThisDiskId;

   sd->BootDiskId  = (d->lvmBootId)    ? d->lvmBootId : dfsa->lvmBootId;
   if (sd->BootDiskId == 0)
   {
      sd->BootDiskId = 0xdf5eed00 +  (dfsa->bmgrDisk) ? dfsa->bmgrDisk : 1;
   }
   if (dfsa->lvmBootId == 0)
   {
      dfsa->lvmBootId = sd->BootDiskId;
   }
   sd->geoCyls  = (ULONG) d->geoCyls;
   sd->geoHeads = (ULONG) d->geoHeads;
   sd->geoSecs  = (ULONG) d->geoSecs;

   VRETURN();
}                                               // end 'dfsInitLvmDskInfo'
/*-----------------------------------------------------------------------------------------------*/


/*************************************************************************************************/
// Initialize LVM-info sector from existing LVM-signature sector when found
// to be refined, phase PERFORM should not be needed, just execute at PREPARE
// Info should remain between PREPARE and PERFORM at caller (but does NOT!)
/*************************************************************************************************/
static BOOL dfsInitInfoFromSig                  // RET   LVM-sig found & used
(
   S_LVINF            *sd,                      // IN    LVM info sector
   ULONG               index,                   // IN    index to be updated
   FDSK_CB_PHASE       phase,                   // IN    callback phase
   DFSPARTINFO        *p                        // IN    Partition info
)
{
   BOOL                rc = FALSE;
   BYTE               *sec = NULL;              // sector buffer

   ENTER();

   switch (phase)
   {
      case FDSK_PREPARE:
      case FDSK_PERFORM:
         if ((sec = (TxAlloc( 1, dfsGetSectorSize()))) != NULL)
         {
            rc = dfstReadPsn( DFSTORE, p->lastPsn, 1, sec); // possible signature sector
            if (rc == NO_ERROR)
            {
               BYTE  sectype = dfsIdentifySector( p->lastPsn, 0, sec);

               if ((sectype == ST_LVSIG) ||     // existing signature sector
                   (sectype == ST_LVREM)  )     // or BBR V-deleted one
               {
                  S_LVSIG  *sg = (S_LVSIG *) sec;
                  S_LVPART *lp = &( sd->part[index]); // new LVPART to work on

                  sd->BootDiskId  = sg->BootDiskId;
                  lp->VolumeId    = sg->VolumeId;
                  lp->PartitId    = sg->PartitId;
                  lp->partPsn     = sg->firstPsn;
                  lp->sectors     = sg->sectors;
                  lp->letter      = sg->letter;

                  memcpy( lp->PartName, sg->PartName, LVM_NAME_L);
                  memcpy( lp->VoluName, sg->VoluName, LVM_NAME_L);

                  if (phase == FDSK_PREPARE)    // just display once
                  {
                     TxPrint("LVM info recovered from LVM-sig sector at 0x%8.8x.\n", (ULONG) p->lastPsn);
                     TxPrint("\n Volume    name   : %-25.25s  Id : 0x%8.8x",    lp->VoluName,
                                                                                 lp->VolumeId);
                     TxPrint("\n Partition name   : %-25.25s  Id : 0x%8.8x",    lp->PartName,
                                                                                 lp->PartitId);
                     TxPrint("\n Drive letter     : %c%c", (lp->letter != 0) ? lp->letter : '-',
                                                           (lp->letter != 0) ? ':' : '-');
                     TxPrint("\n On BMGR menu     : %s\n", (lp->OnBmMenu) ? "Yes" : "No ");
                  }
                  rc = TRUE;
               }
            }
            TxFreeMem( sec);                    // free the memory
         }
         break;

      default:
         break;
   }
   BRETURN( rc);
}                                               // end 'dfsInitInfoFromSig'
/*-----------------------------------------------------------------------------------------------*/


/*************************************************************************************************/
// Set new disk-level information and init BBR area for LVM-signature sector
/*************************************************************************************************/
static ULONG dfsInitLvmSigArea
(
   S_LVSIG            *sg,                      // IN    LVM signature sector
   DFSDISKINFO        *d,                       // IN    Target disk info
   DFSPARTINFO        *p                        // IN    Partition info
)
{
   ULONG               rc;
   S_LVPART           *lp      = &( p->lvm);    // set by LVMINFO init
   ULONG               tabsize = min((lp->sectors / 1024), 4096);
   ULONG               tabsect = ((tabsize * 8) / 496) + 1;
   ULONG               reserve = tabsize + (2 * tabsect) + p->geoSecs + 11;
   S_LVFEAT           *lf;                      // LVM feature info

   ENTER();
   TRACES(( "tabsize: %lu  tabsect: %lu  reserve: %lu\n", tabsize, tabsect, reserve));

   sg->BootDiskId = dfsa->lvmBootId;

   sprintf( sg->pComment, "Created by %s version %s", DFS_N, DFS_V);
   memcpy(  sg->DiskName,  d->DiskName, LVM_NAME_L);
   memcpy(  sg->PartName, lp->PartName, LVM_NAME_L);
   memcpy(  sg->VoluName, lp->VoluName, LVM_NAME_L);

   sg->VolumeId   = lp->VolumeId;
   sg->PartitId   = lp->PartitId;
   sg->firstPsn   = lp->partPsn;
   sg->lastPsn    = lp->partPsn + lp->sectors -1;
   sg->sectors    = lp->sectors;
   sg->letter     = lp->letter;
   sg->majVers    = 1;
   sg->minVers    = 0;
   sg->reserved   = reserve;
   sg->reported   = lp->sectors - reserve;

   //- Init and write the fake-EBR track
   sg->fakeEbrPsn = sg->lastPsn - p->geoSecs;
   sg->fakeEbrSet = TRUE;

   rc = dfsMkFakeEbrSector( p, sg->fakeEbrPsn, sg->firstPsn, sg->reported);
   if (rc == NO_ERROR)
   {
      //- Init the bad-block feature
      lf = &(sg->feature[0]);                   // 1st feature in array
      lf->FeatureId = LVMF_BAD_BLOCKR;
      lf->majVers   = 1;
      lf->minVers   = 0;
      lf->active    = 1;
      lf->sectors   = tabsect +1;
      lf->secPsn    = sg->sectors - reserve + 8; // really an LSN! (LVM design)
      lf->priPsn    = lf->secPsn  + lf->sectors;

      rc = dfsMkBadBlockFeature( lf->priPsn + sg->firstPsn,
                                 lf->priPsn + lf->sectors,
                                 tabsize, tabsect);
      if (rc == NO_ERROR)
      {
         rc = dfsMkBadBlockFeature( lf->secPsn + sg->firstPsn,
                                    lf->priPsn + lf->sectors,
                                    tabsize, tabsect);
         if (rc == NO_ERROR)
         {
            //- Init the drive-linking feature
            lf = &(sg->feature[1]);             // 2nd feature in array
            lf->FeatureId = LVMF_DRIVE_LINK;
            lf->majVers   = 1;
            lf->minVers   = 0;
            lf->active    = 1;
            lf->sectors   = 4;
            lf->secPsn    = sg->lastPsn - reserve + 1;
            lf->priPsn    = lf->secPsn  + lf->sectors;

            rc = dfsMkDriveLinkFeature( lf->priPsn, d->lvmDiskId, sg->PartitId);
            if (rc == NO_ERROR)
            {
               rc = dfsMkDriveLinkFeature( lf->secPsn, d->lvmDiskId, sg->PartitId);
            }
         }
      }
   }
   RETURN( rc);
}                                               // end 'dfsInitLvmSigArea'
/*-----------------------------------------------------------------------------------------------*/


/*****************************************************************************/
// Create and write a fake EBR sector for the LVM reported partition size
/*****************************************************************************/
static ULONG dfsMkFakeEbrSector
(
   DFSPARTINFO        *p,                       // IN    partition information
   ULONG               fakePsn,                 // IN    sector to write to
   ULONG               pStart,                  // IN    partition start PSN
   ULONG               pSize                    // IN    reported part size
)
{
   ULONG               rc = NO_ERROR;           // function return
   S_BOOTR            *ebr;                     // EBR record buffer
   DFSPARTENTRY       *pe;                      // partition-table entry
   ULONG               c, h;
   ULONG               aSize = pSize + DFS_STD_SECT; // aligned size

   ENTER();

   TRACES(( "Psn:%8.8x start:%8.8x size:%8.8x\n", fakePsn, pStart, pSize));
   if ((ebr = (TxAlloc( 1, dfsGetSectorSize()))) != NULL)
   {
      //- LVM aligns the 'fakeEBR' such that the faked partition is always
      //- aligned to a 255/63 geometry, using an exact multiple of the
      //- cylinder size for that. aSize includes the EBR-track too.

      h = min((aSize / DFS_STD_SECT), DFS_STD_HEAD);
      while ((c = (aSize / (h * DFS_STD_SECT))) > 0xFFFF)
      {
         h *= 2;                                // double heads until c fits
      }
      aSize = ((c * h) -1) * DFS_STD_SECT;      // aligned to 255/63 geometry

      pe = &(ebr->PartitionTable[0]);

      pe->Status             = DFS_P_ACTIVE;    // always marked active
      pe->PartitionType      = DFS_P_INST_FS;   // partition type
      pe->BootSectorOffset   = DFS_STD_SECT;    // one-track offset to EBR
      pe->NumberOfSectors    = aSize;           // partition size

      pe->FirstSector.SecCyl = DFSCOMBINE(63, 1023); //- fixed 255/63 geo!
      pe->FirstSector.Head   = (BYTE)    (254);      //- using dummy values

      pe->LastSector.SecCyl = DFSCOMBINE(63, 1023);
      pe->LastSector.Head   = (BYTE)    (254);

      ebr->Signature = SV_BOOTR;
      rc = dfsWrite( fakePsn, 1, (BYTE *) ebr);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsMkFakeEbrSector'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create and write the sectors for sec/prim Bad Block Relocation feature/table
/*****************************************************************************/
static ULONG dfsMkBadBlockFeature
(
   ULONG               featPsn,                 // IN    1st sector to write to
   ULONG               replLSN,                 // IN    1st replacement LSN
   ULONG               tableSize,               // IN    BBR replacement sectors
   ULONG               tableSect                // IN    BBR table sectors
)
{
   ULONG               rc = NO_ERROR;           // function return
   BYTE               *sec;                     // sector buffer

   ENTER();

   TRACES(( "Psn:%8.8x start:%8.8x size:%8.8x\n", featPsn, replLSN, tableSize, tableSect));
   if ((sec = (TxAlloc( 1, dfsGetSectorSize()))) != NULL)
   {
      S_LVBBF         *bbf = (S_LVBBF *) sec;

      memcpy( bbf->Signature, sg_lvbbf, SG_LVBBF);
      bbf->SequenceNr                         = 1;
      bbf->TableEntriesInUse                  = 0;
      bbf->SectorsPerTable                    = tableSect;
      bbf->TableSize                          = tableSize;
      bbf->ReplSectorCount                    = tableSize;
      bbf->FirstReplSector                    = replLSN;
      bbf->LastReplSector                     = replLSN + tableSize -1;
      bbf->FeatureSequenceNr                  = 0;
      bbf->TopOfClass                         = FALSE;
      bbf->ActualClass                        = LVM_CLASS_PARTITION;

      bbf->LvmCRC = TxCalculateLvmCrc((BYTE *) sec, SECTORSIZE);

      rc = dfsWrite( featPsn, 1, sec);
      if (rc == NO_ERROR)
      {
         S_LVBBT      *bbt = (S_LVBBT *) sec;
         ULONG         rsn = replLSN;
         ULONG         lim = replLSN + tableSize;
         ULONG         ts;
         ULONG         i;

         for ( ts = 0; (ts < tableSect) && (rc == NO_ERROR); ts++)
         {
            memset( bbt, 0, SECTORSIZE);
            memcpy( bbt->Signature, sg_lvbbt, SG_LVBBT);
            bbt->SequenceNr = 1;

            for (i = 0; (i < LVBBT_SIZE) && (rsn < lim); i++, rsn++)
            {
               bbt->BbrTable[i].Badsector   = 0xffffffff;
               bbt->BbrTable[i].Replacement = rsn;
            }
            bbt->LvmCRC = TxCalculateLvmCrc((BYTE *) sec, SECTORSIZE);

            rc   = dfsWrite( featPsn + 1 + ts, 1, sec);
         }
      }
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsMkBadBlockFeature'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create and write the four sectors for sec/prim Drive Linking feature/table
/*****************************************************************************/
static ULONG dfsMkDriveLinkFeature
(
   ULONG               featPsn,                 // IN    1st sector to write to
   ULONG               diskId,                  // IN    LVM disk ID
   ULONG               partId                   // IN    LVM part ID
)
{
   ULONG               rc = NO_ERROR;           // function return
   BYTE               *sec;                     // sector buffer

   ENTER();

   TRACES(( "Psn:%8.8x start:%8.8x size:%8.8x\n", featPsn, diskId, partId));
   if ((sec = (TxAlloc( 1, dfsGetSectorSize()))) != NULL)
   {
      S_LVDLF         *dlf = (S_LVDLF *) sec;

      memcpy( dlf->Signature, sg_lvdlf, SG_LVDLF);
      dlf->SequenceNr                         = 1;
      dlf->LinksInUse                         = 1;
      dlf->AggregateSerNr                     = 0;
      dlf->FeatureSequenceNr                  = 0;
      dlf->TopOfClass                         = FALSE;
      dlf->ActualClass                        = LVM_CLASS_PARTITION;
      dlf->LinkTable[ 0].DriveSerialNr        = diskId;
      dlf->LinkTable[ 0].PartitionSerialNr    = partId;

      dlf->LvmCRC = TxCalculateLvmCrc((BYTE *) sec, SECTORSIZE);

      rc = dfsWrite( featPsn, 1, sec);
      if (rc == NO_ERROR)
      {
         S_LVDLT      *dlt = (S_LVDLT *) sec;

         memset( dlt, 0, SECTORSIZE);
         memcpy( dlt->Signature, sg_lvdlt, SG_LVDLT);
         dlt->SequenceNr = 1;                   // P#1262

         dlt->LvmCRC = TxCalculateLvmCrc((BYTE *) sec, SECTORSIZE);

         if    ((rc  = dfsWrite( featPsn +1, 1, sec)) == NO_ERROR)
         {
            if ((rc  = dfsWrite( featPsn +2, 1, sec)) == NO_ERROR)
            {
                rc   = dfsWrite( featPsn +3, 1, sec);
            }
         }
      }
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsMkDriveLinkFeature'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Update lvmsnp and lvmsnd at first match in LVM Drive-linking sector at PSN
/*****************************************************************************/
static ULONG dfsFdskLvmSetDriveLink
(
   ULONG               dLink,                   // IN    Drive-link sector PSN
   ULONG               newSnD,                  // IN    new LVM-this-disk S#
   ULONG               newSnP,                  // IN    new LVM-partition S#
   ULONG               oldSnP                   // IN    current partition S#
)
{
   ULONG               rc = NO_ERROR;           // function return
   BYTE               *sec;                     // sector buffer

   ENTER();

   TRACES(( "Psn:%8.8x newSnD:%8.8x newSnP:%8.8x\n", dLink, newSnD, newSnP));
   if ((sec = (TxAlloc( 1, dfsGetSectorSize()))) != NULL)
   {
      rc = dfstReadPsn( DFSTORE, dLink, 1, sec);
      if (rc == NO_ERROR)
      {
         S_LVDLF      *sd = (S_LVDLF *) sec;    // first DriveLink sector
         BOOL          changes = FALSE;
         ULONG         li;                      // link index
         ULONG         id;                      // id value
         BYTE          sectype = dfsIdentifySector( dLink, 0, sec);

         if (sectype == ST_LVDLF)               // it is a DriveLink sector
         {
            if (sd->LinksInUse > 0)
            {
               for (li = 0; (li < sd->LinksInUse) && (li < LVDLT_SIZE1); li++)
               {
                  if (sd->LinkTable[ li].PartitionSerialNr == oldSnP)
                  {
                     if ((id = sd->LinkTable[ li].PartitionSerialNr) != newSnP)
                     {
                        if (changes == FALSE)   // first incorrect one
                        {
                           TxPrint(" DriveLink sector : 0x%8.8x partS# 0x%8.8x => 0x%8.8x\n",
                                                         dLink, id, newSnP);
                           sd->LinkTable[li].PartitionSerialNr = newSnP;
                           if ((id = sd->LinkTable[li].DriveSerialNr) != newSnD)
                           {
                              TxPrint(" DriveLink sector : 0x%8.8x diskS# 0x%8.8x => 0x%8.8x\n",
                                                            dLink, id, newSnD);
                              sd->LinkTable[li].DriveSerialNr = newSnD;
                           }
                           changes = TRUE;
                        }
                        else                    // there are more than one, just change the FIRST
                        {                       // allowing multiple commands making them unique!
                           TxPrint(" DriveLink sector : 0x%8.8x partS# 0x%8.8x was a DUPLICATE!\n");
                        }
                     }
                  }
               }
            }
            if (changes)                        // write back changed sector
            {
               sd->LvmCRC = 0;                  // recalculate CRC
               sd->LvmCRC = TxCalculateLvmCrc(sec, dfsGetSectorSize());
               rc = dfstWritePsn( DFSTORE, dLink, 1, sec);
            }
         }
      }
      TxFreeMem( sec);                          // free the memory
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsFdskLvmSetDriveLink'
/*---------------------------------------------------------------------------*/

