//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// FDISK GPT related utility functions
//
// Author: J. van Wijk
//
// JvW  12-10-2015   Initial version, split from DFSUFGPT
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfsupart.h>                           // FDISK partition functions
#include <dfsafdsk.h>                           // FDISK display & analysis
#include <dfsufdsk.h>                           // FDISK utility functions
#include <dfsufgpt.h>                           // FDISK GPT utility functions
#include <dfscfgpt.h>                           // FDISK GPT Command functions


// Find an empty entry in the GPT partition table array (PTA), in memory
static ULONG dfsGptFindEmptyEntry               // RET   PTA index, or DFS32MAX
(
   S_GPTENTRY          ptArray[],               // IN    allocated PT array
   ULONG               ptaEntries,              // IN    max nr of entries
   ULONG               ptaEntrySize,            // IN    size of entry in bytes
   S_GPTENTRY        **emptyEntry               // OUT   empty entry ptr, or NULL
);


/*****************************************************************************/
// Execute the 'gpt' command to display or FIX the GPT structures for one disk
/*****************************************************************************/
ULONG dfsGptExcecute
(
   USHORT              disknr,                  // IN    disk to work on
   BOOL                fixup                    // IN    fix GPT by write back
)
{
   ULONG               rc = NO_ERROR;           // rc, alloc / CRC errors
   DFSDISKINFO        *d;
   TXTM                command;

   ENTER();

   if ((d = dfsGetDiskInfo( disknr)) != NULL)
   {
      TxPrint( "\n");
      dfsSelectDisk( disknr, FALSE, FALSE);     // select disk

      if ((d->gptHeader != NULL) && (d->gptArray != NULL))
      {
         S_GPTHDR  *hdr = (S_GPTHDR *) d->gptHeader;

         if (fixup)
         {
            if (hdr->thisPSN == GPTHDR_PRIMARY_PSN) // check alt-PSN against disk-size
            {
               if (hdr->althPSN != (d->sectors - 1))
               {
                  ULONG infoSectors = hdr->althPSN - hdr->lastPSN;

                  if ((d->lastUsedPsn + infoSectors) < d->sectors)  //- sanity check, info still fits?
                  {
                     hdr->althPSN = d->sectors - 1;                 //- update alternate location
                     hdr->lastPSN = hdr->althPSN - infoSectors;     //- and last usable sector
                     TxPrint( "\nRelocating alternate GPT hdr/PTA to currently defined end of disk (size)\n");
                  }
                  else                          // current disk-size too small
                  {                             // for partitions defined in tables!
                     TxPrint( "\n Disk %2hu  WARNING : Location of alternate GPT Header/PTA can NOT be fixed\n"
                                 "                     because the last defined partition ends BEYOND that\n"
                                 "                     position, with the currently reported disk size!\n", disknr);
                  }
               }
            }
            if ((dfsa->batch) || TxConfirm( 5028,
                 "Fix up and write back retrieved GPT structures for disk %hu ? [Y/N] : ", disknr))
            {
               TxPrint( "Writing currently loaded GPT header and partition table array for disk %hu,\n"
                        "fixing missing primary or alternate headers, and fixup CRC errors.\n\n", disknr);

               rc = dfsGptFixupWriteAll( hdr, d->gptArray);
               if (rc == NO_ERROR)
               {
                  TxPrint( "GPT info written successfully to primary and alternate location on disk %hu.\n", disknr);
               }
            }
         }
         else                                   // display the loaded GPT hdr/array
         {
            TxPrint( "Currently loaded GPT header and partition table array for disk %hu:\n\n", disknr);
            sprintf( command, "0x%llx", hdr->thisPSN);
            rc = dfsMultiCommand( command, 0, FALSE, FALSE, TRUE);
         }
      }
      else                                      // no GPT tables read, perhaps autoGPT is off ...
      {                                         // or there is no guard partition defined
         DFSPARTINFO        *p = dfsGetPartInfo( d->firstpart);

         if ((p != NULL) && (p->partent.PartitionType == DFS_P_EFI_GPT)) // there IS a guard
         {
            TxPrint( "\nThere IS a GPT guard partition, but no GPT tables are detected.\n"
                       "Check if 'Auto GPT partition handling' is ON in the Edit menu,\n"
                       "or force it to ON with 'set gpt on'\n\n");
         }
         else                                   // there is no guard, check if GPT tables present
         {
            rc = dfstReadPsn( DFSTORE, 1, 1, brec);
            if (dfsIdentifySector( 1, 0, brec) == ST_GPTHD)
            {
               TxPrint( "\nSpecified disk %hu does not have a GPT-guard partition in the MBR\n"
                          "but there IS a at least a GPT header sector after the MBR sector itself,\n"
                          "any GPT partitions defined there are currently NOT visible in DFSee.\n"
                          "You can either keep the currently shown MBR-style with 'gpt2mbr -m-'\n"
                          "or recover the GPT partitions using the command 'mbr2gpt'\n\n", disknr);

               rc = dfsGptHdr( brec);
            }
            else
            {
               TxPrint( "\nSpecified disk %hu does not seem to be a GPT partitioned disk!\n", disknr);
            }
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsGptExcecute'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create a new GPT style partition in the existing GPT partition array
/*****************************************************************************/
ULONG dfsGptCreatePartition
(
   FDSK_CB_INFO       *cbp                      // IN    call-back-params
)
{
   ULONG               rc = NO_ERROR;           // function return
   DFSDISKINFO        *d  = dfsGetDiskInfo( cbp->disknr);
   S_GPTHDR           *gptHdr  = (S_GPTHDR *)   d->gptHeader;
   S_GPTENTRY         *gpTable = (S_GPTENTRY *) d->gptArray;

   ENTER();

   strcat( cbp->string, "Create the partition as shown ? [Y/N] : ");
   if (dfsa->batch || TxConfirm( 5036, cbp->string))
   {
      S_GPTENTRY      *gpte = NULL;
      ULONG            index;

      gptHdr = (S_GPTHDR *) d->gptHeader;       // current valid header info

      if ((index = dfsGptFindEmptyEntry( gpTable,
                     gptHdr->ptaEntries, gptHdr->ptaEntrySize, &gpte)) != DFS32MAX)
      {
         DFSPARTINFO  *f = (DFSPARTINFO *) cbp->more;

         TxPrint( "\nUsing PTA entry # : %lu for new GPT partition %s\n", index, cbp->pname);

         //- Copy CREATE parameters into the new entry
         gpte->firstPSN = cbp->sn;
         gpte->lastPSN  = cbp->sn + cbp->number -1;

         memcpy( gpte->typeGuid, cbp->typeGuid, DFS_GUID_LENGTH);

         if (!TxAreaEmpty( cbp->partGuid, DFS_GUID_LENGTH, 0)) // use specified one
         {
            memcpy( gpte->partGuid, cbp->partGuid, DFS_GUID_LENGTH);
         }
         else                                   // create a unique part GUID
         {
            dfsGptMkGuid( f->id,   gpte->partGuid);
         }

         TxAscii2Unic( cbp->pname, gpte->partName, GPT_PNAME_LEN);

         sscanf( cbp->cbOptStr1, "%llX", &gpte->attrFlags);

         TxPrint( "Writing the modified GPT header and partition table array for disk %hu.\n", cbp->disknr);

         rc = dfsGptFixupWriteAll( d->gptHeader, d->gptArray);
         if (rc == NO_ERROR)
         {
            TxPrint( "GPT info written successfully to primary and alternate location on disk %hu.\n", cbp->disknr);
            dfsFdskSingleAutoShow( cbp->disknr); // indicate changes
         }
      }
      else
      {
         TxPrint( "\nPartition Table Array (PTA) is full. Create partition failed!\n");
         rc = DFS_CMD_FAILED;
      }
   }
   else
   {
      rc = DFS_USER_ABORT;                      // aborted
   }
   RETURN (rc);
}                                               // end 'dfsGptCreatePartition'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Delete a single GPT partition from specified disk
/*****************************************************************************/
ULONG dfsGptDeletePartition
(
   DFSPARTINFO        *p,                       // IN    partition to work on
   USHORT              disknr                   // IN    verify disknr, or 0
)
{
   ULONG               rc = NO_ERROR;           // function return
   TXLN                s1;

   ENTER();
   TRARGS(("p-info *:%p  ID:%hu  disknr:%hu\n", p, (p) ? p->id : 0, disknr));

   if (p != NULL)
   {
      if ((disknr == 0) || (disknr == FDSK_ANY) || (disknr == p->disknr))
      {
         sprintf(   s1, "Partition %hu disk %hu, %s\nLabel %s by %s\nGPT type: %s named '%s'\n",
                    p->id, p->disknr, p->fsform, p->plabel,   p->creatr, p->descr, p->lvm.VoluName);
         dfstrSizeBps( s1, "Partition size: ", p->sectors, p->bpsector, "\n\n");

         if ((dfsa->batch) || (TxConfirm( 5071, "%sDelete this partition ? [Y/N] : ", s1)))
         {
            rc = dfsSelectDisk( p->disknr, FALSE, FALSE); // Select disk quietly
            if (rc == NO_ERROR)                 // sets base to 0 (lsn == psn)
            {
               DFSDISKINFO *d       =  dfsGetDiskInfo( p->disknr);
               S_GPTHDR    *gptHdr  = (S_GPTHDR *)   d->gptHeader;
               S_GPTENTRY  *gpTable = (S_GPTENTRY *) d->gptArray;

               if (gptHdr && gpTable && (p->partnr < gptHdr->ptaEntries))
               {
                  memset( &gpTable[ p->partnr], 0, gptHdr->ptaEntrySize);

                  disknr = p->disknr;           // Remember disknr (p & d will be invalid!)
                  rc = dfsGptFixupWriteAll( d->gptHeader, d->gptArray);
                  if (rc == NO_ERROR)
                  {
                     TxPrint( "GPT info written successfully to primary and alternate location on disk %hu.\n", disknr);

                     dfsReadDiskInfo( FDSK_QUIET);     //- re-read all partition info
                     TxPrint("\nPartition deleted, GPT partition-table entry set to all zero's\n");
                     dfsFdskSingleAutoShow( disknr );  //- indicate changes
                  }
                  else
                  {
                     TxPrint( "\nWriting the GPT info structures to disk %hu failed!\n", disknr);
                  }
               }
               else
               {
                  rc = DFS_BAD_STRUCTURE;
               }
            }
         }
         else
         {
            rc = DFS_CMD_FAILED;                // aborted
         }
      }
      else
      {
         TxPrint("Partition %hu is not located on disk %hu\n", p->id, disknr);
         rc = DFS_NOT_FOUND;
      }
   }
   RETURN (rc);
}                                               // end 'dfsGptDeletePartition'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Delete ALL GPT partitions from specified disk
/*****************************************************************************/
ULONG dfsGptDeleteAll
(
   DFSDISKINFO        *d                        // IN    info on disk to work on
)
{
   ULONG               rc = NO_ERROR;           // function return
   TXLN                s1;

   ENTER();
   TRARGS(("d-info *:%p   disknr:%hu\n", d, (d) ? d->disknr : 0));

   if (d != NULL)
   {
      USHORT       disknr  = d->disknr;
      S_GPTHDR    *gptHdr  = (S_GPTHDR *)   d->gptHeader;
      S_GPTENTRY  *gpTable = (S_GPTENTRY *) d->gptArray;

      sprintf(   s1, "Disk %hu, GUID: %s\n", disknr, dfsFsUuidValueString(gptHdr->diskGuid));
      dfstrSizeBps( s1, "Disk size from GPT header: ", gptHdr->althPSN + 1, d->bpsector, "\n\n");

      if ((dfsa->batch) || (TxConfirm( 5072, "%sDelete ALL partitions on this disk ? [Y/N] : ", s1)))
      {
         rc = dfsSelectDisk( disknr, FALSE, FALSE); // Select disk quietly
         if (rc == NO_ERROR)                    // sets base to 0 (lsn == psn)
         {
            memset( gpTable, 0, gptHdr->ptaEntries * gptHdr->ptaEntrySize);

            rc = dfsGptFixupWriteAll( d->gptHeader, d->gptArray);
            if (rc == NO_ERROR)
            {
               TxPrint( "GPT info written successfully to primary and alternate location on disk %hu.\n", disknr);

               dfsReadDiskInfo( FDSK_QUIET);    // re-read all partition info
               TxPrint("\nAll partitions deleted from disk, GPT partition-table set to all zero's\n");
               dfsFdskSingleAutoShow( disknr ); // indicate changes
            }
            else
            {
               TxPrint( "\nWriting the GPT info structures to disk %hu failed!\n", disknr);
            }
         }
      }
      else
      {
         rc = DFS_CMD_FAILED;                   // aborted
      }
   }
   RETURN (rc);
}                                               // end 'dfsGptDeleteAll'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Find an empty entry in the GPT partition table array (PTA), in memory
/*****************************************************************************/
static ULONG dfsGptFindEmptyEntry               // RET   PTA index, or DFS32MAX
(
   S_GPTENTRY          ptArray[],               // IN    allocated PT array
   ULONG               ptaEntries,              // IN    max nr of entries
   ULONG               ptaEntrySize,            // IN    size of entry in bytes
   S_GPTENTRY        **emptyEntry               // OUT   empty entry ptr, or NULL
)
{
   ULONG               rc = DFS32MAX;
   ULONG               i;
   BYTE               *ep;                      // BYTE aligned entry pointer

   ENTER();

   for (i = 0,           ep  = (BYTE *) ptArray;
        i < ptaEntries;
        i++,             ep += ptaEntrySize)
   {
      if (TxAreaEmpty( ep, sizeof(S_GPTENTRY), 0)) // use specified one
      {
         *emptyEntry = (S_GPTENTRY *) ep;       // PTA entry structure pointer
         rc = i;
         break;                                 // no need to continue
      }
   }
   RETURN( rc);
}                                               // end 'dfsGptFindEmptyEntry'
/*---------------------------------------------------------------------------*/


