//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// FDISK callback-functions and command implementations
//
// Author: J. van Wijk
//
// JvW  22-03-2001 Updated W2KBM avoiding CHKDSK screen on W2K boot (Daniela)
// JvW  29-06-2000 Initial version, split-off from DFSAFDSK
//

#include <txlib.h>                              // TX library interface

#include <dfsrgkey.h>                           // Registration interface
#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsimage.h>                           // image create restore
#include <dfswipe.h>                            // wipe functions
#include <dfsutil.h>                            // DFS utility functions
#include <dfsupart.h>                           // FDISK partition functions
#include <dfsafdsk.h>                           // FDISK display & analysis
#include <dfscfdsk.h>                           // FDISK callback and commands
#include <dfscflvm.h>                           // FDISK LVM callback and commands
#include <dfsdembr.h>                           // FDISK PTE MBR dialog functions
#include <dfsdflvm.h>                           // FDISK LVM dialog functions
#include <dfsufdsk.h>                           // FDISK utility functions
#include <dfsufgpt.h>                           // FDISK GPT utility functions
#include <dfsdegpt.h>                           // FDISK PTE GPT dialog functions
#include <dfswin.h>                             // windowing functions
#include <dfsver.h>                             // version and naming info
#include <dfsosapi.h>                           // OS specific stuff

#define DFS_PSAVE_SIG     "Psave "              // Signature used in .PDx file
#define DFS_PSAVE_POS     0x76                  // Signature location, fallback


#define BMGR_I13X_SECTOR8  0x106a               // Offset in 'bootsector' = 0x106a
#define BMGR_I13X_OFFSET1  0x38                 // offset 1st jump instruction
#define BMGR_I13X_OFFSET2  0x4b                 // offset 2nd jump instruction
#define BMGR_I13X_SECTORS  9                    // need sectors 0..8 for patch

static char    dfsFdskCchs[] = " set to: ";

//- to be refined need another for Fat16 and Win-98 ...
//- and possibly Fat32 and Win-NT


// Fix BMGR bootsector for W2K immunity and correct CHS for this/data sector
static ULONG dfsFdskBmFixBootW2K                // RET PENDING  => call again
(                                               //     NO_ERROR => all done
   BYTE               *sec,                     // INOUT sectors with BM BR
   FDSK_CB_PHASE       phase,                   // IN    callback phase
   FDSK_CB_INFO       *cbp                      // IN    operation info
);


// Implement the 'PriBmName' callback operation on the bootmanager sectors
// Note: cbp->more holds info on the related primary partition (like disknr)
//       cbp->disknr holds info on BMGR disk and partition
static ULONG dfsFdskPriBmName                   // RET PENDING  => call again
(                                               //     NO_ERROR => all done
   BYTE               *sec,                     // INOUT sectors with BM table
   FDSK_CB_PHASE       phase,                   // IN    callback phase
   FDSK_CB_INFO       *cbp                      // IN    operation info
);


// Implement the 'LogBmName' callback operation, set BM-name in EBR sector
static ULONG dfsFdskLogBmName                   // RET PENDING  => call again
(                                               //     NO_ERROR => all done
   DFSPARTINFO        *p,                       // INOUT partition info
   S_BOOTR            *br,                      // INOUT sector with Ptable
   FDSK_CB_PHASE       phase,                   // IN    callback phase
   FDSK_CB_INFO       *cbp                      // IN    operation info
);

// Implement the 'Psave' callback operation on bootsectors
static ULONG dfsFdskPsaveBoot                   // RET PENDING  => call again
(                                               //     NO_ERROR => all done
   DFSPARTINFO        *p,                       // IN    p-info
   S_BOOTR            *sec,                     // INOUT partition boot record
   FDSK_CB_PHASE       phase,                   // IN    callback phase
   FDSK_CB_INFO       *cbp                      // IN    callback parameters
);


// Implement the 'Psave' callback operation on part-table and LVM-sector
static ULONG dfsFdskPsaveSect                   // RET PENDING  => call again
(                                               //     NO_ERROR => all done
   BYTE               *sec,                     // IN    a MBR/EBR/LVM/BR sect
   FDSK_CB_PHASE       phase,                   // IN    callback phase
   FDSK_CB_INFO       *cbp                      // IN    callback parameters
);

// Implement the 'Psave' sector-save function for GPT specific information
static ULONG dfsFdskPsaveGptInfo
(
   FDSK_CB_INFO       *cbp                      // IN    disk information
);

// Implement the 'Psave' sector-save read/callback for GPT specific sectors
static ULONG dfsFdskPsaveGptSect
(
   FDSK_CB_INFO       *cbp                      // IN    disk/sect information
);

// PRESTORE callback operating on PsaveD sectors + info, restore to disk
static ULONG dfsRestorePsaveD
(
   BYTE               *sec,                     // IN    sector contents
   DFSPS_SEC          *psi,                     // IN    PsaveD sector info
   FDSK_CB_INFO       *info                     // INOUT function info+data
);


// Check and update contents of one partition-table entry for FixChs
static ULONG dfsFixChsEntry                     // RET   PENDING ==> updated
(
   DFSPARTENTRY       *pe,                      // IN    part entry ptr
   FDSK_CB_INFO       *cbp                      // IN    callback parameters
);


/*****************************************************************************/
// Implement the 'Create' callback operation on a partition-table entry
/*****************************************************************************/
ULONG dfsFdskCreate                             // RET PENDING  => call again
(                                               //     NO_ERROR => all done
   DFSPARTINFO        *f,                       // INOUT freespace info
   S_BOOTR            *br,                      // INOUT sector with Ptable
   FDSK_CB_PHASE       phase,                   // IN    callback phase
   FDSK_CB_INFO       *cbp
)
{
   ULONG               rc = NO_ERROR;           // function return
   DFSPARTENTRY       *pe;                      // partition-table entry
   USHORT              partnr;                  // Ptable entry-number
   DFSPARTINFO        *e;                       // new EBR info structure
   DFSPARTINFO        *prevEbr;                 // previous EBR info structure

   ENTER();
   TRACES(("Phase:%u Related:%u Confirmed:%u Option:%u Flag:%u  Disk:%hu thisSn:%llx\n",
            phase, cbp->related, cbp->confirmed, cbp->option, cbp->flag, cbp->disknr, cbp->thisSn));
   TRARGS(("freespace-info *:%p\n", f));

   if (f != NULL)                               // info available
   {
      switch (phase)
      {
         case FDSK_PREPARE:
            strcat( cbp->string, "Create the partition as shown ? [Y/N] : ");

            if ((dfsa->batch    == FALSE) &&
                (cbp->confirmed == FALSE) &&
                !TxConfirm( 5036, cbp->string))
            {
               rc = DFS_CMD_FAILED;             // aborted
            }
            break;

         case FDSK_PERFORM:
            f->partnr = (USHORT) cbp->cbOptNum2; // prefered entry index
            if (dfsFindPtableEntry( br, DFS_P_EMPTY, &f->partnr))
            {
               rc = DFS_PENDING;                // request follow-up action
               TRACES(("Adding new partition itself to partition-table\n"));
               pe = &(br->PartitionTable[f->partnr]);
               dfsFillPtableEntry( f, cbp, pe);
               if (cbp->cbOption1)              // force bootable status flag
               {
                  pe->Status = 0x80;            // set bootable flag in entry
               }
               f->partent = *pe;                // copy to info-struct too
               TRHEXS( 70, pe, 16, "new table entry");
               if (cbp->flag)                   // primary, check visibility
               {
                  if (cbp->cbOption2 == FALSE)  // when hiding not suppressed
                  {
                     if (dfsVisPrimaries( br) > 1) // more than just me ?
                     {
                        if (dfsPartTypeHidable(pe->PartitionType))
                        {
                           pe->PartitionType |= DFS_P_PHIDDEN;

                           TxPrint("The created partition will be invisible "
                                   "because another primary is active.\n");
                           TxPrint("To make it active (visible) use the "
                                   "'SETACCESS' command.\n");
                        }
                     }
                  }
               }
               dfsX10("Add to Ptable  at : ", f->partPsn, CBG, "  ");
               TxPrint("Using entry %u for a type 0x%2.2X partition\n",
                        f->partnr, pe->PartitionType);
               if (cbp->flag == FALSE)          // logical, attach EBR chain
               {
                  if (strlen( cbp->bname))      // and write BMGR-name in EBR
                  {                             // when specified
                     pe->Status      = 0x80;    // set bootable flag in entry
                     br->BootMgrFlag = BT_BM_BOOT; // and in EBR-sector
                     strcpy( br->BootMgrName, cbp->bname);
                  }
                  else                          // write Psn Bm info
                  {
                     sprintf( br->BootMgrName, "%8.8x", (ULONG) cbp->sn);
                  }
                  if (f->ebrChain != NULL )     // 'next' in chain present
                  {
                     sprintf( br->BootMgrNext, "%8.8x", (ULONG) f->ebrChain->basePsn);
                     pe = &f->ebrChain->partent; // address of next's Pentry
                     partnr = 0;                // prefered entry
                     if (dfsFindPtableEntry( br, DFS_P_EMPTY, &partnr))
                     {
                        dfsX10("Link next EBR  in : ", f->partPsn, CNG, "  ");
                        TRACES(("Adding link to next EBR partition-table\n"));
                        TxPrint("using entry %u for a type 0x%2.2X partition\n",
                                 partnr, pe->PartitionType);
                        br->PartitionTable[partnr] = *pe;
                        if (f->ebrChain->partPsn == 0) // was it the ebrBase ?
                        {
                           //- make it relative to this new ebrBase record
                           br->PartitionTable[partnr].BootSectorOffset -= (ULONG) cbp->sn;
                        }
                     }
                     else                       // impossible, the table is
                     {                          // just created and has
                        rc = DFS_BAD_STRUCTURE; // one entry sofar ...
                     }
                  }
                  if ((e = dfsNewEbrInfo( f)) != NULL) // clone f to new-EBR
                  {
                     //- update required EBR info fields in new-EBR block
                     e->basePsn = f->partPsn;   // sn of EBR sector (Ptable)
                     e->lastPsn = cbp->sn + cbp->number   -1;
                     e->sectors = e->lastPsn - e->basePsn +1;

                     if (dfsInsertInMemEbrChain( e->disknr,
                                                 e->ebrChain,
                                                 e,
                                                 &prevEbr))
                     {
                        if (prevEbr == NULL)    // e is now ebrHead -> MBR
                        {
                           e->partPsn = 0;
                        }
                        else                    // in chain, partent at start
                        {                       // of the previous EBR area
                           e->partPsn = prevEbr->basePsn;
                        }
                        //- make in-memory updates to the ext-chain
                        if (dfsFdskUpdateExtChain( e->disknr) == NO_ERROR)
                        {
                           FDSK_CB_INFO  cbs;   // info for 'sub' operations

                           memset( &cbs, 0, sizeof(cbs)); // initial cb info

                           cbs.sn      = e->basePsn;
                           cbs.number  = e->sectors;
                           cbs.ntype   = DFS_P_EXTENDED;
                           cbs.flag    = TRUE;

                           dfsFillPtableEntry( e, &cbs, &(e->partent));
                        }
                        else
                        {
                           rc = DFS_BAD_STRUCTURE;
                        }
                     }
                     else
                     {
                        TxPrint("Error inserting into internal ebrChain\n");
                        rc = DFS_BAD_STRUCTURE;
                     }
                  }
                  else
                  {
                     rc = DFS_ALLOC_ERROR;
                  }
               }
            }
            else
            {
               rc = DFS_CMD_FAILED;             // no free entry, abort
            }
            break;

         case FDSK_PENDING:
            if (cbp->flag == FALSE)             // logical, update EBR-chain
            {                                   // on the disk itself
               FDSK_CB_INFO  cbs;               // info for 'sub' operations

               memset( &cbs, 0, sizeof(cbs));   // initial cb info
               cbs.disknr    = f->disknr;
               cbs.confirmed = TRUE;            // no more prompting ...
               if (cbp->creMbr != FDSK_CR_MBR_OFF)
               {
                  cbs.creMbr = cbp->creMbr;  // inherit ...
               }
               if (dfsIterateExtContainers( dfsFdskWriteExtChain, &cbs) != NO_ERROR)
               {
                  TxPrint("Error updating on-disk ebrChain\n");
                  rc = DFS_BAD_STRUCTURE;
               }
            }
            else                                // primary, set BM-name
            {
               rc = dfsFdskSetPriBm( f, cbp->bname, TRUE);
            }
            if (rc == NO_ERROR)
            {
               if (cbp->ntype == DFS_P_EFI_GPT) // GPT guard partition
               {
                  //- create GPT structures, with optional recovery of existing
                  rc = dfsGptCreateFromMbr((cbp->cbOption3 == FALSE), f);
               }
               else                             // MBR style, check image
               {
                  FILE        *fp;
                  TXLN         fname;
                  TXA_OPTION  *opt;
                  BOOL         bootmgr = FALSE;

                  strcpy( fname, "");
                  if (!TxaOptUnSet('I'))        // unless no image wanted
                  {
                     if ((opt = TxaOptValue('I')) != NULL) // explicit Image
                     {
                        switch (opt->type)
                        {
                           case TXA_NO_VAL: strcpy(  fname, "");                          break;
                           case TXA_STRING: strcpy(  fname,         opt->value.string);   break;
                           default:         sprintf( fname, "%lld", opt->value.number);   break;
                        }
                        if (strlen( fname) == 0)
                        {
                           if (dfsa->batch)     // batch-mode, use default name
                           {
                              sprintf( fname, "dfsnew%2.2hhx.img", cbp->ntype);
                           }
                           else
                           {
                              TxPrompt( 5521, TXMAXLN, fname, "Initialization Image filename");
                           }
                        }
                     }
                     else if (cbp->ntype == DFS_P_BOOTMGR)
                     {
                        strcpy( fname, DFSFDSK_BMGRIMG);
                        bootmgr = TRUE;
                     }
                  }

                  if (strlen( fname) != 0)
                  {
                     TXLN               text;

                     if ((fp = fopen( fname, "rb")) == NULL) // not present here
                     {
                        if (TxFindByPath( fname, "PATH", fname) != NULL)
                        {
                           fp = fopen( fname, "rb");
                        }
                     }
                     if (fp != NULL)            // present
                     {
                        fclose( fp);
                        if ((dfsa->batch) || TxConfirm( 5041, "Initialize partition with the contents "
                                                              "from imagefile '%s' ? [Y/N] : ", fname))
                        {
                           rc = dfsSelectDisk( cbp->disknr, FALSE, FALSE);
                           if (rc == NO_ERROR)
                           {
                              if (DFSTORE_WRITE_ALLOWED)
                              {
                                 TRACES(("Initialize partition using: %s\n", fname));

                                 rc = dfsRestoreImage( cbp->sn, (bootmgr) ? DFSFDSK_BMGRSEC : cbp->number, 0, fname, 1);
                                 if ((rc == NO_ERROR) && (bootmgr))
                                 {
                                    DFSDISKINFO      *d;

                                    if ((d = dfsGetDiskInfo( cbp->disknr)) != NULL)
                                    {
                                       d->bmgrPsn = cbp->sn;

                                       if ((d->disknr == 1) && !dfsa->mbr1I13X) // DFSee BMGR requires I13X!
                                       {
                                          sprintf( text, "IBM BMGR requires 'I13X' but MBR does not seem to set that!\n"
                                                         "Refresh MBR code using the command 'NEWMBR 1' or menu item");
                                          TxNamedMessage( !dfsa->batch, 0, " WARNING: MBR code mismatch ", text);
                                       }
                                    }
                                    rc = dfsFdskBmgrFixBootSec( cbp->disknr, TRUE);
                                 }
                              }
                              else
                              {
                                 rc = DFS_READ_ONLY;
                              }
                           }
                        }
                     }
                     else                       // file open error
                     {
                        sprintf( text, "Imagefile '%s' was not found, new partition not initialized!", fname);
                        if (bootmgr)
                        {
                           strcat( text, "\n\nBootmanager-code is not installed!  Related "
                                         "commands like setname\nand setboot might give "
                                         "unpredictable results");
                        }
                        TxNamedMessage( !dfsa->batch, 0, " ERROR: IBM bootmanager missing ", text);
                        rc = ERROR_FILE_NOT_FOUND;
                     }
                  }
                  else if (cbp->cbOption3)      // no image, CLEAR bootsector ?
                  {
                     rc = dfsSelectDisk( cbp->disknr, FALSE, FALSE);
                     if (rc == NO_ERROR)
                     {
                        if (DFSTORE_WRITE_ALLOWED)
                        {
                           SEC512 pattern;

                           TRACES(("Initialize bootsector with 0xF6 pattern\n"));

                           memset( pattern, 0xf6, SECTORSIZE);
                           rc = dfsWipeArea( cbp->sn, 1, DFSP_NONE, NULL, FALSE, pattern, SECTORSIZE);
                        }
                        else
                        {
                           rc = DFS_READ_ONLY;
                        }
                     }
                  }
               }
            }
            dfsFdskSingleAutoShow( cbp->disknr); // indicate changes
            break;

         default:
            rc = DFS_CMD_FAILED;
            break;
      }
   }
   RETURN (rc);
}                                               // end 'dfsFdskCreate'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Cleanup Ptable status flags and EBR-chain on a disk, removing 'empty' EBR's
/*****************************************************************************/
ULONG dfsCleanupExtChain
(
   USHORT              disknr,                  // IN    disknr to work on
   BOOL                confirmed                // IN    already confirmed
)
{
   ULONG               rc = NO_ERROR;           // function return
   FDSK_CB_INFO  cbs;                           // info for 'sub' operations

   ENTER();

   memset( &cbs, 0, sizeof(cbs));               // initial cb info
   cbs.disknr    = disknr;
   cbs.confirmed = confirmed;

   if (dfsFdskUpdateExtChain( disknr) == NO_ERROR)
   {
      if (dfsIterateExtContainers( dfsFdskWriteExtChain, &cbs) == NO_ERROR)
      {
         rc = dfsIteratePart( dfsFdskCleanupAct, &cbs);

         //- corrupted (in-memory only) chain by dfsFdskUpdateExtChain()
         //- caused symptoms seen with P#1345
         /*
         if (dfsIteratePmbr( dfsFdskCleanupAct, &cbs) == NO_ERROR)
         {
            // cbs.cbOptNum1 = 0;
            rc = dfsIterateExtContainers( dfsFdskCleanupAct, &cbs);
         }
         else
         {
            TxPrint("Error updating MBR tables\n");
            rc = DFS_BAD_STRUCTURE;
         }
         */
      }
      else
      {
         TxPrint("Error updating on-disk ebrChain\n");
         rc = DFS_BAD_STRUCTURE;
      }
   }
   else
   {
      rc = DFS_BAD_STRUCTURE;
   }
   RETURN (rc);
}                                               // end 'dfsCleanupExtChain'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Implement the 'WriteExtChain' callback operation on a partition-table entry
/*****************************************************************************/
ULONG dfsFdskWriteExtChain                      // RET PENDING  => call again
(                                               //     NO_ERROR => all done
   DFSPARTINFO        *p,                       // INOUT partition info
   S_BOOTR            *br,                      // INOUT sector with Ptable
   FDSK_CB_PHASE       phase,                   // IN    callback phase
   FDSK_CB_INFO       *cbp                      // IN    option:zero full entry
)
{
   ULONG               rc = NO_ERROR;           // function return
   DFSPARTENTRY       *pe;                      // partition-table entry
   USHORT              partnr = FDSK_ANY;       // Ptable entry-number
   BYTE                ctype;

   ENTER();
   TRACES(("Phase:%u Related:%u Confirmed:%u Option:%u Flag:%u  Disk:%hu thisSn:%llx\n",
            phase, cbp->related, cbp->confirmed, cbp->option, cbp->flag, cbp->disknr, cbp->thisSn));
   TRARGS(("p:%p  p->id:%2u  p->pitype:%u\n", p, p->id, p->pitype));

   if ((p != NULL)   &&                         // info available
       (p->pitype == DFS_PI_EBR_CHAIN))         // info has valid type
   {
      switch (phase)
      {
         case FDSK_PREPARE:
            if ((dfsa->batch    == FALSE) &&
                (cbp->confirmed == FALSE) &&
                !TxConfirm( 5046,
                           "Cleanup partition-table information for "
                           "disk %u, in table at 0x%0llx ? [Y/N] : ",
                            p->disknr, p->partPsn))
            {
               rc = DFS_CMD_FAILED;             // aborted
            }
            else
            {
               cbp->confirmed = TRUE;           // only confirm once in
            }                                   // a chain update session
            break;

         case FDSK_PERFORM:
            ctype = DFS_P_EXTENDED;
            if (!dfsFindPtableEntry(             br, ctype, &partnr))
            {
               ctype = DFS_P_BIGEXTEND;
               if (!dfsFindPtableEntry(          br, ctype, &partnr))
               {
                  ctype = DFS_P_HIDEXTEND;
                  if (!dfsFindPtableEntry(       br, ctype, &partnr))
                  {
                     ctype = DFS_P_LINUXEXTX;
                     if (!dfsFindPtableEntry(    br, ctype, &partnr))
                     {
                        ctype = DFS_P_BIGLINEXT;
                        if (!dfsFindPtableEntry( br, ctype, &partnr))
                        {
                           ctype = DFS_P_EMPTY;
                           if (p->partPsn == 0)     // it is the MBR extended container
                           {
                              if (TxaOption('X'))   // user overrule prefered slot
                              {
                                 partnr = TxaOptNum( 'X', NULL, FDSK_ANY);
                              }
                              else
                              {
                                 partnr = 3;        // prefer last slot
                              }
                           }
                           else
                           {
                              if (TxaOption('Y'))  // user overrule prefered slot
                              {
                                 partnr = TxaOptNum( 'Y', NULL, FDSK_ANY);
                              }
                              else
                              {
                                 partnr = 1;       // prefer 2nd slot
                              }
                           }
                           dfsFindPtableEntry(   br, ctype, &partnr);
                        }
                     }
                  }
               }
            }
            if (partnr != FDSK_ANY)                 // found entry to be updated
            {
               if ((p->partent.NumberOfSectors)  && // non-zero partition entry
                   (ctype != DFS_P_EMPTY)         ) // and not an existing one
               {
                  p->partent.PartitionType = ctype; // keep same container type
               }
               pe = &(br->PartitionTable[partnr]);
               TRACES(( "EBR at: %llx, entry: %hu\n", p->partPsn, partnr));
               TRHEXS( 70, pe, 16,          "on-disk   table entry");
               TRHEXS( 70, &p->partent, 16, "in-memory table entry");
               if (memcmp( pe, &p->partent, sizeof(DFSPARTENTRY)))
               {
                  dfsX10("Write EBR link in : ", p->partPsn, CBG, "  ");
                  TRACES(("Synchronize on-disk tables with in-memory list\n"));
                  TxPrint("Using entry %u for a type 0x%2.2X partition\n",
                           partnr, p->partent.PartitionType);
                  memcpy(  pe, &p->partent, sizeof(DFSPARTENTRY));
                  rc = DFS_PENDING;
               }
               else                             // no change needed here
               {
                  TRACES(("Ptable at:%llx, entry:%u, type:%2.2x is OK\n",
                           p->partPsn, partnr, p->partent.PartitionType));
                  rc = NO_ERROR;
               }
            }
            else                                // no ext-container entry
            {                                   // or empty entry found
               rc = DFS_BAD_STRUCTURE;
            }
            break;

         case FDSK_PENDING:
            fdsk->AutoShow = TRUE;              // indicate changes
            break;

         default:
            rc = DFS_CMD_FAILED;
            break;
      }
   }
   RETURN (rc);
}                                               // end 'dfsFdskWriteExtChain'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Implement the 'Cleanup Active' callback operation on a partition-table entry
// Note: always called in an iteration for ONE disk at a time!
/*****************************************************************************/
ULONG dfsFdskCleanupAct                         // RET PENDING  => call again
(                                               //     NO_ERROR => all done
   DFSPARTINFO        *p,                       // INOUT partition info
   S_BOOTR            *br,                      // INOUT sector with Ptable
   FDSK_CB_PHASE       phase,                   // IN    callback phase
   FDSK_CB_INFO       *cbp                      // IN    option:zero full entry
)
{
   ULONG               rc = NO_ERROR;           // function return
   DFSPARTENTRY       *pe;                      // partition-table entry

   ENTER();
   TRACES(("Phase:%u Related:%u Confirmed:%u Option:%u Flag:%u  Disk:%hu thisSn:%llx\n",
            phase, cbp->related, cbp->confirmed, cbp->option, cbp->flag, cbp->disknr, cbp->thisSn));
   if (p != NULL)                               // info available
   {
      switch (phase)
      {
         case FDSK_PREPARE:
            break;

         case FDSK_PERFORM:
            pe = &(br->PartitionTable[p->partnr]);
            TRACES(( "Part:%hu table-index:%hu flag:%2.2hx  Actives:%u\n",
                      p->id, p->partnr, pe->Status, cbp->cbOptNum1));
            if (pe->Status != 0)
            {
               if (p->primary)                  // sanity check ...
               {
                  if (pe->Status  & DFS_P_ACTIVE)
                  {
                     cbp->cbOptNum1++;          // actives on THIS disk sofar
                  }
                  if (cbp->cbOptNum1 > 1)       // more than one ACTIVE!
                  {
                     TxPrint("Fix status on Pid : % 2.2hu, "
                             "reset multiple ACTIVE partitions!\n", p->id);
                     pe->Status = 0;            // reset whole byte, incl flag
                     rc = DFS_PENDING;
                  }
               }
               else                             // ACTIVE logical
               {
                  TxPrint("Fix status on Pid : % 2.2hu, "
                          "reset ACTIVE flag for logical partition!\n", p->id);
                  pe->Status = 0;               // reset whole byte, incl flag
                  rc = DFS_PENDING;
               }
               if ((rc == NO_ERROR) && (pe->Status != DFS_P_ACTIVE)) // zero other bits
               {
                  TxPrint("Fix status on Pid : % 2.2hu, reset non standard "
                          "status flag value 0x%2.2hx!\n", p->id, pe->Status);
                  pe->Status &= DFS_P_ACTIVE;   // reset non-std flag-bits
                  rc = DFS_PENDING;
               }
               TRACES(( "Status now %2.2hx, Actives: %u\n", pe->Status, cbp->cbOptNum1));
            }
            break;

         case FDSK_PENDING:
            dfsFdskSingleAutoShow( p->disknr);
            break;

         default:
            rc = DFS_CMD_FAILED;
            break;
      }
   }
   RETURN (rc);
}                                               // end 'dfsFdskCleanupAct'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Update bootmanager BOOTSECTOR for W2K CHKDSK immunity and CHS value issues
/*****************************************************************************/
ULONG dfsFdskBmgrFixBootSec
(
   USHORT              disknr,                  // IN    disk with a BM part
   BOOL                confirmed                // IN    skip confirm prompt
)
{
   ULONG               rc = NO_ERROR;           // function return
   DFSDISKINFO        *d;                       // bmgr disk info
   FDSK_CB_INFO        cbs;                     // info for 'sub' operations

   ENTER();

   memset( &cbs, 0, sizeof(cbs));               // initial cb info
   cbs.disknr    = (disknr) ? disknr : dfsa->bmgrDisk;
   cbs.confirmed =  confirmed;

   if (((d = dfsGetDiskInfo( cbs.disknr)) != NULL) && (d->bmgrPsn != L64_NULL))
   {
      cbs.sn        = d->bmgrPsn;               // first sector in BMGR
      cbs.number    = BMGR_I13X_SECTORS;        // need more than one sector
                                                // for the I13X patch!
      rc = dfsExecOnSectors( dfsFdskBmFixBootW2K, &cbs);
   }
   else
   {
      TxPrint("There is no bootmanager present on ");
      if (cbs.disknr == dfsa->bmgrDisk)
      {
         TxPrint("any disk\n");
      }
      else
      {
         TxPrint("disk %u\n", cbs.disknr);
      }
      rc = DFS_NOT_FOUND;
   }
   RETURN (rc);
}                                               // end 'dfsFdskBmgrFixBootSec'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Fix BMGR bootsector for W2K immunity and correct CHS for this/data sector
/*****************************************************************************/
static ULONG dfsFdskBmFixBootW2K                // RET PENDING  => call again
(                                               //     NO_ERROR => all done
   BYTE               *sec,                     // INOUT sectors with BM BR
   FDSK_CB_PHASE       phase,                   // IN    callback phase
   FDSK_CB_INFO       *cbp                      // IN    operation info
)
{
   ULONG               rc = NO_ERROR;           // function return

   ENTER();
   TRACES(("Phase:%u Related:%u Confirmed:%u Option:%u Flag:%u  Disk:%hu thisSn:%llx\n",
            phase, cbp->related, cbp->confirmed, cbp->option, cbp->flag, cbp->disknr, cbp->thisSn));
   if (sec != NULL)
   {
      S_BOOTR         *br = (S_BOOTR *) sec;
      TXLN             text;
      BOOL             i13x = (br->os.RestCode[ BMGR_I13X_INBOOT] != DFS_B_OPCJMPS);
      BOOL             fix1 = (br->eb.FatOffset <  DFSFDSK_BMGRSEC);
      BOOL             fix2 = dfsFixChsI13Changed( NULL, FALSE, cbp->thisSn,
                                  &br->os.BmBootSecCyl, &br->os.BmBootHead);
      BOOL             fix3 = dfsFixChsI13Changed( NULL, FALSE, cbp->thisSn +1,
                                  &br->os.BmDataSecCyl, &br->os.BmDataHead);
      BOOL             fix4 = (i13x != TxaOption( DFS_O_I13X));

      switch (phase)
      {
         case FDSK_PREPARE:
            sprintf( text,
               "BMGR disk : %hu, W2K CHKDSK survival and CHS value check.\n\n"
               "Bootsector shows %2hu reserved sectors: %sW2K-proof      %s\n"
               "CHS values for BOOT are %svalid for BMGR location      %s\n"
               "CHS values for DATA are %svalid for BMGR location      %s\n"
               "I13X capable MBR code %srequired to boot > Cyl-1024    %s",
                cbp->disknr, br->eb.FatOffset,
                (fix1) ? "NOT " : "",     (fix1) ? "fix!" : "    OK",
                (fix2) ? "NOT " : "",     (fix2) ? "fix!" : "    OK",
                (fix3) ? "NOT " : "",     (fix3) ? "fix!" : "    OK",
                (i13x) ? " is " : "NOT ", (fix4) ? "change?" :  "OK?");

            if ((fix1 || fix2 || fix3 || fix4) && !dfsa->batch && !cbp->confirmed)
            {
               if (!TxConfirm( 5056, "%s\n\nUpdate the bootsector ? [Y/N] : ", text))
               {
                  rc = DFS_CMD_FAILED;          // aborted
               }
            }
            else                                // just print for logging ...
            {
               TxPrint( "%s\n", text);
            }
            break;

         case FDSK_PERFORM:
            if (fix1 || fix2 || fix3 || fix4)   // any change needed ?
            {
               if (fix1)
               {
                  br->eb.ClustSize    = 0;
                  br->eb.FatOffset    = DFSFDSK_BMGRSEC;
                  br->eb.RootEntries  = 0;
                  br->eb.Sectors      = 0;
                  memcpy( br->os.Type, "BMGR    ", 8);
               }
               if (fix2)
               {
                  dfsFixChsI13Changed( "CHS BM Bootsector : ", TRUE, cbp->thisSn,
                                       &br->os.BmBootSecCyl, &br->os.BmBootHead);
               }
               if (fix3)
               {
                  dfsFixChsI13Changed( "CHS BM Datasector : ", TRUE, cbp->thisSn +1,
                                       &br->os.BmDataSecCyl, &br->os.BmDataHead);
               }
               if (fix4)                        // check for correct code-match
               {                                // for W4-FP15 and W4.5x or eCS
                  BYTE    *sectordata = (BYTE *) br;
                  ULONG    patch = 0;           // version specific offset

                  if ((br->os.RestCode[ BMGR_I13X_INBOOT  + 1] == BMGR_I13X_OFFSET1) &&
                      (    (sectordata[ BMGR_I13X_SECTOR8 - 1] == 'X') ||
                           (sectordata[ BMGR_I13X_SECTOR8 + 1] == 'X')))
                  {
                     if (sectordata[ BMGR_I13X_SECTOR8 + 1] == 'X')
                     {
                        patch += 2;
                     }
                     if (TxaOption( DFS_O_I13X)) // keep explicit I13X check
                     {
                        TxPrint("I13X-capable BMGR code, requires I13X capable MBR too!\n");
                        br->os.RestCode[ BMGR_I13X_INBOOT] = DFS_B_OPCJE;
                        sectordata[ patch + BMGR_I13X_SECTOR8    ] = DFS_B_OPCJNE;
                        sectordata[ patch + BMGR_I13X_SECTOR8 + 1] = BMGR_I13X_OFFSET2;
                     }
                     else
                     {
                        TxPrint("I13X-capable BMGR code, no I13X check (allow GRUB etc).\n");
                        br->os.RestCode[ BMGR_I13X_INBOOT] = DFS_B_OPCJMPS;
                        sectordata[ patch + BMGR_I13X_SECTOR8    ] = DFS_B_OPCNOP;
                        sectordata[ patch + BMGR_I13X_SECTOR8 + 1] = DFS_B_OPCNOP;
                     }
                  }
                  else
                  {
                     TxPrint("I13X-capable BMGR code not detected, fix not possible.\n");
                  }
               }
               rc = DFS_PENDING;                // request write-back
            }
            else
            {
               TxPrint( "\nNo change needed ...\n");
            }
            break;

         case FDSK_PENDING:
            TxPrint( "\nIBM BMGR bootsector CHS-consistent, W2K CHKDSK proof, and I13X capability set.\n");
            break;

         default:
            rc = DFS_CMD_FAILED;
            break;
      }
   }
   RETURN (rc);
}                                               // end 'dfsFdskBmFixBootW2K'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set the bootmanager-info for a primary partition, incl extended-container
// Note: Empty name "" will clear the name, and place the basePsn info there
// Note: No name, NULL will clear the name, and zero the whole info entry
/*****************************************************************************/
ULONG dfsFdskSetPriBm
(
   DFSPARTINFO        *p,                       // IN    partition info
   char               *name,                    // IN    BM name, NULL or ""
   BOOL                confirmed                // IN    already confirmed
)
{
   ULONG               rc = NO_ERROR;           // function return

   ENTER();

   if (dfsa->bmgrDisk != 0 )                    // if BM is present
   {
      FDSK_CB_INFO     cbs;                     // info for 'sub' operations
      DFSDISKINFO     *d;                       // bmgr disk info
      USHORT           cdsk = SINF->disknr;     // current disk

      d = dfsGetDiskInfo(dfsa->bmgrDisk);
      if (d != NULL)
      {
         memset( &cbs, 0, sizeof(cbs));         // initial cb info
         cbs.confirmed = confirmed;             // prompting
         cbs.disknr    = dfsa->bmgrDisk;
         cbs.more      = p;                     // disknr, partnr, id, drive
         cbs.sn        = d->bmgrPsn +
                         BMPRIMSECTN;
         cbs.number    = BMPRIMSECTS;

         if (name && strlen( name))             // valid name specified
         {
            memset( cbs.string, ' ', BT_BM_L);
            memcpy( cbs.string, name,  strlen(name));
            cbs.string[ BT_BM_L] = '\0';
         }
         else
         {
            strcpy( cbs.string, "");
            cbs.option = TRUE;                  // reset name in BM structure
            if (name == NULL)                   // delete partition
            {
               cbs.flag = TRUE;
            }
         }
         rc = dfsExecOnSectors( dfsFdskPriBmName, &cbs);
         if (cdsk != SINF->disknr)              // active disk changed ?
         {
            rc = dfsSelectDisk( cdsk, FALSE, FALSE); // Re-Select disk quietly
         }
      }
   }
   else if (name && strlen( name))
   {
      TxPrint("There is no bootmanager menu to set the name in ...\n"
              "The specified name '%s' will be ignored.\n", name);
   }
   RETURN (rc);
}                                               // end 'dfsFdskSetPriBm'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Implement the 'PriBmName' callback operation on the bootmanager sectors
// Note: cbp->more holds info on the related primary partition (like disknr)
//       cbp->disknr holds info on BMGR disk and partition
//       cbp->option indicates clearing of the name    ==> record Psn info
//       cbp->flag   indicates delete of the partition ==> zero whole field
/*****************************************************************************/
static ULONG dfsFdskPriBmName                   // RET PENDING  => call again
(                                               //     NO_ERROR => all done
   BYTE               *sec,                     // INOUT sectors with BM table
   FDSK_CB_PHASE       phase,                   // IN    callback phase
   FDSK_CB_INFO       *cbp                      // IN    operation info
)
{
   ULONG               rc = NO_ERROR;           // function return
   DFSPARTINFO        *p;                       // related partition info
   BMPRIMARY          *bmi = (BMPRIMARY *) sec; // Bootmgr info, one partition
   ULONG               index;

   ENTER();
   TRACES(("Phase:%u Related:%u Confirmed:%u Option:%u Flag:%u  Disk:%hu thisSn:%llx\n",
            phase, cbp->related, cbp->confirmed, cbp->option, cbp->flag, cbp->disknr, cbp->thisSn));
   if ((sec != NULL) && (cbp->more != NULL))    // info available
   {
      p     = (DFSPARTINFO *) cbp->more;        // related partition

      dfsFdskGetBmiIndex( p->disknr, &p->partent, bmi, &index);
      if (index != DFS32MAX)                    // existing/free index found
      {
         switch (phase)
         {
            case FDSK_PREPARE:
               if ((dfsa->batch    == FALSE) &&
                   (cbp->confirmed == FALSE) &&
                   !TxConfirm( 5061,
                              "%s '%s' for partition %02u (%s) ? [Y/N] : ",
                       (cbp->option) ? (cbp->flag) ? "Delete BM-info" :
                                                     "Remove BM-name" :
                                                     "Set as BM-name",
                       (cbp->option) ? bmi[index].BootMgrName
                                     : cbp->string, p->id, p->drive))
               {
                  rc = DFS_CMD_FAILED;          // aborted
               }
               break;

            case FDSK_PERFORM:
               if (cbp->flag == FALSE)          // value update required
               {
                  bmi[index].BootDisk    = (BYTE) (p->disknr -1) |
                                           p->partent.Status;
                  bmi[index].FirstSector = p->partent.FirstSector;
                  if (cbp->option)
                  {
                     bmi[index].BootMgrFlag = 0;
                     sprintf( bmi[index].BootMgrName, "%8.8x", (ULONG) p->basePsn);
                  }
                  else
                  {
                     bmi[index].BootMgrFlag = BT_BM_BOOT;
                     strcpy( bmi[index].BootMgrName, cbp->string);
                  }
               }
               else                             // del partition, zero fields
               {
                  memset( &bmi[index], 0, sizeof(BMPRIMARY));
               }
               rc = DFS_PENDING;
               break;

            case FDSK_PENDING:
               dfsFdskSingleAutoShow( p->disknr);
               break;

            default:
               rc = DFS_CMD_FAILED;
               break;
         }
      }
      else
      {
         TxPrint("Required existing/free entry not found in BMGR info!\n");
         if (cbp->option && cbp->flag)          // delete operation
         {
            TxPrint( "Partition deleted, but some BMGR info is remaining\n");
         }
         else
         {
            TxPrint( "BMGR info for this partition is not created/updated\n");
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsFdskPriBmName'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set the bootmanager-name for a logical partition
/*****************************************************************************/
ULONG dfsFdskSetLogBm
(
   DFSPARTINFO        *p,                       // IN    partition info
   char               *name,                    // IN    BM name, NULL or ""
   BOOL                confirmed                // IN    already confirmed
)
{
   ULONG               rc = NO_ERROR;           // function return
   FDSK_CB_INFO        cbs;                     // info for operations

   ENTER();

   if ((dfsa->bmgrDisk == 0 )  &&               // if BM is not present
       (name && strlen( name))  )               // and name to be set
   {
      TxPrint("WARNING: There is currently no active BMGR menu ...\n");
   }
   memset( &cbs, 0, sizeof(cbs));               // initial cb info
   cbs.confirmed = confirmed;                   // prompting
   cbs.disknr    = p->disknr;

   if (name && strlen( name))                   // valid name specified
   {
      memset( cbs.string, ' ', BT_BM_L);
      memcpy( cbs.string, name,  strlen(name));
      cbs.string[ BT_BM_L] = '\0';
   }
   else
   {
      strcpy( cbs.string, "");
      cbs.option = TRUE;                        // reset name in BM structure
   }
   rc = dfsExecOnBootRec( p, cbs.disknr, dfsFdskLogBmName, &cbs);
   RETURN (rc);
}                                               // end 'dfsFdskSetLogBm'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Implement the 'LogBmName' callback operation, set BM-name in EBR sector
/*****************************************************************************/
static ULONG dfsFdskLogBmName                   // RET PENDING  => call again
(                                               //     NO_ERROR => all done
   DFSPARTINFO        *p,                       // INOUT partition info
   S_BOOTR            *br,                      // INOUT sector with Ptable
   FDSK_CB_PHASE       phase,                   // IN    callback phase
   FDSK_CB_INFO       *cbp                      // IN    operation info
)
{
   ULONG               rc = NO_ERROR;           // function return

   ENTER();
   TRACES(("Phase:%u Related:%u Confirmed:%u Option:%u Flag:%u  Disk:%hu thisSn:%llx\n",
            phase, cbp->related, cbp->confirmed, cbp->option, cbp->flag, cbp->disknr, cbp->thisSn));
   if (p != NULL)                               // info available
   {
      switch (phase)
      {
         case FDSK_PREPARE:
            if ((dfsa->batch    == FALSE) &&
                (cbp->confirmed == FALSE) &&
                !TxConfirm( 5066, "%s '%8.8s' for partition %02u (%s) ? [Y/N] : ",
                    (cbp->option) ? "Remove BM-name" : "Set as BM-name",
                    (cbp->option) ?  br->BootMgrName :  cbp->string,
                                                        p->id, p->drive))
            {
               rc = DFS_CMD_FAILED;             // aborted
            }
            break;

         case FDSK_PERFORM:
            if (cbp->option)
            {
               br->BootMgrFlag = 0;
               sprintf( br->BootMgrName, "%8.8x", (ULONG) p->basePsn);
            }
            else
            {
               br->BootMgrFlag = BT_BM_BOOT;
               strcpy( br->BootMgrName, cbp->string);
            }
            if (p->ebrChain && p->ebrChain->ebrChain)
            {
               sprintf( br->BootMgrNext, "%8.8x", (ULONG) p->ebrChain->ebrChain->basePsn);
            }
            else                                // no next EBR, zero field
            {
               memset(  br->BootMgrNext, 0, sizeof(br->BootMgrNext));
            }
            rc = DFS_PENDING;
            break;

         case FDSK_PENDING:
            br->PartitionTable[p->partnr].Status =
               (BYTE) (cbp->option) ? (BYTE) 0 : (BYTE) 0x80;
            dfsFdskSingleAutoShow( p->disknr);
            break;

         default:
            rc = DFS_CMD_FAILED;
            break;
      }
   }
   RETURN (rc);
}                                               // end 'dfsFdskLogBmName'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Implement the 'Delete' callback operation on a partition-table entry
/*****************************************************************************/
ULONG dfsFdskDelete                             // RET PENDING  => call again
(                                               //     NO_ERROR => all done
   DFSPARTINFO        *p,                       // INOUT partition info
   S_BOOTR            *br,                      // INOUT sector with Ptable
   FDSK_CB_PHASE       phase,                   // IN    callback phase
   FDSK_CB_INFO       *cbp                      // IN    option:zero full entry
)
{
   ULONG               rc = NO_ERROR;           // function return
   DFSPARTENTRY       *pe;                      // partition-table entry

   ENTER();
   TRACES(("Phase:%u Related:%u Confirmed:%u Option:%u Flag:%u  Disk:%hu thisSn:%llx\n",
            phase, cbp->related, cbp->confirmed, cbp->option, cbp->flag, cbp->disknr, cbp->thisSn));
   if (p != NULL)                               // info available
   {
      USHORT           disk = p->disknr;        // keep valid disk nr around
      BOOL             prim = p->primary;       // keep valid primary status
      TXLN             s1;

      switch (phase)
      {
         case FDSK_PREPARE:
            if ((dfsa->batch == FALSE) && (cbp->confirmed == FALSE))
            {
               sprintf(   s1, "Partition %hu disk %hu, %s %s\nLabel '%s' by %s\n",
                          p->id, p->disknr,  (prim) ? "primary" : "logical",
                          p->fsform, p->plabel,   p->creatr);
               dfstrSizeBps( s1, "Partition size: ", p->sectors, p->bpsector, "\n\n");

               if (!TxConfirm( 5071, "%sDelete this partition ? [Y/N] : ", s1))
               {
                  rc = DFS_CMD_FAILED;          // aborted
               }
            }
            #if defined (DEV32)
            if (rc == NO_ERROR)                 // when confirmed/batch
            {
               if ((dfsLvmSyncDesired( p)  ) && // partition, sync wanted
                   (p->lvm.VoluName[0] != 0) && // and a volumename
                   (p->lvm.letter      != 0)  ) // and driveletter
               {
                  TxCopy( s1, p->lvm.VoluName, LVM_NAME_L);
                  dfsLvmSyncEngine( TRUE, FALSE, s1, 0); // hide volume
               }
            }
            #endif
            break;

         case FDSK_PERFORM:
            pe = &(br->PartitionTable[p->partnr]);
            if (cbp->option)                    // zero whole structure
            {
               memset( pe, 0, sizeof(DFSPARTENTRY));
            }
            else                                // just zero minimum fields
            {
               pe->PartitionType = 0;           // 'unused' type
               pe->Status        = 0;           // reset startable/active flag
            }
            rc = DFS_PENDING;
            break;

         case FDSK_PENDING:
            if (prim)                           // remove BM info for deleted
            {
               dfsFdskSetPriBm( p, NULL, TRUE);
            }
            dfsReadDiskInfo( FDSK_QUIET);       // re-read all partition info
            if ((prim) && (p->lvmPresent))      // remove LVM DLAT entry
            {
               FDSK_CB_INFO    cbi;             // use private CB info
               DFSDISKINFO    *d;

               if ( (!TxaOptUnSet('L')) &&      // if not -L- to keep LVM info
                   ((d = dfsGetDiskInfo(disk)) != NULL))
               {
                  memset( &cbi, 0, sizeof(FDSK_CB_INFO));
                  cbi.disknr    = disk;         // no options = crc/DLAT check
                  cbi.confirmed = TRUE;         // no prompting ...
                  cbi.sn        = d->geoSecs-1; // MBR-LVM sector
                  cbi.number    = 1;            // 1 sector only
                  dfsExecOnSectors( dfsFdskLvmSetInfo, &cbi);
               }
            }
            else                                // remove empty containers etc
            {
               dfsCleanupExtChain( disk, TRUE);
            }
            TxPrint("\nPartition deleted, partition-table ");
            if (cbp->option)                    // zero whole structure
            {
               TxPrint("entry set to all zero's\n");
            }
            else
            {
               TxPrint("type-byte set to zero\n");
            }
            dfsFdskSingleAutoShow( disk);       // indicate changes
            break;

         default:
            rc = DFS_CMD_FAILED;
            break;
      }
   }
   RETURN (rc);
}                                               // end 'dfsFdskDelete'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Implement the 'SetType' callback operation on a partition-table entry
// cbp->string holds original 'new' type specification for 'smart' types
/*****************************************************************************/
ULONG dfsFdskSetType                            // RET PENDING  => call again
(                                               //     NO_ERROR => all done
   DFSPARTINFO        *p,                       // INOUT partition info
   S_BOOTR            *br,                      // INOUT sector with Ptable
   FDSK_CB_PHASE       phase,                   // IN    callback phase
   FDSK_CB_INFO       *cbp
)
{
   ULONG               rc = NO_ERROR;           // function return
   DFSPARTENTRY       *pe;                      // partition-table entry
   TXTM                descr;

   ENTER();
   TRACES(("Phase:%u Related:%u Confirmed:%u Option:%u Flag:%u  Disk:%hu thisSn:%llx\n",
            phase, cbp->related, cbp->confirmed, cbp->option, cbp->flag, cbp->disknr, cbp->thisSn));
   if (p != NULL)                               // info available
   {
      pe = &(br->PartitionTable[p->partnr]);
      switch (phase)
      {
         case FDSK_PREPARE:
            cbp->more   = p;                    // attach partition info
            cbp->sn     = p->basePsn;           // start sector
            cbp->number = p->sectors;           // size

            cbp->ntype  = dfsParsePartType( cbp->string, cbp);

            dfsPartTypeDescription( cbp->ntype, descr);
            TxStrip( descr, descr, ' ', ' ');

            if ((cbp->stype == 0) || (cbp->stype == pe->PartitionType))
            {
               if ((dfsa->batch == FALSE) && (cbp->confirmed == FALSE))
               {
                  TXLN    s1;

                  sprintf(   s1, "Partition %hu disk %hu, %s %s\nLabel '%s' by %s\n",
                             p->id, p->disknr,  (p->primary) ? "primary" : "logical",
                             p->fsform, p->plabel,   p->creatr);
                  dfstrSizeBps( s1, "Partition size: ", p->sectors, p->bpsector, "\n\n");

                  if (!TxConfirm( 5081, "%sChange the type for this "
                                        "partition to %2.2x = %s ? [Y/N] : ",
                                         s1, cbp->ntype, descr))

                  {
                     rc = DFS_CMD_FAILED;       // aborted
                  }
               }
            }
            break;

         case FDSK_PERFORM:
            if ((cbp->stype == 0) || (cbp->stype == pe->PartitionType))
            {
               pe->PartitionType = cbp->ntype;
               rc = DFS_PENDING;
            }
            else
            {
               if (cbp->verbose)
               {
                  TxPrint("Partition does not match selection criteria\n");
               }
            }
            break;

         case FDSK_PENDING:
            dfsPartTypeDescription( cbp->ntype, descr);
            p->partent.PartitionType = cbp->ntype;
            strcpy( p->descr, descr);
            dfsFdskSingleAutoShow( p->disknr);
            break;

         default:
            rc = DFS_CMD_FAILED;
            break;
      }
   }
   RETURN (rc);
}                                               // end 'dfsFdskSetType'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Implement the 'SetStatus' callback operation on a partition-table entry
/*****************************************************************************/
ULONG dfsFdskSetStatus                          // RET PENDING  => call again
(                                               //     NO_ERROR => all done
   DFSPARTINFO        *p,                       // INOUT partition info
   S_BOOTR            *br,                      // INOUT sector with Ptable
   FDSK_CB_PHASE       phase,                   // IN    callback phase
   FDSK_CB_INFO       *cbp                      // IN    callback parameters
)
{
   ULONG               rc = NO_ERROR;           // function return
   USHORT              pi;                      // Ptable index
   DFSPARTENTRY       *pe;                      // partition-table entry
   BYTE                type;
   BOOL                autovisible = FALSE;     // set accessible too

   ENTER();
   TRACES(("Phase:%u Related:%u Confirmed:%u Option:%u Flag:%u  Disk:%hu thisSn:%llx\n",
            phase, cbp->related, cbp->confirmed, cbp->option, cbp->flag, cbp->disknr, cbp->thisSn));

   if (p != NULL)                               // info available
   {
      pe   = &(br->PartitionTable[p->partnr]);
      type = pe->PartitionType;
      if ((cbp->option == FALSE) &&             // no 'hide' operation
          (cbp->flag   != FALSE) &&             // and make startable
          (cbp->result == FALSE) )              // and not clear
      {
         autovisible = dfsPartTypeHidable( type);
      }
      switch (phase)
      {
         case FDSK_PREPARE:
            if ( (p->primary  != FALSE) &&      // on primary partition
                 (cbp->result == FALSE) &&      // not 'clear' operation
                ((cbp->option == FALSE) ||      // no 'hide' operation
                 (autovisible != FALSE) ))      // or auto on startable
            {
               if (dfsPartTypeHidable(pe->PartitionType))
               {
                  if (cbp->more == NULL)        // no multiple primaries
                  {
                     TxPrint("Other primaries on same disk will be hidden.\n");
                  }
                  else
                  {
                     TxPrint("Multiple accessible primaries will be allowed.\n");
                  }
               }
            }
            if ((cbp->option) &&                // 'hide' operation
                (p->primary  == FALSE) )        // on logical partition
            {
               TxPrint("Hiding a logical partition might not be supported\n"
                       "by the operating system(s) you are using.\n");
            }
            if ((autovisible) && (type & DFS_P_PHIDDEN))
            {
               TxPrint("Making this partition startable, "
                       "will make it visible too.\n");
            }
            if ((dfsa->batch == FALSE) && (cbp->confirmed == FALSE))
            {
               TXLN    s1;

               sprintf(   s1, "Partition %hu disk %hu, %s %s\nLabel '%s' by %s\n",
                          p->id, p->disknr,  (p->primary) ? "primary" : "logical",
                          p->fsform, p->plabel,   p->creatr);
               // dfstrSiz8( s1, "Partition size: ", p->sectors, "\n\n");
               dfstrSizeBps( s1, "Partition size: ", p->sectors, p->bpsector, "\n\n");

               if (!TxConfirm( 5086, "%sMake this partition %s ? [Y/N] : ", s1,
                            (cbp->flag  ) ?
                            (cbp->result) ? "not-startable"    : "startable" :
                            (cbp->option) ? "hidden"           : "visible"))
               {
                  rc = DFS_CMD_FAILED;          // aborted
               }
            }
            break;

         case FDSK_PERFORM:
            for (pi = 0; pi < 4; pi++)
            {
               pe = &(br->PartitionTable[pi]);
               if (cbp->flag)                   // make startable
               {
                  if (cbp->result)              // clear startable
                  {
                     if (pi == p->partnr)
                     {
                        pe->Status &= ~0x80;    // clear on this part
                     }
                  }
                  else                          // set startable
                  {
                     if (pi == p->partnr)       // selected partition
                     {
                        pe->Status |= 0x80;     // set bootable flag
                     }
                     else                       // reset bootable flag
                     {                          // on ALL others, including
                        pe->Status &= ~0x80;    // posible deleted ones
                     }
                  }
               }
               if (pe->PartitionType != 0)      // used entry
               {
                  if ((!cbp->flag) || (autovisible))
                  {
                     if (pi == p->partnr)       // selected partition
                     {
                        if (cbp->option)        // hide selected partition
                        {
                           pe->PartitionType |= DFS_P_PHIDDEN;
                        }
                        else
                        {
                           pe->PartitionType &= ~DFS_P_PHIDDEN;
                        }
                     }
                     else                       // other primaries, same disk
                     {
                        if ((cbp->option == FALSE) && // hide not-selected
                            (cbp->more   == NULL ) && // if no multiple
                            (p->primary  != FALSE)  ) // and a primary
                        {
                           if (dfsPartTypeHidable(pe->PartitionType))
                           {
                              pe->PartitionType |= DFS_P_PHIDDEN;
                           }
                        }
                     }
                  }
               }
            }
            rc = DFS_PENDING;
            break;

         case FDSK_PENDING:
            dfsFdskSingleAutoShow( p->disknr);
            break;

         default:
            rc = DFS_CMD_FAILED;
            break;
      }
   }
   RETURN (rc);
}                                               // end 'dfsFdskSetStatus'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// Save partition info for specified disk to a file
/*****************************************************************************/
ULONG dfsSavePartInfo                           // RET   function result
(
   USHORT              disk,                    // IN    disk number
   char               *filebase,                // IN    dest file basename
   char               *comment                  // IN    user comment on save
)
{
   ULONG               rc = NO_ERROR;           // function return
   FDSK_CB_INFO        cbi;                     // callback parameter struct
   DFSDISKINFO        *d;                       // ptr to phys disk info
   DFSDISKINFO         dinfo;                   // local (partial/modified) copy
   TX1K                ascii;                   // big ascii buffer
   TXLN                fname;                   // filename
   FILE               *df;                      // diskfile ptr
   TXTM                regstring;

   ENTER();

   sprintf( fname, "%s.pd%u", filebase, disk);
   if ((df = fopen( fname, "wb")) != NULL)      // create new binary file
   {
      if ((d = dfsGetDiskInfo(disk)) != NULL)
      {
         TXTM          text;
         time_t        tt = time( &tt);         // current date/time

         TxPrint( "\nSaving for disk%2u : ", disk);

         #if defined (REGISTRATION)
            dfsRegistrationValid( NULL, regstring);   // get reg description
         #else
            strcpy( regstring, "OEM licensed version");
         #endif

         strftime( text,    TXMAXTM, "%Y-%m-%d %H:%M:%S", localtime( &tt));
         memset(  ascii, 0, TXMAX1K);
         sprintf( ascii, "Saved disk sectors (MBR/EBR, BOOTREC%s LVM)\r\n%s; version %s %s\r\n"
                         "%s%s disk:%u on %s with %u partitions, %u Pri/gpt\r\n"
                         "Geo CHS:%7u %3u %-3u = %4.2lf Mb ",
                         (d->lvmPartitions == 0) ? ", no" : " and", DFS_N, DFS_V, DFS_C, DFS_PSAVE_SIG,
                          d->pStyle,     disk, text,  d->totalpart, d->primaries,
                          d->geoCyls,    d->geoHeads, d->geoSecs, TXSMIB( d->cSC, d->bpsector));

         dfstrSizeBps( ascii, "size : ", d->sectors,  d->bpsector, "\r\n");
         strcat(       ascii, "Reg ");
         strcat(       ascii, regstring);
         strcat(       ascii, "\r\n");          // add registration fragment
         strcat(       ascii, comment);
         strcat(       ascii, "\r\n\r\n\032");  // add comment and EOF

         ascii[ DFSPS_ASCII_SIZE -1] = 0x1a;    // make sure it ends in EOF

         //- Note that the DFSDISKINFO is truncated! (is larger than 128)
         //- And that the first 8 bytes are replaced by two MAGIC numbers
         //- to be able to detect the 64bit PDx version (new in DFSee 14.5)
         memcpy( &dinfo, d, DFSPS_DINFO_SIZE);  // create modifyable copy
         dinfo.ebrHead = DFSPS_MAGIC1;
         dinfo.fspHead = DFSPS_MAGIC2;
         memcpy( ascii + DFSPS_ASCII_SIZE, &dinfo, DFSPS_DINFO_SIZE);

         fwrite( ascii,  DFSPS_HEADERSIZE, 1, df);

         memset( &cbi, 0, sizeof(cbi));         // initial callback info
         cbi.disknr    = disk;                  // physical disk nr
         cbi.more      = df;                    // pass FILE pointer
         cbi.flag      = TRUE;                  // Visit MBR/EBR sectors
         cbi.related   = dfsa->lvmPresent;      // LVM too ?
         cbi.option    = dfsa->lvmPresent;      // LVM signature sects
         strcpy( cbi.string, filebase);         // LVM feature image basename
         rc = dfsIteratePlvm( dfsFdskPsaveSect, &cbi);

         if (rc == NO_ERROR)
         {
            if (d->gptHeader != NULL)
            {
               rc = dfsFdskPsaveGptInfo( &cbi);
            }
         }

         if (rc == NO_ERROR)
         {
            cbi.stype     = 0;                  // any type
            cbi.related   = FALSE;              // Don't visit EBR sectors
            cbi.usebase   = TRUE;               // select boot records
            cbi.cbOption4 = TRUE;               // include unrecognized BR
            rc = dfsIteratePart( dfsFdskPsaveBoot, &cbi);
         }
         TxPrint( "  done saving.\nSectors saved are : %u "
                  "MBR/EBR or LVM and %llu boot sectors in '%s'\n",
                    cbi.cbOptNum1, cbi.cbOptNum2, fname);
      }
      fclose( df);
   }
   else
   {
      TxPrint( "Destination file '%s' cannot be created\n", fname);
      rc = ERROR_FILE_NOT_FOUND;
   }
   RETURN(rc);
}                                               // end 'dfsSavePartInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Fix type of extended partitions for specified disk-nr from 'old' to 'new'
/*****************************************************************************/
ULONG dfsFixExtType                             // RET   function result
(
   USHORT              disk,                    // IN    disk number
   BYTE                old,                     // IN    old, verify type or 0
   BYTE                new                      // IN    new  system type
)
{
   ULONG               rc = NO_ERROR;           // function return
   DFSDISKINFO        *d;                       // ptr to phys disk info
   DFSPARTINFO        *e;                       // related ebr partition
   S_BOOTR            *br;                      // master/extended boot record
   TXTS                descr;                   // type description buffer

   ENTER();

   if ((d = dfsGetDiskInfo(disk)) != NULL)
   {
      rc = dfsSelectDisk( disk, FALSE, FALSE);  // Select disk, no reset
      if ((rc == NO_ERROR) && (d->ebrHead != NULL))
      {
         if ((br = TxAlloc( 1, dfsGetSectorSize())) != NULL)
         {
            for (e = d->ebrHead; (e != NULL) && !TxAbort();)
            {
               if ((old == 0) || (old == e->partent.PartitionType))
               {
                  dfsPartTypeDescription( e->partent.PartitionType, descr);
                  TxPrint( "\nPart-table at PSN : 0x0%llx, type 0x%02.2x = %s ",
                             e->partPsn, e->partent.PartitionType, descr);

                  if (new != e->partent.PartitionType)
                  {
                     dfsPartTypeDescription( new, descr);
                     TxPrint( "change to %2.2x = %s\n", new, descr);
                     rc = dfstReadPsn( DFSTORE, e->partPsn, 1, (BYTE *) br);
                     if (rc == NO_ERROR)
                     {
                        br->PartitionTable[e->partnr].PartitionType = new;
                        if ((dfsa->batch) ||
                            (TxConfirm( 5091,
                                        "Extended container for partition %hu\n\n"
                                        "Change type to %2.2x = %s ? [Y/N] : ",
                                         e->id, new, descr)))
                        {
                           if (DFSTORE_WRITE_ALLOWED)
                           {
                              TxPrint("Update %cBR sector : ",
                                      (e->partPsn == 0) ? 'M' : 'E');
                              if ((rc = dfstWritePsn( DFSTORE, e->partPsn, 1,
                                                     (BYTE *) br)) == NO_ERROR)
                              {
                                 e->partent.PartitionType = new;
                                 strcpy( e->descr, descr);
                                 TxPrint("finished\n");
                              }
                              else
                              {
                                 TxPrint("failed\n");
                              }
                           }
                           else
                           {
                              rc = DFS_READ_ONLY;
                           }
                        }
                        else
                        {
                           TxPrint("Update MBR/EBR skipped, no changes made\n");
                        }
                     }
                     else
                     {
                        TxPrint("Error reading MBR/EBR sector at %llX, Cyl:%u\n",
                                 e->partPsn, dfstPsn2Cyl(DFSTORE, e->partPsn));
                     }
                  }
                  else
                  {
                     TxPrint( "OK : no change needed\n");
                  }
               }
               else
               {
                  TxPrint( "Partition-type does not match criteria, no change\n");
               }
               if ((new == 0x0f) && (!TxaOption('a'))) // not ALL 0x0f ?
               {
                  e = NULL;                     // stop for this disk
               }
               else
               {
                  e = e->ebrChain;
               }
            }
            TxFreeMem( br);
         }
         else
         {
            rc = DFS_ALLOC_ERROR;
         }
      }
      else
      {
         TxPrint( "Cannot locate an extended partition on disk %u\n", disk);
      }
   }
   RETURN(rc);
}                                               // end 'dfsFixExtType'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Align the extended-container (first logical) for a disk to track boundaries
/*****************************************************************************/
ULONG dfsAlignDiskContainer                     // RET   function result
(
   USHORT              disk                     // IN    disk number
)
{
   ULONG               rc = NO_ERROR;           // function return
   DFSDISKINFO        *d;                       // ptr to phys disk info

   ENTER();

   if ((d = dfsGetDiskInfo(disk)) != NULL)
   {
      rc = dfsSelectDisk( disk, FALSE, FALSE);  // Select disk, no reset
      if ((rc == NO_ERROR) && (d->ebrHead != NULL))
      {
         rc = dfsAlignPartContainer( dfsGetPartInfo( d->ebrHead->id));
      }
      else
      {
         TxPrint( "Cannot locate an extended partition on disk %u\n", disk);
      }
   }
   RETURN(rc);
}                                               // end 'dfsAlignDiskContainer'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Align the extended-container (EBR) for specified logical to track boundaries
/*****************************************************************************/
ULONG dfsAlignPartContainer                     // RET   function result
(
   DFSPARTINFO        *p                        // IN    partition info
)
{
   ULONG               rc = NO_ERROR;           // function return

   ENTER();

   if (p != NULL)
   {
      if (p->primary == FALSE)
      {
         ULN64         alignedEbr = p->basePsn - p->geoSecs;
         DFSPARTINFO  *q = dfsGetPartInfo( p->id - 1);

         if ((p->diskPart == 1) || (q && (q->lastPsn < alignedEbr)))
         {
            if (p->partPsn != alignedEbr)       // not aligned already
            {
               if ((dfsa->batch) || TxConfirm( 5023,
                    "Recreate logical partition %hu on disk %hu to align its "
                    "extended container (EBR) to track boundaries ? [Y/N] : ",
                     p->id, p->disknr))
               {
                  TXTM                delCmd;
                  TXTM                creCmd;

                  //- Prepare cmd to recreate this partition (p is invalid after delete)
                  sprintf( creCmd, "cr -B log 0x%2.2hhx -d:%hu -a:0x%llx,s -s:0x%llx,s",
                           p->partent.PartitionType, p->disknr, alignedEbr,
                                                     p->sectors + p->geoSecs);

                  sprintf( delCmd, "delete -B %hu",  p->id);
                  rc = dfsMultiCommand( delCmd, 0, TRUE, FALSE, FALSE);
                  if (rc == NO_ERROR)
                  {
                     rc = dfsMultiCommand( creCmd, 0, TRUE, FALSE, TRUE);
                  }
                  else
                  {
                     TxPrint( "Delete of the partition failed unexpectedly!\n");
                  }
               }
            }
            else
            {
               TxPrint( "Container for logical partition %hu is already aligned\n", p->id);
            }
         }
         else if (q && (p->diskPart > 1))       // no room for alignment
         {
            TxPrint( "No room between partitions %hu en %hu for proper alignment\n", q->id, p->id);
            rc = DFS_BAD_STRUCTURE;
         }
      }
      else
      {
         TxPrint( "Specified partition, PID: %hu, is NOT a logical partition\n", p->id);
         rc = DFS_NOT_FOUND;
      }
   }
   RETURN(rc);
}                                               // end 'dfsAlignPartContainer'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Implement a callback to display a PARTINFO structure
/*****************************************************************************/
ULONG dfsFdskListPinfo                          // RET PENDING  => call again
(                                               //     NO_ERROR => all done
   DFSPARTINFO        *p,                       // INOUT partition/freesp info
   S_BOOTR            *br,                      // IN    dummy
   FDSK_CB_PHASE       phase,                   // IN    callback phase
   FDSK_CB_INFO       *cbp                      // IN    callback info
)
{
   ULONG               rc = NO_ERROR;           // function return
   TXTS                pitype;                  // info type
   TXTS                sizeiB;                  // size in XiB format (11 pos)

   ENTER();
   TRACES(("Phase:%u Related:%u Confirmed:%u Option:%u Flag:%u  Disk:%hu thisSn:%llx\n",
            phase, cbp->related, cbp->confirmed, cbp->option, cbp->flag, cbp->disknr, cbp->thisSn));
   TRARGS(("p-info *:%p\n", p));

   if ((p != NULL) && (phase == FDSK_PERFORM))
   {
      switch (p->pitype)
      {
         case DFS_PI_FREESPACE: strcpy( pitype, "FREE"); break;
         case DFS_PI_EBR_CHAIN: strcpy( pitype, " EBR"); break;
         default:               strcpy( pitype, "PART"); break;
      }
      strcpy( sizeiB, "");
      dfstrSz64XiB( sizeiB, "", p->sectors, p->bpsector, "");
      TxPrint( "\n%s %s ID : %-2hu disk:%2u size: 0x0%llx = %s", cbp->string, pitype, p->id, p->disknr, p->sectors, sizeiB);
      if (p->pitype != DFS_PI_FREESPACE)
      {
         TxPrint( " Ptable-entry: %u\n",   p->partnr);
         dfsGeoDispTransPsn( "Part-table offset :", p->geoHeads, p->geoSecs, p->partPsn);
      }
      else
      {
         TxPrint("\n Freespace   type : 0x%02.2hx = %s%s%s\n", p->partent.PartitionType,  CBM, p->descr, CNN);
      }
      dfsGeoDispTransPsn(    " Area base sector :", p->geoHeads, p->geoSecs, p->basePsn);
      dfsGeoDispTransPsn(    " Area last sector :", p->geoHeads, p->geoSecs, p->lastPsn);
      if (p->pitype != DFS_PI_FREESPACE)
      {
         if ((cbp->usebase) && (p->pitype == DFS_PI_PARTITION))
         {
            TxPrint("\n");
            dfsBootSector((BYTE *) br);
         }
         else                                   // part-table info
         {
            dfsShowPartEntry( " Partition-table  : ",
                              0,                // no navigation updates!
                              FALSE,            // no PID display
                              p->geoHeads,
                              p->geoSecs,
                              p->partPsn,
                            &(p->partent));
         }
      }
      dfsAdd2SectorList( p->basePsn);           // add to sector list
   }
   RETURN (rc);
}                                               // end 'dfsFdskListPinfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Implement the 'Clear' callback operation on partition boot records
/*****************************************************************************/
ULONG dfsFdskClearBoot                          // RET PENDING  => call again
(                                               //     NO_ERROR => all done
   DFSPARTINFO        *p,                       // IN    p-info
   S_BOOTR            *sec,                     // INOUT partition boot record
   FDSK_CB_PHASE       phase,                   // IN    callback phase
   FDSK_CB_INFO       *cbp                      // IN    callback parameters
)
{
   ULONG               rc = NO_ERROR;           // function return

   ENTER();
   TRACES(("Phase:%u Related:%u Confirmed:%u Option:%u Flag:%u  Disk:%hu thisSn:%llx\n",
            phase, cbp->related, cbp->confirmed, cbp->option, cbp->flag, cbp->disknr, cbp->thisSn));

   if ((sec != NULL) && (p != NULL))            // sector and p-info available
   {
      switch (phase)
      {
         case FDSK_PREPARE:
            if ((dfsa->batch == FALSE) && (cbp->confirmed == FALSE))
            {
               TXLN    s1;

               sprintf(   s1, "Partition %hu disk %hu, %s %s\nLabel '%s' by %s\n",
                          p->id, p->disknr,  (p->primary) ? "primary" : "logical",
                          p->fsform, p->plabel,   p->creatr);
               dfstrSz64( s1, "Partition size: ", p->sectors, "\n\n");

               if (!TxConfirm( 5096, "%sClear the bootrecord ? [Y/N] : ", s1))
               {
                  rc = DFS_CMD_FAILED;       // aborted
               }
            }
            break;

         case FDSK_PERFORM:
            memset( sec, 0,   sizeof(S_BOOTR));
            rc = DFS_PENDING;
            break;

         case FDSK_PENDING:
            fdsk->AutoShow = TRUE;              // indicate changes
            break;

         default:
            rc = DFS_CMD_FAILED;
            break;
      }
   }
   RETURN (rc);
}                                               // end 'dfsFdskClearBoot'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Implement the 'Clear' callback operation on part-table and/or LVM-sector
/*****************************************************************************/
ULONG dfsFdskClearSect                          // RET PENDING  => call again
(                                               //     NO_ERROR => all done
   BYTE               *sec,                     // INOUT a MBR/EBR/LVM/BR sect
   FDSK_CB_PHASE       phase,                   // IN    callback phase
   FDSK_CB_INFO       *cbp                      // IN    callback parameters
)
{
   ULONG               rc = NO_ERROR;           // function return

   ENTER();
   TRACES(("Phase:%u Related:%u Confirmed:%u Option:%u Flag:%u  Disk:%hu thisSn:%llx\n",
            phase, cbp->related, cbp->confirmed, cbp->option, cbp->flag, cbp->disknr, cbp->thisSn));

   if (sec != NULL)                             // sector available
   {
      switch (phase)
      {
         case FDSK_PREPARE:
            if ((dfsa->batch    == FALSE) &&
                (cbp->confirmed == FALSE) &&
                !TxConfirm( 5101, "About to clear sector at SN: 0x0%llx\n\n"
                           "Clear information in all %s sectors  "
                           "on disk %u ? [Y/N] : ",
                            cbp->sn, cbp->string, cbp->disknr))
            {
               rc = DFS_CMD_FAILED;             // aborted
            }
            else
            {
               cbp->confirmed = TRUE;           // only confirm once in
            }                                   // an iterate loop
            break;

         case FDSK_PERFORM:
            memset( sec, 0,   dfsGetSectorSize());
            rc = DFS_PENDING;
            break;

         case FDSK_PENDING:
            fdsk->AutoShow = TRUE;              // indicate changes
            break;

         default:
            rc = DFS_CMD_FAILED;
            break;
      }
   }
   RETURN (rc);
}                                               // end 'dfsFdskClearSect'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Implement the 'FixChs' callback operation on part-table entries
/*****************************************************************************/
ULONG dfsFdskFixChs                             // RET PENDING  => call again
(                                               //     NO_ERROR => all done
   BYTE               *sec,                     // INOUT a MBR/EBR sector
   FDSK_CB_PHASE       phase,                   // IN    callback phase
   FDSK_CB_INFO       *cbp                      // IN    callback parameters
)
{
   ULONG               rc = NO_ERROR;           // function return

   ENTER();
   TRACES(("Phase:%u Related:%u Confirmed:%u Option:%u Flag:%u  Disk:%hu thisSn:%llx\n",
            phase, cbp->related, cbp->confirmed, cbp->option, cbp->flag, cbp->disknr, cbp->thisSn));

   if (sec != NULL)                             // sector available
   {
      switch (phase)
      {
         case FDSK_PREPARE:
            if (cbp->confirmed == FALSE)
            {
               ULONG   chsStyle = dfsGetChsStyle( 'c', TXA_CUR, dfsa->batch);

               if ((dfsa->batch    == FALSE) &&
                   !TxConfirm( 5102, "Update partition-table CHS fields to "
                                     "use %s CHS style and match "
                                     "LBA/geometry values starting at %cBR "
                                     "at sector 0x0%llx on disk %u ? [Y/N] : ",
                    (chsStyle == DFS_CHSTYLE_PQ) ? "PowerQuest" :
                    (chsStyle == DFS_CHSTYLE_MS) ? "MicroSoft"  : "IBM/DFSee",
                    (cbp->sn == 0) ? 'M' : 'E', cbp->sn, cbp->disknr))
               {
                  rc = DFS_CMD_FAILED;          // aborted
               }
               else
               {
                  TxPrint("\n");
                  cbp->confirmed = TRUE;        // only confirm once in
               }                                // an iterate loop
            }
            break;

         case FDSK_PERFORM:
            {
               S_BOOTR        *sd = (S_BOOTR *) sec;
               DFSPARTENTRY   *pe;          // part entry ptr
               USHORT          i;

               for ( i=0; i < 4; i++)           // walk partition table
               {
                  pe = &sd->PartitionTable[i];
                  if (pe->PartitionType != DFS_P_EMPTY)
                  {
                     if (dfsFixChsEntry( pe, cbp) == DFS_PENDING)
                     {
                        rc = DFS_PENDING;
                     }
                  }
               }
            }
            TxPrint( "%cBR sector at PSN : 0x0%llx ", (cbp->sn == 0) ? 'M' : 'E', (ULONG) cbp->sn);
            if (rc == DFS_PENDING)
            {
               TxPrint( "had incorrect CHS values, and is updated\n\n");
            }
            else
            {
               TxPrint( "was correct, no update needed\n");
            }
            break;

         case FDSK_PENDING:
            break;

         default:
            rc = DFS_CMD_FAILED;
            break;
      }
   }
   RETURN (rc);
}                                               // end 'dfsFdskFixChs'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Check and update contents of one partition-table entry for FixChs
/*****************************************************************************/
static ULONG dfsFixChsEntry                     // RET   PENDING ==> updated
(
   DFSPARTENTRY       *pe,                      // IN    part entry ptr
   FDSK_CB_INFO       *cbp                      // IN    callback parameters
)
{
   ULONG               rc = NO_ERROR;
   ULONG               calcs;                   // Calculated start psn
   ULONG               calce;                   // Calculated end   psn

   ENTER();

   if (dfsIsExtendedType( pe->PartitionType))
   {
      calcs = pe->BootSectorOffset;             // calculate start
      if (cbp->sn != 0)                         // not MBR
      {
         calcs += dfsEbrBase();                 // add EBR base offset
      }
      calce = calcs + pe->NumberOfSectors -1;   // calculate end
   }
   else                                         // normal partition
   {
      calcs = cbp->sn + pe->BootSectorOffset;
      calce = calcs   + pe->NumberOfSectors -1;
   }

   if (dfsFixChsI13Changed( "CHS for BEGIN PSN : ", TRUE, calcs,
                            &pe->FirstSector.SecCyl,
                            &pe->FirstSector.Head))
   {
      rc = DFS_PENDING;                         // request write-back
   }
   if (dfsFixChsI13Changed( "CHS for  END  PSN : ", TRUE, calce,
                            &pe->LastSector.SecCyl,
                            &pe->LastSector.Head))
   {
      rc = DFS_PENDING;                         // request write-back
   }
   RETURN (rc);
}                                               // end 'dfsFixChsEntry'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Check and update I13 limited (dummy) CHS values for PSN, style and geometry
/*****************************************************************************/
BOOL dfsFixChsI13Changed                        // RET   CHS value was changed
(
   char               *verbose,                 // IN    msg lead text or NULL
   BOOL                update,                  // IN    update fields
   ULONG               psn,                     // IN    absolute sector (LBA)
   USHORT             *chsSecCyl,               // INOUT Sector+cylinder (CHS)
   BYTE               *chsHead                  // INOUT Head value      (CHS)
)
{
   ULONG               rc = FALSE;
   ULONG               chsc, chsh, chss;        // CHS check values
   ULONG               lbac, lbah, lbas;        // LBA check values
   ULONG               dummy_h, dummy_s;        // dummy values for style
   ULONG               geoHeads = dfstGeoHeads(    DFSTORE);
   ULONG               geoSecs  = dfstGeoSectors(  DFSTORE);
   ULONG               chsStyle;
   TXTM                chsText;                 // style description

   ENTER();

   chsStyle = dfsGetChsStyle( 'c', TXA_CUR, (verbose == NULL)); // style from 'c' option
   switch (chsStyle)
   {
      case DFS_CHSTYLE_PQ: sprintf( chsText, "PowerQuest"); break;
      case DFS_CHSTYLE_MS: sprintf( chsText, "MicroSoft "); break;
      default:             sprintf( chsText, "IBM/DFSee "); break;
   }

   chsc = DFSC2CYLIND( *chsSecCyl);
   chsh =              *chsHead;
   chss = DFSC2SECTOR( *chsSecCyl);
   dfstPsn2Chs( DFSTORE, psn, &lbac, &lbah, &lbas);

   TRACES(("LBA Cyl:%7u H:%3u S:%2u   CHS Cyl:%5u H:%3u S:%2u\n",
                lbac, lbah, lbas,             chsc, chsh, chss));

   switch (chsStyle)
   {
      case DFS_CHSTYLE_PQ: dummy_h = lbah;        dummy_s = lbas;    break;
      case DFS_CHSTYLE_MS: dummy_h = 255;         dummy_s = 63;      break;
      default:             dummy_h = geoHeads -1; dummy_s = geoSecs; break;
   }
   if ((chsc != lbac) || (chsh != lbah) || (chss != lbas))
   {
      if (lbac <= 1023)                         // below int-13, should match
      {
         TRACES(("bad CHS values with LBA cyl < 1024\n"));

         if (update)
         {
            *chsSecCyl = DFSCOMBINE(lbas,lbac);
            *chsHead   = (BYTE)     lbah;
         }
         if (verbose)
         {
            dfsX10( verbose,  psn, CNG, dfsFdskCchs);
            TxPrint("%sCHS:%7u %3u %-3u%s  match LBA/geometry\n",
                     CNC, lbac, lbah, lbas,  CNN);
         }
         rc = TRUE;
      }
      else if ((chsc != 1023) || (chsh != dummy_h) || (chss != dummy_s))
      {
         TRACES(("bad dummy CHS  with LBA cyl > 1024  (h:%u)\n", geoHeads));

         if (update)
         {
            *chsSecCyl = DFSCOMBINE( dummy_s, 1023);
            *chsHead   = (BYTE)    ( dummy_h);
         }
         if (verbose)
         {
            dfsX10( verbose,  psn, CNG, dfsFdskCchs);
            TxPrint("%sCHS:%7u %3u %-3u%s  style : %s\n",
                     CNC, 1023, dummy_h, dummy_s, CNN, chsText);
         }
         rc = TRUE;
      }
   }
   BRETURN (rc);
}                                               // end 'dfsFixChsI13Changed'
/*---------------------------------------------------------------------------*/


#if defined (USEWINDOWING)
/*****************************************************************************/
// Implement PTEDIT callback, interface between R/W and update dialog MBR style
/*****************************************************************************/
ULONG dfsFdskPtEditMbr                          // RET PENDING  => call again
(                                               //     NO_ERROR => all done
   BYTE               *sec,                     // INOUT sectors with Ptables
   FDSK_CB_PHASE       phase,                   // IN    callback phase
   FDSK_CB_INFO       *cbp                      // IN    operation info
)
{
   ULONG               rc = NO_ERROR;           // function return

   ENTER();
   TRACES(("Phase:%u Related:%u Confirmed:%u Option:%u Flag:%u  Disk:%hu thisSn:%llx\n",
            phase, cbp->related, cbp->confirmed, cbp->option, cbp->flag, cbp->disknr, cbp->thisSn));
   if (sec != NULL)
   {
      S_BOOTR         *br = (S_BOOTR *) sec;

      switch (phase)
      {
         case FDSK_PREPARE:
            strcpy( cbp->string, dfstStoreDesc2( DFSTORE));

            if (br->Signature != SV_BOOTR)      // invalid signature ?
            {
               TXLN    apmmsg;

               if (!memcmp(((S_MCDDM*)sec)->Signature,sg_mcddm,SG_MCDDM))
               {
                  strcpy( apmmsg, "Adding it will make this a HYBRID MBR/APM disk!\n"
                                  "APM partitions need to be kept in sync manually\n\n");
               }
               if (cbp->cbOption2 || TxConfirm( 5113,
                   "PTEdit: Selected sector has NO valid MBR/EBR signature!\n\n%s"
                   "Add the 0x%4.4X signature to this sector located on\n"
                   "%s ? [Y/N] : ", apmmsg, SV_BOOTR, cbp->string))
               {
                  br->Signature = SV_BOOTR;
                  TxPrint( "MBR/EBR signature : 0x%4.4X added "
                           "to sector ...\n", SV_BOOTR);
               }
               else                             // not confirmed
               {
                  cbp->cbOptNum1 = DFSD_NONE;   // stop NEXT/PREV selection
                  dfsa->explain  = TRUE;
                  rc = DFS_ST_MISMATCH;         // aborted (wrong sector)
               }
            }

            if (br->Signature == SV_BOOTR)      // valid signature ?
            {
               ULONG brCrc = TxCrc32( br, SECTORSIZE);

               rc = dfsFdskPteMbrDialog( cbp, br);
               if (rc != TXDID_CANCEL)
               {
                  if (brCrc != TxCrc32( br, SECTORSIZE)) // changed ?
                  {
                     if (TxConfirm( 5112,
                         "PTEdit: update Part-table sector on\n"
                         "%s ? [Y/N] : ", cbp->string))
                     {
                        rc = NO_ERROR;          // confirmed, write back
                     }
                     else                       // not confirmed
                     {
                        dfsa->explain = TRUE;
                        rc = DFS_CMD_FAILED;    // aborted (no update wanted)
                     }
                  }
                  else
                  {
                     //- TxPrint( "PartitionTable at : 0x%llx is not changed.\n", cbp->thisSn);
                     rc = DFS_CMD_FAILED;       // (no update needed)
                  }
               }
               else
               {
                  rc = DFS_CMD_FAILED;          // CANCEL is not-confirmed
               }
            }
            else
            {
               rc = DFS_CMD_FAILED;             // CANCEL is not-confirmed
            }
            break;

         case FDSK_PERFORM:
            rc = DFS_PENDING;                   // request write-back
            break;

         case FDSK_PENDING:
            TxPrint( "PartitionTable at : 0x0%llx updated and written back.\n", cbp->thisSn);
            break;

         default:
            rc = DFS_CMD_FAILED;
            break;
      }
   }
   RETURN (rc);
}                                               // end 'dfsFdskPtEditMbr'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Implement PTEDIT callback, interface between R/W and update dialog GPT style
/*****************************************************************************/
ULONG dfsFdskPtEditGpt
(
   FDSK_CB_INFO       *cbp                      // INOUT operation info
)
{
   ULONG               rc = NO_ERROR;           // function return
   ULONG               entryCrc;                // CRC for entry to be edited
   S_GPTHDR           *gptHdr;                  // GPT header for the disk
   S_GPTENTRY         *gptTable;
   S_GPTENTRY         *gptEntry;


   ENTER();
   TRACES(("Confirmed:%u Option:%u Flag:%u  Disk:%hu thisSn:%llx entry:%hu\n",
            cbp->confirmed, cbp->option, cbp->flag, cbp->disknr, cbp->thisSn, cbp->partnr));

   if ((gptEntry = dfsGptDiskGetEntry( cbp, &gptHdr, &gptTable)) != NULL)
   {
      DFSDISKINFO     *d = dfsGetDiskInfo( cbp->disknr);

      entryCrc = TxCrc32((BYTE *) gptEntry, sizeof(S_GPTENTRY));
      TRACES(("Entry :%p  Crc:%8.8x\n", gptEntry, entryCrc));

      strcpy( cbp->string, d->DiskName);
      rc = dfsFdskPteGptDialog( cbp, gptEntry);
      if (rc != TXDID_CANCEL)
      {
         if (entryCrc != TxCrc32((BYTE *) gptEntry, sizeof(S_GPTENTRY))) // changed ?
         {
            if (TxConfirm( 5112, "PTEdit: update GPT Partition-Table-Array on\n%s ? [Y/N] : ", cbp->string))
            {
               rc = dfsGptFixupWriteAll( gptHdr, gptTable);
               if (rc == NO_ERROR)
               {
                  TxPrint( "GPT info written successfully to primary and alternate location on disk %hu.\n", cbp->disknr);
               }
            }
            else                                // not confirmed
            {
               dfsa->explain = TRUE;
               rc = DFS_NO_CHANGE;              // aborted (no update wanted)
            }
         }
         else
         {
            //- TxPrint( "PartitionTable at : 0x%llx is not changed.\n", cbp->thisSn);
            rc = DFS_NO_CHANGE;                 // (no update needed)
         }
      }
      else
      {
         rc = DFS_NO_CHANGE;                    // CANCEL is not-confirmed
      }
   }
   else
   {
      rc = DFS_BAD_STRUCTURE;
   }
   RETURN (rc);
}                                               // end 'dfsFdskPtEditGpt'
/*---------------------------------------------------------------------------*/
#endif                                          // USEWINDOWING

/*****************************************************************************/
// Implement GPT setname, interface between R/W and actual name update
/*****************************************************************************/
ULONG dfsFdskGptSetName
(
   FDSK_CB_INFO       *cbp                      // INOUT operation info
)
{
   ULONG               rc = NO_ERROR;           // function return
   S_GPTHDR           *gptHdr;                  // GPT header for the disk
   S_GPTENTRY         *gptTable;
   S_GPTENTRY         *gptEntry;


   ENTER();
   TRACES(("Disk:%hu PID:%llu entry:%hu name: '%s'\n", cbp->disknr, cbp->number, cbp->partnr, cbp->pname));

   if ((gptEntry = dfsGptDiskGetEntry( cbp, &gptHdr, &gptTable)) != NULL)
   {
      TxAscii2Unic( cbp->pname, gptEntry->partName, GPT_PNAME_LEN); // write name in GPT entry

      if (dfsa->batch || (TxConfirm( 5111, "GPT: update entry for partition %llu on disk %u\n"
                         "to new name: '%s' ? [Y/N] : ", cbp->number, cbp->disknr, cbp->pname)))
      {
         rc = dfsGptFixupWriteAll( gptHdr, gptTable);
         if (rc == NO_ERROR)
         {
            TxPrint( "GPT info written successfully to primary and alternate location on disk %hu.\n", cbp->disknr);
         }
      }
      else                                      // not confirmed
      {
         dfsa->explain = TRUE;
         rc = DFS_NO_CHANGE;                    // aborted (no update wanted)
      }
   }
   else
   {
      rc = DFS_BAD_STRUCTURE;
   }
   RETURN (rc);
}                                               // end 'dfsFdskGptSetName'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Implement the 'Psave' callback operation on bootsectors
/*****************************************************************************/
static ULONG dfsFdskPsaveBoot                   // RET PENDING  => call again
(                                               //     NO_ERROR => all done
   DFSPARTINFO        *p,                       // IN    p-info
   S_BOOTR            *sec,                     // INOUT partition boot record
   FDSK_CB_PHASE       phase,                   // IN    callback phase
   FDSK_CB_INFO       *cbp                      // IN    callback parameters
)
{
   ULONG               rc = NO_ERROR;           // function return

   ENTER();
   TRACES(("Phase:%u Related:%u Confirmed:%u Option:%u Flag:%u  Disk:%hu thisSn:%llx\n",
            phase, cbp->related, cbp->confirmed, cbp->option, cbp->flag, cbp->disknr, cbp->thisSn));

   if ((p != NULL) && (sec != NULL) && (phase == FDSK_PERFORM))
   {
      FILE            *df = (FILE *) cbp->more;
      BYTE             st = ST_BOOTR;
      DFSPS_SEC        ps;
      TXTM             text;
      TXTM             descr;                   // type description buffer

      memset( &ps, 0, sizeof(DFSPS_SEC));

      ps.sects    = 1;
      ps.type     = st;
      ps.pdxVer   = DFSPS_V_01_64BIT;
      ps.infosize = DFSPS_SINFO_SIZE;

      //- 32bit value for backward/forward compatibility, and sectortype eyecather
      ps.psn32    = (ULONG) cbp->thisSn;
      dfsSectorTypeAsAscii( st, descr);
      sprintf( text, "%16.16sPSN = 0x%8.8x", descr, (ULONG) cbp->thisSn);
      memcpy(  &ps.info, text, DFSPS_ILEAD_SIZE);

      //- partial copy of PARTINFO, including p->id
      memcpy(  &ps.ibin, p, DFSPS_ITAIL_SIZE);

      //- 64bit sector value and ASCII eyecatcher
      ps.psn      =  cbp->thisSn;
      sprintf( text, "PSN (64bit) = 0x%16.16llx", cbp->thisSn);
      memcpy(  &ps.in64, text, DFSPS_INF64_SIZE);

      fwrite(  &ps, sizeof(DFSPS_SEC), 1, df);
      fwrite(  sec, sizeof(S_BOOTR),   1, df);

      cbp->cbOptNum2++;
      TxPrint( "b");
   }
   RETURN (rc);
}                                               // end 'dfsFdskPsaveBoot'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Implement the 'Psave' callback operation on part-table and LVM-sector
/*****************************************************************************/
static ULONG dfsFdskPsaveSect                   // RET PENDING  => call again
(                                               //     NO_ERROR => all done
   BYTE               *sec,                     // IN    a MBR/EBR/LVM/BR sect
   FDSK_CB_PHASE       phase,                   // IN    callback phase
   FDSK_CB_INFO       *cbp                      // IN    callback parameters
)
{
   ULONG               rc = NO_ERROR;           // function return

   ENTER();
   TRACES(("Phase:%u Related:%u Confirmed:%u Option:%u Flag:%u  Disk:%hu thisSn:%llx\n",
            phase, cbp->related, cbp->confirmed, cbp->option, cbp->flag, cbp->disknr, cbp->thisSn));

   if ((sec != NULL) && (phase == FDSK_PERFORM))
   {
      FILE            *df = (FILE *) cbp->more;
      BYTE             st = dfsIdentifySector( cbp->thisSn, 0, sec);
      DFSPS_SEC        ps;                      // size is 256
      TXTM             text;
      TXTM             descr;                   // type description buffer

      memset( &ps, 0, sizeof(DFSPS_SEC));

      ps.sects    = 1;
      ps.type     = st;
      ps.pdxVer   = DFSPS_V_01_64BIT;
      ps.infosize = DFSPS_SINFO_SIZE;

      //- 32bit value for backward/forward compatibility, and sectortype eyecather
      ps.psn32    = (ULONG) cbp->thisSn;
      dfsSectorTypeAsAscii( st, descr);
      sprintf( text, "%16.16sPSN = 0x%8.8x", descr, (ULONG) cbp->thisSn);
      memcpy(  &ps.info, text, DFSPS_ILEAD_SIZE);

      //- 64bit sector value and ASCII eyecatcher
      ps.psn      =  cbp->thisSn;
      sprintf( text, "PSN (64bit) = 0x%16.16llx", cbp->thisSn);
      memcpy(  &ps.in64, text, DFSPS_INF64_SIZE);

      if (st == ST_LVSIG)                       // LVM BBR, save feature data
      {
         TXLN          imzCmd;                  // cmd making LVM feature IMZ
         S_LVSIG      *sg = (S_LVSIG *) sec;    // as LVM signature sector
         USHORT        partID = dfsDiskPsn2PartId( cbp->disknr, cbp->sn);
         DFSPARTINFO  *p = dfsGetPartInfo( partID);

         if (p != NULL)                         // need PID for the PRESTORE ...
         {
            memcpy(  &ps.ibin, p, DFSPS_ITAIL_SIZE);
         }
         sprintf( imzCmd, "image -B \"%s.f%2.2u\" 0x%x 0x%x -z", cbp->string, partID,
                                        (sg->lastPsn - sg->reserved + 1), sg->reserved);

         rc = dfsMultiCommand( imzCmd, 0, FALSE, FALSE, FALSE); // silent execute
         TxPrint( "%c", (rc == NO_ERROR) ? 'F' : '!');
      }
      fwrite( &ps, sizeof(DFSPS_SEC),  1, df);
      fwrite( sec, dfsGetSectorSize(), 1, df);

      cbp->cbOptNum1++;
      TxPrint( "%c", (cbp->sn == 0) ? 'r' : st);

   }
   RETURN (rc);
}                                               // end 'dfsFdskPsaveSect'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Implement the 'Psave' sector-save function for GPT specific information
/*****************************************************************************/
static ULONG dfsFdskPsaveGptInfo
(
   FDSK_CB_INFO       *cbp                      // IN    disk information
)
{
   ULONG               rc = NO_ERROR;           // function return
   DFSDISKINFO        *d;                       // ptr to phys disk info

   ENTER();
   TRACES(("Disk:%hu\n", cbp->disknr));

   if (((d  = dfsGetDiskInfo( cbp->disknr)) != NULL) &&
       ((rc = dfsSelectDisk(  cbp->disknr, FALSE, FALSE)) == NO_ERROR))
   {
      S_GPTHDR        *gptHdr  = (S_GPTHDR *) d->gptHeader;
      ULONG            ePerSec = dfsGetSectorSize()  / gptHdr->ptaEntrySize;
      ULONG            tbleSec = (gptHdr->ptaEntries + ePerSec - 1) / ePerSec;
      ULONG            usedSec = (d->gptPartitions   + ePerSec - 1) / ePerSec;

      cbp->stype  = ST_GPTHD;                   // header sector
      cbp->thisSn = GPTHDR_PRIMARY_PSN;         // primary
      cbp->number = 1;
      rc = dfsFdskPsaveGptSect( cbp);
      if (rc == NO_ERROR)
      {
         cbp->stype  = ST_GPTAS;                // table sector(s)
         cbp->thisSn = gptHdr->ptaPSN;
         cbp->number = usedSec;
         rc = dfsFdskPsaveGptSect( cbp);
         if (rc == NO_ERROR)
         {
            cbp->stype  = ST_GPTHD;             // header sector
            cbp->thisSn = max( gptHdr->althPSN, gptHdr->thisPSN); // at end
            cbp->number = 1;
            rc = dfsFdskPsaveGptSect( cbp);
            if (rc == NO_ERROR)
            {
               cbp->stype  = ST_GPTAS;          // table sector(s)
               cbp->thisSn = gptHdr->althPSN - tbleSec;
               cbp->number = usedSec;
               rc = dfsFdskPsaveGptSect( cbp);
            }
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsFdskPsaveGptInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Implement the 'Psave' sector-save read/callback for GPT specific sectors
/*****************************************************************************/
static ULONG dfsFdskPsaveGptSect
(
   FDSK_CB_INFO       *cbp                      // IN    disk/sect information
)
{
   ULONG               rc = NO_ERROR;           // function return
   BYTE               *sec = NULL;              // sector buffer

   ENTER();
   TRACES(("Disk:%hu sn:%llx  nr:%llu\n", cbp->disknr, cbp->thisSn, cbp->number));

   if ((sec = (TxAlloc( cbp->number, dfsGetSectorSize()))) != NULL)
   {
      rc = dfstReadPsn( DFSTORE, cbp->thisSn, cbp->number, sec);
      if (rc == NO_ERROR)
      {
         FILE            *df = (FILE *) cbp->more;
         DFSPS_SEC        ps;                   // size is 256
         TXTM             text;
         TXTM             descr;                // type description buffer

         memset( &ps, 0, sizeof(DFSPS_SEC));

         ps.sects    = cbp->number;             // could be multiple sectors
         ps.type     = cbp->stype;
         ps.pdxVer   = DFSPS_V_01_64BIT;
         ps.infosize = DFSPS_SINFO_SIZE;

         //- 32bit value for backward/forward compatibility, and sectortype eyecather
         ps.psn32    = (ULONG) cbp->thisSn;
         dfsSectorTypeAsAscii( cbp->stype, descr);
         sprintf( text, "%16.16sPSN = 0x%8.8x", descr, (ULONG) cbp->thisSn);
         memcpy(  &ps.info, text, DFSPS_ILEAD_SIZE);

         //- 64bit sector value and ASCII eyecatcher
         ps.psn      =  cbp->thisSn;
         sprintf( text, "PSN (64bit) = 0x%16.16llx", cbp->thisSn);
         memcpy(  &ps.in64, text, DFSPS_INF64_SIZE);

         fwrite( &ps, sizeof(DFSPS_SEC),  1, df);
         fwrite( sec, dfsGetSectorSize(), cbp->number, df);

         TxPrint( "%c", cbp->stype);
      }
      TxFreeMem( sec);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsFdskPsaveGptSect'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Restore partition info from specified file to a disk
/*****************************************************************************/
ULONG dfsRestorePartInfo                        // RET   function result
(
   USHORT              disk,                    // IN    disk number
   char               *filebase,                // IN    source file basename
   char               *types                    // IN    types to restore
)
{
   ULONG               rc = NO_ERROR;           // function return
   FDSK_CB_INFO        cbi;                     // callback parameter struct
   DFSDISKINFO        *d;                       // ptr to phys disk info
   TXLN                fname;                   // filename
   TX1K                ascii;                   // big ascii buffer
   TXTS                action;                  // action description
   FILE               *df;                      // diskfile ptr
   char               *sdesc;                   // store description in header

   ENTER();

   if (strchr( filebase, '.') != NULL)          // explicit psaved filename
   {
      strcpy( fname, filebase);
   }
   else                                         // make complete filename
   {
      sprintf( fname, "%s.pd%u", filebase, disk);
   }
   if ((df = fopen( fname, "rb")) != NULL)      // open binary file for read
   {
      if (((d  = dfsGetDiskInfo(disk)) != NULL) &&
          ((rc = dfsSelectDisk( disk, FALSE, FALSE)) == NO_ERROR))
      {
         memset( &cbi, 0, sizeof(cbi));         // initial callback info
         cbi.nowrite = TxaOption('l');          // list only
         cbi.verbose = (dfsa->verbosity >= TXAO_VERBOSE); // verbose
         cbi.more    = df;
         cbi.disknr  = disk;
         strcpy( cbi.cbOptStr1, types);
         strcpy( cbi.string, fname);            // LVM feature image filename
         strcpy( action, cbi.nowrite ? "Display" : "Restore");
         fread(  ascii, DFSPS_HEADERSIZE, 1, df);
         TxRepl( ascii, 0x1a, 0);               // replace EOF by 0-terminator
         TxRepl( ascii, '\r', ' ');             // replace CR by SPACE

         if ((sdesc = strstr( ascii, DFS_PSAVE_SIG)) != NULL)
         {
            sdesc  += strlen( DFS_PSAVE_SIG);   // found exact description
         }
         else
         {
            sdesc   = &ascii[ DFS_PSAVE_POS];   // use most likely position
         }

         TxPrint( "\n====== Saved disk info description ====== (%s%s%s)\n\n%s\n",
                     CBG, fname, CNN, ascii);
         TxPrint(   "------ To be restored on current "
                    "disk %hu with geometry:\n", disk);
         dfstDiskGeometry( DFSTORE, 0, 0, 0, 0, 0, FALSE, TRUE);
         TxPrint( "====== %s    : %s", action,
            (cbi.nowrite || dfsa->batch) ? "above file\n" : "");
         if (cbi.nowrite || dfsa->batch || TxConfirm( 5114,
             "Was %s\n\n%s sectors from %s to %s disk %hu ?  [Y/N] : ",
              sdesc, action, fname, dfsMediaTypeDescr( dfsDid2DiskType(disk)), disk))
         {
            TxPrint( "\n");
            rc = dfsIteratePsaved( dfsRestorePsaveD, &cbi);
         }
         else                                   // skipped whole file
         {
            TxPrint( "Skipped %s of partition info from %s\n",
                      action, fname);
         }
      }
      else                                      // problem with the disk
      {
         TxPrint( "Specified physical disk %ul cannot be accessed\n", disk);
      }
      fclose( df);
   }
   else
   {
      TxPrint( "Source file '%s' cannot be read\n", fname);
      rc = ERROR_FILE_NOT_FOUND;
   }
   RETURN(rc);
}                                               // end 'dfsRestorePartInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// PRESTORE callback operating on PsaveD sectors + info, restore to disk
/*****************************************************************************/
static ULONG dfsRestorePsaveD
(
   BYTE               *sec,                     // IN    sector contents (NOT rbuf!)
   DFSPS_SEC          *psi,                     // IN    PsaveD sector info
   FDSK_CB_INFO       *info                     // INOUT function info+data
)
{
   ULONG               rc = NO_ERROR;           // function return
   ULN64               psn = (psi->pdxVer >= DFSPS_V_01_64BIT) ? psi->psn : psi->psn32;

   ENTER();

   if ((strchr( info->cbOptStr1, psi->type) != NULL) ||
       (strlen( info->cbOptStr1)            == 0)    ||
       (strcmp( info->cbOptStr1, "*" )      == 0)     )
   {
      TXTT             featureString = {0};     // LVM feature feedback text
      TXLN             imzString;               // LVM feature IMZ name/cmd
      DFSPARTINFO     *p = NULL;                // partial part-info in psi

      if (psi->type == ST_LVSIG)
      {
         p = (DFSPARTINFO *) psi->ibin;         // saved PARTINFO original partition
         TxStripExtension( info->string);       // make BASE, remove extension
         sprintf( imzString, "%s.f%2.2u", info->string, p->id);
         if (TxFileExists( imzString))          // feature IMZ file is available ?
         {
            strcpy( featureString, " (+ features)");
         }
      }
      if (info->verbose)
      {
         //- 'sec' should NOT be the generic rbuf, may be overwritten by display!
         TxPrint("\n");
         rc = dfsDisplaySector( psn, psi->type, sec);
         TxPrint("\n");
      }

      TxPrint( "Sector to restore : %s%16.16s%s at PSN: 0x%s%12.12llx%s size: %u sect.%s",
                CBC, psi->info, CNN, CBG, psn, CNN, psi->sects, featureString);

      if (!info->nowrite)                       // write requested ?
      {
         BOOL cf = ((dfsa->batch) || (TxaOptUnSet('c')));

         if (cf || TxConfirm( 5116, "%16.16s at PSN: 0x%12.12llx%s\n\n"
                                    "Restore this sector ? [Y/N] : ",
                                     psi->info, psn, featureString))
         {
            if (DFSTORE_WRITE_ALLOWED)
            {
               if ((rc = dfsWrite( psn, psi->sects, sec)) == NO_ERROR)
               {
                  if (!cf)                      // only when explicit confirm
                  {
                     TxPrint("\nSector contents written to 0x%12.12llX", psn);
                  }
                  if (strlen(featureString))
                  {
                     sprintf( imzString, "restore -B \"%s.f%2.2u\" -S", info->string, p->id);
                     rc = dfsMultiCommand( imzString, 0, FALSE, FALSE, FALSE); // silent
                     if (rc == NO_ERROR)
                     {
                        TxPrint("\nLVM features from : '%s.f%02.2u' restored.", info->string, p->id);
                     }
                  }
               }
               else
               {
                  TxPrint("\nWrite failed, PSN : 0x%12.12llX", psn);
               }
            }
            else
            {
               rc = DFS_READ_ONLY;
            }
         }
      }
      TxPrint( "\n");
   }
   RETURN(rc);
}                                               // end 'dfsRestorePsaveD'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Implement the 'Truncate' callback operation on an MBR partition-table entry
/*****************************************************************************/
ULONG dfsFdskTruncateMbr                        // RET PENDING  => call again
(                                               //     NO_ERROR => all done
   DFSPARTINFO        *p,                       // INOUT partition info
   S_BOOTR            *br,                      // INOUT sector with Ptable
   FDSK_CB_PHASE       phase,                   // IN    callback phase
   FDSK_CB_INFO       *cbp
)
{
   ULONG               rc = NO_ERROR;           // function return
   DFSPARTENTRY       *pe;                      // partition-table entry

   ENTER();
   TRACES(("Phase:%u Related:%u Confirmed:%u Option:%u Flag:%u  Disk:%hu thisSn:%llx\n",
            phase, cbp->related, cbp->confirmed, cbp->option, cbp->flag, cbp->disknr, cbp->thisSn));
   TRARGS(("p-info *:%p  phase:%u\n", p, phase));
   if (p != NULL)                               // info available
   {
      switch (phase)
      {
         case FDSK_PREPARE:                     // confirmed in FsTruncateSize
            break;

         case FDSK_PERFORM:
            dfsX10("Resize part-table : ", p->partPsn, CBG, "  ");
            TRACES(("Truncate partition-table entry\n"));
            TxPrint("Using entry %u for a type %2.2x partition\n", p->partnr, cbp->ntype);
            pe = &(br->PartitionTable[p->partnr]);
            dfsFillPtableEntry( p, cbp, pe);
            p->partent = *pe;                   // copy to info-struct too
            rc = DFS_PENDING;                   // request write & follow-up
            break;

         case FDSK_PENDING:                     // no post processing needed
            break;

         default:
            rc = DFS_CMD_FAILED;
            break;
      }
   }
   RETURN (rc);
}                                               // end 'dfsFdskTruncateMbr'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Implement RESIZE callback, interface between R/W and resize logic GPT style
/*****************************************************************************/
ULONG dfsFdskResizeGptEntry
(
   FDSK_CB_INFO       *cbp                      // INOUT operation info
)
{
   ULONG               rc = NO_ERROR;           // function return
   S_GPTHDR           *gptHdr;                  // GPT header for the disk
   S_GPTENTRY         *gptTable;
   S_GPTENTRY         *gptEntry;

   ENTER();
   TRACES(("Resize GPT entry: %hu to size: 0x%llx\n", cbp->partnr, cbp->size->value));
   //- Note: cbi.number is 32-bit, and set to size->value

   if ((gptEntry = dfsGptDiskGetEntry( cbp, &gptHdr, &gptTable)) != NULL)
   {
      gptEntry->lastPSN = cbp->sn + cbp->size->value -1;

      rc = dfsGptFixupWriteAll( gptHdr, gptTable);
      if (rc == NO_ERROR)
      {
         TxPrint( "GPT info written successfully to primary and alternate location on disk %hu.\n", cbp->disknr);
      }
   }
   else
   {
      rc = DFS_BAD_STRUCTURE;
   }
   RETURN (rc);
}                                               // end 'dfsFdskResizeGptEntry'
/*---------------------------------------------------------------------------*/
