//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// Implement file BROWSING based on contents of the DFSee sector list
// Usually initiated by a 'browse' command or filefind/delfind + browse .
//
// Gets information about the listed file using the FileInformation()
// Creates new (parent/subdir) list contents with a FsMakeBrowseList()
//
// Author: J. van Wijk
//
// JvW  16-09-2016 Initial version, derived from TXLIB txtree/txwstdlg code
//

#include <txlib.h>                              // TX library interface

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfsupart.h>                           // FS partition utilities
#include <dfsufgpt.h>                           // GPT utility functions
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS progress reporting
#include <dfshexed.h>                           // DFS  HEX editor interface
#include <dfsdgen.h>                            // DFS generic dialogs
#include <dfswin.h>                             // DFS windowing, help-id

#include <dfsbrows.h>                           // DFS sectorlist browsing

#if defined (USEWINDOWING)

typedef struct dfsbrowsedata
{
   ULONG               flags;                   // modify BrowseDlg behaviour
   ULONG               result;                  // dialog result (DID_...)
   USHORT              iSort;                   // Initial sort order
   BOOL                recMode;                 // Recovery mode active
   TXTM                bOptions;                // Browse options, like '~'
   TXLN                bFilter;                 // Browse filter string (select)
   TXLN                fPath;                   // initial  IN, current  OUT
   TXLN                fResult;                 // Resulting fully qualified
   TXLN                curPath;                 // Current contents, directory
   char               *title;                   // dialog title, or NULL
   TXLN                footer;                  // dialog footer, dynamic
   TXGW_DATA          *gwdata;                  // generic widget data
   ULONG               helpid;                  // specific help, whole dialog
   ULONG               basehelp;                // Std Open/SaveAs/Dir help
   TXSELIST           *list;                    // multiple-select browse list
   TXSELIST           *orgList;                 // saved copy of original rendered list
   ULN64              *orgSnlist;               // saved sector number list
   USHORT             *orgSninfo;               // saved sector number info
   ULN64               orgBrlist;               // saved Browse, snlist 'this' sn
   ULN64               orgBrinfo;               // saved snlist 'this' info
   TXLN                orgBrdescript;           // saved list/browse description string
   TXLN                fsDotCommand;            // command with .NNNN param to execute
} DFSBROWSEDATA;                                // end of struct "dfsbrowsedata"

//efine DFSBR_DEFAULT_COMMAND   "fatset -v- "
#define DFSBR_DEFAULT_COMMAND   ""


#define DFSBR_WWIDTH           105              // default window  width

//- positions in item->text for data-location, alloc-size, ea-size and basename
#define DFSBR_DLOC    7
#define DFSBR_ALSZ   18
#define DFSBR_EASZ   27
#define DFSBR_NAME   35
// 0      7          18       27        37


#define DFSBR_WID_DFRAME        910             // window-id dialog frame
#define DFSBR_WID_BRLIST        920             // window-id browse list

#define DFSBR_WM_NEWCONTENTS    TXWM_USER + 20
#define DFSBR_WM_SHOWCONTENTS   TXWM_USER + 22

//- Menu/accelerator command codes
#define A_POPUP                 10              // popup itself (from F10/Enter)
#define A_SPACE                 20
#define A_EDATA                 30
#define A_XATTR                 40
#define A_EMETA                 50
#define A_SMETA                 55
#define A_ENAME                 60
#define A_ELCMD                 65
#define A_RECOV                 70
#define A_REDUC                 80

#define DFSH_BR             (DFSH_BROWSE)       // shown from HELP menu

#define DFSH_BR_DFRAME      (DFSH_BR +  2)      // shown help-on-help (2nd)
#define DFSH_BR_BRLIST      (DFSH_BR +  5)      // shown on <F1> from ListBox

#define DFSH_BR_SPACE       (DFSH_BR + A_SPACE) // Action Edit allocated space, auto
#define DFSH_BR_EDATA       (DFSH_BR + A_EDATA) // Action Edit File data area
#define DFSH_BR_XATTR       (DFSH_BR + A_XATTR) // Action View extended attributes
#define DFSH_BR_SMETA       (DFSH_BR + A_SMETA) // Action Show Meta data area
#define DFSH_BR_EMETA       (DFSH_BR + A_EMETA) // Action Edit Meta data area
#define DFSH_BR_ENAME       (DFSH_BR + A_ENAME) // Action Edit Filename
#define DFSH_BR_ELCMD       (DFSH_BR + A_ELCMD) // Action Execute CMD with .NNNN
#define DFSH_BR_RECOV       (DFSH_BR + A_RECOV) // Action Copy/Recover
#define DFSH_BR_REDUC       (DFSH_BR + A_REDUC) // Action Reduce selection


//- Definition of popup submenus for the browser-list dialog

TXSitem(brSpace,A_SPACE,DFSH_BR_SPACE,0,12,"Edit DATA, Allocated for a File/Dir" ,"Edit/view ALLOCATED sectors, limit to exact size, multiple fragments ");
TXSitem(brEdata,A_EDATA,DFSH_BR_EDATA,0,12,"Edit DATA, Raw sectors for File/Dir" ,"Edit/view raw sectors  at the start of the FIRST allocated fragment  ");
TXSitem(brXattr,A_XATTR,DFSH_BR_XATTR,0, 6,"View XATTR or OS/2 EA on a File/Dir" ,"View all extended attributes (Linux/Unix XATTR or OS/2 EA) for a file");
TXSitem(brEmeta,A_EMETA,DFSH_BR_EMETA,0, 6,"Edit META:Inode/Fnode/MFT/Dir-entry" ,"Hex/Ascii/Asm edit on the meta-data (inode/MFT) for a file/directory ");
TXSitem(brSmeta,A_SMETA,DFSH_BR_SMETA,0, 1,"Show META:Inode/Fnode/MFT/Dir-entry" ,"Display meta-data (inode/MFT) in output window, for a file/directory)");
TXSitem(brEname,A_ENAME,DFSH_BR_ENAME,0,12,"Edit the base filename for File/Dir" ,"Edit the base file-NAME (keeping the same length) for file/directory ");
TXSitem(brElCmd,A_ELCMD,DFSH_BR_ELCMD,0, 2,"Execute command with .NNNN list ref" ,"Execute specified DFSee command, passing it a '.NNNN' list reference ");
TXSitem(brRecov,A_RECOV,DFSH_BR_RECOV,0, 1,"Copy/Recover File(s) to other drive" ,"Copy the file(s)/dir(s) to a safe location, incl. attrib/date/time   ");
TXSitem(brReduc,A_REDUC,DFSH_BR_REDUC,0, 1,"Reduce list to SELECTED files only"  ,"Discard UNMARKED items from the list, updating the sectorlist too    ");

static TXS_ITEM *brsingle[] = {&brSpace, &brEdata, &brXattr, &brEmeta, &brSmeta, &brEname, &brRecov, &brElCmd};
TXSlist(dfsbrsingle,8,8,brsingle);

static TXS_ITEM *brmulti[]  = {&brRecov, &brElCmd, &brSmeta, &brReduc};
TXSlist(dfsbrmulti,4,4,brmulti);

static char           *browseHelpText[] =
{
   "",
   " This dialog allows convenient and quick access to the directory tree",
   " present in a (possibly damaged) filesystem much like a filemanager",
   " would in a filesystem that is normally accessible outside of DFSee.",
   " (properly 'seen' or 'mounted' by the operating system)",
   "",
   " It can be activated from various places in the menu, or with <F9>",
   "",
   " When started this way, you select the starting contents for the list",
   " by specifying either a full-path (default is '" FS_PATH_STR "') or just '.'",
   " to signal that you want to (re)use the current contents of the sector",
   " list, which may have come from a previous BROWSE session, or from other",
   " file related actions like searching for deleted files (DELFIND).",
   "",
   " When starting a BROWSE from the ROOT of the filesystem, you can navigate",
   " the directory tree to view all the files and perform operations like",
   " edit/view the contents, or copy/recover a file to a safe location on",
   " another drive.",
   "",
TXHELPITEM(002,"BROWSE file/directory browser dialog")
   " Files and Directories are presented in a single list with information",
   " like date-time and size, as well as the filename/path to allow good",
   " identification of each specific one.",
   "",
   " The selection list for BROWSE looks like:",
   "",
   "     D 100% <UP-DIR>  2.0 KiB        0  ..",
   "    +D 100% <SUBDIR>  2.0 KiB      211  dfs\\",
   "     D 100% <SUBDIR>  2.0 KiB     2365  txlib\\",
   "    +f 100%    1202A  3.5 KiB        0  ibmcpp.txt",
   "    ||  |      |         |           |     |",
   "    ||  |      |         |           |     \\__ Base filename",
   "    ||  |      |         |           |",
   "    ||  |      |         |           \\__ Xattr or OS/2-EA size (bytes)",
   "    ||  |      |         |",
   "    ||  |      |         \\__ Allocated size (multiple of block/sector)",
   "    ||  |      |",
   "    ||  |      \\__ Data location for file, indicator for a directory",
   "    ||  |",
   "    ||  \\__ Recoverability percentage, 100% means file allocation is OK",
   "    ||       (Note: Only available in 'Recovery' mode, blank otherwise)",
   "    ||",
   "    |\\__ Type of this entry, 'D' for a directory, 'f' for normal file",
   "    |                        'H' folder Alias     'h' hardlinked file",
   "    |                            (HFS+ only)      's' symlink",
   "    |                        'y' deleted DIR      'z' deleted file",
   "    |",
   "    |",
   "    \\__ Mark indicator, some operations can be performed on EACH",
   "        marked line, instead of just the CURRENT (highlighted) one.",
   "",
   "",
   " In addition to this, the BOTTOM line of the DFSee windows holds an",
   " extra description for the current file or directory, that looks like:",
   "",
   "        2015-12-12 16:51:22               3.547  f  \\c\\ibmcpp.txt",
   "                  |                        |     |   |",
   "    Date-Time  __/                         |     |   \\__ full path",
   "   (last mod)                              |     |",
   "                      Filesize in bytes __/      \\__ type, for sorting",
   "",
   "",
   " After closing the dialog, the 'down' and 'this' navigation aids are",
   " set to match the CURRENT file, and the browsed directory respectively.",
   " This means that <ENTER> will take you to the file (Fnode/Inode/MFT...)",
   " and the 't' command will take you to the enclosing directory info.",
   "",
TXHELPITEM(005,"BROWSE list, selectable files/directories")
   " For more help on the complete dialog, press <F1> again ...",
   "",
   "",
   "                                  DFSee BROWSE mode keyboard usage",
   "                                  ================================",
   "",
   "    Operations on the whole dialog window",
   "",
   "    <F3>          : Quit the browser dialog (same as Esc)",
   "",
   "    <F4>          : Toggle between 'Fast' and 'Recovery Percentage' mode",
   "                    'Recovery Percentage' can be very slow in loading",
   "                    the file/directory list for huge or many files!",
   "",
   "    Alt + <F4>    : Reload the current file/direcory list (refresh)",
   "",
   "    <F6>          : Replace list by the ORIGINAL, as it was when starting",
   "                    the browse session (ROOT-directory or filefind result)",
   "                    Can be useful on damaged filesystems when the '..'",
   "                    entry is missing from a subdirectory listing.",
   "",
   "    <F12>         : Minimize the browse window to the title-line,",
   "                    allows viewing the text-output area below.",
   "",
   "    Ctrl-\\          Replace list by contents of the ROOT directory",
   "",
   "    Operations on the CURRENT line, all marked ones, or whole list:",
   "",
   "    <ENTER>       : On a file: Menu with possible actions on selected file(s)",
   "                    Directory: Replace list by contents of that directory",
   "",
   "    <F2>          : On a File:      Edit/view allocated sectors as ASCII,",
   "                    on a Directory: Edit/view Meta (Inode/MFT/DIR-entry)",
   "",
   "    <F5>          : Copy/Recover the file/directory to a safe location",
   "",
   "    <F9>          : Filter list on (name) wildcard, recoverability and size",
   "",
   "    <F10>         : Present menu with possible actions on selected file(s)",
   "                    (or when RMBMENU is set, click mouse button 2)",
   "",
   "    <F11>         : Toggle 'Show hidden Dirs/Files' between SHOW and HIDE",
   "                    (initial setting from menu option or dialog checkbox)",
   "",
   "    <SPACE>, +    : Toggle mark for the current line (and advance to next)",
   "    <INSERT>",
   "",
   "    Ctrl-A, *, =  : Mark ALL lines",
   "",
   "    Ctrl-F, Alt-[ : Mark ALL Files, meaning lines starting with 'f' or 'h'",
   "    Ctrl-G, Alt-] : Mark ALL Directories, lines not starting with 'D' or 'H'",
   "",
   "    Ctrl-I,  -    : Toggle mark on ALL lines (Invert marking)",
   "",
#if defined (UNIX)
#else
   "    Ctrl-Z"
#endif
   "    Alt-Z         : Unmark ALL lines",
   "",
   "",
   " The default (D/f + Path) sorting can be changed using:",
   "",
   "   " TXS_SORT_KEYS1 " Filename at last dot (.extension), case sensitive",
   "   " TXS_SORT_KEYS2 " Filename from start  ( filename ), case sensitive",
   "   " TXS_SORT_KEYS3 " Description, col 43  (D/f + Path), ignoring case",
   "   " TXS_SORT_KEYS4 " Descr.  at last dot  (.extension), ignoring case",
   "   " TXS_SORT_KEYS5 " Description, col  1  ( Date-time), descending",
   "   " TXS_SORT_KEYS6 " Description, col 21  ( File-Size), descending",
   "   " TXS_SORT_KEYS7 " Unsort, restore original list ordering",
   "   " TXS_SORT_KEYS8 " Reverse the current list ordering",
   "   " TXS_SORT_KEYS9 " Allocation reliability Percentage, descending",
   "",
   " Quick searching for a specific (start of a) filename can be done by",
   " using the <Alt> key together with the first letter, followed by more",
   " letters of the filename (with the <Alt> key being released then!).",
   " The part of the filename being matched is shown on at the statusline and",
   " is updated at each keypress, while the CURRENT selection moves to the",
   " line containing the first filename starting with those characters.",
   "",
   "",
   "                                  DFSee BROWSE mode mouse usage",
   "                                  =============================",
   "",
   "    MB1-CLICK   : Toggle mark for the current line (multiselect marking)",
   "",
   "    MB1-DRAG    : Toggle mark for a range of lines (multiselect marking)",
   "",
   "    MB1-DBLCLK  : On a file: Menu with possible actions on selected file(s)",
   "                  Directory: Replace list by contents of that directory",
   "",
   "    MB2-CLICK   : Open popup menu with possible actions on selected file(s)",
   "                  (when RMBMENU option set, which is the default behaviour)",
   "",
   "",
TXHELPITEM(020,"BROWSE, action: Edit Contents, autoview on File/Dir")
   " This menu item will start an (Hex/Ascii/Asm) edit session on the contents",
   " of the selected file or directory, with access restricted to the allocated ",
   " sectors for it, to view that or to modify it.",
   "",
   " It will analyse the data to select the most appropriate VIEW for this data,",
   " but you cycle through the available views using the <F2> function key.",
   "",
   " The available views are:",
   "",
   "   - ASCII view, text/strings very much like the Unix 'strings' command",
   "     command, you can use the <F5> and <F6> keys to tune its behaviour",
   "",
   "   - DISASM view, x86 instructions, dis-assembled as 16/32/64 bit code",
   "     you can use function keys to step through code and select bitness",
   "",
   "   - HEXEDIT view, bytes displayed as rows of hexa-decimal values, with",
   "     the same bytes displayed as ASCII beside each line.",
   "     This mode is selected automatically when the contents is NOT mostly",
   "     ASCII, and does not start with recognized x86 instructions",
   "     It is the only mode in which you can also CHANGE the data",
   "",
   " The area to be viewed/edited is taken from the EACT allocation for the",
   " file, as retrieved from the filesystem structures for the file, this is ",
   " useful when a file is fragmented, with multiple areas spread over the",
   " filesystem, and also to avoid the risk of modifying something that does",
   " NOT belong to the selected file.",
   "",
   " When modifying anything the length/size of a file can NOT be changed!",
   "",
TXHELPITEM(030,"BROWSE, action: Edit DATA area sectors for File/Dir")
   " This menu item will start an (Hex/Ascii/Asm) edit session on the raw",
   " data for the selected file or directory, to view that or to modify it.",
   "",
   " This can be useful when the allocation data for the file can not be",
   " retrieved, so the 'Edit contents' selection does not work.",
   "",
   " For fragmented files, not all sectors will be present, or in the correct",
   " ordering sequence (Use 'Edit Contents' for an exact file-data edit)",
   "",
   " Also, for directories, be extremely careful not to damage the structure",
   " since thay may cause serious problems in accessing it afterwards.",
   "",
TXHELPITEM(040,"BROWSE, action: View OS/2 EA or Xattr on File/Dir")
   " This menu item will start an (Hex/Ascii/Asm) edit session on the OS/2 EA",
   " or Linux/Unix XATTR extended attributes for the selected file or directory,",
   " with access restricted to the EA/Xattr area only, to view it.",
   "",
   " The access is read-only, you can NOT write-back the values",
   "",
TXHELPITEM(050,"BROWSE, action: Edit META:Inode/Fnode/MFT/Dir-entry")
   " This menu item will start an (HEX) edit session on the metadata for the",
   " selected file or directory, either to just view that or to modify it.",
   "",
   " The metadata is the information ABOUT the file/directory, usually inside",
   " structures like Inodes, Fnodes, MFT-records or (FAT) directory entries.",
   "",
   " Be extremely careful not to damage the metadata structure since thay may",
   " cause serious problems in accessing the files or directories afterwards.",
   "",
   " One possible reason to edit metadata is to (slightly) change the name of",
   " a file or directory, for example when that can not be done through the",
   " regular operating system commands (due to codepage issues or otherwise).",
   "",
TXHELPITEM(055,"BROWSE, action: Show META:Inode/Fnode/MFT/Dir-entry")
   " This menu item will display metadata for the selected file or directory,",
   " in whatever the default format for that meta data is.",
   "",
   " The metadata is the information ABOUT the file/directory, usually inside",
   " structures like Inodes, Fnodes, MFT-records or (FAT) directory entries.",
   "",
   " It will be displayed in the regular text output window, which will be",
   " mostly BEHIND the browse-window itself. It is mainly intended to get a",
   " contents aware display of the information for one or more files, to be",
   " checked after leaving the browser, or to get that information into a",
   " logfile in a convenient way. You can make it visible using <F12>",
   "",
TXHELPITEM(060,"BROWSE, action: Edit the base filename for File/Dir")
   " This menu item will start an (HEX) edit session on the NAME for the",
   " selected file or directory, to modify it (low-level kind of 'rename')",
   "",
   " When modifying, the length of the filename can NOT be changed.",
   "",
   " For more sophisticated filesystems (HPFS, JFS, NTFS, EXTn and more) you",
   " need to be aware of possible directory search/sort issues later that may",
   " arise if the (start of) the name changes significantly, causing ordering",
   " differences resulting in tree-structure search failures."
   "",
   "    NOTE: Direct filename editing is not implemented for all filesystems,",
   "          for those you need to fall back to editing the meta-data,",
   "          or even exit the browser on the selected file, and work",
   "          on the involved structures manually.",
   "",
TXHELPITEM(065,"BROWSE, action: Execute a command with .NNNN parameter")
   " This menu item will prompt for a DFSee command to be executed,"
   " with a reference to the current/selected File/Dir as a parameter",
   " in the form of a list reference '.NNNN' (index in sectorlist)",
   "",
   " Example: The 'fatset .NNNN' command to recreate the allocation chain",
   "          in the FAT-areas for a contiguous file on FAT filesystems",
   "",
   " There is a default command, specific to the 'Mode=xx' filesystem.",
   "",
TXHELPITEM(070,"BROWSE, action: Copy/Recover File(s) to another drive")
   " This menu item allows COPYING the data, and in most cases additional",
   " information like attributes and data/time for the selected file(s)",
   "",
   " It can either be used on a SINGLE (selected) file, or on all the",
   " files in the list that have been MARKED",
   "",
   " Copy/Recover of a Directory, will NOT copy the the directory CONTENTS,",
   " it will only create that directory in the recovery location, and may",
   " set identical attributes and data/time info (depending on filesystem)",
   " (Future versions may add the option to recurse into the directories",
   "  allowing copy/recover of a complete directory tree in a single go)",
   "",
   " Useful to get files from a filesystem that is otherwise inaccessible",
   " due to damaged bootsectors, mount failures etc, or simply because there",
   " is no filesystem driver for this filesystem in the operating system",
   " being used. (example: Copy file from an EXT4 partition using OS/2)",
   "",
   " On some filesystems, it can also be used to UNDELETE files, in that",
   " case the browsing session needs to start with an operation that finds",
   " deleted files, and places them in the sectorlist (DELFIND command etc).",
   "",
TXHELPITEM(080,"BROWSE, action: Reduce list to SELECTED files only")
   " This menu item will UNLIST all the UNMARKED files/directories from",
   " the browser (as well as in the sectorlist it was based upon).",
   "",
   " This could be useful to allow an EXPORT (of the sectorlist) with",
   " some specific selection of files, that can be IMPORTED at a later",
   " time to continue working with it.",
   "",
   NULL
};

static char  dfs_br_rec_footer[]    = "4=Fast";
static char  dfs_br_fst_footer[]    = "4=Perc";
static char  dfs_br_org_footer[]    = "6=Orig";
static char  dfs_br_no6_footer[]    = "      ";
static char  dfs_br_hid_footer[]    = "Hide";
static char  dfs_br_shw_footer[]    = "Show";
static char  dfs_browse_footer[]    =
   "F1=Help 2=Edit 3=Quit %s 5=Copy %s 9=Filter 10=Menu 11=%s 12=Min SPACE=Mark Sort:Ctl-xentbur"
#if defined (UNIX)
                                                                                               "p                                    ";
#else
                                                                                               "y                                    ";
#endif


// Define sort information for the browse lists
static TXSELSORT       dfs_browse_sort =
{
  {                                             // actual, initial sort order
     DFSBR_SORT_PATH,                           // ascend case-insensitive D/f + path
     TXS_SORT_TEXT | TXS_SORT_ON_CASE | TXS_SORT_ASCEND  | (DFSBR_NAME +1),  // (fname)
     TXS_SORT_TEXT | TXS_SORT_ON_CASE | TXS_SORT_LASTDOT,                    // (f-ext)
     TXS_SORT_TEXT | TXS_SORT_ON_CASE | TXS_SORT_DESCEND |  3,               // (pcent)
     TXS_SORT_DESC | TXS_SORT_ON_CASE | TXS_SORT_DESCEND |  1,               // (date)
     TXS_SORT_DESC | TXS_SORT_ON_CASE | TXS_SORT_DESCEND | 21,               // (size)
     TXS_SORT_DESC | TXS_SORT_IGNCASE | TXS_SORT_ASCEND  | 43,               // (path)
     TXS_SORT_DESC | TXS_SORT_IGNCASE | TXS_SORT_LASTDOT                     // (ext)
  },
  {                                             // reverse   (c-F8 / c-R)
     "Sorted: initial    <->",                  // unsorted  (c-F7 / c-U)
     "Sorted: filename   <->",                  // text   1  (c-F2 / c-X)
     "Sorted: .extension <->",                  // text   2  (c-F1 / c-D)
     "Sorted: Percentage <->",                  // text   3  (c-F9 / c-Y)
     "Sorted: Date-Time  <->",                  // desc   4  (c-F5 / c-T)
     "Sorted: File-Size  <->",                  // desc   5  (c-F6 / c-B)
     "Sorted: D/f + Path <->",                  // desc   6  (c-F3 / c-N)
     "Sorted: .Extension <->"                   // desc   7  (c-F4 / c-E)
  },
  ""                                            // current description
};


static  char           dfsbrSeparatorLine[] =
   ""
   "";


// Present BROWSE files/directories dialog using sector-list contents
static ULONG dfsBrowseDialog
(
   TXWHANDLE           parent,                  // IN    parent window
   DFSBROWSEDATA      *bd                       // IN    browse dialog data
);

// Window procedure, for the Browse Dialog
static ULONG dfsBrowseDlgProc                   // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    msg param 1
   TXWMPARAM           mp2                      // IN    msg param 2
);

// Determine if dir/file should be recovered, based on filetext and history
static BOOL dfsBrowseAuthorizeCopy              // RET   Copy this item
(
   char               *itemtext                 // IN    'D nnn% ..' item text
);

// Update title right-hand text based on current list filtering result
static void dfsBrowseFilter2trhText
(
   TXWHANDLE           hwnd                     // IN    window with attached list
);


// Build selection-list with File BROWSE information from current sector list
static TXSELIST *dfsMkBrowseSelist              // RET   selection list or NULL
(
   char               *filter,                  // IN    Filter string FileInfo
   BOOL                alCheck,                 // IN    Need allocation info
   char               *selName                  // IN    filename to make current
);


// Add file/directory instances from sectorlist to the Selist being built
static ULONG dfsBrowseAdd2List
(
   char               *filter,                  // IN    Filter string FileInfo
   BOOL                alCheck,                 // IN    Need allocation info
   TXSELIST           *list                     // INOUT Selist being built
);

// Save original raw and rendered lists (ONCE!) for later 'restore original'
static ULONG dfsBrowseSaveOriginal
(
   DFSBROWSEDATA      *brd                      // INOUT browse data
);

// Restore original raw and rendered lists from previously saved instance
static ULONG dfsBrowseRestoreOriginal
(
   DFSBROWSEDATA      *brd                      // INOUT browse data
);

// Discard original raw and rendered lists from previously saved instance
static void dfsBrowseDiscardOriginal
(
   DFSBROWSEDATA      *brd                      // INOUT browse data
);

// Build dialog-footer text based on current browse-data contents
static void dfsBrowseSetFooterText
(
   DFSBROWSEDATA      *brd                      // INOUT browse data
);


/*****************************************************************************/
// Initialize help system for file/directory BROWSING (accessible from startup)
/*****************************************************************************/
void dfsBrowseInitHelp
(
   void
)
{
   ENTER();

   txwRegisterHelpText( DFSH_BR, "dfsfilebrowse", "File-BROWSE Dialog help", browseHelpText);

   VRETURN ();
}                                               // end 'dfsBrowseInitHelp'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// BROWSE files/directories in sectorlist, execute actions on selected file(s)
/*****************************************************************************/
ULONG dfsBrowseSectorList
(
   char               *filter,                  // IN    Initial filter string
   ULONG               flags,                   // IN    Dialog behaviour flags
   USHORT              initialSort              // IN    Initial sort order
)
{
   ULONG               rc = NO_ERROR;
   TXLN                initialContents;
   DFSBROWSEDATA       brData;

   ENTER();
   TRACES(("filter: '%s'\n", filter));

   sprintf( initialContents, " Initial: %-*s", DFSBR_WWIDTH, dfsa->brdescript);

   memset( &brData, 0, sizeof(DFSBROWSEDATA));  // start empty

   brData.iSort   = initialSort;
   brData.title   = initialContents;
   brData.flags   = flags;
   brData.recMode = ((flags & DFSBR_RECOVERY) != 0);

   strcpy(  brData.bFilter, filter);

   //- to be refined, perhaps query from current filesystem through a dfsa field
   strcpy(  brData.fsDotCommand, DFSBR_DEFAULT_COMMAND);

   rc = dfsBrowseDialog( TXHWND_DESKTOP, &brData);

   BRETURN (rc);
}                                               // end 'dfsBrowseSectorList'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Present BROWSE files/directories dialog using sector-list contents
/*****************************************************************************/
static ULONG dfsBrowseDialog
(
   TXWHANDLE           parent,                  // IN    parent window
   DFSBROWSEDATA      *bd                       // IN    browse dialog data
)
{
   ULONG               rc = NO_ERROR;           // function return

   ENTER();
   TRACES(( "parent:%8.8x flags:%8.8x title:'%s'\n", parent, bd->flags,    bd->title));
   TRACES(( "Sort:%hu Options:'%s' Filter:'%s'\n", bd->iSort,  bd->bOptions, bd->bFilter));

   if (txwIsWindow( TXHWND_DESKTOP))            // is there a desktop ?
   {
      TXRECT           position;                // reference size/position
      TXWHANDLE        fframe;                  // file-dialog frame
      TXWHANDLE        ffname;                  // filename list
      TXWINDOW         window;                  // setup window data
      ULONG            style;
      short            wvsize;                  // widget vertical size
      short            phsize;                  // parent window width
      short            pvsize;                  // parent window height
      short            etline;                  // empty lines at top
      short            ww;                      // dialog window width
      TXTT             sort_files;              // files     sorting text
      TXTT             foot_count_files;        // #items footer files

      dfs_browse_sort.sort[ TXS_SORT_CURRENT] = bd->iSort; // Initial sort order

      txwQueryWindowPos( parent, FALSE, &position);
      phsize = position.right  - position.left + 1;
      pvsize = position.bottom;

      etline = 0;
      wvsize = txwWidgetSize( bd->gwdata, etline, NULL);
      if ((wvsize != 0) && ((pvsize - wvsize) > 20))
      {
         etline++;                              // empty line at top, readability
         wvsize++;
      }
      ww  = min( DFSBR_WWIDTH, phsize);

      position.left   = (phsize - ww) /2;       // always center the dialog
      position.right  = ww;
      position.top   -= 1;                     // overlay parent window title line
      position.bottom = pvsize - position.top + 1;
      if (pvsize >= 30)
      {
         position.bottom -= 3;                  // keep bottom clear ...
      }

      TRECTA( "pos/size", (&position));

      //- note: no shadow on this dialog, often overlaps bottom/status area
      style = TXWS_DIALOG | TXWS_DISABLED | TXCS_CLOSE_BUTTON | TXWS_LEFTJUSTIFY;
      if (bd->flags & DFSBR_MOVEABLE)
      {
         style |= TXWS_MOVEABLE;                // make frame move/sizeable
      }
      txwSetupWindowData(
         position.top, position.left, position.bottom, position.right, // horizontal size
         style, DFSH_BR_DFRAME,                 // window frame style and help
         ' ', ' ', TXWSCHEME_COLORS,
         bd->title, bd->footer,
         &window);
      window.st.buf     = NULL;                 // NO artwork attached
      window.dlgFocusID = DFSBR_WID_BRLIST;

      fframe = txwCreateWindow( parent, TXW_CANVAS, 0, 0, &window, NULL);
      txwSetWindowUShort( fframe, TXQWS_ID, DFSBR_WID_DFRAME);
      TRACES(("Attached browsedata structure at %8.8x to BR-frame\n", bd));

      if (wvsize != 0)                          // do we have any widgets ?
      {
         if ((rc = txwCreateWidgets( fframe, bd->gwdata, etline, 1)) == NO_ERROR)
         {
            txwSetupWindowData( wvsize, 0, 1, ww,
                  TXWS_OUTPUT | TXWS_HCHILD_SIZE,
                  0, ' ', ' ',  TXWSCHEME_COLORS, "",  "", &window);
            window.sl.buf = dfsbrSeparatorLine;
            txwCreateWindow( fframe, TXW_STLINE, fframe, 0, &window, NULL);
         }
      }
      if (rc == NO_ERROR)
      {
         style = TXWS_LEFTJUSTIFY | TXWS_FRAMED      | TXLS_FOOT_COUNT | TXLS_NEEDDBLCLK |
                 TXWS_HCHILD_SIZE                    | TXWS_VCHILD_SIZE;
         strcpy( foot_count_files, "");
         txwSetupWindowData(
            wvsize +0, 0, position.bottom - 2 - wvsize,  ww -2,
            style, DFSH_BR_BRLIST,              // style and help
            ' ', ' ',
            cFileTextStand,  cFileBorder_top,   // client and border
            cFileCountStand, cFileCountFocus,   // same color as footer (non shaded)
            cFileCountStand, cFileCountFocus,   // Footer count value
            bd->curPath,     foot_count_files,  // dynamic title and footer
            &window);
         window.lb.list = bd->list;
         window.lb.cpos = 0;
         window.frhtext = sort_files;
         strcpy( sort_files, "");
         strcpy( foot_count_files, "");
         ffname = txwCreateWindow( fframe, TXW_LISTBOX, fframe, 0, &window, txwDefWindowProc);
         txwSetWindowUShort( ffname, TXQWS_ID,   DFSBR_WID_BRLIST);

         bd->result = txwDlgBox( parent, parent, dfsBrowseDlgProc, fframe, bd);
      }
      if (bd->flags & DFSBR_RETAIN_LIST)        // when not needed by caller
      {
         txSelDestroy( &bd->list);              // free  list
      }
      dfsBrowseDiscardOriginal( bd);            // free original list copy
   }
   TRACES(( "result:%8.8x  list:%8.8x fpath:'%s\n", bd->result, bd->list, bd->fPath));
   RETURN (rc);
}                                               // end 'dfsBrowseDialog'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Window procedure, for the Browse Dialog
/*****************************************************************************/
static ULONG dfsBrowseDlgProc                   // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    msg param 1
   TXWMPARAM           mp2                      // IN    msg param 2
)
{
   ULONG               rc  = NO_ERROR;

   ENTER();
   if (hwnd != 0)
   {
      ULONG            lp1   = (ULONG) mp1;
      ULONG            lp2   = (ULONG) mp2;
      TXWINDOW        *win   = txwWindowData(     hwnd);
      DFSBROWSEDATA   *bd    = txwQueryWindowPtr( hwnd, TXQWP_USER);
      TXS_ITEM        *item  = (bd->list) ? bd->list->items[bd->list->selected] : NULL;
      TXWHANDLE        lhwnd = txwWindowFromID( hwnd, DFSBR_WID_BRLIST);
      TXWINDOW        *lw;
      ULN64            sn;
      ULONG            snindex = 1;
      ULN64            this = 0;
      ULN64            info = 0;
      BYTE             st = 0;                  // sector type, default none
      DFS_PARAMS       params;                  // generic function params
      ULONG            listcode = TXDID_OK;
      TXLN             tempText;                // dynamic popup dialog title
      BOOL             multiSel = FALSE;
      TXRECT           where = win->client;     // start position

      TRCMSG( hwnd, msg, mp1, mp2);

      if ((item != NULL) && (item->value > TXDID_MAX))
      {
         multiSel = ((bd->list->markinfo) && (bd->list->markinfo->marked != 0));
         snindex = item->value - TXDID_MAX;
         this    = dfsa->snlist[ snindex];      // Inode/Fnode/Dir-entry etc LSN
         if (dfsa->FsModeId  == DFS_FS_NTFS)
         {
            info = dfsa->brlist;                // related info for NTFS is parent LSN!
            TRACES(("Set desired parent MFT LSN: %llx\n", info));
         }
         else
         {
            info = dfsa->sninfo[ snindex];      // related info byte
         }
         TRACES(("this lsn: 0x%llx  info: 0x%llx\n", this, info));
      }
      switch (msg)
      {
         case TXWM_INITDLG:
            rc = txwDefDlgProc( hwnd, msg, mp1, mp2); // set-focus & activate
            txwPostMsg( hwnd, DFSBR_WM_NEWCONTENTS, 0, 0); // then our custom stuff
            break;

         case DFSBR_WM_NEWCONTENTS:             // new snlist, update list
            sprintf(            tempText, "Browse : (re)loading file/directory list%s ...   ",
                                                         (bd->recMode) ? " for Percentage display" : ", Fast");
            txwSetSbviewStatus( tempText, cSchemeColor);

            if (item != NULL)
            {
               strcpy( tempText, item->text);
            }
            else
            {
               strcpy( tempText, "");
            }
            TRACES(("Current item to keep selected: '%s'\n", tempText));

            txSelDestroy( &bd->list);           // free existing list, if any (invalidates item)

            bd->list = dfsMkBrowseSelist( bd->bFilter, bd->recMode, tempText);

            txwPostMsg( hwnd, DFSBR_WM_SHOWCONTENTS, (TXWMPARAM) bd->list->selected, 0); // show prepared browse selist
            break;

         case DFSBR_WM_SHOWCONTENTS:            // show new snlist, updated or restored
            dfsBrowseSetFooterText( bd);
            sprintf( bd->curPath, "%s %s ", (bd->recMode) ? "Recovery" : "Current:", dfsa->brdescript);

            if ((lw = txwWindowData(  lhwnd)) != NULL)
            {
               if (bd->list != NULL)
               {
                  bd->list->vsize = lw->client.bottom - lw->client.top +1;
                  TRACES(("list vsize: %hu\n", bd->list->vsize));
               }
               lw->lb.list = bd->list;
               if (bd->list != bd->orgList)     // don't modify position on restored list
               {
                  lw->lb.cpos = 0;
               }
            }
            dfsBrowseFilter2trhText( lhwnd);    // update filtering TRH text
            txwListSort2frhText( lhwnd);        // update footer indicator

            TRACES(("List vsize: %hu  set selected to: %u, top:%u\n", bd->list->vsize, lp1, lp2));
            if ((lp1 < bd->list->count) && (lp2 < lp1)) // preferred position in list
            {
               bd->list->selected = lp1;
               bd->list->top      = lp2;
               lw->lb.cpos        = lp1 - lp2;
            }

            //- update desktop footer with list-item description
            if ((item = bd->list->items[bd->list->selected]) != NULL)
            {
               txwPostMsg( TXHWND_DESKTOP, TXWM_SETFOOTER, (TXWMPARAM) item->desc, 0);
            }
            txwInvalidateWindow( hwnd, TRUE, TRUE); // possible footer-change (Fast/Recovery)
            txwSetSbviewStatus( "", cSchemeColor);
            break;

         case TXWM_CONTROL:                     // dialog controls, just a list here
            TRACES(("WM_CONTROL list:%8.8x item:%8.8x\n", bd->list, item));
            if (item != NULL)
            {
               switch (TXSH2FROMMP(mp1))
               {
                  case TXLN_SELECT:             // select, not 1st time (FOCUS)
                  case TXLN_SETFOCUS:
                     txwPostMsg( TXHWND_DESKTOP, TXWM_SETFOOTER, (TXWMPARAM) item->desc, 0);
                     break;

                  case TXLN_ENTER:              // ENTER on the item
                  default:
                     if ( (item->text[0] == 'D') || // Regular subdirectory item, or parent DIR
                         ((item->text[0] == 'H') && (atol( item->text + 7) == 0))) // active Hardlink
                     {

                        if (dfsBrowseSaveOriginal( bd) != NO_ERROR)
                        {
                           TxNamedMessage( TRUE, 0, " ERROR: list not saved ", "Saving original list failed!");
                        }
                        TxStrip( tempText, item->desc + 45, ' ', ' '); // strip spaces at EOL
                        TRACES(("Browse D on snlist index: %u = '%s'\n", snindex, tempText));

                        //- Fill sector-list for this DIR, and set browse description
                        TxCancelAbort();        // discard any pending abort!
                        rc = DFSFNCALL(dfsa->FsMakeBrowseList, this, info, "", NULL);
                        if (rc == NO_ERROR)
                        {
                           dfsa->brlist = this; // new list directory LSN (and NTFS desired parent!)
                           dfsa->brinfo = info;
                           TxPrint( "Listed files/dirs : %10llu in %s%s%s\n", dfsa->snlist[0], CBY, tempText, CNN);

                           if (strlen(item->desc) > 45)
                           {
                              sprintf( dfsa->brdescript, "browse: %s", tempText);
                              TxStrip( dfsa->brdescript,  dfsa->brdescript, 0 , ' ');
                           }
                           else
                           {
                              sprintf( dfsa->brdescript, "browse: sub-directory");
                           }
                        }
                        else
                        {
                           TxNamedMessage( TRUE, 0, " ERROR: no browse list ", "Making a BROWSE list for '%s' failed!", tempText);
                        }
                        txwPostMsg( hwnd, DFSBR_WM_NEWCONTENTS, 0, 0); // refresh select/display
                     }
                     else                       // File, present popup-menu with actions
                     {
                        txwPostMsg( hwnd, TXWM_COMMAND, (TXWMPARAM) A_POPUP, (TXWMPARAM) TXCMDSRC_ACCELERATOR);
                     }
                     break;
               }
            }
            //- note: no DefDlgProc default processing (to be refined ?)
            break;

         case TXWM_COMMAND:                     // can only be an mp2=TXCMDSRC_MENU
            if (item != NULL)                   // from popup menu or shortcut keys
            {
               DFSHEXEDINFO  hxi;               // hexedit additional info
               TXLN          recFname  = {0};   // full recovery path + filename

               dfsa->rParams.select    = NULL;  // file selection, wildcard
               dfsa->rParams.newname   = NULL;  // no rename
               dfsa->rParams.recFname  = recFname;


               memset( &hxi, 0, sizeof( DFSHEXEDINFO)); // make sure it is EMPTY

               switch (lp1)                     // menu-item ID
               {
                  case A_POPUP:                 // present popup action menu
                     where.top  += bd->list->selected - bd->list->top;
                     where.left += DFSBR_NAME;  // near start of filename area of list
                                                // to be refined, near screen bottom!
                     if (multiSel)
                     {
                        sprintf( tempText,  " Select action for %u MARKED files/directories ", bd->list->markinfo->marked);
                        listcode = txwListBox( hwnd, hwnd, &where,
                                               tempText, "", DFSH_BR_BRLIST, TXLB_MOVEABLE,
                                               cSchemeColor, cSchemeColor, &dfsbrmulti);
                     }
                     else
                     {
                        sprintf( tempText,  " Select action on SELECTED '%s' ", item->text + DFSBR_NAME);
                        listcode = txwListBox( hwnd, hwnd, &where,
                                               tempText, "", DFSH_BR_BRLIST, TXLB_MOVEABLE,
                                               cSchemeColor, cSchemeColor, &dfsbrsingle);
                     }
                     switch (listcode)
                     {
                        case TXDID_CANCEL:
                           break;

                        default:
                           txwPostMsg( hwnd, TXWM_COMMAND, (TXWMPARAM) listcode, (TXWMPARAM) TXCMDSRC_MENU);
                           break;
                     }
                     break;

                  case A_SPACE:
                  case A_XATTR:
                     //- Get file allocation, including (OS/2) extended attributes in integrated space
                     if ((rc = DFSFNCALL(dfsa->FsGetAllocSpace, this, info, "", &hxi.al)) == NO_ERROR)
                     {
                        if (hxi.al.byteSize == 0) // not supplied by FS allocSpace() ?
                        {
                           hxi.al.byteSize = dfsGetUllDot20( item->desc + 20);
                        }
                        TxCopy( hxi.name, item->text + DFSBR_NAME, TXMAXTM);

                        if (lp1 == A_XATTR)
                        {
                           if (hxi.al.xasize == 0)
                           {
                              TxNamedMessage( TRUE, 0, " ERROR: no extended attributes ", "Selected file '%s' has no extended attributes, nothing to view", hxi.name);
                           }
                           else
                           {
                              TxFreeMem( hxi.al.space);         //- free data space allocation
                              hxi.al.byteSize = hxi.al.xasize;  //- set 'allocated' size
                              hxi.al.space    = hxi.al.xattrs;  //- put EA allocation in space
                              hxi.al.chunks   = 0;              //- Plain memory alloc, no Sspace

                              TRACES(("Sspace byteSize: %llu\n", hxi.al.byteSize));
                              TxPrint("Edit extended attr: # .%7.7u == %s%s%s\n", snindex -1, CBG, item->text + DFSBR_NAME, CNN);
                              dfsHexObjectEditor( 0,   dfsGetClusterSize(), 0, 0, &hxi, DFSH_AUTOVIEW, 0, 0, 0);
                           }
                        }
                        else                    // regular file contents
                        {
                           TxFreeMem( hxi.al.xattrs);  //- free possible EA allocation
                           hxi.al.xatype = DFSXA_NONE; //- reset XA type to none
                           if ((hxi.al.byteSize == 0) || (hxi.al.space == NULL))
                           {
                              TxNamedMessage( TRUE, 0, " ERROR: empty file ", "Selected file '%s' has size 0 (EMPTY), nothing to edit", hxi.name);
                           }
                           else                 // start Hexed, even if no data
                           {
                              TRACES(("Sspace byteSize: %llu\n", hxi.al.byteSize));
                              TxPrint("Edit file contents: # .%7.7u == %s%s%s\n", snindex -1, CBG, item->text + DFSBR_NAME, CNN);
                              dfsHexObjectEditor( 0,   dfsGetClusterSize(), 0, 0, &hxi, DFSH_AUTOVIEW, 0, 0, 0);
                           }
                        }
                        TxFreeMem( hxi.al.space); // free data/xattr space allocation
                     }
                     else
                     {
                        TxNamedMessage( TRUE, 0, " ERROR: No data available ", "Failed to get allocation info for '%s'\n\n"
                                            "Data/Xattr/EA access may not be supported for Mode=%s",
                                             item->text + DFSBR_NAME, SINF->afsys);
                     }
                     break;

                  case A_EDATA:
                     hxi.al.byteSize = dfsGetUllDot20( item->desc + 20);
                     TxCopy( hxi.name, item->text + DFSBR_NAME, TXMAXTM);

                     if (hxi.al.byteSize == 0)
                     {
                        TxNamedMessage( TRUE, 0, " ERROR: empty file ", "Selected file '%s' has size 0 (EMPTY), nothing to edit", hxi.name);
                     }
                     else                       // start Hexed, when data available
                     {
                        sscanf( item->text + DFSBR_DLOC,"%llx", &sn);
                        TxPrint("Edit filedata  on : # .%7.7u == %s%s%s\n",
                                 snindex -1, CBG, item->text + DFSBR_NAME, CNN);
                        dfsHexObjectEditor( sn,   dfsGetClusterSize(), 0, 0, &hxi, DFSH_AUTOVIEW, 0, 0, 0);
                     }
                     break;

                  case A_EMETA:
                     TxPrint("Edit metadata  on : # .%7.7u == %s%s%s\n",
                              snindex -1, CBG, item->text + DFSBR_NAME, CNN);
                     dfsHexObjectEditor( this, dfsGetClusterSize(), 0, 0, NULL, DFSH_HEX_VIEW, 0, 0, 0);
                     break;

                  case A_SMETA:
                     txwSendMsg( hwnd, TXWM_CHAR, 0, (TXWMPARAM) TXk_F12); // minimize browse window
                     if (multiSel)
                     {
                        ULONG           i;

                        TxPrint("Show metadata  on : %u selected files/directories\n", bd->list->markinfo->marked);
                        for (i = 0, sn = 0; (i < bd->list->count) && !TxAbort(); i++)
                        {
                           if (bd->list->markinfo->markings[i] & TXSF_MARK_STD)
                           {
                              item    = bd->list->items[i];
                              snindex = item->value - TXDID_MAX;
                              this    = dfsa->snlist[ snindex]; // Inode/Fnode/Dir-entry etc
                              if (dfsa->FsModeId  == DFS_FS_NTFS)
                              {
                                 info = dfsa->brinfo; // related info, desired parent
                              }
                              else
                              {
                                 info = dfsa->sninfo[ snindex]; // related info byte
                              }

                              TxPrint("\nShow metadata  on : # .%7.7u == %s%s%s\n",
                                       snindex -1, CBG, item->text + DFSBR_NAME, CNN);

                              dfsReadAnDisplay( this, info, &st);
                           }
                        }
                     }
                     else
                     {
                        TxPrint("\nShow metadata  on : # .%7.7u == %s%s%s\n",
                                 snindex -1, CBG, item->text + DFSBR_NAME, CNN);

                        dfsReadAnDisplay( this, info, &st);
                     }
                     TxPrint( "\nPress <SPACE> or <ENTER> to resume the browser ...\n");
                     break;

                  case A_ENAME:
                     if (dfsa->FsUpdateFileName != NULL)
                     {
                        TXLN            oldFileName;
                        TXLN            newFileName;

                        strcpy( oldFileName, item->text + DFSBR_NAME); // current filename
                        if (oldFileName[ strlen( oldFileName) - 1] == FS_PATH_SEP) // directory
                        {
                           oldFileName[ strlen( oldFileName) - 1] = 0; // strip path separator
                        }
                        strcpy( newFileName, oldFileName);

                        TxNamedMessage( TRUE, 5525, " NOTE: file renaming ",
                                   "For most filesystems, it is advised to keep at least the START of the name "
                                   "the same or very close to the original, to avoid sort-order problems with "
                                   "the directory tree structures (B-tree etc) that could cause this file and "
                                   "most likely many other files NOT to be found anymore!\n\n"
                                   "After renaming, it is advised to REBOOT to avoid caching problems when "
                                   "the filesystem is currently mounted (accessible outside DFSee)");
                        do
                        {
                           if (TxPrompt( 5525, strlen( oldFileName), newFileName,
                                         "(Keep the name exactly the same length!)\n\n"
                                         " New filename for '%s'", oldFileName))
                           {
                              if (strlen( newFileName) != strlen( oldFileName))
                              {
                                 TxNamedMessage( TRUE, 5525, " ERROR: invalid filename length ",
                                            "New name must be the EXACT same length (%d) as the "
                                            "current name: '%s'", strlen( oldFileName), oldFileName);
                                 strcpy( oldFileName, item->text + DFSBR_NAME); // restore original filename
                              }
                           }
                           else                 // name made empty, abort the rename
                           {
                              break;
                           }
                        } while (strlen( newFileName) != strlen( oldFileName));

                        if (strlen( newFileName)) // not escaped from prompt dialog
                        {
                           if ((rc = DFSFNCALL( dfsa->FsUpdateFileName,
                                                this, 0, oldFileName, newFileName)) == NO_ERROR)
                           {
                              TxPrint("Edit base filename: # .%7.7u == %s%s%s to %s%s%s\n",
                                       snindex -1, CBG, oldFileName, CNN, CBY, newFileName, CNN);

                              //- refresh list contents, position at same line again
                              txwPostMsg( hwnd, DFSBR_WM_NEWCONTENTS, (TXWMPARAM) bd->list->selected, (TXWMPARAM) bd->list->top);
                           }
                        }
                     }
                     else
                     {
                        TxNamedMessage( TRUE, 0, " INFO: Not implemented ", "Direct Name edit NOT available for 'Mode=%s' yet\n", SINF->afsys);
                     }
                     break;

                  case A_RECOV:
                     txwSendMsg( hwnd, TXWM_CHAR, 0, (TXWMPARAM) TXk_F12); // minimize browse window
                     dfsInitMultiRecover();     // progress/status (single can be a recursive DIR)
                     if (multiSel)
                     {
                        TxPrint("Start recovery on : %u selected files/directories\n", bd->list->markinfo->marked);
                        if (dfsDirSelectDialog(
                              " Select recovery options and destination directory for Copy/Recover ",
                                DFSC_FSAVETO, dfsa->SavetoPath,
                               &dfsa->rParams.recurseDir,   "Recurse into subdirectories, recover whole tree",
                               &dfsa->rParams.name83,       "Force 8.3 compatible (short) filenames and paths",
                               &dfsa->rParams.unique,       "Force unique filenames by prefixing a sector number",
                               &dfsa->rParams.nopath,       "Discard original file path, only use its base-filename",
                               &dfsa->rParams.noAllocCheck, "Do NOT perform an allocation check (ignoring errors)",
                                NULL, NULL, NULL, NULL))
                        {
                           ULONG     i;

                           TxPrint("Copy/Recover save : %10lu items to: '%s'\n",
                                    bd->list->markinfo->marked, dfsa->SavetoPath);

                           for (i = 1; (i < bd->list->count) && !TxAbort(); i++)
                           {
                              if (bd->list->markinfo->markings[i] & TXSF_MARK_STD)
                              {
                                 item    = bd->list->items[i];

                                 if (dfsBrowseAuthorizeCopy( item->text))
                                 {
                                    snindex = item->value - TXDID_MAX;
                                    this    = dfsa->snlist[ snindex]; // Inode/Fnode/Dir-entry etc
                                    info    = dfsa->sninfo[ snindex]; // related info byte
                                    TxPrint("Copy/Recover file : # .%7.7u => ", snindex -1);
                                    rc = dfsRecoverSingle( this, info, dfsa->SavetoPath, &dfsa->rParams);
                                 }
                              }
                           }
                        }
                     }
                     else                       // single file, copy CURRENT (cursor)
                     {
                        if (dfsBrowseAuthorizeCopy( item->text))
                        {
                           TxPrint("Start recovery on : current item: '%s'\n", item->text + DFSBR_NAME);
                           if (dfsDirSelectDialog(
                                 " Select recovery options and destination directory for Copy/Recover ",
                                   DFSC_FSAVETO, dfsa->SavetoPath,
                                  &dfsa->rParams.recurseDir, "Recurse into subdirectories, recover whole tree",
                                  &dfsa->rParams.name83,     "Force 8.3 compatible (short) filenames and paths",
                                  &dfsa->rParams.unique,     "Force unique filenames by prefixing a sector number",
                                  &dfsa->rParams.nopath,     "Discard original file path, only use its base-filename",
                                  &dfsa->rParams.noAllocCheck, "Do NOT perform an allocation check (ignoring errors)",
                                   NULL, NULL, NULL, NULL))
                           {
                              TxPrint("Copy/Recover file : # .%7.7u => ", snindex -1);
                              rc = dfsRecoverSingle( this, info, dfsa->SavetoPath, &dfsa->rParams);

                              if (dfsGetMrHandled() == 1) // just a single item done
                              {
                                 if (rc == NO_ERROR)
                                 {
                                    TxNamedMessage( TRUE, 0, " INFO: Copy successful ", "Successfully copied selected item '%s' to recovery path:\n\n'%s'",
                                                         item->text + DFSBR_NAME, dfsa->rParams.recFname);
                                 }
                                 else
                                 {
                                    TXLN    errormsg;

                                    TxStripAnsiCodes( dfsGetRcString( rc, errormsg));
                                    TxNamedMessage( TRUE, 0, " WARNING: Copy issue ", "Warning on Copy/Recover of selected file:\n\n%s\n\n%s",
                                                         item->desc + 20, errormsg);
                                    rc = NO_ERROR;
                                 }
                              }
                           }
                        }
                     }
                     if (dfsGetMrHandled() > 1) // only when something done ...
                     {
                        TxNamedMessage( TRUE, 0, " INFO: multi-file copy finished ",
                                            "A total of %u items handled of which %u UNRELIABLE,\n"
                                            "with %u items recovered to: '%s'\n\n"
                                            "Last file recovered:\n\n%s",
                                             dfsGetMrHandled(),   dfsGetMrFailed(),
                                             dfsGetMrRecovered(), dfsa->SavetoPath,  dfsa->rParams.recFname);
                     }
                     txwSendMsg( hwnd, TXWM_CHAR, 0, (TXWMPARAM) TXk_F12); // restore minimized browse window
                     break;

                  case A_ELCMD:
                     if (txwPromptBox( TXHWND_DESKTOP, TXHWND_DESKTOP, NULL,
                            "Specify the command to be executed against\n"
                            "the selected file(s) or directories, with\n"
                            "the list index .NNNN as the only parameter\n"
                            " (leave empty to display each .NNNN item)",
                            " Specify .NNNN command to execute ",
                            DFSH_BR_ELCMD, TXPB_MOVEABLE | TXPB_HCENTER | TXPB_VCENTER,
                            TXMAXLN, bd->fsDotCommand) != TXDID_CANCEL)
                     {
                        txwSendMsg( hwnd, TXWM_CHAR, 0, (TXWMPARAM) TXk_F12); // minimize browse window

                        TxStrip( bd->fsDotCommand, bd->fsDotCommand, ' ', ' ');
                        if (strlen( bd->fsDotCommand) > 0)
                        {
                           strcat(  bd->fsDotCommand, " "); // non-empty needs space separator
                        }
                        if (multiSel)
                        {
                           ULONG     i;

                           TxPrint("Start '%s.NNNN' execution on : %u selected files/directories\n",
                                    bd->fsDotCommand, bd->list->markinfo->marked);

                           for (i = 1; (i < bd->list->count) && !TxAbort(); i++)
                           {
                              if (bd->list->markinfo->markings[i] & TXSF_MARK_STD)
                              {
                                 item    = bd->list->items[i];
                                 snindex = item->value - TXDID_MAX;
                                 sprintf( tempText, "%s.%u", bd->fsDotCommand, snindex - 1);
                                 rc = dfsMultiCommand( tempText, 0, FALSE, FALSE, TRUE);
                                 TxPrint("Executed for file : # .%7.7u, command RC: %s%u%s\n",
                                          snindex - 1, (rc == 0) ? CBG : CBR, rc, CNN);
                              }
                           }
                        }
                        else                    // single file, execute on CURRENT (cursor)
                        {
                           sprintf( tempText, "%s.%u", bd->fsDotCommand, snindex - 1);
                           rc = dfsMultiCommand( tempText, 0, FALSE, FALSE, TRUE);
                           TxPrint("Executed for file : # .%7.7u, command '%s' RC: %s%u%s\n",
                                    snindex - 1, bd->fsDotCommand, (rc == 0) ? CBG : CBR, rc, CNN);
                        }
                        txwSendMsg( hwnd, TXWM_CHAR, 0, (TXWMPARAM) TXk_F12); // restore minimized browse window
                     }
                     break;

                  case A_REDUC:
                     if (multiSel)
                     {
                        ULONG     i;
                        ULONG     iNew;
                        TXSELIST *list;         // full (original) list to start with
                        ULONG     count   = bd->list->markinfo->marked;
                        ULN64    *newList = NULL;
                        USHORT   *newInfo = NULL;   //- temporary new snlist/sninfo reduced

                        if (((newList = (TxAlloc( count, sizeof( ULN64 )))) != NULL) &&
                            ((newInfo = (TxAlloc( count, sizeof( USHORT)))) != NULL)  )
                        {
                           if (dfsBrowseSaveOriginal( bd) != NO_ERROR) // save original, when not done yet
                           {
                              TxNamedMessage( TRUE, 0, " ERROR: list not saved ", "Unexpected error on saving original list!");
                           }
                           list = (bd->list != NULL) ? bd->list : bd->orgList;

                           //- first stage, copy to temporary, needed due to unknown sorting order (overwriting)
                           TRACES(("Reduce: copy %u items out of original %u\n", count, list->count));
                           for (i = 0, iNew = 0; (i < list->count) && (iNew < count); i++)
                           {
                              if (list->markinfo->markings[i] & TXSF_MARK_STD)
                              {
                                 item    = list->items[i];
                                 snindex = item->value - TXDID_MAX;

                                 newList[ iNew] = dfsa->snlist[ snindex]; // Inode/Fnode/Dir-entry etc
                                 newInfo[ iNew] = dfsa->sninfo[ snindex]; // related info value

                                 TRACES(("list item %3u set snlist index %3u (from %3u) to 0x%llx ^ 0x%hx\n",
                                                       i, iNew,  snindex, newList[ iNew], newInfo[ iNew]));
                                 iNew++;
                              }
                           }

                           //- second stage, copy to actual Sectorlist
                           strcpy( dfsa->brdescript, "Reduced list");
                           dfsInitList(0, "-f -P", "-d"); // optimal for menu file-recovery (clears sninfo!)
                           dfsa->snlist[0] = count; // update sectorlist count
                           for (i = 0; i < count; i++)
                           {
                              dfsa->snlist[ i + 1] = newList[ i];
                              dfsa->sninfo[ i + 1] = newInfo[ i];
                           }
                           TRACES(("SNlist final size: %llu\n", dfsa->snlist[0]));
                           txwPostMsg( hwnd, DFSBR_WM_NEWCONTENTS, 0, 0); // refresh select/display

                           TxFreeMem( newList);
                           TxFreeMem( newInfo); // Free temporary list
                        }
                        else
                        {
                           TxNamedMessage( TRUE, 0, " ERROR: out of memory ", "Not enough memory to reduce list");
                        }
                     }
                     else
                     {
                        TxNamedMessage( TRUE, 0, " INFO: no files marked ", "No files are marked, retaining complete list");
                     }
                     break;

                  default:
                     break;
               }
               dfsa->rParams.recFname  = NULL;  // out of scope! Not allocated anymore
            }
            break;

         case TXWM_CHAR:                        // mp1 not 0 is control-WID
            if (item != NULL)
            {
               switch (lp2)
               {
                  case TXc_F:                   // make ctrl+F mark all Files
                     txwPostMsg( lhwnd, TXWM_CHAR, 0, (TXWMPARAM) TXa_LBRACKET); // to LIST!
                     break;

                  case TXc_G:                   // make ctrl+G mark all Dirs
                     txwPostMsg( lhwnd, TXWM_CHAR, 0, (TXWMPARAM) TXa_RBRACKET); // to LIST
                     break;

                  case TXc_BACKSLASH:           // change current to ROOT
                     sprintf( dfsa->brdescript, "browse: %c", FS_PATH_SEP);

                     if ((rc = DFSFNCALL(dfsa->FsFindPath, 0, 0, "", &params)) == NO_ERROR)
                     {
                        TRACES(("ROOT LSN: 0x%llx, info: 0x%llx\n", params.Lsn, params.Number));
                        rc = DFSFNCALL(dfsa->FsMakeBrowseList, params.Lsn, params.Number, "", NULL);
                        if (rc == NO_ERROR)
                        {
                           dfsa->brlist = params.Lsn; // can be used with fileinfo() to get path back
                           dfsa->brinfo = params.Number;
                           TxPrint( "Listed files/dirs : %10llu in %sROOT%s\n", dfsa->snlist[0], CBY, CNN);
                           txwPostMsg( hwnd, DFSBR_WM_NEWCONTENTS, 0, 0); // refresh select/display
                        }
                     }
                     break;

                  case TXk_F2:
                     if ((item->text[0] == 'D') || // Directory item, view META data
                         (item->text[0] == 'H') || // Folder Alias/hardlinked DIR (HFS+)
                         (item->text[0] == 'y')  ) // Deleted DIR
                     {
                        txwPostMsg( hwnd, TXWM_COMMAND, (TXWMPARAM) A_EMETA, (TXWMPARAM) TXCMDSRC_ACCELERATOR);
                     }
                     else                       // File item, edit/view in HEX-editor
                     {
                        txwPostMsg( hwnd, TXWM_COMMAND, (TXWMPARAM) A_SPACE, (TXWMPARAM) TXCMDSRC_ACCELERATOR);
                     }
                     break;

                  case TXk_F4:
                     bd->recMode = !bd->recMode; // toggle setting
                  case TXa_F4:
                     txwPostMsg( hwnd, DFSBR_WM_NEWCONTENTS, 0, 0); // refresh select/display
                     break;

                  case TXk_F5:                  // Copy/Recover file(s)
                     txwPostMsg( hwnd, TXWM_COMMAND, (TXWMPARAM) A_RECOV, (TXWMPARAM) TXCMDSRC_ACCELERATOR);
                     break;

                  case TXk_F6:                  // restore original list
                     if (dfsBrowseRestoreOriginal( bd) == NO_ERROR)
                     {
                        //- display restored list, position at original line again
                        txwPostMsg( hwnd, DFSBR_WM_SHOWCONTENTS, (TXWMPARAM) bd->list->selected, (TXWMPARAM) bd->list->top);
                     }
                     break;

                  case TXk_F9:                  // Edit filter criteria
                     dfsFileListDialog( " Specify file selection critera ",
                                         *(dfsa->snlist), TRUE, bd->bFilter);
                     txwPostMsg( hwnd, DFSBR_WM_NEWCONTENTS, 0, 0); // refresh select/display
                     break;

                  case TXk_F10:                 // popup (sub) menu with actions (1 or more files)
                     txwPostMsg( hwnd, TXWM_COMMAND, (TXWMPARAM) A_POPUP, (TXWMPARAM) TXCMDSRC_ACCELERATOR);
                     break;

                  case TXk_F11:                 // toggle show/hide hidden directories/files
                     dfsa->browseShowHidden = (dfsa->browseShowHidden) ? FALSE : TRUE; // toggle showhidden
                     this = dfsa->brlist;       // reload list (brlist destroyed by MakeBrowse!)
                     info = dfsa->brinfo;
                     rc = DFSFNCALL(dfsa->FsMakeBrowseList, this, info, "", NULL); // reload sectorlist
                     if (rc == NO_ERROR)        // list contents is now 'filtered' for show/hide
                     {
                        dfsa->brlist = this;    // remember new list directory LSN for next time :)
                        dfsa->brinfo = info;
                        TxPrint( "Listed files/dirs : %10llu in same directory (%s%s%s hidden items)\n",
                                  dfsa->snlist[0], CBC, (dfsa->browseShowHidden) ? "SHOW" : "HIDE", CNN);
                        txwPostMsg( hwnd, DFSBR_WM_NEWCONTENTS, 0, 0); // refresh select/display
                     }
                     break;

                  default:
                     rc = txwDefDlgProc( hwnd, msg, mp1, mp2);
                     break;
               }
            }
            else                                // possible widgets?
            {
               rc = txwDefDlgProc( hwnd, msg, mp1, mp2);
            }
            break;

         case TXWM_CLOSE:                       // selected to nav.this ...
            if ((item != NULL) && (item->text[0] != '-')) // item, and NOT the empty warning
            {
               nav.down        = this;          // FILE available for follow-up
               nav.down_sninfo = info;          // actions from the commandline

               TxPrint("Leaving BROWSE on : # .%7.7u == %s%s%s\n", snindex -1, CBG, item->text + DFSBR_NAME, CNN);
               TxPrint("Press <Enter> to access its metadata  (Inode, Fnode, MFT-rec, DIR-entry ...)\n");
               if (dfsa->brlist != 0)
               {
                  TxPrint( "Or use the 't' command to redisplay/access the browse directory info.\n");
                  nav.this        = dfsa->brlist; // DIR available for follow-up
                  nav.this_sninfo = dfsa->brinfo; // actions from the commandline
               }
               else                             // no valid browse list info
               {
                  nav.this        = this;       // revert to selected FILE
                  nav.this_sninfo = info;
               }
            }
         default:
            rc = txwDefDlgProc( hwnd, msg, mp1, mp2);
            break;
      }
   }
   else
   {
      rc = TX_INVALID_HANDLE;
   }
   RETURN( rc);
}                                               // end 'dfsBrowseDlgProc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Determine if dir/file should be recovered, based on filetext and history
/*****************************************************************************/
static BOOL dfsBrowseAuthorizeCopy
(
   char               *itemtext                 // IN    'D nnn% ..' item text
)
{
   BOOL                rc = TRUE;
   TXLN                text;                       // One line of text
   static BOOL         noAllocConfirm  = FALSE; // persistent alloc-confirm flag
   static BOOL         doCopyBadAlloc  = FALSE; // persistent Copy Bad

   ENTER();
   TRACES(("item: '%s'\n", itemtext));

   if (itemtext[5] == '%')                      // there IS a precentage, check it
   {
      if (atoi( itemtext + 2) < 100)            // file allocation not consistent
      {
         if (noAllocConfirm == FALSE)           // need confirmation
         {
            sprintf( text, "Allocation consistency for '%s' is less than 100%%.\n"
                           "The resulting file may be damaged, or incomplete!\n\n"
                           "Copy the file anyway ? [Y/N]: ", itemtext + DFSBR_NAME);

            doCopyBadAlloc = dfsConfirmOptDialog( 0, text, &noAllocConfirm,
                               "Remember this choice, do not ask the same question again",
                               NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
         }
         rc = doCopyBadAlloc;                   // use persistent setting
      }
   }
   BRETURN( rc);
}                                               // end 'dfsBrowseAuthorizeCopy'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Update title right-hand text based on current list filtering result
/*****************************************************************************/
static void dfsBrowseFilter2trhText
(
   TXWHANDLE           hwnd                     // IN    window with attached list
)
{
   TXWINDOW           *win;                     // base window structure
   TXSELIST           *list;                    // selection list structure

   ENTER();

   if (((win = txwWindowData( hwnd)) != NULL) && (win->class == TXW_LISTBOX))
   {
      list = win->lb.list;

      TRACES(("list:%8.8x  markinfo:%8.8x\n", list, list->markinfo));
      if (list && list->markinfo)
      {
         //- could use (static) buffer too, but markText is there anyway ...
         win->trhtext = list->markinfo->markText; // point to available buffer
         if (list->count < dfsa->snlist[0])
         {
            sprintf( win->trhtext, "Filtered out:%6llu of %6llu",
                                    dfsa->snlist[0] - list->count, dfsa->snlist[0]);
         }
         else
         {
            strcpy(  win->trhtext, "");         // empty when count EQUAL
         }
         TRACES(("TRH text:'%s'\n", win->frhtext));
      }
   }
   VRETURN();
}                                               // end 'dfsBrowseFilter2trhText'
/*---------------------------------------------------------------------------*/


// Layout of item->text
// D 100%   <SUBDIR>  2.0 KiB      123  test                 -> symlink-path
// |      |          |        |         |
// 0      7          18       27        37

// Layout of item->desc
//  2016-08-30 17:17:35                   0  D  \c\dfs\test
// |                   |                     |  |
// 0                   20                    42 45

/*****************************************************************************/
// Build selection-list with File BROWSE information from current sector list
/*****************************************************************************/
static TXSELIST *dfsMkBrowseSelist              // RET   selection list or NULL
(
   char               *filter,                  // IN    Filter string FileInfo
   BOOL                alCheck,                 // IN    Need allocation info
   char               *selName                  // IN    filename to make current
)
{
   TXSELIST           *list  = NULL;            // total list
   ULONG               size  = 500;             // initial allocated size

   ENTER();

   TRACES(( "Filter string: '%s'\n", filter));

   if (TxSelCreate( size, size, size, TXS_AS_NOSTATIC, TRUE, NULL, &list) == NO_ERROR)
   {
      list->astatus  = TXS_AS_NOSTATIC;         // all dynamic allocated

      if (dfsBrowseAdd2List( filter, alCheck, list) == NO_ERROR)
      {
         if (list->count == 0)                  // no entries matching the filter
         {
            txSelDestroy( &list);
            list = TxSelEmptyList( "- List empty, or no files match the current filter -",
                                   "The sectorlist is empty, or none of its entries "
                                   "match the specified selection string", FALSE);
         }
         if (list != NULL)
         {
            TRACES(("Count:%4u selected now: %u = '%s'\n", list->count, list->selected,
                                   (list->count == 0) ? "Empty list!" : list->items[ list->selected]->text));

            list->uservalue     = 'f';          // select all Files with ctrl+F
            list->flags         = TXSL_MULTI_QUICK; // multi-char quick select (Alt+xxx, not multiple-select!)
            list->multiQuickPos = DFSBR_NAME;   // multi-char quick starts at filename
            list->miscInfo      = 0;            // to be refined
            list->tsize         = list->asize;

            list->selected      = 0;            // always start with first ...

            list->sortinfo = &dfs_browse_sort;
            TxSelSortCurrent( list);
            list->selected      = 0;            // default FIRST as selected after sort!

            if (selName && strlen( selName))    // but reset to previous selected if specified
            {
               ULONG   i;

               for (i = 0; i < list->count; i++)
               {
                  if (strcmp( selName, list->items[ i]->text) == 0)
                  {
                     list->selected  = i;       // make this the selected item in new list
                  }
               }
            }
            TRACES(("Count:%4u selected now: %u = '%s'\n", list->count, list->selected,
                                   (list->count == 0) ? "Empty list!" : list->items[ list->selected]->text));
         }
      }
   }
   RETURN( list);
}                                               // end 'dfsMkBrowseSelist'
/*---------------------------------------------------------------------------*/

// 20170413 JvW Updated to allow 10 hexdigit datalocation (512 TiB disk limit)
// Layout of the first string as returned by FileInformation()
// D 100% 0123ABCDEF  2.0 KiB 01234567  2016-08-30 17:01:25                   0
// |      |          |        |         |                  |
// 0      7          16       25        35                 54  (was 32bit LSN)
// 0      7          18       27        37                 56  (now 40bit LSN)
/*****************************************************************************/
// Add file/directory instances from sectorlist to the Selist being built
/*****************************************************************************/
static ULONG dfsBrowseAdd2List
(
   char               *filter,                  // IN    Filter string FileInfo
   BOOL                alCheck,                 // IN    Need allocation info
   TXSELIST           *list                     // INOUT Selist being built
)
{
   ULONG               rc  = NO_ERROR;
   TXS_ITEM           *item;                    // single item
   ULONG               index;                   // index in snlist
   TXLN                string;                  // filter on INPUT, info OUT
   TXLN                fullPath;                // empty  on INPUT, path OUT
   TXLN                baseName;                // basename
   TXLN                paddPath;                // padded to constant length
   ULN64               snSize = dfsa->snlist[0];
   ULN64               info;                    // current browse dir info
   char               *sym;

   ENTER();
   TRACES(( "Filter string: '%s'\n", filter));

   if (list != NULL)                            // do we have a list ?
   {
      if (snSize > 2)
      {
         dfsProgressInit( 0, snSize, 0, "DIR-entries:", "", DFSP_STAT | DFSP_VDEC, 0);
      }

      for ( index = 1; index <= snSize; index++)
      {
         TRACES(("index: %4u  sn:0x%llx  info:0x%2.2hx\n",
                  index, dfsa->snlist[ index], dfsa->sninfo[index]));

         //- Get FileInformation, using filter, no-match if RC not NO_ERROR

         strcpy( string, filter);
         strcpy( fullPath, (alCheck) ? "" : "!"); // With or without alloc info
         if (dfsa->FsModeId  == DFS_FS_NTFS)
         {
            info = dfsa->brlist;                // related info, desired parent
         }
         else
         {
            info = dfsa->sninfo[ index];        // related info byte for FAT/EXT etc
         }
         rc = DFSFNCALL( dfsa->FsFileInformation, dfsa->snlist[ index], info, string, fullPath);
         if ((rc == NO_ERROR) || (rc == DFS_NOT_FOUND))
         {
            TRACES(("string  : '%s'\n", string));
            TRACES(("fullPath: '%s'\n", fullPath));

            if (rc == DFS_NOT_FOUND)      // partial success, error in string, basename in fullPath
            {
               strcpy( baseName,  fullPath);
               strcpy( fullPath, "File information could not be retrieved!");
               rc = NO_ERROR;
            }
            else                                // regular, complete fileinfo received
            {
               //- Remember symlink part of returned fullPath, before trimming it away
               if ((sym = strstr( fullPath, DFS_SH_LINK_STR)) != NULL) // there is a symlink part
               {
                  sprintf( paddPath, "%-50s", sym + strlen( DFS_SH_LINK_STR));
                  *sym = 0;                                 //- clip fullPath at start of symlink
                  TxStrip( fullPath, fullPath, ' ', ' ');   //- and remove leading/trailing spaces
               }

               //- Get the basename from returned fullpath, after removing trailing DIR seperator
               if (fullPath[strlen(fullPath) - 1] == FS_PATH_SEP)
               {
                  fullPath[strlen(fullPath) - 1] = 0;
               }
               strcpy( baseName, TxGetBaseName( fullPath));

               TRACES(("fullPath: '%s'  baseName: '%s'\n", fullPath, baseName));

               //- re-apply trailing path-separator for directories (make them stand out)
               if (((string[0] == 'D') || (string[0] == 'H') || (string[0] == 'y')) &&
                    (baseName[ strlen(baseName) - 1]) != '.')
               {
                  strcat( fullPath, FS_PATH_STR);
                  strcat( baseName, FS_PATH_STR);
               }
            }

            if (list->count >= list->asize)     // reached maximum array size
            {
               ULONG         newsize = 2 * list->asize + 100;

               TRACES(("realloc items to %u items\n",  newsize));
               if (((list->items              = realloc( list->items, sizeof(TXS_ITEM *) * newsize)) != NULL) &&
                   ((list->markinfo->markings = realloc( list->markinfo->markings,     1 * newsize)) != NULL)  )
               {
                  //- Note: 'new' part of arrays are NOT zeroed by re-alloc!
                  list->asize = newsize;        // size has more than doubled now
               }
               else
               {
                  rc = DFS_ALLOC_ERROR;
               }
            }
            if (rc == NO_ERROR)
            {
               if ((item = TxAlloc( 1, sizeof(TXS_ITEM))) != NULL)
               {
                  TRACES(("Alloc: %u, adding item %u: '%s' for '%s'\n", list->asize, list->count, string, fullPath));

                  list->markinfo->markings[ list->count] = 0; // make sure it is ZERO

                  item->userdata = (void *) list->count; // assign item number for unsort
                  list->items[list->count++] = item; // attach item to list

                  item->value = TXDID_MAX + index; // will be returned value on select

                  if ( (string[0] == 'D') || (string[0] == 'y')      || // Is a Directory, perhaps a Parent
                      ((string[0] == 'H') && (atol( string + 7) == 0))) // Active DIR hardlink (fsize is 0)
                  {
                     if ((index == 1) && (DFSBR_SnlistFlag & DFSBR_1ST_PARENT))
                     {
                        strcpy( baseName, ".."); // force the DOTDOT base name
                        memcpy( string + 7, "  <UP-DIR>", 10);
                     }
                     else                       // regular SUBDIR
                     {
                        memcpy( string + 7, "  <SUBDIR>", 10);
                     }
                     item->flags |= TXSF_TXTCOLOR1; // use 1st alternative color for DIRs
                  }
                  if ((item->text = TxAlloc( 1, strlen(baseName) + 40)) != NULL)
                  {
                     sprintf( item->text, "%-34.34s %s", string, baseName);
                  }

                  if (sym == NULL)              // paddPath not created yet, no symlink
                  {
                     sprintf( paddPath, "%-50s", fullPath);
                  }
                  if ((item->desc = TxAlloc( 1, strlen(paddPath) + 50)) != NULL)
                  {
                     sprintf( item->desc, "%s  %c  %s", string + 36, string[0], paddPath);
                  }
                  item->helpid = TXWH_USE_WIN_HELP; // from list-window itself
                  TRACES(("item: 0x%p sequence:% 5u\n", item, item->userdata));
               }
            }
         }
         if (snSize > 2)
         {
            dfsProgressShow( index, 0, 0, NULL);
         }
      }
      if (snSize > 2)
      {
         dfsProgressTerm();
         //- txwSetSbviewStatus( "", cSchemeColor);
      }
   }
   else
   {
      TRACES(("supplied listpointer is NULL!\n"));
   }
   RETURN (rc);
}                                               // end 'dfsBrowseAdd2List'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Save original raw and rendered lists (ONCE!) for later 'restore original'
// Will leave the ->list unallocated NULL, when identical to saved on exit
/*****************************************************************************/
static ULONG dfsBrowseSaveOriginal
(
   DFSBROWSEDATA      *brd                      // INOUT browse data
)
{
   ULONG               rc = NO_ERROR;           // function return
   size_t              savedSize;               // number of entries in saved list

   ENTER();

   if ((brd->orgSnlist == NULL) && (brd->orgList == NULL)) // save only once
   {
      savedSize = (size_t) ((*(dfsa->snlist)) + 1);

      if ((brd->orgSnlist = TxAlloc( savedSize, sizeof( ULN64))) != NULL)
      {
         if ((brd->orgSninfo = TxAlloc( savedSize, sizeof( USHORT))) != NULL)
         {
            memcpy( brd->orgSnlist, dfsa->snlist, savedSize * sizeof( ULN64 )); // save snlist
            memcpy( brd->orgSninfo, dfsa->sninfo, savedSize * sizeof( USHORT)); // save sninfo

            //- list includes the current 'selection' (position in list)
            brd->orgList = brd->list;           // move the list to 'org' location

            brd->orgBrlist            = dfsa->brlist;
            brd->orgBrinfo            = dfsa->brinfo;
            strcpy( brd->orgBrdescript, dfsa->brdescript);

            TRACES(("Original list saved: orgList:%p\n", brd->orgList));
         }
         else
         {
            TxFreeMem( brd->orgSnlist);
            rc = DFS_ALLOC_ERROR;
         }
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   else
   {
      TRACES(("Original was already saved\n"));
   }
   if (brd->list == brd->orgList)               // avoid destroying our saved list!
   {
      brd->list    = NULL;                      // remove it from BRD for rebuild!
   }
   RETURN (rc);
}                                               // end 'dfsBrowseSaveOriginal'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Restore original raw and rendered lists from previously saved instance
/*****************************************************************************/
static ULONG dfsBrowseRestoreOriginal
(
   DFSBROWSEDATA      *brd                      // INOUT browse data
)
{
   ULONG               rc = DFS_NO_CHANGE;      // function return
   size_t              savedSize;               // number of entries in saved list

   ENTER();

   TRACES(("orgSnlist:%p orgList:%p brd->list:%p\n", brd->orgSnlist, brd->orgList, brd->list));

   if ((brd->orgSnlist != NULL) && (brd->orgList != NULL))
   {
      if (brd->list != brd->orgList)
      {
         savedSize = (size_t) ((*(brd->orgSnlist)) + 1);

         memcpy( dfsa->snlist, brd->orgSnlist, savedSize * sizeof( ULN64 )); // restore snlist
         memcpy( dfsa->sninfo, brd->orgSninfo, savedSize * sizeof( USHORT)); // restore sninfo

         dfsa->brlist            = brd->orgBrlist;
         dfsa->brinfo            = brd->orgBrinfo;
         strcpy( dfsa->brdescript, brd->orgBrdescript);

         txSelDestroy( &brd->list);             // free existing list
         brd->list    = brd->orgList;           // and replace by saved one (alias!)

         TRACES(("Original list restored\n"));
         rc = NO_ERROR;                         // restored original, redisplay needed
      }
   }
   RETURN (rc);
}                                               // end 'dfsBrowseRestoreOriginal'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Discard original raw and rendered lists from previously saved instance
/*****************************************************************************/
static void dfsBrowseDiscardOriginal
(
   DFSBROWSEDATA      *brd                      // INOUT browse data
)
{
   ENTER();

   txSelDestroy( &brd->orgList);                // free  list
   TxFreeMem(     brd->orgSnlist);
   TxFreeMem(     brd->orgSninfo);

   VRETURN();
}                                               // end 'dfsBrowseDiscardOriginal'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// Build dialog-footer text based on current browse-data contents
/*****************************************************************************/
static void dfsBrowseSetFooterText
(
   DFSBROWSEDATA      *brd                      // INOUT browse data
)
{
   ENTER();

   sprintf( brd->footer, dfs_browse_footer,              (brd->recMode) ? dfs_br_rec_footer : dfs_br_fst_footer,
           (brd->orgList && brd->list && (brd->orgList != brd->list))   ? dfs_br_org_footer : dfs_br_no6_footer,
                                             (dfsa->browseShowHidden)   ? dfs_br_hid_footer : dfs_br_shw_footer);
   VRETURN();
}                                               // end 'dfsBrowseSetFooterText'
/*---------------------------------------------------------------------------*/

#endif
