//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// FAT bootsector format/fixboot related functions
//
// Author: J. van Wijk
//
// JvW  14-02-2007 Initial version, split-off from DFSCFDSK
//

#include <txlib.h>                              // TX library interface

#include <dfsver.h>                             // DFS version info
#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfsupart.h>                           // FDISK partition functions
#include <dfsafdsk.h>                           // FDISK display & analysis
#include <dfscfdsk.h>                           // FDISK callback and commands
#include <dfsufdsk.h>                           // FDISK utility functions
#include <dfsbfat.h>                            // FAT bootsector functions


#define        DFSFDSK_MODN_OFFSET     0x1e6    // offset & value MSdos module
static char    dfsFdskFatMsdName[] =   "IO      SYSMSDOS   SYS";
static USHORT  dfsFdskFatDosCstart =   0x33fa;
static ULONG   dfsFdskFatDosCode[] =                //- From IBM DOS 5
{
   0xbcd08ec0, 0x07167c00, 0x360078bb, 0x561e37c5,  //- [м|x67V]
   0x3ebf5316, 0x000bb97c, 0x06a4f3fc, 0xfe45c61f,  //- [S>|E]
   0x180e8b0f, 0xf94d887c, 0xc7024789, 0xfb7c3e07,  //- [|MG>|]
   0x797213cd, 0x0639c033, 0x08747c13, 0x7c130e8b,  //- [ry39|t|]
   0x7c200e89, 0xf77c10a0, 0x037c1626, 0x137c1c06,  //- [ ||&||]
   0x037c1e16, 0x837c0e06, 0x50a300d2, 0x5216897c,  //- [||P|R]
   0x7c49a37c, 0x7c4b1689, 0xf70020b8, 0x8b7c1126,  //- [|I|K| &|]
   0x037c0b1e, 0xf3f748c3, 0x7c490601, 0x7c4b1683,  //- [|HI|K|]
   0x0500bb00, 0x7c52168b, 0xe87c50a1, 0x1d720092,  //- [R|P|r]
   0xace801b0, 0x8b167200, 0x000bb9fb, 0xf37de6be,  //- [r}]
   0x8d0a75a6, 0x0bb9207f, 0x74a6f300, 0x7d9ebe18,  //- [u t}]
   0x33005fe8, 0x5e16cdc0, 0x8f048f1f, 0x19cd0244,  //- [_3^D]
   0xeb585858, 0x1a478be8, 0x1e8a4848, 0xff327c0d,  //- [XXXGHH|2]
   0x0603e3f7, 0x16137c49, 0x00bb7c4b, 0x0003b907,  //- [I|K|]
   0xe8515250, 0xd872003a, 0x54e801b0, 0x585a5900,  //- [PRQ:rذTYZX]
   0x0105bb72, 0x00d28300, 0x7c0b1e03, 0x2e8ae2e2,  //- [r|.]
   0x168a7c15, 0x1e8b7c24, 0x4ba17c49, 0x0000ea7c,  //- [|$|I|K|]
   0x0aac0070, 0xb42974c0, 0x0007bb0e, 0xf2eb10cd,  //- [pt)]
   0x7c18163b, 0x36f71973, 0xc2fe7c18, 0x7c4f1688,  //- [;|s6|O|]
   0x36f7d233, 0x16887c1a, 0x4da37c25, 0xf9c3f87c,  //- [36|%|M|]
   0x8b02b4c3, 0xb17c4d16, 0x0ae6d206, 0x8b7c4f36,  //- [ôM|6O|]
   0x8ae986ca, 0x8a7c2416, 0xcd7c2536, 0x0a0dc313,  //- [ʆ$|6%|]
   0x2d6e6f4e, 0x74737953, 0x64206d65, 0x206b7369,  //- [Non-System disk ]
   0x6420726f, 0x206b7369, 0x6f727265, 0x520a0d72,  //- [or disk errorR]
   0x616c7065, 0x61206563, 0x7020646e, 0x73736572,  //- [eplace and press]
   0x796e6120, 0x79656b20, 0x65687720, 0x6572206e,  //- [ any key when re]
   0x0d796461, 0x4249000a, 0x4f49424d, 0x4f432020,  //- [adyIBMBIO  CO]
   0x4d42494d, 0x20534f44, 0x4d4f4320, 0xaa550000,  //- [MIBMDOS  COMU]
};

static USHORT  dfsFdskFatOs2Cstart =   0x0000;
static ULONG   dfsFdskFatOs2Code[] =                //- From Warp V4
{
   0x00001000, 0x33fa0000, 0xbcd38edb, 0xbafb7bff,  //- [3ێӼ{]
   0xda8e07c0, 0x00243e80, 0x1e3d7500, 0x8e0040b8,  //- [ڀ>$u=@]
   0x0eff26c0, 0x12cd0013, 0x8e06e0c1, 0x33ff33c0,  //- [&33]
   0xc5d88ec0, 0xfc007836, 0xf3000bb9, 0x18a11fa4,  //- [6x]
   0x04a22600, 0xc0331e00, 0x78a3d88e, 0x7a068c00,  //- [&3أxz]
   0x168a1f00, 0x13cd0024, 0x980010a0, 0x001626f7,  //- [$&]
   0x000e0603, 0x20b89150, 0x1126f700, 0x0b1e8b00,  //- [P &]
   0x48c30300, 0x0350f3f7, 0x003ea3c1, 0x8e1000b8,  //- [HP>]
   0x59ff33c0, 0x00440e89, 0x0042a358, 0x73e8d233,  //- [3YDXB3s]
   0x8bdb3300, 0x8b00110e, 0x0bb951fb, 0x01dbbe00,  //- [3ۋQ]
   0x7459a6f3, 0x20c38305, 0x35e3ede2, 0x1c478b26,  //- [Yt 5&G]
   0x1e578b26, 0x000b36f7, 0xc88ac0fe, 0x1a578b26,  //- [&W6&W]
   0x0da04a4a, 0xf7e43200, 0x3e0603e2, 0x00d28300,  //- [JJ2>]
   0x8e0800bb, 0x06ff33c3, 0x0028e857, 0x000b368d,  //- [3W(6]
   0x019ebecb, 0xb3be03eb, 0x0009e801, 0xe801c8be,  //- [˾]
   0xebfb0003, 0xc00aacfe, 0x0eb40974, 0xcd0007bb,  //- [t]
   0xc3f2eb10, 0x03515250, 0x13001c06, 0x50001e16,  //- [PRQP]
   0xf6001aa1, 0x8b001826, 0xf3f758d8, 0x1836f692,  //- [&X6]
   0x8be08600, 0xa1c3fed8, 0xc32a0018, 0x02b45040,  //- [á*@P]
   0xe6d206b1, 0xca8bf30a, 0x168ae986, 0xf78a0024,  //- [ʆ$]
   0x13cddf8b, 0x595ba072, 0x26f7c38b, 0xf803000b,  //- [r[Y&]
   0xc303585a, 0x2a00d283, 0xc3a97fcb, 0x534f0012,  //- [ZXÃ*OS]
   0x2120322f, 0x59532021, 0x34313053, 0x0a0d3537,  //- [/2 !! SYS01475]
   0x4f001200, 0x20322f53, 0x53202121, 0x32305359,  //- [OS/2 !! SYS02]
   0x0d353230, 0x0012000a, 0x322f534f, 0x20212120,  //- [025OS/2 !! ]
   0x30535953, 0x37323032, 0x4f000a0d, 0x4f423253,  //- [SYS02027OS2BO]
   0x2020544f, 0x00002020, 0x00000000, 0x00000000,  //- [OT    ]
   0x00000000, 0x00000000, 0x00000000, 0xaa550000   //- [U]
};


static USHORT  dfsFdskFatOs2Vstart  =   0x534f;
static ULONG   dfsFdskFatOs2VCode[] =               //- From Veit's OS2LDR.SEK
{
   0x52444c32, 0x20202020, 0x7c00bd20, 0xe58b170e,  //- 2LDR     ..|....
   0x56881f0e, 0xb70f6624, 0x00bb0e46, 0x167e8b40,  //- ...V$f..F...@.~.
   0xbce8570e, 0xf6586600, 0x46031066, 0x117e8b0e,  //- .W...fX.f..F..~.
   0x04efc157, 0x8966c703, 0xc72b4946, 0x8e1000bb,  //- W.....f.FI+.....
   0x009de8c3, 0x7c3ebe59, 0x60fcff2b, 0xf3000bb9,  //- ....Y.>|+..`....
   0x0a7461a6, 0xe220c783, 0x7debbef2, 0x8b2670eb,  //- .at... ....}.p&.
   0x00bb1a45, 0x0e0e5320, 0x47ff2b50, 0x4000ba1e,  //- E... S..P+.G...@
   0xf003f08b, 0x50b60273, 0xf1b9da8e, 0x167e83ff,  //- ....s..P......~.
   0x77148b0c, 0xd1f00310, 0x72148bee, 0x04e2c103,  //- ...w.......r....
   0xb504eac1, 0x72d13b0f, 0x1fd22b02, 0x74c23b40,  //- .....;.r.+..@;.t
   0x525866ca, 0xbc0f4848, 0xd3660d4e, 0x66e7d3e0,  //- .fXRHH..N.f....f
   0xe8494603, 0x8558002b, 0x99ac75c0, 0x8d24568a,  //- .FI.+.X..u...V$.
   0xff2b0b76, 0xcb495e8b, 0xdb2b0eb4, 0x2efc10cd,  //- v.+..^I...+.....
   0x75c084ac, 0xcdc02bf3, 0x8a19cd16, 0x13cd2456,  //- ...u.+......V$..
   0x036606c3, 0xb9571c46, 0xf93b0020, 0xf98b0276,  //- ..f.F.W. .;.v...
   0x6860661e, 0xbe1f3000, 0x04c70008, 0x89660010,  //- .f`h.0........f.
   0x2b660844, 0x448966c0, 0x027c890c, 0x89044489,  //- D.f+.f.D..|..D..
   0x41b4065c, 0xe855aabb, 0xfb81ffc1, 0x75f9aa55,  //- \..A..U.....U..u
   0xe842b40d, 0x0672ffb5, 0x3149b866, 0x89665833,  //- ..B...r.f.I13Xf.
   0x8b66f844, 0x89661c46, 0x6166fc44, 0x73cf8b1f,  //- D.f.F.f.D.fa...s
   0x6650664a, 0xf75a5850, 0xca8b1876, 0x2b41f18b,  //- JfPfPXZ.v.....A+
   0x1a76f7d2, 0xe4c0f28a, 0x0bc48606, 0x8bc38ec8,  //- ..v.............
   0xc62b1846, 0x0276c73b, 0x05bec78b, 0x02b46000,  //- F.+.;.v......`..
   0x66e8db2b, 0xb40f73ff, 0xff5fe800, 0xed754e61,  //- +..f.s...._.aNu.
   0xe97ddbbe, 0x8b61ff48, 0x5f5866c8, 0xc3834066,  //- ..}.H.a..fX_f@..
   0x05744f20, 0x4de9f6e2, 0x52c307ff, 0x20646165,  //-  Ot....M...Read
   0x6f727265, 0x0d202172, 0x4e00070a, 0x7973206f,  //- error! ....No sy
   0x6d657473, 0x0d202021, 0x0000070a, 0xaa550000   //- stem!  .......U.
};

static USHORT  dfsFdskFatWntCstart =   0xc033;
static ULONG   dfsFdskFatWntCode[] =                //- From Windows NT 4
{
   0x00bcd08e, 0x07c0687c, 0x0010a01f, 0x001626f7,  //- [м|h&]
   0x000e0603, 0x20b89150, 0x1126f700, 0x0b1e8b00,  //- [P &]
   0x48c30300, 0xc803f3f7, 0x02080e89, 0x07100068,  //- [Hȉh]
   0x068fdb33, 0x1e890213, 0xe80e0215, 0x57720090,  //- [3ۏrW]
   0x0e8bdb33, 0xfb8b0011, 0x000bb951, 0xf301dcbe,  //- [3ۋQ]
   0x057459a6, 0xe220c383, 0x2637e3ed, 0x521a578b,  //- [Yt 7&WR]
   0x680001b8, 0x33072000, 0x48e80edb, 0x5b287200,  //- [h 3Hr([]
   0x000b368d, 0x020b3e8d, 0x02458f1e, 0x00f505c7,  //- [6>E]
   0x06458f1e, 0x0e0445c7, 0x24168a01, 0x0003ea00,  //- [EE$]
   0x86be2000, 0xbe03eb01, 0x09e801a2, 0x01c1be00,  //- [ ]
   0xfb0003e8, 0x0aacfeeb, 0xb40974c0, 0x0007bb0e,  //- [t]
   0xf2eb10cd, 0x4a4a50c3, 0x32000da0, 0x03e2f7e4,  //- [PJJ2]
   0x83020806, 0x13a300d2, 0x15168902, 0x07a25802,  //- [X]
   0x0213a102, 0x0215168b, 0x001c0603, 0x001e1613,  //- []
   0x001836f7, 0x1688c2fe, 0xd2330206, 0x001a36f7,  //- [636]
   0x00251688, 0xa10204a3, 0x062a0018, 0x3a400206,  //- [%*@:]
   0x76020706, 0x0207a005, 0xb450e432, 0x040e8b02,  //- [v2P]
   0x06e5c002, 0x02062e0a, 0x168be986, 0x13cd0024,  //- [.$]
   0x0005830f, 0xf902c483, 0x062858cb, 0x11760207,  //- [X(v]
   0x02130601, 0x02151683, 0x0b26f700, 0xebd80300,  //- [&]
   0x0207a290, 0x4f42cbf8, 0x203a544f, 0x6c756f43,  //- [BOOT: Coul]
   0x74276e64, 0x6e696620, 0x544e2064, 0x0d52444c,  //- [dn't find NTLDR]
   0x4f42000a, 0x203a544f, 0x204f2f49, 0x6f727265,  //- [BOOT: I/O erro]
   0x65722072, 0x6e696461, 0x69642067, 0x0a0d6b73,  //- [r reading disk]
   0x656c5000, 0x20657361, 0x65736e69, 0x61207472,  //- [Please insert a]
   0x68746f6e, 0x64207265, 0x006b7369, 0x444c544e,  //- [nother diskNTLD]
   0x20202052, 0x00202020, 0x00000000, 0x00000000,  //- [R      ]
   0x00000000, 0x00000000, 0x00000000, 0xaa550000,  //- [U]
};


static USHORT  dfsFdskF16W98Cstart =   0xc933;
static ULONG   dfsFdskF16W98Code[] =                //- From Win-98 FAT16
{
   0xfcbcd18e, 0xbd07167b, 0x76c50078, 0x16561e00,  //- [....{...x..v..V.]
   0x0522bf55, 0x89007e89, 0x0bb1024e, 0x06a4f3fc,  //- [U."..~..N.......]
   0x7c00bd1f, 0x0ffe45c6, 0x7d244e38, 0x99c18b20,  //- [...|.E..8N$} ...]
   0x83017ee8, 0xa1663aeb, 0x3b667c1c, 0xfc578a07,  //- [.~...:f..|f;..W.]
   0xca800675, 0x02568802, 0x7310c380, 0xfec933ed,  //- [u.....V....s.3..]
   0x8a7dd806, 0xf7981046, 0x46031666, 0x1e56131c,  //- [..}.F...f..F..V.]
   0x130e4603, 0x11768bd1, 0xfc468960, 0xb8fe5689,  //- [.F....v.`.F..V..]
   0xe6f70020, 0x030b5e8b, 0xf3f748c3, 0x11fc4601,  //- [ ....^...H...F..]
   0xbf61fe4e, 0x28e80700, 0x383e7201, 0x6017742d,  //- [N.a....(.r>8-t.`]
   0xd8be0bb1, 0x61a6f37d, 0x744e3d74, 0x20c78309,  //- [....}..at=Nt... ]
   0xe772fb3b, 0x0efeddeb, 0xa77b7dd8, 0xac7d7fbe,  //- [;.r......}{...}.]
   0xacf00398, 0x0c744098, 0xb4137448, 0x0007bb0e,  //- [.....@t.Ht......]
   0xefeb10cd, 0xeb7d82be, 0x7d80bee6, 0x16cde1eb,  //- [......}....}....]
   0x8f661f5e, 0xbe19cd04, 0x7d8b7d81, 0xfe458d1a,  //- [^.f......}.}..E.]
   0xf70d4e8a, 0xfc4603e1, 0xb1fe5613, 0x00c2e804,  //- [.N....F..V......]
   0x00ead772, 0x52007002, 0x6a530650, 0x91106a01,  //- [r....p.RP.Sj.j..]
   0xa218468b, 0x92960526, 0xf6f7d233, 0x42f6f791,  //- [.F..&...3......B]
   0x76f7ca87, 0x8af28a1a, 0x02ccc0e8, 0x01b8cc0a,  //- [...v............]
   0x027e8002, 0xb404750e, 0x8af48b42, 0x13cd2456,  //- [..~..u..B...V$..]
   0x0a726161, 0x42017540, 0x490b5e03, 0x03c37775,  //- [aar.@u.B.^.Iuw..]
   0x0d270118, 0x766e490a, 0x64696c61, 0x73797320,  //- [..'..Invalid sys]
   0x206d6574, 0x6b736964, 0x440a0dff, 0x206b7369,  //- [tem disk...Disk ]
   0x204f2f49, 0x6f727265, 0x0a0dff72, 0x6c706552,  //- [I/O error...Repl]
   0x20656361, 0x20656874, 0x6b736964, 0x6e61202c,  //- [ace the disk, an]
   0x68742064, 0x70206e65, 0x73736572, 0x796e6120,  //- [d then press any]
   0x79656b20, 0x00000a0d, 0x20204f49, 0x20202020,  //- [ key....IO      ]
   0x4d535953, 0x534f4453, 0x53202020, 0x017f5359,  //- [SYSMSDOS   SYS..]
   0x00bb4100, 0x6a666007, 0xff3be900, 0xaa550000,  //- [.A...`fj..;...U.]
};

static USHORT  dfsFdskF32W98Cstart =   0x33fa;
static ULONG   dfsFdskF32W98Code[] =                //- From Win-98 preload
{
                                       0xbcd18ec9,  //- [            ɎѼ]
   0xc18e7bf8, 0xc50078bd, 0x561e0076, 0x22bf5516,  //- [{xvVU"]
   0x007e8905, 0xb1024e89, 0xa4f3fc0b, 0x00bdd98e,  //- [~Nٽ]
   0xfe45c67c, 0x18468b0f, 0x38f94588, 0x257d404e,  //- [|EFE8N@}%]
   0xbb99c18b, 0x97e80700, 0x831a7200, 0xa1663aeb,  //- [r:f]
   0x3b667c1c, 0xfc578a07, 0xca800675, 0x02568802,  //- [|f;WuV]
   0x7310c380, 0x0002bfed, 0x00167e83, 0x468b4575,  //- [s~uEF]
   0x1e568b1c, 0x490003b9, 0x42017540, 0xe87e00bb,  //- [VI@uB~]
   0x2673005f, 0x744ff8b0, 0x32468b1d, 0x03b9d233,  //- [_s&OtF23ҹ]
   0x77c83b00, 0x0e768b1e, 0x1773ce3b, 0x4603f12b,  //- [;wv;s+F]
   0x1e56131c, 0x0b73d1eb, 0x7e8327eb, 0x0377002a,  //- [Vs'~*w]
   0xbe02fde9, 0x98ac7d7e, 0x84acf003, 0x3c1774c0,  //- [~}t<]
   0xb40974ff, 0x0007bb0e, 0xeeeb10cd, 0xeb7d81be,  //- [t}]
   0x7d7fbee5, 0xcd98e0eb, 0x661f5e16, 0x19cd048f,  //- [}^f]
   0x6a665641, 0x06505200, 0x6a016a53, 0x60f48b10,  //- [AVfjRPSjj`]
   0x0e027e80, 0x42b40475, 0x92911deb, 0x76f7d233,  //- [~uB3v]
   0x76f79118, 0xca874218, 0x8a1a76f7, 0xc0e88af2,  //- [vBv]
   0xcc0a02cc, 0x8a0201b8, 0x13cd4056, 0x10648d61,  //- [̸V@ad]
   0x400a725e, 0x03420175, 0x75490b5e, 0x1803c3b4,  //- [^r@uB^Iu]
   0x0a0d2701, 0x61766e49, 0x2064696c, 0x74737973,  //- ['Invalid syst]
   0x64206d65, 0xff6b7369, 0x69440a0d, 0x49206b73,  //- [em diskDisk I]
   0x65204f2f, 0x726f7272, 0x520a0dff, 0x616c7065,  //- [/O errorRepla]
   0x74206563, 0x64206568, 0x2c6b7369, 0x646e6120,  //- [ce the disk, and]
   0x65687420, 0x7270206e, 0x20737365, 0x20796e61,  //- [ then press any ]
   0x0d79656b, 0x0000000a, 0x20204f49, 0x20202020,  //- [keyIO      ]
   0x4d535953, 0x534f4453, 0x53202020, 0x017e5359,  //- [SYSMSDOS   SYS~]
   0x4e495700, 0x544f4f42, 0x53595320, 0xaa550000,  //- [WINBOOT SYSU]
};



/*****************************************************************************/
// Implement the 'Fat2Boot' callback operation on a partition-table entry
/*****************************************************************************/
ULONG dfsFdskFat2Boot                           // RET PENDING  => call again
(                                               //     NO_ERROR => all done
   DFSPARTINFO        *p,                       // INOUT partition info
   S_BOOTR            *br,                      // INOUT boot sector
   FDSK_CB_PHASE       phase,                   // IN    callback phase
   FDSK_CB_INFO       *cbp                      // IN    string: OS name
)
{
   ULONG               rc = NO_ERROR;           // function return
   TXTS                osname;

   ENTER();
   TRACES(("Phase:%u Related:%u Confirmed:%u Option:%u Flag:%u  Disk:%hu thisSn:%llx\n",
            phase, cbp->related, cbp->confirmed, cbp->option, cbp->flag, cbp->disknr, cbp->thisSn));
   if (p != NULL)                               // info available
   {
      rc = dfsMakeFatOsString( cbp->string[0], osname);
      switch (phase)
      {
         case FDSK_PREPARE:
            if (rc == NO_ERROR)
            {
               if ((dfsa->batch == FALSE) && (cbp->confirmed == FALSE))
               {
                  TXLN    s1;

                  sprintf(   s1, "Partition %hu disk %hu, %s %s\nLabel '%s' by %s\n",
                             p->id, p->disknr,  (p->primary) ? "primary" : "logical",
                             p->fsform, p->plabel,   p->creatr);
                  dfstrSz64( s1, "Partition size: ", p->sectors, "\n\n");

                  if (!TxConfirm( 5076, "%sChange partition to bootable %s "
                                        "? [Y/N] : ", s1, osname))
                  {
                     rc = DFS_CMD_FAILED;       // aborted
                  }
               }
            }
            break;

         case FDSK_PERFORM:
            switch (p->partent.PartitionType & ~DFS_P_PHIDDEN)
            {
               case DFS_P_FAT12:
               case DFS_P_FAT16:
               case DFS_P_BIGDOS:
               case DFS_P_FAT32:
               case DFS_P_FAT32X:
               case DFS_P_ACRONISSZ:
               case DFS_P_FAT16X:
                  if (dfsInsertFatBootCode(
                      (char) toupper(cbp->string[0]), br))
                  {
                     TxPrint( "Partition %2u changed, type: %s to boot %s\n",
                               p->id, p->descr, osname);
                     rc = DFS_PENDING;
                  }
                  else
                  {
                     rc = DFS_VALUE_ERROR;
                  }
                  break;

               default:
                  TxPrint( "Skipped part %2u, no FAT type: %s\n",
                            p->id, p->descr);
                  break;
            }
            break;

         case FDSK_PENDING:
            strcpy( p->descr, "-Fat2Boot");
            TxPrint("\nPartition prepared for bootable FAT %s\n", osname);
            dfsFdskSingleAutoShow( p->disknr);
            break;

         default:
            rc = DFS_CMD_FAILED;
            break;
      }
   }
   RETURN (rc);
}                                               // end 'dfsFdskFat2Boot'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get operating system description for specified os
/*****************************************************************************/
ULONG dfsMakeFatOsString
(
   char                os,                      // IN    OS: I P M N O V W 9
   char               *string                   // OUT   OS descr (max 15 char)
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   switch (toupper(os))
   {
      case 'I':
      case 'P': strcpy( string, "IBM/PCDOS");       break;
      case 'M': strcpy( string, "MSDOS");           break;
      case 'N': strcpy( string, "Win-NT");          break;
      case 'O': strcpy( string, "OS/2");            break;
      case 'V': strcpy( string, "Veit's OS2LDR");   break;
      case 'W': strcpy( string, "Win-9x (FAT16)");  break;
      case '9': strcpy( string, "Win-98 (FAT32)");  break;
      default:  strcpy( string, "Invalid OS code");
         rc = DFS_VALUE_ERROR;
         break;
   }
   TRACES(( "OSstring: '%s'", string));
   RETURN(rc);
}                                               // end 'dfsMakeFatOsString'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Insert bootcode for specified OS in the supplied bootsector
/*****************************************************************************/
BOOL dfsInsertFatBootCode                       // RET   code replaced
(
   char                os,                      // IN    OS: I P M N O V W 9
   S_BOOTR            *br                       // INOUT boot sector
)
{
   BOOL                rc = TRUE;               // function return

   ENTER();
   TRARGS(("Operating system spec: %2.2hx = '%c'\n", os, os));

   memcpy( br->OpSys, (os == 'V') ? "OS2LDR.s" : DFS_VLABEL, BT_SYST_L);
   switch (os)
   {
      case 'I':                                 // IBM-dos
      case 'P':                                 // PC-Dos
      case 'M':                                 // MS-dos
         br->Instruction[0] = DFS_B_OPCO_2;
         br->Instruction[1] = DFS_B_FATDOS;
         br->Instruction[2] = DFS_B_OPCNOP;
         br->os.CodeStart   = dfsFdskFatDosCstart;
         memcpy( br->os.Code, dfsFdskFatDosCode,
                       sizeof(dfsFdskFatDosCode));
         if (os == 'M')
         {
            char        *s = (char *) br;

            memcpy( s + DFSFDSK_MODN_OFFSET,
                        dfsFdskFatMsdName,
                 sizeof(dfsFdskFatMsdName));
         }
         break;

      case 'N':                                 // Windows NT
         br->Instruction[0] = DFS_B_OPCO_2;
         br->Instruction[1] = DFS_B_FATWNT;
         br->Instruction[2] = DFS_B_OPCNOP;
         br->os.CodeStart   = dfsFdskFatWntCstart;
         memcpy( br->os.Code, dfsFdskFatWntCode,
                       sizeof(dfsFdskFatWntCode));
         break;

      case 'O':                                 // OS/2 (eCS, 4.50 kernel)
         br->Instruction[0] = DFS_B_OPCO_2;
         br->Instruction[1] = DFS_B_FATOS2;
         br->Instruction[2] = DFS_B_OPCNOP;
         br->os.CodeStart   = dfsFdskFatOs2Cstart;
         memcpy( br->os.Code, dfsFdskFatOs2Code,
                       sizeof(dfsFdskFatOs2Code));
         break;

      case 'V':                                 // Veit's OS2LDR.SEK
         br->Instruction[0] = DFS_B_OPCO_2;
         br->Instruction[1] = DFS_B_FATVKO;
         br->Instruction[2] = DFS_B_OPCNOP;
         br->os.CodeStart   = dfsFdskFatOs2Vstart;
         memcpy( br->os.Code, dfsFdskFatOs2VCode,
                       sizeof(dfsFdskFatOs2VCode));
         break;

      case 'W':                                 // Win-98 FAT16 bootsector
         br->Instruction[0] = DFS_B_OPCO_2;
         br->Instruction[1] = DFS_B_FATWNT;     // same offset at WNT
         br->Instruction[2] = DFS_B_OPCNOP;
         br->os.CodeStart   = dfsFdskF16W98Cstart;
         memcpy( br->os.Code, dfsFdskF16W98Code,
                       sizeof(dfsFdskF16W98Code));
         break;

      case '9':                                 // Win-98 FAT32 bootsector
         br->Instruction[0] = DFS_B_JMPW98;
         br->Instruction[1] = DFS_B_F32W98;
         br->Instruction[2] = DFS_B_OPCNOP;
         br->f2.CodeStart   = dfsFdskF32W98Cstart;
         memcpy( br->f2.Code, dfsFdskF32W98Code,
                       sizeof(dfsFdskF32W98Code));
         break;

      default:                                  // un-recognized
         rc = FALSE;
         break;
   }
   BRETURN (rc);
}                                               // end 'dfsInsertFatBootCode'
/*---------------------------------------------------------------------------*/

