//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// XFS dump & display Analysis functions
//
// Author: J. van Wijk
//
// JvW  15-07-2007 Initial version, derived from EXT
// JvW  04-03-2018 Update, adding superblock display, and Uuid/VolumeName display/edit
//
// Author: J. van Wijk

// Note: XFS is a BIG-ENDIAN oriented FS, so all field access
//       needs to use the TxBE16/32/64 macros, see txlib.h
//       Structures remain in their original endianess (big) in memory

#include <txlib.h>                              // TX library interface

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfsposix.h>                           // POSIX utility functions
#include <dfsspace.h>                           // DFS file-space interface
#include <dfstable.h>                           // SLT utility functions
#include <dfsdgen.h>                            // Generic dialogs (UUID edit)

#include <dfsaxfs.h>                            // XFS display & analysis
#include <dfsuxfs.h>                            // XFS utility functions
#include <dfslxfs.h>                            // XFS SLT functions

char dfsXfsSectorTypes[] =
{
   ST_XFSSUP,                                   //     XFS  super-block
   0                                            //     string terminating ZERO
};


static DFSAXFS    xfs_anchor =                  // XFS specific global info
{
   0, 0,                                        // zero size
   8,                                           // default 2 sectors/block
   NULL                                         // superblock pointer
};

       DFSAXFS   *xfs = &xfs_anchor;

static char sg_xfssup[SG_XFSSUP] = {SV_XFSSUP};

static  char       *xfs_txt[] =
{
   "",
   "Active filesystem : XFS, specific commands are:",
   "",
   " SUPER           = Display the filesystem SUPERBLOCK sector",
   "",
   NULL
};

// Close XFS filesystem for analysis and free any resources
static ULONG dfsXfsClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
);

// Interpret and execute specific XFS command;
static ULONG dfsXfsCommand
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *none,                    // IN    dummy
   void               *data                     // INOUT dummy
);

// XFS filesystem, supply sector-type description string
static ULONG dfsXfsStype
(
   ULN64               di,                      // IN    sector type
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // OUT   type description
   void               *data                     // INOUT dummy
);

// XFS filesystem, display sector-contents based on type
static ULONG dfsXfsDispl
(
   ULN64               psn,                     // IN    base psn for sector
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // IN    sector contents
);

// Display XFS  super-block
static ULONG dfsXfsSuperBlock                   // RET   rc = 0 if type match
(
   BYTE               *sector,                  // IN    Drive specification
   BOOL                navigation               // IN    update nav values
);


// DFS XFS write-file to disk (SaveTo inode to file)
static ULONG dfsXfsFileSaveAs
(
   ULN64               dse,                     // IN    dirsec/entry LSN
   ULN64               d2,                      // IN    dummy
   char               *path,                    // IN    destination path
   void               *ren                      // IN    new name (like 8.3)
);

/*****************************************************************************/
// Initialize XFS filesystem analysis
/*****************************************************************************/
ULONG dfsXfsInit
(
   char               *fs                       // forced filesystem type
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   dfsa->FsCommand          = dfsXfsCommand;
   dfsa->FsClose            = dfsXfsClose;
   dfsa->FsIdentifySector   = dfsXfsIdent;
   dfsa->FsShowType         = dfsXfsStype;
   dfsa->FsDisplaySector    = dfsXfsDispl;
   dfsa->FsFileInformation  = NULL;
   dfsa->FsGetAllocSpace    = NULL;
   dfsa->FsWriteMetaSpace   = NULL;
   dfsa->FsFindPath         = NULL;
   dfsa->FsUpdateFileName   = NULL;
   dfsa->FsSetAltBrecLabel  = NULL;
   dfsa->FsMakeBrowseList   = NULL;
   dfsa->FsLsnAllocated     = NULL;
   dfsa->FsLsnSetAlloc      = NULL;
   dfsa->FsSltBuild         = dfsXfsSltBuild;
   dfsa->FsNpBuild          = NULL;
   dfsa->FsDisplayError     = dfsXfsDispError;
   dfsa->FsDisplayLsnInfo   = NULL;
   dfsa->FsDirIterator      = NULL;
   dfsa->FsDirFileSaveAs    = dfsXfsFileSaveAs;
   dfsa->FsCl2Lsn           = NULL;
   dfsa->FsLsn2Cl           = NULL;
   dfsa->FsCmdHelp          = xfs_txt;          // FS specific cmdhelp text

   dfsa->FsModeId           = DFS_FS_XFS;       // common for XFS

   dfsa->Fsi                = xfs;
   dfsa->FsSectorTypes      = dfsXfsSectorTypes;

   dfsa->FsEntry            = XFS_LSNSUP1;      // entry-point in filesystem

   if ((xfs->sup = TxAlloc(1, dfsGetSectorSize())) != NULL)
   {
      if ((rc = dfsRead( XFS_LSNSUP1, 1,     (BYTE *) xfs->sup)) == NO_ERROR)
      {
         if (dfsIdentifySector( XFS_LSNSUP1, 0, (BYTE *) xfs->sup)  == ST_XFSSUP)
         {
            xfs->SectorsPerBlock = TxBE32( xfs->sup->BlockSize) / dfsGetSectorSize();
            xfs->Block           = TxBE64( xfs->sup->DtotalBlocks);
            xfs->Sect            = xfs->Block * xfs->SectorsPerBlock;

            /*
            if ((rc = dfsXfsBitMapInit()) == NO_ERROR)
            {
               if (!TxaOptUnSet('a'))           // show allocation by default
               {
                  dfsXfsAllocMap( 0, 0, "@", NULL); // not really needed (no resize)
               }
            }
            */
         }
         else
         {
            TxPrint("No XFS Super-block found at 0x%2.2lx\n", XFS_LSNSUP1);
         }
      }
      else
      {
         TxPrint("Error reading XFS Super-block at 0x%2.2lx\n", XFS_LSNSUP1);
      }
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   dfstSetClusterSize( DFSTORE, (USHORT) xfs->SectorsPerBlock); // make sure!

   RETURN (rc);                                 // when needed
}                                               // end 'dfsXfsInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Close XFS filesystem for analysis and free any resources
/*****************************************************************************/
static ULONG dfsXfsClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   dfsSlTableReset();                           // stop SLT thread and reset

   TxFreeMem( xfs->sup);

   dfsCloseFileSystem();
   RETURN (rc);
}                                               // end 'dfsXfsClose'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Interpret and execute specific XFS command;
/*****************************************************************************/
static ULONG dfsXfsCommand
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *none,                    // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;
   ULN64               sn = 0;                  // sector number input
   TXLN                dc;                      // DFS command
   TXLN                s0;                      // big temporary string space
   TXLN                s1;                      // big temporary string space
   int                 cc;                      // command string count
   char               *c0, *c1, *c2;            // parsed command parts
   char               *pp;                      // parameter pointer

   ENTER();

   pp = TxaGetArgString( TXA_CUR, 1, 0, TXMAXLN, dc); // dc => cmd from arg 1
   cc = TxaArgCount( );                         // number of parameters
   c0 = TxaArgValue(0);
   c1 = TxaArgValue(1);
   c2 = TxaArgValue(2);

   sn = nav.this;                               // default at current sector

   if ((strcmp(c0, "/?") == 0) ||
       (strcmp(c0,  "?") == 0)  )
   {
      TxShowTxt( xfs_txt);
   }
   else if (strcasecmp(c0, "super"    ) == 0)
   {
      sprintf( dc, "0x0%llx", XFS_LSNSUP1);     // most likely location
      TxaReParseCommand( dc);
      rc = DFS_PENDING;                         // display the sector
   }
   else if (strcasecmp(c0, "label"    ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nSet volume label in superblock (16-character, Linux FS-name)\n");
         TxPrint("\nUsage: %s  [label]\n\n"
                 "   label  : new volume label, maximum 16 characters\n"
                 "   -!-    = do not prompt for new value, just set it\n", c0);
      }
      else
      {
         if (cc > 1)
         {
            strcpy( s1, c1);                    // use specified parameter value
         }
         else if (SINF->p)                      // it is a partition
         {
            strcpy( s1, SINF->p->plabel);       // use label determined by readdiskinfo
         }
         else
         {
            strcpy( s1, "");                    // start with empty label
         }
         s1[ XFS_LEN_LBL] = 0;                  // truncate to allowed length

         if (dfsa->batch)                       // s1 has new label value
         {
            memcpy( xfs->sup->VolumeName, s1, XFS_LEN_LBL);
            rc = dfsXfsWriteSuperBlocks();
         }
         else
         {
            #if defined (USEWINDOWING)          // Allow interactive update
            if (!TxaOptUnSet('!') && txwIsWindow( TXHWND_DESKTOP))
            {
               TXLN      prompt;

               sprintf( prompt, "%s\n\nVolume label in superblock: '%s'\n\n"
                                "Specify new %d-character volume label\n",
                                 dfstStoreDesc1( DFSTORE) + 10, s1, XFS_LEN_LBL);

               if (txwPromptBox( TXHWND_DESKTOP, TXHWND_DESKTOP, NULL,
                      prompt, " Set volume label in superblock ",
                      5144, TXPB_MOVEABLE | TXPB_HCENTER | TXPB_VCENTER,
                      XFS_LEN_LBL, s1) != TXDID_CANCEL)
               {
                  TxStrip( s1, s1, ' ', ' ');   // strip leading/trailing spaces
                  memcpy( xfs->sup->VolumeName, s1, XFS_LEN_LBL);
                  rc = dfsXfsWriteSuperBlocks();
               }
            }
            else
            #endif
            {
               if (TxConfirm( 5144, "Write Volume LABEL '%s' to superblock ? [Y/N] : ", s1))
               {
                  memcpy( xfs->sup->VolumeName, s1, XFS_LEN_LBL);
                  rc = dfsXfsWriteSuperBlocks();
               }
            }
         }
      }
   }
   else if (strcasecmp(c0, "uuid"    ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nUsage: %s  [UUID] | [-u:UUID]\n"
                 "\n      UUID : UUID string value, 5 groups separated by hyphens, no brackets"
                 "\n   -u:UUID : 36 character UUID value: 12345678-abcd-0123-4567-0123456789ab\n", c0);
      }
      else
      {
         if (TxaOptMutEx((cc > 1), "u", "if a UUID parameter is specified!", NULL))
         {
            if (cc > 1)
            {
               strcpy( s1, c1);                 // use specified parameter value
            }
            else
            {
               strcpy( s1, TxaOptStr( 'u', NULL, ""));
            }
            sprintf( s0, "{%s}", s1);           // Make fully-formed UUID string

            if (((strlen( s1) == 0) && (!dfsa->batch)) || dfsUidStringIsValid( s0))
            {
               if (dfsa->batch)                 // s0 has valid new UUID
               {
                  dfsUidStringToBinary( s0, xfs->sup->Uuid, TRUE);
                  rc = dfsXfsWriteSuperBlocks();
               }
               else
               {
                  #if defined (USEWINDOWING)    // Allow interactive update
                  if (txwIsWindow( TXHWND_DESKTOP))
                  {
                     if (dfsGuidUuidEditor( "Update UUID value in XFS Superblock",
                                            "Write Back", SINF->partid,
                                             5143, s0, TRUE, xfs->sup->Uuid) == NO_ERROR)
                     {
                        rc = dfsXfsWriteSuperBlocks();
                     }
                  }
                  else
                  #endif
                  {
                     if (TxConfirm( 5143, "Write UUID %s on PID %hu ? [Y/N] : ", s0, SINF->partid))
                     {
                        dfsUidStringToBinary( s0, xfs->sup->Uuid, TRUE);
                        rc = dfsXfsWriteSuperBlocks();
                     }
                  }
               }
            }
            else
            {
               TxPrint( "\nInvalid UUID value: '%s'\n", s1);
            }
         }
      }
      TxPrint( "\nVolume UUID now   : %s\n", dfsFsUuidValueString( xfs->sup->Uuid));
   }
   else
   {
      rc = DFS_CMD_UNKNOWN;                     // cmd not recognized
   }
   RETURN (rc);
}                                               // end 'dfsXfsCommand'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// XFS filesystem, identify specified sector as a valid superblock
/*****************************************************************************/
BOOL dfsXfsIsSuperBlock                         // RET   sector is a valid sb
(
   BYTE               *sec                      // IN    sector contents
)
{
   return (!memcmp(((S_XFS_SUPER*)sec)->Signature,sg_xfssup,SG_XFSSUP));
}                                               // end 'dfsXfsIsSuperBlock'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// XFS filesystem, display sector constents as a superblock
/*****************************************************************************/
void dfsXfsDisplaySuperBlock
(
   BYTE               *sec                      // IN    sector contents
)
{
   dfsXfsSuperBlock( sec, FALSE);
}                                               // end 'dfsXfsDisplaySuperBlock'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// XFS filesystem, identify specified sector
/*****************************************************************************/
ULONG dfsXfsIdent
(
   ULN64               lsn,                     // IN    LSN for sector
   ULN64               d2,                      // IN    dummy
   char               *st,                      // OUT   sector type
   void               *sec                      // IN    sector contents
)
{
   ULONG               dr = NO_ERROR;
   BYTE                rc = ST_UDATA;

   ENTER();

   if (!memcmp(((S_XFS_SUPER*)sec)->Signature,sg_xfssup,SG_XFSSUP)) rc=ST_XFSSUP;
   switch (rc)
   {
      case ST_UDATA:
         dr = DFS_PENDING;
         break;

      default:
         break;
   }
   *st = rc;
   RETURN (dr);
}                                               // end 'dfsXfsIdent'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// XFS filesystem, supply sector-type description string
/*****************************************************************************/
static ULONG dfsXfsStype
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *st,                      // IN    sector type
   void               *data                     // OUT   type description
)
{
   ULONG               dr  = NO_ERROR;
   char               *buf = (char *) data;
   BYTE                tp  = st[0];

   switch (tp)                                  // searchable types
   {
      case ST_XFSSUP: sprintf(buf,"XFS  superblock  "); break;
      default:
         switch (tp | ST__INFO)                 // non-searchable ones
         {
            default:       dr = DFS_PENDING;
               break;
         }
         break;
   }
   return (dr);
}                                               // end 'dfsXfsStype'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// XFS filesystem, display sector-contents based on type
/*****************************************************************************/
static ULONG dfsXfsDispl
(
   ULN64               psn,                     // IN    base psn for sector
   ULN64               d2,                      // IN    dummy
   char               *type,                    // IN    type of sector
   void               *data                     // IN    sector contents
)
{
   ULONG               dr = NO_ERROR;
   BYTE                st = (BYTE) *type;

   ENTER();

   switch (st)
   {
      case ST_XFSSUP: (void) dfsXfsSuperBlock(rbuf, TRUE);       break;

      default:
         dr = DFS_PENDING;
         break;
   }
   RETURN (dr);
}                                               // end 'dfsXfsDispl'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display XFS  super-block
/*****************************************************************************/
static ULONG dfsXfsSuperBlock                   // RET   rc = 0 if type match
(
   BYTE               *sector,                  // IN    Drive specification
   BOOL                navigation               // IN    update nav values
)
{
   ULONG               rc    = 0;               // rc, sector match
   S_XFS_SUPER        *sd    = (S_XFS_SUPER *) sector;
   TXLN                text;

   ENTER();

   TxPrint("FS  version flags : 0x%4.4hx\n", TxBE16( sd->FsVersionFlags));
   dfsSz64("Block Group  size : ", (ULN64)   TxBE32( sd->BlocksPerGroup) * xfs->SectorsPerBlock, "");
   dfsX10( "  blocks : ",          (ULN64)   TxBE32( sd->BlocksPerGroup),  CNN, " ");
   TxPrint("  groups : %lu\n",               TxBE32( sd->NumberOfGroups));
   dfsSz64("FS     total size : ",                   xfs->Sect,  "");
   dfsX10( "  blocks : ",                            xfs->Block, CNN, "\n");
   dfsSz64("Free data blocks  : ",           TxBE64( sd->DataBlocksFree) * xfs->SectorsPerBlock, "\n");
   dfsSiz8("Sectors per Block : ",                   xfs->SectorsPerBlock, "  ");
   TxPrint(" using : %hu byte sectors\n",    TxBE16( sd->SectorSize));
   dfsX10( "Inodes  allocated : ",           TxBE64( sd->InodesAllocated), CNN, " ");
   dfsX10( "free: ",                         TxBE64( sd->InodesFree     ), CNN, " " );
   dfsX10( "used : ",                       (TxBE64( sd->InodesAllocated) -
                                             TxBE64( sd->InodesFree)    ), CNN, " =");
   TxPrint( " %llu\n",                      (TxBE64( sd->InodesAllocated) -
                                             TxBE64( sd->InodesFree)    ));
   TxPrint("Inode size, bytes : %hu, ",      TxBE16( sd->InodeSize));
   TxPrint("with %hu inodes per block\n",    TxBE16( sd->InodesPerBlock));

   if (sd->VolumeName[0] == 0)
   {
      strcpy( text, "-none-");
   }
   else
   {
      TxCopy(  text, (char *) sd->VolumeName, XFS_LEN_LBL + 1);
   }
   TxPrint("Volume label      : %s%s%s\n", CBG, text, CNN);
   TxPrint("Volume UUID       : %s\n", dfsFsUuidValueString( sd->Uuid));

   /*
   if (navigation)                              // set auto display on ENTER to the root Inode
   {
      nav.down         = dfsXfsInode2LsnIndex( sd->RootInode, &nav.down_sninfo);
      nav.down_sninfo |= DFSSNINFO;
   }
   */

   RETURN (rc);
}                                               // end 'dfsXfsSuperBlock'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// DFS XFS write-file to disk (SaveTo to file)
/*****************************************************************************/
static ULONG dfsXfsFileSaveAs
(
   ULN64               dse,                     // IN    dirsec/entry LSN
   ULN64               d2,                      // IN    dummy
   char               *path,                    // IN    destination path
   void               *ren                      // IN    new name, or empty
)
{
   ULONG               rc  = NO_ERROR;

   ENTER();
   TRARGS(("Dirsec/entry LSN : 0x%llX, to path: '%s'\n", dse, path));

   TxPrint( "\ndfsXfsFileSaveAs Not implemented yet\n");

   RETURN(rc);
}                                               // end 'dfsXfsFileSaveAs'
/*---------------------------------------------------------------------------*/
