//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// Auxiliary dump & display Analysis functions
//
// Author: J. van Wijk
//
// JvW  12-03-2000 Initial version
//
// Author: J. van Wijk
//
// 20170818 JvW Removed VRAID support (freeing up sector type letters v w W)
// 20171018 JvW Added not-implemented 'standard' commands like filefind, with a message for usability

#include <txlib.h>                              // TX library interface

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfsauxil.h>                           // AUXILIARY display & analysis


char dfsAuxSectorTypes[] =
{
   0                                            //     string terminating ZERO
};


static  char       *aux_txt[] =
{
   "Active filesystem : AUXILIARY,  specific commands are:",
   "",
   "none (yet)",
   "",
   NULL
};

// Close AUXILIARY filesystem for analysis and free any resources
static ULONG dfsAuxClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
);

// Interpret and execute specific AUXILIARY command;
static ULONG dfsAuxCommand
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *none,                    // IN    dummy
   void               *data                     // INOUT dummy
);

// AUXILIARY filesystem, identify specified sector
ULONG dfsAuxIdent
(
   ULN64               lsn,                     // IN    LSN for sector
   ULN64               d2,                      // IN    dummy
   char               *st,                      // OUT   sector type
   void               *sec                      // IN    sector contents
);

// AUXILIARY filesystem, supply sector-type description string
static ULONG dfsAuxStype
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *st,                      // IN    sector type
   void               *data                     // OUT   type description
);

// AUXILIARY filesystem, display sector-contents based on type
static ULONG dfsAuxDispl
(
   ULN64               psn,                     // IN    base psn for sector
   ULN64               d2,                      // IN    dummy
   char               *type,                    // IN    type of sector
   void               *data                     // IN    sector contents
);


// Truncate the partition size, keeping all data untouched
static ULONG dfsAuxTruncate
(
   ULN64               newsize,                 // IN    new size in sectors
   ULN64               d2,                      // IN    dummy
   char               *string,                  // IN    dummy
   void               *data                     // IN    partition info
);


/*****************************************************************************/
// Initialize AUXILIARY filesystem analysis
/*****************************************************************************/
ULONG dfsAuxInit
(
   char               *fs                       // forced filesystem type
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   dfsa->FsCommand          = dfsAuxCommand;
   dfsa->FsClose            = dfsAuxClose;
   dfsa->FsIdentifySector   = dfsAuxIdent;
   dfsa->FsShowType         = dfsAuxStype;
   dfsa->FsDisplaySector    = dfsAuxDispl;
   dfsa->FsTruncateSize     = dfsAuxTruncate;
   dfsa->FsFileInformation  = NULL;
   dfsa->FsGetAllocSpace    = NULL;
   dfsa->FsWriteMetaSpace   = NULL;
   dfsa->FsFindPath         = NULL;
   dfsa->FsUpdateFileName   = NULL;
   dfsa->FsSetAltBrecLabel  = NULL;
   dfsa->FsMakeBrowseList   = NULL;
   dfsa->FsLsnAllocated     = NULL;
   dfsa->FsLsnSetAlloc      = NULL;
   dfsa->FsSltBuild         = NULL;
   dfsa->FsNpBuild          = NULL;
   dfsa->FsDisplayError     = NULL;
   dfsa->FsDisplayLsnInfo   = NULL;
   dfsa->FsDirIterator      = NULL;
   dfsa->FsDirFileSaveAs    = NULL;
   dfsa->FsCl2Lsn           = NULL;
   dfsa->FsLsn2Cl           = NULL;
   dfsa->Fsi                = NULL;
   dfsa->FsSltRecSize       = 0;

   dfsa->FsModeId           = DFS_FS_AUX;
   dfsa->FsSectorTypes      = dfsAuxSectorTypes;

   dfsa->FsEntry = 0;
   nav.down      = LSN_BOOTR;

   TxPrint("\nUsing DFSee generic/auxiliary analysis mode for '%s'\n\n", fs);

   if ((!TxaOptSet( DFS_O_FILESYS)) && (!TxaOptSet( 'q'))) // explicit AUX and not quiet
   {
      TxNamedMessage( ((!dfsa->batch) && (!TxaOptUnSet('P'))), 5005,
         " WARNING: Unrecognized filesystem ",
         "Filesystem unrecognized by DFSee. Switching to AUXiliary mode.\n\n"
         "Object may be unformatted, encrypted, damaged or contain a filesystem "
         "that is not known by DFSee (yet). Use <F1> for additional info.\n\n"
         "                 This is not always an error!");
   }
   RETURN (rc);
}                                               // end 'dfsAuxInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Close AUXILIARY filesystem for analysis and free any resources
/*****************************************************************************/
static ULONG dfsAuxClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   dfsCloseFileSystem();
   RETURN (rc);
}                                               // end 'dfsAuxClose'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Interpret and execute specific AUXILIARY command;
/*****************************************************************************/
static ULONG dfsAuxCommand
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *none,                    // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;
   char               *c0;                      // parsed command parts

   ENTER();

   c0 = TxaArgValue(0);

   if ((strcmp(c0, "/?") == 0) ||
       (strcmp(c0,  "?") == 0)  )
   {
      TxShowTxt( aux_txt);
   }
   else if ((strcasecmp(c0, "delfind"  ) == 0) ||
            (strcasecmp(c0, "filefind" ) == 0) ||
            (strcasecmp(c0, "findroot" ) == 0) ||
            (strcasecmp(c0, "fixboot"  ) == 0) ||
            (strcasecmp(c0, "bootini"  ) == 0) ||
            (strcasecmp(c0, "bsclear"  ) == 0) ||
            (strcasecmp(c0, "super"    ) == 0) ||
            (strcasecmp(c0, "dirty"    ) == 0) ||
            (strcasecmp(c0, "ino"      ) == 0)  )
   {
      TxPrint( "\nThe '%s' command is NOT implemented for the AUXiliary modus\n\n"
               "you can switch to another mode using the 'fs xxx' command or\n"
               "the menu item: 'Edit -> Force Operation mode xxx'\n\n"
               "In AUX mode, there is NO notion of files, directories, superblocks\n"
               "or anything else that would be associated with a filesystem!\n\n", c0);
   }
   else
   {
      rc = DFS_CMD_UNKNOWN;                     // cmd not recognized
   }
   RETURN (rc);
}                                               // end 'dfsAuxCommand'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// AUXILIARY filesystem, identify specified sector
/*****************************************************************************/
ULONG dfsAuxIdent
(
   ULN64               lsn,                     // IN    LSN for sector
   ULN64               d2,                      // IN    dummy
   char               *st,                      // OUT   sector type
   void               *sec                      // IN    sector contents
)
{
   ULONG               rc = NO_ERROR;
   BYTE                tp = ST_UDATA;

   ENTER();

   TRDUMP(70, "AUX ident, start of sector:\n", sec, 64, 0);

   switch (tp)
   {
      case ST_UDATA:
         rc = DFS_PENDING;
         break;

      default:
         break;
   }
   *st = tp;
   RETURN (rc);
}                                               // end 'dfsAuxIdent'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// AUXILIARY filesystem, supply sector-type description string
/*****************************************************************************/
static ULONG dfsAuxStype
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *st,                      // IN    sector type
   void               *data                     // OUT   type description
)
{
   ULONG               rc  = NO_ERROR;

   switch (*st)                                 // searchable types
   {
      default:         rc = DFS_PENDING;
   }
   return (rc);
}                                               // end 'dfsAuxStype'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// AUXILIARY filesystem, display sector-contents based on type
/*****************************************************************************/
static ULONG dfsAuxDispl
(
   ULN64               psn,                     // IN    base psn for sector
   ULN64               d2,                      // IN    dummy
   char               *type,                    // IN    type of sector
   void               *data                     // IN    sector contents
)
{
   ULONG               rc = NO_ERROR;
   BYTE                st = (BYTE) *type;

   ENTER();

   switch (st)
   {
      default:
         rc = DFS_PENDING;
         break;
   }
   RETURN (rc);
}                                               // end 'dfsAuxDispl'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Truncate the filesystem size, keeping all data intact; (2 sectors in br)
/*****************************************************************************/
static ULONG dfsAuxTruncate
(
   ULN64               newsize,                 // IN    new size in sectors
   ULN64               d2,                      // IN    dummy
   char               *string,                  // IN    dummy
   void               *data                     // IN    partition info
)
{
   ULONG               rc  = NO_ERROR;
   DFSPARTINFO        *p   = (DFSPARTINFO *) data;
   TXTM                text;

   ENTER();
   TRARGS(("New size: 0x%llx\n", newsize));

   strcpy( text, "");
   dfstrSizeBps( text, "to a new size of: ", newsize, (USHORT) p->bpsector, "");

   if ((dfsa->batch) ||
       (TxConfirm( 5140, "Resizing an unrecognized filesystem could cause severe "
                         "damage to the DATA! Resize only if the partition is known "
                         "to be empty, unformatted or the data is obsolete\n\n"
                         "Resize this filesystem %s ? [Y/N] : ", text)))
   {
      rc = NO_ERROR;
   }
   else
   {
      rc = DFS_NO_CHANGE;
   }
   RETURN (rc);
}                                               // end 'dfsAuxTruncate'
/*---------------------------------------------------------------------------*/

