//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// Linux SWAP dump & display Analysis functions
//
// Author: J. van Wijk
//
// JvW  06-08-2007 Initial version, derived from XFS
//
// Author: J. van Wijk

#include <txlib.h>                              // TX library interface

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions

#include <dfsaswap.h>                           // SWAP display & analysis

#define SWAP_STARTSEC  0L                       // SWAP start sector
#define SWAP_INFOSEC   2L                       // SWAP info  sector
#define SWAP_SSECTS    8L                       // one block, 8 sectors

char dfsSwapSectorTypes[] =
{
   ST_SWAPT2,                                   //     SWAP super-block
   0
};


static  char       *swap_txt[] =
{
   "",
   "Active filesystem : Linux SWAP, specific commands are:",
   "",
   " SUPER           = Display the filesystem SUPERBLOCK (first block)",
   "",
   NULL
};

// Close SWAP filesystem for analysis and free any resources
static ULONG dfsSwapClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
);

// Close SWAP filesystem for Area analysis and free any resources
static ULONG dfsSwapAreaClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
);

// Interpret and execute specific SWAP command;
static ULONG dfsSwapCommand
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *none,                    // IN    dummy
   void               *data                     // INOUT dummy
);

// SWAP filesystem, identify specified sector
static ULONG dfsSwapIdent
(
   ULN64               lsn,                     // IN    LSN for sector
   ULN64               d2,                      // IN    dummy
   char               *st,                      // OUT   sector type
   void               *sec                      // IN    sector contents
);

// SWAP filesystem, supply sector-type description string
static ULONG dfsSwapStype
(
   ULN64               di,                      // IN    sector type
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // OUT   type description
   void               *data                     // INOUT dummy
);

// SWAP filesystem, display sector-contents based on type
static ULONG dfsSwapDispl
(
   ULN64               di,                      // IN    sector type
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // IN    sector contents
);

// Display SWAP  super-block
static ULONG dfsSwapSuperBlock                  // RET   rc = 0 if type match
(
   BYTE              *sector                    // IN    Sector contents
);

// Determine allocation-bit for specified LSN, FREE beyond first block!
static ULONG dfsSwapAllocated                   // RET   LSN is allocated
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
);


/*****************************************************************************/
// Initialize SWAP filesystem analysis
/*****************************************************************************/
ULONG dfsSwapInit
(
   char               *fs                       // forced filesystem type
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   dfsa->FsCommand          = dfsSwapCommand;
   dfsa->FsClose            = dfsSwapClose;
   dfsa->FsIdentifySector   = dfsSwapIdent;
   dfsa->FsShowType         = dfsSwapStype;
   dfsa->FsDisplaySector    = dfsSwapDispl;
   dfsa->FsFileInformation  = NULL;
   dfsa->FsGetAllocSpace    = NULL;
   dfsa->FsWriteMetaSpace   = NULL;
   dfsa->FsFindPath         = NULL;
   dfsa->FsLsnAllocated     = dfsSwapAllocated;
   dfsa->FsLsnSetAlloc      = NULL;
   dfsa->FsSltBuild         = NULL;
   dfsa->FsNpBuild          = NULL;
   dfsa->FsDisplayError     = NULL;
   dfsa->FsDisplayLsnInfo   = NULL;
   dfsa->FsDirIterator      = NULL;
   dfsa->FsDirFileSaveAs    = NULL;
   dfsa->FsCl2Lsn           = NULL;             // dummies, one to one
   dfsa->FsLsn2Cl           = NULL;
   dfsa->FsCmdHelp          = swap_txt;         // FS specific cmdhelp text

   dfsa->FsModeId           = DFS_FS_SWAP;      // common for SWAP

   dfsa->Fsi                = NULL;
   dfsa->FsSectorTypes      = dfsSwapSectorTypes;

   dfsa->FsEntry            = SWAP_INFOSEC;     // entry-point in filesystem

   RETURN (rc);                                 // when needed
}                                               // end 'dfsSwapInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Close SWAP filesystem for analysis and free any resources
/*****************************************************************************/
static ULONG dfsSwapClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   dfsCloseFileSystem();
   RETURN (rc);
}                                               // end 'dfsSwapClose'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Initialize SWAP filesystem analysis for Area (FS info in FDISK mode)
/*****************************************************************************/
ULONG dfsSwapAreaInit
(
   void
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   dfsa->FsAreaClose        = dfsSwapAreaClose;
   dfsa->FsAreaLsnAllocated = dfsSwapAllocated;

   RETURN (rc);
}                                               // end 'dfsSwapAreaInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Close SWAP filesystem for Area analysis and free any resources
/*****************************************************************************/
static ULONG dfsSwapAreaClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   dfsa->FsAreaClose        = NULL;
   dfsa->FsAreaLsnAllocated = NULL;

   RETURN (rc);
}                                               // end 'dfsSwapAreaClose'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Interpret and execute specific SWAP command;
/*****************************************************************************/
static ULONG dfsSwapCommand
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *none,                    // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;
   ULN64               sn = 0;                  // sector number input
   TXLN                dc;                      // DFS command
   int                 cc;                      // command string count
   char               *c0;                      // parsed command parts
   char               *pp;                      // parameter pointer

   ENTER();

   pp = TxaGetArgString( TXA_CUR, 1, 0, TXMAXLN, dc); // dc => cmd from arg 1
   cc = TxaArgCount( );                         // number of parameters
   c0 = TxaArgValue(0);

   sn = nav.this;                               // default at current sector

   if ((strcmp(c0, "/?") == 0) ||
       (strcmp(c0,  "?") == 0)  )
   {
      TxShowTxt( swap_txt);
   }
   else if (strcasecmp(c0, "super"    ) == 0)
   {
      sprintf( dc, "0x%lx", SWAP_INFOSEC);      // guaranteed to show as super
      TxaReParseCommand( dc);                   // 0 may be shown as GRUB/LILO
      rc = DFS_PENDING;                         // display the sector
   }
   else
   {
      rc = DFS_CMD_UNKNOWN;                     // cmd not recognized
   }
   RETURN (rc);
}                                               // end 'dfsSwapCommand'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// SWAP filesystem, identify specified sector
/*****************************************************************************/
static ULONG dfsSwapIdent
(
   ULN64               lsn,                     // IN    LSN for sector
   ULN64               d2,                      // IN    dummy
   char               *st,                      // OUT   sector type
   void               *sec                      // IN    sector contents
)
{
   ULONG               dr = NO_ERROR;
   BYTE                rc = ST_UDATA;

   ENTER();

   if (lsn < SWAP_SSECTS)
   {
      if ((lsn != SWAP_STARTSEC) || (((S_BOOTR*)sec)->Signature != SV_BOOTR))
      {
         rc = ST_SWAPT2;                        // SWAP with (GRUB/LILO)
      }                                         // will be seen as bootsec!
   }
   switch (rc)
   {
      case ST_UDATA:
         dr = DFS_PENDING;
         break;

      default:
         break;
   }
   *st = rc;
   RETURN (dr);
}                                               // end 'dfsSwapIdent'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// SWAP filesystem, supply sector-type description string
/*****************************************************************************/
static ULONG dfsSwapStype
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *st,                      // IN    sector type
   void               *data                     // OUT   type description
)
{
   ULONG               dr  = NO_ERROR;
   char               *buf = (char *) data;
   BYTE                tp  = st[0];

   switch (tp)                                  // searchable types
   {
      case ST_SWAPT2: sprintf(buf,"SWAP superblock  "); break;
      default:
         switch (tp | ST__INFO)                 // non-searchable ones
         {
            default:       dr = DFS_PENDING;
               break;
         }
         break;
   }
   return (dr);
}                                               // end 'dfsSwapStype'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// SWAP filesystem, display sector-contents based on type
/*****************************************************************************/
static ULONG dfsSwapDispl
(
   ULN64               psn,                     // IN    base psn for sector
   ULN64               d2,                      // IN    dummy
   char               *type,                    // IN    type of sector
   void               *data                     // IN    sector contents
)
{
   ULONG               dr = NO_ERROR;
   BYTE                st = (BYTE) *type;

   ENTER();

   switch (st)
   {
      case ST_SWAPT2:                           // read total SWAP super
         dfsRead( SWAP_STARTSEC, SWAP_SSECTS, rbuf);
         (void) dfsSwapSuperBlock(rbuf);
         break;

      default:
         dr = DFS_PENDING;
         break;
   }
   RETURN (dr);
}                                               // end 'dfsSwapDispl'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display SWAP  super-block
/*****************************************************************************/
static ULONG dfsSwapSuperBlock                  // RET   rc = 0 if type match
(
   BYTE               *sector                   // IN    sector buffer
)
{
   ULONG               rc   = 0;                // rc, sector match
   DFSWAPBLOCK1       *sd   = (DFSWAPBLOCK1 *) sector;
   TXTS                sw;

   ENTER();

   if (!memcmp( sd->swSignature, "SWAP", 4))
   {
      TxCopy( sw, (char *) sd->swSignature, 11);
   }
   else
   {
      strcpy( sw, "-Unknown-");
   }
   TxPrint("\nLinux SWAP type   : %s = ", sw);

   if (sd->swSignature[9] == '2')
   {
      TxPrint( "Linux SWAP Type 2, Version ");
      switch (sd->swVersion)
      {
         case 0:  TxPrint( "0");         break;
         case 1:  TxPrint( "1");         break;
         default: TxPrint( "-Invalid-"); break;
      }
   }
   else
   {
      TxPrint( "\nLinux SWAP Type 1  (128 MiB limit)");
   }
   dfsSiz8("\nNet  SWAP sectors : ", sd->swBlocks * 8, "\n");
   RETURN (rc);
}                                               // end 'dfsSwapSuperBlock'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Determine allocation-bit for specified LSN, FREE beyond first block!
/*****************************************************************************/
static ULONG dfsSwapAllocated                   // RET   LSN is allocated
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               asn = dfstAreaD2Part( DFSTORE, lsn); // Area aware sn

   if (asn < SWAP_SSECTS)                       // within valid block area
   {
      return( TRUE);
   }
   else
   {
      return( FALSE);
   }
}                                               // end 'dfsSwapAllocated'
/*---------------------------------------------------------------------------*/

