//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// REISER dump & display Analysis functions
//
// Author: J. van Wijk
//
// JvW  31-03-2004 Initial version, derived from EXT
//
// Author: J. van Wijk

#include <txlib.h>                              // TX library interface

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfsspace.h>                           // DFS file-space interface
#include <dfstable.h>                           // SLT utility functions
#include <dfsdgen.h>                            // Generic dialogs (UUID edit)

#include <dfsarsr.h>                            // RSR display & analysis
#include <dfsursr.h>                            // RSR utility functions
#include <dfslrsr.h>                            // RSR SLT functions

char dfsRsrSectorTypes[] =
{
   ST_RSRSUP,                                   //     RSR  super-block
   0                                            //     string terminating ZERO
};


static DFSARSR    rsr_anchor =                  // RSR specific global info
{
   0,                                           // Number of sectors
   0,                                           // Number of blocks
   8,                                           // sectors per block
   0,                                           // Group size in blocks
   NULL,                                        // ptr to superblock
   {FALSE, 0, NULL}                             // Block bitmap cache
};

       DFSARSR   *rsr = &rsr_anchor;

static char sg_rsrsup[SG_RSRSUP] = {SV_RSRSUP};

static  char       *rsr_txt[] =
{
   "",
   "Active filesystem : REISER, specific commands are:",
   "",
   " SUPER           = Display the filesystem SUPERBLOCK sector",
   " UUID  [-u] [id] = Display or set a UUID string, dialog if no id specified",
   "",
   NULL
};

// Close RSR filesystem for analysis and free any resources
static ULONG dfsRsrClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
);

// Close Reiser filesystem for Area analysis and free any resources
static ULONG dfsRsrAreaClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
);

// Interpret and execute specific RSR command;
static ULONG dfsRsrCommand
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *none,                    // IN    dummy
   void               *data                     // INOUT dummy
);

// RSR filesystem, supply sector-type description string
static ULONG dfsRsrStype
(
   ULN64               di,                      // IN    sector type
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // OUT   type description
   void               *data                     // INOUT dummy
);

// RSR filesystem, display sector-contents based on type
static ULONG dfsRsrDispl
(
   ULN64               psn,                     // IN    base psn for sector
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // IN    sector contents
);

// Display RSR  super-block
static ULONG dfsRsrSuperBlock                   // RET   rc = 0 if type match
(
   BYTE               *sector,                  // IN    Drive specification
   BOOL                navigation               // IN    update nav values
);

// Display value of superblock status value
static void dfsRsrDisplayStatus
(
   char               *lead,                    // IN    leading text
   ULONG               st                       // IN    RSR status value
);

// Make filesystem usage map dump on TxPrint output
static ULONG dfsRsrAllocMap
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *options,                 // IN    display options
   void               *data                     // OUT   dummy
);

// DFS RSR write-file to disk (SaveTo inode to file)
static ULONG dfsRsrFileSaveAs
(
   ULN64               dse,                     // IN    dirsec/entry LSN
   ULN64               d2,                      // IN    dummy
   char               *path,                    // IN    destination path
   void               *ren                      // IN    new name (like 8.3)
);

/*****************************************************************************/
// Initialize RSR filesystem analysis
/*****************************************************************************/
ULONG dfsRsrInit
(
   char               *fs                       // forced filesystem type
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   dfsa->FsCommand          = dfsRsrCommand;
   dfsa->FsClose            = dfsRsrClose;
   dfsa->FsIdentifySector   = dfsRsrIdent;
   dfsa->FsShowType         = dfsRsrStype;
   dfsa->FsDisplaySector    = dfsRsrDispl;
   dfsa->FsFileInformation  = NULL;
   dfsa->FsGetAllocSpace    = NULL;
   dfsa->FsWriteMetaSpace   = NULL;
   dfsa->FsFindPath         = NULL;
   dfsa->FsLsnAllocated     = dfsRsrAllocated;
   dfsa->FsLsnSetAlloc      = NULL;
   dfsa->FsSltBuild         = dfsRsrSltBuild;
   dfsa->FsNpBuild          = NULL;
   dfsa->FsDisplayError     = dfsRsrDispError;
   dfsa->FsDisplayLsnInfo   = NULL;
   dfsa->FsDirIterator      = NULL;
   dfsa->FsDirFileSaveAs    = dfsRsrFileSaveAs;
   dfsa->FsAllocDisplay     = dfsRsrAllocMap;
   dfsa->FsCl2Lsn           = dfsRsrCl2Lsn;
   dfsa->FsLsn2Cl           = dfsRsrLsn2Cl;
   dfsa->FsCmdHelp          = rsr_txt;          // FS specific cmdhelp text

   dfsa->FsModeId           = DFS_FS_RSR;       // common for REISER

   dfsa->Fsi                = rsr;
   dfsa->FsSectorTypes      = dfsRsrSectorTypes;

   dfsa->FsEntry            = RSR_LSNSUP2;      // entry-point in filesystem

   if ((rsr->sup = TxAlloc(1, dfsGetSectorSize())) != NULL)
   {
      if ((rc = dfsRead( RSR_LSNSUP2, 1,     (BYTE *) rsr->sup)) == NO_ERROR)
      {
         if (dfsIdentifySector( RSR_LSNSUP2, 0, (BYTE *) rsr->sup)  == ST_RSRSUP)
         {
            if (rsr->sup->State == RSR_FS_CLEAN)
            {
               dfsa->FsDirtyStatus = DFSTAT_CLEAN; // clean AND unmounted
            }
            else
            {
               dfsa->FsDirtyStatus = DFSTAT_DIRTY;
            }
            rsr->BlocksPerGroup  = rsr->sup->BlockSize * 8;
            rsr->SectorsPerBlock = rsr->sup->BlockSize / dfsGetSectorSize();
            rsr->Block           = rsr->sup->Blocks;
            rsr->Sect            = rsr->Block * rsr->SectorsPerBlock;

            if ((rc = dfsRsrBitMapInit()) == NO_ERROR)
            {
               if (TxaOption('a'))              // show NO allocation by default
               {
                  dfsRsrAllocMap( 0, 0, "@", NULL); // not really needed (no resize)
               }
            }
         }
         else
         {
            TxPrint("No ReiserFS Super-block found at 0x%2.2lx\n", RSR_LSNSUP2);
         }
      }
      else
      {
         TxPrint("Error reading ReiserFS Super-block at 0x%2.2lx\n", RSR_LSNSUP2);
      }
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   dfstSetClusterSize( DFSTORE, (USHORT) rsr->SectorsPerBlock); // make sure!

   RETURN (rc);                                 // when needed
}                                               // end 'dfsRsrInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Close RSR filesystem for analysis and free any resources
/*****************************************************************************/
static ULONG dfsRsrClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   dfsSlTableReset();                           // stop SLT thread and reset

   rc = dfsRsrBitmapFlush( TRUE);               // flush, and terminate cache

   TxFreeMem( rsr->sup);

   dfsCloseFileSystem();
   RETURN (rc);
}                                               // end 'dfsRsrClose'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Initialize Reiser filesystem analysis for Area (FS info in FDISK mode)
/*****************************************************************************/
ULONG dfsRsrAreaInit
(
   void
)
{
   ULONG               rc = NO_ERROR;
   char                st;

   ENTER();

   dfsa->FsAreaClose        = dfsRsrAreaClose;
   dfsa->FsAreaLsnAllocated = dfsRsrAllocated;

   if ((rsr->sup = TxAlloc(1, dfsGetSectorSize())) != NULL)
   {
      if ((rc = dfsRead( dfstAreaP2Disk( DFSTORE, RSR_LSNSUP2), 1, (BYTE *) rsr->sup)) == NO_ERROR)
      {
         dfsRsrIdent( RSR_LSNSUP2, 0, &st, rsr->sup); // dfsIdentify not available!
         if (st == ST_RSRSUP)
         {
            rsr->BlocksPerGroup  = rsr->sup->BlockSize * 8;
            rsr->SectorsPerBlock = rsr->sup->BlockSize / dfsGetSectorSize();
            rsr->Block           = rsr->sup->Blocks;
            rsr->Sect            = rsr->Block * rsr->SectorsPerBlock;

            rc = dfsRsrBitMapInit();
         }
      }
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsRsrAreaInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Close Reiser filesystem for Area analysis and free any resources
/*****************************************************************************/
static ULONG dfsRsrAreaClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   dfsa->FsAreaClose        = NULL;
   dfsa->FsAreaLsnAllocated = NULL;

   rc = dfsRsrBitmapFlush( TRUE);               // flush, and terminate cache

   TxFreeMem( rsr->sup);

   RETURN (rc);
}                                               // end 'dfsRsrAreaClose'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Interpret and execute specific RSR command;
/*****************************************************************************/
static ULONG dfsRsrCommand
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *none,                    // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;
   ULN64               sn = 0;                  // sector number input
   TXLN                dc;                      // DFS command
   TXLN                s0;                      // big temporary string space
   TXLN                s1;                      // big temporary string space
   int                 cc;                      // command string count
   char               *c0, *c1, *c2;            // parsed command parts
   char               *pp;                      // parameter pointer

   ENTER();

   pp = TxaGetArgString( TXA_CUR, 1, 0, TXMAXLN, dc); // dc => cmd from arg 1
   cc = TxaArgCount( );                         // number of parameters
   c0 = TxaArgValue(0);
   c1 = TxaArgValue(1);
   c2 = TxaArgValue(2);

   sn = nav.this;                               // default at current sector

   if ((strcmp(c0, "/?") == 0) ||
       (strcmp(c0,  "?") == 0)  )
   {
      TxShowTxt( rsr_txt);
   }
   else if (strcasecmp(c0, "super"    ) == 0)
   {
      sprintf( dc, "0x0%llx", RSR_LSNSUP2);      // most likely location
      TxaReParseCommand( dc);
      rc = DFS_PENDING;                         // display the sector
   }
   else if (strcasecmp(c0, "label"    ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nSet volume label in superblock (16-character, Linux FS-name)\n");
         TxPrint("\nUsage: %s  [label]\n\n"
                 "   label  : new volume label, maximum 16 characters\n"
                 "   -!-    = do not prompt for new value, just set it\n", c0);
      }
      else
      {
         if (cc > 1)
         {
            strcpy( s1, c1);                    // use specified parameter value
         }
         else if (SINF->p)                      // it is a partition
         {
            strcpy( s1, SINF->p->plabel);       // use label determined by readdiskinfo
         }
         else
         {
            strcpy( s1, "");                    // start with empty label
         }
         s1[ RSR_LEN_LBL] = 0;                  // truncate to allowed length

         if (dfsa->batch)                       // s1 has new label value
         {
            memcpy( rsr->sup->VolumeName, s1, RSR_LEN_LBL);
            rc = dfsRsrWriteSuperBlocks();
         }
         else
         {
            #if defined (USEWINDOWING)          // Allow interactive update
            if (!TxaOptUnSet('!') && txwIsWindow( TXHWND_DESKTOP))
            {
               TXLN      prompt;

               sprintf( prompt, "%s\n\nVolume label in superblock: '%s'\n\n"
                                "Specify new %d-character volume label\n",
                                 dfstStoreDesc1( DFSTORE) + 10, s1, RSR_LEN_LBL);

               if (txwPromptBox( TXHWND_DESKTOP, TXHWND_DESKTOP, NULL,
                      prompt, " Set volume label in superblock ",
                      5144, TXPB_MOVEABLE | TXPB_HCENTER | TXPB_VCENTER,
                      RSR_LEN_LBL, s1) != TXDID_CANCEL)
               {
                  TxStrip( s1, s1, ' ', ' ');   // strip leading/trailing spaces
                  memcpy( rsr->sup->VolumeName, s1, RSR_LEN_LBL);
                  rc = dfsRsrWriteSuperBlocks();
               }
            }
            else
            #endif
            {
               if (TxConfirm( 5144, "Write Volume LABEL '%s' to superblock ? [Y/N] : ", s1))
               {
                  memcpy( rsr->sup->VolumeName, s1, RSR_LEN_LBL);
                  rc = dfsRsrWriteSuperBlocks();
               }
            }
         }
      }
   }
   else if (strcasecmp(c0, "uuid"    ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nUsage: %s  [UUID] | [-u:UUID]\n"
                 "\n      UUID : UUID string value, 5 groups separated by hyphens, no brackets"
                 "\n   -u:UUID : 36 character UUID value: 12345678-abcd-0123-4567-0123456789ab\n", c0);
      }
      else
      {
         if (TxaOptMutEx((cc > 1), "u", "if a UUID parameter is specified!", NULL))
         {
            if (cc > 1)
            {
               strcpy( s1, c1);                 // use specified parameter value
            }
            else
            {
               strcpy( s1, TxaOptStr( 'u', NULL, ""));
            }
            sprintf( s0, "{%s}", s1);           // Make fully-formed UUID string

            if (((strlen( s1) == 0) && (!dfsa->batch)) || dfsUidStringIsValid( s0))
            {
               if (dfsa->batch)                 // s0 has valid new UUID
               {
                  dfsUidStringToBinary( s0, rsr->sup->Uuid, TRUE);
                  rc = dfsRsrWriteSuperBlocks();
               }
               else
               {
                  #if defined (USEWINDOWING)    // Allow interactive update
                  if (txwIsWindow( TXHWND_DESKTOP))
                  {
                     if (dfsGuidUuidEditor( "Update UUID value in Reiser Superblock",
                                            "Write Back", SINF->partid,
                                             5143, s0, TRUE, rsr->sup->Uuid) == NO_ERROR)
                     {
                        rc = dfsRsrWriteSuperBlocks();
                     }
                  }
                  else
                  #endif
                  {
                     if (TxConfirm( 5143, "Write UUID %s on PID %hu ? [Y/N] : ", s0, SINF->partid))
                     {
                        dfsUidStringToBinary( s0, rsr->sup->Uuid, TRUE);
                        rc = dfsRsrWriteSuperBlocks();
                     }
                  }
               }
            }
            else
            {
               TxPrint( "\nInvalid UUID value: '%s'\n", s1);
            }
         }
      }
      TxPrint( "\nVolume UUID now   : %s\n", dfsFsUuidValueString( rsr->sup->Uuid));
   }
   else
   {
      rc = DFS_CMD_UNKNOWN;                     // cmd not recognized
   }
   RETURN (rc);
}                                               // end 'dfsRsrCommand'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// RSR  filesystem, identify specified sector as a valid superblock
/*****************************************************************************/
BOOL dfsRsrIsSuperBlock                         // RET   sector is a valid sb
(
   BYTE               *sec                      // IN    sector contents
)
{
   return  (!memcmp(((S_RSR_SUPER*)sec)->Signature,sg_rsrsup,SG_RSRSUP));
}                                               // end 'dfsRsrIsSuperBlock'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// REISER filesystem, display sector constents as a superblock
/*****************************************************************************/
void dfsRsrDisplaySuperBlock
(
   BYTE               *sec                      // IN    sector contents
)
{
   dfsRsrSuperBlock( sec, FALSE);
}                                               // end 'dfsRsrDisplaySuperBlock'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// RSR filesystem, identify specified sector
/*****************************************************************************/
ULONG dfsRsrIdent
(
   ULN64               lsn,                     // IN    LSN for sector
   ULN64               d2,                      // IN    dummy
   char               *st,                      // OUT   sector type
   void               *sec                      // IN    sector contents
)
{
   ULONG               dr = NO_ERROR;
   BYTE                rc = ST_UDATA;

   ENTER();

   if (!memcmp(((S_RSR_SUPER*)sec)->Signature,sg_rsrsup,SG_RSRSUP)) rc=ST_RSRSUP;
   switch (rc)
   {
      case ST_UDATA:
         dr = DFS_PENDING;
         break;

      default:
         break;
   }
   *st = rc;
   RETURN (dr);
}                                               // end 'dfsRsrIdent'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// RSR filesystem, supply sector-type description string
/*****************************************************************************/
static ULONG dfsRsrStype
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *st,                      // IN    sector type
   void               *data                     // OUT   type description
)
{
   ULONG               dr  = NO_ERROR;
   char               *buf = (char *) data;
   BYTE                tp  = st[0];

   switch (tp)                                  // searchable types
   {
      case ST_RSRSUP: sprintf(buf,"RSR  superblock  "); break;
      default:
         switch (tp | ST__INFO)                 // non-searchable ones
         {
            default:       dr = DFS_PENDING;
               break;
         }
         break;
   }
   return (dr);
}                                               // end 'dfsRsrStype'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// RSR filesystem, display sector-contents based on type
/*****************************************************************************/
static ULONG dfsRsrDispl
(
   ULN64               psn,                     // IN    base psn for sector
   ULN64               d2,                      // IN    dummy
   char               *type,                    // IN    type of sector
   void               *data                     // IN    sector contents
)
{
   ULONG               dr = NO_ERROR;
   BYTE                st = (BYTE) *type;

   ENTER();

   switch (st)
   {
      case ST_RSRSUP: (void) dfsRsrSuperBlock(rbuf, TRUE);     break;

      default:
         dr = DFS_PENDING;
         break;
   }
   RETURN (dr);
}                                               // end 'dfsRsrDispl'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display RSR  super-block
/*****************************************************************************/
static ULONG dfsRsrSuperBlock                   // RET   rc = 0 if type match
(
   BYTE               *sector,                  // IN    Drive specification
   BOOL                navigation               // IN    update nav values
)
{
   ULONG               rc    = 0;               // rc, sector match
   S_RSR_SUPER        *sd    = (S_RSR_SUPER *) sector;
   TXTM                text;

   ENTER();

   dfsRsrDisplayStatus( "Filesystem status : ", sd->State);

   TxPrint( "Number of blocks  : 0x%8.8lx     Free: 0x%8.8lx  Blocksize: %hu bytes\n",
             sd->Blocks, sd->FreeBlocks, sd->BlockSize);

   dfsX10(  "Rootblock sector  : ", (ULN64) rsr->SectorsPerBlock * sd->RootBlock, "", " ");
   TxPrint( "on block: 0x%8.8lx  Rootlevel: %hu\n", sd->RootBlock, sd->TreeHeight);

   TxPrint( "Bitmap size  is   :% 4hu blocks = % 7hu sectors\n",
             sd->BitmapBlocks, rsr->SectorsPerBlock * sd->BitmapBlocks);

   dfsSz64( "Partition size    : ", (ULN64) rsr->SectorsPerBlock * sd->Blocks, "         (based on #blocks)\n");

   if (sd->VolumeName[0] == 0)
   {
      strcpy( text, "-none-");
   }
   else
   {
      TxCopy(  text, sd->VolumeName, 17);
   }
   TxPrint("Volume label      : %s%s%s\n", CBG, text, CNN);
   TxPrint("Volume UUID       : %s\n", dfsFsUuidValueString( sd->Uuid));

   RETURN (rc);
}                                               // end 'dfsRsrSuperBlock'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display value of superblock status value
/*****************************************************************************/
static void dfsRsrDisplayStatus
(
   char               *lead,                    // IN    leading text
   ULONG               st                       // IN    RSR status value
)
{
   TxPrint("%s%lu = ", lead, st);
   if      (st == RSR_FS_CLEAN) TxPrint( "%sUnmounted and clean", CBG);
   else
   {
      TxPrint( "%s", CBR);
      if (st & RSR_FS_DIRTY)    TxPrint( "Errors present; ");
   }
   TxPrint( "%s\n", CNN);                       // back to neutral color
}                                               // end 'dfsRsrDisplayStatus'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Make filesystem usage map dump on TxPrint output
/*****************************************************************************/
static ULONG dfsRsrAllocMap
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *options,                 // IN    display options
   void               *data                     // OUT   dummy
)
{
   ULN64               i;                       // index in data-area
   ULONG               b;                       // byte-index for one mapchar
   ULN64               cClus = 0;               // Clusters in current char
   ULN64               lClus = 0;               // Clusters in current line
   ULN64               mClus = 0;               // Clusters in current map
   ULN64               uClus = 1;               // Last cluster that is in use
   ULONG               l;                       // line number, 0 based
   ULONG               a;                       // index in display-array
   ULN64               perc;                    // percentage
   TX1K                ascii;                   // display-array
   ULONG               spe;                     // sectors per entry
   ULONG               acl;                     // ascii alloc chars per line
   ULONG               cpc;                     // clusters per display-char
   ULONG               spc;                     // sectors per display-char
   ULN64               size;                    // nr of clusters to map
   BOOL                verbose = (*options != '@');
   ULONG               bsSmart;
   ULN64               unallocSmart = 0;        // Size of THIS unallocated area
   BOOL                oneLineProgress = FALSE; // alloc in one line, output per character

   ENTER();

   dfsa->FsUnallocSmart = 0;                    // will be recalculated now
   bsSmart = dfsGetBufferSize( DFSOPTIMALBUF, DFSMAXBUFSIZE); // for SMART prediction

   spe   = rsr->SectorsPerBlock;
   size  = rsr->Block;                          // Block 0 .. last-valid

   acl   = dfsAllocCharsPerLine( 'c');
   cpc   = dfsAllocItemsPerChar( 'l', options, size, acl);
   if (cpc >= (size / acl))                     // will be a single line
   {
      oneLineProgress = TRUE;                   // one char at a time, no ascii string used
   }
   spc   = cpc * spe;
   dfsProgressInit( 0, size * spe, 0, "Get ALLOCATION info, at Sector:", dfstStoreDpid( DFSTORE), DFSP_STAT, 0);

   if (verbose)
   {
      dfsSz64("Size for one line : ", (ULN64) acl * spc, "  ");
      TxPrint("with %3lu characters,", acl);
      dfsSiz8(" size : ", spc, "\n");
      TxPrint(" %16.16s = Allocation grade, empty to full.     ", mapchar);
      dfsSiz8(" SmartBuf size : ", bsSmart, "\n");
   }
   TxPrint(   "          %s%*.*s%s\n", CBC, acl, acl, BLIN, CNN);
   dfsX10( CNZ, dfsRsrBlock2Lsn(0), CNZ, "");   // display address for first block
   TxPrint("%s%s", CBC, CNC);

   for (i=0, a=0, l=0; (i < size) && (!TxAbort());)
   {
      if (a == 0)
      {
         memset(ascii, 0, TXMAXLN);
         lClus = 0;
      }
      for (cClus=0, b=0; (b < cpc) && (i < size); b++, i++)
      {
         if (dfsRsrBitmapCache(i, NULL))        // block in use ?
         {
            if (unallocSmart != 0)              // first in-use after unallocated?
            {
               if (unallocSmart >= bsSmart)                  //- if at least ONE smart buffer
               {
                  unallocSmart -= (bsSmart / 2);             //- compensate for alignment issues
                  unallocSmart -= (unallocSmart % bsSmart);  //- clip to whole smart buffer size

                  dfsa->FsUnallocSmart += unallocSmart;      //- add this area to total
               }
               unallocSmart = 0;                // reset for next area to come
            }
            cClus++;
            uClus = i;                          // last in-use one sofar
         }
         else                                   // unallocated, Smart predict
         {
            unallocSmart += spe;                // add to size THIS area
         }
      }
      lClus += cClus;

      dfsProgressShow( (i * spe) + spc, 0, 0, NULL);

      if (oneLineProgress == TRUE)
      {
         TxPrint( "%c", (char) ((cClus == 0) ? ' ' : mapchar[cClus * 8 / cpc])); // slow, acts as progress bar
      }
      else
      {
         ascii[a] = (char) ((cClus == 0) ? ' ' : mapchar[cClus * 8 / cpc]);
      }
      a++;
      if ((i && ((i%(acl*cpc)) == 0)) || (i >= size))
      {
         if (oneLineProgress == FALSE)
         {
            TxPrint( "%s", ascii);              // display accumulated chars (fast)
         }
         perc  = (ULN64)(100*lClus / (a*cpc));
         if (a == acl)
         {
            TxPrint("%s%s% 3lu%%\n", CBC, CNN, perc);
         }
         else
         {
            TxPrint("%s%.*s%s% 3lu%%\n", CBC, (int) (acl-a-1), BLIN, CNN, perc);
         }
         if (i < size)
         {
            dfsX10( CNZ, dfsRsrBlock2Lsn(i), CNZ, "");
            TxPrint("%s%s", CBC, CNC);
            a = 0;                              // keep a value on last line
         }
         mClus += lClus;
         l++;
      }
   }
   if (unallocSmart != 0)                           //- unalloc area pending?
   {
      if (unallocSmart >= bsSmart)                  //- if at least ONE smart buffer
      {
         unallocSmart -= (bsSmart / 2);             //- compensate for alignment issues
         unallocSmart -= (unallocSmart % bsSmart);  //- clip to whole smart buffer size

         dfsa->FsUnallocSmart += unallocSmart;      //- add this area to total
      }
   }
   TxPrint("          %s%.*s%s\n", CBC, (USHORT) a, BLIN, CNN);

   if (TxAbort())
   {
      dfsa->FsUnallocSmart = 0;                 // signal info NOT available!
      TxPrint( "\nAllocation check and display aborted! Used/Free info not available.\n");
   }
   else
   {
      if (size != 0)                            // avoid devide by 0
      {
         ULN64            used = mClus * spe;

         dfsa->FsUnallocated   = size * spe - used; // free space (optimize hint)

         dfsSz64("Unallocated sects : ", dfsa->FsUnallocated, " ");
         dfsSz64("SmartUse ", dfsa->FsUnallocSmart, " ");
         TxPrint("=% 5.1lf%%\n", (double) (100.0 * ((double) dfsa->FsUnallocSmart) / (double) (size * spe)));

         dfsSz64("Allocated sectors : ", used, " ");
         dfsSiz8("of ", size * spe, " ");
         TxPrint("=% 5.1lf%%\n", (double) (100.0 * ((double) mClus) / (double) size));
      }
      dfsa->FsTruncPoint = dfsRsrBlock2Lsn( uClus +1); // first free (truncate)
      dfsa->FsLastInUse  = dfsa->FsTruncPoint -1;

      #if defined (RSR_RESIZING)
         dfsSz64( "Minimum vol. size : ", dfsa->FsTruncPoint, " (for  Resizing)\n");
      #endif
   }
   dfsProgressTerm();
   RETURN (NO_ERROR);
}                                               // end 'dfsRsrAllocMap'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// DFS RSR write-file to disk (SaveTo to file)
/*****************************************************************************/
static ULONG dfsRsrFileSaveAs
(
   ULN64               dse,                     // IN    dirsec/entry LSN
   ULN64               d2,                      // IN    dummy
   char               *path,                    // IN    destination path
   void               *ren                      // IN    new name, or empty
)
{
   ULONG               rc  = NO_ERROR;

   ENTER();
   TRARGS(("Dirsec/entry LSN : 0x%llX, to path: '%s'\n", dse, path));

   TxPrint( "\ndfsRsrFileSaveAs Not implemented yet\n");

   RETURN(rc);
}                                               // end 'dfsRsrFileSaveAs'
/*---------------------------------------------------------------------------*/
