//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// FDISK Area related functionality, allowing some FS-access in FDISK mode
//
// Author: J. van Wijk
//
// JvW  23-05-2006 Initial version, derived from DFSAFDSK
//

#include <txlib.h>                              // TX library interface

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFSee navigation and defs
#include <dfsutil.h>                            // DFSee utility functions
#include <dfsupart.h>                           // FDISK partition functions
#include <dfsufgpt.h>                           // FDISK GPT utility functions
#include <dfsspace.h>                           // DFS file-space interface
#include <dfsarea.h>                            // FDISK Area/Alloc functions
#include <dfsaapfs.h>                           // APFS   display & analysis
#include <dfsadump.h>                           // DUMPFS interface (for Init)
#include <dfsaefat.h>                           // EFAT   display & analysis
#include <dfsafat.h>                            // FAT    display & analysis
#include <dfsahfs.h>                            // HFS+   display & analysis
#include <dfsahpfs.h>                           // HPFS   display & analysis
#include <dfsajfs.h>                            // JFS    display & analysis
#include <dfsaext.h>                            // JFS    display & analysis
#include <dfsantfs.h>                           // NTFS   display & analysis
#include <dfsarsr.h>                            // REISER display & analysis
#include <dfsaswap.h>                           // SWAP   interface (for Init)


// Determine and set type for area specified by a single LSN value
static ULONG dfsFdskSetNewArea                  // RET result
(
   ULN64               lsn,                     // IN    LSN to find area for
   USHORT              disk,                    // IN    current disk number
   ULN64              *start,                   // OUT   start LSN of area
   ULN64              *final,                   // OUT   final LSN of area
   char               *atype                    // OUT   type of area
);


/*****************************************************************************/
// Determine allocation-bit for specified LSN, use Area knowledge if available
/*****************************************************************************/
ULONG dfsFdskAreaAllocated                      // RET   LSN is allocated
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *atype,                   // IN    dummy
   void               *data                     // INOUT dummy
)
{
   static ULONG        al     = TRUE;           // cached alloc status
   static USHORT       aDisk  = 0;              // cached Disk number
   static ULN64        aStart = 0;              // cached Area start
   static ULN64        aFinal = 0;              // cached Area end
   static char         aType  = DFSAREA_BOOT;   // cached type 'Bootarea'

   if ((SINF->disknr != aDisk) || (lsn < aStart) || (lsn > aFinal))
   {
      aDisk = SINF->disknr;
      if (dfsFdskSetNewArea( lsn, aDisk, &aStart, &aFinal, &aType) == NO_ERROR)
      {
         switch (aType)
         {
            case DFSAREA_BOOT: al = TRUE;   break; // whole areas same status
            case DFSAREA_PART: al = TRUE;   break;
            case DFSAREA_FREE: al = FALSE;  break;
            default:                        break; // will be set per sector
         }
      }
      else                                      // some error on initializing
      {                                         // the filesystem for the area
         aType = DFSAREA_PART;                  // set as unsupported FS
         al    = TRUE;                          // assume all allocated
      }
   }

   //- LSN in cache now, cached ALLOC correct except for FSYS-specific areas
   if (aType == DFSAREA_FSYS)                   // determine actual alloc
   {
      al = DFSFNCALL( dfsa->FsAreaLsnAllocated, lsn, 0, NULL, NULL);
   }                                            // else return cached value

   if (atype != NULL)
   {
      *atype = aType;                           // return alloc type as well
   }
   TRLEVX(150,("FdskAreaAllocated at lsn:0x%llu : %s\n", lsn, (al) ? "Yes" : "No"));
   return (al);
}                                               // end 'dfsFdskAreaAllocated'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Determine and set type for area specified by a single LSN value
/*****************************************************************************/
static ULONG dfsFdskSetNewArea                  // RET result
(
   ULN64               lsn,                     // IN    LSN to find area for
   USHORT              disk,                    // IN    current disk number
   ULN64              *start,                   // OUT   start LSN of area
   ULN64              *final,                   // OUT   final LSN of area
   char               *atype                    // OUT   type of area
)
{
   ULONG               rc = DFS_NOT_FOUND;
   DFSDISKINFO        *d;                       // ptr to phys disk info
   TXTS                fs;                      // filesystem name

   ENTER();
   TRACES(( "disk:%hu area lsn: 0x%llx\n", disk, lsn));

   if ((d = dfsGetDiskInfo(disk)) != NULL)
   {
      ULN64            cylSize = (d->geoHeads * d->geoSecs);

      if (dfsGptLsnInTableArea( d, lsn, start, final))
      {
         *atype = DFSAREA_BOOT;
         rc     = NO_ERROR;                     // signal success
      }
      else if (lsn < cylSize)                   // MBR cylinder, may contain bootloader
      {                                         // that is larger than a single track!
         *start = 0;
         *final = cylSize -1;
         *atype = DFSAREA_BOOT;
         rc     = NO_ERROR;                     // signal success
      }
      else
      {
         USHORT        parts  = dfsPartitions();
         USHORT        index;                   // partition index
         ULN64         endsec = d->geoSecs -1;  // last sector prev part
         DFSPARTINFO  *p;                       // ptr to partition info

         for (index = 1; (index <= parts) && (rc == DFS_NOT_FOUND); index++)
         {
            p = dfsGetPartInfo(index);

            if (p && (p->disknr == disk))       // partition on this disk
            {
               if (!p->primary)                 // it is a logical
               {
                  if ((lsn >= p->partPsn) &&    // from EBR sector upto and
                      (lsn <= p->basePsn))      // including bootsector
                  {
                     *start = p->partPsn;
                     *final = p->basePsn;
                     *atype = DFSAREA_BOOT;
                     rc     = NO_ERROR;         // signal success
                     break;                     // and end loop
                  }
               }
               if (lsn < p->basePsn)            // must be freespace
               {
                  *start = endsec +1;
                  *final = ((p->primary) ? p->basePsn : p->partPsn) -1;
                  *atype = DFSAREA_FREE;
                  rc     = NO_ERROR;            // signal success
                  break;                        // and end loop
               }
               else if (lsn == p->basePsn)      // primary, bootsector
               {
                  *start = p->basePsn;          // single sector area :-)
                  *final = p->basePsn;
                  *atype = DFSAREA_BOOT;
                  rc     = NO_ERROR;            // signal success
                  break;                        // and end loop
               }
               else if ((lsn >= p->basePsn) &&  // from bootsector upto and
                        (lsn <= p->lastPsn))    // including last sector
               {
                  strcpy( fs, p->fsform);
                  if ((strncasecmp( fs, "APFS",   4) == 0) ||
                      (strncasecmp( fs, "DUMPFS", 6) == 0) ||
                      (strncasecmp( fs, "EFAT",   4) == 0) ||
                      (strncasecmp( fs, "EXT",    3) == 0) ||
                      (strncasecmp( fs, "FAT",    3) == 0) ||
                      (strncasecmp( fs, "HFS",    3) == 0) ||
                      (strncasecmp( fs, "HPFS",   4) == 0) ||
                      (strncasecmp( fs, "JFS",    3) == 0) ||
                      (strncasecmp( fs, "NTFS",   4) == 0) ||
                      (strncasecmp( fs, "REISER", 6) == 0) ||
                      (strncasecmp( fs, "SWAP",   4) == 0) ||
                      (strncasecmp( fs, "VFAT",   4) == 0)  )
                  {
                     *atype = DFSAREA_FSYS;
                  }
                  else                          // unsupported filesystem,
                  {                             // assume all ALLOCATED
                     *atype = DFSAREA_PART;
                  }
                  *start = p->basePsn;
                  *final = p->lastPsn;
                  rc     = NO_ERROR;            // signal success
                  break;                        // and end loop
               }
               endsec = p->lastPsn;             // new last used sector
            }
         }
         if ((rc == DFS_NOT_FOUND) && (lsn > endsec))
         {
            *start = endsec +1;                 // freespace at end disk
            *final = L64_NULL;                  // rest is free, unless GPT!
            *atype = DFSAREA_FREE;

            //- Freespace at end disk, but possible GPT area at end!
            if (d->gptHeader)
            {
               ULN64   firstGpt;
               ULN64   endGpt;

               if (dfsGptLsnInTableArea( d, d->sectors -1, &firstGpt, &endGpt))
               {
                  if (firstGpt <= *start)       // GPT starts directly after end partition
                  {
                     *atype = DFSAREA_BOOT;     // so change freespace to BOOT
                     *final = endGpt;           // and set correct end for that
                  }
                  else
                  {
                     *final = firstGpt -1;      // end of the Freespace
                  }
               }
            }
            rc = NO_ERROR;                      // signal success
         }
      }
   }
   else
   {
      *start = 0;
      *final = 0;
      *atype = DFSAREA_BOOT;
      rc     = DFS_NO_DEVICE;
   }
   if (rc == NO_ERROR)
   {
      dfstSetAreaLimits( DFSTORE, *start, *final);
      if (*atype == DFSAREA_FSYS)               // need to initialize the FS
      {
         DFSFNCALL(dfsa->FsAreaClose,0,0,NULL,NULL); // FS area handler, close

         if      (strncasecmp( fs, "APFS",   4) == 0) rc = dfsApfsAreaInit();
         else if (strncasecmp( fs, "DUMPFS", 6) == 0) rc = dfsDumpAreaInit();
         else if (strncasecmp( fs, "EFAT",   4) == 0) rc = dfsEfatAreaInit();
         else if (strncasecmp( fs, "EXT",    3) == 0) rc = dfsExtAreaInit();
         else if (strncasecmp( fs, "FAT",    3) == 0) rc = dfsFatAreaInit();
         else if (strncasecmp( fs, "HFS",    3) == 0) rc = dfsHfsAreaInit();
         else if (strncasecmp( fs, "HPFS",   4) == 0) rc = dfsHpfsAreaInit();
         else if (strncasecmp( fs, "JFS",    3) == 0) rc = dfsJfsAreaInit();
         else if (strncasecmp( fs, "NTFS",   4) == 0) rc = dfsNtfsAreaInit();
         else if (strncasecmp( fs, "REISER", 6) == 0) rc = dfsRsrAreaInit();
         else if (strncasecmp( fs, "SWAP",   4) == 0) rc = dfsSwapAreaInit();
         else if (strncasecmp( fs, "VFAT",   4) == 0) rc = dfsFatAreaInit();
      }
      TRACES(( "New area: '%c' start: 0x%llx, final: 0x%llx\n", *atype, *start, *final));
   }
   RETURN( rc);
}                                               // end 'dfsFdskSetNewArea'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display allocation information for whole disk, use Area info if available
/*****************************************************************************/
ULONG dfsFdskAllocMap
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *options,                 // IN    display options
   void               *data                     // OUT   dummy
)
{
   ULN64               i;                       // index in data-area
   ULONG               b;                       // byte-index for one mapchar
   ULN64               cSect;                   // Sectors in current char
   ULN64               lSect = 0;               // Sectors in current line
   ULN64               mSect = 0;               // Sectors in current map
   ULONG               l;                       // line number, 0 based
   ULONG               a;                       // index in display-array
   ULN64               perc;                    // percentage
   char                mChar;                   // single character to display
   TX1K                ascii;                   // display-array
   ULONG               acl;                     // ascii alloc chars per line
   ULN64               spc;                     // sectors per display-char
   ULN64               size;                    // nr of sectors to map
   BOOL                verbose = (*options != '~');
   char                atype;
   char                area;
   ULONG               bsSmart;
   ULN64               unallocSmart = 0;        // Size of THIS unallocated area
   BOOL                oneLineProgress = FALSE; // alloc in one line, output per character

   ENTER();

   dfsa->FsUnallocSmart = 0;                    // will be recalculated now
   bsSmart = dfsGetBufferSize( DFSOPTIMALBUF, DFSMAXBUFSIZE); // for SMART prediction

   size  = dfsGetLogicalSize();                 // all sectors current object

   acl   = dfsAllocCharsPerLine( 'c');
   spc   = dfsAllocItemsPerChar( 'l', options, size, acl);
   if (spc >= (size / acl))                     // will be a single line
   {
      oneLineProgress = TRUE;                   // one char at a time, no ascii string used
   }

   dfsProgressInit( 0, size *   1, 0, "Get ALLOCATION info, at Sector:", dfstStoreDpid( DFSTORE), DFSP_STAT, 0);
   if (verbose)
   {
      dfsSz64("Size for one line : ", (ULN64) acl * spc, "  ");
      TxPrint("with %3lu characters,", acl);
      dfsSiz8(" size : ", spc, "\n");
      TxPrint(" %16.16s = Allocation grade.  f=Free  B=Boot    ", mapchar);
      dfsSiz8(" SmartBuf size : ", bsSmart, "\n");
   }
   TxPrint(   "          %s%*.*s%s\n", CBC, acl, acl, BLIN, CNN);
   dfsX10( CNZ, 0, CNZ, "");                    // display address for first block
   TxPrint("%s%s", CBC, CNC);

   for (i=0, a=0, l=0; (i < size) && (!TxAbort());)
   {
      if (a == 0)
      {
         memset(ascii, 0, TXMAXLN);
         lSect = 0;
      }
      area = ' ';
      for (cSect=0, b=0; (b < spc) && (i < size); b++, i++)
      {
         if (dfsFdskAreaAllocated( i, 0, &atype, NULL))  // sector in use ?
         {
            if (unallocSmart != 0)              // first in-use after unallocated?
            {
               if (unallocSmart >= bsSmart)                  //- if at least ONE smart buffer
               {
                  unallocSmart -= (bsSmart / 2);             //- compensate for alignment issues
                  unallocSmart -= (unallocSmart % bsSmart);  //- clip to whole smart buffer size

                  dfsa->FsUnallocSmart += unallocSmart;      //- add this area to total
               }
               unallocSmart = 0;                // reset for next area to come
            }
            cSect++;
         }
         else                                   // unallocated, Smart predict
         {
            unallocSmart++;                     // add sector to size THIS area
         }
         switch (atype)
         {
            case DFSAREA_BOOT:
            case DFSAREA_FREE:
               if (area != DFSAREA_BOOT)        // BOOT overrules free!
               {
                  area = atype;
               }
               break;

            default:                            // Should not happen, FS-free
               break;
         }
      }
      lSect += cSect;

      dfsProgressShow( i + spc, 0, 0, NULL);

      TRACES(("lSect 2nd Loop i: %llu, b:%lu = %llu area = '%c'\n", i, b, lSect, area));
      switch (area)
      {
         case DFSAREA_BOOT: mChar = 'B'; break;
         case DFSAREA_FREE:
            if (cSect == 0)                     // only when no used sectors
            {
                            mChar = 'f'; break;
            }
         default:           mChar = (char) ((cSect == 0) ? ' ' : mapchar[cSect * 8 / spc]);   break;
      }
      if (oneLineProgress == TRUE)
      {
         TxPrint( "%c", mChar);                 // slow, acts as progress bar
      }
      else
      {
         ascii[a] = mChar;
      }
      a++;
      if ((i && ((i%(acl*spc)) == 0)) || (i >= size))
      {
         if (oneLineProgress == FALSE)
         {
            TxPrint( "%s", ascii);              // display accumulated chars (fast)
         }
         perc  = (ULN64)((100*lSect) / (a*spc));
         if (a == acl)
         {
            TxPrint("%s%s% 3lu%%\n", CBC, CNN, perc);
         }
         else
         {
            TxPrint("%s%.*s%s% 3llu%%\n", CBC, (int) (acl-a-1), BLIN, CNN, perc);
         }
         if (i < size)
         {
            dfsX10( CNZ, i, CNZ, "");
            TxPrint("%s%s", CBC, CNC);
            a = 0;                              // keep a value on last line
         }
         mSect += lSect;
         l++;
      }
   }
   if (unallocSmart != 0)                           //- unalloc area pending?
   {
      if (unallocSmart >= bsSmart)                  //- if at least ONE smart buffer
      {
         unallocSmart -= (bsSmart / 2);             //- compensate for alignment issues
         unallocSmart -= (unallocSmart % bsSmart);  //- clip to whole smart buffer size

         dfsa->FsUnallocSmart += unallocSmart;      //- add this area to total
      }
   }
   TxPrint("          %s%.*s%s\n", CBC, (USHORT) a, BLIN, CNN);

   if (TxAbort())
   {
      dfsa->FsUnallocSmart = 0;                 // signal info NOT available!
      TxPrint( "\nAllocation check and display aborted! Used/Free info not available.\n");
   }
   else
   {
      if (size != 0)                            // avoid devide by 0
      {
         dfsa->FsUnallocated = size - mSect;    // free space (optimize hint)

         dfsSz64("Unallocated sects : ", dfsa->FsUnallocated, " ");
         dfsSz64("SmartUse ", dfsa->FsUnallocSmart, " ");
         TxPrint("=% 5.1lf%%\n", (double) (100.0 * ((double) dfsa->FsUnallocSmart) / (double) size));

         dfsSz64("Allocated sectors : ", mSect, " ");
         dfsSz64("of total ", size, " ");
         TxPrint("=% 5.1lf%%\n", (double) (100.0 * ((double) mSect) / (double) size));
      }
   }
   dfsProgressTerm();
   RETURN (NO_ERROR);
}                                               // end 'dfsFdskAllocMap'
/*---------------------------------------------------------------------------*/

