//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// APFS (container) structure definitions
//
// Author: J. van Wijk
//
// JvW  24-01-2018 Initial version derived from High Performance File System, APFS v1
// JvW  10-02-2019 Updated using the official Apple File System Reference for APFS v2/v3
//
#ifndef    DFSAPFS_H
   #define DFSAPFS_H


#define APFS_BLOCKSIZE                  4096

#define LSN_C_SUPER            ((ULN64)  0x0)

// types and subtypes used in the block header structure
typedef enum block_type
{
   BT_NO_TYPE                    = 0x00,        // Type invalid, or subtype 'none'
   BT_C_SUPERBLOCK               = 0x01,        // Container superblock
   BT_BTREE_ROOT                 = 0x02,        // Btree Node root
   BT_BTREE_NODE                 = 0x03,        // Btree Node regular
   BT_SPACEMANAGER               = 0x05,        // Spacemanager main object
   BT_SPACEMAN_CAB               = 0x06,        // Spacemanager Address Block
   BT_SPACEMAN_CIB               = 0x07,        // Spacemanager Info Block
   BT_SPACEMANBMAP               = 0x08,        // Spacemanager BitMap
   BT_SPACEMANFREE               = 0x09,        // Spacemanager Free queue            (subtype)
   BT_EXTENT_TREE                = 0x0A,        // Extent list tree                   (subtype)
   BT_OBJECT_MAP                 = 0x0B,        // Object map
   BT_CHECKP_MAP                 = 0x0C,        // Checkpoint Map
   BT_V_SUPERBLOCK               = 0x0D,        // Volume SB, filesystem
   BT_V_FS_TREE                  = 0x0E,        // Volume filesystem      tree        (subtype)
   BT_V_BLOCKREFTREE             = 0x0F,        // Block extent reference tree        (subtype)
   BT_V_SNAPMETATREE             = 0x10,        // Snapshot meta data     tree        (subtype)
   BT_REAPER                     = 0x11,        // Reaper object
   BT_REAP_LIST                  = 0x12,        // Reaper list
   BT_OMAP_SNAPSHOT              = 0x13,        // object map snapshot                (subtype)
   BT_EFI_JUMPSTART              = 0x14,        // EFI info used for booting
   BT_FUSION_MIDDLE_TREE         = 0x15,        // Fusion tree, blocks cached on SSD  (subtype)
   BT_FUSION_WRBACK_CACHE        = 0x16,        // Fusion writeback cache status info
   BT_FUSION_WRBACK_LIST         = 0x17,        // Fusion writeback cache list
   BT_ENCR_ROLLING_STATE         = 0x18,        // Encryption rolling key status
   BT_GENERIC_BITMAP             = 0x19,        // General purpose bitmap object
   BT_GENERIC_BITMAP_TREE        = 0x1A,        // Tree of general bitmap objects     (subtype)
   BT_GENERIC_BITMAP_BLK         = 0x1B,        // Block containing a general bitmap
   BT_UNIVERSAL_TEST_TYPE        = 0xFF         // used for testing, should NOT be in GA code
} BLOCK_TYPE;

#define HDR_OBJID_C_SUPER          1

#define HDR_TYPE_VALUE_MASK        0x0000ffff
#define HDR_TYPE_FLAGS_MASK        0xffff0000
#define HDR_TYPE_STORG_MASK        0xc0000000
#define HDR_TYPE_DEFND_MASK        0xf8000000

#define OID_TYPE_VIRT              0x00000000   // virtual-ID, phys-block is in an object-map
#define OID_TYPE_PHYS              0x40000000   // physical block number, directly accessible
#define OID_TYPE_EPHM              0x80000000   // In-memory if mounted, checkpoint otherwise

typedef struct s_b_header                       // APFS Block Header
{
   ULN64               checksum;                // 000 block checksum value
   ULN64               objectId;                // 008 object-ID
   ULN64               transactId;              // 010 transaction-ID
   ULONG               objectType;              // 018 object type
   ULONG               objSubType;              // 01C object subtype
} S_B_HEADER;                                   // end of struct "s_b_header"

// quick access macros to test for object-type using hdr-ptr or obj-ptr
#define APFShdrType(h) ((h)->objectType & HDR_TYPE_VALUE_MASK)
#define APFSobjType(o) ((o)->blockHdr.objectType & HDR_TYPE_VALUE_MASK)

typedef struct s_prange
{
   ULN64               block;                   // on-disk block-number
   ULN64               count;                   // number of blocks in range
} S_PRANGE;                                     // end of struct "s_prange"


#define EFI_JS_RESERVED                 16
#define SG_C_EFI_J      4
#define SV_C_EFI_J      (char)0x4A,(char)0x53,(char)0x44,(char)0x52
typedef struct s_b_efi_j
{
   S_B_HEADER          blockHdr;                // 000 Generic block header
   char                Signature[SG_C_EFI_J];   // 020 Signature 'JSDR'
   ULONG               ejVersion;               // 024 structure version (1)
   ULONG               ejFileLength;            // 028 EFI driver file length in bytes
   ULONG               ejExtentCount;           // 02C EFI driver file nr of extents
   ULN64               ejRes[ EFI_JS_RESERVED]; // 030 reserved/preserved area
   S_PRANGE            ejExtents[1];            // 0B0 Array of EFI driver file extents
} S_B_EFI_J;                                    // end of struct "s_b_efi_j"

//- Container feature flag definitions
#define CFF_DEFRAG                   0x0001LL   // defragging supported
#define CFF_LCFD                     0x0002LL   // low capacity fusion-drive
#define CFF_VALIDMASK                0x0003LL   // defined flags mask

//- Container RO compatible features
#define CFR_VALIDMASK                0x0000LL   // defined flags mask

//- Container incompatible features
#define CFI_VERSION1                 0x0001LL   // not compatible, APFS version from 10.12 (beta)
#define CFI_VERSION2                 0x0002LL   // not compatible, APFS version from 10.13
#define CFI_FUSION                   0x0100LL   // not compatible with fusion drives
#define CFI_VALIDMASK                0x0103LL   // defined flags mask

#define SB_MAX_FS                         100   // maximum nr of filesystems
#define SB_MAX_CNT                         32   // number of reserved counter slots
#define SB_MAX_EPH                          4   // number of fields in ephemeral info array

#define SB_CNT_CHECKSUM_SET                 0   // index of checksum-set  counter
#define SB_CNT_CHECKSUM_FAIL                1   // index of checksum-fail counter



#define SG_C_SUPER      4
#define SV_C_SUPER      (char)0x4e,(char)0x58,(char)0x53,(char)0x42
typedef struct s_c_super                        // APFS container super-block
{
   S_B_HEADER          blockHdr;                // 000 Generic block header
   char                Signature[SG_C_SUPER];   // 020 Signature 'NXSB'
   ULONG               csBlockSize;             // 024 Block size in bytes
   ULN64               csTotalBlocks;           // 028 Total number of blocks
   ULN64               csFeatures;              // 030 Feature flags
   ULN64               csRoCompatible;          // 038 RO Compatible features
   ULN64               csInCompatible;          // 040 Incompatible  features
   DFS_GUID            csContainerGuid;         // 048 128-bit uuid (4-2-2-2-6)
   ULN64               csNextPhysBlkOid;        // 058 next free phys block   id  (OID)
   ULN64               csNextTransactId;        // 060 next transaction ID to use (XID)
   ULONG               csSbDescBlockCount;      // 068 Superblock description area #blocks
   ULONG               csSpDataBlockCount;      // 06C Superblock data        area #blocks
   ULN64               csSbDescBaseBlock;       // 070 First Block# SB desc area (circular)
   ULN64               csSbDataBaseBlock;       // 078 First Block# SB data area (circular)
   ULONG               csNextSbDescBlock;       // 080 Next SB desc block to use
   ULONG               csNextSbDataBlock;       // 084 Next SB data block to use
   ULONG               csCpFirstDescBlock;      // 088 Index 1st valid checkpoint desc block
   ULONG               csCpDescLength;          // 08C Number of used  checkpoint desc blocks
   ULONG               csCpFirstDataBlock;      // 090 Index 1st valid checkpoint data block
   ULONG               csCpDataLength;          // 094 Number of used  checkpoint data blocks
   ULN64               csCpSpManObjId;          // 098 Checkpoint Spacemanager OID    (0x400)
   ULN64               csObjectMapBlock;        // 0A0 Object-map block number
   ULN64               csReaperObjId;           // 0A8 Reaper ephemeral object OID    (0x401)
   ULONG               csTestType;              // 0B0 Used for testing only   (0)
   ULONG               csMaxVolCount;           // 0B4 Maximum volume count (2..100)
   ULN64               csVolumeOid[SB_MAX_FS];  // 0B8 Volume offset table
   ULN64               csCounters[SB_MAX_CNT];  // 3D8 Volume statistics counters
   S_PRANGE            csBlockedOutRange;       // 4D8 Range of blocks not to be allocated
   ULN64               csEvictMappingTreeOid;   // 4E8 Tree to be removed from BlockedOut
   ULN64               csContainerFlags;        // 4F0 Additional container flags
   ULN64               csEfiJumpStart;          // 4F8 Start block reserved for the EFI boot loader
   DFS_GUID            csFusionUuid;            // 500 Fusion drive UUID
   S_PRANGE            csKeyLocker;             // 510 Location of container key-bag
   ULN64               csEphemInfo[SB_MAX_EPH]; // 520 Array of fields to manage epemeral objects
   ULN64               csTestOid;               // 540 Test object ID
   ULN64               csFusionMtOid;           // 548 Fusion Middle-Tree object ID
   ULN64               csFusionWbcOid;          // 550 Fusion writebackcache status object ID
   S_PRANGE            csFusionWbCache;         // 558 Blocks used for writebackcache
                                                // 568 rest of block currently unused
} S_C_SUPER;                                    // FFF of 4096 byte struct "s_c_super"


//- single mapping structure, mapping virtual-object to physical in chkpt data area
typedef struct s_chkpt_mapping
{                                               // shown offsets are for FIRST mapping!
   ULONG               cpm_type;                // 028 checkpoint map type
   ULONG               cpm_subtype;             // 02c checkpoint map subtype
   ULONG               cpm_size;                // 030 size of the object in bytes
   ULONG               cpm_pad;                 // 034 reserved (0?)
   ULN64               cpm_fs_oid;              // 038 Virtual Object-ID related filesystem
   ULN64               cpm_oid;                 // 040 Ephemeral Object-ID for object
   ULN64               cpm_paddr;               // 048 Object address in checkpoint-DATA area
} S_CHKPT_MAPPING;                              // end of struct "s_chkpt_mapping"

#define CPM_LAST_CHECKPOINT   0x00000001        // last entry for THIS mapping
#define CPM_SPMGR_INDEX                0        // SPMGR is first index in the map

// Checkpoint map object (possibly multiple mappings)
typedef struct s_checkpoint_map
{
   S_B_HEADER          blockHdr;                // 000 Generic block header
   ULONG               cpm_flags;               // 020 mapping flags (last)
   ULONG               cpm_count;               // 024 number of mappings
   S_CHKPT_MAPPING     cpm_map[1];              // 028 First mapping in the array
} S_CHECKPOINT_MAP;                             // end of struct "s_checkpoint_map"


#define OMF_MANUALLY_MANAGED         0x0001LL   // Object does NOT support snapshots
#define OMF_ENCRYPTING               0x0002LL   // Storage is being encrypted
#define OMF_DECRYPTING               0x0004LL   // Storage is being decrypted
#define OMF_KEYROLLING               0x0008LL   // Encryption key being changed
#define OMF_CRYPTO_GEN               0x0010LL   // Encryption change in progress
#define OMF_VALIDMASK                0x001FLL   // defined flags mask

typedef struct s_object_map
{
   S_B_HEADER          blockHdr;                // 000 Generic block header
   ULONG               omFlags;                 // 020 Object Map flags
   ULONG               omSnapCount;             // 024 Number of snapshots in map
   ULONG               omTreeType;              // 028 Type of tree being used
   ULONG               omSnapTreeType;          // 02c Snapshot tree type
   ULN64               omTreeOid;               // 030 Tree object ID    (blocknr)
   ULN64               omSnapTreeOid;           // 038 Snapshot Tree object ID
   ULN64               omSnapLastXid;           // 040 XID of most recent Snapshot
   ULN64               omPendingRevertMin;      // 048 smallest XID being reverted
   ULN64               omPendingRevertMax;      // 050 largest  XID being reverted
} S_OBJECT_MAP;                                 // 058 of struct "s_object_map"


#define OMV_DELETED                  0x0001LL   // Object deleted, placeholder only
#define OMV_SAVED                    0x0002LL   // Not to be updated (never used)
#define OMV_ENCRYPTED                0x0004LL   // Object is encrypted
#define OMV_NOHEADER                 0x0008LL   // Object is stored without a header
#define OMV_CRYPTO_GEN               0x0010LL   // Encryption change in progress
#define OMV_VALIDMASK                0x001FLL   // defined flags mask

typedef struct s_omap_key                       // Object Map KEY in tree
{
   ULN64               okOid;                   // Object ID
   ULN64               okXid;                   // Transaction ID
} S_OMAP_KEY;                                   // end of struct "s_omap_key"


typedef struct s_omap_value                     // Object Map value in LEAF NODES
{
   ULONG               ovFlags;                 // Value flags
   ULONG               ovSize;                  // Object size in bytes
   ULN64               ovPaddr;                 // Object phys address (blocknumber)
} S_OMAP_VALUE;                                 // end of struct "s_omap_value"

typedef struct s_omap_vlink                     // Object Map value NON-LEAF NODES
{
   union
   {
      ULN64            ovPaddr;                 // Link physical address (blocknumber)
      void            *ovLnode;                 // Pointer to link node, allocated in-memory
   };
} S_OMAP_VLINK;                                 // end of struct "s_omap_vlink"

#define OMSF_SNAPSHOT_DELETED        0x0001LL   // Snapshot has been deleted
#define OMSF_SNAPSHOT_REVERTED       0x0002LL   // Snapshot has been reverted
#define OMSF_VALIDMASK               0x0003LL   // defined flags mask

typedef struct s_omap_snapshot
{
   ULONG               omsFlags;                // Snapshot flags
   ULONG               omsPad;                  // Padding,   reserved (zero)
   ULN64               omsOid;                  // Object ID, reserved (zero)
} S_OMAP_SNAPSHOT;                              // end of struct "s_omap_snapshot"

#define V_I_NUM        8                        // number of Volume-info strings
#define V_I_LEN        0x20                     // length of Volume-info strings
#define V_N_LEN        0x100                    // length of Volume-name (256)

typedef struct s_v_id_info                      // Volume ID info element
{
   char                vsIdString[V_I_LEN];     // +00 ID-string
   ULN64               vsIdTimestamp;           // +20 ID-timestamp
   ULN64               vsIdVersion;             // +28 ID-generation/version
} S_V_ID_INFO;                                  // end of struct "s_v_id_info"

typedef struct s_v_meta_crypto
{
   USHORT              mcMajorVer;              // 000 major version
   USHORT              mcMinorVer;              // 002 minor version
   ULONG               mcCryptoFlags;           // 004 encryption state flags
   ULONG               mcPersistentClass;       // 008 protection class for key
   ULONG               mcKeyOsVersion;          // 00C maj-min-build coded OS version for key
   USHORT              mcKeyRevision;           // 010 Key revision number
   USHORT              mcUnused;                // 012 Unused
} S_V_META_CRYPTO;                              // 014 end of struct "s_v_meta_crypto"

//- Volume feature flag definitions
#define VFF_OLDDEFRAG                0x0001LL   // pre-release defragging                (10.13 only)
#define VFF_HARDLINKS                0x0002LL   // volume has hardlink map
#define VFF_DEFRAG                   0x0004LL   // defragging supported             (10.14 and later)
#define VFF_VALIDMASK                0x0007LL   // defined flags mask

//- Volume RO compatible features
#define VFR_VALIDMASK                0x0000LL   // defined flags mask

//- Volume incompatible features
#define VFI_CASE_INSENS              0x0001LL   // Filenames are case-insensitive
#define VFI_DATALESS_SNAPS           0x0002LL   // at least one snapshot without data exists
#define VFI_ENCRYPT_ROLLED           0x0004LL   // encryption keys have been rolled at least once
#define VFI_NORMALI_INSENS           0x0008LL   // Filenames are normalization insensitive (language)
#define VFI_VALIDMASK                0x000FLL   // defined flags mask

//- Volume Flags
#define VFL_UNENCRYPTED              0x0001LL   // Volume is not encrypted
#define VFL_EFFACEABLE               0x0002LL   // Reserved flag, crypto related
#define VFL_RESERVED_4               0x0004LL   // Reserved flag, undocumeneted
#define VFL_ONE_KEY                  0x0008LL   // All files encrypted using VEK
#define VFL_SPILLEDOVER              0x0010LL   // Vol ran out of space on SSD (Fusion ?)
#define VFL_SPOVERCLEAN              0x0020LL   // Vol spilled-over, cleaner must be run
#define VFL_CHECKEXTENTREF           0x0040LL   // Alway check extent-ref tree when overwriting extent
#define VFL_VALIDMASK                0x007FLL   // defined flags mask

//- Defined Volume Roles                        // multiple roles are possible!
#define V_ROLE_NONE                  0x0000     // no defined role (seen on external data volumes)
#define V_ROLE_SYSTEM                0x0001     // FS mounted as the systems root '/'
#define V_ROLE_USER                  0x0002     // FS with user home directories  '/Users'
#define V_ROLE_RECOVERY              0x0004     // FS contains a recovery system
#define V_ROLE_VM                    0x0008     // FS contains virual memory swap space
#define V_ROLE_PREBOOT               0x0010     // FS contains files need to boot with encryption
#define V_ROLE_INSTALLER             0x0020     // FS is used by the OS installer
#define V_ROLE_DATA                  0x0040     // FS used for user data                 (iOS only)
#define V_ROLE_BASEBAND              0x0080     // FS contains Radio config data         (iOS only)
#define V_ROLE_RESERVED              0x0200     // Reserved by Apple


#define SG_V_SUPER      4
#define SV_V_SUPER      (char)0x41,(char)0x50,(char)0x53,(char)0x42
typedef struct s_v_super                        // APFS volume super-block
{
   S_B_HEADER          blockHdr;                // 000 Generic block header
   char                Signature[SG_V_SUPER];   // 020 Signature 'APSB'
   ULONG               vsVolumeIndex;           // 024 Index of this volume in container-SB
   ULN64               vsFeatures;              // 028 Feature flags
   ULN64               vsRoCompatible;          // 030 RO Compatible features
   ULN64               vsInCompatible;          // 038 Incompatible  features
   ULN64               vsUnMountTime;           // 040 Timestamp last unmount
   ULN64               vsBlocksReserved;        // 048 Blocks resereved for this volume (min)
   ULN64               vsBlockQuota;            // 050 Blocks quoata    for this volume (max)
   ULN64               vsBlocksAllocated;       // 058 Blocks currently allocated
   S_V_META_CRYPTO     vsMetaCryptoInfo;        // 060 Info on ecryption key for MetaData
   ULONG               vsRootTreeType;          // 074 Object type info on ROOT tree
   ULONG               vsExtentTreeType;        // 078 Object type info on EXTENTS tree
   ULONG               vsSnapMtTreeType;        // 07C Object type info on SNAPSHOT-META tree
   ULN64               vsObjectMapOid;          // 080 Phys Object Map ID       (blocknumber)
   ULN64               vsRootTreeOid;           // 088 Virt ID for Root tree
   ULN64               vsExtentsTreeOid;        // 090 Phys Extent-tree ID      (blocknumber)
   ULN64               vsSnapMetaTreeOid;       // 098 Phys ID for Snapshot-metadata tree
   ULN64               vsRevertToXid;           // 0A0 Revert-to transaction ID
   ULN64               vsRevertToVsbOid;        // 0A8 Phys ID vSB to revert to (blocknumber)
   ULN64               vsNextOid;               // 0B0 Next virt/eph ID to use  (next inum)
   ULN64               vsFileCount;             // 0B8 Number of files on volume
   ULN64               vsDirCount;              // 0C0 Number of directories on volume
   ULN64               vsSymLinkCount;          // 0C8 Number of symbolic-links on volume
   ULN64               vsOtherFsObjCount;       // 0D0 Number of 'other' FS-objects
   ULN64               vsSnapshotCount;         // 0D8 Number of snapshots in volume
   ULN64               vsTotalBlocksAlloc;      // 0E0 Total number of blocks allocated, ever
   ULN64               vsTotalBlocksFreed;      // 0E8 Total number of blocks freed,     ever
   DFS_GUID            vsVolumeGuid;            // 0F0 128-bit uuid (4-2-2-2-6)
   ULN64               vsLastMountTime;         // 100 Last mount timestamp (last modify)
   ULN64               vsVolumeFlags;           // 108 VolumeFlags, see defines above
   S_V_ID_INFO         vsFormattedBy;           // 110 Info on formatting software
   S_V_ID_INFO         vsModHistory[ V_I_NUM];  // 110 Array of modified-by history info
   char                vsVolumeName[V_N_LEN];   // 2C0 Volume Name (UTF-8)
   ULONG               vsNextDocumentId;        // 3C0 Next DOC-ID to be used
   USHORT              vsVolumeRole;            // 3C4 Specific Role for the volume
   USHORT              vsReserved;              // 3C6 reserved (zero)
   ULN64               vsRootToXid;             // 3C8 Xid of snapshot to root from, or 0
   ULN64               vsErStateOid;            // 3D0 Current state of encryption,  or 0
   ULN64               vsUnknownValue;          // 3D8 Unknown value (seen: 0x00000000000000F2)
   ULN64               vsLastMountXid;          // 3E0 Xid when FS was last mounted, shows #changes
} S_V_SUPER;                                    // end of struct "s_v_super"


#define                SPD_MAIN         0       // Fusion main device, SSD (or only one)
#define                SPD_TIER2        1       // Fusion secondary device, rotating disk
#define                SPD_COUNT        2       // Fusion number of devices (spacemanager)

#define SPM_CI_COUNT_MASK    0x000fffff         // valid bits in CI count
#define SPM_CI_C_RES_MASK    0xfff00000         // reserved bits in CI count

//- device specific space manager info (2 devices, for Fusion drives)
typedef struct s_spmgr_device
{
   ULN64               spdBlockCount;           // 000 block count
   ULN64               spdChunkCount;           // 008 chunk count
   ULONG               spdCibCount;             // 010 Number of info-blocks
   ULONG               spdCabCount;             // 014 Number of address-blocks
   ULN64               spdFreeCount;            // 018 Free blocks count
   ULONG               spdIndexOffset;          // 020 Offset to CIB/CAB block# array
   ULONG               spdReserved1;            // 024 Reserved value (0)
   ULN64               spdReserved2;            // 028 Reserved value (0)
} S_SPMGR_DEVICE;                               // 030 end of struct "s_spmgr_device"


#define                SFQ_IPOOL        0       // Free-Queue Internal Pool
#define                SFQ_MAIN         1       // Free-Queue main device, SSD
#define                SFQ_TIER2        2       // Free-Queue secondary device, rotating disk
#define                SFQ_COUNT        3       // Free-Queue count

typedef struct s_spmgr_freeq
{
   ULN64               sfqCount;                // 000 Count (of blocks ?)
   ULN64               sfqTreeOid;              // 010 Tree object ID
   ULN64               sfqOldestXid;            // 008 Oldest Xid in tree
   USHORT              sfqNodeLimit;            // 018 Tree Node Limit
   USHORT              sfqReserved1;            // 01a Reserved value 1
   ULONG               sfqReserved2;            // 01c Reserved value 2
   ULN64               sfqReserved3;            // 020 Reserved value 3
} S_SPMGR_FREEQ;                                // 028 end of struct "s_spmgr_freeq"

typedef struct s_spmgr_azone_bound              // alloc-zone-boundaries
{
   ULN64               azbStart;
   ULN64               azbEnd;
} S_SPMGR_AZONE_BOUND;                          // end of struct "s_spmgr_azone_bound"

#define AZI_INVALID_END                 0       // invalid end boundary
#define AZI_PREV_BSIZE                  7       // number or previous boundaries
typedef struct s_spmgr_azone_info               // alloc-zone info
{
   S_SPMGR_AZONE_BOUND aziCurrentBoundaries;                //- 000 current  azone boundaries
   S_SPMGR_AZONE_BOUND aziPrevBoundaries[ AZI_PREV_BSIZE];  //- 010 previous azone boundaries
   USHORT              aziZoneId;               // 080 zone ID
   USHORT              aziPrevBoundIndix;       // 082 previous boundary index
   ULONG               aziReserved;             // 084 reserved
} S_SPMGR_AZONE_INFO;                           // 088 end of struct "s_spmgr_azone_info"

#define DZI_AZONE_COUNT                 8       // number of azone-info structs per device

typedef struct s_spmgr_dzone_info               // data-zone info (2dim array of azone-info)
{
   S_SPMGR_AZONE_INFO  dziAzone[ SPD_COUNT][ DZI_AZONE_COUNT]; // data-zone info array of a-zone info
} S_SPMGR_DZONE_INFO;                           // 880 end of struct "s_spmgr_dzone_info"


#define SPF_VERSIONED        0x00000001
#define SPF_VALIDMASK        0x00000001         // defined flags mask

#define SPM_CI_COUNT_MASK    0x000fffff         // valid bits in CI count
#define SPM_CI_C_RES_MASK    0xfff00000         // reserved bits in CI count

// Note: Fields at 098 0A0 and 0A4 differ from APPLE spec, since HEXDUMP shows different order!

typedef struct s_spacemanager
{
   S_B_HEADER          blockHdr;                // 000 Generic block header
   ULONG               spBlockSize;             // 020 Blocksize (bytes ?)
   ULONG               spBlocksPerChunk;        // 024 Number of blocks per chunk
   ULONG               spChunksPerCib;          // 028 Number of chunks per cib
   ULONG               spCibsPerCab;            // 02C Number of cibs   per cab
   S_SPMGR_DEVICE      spDev[ SPD_COUNT];       // 030 Device specific info array
   ULONG               spFlags;                 // 090 Spacemanager flags
   ULONG               spIpTxMultiplier;        // 094 Internal Pool TX multiplier
   ULN64               spReaperObjectBlock;     // 098 Seems to be a Reaper object
   ULONG               spIpBlockCount;          // 0A0 Internal Pool Block Count
   ULONG               spIpBmBlockCount;        // 0A4 IP Bitmap Block Count
   ULN64               spIpBmBaseBlock;         // 0A8 IP Bitmap base block
   ULN64               spIpBaseBlock;           // 0B0 Internal Pool base block ???
   ULN64               spFsReserveBlockCount;   // 0B8 FS reserved  blocks ?
   ULN64               spFsReserveAllocCount;   // 0C0 FS allocated blocks ?
   S_SPMGR_FREEQ       spFreeQ[ SFQ_COUNT];     // 0C8 Free-Queue info array
   USHORT              spIpBmFreeHead;          // 140 Head index in circular buffer ?
   USHORT              spIpBmFreeTail;          // 142 Tail index (perhaps buffer 0x160)
   ULONG               spIpBmXidOffset;         // 144 Offset for Int-Pool bitmap Xid
   ULONG               spIpBitMapOffset;        // 148 Offset for Int-Pool bitmap itself
   ULONG               spIpBmFreeNextOffset;    // 14C Offset for Int-Pool next bitmap
   ULONG               spVersion;               // 150 Versioning info
   ULONG               spBaseSize;              // 154 Structure base size (rest dynamic)
   S_SPMGR_DZONE_INFO  spDataZone;              // 158 Data zone array of alloc-zones
} S_SPACEMANAGER;                               // 9D8 end of struct "s_spacemanager"

// Note: Spacemanager uses the space after 9D0 for various info (arrays) accesed using offsets
//       from the start of the block. One example is the list of CIB/CAB blocks to the bitmaps


// a CAB is only used for huge filesystems (terrabytes) as a top-level above CIB's
typedef struct s_sp_cab_block
{
   S_B_HEADER          blockHdr;                // 000 Generic block header
   ULONG               cabIndex;                // 020 Index of this CAB
   ULONG               cabCibCount;             // 024 Number of CIB block numbers
   ULN64               cabCibBlock[1];          // 028 Array  of CIB block numbers
} S_SP_CAB_BLOCK;                               // end of struct "s_sp_cab_block"

// a CIB is used to build a SPARSE table of BITMAP blocks, where ZERO is an unused block range
typedef struct s_sp_cib_entry
{
   ULN64               cieXid;                  // 000 Xid for last change to this bitmap
   ULN64               cieAddr;                 // 008 First Block# represented in bitmap
   ULONG               cieBlockCount;           // 010 Number of blocks in the bitmap
   ULONG               cieFreeCount;            // 014 Blocks still free in this range
   ULN64               cieBmAddr;               // 018 Bitmap Block# 0=unused (sparse!)
} S_SP_CIB_ENTRY;                               // end of struct "s_sp_cib_entry"

typedef struct s_sp_cib_block
{
   S_B_HEADER          blockHdr;                // 000 Generic block header
   ULONG               cibIndex;                // 020 Index of this CIB 0..n
   ULONG               cibEntryCount;           // 024 Number of Bitmap entries
   S_SP_CIB_ENTRY      cibEntry[1];             // 028 CIB entries with bitmap info
} S_SP_CIB_BLOCK;                               // end of struct "s_sp_cib_block"


//------------------------------ FS BTREE --------------------------------------

#define BT_SANE_LEVELS 9                        // reaches 10^14 nodes with 36 records/node

#define BNF_ROOT                     0x0001     // node is a ROOT node
#define BNF_LEAF                     0x0002     // node is a LEAF node
#define BNF_FIXED_SIZE               0x0004     // key and value are fixed size
#define BNF_KOFF_INVAL               0x8000     // offsets are invalid (in memory only)
#define BNF_VALIDMASK                0x8007     // Recognized flag bits

typedef struct s_n_loc                          // Node internal location structure
{
   USHORT              Offset;
   USHORT              Length;
} S_N_LOC;                                      // end of struct "s_n_loc"

typedef struct s_btree_node                     // start of any node
{
   USHORT              btFlags;                 // 020 Flags, bit 04 always set
   USHORT              btLevel;                 // 022 tree level
   ULONG               btCount;                 // 024 number of entries
   S_N_LOC             btToc;                   // 028 Location of the TOC, from btData
   S_N_LOC             btFree;                  // 02C Location of Freespace
   S_N_LOC             btKfreeList;             // 030 Linked list of free key-slots
   S_N_LOC             btVfreeList;             // 034 Linked list of free value-slots
} S_BTREE_NODE;                                 // 038 end of struct "s_btree_node"


#define BTOFF_INVALID                0xFFFF     // invalid offset (ghost key, no value)

#define BTF_UINT64_KEYS          0x00000001     // 64-bit optimization hint
#define BTF_SEQ_INSERT           0x00000002     // Sequential insert optimization hint
#define BTF_ALLOW_GHOSTS         0x00000004     // allow keys with no value (BTOFF_INVALID)
#define BTF_EPHEMERAL            0x00000008     // Btree uses ephemeral links to nodes
#define BTF_PHYSICAL             0x00000010     // Btree uses physical  links to nodes
#define BTF_NONPERSISTENT        0x00000020     // Btree not persistent across unmount
#define BTF_KV_NONALIGNED        0x00000040     // Keys/Values NOT aligned to 8 bytes
#define BTF_VALIDMASK            0x0000007F     // Recognized flag bits

typedef struct s_btree_info                     // end of root-node (btree_info_t)
{
   ULONG               bfFlags;                 // fd8 Tree level flags
   ULONG               bfNodeSize;              // fdc size of node (blocksize 0x1000)
   ULONG               bfMinKeySize;            // fe0 fixed key     size, or 0
   ULONG               bfMinValSize;            // fe4 fixed value   size, or 0
   ULONG               bfMaxKeySize;            // fe8 longest key   size, or 0
   ULONG               bfMaxValSize;            // fec longest value size, or 0
   ULN64               bfEntryCount;            // ff0 Btree number of entries
   ULN64               bfNodeCount;             // ff8 Btree number of nodes
} S_BTREE_INFO;                                 // end of struct "s_btree_info"

//- below construct limits usage to nodes of 4096 bytes only, but that is all that seems to be used
//- Since the NodeSize is only present in the INFO at the end, it is hard to get to (need to probe)
//- After reading a 4096 size TREE-ROOT-NODE, and info is NOT seen at the end, node may be larger ...
#define BTN_DATA_SIZE (APFS_BLOCKSIZE - sizeof(S_B_HEADER) - sizeof(S_BTREE_NODE) - sizeof(S_BTREE_INFO))
typedef struct s_bt_node                        // any node, access to header/footer
{
   S_B_HEADER          blockHdr;                // 000 Generic block header
   S_BTREE_NODE        btNode;                  // 020 Btree generic node header
   BYTE                btData[ BTN_DATA_SIZE];  // 034 Variable: TOC, Keys, Freesp, Values
   S_BTREE_INFO        btInfo;                  // fd8 Tree-level info (TREE node only)
} S_BT_NODE;                                    // end of struct "s_bt_node"


typedef struct s_bt_entry                       // standard entry (kvloc_t)
{                                               // used for variable size key/value
   USHORT              nKoffset;                // 000 offset = base  + KeyOffset
   USHORT              nKlength;                // 002 length of key
   USHORT              nVoffset;                // 004 offset = END  - ValueOffset
   USHORT              nVlength;                // 006 length of value
} S_BT_ENTRY;                                   // end of struct "s_bt_entry"

typedef struct s_bt_fixed                       // fixed entry, fixed key/value size
{
   USHORT              nKoffset;                // 000 offset = base  + KOffset
   USHORT              nVoffset;                // 002 offset = END   - VOffset
} S_BT_FIXED;                                   // end of struct "s_bt_fixed"


typedef struct s_bitmap_ptr
{
   ULN64               bpVersion;
   ULN64               bpOffset;
   ULONG               bpBitsTotal;             // total bits in bitmap
   ULONG               bpBitsAvail;             // available bits (free ?)
   ULN64               bpBlock;                 // block number this bitmap
} S_BITMAP_PTR;                                 // end of struct "s_bitmap_ptr"


// FS (virtual) OID and TYPE masks, for manipulation of hybrid FS key values
#define APFSI_MASK             0x0fffffffffffLL
#define APFST_MASK             0xf00000000000LL
#define APFST_SHIFT                          60

// FS TYPE values
#define APFST_FIRST                  0x0
#define APFST_SNAP_METADATA          0x1
#define APFST_PHYS_EXTENT            0x2
#define APFST_INODE                  0x3
#define APFST_XATTR                  0x4
#define APFST_SIBLING_LINK           0x5
#define APFST_DSTREAM_ID             0x6
#define APFST_CRYPTO_STATE           0x7
#define APFST_FILE_EXTENT            0x8
#define APFST_DIR_RECORD             0x9
#define APFST_DIR_STATS              0xA
#define APFST_SNAP_NAME              0xB
#define APFST_SIBLING_MAP            0xC
#define APFST_COUNT                  0xD

// Quick access macros from fsId value (i)
#define APFSidType(i) ((int)(((ULN64)(i)) >> APFST_SHIFT))
#define APFSidPrim(i)       (((ULN64)(i))  & APFSI_MASK )
#define APFSmkFsId(p,t)     (((ULN64)(t)  << APFST_SHIFT) | (p))

// Note: For many FS-objects, the key is a single ULN64, with TYPE and INODE number
//       For Directory records it is the PARENT directory inode plus a name (hash)
//       The TYPE is the SECONDARY key, and optional parts of the key are TERTAIARY


//-------------------------------- XFIELDS -------------------------------------

// Extended fields (optional, variable) for FS Inode and Directory records

#define XFT_FIRST                   0x00        // Not an extended field
#define XFT_DREC_SIBLING_ID         0x01        // sibling ID, used on hardlinks
#define XFT_INO_SNAPSHOT_ID         0x01        // Xid for a snapshot
#define XFT_INO_DELTATREEID         0x02        // Virtual OID snapshot delta-extents
#define XFT_INO_DOCUMENT_ID         0x03        // Unique document identifier
#define XFT_INO_FILENAME            0x04        // Filename (only for hardlinks ?)
#define XFT_INO_PREV_FSIZE          0x05        // Previous filesize (crash recover)
#define XFT_INO_RESERVED_6          0x06        // Reserved
#define XFT_INO_FINDERINFO          0x07        // Opaque finder indo (32 bytes)
#define XFT_INO_DATASTREAM          0x08        // Data Stream, including size in bytes
#define XFT_INO_RESERVED_9          0x09        // Reserved
#define XFT_INO_DIRSTAT_KEY         0x0A        // Key to Directory statistics Record
#define XFT_INO_FS_UUID             0x0B        // UUID of FS auto-mounted in this DIR
#define XFT_INO_RESERVED_C          0x0C        // Reserved
#define XFT_INO_SPARSEBYTES         0x0D        // Nr of sparse bytes in the stream
#define XFT_INO_RDEVICE_ID          0x0E        // Device identifier CHR/BLK special device
#define XFT_COUNT                   0x0F        // Number of Xfield types


typedef struct s_xf_field                       // extended field meta-data
{                                               // (actual data is outside this struct)
   BYTE                xfType;                  // extended field type
   BYTE                xfFlags;                 // extended field flags
   USHORT              xfSize;                  // extended field size in bytes (NET size)
} S_XF_FIELD;                                   // end of struct "s_xf_field"

// allocated size is always a multiple of 8 for alignment (first starts directly after aray)
// So when calculating the start of the next Xfield, adjust size from array: (s + 7) & fff8)

typedef struct s_xf_blob                        // extended-field blob
{
   USHORT              xfExtents;               // number of XF extents
   USHORT              xfUsedData;              // number of bytes used
   S_XF_FIELD          xfData[];                // extended field array, followed by real data
} S_XF_BLOB;                                    // end of struct "s_xf_blob"

//-------------------------------- DSTREAM -------------------------------------

// Data-stream structure used for XATTR Dstream and Inode Xfield dstream

typedef struct s_dstream
{
   ULN64               dsByteSize;              // Stream data size in bytes
   ULN64               dsAllocSize;             // Allocated data size in bytes
   ULN64               dsCryptoId;              // default crypto ID
   ULN64               dsBytesWritten;          // total bytes written on stream
   ULN64               dsBytesRTead;            // total bytes read on stream
} S_DSTREAM;                                    // end of struct "s_dstream"

typedef struct s_xattr_dstream
{
   ULN64               xdOid;                   // XATTR object ID
   S_DSTREAM           xdStream;                // XATTR data stream
} S_XATTR_DSTREAM;                              // end of struct "s_xattr_dstream"



//------------------------- SF INODE ------------------------------------------

#define APFS_ROOT_DIR_PARENT              1
#define APFS_ROOT_DIR_INODE               2
#define APFS_PRIV_DIR_INODE               3
#define APFS_SNAP_DIR_INODE               6
#define APFS_USER 1ST_INODE              16


typedef struct s_inode_key                      // INODE
{
   ULN64               fsId;                    // FS object-id and type (INO)
} S_INODE_KEY;                                  // end of struct "s_inode_key"

typedef struct s_inode_val                      // INODE value
{
   ULN64               inParentId;              // 000 Parent directory ID     (up-link)
   ULN64               inObjectId;              // 008 OID for the data stream (a phys-extent)
   ULN64               inCreTime;               // 010 Time record (file/dir) was created
   ULN64               inModTime;               // 018 Last record modification time
   ULN64               inChgTime;               // 020 Last Attributes change time
   ULN64               inAccTime;               // 028 Last access time
   ULN64               inNodeFlags;             // 030 Inode internal flags
   ULN64               inRefCount;              // 038 #entries for DIR, #links for FILE
   ULONG               inProtClass;             // 040 Default Protection Class (encryption)
   ULONG               inBsdFlags;              // 044 BSD-style flags; (values: see stat.h)
   ULONG               inUid;                   // 048 user-ID
   ULONG               inGid;                   // 04C group-ID
   USHORT              inMode;                  // 050 file/dir mode
   USHORT              inPad1;                  // 052 reserved (pad)
   ULN64               inPad2;                  // 054 reserved (pad)
   BYTE                inXfields[];             // 05C Inode extended fields (variable)
} S_INODE_VAL;                                  // end of struct "s_inode_value"


//------------------------- SF DIR RECORD -------------------------------------

typedef struct s_drec_key                       // DIR_REC FS directory key (no hash)
{
   ULN64               fsId;                    // FS object-id and type
   USHORT              drNameLength;            // Length of key (dir/file) name
   BYTE                drName[ TXMAXLN];        // Name, UTF8
} S_DREC_KEY;                                   // end of struct "s_drec_key"

#define DRH_LENGTH_MASK        0x000003FF
#define DRH_HASH_MASK          0xFFFFFC00
#define DRH_HASH_SHIFT                 10
#define DRHash(h) ((h) >> DRH_HASH_SHIFT)

typedef struct s_hdrec_key                      // DIR_REC FS hashed directory key
{
   ULN64               fsId;                    // FS object-id and type (PARENT)
   ULONG               drLengthHash;            // FS combined name length and hash
   BYTE                drName[ TXMAXLN];        // Name, UTF8
} S_HDREC_KEY;                                  // end of struct "s_hdrec_key"

#define DRF_UNKNOWN          0x0000             // unknown type
#define DRF_FIFO             0x0001             // named pipe
#define DRF_CHR              0x0002             // character special device
#define DRF_DIR              0x0004             // directory
#define DRF_BLK              0x0006             // block special device
#define DRF_REG              0x0008             // regular file
#define DRF_LNK              0x000A             // symbolic link
#define DRF_SOCK             0x000C             // socket
#define DRF_WHT              0x000E             // whiteout
#define DRF_MASK             0x000F

typedef struct s_drec_val                       // FS directory value
{
   ULN64               fileId;                  // 000 ID of the Inode for this file/dir
   ULN64               drDateAdded;             // 008 Date/time this dir entry was added
   USHORT              drFlags;                 // 010 dir entry flags (DIR/file/Link/CHR/BLK etc)
   BYTE                drXfields[];             // 012 DREC extended fields (variable)
} S_DREC_VAL;                                   // end of struct "s_drec_val"


//------------------------- SF DIR STATS --------------------------------------

typedef struct s_dstats_key                     // DIR STATS
{
   ULN64               fsId;                    // FS object-id and type
} S_DSTATS_KEY;                                 // end of struct "s_dstats_key"

typedef struct s_dstats_val
{
   ULN64               dsNumChildren;           // 000 number of files/dirs contained
   ULN64               dsTotalSize;             // 008 size in bytes taken up by tree
   ULN64               dsParentDir;             // 010 fsId of parent directory (up link)
   ULN64               dsGenCount;              // 018 Generation Count (writes)
} S_DSTATS_VAL;                                 // 020 end of struct "s_dstats_val"


//------------------------- SF XATTR ------------------------------------------

typedef struct s_xattr_key                      // XATTR
{
   ULN64               fsId;                    // FS object-id and type
   USHORT              xaNameLength;            // Length of key (xattr) name
   BYTE                xaName[ TXMAXLN];        // Name, UTF8
} S_XATTR_KEY;                                  // end of struct "s_xattr_key"

#define XAF_DATA_STR               0x0001       // XA value is in data-stream (extent)
#define XAF_EMBEDDED               0x0002       // XA value is embedded in this record
#define XAF_SYSOWNED               0x0004       // XA value is owned by FS (like symlinks)
#define XAF_RESERVED               0x0008       // XA reserved and preserve when set

#define XA_MAX_EMBED_SIZE            3804       // max size of embedded XA value
#define XA_SYMLINK_NAME "com.apple.fs.symlink"

typedef struct s_xattr_val
{
   USHORT              xaFlags;                 // 000 xattr flags
   USHORT              xaLength;                // 002 xattr data length
   BYTE                xaData[];                // 004 xattr data
} S_XATTR_VAL;                                  // VAR end of struct "s_xattr_val"


//------------------------- SF EXTENT -----------------------------------------

typedef struct s_phys_ext_key                   // EXTENT
{
   ULN64               fsId;                    // FS object-id and type
} S_PHYS_EXT_KEY;                               // end of struct "s_phys_ext_key"

typedef struct s_phys_ext_val
{
   ULN64               peLengthKind;            // 000 Length and Kind of extent
   ULN64               peOwningOid;             // 008 Owning object ID
   ULONG               peRefCount;              // 010 Reference count
} S_PHYS_EXT_VAL;                               // 014 end of struct "s_phys_ext_val"

// PHYS EXTENT LENGTH and KIND masks
#define APFSPL_MASK            0x0fffffffffffLL
#define APFSPK_MASK            0xf00000000000LL
#define APFSPK_SHIFT                         60

//------------------------- SF FILE EXTENT ------------------------------------

typedef struct s_file_ext_key                   // FILE EXTENT
{
   ULN64               fsId;                    // FS object-id and type (INO)
   ULN64               feLogAddress;            // relative byte address of extent in file
} S_FILE_EXT_KEY;                               // end of struct "s_file_ext_key"

typedef struct s_file_ext_val
{
   ULN64               feLengthFlags;           // 000 Length and flags for extent
   ULN64               feBlockNr;               // 008 Physical block number
   ULN64               feCryptoId;              // 010 Encryption key or tweak
} S_FILE_EXT_VAL;                               // end of struct "s_file_ext_val"

// FILE EXTENT LENGTH and FLAGS masks
#define APFSFL_MASK            0x00ffffffffffLL
#define APFSFK_MASK            0xff0000000000LL
#define APFSFK_SHIFT                         56


//------------------------- SF DSTREAM ID -------------------------------------
typedef struct s_dstream_id_key                 // DSTREAM ID
{
   ULN64               fsId;                    // FS object-id and type (INO)
} S_DSTREAM_ID_KEY;                             // end of struct "s_dstream_id_key"

typedef struct s_dstream_id_val
{
   ULONG               diRefCount;              // 000 Datastream reference count
} S_DSTREAM_ID_VAL;                             // end of struct "s_dstream_id_val"


//------------------------- SF SIBLING ---------------------------------------

// Link an INODE to all hardlinks that refer to it, lowest sibId is the PRIMARY
typedef struct s_siblink_key                    // SIBLING LINK key
{
   ULN64               fsId;                    // FS object-id and type
   ULN64               slSibId;                 // sibling ID (INO alias)
} S_SIBLINK_KEY;                                // end of struct "s_siblink_key"

typedef struct s_siblink_val                    // SIBLING LINK
{
   ULN64               slParentIno;             // Parent INO (where hardlink lives)
   USHORT              slNameLength;            // Length of key (hardlink) name
   BYTE                slName[];                // Name of the hardlink, UTF8
} S_SIBLINK_VAL;                                // end of struct "s_siblink_val"


// Maps an sibId to the INODE it represents
typedef struct s_siblmap_key                    // SIBLING MAP key
{
   ULN64               fsId;                    // FS object-id and type (matches sibId)
} S_SIBLMAP_KEY;                                // end of struct "s_siblmap_key"

typedef struct s_siblmap_val                    // SIBLING LINK
{
   ULN64               spFileId;                // Inode number underlying file
} S_SIBLMAP_VAL;                                // end of struct "s_siblmap_val"

//------------------------- SF SNAPMETA ---------------------------------------

typedef struct s_snapmeta_key                   // SNAPMETA key
{
   ULN64               fsId;                    // FS object-id and type (snapshot XID)
} S_SNAPMETA_KEY;                               // end of struct "s_snapmeta_key"

#define SMF_SNAP_PENDING_DATALESS   0x00000001

typedef struct s_snapmeta_val
{
   ULN64               smExtTreeOid;            // 000 Blocknr of B-tree storing extents
   ULN64               smVolSuper;              // 008 Blocknr of Volume superblock
   ULN64               smCreateTime;            // 010 Snapshot creation timestamp
   ULN64               smChangeTime;            // 018 Snapshot last modification time
   ULN64               smInum;                  // 020 Unknown purpose
   ULONG               smExtTreeType;           // 028 Type of B-tree with extent info
   ULONG               smFlags;                 // 02C Snapshot metadata flags
   USHORT              smNameLength;            // 030 Length of  (snapshot) name
   BYTE                smName[];                // 032 Name of the snapshot, UTF8
} S_SNAPMETA_VAL;                               // VAR end of struct "s_snapmeta_val"


//------------------------- SF SNAPNAME ---------------------------------------

typedef struct s_snapname_key                   // SNAPNAME key
{
   ULN64               fsId;                    // FS object-id and type
   USHORT              snNameLength;            // Length of key (snapshot) name
   BYTE                snName[  TXMAXLN];       // Name, UTF8
} S_SNAPNAME_KEY;                               // end of struct "s_snapname_key"

typedef struct s_snapname_val
{
   ULN64               snXid;                   // 000 Last XID included in snapshot
} S_SNAPNAME_VAL;                               // VAR end of struct "s_snapname_val"



//------------------------- NODE KEY / VAL PTR UNIONS -------------------------

typedef union s_node_key
{
   void               *data;
   ULN64              *p64;                     // pointer to 64-bit, primary + secondary key
   S_INODE_KEY        *in;
   S_DREC_KEY         *dr;
   S_HDREC_KEY        *dh;
   S_DSTATS_KEY       *ds;
   S_XATTR_KEY        *xa;
   S_DSTREAM_ID_KEY   *di;
   S_PHYS_EXT_KEY     *pe;
   S_FILE_EXT_KEY     *fe;
   S_SIBLINK_KEY      *sl;
   S_SIBLMAP_KEY      *sp;
   S_SNAPMETA_KEY     *sm;
   S_SNAPNAME_KEY     *sn;
} S_NODE_KEY;                                     // end of union "s_node_key"

typedef union s_node_val
{
   void               *data;                    // any pointer type
   ULN64              *p64;                     // pointer to 64-bit, like a (virtual) link
   S_INODE_VAL        *in;
   S_DREC_VAL         *dr;
   S_DSTATS_VAL       *ds;
   S_XATTR_VAL        *xa;
   S_DSTREAM_ID_VAL   *di;
   S_PHYS_EXT_VAL     *pe;
   S_FILE_EXT_VAL     *fe;
   S_SIBLINK_VAL      *sl;
   S_SIBLMAP_VAL      *sp;
   S_SNAPMETA_VAL     *sm;
   S_SNAPNAME_VAL     *sn;
} S_NODE_VAL;                                     // end of union "s_node_val"

#endif
