//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// DFS display and analysis, NTFS specific functions and definitions
//
// Author: J. van Wijk
//
// JvW  21-05-1997 Initial version, derived from DFSHPFS
// JvW  04-04-2017 Updated LSN to 64bit, MFT-numbers to 64bit
//
#ifndef    DFSANTFS_H
   #define DFSANTFS_H

#include <dfsntfs.h>                            // NTFS structure defs

// NTFS specific sector types + ATTRIB variants
                                                // S   used on searching
                                                //  I  identification only
#define ST_S_NTLDR     's'                      //  I  fake superblock, NTLDR
#define ST_MFTFILE     'f'                      //  I  MFT rec for a file
#define ST_MFTDIRB     'D'                      // S   MFT rec for a directory
#define ST_MFTSECF     'm'                      //  I  MFT secondary rec file
#define ST_MFTSECD     'M'                      // S   MFT secondary rec dir
#define ST_MFTDELD     'y'                      // S   MFT deleted directory
#define ST_MFTDELF     'z'                      // S   MFT deleted file
#define ST_DIRINDX     'd'                      // S   Directory index
#define ST_SIIINDX    ('i' + ST__INFO)          //  I  Security SII index
#define ST_SDHINDX    ('S' + ST__INFO)          //  I  Security SDH index
#define ST_LOGRSTR     'j'                      // S   LogFile restart page
#define ST_LOGRCRD     'J'                      // S   LogFile record  page
#define ST_MFGHOST    ('Z' + ST__INFO)          //  I  MFT Ghost (outside MFT)
#define ST_MFTEMPT    ('Y' + ST__INFO)          //  I  MFT Empty (seq nr == 0)
#define ST_CLSLACK    ('c' + ST__INFO)          //  I  Cluster slack area


#define STIB_MFTFILE         0x00               // attribute offset for FILE
#define STIB_MFTDIRB         0x14               // attribute offset for DIR

#define ST_F_STANDARD (STIB_MFTFILE + 0x01)
#define ST_F_ATTRLIST (STIB_MFTFILE + 0x02)     // keep values in sync with
#define ST_F_FILENAME (STIB_MFTFILE + 0x03)     // definitions in DFSNTFS.H
#define ST_F_VERSION  (STIB_MFTFILE + 0x04)     // with values shifted >> 4
#define ST_F_SECURITY (STIB_MFTFILE + 0x05)
#define ST_F_VOLNAME  (STIB_MFTFILE + 0x06)
#define ST_F_VOLINFO  (STIB_MFTFILE + 0x07)
#define ST_F_DATA     (STIB_MFTFILE + 0x08)
#define ST_F_IDXROOT  (STIB_MFTFILE + 0x09)
#define ST_F_DIRINDX  (STIB_MFTFILE + 0x0a)
#define ST_F_BITMAP   (STIB_MFTFILE + 0x0b)
#define ST_F_SYMLINK  (STIB_MFTFILE + 0x0c)
#define ST_F_EAINFO   (STIB_MFTFILE + 0x0d)
#define ST_F_EADATA   (STIB_MFTFILE + 0x0e)
#define ST_F_PROPSET  (STIB_MFTFILE + 0x0f)
#define ST_F_LOGUSTR  (STIB_MFTFILE + 0x10)

#define ST_D_STANDARD (STIB_MFTDIRB + 0x01)
#define ST_D_ATTRLIST (STIB_MFTDIRB + 0x02)     // keep values in sync with
#define ST_D_FILENAME (STIB_MFTDIRB + 0x03)     // definitions in DFSNTFS.H
#define ST_D_VERSION  (STIB_MFTDIRB + 0x04)     // with values shifted >> 4
#define ST_D_SECURITY (STIB_MFTDIRB + 0x05)
#define ST_D_VOLNAME  (STIB_MFTDIRB + 0x06)
#define ST_D_VOLINFO  (STIB_MFTDIRB + 0x07)
#define ST_D_DATA     (STIB_MFTDIRB + 0x08)
#define ST_D_IDXROOT  (STIB_MFTDIRB + 0x09)
#define ST_D_DIRINDX  (STIB_MFTDIRB + 0x0a)
#define ST_D_BITMAP   (STIB_MFTDIRB + 0x0b)
#define ST_D_SYMLINK  (STIB_MFTDIRB + 0x0c)
#define ST_D_EAINFO   (STIB_MFTDIRB + 0x0d)
#define ST_D_EADATA   (STIB_MFTDIRB + 0x0e)
#define ST_D_PROPSET  (STIB_MFTDIRB + 0x0f)
#define ST_D_LOGUSTR  (STIB_MFTDIRB + 0x10)

#define DFS_TZ_VAR     "TZDFSEE"                // Time zone for DFSEE


typedef struct dfsntfsbmcache                   // Cluster bitmap cache
{
   DFSISPACE           All;                     // space allocation for Bitmap
   ULN64               LimitLsn;                // largest LSN inside bitmap
   ULN64               First;                   // first cached rsn
   ULN64               Size;                    // cache size in sectors
   BOOL                Dirty;                   // cache contents modified
   BYTE               *Buffer;
} DFSNTFSBMCACHE;                               // end of struct "dfsntfsbmcache"


typedef struct dfsntfsdacache                   // Directory data cache
{
   DFSISPACE           All;                     // space allocation for DIR
   ULN64               Mft;                     // MFT number for directory
} DFSNTFSDACACHE;                               // end of struct "dfsntfsdacache"


typedef struct dfsntfsmftbmap                   // Mft Bitmap cache
{
   ULONG               Size;                    // size of bitmap in bytes
   BYTE               *Data;                    // Bitmap data
} DFSNTFSMFTBMAP;                               // end of struct "dfsntfsmbcache"


typedef struct dfsantfs                         // NTFS analysis information
{
   ULN64               Clust;                   // Number of clusters
   ULONG               ClustSize;               // sectors per cluster
   ULN64               SpareLsn;                // most likely spare-boot LSN
   ULN64               MftBaseLsn;              // most likely MFT[0] LSN
   ULN64               MftRecCount;             // Number of MFT records
   USHORT              MftRecSize;              // Sectors in MFT record
   USHORT              DirRecSize;              // Sectors in DIR record
   LONG                TimeZone;                // TZ offset to GMT (minutes)
   DFSISPACE           MftAll;                  // space allocation for MFT
   DFSNTFSBMCACHE      Bm;                      // BitMap file cache
   DFSNTFSDACACHE      Da;                      // Directory Alloc cache
   DFSNTFSMFTBMAP      MftMap;                  // Bitmap for MFT record usage
   ULN64               LastDirMftnr;            // MFTnr last used Directory
} DFSANTFS;                                     // end of struct "dfsantfs"

extern DFSANTFS      *ntfs;


// Initialize NTFS filesystem analysis
ULONG dfsNtfsInit
(
   char               *fs                       // forced filesystem type
);

// Initialize NTFS filesystem analysis for Area (FS info in FDISK mode)
ULONG dfsNtfsAreaInit
(
   void
);

// Convert NTFS size-value to actual number of sectors
ULN64 dfsNtfsV2S                                // RET   nr of sectors in value
(
   ULONG               value,                   // IN    NTFS coded size value
   BYTE                csize                    // IN    ClusterSize in sectors
);

// NTFS filesystem, identify specified sector as a valid superblock
BOOL dfsNtfsIsSuperBlock                        // RET   sector is a valid sb
(
   BYTE               *sec                      // IN    sector contents
);

// NTFS filesystem, display sector constents as a superblock
void dfsNtfsDisplaySuperBlock
(
   BYTE               *sec                      // IN    sector contents
);


#endif
