//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// NTFS dump & display Analysis functions
//
// Author: J. van Wijk
//
// JvW  21-05-1997 Initial version, derived from DFSHPFS
// JvW  06-01-2000 Fixed trap on resident MFT allocations
// JvW  17-01-2000 Fixed and updated delfind for (Unicode) specific name
// JvW  09-04-2000 Added disk-key registry display (NT version only)
// JvW  26-08-2005 Added more complete NTFS resizing, $Bitmap and $BadClus
// JvW  04-04-2017 Updated LSN to 64bit, MFT-numbers to 64bit
// JvW  16-11-2017 Introduce 64bit desired-parent MFT-number in various functions
// JvW  08-04-2021 Use rc CMD_WARNING on FileSaveAs alloc errors, considered OK
//

#include <txlib.h>                              // TX library interface

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfsspace.h>                           // DFS file-space interface
#include <dfstable.h>                           // SLT utility functions
#include <dfsantfs.h>                           // NTFS display & analysis
#include <dfsuntfs.h>                           // NTFS utility functions
#include <dfslntfs.h>                           // NTFS SLT functions
#include <dfsntreg.h>                           // NT disk registry functions

char dfsNtfsSectorTypes[] =
{
   ST_S_NTLDR,                                  //  I  fake superblock, NTLDR
   ST_MFTFILE,                                  //  I  MFT rec for a file
   ST_MFTDIRB,                                  // S   MFT rec for a directory
   ST_MFTSECF,                                  //  I  MFT secondary rec file
   ST_MFTSECD,                                  // S   MFT secondary rec dir
   ST_MFTDELD,                                  // S   MFT deleted directory
   ST_MFTDELF,                                  // S   MFT deleted file
   ST_DIRINDX,                                  // S   Directory index
   ST_SIIINDX,                                  //  I  Security SII index
   ST_SDHINDX,                                  //  I  Security SDH index
   ST_LOGRSTR,                                  // S   LogFile restart page
   ST_LOGRCRD,                                  // S   LogFile record  page
   ST_MFGHOST,                                  //  I  MFT Ghost (outside MFT)
   ST_MFTEMPT,                                  //  I  MFT Empty (seq nr == 0)
   ST_CLSLACK,                                  //  I  Cluster slack area
   0                                            //     string terminating ZERO
};


static DFSANTFS    ntfs_anchor =
{
   0,                                           // Number of clusters
   8,                                           // sectors per cluster
   0,                                           // most likely spare-boot LSN
   0x600000,                                    // most likely MFT[0] LSN (XP default)
   16,                                          // number of MFT records
   2,                                           // Sectors in MFT record
   8,                                           // Sectors in DIR record
   0,                                           // TZ offset to GMT (minutes)
   { 0, NULL, 0, 0, 0, 0},                      // MFT space allocation
   {{0, NULL, 0, 0, 0, 0}, 0, 0, 0, 0, NULL},   // Bitmap file cache
   {{0, NULL, 0, 0, 0, 0}, 0},                  // Directory Allocation cache
   {0, NULL},                                   // Bitmap for MFT record usage
   MFT_ROOT
};

       DFSANTFS   *ntfs = &ntfs_anchor;

#define MFT_RELATEDS  128                       // related MFTs seen as unique

static char sg_mftfile[SG_MFTFILE] = {SV_MFTFILE};
static char sg_dirindx[SG_DIRINDX] = {SV_DIRINDX};
static char sg_logrstr[SG_LOGRSTR] = {SV_LOGRSTR};
static char sg_logrcrd[SG_LOGRCRD] = {SV_LOGRCRD};
static char sg_s_ntldr[SG_S_NTLDR] = {SV_S_NTLDR};
static char sg_s_bootm[SG_S_NTLDR] = {SV_S_BOOTM};

static  char       *ntfs_txt[] =
{
   "",
   "Active filesystem : NTFS, specific commands are:",
   "",
   " \\[path-spec]    = find and show ROOT or file/directory relative to root",
   " BOOTINI  [part] = Find the (first) boot.ini file present in the filesystem",
   " BSCLEAR         = Reset bad-sector/cluster administration to ZERO bad sectors",
   " CA   [lsn][opt] = Check Allocation integrity for (current) MFT lsn",
   " DELFIND  [name] = Find deleted files, with MFT containing (partial) name",
   " DFSNTLDR  [img] = Create compressed imagefile containing the NTLDR sectors",
   " DIRTY     [n|d] = Set NTFS $Volume 'CHKDSK required' flag to Normal or Dirty",
   " FILEFIND [name] = Find normal files,  with MFT containing (partial) name",
   " FINDROOT [n][~] = find the Root directory without using the superblock",
   "                   starting the search at LSN [n]; [~] = use 8.3 names",
   " FINDMFT   [lsn] = Locate the MFT (for FIXBOOT when no spare-copy available)",
   " FIXBOOT         = Recover NTFS bootsector from spare-copy, or create new",
   " FIXNTLDR        = Replace (the first sector of) NTLDR for an NTFS partition",
   " LABEL   [label] = Display/edit 32-char volume label in the $Volume MFT record",
   " MFT       [-v-] = Translate and display 'this' LSN as an MFT record nr",
   " MFT  mft-recnr  = Calculate LSN for MFT record-nr and perform default display",
   " MIR [mft-recnr] = Calculate LSN for MFT-mirror record-nr, do default display",
   " MFTEXT          = Find all external MFT records (continuation for base-MFT)",
   " PATH     [n][~] = Show all path-components for MFT, upto root [use 8.3] name",
   "",
   NULL
};


// Display $Volume and $logfile clean/dirty status info, update dfsa anchor
static ULONG dfsNtfsDisplayFsStatus
(
   void
);

// Extract MFT allocation S_SPACE from the MFT base record
static ULONG dfsNtfsMftAlloc                    // RET   rc = 0 if type match
(
   ULN64               lsn,                     // IN    sector number
   BYTE               *sector                   // IN    sector data
);


// Extract BitMap allocation S_SPACE structure and initialize the BitMap cache
static ULONG dfsNtfsBitMapInit                  // RET   rc = 0 if type match
(
   void
);

// Close NTFS filesystem for analysis and free any resources
static ULONG dfsNtfsClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
);

// Close NTFS filesystem for analysis and free any resources
static ULONG dfsNtfsAreaClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
);

// Interpret and execute specific NTFS command;
static ULONG dfsNtfsCommand
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *none,                    // IN    dummy
   void               *data                     // INOUT dummy
);

// NTFS filesystem, identify specified sector
static ULONG dfsNtfsIdent
(
   ULN64               di,                      // IN    LSN for sector
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // OUT   sector type
   void               *data                     // IN    sector contents
);

// NTFS filesystem, supply sector-type description string
static ULONG dfsNtfsStype
(
   ULN64               di,                      // IN    sector type
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // OUT   type description
   void               *data                     // INOUT dummy
);

// NTFS filesystem, display sector-contents based on type
static ULONG dfsNtfsSectorContents
(
   ULN64               psn,                     // IN    base psn for sector
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // IN    sector contents
);

// Display NTFS MFT record with attributes
static ULONG dfsNtfsMftFile                     // RET   rc = 0 if type match
(
   BYTE              *sector,                   // IN    disk sector
   ULN64              lsn                       // IN    LSN of Mft
);

// Display NTFS index structure for DIRECTORY or security SII and SDH data
static ULONG dfsNtfsIndxBlock                   // RET   rc = 0 if type match
(
   BYTE               *sector,                  // IN    sector data
   BYTE                stype                    // IN    sector type
);

// Display NTFS LogFile Restart Area structure
static ULONG dfsNtfsLogRestart                  // RET   rc = 0 if type match
(
   BYTE               *sector                   // IN    sector data
);

// Display NTFS LogFile Restart Area Flags value
static void dfsNtfsLraFlags
(
   USHORT              lraFlags                 // IN    flags value
);

// Display NTFS MFT record Attribute specific value part
static ULONG dfsNtfsAttribValuePart             // RET   rc = 0 if type match
(
   ULN64               mft,                     // IN    Current MFT number
   ULN64               lsn,                     // IN    Current MFT lsn
   S_MFTFILE          *sd,                      // IN    base MFT rec, fixed-up
   S_MFTATTR          *ra                       // in    resident attribute
);

// Display NTFS MFT record AttributeList, get list of related MFT's
static ULONG dfsNtfsAttributList
(
   S_MFTFILE          *sd,                      // IN    base MFT rec, fixed-up
   A_ATTRLIST         *at,                      // IN    Attribute value data
   ULONG               size,                    // IN    size of attribute
   ULN64              *rmft                     // OUT   related MFT (MFT_RELATEDS)
);

// Display NTFS MFT record Standard Attribute
static ULONG dfsNtfsAttrStandard
(
   A_STANDARD         *at                       // IN    Attribute value data
);

// Display NTFS MFT record Filename Attribute
static ULONG dfsNtfsAttrFilename
(
   ULN64               mftLsn,                  // IN    Base MFT lsn
   A_FILENAME         *at                       // IN    Attribute value data
);

// Display NTFS MFT record IdxRoot Attribute
static ULONG dfsNtfsAttrIndxRoot
(
   ULN64               mftnr,                   // IN    Current MFT number
   A_IDXROOT          *at,                      // IN    Attribute value data
   ULONG               size                     // IN    size of attribute
);

// Display NTFS MFT record VulumeName
static ULONG dfsNtfsVolumeName
(
   S_MFTFILE          *sd,                      // IN    base MFT rec, fixed-up
   A_VOLNAME          *at,                      // IN    Attribute value data
   ULONG               size                     // IN    size of attribute
);

// Display NTFS MFT record VolumeInfo
static ULONG dfsNtfsVolumeInfo
(
   S_MFTFILE          *sd,                      // IN    base MFT rec, fixed-up
   A_VOLINFO          *at,                      // IN    Attribute value data
   ULONG               size                     // IN    size of attribute
);

// Display NTFS $Volume Flags value
static void dfsNtfsVolFlags
(
   char               *lead,                    // IN    leading string
   USHORT              volFlags                 // IN    flags value
);

// Display NTFS MFT record Resident Attribute in a hex-dump
static ULONG dfsNtfsResAttribHex
(
   S_MFTFILE          *sd,                      // IN    base MFT rec, fixed-up
   void               *at,                      // IN    Attribute value data
   ULONG               size                     // IN    size of attribute
);

// DFS NTFS write-file to disk (SaveTo fnode to file)
static ULONG dfsNtfsFileSaveAs
(
   ULN64               lsn,                     // IN    MFT LSN
   ULN64               d2,                      // IN    dummy
   char               *path,                    // IN    destination path
   void               *ren                      // IN    new name (like 8.3)
);

// Make filesystem usage map dump on TxPrint output
static ULONG dfsNtfsAllocMap
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *options,                 // IN    display options
   void               *data                     // OUT   dummy
);

// Resize the filesystem, keeping all data intact; (2 sectors in br)
static ULONG dfsNtfsResizeFS
(
   ULN64               newsize,                 // IN    new size in sectors
   ULN64               d2,                      // IN    dummy
   char               *string,                  // IN    dummy
   void               *data                     // IN    partition info
);

#define NTFS_MAXBMPEXT         3                // max extents added to $Bitmap

// Update all attributes of the $Bitmap file for a new size of the filesystem
static ULONG dfsNtfsResizeBitmap                // RET   result
(
   ULN64               size                     // IN    new size in sectors
);

// Set slack bits to allocated, for clusters upto multiple-of-64 for last CL
static ULONG dfsNtfsBitmapSetSlack              // RET   result
(
   ULN64               size,                    // IN    new size in clusters
   BOOL                set                      // IN    allocation SET or reset
);

// Update all attributes of the $BadClus file for a new size of the filesystem
static ULONG dfsNtfsResizeBadClus               // RET   result
(
   ULN64               size                     // IN    new size in sectors
);


// Reset all attributes of the $BadClus file to represent ZERO bad clusters
static ULONG dfsNtfsResetBadClus                // RET   result
(
   ULN64               size                     // IN    part size in sectors
);

/*****************************************************************************/
// Initialize NTFS filesystem analysis
/*****************************************************************************/
ULONG dfsNtfsInit
(
   char               *fs                       // forced filesystem type
)
{
   ULONG               rc = NO_ERROR;
   char               *tz;
   BYTE                ClustSize;
   ULN64               lsn;
   BOOL                fixboot = FALSE;         // fixboot advise

   ENTER();

   dfsa->FsCommand          = dfsNtfsCommand;
   dfsa->FsClose            = dfsNtfsClose;
   dfsa->FsIdentifySector   = dfsNtfsIdent;
   dfsa->FsShowType         = dfsNtfsStype;
   dfsa->FsDisplaySector    = dfsNtfsSectorContents;
   dfsa->FsFileInformation  = dfsNtfsFileInfo;
   dfsa->FsGetAllocSpace    = dfsNtfsGetAllocSpace;
   dfsa->FsWriteMetaSpace   = dfsNtfsWriteMetaSpace;
   dfsa->FsFindPath         = dfsNtfsFindPath;
   dfsa->FsUpdateFileName   = NULL;
   dfsa->FsSetAltBrecLabel  = NULL;
   dfsa->FsMakeBrowseList   = dfsNtfsMakeBrowseList;
   dfsa->FsLsnAllocated     = dfsNtfsAllocated;
   dfsa->FsLsnSetAlloc      = dfsNtfsSetAlloc;
   dfsa->FsSltBuild         = dfsNtfsSltBuild;
   dfsa->FsNpBuild          = NULL;
   dfsa->FsDisplayError     = dfsNtfsDispError;
   dfsa->FsDisplayLsnInfo   = NULL;
   dfsa->FsDirIterator      = NULL;
   dfsa->FsDirFileSaveAs    = dfsNtfsFileSaveAs;
   dfsa->FsTruncateSize     = dfsNtfsResizeFS;
   dfsa->FsAllocDisplay     = dfsNtfsAllocMap;
   dfsa->FsCl2Lsn           = dfsNtfsCl2Lsn;
   dfsa->FsLsn2Cl           = dfsNtfsLsn2Cl;
   dfsa->FsLsnIterator      = dfsNtfsLsnIterator;
   dfsa->FsCmdHelp          = ntfs_txt;         // FS specific cmdhelp text

   dfsa->FsModeId           = DFS_FS_NTFS;

   dfsa->Fsi                = ntfs;
   dfsa->FsSectorTypes      = dfsNtfsSectorTypes;

   TRACES(("sizeof S_MFTFILE      = % 3u\n", sizeof(S_MFTFILE)));
   TRACES(("sizeof S_NONRES       = % 3u\n", sizeof(S_NONRES)));
   TRACES(("sizeof S_RESIDENT     = % 3u\n", sizeof(S_RESIDENT)));
   TRACES(("sizeof S_MFTATTR      = % 3u\n", sizeof(S_MFTATTR)));
   TRACES(("sizeof A_STANDARD     = % 3u\n", sizeof(A_STANDARD)));
   TRACES(("sizeof A_ATTRELEM     = % 3u\n", sizeof(A_ATTRELEM)));
   TRACES(("sizeof A_FILENAME     = % 3u\n", sizeof(A_FILENAME)));
   TRACES(("sizeof A_IDXROOT      = % 3u\n", sizeof(A_IDXROOT)));
   TRACES(("sizeof S_DIRINDX      = % 3u\n", sizeof(S_DIRINDX)));
   TRACES(("sizeof U_MFTATTR      = % 3u\n", sizeof(U_MFTATTR)));

   if ((tz = getenv( DFS_TZ_VAR)) != NULL)      // get correction value in
   {                                            // minutes to compensate for
      ntfs->TimeZone  = atol( tz);              // local timezone offset to
   }                                            // Coordinated Universal Time

   TRACES(( "dfsa->boot: %8.8x\n", dfsa->boot));

   ClustSize          = (BYTE) ((dfsa->boot->eb.ClustSize) ?
                        (BYTE)   dfsa->boot->eb.ClustSize : (BYTE) 8);
   ntfs->ClustSize    =  ClustSize;

   dfstSetClusterSize( DFSTORE, (USHORT) ClustSize);

   ntfs->Clust        =  dfsa->boot->nt.VolSize / ClustSize;

   if (SINF->p != NULL)                         // it is a partition
   {                                            // most likely location
      ntfs->SpareLsn = SINF->p->sectors -1;     // of spare bootsector
   }
   else                                         // use bootsector info
   {
      //- value seems a bit inconsistent, either equal to ptable,
      //- ore one less than that. This is a best guess ...

      ntfs->SpareLsn = dfsa->boot->nt.VolSize;
      if (ntfs->SpareLsn > dfsNtfsClust2Lsn( ntfs->Clust))
      {
         ntfs->SpareLsn--;
      }
   }

   ntfs->MftRecSize = (USHORT) dfsNtfsV2S( dfsa->boot->nt.MftSize, ClustSize);
   ntfs->DirRecSize = (USHORT) dfsNtfsV2S( dfsa->boot->nt.DirSize, ClustSize);

   if ((ntfs->MftRecSize == 0) || (ntfs->MftRecSize > 16))
   {
      dfsa->boot->nt.MftSize = (dfsGetSectorSize() <= SECTORSIZE) ? 2 : 1; // sane default
      fixboot = TRUE;
      TxPrint( "%sMftRecSize wrong%s! : %u, set to %u sectors for analysis\n",
                CBR, CNN, ntfs->MftRecSize, dfsa->boot->nt.MftSize);
      ntfs->MftRecSize = dfsa->boot->nt.MftSize;
   }
   if ((ntfs->DirRecSize == 0) || (ntfs->DirRecSize > 128))
   {
      dfsa->boot->nt.DirSize = max( 1, (NTFS_INDX_BLOCKSIZE / dfsGetSectorSize())); // sane default
      TxPrint( "%sDirRecSize wrong%s! : %u, set to %u sectors for analysis\n",
                CBR, CNN, ntfs->DirRecSize, dfsa->boot->nt.DirSize);
      fixboot = TRUE;
      ntfs->DirRecSize = dfsa->boot->nt.DirSize;
   }
   if (fixboot)
   {
      TxPrint( "\nDamage could be %srecoverable%s using the %sDFSee 'fixboot'%s command.\n"
               "You may need to run the FINDMFT command BEFORE that, to locate the MFT.\n\n"
               "Note that FIXBOOT from the NT/W2K/XP recovery-console is not the same!\n"
               "It solves different types of problems! You can try that if the DFSee\n"
               "fixboot does not solve the problem completely ...\n\n", CBG, CNN, CBG, CNN);
   }
   TRACES(( "MFT record size: % 4u   DIR record size: % 4u\n", ntfs->MftRecSize, ntfs->DirRecSize));
   TRACES(( "Base MFT records at cl: %llx = lsn %llx\n", dfsa->boot->nt.MftClus,
                                       dfsNtfsClust2Lsn( dfsa->boot->nt.MftClus)));
   TRACES(( "Copy MFT records at cl: %llx = lsn %llx\n", dfsa->boot->nt.MftCopy,
                                       dfsNtfsClust2Lsn( dfsa->boot->nt.MftCopy)));

   lsn = dfsNtfsClust2Lsn(dfsa->boot->nt.MftClus);
   if (lsn != 0)                                // MFT present
   {
      if ((rc = dfsRead( lsn, ntfs->MftRecSize, rbuf)) == NO_ERROR)
      {
         rc = dfsNtfsMftAlloc( lsn, rbuf);      // store MFT allocation
      }
      if (rc != NO_ERROR)
      {
         TxPrint( "Using base MFT records failed, retrying with MFT copy ...\n");
         lsn = dfsNtfsClust2Lsn(dfsa->boot->nt.MftCopy);
         if ((rc = dfsRead( lsn, ntfs->MftRecSize, rbuf)) == NO_ERROR)
         {
            rc = dfsNtfsMftAlloc( lsn, rbuf);   // store MFT allocation
         }
      }
      if (rc == NO_ERROR)                       // $MFT record now in rbuf,
      {                                         // fixed up by MftAlloc()
         ULONG         mftCount;

         rc = dfsNtfsGetMftAttrData((S_MFTFILE *) rbuf,
                          MA_BITMAP, IA_ANY, NULL, 0,
                          &ntfs->MftMap.Size,
                (void **) &ntfs->MftMap.Data);  // read into cache area

         //- Note: MFT-count should perhaps be 64-bit, just like MftRecCount
         mftCount = dfsUlMapBitCount( (ULONG *) ntfs->MftMap.Data, ntfs->MftMap.Size / sizeof(ULONG));
         TxPrint( "Used  MFT records : 0x%8.8x = %u decimal\n", mftCount, mftCount);
      }
      dfsNtfsDisplayFsStatus();

      if ((ntfs->MftRecSize    > ClustSize) &&  // MFT-records can fragment
          (ntfs->MftAll.chunks > 1))            // and MFT IS fragmented ...
      {
         TxPrint( "\n%sWARNING: MFT-record size is larger than the clustersize%s, "
                  "and the MFT IS fragmented!\nMay cause problems for some of the "
                  "NTFS specific functions, use at your own risk!\n\n", CBR, CNN);
      }
      rc = dfsNtfsBitMapInit();                 // init Bitmap cache
      if (rc == NO_ERROR)
      {
         DFSPARTINFO   *p;

         if (TxaOption('a'))                    // show NO allocation by default
         {
            dfsNtfsAllocMap( 0, 0, "@", NULL);  // get/show Minimum size
            if ((p = SINF->p) != NULL)          // we have a real partition
            {
               ULONG   kmgAlign = dfsAlignmentSectors((p->tablenr == GPT_STYLE) ? 'G' : 'P', p);

               if ((p->expandSize != 0) &&      // fspAfter, at least one alignment unit
                   (p->expandSize - p->sectors >= kmgAlign))
               {
                  dfsa->FsExpandSize = p->expandSize; // set/show Maximum size

                  dfsSz64( "Maximum vol. size : ", dfsa->FsExpandSize, " (for Expanding)\n");
               }
            }
         }
      }
   }
   dfsa->FsSltRecSize = ntfs->MftRecCount * 8 + 1000; // average 8 extents per file
   if (rc != NO_ERROR)
   {
      ntfs->Bm.Size = 0;                        // signal no BitMaps present
   }
   dfsa->findSpace = ntfs->MftAll;              // find -o:S to use MFT alloc
   dfsa->FsEntry = dfsNtfsMft2Lsn( MFT_ROOT);
   nav.down = LSN_BOOTR;
   RETURN (rc);                                 // when needed
}                                               // end 'dfsNtfsInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Close NTFS filesystem for analysis and free any resources
/*****************************************************************************/
static ULONG dfsNtfsClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   dfsSlTableReset();                           // stop SLT thread and reset

   TxFreeMem( ntfs->MftMap.Data);               // free MFT record bitmap
   TxFreeMem( ntfs->MftAll.space);              // free MFT allocation admin
   dfsa->findSpace.space  = NULL;               // remove MFT alloc reference
   dfsa->findSpace.chunks = 0;

   rc = dfsNtfsBitmapFlush( TRUE);              // flush, and terminate cache

   TxFreeMem( ntfs->Da.All.space);              // free Dir Alloc cache

   dfsCloseFileSystem();
   RETURN (rc);
}                                               // end 'dfsNtfsClose'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Initialize NTFS filesystem analysis for Area (FS info in FDISK mode)
/*****************************************************************************/
ULONG dfsNtfsAreaInit
(
   void
)
{
   ULONG               rc = NO_ERROR;
   BYTE                ClustSize;

   ENTER();

   dfsa->FsAreaClose        = dfsNtfsAreaClose;
   dfsa->FsAreaLsnAllocated = dfsNtfsAllocated;

   dfsa->boot = (S_BOOTR *) brec;
   if ((rc = dfsRead( dfstAreaP2Disk( DFSTORE, LSN_BOOTR), 1, brec)) == NO_ERROR)
   {
      ClustSize          = (BYTE) ((dfsa->boot->eb.ClustSize) ?
                           (BYTE)   dfsa->boot->eb.ClustSize : (BYTE) 8);
      ntfs->ClustSize    =  ClustSize;
      ntfs->Clust        =  dfsa->boot->nt.VolSize / ClustSize;
      ntfs->MftRecSize   = (USHORT) dfsNtfsV2S( dfsa->boot->nt.MftSize, ClustSize);
      ntfs->MftAll.space = NULL;                // no MFT alloc available!

      rc = dfsNtfsBitMapInit();                 // init Bitmap cache
   }
   RETURN (rc);                                 // when needed
}                                               // end 'dfsNtfsAreaInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Close NTFS filesystem for Area analysis and free any resources
/*****************************************************************************/
static ULONG dfsNtfsAreaClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   dfsa->FsAreaClose        = NULL;
   dfsa->FsAreaLsnAllocated = NULL;

   TxFreeMem( ntfs->MftMap.Data);               // free MFT record bitmap

   rc = dfsNtfsBitmapFlush( TRUE);              // flush, and terminate cache

   RETURN (rc);
}                                               // end 'dfsNtfsAreaClose'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display $Volume and $logfile clean/dirty status info, update dfsa anchor
/*****************************************************************************/
static ULONG dfsNtfsDisplayFsStatus
(
   void
)
{
   USHORT              LogRecFlags = 0;
   USHORT              VolumeFlags = 0;
   TXTM                VolumeName  = "";
   double              VolumeVer   = 0.0;

   dfsa->FsDirtyStatus = DFSTAT_UNKNOWN;

   if (dfsNtfsGetSetVolumeInfo (&VolumeFlags, 0,
                                 VolumeName,  FALSE,
                                &VolumeVer,   FALSE) == NO_ERROR)
   {
      TxPrint("NTFS version used : %3.1lf Volume = %s\n", VolumeVer, VolumeName);
      dfsNtfsVolFlags( "MFT $Volume Flags : ", VolumeFlags);

      if (dfsNtfsGetSetLogStatus (&LogRecFlags, 0) == NO_ERROR)
      {
         dfsNtfsLraFlags( LogRecFlags);

         //- no volume flags and journal-log flag set to unmounted
         if ((VolumeFlags == 0) && (LogRecFlags & NTFSLOG_UNMOUNT))
         {
            dfsa->FsDirtyStatus = DFSTAT_CLEAN;
         }
         else
         {
            dfsa->FsDirtyStatus = DFSTAT_DIRTY;
         }
      }
      else
      {
         TxPrint( "Reading journaling logfile flags failed, no status available!\n");
      }
   }
   else
   {
      TxPrint( "Reading $Volume with status flags, label and version info failed!\n");
   }
   RETURN (dfsa->FsDirtyStatus);
}                                               // end 'dfsNtfsDisplayFsStatus'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Convert NTFS size-value to actual number of sectors
/*****************************************************************************/
ULN64 dfsNtfsV2S                                // RET   nr of sectors in value
(
   ULONG               value,                   // IN    NTFS coded size value
   BYTE                csize                    // IN    ClusterSize in sectors
)
{
   ULN64               rc;                      // function return

   ENTER();
   TRACES(("value:0x%x\n", value));

   if (value > 0xee)                            // larger means 2's complement
   {                                            // calculate 2^(-value)
      rc = (1 << (0x100 - (value & 0xff))) / dfsGetSectorSize();
   }
   else
   {
      rc = value * csize;                       // clusters to sectors
   }
   RETN64 (rc);
}                                               // end 'dfsNtfsV2S'
/*---------------------------------------------------------------------------*/



/*****************************************************************************/
// Extract MFT allocation S_SPACE from the MFT base record, DATA attribute
/*****************************************************************************/
static ULONG dfsNtfsMftAlloc                    // RET   rc = 0 if type match
(
   ULN64               lsn,                     // IN    sector number
   BYTE               *sector                   // IN    sector data
)
{
   ULONG               rc = 0;                  // rc, sector match
   S_MFTFILE          *sd = (S_MFTFILE *) sector;
   S_MFTATTR          *at;
   int                 sanity = 30;             // allow 30 attr iterations

   ENTER();

   switch ((rc = dfsNtfsFixupStructure( sector, ntfs->MftRecSize)))
   {
      case NO_ERROR:
         for ( at = (S_MFTATTR *) ((BYTE *) sd + sd->AttribOffset);
              (at->Type != MA_END_LIST) && (rc == NO_ERROR) && (sanity--);
               at = ((S_MFTATTR *)((BYTE *) at + at->Length)))
         {
            TRACES(("MFT at %llx, check attribute at offset: %8.8x\n", lsn, (((char *) at) - ((char *) sd))));
            if ((at->DataExtern) && (at->Type == MA_DATA))
            {
                                                //- to be refined (HUGE)
               ntfs->MftRecCount = at->N.Size / sd->AllLength;
               break;
            }
         }
         if (sanity)
         {
            rc = dfsNtfsSpAlloc( sd, MA_DATA, IA_ANY, NULL, &ntfs->MftAll);
            if ((rc == NO_ERROR) && (ntfs->MftAll.chunks) && (ntfs->MftAll.space))
            {
               ntfs->MftBaseLsn = lsn;          // remember for fixboot

               TxPrint( "Total MFT records : 0x%08llX = %llu decimal   MFT Fragments: %u\n",
                         ntfs->MftRecCount, ntfs->MftRecCount, ntfs->MftAll.chunks);
            }
            else  //- resident MFT (impossible by design!) or alloc error
            {
               TxFreeMem( ntfs->MftAll.space);  // non-res SPACE or res data
               dfsX10( "MFT at sector ", lsn, "", " has bad allocation!\n");
               rc = DFS_BAD_STRUCTURE;
            }
         }
         else
         {
            dfsX10( "MFT at sector ", lsn, "", " has no DATA attribute!\n");
            rc = DFS_BAD_STRUCTURE;
         }
         break;

      case DFS_BAD_STRUCTURE:
         dfsX10( "Base MFT at LSN ", lsn, "", " has bad fixup structures.\n");
         break;

      default:
         dfsX10( "Base MFT at LSN ", lsn, "", " has a fixup error for sector: ");
         TxPrint("%u\n", lsn, rc -1);
         rc = DFS_BAD_STRUCTURE;                // signal invalid structure
         break;
   }
   RETURN (rc);
}                                               // end 'dfsNtfsMftAlloc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Extract BitMap allocation S_SPACE structure and initialize the BitMap cache
// Note: Area aware to allow usage from FDISK mode too
/*****************************************************************************/
static ULONG dfsNtfsBitMapInit                  // RET   rc = 0 if type match
(
   void
)
{
   ULONG               rc = 0;                  // rc, sector match
   BYTE                st;

   ENTER();

   ntfs->Bm.Size = (ULONG) max(ntfs->ClustSize, (ULONG) 8);
   if ((ntfs->Bm.Buffer = (BYTE *) TxAlloc( ntfs->Bm.Size, dfsGetSectorSize())) != NULL)
   {
      //- MFT-LSN is converted for Area-awareness inside Mft2Lsn!
      rc = dfsNtfsMftLsn2Alloc( dfsNtfsMft2Lsn( MFT_BITMAPF), MA_DATA, IA_ANY, &st, &ntfs->Bm.All);
      if (rc == NO_ERROR)
      {
         //- LimitLsn is the last addressable LSN in the bitmap, +1
         ntfs->Bm.LimitLsn = dfsSspaceSectors( FALSE,
                                               ntfs->Bm.All.chunks,
                                               ntfs->Bm.All.space) * //- bitmap sectors present
                                          (8 * dfsGetSectorSize()) * //- #bits, each bit is one cluster
                                               ntfs->ClustSize;      //- #sectors represented
         ntfs->Bm.Dirty    = FALSE;
         ntfs->Bm.First    = 0;
         rc = dfsSspaceReadFilePart( ntfs->Bm.All.chunks,
                                     ntfs->Bm.All.space,
                                     ntfs->Bm.First,
                                     ntfs->Bm.Size,
                            (BYTE *) ntfs->Bm.Buffer);

         #if defined (DUMP)
            if (ntfs->Bm.Buffer != NULL)
            {
               TRHEXS( 70, ntfs->Bm.Buffer, 0xc0, "ntfs->Bm.Buffer");
            }
         #endif
      }
      else
      {
         TxPrint("\nError interpreting $Bitmap allocation, no Bitmap info available!\n");
         ntfs->Bm.Size = 0;                     // signal no BitMaps present
      }
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   TRACES(( "Bm.Size: %u\n", ntfs->Bm.Size));
   RETURN (rc);
}                                               // end 'dfsNtfsBitMapInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Interpret and execute specific NTFS command;
/*****************************************************************************/
static ULONG dfsNtfsCommand
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *none,                    // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;
   ULN64               sl = 0;                  // sector number
   ULN64               sn = 0;                  // sector number input
   BYTE                st = 0;                  // sector type wanted
   LONG                nr = 0;
   ULN64               mft = 0;
   TXLN                dc;                      // DFS command
   int                 cc;                      // command string count
   char               *c0, *c1, *c2;            // parsed command parts
   TXLN                s0;                      // big temporary string space
   TXLN                s1;                      // big temporary string space
   char               *pp;                      // parameter pointer

   ENTER();

   pp = TxaGetArgString( TXA_CUR, 1, 0, TXMAXLN, dc); // dc => cmd from arg 1
   cc = TxaArgCount( );                         // number of parameters
   c0 = TxaArgValue(0);
   c1 = TxaArgValue(1);
   c2 = TxaArgValue(2);

   sn = nav.this;                               // default at current sector

   if ((strcmp(c0, "/?") == 0) ||
       (strcmp(c0,  "?") == 0)  )
   {
      TxShowTxt( ntfs_txt);
   }
   else if ((strcasecmp(c0, "findroot" ) == 0) ||
            (strcasecmp(c0, "path"     ) == 0) )
   {
      if (TxaOption('?'))
      {
         TxPrint("\nLocate root-directory (lsn) and show path to root\n");
         TxPrint("\n Usage:  %s  [start-lsn [~]]\n\n", c0);
      }
      else
      {
         sn = dfsGetSymbolicSN( c1, nav.this);  // default current LSN
         if (dfsNtfsFindRootMft( sn, TRUE, (cc > 2), 0, s0, &sn))
         {
            if (strcasecmp(c0, "findroot" ) == 0)
            {
               rc = dfsReadAnDisplay(sn, 0, &st);
            }
         }
         else
         {
            TxPrint("No NTFS Mft with path to Root found\n");
            rc = DFS_NOT_FOUND;
         }
      }
   }
   else if (strcasecmp(c0, "findmft" ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nLocate Master File Table (MFT), by searching for $AttrDef record"
                 "\n(result can be used by 'fixboot' to point bootsector to the MFT)\n");
         TxPrint("\n Usage:  %s\n\n", c0);
      }
      else
      {
         TxPrint( "\nSearch for the (unique) $AttrDef MFT record, to locate the MFT itself ...\n");
         strcpy( dc, "find -o:%! -i:2 -t:f -u:'$$AttrDef'");
         rc = dfsMultiCommand( dc, 0, TRUE, FALSE, TRUE);
         if (rc == NO_ERROR)                    // $AttrDef seems to be there
         {
            if (dfsGetSectorSize() == SECTORSIZE) // two sectors per MFT-record
            {
               ntfs->MftBaseLsn = nav.this - 8; // MFT start is 4 records back, 8 sectors
            }
            else
            {
               ntfs->MftBaseLsn = nav.this - 4;  // MFT start is 4 records back, 4 sectors
            }
            dfsX10( "\nStart of MFT area : ", ntfs->MftBaseLsn, CBY, "  (to be used by FIXBOOT)\n\n");

            sprintf( dc, "0x0%llX  -v-", ntfs->MftBaseLsn); // compact display, non-verbose
            TxaReParseCommand( dc);
            rc = DFS_PENDING;                   // handle translated command
         }
         else
         {
            TxPrint( "\nThe $AttrDef MFT record was NOT found, seems like this is not"
                     "\nan NTFS filesystem, or it is damaged beyond repair ...\n");
         }
      }
   }
   else if (strcasecmp(c0, "ca"   ) == 0)          // check allocation
   {
      ULN64            sect = 0;
      ULN64            bads = 0;

      sn = dfsGetSymbolicSN( c1, nav.this);     // default current LSN
      if (cc > 2)                               // no options
      {                                         // specified
         strcpy(s1, c2);
      }
      else
      {
         strcpy(s1, "v");                       // default verbose
      }
      rc = dfsNtfsMftLsnCheckAlloc( sn, s1, &st, &sect, &bads, &sl);
      if ((rc == NO_ERROR) || (rc == DFS_BAD_STRUCTURE))
      {
         dfsSz64("Total filesectors : ", sect, "\n");
         dfsSz64("Allocation errors : ", bads, "\n");
      }
      else
      {
         TxPrint( "\nSector 0x%llx does not seem to be a valid MFT record!\n", sn);
         rc = DFS_ST_MISMATCH;
      }
   }
   else if (strcasecmp(c0, "bootini"  ) == 0)      // find (first) boot.ini
   {
      if (TxaOption('?'))
      {
         TxPrint("\nDisplay and optionally fix the partition-index to boot for\n"
                   "the default line in the Windows-NT/W2K/XP BOOT.INI file.\n");
         TxPrint("\n Usage:  %s  [* | index]   [-2] [-c]\n\n"
                   "   *     = use DFSee calculated value  as new partition-index\n"
                   "   index = use specified numeric value as new partition-index\n\n"
                   "   -2    = update/synchronize index in 2nd BOOT.INI line too\n\n"
                   "   -c    = work on current sector, do not search BOOT.INI\n", c0);
      }
      else
      {
         if (TxaOption('c'))                    // force current sector ?
         {
            sn = nav.this;                      // use current, no search
            TRACES(("-c = Assume bootini is in current sector: 0x%llx\n", sn));
         }
         else
         {
            nav.this = dfsNtfsMft2Lsn( MFT_1STUSER) -1; // from first user MFT record
            TxPrint( "\nSearch for the (first) boot.ini file in the filesystem ...\n");
            strcpy( dc, "find -t:f -o:$/% boot.ini");
            rc = dfsMultiCommand( dc, 0, TRUE, FALSE, TRUE);
            if (rc == NO_ERROR)                 // boot.ini seems to be there
            {
               //- to be refined, could check here if the nav.this represents
               //- an MFT with ParentMft is 5 (ROOT, hence 'C:\boot.ini')
               //- needs a specific function, first load/fixup MFT, then check ParentMft

               if ((nav.down - nav.this) == ntfs->MftRecSize)
               {
                  sn = nav.this;                // internal data attribute
               }
               else
               {
                  sn = nav.down;                // LSN external data attribute
               }
            }
         }
         if (rc == NO_ERROR)                    // boot.ini seems to be there
         {
            rc = dfsCheckFixNtBootIni( sn, c1, TxaOption('2'));
         }
         else                                   // no boot.ini found, return
         {                                      // calculated boot.ini id
            if (SINF->p != NULL)                // do we have a partition id ?
            {
               rc = SINF->p->diskPart;          // ARC style partition index
               dfsa->explain = FALSE;           // don't give incorrect
            }                                   // explanations about this :-)
         }
      }
   }
   else if (strcasecmp(c0, "dirty"    ) == 0)
   {
      if (cc > 1)
      {
         USHORT        flag = 0;
         USHORT        mask = VF_VOLUME_IS_DIRTY;

         if ((c1[0] == 'd') || (c1[0] == 'D') || (c1[0] == '1'))
         {
            flag = VF_VOLUME_IS_DIRTY;
         }
         rc = dfsNtfsGetSetVolumeInfo( &flag, mask, NULL, FALSE, NULL, FALSE);
      }
      else
      {
         TxPrint("\nSet or reset the 'CHKDSK' required flag in $Volume\n\n"
                 "  Usage: %s  normal | dirty (CHKDSK required)\n", c0);
      }
      TxPrint( "\n");
      dfsNtfsDisplayFsStatus();
   }
   else if (strcasecmp(c0, "label"    ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nSet volume label in $Volume MFT-record (32-character, Windows volume name)\n");
         TxPrint("\nUsage: %s  [label]\n\n"
                 "   label  : new volume label, maximum 32 characters\n"
                 "   -!-    = do not prompt for new value, just set it\n", c0);
      }
      else
      {
         USHORT        flag = 0;                // volume flags, unchanged

         if (cc > 1)
         {
            strcpy( s1, c1);                    // use specified parameter value
         }
         else                                   // get current value from $Volume
         {
            rc = dfsNtfsGetSetVolumeInfo( &flag, 0, s1, FALSE, NULL, FALSE);
         }
         s1[ NTFS_LEN_LBL] = 0;                 // truncate to allowed length

         if (dfsa->batch)                       // s1 has new label value
         {
            rc = dfsNtfsGetSetVolumeInfo( &flag, 0, s1, TRUE, NULL, FALSE);
         }
         else
         {
            #if defined (USEWINDOWING)          // Allow interactive update
            if (!TxaOptUnSet('!') && txwIsWindow( TXHWND_DESKTOP))
            {
               TXLN      prompt;

               sprintf( prompt, "%s\n\nVolume label in $Volume MFT record: '%s'\n\n"
                                "Specify new %d-character volume label\n",
                                 dfstStoreDesc1( DFSTORE) + 10, s1, NTFS_LEN_LBL);

               if (txwPromptBox( TXHWND_DESKTOP, TXHWND_DESKTOP, NULL,
                      prompt, " Set volume label in $Volume MFT record ",
                      5164, TXPB_MOVEABLE | TXPB_HCENTER | TXPB_VCENTER,
                      NTFS_LEN_LBL, s1) != TXDID_CANCEL)
               {
                  TxStrip( s1, s1, ' ', ' ');   // strip leading/trailing spaces
                  rc = dfsNtfsGetSetVolumeInfo( &flag, 0, s1, TRUE, NULL, FALSE);
               }
            }
            else
            #endif
            {
               if (TxConfirm( 5164, "Write Volume LABEL '%s' to $Volume ? [Y/N] : ", s1))
               {
                  rc = dfsNtfsGetSetVolumeInfo( &flag, 0, s1, TRUE, NULL, FALSE);
               }
            }
         }
      }
   }
   else if (strcasecmp(c0, "mft"      ) == 0)   // calculate LSN for MFT-nr
   {                                            // optional attribute MkRunList
      if (cc > 1)
      {
         nr = dfsGetSymbolicSN( c1, 0);         // default MFT 0
         sn = dfsNtfsMft2Lsn( nr);
         if (sn != L64_NULL)
         {
            sprintf( dc, "0x0%llX", sn);
            if (TxaOptUnSet('v'))
            {
               strcat( dc, " -v-");
            }
            rc = dfsMultiCommand( dc, 0, TRUE, FALSE, TRUE);

            if ((rc == NO_ERROR) && (TxaOption('R'))) // Test MkRunList
            {
               DFSISPACE isp;                   // allocation SPACE info
               BYTE      st;
               USHORT    id = (USHORT) TxaOptNum('R', NULL, IA_ANY);

               rc = dfsNtfsMftLsn2Alloc( sn, (id == IA_ANY) ? MA_DATA : MA_ALLATTRS, id, &st, &isp);
               if (rc == NO_ERROR)
               {
                  if (isp.space)
                  {
                     USHORT rlsize;
                     BYTE  *rl;

                     if (dfsNtfsMkRunList( isp.chunks, isp.space, &rlsize, &rl) == NO_ERROR)
                     {
                        TxPrint( "Runlist   length  : 0x%3.3hx = %u, for ", rlsize, rlsize);
                        if (id == IA_ANY)
                        {
                           TxPrint( "unnamed %sData Stream%s\n", CBC, CNN);
                        }
                        else
                        {
                           TxPrint( "attribute with Id: %s% 2hx%s\n", CBC, id, CNN);
                        }
                        TxDisplayHex( "", (BYTE *) rl, (ULONG) rlsize, 0);
                        TxPrint( "\n");

                        TxFreeMem( rl);         // free RunList data
                     }
                     TxFreeMem( isp.space);     // free S_SPACE structure
                  }
                  else
                  {
                     TxPrint( "MFT %x has no RunList for the specified attribute!\n", nr);
                  }
               }
            }
         }
         else
         {
            dfsX10(  "Invalid MFT    nr : ", nr, CBR, "\n");
            dfsDec8( "Max MFT record nr : ", ntfs->MftRecCount, "\n");
         }
      }
      else
      {
         dfsX10("Current MFT rec nr: ", dfsNtfsLsn2Mft(nav.this), CBC, "\n");
      }
   }
   else if (strcasecmp(c0, "mir"      ) == 0)      // calc LSN for MFT-mirror nr
   {
      nr = dfsGetSymbolicSN( c1, 0);            // default MFT 0
      sprintf( dc, "0x0%llX", dfsNtfsMir2Lsn( nr));
      TxaReParseCommand( dc);
      rc = DFS_PENDING;                         // handle translated command
   }
   else if (strcasecmp(c0, "mftext"   ) == 0)      // find external MFT records
   {
      strcpy( dc, "find -o:*%^@32 -t:f '0000 0000 0000 0000'");
      TxaReParseCommand( dc);
      rc = DFS_PENDING;                         // handle translated command
   }
   else if (strcasecmp(c0, "mftused"      ) == 0)  // check if MFTrec is in-use
   {
      mft = dfsGetSymbolicSN( c1, 0);
      TxPrint(  "MFT record number : 0x%16.16lx is %s%s%sin use\n",
                 nr, CBR, dfsNtfsMftMapped( mft) ? "" : "NOT ", CNN);
   }
   else if ((strcasecmp(c0, "bsclear"   ) == 0) || // almost like 'bsimport -clear'
            (strcasecmp(c0, "nobads"    ) == 0)  ) // deprecated
   {
      if (TxaOption('?'))
      {
         TxPrint("\nReset bad-sector administration in $BadClus to ZERO bads\n");
         TxPrint("\n Usage:  %s\n\n", c0);
      }
      else
      {
         rc = dfsNtfsResetBadClus( ntfs->Clust * ntfs->ClustSize);
      }
   }
   else if (strcasecmp(c0, "fixboot"  ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nFix the bootsector for an NTFS partition by copying\n"
                   "the spare one or creating a new one from a template\n");
         TxPrint("\n Usage:  %s  [mftLSN]   [-c:n]  [-s-] [-V-]\n\n"
                   "   mftLSN  = sectornumber for $MFT record   (default is 0x600000)\n\n"
                   "   -c[:n]  = use current, or specified clustersize (default is 8)\n"
                   "   -s-     = do NOT try to copy the existing spare-bootsector first\n"
                   "   -V-     = Use Windows NT/XP 'NTLDR', not Vista/W7/8/10 'BOOTMGR'\n", c0);
      }
      else
      {
         if (SINF->p != NULL)                   // partition info available
         {
            if (!TxaOptUnSet('s'))              // unless -s-, try spare first
            {
               rc = dfsNtfsCopySpareBR();
            }
            else
            {
               rc = DFS_ST_MISMATCH;
            }
            if ((rc != NO_ERROR)    &&          // unless replaced by spare
                (rc != DFS_NO_CHANGE))          // or replace canceled
            {
               TxPrint( "Preparing new NTFS bootsector from a template ...\n");
               rc = dfsNtfsMkBootRec( dfsGetSymbolicSN( c1, 0), !TxaOptUnSet('V'));
            }
            if (rc == NO_ERROR)
            {
               if (SINF->partid != 0)           // was it a partition ?
               {
                  TxPrint( "Resulting partition after replacing the bootsector:\n\n");
                  sprintf( dc, "part -r -q %hu#d", SINF->partid);
                  rc = dfsMultiCommand( dc, 0, FALSE, FALSE, TRUE);
               }
            }
         }
         else                                   // not supported!
         {
            TxPrint("No partition information available, fix not possible\n");
         }
      }
   }
   else if (strcasecmp(c0, "fixntldr"  ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nReplace (the first sector of) NTLDR/BOOTMGR for an NTFS partition\n");
         TxPrint("\n Usage:  %s  [-V- | -I:image]\n"
                 "\n  -V-        = Use Windows NT/XP 'NTLDR', not Vista/W7/8/10 'BOOTMGR'\n"
                 "\n  -I[:image] = Name of imagefile with NTLDR code (%u sectors)"
                 "\n               with a default of '%s'\n"
                 "\n If no parameter is specified, a builtin 1st sector of W7 BOOTMGR"
                 "\n will be used to replace the possibly corrupted one.\n",
                     c0, NTLDR_SIZE, NTLDR_IMGNAME);
      }
      else
      {
         if (TxaOptSet('I'))                    // read NTLDR from image
         {
            char *fname = TxaOptStr( 'I', "NTLDR image", NTLDR_IMGNAME);

            sprintf( s1, "restore \"%s\" 0x0%llx %u", fname, NTLDR_LSN, NTLDR_SIZE);
            rc = dfsMultiCommand( s1, 0, TRUE, FALSE, TRUE);
         }
         else
         {
            rc = dfsNtfsMkNtldrSec( !TxaOptUnSet('V'));
         }
      }
   }
   else if (strcasecmp(c0, "dfsntldr"  ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nCreate compressed imagefile containing the NTLDR/BOOTMGR sectors\n");
         TxPrint("\n Usage:  %s   [image]\n"
                 "\n  image  = Name of imagefile to be created (%u sectors) without"
                 "\n           a file-extension, and a default name of '%s'\n",
                               c0, NTLDR_SIZE, NTLDR_IMGNAME);
      }
      else
      {
         strcpy(  s0, (cc > 1) ? c1 : NTLDR_IMGNAME);
         sprintf( s1, "image \"%s\" 0x0%llx %u  -z", s0, NTLDR_LSN, NTLDR_SIZE);
         rc = dfsMultiCommand( s1, 0, TRUE, FALSE, TRUE);
      }
   }
   else if ((strcasecmp(c0, "delfind"  ) == 0) ||
            (strcasecmp(c0, "filefind" ) == 0)  )
   {

      if (TxaOption('?'))
      {
         TxPrint("\nFind (deleted) files by direct searching the MFT-records\n");
         TxPrint("\n Usage:  %s  [name]  [-a] [-D | -f] [-c] [-d-] [-v]\n\n", c0);
         TxPrint("   name      = part of filename wanted, not a true wildcard,\n"
                 "               but may start and end in a '*' character\n"
                 "   -a        = search ANY/ALL files, deleted and normal\n"
                 "   -D        = search  directories  only, not files+directories\n"
                 "   -f        = search regular files only, not files+directories\n"
                 "   -c        = start from current sector, not start of volume\n"
                 "   -d-       = search outside Master File Table too   (SLOW!)\n"
                 "   -v        = verbose search, list files while found (SLOW!)\n");
      }
      else
      {
         TXTS       findtype;                   // sector types to be searched
         BOOL       deleted  = (strcasecmp(c0, "delfind"  ) == 0);

         if (TxaOption('D'))                    // Directories only
         {
            strcpy( findtype, TxaOption('a') ? "Dy" : (deleted)  ? "y" :  "D");
         }
         else if (TxaOption('f'))               // files only
         {
            strcpy( findtype, TxaOption('a') ? "fz" : (deleted)  ? "z" :  "f");
         }
         else                                   // Files + Directories
         {
            strcpy( findtype, TxaOption('a') ? "fzDy" : (deleted)  ? "zy" :  "fD");
         }
         sprintf( dfsa->brdescript, "NTFS %s %s", c0, pp);
         sprintf( s1, "find -t:%s -o:*$/%%%s!%s -i:2 ", findtype,
                 (TxaOptUnSet('d')) ? "" : "S", // MFT only (-o:S spaced)
                 (TxaOption(  'v')) ? "" : "Q");  // verbose or quiet, FAST
         if (cc > 1)                            // ascii wildcard search-string
         {
            TxStrip( s0, pp, '*', '*');         // remove leading/trailing *
            strcat(  s1, s0);                   // add search string-value
         }
         strcpy( dc, s1);                       // copy complete command
         TxPrint("Find %s: NTFS specific, using: %s%s%s\n",
                  TxaOption('a') ? "any/all files" :
                       (deleted) ? "deleted files" :
                                   "normal files ", CBC, dc, CNN);
         if (!TxaOption('c'))
         {
            nav.this = dfsNtfsMft2Lsn( MFT_1STUSER); // from first user MFT record
         }
         TxaReParseCommand( dc);
         rc = DFS_PENDING;                      // handle translated command
      }
   }
   else
   {
      rc = DFS_CMD_UNKNOWN;                     // cmd not recognized
   }
   RETURN (rc);
}                                               // end 'dfsNtfsCommand'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// NTFS filesystem, identify specified sector as a valid superblock
/*****************************************************************************/
BOOL dfsNtfsIsSuperBlock                        // RET   sector is a valid sb
(
   BYTE               *sec                      // IN    sector contents
)
{
   BOOL                rc = FALSE;
   S_S_NTLDR          *sp = (S_S_NTLDR*) sec;

   if ((memcmp(sp->Signature,sg_s_ntldr,SG_S_NTLDR) == 0) ||
       (memcmp(sp->Signature,sg_s_bootm,SG_S_NTLDR) == 0)  )
   {
      rc = TRUE;
   }
   return (rc);
}                                               // end 'dfsNtfsIsSuperBlock'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// NTFS filesystem, display sector constents as a superblock
/*****************************************************************************/
void dfsNtfsDisplaySuperBlock
(
   BYTE               *sec                      // IN    sector contents
)
{
   TxPrint( "Sector contains 'loader' code for a Windows-Nt style operating system\n");
}                                               // end 'dfsNtfsDisplaySuperBlock'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// NTFS filesystem, identify specified sector
/*****************************************************************************/
static ULONG dfsNtfsIdent
(
   ULN64               lsn,                     // IN    LSN for sector
   ULN64               d2,                      // IN    dummy
   char               *st,                      // OUT   sector type
   void               *sec                      // IN    sector contents
)
{
   ULONG               dr = NO_ERROR;
   BYTE                rc = ST_UDATA;

   ENTER();

   TRDUMP(70, "NTFS ident, start of sector:\n", sec, 64, 0);
   if (!memcmp(((S_MFTFILE*)sec)->Signature,sg_mftfile,SG_MFTFILE)) rc=ST_MFTFILE;
   if (!memcmp(((S_DIRINDX*)sec)->Signature,sg_dirindx,SG_DIRINDX)) rc=ST_DIRINDX;
   if (!memcmp(((S_LOGRSTR*)sec)->Signature,sg_logrstr,SG_LOGRSTR)) rc=ST_LOGRSTR;
   if (!memcmp(((S_LOGRCRD*)sec)->Signature,sg_logrcrd,SG_LOGRCRD)) rc=ST_LOGRCRD;
   if (!memcmp(((S_S_NTLDR*)sec)->Signature,sg_s_ntldr,SG_S_NTLDR)) rc=ST_S_NTLDR;
   if (!memcmp(((S_S_NTLDR*)sec)->Signature,sg_s_bootm,SG_S_NTLDR)) rc=ST_S_NTLDR;
   switch (rc)
   {
      case ST_UDATA:
         dr = DFS_PENDING;
         break;

      case ST_MFTFILE:                          // perform sanity checks
         {
            S_MFTFILE *sd   = (S_MFTFILE *) sec;
            USHORT     size = (ntfs->MftRecSize * dfsGetSectorSize());

            if ((sd->FixupSize    < 2                       ) ||
                (sd->FixupSize -1 > (size / FIXUP_AREA_SIZE)) ||
                (sd->FixupOffset  < 0x20                    ) ||
                (sd->FixupOffset  > FIXUP_AREA_SIZE         ) ||
                (sd->AttribOffset > size                    ) ||
                (sd->RecLength    > sd->AllLength           ) ||
                (sd->AllLength   != size                    ) )
            {
               TRACES(( "fixup: %hu at %hu attrib: %hu  Rec:%hu All:%hu\n",
                        sd->FixupSize, sd->FixupOffset, sd->AttribOffset,
                        sd->RecLength, sd->AllLength));
               TRACES(( "MftRecSize: %hu  sector size: %hu  size: %hu\n",
                         ntfs->MftRecSize, dfsGetSectorSize(), size));

               rc = ST_UDATA;
               dr = DFS_PENDING;
            }
            else                                // determine exact type
            {
               if (sd->Sequence == 0)           // Empty MFT, reserved
               {
                  rc = ST_MFTEMPT;
               }
               else if (dfsNtfsLsn2Mft(lsn) != L64_NULL)
               {
                  if (sd->MftFlags & AF_REC_INUSE) // entity not deleted
                  {
                     if (sd->MftFlags & AF_DIRECTORY)
                     {
                        if (sd->BaseMftRec != 0)
                        {
                           rc = ST_MFTSECD;
                        }
                        else
                        {
                           rc = ST_MFTDIRB;
                        }
                     }
                     else
                     {
                        if (sd->BaseMftRec != 0)
                        {
                           rc = ST_MFTSECF;
                        }
                        else
                        {
                           rc = ST_MFTFILE;
                        }
                     }
                  }
                  else                          // deleted
                  {
                     if (sd->MftFlags & AF_DIRECTORY)
                     {
                        rc = ST_MFTDELD;
                     }
                     else
                     {
                        rc = ST_MFTDELF;
                     }
                  }
               }
               else                             // MFT not in MFT-file, this
               {                                // could be from a previous
                  rc = ST_MFGHOST;              // format or after shrinking
               }                                // of the MFT (possible ?)
            }
         }
         break;

      case ST_DIRINDX:                          // perform sanity checks
         {                                      // and split SEC indexes
            S_DIRINDX *sd   = (S_DIRINDX *) sec;
            ULONG      size = (ntfs->DirRecSize * dfsGetSectorSize());

            if ((sd->FixupSize    < 2                       ) ||
                (sd->FixupSize -1 > (size / FIXUP_AREA_SIZE)) ||
                (sd->FixupOffset  > FIXUP_AREA_SIZE         ) ||
                (sd->HeaderSize   > size                    ) )
            {
               rc = ST_UDATA;
               dr = DFS_PENDING;
            }
            else
            {
               S_DIRENTRY *entry = (S_DIRENTRY *) (((BYTE *) &sd->HeaderSize) +
                                                              sd->HeaderSize);
               switch (entry->Magic)            // hack to identify SDH/SII
               {
                  case IHDR_SDH_IND: rc = ST_SDHINDX; break;
                  case IHDR_SII_IND: rc = ST_SIIINDX; break;
                  default:                            break;
               }
            }
         }
         break;

      default:
         break;
   }
   *st = rc;
   RETURN (dr);
}                                               // end 'dfsNtfsIdent'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// NTFS filesystem, supply sector-type description string, incl ATTRIBUTE
/*****************************************************************************/
static ULONG dfsNtfsStype
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *st,                      // IN    sector type
   void               *data                     // OUT   type description
)
{
   ULONG               rc  = NO_ERROR;
   char               *buf = (char *) data;

   switch (*st)
   {
      case ST_MFTFILE    : sprintf(buf,"MFT regular File "); break;
      case ST_F_STANDARD : sprintf(buf,"FILE StandardInfo"); break;
      case ST_F_ATTRLIST : sprintf(buf,"FILE AttributeLst"); break;
      case ST_F_FILENAME : sprintf(buf,"FILE File Name   "); break;
      case ST_F_VERSION  : sprintf(buf,"FILE Object Id   "); break;
      case ST_F_SECURITY : sprintf(buf,"FILE SecurityDesc"); break;
      case ST_F_VOLNAME  : sprintf(buf,"FILE Volume name "); break;
      case ST_F_VOLINFO  : sprintf(buf,"FILE Volume Info "); break;
      case ST_F_DATA     : sprintf(buf,"FILE Data Stream "); break;
      case ST_F_IDXROOT  : sprintf(buf,"FILE Index Root  "); break;
      case ST_F_DIRINDX  : sprintf(buf,"FILE Index Alloc "); break;
      case ST_F_BITMAP   : sprintf(buf,"FILE Alloc Bitmap"); break;
      case ST_F_SYMLINK  : sprintf(buf,"FILE ReparsePoint"); break;
      case ST_F_EAINFO   : sprintf(buf,"FILE ExtAttr Info"); break;
      case ST_F_EADATA   : sprintf(buf,"FILE ExtAttr Data"); break;
      case ST_F_PROPSET  : sprintf(buf,"FILE Property Set"); break;
      case ST_F_LOGUSTR  : sprintf(buf,"FILE LoggedUtStrm"); break;
      case ST_MFTDIRB    : sprintf(buf,"MFT  regular Dir "); break;
      case ST_LOGRSTR    : sprintf(buf,"LOG  restart Area"); break;
      case ST_LOGRCRD    : sprintf(buf,"LOG  record  Page"); break;
      case ST_D_STANDARD : sprintf(buf,"DIR  StandardInfo"); break;
      case ST_D_ATTRLIST : sprintf(buf,"DIR  AttributeLst"); break;
      case ST_D_FILENAME : sprintf(buf,"DIR  File Name   "); break;
      case ST_D_VERSION  : sprintf(buf,"DIR  Object Id   "); break;
      case ST_D_SECURITY : sprintf(buf,"DIR  SecurityDesc"); break;
      case ST_D_VOLNAME  : sprintf(buf,"DIR  Volume name "); break;
      case ST_D_VOLINFO  : sprintf(buf,"DIR  Volume Info "); break;
      case ST_D_DATA     : sprintf(buf,"DIR  Data Stream "); break;
      case ST_D_IDXROOT  : sprintf(buf,"DIR  Index Root  "); break;
      case ST_D_DIRINDX  : sprintf(buf,"DIR  Index Alloc "); break;
      case ST_D_BITMAP   : sprintf(buf,"DIR  Alloc Bitmap"); break;
      case ST_D_SYMLINK  : sprintf(buf,"DIR  ReparsePoint"); break;
      case ST_D_EAINFO   : sprintf(buf,"DIR  ExtAttr Info"); break;
      case ST_D_EADATA   : sprintf(buf,"DIR  ExtAttr Data"); break;
      case ST_D_PROPSET  : sprintf(buf,"DIR  Property Set"); break;
      case ST_D_LOGUSTR  : sprintf(buf,"DIR  LoggedUtStrm"); break;
      case ST_MFTSECF    : sprintf(buf,"MFT  2ndary  File"); break;
      case ST_MFTSECD    : sprintf(buf,"MFT  2ndary  Dir "); break;
      case ST_MFTDELF    : sprintf(buf,"MFT  deleted File"); break;
      case ST_MFTDELD    : sprintf(buf,"MFT  deleted Dir "); break;
      case ST_DIRINDX    : sprintf(buf,"DIR filename INDX"); break;
      case ST_S_NTLDR    : sprintf(buf,"NTLDR, 1st sector"); break;
      default:
         switch ((*st) | ST__INFO)              // non-searchable ones
         {
            case ST_MFGHOST    : sprintf(buf,"MFT Ghost record "); break;
            case ST_MFTEMPT    : sprintf(buf,"MFT empty record "); break;
            case ST_CLSLACK    : sprintf(buf,"Past last cluster"); break;
            case ST_SIIINDX    : sprintf(buf,"SII security INDX"); break;
            case ST_SDHINDX    : sprintf(buf,"SDH security INDX"); break;
            default:       rc = DFS_PENDING;
         }
         break;
   }
   return (rc);
}                                               // end 'dfsNtfsStype'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// NTFS filesystem, display sector-contents based on type
/*****************************************************************************/
static ULONG dfsNtfsSectorContents
(
   ULN64               psn,                     // IN    base psn for sector
   ULN64               d2,                      // IN    dummy
   char               *type,                    // IN    type of sector
   void               *data                     // IN    sector contents
)
{
   ULONG               rc = NO_ERROR;
   BYTE                st = (BYTE) *type;
   ULN64               lsn = dfstPsn2LSN( DFSTORE, psn);
   USHORT              nr;                      // nr of additional sectors

   ENTER();

   switch (st)
   {
      case ST_MFTFILE:                          // read rest of MFT FileRecord
      case ST_MFTDIRB:
      case ST_MFTSECF:
      case ST_MFTSECD:
      case ST_MFTDELD:
      case ST_MFTDELF:
      case ST_MFGHOST:
      case ST_MFTEMPT:
         nr = ntfs->MftRecSize;

         //- Note: this assumes MFT-records are always contiguous, they could be
         //-       at DIFFERENT clusters with a 512 byte clustersize!
         //-       In that case you need to read the 2nd half by interpreting
         //-       the allocation chain, not simply the next LSN!

         //- Might need to split dfsNtfsReadFixMft into a Read and a Fixup part
         //- and use that everywhere (read for display should NOT fixup yet :-)

         if (nr > 1)                            // if there are more ...
         {
            rc = dfstReadPsn( DFSTORE, psn+1, nr -1, rbuf + dfsGetSectorSize());
         }
         nav.xtra = nav.this + ntfs->MftRecSize;
         nav.down = nav.xtra;
         rc = dfsNtfsMftFile( rbuf, lsn);
         break;

      case ST_SIIINDX:
      case ST_SDHINDX:
      case ST_DIRINDX:                          // read rest of INDX block
      case ST_LOGRSTR:                          // or of restart Area
         nr = ntfs->DirRecSize;
         if (nr > 1)                            // if there are more ...
         {
            rc = dfstReadPsn( DFSTORE, psn+1, nr -1, rbuf + dfsGetSectorSize());
         }
         if (st == ST_LOGRSTR)
         {
            rc = dfsNtfsLogRestart( rbuf);
         }
         else
         {
            rc = dfsNtfsIndxBlock( rbuf, st);
         }
         break;

      default:
         rc = DFS_PENDING;
         break;
   }
   RETURN (rc);
}                                               // end 'dfsNtfsSectorContents'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display NTFS MFT record with attributes
// 20180614 JvW Added -v- non-verbose dislay option, with name attributes only
/*****************************************************************************/
static ULONG dfsNtfsMftFile                     // RET   rc = 0 if type match
(
   BYTE                *sector,                 // IN    sector data
   ULN64               lsn                      // IN    LSN of Mft
)
{
   ULONG               rc = NO_ERROR;           // rc, sector match
   S_MFTFILE          *sd = (S_MFTFILE *) sector;
   S_MFTATTR          *at;
   TXLN                text;                    // ASCII buffer
   TXTM                tbuf;                    // ASCII buffer
   USHORT             *generation;              // ptr to sector generation
   TXLN                fpath;                   // fname-path
   ULN64               related[MFT_RELATEDS];   // seen, related MFT numbers
   ULN64               baseMft = dfsNtfsLsn2Mft(lsn);
   ULONG               nr;

   ENTER();

   //- set all entries to a non-existing value (results in MFT_NULL)
   memset( &related, 0xff, (size_t) (MFT_RELATEDS * sizeof(ULN64)));
   related[0] = baseMft;

   generation = (USHORT *) (sector + sd->FixupOffset);

   strcpy( text, "");
   dfstrX10( text, "Calculated MFT nr : ", baseMft, CBC, " ");
   TxPrint( "%ssequence nr : 0x%4.4X  "
            "Nr of hard-links  : %6u\n",
             text, sd->Sequence, sd->HardLinks);

   if (sd->BaseMftRec != 0)
   {
      strcpy( text, "");
      dfstrX10( text, "BaseRecord MFT nr : ",      dfsMftPure( sd->BaseMftRec), CBY, " ");
      TxPrint( "%ssequence nr : 0x%4.4hx\n", text, dfsMftSqnr( sd->BaseMftRec));
      nav.up   = dfsNtfsMft2Lsn( dfsMftPure( sd->BaseMftRec));
      nav.xtra = nav.up;
   }
   TxPrint( "Fixup-list offset : 0x%4.4X     Fixup areas : %6u  Nr of generations : 0x%4.4x\n",
                                       sd->FixupOffset, sd->FixupSize -1, *generation);
   TxPrint( "Offset 1st Attrib : 0x%4.4X   Actual length : 0x%4.4X  Allocated length  : 0x%4.4X\n",
                                       sd->AttribOffset, sd->RecLength, sd->AllLength);

   TxPrint("MFT rec is in-use : %s MFT for : %s\n",
           (sd->MftFlags & AF_REC_INUSE) ? "Yes,   regular" : "No,    deleted",
           (sd->MftFlags & AF_DIRECTORY) ? "Directory"    : "File");

   TxPrint( "Logfile sequence# : 0x%16.16llX%11.11sNext attribute Id : 0x%2.2hx\n",
                                 sd->LogSeqNr, "", sd->NextAttributeId & 0xff);

   if (sd->BaseMftRec)                          // display deafult path to root for the
   {                                            // BASE MFT (this is a secondary one)
      dfsNtfsFindRootMft( nav.up, FALSE, FALSE, 0, fpath, NULL);
      TxPrint( "Base-MFT fullpath : %s%s%s\n", CBY, fpath, CNN);
   }

   switch ((rc = dfsNtfsFixupStructure( sector, ntfs->MftRecSize)))
   {
      case NO_ERROR:
         break;

      default:
         TxPrint("%sError in fixup%s at : MFT sector: %u \n", CBR, CNN, rc -1);
         rc = DFS_BAD_STRUCTURE;                // signal invalid structure
         break;
   }
   if (sd->MftFlags & AF_DIRECTORY)             // remember MFT number as last
   {                                            // resort for later display of
      ntfs->LastDirMftnr = baseMft;             // INDX ALLOC (directories)
   }
   for ( nr = 0,      at = (S_MFTATTR *) ((BYTE *) sd + sd->AttribOffset);
        (nr < 30) && (at->Type != MA_END_LIST) && !TxAbort();
         nr++,        at = ((S_MFTATTR *)((BYTE *) at + at->Length)))
   {
      if ((!TxaOptUnSet('v')) || (at->Type == MA_FILENAME)) // in non-verbose, only NAME
      {
         USHORT           offset  = ((USHORT) (((BYTE *) at) - ((BYTE *) sd)));
         USHORT           maxsize = (ntfs->MftRecSize * dfsGetSectorSize()) - offset;

         dfsNtfsShowAttrType( "   Attribute ",   at->Type, CNC, " Id:");
         sprintf( text, "%s%4X   %s%s %s",  CBC, at->Ident,
            (at->DataExtern) ?  CBG       :  CNG,
            (at->DataExtern) ? "External" : "Resident", CNN);
         if (at->DataExtern)
         {
            dfstrSz64Byte( text, "", at->N.Size, "");
         }
         else
         {
            sprintf( tbuf, "at: 0x%4.4X Size%6u Dec", offset + at->R.Offset, at->R.Size);
            strcat(  text, tbuf);
         }
         TxPrint( "%s\n", text);
         if ((at->DataExtern) && (at->N.ComprUnitSize != 0))
         {
            TxPrint( "   Compr-Unitsize : 0x%4.4hx     = 2^%hu or %u clusters\n",
                                  at->N.ComprUnitSize,
                                  at->N.ComprUnitSize,
                     (ULONG)(1 << at->N.ComprUnitSize));
         }
         if (at->AttFlags != 0)                 // any flags for attribute ?
         {
            TxPrint( "  attribute flags : %s%s%s%s%s\n",                    CBY,
                       (at->AttFlags & AT_COMPRESSED) ? "Compressed " : "",
                       (at->AttFlags & AT_ENCRYPTED)  ? "Encrypted "  : "",
                       (at->AttFlags & AT_SPARSE)     ? "Sparse "     : "", CNN);
         }
         if (at->NameLen != 0)                  // attr name is in Unicode!
         {
            TxUnic2Ascii((USHORT *) ((BYTE *) at + at->Offset), text, at->NameLen);
            TxPrint( "   attribute name : %s%s%s\n", CBM, text, CNN);
         }
         if (at->DataExtern)                    // non resident data
         {
            TxPrint( "   Cluster LCN's  : 0x%llx - 0x%llx\n", at->N.SegFirst, at->N.SegLast);

            strcpy(  text, "");
            dfstrSz64Byte( text, "  Allocated size  : ", at->N.Allocated,  " = ");
            dfsUllDot20(   text,                         at->N.Allocated,  " bytes\n");

            strcpy(  text, "");
            dfstrSz64Byte( text, (at->N.Compressed == at->N.Size) ?
                                 "    Reported size : " :
                                 "  Compressed size : ", at->N.Compressed, " = ");
            dfsUllDot20(   text,                         at->N.Compressed, " bytes\n");

            TxPrint( "   Runlist offset : 0x%4.4hx Max-Length : 0x%4.4hx = %hu"
                     " bytes     Runlist:\n", offset + at->N.Offset,
                                          at->Length - at->N.Offset,
                                          at->Length - at->N.Offset);

            TxDisplayHex( "",            (BYTE *) at + at->N.Offset,
                  (ULONG) (min( (USHORT) (at->Length - at->N.Offset),
                                             maxsize - at->N.Offset)),
                                              offset + at->N.Offset);

            TxPrint( "\n");
            dfsNtfsSpDisplay( sd, at->Type, at->Ident); // RAW external allocation
         }
         else                                   // Resident
         {
            if (at->Type == MA_DATA)            // report undelete status
            {
               TxPrint( "%s : %sPossible%s   (Resident, always OK)\n",
                  (sd->MftFlags & AF_REC_INUSE) ? "Saveto cmd (copy)"
                                             : "Undelete filedata", CBG, CNN );
            }
            dfsNtfsAttribValuePart( baseMft, lsn, sd, at);
         }
         if (at->Type == MA_ATTRLIST)           // display contents too
         {                                      // resident or non-resident
            ULONG        atSize;
            A_ATTRLIST  *atList;

            TxPrint( "   attribute 0020 : list contents\n");
            if (dfsNtfsGetMftAttrData( sd, MA_ATTRLIST, at->Ident, NULL, 0,
                                       &atSize, (void **) &atList) == NO_ERROR)
            {
               dfsNtfsAttributList( sd, atList, atSize, related);
               TxFreeMem( atList);
            }
         }
         else if (at->Type > MA_ALLATTRS)       // illegal value for attr
         {
            TxPrint( "\nIllegal attribute type 0x%8.8x at offset %4.4hx!"
                     "  Aborting display ...\n", at->Type, offset);
            rc = DFS_BAD_STRUCTURE;
            break;
         }
      }
   }
   for (nr = 0; (nr < MFT_RELATEDS) && (!TxAbort()); nr++) // first is baseMft (done :-)
   {
      if      (related[nr] == baseMft)          // baseMft, no recursion
      {
         //- skip references to same MFT number, already displayed
      }
      else if (related[nr] != MFT_NULL)
      {
         BYTE          st = 0;                  // use autodetect
         ULN64         nd = nav.down;

         dfsX10( "\nAutomatic display : Related MFT record nr  : ", related[ nr], CBY, "\n");
         dfsReadAnDisplay( dfsNtfsMft2Lsn( related[ nr]), 0, &st);

         nav.down = nd;                         // restore next lsn (contents)
      }
      else                                      // reached end of list
      {
         break;
      }
   }
   RETURN (rc);
}                                               // end 'dfsNtfsMftFile'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display NTFS index structure for DIRECTORY or security SII and SDH data
/*****************************************************************************/
static ULONG dfsNtfsIndxBlock                   // RET   rc = 0 if type match
(
   BYTE               *sector,                  // IN    sector data
   BYTE                stype                    // IN    sector type
)
{
   ULONG               rc = 0;                  // rc, sector match
   S_DIRINDX          *sd = (S_DIRINDX *) sector;
   S_DIRENTRY         *list;                    // List of Dir entries
   USHORT             *generation;              // ptr to sector generation
   TXLN                fpath;                   // fname-path
   ULN64               mftnr = ntfs->LastDirMftnr;

   ENTER();

   generation = (USHORT *) (sector + sd->FixupOffset);

   TxPrint( "INDX Header size  : 0x%4.4X   "
                "Bytes in use  : 0x%4.4x  "
            "Total nr of bytes : 0x%4.4x\n",
             sd->HeaderSize, sd->InUseLength, sd->TotalLength);
   TxPrint( "Header field nr 1 : 0x%4.4hx   "
                "Header field2 : 0x%4.4hx  "
            "Header field nr 3 : 0x%x\n",
             sd->hf1, sd->hf2, sd->hf3);
   TxPrint( "Fixup-list offset : 0x%4.4X   "
                "  Fixup areas : %6u  "
            "Nr of generations : 0x%4.4x\n",
             sd->FixupOffset, sd->FixupSize -1, *generation);

   switch ((rc = dfsNtfsFixupStructure( sector, ntfs->DirRecSize)))
   {
      case NO_ERROR:
         break;

      default:
         TxPrint("Error in fixup for INDX structure at sector: %u\n", rc -1);
         rc = DFS_ST_MISMATCH;                  // signal invalid structure
         break;
   }
   switch (stype)
   {
      case ST_SIIINDX:
         TxPrint( "Part of index for : Security data, SII\n");
         break;

      case ST_SDHINDX:
         TxPrint( "Part of index for : Security data, SDH\n");
         break;

      case ST_DIRINDX:
      default:
         list = (S_DIRENTRY *) (((BYTE *) &sd->HeaderSize) + sd->HeaderSize);

         if (list->Length > 0x20)               // first entry a real filename
         {                                      // so has a Parent reference
            TxPrint( "Part of directory : ");
            mftnr = list->Parent;               // explicit mftnr is known
         }
         else
         {
            TxPrint( "Guessed directory : ");   // use LastDirMftnr
         }
         dfsNtfsFindRootMft( dfsNtfsMft2Lsn( mftnr), FALSE, FALSE, 0, fpath, NULL);
         TxPrint( "%s%s%s\n", CBY, fpath, CNN);
         dfsNtfsDirEntries( mftnr, list, TRUE); // list and record the entries
         break;
   }
   RETURN (rc);
}                                               // end 'dfsNtfsIndxBlock'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display NTFS LogFile Restart Area structure
/*****************************************************************************/
static ULONG dfsNtfsLogRestart                  // RET   rc = 0 if type match
(
   BYTE               *sector                   // IN    sector data
)
{
   ULONG               rc = 0;                  // rc, sector match
   S_LOGRSTR          *sd = (S_LOGRSTR *) sector;
   USHORT             *generation;              // ptr to sector generation

   ENTER();

   generation = (USHORT *) (sector + sd->FixupOffset);

   TxPrint( "Fixup-list offset : %4.4X   ",  sd->FixupOffset);
   TxPrint( "Nr of fixup areas : %4u  ",     sd->FixupSize -1);
   TxPrint( "Nr of generations : %4.4x\n",  *generation);

   switch ((rc = dfsNtfsFixupStructure( sector, SC_LOGRSTR)))
   {
      case NO_ERROR:
         break;

      default:
         TxPrint("Error in fixup for RSTR structure at sector: %u\n", rc -1);
         rc = DFS_ST_MISMATCH;                  // signal invalid structure
         break;
   }

   TxPrint( "System page size  : 0x%8.8x =% 5u bytes\n", sd->SysPageSize, sd->SysPageSize);
   TxPrint( "LogRec page size  : 0x%8.8x =% 5u bytes\n", sd->LogPageSize, sd->LogPageSize);
   TxPrint( "LogFile version   : %hu.%hu\n",             sd->LogMajor,    sd->LogMinor);

   if (sd->LraOffset < dfsGetSectorSize())      // should be in first sector
   {
      S_LRAREA        *area  = (S_LRAREA *) (sector + sd->LraOffset);

      dfsNtfsLraFlags( area->LraFlags);
      TxPrint( "# Client records  : %hu\n",                   area->Clients);
      TxPrint( "# in the FreeList : 0x%4.4hx     = %hu %s\n", area->FreeList,
                                                              area->FreeList,
                  (area->FreeList == NTFSLOG_LISTEMPTY) ? "(empty)" : "");
      TxPrint( "# in the UsedList : 0x%4.4hx     = %hu %s\n", area->UsedList,
                                                              area->UsedList,
                  (area->UsedList == NTFSLOG_LISTEMPTY) ? "(empty)" : "");

      //- to be refined, add more fields and client records
   }
   RETURN (rc);
}                                               // end 'dfsNtfsLogRestart'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display NTFS LogFile Restart Area Flags value
/*****************************************************************************/
static void dfsNtfsLraFlags
(
   USHORT              lraFlags                 // IN    flags value
)
{
   BOOL             clean = ((lraFlags & NTFSLOG_UNMOUNT) != 0);
   BOOL             cpend = ((lraFlags & NTFSLOG_CHANGES) != 0);

   TxPrint( "RestartArea Flags : 0x%4.4hX     = %s%s%s, %schanges pending\n",
             lraFlags, (clean) ? CBG : CBR,
                       (clean) ? "Unmounted" : "Mounted", CNN,
                       (cpend) ? "" : "no ");
}                                               // end 'dfsNtfsLraFlags'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display NTFS MFT record Attribute specific value part
/*****************************************************************************/
static ULONG dfsNtfsAttribValuePart             // RET   rc = 0 if type match
(
   ULN64               mft,                     // IN    Current MFT number
   ULN64               lsn,                     // IN    Current MFT lsn
   S_MFTFILE          *sd,                      // IN    base MFT rec, fixed-up
   S_MFTATTR          *ra                       // in    resident attribute
)
{
   ULONG               rc = 0;                  // rc, sector match
   ULONG               size  = ra->R.Size;      // size of attribute data
   U_MFTATTR          *av    = (U_MFTATTR *)    // Attribute value data
                              (((BYTE *) ra) + ra->R.Offset);
   ENTER();
   TRARGS(("sd:%8.8x  atype:%4x  size:%8hx\n", sd, ra->Type, size));

   //- to be refined, assumes all these are always located in same MFT
   //- might use NtfsGetMftAttrData for ones that can be external.
   //- (generic solution for what is now used with attribute-list 0020)

   switch (ra->Type)
   {
      case MA_STANDARD:        rc = dfsNtfsAttrStandard(      &av->Stand);       break;
      case MA_FILENAME:        rc = dfsNtfsAttrFilename( lsn, &av->Fname);       break;
      case MA_VERSION:         rc = dfsNtfsResAttribHex( sd,   av, size);        break;
      case MA_SECURITY:        rc = dfsNtfsResAttribHex( sd,   av, size);        break;
      case MA_VOLNAME:         rc = dfsNtfsVolumeName(   sd,  &av->Vname, size); break;
      case MA_VOLINFO:         rc = dfsNtfsVolumeInfo(   sd,  &av->Vinfo, size); break;

      case MA_IDXROOT:
         switch (av->Index.Id)                  // different kinds of indexes
         {                                      // to be refined for SII/SDH (Magic)
            case INDX_I30_DIR: rc = dfsNtfsAttrIndxRoot( mft, &av->Index, size); break;
            default:           rc = dfsNtfsResAttribHex( sd,   av, size);        break;
         }
         break;

      case MA_DIRINDX:         rc = dfsNtfsResAttribHex( sd,   av, size);        break;
      case MA_BITMAP:          rc = dfsNtfsResAttribHex( sd,   av, size);        break;
      case MA_SYMLINK:         rc = dfsNtfsResAttribHex( sd,   av, size);        break;
      case MA_EAINFO:          rc = dfsNtfsResAttribHex( sd,   av, size);        break;
      case MA_EADATA:          rc = dfsNtfsResAttribHex( sd,   av, size);        break;
      case MA_PROPSET:         rc = dfsNtfsResAttribHex( sd,   av, size);        break;
      case MA_LOGUSTR:         rc = dfsNtfsResAttribHex( sd,   av, size);        break;
      case MA_DATA:            rc = dfsNtfsResAttribHex( sd,   av, size);        break;
      default:                 rc = DFS_ST_MISMATCH;                             break;
   }
   RETURN (rc);
}                                               // end 'dfsNtfsAttribValuePart'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display NTFS MFT record AttributeList, get list of related MFT's
/*****************************************************************************/
static ULONG dfsNtfsAttributList
(
   S_MFTFILE          *sd,                      // IN    base MFT rec, fixed-up
   A_ATTRLIST         *at,                      // IN    Attribute value data
   ULONG               size,                    // IN    size of attribute
   ULN64              *rmft                     // INOUT related MFT (MFT_RELATEDS)
)                                               //       first item is baseMft
{
   ULONG               rc = 0;                  // rc, sector match
   ULONG               this;                    // size of this element
   ULONG               done;
   ULN64               mft;                     // mft record nr
   char               *same;                    // color for same / different
   A_ATTRELEM         *el;
   TXTM                text;

   ENTER();

   TRHEXS( 70, sd, 0xc0, "S_MFTFILE");
   for ( done = 0;
        (done < size) && (!TxAbort()) && (rc == NO_ERROR);
         done += this)                          // iterate over attr elements
   {
      el   = (A_ATTRELEM *) (((char *) at) + done);
      this = el->RecLength;
      TRHEXS( 70, at, 0xc0, "A_ATTRLIST");
      TRACES(("Type:%4x  done:%4x  this:%4x  size:%4x\n",
               el->Type, done, this, size));
      if ((this != 0) && (this <= (size - done)))
      {
         mft = dfsMftPure( el->MftRecordNr);    // MFT nr for this attribute

         if (mft == rmft[0])                    // is this the base ?
         {
            same = CBC;
         }
         else                                   // located in other MFT!
         {
            int        i;

            for (i = 0; (i < MFT_RELATEDS) && (!TxAbort()); i++)
            {
               if (rmft[i] == mft)              // already recorded
               {
                  break;                        // no action needed
               }
               else if (rmft[i] == MFT_NULL)    // end of list reached
               {
                  rmft[i] = mft;                // record this MFT as handled
                  break;
               }
            }
            same = CBY;
            nav.xtra = dfsMftPure( mft);
         }
         dfsNtfsShowAttrType( "        Attr ", el->Type,  CBG, " ");
         TxPrint( "Id:%s%4X   %s",       same, el->Ident, CNN);
         dfsX10( "MFT: ",          dfsMftPure( el->MftRecordNr), same, " ");

         if (el->NameLen != 0)                  // attr name is in Unicode!
         {
            TxUnic2Ascii( &(el->Name), text, el->NameLen);
            TxPrint( "Name : %s%s%s\n", CBM, text, CNN);
         }
         else
         {
            TxPrint( "Name : -unnamed-\n");
         }
      }
      else
      {
         TxPrint( "Invalid element length %hu at offset %hu\n", this, done);
         rc = DFS_BAD_STRUCTURE;
      }
   }
   RETURN (rc);
}                                               // end 'dfsNtfsAttributList'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display NTFS MFT record Standard Attribute
/*****************************************************************************/
static ULONG dfsNtfsAttrStandard
(
   A_STANDARD         *at                       // IN    Attribute value data
)
{
   ULONG               rc = 0;                  // rc, sector match
   TXTM                tbuf;                    // temporary buffer
   TXLN                text;

   ENTER();

   strcpy( text, "         Created  : ");
   strcat( text, dfsNtfsTime2str( &at->Create, tbuf));
   strcat( text, "   ");
   if ((at->Modify.hi != at->Create.hi) ||
       (at->Modify.lo != at->Create.lo)  )
   {
      strcat( text, "Modified : ");
      strcat( text, dfsNtfsTime2str( &at->Modify, tbuf));
   }
   TxPrint( "%s\n", text);
   if ((at->Access.hi != at->Create.hi) ||
       (at->Access.lo != at->Create.lo)  )
   {
      strcpy( text, "         Accessed : ");
      strcat( text, dfsNtfsTime2str( &at->Access, tbuf));
      strcat( text, "   ");
      if ((at->Modfil.hi != at->Modify.hi) ||
          (at->Modfil.lo != at->Modify.lo)  )
      {
         strcat( text, "Rec. mod : ");
         strcat( text, dfsNtfsTime2str( &at->Modfil, tbuf));
      }
      TxPrint( "%s\n", text);
   }
   sprintf( text, "    StdAttributes : 0x%8.8x = ", at->FileAttributes);
   dfstrNtfsAttrib( text, at->FileAttributes);
   TxPrint("%s\n", text);
   if (at->ReservedField != 0)
   {
      TxPrint( "      ReservedFld : 0x%8.8x\n",  at->ReservedField);
   }
   RETURN (rc);
}                                               // end 'dfsNtfsAttrStandard'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display NTFS MFT record Filename Attribute
// 20171114 JvW Use baseMft to show full-path-to-root for each name attribute
/*****************************************************************************/
static ULONG dfsNtfsAttrFilename
(
   ULN64               mftLsn,                  // IN    Base MFT lsn
   A_FILENAME         *at                       // IN    Attribute value data
)
{
   ULONG               rc = 0;                  // rc, sector match
   TXLN                fn;                      // ASCII buffer
   TXTM                dtime;                   // ASCII string buffer
   TXLN                fpath;

   ENTER();

   TxUnic2Ascii( at->Filename, fn, at->FileNameLength);

   TxPrint(     "        ParentMft : 0x%s%12.12llX%s sequence nr : 0x%4.4hX   fnType:0x%2.2hx  = %s%s%s\n",
                                                CBC, dfsMftPure( at->MftParentMft),
                                                CNN, dfsMftSqnr( at->MftParentMft),
                                                                 at->FileNameType,
                                                CNC,            (at->FileNameType == FN_DOS83) ? "8.3 " :
                                                                (at->FileNameType == FN_UNICO) ? "Long" :
                                                                                                 "Link", CNN);

   dfsNtfsFindRootMft( mftLsn, FALSE, (at->FileNameType == FN_DOS83),
                           dfsMftPure( at->MftParentMft), fpath, NULL);

   TxPrint(     "        Full Path : %s%s%s\n", CBY, fpath, CNN);

   TxPrint(     "         Filename : %s%s%s\n", CBG, fn, CNN);

   TxPrint(     "         Created  : %s      ", dfsNtfsTime2str( &at->Create, dtime));
   if ((at->Modify.hi != at->Create.hi) || (at->Modify.lo != at->Create.lo))
   {
      TxPrint( "Modified : %s", dfsNtfsTime2str( &at->Modify, dtime));
   }
   if ((at->Access.hi != at->Create.hi) || (at->Access.lo != at->Create.lo))
   {
      TxPrint("\n         Accessed : %s      ", dfsNtfsTime2str( &at->Access, dtime));
      if ((at->Modfil.hi != at->Modify.hi) || (at->Modfil.lo != at->Modify.lo))
      {
         TxPrint("Rec. mod : %s", dfsNtfsTime2str( &at->Modfil, dtime));
      }
   }
   TxPrint("\n");

   sprintf(       fn, "         All-size : ");
   dfstrSz64Byte( fn, "",       at->Allocated, "  ");
   dfstrSz64Byte( fn, "Real :", at->RealSize,  "\n");
   TxPrint( "%s", fn);

   if (at->FileAttributes)
   {
      sprintf( fn, "    StdAttributes : 0x%8.8x = ", at->FileAttributes);
      dfstrNtfsAttrib( fn, at->FileAttributes);
      TxPrint("%s\n", fn);
      if (at->FileAttributes & FA_REPARSE_POINT)
      {
         TxPrint( "      Reparse-Tag : 0x%8.8x\n",  at->ReparsePointTag);
      }
   }
   nav.up = dfsNtfsMft2Lsn( dfsMftPure( at->MftParentMft));
   RETURN (rc);
}                                               // end 'dfsNtfsAttrFilename'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display NTFS MFT record IdxRoot Attribute
/*****************************************************************************/
static ULONG dfsNtfsAttrIndxRoot
(
   ULN64               mftnr,                   // IN    Current MFT number
   A_IDXROOT          *at,                      // IN    Attribute value data
   ULONG               size                     // IN    size of attribute
)
{
   ULONG               rc = 0;                  // rc, sector match

   ENTER();

   dfsNtfsDirEntries( mftnr, at->EntryList, (size > 0x40));
   TxPrint( "\n");

   RETURN (rc);
}                                               // end 'dfsNtfsAttrIndxRoot'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display NTFS MFT record VolumeName
/*****************************************************************************/
static ULONG dfsNtfsVolumeName
(
   S_MFTFILE          *sd,                      // IN    base MFT rec, fixed-up
   A_VOLNAME          *at,                      // IN    Attribute value data
   ULONG               size                     // IN    size of attribute
)
{
   ULONG               rc = 0;                  // rc, sector match
   TXLN                fn;                      // ASCII buffer

   ENTER();

   TxUnic2Ascii( at->VolumeName, fn, (USHORT) (size / 2));
   TxPrint(     "         Vol-name : %s%s%s\n", CBY, fn, CNN);
   RETURN (rc);
}                                               // end 'dfsNtfsVolumeName'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display NTFS MFT record VolumeInfo
/*****************************************************************************/
static ULONG dfsNtfsVolumeInfo
(
   S_MFTFILE          *sd,                      // IN    base MFT rec, fixed-up
   A_VOLINFO          *at,                      // IN    Attribute value data
   ULONG               size                     // IN    size of attribute
)
{
   ULONG               rc = 0;                  // rc, sector match

   ENTER();

   TxPrint( "    NTFS  version : %s%hu.%hu%s = ", CBM, at->MajorVer,
                                                       at->MinorVer, CNN);
   if (at->MajorVer < 3)
   {
      TxPrint( "NT level, upto NT4-SP6\n");
   }
   else if (at->MinorVer == 0)
   {
      TxPrint( "Win-2000 level\n");
   }
   else if (at->MinorVer == 1)
   {
      TxPrint( "Win-XP level\n");
   }
   else
   {
      TxPrint( "NEWER than XP (.NET ?), %suntested with DFSee!%s\n", CBR, CNN);
   }
   dfsNtfsVolFlags( "   Dirty / CHKDSK : ", at->VolFlags);
   RETURN (rc);
}                                               // end 'dfsNtfsVolumeInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display NTFS $Volume Flags value
/*****************************************************************************/
static void dfsNtfsVolFlags
(
   char               *lead,                    // IN    leading string
   USHORT              volFlags                 // IN    flags value
)
{
   TxPrint("%s0x%4.4hx     = %s%s%s\n", lead, volFlags,
      (volFlags & VF_VOLUME_IS_DIRTY)  ? CBR                      : CBG,
      (volFlags & VF_VOLUME_IS_DIRTY)  ? "Dirty, CHKDSK required" : "Normal", CNN);
   if (volFlags & VF_RESIZE_LOG_FILE ) TxPrint("   Resize logfile : Yes\n");
   if (volFlags & VF_UPGRADE_ON_MOUNT) TxPrint("   Upgr. on mount : Yes\n");
   if (volFlags & VF_MOUNTED_ON_NT4  ) TxPrint("   Mounted on NT4 : Yes\n");
   if (volFlags & VF_DEL_USN_UNDERWAY) TxPrint("   Del USN underw : Yes\n");
   if (volFlags & VF_REPAIR_OBJ_ID   ) TxPrint("   Repair obj. id : Yes\n");
   if (volFlags & VF_MODIF_BY_CHKDSK ) TxPrint("   Modified by    : CHKDSK\n");
   if (volFlags & VF_UNKNOWN_VALUE   ) TxPrint("   Unknown value! : %s0x%4.4x%s\n", CBR,
       volFlags & VF_UNKNOWN_VALUE, CNN);
}                                               // end 'dfsNtfsVolFlags'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display NTFS MFT record Resident Attribute in a hex-dump
/*****************************************************************************/
static ULONG dfsNtfsResAttribHex
(
   S_MFTFILE          *sd,                      // IN    base MFT rec, fixed-up
   void               *at,                      // IN    Attribute value data
   ULONG               size                     // IN    size of attribute
)
{
   ULONG               rc = 0;                  // rc, sector match
   ULONG               offset  = ((ULONG) (((char *) at) - ((char *) sd)));
   ULONG               maxsize = (ntfs->MftRecSize * SECTORSIZE) - offset;

   ENTER();

   TxDisplayHex( "", (BYTE *) at, min( size, maxsize), offset);
   RETURN (rc);
}                                               // end 'dfsNtfsResAttribHex'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// DFS NTFS write-file to disk (SaveTo MFT first data-attribute to file)
// 20171116 JvW Changed 2nd param to be an optional 'desried parent' MFT number
/*****************************************************************************/
static ULONG dfsNtfsFileSaveAs
(
   ULN64               lsn,                     // IN    MFT LSN
   ULN64               parent,                  // IN    desired parent MFT or 0
   char               *path,                    // IN    destination path
   void               *recp                     // INOUT recovery parameters
)
{
   ULONG               rc = NO_ERROR;
   BYTE                st = ST_UDATA;
   S_RECOVER_PARAM    *param = (S_RECOVER_PARAM *) recp;
   S_MFTFILE          *mft;
   DFSISPACE           isp;                     // allocation SPACE info
   TXLN                uniq;                    // unique filename prefix
   TXLN                fname;                   // filename component
   TXLN                safname;                 // filename from SaveAs
   ULN64               desiredParent = parent;  // desired parent MFT

   ENTER();
   TRARGS(("MFT LSN:0x%llX, desired-parent:0x%llx to path: '%s'\n", lsn, parent, path));

   isp.meta = lsn;                              // may be updated for extern-resident

   if ((rc = dfsNtfsReadFixMft( lsn, &st, &mft)) == NO_ERROR)
   {
      if ((rc = dfsNtfsSpAlloc( mft, MA_DATA, IA_ANY, NULL, &isp)) == NO_ERROR)
      {
         rc = dfsNtfsMft2NameInfo( mft, FALSE, &desiredParent, NULL, fname); // get filename

         if ((param->newname == NULL) || (*param->newname == 0)) // no newname present
         {
            if (strlen( fname) == 0)            // name was not retrieved
            {
               sprintf( fname, "-mft-%12.12llx-at-%12.12llx-", dfsNtfsLsn2Mft( lsn), lsn);
            }
         }
         else                                   // rename specified
         {
            strcpy( fname, param->newname);
         }
         if (param->unique)                     // force unique naming
         {                                      // on PATH and FILE components
            sprintf(uniq, "%12.12llX_%s", lsn, fname);
            strcpy( fname, uniq);               // and move back to fname
         }
         if (param->recFname)
         {
            strcpy( param->recFname, fname);    // return full base-filename
         }
         rc = dfsSspaceFileSaveAs( &isp, ((mft->MftFlags & AF_DIRECTORY) != 0),
                                         ((mft->MftFlags & AF_REC_INUSE) == 0),
                                           param->noAllocCheck, param->name83, path, fname, safname);
         if ((rc == NO_ERROR)       ||          // set original timestamps
             (rc == DFS_CMD_WARNING) )          // even when allocation errors present
         {
            S_MFTATTR  *at;                     // attribute data

            if (dfsNtfsGetMftAttrRef( mft, MA_STANDARD, IA_ANY, NULL, &at) == NO_ERROR)
            {
               A_STANDARD *sad = (A_STANDARD *) (((BYTE *) at) + at->R.Offset);
               time_t      cre = txWinFileTime2t( &sad->Create, 0);
               time_t      acc = txWinFileTime2t( &sad->Access, 0);
               time_t      mod = txWinFileTime2t( &sad->Modify, 0);

               TxSetFileTime( safname, &cre, &acc, &mod);

               if (param->recFname)
               {
                  strcpy( param->recFname, safname);
               }
            }
         }
         TxFreeMem( isp.space);                 // non-res SPACE or res data
      }
      TxFreeMem( mft);                          // MFT record
   }
   RETURN(rc);
}                                               // end 'dfsNtfsFileSaveAs'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Make filesystem usage map dump on TxPrint output
// 20170530: Enhanced with Smart-usage prediction
/*****************************************************************************/
static ULONG dfsNtfsAllocMap
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *options,                 // IN    display options
   void               *data                     // OUT   dummy
)
{
   ULN64               i;                       // index in data-area
   ULONG               b;                       // byte-index for one mapchar
   ULN64               cClus = 0;               // Clusters in current char
   ULN64               lClus = 0;               // Clusters in current line
   ULN64               mClus = 0;               // Clusters in current map
   ULN64               uClus = 1;               // Last cluster that is in use
   ULONG               l;                       // line number, 0 based
   ULONG               a;                       // index in display-array
   ULN64               perc;                    // percentage
   TX1K                ascii;                   // display-array
   ULONG               spe;                     // sectors per entry
   ULONG               acl;                     // ascii alloc chars per line
   ULONG               cpc;                     // clusters per display-char
   ULONG               spc;                     // sectors per display-char
   ULN64               size;                    // nr of clusters to map
   BOOL                verbose = (*options != '@');
   ULONG               bsSmart;
   ULN64               unallocSmart = 0;        // Size of THIS unallocated area
   BOOL                oneLineProgress = FALSE; // alloc in one line, output per character

   ENTER();

   dfsa->FsUnallocSmart = 0;                    // will be recalculated now
   bsSmart = dfsGetBufferSize( DFSOPTIMALBUF, DFSMAXBUFSIZE); // for SMART prediction

   spe   = ntfs->ClustSize;
   size  = ntfs->Clust;                         // Cluster 0 .. last-valid

   acl   = dfsAllocCharsPerLine( 'c');
   cpc   = dfsAllocItemsPerChar( 'l', options, size, acl);
   if (cpc >= (size / acl))                     // will be a single line
   {
      oneLineProgress = TRUE;                   // one char at a time, no ascii string used
   }
   spc   = cpc * spe;
   dfsProgressInit( 0, size * spe, 0, "Get ALLOCATION info, at Sector:", dfstStoreDpid( DFSTORE), DFSP_STAT, 0);

   if (verbose)
   {
      dfsSz64("Size for one line : ", (ULN64) acl * spc, "  ");
      TxPrint("with %3lu characters,", acl);
      dfsSiz8(" size : ", spc, "\n");
      TxPrint(" %16.16s = Allocation grade, empty to full.     ", mapchar);
      dfsSiz8(" SmartBuf size : ", bsSmart, "\n");
   }
   TxPrint(   "          %s%*.*s%s\n", CBC, acl, acl, BLIN, CNN);
   dfsX10( CNZ, dfsNtfsClust2Lsn(0), CNZ, "");   // display address for first block
   TxPrint("%s%s", CBC, CNC);

   for (i=0, a=0, l=0; (i < size) && (!TxAbort());)
   {
      if (a == 0)
      {
         memset(ascii, 0, TXMAXLN);
         lClus = 0;
      }
      for (cClus=0, b=0; (b < cpc) && (i < size); b++, i++)
      {
         if (dfsNtfsBitmapCache(i, NULL))       // cluster in use ?
         {
            if (unallocSmart != 0)              // first in-use after unallocated?
            {
               if (unallocSmart >= bsSmart)                  //- if at least ONE smart buffer
               {
                  unallocSmart -= (bsSmart / 2);             //- compensate for alignment issues
                  unallocSmart -= (unallocSmart % bsSmart);  //- clip to whole smart buffer size

                  dfsa->FsUnallocSmart += unallocSmart;      //- add this area to total
               }
               unallocSmart = 0;                // reset for next area to come
            }
            cClus++;
            uClus = i;                          // last in-use one sofar
         }
         else                                   // unallocated, Smart predict
         {
            unallocSmart += spe;                // add to size THIS area
         }
      }
      lClus += cClus;

      dfsProgressShow( (i * spe) + spc, 0, 0, NULL);

      TRACES(("lClus 2nd Loop i: %llu, b:%u = %llu\n", i, b, lClus));
      if (oneLineProgress == TRUE)
      {
         TxPrint( "%c", (char) ((cClus == 0) ? ' ' : mapchar[cClus * 8 / cpc])); // slow, acts as progress bar
      }
      else
      {
         ascii[a] = (char) ((cClus == 0) ? ' ' : mapchar[cClus * 8 / cpc]);
      }
      a++;
      if ((i && ((i%(acl*cpc)) == 0)) || (i >= size))
      {
         if (oneLineProgress == FALSE)
         {
            TxPrint( "%s", ascii);              // display accumulated chars (fast)
         }
         perc  = (ULN64)(100*lClus / (a*cpc));
         if (a == acl)
         {
            TxPrint("%s%s% 3lu%%\n", CBC, CNN, perc);
         }
         else
         {
            TxPrint("%s%.*s%s% 3llu%%\n", CBC, (int) (acl-a-1), BLIN, CNN, perc);
         }
         if (i < size)
         {
            dfsX10( CNZ, dfsNtfsClust2Lsn(i), CNZ, "");
            TxPrint("%s%s", CBC, CNC);
            a = 0;                              // keep a value on last line
         }
         mClus += lClus;
         l++;
      }
   }
   if (unallocSmart != 0)                           //- unalloc area pending?
   {
      if (unallocSmart >= bsSmart)                  //- if at least ONE smart buffer
      {
         unallocSmart -= (bsSmart / 2);             //- compensate for alignment issues
         unallocSmart -= (unallocSmart % bsSmart);  //- clip to whole smart buffer size

         dfsa->FsUnallocSmart += unallocSmart;      //- add this area to total
      }
   }
   TxPrint("          %s%.*s%s\n", CBC, (USHORT) a, BLIN, CNN);

   if (TxAbort())
   {
      dfsa->FsUnallocSmart = 0;                 // signal info NOT available!
      TxPrint( "\nAllocation check and display aborted! Used/Free info not available.\n");
   }
   else
   {
      if (size != 0)                            // avoid devide by 0
      {
         ULN64          used = mClus * spe;

         dfsa->FsUnallocated = size * spe - used; // free space (optimize hint)

         dfsSz64("Unallocated sects : ", dfsa->FsUnallocated, " ");
         dfsSz64("SmartUse ", dfsa->FsUnallocSmart, " ");
         TxPrint("=% 5.1lf%%\n", (double) (100.0 * ((double) dfsa->FsUnallocSmart) / (double) (size * spe)));

         dfsSz64("Allocated sectors : ", used, " ");
         dfsSz64("of total ", size * spe, " ");
         TxPrint("=% 5.1lf%%\n", (double) (100.0 * ((double) mClus) / (double) size));
      }
      dfsa->FsTruncPoint = dfsNtfsClust2Lsn( uClus +1); // first free (truncate)
      dfsa->FsLastInUse  = dfsa->FsTruncPoint -1;
      dfsSz64( "Minimum vol. size : ", dfsa->FsTruncPoint, " (for  Resizing)\n");
   }
   dfsProgressTerm();
   RETURN (NO_ERROR);
}                                               // end 'dfsNtfsAllocMap'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Truncate the filesystem size, keeping all data intact; (2 sectors in br)
/*****************************************************************************/
static ULONG dfsNtfsResizeFS
(
   ULN64               newsize,                 // IN    new size in sectors
   ULN64               d2,                      // IN    dummy
   char               *string,                  // IN    dummy
   void               *data                     // IN    partition info
)
{
   ULONG               rc  = NO_ERROR;
   DFSPARTINFO        *p   = (DFSPARTINFO *) data;
   TXTM                text;

   ENTER();
   TRARGS(("New size: %llx\n", newsize));

   //- to be refined, add check of LOG area to see if filesystem is CLEAN

   strcpy( text, "");
   dfstrSz64Bps( text, "to a new size of: ", newsize, (USHORT) p->bpsector, "");
   if ((dfsa->batch) || (TxConfirm( 5138, "Resize this NTFS filesystem %s ? [Y/N] : ", text)))
   {
      S_BOOTR         *br = (S_BOOTR *) brec;

      br->nt.VolSize = newsize -1;
      if (DFSTORE_WRITE_ALLOWED)
      {
         TxPrint(    "Updating size-information fields in the bootsector ...\n");
         rc = dfsWrite( LSN_BOOTR, 1, brec);    // point of no return, RC not
         if (rc == NO_ERROR)                    // affected by remaining stuff
         {
            TxPrint( "Updating sparse data stream in $Badclus for new size : ");
            if ((dfsNtfsResizeBadClus( newsize)) == NO_ERROR)
            {
               TxPrint( "OK\n");
            }
            else                                // not done (completely)
            {
               TxPrint( "incomplete\n");
            }

            TxPrint( "Updating allocation in the $Bitmap file for new size : ");
            if ((dfsNtfsResizeBitmap( newsize)) == NO_ERROR)
            {
               TxPrint( "OK\n");
            }
            else                                // not done (completely)
            {
               TxPrint( "incomplete\n");
            }

            //- check if expanded bitmap is already used now for set allocation!
            TxPrint( "Updating the spare bootsector at end of the volume   : ");
            if ((dfsWrite( newsize -1, 1, brec)) == NO_ERROR)
            {
               //- set allocation for spare, even if outside cluster-area
               dfsSetAllocForRange( newsize -1, 1, 1, TRUE);
               TxPrint( "OK\n");
            }
            else                                // not written at all
            {
               TxPrint( "failed\n");
            }

            TxPrint("\nNTFS filesystem resized %s ...\n", text);
         }
      }
      else
      {
         rc = DFS_READ_ONLY;
      }
   }
   else
   {
      rc = DFS_NO_CHANGE;
   }
   RETURN (rc);
}                                               // end 'dfsNtfsResizeFS'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Update all attributes of the $Bitmap file for a new size of the filesystem
/*****************************************************************************/
static ULONG dfsNtfsResizeBitmap                // RET   result
(
   ULN64               size                     // IN    new size in sectors
)
{
   ULONG               rc = NO_ERROR;
   S_MFTFILE          *mft;
   BYTE                st  = ST_UDATA;
   ULN64               lsn = dfsNtfsMft2Lsn( MFT_BITMAPF);
   ULONG               bmbyte    = TXMULTI08(((size / ntfs->ClustSize) +7) / 8);
   ULONG               bmsect    = ((bmbyte -1) / SECTORSIZE) +1;
   ULONG               bmclus    = ((bmsect -1) / ntfs->ClustSize) +1;
   ULONG               sectalloc = bmclus * ntfs->ClustSize;
   ULONG               bytealloc = sectalloc * SECTORSIZE;
   S_MFTATTR          *at;                      // Attribute reference

   ENTER();
   TRACES(( "Size:%llx  bmbyte:%8.8x  bmsect:%8.8x  bmclus:%8.8x  bytes:0x%x\n",
             size,      bmbyte,       bmsect,       bmclus,       bytealloc));

   if ((rc = dfsNtfsReadFixMft( lsn, &st, &mft)) == NO_ERROR)
   {
      if (dfsNtfsGetMftAttrRef( mft, MA_FILENAME, IA_ANY, NULL, &at) == NO_ERROR)
      {
         if ((at->DataExtern == FALSE) && (at->Type == MA_FILENAME))
         {
            A_FILENAME         *fnat;           // Filename Attribute data

            fnat = (A_FILENAME *) (((BYTE *) at) + at->R.Offset);

            if (dfsNtfsGetMftAttrRef( mft, MA_DATA, IA_ANY, NULL, &at) == NO_ERROR)
            {
               if ((at->DataExtern) && (at->Type == MA_DATA))
               {
                  DFSISPACE  is;                // data for bitmap
                  USHORT     rlsize;
                  BYTE      *rldata;            // runlist

                  is.clsize = ntfs->ClustSize;
                  if ((rc = dfsNtfsSpNonRes( mft, MA_DATA, IA_ANY, NULL, &is)) == NO_ERROR)
                  {
                     if ((is.space != NULL) && (is.chunks != 0)) // non-res!
                     {
                        //- Reset old bitmap slack-area (8-byte granularity)
                        //- for current size, making those clusters FREE
                        dfsNtfsBitmapSetSlack( ntfs->Clust, FALSE);

                        if (bytealloc <= fnat->Allocated) // smaller, truncate
                        {
                           //- Set new bitmap slack-area (8-byte granularity)
                           dfsNtfsBitmapSetSlack((size / ntfs->ClustSize), TRUE);

                           //- set alloc bits to FREE, then truncate
                           dfsSetAllocForSpace( &is, sectalloc, FALSE);
                           dfsSspaceUpdateSize( &is, sectalloc, FALSE);

                           //- Make new runList for truncated space
                           if ((rc = dfsNtfsMkRunList( is.chunks, is.space,
                                           &rlsize, &rldata )) == NO_ERROR)
                           {
                              fnat->Allocated = bytealloc;
                              fnat->RealSize  = bmbyte; // set new alloc/real size

                              if (rlsize <= (at->Length - at->N.Offset))
                              {
                                 memcpy( ((BYTE *) at) + at->N.Offset,
                                         rldata, rlsize);

                                 at->N.SegFirst   = 0;
                                 at->N.SegLast    = bmclus -1;
                                 at->N.Allocated  = bytealloc;
                                 at->N.Size       = bmbyte;
                                 at->N.Compressed = bmbyte;

                                 rc = dfsNtfsBitmapFlush( FALSE); // to disk
                              }
                              TxFreeMem( rldata); // free RunList data
                           }
                        }
                        else                    // expand FS, larger bitmap
                        {
                           DFSISPACE    ex;     // expand allocation for bitmap

                           ULONG   bmapSize = dfsSspaceSectors( FALSE, is.chunks, is.space);
                           ULONG   adjacent = dfsSspaceRsn2Lsn( is.chunks, is.space, bmapSize -1) +1;

                           TRACES(( "Prefered new area: 0x%llx size: %8.8x\n", adjacent, sectalloc - bmapSize));

                           ex.chunks = 1;
                           ex.space  = TxAlloc( NTFS_MAXBMPEXT, sizeof( S_SPACE));
                           ex.clsize = ntfs->ClustSize;

                           if (ex.space != NULL)
                           {
                              ex.space[0].start = adjacent;
                              ex.space[0].size  = sectalloc - bmapSize;

                              //- check if prefered area is (still) FREE
                              if (dfsCheckAllocForSpace( &ex, 0, FALSE, NULL) != NO_ERROR)
                              {
                                 //- not FREE, find other area using the SLT
                                 //- create SLT now, if not there yet

                                 TxPrint( "needs expanding ...\n");
                                 if ((dfsSlTableStatus(NULL)) != SLT_READY)
                                 {
                                    TxPrint( "Building SLT to find free disk-space");
                                    dfsa->sltProgress = TRUE;
                                    dfsSlTableCreate(dfsa->Fsi); // build SLT now
                                    TxPrint( "\n"); // terminate progress bars
                                 }
                                 if ((dfsSlTableStatus(NULL)) == SLT_READY)
                                 {
                                    dfsa->sltProgress = FALSE; // no screen output
                                    TxPrint( "Expanding the $Bitmap system "
                                             "file for larger FS now  : ");
                                    //- then use allocation function to get S_SPACE

                                    if (dfsSlTableSearchFree( sectalloc - bmapSize,
                                                              ntfs->ClustSize,
                                                              NTFS_MAXBMPEXT,
                                                             &ex.chunks,
                                                              ex.space))
                                    {
                                       dfsSlTableAllocate( sectalloc - bmapSize,
                                                           ntfs->ClustSize,
                                                           ex.chunks,
                                                           ST_FDATA, lsn, 0,
                                                           ex.space);
                                    }
                                    else        // no freespace on disk ...
                                    {
                                       rc = DFS_CMD_FAILED;
                                    }
                                    dfsa->sltProgress = TRUE; // end no output
                                 }
                                 else           // SLT not built, aborted ?
                                 {
                                    rc = DFS_CMD_FAILED;
                                 }
                              }
                              else              // direct allocate, set bitmap!
                              {
                                 //- set alloc bits to ALLOCATED
                                 dfsSetAllocForSpace( &ex, 0, TRUE);
                              }
                              if (rc == NO_ERROR)
                              {
                                 //- initialize expanded bitmap sectors to zero
                                 st = 0;        // wipe pattern, zero
                                 rc = dfsWipeTailForSpace( &ex, 0, &st, 1);
                                 if (rc == NO_ERROR)
                                 {
                                    //- Join old and new area of bitmap S_SPACE
                                    rc = dfsSspaceJoin( is.chunks,  is.space,
                                                        ex.chunks,  ex.space,
                                                       &is.chunks, &is.space);
                                    if (rc == NO_ERROR)
                                    {
                                       //- Make new runList for expanded space
                                       if ((rc = dfsNtfsMkRunList( is.chunks, is.space,
                                                       &rlsize, &rldata )) == NO_ERROR)
                                       {
                                          DFSISPACE swap; // to swap with BM cache

                                          fnat->Allocated = bytealloc;
                                          fnat->RealSize  = bmbyte; // set new alloc/real size

                                          if (rlsize > (at->Length - at->N.Offset))
                                          {
                                             BYTE  *from = ((BYTE *) at) + at->Length;
                                             ULONG  grow = rlsize - (at->Length - at->N.Offset);
                                             ULONG  rest = (mft->AllLength) -
                                                           (from - ((BYTE *) mft)) - grow;

                                             TRACES(( "Move %u bytes REST over %u bytes\n",
                                                       rest, grow));

                                             mft->RecLength += grow; // must be 8-multiple
                                             at->Length     += grow;
                                             memmove( from  +  grow, from, rest); // overlap!
                                          }
                                          memcpy( ((BYTE *) at) + at->N.Offset, rldata, rlsize);

                                          TxFreeMem( rldata); // free RunList data

                                          at->N.SegFirst   = 0;
                                          at->N.SegLast    = bmclus -1;
                                          at->N.Allocated  = bytealloc;
                                          at->N.Size       = bmbyte;
                                          at->N.Compressed = bmbyte;

                                          dfsNtfsBitmapFlush( FALSE); // to disk

                                          swap.chunks         = ntfs->Bm.All.chunks;
                                          swap.space          = ntfs->Bm.All.space;

                                          ntfs->Bm.All.chunks = is.chunks;
                                          ntfs->Bm.All.space  = is.space;
                                          ntfs->Bm.LimitLsn = dfsSspaceSectors(
                                                                 FALSE,
                                                                 ntfs->Bm.All.chunks,
                                                                 ntfs->Bm.All.space) * //- bitmap sectors present
                                                            (8 * dfsGetSectorSize()) * //- #bits, each bit is one cluster
                                                                 ntfs->ClustSize;      //- #sectors represented
                                          is.chunks         = swap.chunks;
                                          is.space          = swap.space;

                                          //- Set new bitmap slack-area (8-byte granularity)
                                          dfsNtfsBitmapSetSlack((size / ntfs->ClustSize), TRUE);
                                       }
                                    }
                                 }
                              }
                              TxFreeMem( ex.space);
                           }
                           else
                           {
                              rc = DFS_ALLOC_ERROR;
                           }
                        }
                     }
                     TxFreeMem( is.space);
                  }
               }
               else                             // internal Bitmap data
               {
                  rc = DFS_BAD_STRUCTURE;
               }
            }
            if (rc == NO_ERROR)                 // write modified BM-MFT back
            {
               rc = dfsNtfsFixWriteMft( lsn, mft);
            }
         }
         else                                   // external Bitmap filename
         {
            rc = DFS_BAD_STRUCTURE;
         }
      }
      TxFreeMem( mft);
   }
   RETURN (rc);
}                                               // end 'dfsNtfsResizeBitmap'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set/Reset slack bits for clusters upto multiple-of-64 for last CL
// Bitmap used filesize is a multiple of 8 bytes, extra bits beyond being
// used for the last real cluster must be set to 1 (allocated)
/*****************************************************************************/
static ULONG dfsNtfsBitmapSetSlack              // RET   result
(
   ULN64               size,                    // IN    new size in clusters
   BOOL                set                      // IN    allocation SET or reset
)
{
   ULONG               rc = NO_ERROR;
   ULN64               slack = TXMULTI64(size); // rounded up to 64-multiple
   ULN64               cl;

   ENTER();
   TRACES(( "Size:0x%llx cl, slack:0x%llx set to:%s\n", size, slack, (set) ? "ALLOCATED" : "FREE"));

   for (cl = size; cl < slack; cl++)            // slack range to set
   {
      rc = dfsNtfsSetCluster( cl, set);         // set/reset allocation bit
   }
   RETURN (rc);
}                                               // end 'dfsNtfsBitmapSetSlack'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Update all attributes of the $BadClus file for a new size of the filesystem
/*****************************************************************************/
static ULONG dfsNtfsResizeBadClus               // RET   result
(
   ULN64               size                     // IN    new size in sectors
)
{
   ULONG               rc = NO_ERROR;
   S_MFTFILE          *mft;
   BYTE                st  = ST_UDATA;
   ULN64               lsn = dfsNtfsMft2Lsn( MFT_BADCLUS);
   S_MFTATTR          *at;                      // Attribute reference
   DFSISPACE           is;                      // sparse data stream

   ENTER();
   TRACES(( "Size:0x%llx\n", size));

   if ((rc = dfsNtfsReadFixMft( lsn, &st, &mft)) == NO_ERROR)
   {
      if (dfsNtfsGetMftAttrRef( mft, MA_DATA, IA_ANY, "$Bad", &at) == NO_ERROR)
      {
         if ((at->DataExtern) && (at->Type == MA_DATA))
         {
            is.clsize = ntfs->ClustSize;
            if ((rc = dfsNtfsSpNonRes( mft, MA_DATA, IA_ANY, "$Bad", &is)) == NO_ERROR)
            {
               if ((is.space != NULL) && (is.chunks != 0)) // non-res!
               {
                  USHORT     rlsize;
                  BYTE      *rldata;            // runlist
                  ULONG      clusters  = size     / ntfs->ClustSize;
                  LLONG      sectalloc = clusters * ntfs->ClustSize;

                  dfsSspaceUpdateSize( &is, sectalloc, TRUE);

                  //- Make new runList for truncated/expanded space
                  if ((rc = dfsNtfsMkRunList( is.chunks, is.space,
                                              &rlsize, &rldata )) == NO_ERROR)
                  {
                     if (rlsize <= (at->Length - at->N.Offset))
                     {
                        memcpy( ((BYTE *) at) + at->N.Offset, rldata, rlsize);

                        at->N.SegFirst   = 0;
                        at->N.SegLast    = clusters -1;
                        at->N.Allocated  = sectalloc * dfsGetSectorSize();
                        at->N.Size       = at->N.Allocated;

                        //- note: compressed size stays the same,
                        //-       represents the actual Bad space
                     }
                     else                       // run list grown, error!
                     {
                        rc = DFS_CMD_FAILED;
                     }
                     TxFreeMem( rldata);        // free RunList data
                  }
               }
               TxFreeMem( is.space);
            }
            if (rc == NO_ERROR)                 // write modified BC-MFT back
            {
               rc = dfsNtfsFixWriteMft( lsn, mft);
            }
         }
         else                                   // internal Bitmap data
         {
            rc = DFS_BAD_STRUCTURE;
         }
      }
      TxFreeMem( mft);
   }
   RETURN (rc);
}                                               // end 'dfsNtfsResizeBadClus'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Reset all attributes of the $BadClus file to represent ZERO bad clusters
/*****************************************************************************/
static ULONG dfsNtfsResetBadClus                // RET   result
(
   ULN64               size                     // IN    part size in sectors
)
{
   ULONG               rc = NO_ERROR;
   S_MFTFILE          *mft;
   BYTE                st  = ST_UDATA;
   ULN64               lsn = dfsNtfsMft2Lsn( MFT_BADCLUS);
   S_MFTATTR          *at;                      // Attribute reference
   DFSISPACE           is;                      // sparse data stream

   ENTER();
   TRACES(( "Size:0x%llx\n", size));

   if ((rc = dfsNtfsReadFixMft( lsn, &st, &mft)) == NO_ERROR)
   {
      if (dfsNtfsGetMftAttrRef( mft, MA_DATA, IA_ANY, "$Bad", &at) == NO_ERROR)
      {
         if ((at->DataExtern) && (at->Type == MA_DATA))
         {
            is.clsize = ntfs->ClustSize;
            if ((rc = dfsNtfsSpNonRes( mft, MA_DATA, IA_ANY, "$Bad", &is)) == NO_ERROR)
            {
               if ((is.space != NULL) && (is.chunks != 0)) // non-res!
               {
                  USHORT     rlsize;
                  BYTE      *rldata;                    // runlist
                  ULN64      clusters  = size     / ntfs->ClustSize;
                  ULN64      sectalloc = clusters * ntfs->ClustSize;
                  ULN64      bads = dfsSspaceSectors( TRUE, is.chunks, is.space);

                  if (bads != 0)
                  {
                     dfsSz64( "\nTotal bad sectors : ", bads, ", resetting ...\n");
                     dfsNtfsSetSpaceAlloc( &is, FALSE); //- FREE in $BitMap

                     is.chunks = 1;                     //- single extent
                     is.space[0].start = LSN_SPARSE;    //- SPARSE element
                     is.space[0].size  = sectalloc;     //- full partition size

                     //- Make new runList for ZERO-bad-cluster space
                     if ((rc = dfsNtfsMkRunList( is.chunks, is.space,
                                                 &rlsize, &rldata )) == NO_ERROR)
                     {
                        if (rlsize <= (at->Length - at->N.Offset))
                        {
                           memcpy( ((BYTE *) at) + at->N.Offset, rldata, rlsize);

                           at->N.SegFirst   = 0;
                           at->N.SegLast    = clusters -1;
                           at->N.Allocated  = sectalloc * dfsGetSectorSize();
                           at->N.Size       = at->N.Allocated;
                           at->N.Compressed = 0; // NO bad clusters

                           if (rlsize <= 8)       // we are back at default list size
                           {
                              S_MFTATTR  *last;   // Sentinel, end-of-list element

                              at->Length = 0x50;  // set to default attribute size

                              last = (S_MFTATTR *) ((BYTE *) at) + at->Length;

                              last->Type   = MA_END_LIST;
                              last->Length = 0;
                           }
                           TxPrint( "$BadClus administration has been "
                                    "reset to ZERO bad sectors.\n");
                        }
                        else                    // run list grown, error!
                        {
                           rc = DFS_CMD_FAILED;
                        }
                        TxFreeMem( rldata);     // free RunList data
                     }
                  }
                  else                          // nothing to do ...
                  {
                     TxPrint( "\nNo bad sectors present, no action required ...\n");
                  }
               }
               TxFreeMem( is.space);
            }
            if (rc == NO_ERROR)                 // write modified BC-MFT back
            {
               rc = dfsNtfsFixWriteMft( lsn, mft);
            }
         }
         else                                   // internal Bitmap data
         {
            rc = DFS_BAD_STRUCTURE;
         }
      }
      TxFreeMem( mft);
   }
   RETURN (rc);
}                                               // end 'dfsNtfsResetBadClus'
/*---------------------------------------------------------------------------*/

