//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// DFSee utility services, Arealist
//
// Author: J. van Wijk
//
// JvW  26-08-2005 Initial version
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfshpfs.h>                            // HPFS structure defs
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS  navigation and defs
#include <dfsutil.h>                            // DFS utility functons
#include <dfsalist.h>                           // sector Area list handling

/*****************************************************************************/
// Allocate a new Sector-Area list with description, empty
/*****************************************************************************/
S_AREALIST *dfsNewAreaList                      // RET   new list or NULL
(
   char               *desc                     // IN    new list description
)
{
   S_AREALIST         *list;                    // new area list

   ENTER();

   if ((list = TxAlloc( 1, sizeof( S_AREALIST))) != NULL)
   {
      strcpy( list->descr, desc);
   }
   RETURN (list);
}                                               // end 'dfsInitAreaList'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Free area list and attched elements
/*****************************************************************************/
void dfsFreeAreaList
(
   S_AREALIST        **listAddress              // IN    address of list
)
{
   S_AREALIST         *list = *listAddress;

   ENTER();

   if (list && list->head)
   {
      while (list->head)
      {
         list->tail = list->head->next;
         TxFreeMem( list->head);
         list->head = list->tail;
      }
   }
   TxFreeMem( *listAddress);
   VRETURN ();
}                                               // end 'dfsFreeAreaList'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add new area, append to last element for adjacent, else create a new one
/*****************************************************************************/
S_AREAELEM *dfsAddListArea                      // RET   result element or NULL
(
   S_AREALIST         *list,                    // IN    area list
   ULN64               start,                   // IN    start of new area
   ULN64               size                     // IN    size area in sectors
)
{
   S_AREAELEM         *elem;                    // new area list element

   ENTER();

   TRACES(( "List:%8.8lx, start:0x%llx size:%llu\n", list, start, size));
   if (list->tail && (list->tail->start + list->tail->size == start)) // adjacent
   {
      elem = list->tail;
      elem->size += size;                       // add to current last element
   }
   else                                         // create a new element
   {
      if ((elem = TxAlloc( 1, sizeof( S_AREAELEM))) != NULL)
      {
         elem->start = start;
         elem->size  = size;

         if (list->tail)                        // we have elements already
         {
            list->tail->next = elem;
         }
         else
         {
            list->head = elem;
         }
         list->tail = elem;
      }
   }
   RETURN (elem);
}                                               // end 'dfsAddListArea'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display contents of a sector-area list
/*****************************************************************************/
void dfsDisplayAreaList
(
   S_AREALIST         *list                     // IN    area list
)
{
   ULN64               totalSize = 0;
   S_AREAELEM         *area;

   ENTER();

   if (list)
   {
      TxPrint( "\n%s\n", list->descr);

      for (area = list->head; area; area = area->next)
      {
         dfsX10(  "Area at sector nr : ", area->start, "", "  ");
         dfsSz64( "size : ", area->size, "\n");
         totalSize += area->size;
      }
   }
   dfsSz64( "\nArea list size is : ", totalSize, "\n");
   VRETURN();
}                                               // end 'dfsDisplayAreaList'
/*---------------------------------------------------------------------------*/

