//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// DFS display and analysis, JFS specific functions and definitions
//
// Author: J. van Wijk
//
// JvW  21-05-97   Initial version, derived from DFSJFS
//
#ifndef    DFSAJFS_H
   #define DFSAJFS_H

#include <dfsjfs.h>                             // JFS  structure defs

// JFS specific sector types
                                                // S   used on searching
                                                //  I  identification only
#define ST_JFSUP   's'                          // S   JFS  super-block
#define ST_JSBAD   'U'                          // S   JFS  BAD super-block
#define ST_JINOD   'f'                          // S   FS1 Inode (normal file)
#define ST_JIDIR   'D'                          // S   FS1 Inode (directory)
#define ST_JIDDI   'y'                          // S   Inode, deleted directory
#define ST_JIDEL   'z'                          // S   Inode, deleted file
#define ST_JINAG   'k'                          // S   JFS  Aggregate  Inode
#define ST_JIBAD   'u'                          // S   JFS  unused/BAD Inode
#define ST_JXTPG   'a'                          // S   Allocation, Xtree page
#define ST_JDTPG   'd'                          // S   Directory,  Dtree page

#define ST_JIMAG   'K'                          // S   aggregate Inode Map
#define ST_JIAGA   'v'                          // S   JFS  IAG for aggregate
#define ST_JIAGF   'w'                          // S   JFS  IAG for fileset-1
#define ST_JIMFS   'F'                          // S   fileset-1 Inode Map

#define ST_JEASD  ('E' + ST__INFO)              //  I  EA data
#define ST_JACLD  ('A' + ST__INFO)              //  I  ACL data
#define ST_JIMA2  ('m' + ST__INFO)              //  I  2nd aggr. Inode Map
#define ST_JI2AG  ('i' + ST__INFO)              //  I  JFS  2ndary agg Inode
#define ST_JIXFS  ('x' + ST__INFO)              //  I  fileset-1 Inode Extent
#define ST_JIXAG  ('S' + ST__INFO)              //  I  aggregate Inode Extent
#define ST_JIXA2  ('o' + ST__INFO)              //  I  2nd aggr. Inode Extent
#define ST_JBLAM  ('M' + ST__INFO)              //  I  Block allocation Map
#define ST_JFSCK  ('C' + ST__INFO)              //  I  FSCK work-area
#define ST_JINLL  ('J' + ST__INFO)              //  I  Inline-Log area
#define ST_JBOOT  ('B' + ST__INFO)              //  I  Micro-FS bootcode
#define ST_SLACK  ('c' + ST__INFO)              //  I  Last-block slack


#define JFS_INOSIG_LEN      4
#define JFS_AGGSIG_LEN      8
#define JFS_FS1SIG_LEN      8

typedef struct dfsjfsbmcache
{
   DFSISPACE           File;                    // space allocation for file
   ULN64               LimitLsn;                // largest LSN inside bitmap
   ULN64               FreeSect;                // free sectors, from bitmap
   ULONG               Rdmap;                   // cached relative dmap page
   BOOL                Dirty;                   // cache contents modified
   S_DMAP             *dmap;                    // DMAP page (maps 8K blocks)
} DFSJFSBMCACHE;                                // end of struct "dfsjfsbmcache"

typedef struct dfsjfsiname                      // inode2name caching element
{
   ULONG               Parent;                  // Parent Inode number
   char               *Name;                    // related name (allocated)
} DFSJFSINAME;                                  // end of struct "dfsjfsiname"

typedef struct s_imap                           // inode allocation info
{
   ULONG               Max;                     // Maximum Inode nr
   ULONG               Used;                    // Used Inodes
   ULONG               Alloc;                   // Allocated Inodes
   ULN64              *Inode;                   // Array of Inode LSNs 64bit
   ULONG               Chunks;                  // space chunks     IAG's
   S_SPACE            *Space;                   // space allocation IAG's
} S_IMAP;                                       // end of struct "s_imap"

typedef struct dfsajfs                          // JFS analysis information
{
   ULN64               Sect;                    // Number of sectors
   ULN64               TotalBlocks;             // Total Number of blocks
   ULN64               VolumeBlocks;            // Net Blocks in the volume (IAG)
   ULONG               SectorsPerBlock;         // sectors per block
   S_JFSUP            *sup;                     // ptr to superblock
   ULONG               InoStamp;                // Inode Signature value
   ULONG               InoRecoveryStamp;        // Inode Recovery Signature
   S_IMAP              Agg;                     // Aggregate combined Inode Map
   S_IMAP              Fs1;                     // Fileset-1 combined Inode Map
   DFSJFSINAME        *Ic;                      // Name and Parent inode cache
   DFSJFSBMCACHE       Bm;                      // BitMap file cache
} DFSAJFS;                                      // end of struct "dfsajfs"

extern DFSAJFS        *jfs;

// Initialize JFS filesystem analysis
ULONG dfsJfsInit
(
   char               *fs                       // forced filesystem type
);

// Display code version and alternate driveletter for JFS bootsector
void dfsJfsBootSectorDetails
(
   S_BOOTR            *sd                       // IN    BootSector buffer
);

// Initialize JFS filesystem analysis for Area (FS info in FDISK mode)
ULONG dfsJfsAreaInit
(
   void
);

// Check if boot-area contains a valid JFS superblock, optional label return
// Ignores the actual bootsector, since thet may not be present (Linux JFS)
BOOL dfsBootAreaIsJFS                           // RET   boot area is JFS
(
   BYTE               *brec,                    // IN    bootsector contents
   ULONG               sectors,                 // IN    boot area read (>= 0x40)
   char               *label                    // OUT   buffer for label, or NULL
);

// JFS filesystem, identify specified sector as a valid superblock
BOOL dfsJfsIsSuperBlock                         // RET   sector is a valid sb
(
   BYTE               *sec                      // IN    sector contents
);

// JFS filesystem, display sector constents as a superblock
void dfsJfsDisplaySuperBlock
(
   BYTE               *sec                      // IN    sector contents
);

// JFS filesystem, identify specified sector
ULONG dfsJfsIdent
(
   ULN64               di,                      // IN    LSN for sector
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // OUT   sector type
   void               *data                     // IN    sector contents
);

#endif
