//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// JFS dump & display Analysis functions
//
// Author: J. van Wijk
//
// JvW  19-04-2001 Initial version, derived from HPFS
// JvW  03-04-2017 Updated for 64bit sectornumbers
// JvW  27-06-2017 Updated to recognize Linux formatted filesystems (mkfs.jfs, no bootsector)
// JvW  08-04-2021 Use rc CMD_WARNING on FileSaveAs alloc errors, considered OK
//
// Author: J. van Wijk
//

#include <txlib.h>                              // TX library interface

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfsposix.h>                           // POSIX utility functions
#include <dfsspace.h>                           // DFS file-space interface
#include <dfstable.h>                           // SLT utility functions
#include <dfsdgen.h>                            // Generic dialogs (UUID edit)

#include <dfsjfs.h>                             // JFS disk structure defs
#include <dfsajfs.h>                            // JFS display & analysis
#include <dfsujfs.h>                            // JFS utility functions
#include <dfsljfs.h>                            // JFS SLT functions
#include <dfsos2ea.h>                           // HPFS/JFS/FAT EA handling

// recovery wildcard minimum and maximum timestamps
#define JFS_REC_TSMIN   0x30000000              // somewhere in 1995
#define JFS_REC_TSMAX   0x70000000              // somewhere in 2029

char dfsJfsSectorTypes[] =
{
   ST_JFSUP,                                    // S   JFS  super-block
   ST_JSBAD,                                    // S   JFS  BAD super-block
   ST_JINOD,                                    // S   FS1 Inode (normal file)
   ST_JIDIR,                                    // S   FS1 Inode (directory)
   ST_JIDDI,                                    // S   Inode, deleted directory
   ST_JIDEL,                                    // S   Inode, deleted file
   ST_JINAG,                                    // S   JFS  Aggregate  Inode
   ST_JIBAD,                                    // S   JFS  unused/BAD Inode
   ST_JXTPG,                                    // S   Allocation, Xtree page
   ST_JDTPG,                                    // S   Directory,  Dtree page
   ST_JEASD,                                    //     EA data
   ST_JACLD,                                    //     ACL data
   ST_JIMAG,                                    //  I  aggregate Inode Map
   ST_JIMA2,                                    //  I  2nd aggr. Inode Map
   ST_JI2AG,                                    //  I  JFS  2ndary agg Inode
   ST_JIAGA,                                    //  I  JFS  IAG for aggregate
   ST_JIAGF,                                    //  I  JFS  IAG for fileset-1
   ST_JIMFS,                                    //  I  fileset-1 Inode Map
   ST_JIXFS,                                    //  I  fileset-1 Inode Extent
   ST_JIXAG,                                    //  I  aggregate Inode Extent
   ST_JIXA2,                                    //  I  2nd aggr. Inode Extent
   ST_JBLAM,                                    //  I  Block allocation Map
   ST_JFSCK,                                    //  I  FSCK work-area
   ST_JINLL,                                    //  I  Inline-Log area
   ST_JBOOT,                                    //  I  Micro-FS bootcode
   ST_SLACK,                                    //  I  Last-block slack
   0                                            //     string terminating ZERO
};


static DFSAJFS    jfs_anchor =                  // JFS specific global info
{
   0, 0, 0,                                     // total sectors and blocks
   8,                                           // sectors / block (of 4096)
   NULL,                                        // superblock pointer
   0xbeefface,                                  // Inode Signature (dummy)
   0xdeadbeef,                                  // Inode Recovery Signature (dummy)
   {0,0,0,NULL,0,NULL},                         // Aggregate combined Inode map
   {0,0,0,NULL,0,NULL},                         // Fileset-1 combined Inode map
   NULL,                                        // name and parent Inode cache
};

       DFSAJFS   *jfs = &jfs_anchor;

static char sg_jfsup[SG_JFSUP] = {SV_JFSUP};

static  char       *jfs_txt[] =
{
   "",
   "Active filesystem : JFS, specific commands are:",
   "",
   " \\path-spec       = find and show file/directory relative to root (FINDPATH)",
   " BL               = Translate and display 'this' LSN as a block number",
   " BL block  [cmd]  = Translate specified block-number to LSN, then execute 'cmd'",
   " BSCLEAR          = Reset bad block administration to ZERO bad blocks/sectors",
   " BSEXPORT         = Export JFS bad-block Inode-5c to DFSee (bad) Sectorlist",
   " BSIMPORT         = Import DFSee (bad) Sectorlist into JFS bad-block Inode-5c",
   " CA   [lsn][opt]  = Check Allocation integrity for (current) INODE lsn",
   " DELFIND  [name]  = Find deleted files, with name-info matching (partial) name",
   " DIRTY [d|c|m|#]  = Display/set DIRTY, CLEAN, MOUNT status or numeric value #",
   " DFSJFLDR  [img]  = Create compressed imagefile containing the JFS LDR sectors",
   " FILEFIND [name]  = Find normal files, with name-info matching (partial) name",
   " FINO  [fDyzgui]  = Search all or specified type of Inodes (f=file, D=directory)",
   " FIXBOOT          = Create JFS bootsector from superblock, template and p-table",
   " FIXJFLDR  [img]  = Replace the JFS bootloader on a bootable JFS partition",
   " FRAGFILES        = Search File Inode's for fragmented files  (internal nodes)",
   " IAG  [nr] [opt]  = Show details about Inode Allocation Group data structure",
   " INAG [inode-nr]  = Calculate LSN for aggregate Inode, or translate 'this' LSN",
   " INO              = Translate and display 'this' LSN as a Fileset1 INODE number",
   " INO   inode-nr   = Calculate LSN for a Fileset1 INODE number, default display",
   " IRS       [irs]  = Display/set 'Inode Recovery Signature' after (quick) FORMAT",
   " LABEL   [label]  = Display/edit 11/16-char volume label in bootsector/superblock",
   " SBVER [i|e|l|#]  = Display/set JFS version IBM, ECS, LINUX or numeric value #",
   " SUPER            = Display the filesystem SUPERBLOCK sector",
   " SYNC       1|2   = Sync superblock, copy from 1st or from 2nd instance",
   " UUID [[-u:]uuid] = Display or set a UUID string, dialog if no id specified",
   "",
   NULL
};

// Close JFS filesystem for analysis and free any resources
static ULONG dfsJfsClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
);

// Close JFS filesystem for Area analysis and free any resources
static ULONG dfsJfsAreaClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
);

// Interpret and execute specific JFS command;
static ULONG dfsJfsCommand
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *none,                    // IN    dummy
   void               *data                     // INOUT dummy
);

// JFS filesystem, supply sector-type description string
static ULONG dfsJfsStype
(
   ULN64               di,                      // IN    sector type
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // OUT   type description
   void               *data                     // INOUT dummy
);

// JFS filesystem, display sector-contents based on type
static ULONG dfsJfsDispl
(
   ULN64               psn,                     // IN    base psn for sector
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // IN    sector contents
);

// Display JFS  super-block
static ULONG dfsJfsSuperBlock                   // RET   rc = 0 if type match
(
   BYTE               *sector,                  // IN    Drive specification
   BOOL                navigation               // IN    update nav values
);

// Display value of superblock status value
static void dfsJfsDisplayStatus
(
   char               *lead,                    // IN    leading text
   ULONG               st                       // IN    JFS status value
);


// Display JFS Inode sector contents
static ULONG dfsJfsInodeSec                     // RET   rc = 0 if type match
(
   BYTE               *sector,                  // IN    Sector contents
   ULN64               lsn,                     // IN    sector number
   BYTE                st                       // IN    INODE type
);

// Display JFS directory Dtree page (for the single page, not whole tree)
static ULONG dfsJfsDtreePage                    // RET   rc = 0 if type match
(
   BYTE               *sector                   // IN    data sector(s)
);

// Display JFS allocation Xtree page
static ULONG dfsJfsXtreePage                    // RET   rc = 0 if type match
(
   BYTE               *sector                   // IN    Sector contents
);

// Display JFS Inode Allocation Group structure (IAG) mapping max 4096 Inodes
static ULONG dfsJfsIagPage                      // RET   rc = 0 if type match
(
   BYTE               *sector                   // IN    data sector(s)
);

// Display IAG summary information for the specified Inode map
static ULONG dfsJfsIagSummary
(
   ULONG               verbose,                 // IN    detail level
   char               *lead9,                   // IN    9 char lead text
   S_IMAP             *map                      // IN    Inode map info
);

#define DFS_JDIR_THEADER   0x0001               // add directory header at top
#define DFS_JDIR_BHEADER   0x0002               // optional header at bottom
#define DFS_JDIR_LISTADD   0x0010               // add entries to the Sectorlist
#define DFS_JDIR_BTSPLIT   0x0100               // display in btree-split format
#define DFS_JDIR_USEINODE  0x0200               // use inode for filesize/mode

#define DFS_JDIR_STD  (DFS_JDIR_THEADER | DFS_JDIR_BHEADER | DFS_JDIR_USEINODE)

// Display sequence of JFS Directory entries, optional add to LSN list
static ULONG dfsJfsDirEntries                   // RET   nr of entries listed
(
   S_JDSLOT           *slot,                    // IN    Directory slot array
   BYTE                maxslot,                 // IN    maximum nr of slots
   BYTE               *sTable,                  // IN    sort-table array
   BYTE                sCount,                  // IN    used entries in sTable
   ULONG               dirFlags                 // IN    presentation flags
);

// Make filesystem usage map dump on TxPrint output
static ULONG dfsJfsAllocMap
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *options,                 // IN    display options
   void               *data                     // OUT   dummy
);

// DFS JFS write-file to disk (SaveTo inode to file)
static ULONG dfsJfsFileSaveAs
(
   ULN64               lsn,                     // IN    Inode LSN
   ULN64               d2,                      // IN    dummy
   char               *path,                    // IN    destination path
   void               *opt                      // IN    options (like 8.3)
);


static char     jfsDirHeaderText[] =
 " Nr     Inode-Lsn Creation/LastAccess Attr Modified / Filename             Filesize    Inode  EA's";

static char     jfsDirHeaderLine[] =
 " ===== ========== =================== ==== =================== ==================== ======== =====";


static char     jfsBtsHeaderText[] =
 " Nr    BtrPageLsn  Block-nr  Split-point name (abbreviation)";

static char     jfsBtsHeaderLine[] =
 " ===== ========== ========== ===============================";


/*****************************************************************************/
// Initialize JFS filesystem analysis
/*****************************************************************************/
ULONG dfsJfsInit
(
   char               *fs                       // forced filesystem type
)
{
   ULONG               rc = NO_ERROR;
   TXLN                text;

   ENTER();

   dfsa->FsCommand          = dfsJfsCommand;
   dfsa->FsClose            = dfsJfsClose;
   dfsa->FsIdentifySector   = dfsJfsIdent;
   dfsa->FsShowType         = dfsJfsStype;
   dfsa->FsDisplaySector    = dfsJfsDispl;
   dfsa->FsFileInformation  = dfsJfsFileInfo;
   dfsa->FsGetAllocSpace    = dfsJfsGetAllocSpace;
   dfsa->FsWriteMetaSpace   = NULL;
   dfsa->FsUpdateFileName   = dfsJfsUpdateFileName;
   dfsa->FsSetAltBrecLabel  = dfsJfsSetAltBrecLabel;
   dfsa->FsMakeBrowseList   = dfsJfsMakeBrowseList;
   dfsa->FsFindPath         = dfsJfsFindPath;
   dfsa->FsLsnAllocated     = dfsJfsAllocated;
   dfsa->FsLsnSetAlloc      = dfsJfsSetAlloc;
   dfsa->FsSltBuild         = dfsJfsSltBuild;
   dfsa->FsNpBuild          = dfsJfsNpBuild;
   dfsa->FsDisplayError     = dfsJfsDispError;
   dfsa->FsDisplayLsnInfo   = NULL;
   dfsa->FsDirIterator      = NULL;
   dfsa->FsDirFileSaveAs    = dfsJfsFileSaveAs;
   dfsa->FsTruncateSize     = NULL;
   dfsa->FsAllocDisplay     = dfsJfsAllocMap;
   dfsa->FsCl2Lsn           = dfsJfsCl2Lsn;
   dfsa->FsLsn2Cl           = dfsJfsLsn2Cl;
   dfsa->FsCmdHelp          = jfs_txt;          // FS specific cmdhelp text

   dfsa->FsModeId           = DFS_FS_JFS;

   dfsa->Fsi                = jfs;
   dfsa->FsSectorTypes      = dfsJfsSectorTypes;

   TRACES(("sizeof S_PXD          = % 4u\n", sizeof(S_PXD)));
   TRACES(("sizeof S_DXD          = % 4u\n", sizeof(S_DXD)));
   TRACES(("sizeof JTIME          = % 4u\n", sizeof(JTIME)));
   TRACES(("sizeof S_JFSUP        = % 4u\n", sizeof(S_JFSUP)));
   TRACES(("sizeof S_JINODE       = % 4u\n", sizeof(S_JINODE)));
   TRACES(("sizeof S_JDSLOT       = % 4u\n", sizeof(S_JDSLOT)));
   TRACES(("sizeof S_DTPAGE       = % 4u\n", sizeof(S_DTPAGE)));
   TRACES(("sizeof S_IAG          = % 4u\n", sizeof(S_IAG)));
   TRACES(("sizeof S_BMAP         = % 4u\n", sizeof(S_BMAP)));
   TRACES(("sizeof S_DMAP         = % 4u\n", sizeof(S_DMAP)));
   TRACES(("sizeof DFSJFSINAME    = % 4u\n", sizeof(DFSJFSINAME)));

   dfsa->FsEntry            = LSN_JFSUP;        // entry-point in filesystem

   dfsJfsBootSectorDetails( dfsa->boot);

   jfs->SectorsPerBlock = 8;                    // sane default

   if ((dfsa->boot->Signature == SV_BOOTR) &&   // it is a valid JFS bootsector
       (strncmp( dfsa->boot->os.Type, "JFS", 3) == 0))
   {
      jfs->Sect = max(dfsa->boot->eb.BigSectors, (ULONG) dfsa->boot->eb.Sectors);
   }
   else                                         // empty, GRUB or whatever (Linux)
   {
      jfs->Sect = dfsGetLogicalSize();          // just take the partition size
   }

   if ((jfs->sup = TxAlloc(1, dfsGetSectorSize())) != NULL)
   {
      ULONG            checkSuper2 = 0;         // checksum 2nd superblock
      ULONG            checkClean2 = 0;         // checksum cleaned superblock
      ULN64            TotalSize;
      BYTE             st;                      // sector-type

      if ((rc = dfsRead(LSN_JFSU2, 1,       (BYTE   *) jfs->sup)) == NO_ERROR)
      {
         st = dfsIdentifySector(LSN_JFSU2,  0,  (BYTE   *) jfs->sup);
         if ((st == ST_JFSUP) || (st == ST_JSBAD))
         {
            checkSuper2          = TxHpfsCheckSum((char *) jfs->sup);
            jfs->sup->Status     = 0;
            jfs->sup->LogDevAddr = 0;
            jfs->sup->LogSerial  = 0;
            checkClean2          = TxHpfsCheckSum((char *) jfs->sup);
            if (st == ST_JSBAD)
            {
               TxPrint("BAD signature on  : 2nd superblock at 0x%2.2x\n", LSN_JFSU2);
            }
         }
         else
         {
            TxPrint("No 2nd JFS Super-block found at 0x%2.2x\n", LSN_JFSU2);
         }
      }
      else
      {
         TxPrint("Error reading 2nd JFS Super-block\n");
      }
      if ((rc = dfsRead(LSN_JFSUP, 1,       (BYTE   *) jfs->sup)) == NO_ERROR)
      {
         st = dfsIdentifySector(LSN_JFSUP,  0,  (BYTE   *) jfs->sup);
         if ((st == ST_JFSUP) || (st == ST_JSBAD))
         {
            if (st == ST_JSBAD)
            {
               TxPrint("BAD signature on  : 1st superblock at 0x%2.2x\n", LSN_JFSUP);
            }
            TxPrint( "Superblocks read  : ");
            if (checkSuper2 != 0)
            {
               if (checkSuper2 == TxHpfsCheckSum((char *) jfs->sup))
               {
                  TxPrint( "2, exactly equal, which is unusual\n");
               }
               else                             // retry with cleaned Super
               {
                  ULONG     Status     = jfs->sup->Status;
                  ULONG     LogDevAddr = jfs->sup->LogDevAddr;
                  ULONG     LogSerial  = jfs->sup->LogSerial;

                  jfs->sup->Status     = 0;     // clean variable fields
                  jfs->sup->LogDevAddr = 0;
                  jfs->sup->LogSerial  = 0;

                  if (checkClean2 == TxHpfsCheckSum((char *) jfs->sup))
                  {
                     TxPrint( "2, identical, except Status & LogDev/Serial"
                                                  " (%sOK%s)\n", CBG, CNN);
                  }
                  else
                  {
                     TxPrint( "2, with more differences than expected "
                                          "(%sSUSPICIOUS%s)\n", CBR, CNN);
                  }
                  jfs->sup->Status     = Status;
                  jfs->sup->LogDevAddr = LogDevAddr;
                  jfs->sup->LogSerial  = LogSerial;
               }
            }
            else
            {
               TxPrint( "1, 2nd copy unavailable\n");
            }

            jfs->SectorsPerBlock = jfs->sup->BlockSize / jfs->sup->SectSize;
            jfs->TotalBlocks     = jfs->Sect           / jfs->SectorsPerBlock;
            jfs->VolumeBlocks    = jfs->sup->TotalSec  / jfs->SectorsPerBlock;

            TotalSize  = jfs->Sect - (jfs->sup->InlineLog.len * jfs->SectorsPerBlock)
                                   - (jfs->sup->FsckSpace.len * jfs->SectorsPerBlock);
            TotalSize -= (TotalSize % jfs->SectorsPerBlock);

            if ((dfsa->verbosity > TXAO_QUIET) || (TotalSize != jfs->sup->TotalSec))
            {
               sprintf( text, "  VolBlk : 0x%12.12llX = %llu\n", jfs->VolumeBlocks, jfs->VolumeBlocks);
               dfsSz64( "Superblock #sects : ",  jfs->sup->TotalSec, text);
               if (TotalSize != jfs->sup->TotalSec)
               {
                  dfsSz64( "Calculated #sects : ", TotalSize, " (should be nearly equal to SB!)\n");
               }
            }
            if (jfs->sup->Status == FM_CLEAN)
            {
               dfsa->FsDirtyStatus = DFSTAT_CLEAN; // clean AND unmounted
            }
            else
            {
               dfsa->FsDirtyStatus = DFSTAT_DIRTY;
            }
         }
         else
         {
            TxPrint("No 1st JFS Super-block found at 0x%2.2x\n", LSN_JFSUP);
         }
      }
      else
      {
         TxPrint("Error reading 1st JFS Super-block\n");
      }
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   dfstSetClusterSize( DFSTORE, (USHORT) jfs->SectorsPerBlock); // make sure!

   if (dfsa->verbosity > TXAO_QUIET)
   {
      sprintf( text, "  Blocks : 0x%12.12llX = %llu\n", jfs->TotalBlocks, jfs->TotalBlocks);
      dfsSz64( "Bootsector #sects : ",                  jfs->Sect, text);
      dfsSiz8( "Sectors per block : ",                  jfs->SectorsPerBlock, "\n");
   }
   if (rc == NO_ERROR)
   {
      if ((rc = dfsJfsBitMapInit()) == NO_ERROR)
      {
         dfsSz64( "Bitmap sect range : ", jfs->Bm.LimitLsn, "  ");
         dfsSz64( "Freesp : ",            jfs->Bm.FreeSect, "\n");

         if (TxaOption('a'))                    // show NO allocation by default
         {
            dfsJfsAllocMap( 0, 0, "@", NULL);   // not really needed (no resize)
         }
         if ((rc = dfsRead(LSN_AITAB + JFS_FS1_SUPER_INODE, 1, rbuf)) == NO_ERROR)
         {
            S_JINODE        *inode = (S_JINODE *) rbuf;

            if ((inode->Fileset == 1) && (inode->Self == JFS_FS1_SUPER_INODE))
            {
               jfs->InoStamp = inode->InoStamp;
               jfs->InoRecoveryStamp = jfs->InoStamp; // start at same value
               if (dfsa->verbosity > TXAO_QUIET)
               {
                  dfsJfsTime2str( (JTIME *) &jfs->InoStamp, text);
                  TxPrint( "Filesystem Create : %s\n", text);
               }
               if ((rc = dfsJfsFsInodeAlloc( inode, &jfs->Fs1)) == NO_ERROR)
               {
                  dfsJfsIagSummary( dfsa->verbosity, "Fileset1 ", &jfs->Fs1);
               }
               else
               {
                  TxPrint( "Fileset-1 Inode allocation RC: %u, filesystem seems damaged!\n");
               }
            }
            else
            {
               TxPrint( "Invalid super Inode in Fileset-1, no auto Inode recognition!\n");
            }
         }
         else
         {
            TxPrint("Error reading Fileset-1 Inode, no auto Inode recognition!\n");
         }
         if ((rc = dfsRead(LSN_AITAB + JFS_AGGR_SELF_INODE, 1, rbuf)) == NO_ERROR)
         {
            S_JINODE        *inode = (S_JINODE *) rbuf;

            if ((inode->Fileset == 1) && (inode->Self == JFS_AGGR_SELF_INODE))
            {
               if ((rc = dfsJfsFsInodeAlloc( inode, &jfs->Agg)) == NO_ERROR)
               {
                  dfsJfsIagSummary( (dfsa->verbosity & ~TXAO_VERBOSE), "Aggregate", &jfs->Agg);
               }
               else
               {
                  TxPrint( "Aggregate Inode allocation RC: %u, filesystem seems damaged!\n");
               }
            }
            else
            {
               TxPrint( "Invalid self Inode for Aggregate, filesystem may be damaged!\n");
            }
         }
         else
         {
            TxPrint("Error reading Aggregate self Inode!\n");
         }
      }
   }
   dfsa->FsSltRecSize = jfs->Fs1.Used * 8 + 1000; // average 8 extents per file

   nav.up   = LSN_AITAB + JFS_AGGR_SELF_INODE;  // aggregate self inode
   nav.xtra = LSN_JFSU2;                        // navigate to spare-superblock

   RETURN (rc);                                 // when needed
}                                               // end 'dfsJfsInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display code version and alternate driveletter for JFS bootsector
/*****************************************************************************/
void dfsJfsBootSectorDetails
(
   S_BOOTR            *sd                       // IN    BootSector buffer
)
{
   TxPrint( "Boot code version : ");
   if (sd->Signature != SV_BOOTR)          // it is NOT a valid bootsector
   {
      TxPrint( "%sLinux%s      = Linux JFS without a bootsector, not bootable\n",  CBM, CNN);
   }
   else if (TxMemStr( sd, "GRUB", SECTORSIZE) != NULL)
   {
      TxPrint( "%sLinux-GRUB%s = Linux JFS with a 'GRUB' bootsector, may be bootable\n",  CBM, CNN);
   }
   else if (TxMemStr( sd, "OS2BOOT", SECTORSIZE) != NULL)
   {
      BYTE      *sectordata = (BYTE *) sd;

      TxPrint( "%sArcaOS/eCS%s = Modern JFS, Alternate boot drive-letter = %s%c:%s\n",
                CBM, CNN, CBY, sectordata[ JFSBOOTALTERNATELETTER] + 'C' - 0x80, CNN);
   }
   else if (strncmp( sd->os.Type, "JFS", 3) == 0)
   {
      TxPrint( "%sStandard%s   = IBM classic JFS, usually not bootable\n",  CBM, CNN);
   }
   else
   {
      TxPrint( "%sUnknown%s    = Unknown bootsector code, check superblock for details\n",  CBM, CNN);
   }
}                                               // end 'dfsJfsBootSectorDetails'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Close JFS filesystem for analysis and free any resources
/*****************************************************************************/
static ULONG dfsJfsClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;
   ULONG               i;

   ENTER();

   dfsSlTableReset();                           // stop SLT thread and reset

   rc = dfsJfsBitmapFlush( TRUE);               // flush, and terminate cache

   TxFreeMem( jfs->sup);
   jfs->Sect            = 0;
   jfs->TotalBlocks     = 0;
   jfs->SectorsPerBlock = 8;                    // back to defaults

   if (jfs->Ic)
   {
      for (i = 0; i < jfs->Fs1.Max; i++)
      {
         TxFreeMem( jfs->Ic[i].Name);           // free cached Inode names
      }
      TxFreeMem( jfs->Ic);                      // free name/parent cache
   }

   jfs->Fs1.Chunks = 0;
   TxFreeMem( jfs->Fs1.Space);                  // free allocation space
   TxFreeMem( jfs->Fs1.Inode);                  // free inode table
   jfs->Fs1.Max  = 0;                           // nr of allocated/used

   jfs->Agg.Chunks = 0;
   TxFreeMem( jfs->Agg.Space);                  // free allocation space
   TxFreeMem( jfs->Agg.Inode);                  // free inode table
   jfs->Agg.Max  = 0;                           // nr of allocated/used

   dfsCloseFileSystem();
   RETURN (rc);
}                                               // end 'dfsJfsClose'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Initialize JFS filesystem analysis for Area (FS info in FDISK mode)
/*****************************************************************************/
ULONG dfsJfsAreaInit
(
   void
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   dfsa->FsAreaClose        = dfsJfsAreaClose;
   dfsa->FsAreaLsnAllocated = dfsJfsAllocated;

   dfsa->boot = (S_BOOTR *) brec;
   if ((rc = dfsRead( dfstAreaP2Disk( DFSTORE, LSN_BOOTR), 1, brec)) == NO_ERROR)
   {
      if (dfsa->boot->Signature != SV_BOOTR)    // it is NOT a valid bootsector
      {
         jfs->Sect = dfsGetLogicalSize();       // just take the partition size
      }
      else
      {
         jfs->Sect = max(dfsa->boot->eb.BigSectors, (ULONG) dfsa->boot->eb.Sectors);
      }

      if ((jfs->sup = TxAlloc(1, dfsGetSectorSize())) != NULL)
      {
         if ((rc = dfsRead( dfstAreaP2Disk( DFSTORE, LSN_JFSUP), 1,(BYTE   *) jfs->sup)) == NO_ERROR)
         {
            jfs->SectorsPerBlock = jfs->sup->BlockSize / jfs->sup->SectSize;
            jfs->TotalBlocks     = jfs->Sect           / jfs->SectorsPerBlock;
            jfs->VolumeBlocks    = jfs->sup->TotalSec  / jfs->SectorsPerBlock;

            rc = dfsJfsBitMapInit();
         }
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   RETURN (rc);                                 // when needed
}                                               // end 'dfsJfsAreaInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Close JFS filesystem for Area analysis and free any resources
/*****************************************************************************/
static ULONG dfsJfsAreaClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;
   ULONG               i;

   ENTER();

   dfsa->FsAreaClose        = NULL;
   dfsa->FsAreaLsnAllocated = NULL;

   rc = dfsJfsBitmapFlush( TRUE);               // flush, and terminate cache

   TxFreeMem( jfs->sup);
   jfs->Sect            = 0;
   jfs->TotalBlocks     = 0;
   jfs->SectorsPerBlock = 8;                    // back to defaults

   if (jfs->Ic)
   {
      for (i = 0; i < jfs->Fs1.Max; i++)
      {
         TxFreeMem( jfs->Ic[i].Name);           // free cached Inode names
      }
      TxFreeMem( jfs->Ic);                      // free name/parent cache
   }

   TxFreeMem( jfs->Fs1.Inode);                  // free existing, if any
   jfs->Fs1.Max  = 0;                           // nr of allocated/used

   TxFreeMem( jfs->Agg.Inode);                  // free existing, if any
   jfs->Agg.Max  = 0;                           // nr of allocated/used

   RETURN (rc);
}                                               // end 'dfsJfsAreaClose'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Interpret and execute specific JFS command;
/*****************************************************************************/
static ULONG dfsJfsCommand
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *none,                    // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;
   ULN64               sn = 0;                  // sector number input
   BYTE                st = 0;                  // sector type wanted
   LONG                nr = 0;
   TXLN                dc;                      // DFS command
   int                 cc;                      // command string count
   char               *c0, *c1, *c2;            // parsed command parts
   TXLN                s0;                      // big temporary string space
   TXLN                s1;                      // big temporary string space
   char               *pp;                      // parameter pointer

   ENTER();

   pp = TxaGetArgString( TXA_CUR, 1, 0, TXMAXLN, dc); // dc => cmd from arg 1
   cc = TxaArgCount( );                         // number of parameters
   c0 = TxaArgValue(0);
   c1 = TxaArgValue(1);
   c2 = TxaArgValue(2);

   sn = nav.this;                               // default at current sector

   if ((strcmp(c0, "/?") == 0) ||
       (strcmp(c0,  "?") == 0)  )
   {
      TxShowTxt( jfs_txt);
   }
   else if (strcasecmp(c0, "super"    ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nDisplay the superblock for this JFS filesystem\n");
         TxPrint("\n Usage:  %s  [ 2 ]\n\n"
                   "       2  =  Display the SECONDARY superblock (0x78)\n\n", c0);
      }
      else
      {
         st = ST_JFSUP;
         rc = dfsReadAnDisplay( (cc > 1) ? LSN_JFSU2 : LSN_JFSUP, 0, &st);
      }
   }
   else if ((strncasecmp(c0, "bsc", 3) == 0) || // clear FS badsector admin
            (strcasecmp( c0, "nobads") == 0)  ) // deprecated
   {
      if (TxaOption('?'))
      {
         TxPrint("\nReset bad-block administration in special Inode 0x5c\n");
         TxPrint("\n Usage:  %s\n\n", c0);
      }
      else
      {
         dfsInitList( 0, "-w", "-f -P");        // with optimal list options
         if (DFSTORE_WRITE_ALLOWED)
         {
            rc = dfsJfsImportBadList( dfsa->snlist);
            if (rc == NO_ERROR)
            {
               TxPrint("\nBad-block administration in special Inode 0x5c reset to ZERO,\n"
                       "bad-block list updated, a reboot or 'CHKDSK /f' is needed to update\n"
                       "JFS.IFS and CHKDSK statistics like reserved and free-space.\n");
            }
         }
         else
         {
            rc = DFS_READ_ONLY;
         }
      }
   }
   else if ((strncasecmp(c0, "bse", 3) == 0) || // FS to snlist
            (strcasecmp( c0, "getbs" ) == 0)  ) // deprecated
   {
      if (TxaOption('?'))
      {
         TxPrint("\nExport bad-block list in special Inode 0x5c to DFSee (bad) Sectorlist\n");
         TxPrint("\n Usage:  %s\n\n", c0);
      }
      else
      {
         rc = dfsJfsExportBadList( dfsa->snlist, dfsa->snsize -1);
         if (rc == NO_ERROR)
         {
            TxPrint( "\nSuccessfully exported %u bad sectors to Sectorlist:\n", dfsa->snlist[0]);
            strcpy(  dc, "list -r");            // display in range format
            TxaReParseCommand( dc);
            rc = DFS_PENDING;                   // handle translated command
         }
         else
         {
            TxPrint( "\nExport of bad sectors failed!\n");
         }
      }
   }
   else if ((strncasecmp(c0, "bsi", 3) == 0) || // snlist to FS
            (strcasecmp( c0, "fixbs" ) == 0)  ) // deprecated
   {
      if (TxaOption('?'))
      {
         TxPrint("\nImport DFSee (bad) Sectorlist to bad-block list in special Inode 0x5c\n");
         TxPrint("\n Usage:  %s\n\n", c0);
      }
      else
      {
         if (dfsa->snlist[0] != 0)              // bad sectors defined ?
         {
            if (DFSTORE_WRITE_ALLOWED)
            {
               rc = dfsJfsImportBadList( dfsa->snlist);
               if (rc == NO_ERROR)
               {
                  TxPrint("\nJFS bad-block list updated, a reboot or 'CHKDSK /f' is needed to\n"
                          "update JFS.IFS and CHKDSK statistics like reserved and free-space.\n\n"
                          "Bad blocks Inode contents now:\n");

                  strcpy(  dc, "0x5c      ;Display bad-blocks Inode");
                  TxaReParseCommand( dc);
                  rc = DFS_PENDING;     // handle translated command
               }
               else
               {
                  TxPrint( "\nImport of bad sectors failed!\n");
               }
            }
            else
            {
               rc = DFS_READ_ONLY;
            }
         }
         else
         {
            TxPrint("DFSee (bad) Sectorlist is empty, use 'scan', 'listset' "
                    "or 'import' command and try again\n");
         }
      }
   }
   else if (strcasecmp(c0, "ca"   ) == 0)          // check allocation
   {
      S_JINODE        *ino;
      ULN64            size = 0;
      ULN64            bads = 0;

      sn = dfsGetSymbolicSN( c1, nav.this);     // default current LSN
      if (cc > 2)                               // no options
      {                                         // specified
         strcpy(s1, c2);
      }
      else
      {
         strcpy(s1, "v");                       // default verbose

      }
      if ((rc = dfsJfsReadChkInode( sn, &st, &ino)) == NO_ERROR)
      {
         rc = dfsJfsInodeCheckAlloc( ino, st, s1, &size, &bads, NULL);
         if ((rc == NO_ERROR) || (rc == DFS_BAD_STRUCTURE))
         {
            dfsSz64("Total filesectors : ", size, "\n");
            dfsSz64("Allocation errors : ", bads,   "\n");
         }
         TxFreeMem( ino);
      }
   }
   else if ((strcasecmp(c0, "cl"       ) == 0) ||
            (strcasecmp(c0, "bl"       ) == 0)  )
   {
      if (cc > 1)
      {
         sn = dfsGetSymbolicSN( c1, 0);
         if (sn <= jfs->TotalBlocks)
         {
            if (cc > 2)
            {
               sprintf( dc, "%s ", c2);
            }
            else
            {
               strcpy(  dc, "");
            }
            sprintf( s1, "0x%llX", dfsJfsBlock2Lsn(sn));
            strcat(  dc, s1);
            TxaReParseCommand( dc);
            rc = DFS_PENDING;                   // handle translated command
         }
         else
         {
            dfsX10( "Invalid block   nr: ", sn, CBR, "\n");
         }
      }
      else
      {
         dfsX10("Current block   nr: ", dfsJfsLsn2Block(nav.this), CBM, "\n");
      }
   }
   else if (strcasecmp(c0, "dirty"    ) == 0)
   {
      if (cc > 1)
      {
         ULONG         sb = (ULONG) atol( c2);

         if (toupper( c1[0]) == 'Q')            // query only
         {
            rc = jfs->sup->Status;              // set RC value to status
            dfsa->explain = FALSE;              // don't give incorrect explanation
         }
         else                                   // set new status value
         {
            switch (c1[0])
            {
               case 'd': case 'D': jfs->sup->Status = FM_DIRTY;          break;
               case 'm': case 'M': jfs->sup->Status = FM_MOUNT;          break;
               case 'c': case 'C': jfs->sup->Status = FM_CLEAN;          break;
               default:            jfs->sup->Status = (ULONG) atol( c1); break;
            }
            if (sb == JFSB_2ND)                 // second only, display it!
            {
               dfsJfsDisplayStatus( "\nSB2 status set to : ", jfs->sup->Status);
            }
            if (jfs->sup->Status == FM_CLEAN)   // sync global dirty flag!
            {
               dfsa->FsDirtyStatus = DFSTAT_CLEAN; // clean AND unmounted
            }
            else
            {
               dfsa->FsDirtyStatus = DFSTAT_DIRTY;
            }
            rc = dfsJfsWriteSuperBlocks((cc > 2 ) ? sb : JFSB_BOTH);
         }
      }
      else
      {
         TxPrint("\nUsage: %s  [query | clean | mounted | dirty | numeric-value]  [1|2]\n", c0);
         TxPrint("\n  query      = Query, set RC to FS status value, as specified below\n");
         TxPrint("\n  clean      = FS unmounted and clean   (NOT 'dirty')");
         TxPrint("\n  mounted    = FS mounted cleanly   (OK, but 'dirty')");
         TxPrint("\n  dirty      = FS needs CHKDSK (INCONSISTENT 'dirty')\n");
         TxPrint("\n  numeric: 0 = FS unmounted and clean   (NOT 'dirty')");
         TxPrint("\n           1 = FS mounted cleanly   (OK, but 'dirty')");
         TxPrint("\n           2 = FS needs CHKDSK (INCONSISTENT 'dirty')");
         TxPrint("\n           4 = recovery failed (logredo())");
         TxPrint("\n           8 = extendfs() is in progress\n");
         TxPrint("\n           4 & 8 can be combined with others\n");
         TxPrint("\n  [1|2] : Update 1st or 2nd superblock only (default is update both)\n");
      }
      dfsJfsDisplayStatus( "\nJFS SB1 status is : ", jfs->sup->Status);
   }
   else if (strcasecmp(c0, "label"    ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nSet volume label in superblock (16-character, Linux FS-name)\n");
         TxPrint("\nUsage: %s -l  [label]\n\n"
                 "   label  : new volume label, maximum 16 characters\n"
                 "   -l     = request to set the Linux-style label (FS-Name)\n"
                 "   -!-    = do not prompt for new value, just set it\n\n"
                 "OR, set the classic PC-style label, like:\n\n", c0);
          rc = DFS_PENDING;                     // add generic label help/usage
      }
      else
      {
         if (TxaOption( 'l'))
         {
            if (cc > 1)
            {
               strcpy( s1, c1);                 // use specified parameter value
            }
            else
            {
               TxCopy( s1, jfs->sup->linux.lxVolName, JFS_LEN_LBL + 1);
            }
            s1[ JFS_LEN_LBL] = 0;               // truncate to allowed length

            if (dfsa->batch)                    // s1 has new label value
            {
               memcpy( jfs->sup->linux.lxVolName, s1, JFS_LEN_LBL);
               rc = dfsJfsWriteSuperBlocks( TxaOptNum('s', NULL, JFSB_BOTH));
            }
            else
            {
               #if defined (USEWINDOWING)       // Allow interactive update
               if (!TxaOptUnSet('!') && txwIsWindow( TXHWND_DESKTOP))
               {
                  TXLN      prompt;

                  sprintf( prompt, "%s\n\nVolume label in superblock: '%s'\n\n"
                                   "Specify new %d-character volume label\n",
                                    dfstStoreDesc1( DFSTORE) + 10, s1, JFS_LEN_LBL);

                  if (txwPromptBox( TXHWND_DESKTOP, TXHWND_DESKTOP, NULL,
                         prompt, " Set volume label in superblock ",
                         5144, TXPB_MOVEABLE | TXPB_HCENTER | TXPB_VCENTER,
                         JFS_LEN_LBL, s1) != TXDID_CANCEL)
                  {
                     TxStrip( s1, s1, ' ', ' '); // strip leading/trailing spaces
                     memcpy( jfs->sup->linux.lxVolName, s1, JFS_LEN_LBL);
                     rc = dfsJfsWriteSuperBlocks( TxaOptNum('s', NULL, JFSB_BOTH));
                  }
               }
               else
               #endif
               {
                  if (TxConfirm( 5144, "Write Volume LABEL '%s' to superblock ? [Y/N] : ", s1))
                  {
                     memcpy( jfs->sup->linux.lxVolName, s1, JFS_LEN_LBL);
                     rc = dfsJfsWriteSuperBlocks( TxaOptNum('s', NULL, JFSB_BOTH));
                  }
               }
            }
         }
         else
         {
            rc = DFS_PENDING;
         }
      }
   }
   else if (strcasecmp(c0, "uuid"    ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nUsage: %s  [UUID] | [-u:UUID]  [-s:n]\n"
                 "\n      UUID : UUID string value, 5 groups separated by hyphens, no brackets"
                 "\n   -u:UUID : 36 character UUID value: 12345678-abcd-0123-4567-0123456789ab\n",
                 "\n   -s:n    : Write back to superblock n, where n = 1 or 2, default is BOTH\n", c0);
      }
      else
      {
         //- to be refined, UUID value only present on JFS2 (newer Linux)
         //- and there is a second UUID field for the log
         //- Perhaps need to be able to show/edit both

         if (TxaOptMutEx((cc > 1), "u", "if a UUID parameter is specified!", NULL))
         {
            if (cc > 1)
            {
               strcpy( s1, c1);                 // use specified parameter value
            }
            else
            {
               strcpy( s1, TxaOptStr( 'u', NULL, ""));
            }
            sprintf( s0, "{%s}", s1);           // Make fully-formed UUID string

            if (((strlen( s1) == 0) && (!dfsa->batch)) || dfsUidStringIsValid( s0))
            {
               if (dfsa->batch)                 // s0 has valid new UUID
               {
                  dfsUidStringToBinary( s0, jfs->sup->linux.lxVolUuid, FALSE);
                  rc = dfsJfsWriteSuperBlocks( TxaOptNum('s', NULL, JFSB_BOTH));
               }
               else
               {
                  #if defined (USEWINDOWING)    // Allow interactive update
                  if (txwIsWindow( TXHWND_DESKTOP))
                  {
                     if (dfsGuidUuidEditor( "Update UUID value in EXT Superblock",
                                            "Write Back", SINF->partid,
                                             5143, s0, FALSE, jfs->sup->linux.lxVolUuid) == NO_ERROR)
                     {
                        rc = dfsJfsWriteSuperBlocks( TxaOptNum('s', NULL, JFSB_BOTH));
                     }
                  }
                  else
                  #endif
                  {
                     if (TxConfirm( 5143, "Write UUID %s on PID %hu ? [Y/N] : ", s0, SINF->partid))
                     {
                        dfsUidStringToBinary( s0, jfs->sup->linux.lxVolUuid, FALSE);
                        rc = dfsJfsWriteSuperBlocks( TxaOptNum('s', NULL, JFSB_BOTH));
                     }
                  }
               }
            }
            else
            {
               TxPrint( "\nInvalid UUID value: '%s'\n", s1);
            }
         }
      }
      TxPrint( "\nVolume UUID now   : %s\n", dfsFsUuidValueString( jfs->sup->linux.lxVolUuid));
   }
   else if (strcasecmp(c0, "sbver"    ) == 0)
   {
      if (cc > 1)
      {
         if (isdigit( c1[0]))          // symbolic
         {
            jfs->sup->Version = (ULONG) atol( c1);
         }
         else
         {
            switch (c1[0])
            {
               case 'l': case 'L': jfs->sup->Version = 2; break; // Linux
               default:            jfs->sup->Version = 1; break; // Ibm/ArcaOS/Ecs
            }
         }
         rc = dfsJfsWriteSuperBlocks( JFSB_BOTH);
      }
      else
      {
         TxPrint("\nUsage: %s  IBM | ArcaOS | eCS | Linux | #num\n", c0);
         TxPrint("\n    IBM = Classic IBM/ArcaOS/eCS JFS (1)");
         TxPrint("\n ArcaOS = Classic IBM/ArcaOS/eCS JFS (1)");
         TxPrint("\n    eCS = Classic IBM/ArcaOS/eCS JFS (1)");
         TxPrint("\n  Linux = Open Source Linux          (2)");
         TxPrint("\n   #num = Explicit numeric version\n\n");
      }
      rc = dfsReadAnDisplay( LSN_JFSUP, 0, &st);
   }
   else if (strcasecmp(c0, "sbsig"    ) == 0)
   {
      if (cc > 1)
      {
         jfs->sup->Signature[0] = 'J';
         jfs->sup->Signature[1] = 'F';
         jfs->sup->Signature[2] = 'S';
         jfs->sup->SignatNum    = (isdigit( c1[0])) ? c1[0] : '1';
         rc = dfsJfsWriteSuperBlocks( JFSB_BOTH);
      }
      else
      {
         TxPrint("\nUsage: %s  #num | fix\n", c0);
         TxPrint("\n   #num = Explicit numeric signature digit");
         TxPrint("\n    fix = any non-digit sets it to the default, '1'\n\n");
      }
      rc = dfsReadAnDisplay( LSN_JFSUP, 0, &st);
   }
   else if (strcasecmp(c0, "fixboot"  ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nFix the bootsector for an OS/2, eCS or ArcaOS JFS partition (not Linux)\n");
         TxPrint("\n Usage:  %s  [-s] [drive | -]\n\n"
                   " -s        = Use older STANDARD (IBM classic) bootsector contents\n", c0);
         TxPrint("   drive | - = Optional boot driveletter, - resets to 'data'\n"
                 "               Determined automatically when not specified.\n");
      }
      else
      {
         if ((rc = dfsJfsMkBootRec( jfs->sup, TxaOption('s'), (cc > 1) ? c1 : SINF->drive)) == NO_ERROR)
         {
            if (SINF->partid != 0)              // was it a partition ?
            {
               TxPrint( "Resulting partition after fixing the bootsector:\n\n");
               sprintf( dc, "part -r -q %hu#d", SINF->partid);
               rc = dfsMultiCommand( dc, 0, FALSE, FALSE, TRUE);
            }
         }
      }
   }
   else if (strcasecmp(c0, "fixjfldr"  ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nReplace the JFS micro filesystem for a JFS partition\n");
         TxPrint("\n Usage:  %s   [image]\n"
                 "\n  image] = Name of imagefile with JFS LDR code (%u sectors)"
                 "\n           with a default of '%s'\n",
                               c0, JFLDR_SIZE, JFLDR_IMGNAME);
      }
      else
      {
         strcpy(  s0, (cc > 1) ? c1 : JFLDR_IMGNAME);
         sprintf( s1, "restore \"%s\" 0x0%llx %u", s0, JFLDR_LSN, JFLDR_SIZE);
         rc = dfsMultiCommand( s1, 0, TRUE, FALSE, TRUE);
      }
   }
   else if (strcasecmp(c0, "dfsjfldr"  ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nCreate compressed imagefile containing the JFS micro filesystem\n");
         TxPrint("\n Usage:  %s   [image]\n"
                 "\n  image  = Name of imagefile to be created (%u sectors) without"
                 "\n           a file-extension, and a default name of '%s'\n",
                               c0, JFLDR_SIZE, JFLDR_IMGNAME);
      }
      else
      {
         strcpy(  s0, (cc > 1) ? c1 : JFLDR_IMGNAME);
         sprintf( s1, "image %s 0x0%llx %u  -z", s0, JFLDR_LSN, JFLDR_SIZE);
         rc = dfsMultiCommand( s1, 0, TRUE, FALSE, TRUE);
      }
   }
   else if (strncasecmp(c0, "fragfil", 7 ) == 0)     // Inodes with internal nodes
   {
      if (TxaOption('?'))
      {
         TxPrint("\nFind File Inode's with INTERNAL tree nodes (fragmented file)\n");
         TxPrint("\n Usage:  %s\n\n", c0);
      }
      else
      {
         TxPrint( "JFS specific find : Fragmented files (internal tree nodes)\n");
         sprintf( dc, "find -o:$%%*@240 -t:'f' -h:'85' -f:0");
         TxaReParseCommand( dc);
         rc = DFS_PENDING;                      // handle translated command
      }
   }
   else if (strncasecmp(c0, "fino", 4 ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nFind all or specified Inode types\n");
         TxPrint("\n Usage:  %s   [fDyzgi]\n\n", c0);
         TxPrint("   f     = File Inodes, all regular files\n"
                 "   D     = Directory Inodes\n\n"
                 "   y     = Deleted directory Inodes'\n\n"
                 "   z     = Deleted file Inodes\n"
                 "   g     = Aggregate Inodes (system/meta files)\n"
                 "   i     = Invalid Inodes (ZERO timestamps)\n"
                 "           Default is search file/directory inodes\n\n");
         TxPrint("   -v    = Verbose search, list files while found (SLOW!)\n\n");
      }
      else
      {
         TxPrint( "JFS specific find : Inode sectors\n");
         sprintf( dc, "find -t:%s -o:*%%!%s -f:0",
                  (cc > 1) ? c1 : "fD",
                  (TxaOption('v')) ? "" : "Q"); // verbose or quiet, FAST
         TxaReParseCommand( dc);
         rc = DFS_PENDING;                      // handle translated command
      }
   }
   else if ((strcasecmp(c0, "delfind"  ) == 0) ||
            (strcasecmp(c0, "filefind" ) == 0)  )
   {

      if (TxaOption('?'))
      {
         TxPrint( "\nFind (deleted) files by direct searching the file-inodes\n");
         TxPrint( "\n Usage: %s  [name]  [-a] [-D | -f] [-c] [-d-] [-f:first] [-v]\n\n", c0);
         TxPrint( "   name      = partial or whole filename wanted (NOT a wildcard!)\n"
                  "               matches on filename only, not the path component!\n"
                  "   -a        = search ANY files, deleted and normal  (with -d- only)\n"
                  "   -D        = search  directories  only, not files+directories\n"
                  "   -f        = search regular files only, not files+directories\n"
                  "   -c        = start from current sector, not start of volume\n"
                  "   -d-       = search outside known INODE areas too   (SLOW!)\n"
                  "   -F:first  = start at given Inode nr, default = 4   (unless -d-)"
                  "   -v        = verbose search, list files while found (SLOW!)\n");
      }
      else                                      // search Inodes
      {
         TXTS          findtype;
         BOOL          deleted  = (strcasecmp(c0, "delfind"  ) == 0);

         if      (TxaOption('D'))                    // Directories only
         {
            strcpy( findtype, TxaOption('a') ? "Dy" : (deleted)  ? "y" :  "D");
         }
         else if (TxaOption('f'))                    // Directories only
         {
            strcpy( findtype, TxaOption('a') ? "fz" : (deleted)  ? "z" :  "f");
         }
         else                                   // Files + Directories
         {
            strcpy( findtype, TxaOption('a') ? "fzDy" : (deleted)  ? "zy" :  "fD");
         }

         //- note: deleted inodes can be in allocated OR freespace!
         if ((!deleted) && (dfsSlTableStatus( NULL) != SLT_READY))
         {
            //- TxPrint( "\nBuilding SLT for filename lookup, please wait ..."
            //-         "   (or abort that using <Esc>)\n");
            //- dfsMultiCommand( "slt 0 1 -v- -d-", 0, TRUE, FALSE, FALSE);
            dfsSltPrepareNameLookup();
            TxCancelAbort();                    // might have aborted
         }
         if (TxaOptUnSet('d'))                  // use brute force (slow)
         {
            sprintf( dfsa->brdescript, "JFS Slow %s %s", c0, pp);
            sprintf( s1, "find -t:%s -o:*$%%!%s%c ", findtype,
                    (TxaOption('v')) ? "" : "Q", // verbose or quiet, FAST
                    (deleted)  ? ' ' : ']');    // search freespace/allocated
            if (cc > 1)                         // ascii wildcard search-string
            {
               TxStrip( s0, pp, '*', '*');      // remove leading/trailing *
               strcat(  s1, s0);                // add search string-value
            }
            strcpy( dc, s1);                    // copy complete command
            TxPrint("Find %s files: JFS specific, using: %s%s%s\n",
                     TxaOption('a') ? "any/all files" :
                          (deleted) ? "deleted files" :
                                      "normal files ", CBC, dc, CNN);
            if (!TxaOption('c'))
            {
               nav.this = dfsJfsInode2Lsn( JFS_FIRSTINODE, &jfs->Fs1);
            }
            TxaReParseCommand( dc);
            rc = DFS_PENDING;                   // handle translated command
         }
         else                                   // search INODE area only (QFI)
         {
            sprintf( dfsa->brdescript, "JFS Quick %s %s", c0, c1);
            sn = TxaOptNum('F', "",  JFS_FIRSTINODE); // First Inode nr todo
            TxPrint( "\nQuick find '%s' Inodes matching %s%s%s\n",
                 findtype, (cc > 1) ? CBG : "", (cc > 1) ? c1 : "any name", CNN);
            rc = dfsJfsQuickFindInodes( TxaOption('v'), sn, findtype, c1);
            if ((rc == NO_ERROR) && (dfsa->snlist[0] != 0))
            {
               TxPrint( "\nYou can use the 'list -f' or 'browse' command (key <F9>),\n"
                        "to display/browse the found files, and copy/recover them.\n");
            }
         }
      }
   }
   else if ((strcasecmp(c0, "inag"     ) == 0) ||  // calculate LSN for AGG INODE
            (strcasecmp(c0, "ino"      ) == 0)  )  // calculate LSN for FS1 INODE
   {
      S_IMAP *imap = (strcasecmp( c0, "ino")) ? &jfs->Agg : &jfs->Fs1;

      if (cc > 1)
      {
         nr = dfsGetSymbolicSN( c1, 2);         // default INODE 2 (root)
         sn = dfsJfsInode2Lsn( nr, imap);
         if      (sn == L64_NULL)
         {
            dfsX10(  "\nInvalid INODE  nr : ", nr, CBR, "\n");
            dfsDec8( "Max  INODE number : ", imap->Max -1, "\n");
         }
         else if (sn == 0)
         {
            dfsX10(  "\nINODE extent for  : ", nr, CBM, " is not allocated yet (not in use)\n");
         }
         else
         {
            sprintf( dc, "0x0%llX", sn);
            rc = dfsMultiCommand( dc, 0, TRUE, FALSE, TRUE);
         }
      }
      else
      {
         if ((nr = dfsJfsLsn2Inode( nav.this, imap)) != L64_NULL)
         {
            dfsX10("Current  INODE  nr: ", nr, CBC, "\n");
         }
         else
         {
            dfsX10("Current sector Lsn: ", nav.this, CBM, " is not an allocated Inode\n");
         }
      }
   }
   else if (strcasecmp(c0, "iag"     ) == 0)       // show IAG details
   {
      if (TxaOption('?'))
      {
         TxPrint("\nShow details about Inode allocation groups (IAG)\n");
         TxPrint("\n Usage:  %s  [iagnr | *]\n", c0);
         TxPrint("\n   iagnr = Number of IAG to display, 0 .. n, or '*' for ALL"
                 "\n           When not specified, an IAG summary is displayed\n"
                 "\n   -a    : Show aggregate IAG (system), not Fileset-1 (user)"
                 "\n   -v    : Verbose display of IAG control-page summary data\n\n");
      }
      else if (cc == 1)
      {
         TxPrint( "\n");
         rc = dfsJfsIagSummary( TXAO_VERBOSE, (TxaOption('a')) ? "Aggregate" : "Fileset1 ",
                                              (TxaOption('a')) ? &jfs->Agg  : &jfs->Fs1);
      }
      else                                      // display specific
      {
         LONG          first = dfsGetMcsNumber( c1, 0);
         LONG          last  = first;
         LONG          iag;

         nr = TxaOption('a') ? jfs->Agg.Chunks : jfs->Fs1.Chunks;

         if (c1[0] == '*')                      // list all of them
         {
            first = 0;
            last  = nr -1;
         }
         for (iag = first; iag <= last; iag++)
         {
            TxPrint( "\n");
            if ((sn = dfsSspaceRsn2Lsn( nr,
                     (TxaOption('a')) ? jfs->Agg.Space  : jfs->Fs1.Space,
                     ((ULN64) iag * JFS_IAGSEC + JFS_IAGCPG))) < LSN_SPARSE)
            {
               st =  (TxaOption('a')) ? ST_JIAGA : ST_JIAGF; // force 'IAG' display
               rc = dfsReadAnDisplay( sn, 0, &st);
            }
            else
            {
               TxPrint( "Invalid IAG number, allowed range is 0 .. %u\n", nr -1);
            }
         }
      }
   }
   else if (strcasecmp(c0, "irs"     ) == 0)    // Inode Recovery stamp display/set
   {
      if (TxaOption('?'))
      {
         TxPrint("\nShow and optional SET the 'Inode Recovery Signature'\n");
         TxPrint("\n Usage:  %s  [irs] [-!]\n", c0);
         TxPrint("\n   irs   = Inode recovery signature as an 8-digit hexadecimal value."
                 "\n           or 0 as a 'wildcard' that will match any possible Inode."
                 "\n   -!    : Force prompt dialog to specify the signature value.\n"
                 "\n   Signature value is first 4 bytes of (lost) Inodes to be recovered."
                 "\n   Example HEX-edit '6b 5b d6 4e' in Inode is irs value: 0x4ed65b6b\n"
                 "\n   The value assigned to newly created Inodes is the FORMAT\n"
                 "\n   timestamp so it changes after each (quick) FORMAT done!\n\n");
      }
      else
      {
         if (cc > 1)
         {                                      // specified
            jfs->InoRecoveryStamp = dfsGetMcsNumber( c1, jfs->InoStamp); // default is InoStamp
            if (jfs->InoRecoveryStamp == jfs->InoStamp)
            {
               strcpy( dfsa->snifsmcmd, "");    // reset, so reset list CMD too
            }
         }
         #if defined (USEWINDOWING)          // Allow interactive update
         if (TxaOptSet('!') || TxaOptSet('P'))
         {
            TXLN      prompt;
            TXTT      value;

            sprintf( prompt, "Inode Recovery Signature now: %8.8x\n\n"
                             "Specify max 8-digit hexadecimal Signature value, "
                             "or keep 0 as a wildcard to find ANY Inode\n",
                              jfs->InoRecoveryStamp);
            strcpy( value,  "0");

            if (txwPromptBox( TXHWND_DESKTOP, TXHWND_DESKTOP, NULL,
                   prompt, " Set Inode-Recovery-Signature ",
                   5161, TXPB_MOVEABLE | TXPB_HCENTER | TXPB_VCENTER,
                   11, value) != TXDID_CANCEL)
            {
               sscanf( value, "%x", &jfs->InoRecoveryStamp); // get specified hex value
            }
         }
         #endif
         if (jfs->InoRecoveryStamp == 0)
         {
            strcpy( s0, "0 (WildCard, ANY Inode)");
         }
         else
         {
            sprintf( s0, "0x%8.8x", jfs->InoRecoveryStamp);
         }
         TxPrint("\nInode RecoverySig : %s, current InoStamp: 0x%8.8x\n\n", s0, jfs->InoStamp);

         if (jfs->InoRecoveryStamp != jfs->InoStamp)
         {
            sprintf( dfsa->snifsmcmd, "irs 0x%x",jfs->InoRecoveryStamp);
         }
      }

   }
   else if (strcasecmp(c0, "sync"    ) == 0)
   {
      if (cc > 1)                               // mandatory parameter
      {
         if (c1[0] == '2')                      // sync 2nd to 1st
         {
            if ((rc = dfsRead(LSN_JFSU2, 1, (BYTE   *) jfs->sup)) == NO_ERROR)
            {
               if (dfsIdentifySector(LSN_JFSU2, 0, (BYTE   *) jfs->sup) != ST_JFSUP)
               {
                  if ((dfsa->batch) || (TxConfirm( 5031,
                      "No valid JFS superblock found at "
                      "0x%2.2x. Sync anyway ? [Y/N] :\n", LSN_JFSU2)))
                  {
                     rc = NO_ERROR;
                  }
                  else
                  {
                     rc = DFS_NO_CHANGE;
                  }
               }
            }
            else
            {
               TxPrint("Error reading 2nd JFS Super-block\n");
            }
         }
         if (rc == NO_ERROR)
         {
            rc = dfsJfsWriteSuperBlocks( JFSB_BOTH);
         }
      }
      else
      {
         TxPrint("\nUsage: %s   1 | 2\n", c0);
         TxPrint("\n   1 = sync 1st copy at 0x40 to 2nd at 0x78\n");
         TxPrint(  "   2 = sync 2nd copy at 0x78 to 1st at 0x40\n");
      }
   }
   else if (strcasecmp(c0, "write"    ) == 0)
   {
      sn = dfsGetSymbolicSN( c1, nav.this);     // default current LSN
      if (sn == LSN_JFSUP)
      {
         TxPrint("\nInfo: Update to 1st superblock will be "
                 "automatically synchronised to 2nd!\n");
         memcpy(jfs->sup, rbuf, dfsGetSectorSize());
         rc = dfsJfsWriteSuperBlocks( JFSB_BOTH);
      }
      else
      {
         if (sn == LSN_JFSU2)
         {
            TxPrint("\nWarning: Updated 2nd superblock is NOT "
                    "automatically synchronised to 1st!\n");
         }
         rc = DFS_PENDING;                      // handle actual write
      }
   }
   else
   {
      rc = DFS_CMD_UNKNOWN;                     // cmd not recognized
   }
   RETURN (rc);
}                                               // end 'dfsJfsCommand'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Check if boot-area contains a valid JFS superblock, optional label return
// Ignores the actual bootsector, since thet may not be present (Linux JFS)
/*****************************************************************************/
BOOL dfsBootAreaIsJFS                           // RET   boot area is JFS
(
   BYTE               *brec,                    // IN    bootsector contents
   ULONG               sectors,                 // IN    boot area read (>= 0x40)
   char               *label                    // OUT   buffer for label, or NULL
)
{
   BOOL                rc = FALSE;

   ENTER();

   if (sectors >= LSN_JFSUP)                    // superblock should be in there
   {
      BYTE            *jfsSuperBlock;           // possible superblock location (JFS)

      jfsSuperBlock   = brec + (LSN_JFSUP * dfsGetSectorSize());
      if (dfsJfsIsSuperBlock( jfsSuperBlock))
      {
         rc = TRUE;
      }
      if (label)
      {
         TxCopy( label, ((S_JFSUP *) jfsSuperBlock)->FsVolume, BT_LABL_L);
      }
   }
   BRETURN (rc);
}                                               // end 'dfsBootAreaIsJFS'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// JFS filesystem, identify specified sector as a valid superblock
/*****************************************************************************/
BOOL dfsJfsIsSuperBlock                         // RET   sector is a valid sb
(
   BYTE               *sec                      // IN    sector contents
)
{
   return (!memcmp(((S_JFSUP *)sec)->Signature, sg_jfsup, SG_JFSUP));
}                                               // end 'dfsJfsIsSuperBlock'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// JFS filesystem, display sector constents as a superblock
/*****************************************************************************/
void dfsJfsDisplaySuperBlock
(
   BYTE               *sec                      // IN    sector contents
)
{
   dfsJfsSuperBlock( sec, FALSE);
}                                               // end 'dfsJfsDisplaySuperBlock'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// JFS filesystem, identify specified sector
/*****************************************************************************/
ULONG dfsJfsIdent
(
   ULN64               lsn,                     // IN    LSN for sector
   ULN64               d2,                      // IN    dummy
   char               *st,                      // OUT   sector type
   void               *sec                      // IN    sector contents
)
{
   ULONG               dr = NO_ERROR;
   BYTE                rc = ST_UDATA;
   ULN64               bl = dfsJfsLsn2Block( lsn);
   S_JFSUP            *sd = (S_JFSUP  *) sec;   // as Superblock
   S_JINODE           *in = (S_JINODE *) sec;   // as Inode
   S_XTHEAD           *xt = (S_XTHEAD *) sec;   // as Xtree page header
   S_DTHEAD           *dt = (S_DTHEAD *) sec;   // as Dtree page header
   ULONG               ps = xt->self.len * jfs->SectorsPerBlock; // sectors in page
   ULN64               sn;

   ENTER();

   if      (strncasecmp(sd->Signature,sg_jfsup,SG_JFSUP) == 0) rc=ST_JFSUP;
   if      (in->InoStamp == jfs->InoStamp)                     rc=ST_JINOD; // current format

   //- Should allow more than 32-bit for 64bit blocknrs, allow 40 non-zero bits now (1 tera blocks)
   if      ((xt->next <= 0xffffffffffULL) && (xt->prev <= 0xffffffffffULL) && (bl == dfsJfsPxdValue( xt->self)) &&
           ((ps == 1) || (ps == 2) || (ps == 4) || (ps == 8))) rc=ST_JXTPG;

   if (rc == ST_UDATA)
   {
      if (jfs->InoRecoveryStamp == 0)           // wildcard, between 1996 and 2021
      {
         if  ((in->InoStamp      > JFS_REC_TSMIN) && (in->InoStamp      < JFS_REC_TSMAX) &&
              (in->LastAccess.hi > JFS_REC_TSMIN) && (in->LastAccess.hi < JFS_REC_TSMAX) &&
              (in->LastStatus.hi > JFS_REC_TSMIN) && (in->LastStatus.hi < JFS_REC_TSMAX) &&
              (in->LastModify.hi > JFS_REC_TSMIN) && (in->LastModify.hi < JFS_REC_TSMAX) &&
              (in->Creation.hi   > JFS_REC_TSMIN) && (in->Creation.hi   < JFS_REC_TSMAX) )
         {
            rc=ST_JINOD;                        // possible INODE, do sanity check later
         }
      }
      else
      {
         if (in->InoStamp == jfs->InoRecoveryStamp)
         {
            rc=ST_JINOD;                        // older   format
         }
      }
   }

   switch (rc)
   {
      case ST_UDATA:
         if      ((sn = dfsSspaceLsn2Rsn( jfs->Fs1.Chunks, jfs->Fs1.Space, lsn)) != L64_NULL)
         {
            rc = (sn && ((sn % JFS_IAGSEC) == 0)) ? ST_JIAGF : ST_JIMFS;
         }
         else if ((sn = dfsSspaceLsn2Rsn( jfs->Agg.Chunks, jfs->Agg.Space, lsn)) != L64_NULL)
         {
            rc = (sn && ((sn % JFS_IAGSEC) == 0)) ? ST_JIAGA : ST_JIMAG;
         }
         else
         {
            dr = DFS_PENDING;
         }
         break;

      case ST_JXTPG:                            // valid tree page sofar
         if      ((dt->maxslot == (JFS_DB_SLOTS * ps)) &&
                  (dt->sCount  <  (JFS_DB_SLOTS * ps)) &&
                  (dt->fCount  <  (JFS_DB_SLOTS * ps)) &&
                 ((dt->fList   <  (JFS_DB_SLOTS * ps)) ||
                  (dt->fList  ==   JFS_EMPYLIST))      &&
                  (dt->stIndex <   JFS_MAX_STIN)) // directory tree page
         {
            rc = ST_JDTPG;
         }
         else if ((xt->maxentry != (JFS_XB_XADS * ps)) ||
                  (xt->entries  >  (JFS_XB_XADS * ps))  ) // no allocation tree page
         {
            rc = ST_UDATA;
            dr = DFS_PENDING;
         }
         break;

      case ST_JFSUP:                            // Superblock sanity checks
         if ((memcmp(  sd->Signature,sg_jfsup,SG_JFSUP)) || // non-std signature
             (!isdigit(sd->SignatNum))                   || // non-std sig digit
                      (sd->Version > 10)                 || // non-std version/status
                      (sd->Status  & 0xffffff00))
         {
            rc = ST_JSBAD;
         }
         break;

      case ST_JINOD:                            // Inode, sanity check
         sn = dfsJfsPxdValue( in->Extent) * jfs->SectorsPerBlock; // Extent Lsn
         if ((in->Uid       <= JFS_MAX_UID)  && // Uid/Gid usually 16 bit
             (in->Gid       <= JFS_MAX_GID)  && // but could be larger on UNIX
             (in->Fileset   <= JFS_MAXFSET)  &&
             (sn <= lsn) && ((sn + JFS_INODE_EXTENT_SIZE) > lsn) &&
            ((in->Extent.len * jfs->SectorsPerBlock) == JFS_INODE_EXTENT_SIZE))
         {
            if  ((in->LastAccess.hi == 0) &&
                 (in->LastStatus.hi == 0) &&    // or none of the timestamps
                 (in->LastModify.hi == 0) &&    // being a valid date
                 (in->Creation.hi   == 0)  )
            {
               rc = ST_JIBAD;                   // BAD Inode
            }
            else
            {
               switch (in->Fileset)
               {
                  case JFS_INODE_FILESET_1:
                     if (in->Mode & 0x4000)     // Posix directory bit ?
                     {
                        if (in->Links != 0)
                        {
                           rc = ST_JIDIR;
                        }
                        else
                        {
                           rc = ST_JIDDI;
                        }
                     }
                     else
                     {
                        if (in->Links != 0)
                        {
                           rc = ST_JINOD;
                        }
                        else
                        {
                           rc = ST_JIDEL;
                        }
                     }
                     break;

                  case JFS_INODE_AGGREGATE:  rc = ST_JINAG; break;
                  default: dr = DFS_PENDING; rc = ST_UDATA; break;
               }
            }
         }
         else
         {
            rc = ST_UDATA;
            dr = DFS_PENDING;
         }
         break;

      default:
         break;
   }
   *st = rc;
   RETURN (dr);
}                                               // end 'dfsJfsIdent'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// JFS filesystem, supply sector-type description string
/*****************************************************************************/
static ULONG dfsJfsStype
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *st,                      // IN    sector type
   void               *data                     // OUT   type description
)
{
   ULONG               rc  = NO_ERROR;
   char               *buf = (char *) data;
   BYTE                tp  = st[0];

   switch (tp)                                  // searchable types
   {
      case ST_JFSUP: sprintf(buf,"JFS  superblock  "); break;
      case ST_JSBAD: sprintf(buf,"JFS  BAD superblk"); break;
      case ST_JINOD: sprintf(buf,"File        Inode"); break;
      case ST_JIDIR: sprintf(buf,"Directory   Inode"); break;
      case ST_JIDEL: sprintf(buf,"Deleted FileInode"); break;
      case ST_JIDDI: sprintf(buf,"Deleted Dir Inode"); break;
      case ST_JINAG: sprintf(buf,"Aggregate   Inode"); break;
      case ST_JIBAD: sprintf(buf,"Unused/BAD  Inode"); break;
      case ST_JXTPG: sprintf(buf,"Data Btree+  Page"); break;
      case ST_JDTPG: sprintf(buf,"Dir  Btree+  Page"); break;
      case ST_JIMAG: sprintf(buf,"Pri Agg Inode Map"); break;
      case ST_JIAGA: sprintf(buf,"Agg Inode map IAG"); break;
      case ST_JIAGF: sprintf(buf,"Fs1 Inode map IAG"); break;
      case ST_JIMFS: sprintf(buf,"Fileset Inode Map"); break;
      default:
         switch (tp | ST__INFO)                 // non-searchable ones
         {
            case ST_JIMA2: sprintf(buf,"Sec Agg Inode Map"); break;
            case ST_JI2AG: sprintf(buf,"Sec Aggreg. Inode"); break;
            case ST_JIXAG: sprintf(buf,"Pri Agg Inode Ext"); break;
            case ST_JIXA2: sprintf(buf,"Sec Agg Inode Ext"); break;
            case ST_JIXFS: sprintf(buf,"Fileset Inode Ext"); break;
            case ST_JBLAM: sprintf(buf,"Block Alloc.  Map"); break;
            case ST_JFSCK: sprintf(buf,"FSCK working area"); break;
            case ST_JINLL: sprintf(buf,"Inline Log area  "); break;
            case ST_JEASD: sprintf(buf,"EA   data        "); break;
            case ST_JACLD: sprintf(buf,"ACL  data        "); break;
            case ST_JBOOT: sprintf(buf,"Micro-FS bootcode"); break;
            case ST_SLACK: sprintf(buf,"Last-block slack "); break;
            default:       rc = DFS_PENDING;                 break;
         }
         break;
   }
   return (rc);
}                                               // end 'dfsJfsStype'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// JFS filesystem, display sector-contents based on type
/*****************************************************************************/
static ULONG dfsJfsDispl
(
   ULN64               psn,                     // IN    base psn for sector
   ULN64               d2,                      // IN    dummy
   char               *type,                    // IN    type of sector
   void               *data                     // IN    sector contents
)
{
   ULONG               rc  = NO_ERROR;
   BYTE                st  = (BYTE) *type;
   ULN64               lsn = dfstPsn2LSN( DFSTORE, psn);

   ENTER();

   switch (st)
   {
      case ST_JSBAD:
      case ST_JFSUP: (void) dfsJfsSuperBlock( rbuf, TRUE);        break;
      case ST_JINAG:
      case ST_JIBAD:
      case ST_JIDEL:
      case ST_JIDDI:
      case ST_JIDIR:
      case ST_JINOD: (void) dfsJfsInodeSec(   rbuf, lsn, st);  break;

      case ST_JXTPG:
         rc = dfstReadPsn( DFSTORE, psn+1, 7, rbuf + dfsGetSectorSize()); // Read rest Xtree page
         (void) dfsJfsXtreePage( rbuf);
         break;

      case ST_JDTPG:
         rc = dfstReadPsn( DFSTORE, psn+1, 7, rbuf + dfsGetSectorSize()); // Read rest Dtree page
         (void) dfsJfsDtreePage( rbuf);
         break;

      case ST_JIAGA:
      case ST_JIAGF:
         rc = dfstReadPsn( DFSTORE, psn+1, 7, rbuf + dfsGetSectorSize()); // Read rest IAG page
         (void) dfsJfsIagPage( rbuf);
         break;

      default:
         rc = DFS_PENDING;
         break;
   }
   RETURN (rc);
}                                               // end 'dfsJfsDispl'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display JFS  super-block
/*****************************************************************************/
static ULONG dfsJfsSuperBlock                   // RET   rc = 0 if type match
(
   BYTE               *sector,                  // IN    Drive specification
   BOOL                navigation               // IN    update nav values
)
{
   ULONG               rc = 0;                  // rc, sector match
   S_JFSUP            *sd = (S_JFSUP *) sector;
   ULONG               fl = sd->Flag;
   ULN64               AgSize   = (sd->AllocSize * jfs->SectorsPerBlock);
   time_t              tm;                      // c-runtime time_t format
   TXTM                text;

   ENTER();
   if ((sd->SignatNum == '1') && (memcmp( sd->Signature,sg_jfsup,SG_JFSUP) == 0))
   {
      TxPrint("Superbl signature : 0x%s%2.2hx%s '%s%4.4s%s'\n",
                                     CBG, sd->SignatNum, CNN, CBG, sd->Signature, CNN);
   }
   else                                         // non-standard signature
   {
      TxPrint("Superbl signature : 0x%s%2.2hx%s '%s%4.4s%s'"
              ", Non-std signature, use 'sbsig fix' to correct!\n",
                 CBR, sd->SignatNum, CNN, CBR, sd->Signature, CNN);

   }
   TxPrint("Fmt by JFS version: %u          = ", sd->Version);
   switch (sd->Version)
   {
      case 1:
         TxPrint("I386 %sOS2 %sArcaOS %seCS %sWsEB%s (original IBM)\n", CBG, CBC, CBY, CNG, CNN);
         break;

      case 2:
         TxPrint("I386 %sLinux%s Open Source (OS/2 compatible)\n", CBG, CNN);
         break;

      default:
         TxPrint("%sNew, unknown JFS version!%s, Update time!\n", CBR, CNN);
         break;
   }
   dfsJfsDisplayStatus( "Filesystem status : ", sd->Status);

   if (fl & JFS_AIX        ) TxPrint("AIX  style naming : %sYes%s\n", CBR, CNN);
   if (fl & JFS_OS2        ) TxPrint("OS/2 style naming : Yes\n");
   if (fl & JFS_DFS        ) TxPrint("DCE  style naming : %sYes%s\n", CBR, CNN);
   if (fl & JFS_UNICODE    ) TxPrint("Unicode    naming : Yes\n");
   if (fl & JFS_GROUPCOMMIT) TxPrint("Group (1)  commit : Yes\n");
   if (fl & JFS_LAZYCOMMIT ) TxPrint("Lazy       commit : Yes\n");
   if (fl & JFS_TMPFS      ) TxPrint("Temporary Fsystem : Yes, no commit/Log\n");
   if (fl & JFS_INLINEMOVE ) TxPrint("Inline log status : Being moved!\n");
   if (fl & JFS_BAD_SAIT   ) TxPrint("Sec A-Inode table : %sBad%s\n", CBR, CNN);
   if (fl & JFS_SPARSE     ) TxPrint("Sparse files used : Yes\n");
   if (fl & JFS_DASD_ENABLED)TxPrint("DASD  user limits : Enabled\n");
   if (fl & JFS_DASD_PRIME ) TxPrint("DASD  CHK on boot : Yes\n");
   if (fl & JFS_DIR_INDEX  ) TxPrint("Linux DIR Indexes : Yes\n");

   if ((jfs->SectorsPerBlock) && (sd->SectSize)) // avoid divide by zero
   {
      dfsSz64("FS-Aggregate size : ",      sd->TotalSec,               "      ");
      TxPrint("   = 0x%10.10llX blocks\n", sd->TotalSec  / jfs->SectorsPerBlock);
      dfsSiz8("AllocBlock   size : ",      sd->BlockSize / sd->SectSize,   "\n");
      dfsSz64("AllocGroup   size : ",      AgSize,                         "\n");
      dfsDec8("Blocks/AllocGroup : ",      sd->AllocSize,                    "");
      dfsDec8(       " => Groups : ",    ((sd->TotalSec -1) / AgSize) + 1, "\n");
   }
   dfsSpxd("SA Inode map  @Bl : ",  jfs->SectorsPerBlock, &(sd->SaInodeMap),    "\n");
   dfsSpxd("SA InodeTable @Bl : ",  jfs->SectorsPerBlock, &(sd->SaInodeTable),  "\n");

   dfsSpxd("FSCK workarea @Bl : ",  jfs->SectorsPerBlock, &(sd->FsckSpace),     "\n");
   dfsSiz8("Area size FsckLog : ", (jfs->SectorsPerBlock *  sd->FsckLogLen),    "  ");
   TxPrint("Active = %u ",                                  sd->FsckRecent);
   switch (sd->FsckRecent)
   {
      case 0:  TxPrint( "(None)\n");                   break;
      case 1:  TxPrint( "(First)\n");                  break;
      case 2:  TxPrint( "(Second)\n");                 break;
      default: TxPrint( "(%sIllegal%s)\n", CBR, CNN);  break;
   }
   dfsSpxd("Inline log    @Bl : ",   jfs->SectorsPerBlock, &(sd->InlineLog),    "\n");
   TxPrint("LOG DeviceAddress : 0x%8.8x   Log serial at mount : 0x%8.8x\n",
                                            sd->LogDevAddr,  sd->LogSerial);
   tm = (time_t)                            sd->LastUpdate.hi;
   TxPrint("Last Update  time : %-24.24s", (sd->LastUpdate.hi == 0) ? "Never" : ctime( &tm));
   TxPrint("    nsec : 0x%8.8x\n",         sd->LastUpdate.lo);

   TxPrint("FS volume name    : %s% -11.11s%s\n", CBY, sd->FsVolume, CNN);

#if defined (OS2_NEW_JFS_KNOWN)
   if ((sd->os2.zero1 == 0) &&                  // most likely an OS/2 style JFS
       (sd->os2.zero2 == 0) &&
       (sd->os2.zero3 == 0)  )
   {
      //- to be refined, may get some info from AN (Steven) about new JFS superblock
      TxPrint("OS/2 unknown info : 0x%8.8x (@x94) - 0x%8.8x (@a0)\n",
               sd->os2.unknown1,  sd->os2.unknown2);
   }
   else                                         // Linux style JFS, UUID/VolName
#endif

   {
      if (!isprint( sd->linux.lxVolName[0]))
      {
         strcpy( text, "-none-");
      }
      else
      {
         TxCopy(  text, sd->linux.lxVolName, JFS_LEN_LBL + 1);
      }
      TxPrint("Linux Vol label   : %s%s%s\n", CBG, text, CNN);

      //- Note that the endian-swap in the UUID is already IN the on-disk UUID, no swap now!
      TxPrint("Linux Vol UUID    : %s\n", dfsFsUuidValueNoSwap( sd->linux.lxVolUuid));

      if ((sd->linux.lxLogUuid[0] != 0) && (sd->linux.lxLogUuid[1] != 0))
      {
         TxPrint("Linux Log UUID    : %s\n", dfsFsUuidValueNoSwap( sd->linux.lxLogUuid));
      }
   }

   if (navigation)
   {
      nav.down = dfsJfsInode2Lsn(JFS_ROOT_INODE, &jfs->Fs1); // Root Inode FS1
      nav.xtra = LSN_JFSU2;                     // navigate to spare-superblock
   }
   RETURN (rc);
}                                               // end 'dfsJfsSuperBlock'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display value of superblock status value
/*****************************************************************************/
static void dfsJfsDisplayStatus
(
   char               *lead,                    // IN    leading text
   ULONG               st                       // IN    JFS status value
)
{
   TxPrint("%s%u          = ", lead, st);
   if      (st == FM_CLEAN) TxPrint( "%sUnmounted and clean %s(not dirty)", CBG, CNN);
   else if (st == FM_MOUNT) TxPrint( "%sMounted cleanly %s(OK, but dirty)", CBG, CNN);
   else
   {
      TxPrint( "%s", CBR);
      if (st & FM_MOUNT)    TxPrint( "Mounted, but ");
      if (st & FM_DIRTY)    TxPrint( "INCONSISTENT Dirty; ");
      if (st & FM_LOGREDO)  TxPrint( "Log redo failed. CHK needed ");
      if (st & FM_EXTENDFS) TxPrint( "Extend FS in progress. CHK needed");
   }
   TxPrint( "%s\n", CNN);                       // back to neutral color
}                                               // end 'dfsJfsDisplayStatus'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display JFS Inode sector contents
/*****************************************************************************/
static ULONG dfsJfsInodeSec                     // RET   rc = 0 if type match
(
   BYTE               *sector,                  // IN    Sector contents
   ULN64               lsn,                     // IN    sector number
   BYTE                st                       // IN    INODE type
)
{
   ULONG               rc    = NO_ERROR;        // rc, sector match
   S_JINODE           *sd    = (S_JINODE *) sector;
   BOOL                isDir = (S_ISDIR( sd->Mode));
   TXLN                text;                    // ASCII buffer, large
   TXTM                desc;                    // ASCII buffer, medium
   ULN64               alloc = (*((ULN64 *) &sd->Allocated)) * jfs->sup->BlockSize;
   DFSISPACE           fdata;
   ULONG               nr;
   ULN64               root;                    // root LSN

   ENTER();
   nav.up = lsn;                                // allow easy way back ...

   dfsJfsTime2str( (JTIME *) &sd->InoStamp, text);

   TxPrint(       "Fileset  InoStamp : 0x%8.8x = %s\n", sd->InoStamp, text);
   sprintf( text, "Inode self number : 0x%8.8x = ",     sd->Self);
   if ((sd->Self == JFS_FS1_SUPER_INODE) && (sd->Fileset == JFS_INODE_AGGREGATE))
   {
      TxPrint( "%s%-2u (%sSuper%s)  ", text, sd->Self, CBC, CNN);
   }
   else if ((sd->Self == JFS_ROOT_INODE) && (sd->Fileset == JFS_INODE_FILESET_1))
   {
      TxPrint( "%s%-2u  (%sroot%s)", text,  sd->Self, CBY, CNN);
   }
   else                                         // not a special Inode
   {
      TxPrint( "%s%-10u", text, sd->Self);
   }
   TxPrint( "  FileSet : 0x%2.2x  Generation : %u\n",       sd->Fileset,
                                                             sd->Generation);
   dfsPosixUidGidPrint( sd->Uid, sd->Gid);
   TxPrint( "File Mode   (%03o) : %s\n", (sd->Mode & 0777),
                      dfsPosixMode2String( sd->Mode, TXMAXLN, text));

   strcpy( text, "");
   dfstrFatAttrib( text, (BYTE) (sd->Mode >> JFS_FATTRSHIFT));
   TxPrint( "FAT style Attrib. : %s%s%s    ",  CBC, text, CNN );

   TxPrint( "                 Inode Links : %u   %s\n", sd->Links,
            ((sd->Fileset == JFS_INODE_FILESET_1) &&
             (sd->Links   == 0)) ? "(Deleted)" : "");

   if (sd->Fileset == JFS_INODE_FILESET_1)      // skip for aggregate inodes
   {
      if (dfsJfsResolveInodeName( lsn, sd, st, &nr, text))
      {
         if (nr != L32_NULL)                    // parent resolved too ?
         {
            TxPrint( "Parent   Inode nr : 0x%8.8x\n", nr);
            nav.up = dfsJfsInode2Lsn( nr, &jfs->Fs1);
         }
         if (strlen( text))
         {
            TxPrint( "1st link Filename : %s%s%s\n", CBY, text, CNN);
         }
         if (dfsJfsFindRootInode( lsn, text, &root))
         {
            TxPrint( "Filename fullpath : %s%s%s\n", CBY, text, CNN);
         }
      }
   }
   text[0] = 0;
   dfstrSz64Byte( text, "File size (exact) : ", sd->FileSize, " = ");
   dfstrUllDot20( text, "", sd->FileSize, " bytes");
   if (sd->Allocated == 0)
   {
      if      (sd->FileSize == 0)               // Empty
      {
         strcat( text, "   (Empty file)");
      }
      else if (sd->FileSize <= 256)             // Resident
      {
         strcat( text, "   (Resident)");
      }
   }
   TxPrint( "%s\n", text);

   text[0] = 0;
   dfstrSz64Byte( text, "Allocated  size   : ", alloc, "");
   TxPrint( "%s\n", text);

   if (sd->Eax.size != 0)
   {
      TxPrint( "Attribute EA Size : 0x%8.8x = %10u   Flags  : 0x%2.2hx %2.2hx %2.2hx %2.2hx\n",
                sd->Eax.size, sd->Eax.size, sd->Eax.dflag, sd->Eax.db1,  sd->Eax.db2,  sd->Eax.db3);
      if (sd->Eax.dflag & JFS_EA_INTERN)
      {
         dfsEaBlockDisplay( "EA  internal data : ", sd->Eai.value, sd->Eai.size);
      }
      else                                      // External EA
      {
         dfsSpxd( "EA  external  @Bl : ", jfs->SectorsPerBlock, &(sd->Eax.pe), "\n");
         nav.xtra = jfs->SectorsPerBlock * sd->Eax.pe.lo;
      }
   }
   if ((sd->Acl.size != 0) && (sd->Acl.pe.len != 0))
   {  //- to be refined, could ACL be internal too (like the EA) ?
      TxPrint( "Inode ACL   size  : 0x%8.8x = %10u   "
               "Flags: 0x%2.2hx %2.2hx %2.2hx %2.2hx\n",
                sd->Acl.size, sd->Acl.size, sd->Acl.dflag,
                sd->Acl.db1,  sd->Acl.db2,  sd->Acl.db3);
      dfsSpxd( "ACL external  @Bl : ", jfs->SectorsPerBlock, &(sd->Acl.pe), "\n");
      nav.xtra = jfs->SectorsPerBlock * sd->Acl.pe.lo;
   }
   dfsSpxd( "Inode extent  @Bl : ", jfs->SectorsPerBlock, &(sd->Extent), "\n");

   TxPrint( "Last Access time  : %s",   dfsJfsTime2str( &sd->LastAccess, text));
   TxPrint(" Last Inode Change : %s\n", dfsJfsTime2str( &sd->LastStatus, text));
   TxPrint( "Last Modify time  : %s",   dfsJfsTime2str( &sd->LastModify, text));
   TxPrint(" File Create time  : %s\n", dfsJfsTime2str( &sd->Creation,   text));

   strcpy(  desc, CBY);
   if (sd->Dir.rflag & JFS_DXD_INDEX)    strcat (desc, "Btree ");
   if (sd->Dir.rflag & JFS_BT_ROOT)      strcat (desc, "Root + ");
   if (sd->Dir.rflag & JFS_BT_LEAF)      strcat (desc, "Leaf nodes ");
   if (sd->Dir.rflag & JFS_BT_INTN)      strcat (desc, "Internal nodes");
   if (sd->Dir.rflag & JFS_DXD_INLINE)   strcat (desc, "Inline ");
   if (sd->Dir.rflag & JFS_DXD_EXTENT)   strcat (desc, "Extent ");
   if (sd->Dir.rflag & JFS_DXD_FILE)     strcat (desc, "File Inode ");
   if (sd->Dir.rflag & JFS_DXD_DAMAGED)
   {
     strcat( desc,  CBR);                strcat( desc, "Corrupt!");
   }
   TxPrint( "Description for a : %s %s%s\n", (isDir) ? "Directory   " :
                (sd->Fileset == JFS_INODE_FILESET_1) ? "Regular File" :
                                                       "Aggreg Inode",
                                                        desc, CNN);

   if (isDir) //- directory bit is NOT reliable for deleted directories!
   {
      ULONG   format = DFS_JDIR_STD;

      if (sd->Dir.rflag & JFS_BT_INTN)          // Btree using internal nodes
      {
         format |= DFS_JDIR_BTSPLIT;
      }
      if (!TxaOptUnSet('l'))                    // allow create Sectorlist ?
      {
         format |= DFS_JDIR_LISTADD;
      }
      dfsJfsDirEntries( sd->Slot,
                        JFS_IN_SLOTS,
                        sd->Dir.sTable,         // sort-table
                        sd->Dir.sCount,
                        format);
   }
   else                                         // File, convert to allocation
   {                                            // S_SPACE and display that
      if (sd->File.xCount > 2)
      {
         if ((rc = dfsJfsInodeData2Space( sd, JFS_TREE_INDENT, &fdata, NULL, NULL)) == NO_ERROR)
         {
            ULONG      sdo = SD_BLOCKS | SD_TOPLINE;
            ULN64      totals   = 0;
            ULN64      invalids = 0;
            ULN64      maxSects = (((sd->FileSize -1) / jfs->sup->BlockSize) +1) * jfs->SectorsPerBlock;

            rc = dfsJfsSpaceCheckAlloc( &fdata, ((st != ST_JINOD) &&
                                                 (st != ST_JIDIR) &&
                                                 (st != ST_JINAG)),
                                                  "", maxSects, &totals, &invalids);
            TxPrint( "\n Allocation Check : ");
            switch (rc)
            {
               case DFS_BAD_STRUCTURE:
                  TxPrint("%sFAIL%s     ", CBR, CNN);
                  dfsSz64("failing sectors : ",  invalids, "\n");
                  break;

               case DFS_PSN_LIMIT:
                  TxPrint("%sERROR%s, sector nrs are too large!\n", CBR, CNN);
                  break;

               default:
                  TxPrint("%sOK%s%s\n", CBG, CNN,
                         ((st == ST_JINOD) || (st == ST_JIDEL)) ?
                   "         Saveto/Recover/Undelete of filedata is possible" : "");
                  break;
            }
            if (fdata.chunks > 1)
            {
               sdo |= SD_SUMMARY;
            }
            if (fdata.chunks > 32)              // multi-page is likely
            {
               if (dfsa->verbosity >= TXAO_VERBOSE) // explicit verbose
               {
                  sdo |= SD_BOTLINE;
               }
               else
               {
                  sdo |= SD_LIMIT32;            // just first 32 + summary
               }
            }
            if (!TxaOptUnSet('X'))              // unless -X- specified
            {
               sdo |= SD_NAVDOWN;               // next is data sector
            }
            if (TxaOption('X'))
            {
               sdo |= SD_ALLINFO;               // incl Index and Info
            }
            dfsSspaceDisplay( sdo, fdata.chunks, fdata.space);

            TxFreeMem( fdata.space);
         }
         else
         {
            TxPrint( "Data allocation   : type or structure contents are inconsistent!\n");
            TxDisplayHex( "", (BYTE *) &(sd->Dir.rflag), 64L, 0xF0);
         }
      }
      else
      {
         TxPrint( "Data file extents : ZERO (empty file)\n");
      }
   }
   //- TxDisplayHex( "\nSecond Inode half : \n",
   //-      (char *) &(sd->SymLink), 256L, 0x100);


   RETURN (rc);
}                                               // end 'dfsJfsInodeSec'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display JFS directory Dtree page (for the single page, not whole tree)
/*****************************************************************************/
static ULONG dfsJfsDtreePage                    // RET   rc = 0 if type match
(
   BYTE               *sector                   // IN    data sector(s)
)
{
   ULONG               rc = 0;                  // rc, sector match
   S_DTHEAD           *sd = (S_DTHEAD *) sector;
   BYTE               *st;                      // sort table location
   ULONG               format = DFS_JDIR_STD;

   ENTER();

   dfsSpxd("Dtree Pg Self @Bl : ", jfs->SectorsPerBlock, &(sd->self), "\n");
   dfsX10( "Next     page @Bl : ", sd->next, "", "  ");
   dfsX10(        "sector LSN : ", sd->next * jfs->SectorsPerBlock, "", "\n");
   dfsX10( "Previous page @Bl : ", sd->prev, "", "  ");
   dfsX10(        "sector LSN : ", sd->prev * jfs->SectorsPerBlock, "", "\n");
   TxPrint("Dtree page flags  : 0x%2.2hx  0x%2.2hx  Reserved-2 : 0x%4.4hx      entries: %hu\n",
                                   sd->rflag, sd->stIndex, sd->reserve1, sd->sCount);

   st = ((BYTE *) sd) + (sd->stIndex * JFS_SLOTSIZE); // start of sort-table

   if (sd->rflag & JFS_BT_INTN)                 // Btree using internal nodes
   {
      format |= DFS_JDIR_BTSPLIT;
   }
   if (!TxaOptUnSet('l'))                       // allow create Sectorlist ?
   {
      format |= DFS_JDIR_LISTADD;
   }
   dfsJfsDirEntries( sd->slot,                  // slot array after header
                    (sd->self.len * jfs->SectorsPerBlock * JFS_DB_SLOTS) -1,
                     st, sd->sCount,            // sort-table
                     format);
   RETURN (rc);
}                                               // end 'dfsJfsDtreePage'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display JFS allocation Xtree page (for the single page, not whole tree)
/*****************************************************************************/
static ULONG dfsJfsXtreePage                    // RET   rc = 0 if type match
(
   BYTE               *sector                   // IN    data sector(s)
)
{
   ULONG               rc = 0;                  // rc, sector match
   S_XTHEAD           *sd = (S_XTHEAD *) sector;
   DFSISPACE           fdata;

   ENTER();

   dfsSpxd("Xtree Pg Self @Bl : ", jfs->SectorsPerBlock, &(sd->self), "\n");
   dfsX10( "Next     page @Bl : ", sd->next, "", "  ");
   dfsX10(        "sector LSN : ", sd->next * jfs->SectorsPerBlock, "", "\n");
   dfsX10( "Previous page @Bl : ", sd->prev, "", "  ");
   dfsX10(        "sector LSN : ", sd->prev * jfs->SectorsPerBlock, "", "\n");
   TxPrint("Xtree page flags  : 0x%2.2hx 0x%2.2hx Reserved-2 : 0x%4.4hx  entries: %hu\n",
                                   sd->tflag, sd->tb1, sd->ts1, sd->entries);
   if (sd->entries > 2)
   {
      if (dfsJfsXads2Space( sd->extent,  sd->entries -2, &fdata) == NO_ERROR)
      {
         ULONG sdo = SD_BLOCKS | SD_TOPLINE;
         if (fdata.chunks > 1)
         {
            sdo |= SD_SUMMARY;
         }
         if (fdata.chunks > 32)                 // multi-page is likely
         {
            if (dfsa->verbosity >= TXAO_VERBOSE) // explicit verbose
            {
               sdo |= SD_BOTLINE;
            }
            else
            {
               sdo |= SD_LIMIT32;               // just first 32 + summary
            }
         }
         if (!TxaOptUnSet('X'))                 // unless -X- specified
         {
            sdo |= SD_NAVDOWN;                  // next is data sector
         }
         if (TxaOption('X'))
         {
            sdo |= SD_ALLINFO;                  // incl Index and Info
         }
         dfsSspaceDisplay( sdo, fdata.chunks, fdata.space);

         TxFreeMem( fdata.space);
      }
   }
   else
   {
      TxPrint( "Data file extents : ZERO (empty file)\n");
   }
   RETURN (rc);
}                                               // end 'dfsJfsXtreePage'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display sequence of JFS Directory entries, optional add to LSN list
/*****************************************************************************/
static ULONG dfsJfsDirEntries                   // RET   nr of entries listed
(
   S_JDSLOT           *slot,                    // IN    Directory slot array
   BYTE                maxslot,                 // IN    maximum nr of slots
   BYTE               *sTable,                  // IN    sort-table array
   BYTE                sCount,                  // IN    used entries in sTable
   ULONG               dirFlags                 // IN    presentation flags
)
{
   S_JDSLOT           *de;                      // single directory slot
   BYTE                si;                      // sort index
   ULONG               ei = 0;                  // entry index
   int                 cl = TXaNWnZ;            // LSN color
   int                 ac;                      // alternate color fg+bg
   int                 gc;                      // grey FG, alternate BG
   TX1K                text;                    // text buffer, MAX_PATH + line
   TXLN                tbuf;                    // temporary text buffer
   S_JINODE           *ino;                     // Inode sector data
   BYTE                slotnr;

   ENTER();
   TRACES(( "Slot:%8.8x maxslot:%u  sTable:%8.8x sCount:%u  Flags:%8.8x\n",
             slot, maxslot, sTable, sCount, dirFlags));

   if ((ino = TxAlloc( 1, sizeof( S_JINODE))) != NULL)
   {
      if (dirFlags & DFS_JDIR_THEADER)
      {
         if (dirFlags & DFS_JDIR_BTSPLIT)       // Btree split format
         {
            TxPrint( "\n%s\n%s", jfsBtsHeaderText, jfsBtsHeaderLine);
         }
         else
         {
            TxPrint( "\n%s\n%s", jfsDirHeaderText, jfsDirHeaderLine);
         }
      }
      TxPrint( "\n");
      if (dirFlags & DFS_JDIR_LISTADD)          // create a new list for DIR
      {
         dfsInitList(0, "-f -P", "-d");         // optimal for menu file-recovery
      }

      for ( si = 0, ei = 0;
           (si < sCount) && !TxAbort();         // used entries in sort-table
            si++)
      {
         slotnr = sTable[(int) si] -1;          // ZERO based slotnumber
         de     = slot + slotnr;                // Slot with Inode + name

         TRACES(( "si: %u  ei: %u  index: %hu = slotnr: %hu\n",
                   si, ei,  sTable[(int) si], slotnr));

         if (dirFlags & DFS_JDIR_BTSPLIT)       // Btree split format
         {
            ULN64         bdown;                // B-tree PAGE block LSN

            bdown = dfsJfsBlock2Lsn( dfsJfsPxdValue( de->t.Page));

            TRACES(( "Btree-down: 0x0%llx  Dnext:0x%2.2hx Len:%hu\n", bdown, de->t.Dnext, de->t.NameLen));

            sprintf(  text, ".%5.5u", ei);
            dfstrX10( text, " ", bdown, (char *) ((ei == 0) ? CBG : CNN), "");
            dfstrX10( text, " ", dfsJfsPxdValue(de->t.Page), "", " ");

            dfsJfsBtspNameAppend( slot, text, slotnr, maxslot);

            if (ei == 0)
            {
               nav.down = bdown;
            }
            TxPrint( "%s\n", text);
            ei++;
            if (dirFlags & DFS_JDIR_LISTADD) // create a new list for DIR
            {
               dfsAdd2SectorList( bdown);       // add to list
            }
         }
         else                                   // regular directory format
         {
            ULN64         inolsn = 0;           // Inode LSN number

            TRACES(( "Inode: 0x%8.8x  Dnext:0x%2.2hx Len:%hu\n",
                      de->i.Inode, de->i.Dnext, de->i.NameLen));

            if (dirFlags & DFS_JDIR_USEINODE)
            {
               inolsn = dfsJfsInode2Lsn( de->i.Inode, &jfs->Fs1);
               if ((inolsn != 0) && (inolsn != L64_NULL))
               {
                  dfsRead( inolsn, 1, (BYTE   *) ino);
               }
            }

            if (ei % 2)
            {
               ac = TXaBWnC;
               gc = TXaBZnC;
            }
            else
            {
               ac = TXaNWnZ;
               gc = TXaBZnZ;
            }
            if (ei == 0)                        // first entry in the list
            {
               cl = TXaBGnZ;
               if (inolsn)
               {
                  nav.down = inolsn;
               }
            }
            else
            {
               cl = ac;
            }
            sprintf(  text, "%s.%5.5u", ansi[ac], ei);
            dfstrX10( text, "", inolsn,  ansi[cl], ansi[ac]);
            strcat(   text, " ");

            strcat(   text, dfsJfsTime2str( &ino->Creation, tbuf));

            strcpy(   tbuf, "");
            dfstrFatAttrib( tbuf, (BYTE) (ino->Mode >> JFS_FATTRSHIFT));
            tbuf[0] = ' ';                      // replace 'Label' by space
            tbuf[5] = '\0';                     // cut-off 'Directory' letter
            strcat(   text, tbuf);              // middle 4 FAT attributes

            if ((ino->LastModify.hi != ino->Creation.hi) ||
                (ino->LastModify.lo != ino->Creation.lo)  )
            {
               strcat( text, " ");
               strcat( text, dfsJfsTime2str(&ino->LastModify, tbuf));
            }
            else
            {
               strcat( text, "                    ");
            }

            if ((ino->FileSize == 0) && (ino->Allocated == 0))
            {
               strcat( text, "                      ");
            }
            else
            {
               dfstrUllDot20( text, " ", ino->FileSize, " ");
            }
            sprintf( tbuf, "%8x%s%s", de->i.Inode, CGE, ansi[ac]);
            strcat(  text, tbuf);

            if (ino->Eax.size)
            {
               sprintf( tbuf, " %5u\n", ino->Eax.size);
            }
            else
            {
               strcpy( tbuf, "\n");
            }
            strcat(  text, tbuf);

            if ((ino->LastAccess.hi != ino->Creation.hi) ||
                (ino->LastAccess.lo != ino->Creation.lo)  )
            {
               sprintf( tbuf, "  %saccessed%s%s ", ansi[gc], CNN, ansi[ac]);
               strcat(  text, tbuf);
               strcat(  text, dfsJfsTime2str(&ino->LastAccess, tbuf));
            }
            else
            {
               sprintf( tbuf, "  %28.28s", "");
               strcat(  text, tbuf);
            }
            strcat( text, (((ino->Mode >> JFS_FATTRSHIFT) & FATTR_DIRECTORY) ||
                    S_ISDIR( ino->Mode)) ? " D" : "  ");
            if (ino->Mode & 0777)
            {
               sprintf( tbuf, "%3o ", (ino->Mode & 0777));
               strcat(  text, tbuf);
            }
            else
            {
               strcat(  text, "    ");
            }
            strcat(  text, ansi[Ccol((CcY | CcI), Ccbg(ac))]); // Yellow
            dfsJfsFileNameAppend( slot, text, slotnr, maxslot);
            sprintf( tbuf, "%s%s%s\n", ansi[Ccol( CcW, Ccbg(ac))], CGE, CNN);
            strcat(  text, tbuf);

            TxPrint( "%s", text);

            ei++;
            if (dirFlags & DFS_JDIR_LISTADD)    // create a new list for DIR
            {
               dfsAdd2SectorList( inolsn);      // add to list
            }
         }
      }

      if ((dirFlags & DFS_JDIR_BHEADER) && (ei > 12)) // reversed header at end
      {
         if (dirFlags & DFS_JDIR_BTSPLIT)       // Btree split format
         {
            TxPrint( "%s\n%s\n", jfsBtsHeaderLine, jfsBtsHeaderText);
         }
         else
         {
            TxPrint( "%s\n%s\n", jfsDirHeaderLine, jfsDirHeaderText);
         }
      }
      TxFreeMem( ino);                          // free Inode data
   }
   RETURN (ei);
}                                               // end 'dfsJfsDirEntries'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display JFS Inode Allocation Group structure (IAG) mapping max 4096 Inodes
/*****************************************************************************/
static ULONG dfsJfsIagPage                      // RET   rc = 0 if type match
(
   BYTE               *sector                   // IN    data sector(s)
)
{
   ULONG               rc = 0;                  // rc, sector match
   S_IAG              *sd = (S_IAG *) sector;
   int                 i;
   ULONG               allocated[ JFS_SMAPSZ];
   ULONG               freeInode[ JFS_SMAPSZ];
   ULONG               exts;                    // nr of extents
   ULONG               inos;                    // nr of inodes
   ULONG               iChunks;
   S_SPACE            *iSpace;
   ULONG               lsn;
   ULONG               bpe = (JFS_INOEXT / jfs->SectorsPerBlock); // blocks per extent

   ENTER();

   TxPrint("Alloc Group   @Bl : 0x%8.8lx  sector LSN : 0x%8.8llx  IAG number : %u\n",
                                          sd->blAgStart, jfs->SectorsPerBlock * sd->blAgStart,
                                          sd->iagNumber);

   if (dfsJfsIagMkSummary( sd, allocated, freeInode, &exts, &inos) == NO_ERROR)
   {
      ULONG            cfe = JFS_IAGEXT - exts;
      ULONG            cfi = JFS_INOEXT * exts - inos;

      if ((dfsa->verbosity  >= TXAO_VERBOSE)    ||
          (sd->iagNumber    >= jfs->Fs1.Chunks) ||
          (sd->extFreeCount != cfe) ||
          (sd->inoFreeCount != cfi)  )          // verbose or inconsistent
      {
         TxPrint("FreeList pointers : 0x%8.8x, 0x%8.8x,  0x%8.8x, 0x%8.8x, 0x%8.8x\n",
                                                 sd->inoFreeHead, sd->inoFreeTail,
                                                 sd->extFreeHead, sd->extFreeTail,
                                                 sd->iagFreeList);
         TxPrint("Inode status @x20 :");
         for (i = 0; i < JFS_SMAPSZ; i++)
         {
            TxPrint( " %2.2x %2.2x %2.2x %2.2x",
                       (sd->inoSumMap[i]      ) & 0xff,
                       (sd->inoSumMap[i] >>  8) & 0xff,
                       (sd->inoSumMap[i] >> 16) & 0xff,
                       (sd->inoSumMap[i] >> 24) & 0xff);
         }
         TxPrint("\nCalculated Status :");
         for (i = 0; i < JFS_SMAPSZ; i++)
         {
            TxPrint( " %2.2x %2.2x %2.2x %2.2x",
                       (freeInode[i]      ) & 0xff,
                       (freeInode[i] >>  8) & 0xff,
                       (freeInode[i] >> 16) & 0xff,
                       (freeInode[i] >> 24) & 0xff);
         }
         TxPrint("\nExtent Alloc @x30 :");
         for (i = 0; i < JFS_SMAPSZ; i++)
         {
            TxPrint( " %2.2x %2.2x %2.2x %2.2x",
                       (sd->extBitMap[i]      ) & 0xff,
                       (sd->extBitMap[i] >>  8) & 0xff,
                       (sd->extBitMap[i] >> 16) & 0xff,
                       (sd->extBitMap[i] >> 24) & 0xff);
         }
         TxPrint("\nCalculated Alloc  :");
         for (i = 0; i < JFS_SMAPSZ; i++)
         {
            TxPrint( " %2.2x %2.2x %2.2x %2.2x",
                       (allocated[i]      ) & 0xff,
                       (allocated[i] >>  8) & 0xff,
                       (allocated[i] >> 16) & 0xff,
                       (allocated[i] >> 24) & 0xff);
         }
         TxPrint("\nCalc free Extents : 0x%4.4x = % 3u, used %u    "
                              "Inodes : 0x%4.4x = % 4u, used %u\n",
                               cfe, cfe, exts, cfi, cfi, inos);
      }
      TxPrint("IAG  free Extents : 0x%4.4x = % 3u, used %u    "
                         "Inodes : 0x%4.4x = % 4u, used %u\n",
         sd->extFreeCount, sd->extFreeCount,
              JFS_IAGEXT - sd->extFreeCount,
         sd->inoFreeCount, sd->inoFreeCount,   JFS_INOEXT *
             (JFS_IAGEXT - sd->extFreeCount) - sd->inoFreeCount);

      if ((iSpace = TxAlloc( JFS_IAGEXT, sizeof(S_SPACE))) != NULL)
      {
         ULONG          sdo = SD_BLOCKS | SD_TOPLINE;
         int            ei;                // extent index
         int            mw;                // mapword index
         int            mb;                // bit in a mapword
         ULONG          mword;

         for (mw = 0, ei = 0, iChunks = 0;
              mw < JFS_SMAPSZ;
              mw++)                             // iterate over all possible
         {                                      // Inode extents in the IAG
            mword = sd->extBitMap[ mw];
            for (mb = 0; mb < 32; mb++, ei++)
            {
               if (mword & 0x80000000)          // Allocated if MSB set
               {
                  lsn = dfsJfsBlock2Lsn( dfsJfsPxdValue( sd->inoExtent[ei]));
                  if (lsn && (lsn < jfs->Sect) && (sd->inoExtent[ei].len == bpe))
                  {
                     iSpace[iChunks].start = lsn; // location
                     iSpace[iChunks].size  = JFS_INOEXT; // #sectors in extent
                     iChunks++;
                  }
               }
               mword <<= 1;                     // shift left by 1 bit
            }
         }
         if (iChunks > 1)
         {
            sdo |= SD_SUMMARY;
         }
         dfsSspaceDisplay( sdo, iChunks, iSpace);
         TxFreeMem( iSpace);
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   RETURN (rc);
}                                               // end 'dfsJfsIagPage'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display IAG summary information for the specified Inode map
/*****************************************************************************/
static ULONG dfsJfsIagSummary
(
   ULONG               verbose,                 // IN    detail level
   char               *lead9,                   // IN    9 char lead text
   S_IMAP             *map                      // IN    Inode map info
)
{
   ULONG               rc  = NO_ERROR;
   ULONG               size;                    // sectors in IAG file
   ULONG               rsn;                     // RSN in IAM file
   ULONG               isi;                     // startindex in Inode map
   ULONG               iagNr;                   // iag sequence number
   ULONG               extents;                 // nr of Inode extents
   ULONG               invalids = 0;            // invalid extent LSN's
   ULONG               badiags  = 0;            // invalid IAG strcutures
   ULONG               inIag;                   // allocated Inodes in IAG
   S_IAG              *iag;                     // IAG structure buffer

   ENTER();

   if ((iag = TxAlloc( 1, sizeof( S_IAG))) != NULL)
   {
      size        = dfsSspaceSectors( FALSE, map->Chunks, map->Space);
      extents     = ( (size - JFS_IAGCPG) / JFS_IAGSEC) * JFS_IAGEXT;

      if (extents > 200)
      {
         dfsProgressInit( 0, (extents / JFS_IAGEXT), 0, "InodeAG", "usage checked", DFSP_STAT | DFSP_VDEC, 0);
      }
      if (verbose & TXAO_VERBOSE)
      {
         TxPrint( "Inode Alloc Group   Allocated/used Inodes, decimal   AllocGroup Block    1st Inode\n");
         TxPrint( "=================   ==============================   ================   ==========\n");
      }
      for (  rsn  = JFS_IAGCPG, isi  = 0,          iagNr = 0;
           ((rsn +  JFS_IAGSEC)     <= size)  &&   (rc == NO_ERROR);
             rsn += JFS_IAGSEC, isi += JFS_IAGEXT, iagNr++)
      {
         rc = dfsSspaceReadFilePart( map->Chunks, map->Space,
                                     rsn, JFS_IAGSEC, (BYTE *) iag);
         if (rc == NO_ERROR)
         {
            if ((iag->iagNumber < JFS_IAGMAX      ) &&
                (iag->blAgStart < jfs->TotalBlocks)  ) // IAG sanity check
            {
               inIag = (JFS_IAGEXT - iag->extFreeCount) * JFS_INOEXT;

               if (verbose & TXAO_VERBOSE)
               {
                  TxPrint( "InodeAllocGrp% 4u :% 8u Inodes, used :% 8u   @Bl 0x%10.10llx : 0x%8.8x\n",
                        iag->iagNumber, inIag, (inIag - iag->inoFreeCount),
                        iag->blAgStart, isi * JFS_INOEXT);
               }
               rc = dfsJfsIagAddExtents( iag, isi, extents, NULL, &invalids);
            }
            else
            {
               badiags++;
               if (verbose & TXAO_VERBOSE)
               {
                  TxPrint( "InodeAllocGrp% 4u : IagNumber : 0x%8.8x   @Bl 0x%10.10llx : 0x%8.8x\n",
                            iagNr, iag->iagNumber, iag->blAgStart, isi * JFS_INOEXT);
               }
            }
         }
         if (extents > 200)
         {
            dfsProgressShow( (rsn / JFS_IAGSEC), 0, 0, NULL);
         }
      }
      if (extents > 200)
      {
         dfsProgressTerm();
      }
      if (verbose)
      {
         if (verbose & TXAO_VERBOSE)
         {
            TxPrint( "             ----   =======+               =======+\n");
         }
         TxPrint( "%sIAGs% 4u :% 8u Inodes, used :% 8u   MaxInodeNumber : 0x%8.8x\n", lead9,
           (rsn / JFS_IAGSEC) -1, map->Alloc, map->Used, map->Max -1);
      }
      if (invalids != 0)
      {
         TxPrint( "BAD Inode extents : %u extents with INVALID location!\n", invalids);
      }
      if (badiags != 0)
      {
         TxPrint( "BAD IAG structures: %u with incorrect header info!\n", badiags);
      }
      if (verbose & TXAO_VERBOSE)
      {
         ULONG sdo = SD_BLOCKS | SD_TOPLINE;

         if (map->Chunks > 1)
         {
            sdo |= SD_SUMMARY;
         }
         dfsSspaceDisplay( sdo, map->Chunks, map->Space);
      }
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN(rc);
}                                               // end 'dfsJfsIagSummary'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Make filesystem usage map dump on TxPrint output
/*****************************************************************************/
static ULONG dfsJfsAllocMap
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *options,                 // IN    display options
   void               *data                     // OUT   dummy
)
{
   ULN64               i;                       // index iterating over blocks
   ULONG               b;                       // byte-index for one mapchar
   ULN64               cClus = 0;               // Clusters in current char
   ULN64               lClus = 0;               // Clusters in current line
   ULN64               mClus = 0;               // Clusters in current map
   ULN64               uClus = 1;               // Last cluster that is in use
   ULONG               l;                       // line number, 0 based
   ULONG               a;                       // index in display-array
   ULN64               perc;                    // percentage
   TX1K                ascii;                   // display-array
   ULONG               spe;                     // sectors per entry
   ULONG               acl;                     // ascii alloc chars per line
   ULONG               cpc;                     // clusters per display-char
   ULONG               spc;                     // sectors per display-char
   ULN64               size;                    // nr of clusters to map
   BOOL                verbose = (*options != '@');
   ULONG               bsSmart;
   ULN64               unallocSmart = 0;        // Size of THIS unallocated area
   BOOL                oneLineProgress = FALSE; // alloc in one line, output per character

   ENTER();

   dfsa->FsUnallocSmart = 0;                    // will be recalculated now
   bsSmart = dfsGetBufferSize( DFSOPTIMALBUF, DFSMAXBUFSIZE); // for SMART prediction

   spe   = jfs->SectorsPerBlock;
   size  = jfs->VolumeBlocks;                   // Block 0 .. last-valid

   acl   = dfsAllocCharsPerLine( 'c');
   cpc   = dfsAllocItemsPerChar( 'l', options, size, acl);
   if (cpc >= (size / acl))                     // will be a single line
   {
      oneLineProgress = TRUE;                   // one char at a time, no ascii string used
   }
   spc   = cpc * spe;
   dfsProgressInit( 0, size * spe, 0, "Get ALLOCATION info, at Sector:", dfstStoreDpid( DFSTORE), DFSP_STAT, 0);

   if (verbose)
   {
      dfsSz64("Size for one line : ", (ULN64) acl * spc, "  ");
      TxPrint("with %3u characters,", acl);
      dfsSiz8(" size : ", spc, "\n");
      TxPrint(" %16.16s = Allocation grade, empty to full.     ", mapchar);
      dfsSiz8(" SmartBuf size : ", bsSmart, "\n");
   }
   TxPrint(   "          %s%*.*s%s\n", CBC, acl, acl, BLIN, CNN);
   dfsX10( CNZ, dfsJfsBlock2Lsn(0), CNZ, "");   // display address for first block
   TxPrint("%s%s", CBC, CNC);

   for (i=0, a=0, l=0; (i < size) && (!TxAbort());)
   {
      if (a == 0)
      {
         memset(ascii, 0, TXMAXLN);
         lClus = 0;
      }
      for (cClus=0, b=0; (b < cpc) && (i < size); b++, i++)
      {
         if (dfsJfsBitmapCache(i, NULL))        // block in use ?
         {
            if (unallocSmart != 0)              // first in-use after unallocated?
            {
               if (unallocSmart >= bsSmart)                  //- if at least ONE smart buffer
               {
                  unallocSmart -= (bsSmart / 2);             //- compensate for alignment issues
                  unallocSmart -= (unallocSmart % bsSmart);  //- clip to whole smart buffer size

                  dfsa->FsUnallocSmart += unallocSmart;      //- add this area to total
               }
               unallocSmart = 0;                // reset for next area to come
            }
            cClus++;
            uClus = i;                          // last in-use one sofar
         }
         else                                   // unallocated, Smart predict
         {
            unallocSmart += spe;                // add to size THIS area
         }
      }
      lClus += cClus;

      dfsProgressShow( (i * spe) + spc, 0, 0, NULL);

      TRACES(("lClus 2nd Loop i: %llu, b:%u = %llu\n", i, b, lClus));
      if (oneLineProgress == TRUE)
      {
         TxPrint( "%c", (char) ((cClus == 0) ? ' ' : mapchar[cClus * 8 / cpc])); // slow, acts as progress bar
      }
      else
      {
         ascii[a] = (char) ((cClus == 0) ? ' ' : mapchar[cClus * 8 / cpc]);
      }
      a++;
      if ((i && ((i%(acl*cpc)) == 0)) || (i >= size)) //- at end of a display line / begin next
      {
         if (oneLineProgress == FALSE)
         {
            TxPrint( "%s", ascii);              // display accumulated chars (fast)
         }
         perc  = (ULN64) (100 * lClus / (a * cpc));
         if (a == acl)                          // at the very end of a line
         {
            TxPrint("%s%s%3llu%%\n", CBC, CNN, perc);
         }
         else                                   // last line, when a partial one
         {
            TxPrint("%s%.*s%s%3llu%%\n", CBC, (int) (acl-a-1), BLIN, CNN, perc);
         }
         if (i < size)                          // not at end yet, display address for next line
         {
            dfsX10( CNZ, dfsJfsBlock2Lsn(i), CNZ, "");
            TxPrint("%s%s", CBC, CNC);
            a = 0;                              // reset for next, but keep value on last line
         }
         mClus += lClus;
         l++;
      }
   }
   if (unallocSmart != 0)                           //- unalloc area pending?
   {
      if (unallocSmart >= bsSmart)                  //- if at least ONE smart buffer
      {
         unallocSmart -= (bsSmart / 2);             //- compensate for alignment issues
         unallocSmart -= (unallocSmart % bsSmart);  //- clip to whole smart buffer size

         dfsa->FsUnallocSmart += unallocSmart;      //- add this area to total
      }
   }
   TxPrint("          %s%.*s%s\n", CBC, (int) a, BLIN, CNN);

   if (TxAbort())
   {
      dfsa->FsUnallocSmart = 0;                 // signal info NOT available!
      TxPrint( "\nAllocation check and display aborted! Used/Free info not available.\n");
   }
   else
   {
      if (size != 0)                            // avoid devide by 0
      {
         ULN64          used = mClus * spe;

         dfsa->FsUnallocated = size * spe - used; // free space (optimize hint)

         dfsSz64("Unallocated sects : ", dfsa->FsUnallocated, " ");
         dfsSz64("SmartUse ", dfsa->FsUnallocSmart, " ");
         TxPrint("=% 5.1lf%%\n", (double) (100.0 * ((double) dfsa->FsUnallocSmart) / (double) (size * spe)));

         dfsSz64("Allocated sectors : ", used, " ");
         dfsSz64("of total ", size * spe, " ");
         TxPrint("=% 5.1lf%%\n", (double) (100.0 * ((double) mClus) / (double) size));
      }
      dfsa->FsTruncPoint = dfsJfsBlock2Lsn( uClus +1); // first free (truncate)
      dfsa->FsLastInUse  = dfsa->FsTruncPoint -1;

      #if defined (JFS_RESIZING)
         dfsSz64( "Minimum vol. size : ", dfsa->FsTruncPoint, " (for  Resizing)\n");
      #endif
   }
   dfsProgressTerm();
   RETURN (NO_ERROR);
}                                               // end 'dfsJfsAllocMap'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// DFS JFS write-file to disk (SaveTo to file)
/*****************************************************************************/
static ULONG dfsJfsFileSaveAs
(
   ULN64               lsn,                     // IN    Inode LSN
   ULN64               d2,                      // IN    dummy
   char               *path,                    // IN    destination path
   void               *recp                     // INOUT recovery parameters
)
{
   ULONG               rc = NO_ERROR;
   BYTE                st;
   S_RECOVER_PARAM    *param = (S_RECOVER_PARAM *) recp;
   S_JINODE           *ino;
   DFSISPACE           isp;                     // allocation SPACE info
   TXLN                uniq;                    // unique filename prefix
   TXLN                fname;                   // filename component
   TXLN                safname;                 // resulting SaveAs name

   ENTER();
   TRARGS(("Inode LSN: 0x%llX, to path: '%s'\n", lsn, path));

   if ((rc = dfsJfsReadChkInode( lsn, &st, &ino)) == NO_ERROR)
   {
      memset( &isp, 0, sizeof( DFSISPACE));
      switch (st)
      {
         case ST_JINAG:                         // only data-type inodes
         case ST_JINOD:                         // have regular allocation
         case ST_JIDEL:
            rc = dfsJfsInodeData2Space( ino, 0, &isp, NULL, NULL);
         case ST_JIDIR:                         // Dirs result in empty SPACE
         case ST_JIDDI:
            if (rc == NO_ERROR)
            {
               #if defined (DEV32)
                  ULONG       ea2Size;
                  S_FEA2LIST *ea2List;

                  dfsJfsReadEaInode( ino, &ea2Size, &ea2List);

                  if (ea2List  != NULL)
                  {
                     TRACES(("Attach xatts ea2List: %p\n", ea2List));
                     isp.xattrs = ea2List;
                     isp.xasize = ea2Size;
                     isp.xatype = DFSXA_FEA2;
                  }
               #endif
               if ((param->newname == NULL) || (*param->newname == 0)) // no newname present
               {
                  dfsJfsResolveInodeName( lsn, ino, st, NULL, fname);
               }
               else                             // rename specified
               {
                  strcpy( fname, param->newname);
               }
               if (param->unique)               // force unique naming
               {                                // on PATH and FILE components
                  sprintf(uniq, "%8.8llX_%s", lsn, fname);
                  strcpy( fname, uniq);         // and move back to fname
               }
               if (param->recFname)
               {
                  strcpy( param->recFname, fname); // return full base-filename
               }

               isp.byteSize = ino->FileSize;
               rc = dfsSspaceFileSaveAs( &isp, (S_ISDIR( ino->Mode)), (ino->Links == 0),
                                                param->noAllocCheck, param->name83, path, fname, safname);
               if ((rc == NO_ERROR)       ||    // set original timestamps
                   (rc == DFS_CMD_WARNING) )    // even when allocation errors present
               {
                  TxSetFileTime( safname, (time_t *) &ino->Creation.hi,
                                          (time_t *) &ino->LastAccess.hi,
                                          (time_t *) &ino->LastModify.hi);
                  if (param->recFname)
                  {
                     strcpy( param->recFname, safname);
                  }
               }
               #if defined (DEV32)
                  TxFreeMem( ea2List);
               #endif
            }
            TxFreeMem( isp.space);               // free SPACE, if any
            break;

         default:
            rc = DFS_ST_MISMATCH;
            break;
      }
      TxFreeMem( ino);
   }
   RETURN(rc);
}                                               // end 'dfsJfsFileSaveAs'
/*---------------------------------------------------------------------------*/

