//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// DFS display and analysis, ISO specific functions and definitions
//
// Author: J. van Wijk
//
// JvW  10-01-2019 Initial version, derived from HFS
//

#ifndef    DFSAISO_H
   #define DFSAISO_H

#include <dfsiso.h>                             // ISO  structure defs

// ISO specific sector types
                                                // S   used on searching
                                                //  I  identification only
#define ST_ISOSVD  's'                          //     ISO  volume Descriptor
#define ST_ISODIR ('d')                         //     Directory information
#define ST_ISOELT ('e')                         //     El Torito header
#define ST_ISORRC ('c')                         //     Rock Ridge Cont. Area


#define ISO_SDC_MIN_SIZE                10000   // min nr of subdir sectors paths cached
#define ISO_MAX_DIR_DEPTH                  32   // recursion sanity maximum dir depth

typedef struct isosdcelem                       // subdirectory cache element lsn->path
{
   ULONG               DirBlockLsn;             // (sub) directory block number
   char               *Path;                    // related path, from ROOT to here
} ISOSDCELEM;                                   // end of struct "isosdcelem"

typedef struct isopathcache
{
   ULONG               sdcDirs;                 // DIR  count statistic, reporting only
   ULONG               sdcFiles;                // FILE count statistic, reporting only
   ULONG               sdcSize;                 // allocated entries in SD path cache
   ULONG               sdcUsed;                 // used entries in SD path cache
   ISOSDCELEM         *sdCache;                 // subdir block to path cache
} ISOPATHCACHE;                                 // end of struct "isopathcache"


typedef struct isodircache
{
   ULONG               SubdirLsn;               // One cached dir/file LSN for 1st sector
   ULONG               ParentLsn;               // Lsn for parent directory or 0 (root)
   ULONG               ParentEntry;             // DIR-entry index | SNINFO or 0 (root)
} ISODIRCACHE;                                  // end of struct "isodircache"


typedef struct dfsaiso                          // ISO analysis information
{
   ULONG               BlockCount;              // Number of logical blocks
   ULONG               BlockSize;               // Logical block size in bytes
   ISODIRCACHE         dirCache;                // Single Directory information cache
   ISOPATHCACHE        pathCache;               // All directories path-to-root cache
   ULONG               SuspSkipBytes;           // Bytes to skip on 1st SUSP DIR record
   ULONG               vdi;                     // active volume descriptor index
   BOOL                vdUnicode;               // VD uses Unicode Naming
   ULONG               vdCount;                 // number of valid VD's
   S_ISO_VOLUMEDESC   *vd;                      // ptr to active volume-descriptor
} DFSAISO;                                      // end of struct "dfsaiso"

extern DFSAISO        *iso;

// Initialize ISO filesystem analysis
ULONG dfsIsoInit
(
   char               *fs                       // forced filesystem type
);

// Initialize ISO filesystem analysis for Area (FS info in FDISK mode)
ULONG dfsIsoAreaInit
(
   void
);

// ISO  filesystem, test if (large) buffer contains any valid volume descriptor
BOOL dfsIsoFsIsPresent                          // RET   Buffer contains ISO FS
(
   BYTE               *buffer,                  // IN    buffer contents
   ULONG               bytes                    // IN    size of buffer
);

// ISO  filesystem, identify specified sector as a valid volumedesc
BOOL dfsIsoIsVolumeDesc                         // RET   sector is a valid vd
(
   BYTE               *sec                      // IN    sector contents
);

// ISO  filesystem, identify specified sector as a valid Directory block
// Uses single-directory cache to check if 'same' as last viewed one
BOOL dfsIsoIsDirectoryBlock                     // RET   sector is a valid DB
(
   BYTE               *sec,                     // IN    sector contents
   ULN64               lsn                      // IN    lsn for this sector
);

// ISO  filesystem, identify EL TORITO header sector
BOOL dfsIsoIsElToritoHdr                        // RET   sector is a valid ETH
(
   BYTE               *sec                      // IN    sector contents
);

// ISO  filesystem, identify SUSP / ROCK RIDGE continuation area (in a block)
BOOL dfsIsoIsRockRidgeCont                      // RET   sector is a valid RRC
(
   BYTE               *sec                      // IN    sector contents
);

// ISO filesystem, display sector constents as a volumedesc
void dfsIsoDisplayVolumeDesc
(
   BYTE               *sec                      // IN    sector contents
);

// ISO filesystem, identify specified sector
ULONG dfsIsoIdent
(
   ULN64               di,                      // IN    LSN for sector
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // OUT   sector type
   void               *data                     // IN    sector contents
);


#endif
