//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// ISO dump & display Analysis functions
//
// Author: J. van Wijk
//
// JvW  08-04-2021 Use rc CMD_WARNING on FileSaveAs alloc errors, considered OK
// JvW  10-01-2019 Initial version, derived from HFS
//
// Note: ISO is a dual LITTLE- plus BIG-ENDIAN oriented FS, so some field access
//       needs to use the specific access functions (using TxBE16/32/64 macros)
//       to get the value, and verify LE/BE values are identical  (flag if not)
//       Structures remain in their original endianess (big) in memory


#include <txlib.h>                              // TX library interface

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfsspace.h>                           // DFS file-space interface
#include <dfstable.h>                           // SLT utility functions

#include <dfsupart.h>                           // prereq. for UFDSK
#include <dfsufdsk.h>                           // FDISK generic Bootsector display

#include <dfsaiso.h>                            // ISO display & analysis
#include <dfsuiso.h>                            // ISO utility functions
#include <dfsliso.h>                            // ISO SLT functions

char dfsIsoSectorTypes[] =
{
   ST_ISOSVD,                                   //     ISO  super-block, volume descriptor
   ST_ISODIR,                                   //     Directory block
   ST_ISOELT,                                   //     El Torito header
   ST_ISORRC,                                   //     Rock Ridge Cont. Area
   0                                            //     string terminating ZERO
};


static DFSAISO    iso_anchor =                  // ISO specific global info
{
   ISO_STD_CDROM_BLOCKS,                        // 700 MiB CDROM
   ISO_STD_BLOCK_SIZE,                          // with 2048 byte blocks
   {
      0, 0, 0                                   // single dir Cache, empty
   },
   {
      0, 0, 0, 0, NULL                          // path Cache, empty
   },
   1,                                           // usually the primary VD
   0,                                           // SUSP skip bytes
   FALSE,                                       // default no Unicode
   0,                                           // number of VD's
   NULL,                                        // not read yet ...
};

       DFSAISO   *iso = &iso_anchor;


// ISO9660 signature string 'CD001'
static char sg_iso_sig[SG_ISO_SIG] = {SV_ISO_SIG};

static  char       *iso_txt[] =
{
   "",
   "Active filesystem : ISO, specific commands are:",
   "",
   " SUPER    [vdi]  = Display the filesystem VOLUME DESCRIPTOR sector",
   " VDI  [-P][vdi]  = Select Volume Descriptor / directory-tree to use",
   "",
   NULL
};


// Initialize specific Volume Descriptor in ISO filesystem
static ULONG dfsIsoVdiInit
(
   ULONG               vdi,                     // IN    VDI to initialize for
   BOOL                showGeo                  // IN    Show geometry being set
);

// Close ISO filesystem for analysis and free any resources
static ULONG dfsIsoClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
);

// Close ISO filesystem for Area analysis and free any resources
static ULONG dfsIsoAreaClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
);

// Sets the default VD-index to be used (1st 2/joliet, or 1st 1/primary)
// Build selection-list with selectable Volume-Descriptor entries for an ISO
static TXSELIST *dfsIsoVdiDefaultSelist         // RET   selection list or NULL
(
   BOOL                makeList                 // IN    make a SELIST too
);

// Interpret and execute specific ISO command;
static ULONG dfsIsoCommand
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *none,                    // IN    dummy
   void               *data                     // INOUT dummy
);

// ISO filesystem, supply sector-type description string
static ULONG dfsIsoStype
(
   ULN64               di,                      // IN    sector type
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // OUT   type description
   void               *data                     // INOUT dummy
);

// ISO filesystem, display sector-contents based on type
static ULONG dfsIsoSectorContents
(
   ULN64               psn,                     // IN    base psn for sector
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // IN    sector contents
);

// ISO filesystem, display special LSN info (Dirblock lsn + dir-record-index)
static ULONG dfsIsoDisplayLsnInfo
(
   ULN64               lsn,                     // IN    possible directory lsn
   ULN64               info,                    // IN    possible directory index
   char               *dc,                      // IN    dummy
   void               *data                     // IN    dummy
);

// Display Directory Block(s), iterating over all directory entries
static ULONG dfsIsoDirSpaceDisplay
(
   DFSISPACE          *dirIsp                   // IN    DIR allocation in ISPACE
);

// Display Directory Block directly from memory sector buffer
static ULONG dfsIsoDirectoryBlock
(
   ULN64               lsn,                     // IN    Dir LSN, for list
   BYTE               *block,                   // IN    Directory block
   ULONG               dirSize                  // IN    DIR size in bytes
);

// Display ISO  super-block
static ULONG dfsIsoVolumeDesc                   // RET   rc = 0 if type match
(
   BYTE               *sector,                  // IN    Superblock sector data
   BOOL                navigation               // IN    update nav values
);

// Display ISO  El Torito header (link to boot image)
static ULONG dfsIsoElToritoHdr                  // RET   rc = 0 if type match
(
   BYTE               *sector,                  // IN    sector data
   BOOL                navigation               // IN    update nav values
);


// Display ISO  SUSP / Rock Ridge Continuation Area in separate block
static ULONG dfsIsoRockRidgeCont                // RET   rc = 0 if type match
(
   BYTE               *sector,                  // IN    sector data
   BOOL                navigation               // IN    update nav values
);

// Make filesystem usage map dump on TxPrint output
static ULONG dfsIsoAllocMap
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *options,                 // IN    display options
   void               *data                     // OUT   dummy
);


// DFS ISO write-file to disk (SaveTo inode to file)
static ULONG dfsIsoFileSaveAs
(
   ULN64               dirLsn,                  // IN    DirBlock sectornumber
   ULN64               fileIdx,                 // IN    File dir-record index
   char               *path,                    // IN    destination path
   void               *recp                     // IN    recovery params (rename)
);

/*****************************************************************************/
// Initialize ISO filesystem analysis
/*****************************************************************************/
ULONG dfsIsoInit
(
   char               *fs                       // forced filesystem type
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   TRACES(("Sizeof     S_ISO_VD_BOOTREC   : %d\n", sizeof( S_ISO_VD_BOOTREC)));
   TRACES(("Sizeof     S_ISO_VD_PRIMSUP   : %d\n", sizeof( S_ISO_VD_PRIMSUP)));
   TRACES(("Sizeof     S_ISO_DIRENTRY     : %d\n", sizeof( S_ISO_DIRENTRY)));

   dfsa->FsCommand          = dfsIsoCommand;
   dfsa->FsClose            = dfsIsoClose;
   dfsa->FsIdentifySector   = dfsIsoIdent;
   dfsa->FsShowType         = dfsIsoStype;
   dfsa->FsDisplaySector    = dfsIsoSectorContents ;
   dfsa->FsFileInformation  = dfsIsoFileInfo;   // FS file alloc/path/size/date
   dfsa->FsGetAllocSpace    = dfsIsoGetAllocSpace;
   dfsa->FsWriteMetaSpace   = NULL;
   dfsa->FsMakeBrowseList   = dfsIsoMakeBrowseList;
   dfsa->FsFindPath         = dfsIsoFindPath;
   dfsa->FsLsnAllocated     = dfsIsoAllocated;
   dfsa->FsLsnSetAlloc      = NULL;
   dfsa->FsSltBuild         = dfsIsoSltBuild;
   dfsa->FsNpBuild          = NULL;             // dfsIsoNpBuild (too slow, and not needed)
   dfsa->FsDisplayError     = dfsIsoDispError;
   dfsa->FsDisplayLsnInfo   = dfsIsoDisplayLsnInfo;
   dfsa->FsDirIterator      = NULL;
   dfsa->FsDirFileSaveAs    = dfsIsoFileSaveAs;
   dfsa->FsTruncateSize     = NULL;
   dfsa->FsAllocDisplay     = dfsIsoAllocMap;
   dfsa->FsCl2Lsn           = NULL;
   dfsa->FsLsn2Cl           = NULL;
   dfsa->FsCmdHelp          = iso_txt;          // FS specific cmdhelp text

   dfsa->FsModeId           = DFS_FS_ISO;       // common for ISO

   dfsa->Fsi                = iso;
   dfsa->FsSectorTypes      = dfsIsoSectorTypes;


   dfstSetClusterSize( DFSTORE, 1);
   dfstSetSectorSize(  DFSTORE, ISO_STD_BLOCK_SIZE);

   //- To be refined, detect and briefly display MBR/GPT or other system area contents

   dfsa->FsModeSelist = dfsIsoVdiDefaultSelist( TRUE); // Determine default vdi, build selist

   rc = dfsIsoVdiInit( iso->vdi, TRUE);         // Init for resolved default vdi

   RETURN (rc);                                 // when needed
}                                               // end 'dfsIsoInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Initialize specific Volume Descriptor in ISO filesystem
/*****************************************************************************/
static ULONG dfsIsoVdiInit
(
   ULONG               vdi,                     // IN    VDI to initialize for
   BOOL                showGeo                  // IN    Show geometry being set
)
{
   ULONG               rc = NO_ERROR;

   ENTER();
   TRACES(("Init for vdi: %u\n", vdi));

   if (vdi < iso->vdCount)                      // vdi value is valid ?
   {
      TxFreeMem( iso->vd);                      // free existing VD, if any

      dfsInitList(0, "-f -P", "-d");            // empty snlist!

      iso->vdi = vdi;

      TxPrint( "\nVolume Descriptor : %u = Type %s\n", iso->vdi, (dfsa->FsModeSelist != NULL) ?
                                   dfsa->FsModeSelist->items[ iso->vdi]->text : "Unknown");

      dfsa->FsEntry = ISO_LSNVD( vdi);          // entry-point in FS (after bootrec)
      nav.down      = ISO_LSNVD( vdi);          // first sector to see after open

      if (((iso->vd = dfsIsoGetVolumeDesc( vdi)) != NULL)   &&
            (dfsIdentifySector( ISO_LSNVD( vdi), 0, (BYTE *) iso->vd) == ST_ISOSVD) &&
           (iso->vd->vdType > ISO_VDT_BOOT_RECORD))
      {
         iso->BlockCount = dfsIsoLE32( iso->vd->p.LogicalBlocks);
         iso->BlockSize  = dfsIsoLE32( iso->vd->p.LogBlockSize);
      }
      else
      {
         if (iso->vd->vdType > ISO_VDT_BOOT_RECORD)
         {
            TxPrint("Error reading ISO Volume Descriptor %u, defaults used!\n", vdi);
         }
         iso->BlockSize  = ISO_STD_BLOCK_SIZE;
         iso->BlockCount = dfsGetLogicalSize();
      }

      if (iso->BlockCount != 0)
      {
         if (iso->vd->vdType > ISO_VDT_BOOT_RECORD)
         {
            switch (dfstStoreType(  DFSTORE))
            {
               case DFST_IMZIMAGE:
               case DFST_RAWIMAGE:
                  //- Set explicit DFSee logical and system geometry when this is an imagefile
                  dfstDiskGeometry( DFSTORE, 0, (iso->BlockCount + (16 * 32) - 1) / (16 * 32),
                                    16,  32, ISO_STD_BLOCK_SIZE, TRUE, showGeo);
                  break;

               default:                         // no action needed
                  break;
            }
            dfstSetSectorSize(  DFSTORE, (USHORT) iso->BlockSize);
            TxPrint("Size each  block  : %10u   Bytes\n", iso->BlockSize);
            TxPrint("Blocks on volume  : %10u   ",        iso->BlockCount);
            dfsSz64("Size : ", ((ULN64) iso->BlockCount), "\n");
         }
         dfsIsoVolumeDesc( (BYTE *) iso->vd, TRUE);

         rc = dfsIsoBuildDirPathCache();        // build dirblock to path cache

         TxPrint("PathCache holding : %10u   Blocks     with %u files in %u directories\n",
                  iso->pathCache.sdcUsed, iso->pathCache.sdcFiles, iso->pathCache.sdcDirs);
      }
   }
   else
   {
      TxPrint( "\nInvalid vdi value : %u\n", vdi);
   }
   RETURN (rc);                                 // when needed
}                                               // end 'dfsIsoVdiInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Close ISO filesystem for analysis and free any resources
/*****************************************************************************/
static ULONG dfsIsoClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   dfsSlTableReset();                           // stop SLT thread and reset

   dfsIsoFreeDirPathCache();                    // free existing cache, and re-init

   TxFreeMem( iso->vd);

   dfsCloseFileSystem();
   RETURN (rc);
}                                               // end 'dfsIsoClose'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Initialize ISO filesystem analysis for Area (FS info in FDISK mode)
/*****************************************************************************/
ULONG dfsIsoAreaInit
(
   void
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   dfsa->FsAreaClose        = dfsIsoAreaClose;
   dfsa->FsAreaLsnAllocated = dfsIsoAllocated;

   //- no further Init needed, sectors are ALWAYS allocated

   RETURN (rc);                                 // when needed
}                                               // end 'dfsIsoAreaInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Close ISO filesystem for Area analysis and free any resources
/*****************************************************************************/
static ULONG dfsIsoAreaClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   dfsa->FsAreaClose        = NULL;
   dfsa->FsAreaLsnAllocated = NULL;

   RETURN (rc);
}                                               // end 'dfsIsoAreaClose'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Sets the default VD-index to be used (1st 2/joliet, or 1st 1/primary)
// Build selection-list with selectable Volume-Descriptor entries for an ISO
/*****************************************************************************/
static TXSELIST *dfsIsoVdiDefaultSelist         // RET   selection list or NULL
(
   BOOL                makeList                 // IN    make a SELIST too
)
{
   TXSELIST           *list  = NULL;            // total list
#if defined (USEWINDOWING)
   TXS_ITEM           *item;                    // single item
   ULONG               lsize;                   // list-size
   int                 i;
#endif
   BYTE               *block;                   // one sector/block buffer
   S_ISO_VOLUMEDESC   *vd;                      // buffer alias, vol-descriptor
   ULONG               vdIndex;
   int                 sysIdLen = 8;            // Minimum width of sysId column
   TXTT                sysId;
   ULONG               vdiType  = 0;            // Type of selected vd, sofar

   ENTER();

   if ((block = TxAlloc( 1, dfsGetSectorSize())) != NULL)
   {
      iso->vdi     = 0;
      iso->vdCount = 0;

      vd = (S_ISO_VOLUMEDESC *) block;          // set the alias

      for (vdIndex = 0; (vdIndex < ISO_VD_MAX) && (vd->vdType != ISO_VDT_TERMINATOR); vdIndex++)
      {
         if (dfsRead( ISO_LSNVD( vdIndex), 1, block) == NO_ERROR)
         {
            if ((dfsIsoIsVolumeDesc( block)) && (vd->vdType <= ISO_VDT_PARTITION))
            {
               iso->vdCount++;

               switch (vd->vdType)
               {
                  case ISO_VDT_BOOT_RECORD: TxCopy( sysId, (char *) vd->b.BtSysId,  ISO_SIDLEN + 1); break;
                  case ISO_VDT_PRIMARY:     TxCopy( sysId, (char *) vd->p.SystemId, ISO_SIDLEN + 1); break;
                  case ISO_VDT_SUPPLEMENTAL:
                     if ( (vd->p.EscapeSeq[0] == '%') && (vd->p.EscapeSeq[1] == '/') &&
                         ((vd->p.EscapeSeq[2] == '@') || (vd->p.EscapeSeq[2] == 'C')  || (vd->p.EscapeSeq[2] == 'E')))
                     {
                        TxUnic2Ascii( (USHORT *) vd->p.SystemId, sysId, ISO_SIDLEN / 2);
                     }
                     else                       // no 'Joliet' Escape sequence, assume plain ASCII
                     {
                        TxCopy( sysId, (char *) vd->p.SystemId, ISO_SIDLEN + 1);
                     }
                     break;

                  default:                  strcpy( sysId, "");                         break;
               }
               TxStrip( sysId, sysId, ' ', ' '); // strip leading/trailing spaces
               if (strlen( sysId) >= sysIdLen)
               {
                  sysIdLen = strlen( sysId) + 1; // widen sysId column
               }
               if ((vd->vdType > vdiType) &&    // This a 'better' default VD than the current one
                   (vd->vdType < ISO_VDT_PARTITION))
               {
                  iso->vdi     = vdIndex;
               }
            }
         }
         else
         {
            break;                              // abort
         }
      }

      TRACES(( "Volume Descriptors: %u   Default vdi: %hu\n", iso->vdCount, iso->vdi));

      #if defined (USEWINDOWING)

      if ((iso->vdCount != 0) && (makeList))    // some Volume Descriptors present
      {
         lsize = iso->vdCount;
         if (TxSelCreate( lsize, lsize, lsize, TXS_AS_NOSTATIC, FALSE, NULL, &list) == NO_ERROR)
         {
            char           *listdescr;          // list level description

            list->astatus = TXS_AS_NOSTATIC      | // all dynamic allocated
                            TXS_LST_DESC_PRESENT | // with list description
                            TXS_LST_DYN_CONTENTS;

            if ((listdescr  = TxAlloc( 1, TXMAXTM)) != NULL)
            {
               sprintf( listdescr, "Valid ISO9660 Volume Descriptors (CD/DVD/iso image)");

               list->miscInfo = listdescr;
            }
            list->selected = iso->vdi;          // as determined while counting ...

            for (i = 0; i < lsize; i++)         // all list entries
            {
               TRACES(("format list entry: %d for total of %u\n", i, lsize));
               if ((item  = TxAlloc( 1, sizeof(TXS_ITEM))) != NULL)
               {
                  list->count    = i +1;        // actual items in list
                  list->items[i] = item;        // attach item to list

                  item->helpid = TXWH_USE_OWNER_HELP; // from owner-menu-item

                  if (dfsRead( ISO_LSNVD( i), 1, block) == NO_ERROR) // read the VD
                  {
                     if (((item->text = TxAlloc( 1, TXMAXTM)) != NULL) &&
                         ((item->desc = TxAlloc( 1, TXMAXTM)) != NULL)  )
                     {
                        TXTT volId;
                        TXTT stdId;

                        item->value   = TXDID_MAX + i; // i is the vdIndex

                        switch (vd->vdType)
                        {
                           case ISO_VDT_BOOT_RECORD: // like EL TORITO
                              TxCopy( sysId, (char *) vd->b.BtSysId, ISO_SIDLEN + 1);
                              TxCopy( volId, (char *) vd->b.BootId,  ISO_SIDLEN + 1);
                              strcpy( stdId, "BOOT");

                              sprintf( item->desc, "Index: %u  Type: %u = ISO9660 Boot Record, like El Torito  ", i, vd->vdType);
                              break;

                           case ISO_VDT_PRIMARY: // Usually 8.3 uppercase
                              TxCopy( sysId, (char *) vd->p.SystemId, ISO_SIDLEN + 1);
                              TxCopy( volId, (char *) vd->p.VolumeId, ISO_SIDLEN + 1);
                              strcpy( stdId, "UPCASE");

                              sprintf( item->desc, "Index: %u  Type: %u = Primary, IS9660 (8.3) uppercase      ", i, vd->vdType);
                              break;

                           case ISO_VDT_SUPPLEMENTAL:
                              if ( (vd->p.EscapeSeq[0] == '%') && (vd->p.EscapeSeq[1] == '/') &&
                                  ((vd->p.EscapeSeq[2] == '@') || (vd->p.EscapeSeq[2] == 'C')  || (vd->p.EscapeSeq[2] == 'E')))
                              {
                                 TxUnic2Ascii( (USHORT *) vd->p.SystemId, sysId, ISO_SIDLEN / 2);
                                 TxUnic2Ascii( (USHORT *) vd->p.VolumeId, volId, ISO_SIDLEN / 2);

                                 strcpy( stdId, "Joliet");
                                 sprintf( item->desc, "Index: %u  Type: %u = Supplemental, Joliet Long File Names", i, vd->vdType);
                              }
                              else              // no 'Joliet' Escape sequence, assume plain ASCII
                              {
                                 TxCopy( sysId, (char *) vd->p.SystemId, ISO_SIDLEN + 1);
                                 TxCopy( volId, (char *) vd->p.VolumeId, ISO_SIDLEN + 1);
                                 strcpy( stdId, "Suppl.");
                                 sprintf( item->desc, "Index: %u  Type: %u = Supplemental, unknown std, no Joliet", i, vd->vdType);
                              }
                              break;

                           case ISO_VDT_TERMINATOR: // should not occur here, limited by vdCount/lsize
                              break;

                           case ISO_VDT_PARTITION:
                           default:
                              strcpy( sysId, (vd->vdType == ISO_VDT_PARTITION) ? "Partition" : "Unknown");
                              strcpy( volId, "Unsupported");
                              strcpy( stdId, "Ignore");
                              sprintf( item->desc, "Index: %u  Type: %u = %s,  unsupported, ignored  ", i, vd->vdType,
                                             (vd->vdType == ISO_VDT_PARTITION) ? "Partition" : "Unknown");
                              item->flags = TXSF_DISABLED;
                              break;
                        }
                        TxStrip( volId, volId, ' ', ' '); // strip trailing spaces from volume ID
                        sprintf( item->text, "%u   %-6.6s %-*.*s %s", i,  stdId, sysIdLen, sysIdLen, sysId, volId);

                        //- get unique SelectChar starting at text[1] ...
                        item->index = TxSelGetCharSelect( list, i, 1);

                        TRACES(("text: %d '%s'\n", strlen(item->text), item->text));
                        TRACES(("desc: %d '%s'\n", strlen(item->desc), item->desc));
                     }
                  }
               }
            }
         }
      }
      else
      {
         list = TxSelEmptyList( "- No Volume descriptors present in ISO -",
               "No Volume descriptors found, ISO is empty, or this is not an ISO9660 filesystem at all", FALSE);

      }
      #endif

      TxFreeMem( block);
   }
   RETURN( list);
}                                               // end 'dfsIsoVdiDefaultSelist'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Interpret and execute specific ISO command;
/*****************************************************************************/
static ULONG dfsIsoCommand
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *none,                    // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;
   ULN64               sn = 0;                  // sector number input
   TXLN                s1;                      // large text buffer 1
   TXLN                dc;                      // DFS command
   int                 cc;                      // command string count
   char               *c0, *c1, *c2;            // parsed command parts
   char               *pp;                      // parameter pointer
   ULONG               index = 0;               // file index

   ENTER();

   pp = TxaGetArgString( TXA_CUR, 1, 0, TXMAXLN, dc); // dc => cmd from arg 1
   cc = TxaArgCount( );                         // number of parameters
   c0 = TxaArgValue(0);
   c1 = TxaArgValue(1);
   c2 = TxaArgValue(2);

   sn = nav.this;                               // default at current sector

   if ((strcmp(c0, "/?") == 0) ||
       (strcmp(c0,  "?") == 0)  )
   {
      TxShowTxt( iso_txt);
   }
   else if ((strcasecmp(c0, "filefind") == 0)) // to be refined, build NP cache for now
   {
      dfsSltPrepareNameLookup();
      TxCancelAbort();                          // might have aborted
   }
   else if (strcasecmp(c0, "super"    ) == 0)
   {
      sprintf( dc, "0x0%x", ISO_LSNVD( iso->vdi)); // most likely location
      TxaReParseCommand( dc);
      rc = DFS_PENDING;                         // display the sector
   }
   else if (strcasecmp(c0, "vdi"      ) == 0)
   {
      strcpy( s1, "");                          // init to no value
      if (!TxaOption('?'))                      // No explicit help request
      {
         ULONG  listcode  = TXDID_OK;

         #if defined (USEWINDOWING)
            TXRECT where  = {18,0,0,0};         // fixed position

            if ((dfsa->FsModeSelist != NULL) && (TxaOption('P')))
            {
               dfsa->FsModeSelist->selected = iso->vdi; // default to CURRENT vdi

               listcode = txwListBox( TXHWND_DESKTOP, TXHWND_DESKTOP, &where,
                        " Select Volume-Descriptor / DIR-tree to use ", "",
                          5421, TXLB_MOVEABLE,
                          cMenuTextStand, cMenuBorder_top, // same color as menu
                          dfsa->FsModeSelist);

               if (listcode >= TXDID_MAX)       // return list selection
               {                                // plus current item base value
                  sprintf( s1, "%u", listcode - TXDID_MAX); // calculate vdi
               }
            }
         #endif                                 // USEWINDOWING

         if (listcode != TXDID_CANCEL)          // not escaped ...
         {
            if ((cc > 1) && (strlen(s1) == 0))  // No selection from list, use param
            {
               strcpy(  s1, c1);
            }
            if (strlen( s1))                    // We have some vdi specification
            {
               index = atol( s1);

               rc = dfsIsoVdiInit( index, FALSE); // activate it, when valid
            }
         }
      }
      else
      {
         TxPrint( "Select a specific Volume Descriptor and associated directory-tree\n\n");
         TxPrint( " Usage: %s  [index]\n\n"
                  "  index = Volume description index, 0 .. %u\n"
                  "  -P    = Select vdi from list of available ones\n", c0, iso->vdCount - 1);
      }
   }
   else
   {
      rc = DFS_CMD_UNKNOWN;                     // cmd not recognized
   }
   RETURN (rc);
}                                               // end 'dfsIsoCommand'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// ISO  filesystem, test if (large) buffer contains any valid volume descriptor
/*****************************************************************************/
BOOL dfsIsoFsIsPresent                          // RET   Buffer contains ISO FS
(
   BYTE               *buffer,                  // IN    buffer contents
   ULONG               bytes                    // IN    size of buffer
)
{
   BOOL                rc = FALSE;
   ULONG               blocks;
   BYTE               *sec;

   ENTER();

   for (sec = buffer,               blocks = (bytes / ISO_STD_BLOCK_SIZE);
        (rc == FALSE)  &&          (blocks > 0);
        sec += ISO_STD_BLOCK_SIZE,  blocks--)
   {
      rc = (dfsIsoIsVolumeDesc( sec));
   }
   BRETURN (rc);
}                                               // end 'dfsIsoFsIsPresent'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// ISO  filesystem, identify specified sector as a valid volumedesc
/*****************************************************************************/
BOOL dfsIsoIsVolumeDesc                         // RET   sector is a valid sb
(
   BYTE               *sec                      // IN    sector contents
)
{
   BOOL                rc = FALSE;

   ENTER();

   if (!memcmp(((S_ISO_VOLUMEDESC *)sec)->Signature,sg_iso_sig,SG_ISO_SIG))
   {
      rc = TRUE;
   }
   BRETURN (rc);
}                                               // end 'dfsIsoIsVolumeDesc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// ISO  filesystem, identify specified sector as a valid Directory block
// Uses single-directory cache to check if 'same' as last viewed one
/*****************************************************************************/
BOOL dfsIsoIsDirectoryBlock                     // RET   sector is a valid DB
(
   BYTE               *sec,                     // IN    sector contents
   ULN64               lsn                      // IN    lsn for this sector
)
{
   BOOL                rc = FALSE;

   ENTER();

   if      (((ULN64) iso->dirCache.SubdirLsn) == lsn)
   {
      TRACES(("Sector is FIRST, cached subdir LSN!\n"));
      rc = TRUE;                                // sector is the first sector of cached DIR
   }
   else                                         // check for . and .. entries in sector
   {
      BYTE               *at = sec;
      S_ISO_DIRENTRY     *de = (S_ISO_DIRENTRY * ) at;

      //- Fast check for presence or . and .. entries
      if ((de->NameLength == 1) && (de->FileName[0] == 0))
      {
         at += de->RecordLength;                // to second entry
         de = (S_ISO_DIRENTRY * ) at;
         if ((de->NameLength == 1) && (de->FileName[0] == 1))
         {
            TRACES(("DOT and DOTDOT Entry found, dirblock!\n"));
            rc = TRUE;                          // . and .. entries are present
         }
      }
      else                                      // check all entries for LE/BE consistency
      {
         ULONG blockRemain = iso->BlockSize;    // bytes remaining in block
         ULONG entry       = 0;                 // first entry in this block

         TRACES(("Check all entries %u LE/BE consistency ...\n"));

         rc = TRUE;                             // OK until proven otherwise
         do
         {
            de = (S_ISO_DIRENTRY * ) at;

            if (de->RecordLength > 0)       // check this DIR record for consistency
            {
               if ((dfsIsoLE32( de->Location) != dfsIsoBE32( de->Location)) ||
                   (dfsIsoLE32( de->FileSize) != dfsIsoBE32( de->FileSize))  )
               {
                  TRACES(("Entry %u LE/BE inconsistency, not a dirblock!\n"));
                  rc = FALSE;                        // Little-Endian / Big-Endian mismatch
               }
               entry++;

               at          += de->RecordLength; // advance to next
               blockRemain -= de->RecordLength; // and keep count ...
            }
         } while ((blockRemain > 0) && (de->RecordLength > 0) && (rc == TRUE));
      }
   }
   BRETURN (rc);
}                                               // end 'dfsIsoIsDirectoryBlock'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// ISO  filesystem, identify EL TORITO header sector
/*****************************************************************************/
BOOL dfsIsoIsElToritoHdr                        // RET   sector is a valid ETH
(
   BYTE               *sec                      // IN    sector contents
)
{
   BOOL                rc = FALSE;
   S_ISO_EL_TORITO    *et = (S_ISO_EL_TORITO * ) sec;

   ENTER();

   if ((et->hdrId == ELT_HEADER_ID) && (et->Signature == ELT_SIGNATURE) && (et->resZero2 == 0))
   {
      rc = TRUE;                                // . and .. entries are present
   }
   BRETURN (rc);
}                                               // end 'dfsIsoIsElToritoHdr'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// ISO  filesystem, identify SUSP / ROCK RIDGE continuation area (in a block)
/*****************************************************************************/
BOOL dfsIsoIsRockRidgeCont                      // RET   sector is a valid RRC
(
   BYTE               *sec                      // IN    sector contents
)
{
   BOOL                rc = FALSE;
   SRR_CONT_RECORD    *er = (SRR_CONT_RECORD * ) sec;

   ENTER();

   if ((er->Signature == RRP_CONTA_DATA_SIG)    && (er->Version == 1)  &&
       (er->Length != 0) && (er->IdLength != 0) && (er->ExtVersion != 0))
   {
      rc = TRUE;                                // . and .. entries are present
   }
   BRETURN (rc);
}                                               // end 'dfsIsoIsRockRidgeCont'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// ISO filesystem, display sector constents as a volumedesc
/*****************************************************************************/
void dfsIsoDisplayVolumeDesc
(
   BYTE               *sec                      // IN    sector contents
)
{
   dfsIsoVolumeDesc( sec, FALSE);
}                                               // end 'dfsIsoDisplayVolumeDesc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// ISO filesystem, identify specified sector
/*****************************************************************************/
ULONG dfsIsoIdent
(
   ULN64               lsn,                     // IN    LSN for sector
   ULN64               d2,                      // IN    dummy
   char               *st,                      // OUT   sector type
   void               *sec                      // IN    sector contents
)
{
   ULONG               dr = NO_ERROR;
   BYTE                rc = ST_UDATA;

   ENTER();

   if      (dfsIsoIsVolumeDesc(     (BYTE *) sec))          rc = ST_ISOSVD;
   else if (dfsIsoIsElToritoHdr(    (BYTE *) sec))          rc = ST_ISOELT;
   else if (dfsIsoIsRockRidgeCont(  (BYTE *) sec))          rc = ST_ISORRC;
   else if (dfsIsoIsDirectoryBlock( (BYTE *) sec, lsn))     rc = ST_ISODIR;

   switch (rc)
   {
      case ST_UDATA:
         dr = DFS_PENDING;
         break;

      default:
         break;
   }
   *st = rc;
   RETURN (dr);
}                                               // end 'dfsIsoIdent'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// ISO filesystem, supply sector-type description string
/*****************************************************************************/
static ULONG dfsIsoStype
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *st,                      // IN    sector type
   void               *data                     // OUT   type description
)
{
   ULONG               dr  = NO_ERROR;
   char               *buf = (char *) data;
   BYTE                tp  = st[0];

   switch (tp)                                  // searchable types
   {
      case ST_ISOSVD: sprintf(buf,"ISO Volume Descr."); break;
      case ST_ISODIR: sprintf(buf,"ISO Directory blk"); break;
      case ST_ISOELT: sprintf(buf,"ISO El Torito hdr"); break;
      case ST_ISORRC: sprintf(buf,"ISO RockRidge blk"); break;
      default:
         switch (tp | ST__INFO)                 // non-searchable ones
         {
            default:       dr = DFS_PENDING;
               break;
         }
         break;
   }
   return (dr);
}                                               // end 'dfsIsoStype'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// ISO filesystem, display sector-contents based on type
/*****************************************************************************/
static ULONG dfsIsoSectorContents
(
   ULN64               psn,                     // IN    base psn for sector
   ULN64               d2,                      // IN    dummy
   char               *type,                    // IN    type of sector
   void               *data                     // IN    sector contents
)
{
   ULONG               rc   = NO_ERROR;
   BYTE               *sect = (BYTE *) data;
   BYTE                st   = (BYTE) *type;
   ULN64               lsn = dfstPsn2LSN( DFSTORE, psn);

   ENTER();

   switch (st)
   {
      case ST_ISOSVD:
         rc = dfsIsoVolumeDesc( sect, TRUE);
         break;

      case ST_ISODIR:
         if (lsn == iso->dirCache.SubdirLsn)    // read from disk using parent-dir lsn+entry info
         {
            DFSISPACE           dirIsp;         // Directory in ISPACE

            if ((rc = dfsIsoGetAllocSpace( iso->dirCache.ParentLsn, iso->dirCache.ParentEntry, NULL, &dirIsp)) == NO_ERROR)
            {
               rc = dfsIsoDirSpaceDisplay( &dirIsp); // display the directory
               TxFreeMem( dirIsp.space);        // free space memory
            }
         }
         else                                   // display directly from sector contents
         {
            rc = dfsIsoDirectoryBlock( lsn, sect, iso->BlockSize);
         }
         break;

      case ST_ISOELT:
         rc = dfsIsoElToritoHdr( sect, TRUE);
         break;

      case ST_ISORRC:
         rc = dfsIsoRockRidgeCont( sect, TRUE);
         break;

      default:
         rc = DFS_PENDING;
         break;
   }
   RETURN (rc);
}                                               // end 'dfsIsoSectorContents '
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// ISO filesystem, display special LSN info (Dirblock lsn + dir-record-index)
/*****************************************************************************/
static ULONG dfsIsoDisplayLsnInfo
(
   ULN64               lsn,                     // IN    possible directory lsn
   ULN64               info,                    // IN    possible directory index
   char               *dc,                      // IN    dummy
   void               *data                     // IN    dummy
)
{
   ULONG               rc = DFS_PENDING;
   TXLN                path;

   ENTER();
   TRACES(("lsn: 0x%llx  info: 0x%x\n", lsn, info));

   if (info & DFSSNINFO)                        // extra info present
   {
      if (dfsIsoLsnInfo2Path( lsn, info, path) == NO_ERROR)
      {
         TxPrint("\nFound path to root: '%s%s%s' %s\n", CBY, path, CNN, (strlen(path) ? "" : "(root itself)"));
      }
      nav.xtra = lsn;                           // keep reference around
      rc = dfsIsoShowDirectoryEntry( lsn, DFSSNIGET( info));
      if (rc == NO_ERROR)
      {
         DFSISPACE           dirIsp;            // Directory in ISPACE

         if ((rc = dfsIsoGetAllocSpace( lsn, info, NULL, &dirIsp)) == NO_ERROR)
         {
            dfsSspaceDisplay( SD_SUMMARY | SD_TOPLINE | SD_NAVDOWN, dirIsp.chunks, dirIsp.space);
            TxFreeMem( dirIsp.space);           // free space memory
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsIsoDisplayLsnInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display Directory Block(s), iterating over all directory entries
/*****************************************************************************/
static ULONG dfsIsoDirSpaceDisplay
(
   DFSISPACE          *dirIsp                   // IN    DIR allocation in ISPACE
)
{
   ULONG               rc = NO_ERROR;
   ULONG               dirSize;                 // total DIR size in SPACE
   ULN64               dirsec = 0;              // current relative dirblock LSN
   BYTE               *block;
   BYTE               *at;                      // current position
   S_ISO_DIRENTRY     *curDir;                  // current as directory struct
   ULONG               dirSizeDone;             // DIR block size handled
   ULONG               blockRemain;             // remaining size in this block
   ULONG               entry;                   // index in fat directory
   ULN64               data;                    // 1st data sector
   USHORT              ei = 0;                  // entry index
   int                 cl = TXaNWnZ;            // LSN color
   int                 ac;                      // alternate color fg+bg
   TXLN                path;

   ENTER();

   if ((block = TxAlloc( 1, iso->BlockSize)) != NULL)
   {
      if (dfsIsoLsnInfo2Path( dirIsp->space[0].start, DFSSNINFO, path) == NO_ERROR)
      {
         TxPrint("\nPath for 1st entry: '%s%s%s' %s\n", CBY, path, CNN, (strlen(path) ? "" : "(root itself)"));
      }
      TxPrint( "\n");
      dfsIsoDirHeader( "Nr       Block  ", 0);  // display header
      if (!TxaOptUnSet('l'))                    // create a new list for DIR
      {
         dfsInitList(0, "-f -P", "-d");         // optimal for menu file-recovery
      }

      dirSize     = dfsSspaceSectors( TRUE, dirIsp->chunks, dirIsp->space) * iso->BlockSize;
      dirSizeDone = 0;
      do
      {
         if ((rc = dfsSspaceReadFilePart( dirIsp->chunks, dirIsp->space, dirsec, 1, block)) == NO_ERROR)
         {
            blockRemain = iso->BlockSize;       // bytes remaining in block
            at          = block;                // set to start of this block
            entry       = 0;                    // first entry in this block

            do
            {
               curDir = (S_ISO_DIRENTRY *) at;

               if (curDir->RecordLength > 0)    // display this DIR entry record
               {
                  data = dfsIsoLE32( curDir->Location);

                  if (ei % 2)
                  {
                     ac = TXaBWnC;
                  }
                  else
                  {
                     ac = TXaNWnZ;
                  }
                  switch (ei)
                  {
                     case 2:  cl = TXaBGnZ; nav.down = data; break;
                     case 1:  cl = TXaBCnC; nav.up   = data; break;
                     default: cl = ac;                       break;
                  }
                  TxPrint("%s.%05.5u", ansi[ac], ei);
                  dfsX10( " ", data,   ansi[cl], ansi[ac]);

                  dfsIsoShowDirEntry( curDir, Ccbg(ac));
                  TxPrint("%s\n", CGE);

                  if (!TxaOptUnSet('l'))        // add real LSN+entry to sectorlist
                  {
                     dfsAddSI2List( dfsSspaceRsn2Lsn( dirIsp->chunks, dirIsp->space, dirsec), entry);
                  }
                  ei++;
                  entry++;

                  at          += curDir->RecordLength; // advance to next
                  blockRemain -= curDir->RecordLength; // and keep count ...
               }
            } while ((blockRemain > 0) && (curDir->RecordLength > 0) && (rc == NO_ERROR) && (!TxAbort()));

            dirSizeDone += iso->BlockSize;      // one more DIR block handled
            dirsec++;                           // to next relative sector in ISPACE
         }
      } while ((dirSizeDone < dirSize) && (rc == NO_ERROR) && (!TxAbort()));
      TxPrint( "%s", CNN);

      if (ei > 12)
      {
         dfsIsoDirHeader( "Nr       Block  ", ei); // display footer
      }
      TxFreeMem( block);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN( rc);
}                                               // end 'dfsIsoDirSpaceDisplay'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display Directory Block directly from memory sector buffer
/*****************************************************************************/
static ULONG dfsIsoDirectoryBlock
(
   ULN64               lsn,                     // IN    Dir LSN, for list
   BYTE               *block,                   // IN    Directory block
   ULONG               dirSize                  // IN    DIR size in bytes
)
{
   ULONG               rc = NO_ERROR;
   BYTE               *at;                      // current position
   S_ISO_DIRENTRY     *curDir;                  // current as directory struct
   ULONG               blockRemain;             // remaining size in this block
   ULONG               entry;                   // index in fat directory
   ULN64               data;                    // 1st data sector
   USHORT              ei = 0;                  // entry index
   int                 cl = TXaNWnZ;            // LSN color
   int                 ac;                      // alternate color fg+bg
   TXLN                path;

   ENTER();
   TRACES(("DirSize: %u\n", dirSize));

   if (dfsIsoLsnInfo2Path( lsn, DFSSNINFO, path) == NO_ERROR)
   {
      TxPrint("\nPath for 1st entry: '%s%s%s' %s\n", CBY, path, CNN, (strlen(path) ? "" : "(root itself)"));
   }
   TxPrint( "\n");
   dfsIsoDirHeader( "Nr       Block  ", 0);     // display header
   if (!TxaOptUnSet('l'))                       // create a new list for DIR
   {
      dfsInitList(0, "-f -P", "-d");            // optimal for menu file-recovery
   }

   blockRemain = iso->BlockSize;                // bytes remaining in block
   at          = block;                         // set to start of this block
   entry       = 0;                             // first entry in this block

   do
   {
      curDir = (S_ISO_DIRENTRY *) at;

      if (curDir->RecordLength > 0)             // display this DIR entry record
      {
         data = dfsIsoLE32( curDir->Location);

         if (ei % 2)
         {
            ac = TXaBWnC;
         }
         else
         {
            ac = TXaNWnZ;
         }
         switch (ei)
         {
            case 2:  cl = TXaBGnZ; nav.down = data; break;
            case 1:  cl = TXaBCnC; nav.up   = data; break;
            default: cl = ac;                       break;
         }
         TxPrint("%s.%05.5u", ansi[ac], ei);
         dfsX10( " ", data,   ansi[cl], ansi[ac]);

         dfsIsoShowDirEntry( curDir, Ccbg(ac));
         TxPrint("%s\n", CGE);

         if (!TxaOptUnSet('l'))
         {
            dfsAddSI2List( lsn, entry);
         }
         ei++;
         entry++;

         at          += curDir->RecordLength;   // advance to next
         blockRemain -= curDir->RecordLength;   // and keep count ...
      }
   } while ((blockRemain > 0) && (curDir->RecordLength > 0) && (rc == NO_ERROR) && (!TxAbort()));

   TxPrint( "%s", CNN);

   if (ei > 12)
   {
      dfsIsoDirHeader( "Nr       Block  ", ei); // display footer
   }
   RETURN( rc);
}                                               // end 'dfsIsoDirectoryBlock'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display ISO  Volume Descriptor (aka superblock)
/*****************************************************************************/
static ULONG dfsIsoVolumeDesc                   // RET   rc = 0 if type match
(
   BYTE               *sector,                  // IN    sector data
   BOOL                navigation               // IN    update nav values
)
{
   ULONG               rc    = 0;               // rc, sector match
   S_ISO_VOLUMEDESC   *vd    = (S_ISO_VOLUMEDESC *) sector;
   TXTT                sysId;
   TXTT                volId;
   TXLN                text;
   time_t              tt;

   ENTER();

   TxPrint("Volume Descr Type : 0x%2.2hhx         Version: 0x%2.2hhx         Signature: %5.5s\n",
                   vd->vdType,              vd->vdVersion,               vd->Signature);

   if (vd->vdType != ISO_VDT_BOOT_RECORD)
   {
      TxPrint("Volume Flag Value : 0x%2.2hhx         %s Esc-seqences conforming to ISO-2375 (bit 0)\n",
               vd->p.VolumeFlags, (vd->p.VolumeFlags & ISO_VD_FLAG_ESCSEQ) ? "NOT all" : "All");

      TxPrint("Esc-seq (charset) : %-12.12s %s\n",  vd->p.EscapeSeq,
                                                   (vd->p.EscapeSeq[0] == 0) ? "UPCASE only" :
                  ( (vd->p.EscapeSeq[0] == '%') && (vd->p.EscapeSeq[1] == '/') &&
                   ((vd->p.EscapeSeq[2] == '@') || (vd->p.EscapeSeq[2] == 'C')  || (vd->p.EscapeSeq[2] == 'E'))) ?
                                                   "Joliet, mixed case, long filenames"      :
                                                   "Custom escape sequence, no Joliet");

      TxPrint("Volume Set Size   : 0x%4.4hx       ", dfsIsoLE16( vd->p.VolSetSize));
      TxPrint( "Volume Seq Number: 0x%4.4hx\n",      dfsIsoLE16( vd->p.VolSequenceNr));

      TxPrint("Size each  block  : 0x%4.4hx     = %10hu Bytes\n", dfsIsoLE16( vd->p.LogBlockSize),  dfsIsoLE16( vd->p.LogBlockSize));
      TxPrint("Blocks on volume  : 0x%8.8x = %10u ",              dfsIsoLE32( vd->p.LogicalBlocks), dfsIsoLE32( vd->p.LogicalBlocks));
      dfsSz64("Size  : ", ((ULN64) dfsIsoLE32( vd->p.LogicalBlocks)), "\n");

      if (navigation)
      {
         iso->vdUnicode = FALSE;
      }
   }

   switch (vd->vdType)
   {
      case ISO_VDT_BOOT_RECORD:                 // like EL TORITO
         TxCopy( sysId, (char *) vd->b.BtSysId, ISO_SIDLEN + 1);
         TxCopy( volId, (char *) vd->b.BootId,  ISO_SIDLEN + 1);
         break;

      case ISO_VDT_PRIMARY:                     // Usually 8.3 uppercase
         TxCopy( sysId, (char *) vd->p.SystemId, ISO_SIDLEN + 1);
         TxCopy( volId, (char *) vd->p.VolumeId, ISO_SIDLEN + 1);
         break;

      case ISO_VDT_SUPPLEMENTAL:
         if ( (vd->p.EscapeSeq[0] == '%') && (vd->p.EscapeSeq[1] == '/') &&
             ((vd->p.EscapeSeq[2] == '@') || (vd->p.EscapeSeq[2] == 'C')  || (vd->p.EscapeSeq[2] == 'E')))
         {
            TxUnic2Ascii( (USHORT *) vd->p.SystemId, sysId, ISO_SIDLEN / 2);
            TxUnic2Ascii( (USHORT *) vd->p.VolumeId, volId, ISO_SIDLEN / 2);
            if (navigation)
            {
               iso->vdUnicode = TRUE;
            }
         }
         else                                   // no 'Joliet' Escape sequence, assume plain ASCII
         {
            TxCopy( sysId, (char *) vd->p.SystemId, ISO_SIDLEN + 1);
            TxCopy( volId, (char *) vd->p.VolumeId, ISO_SIDLEN + 1);
         }
         break;

      case ISO_VDT_TERMINATOR:
         strcpy( sysId, "Last-VD");
         strcpy( volId, "Terminator");
         break;

      case ISO_VDT_PARTITION:
      default:
         strcpy( sysId, (vd->vdType == ISO_VDT_PARTITION) ? "Partition" : "Unknown");
         strcpy( volId, "Unsupported");
         break;
   }

   if (vd->vdType != ISO_VDT_BOOT_RECORD)
   {
      TxPrint("PathTable, LE  at : 0x%8.8x   %10u Bytes   Table Big Endian: 0x%8.8x\n",
                                vd->p.LePathTableLba, dfsIsoLE32( vd->p.PathTableSize),
                        TxBE32( vd->p.BePathTableLba));

      TxUnicAscii2Ascii( (USHORT *) vd->p.VolumeSet, text, ISO_LIDLEN); // auto detect Ascii/Unicode
      TxStrip( text, text, ' ', ' ');
      if (strlen( text))                        // identifier present
      {
         TxPrint("Volume Set   name : %s\n", text);
      }
      TxUnicAscii2Ascii( (USHORT *) vd->p.Publisher, text, ISO_LIDLEN); // auto detect Ascii/Unicode
      TxStrip( text, text, ' ', ' ');
      if (strlen( text))                        // identifier present
      {
         TxPrint("Publisher    name : %s\n", text);
      }
      TxUnicAscii2Ascii( (USHORT *) vd->p.Preparer,  text, ISO_LIDLEN); // auto detect Ascii/Unicode
      TxStrip( text, text, ' ', ' ');
      if (strlen( text))                        // identifier present
      {
         TxPrint("Preparer     name : %s\n", text);
      }
      TxUnicAscii2Ascii( (USHORT *) vd->p.Applicat,  text, ISO_LIDLEN); // auto detect Ascii/Unicode
      TxStrip( text, text, ' ', ' ');
      if (strlen( text))                        // identifier present
      {
         TxPrint("Application  name : %s\n", text);
      }
   }

   TxPrint("System Identifier : %s%s%s\n", CBC, sysId, CNN);
   TxPrint("Volume Identifier : %s%s%s\n", CBY, volId, CNN);

   if (vd->vdType != ISO_VDT_BOOT_RECORD)
   {
      TxPrint("Creation     Time : %s  ", dfsIsoVtime2str( &vd->p.VdtCreation,     text));
      TxPrint(      "Modify  Time: %s\n", dfsIsoVtime2str( &vd->p.VdtModification, text));
      TxPrint("Expiration   Time : %s  ", dfsIsoVtime2str( &vd->p.VdtExpiration,   text));
      TxPrint(      "Effect. Time: %s\n", dfsIsoVtime2str( &vd->p.VdtEffective,    text));

      tt = dfsIsoFileTime2t( &vd->p.RootDirEntry.FileDateTime);
      strftime( text, TXMAXLN, "%Y-%m-%d %H:%M:%S", localtime( &tt));
      TxPrint("ROOT Direct. Time : %s\n", text);

      strcpy( text, "");
      dfstrUlDot13( text, "", dfsIsoLE32( vd->p.RootDirEntry.FileSize), "");
      TxPrint("ROOT Directory at : 0x%s%8.8x%s %s Bytes  VolumeSequenceNr: 0x%4.4x\n",
            CBG, dfsIsoLE32( vd->p.RootDirEntry.Location), CNN, text,
                 dfsIsoLE16( vd->p.RootDirEntry.VolSequenceNr));
      nav.down = dfsIsoLE32( vd->p.RootDirEntry.Location);
   }
   else if (strstr( sysId, "EL TORITO") != NULL)
   {
      TxPrint("El Torito Catalog : 0x%s%8.8x%s\n", CBG, vd->b.ElToritoBcBlock, CNN);
      nav.down = vd->b.ElToritoBcBlock;
   }
   RETURN (rc);
}                                               // end 'dfsIsoVolumeDesc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display ISO  El Torito header (link to boot image)
/*****************************************************************************/
static ULONG dfsIsoElToritoHdr                  // RET   rc = 0 if type match
(
   BYTE               *sector,                  // IN    sector data
   BOOL                navigation               // IN    update nav values
)
{
   ULONG               rc    = 0;               // rc, sector match
   S_ISO_EL_TORITO    *et    = (S_ISO_EL_TORITO *) sector;
   TXLN                text;

   ENTER();

   switch (et->archId)
   {
      case ELT_ARCH_80x86     : strcpy( text, "Intel 80x86");             break;
      case ELT_ARCH_POWER     : strcpy( text, "PowerPC");                 break;
      case ELT_ARCH_MACOS     : strcpy( text, "MAC 68K/PPC");             break;
      case ELT_ARCH_UEFI      : strcpy( text, "UEFI");                    break;
         default:               strcpy( text, "Unknown, invalid!");       break;
   }
   TxPrint("El Torito Arch ID : 0x%2.2hhx = %s\n", et->archId, text);

   switch (et->media)
   {
      case ELT_MEDIA_NO_EMUL  : strcpy( text, "No emulation mode");       break;
      case ELT_MEDIA_DSK_120  : strcpy( text, "Diskette 1.2  MB");        break;
      case ELT_MEDIA_DSK_144  : strcpy( text, "Diskette 1.44 MB");        break;
      case ELT_MEDIA_DSK_284  : strcpy( text, "Diskette 2.88 MB");        break;
      case ELT_MEDIA_HARDISK  : strcpy( text, "Hard disk emulation");     break;
         default:               strcpy( text, "Unknown, invalid!");       break;
   }
   TxPrint("Media emulation   : 0x%2.2hhx = %s\n", et->archId, text);

   TxPrint("Load segment set  : 0x%4.4hx       (0 defaults to standard 0x07C0)\n", et->loadSeg);
   TxPrint("Nr of 512b sects  : 0x%4.4hx\n",          et->sectCount);
   TxPrint("Bootimg loadBlock : %s0x%8.8x%s\n",  CBG, et->loadBlock, CNN);

   nav.down = et->loadBlock;

   RETURN (rc);
}                                               // end 'dfsIsoElToritoHdr'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display ISO  SUSP / Rock Ridge Continuation Area in separate block
/*****************************************************************************/
static ULONG dfsIsoRockRidgeCont                // RET   rc = 0 if type match
(
   BYTE               *sector,                  // IN    sector data
   BOOL                navigation               // IN    update nav values
)
{
   ULONG               rc    = 0;               // rc, sector match
   SRR_CONT_RECORD    *er    = (SRR_CONT_RECORD *) sector;
   TXLN                text;

   ENTER();

   TxPrint("RRContArea Length : 0x%2.2hhx\n", er->Length);

   if (er->IdLength != 0)
   {
      TxCopy( text, (char *) er->Data,  er->IdLength + 1);
      TxPrint("RRContArea     Id : %s\n", text);
   }
   if (er->DescLength != 0)
   {
      TxCopy( text, (char *) er->Data + er->IdLength,  er->DescLength + 1);
      TxPrint("RRContArea  Descr : %s\n", text);
   }
   if (er->SrcLength != 0)
   {
      TxCopy( text, (char *) er->Data + er->IdLength + er->DescLength,  er->SrcLength + 1);
      TxPrint("RRContArea Source : %s\n", text);
   }
   RETURN (rc);
}                                               // end 'dfsIsoRockRidgeCont'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Make filesystem usage map dump on TxPrint output
/*****************************************************************************/
static ULONG dfsIsoAllocMap
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *options,                 // IN    display options
   void               *data                     // OUT   dummy
)
{
   ENTER();

   //- to be refined, always all allocated!

   RETURN (NO_ERROR);
}                                               // end 'dfsIsoAllocMap'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// DFS ISO write-file to disk (SaveTo to file)
/*****************************************************************************/
static ULONG dfsIsoFileSaveAs
(
   ULN64               dirLsn,                  // IN    DirBlock sectornumber
   ULN64               fileIdx,                 // IN    File dir-record index
   char               *path,                    // IN    destination path
   void               *recp                     // IN    recovery params (rename)
)
{
   ULONG               rc  = NO_ERROR;
   S_RECOVER_PARAM    *param = (S_RECOVER_PARAM *) recp;
   DFSISPACE           isp;                     // integrated alloc SPACE info
   BYTE               *sb = NULL;               // sector buffer
   S_ISO_DIRENTRY     *de;                      // ISO directory entry
   TXLN                fname;                   // destination base filename
   TXLN                fullfn;                  // unique filename prefix

   ENTER();
   TRARGS(("Dir block LSN : 0x%llX, file index: %lld, to path: '%s'\n", dirLsn, fileIdx, path));

   if ((sb = TxAlloc( 1, iso->BlockSize)) != NULL)
   {
      if ((rc = dfsRead( dirLsn, 1, sb)) == NO_ERROR)
      {
         switch (dfsIdentifySector( dirLsn, 0, sb))
         {
            case ST_ISODIR:
               if ((de = dfsIsoDirIndex2Entry( sb, DFSSNIGET( fileIdx))) != NULL)
               {
                  if      ((de->NameLength > 1) || (de->FileName[0] > 0x01))
                  {
                     if ((rc = dfsIsoGetAllocSpace( dirLsn, fileIdx, NULL, &isp)) == NO_ERROR)
                     {
                        if ((param->newname == NULL) || (*param->newname == 0)) // no newname present
                        {
                           if (iso->vdUnicode)  // This VD uses Unicode (Joliet)
                           {
                              TxUnic2Ascii( (USHORT *) de->FileName, fname, de->NameLength / 2);
                           }
                           else
                           {
                              TxCopy( fname, (char *) de->FileName,  de->NameLength + 1);
                           }
                           if (fname[ strlen( fname) - 2] == ';') // fileversion postfix
                           {
                              fname[ strlen( fname) - 2] = 0; // remove the postfix
                           }
                        }
                        else
                        {
                           strcpy( fname, param->newname); // replace by specified new name
                        }
                        if (param->unique)      // force unique naming
                        {                       // on PATH and FILE components
                           sprintf( fullfn, "%12.12llX_%4.4llX_%s", dirLsn, fileIdx, fname);
                           strcpy(  fname, fullfn); // and move back to fname
                        }
                        isp.byteSize = dfsIsoBE32( de->FileSize); // set exact size of file for SaveAs

                        rc = dfsSspaceFileSaveAs( &isp, ((de->FileFlags & ISO_FLAG_SUBDIR) != 0),
                                                  FALSE, param->noAllocCheck, param->name83, path, fname, fullfn);
                        if ((rc == NO_ERROR)       || // set original timestamps
                            (rc == DFS_CMD_WARNING) ) // even when allocation errors present
                        {
                           time_t  mod = dfsIsoFileTime2t( &de->FileDateTime);

                           TxSetFileTime( fullfn, &mod, &mod, &mod); // set same time on full-fn from SaveAs

                           if (param->recFname) // return full recovered path+filename
                           {
                              strcpy( param->recFname, fullfn);
                           }
                        }
                        TxFreeMem( isp.space);
                     }
                  }
               }
               else
               {
                  rc = DFS_ST_MISMATCH;         // entry not present in sector
               }
               break;

            default:
               rc = DFS_PENDING;                // not a directory sector
               break;
         }
      }
      TxFreeMem(sb);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN(rc);
}                                               // end 'dfsIsoFileSaveAs'
/*---------------------------------------------------------------------------*/

