//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// HPFS dump & display Analysis functions
//
// Author: J. van Wijk
//
// JvW  12-01-1997 Initial version, split off from DFS.C
// JvW  04-04-2000 Fixed traps in HPFS CodePage & BMGR names
// JvW  05-07-2001 Added hotfix table display to spareblock
// JvW  18-11-2001 Used TXA as command parser, reducing stack usage
// JvW  21-12-2001 Added DIRTY status reporting to HpfsInit
// JvW  13-02-2002 Support HPFS386 deleted FNODES (only 1st byte is zapped)
//
// Author: J. van Wijk
//

#include <txlib.h>                              // TX library interface

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsfind.h>                            // CMD find sector data
#include <dfsutil.h>                            // DFS utility functions
#include <dfsspace.h>                           // DFS file-space interface
#include <dfstable.h>                           // SLT utility functions
#include <dfsahpfs.h>                           // HPFS display & analysis
#include <dfsuhpfs.h>                           // HPFS utility functions
#include <dfslhpfs.h>                           // HPFS SLT functions
#include <dfsos2ea.h>                           // HPFS/JFS/FAT EA handling


                                                // Bitmap framing definitions
#define BLNO           0x00                     // no framing at all
#define BLTOP          0x01                     // framing before top-line
#define BLEND          0x02                     // framing after  end-line
#define BLBOTH         0x03                     // framing both top and end

                                                // Types for Fnode searching
#define FNODES_NONE    0                        // NONE, no FNODE area set yet
#define FNODES_DEL     1                        // DEL, Fnodes for any deleted file
#define FNODES_REF     2                        // REF, Fnodes referenced from directories
#define FNODES_ALL     3                        // REF, Fnodes for any file, anywhere

#define DL_COMPACT     1
#define DL_DEFAULT     5
#define DL_MAXIMUM     9

char dfsHpfsSectorTypes[] =
{
   ST_EDATA,                                    //     EA data
   ST_ADATA,                                    //     ACL data
   ST_BOOTA,                                    //  I  Boot code (micro-FSD)
   ST_SUPER,                                    //     HPFS super-block
   ST_SPARE,                                    //     HPFS spare-block
   ST_HOTFT,                                    //  I  Hotfix table sector
   ST_HOTFD,                                    //  I  Hotfix data sector
   ST_SPDIR,                                    //  I  Spare dirblocks
   ST_BADSL,                                    //  I  Bad sector-list
   ST_FNODE,                                    // S   Fnode, file
   ST_FNDIR,                                    // S   Fnode, directory
   ST_FNDDI,                                    // S   Fnode, deleted directory
   ST_FNDEL,                                    // S   Fnode, deleted file
   ST_ALLOC,                                    // S   Allocation sector
   ST_ALDEL,                                    // S   Alloc sector, deleted
   ST_DIRBL,                                    // S   Directory block
   ST_DBAND,                                    //  I  Pre-allocated Dir-Band
   ST_DBMAP,                                    //  I  Dir-Band bitmap sector
   ST_CPINF,                                    // S   Codepage info
   ST_CPDAT,                                    // S   Codepage data table
   ST_UITAB,                                    //     UserId table (HPFS-386)
   ST_BMAPI,                                    //     Bitmap indirect sector
   ST_BMAPD,                                    //  I  Bitmap data sector
   0                                            //     string terminating ZERO
};


static DFSAHPFS    hpfs_anchor =
{
   0,                                           // root LSN
   0,                                           // codepage LSN
   0,                                           // nr of codepages
   NULL,                                        // super-block
   NULL,                                        // spare-block
   NULL,                                        // bitmap table
   NULL,                                        // bitmap cache
   {0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15}      // Code-page id's
};

       DFSAHPFS   *hpfs = &hpfs_anchor;

static char sg_super[SG_SUPER] = {SV_SUPER};
static char sg_spare[SG_SPARE] = {SV_SPARE};
static char sg_fnode[SG_FNODE] = {SV_FNODE};
static char sg_fndel[SG_FNDEL] = {SV_FNDEL};
static char sg_fd386[SG_FD386] = {SV_FD386};
static char sg_dirbl[SG_DIRBL] = {SV_DIRBL};
static char sg_alloc[SG_ALLOC] = {SV_ALLOC};
static char sg_cpinf[SG_CPINF] = {SV_CPINF};
static char sg_cpdat[SG_CPDAT] = {SV_CPDAT};
static char sg_uitab[SG_UITAB] = {SV_UITAB};

static  char       *hpfs_txt[] =
{
   "",
   "Active filesystem : HPFS, specific commands are:",
   "",
   " \\[path-spec]    = find and show ROOT or file/directory relative to root",
   " BITMAP [xx,s,D] = Show bitmap at LSN xx, size s, in alloc or [D]ir format",
   " BSCLEAR         = Reset  HPFS bad-sector table to ZERO bad sectors",
   " BSEXPORT        = Export HPFS bad-sector table to DFSee Sectorlist",
   " BSIMPORT        = Import DFSee Sectorlist into HPFS bad-sector table",
   " CA   [lsn][opt] = Check Allocation integrity for (current) fnode lsn",
   " DELFIND  [name] = Find deleted files, with FNODE containing (partial) name",
   " DIRTY [clean|d] = Make HPFS dirty-bit Clean (avoids CHKDSK) or set it Dirty",
   " DIRMAP          = Show directory band allocation and usage map",
   " FILEFIND [name] = Find normal files, with FNODE containing (partial) name",
   " FINDBASE [t][l] = find the start of an HPFS partition by searching sectors",
   " FINDCP          = find the Codepage info sectors (before FIXCP and FIXSPARE)",
   " FINDROOT  [lsn] = find the Root directory starting at LSN  (before FIXSUPER)",
   " FIXBOOT         = Create HPFS bootsector from template and partition-table",
   " FIXCP     [cpi] = Update spareblock with found LSN for codepage info",
   " FIXCS           = Calculate checksums and write superblock and spare-block",
   " FIXROOT  [root] = Fix superblock using found/given root LSN and partition-info",
   " FIXSUPER [root] = Fix superblock using found/given root LSN and partition-info",
   " FIXSPARE  [cpi] = Fix spareblock using found/given codepage LSN and superblock",
   " HPFSMAP   [all] = Show data-band allocation bitmaps, HPFS specific format",
   " LABEL   [label] = Display/edit the 11-character volume label in the bootsector",
   " PATH        [n] = Show all path-components for current fnode, upto root",
   " SUPER           = Display the filesystem SPARE and SUPERBLOCK sectors",
   "",
   NULL
};
static char hpfsDirHeaderLine[] =
 " ===== ======== =================== ===== =================== ============= ===== === ====";

static char hpfsDirHeaderText[] =
 " Nr    Fnode    Creation/LastAccess Attr. Modified / Filename     Filesize  EA's  Cp  Flag";


// Close HPFS filesystem for analysis and free any resources
static ULONG dfsHpfsClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
);

// Close HPFS filesystem for Area analysis and free any resources
static ULONG dfsHpfsAreaClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
);

// Create new Superblock sector from defaults and partition-table info
static ULONG dfsHpfsCreateSuper
(
   DFSPARTINFO        *p,                       // IN    partition info
   S_SUPER            *sup                      // OUT   Superblock
);

// Create new Spareblock sector from defaults and Superblock
static ULONG dfsHpfsSuper2Spare
(
   S_SUPER            *sup,                     // IN    Superblock
   S_SPARE            *spr                      // OUT   Spareblock
);

// Add Codepage sectors info to DFSAHPFS structure
static ULONG dfsHpfsSetCodePage
(
   ULONG               cpi                      // IN    Codepage info LSN
);

// Interpret and execute specific HPFS command;
static ULONG dfsHpfsCommand
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *none,                    // IN    dummy
   void               *data                     // INOUT dummy
);

// HPFS filesystem, identify specified sector
static ULONG dfsHpfsIdent
(
   ULN64               sn64,                    // IN    LSN for sector
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // OUT   sector type
   void               *data                     // IN    sector contents
);

// HPFS filesystem, supply sector-type description string
static ULONG dfsHpfsStype
(
   ULN64               di,                      // IN    sector type
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // OUT   type description
   void               *data                     // INOUT dummy
);

// HPFS filesystem, display sector-contents based on type
static ULONG dfsHpfsDispl
(
   ULN64               psn,                     // IN    base psn for sector
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // IN    sector contents
);

// Read the bitmap indirection table and connect to disk structure
static ULONG dfsHpfsReadBmTable
(
   void
);

// Read and display the allocation bitmaps, HPFS specific display
static ULONG dfsHpfsReadBitmap
(
   USHORT             dlvl                      // IN    Detail level
);

// Read and display the allocation bitmaps, standard display
static ULONG dfsHpfsAllocMap
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *options,                 // IN    display options
   void               *data                     // OUT   dummy
);

// Show current sector contents as a bitmap
static ULONG dfsHpfsMapSector                   // RET   nr of 'SET' bits
(
   ULONG              sn,                       // IN    lsn of sector
   ULONG              size,                     // IN    Size to display
   ULONG              bsn,                      // IN    base lsn mapped area
   ULONG              spb,                      // IN    sectors per bit
   ULONG              bpc,                      // IN    bytes per display-char
   USHORT             framing                   // IN    framing specification
);

// Display HPFS super-block
static ULONG dfsHpfsSuperSec                    // RET   rc = 0 if type match
(
   BYTE               *sector,                  // IN    superblock sector
   BOOL                navigation               // IN    update nav values
);

// Display HPFS spare-block
static ULONG dfsHpfsSpareSec                    // RET   rc = 0 if type match
(
   BYTE              *sector                    // IN    Sector contents
);

// Display HPFS file-node
static ULONG dfsHpfsFnodeSec                    // RET   rc = 0 if type match
(
   BYTE              *sector,                   // IN    Sector contents
   BYTE               st,                       // IN    sector type (Fnode)
   ULONG              lsn                       // IN    LSN of fnode
);


// Display HPFS directory-block
static ULONG dfsHpfsDirblock                    // RET   rc = 0 if type match
(
   BYTE              *sector,                   // IN    Sector contents
   ULONG              lsn                       // IN    LSN of sector
);


// Display HPFS allocation sector
static ULONG dfsHpfsAllocSec                    // RET   rc = 0 if type match
(
   BYTE              *sector,                   // IN    Sector contents
   ULONG              lsn                       // IN    LSN of sector
);

// Display HPFS codepage information sector
static ULONG dfsHpfsCpInf                       // RET   rc = 0 if type match
(
   BYTE              *sector,                   // IN    Sector contents
   ULONG              lsn                       // IN    LSN of sector
);


// Display HPFS codepage data sector
static ULONG dfsHpfsCpDat                       // RET   rc = 0 if type match
(
   BYTE              *sector,                   // IN    Sector contents
   ULONG              lsn                       // IN    LSN of sector
);


// Display HPFS (386) user-id information table
static ULONG dfsHpfsUiTab                       // RET   rc = 0 if type match
(
   BYTE              *sector,                   // IN    Sector contents
   ULONG              lsn                       // IN    LSN of sector
);


// Display HPFS allocation bitmap sector
static ULONG dfsHpfsAllocBitmap                 // RET   rc = 0 if type match
(
   BYTE              *sector,                   // IN    Sector data
   USHORT             dlvl                      // IN    Detail level
);

// Display ACL information
static void dfsHpfsAclArea
(
   char              *text,                     // IN    leading text
   S_ACLBLK          *first,                    // IN    first ACL block
   ULONG              size                      // IN    size of ACL area
);

// Make Codepage character-map dump on TxPrint output, 1 line of 64 chars in brackets;
static void dfsHpfsCpCharMap
(
   char              *data,                     // IN    data area
   int                offset                    // IN    offset in map-table
);

// Make Bitmap dump of data area on TxPrint output
static ULONG dfsHpfsBitmapArea                  // RET   nr of 'SET' bits
(
   BYTE              *data,                     // IN    data area
   ULONG              size,                     // IN    size to dump
   ULONG              first,                    // IN    LSN of first 'bit'
   ULONG              bsn,                      // IN    base lsn mapped area
   ULONG              spb,                      // IN    sectors per bit
   ULONG              bpc,                      // IN    bytes per display-char
   USHORT             framing                   // IN    framing specification
);

// Truncate the filesystem size, keeping all data intact; (2 sectors in br)
static ULONG dfsHpfsTruncate
(
   ULN64               newsize,                 // IN    new size in sectors
   ULN64               d2,                      // IN    dummy
   char               *string,                  // IN    dummy
   void               *data                     // IN    partition info
);


/*****************************************************************************/
// Initialize HPFS filesystem analysis
/*****************************************************************************/
ULONG dfsHpfsInit
(
   char               *fs                       // forced filesystem type
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   dfsa->FsCommand          = dfsHpfsCommand;
   dfsa->FsClose            = dfsHpfsClose;
   dfsa->FsIdentifySector   = dfsHpfsIdent;
   dfsa->FsShowType         = dfsHpfsStype;
   dfsa->FsDisplaySector    = dfsHpfsDispl;
   dfsa->FsFileInformation  = dfsHpfsFileInfo;
   dfsa->FsGetAllocSpace    = dfsHpfsGetAllocSpace;
   dfsa->FsWriteMetaSpace   = NULL;
   dfsa->FsFindPath         = dfsHpfsFindPath;
   dfsa->FsUpdateFileName   = dfsHpfsUpdateFileName;
   dfsa->FsSetAltBrecLabel  = NULL;
   dfsa->FsMakeBrowseList   = dfsHpfsMakeBrowseList;
   dfsa->FsLsnAllocated     = dfsHpfsAllocated;
   dfsa->FsLsnSetAlloc      = dfsHpfsSetAlloc;
   dfsa->FsSltBuild         = dfsHpfsSltBuild;
   dfsa->FsNpBuild          = NULL;
   dfsa->FsDisplayError     = dfsHpfsDispError;
   dfsa->FsDisplayLsnInfo   = NULL;
   dfsa->FsDirIterator      = dfsHpfsIterator;
   dfsa->FsDirFileSaveAs    = dfsHpfsFnodeSaveAs;
   dfsa->FsTruncateSize     = dfsHpfsTruncate;
   dfsa->FsAllocDisplay     = dfsHpfsAllocMap;
   dfsa->FsCl2Lsn           = NULL;             // dummies, one to one
   dfsa->FsLsn2Cl           = NULL;
   dfsa->Fsi                = hpfs;
   dfsa->FsSectorTypes      = dfsHpfsSectorTypes;
   dfsa->FsCmdHelp          = hpfs_txt;         // FS specific cmdhelp text

   dfsa->FsModeId           = DFS_FS_HPFS;

   dfsa->FsEntry            = LSN_SUPER;        // entry-point in filesystem

   TRACES(("sizeof S_SUPER        = % 3u\n", sizeof(S_SUPER)));
   TRACES(("sizeof S_SPARE        = % 3u\n", sizeof(S_SPARE)));
   TRACES(("sizeof S_FNODE        = % 3u\n", sizeof(S_FNODE)));
   TRACES(("sizeof S_DIRENT       = % 3u\n", sizeof(S_DIRENT)));
   TRACES(("sizeof S_DIRBL        = % 3u\n", sizeof(S_DIRBL)));
   TRACES(("sizeof S_ALLOC        = % 3u\n", sizeof(S_ALLOC)));
   TRACES(("sizeof S_CPINF        = % 3u\n", sizeof(S_CPINF)));
   TRACES(("sizeof S_CPIB         = % 3u\n", sizeof(S_CPIB)));
   TRACES(("sizeof S_CPDB         = % 3u\n", sizeof(S_CPDB)));
   TRACES(("sizeof S_CPDAT        = % 3u\n", sizeof(S_CPDAT)));
   TRACES(("sizeof S_UITAB        = % 3u\n", sizeof(S_UITAB)));

   dfstSetClusterSize( DFSTORE, 1);        // reset to only valid value!

   hpfs->root  = L32_NULL;
   hpfs->cpinf = L32_NULL;

   if (((hpfs->sup = TxAlloc(1, dfstGetSectorSize(DFSTORE))) != NULL) &&
       ((hpfs->spr = TxAlloc(1, dfstGetSectorSize(DFSTORE))) != NULL) &&
       ((hpfs->bmc = TxAlloc(4, dfstGetSectorSize(DFSTORE))) != NULL) )
   {
      TRACES(( "hpfs->bmt: %8.8x,    bmc:%8.8x\n", hpfs->bmt, hpfs->bmc));
      TRACES(( "hpfs->spr: %8.8x   super:%8.8x\n", hpfs->spr, hpfs->sup));
      if ((rc = dfsRead(LSN_SUPER, 1, (BYTE   *) hpfs->sup)) != NO_ERROR)
      {
         TxPrint("Error reading the HPFS Super-block from disk, possible bad-sector!\n");
      }
      if (dfsIdentifySector(LSN_SUPER, 0, (BYTE   *) hpfs->sup) != ST_SUPER)
      {
         TxPrint("\nNo valid HPFS Super-block found, using defaults\n\n"
                 "You might want to use the 'findroot' and 'fixroot'\n"
                 "commands to get the correct root-directory value, and\n"
                 "to write the now created superblock back to the disk\n");

         rc = dfsHpfsCreateSuper( SINF->p, hpfs->sup);
      }
      TRHEXS( 70,  hpfs->sup,  sizeof(S_SUPER), "hpfs superblock");
      if (hpfs->sup != 0)
      {
         TRACES(( "Totalsec: %8.8x\n", hpfs->sup->TotalSec));
      }
      hpfs->root         = hpfs->sup->RootFnode; // keep root LSN
      dfsa->FsSltRecSize = hpfs->sup->TotalSec / 8  + 1000; // average 4KB per file

      dfsa->findSpace.spid = FNODES_NONE;       // No cached Fnode search space

      rc = dfsHpfsReadBmTable();
      if (rc != NO_ERROR)
      {
         TxPrint("\nErrors getting bitmap-tables, allocation information unreliable!\n");
      }

      if ((rc = dfsRead(LSN_SPARE, 1, (BYTE   *) hpfs->spr)) != NO_ERROR)
      {
         TxPrint("Error reading the HPFS Spare-block from disk, possible bad-sector!\n");
      }
      if (dfsIdentifySector(LSN_SPARE, 0, (BYTE   *) hpfs->spr) != ST_SPARE)
      {
         dfsHpfsSuper2Spare( hpfs->sup, hpfs->spr);
         TxPrint("\nNo valid HPFS Spare-block found, using defaults\n"
                 "Use the 'FIXSPARE' command to replace the "
                 "spareblock on the disk too\n\n");
      }
      else
      {
         if (hpfs->spr->StatusFlag & HPFS_SP_DIRTY)
         {
            dfsa->FsDirtyStatus = DFSTAT_DIRTY;
         }
         else
         {
            dfsa->FsDirtyStatus = DFSTAT_CLEAN;
         }
         TxPrint("FileSys Dirty bit : %s\n",
            (hpfs->spr->StatusFlag & HPFS_SP_DIRTY) ? "Dirty" : "Clean");
      }
      if (dfsHpfsSetCodePage( hpfs->spr->CodePageInfo) != NO_ERROR)
      {
         TxPrint("\nErrors getting codepage-tables, codepage information unreliable!\n");
      }
      if (TxaOption('a'))                       // show NO allocation by default
      {
         DFSPARTINFO   *p;

         dfsHpfsAllocMap( 0, 0, "@", NULL);     // get/show Minimum size
         if ((p = SINF->p) != NULL)             // we have a real partition
         {
            ULONG   kmgAlign = dfsAlignmentSectors((p->tablenr == GPT_STYLE) ? 'G' : 'P', p);

            if ((p->expandSize != 0) &&         // fspAfter, at least one alignment unit
                (p->expandSize - p->sectors >= kmgAlign))
            {
               TXTM    description;
               ULONG   crLimit = HPFS_LIMIT - kmgAlign + 1;

               strcpy( description, " (for Expanding");

               if (p->expandSize > crLimit)     // hit OS/2 size limit for HPFS
               {
                  strcat( description, ", OS/2 HPFS limit)\n");
                  dfsa->FsExpandSize = crLimit;
               }
               else
               {
                  strcat( description, " into freespace)\n");
                  dfsa->FsExpandSize = p->expandSize;
               }
               dfsSz64( "Maximum vol. size : ", dfsa->FsExpandSize, description);
            }
         }
      }
      TRACES(( "hpfs->bmt: %8.8x,    bmc:%8.8x\n", hpfs->bmt, hpfs->bmc));
      TRACES(( "hpfs->spr: %8.8x   super:%8.8x\n", hpfs->spr, hpfs->sup));
      TRHEXS( 70,  hpfs->sup,  sizeof(S_SUPER), "hpfs superblock");
      if (hpfs->sup != 0)
      {
         TRACES(( "Totalsec: %8.8x\n", hpfs->sup->TotalSec));
      }
   }
   else
   {
      TxFreeMem(hpfs->sup);
      TxFreeMem(hpfs->spr);
      TxFreeMem(hpfs->bmc);
      printf( "\nError allocating buffer memory\n");
      rc = DFS_ALLOC_ERROR;
   }
   TRHEXS( 70,  hpfs,  sizeof(DFSAHPFS), "hpfs global info");
   RETURN (rc);
}                                               // end 'dfsHpfsInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Close HPFS filesystem for analysis and free any resources
/*****************************************************************************/
static ULONG dfsHpfsClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   dfsSlTableReset();                           // stop SLT thread and reset
   TxFreeMem( hpfs->sup);
   TxFreeMem( hpfs->spr);
   TxFreeMem( hpfs->bmc);
   TxFreeMem( hpfs->bmt);
   TxFreeMem( dfsa->findSpace.space);           // free the FNODE FIND SPACE
   dfsa->findSpace.spid = FNODES_NONE;          // No cached Fnode search space
   dfsCloseFileSystem();
   RETURN (rc);
}                                               // end 'dfsHpfsClose'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Initialize HPFS filesystem analysis for Area (FS info in FDISK mode)
/*****************************************************************************/
ULONG dfsHpfsAreaInit
(
   void
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   dfsa->FsAreaClose        = dfsHpfsAreaClose;
   dfsa->FsAreaLsnAllocated = dfsHpfsAllocated;

   if (((hpfs->sup = TxAlloc(1, dfstGetSectorSize(DFSTORE))) != NULL) &&
       ((hpfs->bmc = TxAlloc(4, dfstGetSectorSize(DFSTORE))) != NULL) )
   {
      if ((rc = dfsRead( dfstAreaP2Disk( DFSTORE, LSN_SUPER), 1, (BYTE   *) hpfs->sup)) == NO_ERROR)
      {
         rc = dfsHpfsReadBmTable();
      }
      TRACES(( "hpfs->bmt: %8.8x,    bmc:%8.8x\n", hpfs->bmt, hpfs->bmc));
   }
   else
   {
      TxFreeMem(hpfs->sup);
      TxFreeMem(hpfs->bmc);
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsHpfsAreaInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Close HPFS filesystem for Area analysis and free any resources
/*****************************************************************************/
static ULONG dfsHpfsAreaClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   dfsa->FsAreaClose        = NULL;
   dfsa->FsAreaLsnAllocated = NULL;

   TxFreeMem(hpfs->sup);
   TxFreeMem(hpfs->bmc);
   TxFreeMem(hpfs->bmt);
   RETURN (rc);
}                                               // end 'dfsHpfsAreaClose'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create new Superblock sector from defaults and partition-table info
/*****************************************************************************/
static ULONG dfsHpfsCreateSuper
(
   DFSPARTINFO        *p,                       // IN    partition info
   S_SUPER            *sup                      // OUT   Superblock
)
{
   ULONG               rc  = NO_ERROR;
   ULONG               bmilen;

   ENTER();

   memset( (char *) sup, 0, SECTORSIZE);        // start with zeroes
   memcpy( sup->Signature, sg_super, SG_SUPER);

   if (p != NULL)
   {
      hpfs->sup->TotalSec  = SINF->p->sectors & 0xfffffffc;
   }
   else
   {
      hpfs->sup->TotalSec  = 0x7d43;            // size of 2-cylinder logical
   }
   hpfs->sup->VerMajor     = 0x02;              // Assume HPFS 2.2
   hpfs->sup->VerMinor     = 0x02;
   hpfs->sup->BitmapInd    = LSN_BMIDX;         // use fixed position

   bmilen = ((hpfs->sup->TotalSec >> 23) +1) * 4; // 4 sectors for every 4GB
   hpfs->sup->BadList = LSN_BMIDX + bmilen   + 4; // skip BMI and 1st bitmap

   RETURN( rc);
}                                               // end 'dfsHpfsCreateSuper'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create new Spareblock sector from defaults and Superblock
/*****************************************************************************/
static ULONG dfsHpfsSuper2Spare
(
   S_SUPER            *sup,                     // IN    Superblock
   S_SPARE            *spr                      // OUT   Spareblock
)
{
   ULONG               rc  = NO_ERROR;
   ULONG               SpareDirLSN;
   ULONG               i;

   ENTER();

   memset( (char *) spr, 0, SECTORSIZE);        // start with zeroes
   memcpy( spr->Signature, sg_spare, SG_SPARE);

   spr->StatusFlag      = HPFS_SP_OLDUTIL;      // possibly inconsistent
   spr->SpareDirMax     = 20;                   // most likely default
   spr->SpareDirUsed    = 20;                   // most likely default

   if (hpfs->cpinf == 0)                        // no CP-info read yet
   {
      spr->CodePageInfo = 88;
      spr->CodePages    = 2;
   }
   else
   {
      spr->CodePageInfo = hpfs->cpinf;
      spr->CodePages    = hpfs->cpages;
   }

   if (sup->TotalSec    > 0x4000)               // larger than 8MB
   {
      if (sup->TotalSec > 0x8000)               // larger than 16MB
      {
         spr->HotFixMax = 100;                  // most likely default
      }
      else
      {
         spr->HotFixMax = 64;            // most likely default
      }
      spr->HotFixList   = sup->BadList      +4; // most likely position
      SpareDirLSN       = sup->DirBandEnd   +1; // most likely position
   }
   else                                         // tiny HPFS partitions
   {
      spr->HotFixMax    = 40;                   // most likely default
      spr->HotFixList   = sup->DirBandEnd   +5; // most likely position
      SpareDirLSN       = spr->CodePageInfo +4; // most likely position
   }
   for (i = 0;   i < spr->SpareDirMax;  i++)
   {
      spr->SpareDir[ i] = SpareDirLSN + i * 4;
   }
   RETURN( rc);
}                                               // end 'dfsHpfsSuper2Spare'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add Codepage sectors info to DFSAHPFS structure
/*****************************************************************************/
static ULONG dfsHpfsSetCodePage
(
   ULONG               cpi                      // IN    Codepage info LSN
)
{
   ULONG               rc  = NO_ERROR;
   BYTE               *cps = NULL;              // CodePage sector

   ENTER();

   if ((cps = TxAlloc( BBUFS, SECTORSIZE)) != NULL)
   {
      rc = dfsRead( cpi, 1, cps);
      if (rc == NO_ERROR)
      {
         if (dfsIdentifySector( cpi, 0, cps) == ST_CPINF)
         {
            S_CPINF   *sd = (S_CPINF *) cps;
            ULONG      ib;

            hpfs->cpinf  = cpi;                 // update global CP info
            hpfs->cpages = sd->CpCount;

            for ( ib = 0;
                 (ib < sd->CpCount) && (ib < S_MAX_CPIB);
                  ib++)                         // iterate attached CpData
            {
               hpfs->cpg[ib] = sd->Cpib[ib].CodePageId; // keep Cp around
            }
            if (ib >= S_MAX_CPIB)
            {
               TxPrint("Error in CodePage sectors\n");
               rc = DFS_BAD_STRUCTURE;
            }
         }
         else
         {
            TxPrint("\nInvalid lsn for CodePage sector, you might want to\n"
                    "use the 'findcp' and 'fixcp' commands to fix this\n");
            rc = DFS_BAD_STRUCTURE;
         }
      }
      TxFreeMem( cps);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN( rc);
}                                               // end 'dfsHpfsSetCodePage'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Interpret and execute specific HPFS command;
/*****************************************************************************/
static ULONG dfsHpfsCommand
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *none,                    // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;
   ULN64               sn64;                    // 64bit sectornumber
   ULONG               sn = 0;                  // sector number input
   ULONG               sl = 0;                  // sector LSN
   BYTE                st = 0;                  // sector type wanted
   ULONG               nr = 0;
   TXLN                dc;                      // DFS command
   int                 cc;                      // command string count
   char               *c0, *c1, *c2;            // parsed command parts
   char               *pp;                      // parameter pointer
   TXLN                s0;                      // big temporary string space
   TXLN                s1;                      // big temporary string space
   ULONG               size  = 1;               // default display size

   ENTER();

   pp = TxaGetArgString( TXA_CUR, 1, 0, TXMAXLN, dc); // dc => cmd from arg 1
   cc = TxaArgCount( );                         // number of parameters
   c0 = TxaArgValue(0);
   c1 = TxaArgValue(1);
   c2 = TxaArgValue(2);

   sn = (ULONG) nav.this;                       // default at current sector

   if ((strcmp(c0, "/?") == 0) ||
       (strcmp(c0,  "?") == 0)  )
   {
      TxShowTxt( hpfs_txt);
   }
   else if (strcasecmp(c0, "super"    ) == 0)
   {
      sprintf( dc, "0x%x#0x%x", LSN_SPARE, LSN_SUPER);
      rc = dfsMultiCommand( dc, 0, FALSE, FALSE, TRUE);
   }
   else if ((strcasecmp(c0, "findroot" ) == 0) ||
            (strcasecmp(c0, "path"     ) == 0) )
   {
      if (TxaOption('?'))
      {
         TxPrint("\nLocate root-directory (lsn) and show path to root\n");
         TxPrint("\n Usage:  %s  [start-lsn]\n\n", c0);
      }
      else
      {
         sn = dfsGetSymbolicSN( c1, nav.this);  // default current LSN
         if (dfsHpfsFindRootFnode( sn, TRUE, TRUE, s0, &sn))
         {
            if (strcasecmp(c0, "findroot" ) == 0)
            {
               hpfs->root = sn;                 // remember Root
               rc = dfsReadAnDisplay(sn, 0, &st);
            }
         }
         else
         {
            TxPrint("No HPFS Fnode with path to Root found\n");
            rc = DFS_NOT_FOUND;
         }
      }
   }
   else if ((strncasecmp(c0, "bsc", 3) == 0) || // clear FS badsector admin
            (strcasecmp( c0, "nobads") == 0)  ) // deprecated
   {
      if (TxaOption('?'))
      {
         TxPrint("\nReset bad-sector administration in the superblock\n");
         TxPrint("\n Usage:  %s\n\n", c0);
      }
      else
      {
         if (hpfs->sup->BadSec != 0)
         {
            dfsSiz8( "\nTotal bad sectors : ", hpfs->sup->BadSec, ", resetting ...\n");
            dfsInitList( 0, "-w", "-f -P");     // with optimal list options
            if (DFSTORE_WRITE_ALLOWED)
            {
               rc = dfsHpfsImportBadList( dfsa->snlist);
               if (rc == NO_ERROR)
               {
                  rc = dfsHpfsWriteSuperSpare();
               }
            }
            else
            {
               rc = DFS_READ_ONLY;
            }
         }
         else
         {
            TxPrint( "\nNo bad sectors present, no action required ...\n");
         }
      }
   }
   else if ((strncasecmp(c0, "bse", 3) == 0) || // FS to snlist
            (strcasecmp( c0, "getbs" ) == 0)  ) // deprecated
   {
      if (TxaOption('?'))
      {
         TxPrint("\nExport HPFS bad-sector table in superblock to DFSee Sectorlist\n");
         TxPrint("\n Usage:  %s\n\n", c0);
      }
      else
      {
         rc = dfsHpfsExportBadList( dfsa->snlist, dfsa->snsize -1);
         if (rc == NO_ERROR)
         {
            TxPrint( "\nSuccessfully exported %u bad sectors to Sectorlist:\n", dfsa->snlist[0]);
            strcpy(  dc, "list -r");            // display in range format
            TxaReParseCommand( dc);
            rc = DFS_PENDING;                   // handle translated command
         }
         else
         {
            TxPrint( "\nExport of bad sectors failed!\n");
         }
      }
   }
   else if ((strncasecmp(c0, "bsi", 3) == 0) || // snlist to FS
            (strcasecmp( c0, "fixbs" ) == 0)  ) // deprecated
   {
      if (TxaOption('?'))
      {
         TxPrint("\nImport DFSee Sectorlist into HPFS bad-sector table in superblock\n");
         TxPrint("\n Usage:  %s\n\n", c0);
      }
      else
      {
         if (dfsa->snlist[0] != 0)              // bad sectors defined ?
         {
            if (DFSTORE_WRITE_ALLOWED)
            {
               if ((rc = dfsSlTableStatus(&size)) == SLT_READY)
               {
                  rc = dfsHpfsImportBadList( dfsa->snlist);
                  if (rc == NO_ERROR)
                  {
                     rc = dfsHpfsWriteSuperSpare();
                     dfsSz64("Bad-sector count  : ", dfsa->snlist[0], "\n\n");
                     TxPrint("HPFS bad-sector table updated, a reboot is needed "
                             "to update HPFS.IFS\nand CHKDSK statistics like "
                             "reserved and free-space.\n");
                  }
               }
               else
               {
                  TxPrint( "Before updating the bad-sector table, run the 'CHECK'"
                           "command, and try again\n");
                  rc = DFS_CMD_FAILED;
               }
            }
            else
            {
               rc = DFS_READ_ONLY;
            }
         }
         else
         {
            TxPrint("DFSee Sectorlist is empty, use 'scan', 'listset' "
                    "or 'import' command and try again\n");
         }
      }
   }
   else if (strcasecmp(c0, "dirty"    ) == 0)
   {
      if (cc > 1)
      {
         if ((c1[0] == 'd') || (c1[0] == 'D') || (c1[0] == '1'))
         {
            hpfs->spr->StatusFlag |= HPFS_SP_DIRTY; // set Dirty bit
         }
         else
         {
            hpfs->spr->StatusFlag &= ~HPFS_SP_DIRTY; // clear Dirty bit
         }
         if (hpfs->spr->StatusFlag & HPFS_SP_DIRTY) // sync global dirty flag!
         {
            dfsa->FsDirtyStatus = DFSTAT_DIRTY;
         }
         else
         {
            dfsa->FsDirtyStatus = DFSTAT_CLEAN;
         }
         rc = dfsHpfsWriteSuperSpare();
      }
      else
      {
         TxPrint("\nUsage: %s  clean | dirty\n", c0);
      }
      TxPrint("\nHPFS Dirty bit is : %s\n",
                 (hpfs->spr->StatusFlag & HPFS_SP_DIRTY) ? "Dirty" : "Clean");
   }
   else if ((strcasecmp(c0, "fixroot"  ) == 0) ||
            (strcasecmp(c0, "fixsuper" ) == 0)  )
   {
      hpfs->root = dfsGetSymbolicSN( c1, hpfs->root);
      if (hpfs->root != L32_NULL)
      {
         nr = hpfs->sup->RootFnode;
         if ((hpfs->sup->DirBandMap == 0) ||    // Created new, make complete
             (TxaOption('r')))                  // or recalculculate request
         {

            ULONG tot = hpfs->sup->TotalSec;
            ULONG dbs = (((ULONG) (tot * 10 ) / 1024)         ) & 0xfffffffc;
            ULONG dbp = (((ULONG) (tot - dbs) /    2) + 0x2000) & 0xffffc000;

            hpfs->sup->DirBandMap = dbp - 12;
            if (hpfs->root <= LSN_RESRV)        // invalid, use fixed default
            {
               hpfs->root = dbp - 4;
            }
            hpfs->sup->DirBandSec = dbs;
            if ((dbp & 0x0000ffff) == 0)        // at a regular bitmap LSN
            {
               dbp += 4;                        // skip the bitmap
            }
            hpfs->sup->DirBand    = dbp;
            hpfs->sup->DirBandEnd = dbp + dbs -1;
            TxPrint("\nSUPERBLOCK %s from remaining info:\n\n",
                     TxaOption('r') ? "recalculated" : "created new");
         }
         else
         {
            TxPrint("\nSUPERBLOCK updated with RootDirectory LSN:\n\n");
         }
         hpfs->sup->RootFnode = hpfs->root;             //- Set root LSN
         dfsHpfsSuperSec((BYTE   *) hpfs->sup, TRUE);   //- display the result
         if ((dfsa->batch) || (TxConfirm( 5026,
                "Do you want to replace the superblock "
                "(sector 0x10) with the updated one ? [Y/N] :\n")))
         {
            if ((rc = dfsHpfsWriteSuperSpare()) == NO_ERROR)
            {
               if (nr != hpfs->root)
               {
                  TxPrint("Rootdir LSN changed from 0x%8.8X to 0x%8.8X\n",
                           nr, hpfs->root);
               }
            }
         }
      }
      else
      {
         TxPrint("RootDir sector not known yet, use 'findroot' command "
                 "to find it; and try again\n");
      }
   }
   else if (strcasecmp(c0, "fixboot"  ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nFix the bootsector for an HPFS partition\n");
         TxPrint("\n Usage:  %s  [drive | -] [-c | -i13X]\n\n", c0);
         TxPrint("   drive | - = Optional boot driveletter, - resets to 'data'\n"
                 "               Determined automatically when not specified.\n");
         TxPrint("   -c        = Create a Classic (OS2 20.0) type HPFS bootsector\n"
                 "               instead of the newer I13X (LVM) capable bootcode\n");
         TxPrint("   -I13X     = Require I13X capable MBR code if beyond cyl 1024\n");
      }
      else
      {
         if ((rc = dfsHpfsMkBootRec( hpfs->sup, c1)) == NO_ERROR)
         {
            if (SINF->partid != 0)              // was it a partition ?
            {
               TxPrint( "Resulting partition after fixing the bootsector:\n\n");
               sprintf( dc, "part -r -q %hu#d", SINF->partid);
               rc = dfsMultiCommand( dc, 0, FALSE, FALSE, TRUE);
            }
         }
      }
   }
   else if (strcasecmp(c0, "fixhpldr"  ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nReplace the HPFS micro filesystem for a HPFS partition\n");
         TxPrint("\n Usage:  %s   [image]\n"
                 "\n  image] = Name of imagefile with HPFS LDR code (%u sectors)"
                 "\n           with a default of '%s'\n",
                               c0, HPLDR_SIZE, HPLDR_IMGNAME);
      }
      else
      {
         strcpy(  s0, (cc > 1) ? c1 : HPLDR_IMGNAME);
         sprintf( s1, "restore \"%s\" 0%x %u", s0, HPLDR_LSN, HPLDR_SIZE);
         rc = dfsMultiCommand( s1, 0, TRUE, FALSE, TRUE);
      }
   }
   else if (strcasecmp(c0, "dfshpldr"  ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nCreate compressed imagefile containing the HPFS micro filesystem\n");
         TxPrint("\n Usage:  %s   [image]\n"
                 "\n  image  = Name of imagefile to be created (%u sectors) without"
                 "\n           a file-extension, and a default name of '%s'\n",
                               c0, HPLDR_SIZE, HPLDR_IMGNAME);
      }
      else
      {
         strcpy(  s0, (cc > 1) ? c1 : HPLDR_IMGNAME);
         sprintf( s1, "image %s 0x0%x %u  -z", s0, HPLDR_LSN, HPLDR_SIZE);
         rc = dfsMultiCommand( s1, 0, TRUE, FALSE, TRUE);
      }
   }
   else if (strcasecmp(c0, "write"    ) == 0)
   {
      sn = (ULONG) dfsGetSymbolicSN( c1, nav.this); // default current LSN
      if      (sn == LSN_SUPER)                 // update cached sectors
      {
         memcpy( hpfs->sup, rbuf, dfsGetSectorSize());
      }
      else if (sn == LSN_SPARE)
      {
         memcpy( hpfs->spr, rbuf, dfsGetSectorSize());
      }
      rc = DFS_PENDING;                         // handle actual write
   }
   else if (strcasecmp(c0, "findcp"   ) == 0)
   {
      nav.this = 0;                             // start from LSN 0
      strcpy( dc, "find -t:c -o:");
      if (TxaOption('a'))
      {
         strcat( dc, "*");                      // keep searching after 1st
      }
      TxaReParseCommand( dc);
      rc = DFS_PENDING;                         // execute the "find"
   }
   else if (strcasecmp(c0, "fixcp"    ) == 0)
   {
      if ((hpfs->cpinf != L32_NULL) || (cc > 1))
      {
         hpfs->cpinf = dfsGetSymbolicSN( c1, hpfs->cpinf);
         nr = hpfs->spr->CodePageInfo;
         hpfs->spr->CodePageInfo = hpfs->cpinf; // Correct CP LSN
         if ((rc = dfsHpfsWriteSuperSpare()) == NO_ERROR)
         {
            if (nr != hpfs->cpinf)
            {
               TxPrint("\nCodePage LSN changed from 0x%8.8X to 0x%8.8X\n",
                          nr, hpfs->cpinf);
            }
         }
      }
      else
      {
         TxPrint("\nCODEPAGE sector not known yet, use 'findcp' to find it,\n"
                   "or force update by using %s known-cp-value or 0\n", c0);
      }
   }
   else if (strcasecmp(c0, "fixspare" ) == 0)
   {
      if ((hpfs->cpinf != L32_NULL) || (cc > 1))
      {
         hpfs->cpinf = dfsGetSymbolicSN( c1, hpfs->cpinf);
         dfsHpfsSuper2Spare( hpfs->sup, hpfs->spr);
         TxPrint("\nNew SPAREBLOCK created from superblock "
                   "and CodePage sector\n\n");
         dfsHpfsSpareSec((BYTE   *) hpfs->spr); // display the result
         if ((dfsa->batch) || (TxConfirm( 5027,
                "Do you want to replace the spareblock "
                "(sector 0x11) with the created one ? [Y/N] :\n")))
         {
            if ((rc = dfsHpfsWriteSuperSpare()) == NO_ERROR)
            {
               TxPrint("New SPAREBLOCK with updated checksums written to disk\n");
            }
         }
      }
      else
      {
         TxPrint("\nCODEPAGE sector not known yet, use 'findcp' to find it,\n"
                   "or force update by using %s known-cp-value or 0\n", c0);
      }
   }
   else if (strcasecmp(c0, "fixcs"    ) == 0)
   {
      rc = dfsHpfsWriteSuperSpare();
   }
   else if ((strcasecmp(c0, "findbase") == 0) ||
            (strcasecmp(c0, "autobase") == 0)  )
   {
      BYTE            *sect;
      ULN64            snlimit = dfsGetLogicalSize();

      if ((sect = TxAlloc( 1, dfsGetSectorSize())) != NULL)
      {
         sn64 = nav.this;                       // default base from 'this'
         sl   = (ULONG) dfsGetSymbolicSize( c2, snlimit, DFSTORE, sn64); // to end
         strcpy( s0, (cc > 1) ? c1 : "spad");

         dfsProgressInit( sn, snlimit - sn, 0, "Sector:", "searched", DFSP_STAT, 0);
         do
         {
            nr = dfsGetSectorSize();
            dfsSectorTypeAsAscii( s0[0], s1);
            if (dfsFindTypeContaining( "", sn64, 1, s0, 0, NULL, 0, NULL, snlimit, &sn64, &st, &nr))
            {
               sn = (ULONG) sn64;
               if ((rc = dfsRead( sn, 1, sect)) == NO_ERROR)
               {
                  st = dfsIdentifySector(sn, 0, sect);
                  dfsSectorTypeAsAscii(  st, s1);
                  dfsDisplaySnType("Original found SN : ", sn, 0, st);
                  sn = dfstLastPsn( DFSTORE);   // physical position of sect
                  switch (st)                   // correct with logical pos
                  {
                     case ST_DIRBL: sn -= ((S_DIRBL*) sect)->This; break;
                     case ST_ALDEL:
                     case ST_ALLOC: sn -= ((S_ALLOC*) sect)->This; break;
                     case ST_SUPER: sn -= LSN_SUPER;               break;
                     case ST_SPARE: sn -= LSN_SPARE;               break;
                     case ST_FNDEL:
                     case ST_FNODE:
                        {
                           S_FNODE *sd = (S_FNODE *) sect;
                           if ((sd->AlInfo.Flag & HPFS_AL_NODE) == 0) // Leaf ?
                           {
                              sn -= sd->Al.Leaf[0].DataBlock -1;
                              TxPrint("\nFindbase on FNODE relies "
                                      "on default HPFS allocation "
                                      "and is NOT 100%% reliable!\n\n");
                           }
                           else
                           {
                              rc = DFS_ST_MISMATCH; // keep searching
                           }
                        }
                        break;

                     default:
                        TxPrint("Findbase can't use: %s%s%s\n", CBM, s1, CNN);
                        if (strlen(s0) > 1)
                        {
                           rc = DFS_ST_MISMATCH; // keep searching
                        }
                        else
                        {
                           rc = DFS_CMD_FAILED;
                        }
                        break;
                  }
                  if (rc == NO_ERROR)
                  {
                     S_SUPER *sd = (S_SUPER *) sect;

                     if (st != ST_SUPER)        // if not superblock found
                     {
                        rc = dfstReadPsn( DFSTORE, sn + LSN_SUPER, 1, sect);
                     }
                     if ((rc == NO_ERROR) && (sd->TotalSec > 0))
                     {
                        sl = sn + sd->TotalSec -1;
                     }
                     dfstLogicalLimits( DFSTORE, sn, sl);
                     nav.up   = LSN_SUPER;
                     nav.xtra = LSN_SPARE;
                     nav.down = nav.this = dfstPsn2LSN( DFSTORE,
                                           dfstLastPsn( DFSTORE));
                     TxPrint("Base limits set to: 0x%8.8X - 0x%8.8X\n", sn, sl);
                     dfsMultiCommand( "base", 0, FALSE, FALSE, TRUE);
                     TxPrint("Re-initializing to 'fs HPFS' ...\n");
                     strcpy( dc, "fs HPFS");
                     TxaReParseCommand( dc);
                     rc = DFS_PENDING;          // execute the "fs"
                  }
               }
            }
            else
            {
               TxPrint("\nNo %s%s%s found\n", CBM, s1, CNN);
               rc = DFS_CMD_FAILED;
            }
         } while ((rc == DFS_ST_MISMATCH) && (!TxAbort()));
         dfsProgressTerm();
         TxFreeMem( sect);
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   else if (strcasecmp(c0, "ca"   ) == 0)          // check allocation
   {
      S_FNODE            *sd;

      sn = (ULONG) dfsGetSymbolicSN( c1, nav.this); // default current LSN
      if ((rc = dfsHpfsReadChkFnode( sn, &st, &sd)) == NO_ERROR)
      {
         strcpy(s0, (dfsa->verbosity >= TXAO_NORMAL) ? "v" : ""); // specified
         nr = size = 0;
         rc = dfsHpfsFnodeCheckAlloc( sd, s0, st, &size, &nr, &sl);
         if ((rc == NO_ERROR) || (rc == DFS_BAD_STRUCTURE))
         {
            dfsSiz8("Total filesectors : ", size, "\n");
            dfsSiz8("Allocation errors : ", nr,   "\n");
         }
         else
         {
            TxPrint( "\nSector 0x%8.8x does not seem to be a valid FNODE!\n", sn);
            rc = DFS_ST_MISMATCH;
         }
         TxFreeMem( sd);
      }
      else                                      // not a valid Fnode
      {
         TxPrint("CheckAlloc only implemented for Fnodes\n");
      }
   }
   else if (strcasecmp(c0, "hpfsmap"  ) == 0)
   {
      if (cc > 1)                               // maximum detail
      {
         rc = dfsHpfsReadBitmap(DL_MAXIMUM);
      }
      else
      {
         rc = dfsHpfsReadBitmap(DL_COMPACT);
      }
   }
   else if (strcasecmp(c0, "dirmap"   ) == 0)
   {
      dfsDec8("Dir. band sectors : ", hpfs->sup->DirBandSec, "\n");
      dfsHpfsMapSector( hpfs->sup->DirBandMap,
                       (hpfs->sup->DirBandSec + 31) / 32,
                        hpfs->sup->DirBand,
                        4, 1, BLBOTH);
   }
   else if (strcasecmp(c0, "bitmap"     ) == 0)
   {
      if ((cc > 1) && (!TxaOption('?')))
      {
         sn   = dfsGetSymbolicSN(   c1, 0);
         size = dfsGetSymbolicSize( c2, 0, DFSTORE, sn);
         if (size)
         {
            if (size < RBUFS)
            {
               size *= dfsGetSectorSize();
            }
         }
         else
         {
            size = dfsGetSectorSize() * 4;      // default bitmap size
         }
         if (TxaOption('d'))                    // DIR format
         {
            dfsHpfsMapSector(sn, size, 0, 4, 1, BLBOTH);
         }
         else                                   // alloc format
         {
            dfsHpfsMapSector(sn, size, sn & 0xffffc000, 1, 2, BLBOTH);
         }
      }
      else                                      // no arguments, default
      {
         TxPrint("Usage: %s bitmap-LSN [nr of sectors]  [-dirformat]\n", c0);
      }
   }
   else if ((strcasecmp(c0, "delfind"  ) == 0) ||
            (strcasecmp(c0, "filefind" ) == 0)  )
   {
      BOOL             deleted  = (strcasecmp(c0, "delfind"  ) == 0);
      USHORT           findId;

      if (TxaOption('?'))
      {
         TxPrint("\nFind (deleted) files by direct searching the file-fnodes\n");
         TxPrint("\n Usage:  %s  [name]  [-a] [-D | -f] [-c] [-d-] [-v]\n\n", c0);
         TxPrint("   name      = part of filename wanted, not a true wildcard,\n"
                 "               but may start and end in a '*' character\n"
                 "   -a        = search ANY/ALL files, deleted and normal\n"
                 "   -D        = search  directories  only, not files+directories\n"
                 "   -f        = search regular files only, not files+directories\n"
                 "   -c        = start from current sector, not start of volume\n"
                 "   -d-       = search outside known FNODE areas too   (SLOW!)\n"
                 "   -v        = verbose search, list files while found (SLOW!)\n");
      }
      else
      {
         if (TxaOption( 'a'))
         {
            findId = FNODES_ALL;
         }
         else
         {
            findId = (deleted) ? FNODES_DEL : FNODES_REF;
         }

         if ((dfsa->findSpace.space == NULL) || // no FNODE areas determined yet
             (dfsa->findSpace.spid  != findId)) // or not the right type cached
         {
            TxFreeMem( dfsa->findSpace.space);  // free the FNODE FIND SPACE, if any

            switch (findId)
            {
               case FNODES_DEL:
                  //- Create the FINDSPACE ('find -o:S' etc) from brute-fore FNODE search
                  if (dfsHpfsDelFnodes2Space( TRUE, &dfsa->findSpace.chunks, &dfsa->findSpace.space) == NO_ERROR)
                  {
                     TxPrint( "Del.  #Files+Dirs : %-8u   (Fnodes for DELETED files/directories)\n",
                               dfsSspaceSectors( TRUE, dfsa->findSpace.chunks, dfsa->findSpace.space));
                  }
                  break;

               case FNODES_ALL:
                  //- Create the FINDSPACE ('find -o:S' etc) from brute-fore FNODE search
                  if (dfsHpfsDelFnodes2Space( FALSE, &dfsa->findSpace.chunks, &dfsa->findSpace.space) == NO_ERROR)
                  {
                     TxPrint( "Total #Files+Dirs : %-8u   (Fnodes found anywhere on the volume)\n",
                               dfsSspaceSectors( TRUE, dfsa->findSpace.chunks, dfsa->findSpace.space));
                  }
                  break;

               default:
                  //- Create the FINDSPACE structure (for 'find -o:S' etc) from directory-tree
                  if (dfsHpfsRefFnodes2Space( FALSE, &dfsa->findSpace.chunks, &dfsa->findSpace.space) == NO_ERROR)
                  {
                     TxPrint( "Total #Files+Dirs : %-8u   (Referenced from existing directories)\n",
                               dfsSspaceSectors( TRUE, dfsa->findSpace.chunks, dfsa->findSpace.space));
                  }
                  break;
            }
            dfsa->findSpace.spid = findId;
            TxCancelAbort();                    // reset pending abort status, allow partial SPACE
         }
         else
         {
            TxPrint( "Total #Files+Dirs : %-8u   (Cached SPACE from identical searchtype performed)\n",
                      dfsSspaceSectors( TRUE, dfsa->findSpace.chunks, dfsa->findSpace.space));
         }
         if ((dfsa->findSpace.chunks > 0) || (TxaOptUnSet('d'))) // findSpace or slow search
         {
            TXTS       findtype;                // sector types to be searched

            if      (TxaOption('D'))            // Directories only
            {
               strcpy( findtype, TxaOption('a') ? "Dy" : (deleted)  ? "y" :  "D");
            }
            else if (TxaOption('f'))            // files only
            {
               strcpy( findtype, TxaOption('a') ? "fz" : (deleted)  ? "z" :  "f");
            }
            else                                // Files + Directories
            {
               strcpy( findtype, TxaOption('a') ? "fzDy" : (deleted)  ? "zy" :  "fD");
            }

            sprintf( dfsa->brdescript, "HPFS %s %s", c0, pp);
            sprintf( s1, "find -t:%s -o:*$%%%s!%s%c ", findtype,
                    (TxaOptUnSet('d')) ? "" : "S", // FNODES only (-o:S spaced)
                         (TxaOption('v')) ? "" : "Q", // verbose or quiet, FAST
                         (deleted)  ? '[' : ' '); // search freespace/anywhere
            if (cc > 1)                         // ascii wildcard search-string
            {
               TxStrip( s0, pp, '*', '*');      // remove leading/trailing *
               strcat(  s1, s0);                // add search string-value
            }
            strcpy( dc, s1);                    // copy complete command
            TxPrint("Find %s %s: HPFS specific, using: %s%s%s\n",
                     TxaOption('a') ? "any/all" : (deleted) ? "deleted" : "normal",
                     TxaOption('D') ? "DIRs only" : "Files+Dirs", CBC, dc, CNN);
            if (!TxaOption('c'))
            {
               nav.this = LSN_RESRV;            // search from first usable ...
            }
            TxaReParseCommand( dc);
            rc = DFS_PENDING;                   // handle translated command
         }
         else
         {
            TxPrint("No relevant Fnode's are present, '%s' aborted\n", c0);
         }
      }
   }
   else
   {
      rc = DFS_CMD_UNKNOWN;                     // cmd not recognized
   }
   RETURN (rc);
}                                               // end 'dfsHpfsCommand'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// HPFS filesystem, identify specified sector as a valid superblock
/*****************************************************************************/
BOOL dfsHpfsIsSuperBlock                        // RET   sector is a valid sb
(
   BYTE               *sec                      // IN    sector contents
)
{
   return (!memcmp(((S_SUPER*)sec)->Signature,sg_super,SG_SUPER));
}                                               // end 'dfsHpfsIsSuperBlock'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// HPFS filesystem, display sector constents as a superblock
/*****************************************************************************/
void dfsHpfsDisplaySuperBlock
(
   BYTE               *sec                      // IN    sector contents
)
{
   dfsHpfsSuperSec( sec, FALSE);
}                                               // end 'dfsHpfsDisplaySuperBlock'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// HPFS filesystem, identify specified sector
/*****************************************************************************/
static ULONG dfsHpfsIdent
(
   ULN64               sn64,                    // IN    LSN for sector
   ULN64               d2,                      // IN    dummy
   char               *st,                      // OUT   sector type
   void               *sec                      // IN    sector contents
)
{
   ULONG               dr  = NO_ERROR;
   ULONG               lsn = (ULONG) sn64;
   BYTE                rc  = ST_UDATA;
   S_FNODE            *fn  = (S_FNODE*) sec;
   S_SPARE            *sp  = (S_SPARE*) sec;

   ENTER();

   if      (!memcmp(((S_FNODE*)sec)->Signature,sg_fnode,SG_FNODE)) rc=ST_FNODE;
   else if (!memcmp(((S_FNODE*)sec)->Signature,sg_fndel,SG_FNDEL)) rc=ST_FNDEL;
   else if (!memcmp(((S_FNODE*)sec)->Signature,sg_fd386,SG_FD386)) rc=ST_FNDEL;
   else if (!memcmp(((S_SUPER*)sec)->Signature,sg_super,SG_SUPER)) rc=ST_SUPER;
   else if (!memcmp(((S_SPARE*)sec)->Signature,sg_spare,SG_SPARE)) rc=ST_SPARE;
   else if (!memcmp(((S_DIRBL*)sec)->Signature,sg_dirbl,SG_DIRBL)) rc=ST_DIRBL;
   else if (!memcmp(((S_ALLOC*)sec)->Signature,sg_alloc,SG_ALLOC)) rc=ST_ALLOC;
   else if (!memcmp(((S_CPINF*)sec)->Signature,sg_cpinf,SG_CPINF)) rc=ST_CPINF;
   else if (!memcmp(((S_CPDAT*)sec)->Signature,sg_cpdat,SG_CPDAT)) rc=ST_CPDAT;
   else if (!memcmp(((S_UITAB*)sec)->Signature,sg_uitab,SG_UITAB)) rc=ST_UITAB;
   else if ((lsn != LSN_BOOTR)  &&   (hpfs->sup != NULL)   &&
            (lsn == hpfs->sup->BitmapInd))                         rc=ST_BMAPI;
   switch (rc)
   {
      case ST_UDATA:
         dr = DFS_PENDING;
         break;

      case ST_FNDEL:                            // possible deleted FNODE/ALLOC
         if ((fn->DirFlag              == 1) || // not a file but Directory
             (fn->NameLength           == 0) || // name too short
             (fn->Name[0]              < 32) || // invalid 1st char
             (fn->ParentDir <=    LSN_RESRV) || // no ptr to Dir or too large
             (fn->ParentDir >  hpfs->sup->TotalSec) ||
            ((fn->AlInfo.FreeEntries   +
              fn->AlInfo.UsedEntries)  != 8) || // Alloc free/used incorrect
             (fn->AlInfo.OfsFirstFree   < 8) || // Alloc first incorrect
             (fn->Al.Leaf[0].DataBlock == 0)  ) // no allocated data area
         {
            if ((fn->DirFlag              == 0) || // not a Directory
                (fn->NameLength           == 0) || // name too short
                (fn->Name[0]              < 32) || // invalid 1st char
                (fn->FileLength           != 0) || // not length zero (data)
                (fn->ParentDir    <= LSN_RESRV) || // no ptr to Dir or too large
                (fn->ParentDir            > hpfs->sup->TotalSec) ||
                (fn->AlInfo.FreeEntries   != 0) || // Alloc-info for a file
                (fn->AlInfo.OfsFirstFree  != 0) || // Alloc-info for a file
                (fn->Al.Leaf[0].DataBlock == 0) || // no dir-block or too large
                (fn->Al.Leaf[0].DataBlock > hpfs->sup->TotalSec))
            {
               S_ALLOC   *al = (S_ALLOC*) sec;

               if (lsn && al->Parent && (lsn == al->This))
               {
                  rc = ST_ALDEL;
               }
               else                             // deleted Alloc sector
               {
                  rc = ST_UDATA;
                  dr = DFS_PENDING;
               }
            }
            else                                // Deleted Fnode for Directory
            {
               rc = ST_FNDDI;
            }
         }
         else                                   // Deleted Fnode for file
         {                                      // check Max-LSN boundary
            if (fn->Al.Leaf[0].RelSecData == 0) // Leaf
            {
               if (fn->Al.Leaf[0].DataBlock > hpfs->sup->TotalSec)
               {
                  rc = ST_UDATA;
                  dr = DFS_PENDING;
               }
            }
            else                                // Node
            {
               if (fn->Al.Node[0].ChildBlock > hpfs->sup->TotalSec)
               {
                  rc = ST_UDATA;
                  dr = DFS_PENDING;
               }
            }
         }
         break;

      case ST_FNODE:
         if (fn->DirFlag == 1)
         {
            rc = ST_FNDIR;                      // directory Fnode
         }
         break;

      case ST_SUPER:
         if ((((S_SUPER*)sec)->VerMajor > 3) || (((S_SUPER*)sec)->VerMinor > 5))
         {
            rc = ST_UDATA;
            dr = DFS_PENDING;
         }
         break;

      case ST_SPARE:
         if ((sp->HotFixMax   > 512) || (sp->HotFixes     > sp->HotFixMax  ) ||
             (sp->SpareDirMax > 512) || (sp->SpareDirUsed > sp->SpareDirMax) ||
             (sp->CodePages   >   9) )
         {
            rc = ST_UDATA;
            dr = DFS_PENDING;
         }
         break;

      case ST_CPINF:
         hpfs->cpinf  = lsn;                     // remember CodePage data
         hpfs->cpages = ((S_CPINF*)sec)->CpCount;
      default:
         break;
   }
   *st = rc;
   RETURN (dr);
}                                               // end 'dfsHpfsIdent'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// HPFS filesystem, supply sector-type description string
/*****************************************************************************/
static ULONG dfsHpfsStype
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *st,                      // IN    sector type
   void               *data                     // OUT   type description
)
{
   ULONG               rc  = NO_ERROR;
   char               *buf = (char *) data;

   switch (*st)                                 // searchable types
   {
      case ST_SUPER: sprintf(buf,"HPFS superblock  "); break;
      case ST_SPARE: sprintf(buf,"HPFS spareblock  "); break;
      case ST_FNODE: sprintf(buf,"File        Fnode"); break;
      case ST_FNDEL: sprintf(buf,"Deleted FileFnode"); break;
      case ST_FNDDI: sprintf(buf,"Deleted Dir Fnode"); break;
      case ST_FNDIR: sprintf(buf,"Directory   Fnode"); break;
      case ST_DIRBL: sprintf(buf,"Directory   block"); break;
      case ST_ALLOC: sprintf(buf,"Allocation  block"); break;
      case ST_ALDEL: sprintf(buf,"Deleted All-block"); break;
      case ST_CPINF: sprintf(buf,"Codepage info    "); break;
      case ST_CPDAT: sprintf(buf,"Codepage data    "); break;
      case ST_BMAPI: sprintf(buf,"Bitmap Tables    "); break;
      default:
         switch ((*st) | ST__INFO)              // non-searchable ones
         {
            case ST_EDATA: sprintf(buf,"EA   data        "); break;
            case ST_ADATA: sprintf(buf,"ACL  data        "); break;
            case ST_BOOTA: sprintf(buf,"HPFS Micro-FSD   "); break;
            case ST_HOTFT: sprintf(buf,"Hotfix table     "); break;
            case ST_HOTFD: sprintf(buf,"Hotfix data      "); break;
            case ST_SPDIR: sprintf(buf,"Spare dirblocks  "); break;
            case ST_BADSL: sprintf(buf,"Bad-sector list  "); break;
            case ST_DBAND: sprintf(buf,"Dir-band (free)  "); break;
            case ST_DBMAP: sprintf(buf,"Dir-band bitmap  "); break;
            case ST_UITAB: sprintf(buf,"HPFS386 User-id  "); break;
            case ST_BMAPD: sprintf(buf,"Bitmap data      "); break;
            default:       rc = DFS_PENDING;
         }
         break;
   }
   return (rc);
}                                               // end 'dfsHpfsStype'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// HPFS filesystem, display sector-contents based on type
/*****************************************************************************/
static ULONG dfsHpfsDispl
(
   ULN64               psn,                     // IN    base psn for sector
   ULN64               d2,                      // IN    dummy
   char               *type,                    // IN    type of sector
   void               *dummy                    // IN    sector contents
)
{
   ULONG               rc = NO_ERROR;
   BYTE                st = (BYTE) *type;
   ULONG               lsn = dfstPsn2LSN( DFSTORE, psn);

   ENTER();

   switch (st)
   {
      case ST_FNDIR:
      case ST_FNDDI:
      case ST_FNDEL:
      case ST_FNODE: (void) dfsHpfsFnodeSec(rbuf, st, lsn);  break;
      case ST_SUPER: (void) dfsHpfsSuperSec(rbuf, TRUE);     break;
      case ST_SPARE: (void) dfsHpfsSpareSec(rbuf);           break;
      case ST_ALDEL:
      case ST_ALLOC: (void) dfsHpfsAllocSec(rbuf, lsn);      break;
      case ST_CPINF: (void) dfsHpfsCpInf(rbuf, lsn);         break;
      case ST_CPDAT: (void) dfsHpfsCpDat(rbuf, lsn);         break;
      case ST_UITAB: (void) dfsHpfsUiTab(rbuf, lsn);         break;

      case ST_DIRBL:
         rc = dfstReadPsn( DFSTORE, psn+1, 3, rbuf + dfsGetSectorSize()); // rest dirblock
         (void) dfsHpfsDirblock(rbuf, lsn);
         break;

      case ST_BMAPI:
         rc = dfstReadPsn( DFSTORE, psn+1, 3, rbuf + dfsGetSectorSize()); // rest bitmap
         (void) dfsHpfsAllocBitmap(rbuf, DL_DEFAULT);
         break;

      default:
         rc = DFS_PENDING;
         break;
   }
   RETURN (rc);
}                                               // end 'dfsHpfsDispl'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read and display the allocation bitmaps
/*****************************************************************************/
static ULONG dfsHpfsAllocMap
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *options,                 // IN    display options
   void               *data                     // OUT   dummy
)
{
   ULONG               i;                       // index in data-area
   ULONG               b;                       // byte-index for one mapchar
   ULONG               cClus;                   // Clusters in current char
   LLONG               lClus = 0;               // Clusters in current line
   ULONG               mClus = 0;               // Clusters in current map
   ULONG               uClus = 1;               // Last cluster that is in use
   ULONG               l;                       // line number, 0 based
   ULONG               a;                       // index in display-array
   ULONG               perc;                    // percentage
   TX1K                ascii;                   // display-array
   ULONG               spe;                     // sectors per entry
   ULONG               acl;                     // ascii alloc chars per line
   ULONG               cpc;                     // clusters per display-char
   ULONG               spc;                     // sectors per display-char
   ULONG               size;                    // nr of clusters to map
   ULONG               value;                   // display detail
   BOOL                verbose = (*options != '@');
   ULONG               bsSmart;
   ULN64               unallocSmart = 0;        // Size of THIS unallocated area
   BOOL                oneLineProgress = FALSE; // alloc in one line, output per character

   ENTER();

   dfsa->FsUnallocSmart = 0;                    // will be recalculated now
   bsSmart = dfsGetBufferSize( DFSOPTIMALBUF, DFSMAXBUFSIZE); // for SMART prediction

   spe   = 1;                                   // one sector per cluster
   size  = hpfs->sup->TotalSec;                 // 0 .. last-valid

   acl   = dfsAllocCharsPerLine( 'c');
   cpc   = dfsAllocItemsPerChar( 'l', options, size, acl);
   if (cpc >= (size / acl))                     // will be a single line
   {
      oneLineProgress = TRUE;                   // one char at a time, no ascii string used
   }
   spc   = cpc * spe;
   dfsProgressInit( 0, size * spe, 0, "Get ALLOCATION info, at Sector:", dfstStoreDpid( DFSTORE), DFSP_STAT, 0);

   if (verbose)
   {
      dfsSz64("Size for one line : ", (ULN64) acl * spc, "  ");
      TxPrint("with %3lu characters,", acl);
      dfsSiz8(" size : ", spc, "\n");
      TxPrint(" %16.16s = Allocation grade, empty to full.     ", mapchar);
      dfsSiz8(" SmartBuf size : ", bsSmart, "\n");
   }
   TxPrint(   "          %s%*.*s%s\n", CBC, acl, acl, BLIN, CNN);
   dfsX10( CNZ, 0, CNZ, "");                    // display address for first block
   TxPrint("%s%s", CBC, CNC);

   for (i=0, a=0, l=0; (i < size) && (!TxAbort());)
   {
      if (a == 0)
      {
         memset(ascii, 0, TXMAXLN);
         lClus = 0;
      }
      for (cClus=0, b=0; (b < cpc) && (i < size); b++, i++)
      {
         if (dfsHpfsAllocated( i, 0, NULL, NULL))  // cluster in use ?
         {
            if (unallocSmart != 0)              // first in-use after unallocated?
            {
               if (unallocSmart >= bsSmart)                  //- if at least ONE smart buffer
               {
                  unallocSmart -= (bsSmart / 2);             //- compensate for alignment issues
                  unallocSmart -= (unallocSmart % bsSmart);  //- clip to whole smart buffer size

                  dfsa->FsUnallocSmart += unallocSmart;      //- add this area to total
               }
               unallocSmart = 0;                // reset for next area to come
            }
            cClus++;
            if (!dfsInBitMapArea( i))           // not a band bitmap sector ?
            {
               uClus = i;                       // last in-use one sofar
            }
         }
         else                                   // unallocated, Smart predict
         {
            unallocSmart += spe;                // add to size THIS area
         }
      }
      lClus += cClus;

      dfsProgressShow( (i * spe) + spc, 0, 0, NULL);

      TRACES(("lClus 2nd Loop i: %u, b:%u = %u\n", i, b, lClus));
      if (oneLineProgress == TRUE)
      {
         TxPrint( "%c", (char) ((cClus == 0) ? ' ' : mapchar[cClus * 8 / cpc])); // slow, acts as progress bar
      }
      else
      {
         ascii[a] = (char) ((cClus == 0) ? ' ' : mapchar[cClus * 8 / cpc]);
      }
      a++;
      if ((i && ((i%(acl*cpc)) == 0)) || (i >= size))
      {
         if (oneLineProgress == FALSE)
         {
            TxPrint( "%s", ascii);              // display accumulated chars (fast)
         }
         perc  = (ULONG)(100*lClus / (a*cpc));
         if (a == acl)
         {
            TxPrint("%s%s% 3lu%%\n", CBC, CNN, perc);
         }
         else
         {
            TxPrint("%s%.*s%s% 3lu%%\n",
                    CBC, (int) (acl-a-1), BLIN, CNN, perc);
         }
         if (i < size)
         {
            dfsX10( CNZ, i, CNZ, "");
            TxPrint("%s%s", CBC, CNC);
            a = 0;                              // keep a value on last line
         }
         mClus += lClus;
         l++;
      }
   }
   if (unallocSmart != 0)                           //- unalloc area pending?
   {
      if (unallocSmart >= bsSmart)                  //- if at least ONE smart buffer
      {
         unallocSmart -= (bsSmart / 2);             //- compensate for alignment issues
         unallocSmart -= (unallocSmart % bsSmart);  //- clip to whole smart buffer size

         dfsa->FsUnallocSmart += unallocSmart;      //- add this area to total
      }
   }
   TxPrint("          %s%.*s%s\n", CBC, (USHORT) a, BLIN, CNN);

   if (TxAbort())
   {
      dfsa->FsUnallocSmart = 0;                 // signal info NOT available!
      TxPrint( "\nAllocation check and display aborted! Used/Free info not available.\n");
   }
   else
   {
      if (size != 0)                            // avoid devide by 0
      {
         ULONG            used = mClus * spe;

         dfsa->FsUnallocated   = size * spe - used; // free space (optimize hint)

         dfsSz64("Unallocated sects : ", dfsa->FsUnallocated, " ");
         dfsSz64("SmartUse ", dfsa->FsUnallocSmart, " ");
         TxPrint("=% 5.1lf%%\n", (double) (100.0 * ((double) dfsa->FsUnallocSmart) / (double) (size * spe)));

         dfsSz64("Allocated sectors : ", used, " ");
         dfsSz64("of ", size * spe, " ");
         TxPrint("=% 5.1lf%%\n", (double) (100.0 * ((double) mClus) / (double) size));
      }
      dfsa->FsLastInUse  = uClus;
      value = hpfs->bmt[ uClus >> 14];          // lsn of corresponding bitmap
      TRACES(( "Last used sector : %8.8X, bitmap-position : %8.8X\n", uClus, value));
      if ((value > uClus) && (value < size))    // bitmap AFTER last cluster!
      {
         dfsa->FsTruncPoint = value + 4;        // first sector after bitmap
         TRACES(( "Bitmap AFTER uClus  for index: %u\n", (uClus >> 14)));
      }
      else                                      // bitmap is BEFORE ...
      {
         if (value >= size)
         {
            TxPrint( "WARNING bitmap at : 0x%8.8X is in an invalid position!\n", value);
         }
         dfsa->FsTruncPoint = uClus + 1;        // first sector after uClus
         TRACES(( "Bitmap BEFORE uClus for index: %u\n", (uClus >> 14)));
      }
      dfsSz64( "Minimum vol. size : ", dfsa->FsTruncPoint, " (for  Resizing)\n");
   }
   dfsProgressTerm();
   RETURN (NO_ERROR);
}                                               // end 'dfsHpfsAllocMap'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read the bitmap indirection table and connect to disk structure
// Note: Area aware to allow usage from FDISK mode too
/*****************************************************************************/
static ULONG dfsHpfsReadBmTable
(
   void
)
{
   ULONG               rc = NO_ERROR;
   ULONG               lsn = hpfs->sup->BitmapInd;
   ULONG               bis;                     // nr of bitmapInd sectors
   BYTE               *bmt = NULL;              // BitMap table

   ENTER();
   if (hpfs->bmt != NULL)
   {
      TxFreeMem(hpfs->bmt);
      hpfs->bmt = NULL;
   }
   bis = (((hpfs->sup->TotalSec >> 14) +2) * sizeof(ULONG) / dfsGetSectorSize()) +1;

   TRACES(("BitmapTable size in sectors: %u, bps:%hu\n", bis, dfsGetSectorSize()));
   if ((bmt = TxAlloc(bis, dfsGetSectorSize())) != NULL)
   {
      TRACES(("BitmapTable LSN: %8.8x, buffer: %8.8x\n", lsn, bmt));
      rc = dfsRead( dfstAreaP2Disk( DFSTORE, lsn), bis, bmt);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   hpfs->bmt = (ULONG *) bmt;
   RETURN (rc);
}                                               // end 'dfsHpfsReadBmTable'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// Read and display the allocation bitmaps
/*****************************************************************************/
static ULONG dfsHpfsReadBitmap
(
   USHORT             dlvl                      // IN    Detail level
)
{
   ULONG              lsn = hpfs->sup->BitmapInd;

   ENTER();
   if (hpfs->bmt != NULL)
   {
      nav.this = lsn;
      nav.down = *( hpfs->bmt);         // first bitmap sector
      TxPrint("Shown sector  LSN : 0x%s%08.8X%s = %s", CBM, lsn, CNN, CBM);
      TxPrint("HPFS Bitmap indirection table%s\n", CNN);
      (void) dfsHpfsAllocBitmap((BYTE   *) hpfs->bmt, dlvl);
   }
   RETURN (0);
}                                               // end 'dfsHpfsReadBitmap'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display HPFS allocation bitmap sector
/*****************************************************************************/
static ULONG dfsHpfsAllocBitmap                 // RET   rc = 0 if type match
(
   BYTE               *sector,                  // IN    Sector data
   USHORT              dlvl                     // IN    Detail level
)
{
   ULONG               rc = 0;                  // rc, sector match
   ULONG              *mp;                      // LSN array, equals 4 sectors
   ULONG               bl;                      // base LSN for bitmap
   ULONG               mapsize = SECTORSIZE*4;  // standard bitmap size (2KB)
   ULONG               endband;                 // base LSN of last band
   USHORT              frame = BLBOTH;
   ULONG               used  = 0;               // nr of SET bits in bitmap
   ULONG               total = 0;               // total nr of bits in bitmap

   ENTER();
   endband = (hpfs->sup->TotalSec) & 0xffffc000;  // mask off lower 14 bits

   for (mp = (ULONG *) sector; (*mp != 0) && !TxAbort(); mp++)
   {
      bl = *mp & 0xffffc000;                    // reset to start of 8MB block
      if (bl == endband)
      {
         mapsize = ((hpfs->sup->TotalSec) & 0x3fff) / 8;
      }
      if (dlvl <= DL_DEFAULT)                   // all bands in single block
      {
         frame = BLNO;
         if (bl == 0)                           // first band
         {
            frame |= BLTOP;
         }
         if (*(mp+1) == 0)                      // last band
         {
            frame |= BLEND;
         }
         total += (mapsize * 8);
         used  += dfsHpfsMapSector(*mp, mapsize, bl, 1, 32,  frame);
      }
      else                                      // full format for each band
      {
         TxPrint("\n");
         total += mapsize * 8;
         used  += dfsHpfsMapSector(*mp, mapsize, bl, 1, 2, BLBOTH);
      }
   }
   dfsSiz8("Total  size:", total, "  ");
   if (total != 0)                              // avoid devide by 0
   {
      dfsSiz8("Used: ", used, "        = ");
      TxPrint("% 3lu%%\n", (100 * used / total));
   }
   RETURN (rc);
}                                               // end 'dfsHpfsAllocBitmap'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Show current sector contents as a bitmap
/*****************************************************************************/
static ULONG dfsHpfsMapSector                   // RET   nr of 'SET' bits
(
   ULONG              sn,                       // IN    lsn of sector
   ULONG              size,                     // IN    Size to display
   ULONG              bsn,                      // IN    base lsn mapped area
   ULONG              spb,                      // IN    sectors per bit
   ULONG              bpc,                      // IN    bytes per display-char
   USHORT             framing                   // IN    framing specification
)
{
   ULONG              sb = 0;

   ENTER();
   if ((dfsRead(sn, (size / dfsGetSectorSize()) +1, rbuf)) == NO_ERROR)
   {
      sb = dfsHpfsBitmapArea( rbuf, size, sn, bsn, spb, bpc, framing);
   }
   TRARGS(("Nr of bits set (used): %u\n", sb));
   RETURN (sb);
}                                               // end 'dfsHpfsMapSector'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Make Bitmap dump of data area on TxPrint output
/*****************************************************************************/
static ULONG dfsHpfsBitmapArea                  // RET   nr of 'SET' bits
(
   BYTE              *data,                     // IN    data area
   ULONG              size,                     // IN    size to dump
   ULONG              first,                    // IN    LSN of first 'bit'
   ULONG              bsn,                      // IN    base lsn mapped area
   ULONG              spb,                      // IN    sectors per bit
   ULONG              bpc,                      // IN    bytes per display-char
   USHORT             framing                   // IN    framing specification
)
{
   ULONG               i;                       // index in data-area
   ULONG               b;                       // byte-index for one mapchar
   ULONG               cbits = 0;               // bits in current char
   ULONG               lbits = 0;               // bits in current line
   ULONG               mbits = 0;               // bits in current map
   ULONG               l;                       // line number, 0 based
   ULONG               a;                       // index in display-array
   ULONG               perc;                    // percentage
   TXTM                ascii;                   // display-array
   ULONG               lbase = 0;               // current line base LSN
   ULONG               cbase;                   // current char base LSN
   ULONG               clast;                   // current char last LSN

   ENTER();
   if (framing & BLTOP)
   {
      TxPrint("Bitmap sector LSN : 0x%08.8X", first);
      dfsSiz8("   Sectors per char: ", spb*bpc*8, "\n");
      TxPrint(" %16.16s = Allocation grade, empty upto fully occupied\n", mapchar);
      if (spb == 1)
      {
         TxPrint("         %-8.8s = HPFS system sectors\n", "S");
         TxPrint("         %-8.8s = Directory band area\n", "D");
      }
      TxPrint(   "        %s%s%s\n", CBC, BLIN, CNN);
   }
   for (i=0, a=0, l=0; (i < size) && (!TxAbort());)
   {
      if (a == 0)
      {
         memset(ascii, 0, TXMAXTM);
         lbits = 0;
         lbase = bsn +(l*spb*64*bpc*8);
      }
      for (cbits = 0, b = 0; (b < bpc) && (i < size); b++)
      {
         cbits += (8 - dfsHpfsBitCount(data[i++]));
      }
      lbits += cbits;
      cbase = lbase + ( a   *bpc*8);            // base LSN for this character
      clast = lbase + ((a+1)*bpc*8);            // last LSN for this character

      if ((spb == 1) &&
          (cbase < hpfs->spr->CodePageInfo))
      {
         ascii[a] = 'S';                        // HPFS system sector(s)
      }
      else if ((spb == 1) &&
               (clast > hpfs->sup->DirBand)    &&
               (cbase < hpfs->sup->DirBandEnd) )
      {
         ascii[a] = 'D';                        // HPFS dirband sector(s)
      }
      else                                      // other sectors, show usage
      {
         ascii[a] = (char) ((cbits == 0) ? ' ' : mapchar[cbits / bpc]);
      }
      a++;
      if ((i && ((i%(64*bpc)) == 0)) || (i >= size))
      {
         perc  = (ULONG)(100*lbits / (a*bpc*8));
         TxPrint("%08.08X%s%s%s%s", lbase, CBC, CNC, ascii, CNN);
         if (a == 64)
         {
            TxPrint("%s%s% 3lu%%\n", CBC, CNN, perc);
         }
         else
         {
            TxPrint("%s%.*s%s% 3lu%%\n",
                    CBC, (USHORT) (64-a-1), BLIN, CNN, perc);
         }
         if (i < size)
         {
            a = 0;                              // keep a value on last line
         }
         mbits += lbits;
         l++;
      }
   }
   if (framing & BLEND)
   {
      TxPrint("        %s%.*s%s\n", CBC, (USHORT) a, BLIN, CNN);
      dfsSiz8("Size 1-line:", 64*bpc*spb*8, "\n");
      if (framing & BLTOP)
      {
         dfsSiz8("Bitmap size:", size*spb*8, "  ");
         if (size != 0)                         // avoid devide by 0
         {
            dfsSiz8("Used: ", mbits * spb, "        = ");
            TxPrint("% 3lu%%\n", (100 * mbits / (size*8)));
         }
      }
   }
   RETURN (mbits);
}                                               // end 'dfsHpfsBitmapArea'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display HPFS super-block
/*****************************************************************************/
static ULONG dfsHpfsSuperSec                    // RET   rc = 0 if type match
(
   BYTE               *sector,                  // IN    superblock sector
   BOOL                navigation               // IN    update nav values
)
{
   ULONG               rc = 0;                  // rc, sector match
   S_SUPER            *sd = (S_SUPER *) sector;
   time_t              tm;                      // c-runtime time_t format

   ENTER();
   TxPrint("HPFS FileSys vers : %u.%u        = ", sd->VerMajor, sd->VerMinor);
   if ((sd->VerMajor > 2) || (sd->VerMinor > 3))
   {
      TxPrint("%sNew HPFS version!%s, DFSee update time!\n", CBR, CNN);
   }
   else
   {
      switch (sd->VerMinor)
      {
         case 0:
         case 1:
         case 2:
            TxPrint("Classic HPFS\n");
            break;

         case 3:
            TxPrint("Volume > 2 GiB\n");
            break;

         default:
            TxPrint("HPFS386\n");
            break;
      }
   }
   dfsX10( "ROOT dir      LSN : ",              sd->RootFnode,    CBG, "\n");
   TxPrint("Dir band Map  LSN : 0x%08.8X\n",    sd->DirBandMap);
   TxPrint("Dir band      LSN : 0x%08.8X\n",    sd->DirBand);
   TxPrint("Dir band End  LSN : 0x%08.8X\n",    sd->DirBandEnd);
   dfsSiz8("Dir band size     : ",              sd->DirBandSec,        "\n");
   dfsX10( "Bitmap Indir. LSN : ",              sd->BitmapInd,    CBY, "\n");
   dfsSiz8("Total sectors     : ",              sd->TotalSec,          "\n");
   dfsSiz8("Bad sectors       : ",              sd->BadSec,            "\n");
   TxPrint("Bad list      LSN : 0x%08.8X\n",    sd->BadList);
   tm = (time_t)                                sd->ChkdskDate;
   TxPrint("Last Chkdsk  (/f) : %-24.24s\n",   (sd->ChkdskDate == 0) ?
                                                      "Never" : ctime( &tm));
   tm = (time_t)                                sd->OptimDate;
   TxPrint("Last Optimize     : %-24.24s\n",   (sd->OptimDate  == 0) ?
                                                      "Never" : ctime( &tm));
   TxPrint("VolumeName HPFS   : %-32.32s\n",    sd->VolumeName);
   TxPrint("UserIdTable   LSN : 0x%08.8X\n",    sd->UserIdTable);

   if (navigation)
   {
      nav.down = sd->RootFnode;
      nav.xtra = sd->UserIdTable;
   }
   RETURN (rc);
}                                               // end 'dfsHpfsSuperSec'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display HPFS spare-block
/*****************************************************************************/
static ULONG dfsHpfsSpareSec                    // RET   rc = 0 if type match
(
   BYTE               *sector                   // IN    Drive specification
)
{
   ULONG               rc = 0;                  // rc, sector match
   S_SPARE            *sd = (S_SPARE *) sector;
   BYTE                st = sd->StatusFlag;
   BYTE                s3 = sd->HPFS386Flag;
   ULONG               l;
   TXTM                text;
   TXTM                tbuf;

   ENTER();
   TxPrint("Dirty bit         : %s\n", (st & HPFS_SP_DIRTY) ? "Dirty" : "Clean");

   if (st & HPFS_SP_FASTFMT) TxPrint("Fast Format       : %s\n", "Yes");
   if (st & HPFS_SP_SPARDIR) TxPrint("SpareDir used     : %s\n", "Yes");
   if (st & HPFS_SP_HOTFIXS) TxPrint("Hotfixes used     : %s\n", "Yes");
   if (st & HPFS_SP_CORRUPT) TxPrint("Volume Corrupt    : %s\n", "Yes");
   if (st & HPFS_SP_BADBITM) TxPrint("Bad Bitmap        : %s\n", "Yes");
   if (st & HPFS_SP_RESERVE) TxPrint("Reserved bit      : %s\n", "Set");
   if (st & HPFS_SP_OLDUTIL) TxPrint("Old HPFS/UTIL used: %s\n", "Yes");

   if (s3 & 0x01) TxPrint("Install DASD lim. : %s\n", "Yes");
   if (s3 & 0x02) TxPrint("Resync. DASD info : %s\n", "Yes");
   if (s3 & 0x04)
   {
      TxPrint("DASD limit active : %s\n", "Yes");
      TxPrint("DASD limits state : %s\n", (s3 & 0x20) ? "Dirty" : "Clean");
   }
   if (s3 & 0x10) TxPrint("DCE ACL's  active : %s\n", "Yes");
   if (s3 & 0x08)
   {
      TxPrint("Multimedia active : %s\n", "Yes");
      TxPrint("MM contiguity fac : %u\n", (USHORT) sd->MMContiguityFactor);
   }

   dfsSiz8("Hot fixes         : ",            sd->HotFixes,          "\n");
   dfsSiz8("Max Hotfixes      : ",            sd->HotFixMax,         "\n");
   dfsX10( "Hot list      LSN : ",            sd->HotFixList,   CBY, "\n");
   dfsDec8("Code pages        : ",            sd->CodePages,         "\n");
   dfsX10( "CodePage      LSN : ",            sd->CodePageInfo, CBG, "\n");
   TxPrint("Chksum SuperBlock : 0x%08.8X\n",  sd->SuChecksum);
   TxPrint("Chksum SpareBlock : 0x%08.8X\n",  sd->SpChecksum);
   dfsDec8("Max  SpareDirs    : ",            sd->SpareDirMax,       "\n");
   dfsDec8("Used SpareDirs    : ",            sd->SpareDirUsed,      "\n");

   strcpy( text, "SpareDirs sectors :");
   if (sd->SpareDirUsed < 128)
   {
      for (l = 0; (l < sd->SpareDirUsed) && !TxAbort(); l++)
      {
         if ((l % 8) == 0)
         {
            TxPrint("%s\n", text);
            strcpy( text, " ");
         }
         sprintf( tbuf, " %8.8X", sd->SpareDir[l]);
         strcat(  text, tbuf);
      }
      TxPrint("%s\n", text);
   }
   else
   {
      TxPrint( "\nNr of spare directories exceeds maximum!\n");
      rc = DFS_BAD_STRUCTURE;
   }
   if (sd->HotFixes != 0)
   {
      ULONG           *hft;                     // hotfix table in memory
      ULONG           *rep;                     // hotfix table in memory

      if ((hft = TxAlloc( 4, dfsGetSectorSize())) != NULL)
      {
         rep = &(hft[ (ULONG) min( sd->HotFixMax, 256L)]);
         rc = dfsRead( sd->HotFixList, 4, (BYTE   *) hft);
         if (rc == NO_ERROR)
         {
            strcpy( text, "\nHot-fix map table : Hot-fixed ==> replacement sectors :");
            for ( l = 0;
                 (l < sd->HotFixes) && (l < 256) && !TxAbort();
                  l++)
            {
               if ((l % 3) == 0)
               {
                  TxPrint("%s\n", text);
                  strcpy( text, " ");
               }
               sprintf( tbuf, " %8.8X ==> %8.8X   ",
                                hft[l],    rep[l] );
               strcat(  text, tbuf);
            }
            TxPrint("%s\n", text);
         }
         else
         {
            printf( "\nError reading Hotfix table from disk\n");
         }
         TxFreeMem( hft);
      }
      else
      {
         printf( "\nError allocating buffer memory\n");
         rc = DFS_ALLOC_ERROR;
      }
   }
   nav.down = sd->CodePageInfo;
   nav.xtra = sd->HotFixList;
   RETURN (rc);
}                                               // end 'dfsHpfsSpareSec'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display HPFS file-node
/*****************************************************************************/
static ULONG dfsHpfsFnodeSec                    // RET   rc = 0 if type match
(
   BYTE               *sector,                  // IN    Sector contents
   BYTE                st,                      // IN    sector type (Fnode)
   ULONG               lsn                      // IN    LSN of fnode
)
{
   ULONG               rc = 0;                  // rc, sector match
   S_FNODE            *sd = (S_FNODE *) sector;
   BYTE                af = sd->AlInfo.Flag;
   int                 al;                      // Allocation array items
   char               *cl = CBG;                // LSN color number
   int                 i;
   TXTM                name;                    // shortname this fnode
   TX1K                spath;                   // shortname-path
   BOOL                node;                    // fnode contains nodes
   BOOL                done = FALSE;            // stop-condition in display
   ULONG               RelSecThis = 0;
   ULONG               RelSecNext = 0;
   ULONG               size;
   ULONG               nr;
   ULONG               sl;                      // sector number (data)
   BOOL                multic = (sd->AlInfo.UsedEntries != 1);

   ENTER();

   //- Note: AlInfo Flag is not reliable, it is zapped when deleting a file
   node = (sd->Al.Leaf[0].RelSecData == 0) ? FALSE : TRUE;

   if (sd->NameLength > 0)
   {
      memset(name, '\0', TXMAXTM);
      memcpy(name, sd->Name, min(HPFS_FNAME_LEN,(size_t) sd->NameLength));
   }
   else if (nav.this == sd->ParentDir)
   {
      strcpy( name, "<empty>  = Root Directory");
   }
   TxPrint( "Fnode Name string : %s%s%s\n", CBC, name,  CNN );
   TxPrint( "Full  name length : %u\n", sd->NameLength);
   dfsHpfsFindRootFnode( lsn, FALSE, FALSE, spath, &nr);
   TxPrint( "Path from RootDIR : %s%s%s\n", CBY, spath, CNN);

   if (TxaExeSwitch('g'))
   {
      printf( "%s\n", spath);
      fflush( stdout);
   }

   if (hpfs->spr->HPFS386Flag & 0x04)           // DASD limits active
   {
      TxPrint("First DASD alert  : %u %%\n", (USHORT) sd->FirstDasdAlertThresh);
      TxPrint("Delta DASD alert  : %u %%\n", (USHORT) sd->DeltaDasdAlertThresh);
      dfsSiz8("DASD limit (sect) : ", sd->DasdLimit, "\n");
      dfsSiz8("DASD usage (sect) : ", sd->DasdUsage, "\n");
   }

   nav.up = sd->ParentDir;
   dfsX10( "Parent Dir    LSN : ",    sd->ParentDir,  CBC, "\n");

   TxPrint("Parent structure  : %s\n", (af & 0x20) ? "Fnode" : "Dir Entry");
   TRHEXS( 70,  sd,  0xc0, "FNODE start");
   if ((sd->DirFlag == 1) && (sd->Al.Directory.Zero == 0)) // Valid Directory
   {
      dfsX10( "DirBlock     LSN : ", sd->Al.Directory.DirBlock, cl, "");
      nav.down = sd->Al.Directory.DirBlock;
   }
   else                                         // FILE, limit maximum
   {
      al = sd->AlInfo.UsedEntries + sd->AlInfo.FreeEntries;
      TxPrint("Allocation type   : %s\n", (af & HPFS_AL_NODE) ? "Node"  : "Leaf");
      TxPrint("Free/used entries : %u / %u\n", (USHORT) sd->AlInfo.FreeEntries,
                                               (USHORT) sd->AlInfo.UsedEntries);
      if (node && (af == 0))                    // deleted fnode with Nodes
      {
         TxPrint("Real Alloc type   : %s\n", (node) ? "Node"  : "Leaf");
      }
      dfsDec8("Valid FileLength  : ", sd->FileLength, "");
      dfsSize(" = ",                 (sd->FileLength / dfsGetSectorSize()), "\n");
      TxPrint("%s : %u\n", (node) ?
              "AllocationSectors" :
              "Number of extents",    sd->AlInfo.UsedEntries);

      nr = size = 0;
      rc = dfsHpfsFnodeCheckAlloc( sd, "", st, &size, &nr, &sl);
      TxPrint( "%s : ", (st == ST_FNODE) ? "Saveto cmd (copy)"
                                         : "Undelete filedata");
      switch (rc)
      {
         case DFS_BAD_STRUCTURE:
            TxPrint("%sUnreliable%s  ", CBR, CNN);
            dfsSiz8("failing sectors : ", nr, "\n");
            break;

         case DFS_PSN_LIMIT:
            TxPrint("%sImpossible%s, sector nrs are too large!\n", CBR, CNN);
            break;

         default:
            TxPrint("%sPossible%s   (Allocation integrity is OK)\n", CBG, CNN);
            break;
      }
      if (node)
      {
         TxPrint(   "\n RelSecNext AllocLSN  Allocated-size      %s",
            (multic) ? "RelSecNext AllocLSN  Allocated-size" : "");
         TxPrint(   "\n ========== ========  ==============      %s",
            (multic) ? "========== ========  ==============" : "");
      }
      else                                      // LEAF, extents
      {
         TxPrint(   "\n RelSector  Data-LSN Extent-size          %s",
            (multic) ? "RelSector  Data-LSN Extent-size    " : "");
         TxPrint(   "\n =========  ======== ===============      %s",
            (multic) ? "=========  ======== ===============" : "");
      }

      //- AlInfo is not reliable in deleted fnode, used as upper limit only
      for (i = 0; (i < al) && (!done) && !TxAbort(); i++)
      {
         TxPrint("%s", (i % 2) ? "      " : "\n ");
         if (node)                              // NODE allocation block
         {
            RelSecNext = sd->Al.Node[i].RelSecNext;
            done = (RelSecNext == L32_NULL);
            if (done)
            {
               TxPrint("          ");
               RelSecNext = (sd->FileLength / dfsGetSectorSize()) +1;
            }
            else
            {
               TxPrint("Rs %06.6X ", RelSecNext);
            }
            dfsLX8( " ", sd->Al.Node[i].ChildBlock, cl, "");
            if (RelSecNext > RelSecThis)        // valid filesize
            {
               dfsUlSiz4(" ", RelSecNext - RelSecThis, "");
            }
            else                                // Filesize probably Zero
            {                                   // size only known in DIR
               TxPrint(" Unknown in Fnode");    // and in alloc-sector
            }
            RelSecThis  = RelSecNext;
            if (i == 0)
            {
               if (!TxaOptUnSet('X'))           // unless -X- specified
               {
                  nav.down = sd->Al.Node[i].ChildBlock;
               }
               cl = CNN;
            }
         }
         else                                   // LEAF allocation block
         {
            TxPrint("Rs %06.6X",    sd->Al.Leaf[i].RelSecData);
            dfsLX8( "  ",           sd->Al.Leaf[i].DataBlock, cl, "");
            dfsUlSiz4(" ",          sd->Al.Leaf[i].LengthData, "");
            done = (((sd->Al.Leaf[i].RelSecData  +
                      sd->Al.Leaf[i].LengthData) *
                      dfsGetSectorSize()) >= sd->FileLength);

            if (i == 0)
            {
               nav.down = sd->Al.Leaf[i].DataBlock;
               cl = CNN;
            }
         }
      }
      TxPrint("\n\n");
   }

   if (sd->EAsExtern.size != 0)
   {
      dfsDec8("External EA's     : ", sd->EAsExtern.size, "");
      TxPrint(", LSN : 0x%08.8X\n",   sd->EAsExtern.ptr);
   }
   if (sd->AclExtern.size != 0)
   {
      dfsDec8("External ACL's    : ", sd->AclExtern.size, "");
      TxPrint(", LSN : 0x%08.8X\n",   sd->AclExtern.ptr);
   }
   else if ((nr = dfsHpfsSizeEaFnode(lsn, sd)) != 0) // possible internal EA's
   {
      dfsEaBlockDisplay( "Internal EA's     : ", (S_EABLK *) sd->EasOrAcl, nr);
   }
   if (sd->AclFnLength > 0)
   {
      if (sd->AclFnLength <= HPFS_MAX_ACL_EA)
      {
         dfsHpfsAclArea( "Internal ACL's    : ", (S_ACLBLK *) sd->EasOrAcl, (ULONG) sd->AclFnLength);
      }
      else
      {
         TxPrint("Size of internal ACL exceeds maximum length!\n");
         rc = DFS_BAD_STRUCTURE;
      }
   }
   RETURN (rc);
}                                               // end 'dfsHpfsFnodeSec'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display HPFS allocation sector
/*****************************************************************************/
static ULONG dfsHpfsAllocSec                    // RET   rc = 0 if type match
(
   BYTE              *sector,                   // IN    Sector contents
   ULONG              lsn                       // IN    LSN of sector
)
{
   ULONG              rc = 0;                   // rc, sector match
   S_ALLOC           *sd = (S_ALLOC *) sector;
   BYTE               af = sd->AlInfo.Flag;
   int                al;
   char              *cl = CBG;                 // LSN color number
   int                i;
   BOOL               node;                     // fnode contains nodes
   BOOL               done = FALSE;             // stop-condition in display
   ULONG              RelSecThis = 0;
   ULONG              TotalSects = 0;
   BOOL               multic = (sd->AlInfo.UsedEntries != 1);

   ENTER();

   //- Note: AlInfo Flag alone is unreliable, it is zapped when deleting a file
   node = (sd->Al.Leaf[0].RelSecData + sd->Al.Leaf[0].LengthData !=
           sd->Al.Leaf[1].RelSecData) || (af & HPFS_AL_NODE);
   al   =  sd->AlInfo.UsedEntries + sd->AlInfo.FreeEntries;

   TxPrint("This sector   LSN : 0x%08.8X\n", sd->This);
   nav.up = sd->Parent;
   dfsX10( "Parent        LSN : ",   sd->Parent, CBC, "\n");
   if (node && ((af & HPFS_AL_NODE) == 0))      // deleted fnode with Nodes
   {
      TxPrint("Real Alloc type   : %s\n", (node) ? "Node"  : "Leaf");
   }
   TxPrint(  "Parent structure  : %s",  (af & HPFS_P_FNODE) ? "Fnode  " : "Dir Entry");
   TxPrint(       "  Alloc type : %s",  (af & HPFS_AL_NODE) ? "Node"    : "Leaf");
   TxPrint(     " Nr of extents : %u\n", sd->AlInfo.UsedEntries);

   if (node)
   {
      TxPrint(   "\n RelSecNext AllocLSN  Allocated-size      %s",
         (multic) ? "RelSecNext AllocLSN  Allocated-size" : "");
      TxPrint(   "\n ========== ========  ==============      %s",
         (multic) ? "========== ========  ==============" : "");
   }
   else
   {
      TxPrint(   "\n RelSector  Data-LSN Extent-size          %s",
         (multic) ? "RelSector  Data-LSN Extent-size    " : "");
      TxPrint(   "\n =========  ======== ===============      %s",
         (multic) ? "=========  ======== ===============" : "");
   }
   for (i=0; (i < al) && (!done) && !TxAbort(); i++)
   {
      if (i % 2)
      {
         TxPrint("      ");
      }
      else
      {
         TxPrint("\n ");
      }
      if (node)                                 // NODE allocation block
      {
         done = (sd->Al.Node[i].RelSecNext == L32_NULL);
         if (done)
         {
            TxPrint("          ");
         }
         else
         {
            TxPrint("Rs %06.6X ", sd->Al.Node[i].RelSecNext);
         }
         dfsLX8( " ",      sd->Al.Node[i].ChildBlock, cl, "");

         if (done)
         {
            TxPrint("  <rest-of-file>");
         }
         else
         {
            dfsUlSiz4(" ", sd->Al.Node[i].RelSecNext - RelSecThis, "");
         }
         RelSecThis = sd->Al.Node[i].RelSecNext;
         if (i == 0)
         {
            nav.down = sd->Al.Node[i].ChildBlock;
            cl = CNN;
         }
      }
      else                                      // LEAF allocation block
      {
         TxPrint("Rs %06.6X",  sd->Al.Leaf[i].RelSecData);
         dfsLX8( "  ",         sd->Al.Leaf[i].DataBlock, cl, "");
         dfsUlSiz4(" ",        sd->Al.Leaf[i].LengthData, "");
         TotalSects += sd->Al.Leaf[i].LengthData;
         if (i == 0)
         {
            nav.down = sd->Al.Leaf[i].DataBlock;
            cl = CNN;
         }
         done = ( sd->Al.Leaf[i+1].RelSecData !=
                 (sd->Al.Leaf[i  ].RelSecData +
                  sd->Al.Leaf[i  ].LengthData));
      }
   }
   TxPrint("\n\n");
   if (!node)
   {
      dfsSiz8("Sector alloc-size : ", TotalSects, "\n");
   }
   RETURN (rc);
}                                               // end 'dfsHpfsAllocSec'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display HPFS directory-block
/*****************************************************************************/
static ULONG dfsHpfsDirblock                    // RET   rc = 0 if type match
(
   BYTE               *sector,                  // IN    Sector contents
   ULONG               lsn                      // IN    LSN of sector
)
{
   ULONG               rc = 0;                  // rc, sector match
   S_DIRBL            *sd = (S_DIRBL *) sector;
   ULONG               cc = sd->ChangeCount;
   S_DIRENT           *de;
   USHORT              ei;                      // entry index
   BYTE                df;
   int                 cl = TXaNWnZ;            // LSN color
   int                 ac;                      // alternate color fg+bg
   int                 gc;                      // grey FG, alternate BG
   ULONG              *bdown;
   TXLN                tbuf;                    // temporary text buffer
   TX1K                text;                    // text string, MAX_PATH + line
   USHORT              tl;                      // traversed length (sanity)

   ENTER();
   nav.xtra = sd->This;
   nav.up   = sd->ParentDir;
   dfsX10( "Parent        LSN : ",        sd->ParentDir, CBC, "\n");
   TxPrint("Parent structure  : %s\n", (cc & 0x01) ? "Fnode": "DirBlock");
   TxPrint("ChangeCount       : 0x%08.8X\n", cc);
   TxPrint("ThisBlock     LSN : 0x%08.8X\n", sd->This);

   TxPrint( "\n%s\n%s\n", hpfsDirHeaderText, hpfsDirHeaderLine);

   if (!TxaOptUnSet('l'))                       // create a new list for DIR
   {
      dfsInitList(0, "-f -P", "-d");            // optimal for menu file-recovery
   }
   for ( tl = 20,  ei=0, de = &(sd->FirstEntry);
        (tl < 2048) &&   (de->Length != 0) && (de->Length < 290) && !TxAbort();
         tl += de->Length, de = (S_DIRENT *) ((char *) (de) + de->Length))
   {
      strcpy( text, "");
      df = de->Flag;
      if (df & HPFS_DF_BGDIR)                   // Special "BEGIN" entry
      {
         ei = 0;                                // next is first real entry
      }
      if (df & HPFS_DF_BDOWN)                   // B-tree DOWN pointer
      {
         bdown = (ULONG *) ((char *) (de) + de->Length - sizeof(ULONG));
         sprintf( tbuf, ".%5.5u", ei);
         strcat(   text, tbuf);
         dfstrLX8( text, " ", *bdown, (char *) ((ei == 0) ? CBG : CNN),
                 " -- Btree ptr LSN --\n");
         if (ei == 0)
         {
            nav.down = *bdown;
         }
         ei++;
         if (!TxaOptUnSet('l'))                 // create a new list for DIR
         {
            dfsAdd2SectorList( *bdown);         // add to list
         }
      }
      if ((df & HPFS_DF_BGEND) == 0)            // no special entry
      {
         if (ei % 2)
         {
            ac = TXaBWnC;
            gc = TXaBZnC;
         }
         else
         {
            ac = TXaNWnZ;
            gc = TXaBZnZ;
         }
         switch (ei)
         {
            case 0:  cl = TXaBGnZ; nav.down = de->Fnode; break;
            case 1:  cl = TXaBYnC; nav.xtra = de->Fnode; break;
            default: cl = ac;                            break;
         }
         sprintf(  tbuf, "%s.%5.5u",     ansi[ac], ei);
         strcat(   text, tbuf);
         dfstrLX8( text, "", de->Fnode, ansi[cl], ansi[ac]);
         strcat( text, " ");
         dfsHpfsTime2str( de->CreationTime, tbuf);
         strcat(   text, tbuf);
         dfstrFatAttrib( text,  de->FatAttrib);
         if (de->LastModDate != de->CreationTime)
         {
            strcat( text, " ");
            dfsHpfsTime2str( de->LastModDate, tbuf);
            strcat(  text, tbuf);
         }
         else
         {
            strcat( text, "                    ");
         }
         if (de->FatAttrib & FATTR_DIRECTORY)
         {
            strcat( text, "              ");
         }
         else
         {
            dfstrUlDot13( text, " ", de->FileSize, "");
         }

         if (de->EALength)
         {
            sprintf( tbuf, " %5u", de->EALength);
            strcat(  text, tbuf);
         }
         else
         {
            strcat( text, "      ");
         }
         sprintf( tbuf, " %-3u 0x%2.2hhx%s\n%s", hpfs->cpg[(int)de->CodePageIndex],
                                                 df, CGE, ansi[ac]);
         strcat(  text, tbuf);

         if (de->LastAccessTime != de->CreationTime)
         {
            sprintf( tbuf, " %saccessed%s%s ", ansi[gc], CNN, ansi[ac]);
            strcat(  text, tbuf);
            dfsHpfsTime2str( de->LastAccessTime, tbuf);
            strcat(  text, tbuf);
            strcat(  text, " ");
         }
         else
         {
            sprintf( tbuf, " %29.29s", "");
            strcat(   text, tbuf);
         }
         dfstrHpfsAttrib( text, df);
         sprintf( tbuf, " %s%*.*s%s%s%s\n", ansi[Ccol((CcY | CcI), Ccbg(ac))], // Yellow
                                            de->NameLength, de->NameLength, de->FileName,
                                            ansi[Ccol( CcW       , Ccbg(ac))], CGE, CNN);
         strcat(  text, tbuf);

         ei++;
         if (!TxaOptUnSet('l'))                 // create a new list for DIR
         {
            dfsAdd2SectorList( de->Fnode);      // add to list
         }
      }
      TxPrint( "%s", text);                     // print before break!
      if (df & HPFS_DF_EODIR)                   // Special "END" entry
      {
         break;                                 // out of entry-loop
      }
   }
   if (ei > 12)
   {
      TxPrint( "%s\n%s\n", hpfsDirHeaderLine, hpfsDirHeaderText);
   }
   RETURN (rc);
}                                               // end 'dfsHpfsDirblock'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display HPFS codepage information sector
/*****************************************************************************/
static ULONG dfsHpfsCpInf                       // RET   rc = 0 if type match
(
   BYTE              *sector,                   // IN    Sector contents
   ULONG              lsn                       // IN    LSN of sector
)
{
   ULONG              rc = 0;                   // rc, sector match
   S_CPINF           *sd = (S_CPINF *) sector;
   ULONG              ib;
   S_CPIB            *cpib;
   #if   defined (DEV32)
      COUNTRYCODE     ccode = {0};              // 0 = default codepage
      COUNTRYINFO     cinfo;
      ULONG           csize;                    // returned size in cinfo
   #endif

   ENTER();
   #if   defined (DEV32)
      DosGetCtryInfo( sizeof(cinfo), &ccode, &cinfo, &csize);
   #endif
   TxPrint("Cpib's in sector  : %u\n",       sd->CpCount);
   TxPrint("Next CpInfo   LSN : 0x%08.8X\n", sd->NextCpInfo);

   if (sd->CpCount <= S_MAX_CPIB)
   {
      for (ib = 0; (ib < sd->CpCount) && (!TxAbort()); ib++)
      {
         cpib = &(sd->Cpib[ib]);
         TxPrint(" Vol. Specific Id : %-8u",        cpib->VolCpId);
         TxPrint("   = Country code : %02.2u",      cpib->Country);
         TxPrint(       "  Codepage : %4u\n",       cpib->CodePageId);
         TxPrint("  Nr DBCS ranges  : %u\n",        cpib->DbcsRanges);
         TxPrint("  Checksum of Cp  : 0x%08.8X\n",  cpib->CpChecksum);
         dfsX10( "  Cp data     LSN : ",            cpib->CpData, CBG, "\n");

         hpfs->cpg[ib] = cpib->CodePageId;      // keep Cp around
      }
   }
   else
   {
      TxPrint( "\nNr of CP-info blocks exceeds maximum!\n");
      rc = DFS_BAD_STRUCTURE;
   }
   #if   defined (DEV32)
   TxPrint("Currently active  :         ");
   TxPrint(  "   Country code : %02.2u",      cinfo.country);
   TxPrint(       "  Codepage : %4u\n",       cinfo.codepage);
   #endif
   RETURN (rc);
}                                               // end 'dfsHpfsCpInf'
/*---------------------------------------------------------------------------*/



/*****************************************************************************/
// Display HPFS codepage data sector
/*****************************************************************************/
static ULONG dfsHpfsCpDat                       // RET   rc = 0 if type match
(
   BYTE              *sector,                   // IN    Sector contents
   ULONG              lsn                       // IN    LSN of sector
)
{
   ULONG              rc = 0;                   // rc, sector match
   S_CPDAT           *sd = (S_CPDAT *) sector;
   USHORT             db;
   S_CPDB            *cpdb;
   BOOL               cpok = FALSE;             // Cp is current codepage
   #if   defined (DEV32)
      COUNTRYCODE     ccode = {0};              // 0 = default codepage
      COUNTRYINFO     cinfo;
      ULONG           csize;                    // returned size in cinfo
   #endif

   ENTER();

   #if   defined (DEV32)
      DosGetCtryInfo( sizeof(cinfo), &ccode, &cinfo, &csize);
   #endif

   TxPrint("Cpdb's in sector  : %-8u\n", sd->CpCount);
   if (sd->CpCount <= 3)
   {
      for (db = 0; (db < sd->CpCount) && (!TxAbort()); db++)
      {
         cpdb = (S_CPDB *) (((char *) sd) + sd->OfsCpdb[db]);
         #if   defined (DEV32)
            cpok = (cinfo.codepage == (ULONG) cpdb->CodePageId);
         #endif
         TxPrint("\n Cp  Country code : %02.2u",  cpdb->Country);
         TxPrint(         "  Codepage : %s%4u%s   %s\n",
                           cpok ? CBG : CBR,        cpdb->CodePageId, CNN,
                           cpok ? "(Active codepage, display is OK)"
                                : "(Not active!, display incorrect)");
         TxPrint(  "  Checksum CpData : 0x%08.8X",  sd->Checksum[db]);
         TxPrint(    "   DBCS ranges  : %u\n",      cpdb->DbcsRanges);
         TxPrint(  "  Codepage map    : \n");
         dfsHpfsCpCharMap(cpdb->Table, 0x00);
         dfsHpfsCpCharMap(cpdb->Table, 0x40);
      }
   }
   else
   {
      TxPrint( "\nNr of CP-data blocks exceeds maximum!\n");
      rc = DFS_BAD_STRUCTURE;
   }
   RETURN (rc);
}                                               // end 'dfsHpfsCpDat'
/*---------------------------------------------------------------------------*/



/*****************************************************************************/
// Display HPFS (386) user-id information table (to be refined)
/*****************************************************************************/
static ULONG dfsHpfsUiTab                       // RET   rc = 0 if type match
(
   BYTE              *sector,                   // IN    Sector contents
   ULONG              lsn                       // IN    LSN of sector
)
{
   ULONG              rc = 0;                   // rc, sector match
   S_UITAB           *sd = (S_UITAB *) sector;
   ULONG              ui;

   ENTER();
   TxPrint("Uinf's  in sector : %-8u\n", sd->UiCount -1);
   for (ui = 0; (ui < sd->UiCount -1) && (!TxAbort()); ui++)
   {
      TxPrint(  "  Uinf bytes 1..4 : %02.2X %02.2X %02.2X %02.2X\n",
                                       sd->UiInfo[ui].Byte1,
                                       sd->UiInfo[ui].Byte2,
                                       sd->UiInfo[ui].Byte3,
                                       sd->UiInfo[ui].Byte4);
   }
   nav.xtra = nav.this;                         // remember Uid sector LSN
   RETURN (rc);
}                                               // end 'dfsHpfsUiTab'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display ACL information
/*****************************************************************************/
static void dfsHpfsAclArea
(
   char              *text,                     // IN    leading text
   S_ACLBLK          *first,                    // IN    first ACL block
   ULONG              size                      // IN    size of ACL area
)
{
   S_ACLBLK          *acl = first;              // ACL block
   ULONG              acls = size / 20;         // nr of ACL blocks
   ULONG              i;

   ENTER();

   TxPrint( "Internal ACl size : %4.4X = %u Dec for %d ACL entries\n",
                                 size, size, acls);

   for ( i = 0,        acl = first;
        (i < acls) && (!TxAbort());
         i++,          acl++)
   {
      TxPrint( " ACL entry nr: %2u : %8.8X  "
                               "Uinf: %02.2X %02.2X %02.2X %02.2X  "
                                  "R: %8.8X %8.8X %8.8X\n",
                                   i +1,
                                   acl->access,
                                   acl->userid.Byte1,
                                   acl->userid.Byte2,
                                   acl->userid.Byte3,
                                   acl->userid.Byte4,
                                   acl->reserv_1,
                                   acl->reserv_2,
                                   acl->reserv_3);
   }
   VRETURN();
}                                               // end 'dfsHpfsAclArea'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Make Codepage character-map dump on TxPrint output, 1 line of 64 chars in brackets;
/*****************************************************************************/
static void dfsHpfsCpCharMap
(
   char              *data,                     // IN    data area
   int                offset                    // IN    offset in map-table
)
{
   char                c;
   int                 i;
   TXTM                ascii;

   ENTER();
   memset(ascii, 0, TXMAXTM);
   for (i=0; i<64; i++)
   {
      c = (char) (128 + offset + i);
      ascii[i] = TxPrintSafe(c);
   }
   TxPrint("\n    [%s]\n", ascii);

   memset(ascii, 0, TXMAXTM);
   for (i=0; i<64; i++)
   {
      c = data[offset +i];
      ascii[i] = TxPrintSafe(c);
   }
   TxPrint(  "    [%s]\n", ascii);
   VRETURN();
}                                               // end 'dfsHpfsCpCharMap'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Truncate the filesystem size, keeping all data intact; (2 sectors in br)
// This is used for expanding as well as reducing the size of the filesystem
/*****************************************************************************/
static ULONG dfsHpfsTruncate
(
   ULN64               newsize,                 // IN    new size in sectors
   ULN64               d2,                      // IN    dummy
   char               *string,                  // IN    dummy
   void               *data                     // IN    partition info
)
{
   ULONG               rc  = NO_ERROR;
   DFSPARTINFO        *p   = (DFSPARTINFO *) data;
   TXTM                text;

   ENTER();
   TRARGS(("New size: %8.8x\n", newsize));

   if (newsize > HPFS_LIMIT)                    // limit to (OS/2) HPFS limit
   {
      dfsSiz8("\nNew FS size set to the maximum of: ", HPFS_LIMIT,
              " (OS/2 implementation limit)\n");
      newsize = HPFS_LIMIT;
   }

   strcpy( text, "");
   dfstrSizeBps( text, "to a new size of: ", newsize, (USHORT) p->bpsector, "");

   if ((dfsa->batch) || (TxConfirm( 5127, "Resize this HPFS filesystem "
                                          "%s ? [Y/N] : ", text)))
   {
      S_BOOTR         *br = (S_BOOTR *) brec;
      ULONG            bitmap;

      if (newsize > 0xffff)
      {
         br->eb.Sectors    = 0;
         br->eb.BigSectors = newsize;
      }
      else
      {
         br->eb.Sectors    = (USHORT) newsize;
         br->eb.BigSectors = 0;
      }

      if (DFSTORE_WRITE_ALLOWED)
      {
         ULONG      maxsize = newsize;

         if (maxsize < hpfs->sup->TotalSec)     // smaller than existing FS ?
         {                                      // May need fixup of last bitmap
            bitmap = hpfs->bmt[ newsize >> 14]; // get closest existing bitmap
            if (bitmap > newsize)               // outside requested size
            {
               maxsize = bitmap +4 - 0x4000;    // set to begin of this band
               TxPrint( "\nRounded down net size to nearest 8MB band, reboot\n"
                        "and 'CHKDSK /f' required for maximum available size!\n");
            }
         }
         hpfs->sup->TotalSec    = maxsize & 0xfffffffc;
         hpfs->spr->StatusFlag |= HPFS_SP_DIRTY; // force CHKDSK on next boot

         if ((rc = dfsHpfsWriteSuperSpare()) == NO_ERROR)
         {
            dfsSiz8("\nHPFS filesys-size : ",
                            hpfs->sup->TotalSec, " after resizing\n");
            rc = dfsWrite( LSN_BOOTR, 1, brec);
            if (rc != NO_ERROR)
            {
               TxPrint( "\nSuperblock updated, but boot record failed!\n");
            }
         }
      }
      else
      {
         rc = DFS_READ_ONLY;
      }
   }
   else
   {
      rc = DFS_NO_CHANGE;
   }

   RETURN (rc);
}                                               // end 'dfsHpfsTruncate'
/*---------------------------------------------------------------------------*/

