//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// DFS display and analysis, HFS specific functions and definitions
//
// Author: J. van Wijk
//
// JvW  01-04-2012 Initial version, derived from JFS / EXT
//
// Author: J. van Wijk

#ifndef    DFSAHFS_H
   #define DFSAHFS_H

#include <dfshfs.h>                             // HFS  structure defs

// HFS specific sector types
                                                // S   used on searching
                                                //  I  identification only
#define ST_HFSSUP  's'                          //     HFS  super-block         (512 BPS)
#define ST_HFSBSU  'S'                          //  I  HFS  boot+superblock    (4096 BPS)
#define ST_HFSBFL ('B' + ST__INFO)              //     Boot (loader) file
#define ST_HFSBMP ('M' + ST__INFO)              //     Bitmap sector
#define ST_HFSEXT ('E' + ST__INFO)              //     Extents B-tree
#define ST_HFSATR ('A' + ST__INFO)              //     Attributes B-tree
#define ST_HFSCAT ('c' + ST__INFO)              //     Catalog B-tree unused space
#define ST_HFSCIN ('i' + ST__INFO)              //     Catalog B-tree index-node
#define ST_HFSCLF ('f' + ST__INFO)              //     Catalog B-tree leaf-node
#define ST_HFSCMP ('m' + ST__INFO)              //     Catalog B-tree map-node
#define ST_HFSCHD ('F' + ST__INFO)              //     Catalog B-tree header node


// Menu-settable option boolean for hardlink treatment and show/hide hidden files/directories
#define HfsHardlinksAutoResolve  dfsa->FsToggleOption1
#define HfsFldrAliasAutoResolve  dfsa->FsToggleOption2

// Caches exactly ONE block of bitmap data
typedef struct dfshfsbmcache
{
   DFSISPACE           File;                    // space allocation for file
   ULN64               LimitLsn;                // largest LSN inside bitmap
   ULONG               Rbmap;                   // cached relative bitmap page
   BOOL                Dirty;                   // cache contents modified
   BYTE               *Map;                     // bitMAP array, maps 8 * blocksize
} DFSHFSBMCACHE;                                // end of struct "dfshfsbmcache"

typedef struct dfshfsiname                      // cnID to parent/name cache element
{
   ULONG               Parent;                  // Parent (folder) CnID
   char               *Name;                    // related name (allocated)
} DFSHFSINAME;                                  // end of struct "dfshfsiname"

// Caches Catalog-file allocation SPACE and the Btree ROOT-NODE
typedef struct dfshfscatalog
{
   DFSISPACE           File;                    // space allocation for file
   ULONG               spNode;                  // sectors per node
   ULONG               bpNode;                  // bytes   per node
   S_BT_HDR_NODE      *Btree;                   // B-tree HDR node copy (BIG ENDIAN!)
} DFSHFSCATALOG;                                // end of struct "dfshfsbmcache"


typedef struct dfsahfs                          // HFS analysis information
{
   ULN64               SectorCount;             // Number of sectors
   ULONG               BlockCount;              // Number of blocks
   ULONG               SectorsPerBlock;         // sectors per block
   ULONG               BlockBits;               // Nr of bits per block (bitmap)
   ULONG               preferedFolderId;        // prefered on catalog leaf display
   ULONG               hlMetaFolderId;          // Cnid of hardlink    folder, or 0
   ULONG               fdAliasFolderId;         // Cnid of folderalias folder, or 0
   ULONG               icSize;                  // allocated Ic size
   DFSHFSINAME        *Ic;                      // Name and Parent ID cache
   S_HFS_SUPER        *sup;                     // ptr to superblock
   DFSHFSBMCACHE       Bm;                      // BitMap file cache
   DFSHFSCATALOG       Catalog;                 // catalog allocation + info
} DFSAHFS;                                      // end of struct "dfsahfs"

extern DFSAHFS        *hfs;

// Initialize HFS filesystem analysis
ULONG dfsHfsInit
(
   char               *fs                       // forced filesystem type
);

// Initialize HFS filesystem analysis for Area (FS info in FDISK mode)
ULONG dfsHfsAreaInit
(
   void
);

// HFS  filesystem, identify specified sector as a valid superblock
BOOL dfsHfsIsSuperBlock                         // RET   sector is a valid sb
(
   BYTE               *sec                      // IN    sector contents
);

// HFS filesystem, display sector constents as a superblock
void dfsHfsDisplaySuperBlock
(
   BYTE               *sec                      // IN    sector contents
);

// HFS filesystem, identify specified sector
ULONG dfsHfsIdent
(
   ULN64               di,                      // IN    LSN for sector
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // OUT   sector type
   void               *data                     // IN    sector contents
);


#endif
