//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// HFS dump & display Analysis functions
//
// Author: J. van Wijk
//
// JvW  08-04-2021 Use rc CMD_WARNING on FileSaveAs alloc errors, considered OK
// JvW  17-05-2012 Interpret of allocation special file, including ALLOC display
// JvW  02-05-2012 Display of superblock information including 'special' files
// JvW  16-07-2007 Initial version, derived from RSR
//
// Author: J. van Wijk
//
// Note: HFS is a BIG-ENDIAN oriented FS, so all field access
//       needs to use the TxBE16/32/64 macros, see txlib.h
//       Structures remain in their original endianess (big) in memory


#include <txlib.h>                              // TX library interface

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfsspace.h>                           // DFS file-space interface
#include <dfstable.h>                           // SLT utility functions

#include <dfsupart.h>                           // prereq. for UFDSK
#include <dfsufdsk.h>                           // FDISK generic Bootsector display

#include <dfsahfs.h>                            // HFS display & analysis
#include <dfsuhfs.h>                            // HFS utility functions
#include <dfslhfs.h>                            // HFS SLT functions

char dfsHfsSectorTypes[] =
{
   ST_HFSSUP,                                   //     HFS  super-block
   ST_HFSBFL,                                   //     Boot (loader) file
   ST_HFSBMP,                                   //     Bitmap sector
   ST_HFSEXT,                                   //     Extents B-tree
   ST_HFSATR,                                   //     Attributes B-tree
   ST_HFSCAT,                                   //     Catalog B-tree
   ST_HFSCIN,                                   //     Catalog B-tree index-node
   ST_HFSCLF,                                   //     Catalog B-tree leaf-node
   ST_HFSCMP,                                   //     Catalog B-tree map-node
   ST_HFSCHD,                                   //     Catalog B-tree header node
   0                                            //     string terminating ZERO
};


static DFSAHFS    hfs_anchor =                  // HFS specific global info
{
   0, 0,                                        // zero size
   8,                                           // default 8 (512 byte) sectors/block
   8 * (8 * SECTORSIZE),                        // number of bits in one block (bitmap)
   0,                                           // prefered folder ID on catalog leaf display
   0,                                           // Cnid of hardlink    folder, or 0
   0,                                           // Cnid of folderalias folder, or 0
   0, NULL,                                     // CnID cache size and allocated array
   NULL,                                        // superblock pointer
  {{0,NULL,0,0,0,0,0,NULL,0,0},0,0,FALSE,NULL}, // bitmap space and additional info
  {{0,NULL,0,0,0,0,0,NULL,0,0},0,0,NULL}        // catalog space and Btree rootnode
};

       DFSAHFS   *hfs = &hfs_anchor;

static  char       *hfs_txt[] =
{
   "",
   "Active filesystem : HFS, specific commands are:",
   "",
   " BL               = Translate and display 'this' LSN as a block number",
   " BL block   [cmd] = Translate specified block-number to LSN, display using 'cmd'",
   " CATS Fid  [name] = Catalog search for folder-CnID, [containing filename]",
   " FOLDER     [Fid] = Display Catalog directory data for specified folder CnID",
   " FDALIAS [on|off] = Folder Alias transparantly (ON) or treat as files  (off)",
   " HLINKS  [on|off] = Hardlinks transparantly (ON) or treat as softlinks (off)",
   " LABEL    [label] = Display/edit 255-char volume label in the Catalog file",
   " NODE             = Translate and display 'this' LSN as a Catalog-node number",
   " NODE cat-nodenr  = Calculate LSN for a Catalog-node number, default display",
   " SUPER            = Display the filesystem SUPERBLOCK sector",
   "",
   NULL
};

// Close HFS filesystem for analysis and free any resources
static ULONG dfsHfsClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
);

// Close HFS filesystem for Area analysis and free any resources
static ULONG dfsHfsAreaClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
);

// Interpret and execute specific HFS command;
static ULONG dfsHfsCommand
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *none,                    // IN    dummy
   void               *data                     // INOUT dummy
);

// HFS filesystem, supply sector-type description string
static ULONG dfsHfsStype
(
   ULN64               di,                      // IN    sector type
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // OUT   type description
   void               *data                     // INOUT dummy
);

// HFS filesystem, display sector-contents based on type
static ULONG dfsHfsSectorContents
(
   ULN64               psn,                     // IN    base psn for sector
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // IN    sector contents
);

// HFS filesystem, display special LSN info (Node lsn + record-index value)
static ULONG dfsHfsDisplayLsnInfo
(
   ULN64               lsn,                     // IN    possible special lsn
   ULN64               info,                    // IN    possible REC index
   char               *dc,                      // IN    dummy
   void               *data                     // IN    dummy
);

// Display all available info about a Node Record, in a Catalog Node
static ULONG dfsHfsCatalogRecord
(
   ULN64               nodeLsn,                 // IN    lsn of Catalog Node
   USHORT              index                    // IN    index of record 0..N
);

// Display HFS  super-block
static ULONG dfsHfsSuperBlock                   // RET   rc = 0 if type match
(
   BYTE               *sector,                  // IN    Superblock sector data
   BOOL                navigation               // IN    update nav values
);

// Display HFS Catalog node, header-node for a B-tree
static ULONG dfsHfsCatHeader                    // RET   rc = 0 if type match
(
   ULN64               lsn,                     // IN    lsn for sector
   BYTE               *sector,                  // IN    Node sector data
   BOOL                navigation               // IN    update nav values
);

// Display HFS Catalog node, leaf-node with directory and file info
static ULONG dfsHfsCatLeafNode                  // RET   rc = 0 if type match
(
   ULN64               lsn,                     // IN    lsn for sector
   BYTE               *sector,                  // IN    Node sector data
   BOOL                navigation               // IN    update nav values
);

// Display HFS Catalog index-node with tree split-points to other nodes
static ULONG dfsHfsCatIndexNode                 // RET   rc = 0 if type match
(
   ULN64               lsn,                     // IN    lsn for sector
   BYTE               *sector,                  // IN    Node sector data
   BOOL                navigation               // IN    update nav values
);

// Display HFS Catalog node, unused or unknown/invalid contents
static ULONG dfsHfsCatUnused                    // RET   rc = 0 if type match
(
   ULN64               lsn,                     // IN    lsn for sector
   BYTE               *sector,                  // IN    Node sector data
   BOOL                navigation,              // IN    update nav values
   ULONG               dumpsize                 // IN    number of bytes to dump
);

// Display HFS node-descriptor structure (at start each node)
static ULONG dfsHfsNodeDescr                    // RET   rc = 0 if type match
(
   ULN64               lsn,                     // IN    lsn for sector
   BYTE               *sector,                  // IN    Node sector data
   BOOL                navigation               // IN    update nav values
);

// Make filesystem usage map dump on TxPrint output
static ULONG dfsHfsAllocMap
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *options,                 // IN    display options
   void               *data                     // OUT   dummy
);


// DFS HFS write-file to disk (SaveTo inode to file)
static ULONG dfsHfsFileSaveAs
(
   ULN64               leafLsn,                 // IN    Leaf-node sectornumber
   ULN64               leafIdx,                 // IN    Leaf-node record index
   char               *path,                    // IN    destination path
   void               *recp                     // IN    recovery params (rename)
);

/*****************************************************************************/
// Initialize HFS filesystem analysis
/*****************************************************************************/
ULONG dfsHfsInit
(
   char               *fs                       // forced filesystem type
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   dfsa->FsCommand          = dfsHfsCommand;
   dfsa->FsClose            = dfsHfsClose;
   dfsa->FsIdentifySector   = dfsHfsIdent;
   dfsa->FsShowType         = dfsHfsStype;
   dfsa->FsDisplaySector    = dfsHfsSectorContents ;
   dfsa->FsFileInformation  = dfsHfsFileInfo;   // FS file alloc/path/size/date
   dfsa->FsGetAllocSpace    = dfsHfsGetAllocSpace;
   dfsa->FsWriteMetaSpace   = NULL;
   dfsa->FsMakeBrowseList   = dfsHfsMakeBrowseList;
   dfsa->FsFindPath         = dfsHfsFindPath;
   dfsa->FsLsnAllocated     = dfsHfsAllocated;
   dfsa->FsLsnSetAlloc      = NULL;
   dfsa->FsSltBuild         = dfsHfsSltBuild;
   dfsa->FsNpBuild          = NULL;             // dfsHfsNpBuild (too slow, and not needed)
   dfsa->FsDisplayError     = dfsHfsDispError;
   dfsa->FsDisplayLsnInfo   = dfsHfsDisplayLsnInfo;
   dfsa->FsDirIterator      = NULL;
   dfsa->FsDirFileSaveAs    = dfsHfsFileSaveAs;
   dfsa->FsTruncateSize     = NULL;
   dfsa->FsAllocDisplay     = dfsHfsAllocMap;
   dfsa->FsCl2Lsn           = dfsHfsCl2Lsn;
   dfsa->FsLsn2Cl           = dfsHfsLsn2Cl;
   dfsa->FsCmdHelp          = hfs_txt;          // FS specific cmdhelp text

   dfsa->FsModeId           = DFS_FS_HFS;       // common for HFS

   dfsa->Fsi                = hfs;
   dfsa->FsSectorTypes      = dfsHfsSectorTypes;

   dfsa->FsEntry            = HFS_LSNSUP1;      // entry-point in FS (after bootrec)
   nav.down                 = HFS_LSNSUP1;      // first sector to see after open

   HfsHardlinksAutoResolve = TRUE;
   HfsFldrAliasAutoResolve = TRUE;

   if (((hfs->sup = dfsHfsGetSuperBlock( NULL)) != NULL)   &&
       (dfsIdentifySector( HFS_LSNSUP1, 0, (BYTE *) hfs->sup)  == ST_HFSSUP))
   {
      hfs->SectorsPerBlock = TxBE32( hfs->sup->BlockSize) / dfsGetSectorSize();
      hfs->BlockCount      = TxBE32( hfs->sup->TotalBlocks);
      hfs->SectorCount     = (ULN64) hfs->BlockCount  * hfs->SectorsPerBlock;
      hfs->BlockBits       = (8 * dfsGetSectorSize()) * hfs->SectorsPerBlock;

      dfstSetClusterSize( DFSTORE, (USHORT) hfs->SectorsPerBlock);

      TxPrint("Number of files   : %10u   Directories: %10u\n",
               TxBE32( hfs->sup->FileCount), TxBE32( hfs->sup->DirCount));
      TxPrint("Size each  block  : %10u   Bytes, %u sectors/block\n",
               TxBE32( hfs->sup->BlockSize), hfs->SectorsPerBlock);
      TxPrint("Blocks on volume  : %10u,  ",       hfs->BlockCount);
      dfsSz64("Size : ", ((ULN64) hfs->BlockCount * hfs->SectorsPerBlock), "\n");
   }
   else
   {
      TxPrint("Error reading 1st EXT Super-block at byte offset 1024, defaults used!\n");

      hfs->SectorsPerBlock = 4096 / dfsGetSectorSize();
      hfs->SectorCount     = dfsGetLogicalSize();
      hfs->BlockCount      = hfs->SectorCount / hfs->SectorsPerBlock;
      hfs->BlockBits       = (8 * dfsGetSectorSize()) * hfs->SectorsPerBlock;

      dfstSetClusterSize( DFSTORE, (USHORT) hfs->SectorsPerBlock);
   }
   if ((rc == NO_ERROR) && (hfs->BlockCount != 0) && (hfs->SectorsPerBlock != 0))
   {
      if ((rc = dfsHfsBitMapInit()) == NO_ERROR)
      {
         dfsSz64( "Bitmap sect range : ", hfs->Bm.LimitLsn, "\n");

         if ((rc = dfsHfsCatalogInit()) == NO_ERROR)
         {
            BYTE      *node = NULL;
            ULONG      nodeNr;
            USHORT     index;

            nodeNr = dfsHfsSearchCatalog( HFS_ID_RootFolder, HFS_SD_HARDLINK, &index, &node);
            if ((nodeNr != 0) && ((nodeNr != L32_NULL)))
            {
               U_HFS_CAT_RECORD   *catData;
               USHORT              recType;

               recType = dfsHfsNodeIdx2Rec( node, index, &catData);
               if (recType == HFS_CR_Folder)
               {
                  hfs->hlMetaFolderId = TxBE32( catData->d.crCnID);
               }
               TxFreeMem( node);
            }
            nodeNr = dfsHfsSearchCatalog( HFS_ID_RootFolder, HFS_SD_FLDALIAS, &index, &node);
            if ((nodeNr != 0) && ((nodeNr != L32_NULL)))
            {
               U_HFS_CAT_RECORD   *catData;
               USHORT              recType;

               recType = dfsHfsNodeIdx2Rec( node, index, &catData);
               if (recType == HFS_CR_Folder)
               {
                  hfs->fdAliasFolderId = TxBE32( catData->d.crCnID);
               }
               TxFreeMem( node);
            }
            if (dfsa->verbosity > TXAO_QUIET)
            {
               if (hfs->fdAliasFolderId != 0)
               {
                  TxPrint( "Folder-Alias Cnid : 0x%8.8x for '/%s', to resolve Folder Aliases\n",
                            hfs->fdAliasFolderId, HFS_SD_FLDALIAS);
               }
               else
               {
                  TxPrint( "FolderAlias  name : '/%s' not found, no folder aliases present ...\n", HFS_SD_FLDALIAS);
               }
               if (hfs->hlMetaFolderId != 0)
               {
                  TxPrint( "Meta Folder  Cnid : 0x%8.8x for '/%s', to resolve hardlinks\n",
                            hfs->hlMetaFolderId, HFS_SD_HARDLINK);
               }
               else
               {
                  TxPrint( "Meta Folder  name : '/%s' not found, no hardlinks present ...\n", HFS_SD_HARDLINK);
               }
               TxPrint(    "Catalog node size :%5hu bytes   Root Node : 0x%8.8x  CompareCase: %ssensitive\n",
                            TxBE16( hfs->Catalog.Btree->hdr.bthNodeSize),
                            TxBE32( hfs->Catalog.Btree->hdr.bthRootNode),
                                   (hfs->Catalog.Btree->hdr.bthCompareType == HFS_CaseFolding) ? "in" : "");

               if (dfsa->verbosity > TXAO_NORMAL)
               {
                  TxPrint( "      Btree depth :%4hu levels   NodeCount : 0x%8.8x  Free nodes : 0x%8.8x  Used:0x%8.8x\n",
                            TxBE16( hfs->Catalog.Btree->hdr.bthDepth),
                            TxBE32( hfs->Catalog.Btree->hdr.bthTotalNodes),
                            TxBE32( hfs->Catalog.Btree->hdr.bthFreeNodes ),
                            TxBE32( hfs->Catalog.Btree->hdr.bthTotalNodes) -
                            TxBE32( hfs->Catalog.Btree->hdr.bthFreeNodes));
               }
            }
            if (TxaOption('a'))                 // show NO allocation by default
            {
               dfsHfsAllocMap( 0, 0, "@", NULL); // not really needed (no resize)
            }
         }
      }
   }
   RETURN (rc);                                 // when needed
}                                               // end 'dfsHfsInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Close HFS filesystem for analysis and free any resources
/*****************************************************************************/
static ULONG dfsHfsClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;
   ULONG               i;

   ENTER();

   dfsSlTableReset();                           // stop SLT thread and reset

   rc = dfsHfsBitmapFlush( TRUE);               // flush, and terminate cache

   TxFreeMem( hfs->Catalog.File.space);
   TxFreeMem( hfs->Catalog.Btree);

   if (hfs->Ic)
   {
      for (i = 0; i < hfs->icSize; i++)
      {
         TxFreeMem( hfs->Ic[i].Name);           // free cached CnID names
      }
      TxFreeMem( hfs->Ic);                      // free name/parent cache
   }
   hfs->hlMetaFolderId  = 0;
   hfs->fdAliasFolderId = 0;

   TxFreeMem( hfs->sup);

   dfsCloseFileSystem();
   RETURN (rc);
}                                               // end 'dfsHfsClose'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Initialize HFS filesystem analysis for Area (FS info in FDISK mode)
/*****************************************************************************/
ULONG dfsHfsAreaInit
(
   void
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   dfsa->FsAreaClose        = dfsHfsAreaClose;
   dfsa->FsAreaLsnAllocated = dfsHfsAllocated;

   if ((hfs->sup = dfsHfsGetSuperBlock( NULL)) != NULL)
   {
      hfs->SectorsPerBlock = TxBE32( hfs->sup->BlockSize) / dfsGetSectorSize();
      hfs->BlockCount      = TxBE32( hfs->sup->TotalBlocks);
      hfs->SectorCount     = (ULN64) hfs->BlockCount  * hfs->SectorsPerBlock;
      hfs->BlockBits       = (8 * dfsGetSectorSize()) * hfs->SectorsPerBlock;

      rc = dfsHfsBitMapInit();
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);                                 // when needed
}                                               // end 'dfsHfsAreaInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Close HFS filesystem for Area analysis and free any resources
/*****************************************************************************/
static ULONG dfsHfsAreaClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;
   ULONG               i;

   ENTER();

   dfsa->FsAreaClose        = NULL;
   dfsa->FsAreaLsnAllocated = NULL;

   rc = dfsHfsBitmapFlush( TRUE);               // flush, and terminate cache

   if (hfs->Ic)
   {
      for (i = 0; i < hfs->icSize; i++)
      {
         TxFreeMem( hfs->Ic[i].Name);           // free cached CnID names
      }
      TxFreeMem( hfs->Ic);                      // free name/parent cache
   }

   TxFreeMem( hfs->sup);
   hfs->SectorsPerBlock = 8;
   hfs->BlockCount      = 0;
   hfs->SectorCount     = 0;
   hfs->BlockBits       = 8 * (8 * dfsGetSectorSize()); // back to defaults

   RETURN (rc);
}                                               // end 'dfsHfsAreaClose'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Interpret and execute specific HFS command;
/*****************************************************************************/
static ULONG dfsHfsCommand
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *none,                    // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;
   ULN64               sn = 0;                  // sector number input
   TXLN                s0;                      // large text buffer 0
   TXLN                s1;                      // large text buffer 1
   TXLN                dc;                      // DFS command
   int                 cc;                      // command string count
   char               *c0, *c1, *c2;            // parsed command parts
   char               *pp;                      // parameter pointer
   ULONG               cnId;                    // folder or file ID value
   ULONG               nodeNr;                  // node number (in catalog)
   USHORT              index;                   // node record index

   ENTER();

   pp = TxaGetArgString( TXA_CUR, 1, 0, TXMAXLN, dc); // dc => cmd from arg 1
   cc = TxaArgCount( );                         // number of parameters
   c0 = TxaArgValue(0);
   c1 = TxaArgValue(1);
   c2 = TxaArgValue(2);

   sn = nav.this;                               // default at current sector

   if ((strcmp(c0, "/?") == 0) ||
       (strcmp(c0,  "?") == 0)  )
   {
      TxShowTxt( hfs_txt);
   }
   else if ((strcasecmp(c0, "cl"       ) == 0) ||
            (strcasecmp(c0, "bl"       ) == 0)  )
   {
      if (cc > 1)
      {
         sn = dfsGetSymbolicSN( c1, 0);
         if (sn <= hfs->BlockCount)
         {
            if (cc > 2)
            {
               sprintf( dc, "%s ", c2);
            }
            else
            {
               strcpy(  dc, "");
            }
            sprintf( s1, "0x0%llX", dfsHfsBlock2Lsn(sn));
            strcat(  dc, s1);
            TxaReParseCommand( dc);
            rc = DFS_PENDING;                   // handle translated command
         }
         else
         {
            dfsX10( "Invalid block   nr: ", sn, CBR, "\n");
         }
      }
      else
      {
         dfsX10("Current block   nr: ", dfsHfsLsn2Block(nav.this), CBM, "\n");
      }
   }
   else if ((strcasecmp(c0, "cats"  ) == 0))    // CATalog Search for parent + optional name
   {
      if (TxaOption('?') || (cc < 2))
      {
         TxPrint("\nSearch catalog for folder-ID and optional file/folder name, then display result\n");
         TxPrint("\n Usage:  %s  CnID [name]\n\n"
                   "   CnID = Hexadecimal ID for folder to search, often refered to as  CnID\n"
                   "   name = Name of file or folder to search for in the parent folder CnID\n\n"
                   "          Without a name, it will search for a threadrecord for the CnID\n"
                   "          displaying the parent CnID (folder) and the name for this CniD\n\n", c0);
      }
      else if (hfs->Catalog.Btree != NULL)
      {
         cnId   = dfsGetSymbolicSN( c1, 2);
         nodeNr = dfsHfsSearchCatalog( cnId, c2, &index, NULL);

         if      (nodeNr == 0)
         {
            dfsX10(  "\nBad / unused CnID : ", cnId, CBR, "\n");
         }
         else if (nodeNr == L32_NULL)
         {
            if (cc > 2)
            {
               TxPrint( "\nFile/Folder  name : '%s' not found in folder CnID %x\n", c2, cnId);
            }
            else
            {
               TxPrint( "\nNo thread record found for CnID %x\n", cnId);
            }
         }
         else
         {
            if (cc > 2)
            {
               TxPrint( "\nFolder/File record found in node 0x%8.8x for CnID %x name '%s'\n", nodeNr, cnId, c2);
            }
            else
            {
               TxPrint( "\nThread record found in node 0x%8.8x for CnID %x\n", nodeNr, cnId);
            }
            sn = dfsHfsNode2Lsn( nodeNr);
            hfs->preferedFolderId = 0;          // default ANY folder (to be refined)
            sprintf( dc, "0x0%llX -e:%hu", sn, index); // display as SN+INFO
            TxaReParseKeepOpts( dc);            // reparse but keep option values
            rc = DFS_PENDING;                   // handle translated command
         }
      }
   }
   else if ((strcasecmp(c0, "filefind") == 0))  // to be refined, build NP cache for now
   {
      dfsSltPrepareNameLookup();
      TxCancelAbort();                          // might have aborted
   }
   else if ((strcasecmp(c0, "folder"  ) == 0))  // display folder contents, by leaf-iterate
   {
      if (TxaOption('?'))
      {
         TxPrint("\nDisplay contents of catalog (leaf nodes) for specified folder-ID in directory format\n");
         TxPrint("\n Usage:  %s  [folderId] [-trace]\n\n"
                   "   folderId = Hexadecimal ID for folder to display, often refered to as CnID\n\n"
                   "   -l-      = Do NOT create a new sector list from the folder contents\n"
                   "   -trace   = Include all leafnode headers for folderId 0 (ALL records)\n", c0);
      }
      else if (hfs->Catalog.Btree != NULL)
      {
         cnId   = dfsGetSymbolicSN( c1, 2);

         if (cnId < HFS_ID_RootFolder)          // special folders ALL and LABEL
         {
            nodeNr = TxBE32( hfs->Catalog.Btree->hdr.bthFirstLeaf);
         }
         else                                   // should be real, search in Catalog
         {
            nodeNr = dfsHfsSearchCatalog( cnId, NULL, NULL, NULL);
         }

         if      ((nodeNr == 0) || (nodeNr == L32_NULL))
         {
            dfsX10(  "\nInvalid folder ID : ", cnId, CBR, "\n");
         }
         else
         {
            sn = dfsHfsNode2Lsn( nodeNr);
            hfs->preferedFolderId = cnId;       // show entries for specified folder
            sprintf( dc, "0x0%llX -auto", sn);
            TxaReParseKeepOpts( dc);            // reparse but keep option values
            rc = DFS_PENDING;                   // handle translated command
         }
      }
   }
   else if (strcasecmp(c0, "fdalias"  ) == 0)   // Folder-Alias treatment ON/OFF
   {
      if (TxaOption('?'))
      {
         TxPrint("\nSet Folder-Alias treatment to auto-resolve (transparent) or handle like a file\n");
         TxPrint("\nUsage: %s  [1 | on | 0 | off]\n\n"
                 " 1 |  on  : Auto-resolve, transparent, show/recover actual directory CONTENTS (default)\n"
                 " 0 |  off : Treat the alias as a file, show/recover linked file (resource fork)\n", c0);
      }
      else
      {
         if (cc > 1)                            // there is a parameter
         {
            if ((strcasecmp(c1, "on") == 0) || (atol( c2) != 0))
            {
               HfsFldrAliasAutoResolve = TRUE;
            }
            else
            {
               HfsFldrAliasAutoResolve = FALSE;
            }
         }
         TxPrint( "\nFldr Alias Resolve: %s\n", (HfsFldrAliasAutoResolve)
                   ? "ON, transparent handling, show/recover actual directory contents"
                   : "OFF, treat like a file, show/recover file resource-fork");
      }
   }
   else if (strcasecmp(c0, "hlinks"   ) == 0)   // hardlink treatment ON/OFF
   {
      if (TxaOption('?'))
      {
         TxPrint("\nSet Hardlink treatment to auto-resolve (transparent) or handle like softlinks\n");
         TxPrint("\nUsage: %s  [1 | on | 0 | off]\n\n"
                 " 1 |  on  : Auto-resolve, transparent,  show/recover actual file CONTENTS (default)\n"
                 " 0 |  off : Treat hardlink as softlink, show/recover linked file NAME\n", c0);
      }
      else
      {
         if (cc > 1)                            // there is a parameter
         {
            if ((strcasecmp(c1, "on") == 0) || (atol( c2) != 0))
            {
               HfsHardlinksAutoResolve = TRUE;
            }
            else
            {
               HfsHardlinksAutoResolve = FALSE;
            }
         }
         TxPrint( "\nHardlink Resolve  : %s\n", (HfsHardlinksAutoResolve)
                   ? "ON, transparent handling, show/recover actual file contents"
                   : "OFF, treat like softlink, show/recover linked filename");
      }
   }
   else if (strcasecmp(c0, "label"    ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nSet volume label in Catalog file (length max 255, but limited by space)\n");
         TxPrint("\nUsage: %s  [label]\n\n"
                 "   label  : new volume label, max length depends on available space\n"
                 "   -!-    = do not prompt for new value, just set it\n", c0);
      }
      else if (hfs->Catalog.Btree != NULL)
      {
         ULONG         rootNode = TxBE32( hfs->Catalog.Btree->hdr.bthRootNode);

         if (SINF->p)                           // it is a partition
         {
            strcpy( s0, SINF->p->plabel);       // old label determined by readdiskinfo
         }                                      // might be truncated to 31 characters!
         else
         {
            strcpy( s0, "");                    // unknown, set to empty
         }
         if (cc > 1)
         {
            strcpy( s1, c1);                    // use specified parameter value
         }
         else
         {
            strcpy( s1, s0);                    // use old label, could be empty
         }                                      // or truncated to 31 characters!

         if (dfsa->batch)                       // s1 has new label value
         {
            //- apply the label, starting at the Catalog tree-root node
            rc = dfsHfsNodeSetLabel( rootNode, BT_SANE_LEVELS, s0, s1);
         }
         else
         {
            #if defined (USEWINDOWING)          // Allow interactive update
            if (!TxaOptUnSet('!') && txwIsWindow( TXHWND_DESKTOP))
            {
               TXLN      prompt;

               sprintf( prompt, "%s\n\nVolume label in Catalog: '%s'\n\n"
                                "Specify new %d-character volume label\n",
                                 dfstStoreDesc1( DFSTORE) + 10, s1, MACUNI_MAX_LEN);

               if (txwPromptBox( TXHWND_DESKTOP, TXHWND_DESKTOP, NULL,
                      prompt, " Set volume label in CATALOG file ",
                      5165, TXPB_MOVEABLE | TXPB_HCENTER | TXPB_VCENTER,
                      MACUNI_MAX_LEN, s1) != TXDID_CANCEL)

               {
                  TxStrip( s1, s1, ' ', ' ');   // strip leading/trailing spaces
                  rc = dfsHfsNodeSetLabel( rootNode, BT_SANE_LEVELS, s0, s1);
               }
            }
            else
            #endif
            {
               if (TxConfirm( 5165, "Write Volume LABEL '%s' to Catalog ? [Y/N] : ", s1))
               {
                  rc = dfsHfsNodeSetLabel( rootNode, BT_SANE_LEVELS, s0, s1);
               }
            }
         }
      }
   }
   else if ((strcasecmp(c0, "node"     ) == 0)) // calculate LSN for Catalog node N, and display
   {
      if (TxaOption('?'))
      {
         TxPrint("\nTranslate Catalog-Node-number to sectornumber, and display the node contents,\n"
                   "or tranlate the current sectornumber 'this' to a NODE-number, if any.\n");
         TxPrint("\n Usage:  %s  [node  [-r:rsel] [-p:psel] [-v]]\n\n"
                   "   node    = Catalog Node number, sequence number in Catalog-file 0 .. N\n\n"
                   "   -r:rsel = Rec-type: 0=all 1=DIR 2=FILE 3=Tdir 4=Tfile 5=D+F 6=Thread\n"
                   "   -p:rsel = Parent CNID selection, 0=all other= CNID of parent DIR\n"
                   "   -auto   = Automatic traversal of all leaf-nodes for a parent DIR\n"
                   "   -trace  = Include all leafnode headers for -auto and -p:0 (ALL records)\n"
                   "   -v      = verbose, dump start/end of node and record data as hex\n\n", c0);
      }
      else if (hfs->Catalog.Btree != NULL)
      {
         if (cc > 1)
         {
            nodeNr = dfsGetSymbolicSN( c1, 1);  // default NODE 1 (lowest leaf ?)
            sn = dfsHfsNode2Lsn( nodeNr);
            if      (sn == L64_NULL)
            {
               dfsX10(  "\nInvalid  NODE  nr : ", nodeNr, CBR, "\n");
               dfsDec8( "Max  NODE  number : ", TxBE32( hfs->Catalog.Btree->hdr.bthTotalNodes) -1, "\n");
            }
            else
            {
               hfs->preferedFolderId = 0;       // default ANY folder (to be refined)
               sprintf( dc, "0x0%llX", sn);
               TxaReParseKeepOpts( dc);         // reparse but keep option values
               rc = DFS_PENDING;                // handle translated command
            }
         }
         else
         {
            if ((nodeNr = dfsHfsLsn2Node( nav.this)) != L32_NULL)
            {
               dfsX10("Current   NODE  nr: ", nodeNr, CBC, "\n");
            }
            else
            {
               dfsX10("Current sector Lsn: ", nav.this, CBM, " is not a Catalog NODE\n");
            }
         }
      }
   }
   else if (strcasecmp(c0, "super"    ) == 0)
   {
      sprintf( dc, "0x0%llx", HFS_LSNSUP1);      // most likely location
      TxaReParseCommand( dc);
      rc = DFS_PENDING;                         // display the sector
   }
   else
   {
      rc = DFS_CMD_UNKNOWN;                     // cmd not recognized
   }
   RETURN (rc);
}                                               // end 'dfsHfsCommand'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// HFS  filesystem, identify specified sector as a valid superblock
// 20180613 JvW: Added more checks to avoid too many false positives in bsfind
/*****************************************************************************/
BOOL dfsHfsIsSuperBlock                         // RET   sector is a valid sb
(
   BYTE               *sec                      // IN    sector contents
)
{
   BOOL                rc = FALSE;
   S_HFS_SUPER        *sp = (S_HFS_SUPER *) sec;

   if ((TxBE16( sp->Signature) == HFS_CLAS_SIGNATURE) ||
       (TxBE16( sp->Signature) == HFS_PLUS_SIGNATURE) ||
       (TxBE16( sp->Signature) == HFS_XTND_SIGNATURE)  )
   {
      if ((sp->Version != 0) && (TxBE16( sp->Version) < 10)) // current is 4 or 5 ...
      {
         if ((sp->CreateDate != 0) && (sp->ModifyDate != 0))
         {
            switch (TxBE32(sp->LastMountedBy))  // only accept the four still used ...
            {
               case HFS_PLUS_MOUNTED:
               case HFS_PJRN_MOUNTED:
               case HFS_PFSK_MOUNTED:
               case HFS_CERD_MOUNTED:
                  rc = TRUE;
                  break;

               default:
                  break;
            }
         }
      }
   }
   return (rc);
}                                               // end 'dfsHfsIsSuperBlock'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// HFS filesystem, display sector constents as a superblock
/*****************************************************************************/
void dfsHfsDisplaySuperBlock
(
   BYTE               *sec                      // IN    sector contents
)
{
   dfsHfsSuperBlock( sec, FALSE);
}                                               // end 'dfsHfsDisplaySuperBlock'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// HFS filesystem, identify specified sector
/*****************************************************************************/
ULONG dfsHfsIdent
(
   ULN64               lsn,                     // IN    LSN for sector
   ULN64               d2,                      // IN    dummy
   char               *st,                      // OUT   sector type
   void               *sec                      // IN    sector contents
)
{
   ULONG               dr = NO_ERROR;
   BYTE                rc = ST_UDATA;
   S_BT_NODE_DESC     *nd = (S_BT_NODE_DESC *) sec;

   ENTER();

   if (dfsHfsIsSuperBlock( (BYTE *) sec))
   {
      rc = ST_HFSSUP;
   }
   else if ((dfsGetSectorSize() >= 2048) && (dfsHfsIsSuperBlock( (BYTE *) sec + 1024)))
   {
      rc = ST_HFSBSU;                           // combined Boot+Superblock sector
   }
   if (dfsHfsLsn2Node( lsn) != L32_NULL)        // inside Catalog file ?
   {
      rc = ST_HFSCAT;                           // default to unused space in catalog

      if ((nd->lPrev != L32_NULL) && (nd->lNext != L32_NULL))
      {
         switch (nd->nKind)                     // check kind of node
         {
            case BT_HeaderNode:      rc = ST_HFSCHD; break;
            case BT_LeafNode:        rc = ST_HFSCLF; break;
            case BT_MapNode:         rc = ST_HFSCMP; break;
            case BT_IndexNode:                  // value 0x00, check more fields
            default:
               if ((nd->nHeight != 0) && (nd->nRecords != 0))
               {
                  rc = ST_HFSCIN;               // seems like valid index node
               }
               break;
         }
      }
   }
   switch (rc)
   {
      case ST_UDATA:
         dr = DFS_PENDING;
         break;

      default:
         break;
   }
   *st = rc;
   RETURN (dr);
}                                               // end 'dfsHfsIdent'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// HFS filesystem, supply sector-type description string
/*****************************************************************************/
static ULONG dfsHfsStype
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *st,                      // IN    sector type
   void               *data                     // OUT   type description
)
{
   ULONG               dr  = NO_ERROR;
   char               *buf = (char *) data;
   BYTE                tp  = st[0];

   switch (tp)                                  // searchable types
   {
      case ST_HFSSUP: sprintf(buf,"HFS+ superblock  "); break;
      case ST_HFSBSU: sprintf(buf,"HFS boot+superblk"); break;
      default:
         switch (tp | ST__INFO)                 // non-searchable ones
         {
            case ST_HFSBFL: sprintf(buf,"Boot(loader) file"); break;
            case ST_HFSBMP: sprintf(buf,"Allocation Bitmap"); break;
            case ST_HFSEXT: sprintf(buf,"Extents    B-tree"); break;
            case ST_HFSATR: sprintf(buf,"Attributes B-tree"); break;
            case ST_HFSCAT: sprintf(buf,"Catalog  (unused)"); break;
            case ST_HFSCIN: sprintf(buf,"Catalog IndexNode"); break;
            case ST_HFSCLF: sprintf(buf,"Catalog  LeafNode"); break;
            case ST_HFSCMP: sprintf(buf,"Catalog   MapNode"); break;
            case ST_HFSCHD: sprintf(buf,"Catalog    Header"); break;
            default:       dr = DFS_PENDING;
               break;
         }
         break;
   }
   return (dr);
}                                               // end 'dfsHfsStype'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// HFS filesystem, display sector-contents based on type
/*****************************************************************************/
static ULONG dfsHfsSectorContents
(
   ULN64               psn,                     // IN    base psn for sector
   ULN64               d2,                      // IN    dummy
   char               *type,                    // IN    type of sector
   void               *data                     // IN    sector contents
)
{
   ULONG               dr   = NO_ERROR;
   BYTE               *sect = (BYTE *) data;
   BYTE                st   = (BYTE) *type;
   ULN64               lsn = dfstPsn2LSN( DFSTORE, psn);

   ENTER();

   switch (st)
   {
      case ST_HFSSUP:    dfsHfsSuperBlock(         sect, TRUE);               break;
      case ST_HFSCAT:    dfsHfsCatUnused(     lsn, sect, TRUE, 0xc0);         break;
      case ST_HFSCIN:    dfsHfsCatIndexNode(  lsn, sect, TRUE);               break;
      case ST_HFSCLF:    dfsHfsCatLeafNode(   lsn, sect, TRUE);               break;
      case ST_HFSCMP:    dfsHfsCatUnused(     lsn, sect, TRUE, 0x40);         break;
      case ST_HFSCHD:    dfsHfsCatHeader(     lsn, sect, TRUE);               break;

      case ST_HFSBSU:
         if (dfsGetSectorSize() >= 2048)        // safety check, avoid traps
         {
            if (((S_BOOTR*)sect)->Signature == SV_BOOTR)
            {
               TxPrint( "\nHFS+ Combined Boot+Superblock, with boot signature, BOOTRECORD contents:\n\n");
               dfsBootSector( sect);
            }
            else if (((S_CORE_ST*)sect)->Signature == CORE_ST_SIG) // possible CoreStorage info
            {
               TxPrint( "\nHFS+ Combined CoreStorage+Superblock, Core Storage information:\n\n");
               dfsFdskCoreStg( sect);
            }
            else                                // start does NOT have bootcode
            {
               TxPrint( "\nHFS+ Combined Boot+Superblock, WITHOUT boot/CoreStorage present, superblock only.\n\n");
            }
            dfsHfsSuperBlock( sect + 1024, TRUE);
         }
         else                                   // should not happen (for real EXTBSU)
         {
            dr = DFS_PENDING;
         }
         break;

      default:
         dr = DFS_PENDING;
         break;
   }
   RETURN (dr);
}                                               // end 'dfsHfsSectorContents '
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// HFS filesystem, display special LSN info (Node lsn + record-index value)
/*****************************************************************************/
static ULONG dfsHfsDisplayLsnInfo
(
   ULN64               lsn,                     // IN    possible special lsn
   ULN64               info,                    // IN    possible REC index
   char               *dc,                      // IN    dummy
   void               *data                     // IN    dummy
)
{
   ULONG               rc = DFS_PENDING;
   TXLN                path;

   ENTER();

   if (info & DFSSNINFO)                        // extra info present
   {
      TxPrint( "\n");
      if ((dfsHfsLsnInfo2Path( lsn, DFSSNIGET( info), path) == NO_ERROR) && (strlen( path)))
      {
         TxPrint("Found path to root: '%s%s%s'\n\n", CBY, path, CNN);
      }
      nav.xtra = lsn;                           // keep reference arround
      rc = dfsHfsCatalogRecord( lsn, DFSSNIGET( info));
   }
   RETURN (rc);
}                                               // end 'dfsHfsDisplayLsnInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display all available info about a Node Record, in a Catalog Node
/*****************************************************************************/
static ULONG dfsHfsCatalogRecord
(
   ULN64               nodeLsn,                 // IN    lsn of Catalog Node
   USHORT              index                    // IN    index of record 0..N
)
{
   ULONG               rc;
   BYTE               *node = NULL;             // Catalog node buffer
   BYTE                st;                      // sector type (node type)
   USHORT              offset;

   ENTER();
   TRACES(("nodeLsn: 0x%llx, entry: 0x%4.4hx\n", nodeLsn, index));

   if ((rc = dfsHfsReadChkCatNode( nodeLsn, &st, &node)) == NO_ERROR)
   {
      if ((offset = dfsHfsRecordOffset( index, node)) != 0)
      {
         S_HFS_CATALOG_KEY *key       = (S_HFS_CATALOG_KEY *) (node + offset);
         USHORT             keyLength = TxBE16( key->keyLength);
         U_HFS_CAT_RECORD  *catData   = (U_HFS_CAT_RECORD *) (((BYTE *) key) + keyLength + 2);
         USHORT             recType   = TxBE16( catData->t.crRecType);

         if ((recType == HFS_CR_Dthread) || (recType == HFS_CR_Fthread))
         {
            TXLN       threadName;

            TxPrint( "  Node-sector - Index R-Type this CnID   Parent-CnID   this Name\n");
            TxPrint( " ==================== ====== =========== ============= ============\n");
            dfsX10(  "   ", nodeLsn, CBC, "    ");
            TxPrint( "%s%4hu%s",     CNC, index, CNN);
            TxPrint( " %s id %8x Fldr %8x %s%s%s\n", dfsHfsCatRecType( recType),
                       TxBE32( key->parentId), TxBE32( catData->t.parentId),
                             CBY, TxMacUniStr2Ascii( &(catData->t.name), threadName),  CNN);
         }
         else                                   // must be regular File or Folder record
         {
            dfsHfsDirHeader( " Node-sector - Index", 0);

            dfsX10(  "   ", nodeLsn, CBC, "    ");
            TxPrint( "%s%4hu%s",     CNC, index, CNN);
            dfsHfsShowCatRecord( node + offset, CcZ);

            if      (recType == HFS_CR_File)
            {
               ULN64 forkSize = TxBE64( catData->file.crDataFork.logicalSize); // remember datafork size

               dfsHfsPrintForkdata( "\nStd Datafork size : ", SD_BOTLINE | SD_NAVDOWN, &(catData->file.crDataFork));
               if (TxBE64( catData->file.crRsrcFork.logicalSize) != 0)
               {
                  dfsHfsPrintForkdata( "\nResourcefork size : ", // set navigation when datafork empty
                                    (forkSize) ? SD_BOTLINE : SD_BOTLINE | SD_NAVDOWN, &(catData->file.crRsrcFork));
               }
            }
            else if (recType == HFS_CR_Folder)  // auto display folder contents
            {
               TXTM             command;

               sprintf(         command, "folder 0x%x", TxBE32( catData->d.crCnID));
               dfsMultiCommand( command, 0, TRUE, FALSE, TRUE); // include command echo
            }
         }
      }
      else
      {
         TxPrint( "Invalid node contents or record index: %hu\n", index);
      }
      TxFreeMem( node);
   }
   RETURN( rc);
}                                               // end 'dfsHfsCatalogRecord'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display HFS  super-block (Apple terminology: Volume Header)
/*****************************************************************************/
static ULONG dfsHfsSuperBlock                   // RET   rc = 0 if type match
(
   BYTE               *sector,                  // IN    sector data
   BOOL                navigation               // IN    update nav values
)
{
   ULONG               rc    = 0;               // rc, sector match
   S_HFS_SUPER        *sd    = (S_HFS_SUPER *) sector;
   TXTM                text;
   ULONG               VolAttributes;

   ENTER();
   TxPrint("HFS version value : 0x%4.4hx,      Sig: 0x%4.4hx = ",
                                    TxBE16(sd->Version), TxBE16(sd->Signature));
   switch (TxBE16(sd->Signature))
   {
      case HFS_CLAS_SIGNATURE: TxPrint( "Classic Apple HFS\n");           break;
      case HFS_PLUS_SIGNATURE: TxPrint( "HFS+, case insensitive\n");      break;
      case HFS_XTND_SIGNATURE: TxPrint( "HFS extended case sensitive\n"); break;
      default:                 TxPrint( "Unknown/invalid signature!\n");  break;
   }
   TxCopy(  text, (char *) &(sd->LastMountedBy), 5);
   TxPrint("Last mounted   by : 0x%8.8x = %s%s%s        = ",
                                     TxBE32(sd->LastMountedBy), CBY, text, CNN);
   switch (TxBE32(sd->LastMountedBy))
   {
      //-  Pre OSX versions have been removed from official macOS include files
      //-  HFS_OS89_MOUNTED: TxPrint( "HFS+ Mac OS 8.1 up to 9.2.2  \n"); break;
      case HFS_PLUS_MOUNTED: TxPrint( "HFS+ Mac OS X, non journaled \n"); break;
      case HFS_PJRN_MOUNTED: TxPrint( "HFS+ Mac OS X, journaled\n");      break;
      case HFS_PFSK_MOUNTED: TxPrint( "HFS+ Failed journal replay \n");   break;
      case HFS_CERD_MOUNTED: TxPrint( "HFS+ as used in disk images?\n");  break;
      default:               TxPrint( "Unknown, or not supported!\n");    break;
   }
   VolAttributes = TxBE32(sd->Attributes);
   TxPrint("Volume attributes : 0x%8.8x = ",   VolAttributes);
   if (VolAttributes)
   {
      if (VolAttributes & HFS_VA_HAS_A_JOURNAL)
      {
         TxPrint( "Has-Journal   ");
      }
      if (VolAttributes & HFS_VA_UNUSEDNODEFIX)
      {
         TxPrint( "Uses-NodeFix   ");
      }
      if (VolAttributes & HFS_VA_VOL_UNMOUNTED)
      {
         TxPrint( "Unmounted (clean)");
      }
      else
      {
         TxPrint( "Mounted  (dirty!)");
      }
      if (VolAttributes & HFS_VA_HARDWARE_LOCK)
      {
         TxPrint( "\n                             + Volume is locked by hardware");
      }
      if (VolAttributes & HFS_VA_SPARED_BLOCKS)
      {
         TxPrint( "\n                             + Volume has bad blocks spared");
      }
      if (VolAttributes & HFS_VA_NO_CACHE_REQD)
      {
         TxPrint( "\n                             + Don't cache volume blocks (RAMdisk)");
      }
      if (VolAttributes & HFS_VA_BOOTVOL_INCON)
      {
         TxPrint( "\n                             + Boot volume is inconsistent");
      }
      if (VolAttributes & HFS_VA_VOL_INCONSIST)
      {
         TxPrint( "\n                             + Serious inconsistencies detected at runtime");
      }
      if (VolAttributes & HFS_VA_SOFTWARE_LOCK)
      {
         TxPrint( "\n                             + Volume is locked by software");
      }
      if (VolAttributes & HFS_VA_NODEID_REUSED)
      {
         TxPrint( "\n                             + Catalog Node ID's (CnID) are reused");
      }
   }
   else
   {
      TxPrint( "No attributes at all, unusual!");
   }
   TxPrint("\n");

   dfsHfsPrintDateTime( "Volume created on : ", sd->CreateDate);
   dfsHfsPrintDateTime( "Last modified  on : ", sd->ModifyDate);
   dfsHfsPrintDateTime( "Last backed up on : ", sd->BackupDate);
   dfsHfsPrintDateTime( "Last checked   on : ", sd->CheckedDate);

   TxPrint("Number of files   : %10u   Directories : %10u   Next CnID  :  0x%8.8x\n",
         TxBE32( sd->FileCount), TxBE32( sd->DirCount), TxBE32( sd->NextCatalogID));
   TxPrint(  "Size each  block  : %10u   Bytes\n",      TxBE32( sd->BlockSize));
   TxPrint(  "Blocks on volume  : %10u,       ",        TxBE32( sd->TotalBlocks));
   dfsSz64("     Volume    size : ", ((ULN64)           TxBE32( sd->TotalBlocks) * hfs->SectorsPerBlock), "\n");
   TxPrint(  "Blocks still free : %10u,       ",        TxBE32( sd->FreeBlocks));
   dfsSz64("     Freespace size : ", ((ULN64)           TxBE32( sd->FreeBlocks ) * hfs->SectorsPerBlock), "\n");

   dfsHfsPrintForkdata( "\nBoot(loader) file : ", SD_DEFAULT, &(sd->Startupfile));
   dfsHfsPrintForkdata( "\nAllocation bitmap : ", SD_DEFAULT, &(sd->AllocationFile));
   dfsHfsPrintForkdata( "\nExtents    B-tree : ", SD_DEFAULT, &(sd->ExtentsFile));
   dfsHfsPrintForkdata( "\nAttributes B-tree : ", SD_DEFAULT, &(sd->AttributesFile));
   dfsHfsPrintForkdata( "\nCatalog    B-tree : ",
         (navigation) ? SD_BOTLINE | SD_NAVDOWN : SD_DEFAULT, &(sd->CatalogFile));

   if (navigation)
   {
      hfs->preferedFolderId = 2;                // default ROOT folder
      nav.down = dfsHfsNode2Lsn( 0);            // Next will be the Catalog Header, and
   }                                            // <Enter> goes to ROOT directory then
   RETURN (rc);
}                                               // end 'dfsHfsSuperBlock'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display HFS Catalog node, header-node for a B-tree
/*****************************************************************************/
static ULONG dfsHfsCatHeader                    // RET   rc = 0 if type match
(
   ULN64               lsn,                     // IN    lsn for sector
   BYTE               *sector,                  // IN    Node sector data
   BOOL                navigation               // IN    update nav values
)
{
   ULONG               rc    = 0;               // rc, sector match
   S_BT_HDR_NODE      *sd    = (S_BT_HDR_NODE *) sector;

   ENTER();

   dfsHfsNodeDescr( lsn, sector, FALSE);

   TxPrint( "Catalog node size :%5hu bytes   Root Node : 0x%s%8.8x%s  CompareCase: %ssensitive\n",
             TxBE16( sd->hdr.bthNodeSize),
        CBC, TxBE32( sd->hdr.bthRootNode), CNN,
                    (sd->hdr.bthCompareType == HFS_CaseFolding) ? "in" : "");

   TxPrint( "  Cat Btree depth :%4hu levels   NodeCount : 0x%8.8x  Free nodes : 0x%8.8x  Used:0x%8.8x\n",
             TxBE16( sd->hdr.bthDepth),
             TxBE32( sd->hdr.bthTotalNodes),
             TxBE32( sd->hdr.bthFreeNodes ),
             TxBE32( sd->hdr.bthTotalNodes) -
             TxBE32( sd->hdr.bthFreeNodes));

   TxPrint( "  Tree attributes : 0x%8.8x   LeafCount : 0x%8.8x  First Leaf : 0x%s%8.8x%s  Last:0x%s%8.8x%s\n",
             TxBE32( sd->hdr.bthAttributes),    // to be refined, big-endian ??
             TxBE32( sd->hdr.bthLeafCount),
        CBG, TxBE32( sd->hdr.bthFirstLeaf), CNN,
        CBY, TxBE32( sd->hdr.bthLastLeaf ), CNN);

   TxPrint("\nStart of node usage bitmap, in the catalog file:\n");
   TxDisplHexDump( sd->bTreeMap, 0x40);

   if (navigation)
   {
      ULN64            FirstLeafLsn;            // down
      ULN64            LastLeafLsn;             // xtra
      ULN64            RootNodeLsn;             // up

      FirstLeafLsn = dfsHfsNode2Lsn( TxBE32( sd->hdr.bthFirstLeaf));
      if ((FirstLeafLsn != 0) && (FirstLeafLsn != L64_NULL))
      {
         nav.down = FirstLeafLsn;               // valid LSN for first leaf node
      }
      LastLeafLsn = dfsHfsNode2Lsn( TxBE32( sd->hdr.bthLastLeaf));
      if ((LastLeafLsn != 0) && (LastLeafLsn != L64_NULL))
      {
         nav.xtra = LastLeafLsn;                // valid LSN for last leaf node
      }
      RootNodeLsn = dfsHfsNode2Lsn( TxBE32( sd->hdr.bthRootNode));
      if ((RootNodeLsn != 0) && (RootNodeLsn != L64_NULL))
      {
         nav.up   = RootNodeLsn;                // valid LSN for root node
      }
   }
   RETURN (rc);
}                                               // end 'dfsHfsCatHeader'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display HFS Catalog node, leaf-node with directory and file info
/*****************************************************************************/
static ULONG dfsHfsCatLeafNode                  // RET   rc = 0 if type match
(
   ULN64               lsn,                     // IN    lsn for sector
   BYTE               *sector,                  // IN    Node sector data
   BOOL                navigation               // IN    update nav values
)
{
   ULONG               rc    = 0;               // rc, sector match
   BYTE               *node;

   ENTER();

   //- read full node contents
   if ((node = TxAlloc( hfs->Catalog.spNode, dfsGetSectorSize())) != NULL)
   {
      S_BT_NODE_DESC  *desc = (S_BT_NODE_DESC *) node;
      ULONG            rsel = (TxaOptNum('r', NULL, 5));                      //- record type select
      ULONG            psel = (TxaOptNum('p', NULL, hfs->preferedFolderId));  //- parent selection
      ULN64            thisNodeNr;              // node nr  to be handled next (auto)
      ULN64            thisNodeLsn = lsn;       // node LSN to be handled next (auto)
      ULONG            parentId = 0;            // last parent-ID evaluated
      ULONG            listed = 0;              // number of records added to snlist
      ULONG            lastHeaderLsn = 0;       // Last leaf-node that header was shown for

      if (!TxaOptUnSet('l'))                    // create a new list for DIR
      {
         dfsInitList(0, "-f -P", "-d");         // optimal for menu file-recovery
      }

      do
      {
         if (dfsRead( thisNodeLsn, hfs->Catalog.spNode, node) == NO_ERROR)
         {
            S_HFS_CATALOG_KEY *key;
            U_HFS_CAT_RECORD  *catData;         // catalog data record folder/file/thread
            int                ac;              // alternate color fg+bg
            USHORT             rec;
            USHORT             nRecs = TxBE16( desc->nRecords); // number of records
            USHORT             freeB = hfs->Catalog.bpNode - ((nRecs + 1) * sizeof( USHORT))
                                         - dfsHfsRecordOffset( nRecs, node);
            if (dfsa->verbosity >= TXAO_VERBOSE)
            {
               TxDisplayHex( "\nStart of Leaf node data:\n", node, 0x140, 0);
               TxDisplayHex( "\nEnd of Leaf node, indexes:\n", node + hfs->Catalog.bpNode - 0xc0, 0xc0,
                                                                      hfs->Catalog.bpNode - 0xc0);
            }
            for (rec = 0; rec < nRecs; rec++)
            {
               USHORT     this = dfsHfsRecordOffset( rec,     node);
               USHORT     next = dfsHfsRecordOffset( rec + 1, node);
               USHORT     keyLength;
               USHORT     recType;              // record-type folder/file (or -thread)
               TXLN       keyName;
               TXTS       threadP;              // thread-parent-ID string
               ULONG      cnId;                 // CNID for this record

               key       = (S_HFS_CATALOG_KEY *) (node + this);
               keyLength = TxBE16( key->keyLength);
               parentId  = TxBE32( key->parentId);

               catData = (U_HFS_CAT_RECORD *) (((BYTE *) key) + keyLength + 2);
               recType = TxBE16( catData->t.crRecType);

               if (((rsel == 0) || (rsel   == recType)      ||
                   ((rsel == 5) && recType <= HFS_CR_File)  ||
                   ((rsel == 6) && recType >  HFS_CR_File)) &&
                   ((psel == 0) || (psel   == parentId)     ))
               {
                  //- Display Node header info when this is the FIRST listed item for that node
                  if ((thisNodeLsn != lastHeaderLsn)      &&
                      ((psel != 0) || (thisNodeLsn == lsn)||  //- Only 1st one when showing ALL
                        (TxaOption( TXA_O_TRACE))))           //- or each leafnode when -trace
                  {
                     TxPrint( "\n");
                     dfsHfsNodeDescr( thisNodeLsn, node, navigation);
                     TxPrint(  "Freespace in node :%5hu bytes   Showing records for parent-folderID: 0x%8.8x%s%s%s%s\n",
                                freeB, psel,  (psel <= 2) ? "  = "  : "", CBY,
                                              (psel == 2) ? "ROOT"  :
                                              (psel == 1) ? "LABEL" :
                                              (psel == 0) ? "ANY"   : "", CNN);
                     if (dfsa->verbosity < TXAO_VERBOSE)
                     {
                        TxPrint( "\n");
                        if ((rsel == 3) || (rsel == 4) || (rsel == 6)) // Thread format only ?
                        {
                           dfsHfsThreadHeader( listed);
                        }
                        else                    // Directory or mixed format
                        {
                           dfsHfsDirHeader( "Nr    Folder/File ID", listed);
                        }
                     }
                     lastHeaderLsn = thisNodeLsn;
                  }

                  if ((recType == HFS_CR_Dthread) || (recType == HFS_CR_Fthread))
                  {
                     TxMacUniStr2Ascii( &(catData->t.name), keyName);
                     sprintf( threadP, "Fldr %8x", TxBE32( catData->t.parentId));
                     cnId = parentId;           // record ID is in key 'parent' field!
                  }
                  else                          // regular File/Dir record
                  {
                     TxMacUniStr2Ascii( &(key->name), keyName);
                     strcpy(  threadP, "             ");
                     cnId = TxBE32( catData->d.crCnID);  // record ID is in the file/folder record itself
                  }

                  if (!TxaOptUnSet('l'))        // create a new list, allowing selective access
                  {
                     dfsAddSI2List( thisNodeLsn, rec);  // add to list
                  }

                  ac = (listed % 2) ? TXaBWnC : TXaNWnZ;
                  TxPrint( "%s.%07.7u r%03.3hx %s ", ansi[ac], listed, rec, dfsHfsCatRecType( recType));

                  if ((dfsa->verbosity >= TXAO_VERBOSE) || ((recType == HFS_CR_Dthread) || (recType == HFS_CR_Fthread)))
                  {
                     TxPrint( "id %8x %s", cnId, threadP); // actual record ID and parent for thread-records
                     TxPrint( " %s%s",      ansi[Ccol((CcY | CcI), Ccbg(ac))], keyName);
                     TxPrint( " %s%s%s\n",  ansi[Ccol( CcW, Ccbg(ac))], CGE, CNN);

                     if (dfsa->verbosity >= TXAO_VERBOSE)
                     {
                        USHORT  recStart = this + keyLength + 2;

                        TxDisplayHex( "Record data:\n", node + recStart, next - recStart, recStart);
                     }
                  }
                  else                          // display in DFSee 'directory' format
                  {
                     dfsHfsShowCatRecord( (BYTE *) key, Ccbg(ac));
                  }
                  listed++;
               }
            }
         }
         if (TxaOption( TXA_O_AUTO) &&          // when auto node requested
            ((listed == 0) ||                   // and nothing listed yet, or
             (psel   == 0) ||                   // any parent (list ALL!)
             (psel   == parentId)))             // or matching parent-ID
         {
            thisNodeNr  = TxBE32( desc->lNext);
            thisNodeLsn = dfsHfsNode2Lsn( thisNodeNr);
         }
         else                                   // last Node (for this parent) handled
         {
            thisNodeNr  = 0;                    // terminate ...
         }
      } while ((rc == NO_ERROR) && (thisNodeNr != 0) && !TxAbort());

      if (listed > 12)
      {
         if (dfsa->verbosity < TXAO_VERBOSE)
         {
            if ((rsel == 3) || (rsel == 4) || (rsel == 6)) // Thread format only ?
            {
               dfsHfsThreadHeader( listed);
            }
            else                                // Directory or mixed format
            {
               dfsHfsDirHeader( "Nr    Folder/File ID", listed);
            }
         }
      }
      else if (listed == 0)
      {
         TxPrint( "\nNo catalog records matching parent folder-ID %u = 0x%x\n", psel, psel);
      }
      TxFreeMem( node);
   }
   RETURN (rc);
}                                               // end 'dfsHfsCatLeafNode'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display HFS Catalog index-node with tree split-points to other nodes
/*****************************************************************************/
static ULONG dfsHfsCatIndexNode                 // RET   rc = 0 if type match
(
   ULN64               lsn,                     // IN    lsn for sector
   BYTE               *sector,                  // IN    Node sector data
   BOOL                navigation               // IN    update nav values
)
{
   ULONG               rc    = 0;               // rc, sector match
   S_BT_NODE_DESC     *desc = (S_BT_NODE_DESC *) sector;
   BYTE               *node;
   ULONG               psel = (TxaOptNum('p', NULL, 0)); // parent selection

   ENTER();

   dfsHfsNodeDescr( lsn, sector, navigation);

   //- read full node contents
   if ((node = TxAlloc( hfs->Catalog.spNode, dfsGetSectorSize())) != NULL)
   {
      if (dfsRead( lsn, hfs->Catalog.spNode, node) == NO_ERROR)
      {
         S_HFS_CATALOG_KEY *key;
         USHORT             listed = 0;
         USHORT             rec;
         USHORT             nRecs = TxBE16( desc->nRecords); // number of records
         USHORT             freeB = hfs->Catalog.bpNode - ((nRecs + 1) * sizeof( USHORT))
                                      - dfsHfsRecordOffset( nRecs, node);

         TxPrint(  "Freespace in node :%5hu bytes   Showing records for parent-folderID: 0x%8.8x\n", freeB, psel);

         if (dfsa->verbosity >= TXAO_VERBOSE)
         {
            TxDisplayHex( "\nStart of Index node data:\n", node, 0x140, 0);
            TxDisplayHex( "\nEnd of node, indexes:\n", node + hfs->Catalog.bpNode - 0xc0, 0xc0,
                                                              hfs->Catalog.bpNode - 0xc0);
         }

         if (!TxaOptUnSet('l'))                 // create a new list for DIR
         {
            dfsInitList( 0, "-w", "-d");        // with optimal list options
         }
         TxPrint( "\n");
         for (rec = 0; rec < nRecs; rec++)
         {
            USHORT     this = dfsHfsRecordOffset( rec,     node);
            USHORT     keyLength;
            TXLN       keyName;
            ULONG      parentId;
            ULONG      branchId;                // tree branch ID for index record

            key       = (S_HFS_CATALOG_KEY *) (node + this);
            keyLength = TxBE16( key->keyLength);
            parentId  = TxBE32( key->parentId);
            branchId  = TxBE32( *((ULONG *) (node + this + keyLength + 2)));

            TxMacUniStr2Ascii( &(key->name), keyName);

            if ((psel == 0) || (psel == parentId))
            {
               TxPrint( ".%7.7u %4hu Parent:0x%8.8x => Branch: 0x%8.8x name: '%s'\n",
                          listed++, rec, parentId, branchId, keyName);

               if (!TxaOptUnSet('l'))           // create a new list, allowing selective access
               {
                  dfsAdd2SectorList( dfsHfsNode2Lsn( branchId)); // add to list
               }
            }
         }
      }
      TxFreeMem( node);
   }
   RETURN (rc);
}                                               // end 'dfsHfsCatIndexNode'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display HFS Catalog node, unused or unknown/invalid contents
/*****************************************************************************/
static ULONG dfsHfsCatUnused                    // RET   rc = 0 if type match
(
   ULN64               lsn,                     // IN    lsn for sector
   BYTE               *sector,                  // IN    Node sector data
   BOOL                navigation,              // IN    update nav values
   ULONG               dumpsize                 // IN    number of bytes to dump
)
{
   ULONG               rc    = 0;               // rc, sector match

   ENTER();

   dfsHfsNodeDescr( lsn, sector, navigation);
   if (dumpsize)
   {
      TxPrint("\nStart of node:\n");
      TxDisplHexDump( sector, dumpsize);
   }
   RETURN (rc);
}                                               // end 'dfsHfsCatUnused'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display HFS node-descriptor structure (at start each node)
/*****************************************************************************/
static ULONG dfsHfsNodeDescr                    // RET   rc = 0 if type match
(
   ULN64               lsn,                     // IN    lsn for sector
   BYTE               *sector,                  // IN    Node sector data
   BOOL                navigation               // IN    update nav values
)
{
   ULONG               rc    = 0;               // rc, sector match
   S_BT_NODE_DESC     *sd    = (S_BT_NODE_DESC *) sector;

   ENTER();

   TxPrint("%s%s%s Nodeheight :%3hhu #rec:%3hu  "
           "This-Node : 0x%8.8x  Next-Node  : 0x%s%8.8x%s  Prev:0x%s%8.8x%s\n",
       CBM, dfsHfsNodeType( sd->nKind), CNN, sd->nHeight, TxBE16( sd->nRecords), dfsHfsLsn2Node( lsn),
            (navigation) ? CBG : "", TxBE32( sd->lNext), CNN,
            (navigation) ? CBY : "", TxBE32( sd->lPrev), CNN);

   if (navigation)
   {
      ULN64            NextNodeLsn;             // down
      ULN64            PrevNodeLsn;             // up

      NextNodeLsn = dfsHfsNode2Lsn( TxBE32( sd->lNext));
      if ((NextNodeLsn != 0) && (NextNodeLsn != L64_NULL))
      {
         nav.down = NextNodeLsn;                // valid LSN for next node
      }
      PrevNodeLsn = dfsHfsNode2Lsn( TxBE32( sd->lPrev));
      if ((PrevNodeLsn != 0) && (PrevNodeLsn != L64_NULL))
      {
         nav.xtra = PrevNodeLsn;                // valid LSN for prev node
      }
   }
   RETURN (rc);
}                                               // end 'dfsHfsNodeDescr'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Make filesystem usage map dump on TxPrint output
/*****************************************************************************/
static ULONG dfsHfsAllocMap
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *options,                 // IN    display options
   void               *data                     // OUT   dummy
)
{
   ULN64               i;                       // index in data-area
   ULONG               b;                       // byte-index for one mapchar
   ULN64               cClus = 0;               // Clusters in current char
   ULN64               lClus = 0;               // Clusters in current line
   ULN64               mClus = 0;               // Clusters in current map
   ULN64               uClus = 1;               // Last cluster that is in use
   ULONG               l;                       // line number, 0 based
   ULONG               a;                       // index in display-array
   ULN64               perc;                    // percentage
   TX1K                ascii;                   // display-array
   ULONG               spe;                     // sectors per entry
   ULONG               acl;                     // ascii alloc chars per line
   ULONG               cpc;                     // clusters per display-char
   ULONG               spc;                     // sectors per display-char
   ULN64               size;                    // nr of clusters to map
   BOOL                verbose = (*options != '@');
   ULONG               bsSmart;
   ULN64               unallocSmart = 0;        // Size of THIS unallocated area
   BOOL                oneLineProgress = FALSE; // alloc in one line, output per character

   ENTER();

   dfsa->FsUnallocSmart = 0;                    // will be recalculated now
   bsSmart = dfsGetBufferSize( DFSOPTIMALBUF, DFSMAXBUFSIZE); // for SMART prediction

   spe   = hfs->SectorsPerBlock;
   size  = hfs->BlockCount;                     // Block 0 .. last-valid

   acl   = dfsAllocCharsPerLine( 'c');
   cpc   = dfsAllocItemsPerChar( 'l', options, size, acl);
   if (cpc >= (size / acl))                     // will be a single line
   {
      oneLineProgress = TRUE;                   // one char at a time, no ascii string used
   }
   spc   = cpc * spe;
   dfsProgressInit( 0, size * spe, 0, "Get ALLOCATION info, at Sector:", dfstStoreDpid( DFSTORE), DFSP_STAT, 0);

   if (verbose)
   {
      dfsSz64("Size for one line : ", (ULN64) acl * spc, "  ");
      TxPrint("with %3u characters,", acl);
      dfsSiz8(" size : ", spc, "\n");
      TxPrint(" %16.16s = Allocation grade, empty to full.     ", mapchar);
      dfsSiz8(" SmartBuf size : ", bsSmart, "\n");
   }
   TxPrint(   "          %s%*.*s%s\n", CBC, acl, acl, BLIN, CNN);
   dfsX10( CNZ, dfsHfsBlock2Lsn(0), CNZ, "");  // display address for first block
   TxPrint("%s%s", CBC, CNC);

   for (i=0, a=0, l=0; (i < size) && (!TxAbort());)
   {
      if (a == 0)
      {
         memset(ascii, 0, TXMAXLN);
         lClus = 0;
      }
      for (cClus=0, b=0; (b < cpc) && (i < size); b++, i++)
      {
         if (dfsHfsBitmapCache(i, NULL))        // block in use ?
         {
            if (unallocSmart != 0)              // first in-use after unallocated?
            {
               if (unallocSmart >= bsSmart)                  //- if at least ONE smart buffer
               {
                  unallocSmart -= (bsSmart / 2);             //- compensate for alignment issues
                  unallocSmart -= (unallocSmart % bsSmart);  //- clip to whole smart buffer size

                  dfsa->FsUnallocSmart += unallocSmart;      //- add this area to total
               }
               unallocSmart = 0;                // reset for next area to come
            }
            cClus++;
            uClus = i;                          // last in-use one sofar
         }
         else                                   // unallocated, Smart predict
         {
            unallocSmart += spe;                // add to size THIS area
         }
      }
      lClus += cClus;

      dfsProgressShow( (i * spe) + spc, 0, 0, NULL);

      TRACES(("lClus 2nd Loop i: %llu, b:%lu = %llu\n", i, b, lClus));
      if (oneLineProgress == TRUE)
      {
         TxPrint( "%c", (char) ((cClus == 0) ? ' ' : mapchar[cClus * 8 / cpc])); // slow, acts as progress bar
      }
      else
      {
         ascii[a] = (char) ((cClus == 0) ? ' ' : mapchar[cClus * 8 / cpc]);
      }
      a++;
      if ((i && ((i%(acl*cpc)) == 0)) || (i >= size))
      {
         if (oneLineProgress == FALSE)
         {
            TxPrint( "%s", ascii);              // display accumulated chars (fast)
         }
         perc  = (ULN64)(100*lClus / (a*cpc));
         if (a == acl)
         {
            TxPrint("%s%s% 3u%%\n", CBC, CNN, perc);
         }
         else
         {
            TxPrint("%s%.*s%s% 3u%%\n", CBC, (int) (acl-a-1), BLIN, CNN, perc);
         }
         if (i < size)
         {
            dfsX10( CNZ, dfsHfsBlock2Lsn(i), CNZ, "");
            TxPrint("%s%s", CBC, CNC);
            a = 0;                              // keep a value on last line
         }
         mClus += lClus;
         l++;
      }
   }
   if (unallocSmart != 0)                           //- unalloc area pending?
   {
      if (unallocSmart >= bsSmart)                  //- if at least ONE smart buffer
      {
         unallocSmart -= (bsSmart / 2);             //- compensate for alignment issues
         unallocSmart -= (unallocSmart % bsSmart);  //- clip to whole smart buffer size

         dfsa->FsUnallocSmart += unallocSmart;      //- add this area to total
      }
   }
   TxPrint("          %s%.*s%s\n", CBC, (USHORT) a, BLIN, CNN);

   if (TxAbort())
   {
      dfsa->FsUnallocSmart = 0;                 // signal info NOT available!
      TxPrint( "\nAllocation check and display aborted! Used/Free info not available.\n");
   }
   else
   {
      if (size != 0)                            // avoid devide by 0
      {
         ULN64            used = mClus * spe;

         dfsa->FsUnallocated   = size * spe - used; // free space (optimize hint)

         dfsSz64("Unallocated sects : ", dfsa->FsUnallocated, " ");
         dfsSz64("SmartUse ", dfsa->FsUnallocSmart, " ");
         TxPrint("=% 5.1lf%%\n", (double) (100.0 * ((double) dfsa->FsUnallocSmart) / (double) (size * spe)));

         dfsSz64("Allocated sectors : ", used, " ");
         dfsSz64("of total ", size * spe, " ");
         TxPrint("=% 5.1lf%%\n", (double) (100.0 * ((double) mClus) / (double) size));
      }
      dfsa->FsTruncPoint = dfsHfsBlock2Lsn( uClus +1); // first free (truncate)
      dfsa->FsLastInUse  = dfsa->FsTruncPoint -1;

      #if defined (HFS_RESIZING)
         dfsSz64( "Minimum vol. size : ", dfsa->FsTruncPoint, " (for  Resizing)\n");
      #endif
   }
   dfsProgressTerm();
   RETURN (NO_ERROR);
}                                               // end 'dfsHfsAllocMap'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// DFS HFS write-file to disk (SaveTo to file)
// Supports automatic resolve of hardlinks, when enabled in the (menu) option
/*****************************************************************************/
static ULONG dfsHfsFileSaveAs
(
   ULN64               leafLsn,                 // IN    Leaf-node sectornumber
   ULN64               leafIdx,                 // IN    Leaf-node record index
   char               *path,                    // IN    destination path
   void               *recp                     // IN    recovery params (rename)
)
{
   ULONG               rc  = NO_ERROR;
   S_RECOVER_PARAM    *param = (S_RECOVER_PARAM *) recp;
   DFSISPACE           isp;                     // integrated alloc SPACE info
   BOOL                isDir = FALSE;
   TXLN                fname;                   // destination base filename
   TXLN                fullfn;                  // unique filename prefix
   ULONG               parent;                  // parent CnId (unused)

   ENTER();
   TRARGS(("Leaf node LSN : 0x%llX, record index: %lld, to path: '%s'\n", leafLsn, leafIdx, path));

   //- to be refined for files with Resource, but no data-fork ?
   if ((rc = dfsHfsGetAllocSpace( leafLsn, leafIdx, NULL, &isp)) == NO_ERROR)
   {
      if ((param->newname == NULL) || (*param->newname == 0)) // no newname present
      {
         if (dfsHfsCnId2Parent( dfsHfsLsnInfo2CnId( leafLsn, DFSSNIGET( leafIdx), NULL),
                                                    &parent, fname, &isDir) != NO_ERROR)
         {
            sprintf( fname, "Node_%12.12llX_%4.4hX", leafLsn, DFSSNIGET( leafIdx));
         }
      }
      else                                      // rename specified
      {
         strcpy( fname, param->newname);
      }
      if (param->unique)                        // force unique naming
      {                                         // on PATH and FILE components
         sprintf( fullfn, "%12.12llX_%4.4llX_%s", leafLsn, leafIdx, fname);
         strcpy(  fname, fullfn);               // and move back to fname
      }

      rc = dfsSspaceFileSaveAs( &isp, isDir, FALSE, param->noAllocCheck, param->name83, path, fname, fullfn);
      if ((rc == NO_ERROR)       ||             // set original timestamps
          (rc == DFS_CMD_WARNING) )             // even when allocation errors present
      {
         BYTE               *node = NULL;       // Catalog node buffer
         BYTE                st;                // sector type (node type)
         USHORT              offset;

         if ((rc = dfsHfsReadChkCatNode( leafLsn, &st, &node)) == NO_ERROR)
         {
            if ((offset = dfsHfsRecordOffset( DFSSNIGET( leafIdx), node)) != 0)
            {
               time_t      mod;
               time_t      cre;
               time_t      acc;
               S_HFS_CATALOG_KEY *key       = (S_HFS_CATALOG_KEY *) (node + offset);
               USHORT             keyLength = TxBE16( key->keyLength);
               U_HFS_CAT_RECORD  *catData   = (U_HFS_CAT_RECORD *) (((BYTE *) key) + keyLength + 2);
               BOOL isHrdLink = ((catData->d.f.type == HFS_FT_HRDLINK) && (catData->d.f.creator == HFS_FC_HRDLINK));

               //- resolve hardlink, if any, to get proper data values from the actual file Inode
               if ((isHrdLink) && (hfs->hlMetaFolderId != 0) && (HfsHardlinksAutoResolve == TRUE))
               {
                  ULONG              nodeNr;
                  USHORT             index;     // index in new node
                  TXTT               tbuf;      // hardlink iNode name buffer

                  TxFreeMem( node);         // free original node, to be replaced with resolved one

                  sprintf( tbuf, "iNode%d", TxBE32( catData->d.crBsdInfo.iNodeNum));
                  nodeNr = dfsHfsSearchCatalog( hfs->hlMetaFolderId, tbuf, &index, &node);
                  if ((nodeNr != 0) && ((nodeNr != L32_NULL)))
                  {
                     if (dfsHfsNodeIdx2Rec( node, index, &catData) == HFS_CR_File)
                     {
                        TRACES(("Resolved hardlink, morphed catData to that record for date/time\n"));
                     }
                  }
               }
               mod = dfsHfsFileTime2t( catData->d.crModContTime);
               cre = dfsHfsFileTime2t( catData->d.crCreTime);
               acc = dfsHfsFileTime2t( catData->d.crAccTime);

               TRACES(("Set original date/time for: '%s'\n", fullfn));

               TxSetFileTime( fullfn, &cre, &acc, &mod); // set time on full-fn from SaveAs

               if (param->recFname)             // return full recovered path+filename
               {
                  strcpy( param->recFname, fullfn);
               }
            }
            TxFreeMem( node);
         }
      }
      TxFreeMem( isp.space);
   }
   RETURN(rc);
}                                               // end 'dfsHfsFileSaveAs'
/*---------------------------------------------------------------------------*/

