//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// Filesystem dump & display utility, generic commands
//
// Author: J. van Wijk
//

#include <txlib.h>                              // TX library interface

#include <dfsver.h>                             // DFSee version info
#include <dfsrgkey.h>                           // Registration interface
#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfswin.h>                             // windowed entry point
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsdgen.h>                            // DFS generic dialogs
#include <dfsmdisk.h>                           // Memory disk interface
#include <dfsvoem.h>                            // DFS  SVISTA major functions
#include <dfsimage.h>                           // image create restore
#include <dfsclone.h>                           // cloning and moving
#include <dfsrsize.h>                           // partition resize/expand
#include <dfsshift.h>                           // sector data shifting
#include <dfsscan.h>                            // bad-sector & speed scan
#include <dfsfind.h>                            // find sector data
#include <dfswipe.h>                            // wipe functions
#include <dfsutil.h>                            // DFSee utility functions
#include <dfsspace.h>                           // DFS  file-space interface
#include <dfshexed.h>                           // DFS  HEX editor interface
#include <dfstable.h>                           // SLT utility functions
#include <dfsupart.h>                           // FDISK partition functions
#include <dfsauxil.h>                           // Auxiliary analysis module
#include <dfsagen.h>                            // Generic command handling
#include <dfsahpfs.h>                           // HPFS  interface (for Init)
#include <dfsafat.h>                            // FAT   interface (for Init)
#include <dfsantfs.h>                           // NTFS  interface (for Init)
#include <dfsajfs.h>                            // JFS   interface (for Init)
#include <dfsafdsk.h>                           // FDISK interface (for Init)
#include <dfsufdsk.h>                           // FDISK utility functions
#include <dfscfdsk.h>                           // FDISK callback functions
#include <dfsdembr.h>                           // FDISK PTE MBR dialog functions
#include <dfsufgpt.h>                           // FDISK GPT utility functions
#include <dfsdegpt.h>                           // FDISK PTE GPT dialog functions
#include <dfsalist.h>                           // sector Area list handling
#include <dfsbrows.h>                           // Sectorlist browsing functions

char         *dfsGenericOpts[] =
{
   "",
   "Many DFSee commands accept options, as a '-' sign plus a letter or name string",
   "and an optional value that are specific to the command. There are a few options",
   "however that are implemented on a generic level and work for ALL commands:",
   "",
   "  -?            = Give usage on this specific command   (first and only option)",
   "  -B            = Batch mode ON, so no confirmations asked     (use with care!)",
   "  -B-           = Batch mode OFF regular confirmations asked          (default)",
   "  -D            = Display DELETED files in FAT/EFAT directory views",
   "  -D-           = Ignore  DELETED files in FAT/EFAT (like DIR command, default)",
   "  -E:[c|i|q]    = error strategy : Confirm, Ignore or Quit (confirm is default)",
   "  -echo         = Display  the 'DFSxxx executing: ...' echo line      (default)",
   "  -echo-        = suppress the 'DFSxxx executing: ...' echo for this command",
   "  -l            = Create new sectorlist when displaying (DIR) data    (default)",
   "  -l-           = Do NOT create new list when displaying (directory) data",
   "  -O:s|q|n|v|m  = Set output verbosity Silent, Quiet, Normal, Verbose, Maximum",
   "  -P            = Use prompting and input dialog windows where needed (default)",
   "  -P-           = Don't use prompting and input dialog windows (aka batch mode)",
   "  -q            = Command output reduction on some commands,   similar to -O:q",
   "  -Q   or -Q:3  = Quiet screen mode, turn screen off while executing command",
   "  -Q:2 or -Q:3  = Quiet log mode, turn logfile off while executing command",
   "  -silent       = No command output at all, on some commands,  similar to -O:s",
   "  -trace        = Explicitly switch tracing ON  for this command, level 100",
   "  -trace-       = Explicitly switch tracing OFF for this command",
   "  -v            = Verbose command output   on some commands,   similar to -O:v",
   "  -X            = Set next sector to the MFT/Fnode/Inode/Fat data     (default)",
   "  -X-           = Do NOT set next sector to the MFT/Fnode/Inode/Fat file data",
   "",
   NULL
};


char         *dfsGenericHelp[] =
{
   "",
   "DFSee application specific commands",
   "===================================",
   "",
   " mcs-number       = Analyse & display sector, SN using mcs-number sector LSN",
   " ?                = Show current FS (mode), specific commands and help info",
   " ??               = Show list of all generic commands with short description",
   " ???              = Show all recognized sector-types for current filesystem",
   " <Enter>          = Show next sector with <Enter>   (same as d/DOWN/n/NEXT)",
   "",
   " .NNN             = Display numbered FS-entity marked .NNN from Sectorlist",
   " A|a  sSN  [size] = ASCII-dump (half)sector, at sSN, 'size' sectors or bytes",
   " ABOUT            = Show version and registration information for DFSee/TPlib",
   " AL [sn] [sz] [o] = Display/Write allocation status for sector/cl/block range",
   " ALBIT a [lsn sz] = Set/reset allocation bit in bitmap for sector/cluster range",
   " ALLOC  [options] = Display allocation map in various size, show RESIZE limits",
   " BASE   [options] = Show (and set) base start and end values in several ways",
   " BATCH  [on |off] = Set batch-mode, no confirmation or locking-error reporting",
   " BPS      secsize = Force sector size to specified value (decimal, bytes)",
   " BROWSE [. | dir] = Browse contents of (DIR) Sectorlist, or specified directory",
   " BSFIND [type][a] = Find MBR/EBR/boot/LVM-sectors, on cylinders or ALL sectors",
   " CV [select][opt] = Copy selected value to the clipboard as a (Hex) string",
   " CL               = Translate and display 'this' LSN as a cluster number",
   " CL   clust [cmd] = Translate cluster number to hex LSN, display using cmd",
   " CL2S clust       = Translate cluster number to hex LSN, display LSN value",
   " CS          [sn] = Calculate 32-bit checksum for 'this' sector; HPFS aware",
   " CHECK    [dr/nr] = Report all detected filesystem errors using SLT info",
   " CHS c  h  s      = Analyse & display sector, using CHS (cylinder head sector)",
   " CLONE  [options] = Copy sectors from other (disk/part/vol) to current store",
   " CMD    [command] = Force execution of command by Operating System, not DFSee",
   " COMP   [options] = Compare sectors from other (disk/part/vol) to current store",
   " d | DOWN [Stype] = Display 'next' sector as Hex or specified sector-type",
   " DELSHOW [sel]    = Show delfind list, with selection wildcard and threshold",
   " DFSCHECK[part|*] = Run DOxxx analysis for one or all partitions => DFSC*.P*",
   " DFSDISK [disk|*] = Run DFSDISK analysis for one or all disks => dfsdiski.*",
   " DIRFIND [l][dir] = Put [l] levels of filenames from [dir] in the Sectorlist",
   " DISK   [disk-nr] = Select a physical disk and show the MBR, [-r] = reset geo",
   " DISKLIST         = List available disk numbers plus info (for stdout, use -G)",
   " EDIT  [-a -h -u] = String replacement edit, using Ascii, Hex or Unicode string",
   " EDIT   [options] = Start HEX sector-editor on current or specified location",
   " EDIT    filename = Start HEX sector-editor on specified file (binary edit)",
   " EXPAND [options] = EXPAND FS to a new LARGER  size, incl. part-table update",
   " EXPORT      [fn] = Export  LSNs in the Sectorlist, to file [fn]",
   " FDISK [FdiskCmd] = Activate the FDISK fs (mode) and optionally execute cmd",
   " FI         [sSN] = File information on specified symbolic-SN, default 'this'",
   " FIND options [s] = Find sectors based on options, and optional string [s]",
   " FINDPATH   path  = Find and show file/directory specified by path-spec",
   " FINDSPACE        = Show filefind/delfind find-SPACE (DIR/MFT/FNODES/INODES)",
   " FIXPBR [options] = Fix HiddenSectors and Geometry bootsector to match p-table",
   " FS fsys-name     = Force analysis mode for filesystem 'fsys-name'",
   " GB    [distance] = Go backwards 'distance'; an mcs-number, default is 1 sector",
   " GF    [distance] = Go forward   'distance'; an mcs-number, default is 1 sector",
   " GEO  ?|c h s f b = Set/Display specified geometry on opened object, ?=adjust",
   " H|h   SSN [size] = Hex-dump   (half)sector, at sSN, 'size' sectors or bytes",
   " ID  [ SSN | -L ] = Identify 'this', specified sSN or LAST sector, using SLT",
   " IM    file [opt] = Open a file with FS-image (.img/.imz) for display & analysis",
   " IMAGE file [opt] = Save an object to imagefile(s) using specified options",
   " IMZ    imz [opt] = Start BROWSE session on selected IMZ compressed image file",
   " IMZ2RAW  imz raw = Create uncompressed (RAW) image from an IMZ compressed one",
   " IMPORT      [fn] = Import  LSNs to the Sectorlist, from file [fn]",
   " LABEL   [label]  = Display/edit 11-char volume label in bootsector / alt-loc",
   " LIST [opt] [str] = Display LSNs in the Sectorlist, using options/string",
   " LISTSET  sn [sz] = Add 'sz' sectornumbers starting from 'sn' to the Sectorlist",
   " LOCK             = Lock physical disk or logical partition to avoid conflicts",
   " MAP [disknr opt] = Display graphical map of partitions on a disk",
   " MEDIA            = Display mapping for Partitionable Media to phys/mem/imz ...",
   " MEMSTAT          = Display properties and status for DFSee in-Memory disks",
   " MENU [pd-letter] = Activate the menu, opening default or selected pulldown",
   " MENU2HTML   [pd] = Generate HTML/PHP documentation for specified pull-down menu",
   " MOVE -b|-e | -f  = Move towards Begin or End, or move/copy to Freespace area",
   " PART [pid|drive] = Show partitions,  or select one using nr or drive-letter",
   " n | NEXT [Stype] = Display 'next' sector as Hex or specified sector-type",
   " PRITEMS x [name] = Set number of items/passes for the progress indicators",
   " PSN    xx        = Analyse & display sector xx, using physical sector-number",
   " PT   [pid|drive] = Display partition-table for current or specified pid/drive",
   " PTE   [xx | pid] = Show & edit partition-table at sector [xx] or part [pid]",
   " QUERY   item [n] = Query specified item value for disk/partition/freesp [n]",
   " RAW2IMZ  raw imz = Create a compressed imagefile (IMZ) from a RAW image file",
   " RECOVER path sel = Recover files from Sectorlist to a path, using selection",
   " REPEAT [command] = Repeat a (multi) command string, counted or forever ...",
   " RESIZE [options] = RESIZE FS to a new SMALLER size, incl. part-table update",
   " RESTORE    file  = Restore an imagefile to the currently selected object",
   " RUN [script] [p] = Run a DFSee script, with optional parameters and options",
   " S2CL lsn         = Translate hex LSN to hex cluster number and display that",
   " SAVETO [d][l][n] = Save filedata for current file to new file, opt. rename",
   " SCAN [wr [NoId]] = Scan bad-sectors; use read/Write verify; No automatic id",
   " SCL       clsize = Force cluster size to specified value (decimal, sectors)",
   " SERIAL  [serial] = Set new Volume Serial Number in the bootsector for a FS",
   " SET [prop value] = Set DFSee property to a value, 'SET' lists all properties",
   " SHIFTL [options] = Shift data inside sectors towards begining (aka LEFT)",
   " SHIFTR [options] = Shift data inside sectors towards the end (aka RIGHT)",
   " SLT  [type i ln] = Display SLT for sectors of 'type' at index i, ln lines",
   " SORT   [-d] [-u] = Sort Sectorlist; [-d] = descending [-u] = no duplicates",
   " STORE [store-nr] = Select store to be the current store; or display stores",
#if defined (DEV32)
   " SYNCLVM vol [dr] = Sync LVM engine to use [dr] letter for volume, or HIDE it",
#endif
   " t | this [Stype] = Display 'this' sector as Hex or specified sector-type",
   " TRANS [-type] xx = Translate and display value 'xx' as Lsn, I13, Chs and Psn",
   " u | UP   [Stype] = Display 'up'   sector as Hex or specified sector-type",
   " UNDELETE [fname] = Find deleted files and Browse the resulting Sectorlist",
   " UNLOCK           = Unlock physical disk or logical partition",
   " VDI  [vdi] [opt] = Open VirtualBox Base/Snapshot diskimage for BROWSE/Analysis",
   " VER [-g][-t:str] = Display a single line of version/builddate information",
   " VOL      [drive] = Show all volumes, or select one using drive-letter",
   " WIPE   pat [f s] = Wipe disk/vol/part area with pattern at LSN [f], size [s]",
   " WIPEFREE  [part] = Wipe freespace in a filesystem to all zeroes (wipe -f)",
   " WIPELST [sel][p] = Wipe [sel]ected sectors in list with specified [p]attern",
   " WIPESECURE  [-f] = Wipe disk/vol/part area in a SECURE way (DoD spec, slow!)",
   " WRITE [ . | sn ] = Write sector back to current or specified sector number",
   " x | XTRA [Stype] = Display 'xtra' sector as Hex or specified sector-type",
   " q                = Quit",
   NULL
};

static  char       *cmd_slt[] =
{
   "",
   "Display Sector Lookup Table at specified position and size",
   "",
   "usage: SLT   [pos  [size]]   [options]",
   "",
   "   pos     : Position, start index in the SLT to display as one of:",
   "             n     = start display at absolute number 'n'",
   "             +[n]  = forward  one screen or 'n' lines from current pos",
   "             -[n]  = backward one screen or 'n' lines from current pos",
   "             'a'   = all, display whole SLT from index 0, start to end",
   "             'e'   = end, display the LAST page in the SLT, to the end",
   "",
   "   size    : lines to display, default one screen full; '*' display ALL",
   "",
   "   -1        = Output on a single line, with -v verbose       (default)",
   "   -1-       = Output on multiple lines with verbose -v",
   "   -a[:lsn]  = Display one page of SLT for sectornumbers around 'lsn'",
   "               with a default of the current sectornumber 'this'.",
   "   -c        = Check sector/clust allocation consistency   (implies -r)",
   "   -d        = Use the given -v and -1 values as new defaults (default)",
   "   -d-       = Do not set the given -v and -1 values as new defaults",
   "   -e[:size] = Display last page or 'size' lines upto end of table",
   "   -lfn      = Show long (FAT) filenames not the short ones   (default)",
   "   -lfn-     = Do NOT show long (FAT) filenames but short ones",
   "   -m[:mask] = Display errors only, optionally filtered by a HEX mask.",
   "               Each bit set represents a specific error    (implies -c)",
   "   -M        = Force creation of 'magic' recovery names to",
   "               (unmounted) JFS filesystems. (JFS only)",
   "   -r        = Refresh SLT even when one already exists",
   "   -t:type   = Restrict display to specified area types only c(??? cmd)",
   "   -v        = Set verbose error and path+filename display    (default)",
   "   -v-       = Set verbose OFF for error-text and path+filename display",
   "   -x        = Stop building SLT in background, discard SLT",

   "",
#if defined (HAVETHREADS)
   "   -x        = Stop building SLT in background, discard collected info",
#endif
   "",
   "   -P        = Prompt mode, use an option dialog to specify parameters",
#if defined (HAVETHREADS)
   "               while already building the SLT in the background.",
#endif
   "",
   NULL
};

static  char       *cmd_part[] =
{
   "",
   "Display partition overview as a table, or select specific partition",
   "",
   " Usage: PART  [options]  [part-id | part-id,r | driveletter | . ]",
   "",
   "   part-id      : Absolute id (pid) for the partition to select",
   "   number,r     : Relative nr on CURRENT or '-d:nr' specified disk",
   "   .            : Reselect current partition if any, or just created",
   "   driveletter  : driveletter (A..Z) for the partition to select",
   "",
   "   -2           : Check consistency 1st versus 2nd FAT      (default)",
   "   -2-          : Suppress checking of 2nd FAT consistency (FAT only)",
   "   -a           : Show allocation map for partition on open / listing",
   "   -a-          : Don't show allocation map for partition   (default)",
#if defined (UNIX)
   "   -b           : UNIX-style table, values in blocks        (default)",
   "   -b-          : PC-style table, with label info, no blocks",
#else
   "   -b           : UNIX-style table, label plus values in blocks",
   "   -b-          : PC-style table, label info, values in MB  (default)",
#endif
   "   -c           : Perform a filesystem-check on each partition",
   "   -C[:'str']   : Compact space or 'str' separated partition list",
   "   -d[:disknr]  : Display/select for current or specified disk only",
   "   -e or -e:1   : To add info on (1st) extended containers to table",
   "   -F           : Force use of First EXT-container",
   "   -f           : Display freespace areas as separate lines (default)",
   "   -f-          : Do NOT display any separate lines for freespace",
   "   -fs:'fsname' : Set mode to filesystem 'fsname' on selected partition",
   "   -file:[name] : XML name, default dfsdisks.rdf, -file- is STDOUT",
   "   -g           : Show disk geometry too           (implies -r)",
   "   -i           : Industry std letter assignment   (implies -r)",
   "   -L[:0|1|2]   : Show prefered: none (0), LVM (1), Windows (2)",
   "   -label:XXX   : XML identifying string, default 'dfs' or 'ecs'",
   "   -m           : Identify MBR, and show I13 limit (implies -g)",
   "   -n           : To get a table with all numerical information",
   "   -O:q|n|v|m   : Output: Quiet, Normal, Verbose, Maximum; default normal",
   "   -p           : Display partition information lines       (default)",
   "   -p-          : Do NOT display partition lines, just freespace",
   "   -q           : Quiet, display minimal partition details",
   "   -R           : Select the partition in Read-only mode",
   "   -r           : Re-read partition-tables from disk  (refresh)",
   "   -raw         : Allow type 0x00 (Deleted) with valid start/size",
   "   -raw-        : Needs non-zero type/size, start may be 0  (default)",
   "   -s[:'str']   : Verbose comma or 'str' separated partition list",
   "   -S           : Use double-Spaced lines with -s or -C formats",
   "   -t           : Display partition information as a table  (default)",
   "   -t-          : Do NOT display partitions as table, just warnings",
   "   -u           : Add UUID 16-character string when supported by FS",
   "   -w           : Include ALL warnings (default is MINOR ones only)",
   "   -w-          : Do NOT display any warnings at all, just the table",
   "   -w:m or -w:t : Show all MINOR warnings, or just the non-trivial",
   "   -x[:fsnames] : Add FS-specific info to 'part -s' or XML output",
   "   -xml         : Partition info in XML format to a file or STDOUT",
   NULL
};

static  char       *cmd_map[] =
{
   "",
   "Show pseudo graphical allocation map for one or more disks",
   "",
   " Usage: MAP   [disknr | * | .]  [+ | -]",
   "",
   "   disknr|*|.   : disknumber to show map for, * = all; . = current",
   "   + | -        : + is more verbose map, - is minimal sized map",
   "",
   "   -d[:disknr]  : Display for current or specified disk only",
   "   -g           : Show disk geometry too           (implies -r)",
   "   -i           : Industry std letter assignment   (implies -r)",
   "   -m           : Identify MBR, and show I13 limit (implies -g)",
   "   -M           : Display a partitionable Media map too",
   "   -r           : Re-read partition-tables from disk  (refresh)",
   "",
   NULL
};

#if defined (USEWINDOWING)
static  char       *cmd_pte[] =
{
   "",
   "Edit the partition table(s) for a disk or specific partition",
   "",
   " Usage: PTE  [options]  [part-id] | [sector-nr   [entry-nr]]",
   "",
   "   part-id      : id for the partition to edit the table for",
   "   sector-nr    : partition-table sector number, default current",
   "   entry-nr     : partition-table entry to get focus, default 1st",
   "",
   "   -f           : Force adding 0x55AA signature if not present",
   "   -d:disknr    : disk to edit partition-tables on, default current",
   "   -i:img       : image filename containing the partition-table",
   "                  (should be a disk-image to be useful :-)",
   "   -p:part      : partition to edit table for, number or letter",
   "   -s-          : Turn smart field updating off, startup default is on",
   "",
   NULL
};
#endif                                          // USEWINDOWING


static  char       *cmd_base[] =
{
   "",
   " Usage: BASE  . | * | sSN  [end-sn]",
   "    or: BASE  -a:start-pos [-s:size] [-logical]",
   "    or: BASE  -p:part-id | -t:part-id | -f:fspace-id [-logical]",
   "",
   "   .            : set base to start at current sector, to end of disk",
   "   *            : keep base the same, but specify a different sl value",
   "   sSN          : starting sector number for area to view, hexadecimal",
   "                : as direct or a symbolic-sector number (see DFSTERMS)",
   "   end-sn       : ending   sector number for area to view, hexadecimal",
   "   -a:pos       : area start position as mcs-number (see DFSTERMS.TXT)",
   "   -f:fspace-id : set start to begin of the freespace area specified",
   "   -l           : track-align start-position for a logical partition",
   "   -p:part-id   : set start to bootsector of specified partition id",
   "   -s:size      : area size as mcs-number, end area is start + size",
   "   -t           : set start to partition-Table for current partition",
   "   -t:part-id   : set start to partition-Table for specified part id",
   "                  ID for freespace (and partition) from table left-column ",
   "",
   " The mcs-number format is [0x|0t]nnnn[,g|m|k|c|h|s] HEX or decimal value,",
   " in GiB, MiB, KiB, Cylinders or Sectors. Default is decimal MiB.",
   "",
   NULL
};


static  char       *cmd_edit[] =
{
   "",
   "Edit sectors from currently opened object or from specified (binary) file",
   "using the interactive sector editor with full navigation through the object,",
   "or modify current sector using a simple hex/ascii/unicode string-replacement.",
   "",
   " Usage: EDIT  [SN | filename | [-dir:path [wildc]] [-p:pos] [Options]",
   "",
   "   SN           : starting sector number for interactive sector editor",
   "                : as direct or a symbolic-sector number (see DFSTERMS)",
   "                  (alternative to -F option, default is current sector)",
   "",
   "   filename     : Name of a file to be edited in binary (HEX/ASCII)",
   "",
   "   -dir[:path]  : Select file from FileDialog, start in 'path' directory",
   "   wildc        : Wildcard file selection in FileDialog, like *.bin",
   "",
   "   -p:pos       : numeric offset in sector where editing starts",
   "",
   "                  Options valid with interactive HEX sector editor:",
   "   -A           : Allow append-to and delete-from (binary) file   (default)",
   "   -A-          : No append-to or delete-from (binary) file, size is fixed",
   "   -B           : Batch mode while editing, no confirm on writing changes",
   "   -dir[:path]  : Select file from FileDialog, start in 'path' directory",
   "   -c:cols      : number of columns in the edit window (1..256, default 16)",
   "   -v:view      : Start view, default: 0=HEX, 1=DISASM, 2=ASCII, 9=AUTO",
   "   -e:entry     : Directory entry number for FAT DIR sector 0 .. 15",
   "                  Useful only if the current sector (this) is a FAT",
   "                  directory sector, with 16 entries of 32 bytes each",
   "   -F:mcs       : start sector-nr to be edited     (default current sector)",
   "   -m:bytes     : size in bytes for initial marked area   (see Alt-B/L/U/F)",
   "   -M:position  : start position of an initial marked area in edit window",
   "   -r:rows      : number of rows in edit dialog  (default screen -5)",
   "   -s:size      : size of each item in sectors  (default one cluster/block)",
   "   -screen      : start HEX-edit window maximized, full-screen",
   "",
   "                  Options valid with STR (string-replacement) editor:",
   "   -P           : Prompt dialogs for position and replacement string",
   "   -a[:'str']   : ASCII     replacement string, prompt when no value",
   "   -h[:'hex']   : HEX pairs replacement string, prompt when no value",
   "   -u[:'str']   : UNICODE   replacement string, prompt when no value",
   "",
   NULL
};



static  char       *cmd_list[] =
{
   "Display sector numbers from the Sectorlist, using optional wildcard string",
   "",
   " USAGE: list  [options]  [param]",
   "",
   "   Options: -b        = Verbose bootsec, for BSFIND/UNFDISK analysis",
   "            -C        = FAT only, display location as CLUSTER, not sector",
   "                        Combine with '-e' for EA or '-I' for EA-RelativeClust",
   "            -c        = Execute a command for each LSN, plus 1 line contents",
   "            -d        = Display sector contents, in their default format",
   "            -d-       = Suppress contents display (used with '-c' option)",
   "            -d:hex    = Sector contents, default + hex dump",
   "            -e        = FAT only, display EA-data location, not regular data",
   "            -f        = Verbose PATH info when available (is default)",
   "            -f:i      = Verbose PATH, indirect using SLT information",
   "            -f:8.3    = Verbose PATH, use short 8.3 name (NTFS only)",
   "            -g        = Show lsn -> psn -> CHS logical geometry",
   "            -g:system = Show lsn -> psn -> CHS system  geometry",
   "            -I        = FAT only, show decimal EA-index value as location",
   "            -lfn      = FAT only, show long-filenames, when present  (default)",
   "            -lfn-     = FAT only, show short instead of long filenames with -f",
   "            -l        = Displaying a directory REPLACES current list (default)",
   "            -l-       = Do NOT create new list when displaying directory",
   "                        information. This allows displaying a list of found",
   "                        directories, like resulting from the SUBFIND command.",
   "                        The -l- option is FORCED when using the '-d' option.",
   "                        (this option is also added to the cmd given with -c)",
   "            -n        = No contents list, just show number of entries",
   "            -r        = Show as a cluster/block (badsect) Range, incl. size",
   "            -s        = Show size-offset in MiB, type from list, CHS and PSN",
   "            -t        = Show size-offset in MiB, type from data, CHS and PSN",
   "            -t        = TEST    options are passed-trough to a cmd given with -c",
   "            -t-       = TEST    options NOT passed-trough to a cmd given with -c",
   "            -v        = Use verbose output, one line per LSN        (default)",
   "            -v        = VERBOSE options are passed-trough to a cmd given with -c",
   "            -v-       = VERBOSE options NOT passed-trough to a cmd given with -c",
   "            -w        = Show in Wide format, 6 LSNs and INFO on each line",
   "            -X        = Only display files that have EA-data attached",
   "",
   "   param   optional   For the '-c' option, it is the command to be executed for",
   "                      each sectornumber in the list (but options are discarded).",
   "                      Only '-v', '-v-', '-t' or '-t-' options will be passed on.",
   "                      The options are mutual exclusive, except for the -f:",
   "                      where option values may be combined as in -f:i8.3",
   "                      and the -C, -e, and -I modifiers for that -f option.",
   "                      Only the 1st char of an option value is significant.",
   "",
   "                      For the '-f' option it is a selection string with criteria",
   "                      separated by '%' characters, in the following syntax/order:",
   "",
   "    w%t%m%M%S%d     = File selection filter string, containing:",
   "",
   "    w = wildcard    : wildcard selection on full path and filename string",
   "    t = percentage  : minimum file-allocation-OK percentage",
   "    m = minsize     : minimum filesize as MCS number like 1,s",
   "    M = maxsize     : maximum filesize as MCS number like 999,k",
   "    S = filter type : D=Directories F=Files B=Browse-mode N=Normal (all)",
   "    d = modifydate  : Restrict to files modified AFTER date or #days-ago",
   "",
   "    Default filter  : '*%0%0%*%N%*' => any percentage/size, Normal, any date",
   "",
   NULL
};

#if defined (USEWINDOWING)
static char dfsRecoverHints[] =
   "When selecting a destination directory for the recovery, make sure it is "
   "on ANOTHER partition to avoid overwriting any still recoverable data.\n\n"
   "Review the recovery options (checkboxes) at the top of that dialog too!";
#endif

#if !defined (OEM_BRANDED)
// printf template for HTML menu header, need to fill in basename and to other menu ...
static char dfsMenu2HtmlNavFmt19[] =
   "<TABLE WIDTH=\"100%%\" BORDER=\"1\" FRAME=\"hsides\" CELLPADDING=\"0\">\n"
   " <TR ALIGN=\"CENTER\" BGCOLOR=\"#a8b8ff\"> <TD> <B>\n"
   "  <A HREF=\"%s-file.htm\">    File</A> &nbsp;&#124;&nbsp;\n"
   "  <A HREF=\"%s-edit.htm\">    Edit</A> &nbsp;&#124;&nbsp;\n"
   "  <A HREF=\"%s-actions.htm\"> Actions</A> &nbsp;&#124;&nbsp;\n"
   "  <A HREF=\"%s-image.htm\">   Image</A> &nbsp;&#124;&nbsp;\n"
   "  <A HREF=\"%s-display.htm\"> Display</A> &nbsp;&#124;&nbsp;\n"
   "  <A HREF=\"%s-scripts.htm\"> Scripts</A> &nbsp;&#124;&nbsp;\n"
   "  <A HREF=\"%s-help.htm\">    Help</A>  &nbsp;&#124;&nbsp;\n"
   "  <A HREF=\"%s-file.htm\">    (to %s menu)</A> </B> </TD> </TR>\n"
   " <TR ALIGN=\"CENTER\" BGCOLOR=\"#a8b8ff\">  <TD> <B>\n"
   "  <A HREF=\"%s-fdisk.htm\">   Mode=FDISK</A> &nbsp;&#124;&nbsp;\n"
   "  <A HREF=\"%s-aux.htm\">     AUX</A> &nbsp;&#124;&nbsp;\n"
   "  <A HREF=\"%s-extn.htm\">    EXTn</A> &nbsp;&#124;&nbsp;\n"
   "  <A HREF=\"%s-fat.htm\">     FAT</A> &nbsp;&#124;&nbsp;\n"
   "  <A HREF=\"%s-hfs.htm\">     HFS+</A> &nbsp;&#124;&nbsp;\n"
   "  <A HREF=\"%s-hpfs.htm\">    HPFS</A> &nbsp;&#124;&nbsp;\n"
   "  <A HREF=\"%s-jfs.htm\">     JFS</A> &nbsp;&#124;&nbsp;\n"
   "  <A HREF=\"%s-ntfs.htm\">    NTFS</A> &nbsp;&#124;&nbsp;\n"
   "  <A HREF=\"%s-reiser.htm\">  REISER</A> &nbsp;&#124;&nbsp;\n"
   "  <A HREF=\"%s-xfs.htm\">     XFS</A> &nbsp;&#124;&nbsp;\n"
   "  <A HREF=\"%s-swap.htm\">    SWAP</A> </B> </TD> </TR>\n"
   "</TABLE>\n";
#endif





/*****************************************************************************/
// Interpret and execute the generic DFSee commands
/*****************************************************************************/
ULONG dfsGenericCommand
(
   void                                         // arguments using TxParse
)
{
   ULONG               rc;
   ULONG               ei = 0;                  // entry index
   ULN64               sn = 0;                  // sector number input
   ULN64               sl = 0;                  // sector LSN
   BYTE                st = 0;                  // sector type, default none
   LONG                nr = 0;
   TXLN                dc;                      // DFS command
   int                 cc = 0;                  // command string count
   char               *c0, *c1, *c2, *c3;       // parsed command parts
   char               *c4, *c5, *c6, *c7;
   TXLN                s0, s1;                  // temporary string space
   char               *pp;                      // parameter pointer
   DFS_PARAMS          params;                  // generic function params
   ULONG               ef = 0;                  // Error flags mask
   TXA_OPTION         *opt;                     // option pointer
   SEC512              ms;                      // mixed-string buffer
   USHORT              ml  = 0;                 // mixed string length
   USHORT              ml2 = 0;                 // mixed string-2 length
   ULN64               sectors = 1;             // default display size
   USHORT              index = 0;               // part or freespace id
   DFSPARTINFO        *p = NULL;                // part or freespace info
   ULONG               h1, h2, h3, h4;          // hex values
   ULN64               size = 0;                // size value (to be cleaned up!)
   ULONG               sltSize;
   TXLN                msgText;
   USHORT              sninfo = 0;              // explicit sninfo (-e option)

   //- Note: could save 2.5 K stackspace here (and other command-handlers)
   //- by allocating 5 512-byte buffers in dfsa->dc, s0, s0, s1, msgTxt (char *) and sec512 (BYTE *)
   //- and exit when allocation fails. Then just assign here: s0 = dfsa->s0; etc ...

   ENTER();

   pp = TxaGetArgString( TXA_CUR, 0, 0, TXMAXLN, dc); // dc = pp => cmd from arg 0
   cc = TxaArgCount( );                         // number of parameters
   c0 = TxaArgValue(0);
   c1 = TxaArgValue(1);
   c2 = TxaArgValue(2);
   c3 = TxaArgValue(3);
   c4 = TxaArgValue(4);
   c5 = TxaArgValue(5);
   c6 = TxaArgValue(6);
   c7 = TxaArgValue(7);

   #if defined (USEWINDOWING)
      if (dfsa->sbwindow)
      {
         TxaGetArgString( TXA_CUR, 0, TXA_OPT, TXMAXLN, s1);
         sprintf( s0, "DFSee executing: %s", s1);
         txwSendMsg( dfsa->sbwindow, TXWM_STATUS, (TXWMPARAM) s0, (TXWMPARAM) cSchemeColor);
      }
   #endif                                       // USEWINDOWING
   while (*pp && (*pp != ' ')) pp++;            // skip 1st word = command
   while (*pp && (*pp == ' ')) pp++;            // skip spaces, point at parameters

   rc = NO_ERROR;                               // default result
   sn = nav.this;                               // default at 'this' sector

   TRACES(("cc: %u c0:'%s' c1:'%s' c2:'%s' c3:'%s' c4:'%s' c5:'%s' c6:'%s' c7:'%s' pp:'%s'\n",
            cc,    c0,     c1,     c2,     c3,     c4,     c5,     c6,     c7,     pp));

   if (TxaOptUnSet('e'))
   {
      nav.this_sninfo = 0;                      // reset auto sninfo (saveto)
   }

   if ((strlen(c0) > 1) &&                      // more than one char
       (strchr(". +-",  c0[0]) == NULL)  &&     // not a special format
       (txIsValidHex(   c0)    == FALSE) &&     // no valid hex number
       (txHasMcsPrefix( c0)    == FALSE))       // and no MCS-prefix
   {
      if (strcasecmp(c0, "???"      ) == 0)
      {
         dfsDisplaySectorTypes();
      }
      #if defined (DEV32)
      else if (strcasecmp(c0, "configsys"   ) == 0)
      {
         ULONG       *data = NULL;

         if (DosAllocMem( (PPVOID)&data,
                          65536L,
                          PAG_COMMIT|PAG_READ|PAG_WRITE)
                          == NO_ERROR)
         {
            *data = 0;

            rc = DosSysCtl( SYSCTL_FUNC_CONFIG, data);

            TxDisplHexDump( (BYTE *) data, 8192);

            DosFreeMem( data);
         }
      }
      #endif
      else if (strcasecmp(c0, "raw"   ) == 0)
      {
         strcpy( s1, "");
         dfstrSizeBps( s1, "Raw disk size is  : ", dfsRawDiskSize(), dfsGetSectorSize(), "");
         TxPrint( "\n%s\n\n", s1);
      }
      else if (strcasecmp(c0, "base"   ) == 0)
      {
         BOOL             ok = TRUE;            // arg/option combinations OK

         ok &= TxaOptMutEx((cc > 1), "fapt", "if 'start-sn' is specified!", NULL);

         if (cc > 1)                            // at least one parameter
         {
            if      ((c1[0] == '.')  ||         // use current sector as base
                     (strcasecmp(c1, "this") == 0))
            {
               TxPrint( "\nSet base to current sector 'this'\n");
               sn = dfstLSN2Psn( DFSTORE, nav.this);
            }
            else if ((c1[0] == '*')  ||         // keep current value
                     (strcasecmp(c1, "same") == 0))
            {
               TxPrint( "\nKeep base-start the same\n");
               sn = dfstLSN2Psn( DFSTORE, 0);
            }
            else                                // values supplied
            {
               TxPrint( "\nSet base to specified HEX / symbolic value\n");
               sn = dfsGetSymbolicSN( c1, 0);
            }
            if (cc > 2)                         // two HEX parameters
            {
               if ((c2[0] == '.')  ||           // use current sector as end
                  (strcasecmp(c2, "this") == 0))
               {
                  TxPrint( "\nSet base-end to current sector 'this'\n");
                  sl = dfstLSN2Psn( DFSTORE, nav.this);
               }
               else
               {
                  sl = dfsGetSymbolicSN( c2, 0);
               }
            }
            else
            {
               sl = dfstLSN2Psn( DFSTORE, 0) + dfsGetLogicalSize() -1;
            }
         }
         else                                   // no start-sn, check options
         {
            if (TxaOption('s') && (!TxaOptSet('a')))
            {
               TxPrint( "\n* '-s' size only valid with '-a' absolute position");
               ok = FALSE;
            }
            else if ((opt = TxaOptValue('a')) != NULL) // absolute base start
            {
               ULONG   cSC = ((ULONG) dfstGeoHeads(  DFSTORE) *
                                      dfstGeoSectors(DFSTORE));

               ok &= dfsOptionSizeSectors( 's', DFSTORE, 'm', 0, &size);

               TxPrint( "\nSet base to specified absolute position\n");
               switch (opt->type)
               {
                  case TXA_STRING:
                     TxPrint( "\n* Need a NUMBER value for '-a' absolute position");
                     ok = FALSE;
                     break;

                  default:
                     switch (opt->unit)
                     {
                        case 's':               // sectors, OK as is
                           sn = opt->value.number;
                           break;

                        case 'h': case 't':     // heads (or tracks :-)
                           sn = opt->value.number * dfstGeoSectors(DFSTORE);
                           break;

                        case 'c':               // cylinders
                           sn = opt->value.number * cSC;
                           break;

                        case 'k':               // Kilobytes
                           sn = DFSKB2SECTS( opt->value.number, dfsGetSectorSize());
                           if (sn != 0)         // perform cylinder roundup
                           {
                              sn--;             // subtract 1 before rounding
                              sn /= cSC;        // cylinders, rounded down
                              sn += 1;          // round up nr of cylinders
                              sn *= cSC;        // multiply with cyl-size
                           }
                           break;

                        default:                // Megabytes or Gigabytes
                           sn = DFSMB2SECTS( opt->value.number, dfsGetSectorSize());
                           if (opt->unit == 'g')
                           {
                              sn *= 1024;       // Giga correction
                           }
                           if (sn != 0)         // perform cylinder roundup
                           {
                              sn--;             // subtract 1 before rounding
                              sn /= cSC;        // cylinders, rounded down
                              sn += 1;          // round up nr of cylinders
                              sn *= cSC;        // multiply with cyl-size
                           }
                           break;
                     }
                     if (opt->unit != 's')
                     {
                        if ((TxaOption('l')) || // adjust for logical (EBR)
                            (sn == 0))          // or first primary   (MBR)
                        {
                           sn      += dfstGeoSectors(DFSTORE);
                           if (size > dfstGeoSectors(DFSTORE))
                           {
                              size -= dfstGeoSectors(DFSTORE);
                           }
                        }
                     }
                     if (size != 0)
                     {
                        sl = sn + size -1;
                     }
                     else
                     {
                        sl  = DFS_MAX_PSN;
                     }
                     break;
               }
            }
            else if ((opt = TxaOptValue('f')) != NULL) // freespace
            {
               switch (opt->type)
               {
                  case TXA_NO_VAL:
                  case TXA_STRING:
                     TxPrint( "\n* Need a NUMBER value for '-f' freespace option!");
                     ok = FALSE;
                     break;

                  default:
                     index = (USHORT) opt->value.number;
                     break;
               }
               if ((p = dfsGetFreeInfo( index)) != NULL)
               {
                  TxPrint( "\nSet base to specified freespace area\n");
                  sn = p->basePsn;
                  sl = p->lastPsn;
                  if (sl > p->geoSecs)
                  {
                     if (TxaOption('l'))        // logical ?
                     {
                        if ((sn == 0) &&        // start of disk, 1st logical
                            (sl > (p->cSC + p->geoSecs)))
                        {
                           sn += p->cSC;        // skip 1st cylinder
                        }
                        sn += p->geoSecs;       // skip 1st track (EBR)
                     }
                     else if (sn == 0)          // 1st primary
                     {
                        sn += p->geoSecs;       // skip 1st track (MBR)
                     }
                  }
                  dfsSelectDisk( p->disknr, FALSE, FALSE); // quietly select disk

                  if ((SINF->object == DFSO_DISK) && (sn != 0))
                  {
                     SINF->object = DFSO_FRSP;  // freespace area
                  }
               }
               else
               {
                  TxPrint("\n* Invalid freespace specification\n");
                  ok = FALSE;
               }
            }
            else if (((opt = TxaOptValue('p')) != NULL) || // partition
                     ((opt = TxaOptValue('t')) != NULL)  ) // Table sector
            {
               switch (opt->type)
               {
                  case TXA_NO_VAL:
                     if (SINF->p != NULL)       // partition is selected
                     {
                        index = SINF->partid;
                     }
                     else
                     {
                        TxPrint( "\n* Need a PARTITION to be selected!");
                        ok = FALSE;
                     }
                     break;

                  case TXA_STRING:
                     index = dfsDrive2PartId( opt->value.string);
                     if (index == 0)            // not found
                     {
                        ok = FALSE;
                        TxPrint("\n* Partition for drive '%c': not found\n",
                                     opt->value.string[0]);
                     }
                     break;

                  default:
                     index = (USHORT) opt->value.number;
                     break;
               }
               if ((p = dfsGetPartInfo( index)) != NULL)
               {
                  if (TxaOptSet('t'))
                  {
                     TxPrint( "\nSet base to P-table of specified partition\n");
                     sn = p->partPsn;
                  }
                  else
                  {
                     TxPrint( "\nSet base to specified partition area\n");
                     sn = p->basePsn;
                  }
                  sl = p->lastPsn;
                  dfsSelectDisk( p->disknr, FALSE, FALSE); // quietly select disk
               }
               else
               {
                  TxPrint("\n* Invalid partition specification\n");
                  ok = FALSE;
               }
            }
            else
            {
               ok = FALSE;
            }
         }
         if (ok)
         {
            nav.down = nav.up = nav.this = 0;   // reset to begin!
            nav.this_sninfo = 0;
            if (sl < sn)
            {
               sl = sn;
            }
            dfstLogicalLimits( DFSTORE, sn, sl);
            if (sn != 0)
            {
               SINF->object = DFSO_BASE;        // not whole object anymore
            }
            if ((SINF->p          != NULL) &&   // it was a partition
                (SINF->p->basePsn != sn  )  )   // and not at start now
            {
               SINF->p      = NULL;             // reset partition selection
               SINF->partid = 0;
               strcpy(  SINF->drive, "--");
            }
         }
         if (TxaOption('?'))                    // explicit help request
         {
            TxShowTxt( cmd_base);
         }
         else
         {
            strcpy( s1, dfstStoreName( DFSTORE));
            s1[12] = 0;                        // get 12-character name

            nav.xtra = dfsGetLogicalSize() -1;
            sn = dfstLSN2Psn( DFSTORE, 0); // base  .. end
            sl = sn + nav.xtra;                 // start + size -1
            TxPrint("\n%s%s%s Base : 0x0%llX - 0x0%llX", CBG, s1, CNN, sn, sl);
            dfsSz64( "     Size : ", sl - sn +1, "\n");
            if (SINF->disknr != 0)
            {
               dfsGeoDispTransPsn( "Base start sector :", dfstGeoHeads(  DFSTORE), dfstGeoSectors(DFSTORE), sn);
               dfsGeoDispTransPsn( "Base last  sector :", dfstGeoHeads(  DFSTORE), dfstGeoSectors(DFSTORE), sl);
            }
         }
      }
      else if (strcasecmp(c0, "grep"     ) == 0)
      {
         #if defined (USEWINDOWING)
            if (TxaOption('?'))                 // explicit help request
            {
               TxPrint("\nSearch for a string in output window, show search result list\n");
               TxPrint("\n Usage:  %s  [phrase]\n"
                       "\n   phrase = One or more words to search for\n", c0);
            }
            else
            {
               if (cc > 1)                      // at least one parameter
               {
                  txwSbViewFindRequest( TXWS_REQ_GREPSTRING, pp);
               }
               else
               {
                  txwSbViewFindRequest( TXWS_REQ_GREPDIALOG, "");
               }
            }
         #else
            TxPrint("Grep only supported in windowed mode\n");
         #endif
      }
      else if ((strcasecmp(c0, "search") == 0) || // generic scrolbuffer search
               (c0[0]) == '/')
      {
         #if defined (USEWINDOWING)
            if (TxaOption('?'))                 // explicit help request
            {
               TxPrint("\nSearch for a string in output window, highlight search result\n");
               TxPrint("\n Usage:  %s [phrase]\n"
                       "\n   phrase = One or more words to search for\n", c0);
            }
            else
            {
               if (c0[0] == '/')
               {
                  pp = dc+1;                    // start of search argument
               }
               if ((cc > 1) || (strlen(pp)))    // at least one parameter
               {
                  txwSbViewFindRequest( TXWS_REQ_FINDSTRING, pp);
               }
               else
               {
                  txwSbViewFindRequest( TXWS_REQ_FINDDIALOG, "");
               }
            }
         #else
            TxPrint("Search only supported in windowed mode\n");
         #endif
      }
      else if (strcasecmp(c0, "label"  ) == 0)
      {
         if (TxaOption('?'))                    // explicit help request
         {
            TxPrint("\nSet volume label in partition bootsector and alternate location.\n");
            TxPrint("\n Usage:  %s  [label]  [options]\n\n"
                    "   label  : new volume label, maximum 11 characters\n"
                    "   -!     = Prompt for new value, before setting it (default)\n"
                    "   -!-    = do not prompt for new value, just set it\n"
                    "   -a     = Write label to alternate location too   (default)\n"
                    "   -a-    = do not write alternate location\n", c0);

            TxPrint("\n NOTE: This will update the ON-DISK label information, which may\n"
                    "       or may not be directly visible with commands like DIR or the\n"
                    "       DFSee volume selection lists. It migh require a REBOOT of\n"
                    "       the system for the new label to be visible everywhere!\n\n"
                    "       A DFSee 'part' command (<F5> key) should show it immediately.\n");

         }
         else
         {
            if (cc > 1)
            {
               strcpy( s1, c1);                 // writable copy of parameter, or empty
            }
            else if (SINF->p)                   // it is a partition
            {
               strcpy( s1, SINF->p->plabel);    // use label determined by readdiskinfo
            }
            else
            {
               strcpy( s1, "");                 // let LabelBrec get it from boot record
            }
            if ((rc = dfsSetVolumeLabelBrec( !TxaOptUnSet('!'), s1)) == NO_ERROR)
            {
               TxCancelAbort();                 // might have canceled the dialog

               if (dfsa->FsSetAltBrecLabel != NULL)
               {
                  rc = DFSFNCALL( dfsa->FsSetAltBrecLabel, 0, 0, s1, NULL);
               }
            }
         }
      }
      else if (strcasecmp(c0, "ver" ) == 0)     // single line of version information
      {
         if (TxaOption('?'))                    // explicit help request
         {
            TxPrint("\nDisplay a line of version/builddate/packaging information\n");
            TxPrint("\n Usage:  %s  [options]\n\n"
                      "  -update : Mark as 'Update' not 'Current', add packaging date/time\n"
                      "  -lines  : Embed version information in a partial box of  '*' lines\n"
                      " EXE-switch '-g' : Output line to STDOUT, even in -q quiet mode\n", c0);
         }
         else
         {
            TXLN       line;                    // version info line
            TXTM       star = {0};              // star embedding line
            time_t     tt = time( &tt);         // current date/time

            TxPClip(  star, 83, '*');
            if (TxaOption( TXA_O_UPDATE))
            {
               TXTM    pack;                    // packaging date/time line (update only)

               strftime( pack, TXMAXTM, "(Packaged: %Y-%m-%d %H:%M:%S)", localtime( &tt));
               sprintf( line, "Update  %s version: %s, build date: %s %s\n",  DFS_NP, DFS_VN, DFS_VD, pack);
               if (TxaOption( TXA_O_LINES))
               {
                  strcat( line, star);
                  strcat( line, "\n");
               }
            }
            else                                // currently running version
            {
               sprintf( line, "%s\nCurrent %s version: %s, build date: %s\n",
                                       (TxaOption( TXA_O_LINES)) ? star : "", DFS_NP, DFS_VN, DFS_VD);
            }

            TxPrint( "%s", line);
            if (TxaExeSwitch('g'))
            {
               printf( "%s", line);
               fflush( stdout);
            }
         }
      }
      else if (strcasecmp(c0, "about"   ) == 0)
      {
         BOOL          plaintext;
         TX1K          about;                   // about text
         TXLN          text;                    // one line of text
         TXTS          alead;                   // leader text

         #if defined (USEWINDOWING)
            plaintext = (TxaOptUnSet('P') || (!txwIsWindow( TXHWND_DESKTOP)));
            strcpy( alead,   (plaintext)   ? ( txwIsWindow( TXHWND_DESKTOP)) ? "     " : " " : "");
         #else
            plaintext = TxaOptUnSet('P')  || dfsa->autoquit;
            strcpy( alead, (plaintext) ? " " : "");
         #endif                                 // USEWINDOWING

         //- always print the (c) for the DFSee program itself (changed in 15.1)
         sprintf( about, "%s              Fsys Software - https: //www.dfsee.com\n\n"
                         "%s %s : %s%s%s   (%2d-bit)\n", alead, alead, DFS_N, DFS_V,
                                            (strlen( DFS_VN) > 4) ? "" : " ", DFS_C,
         #if defined (__LP64__)
            64);
         #else
            32);
         #endif
         sprintf( text,  "%sTxWin UI Lib : %s\n", alead, txVersionString());
         strcat( about, text);
         if (!plaintext)                        // add to dialog only
         {
            #if defined (OEMSV)
               sprintf( text, "%sOEM  version : SVISTA commands (c) 2004-2008: Serenity Systems Int.\n", alead);
            #elif defined (OEMAN)
               sprintf( text, "%sOEM  version : OS2 disktool    (c) 2015-2018: Arca Noae, LLC.\n", alead);
            #else
               sprintf( text, "%sDisassembler : diStorm  1.7.25 (c) 2003-2007: Gil Dabah, Rage Storm\n", alead);
            #endif
            strcat( about, text);
            sprintf( text,  "%s'C' compiler : ",     alead);
            strcat( about, text);
            #if defined (__WATCOMC__)
               #if      (__WATCOMC__ > 1900)
                  //- must be OpenWatcom 2.0 or later (changed version convention)
                  sprintf( text, "OpenWatcom %4.2lf (c) 1988-2018: Sybase and OpenWatcom\n",
                                 ((double) (( __WATCOMC__ )) / 1000));
               #elif    (__WATCOMC__ > 1100)
                  //- must be OpenWatcom before 2.0 (official builds)
                  sprintf( text, "OpenWatcom %4.2lf (c) 1988-2010: Sybase and OpenWatcom\n",
                                 ((double) (( __WATCOMC__ - 1100)) / 100));
               #else
                  sprintf( text, "Watcom C++ %4.2lf (c) 1988-1999: Sybase, Inc. and Watcom\n",
                                 ((double) (( __WATCOMC__ )) / 100));
               #endif
            #elif defined (DARWIN)
               #if defined (__LP64__)
                  sprintf( text, "LLVM-gcc 10.0.1 and Clang-1001.0.46.4: (c) 2019 Apple\n");
               #else
                  sprintf( text, "GNU  gcc  4.0.1  on OSX 10.6.8 (c) 2007 Apple computer\n");
               #endif
            #else
               #if defined (DEV32)
                  sprintf( text, "VisualAge  3.65 (c) 1991-1997: IBM Corporation\n");
               #else
                  sprintf( text, "Visual C++  5.0 (c) 1986-1997: Microsoft Corporation\n");
               #endif
            #endif
            strcat( about, text);

            #if defined (DEV32)
               sprintf( text, "%sEXE compress : lxLite     1.33 (c) 1996-2003: Max Alekseyev\n", alead);
               strcat( about, text);
            #elif !defined (DARWIN) || defined (__LP64__)
               sprintf( text, "%sEXE compress : UPX        3.94 (c) 1996-2017: Oberhumer/Molnar/Reiser\n", alead);
               strcat( about, text);
            #endif

            #if defined (DOS32)
               sprintf( text,  "%sDOS extender : %s\n", alead, txDosExtVersion());
               strcat(  about, text);
            #endif
         }
         strcpy( s0, "Initial version description string");
         (void) TxOsVersion( s0);               // Get operating system version
         sprintf( text,  "%sOS   version : %s\n", alead, s0);
         strcat( about, text);
         TRACES(("about length:%d s0:'%s' text:'%s' about:'%s'\n", strlen(about), s0, text, about));

         if (TxOsAdditionalInfo( alead, text) == NO_ERROR) // additional info for OS
         {
            strcat( about, text);
            TRACES(("about length:%d text:'%s' about:'%s'\n", strlen(about), text, about));
         }

         #if !defined (OEM_BRANDED)
            #if defined (REGISTRATION)
               if (!TxaOptUnSet('c'))           // show program copyright and
               {                                // short registration info ?
                  if (!TxaOptUnSet('P'))        // show program copyright and
                  {
                     sprintf( text, "\nAdditional license and copyright "
                                    "notices listed with HELP, use <F1>\n");
                     strcat( about, text);
                  }
                  sprintf( text, "\n%sRegistration : ", alead);
                  strcat( about, text);
                  strcat( about, dfsRegistrationString( text));
                  sprintf( text, "\n%sRgKey status : ", alead);
                  strcat( about, text);
                  strcat( about, dfsRegistrationStatus( text));
                  strcat( about, "\n");
               }
               else                             // separate (c) and reg confirm
               {
                  dfsRegistrationInfo( DFSRG_DISPLAY_CONFIRM, DFS_V, dfsa->regconfirm);
               }
            #else
               sprintf( text, "\nAdditional copyright notices listed with HELP, use <F1>\n");
               strcat( about, text);
               sprintf( text, "\nOPEN SOURCE edition, no registration required\n");
               strcat( about, text);
            #endif
            TxNamedMessage( !(dfsa->batch || plaintext), 5003, " About the DFSee program ", about);

            if (strstr( DFS_V, ".b") != NULL)   // BETA status ?
            {
               static BOOL beta_confirmed = FALSE;

               TxNamedMessage( !(dfsa->batch || beta_confirmed), 0, " INFO: Beta / Experimental RELEASE! ",
                 "\n       This release of the software has BETA status!\n\n"
                 " There may be incomplete or incorrectly documented features\n"
                 "and the chance of program-bugs is a bit higher than usual :-)\n");
               beta_confirmed = TRUE;           // confirm just once, but show
            }                                   // text on all invocations
         #else
            #if defined (OEMSV)
               sprintf( text, "\n%sOEM Engine use only, licensed to ", alead);
               strcat(  text, "SVISTA Disk Utility and eComStation");
               strcat(  about, text);
               TxNamedMessage( FALSE, 0, " OEM ", about); // NEVER ask confirmation
            #elif defined (OEMAN)
               sprintf( text, "\n%sOEM Engine use only, licensed to ", alead);
               strcat(  text, "Arca Noae for installation\n");
               strcat(  text,  alead);
               strcat(  text, "and maintenance on ARCA NOAE products");
               strcat(  about, text);
               TxNamedMessage( FALSE, 0, " OEM ", about); // NEVER ask confirmation
            #else
               sprintf( about, "\n%sRegistration Free use, OPEN SOURCE edition", alead);
               TxNamedMessage( FALSE, 0, " OPEN SOURCE OEM ", about); // NEVER ask confirmation
            #endif

         #endif
      }
      else if (strcasecmp(c0, "??"      ) == 0)
      {
         TxShowTxt( TxGetStdCmdHelp());
         TxShowTxt( dfsGenericHelp);
         TxPrint(  " %s %s %s\n", DFS_N, DFS_V, DFS_C);
      }
      else if (strcasecmp(c0, "map"     ) == 0)
      {
         if (TxaOption('?'))                    // explicit help request
         {
            TxShowTxt( cmd_map);                // give usage
         }
         else
         {
            BOOL refresh = TxaOptSet('r') ||    // force re-read
                           TxaOptSet('F') ||    // First EXT implies -r
                           TxaOptSet('m') ||    // Show MBR, implies -r
                           TxaOptSet('g') ||    // show geo, implies -r
                           TxaOptSet('i') ||    // ind. std, implies -r
                           TxaOptSet('i') ||    // ind. std, implies -r
                           TxaOptSet('L');      // specials, implies -r

            if (TxaOption('M'))                 // Show Media map too
            {
               dfsShowMediaMap( 0, 0);          // show partitionable media map
            }
            dfsShowMap( dfsGetDiskNumber( 'd', "*", c1, refresh), c2[0]);
         }
      }
      #if defined (USEWINDOWING)
      else if (strcasecmp(c0, "menu2html" ) == 0)
      {
         if (TxaOption('?'))                    // explicit help request
         {
            TxPrint("\nGenerate PHP or HTML documentation for specified pull-down menu (Ctrl-2)\n");
            TxPrint("\n Usage:  %s  [pulldown] [options]\n\n"
                      "    pulldown      = First letter of pulldown menu name\n"
                      "    -h            : Generate static HTML instead of PHP\n"
                      "    -f:filename   : Explicit filename for generated HTML or PHP\n", c0);
         }
         else
         {
            TX2K         htmlNavigation;
            TXTM         thisName;
            TXTM         alt_Name;
            TXTM         alt_Text;
            TXTM         sel;
            TXS_MENUBAR *dfsmenu = dfsGetMainMenu();

            if (dfsa->expertui)
            {
               strcpy(  thisName, "dfsmenu-expert");
               strcpy(  alt_Name, "dfsmenu-basic" );
               strcpy(  alt_Text, "BASIC" );
            }
            else
            {
               strcpy(  thisName, "dfsmenu-basic" );
               strcpy(  alt_Name, "dfsmenu-expert");
               strcpy(  alt_Text, "EXPERT");
            }
            sprintf( htmlNavigation, dfsMenu2HtmlNavFmt19,
                     thisName, thisName, thisName, thisName, thisName, thisName, thisName,
                     alt_Name, alt_Text,
                     thisName, thisName, thisName, thisName, thisName, thisName,
                     thisName, thisName, thisName, thisName, thisName);

            strcpy(  sel, (cc > 1) ? c1 : dfsmenu->menu[dfsmenu->defopen]->text);
            TxStrip( sel, sel, ' ', ' ');
            rc = txSelPdMenu2Html( DFS_REG_NAME, dfsmenu, sel,
                                   TxaOption( 'h') ? htmlNavigation : NULL,
                                   TxaOptStr( 'f', NULL, ""), thisName, 2);
         }
      }
      else if (strcasecmp(c0, "menu"      ) == 0)
      {
         if (TxaOption('?'))                    // explicit help request
         {
            TxPrint("\nActivate the menu with default or selected pulldown\n");
            TxPrint("\n Usage:  %s  [pulldown-select-letter]\n\n"
                      "                f = File\n"
                      "                e = Edit\n"
                      "                m = Mode=xxx\n"
                      "                a = Actions\n"
                      "                d = Display\n"
                      "                h = Help\n", c0);
         }
         else
         {
            dfsa->menuOwner = TXHWND_DESKTOP;
            dfsa->menuopen  = (ULONG) c1[0];
         }
      }
      #endif                                    // USEWINDOWING
      else if (strcasecmp(c0, "show"   ) == 0)  // Show sector contents (hex)
      {
         if ((nr = dfsGetSymbolicSize( c1, 0, DFSTORE, 0)) != 0)
         {
            if ((nr <= DFS_MAX_DISPSECTOR) || (toupper(c2[0]) == 'S'))
            {
               nr *= dfsGetSectorSize();   // make it bytes ...
            }
            if (nr > RBUFBYTES)
            {
               nr = RBUFBYTES;                  // limit to maximum rbuf
            }
         }
         else
         {
            dfsa->defDispSize = (isupper(c0[0])) ? dfsGetSectorSize() : SECTORSIZE/2;
            nr = dfsa->defDispSize;
         }
         TxPrint("Current edit/write buffer contents:\n");
         TxDisplHexDump(rbuf, nr);
      }
      else if (strcasecmp(c0, "lock"   ) == 0)     // Lock store
      {
         rc = dfstLocking( DFSTORE, DFS_LOCK, (cc > 1), TRUE);
      }
      else if (strcasecmp(c0, "unlock" ) == 0)     // Unlock store
      {
         rc = dfstLocking( DFSTORE, DFS_UNLOCK, FALSE,  TRUE);
      }
      else if (strcasecmp(c0, "eject"   ) == 0)    // Eject drive (by letter)
      {
         if (TxaOption('?') || ((cc == 1) && (SINF->drive[0] == '-')))
         {
            TxPrint("\nEJECT a removable disk (like USB) using a driveletter\n");
            TxPrint("\n Usage:  %s  [drive]\n\n"
                      "   drive = driveletter for disk to be ejected\n", c0);
         }
         else
         {
            //- to be refined, when no param, present SELIST from volumes, incl FLOP
            //- remove the dfsa->slReMedia, to save menu startup time
            strcpy( s1, (cc > 1) ? c1 : SINF->drive);
            if ((rc = TxFsEjectRemovable( s1)) == NO_ERROR)
            {
               TxPrint( "\nDriveletter %s ejected ...\n", s1);
               dfsReadDiskInfo( FDSK_QUIET);    // re-read all partition info
            }
            else if (rc == ERROR_INVALID_PARAMETER)
            {
               TxPrint( "\nDriveletter %s is not an ejectable drive.\n", s1);
               dfsa->explain = FALSE;           // avoid garbage explanation
            }
         }
      }
      #if defined (DEV32)
      else if (strcasecmp(c0, "synclvm") == 0)  // synchronize LVM info on-disk
      {                                         // to the LVM engine in-memory
         index = dfsAccessibleDisks();          // by setting letter for volume
         rc = dfsLvmSyncEngine( TRUE, TRUE,     // and update diskinfo too
                                c1, c2[0]);
         if (cc == 1)                           // just refreshed removables
         {
            TxPrint( "Accessible disks  : %hu, discovered new: %hd\n",
                      dfsAccessibleDisks(), dfsAccessibleDisks() - index);
         }
      }
      #endif
      else if (strcasecmp(c0, "scan"   ) == 0)
      {
         if (TxaOption('?'))                    // explicit help request
         {
            TxPrint("\nSCAN disk or partition for unreadable (bad) sectors\n");
            TxPrint("\n Usage:  %s  [sn] [-b:sec] [-w] [-a-]\n\n"
                      "   sn          = Start sector as MCS-number, default current sector\n"
                      "   -a          = Perform automatic id (SLT) display for bad sectors\n"
                      "   -a-         = No automatic id (SLT) display for bad sectors\n"
                      "   -B          = Supress confirmation message, start immediately\n"
                      "   -b:N        = Use a user-defined buffer size, in SECTORS\n"
                      "   -skipbads:n = Skip 'n' sectors on each bad-sector area detected\n"
                      "   -w          = Use a full Read/Write/Verify/Write cycle (slow!)\n", c0);
         }
         else
         {
            BOOL  write = (toupper(c1[0]) == 'W') || TxaOption(  'w');
            BOOL  useid = !TxaOptUnSet('a');

            TRACES(("BADSEC scan Auto ID : %s", (useid) ? "TRUE" : "FALSE"));

            sn = dfsGetSymbolicSN( c1, nav.this); // current sector is default
            sectors = dfsGetLogicalSize() - sn; // size to scan, to end of object

            strcpy( s1, "");
            dfstrSz64( s1, "", sectors, "");
            if (sn == 0)
            {
               strcpy(  s0, "start of object");
            }
            else
            {
               strcpy( s0, "");
               dfstrSz64( s0, "", sn, " from the start");
            }


            if ((dfsa->batch) || (TxConfirm( 5029,
               "%s scan for bad-sectors on the current object:\n%s\n%s\n\nStart Scanning : %s, starting at: %s ? [Y/N] : ",
                (write) ? "R/W/VERIFY" : "READ-ONLY", dfstStoreDesc1( DFSTORE), dfstStoreDesc2( DFSTORE), s1, s0)))
            {
               rc = dfsBadSectorScan( write, useid, sn);
            }
            else
            {
               rc = DFS_NO_CHANGE;
            }

            if (rc == NO_ERROR)
            {
               if ((rc = (ULONG) dfsa->snlist[0]) == 0)
               {
                  TxPrint( "\nNo bad sectors were detected during the scan.\n");
                  if (!write)
                  {
                     TxPrint( "\nUse the '-w' option for a full Read/Write/Verify/Write "
                              "scan,\nvery exhaustive and it will leave all data intact.\n");
                  }
               }
               else
               {
                  TxPrint( "%u Bad sectors found!  You can use the 'list -w' command to show,\n"
                           "or the 'export' command to save the list to a file.\n"
                           "The list shows EACH individual bad sector found!\n"
                           "Be aware that verbose listing (list -d) will re-read all the\n"
                           "bad sectors, which will be very time consuming!\n", rc );
                  dfsa->explain = FALSE;        // avoid later garbage explanation

                  #if defined (WIN32) || defined (DARWIN)
                     if (write)
                     {
                        TxNamedMessage( !dfsa->batch, 0, " INFO: Bad sector false alarm ",
                                        "Bad sectors found MAY be a false-alarm due to the OS LOCKING "
                                        "a partition with a mounted filesystem, making it read-only!");
                     }
                  #endif
               }
            }
         }
      }
      else if (strcasecmp(c0, "speed" ) == 0)
      {
         if (TxaOption('?'))                    // explicit help request
         {
            TxPrint("\nTest RAW Read-only or Read/Write speed\n");
            TxPrint("\n Usage:  %s  [-b:N]  [-w]\n\n"
                      "   -b:N   = User-defined buffer size, in SECTORS\n"
                      "   -w     = Use a Read/Write cycle on every buffer\n", c0);
         }
         else
         {
            dfsMeasureIoSpeed( TxaOption('w'), nav.this);
         }
      }
      else if (strcasecmp(c0, "cs"   ) == 0)       // check sum
      {
         if (TxaOption('?'))                    // explicit help request
         {
            TxPrint("\nCalculate checksum for a sector, optional check againts expected value.\n");
            TxPrint("\n Usage:  %s  [sn] [-v:value] [-m:msg]\n\n"
                      "   sn       = Sectornumber to check, default current sector 'this'\n",
                      "   -m[:msg] = Message to show on checksum mismatch\n",
                      "   -v:value = Verify checksum against specified value\n", c0);
         }
         else
         {
            sn = dfsGetSymbolicSN( c1, nav.this); // default 'this' LSN
            if (cc > 1)
            {
               rc = dfsRead( sn, 1, rbuf);
            }
            if (rc == NO_ERROR)
            {
               ULONG cs = dfsCheckSum( rbuf);

               dfsX10("Checksum sector   : ", sn, "", " = ");
               nav.this = sn;
               if (TxaOptSet('v'))              // verify checksum
               {
                  ULONG   vs = TxaOptNum('v', "Checksum", 0);

                  TxPrint("%s%08.8X%s, ", (cs == vs) ? CBG : CBR, cs, CNN);
                  if (cs != vs)                 // Checksum mismatch
                  {
                     TxPrint( "mismatch, expected: %8.8x\n", vs);
                     if (TxaOptSet('m'))        // checksum mismatch message
                     {
                        strcpy( s0, TxaOptStr( 'm', NULL, "Checksum mismatch on sector!"));
                        TxRepl( s0, '~', '\n');
                        TxNamedMessage( !dfsa->batch, 5575, " ERROR Checksum mismatch ", s0);
                     }
                     rc = DFS_ST_MISMATCH;
                  }
                  else
                  {
                     TxPrint( "verify : OK\n");
                  }
               }
               else
               {
                  TxPrint( "%s%08.8X%s\n", CBG, cs, CNN);
               }
            }
         }
      }
      else if (strcasecmp(c0, "cv"   ) == 0)    // Copy Value to clipboard
      {
         if (TxaOption('?'))                    // explicit help request
         {
            TxPrint("\nCopy a selected value as a (Hex) string to the clipboard\n");
            TxPrint("\n Usage:  %s  [b|d|s|t|u|x] [-d]\n\n"
                      "   b  = Base    PSN of first sector of current object\n"
                      "   s  = Size    Size of the current object in sectors\n\n"
                      "   d  = Down    LSN of next sector displayed, on <ENTER>\n"
                      "   t  = This    LSN of the CURRENT sector (default)\n"
                      "   u  = Up      LSN of the 'upward' related sector\n"
                      "   x  = Xtra    LSN of an otherwise related sector\n\n"
                      "   -d : Copy as a decimal formatted string, not Hex\n", c0);
         }
         else
         {
            switch (tolower(c1[0]))
            {
               case 'b': sn = dfstLSN2Psn( DFSTORE, 0);   strcpy( s0, "BASE"); break;
               case 'd': sn = nav.down;                   strcpy( s0, "DOWN"); break;
               case 's': sn = dfsGetLogicalSize();        strcpy( s0, "SIZE"); break;
               case 'u': sn = nav.up;                     strcpy( s0, "UP"  ); break;
               case 'x': sn = nav.xtra;                   strcpy( s0, "XTRA"); break;
               default : sn = nav.this;                   strcpy( s0, "THIS"); break;
            }
            if (TxaOption('d'))
            {
               sprintf( s1, "%llu", sn);
            }
            else
            {
               sprintf( s1, "0x%llx", sn);
            }
            if (TxCopyClipBoardText( s1) == NO_ERROR) // string to clipboard
            {
               sprintf( msgText, "Copied value for %s = %s to the clipboard", s0, s1);
               TxPrint( "\n%s\n", msgText);
            }
         }
      }
      else if (strcasecmp(c0, "saveto"   ) == 0)
      {
         if (TxaOption('?'))                    // explicit help request
         {
            TxPrint("\nSave filedata for current file (FNODE/MFT/INODE) to a file\n");
            TxPrint("\n Usage:  %s [options] [path] [lsn] [name]\n"
                    "\n     path = Path to save file, default . or last path used"
                    "\n     lsn  = SN of LSN/MFT/INODE reference, default is 'this'"
                    "\n     name = New name to use for the file (rename on copy)\n", c0);

            TxPrint("\n Options are:"
                    "\n     -8.3 : Force 8.3 names on output, as required on most FAT16"
                    "\n     -A   : Determine recovery allocation percentage      (default)"
                    "\n     -A-  : NO alloc check, assume 100% (faster, implies -O:n)"
                    "\n -e:entry : Directory entry number for FAT DIR sector 0..15"
                    "\n            (automatic if 'this' was selected from Sectorlist)"
                    "\n     -O:q : Quiet mode, just report on errors"
                    "\n     -O:n : Normal verbosity, a single line per saved file"
                    "\n     -O:v : Verbose mode, some file detail in 3 lines     (default)"
                    "\n     -O:x : Extreme verbose, include progress indicator 'dots'"
                    "\n     -O:m : Maximum verbose mode, include 'skipping' messages"
                    "\n     -p   : Append original path to specified destination (default)"
                    "\n     -p-  : Do NOT append original path to specified destination"
                    "\n     -r   : Recurse into subdirectories, tree recovery    (default)"
                    "\n     -r-  : Do NOT recurse into subdirectories (no DIR contents)"
                    "\n     -u   : Force unique names using hex sector number prefix\n");
         }
         else
         {
            TXLN             recFname  = {0};   // rename / full recovery filename

            if (TxaOptSet('r'))
            {
               dfsa->rParams.recurseDir   = TxaOption('r');
            }
            if (TxaOptSet('A'))
            {
               dfsa->rParams.noAllocCheck = TxaOptUnSet('A'); // FALSE if -A, TRUE if -A-
            }
            if (TxaOptSet('p'))
            {
               dfsa->rParams.nopath       = TxaOptUnSet('p'); // FALSE if -p, TRUE if -p-
            }
            if (TxaOptSet('u'))
            {
               dfsa->rParams.unique       = TxaOption('u');
            }
            if (TxaOptSet('8'))
            {
               dfsa->rParams.name83       = TxaOption('8');
            }
            dfsa->rParams.newname  = c3;        // possible rename
            dfsa->rParams.recFname = recFname;

            if ((cc > 1) && (c1[0] != '.'))
            {
               strcpy( dfsa->SavetoPath, c1);
            }
            #if defined (USEWINDOWING)
               if (TxaOption('P'))              // explicit prompting
               {
                  TxNamedMessage( !dfsa->batch, DFSC_RECOVER, " INFO: File recovery hints ", dfsRecoverHints);

                  if (dfsDirSelectDialog(
                        " Select recovery options and destination directory for file(s) ",
                          DFSC_FSAVETO, dfsa->SavetoPath,
                         &dfsa->rParams.recurseDir,   "Recurse into subdirectories, recover whole tree",
                         &dfsa->rParams.name83,       "Force 8.3 compatible (short) filenames and paths",
                         &dfsa->rParams.unique,       "Force unique filenames by prefixing a sector number",
                         &dfsa->rParams.nopath,       "Discard original file path, only use its base-filename",
                         &dfsa->rParams.noAllocCheck, "Do NOT perform an allocation check (is slightly faster)",
                          NULL, NULL, NULL, NULL))
                  {
                     //- no action needed here
                  }
                  else
                  {
                     rc = DFS_NO_CHANGE;        // Abort without changes
                  }
               }
            #endif                              // USEWINDOWING
            if (rc == NO_ERROR)
            {
               if (TxaOptSet('e'))              // e option specified ?
               {
                  nav.this_sninfo = DFSSNIGET(TxaOptNum('e', "Entry", 0)) | DFSSNINFO;
               }
               sn = dfsGetSymbolicSN( c2, nav.this); // default 'this' LSN

               h1 = dfsa->verbosity;            // save current setting
               if (!TxaOptSet('O'))             // unless explicit verbosity
               {                                // set other default value
                  dfsa->verbosity = TXAO_VERBOSE;
               }
               dfsInitMultiRecover();           // progress/status (recursive DIR)
               rc = dfsRecoverSingle( sn, nav.this_sninfo, dfsa->SavetoPath, &dfsa->rParams);

               dfsa->verbosity = h1;            // restore

               if (rc == DFS_PENDING)           // skipped, wrong type
               {
                  rc = DFS_ST_MISMATCH;
               }
            }
            dfsa->rParams.recFname  = NULL;     // out of scope! Not allocated anymore
            dfsa->rParams.newname   = NULL;     // out of scope! Not allocated anymore
         }
      }
      else if (strncasecmp(c0, "recov", 5 ) == 0)
      {
         if (TxaOption('?'))                    // explicit help request
         {
            TxPrint("\nSave filedata for files from the filelist to a file\n");
            TxPrint("\n Usage:  %s  [options] [path] [sel]"
                    "\n   path = Path to save file, default . or last path used"
                    "\n   sel  = File selection parameters, in the form:"
                    "\n          wildcard%%threshold%%minsize%%maxsize%%type%%modifydate\n"
                    "\n          wildcard   =  wildcard selection on full path string"
                    "\n          threshold  =  +nn = Minimum allocation-OK percentage"
                    "\n                        -nn = Maximum allocation-OK percentage"
                    "\n          minsize    = minimum filesize as MCS number like 1,s"
                    "\n          maxsize    = maximum filesize as MCS number like 9,m"
                    "\n          type       = D=Directories F=Files B=Browse-mode N=Normal (all)",
                    "\n          modifydate = Restrict to files modified AFTER date or #days\n"
                    "\n          Default is :  '*%%100%%1,s%%*,s%%N%%*' ==> any 100%% one\n", c0);

            TxPrint("\n Options are:"
                    "\n     -8.3 : Force 8.3 names on output, as required on most FAT16"
                    "\n     -A   : Determine recovery allocation percentage      (default)"
                    "\n     -A-  : NO alloc check, assume 100% (faster, implies -O:n)"
                    "\n     -E:i : Ignore errors, continue execution when RC is not 0"
                    "\n     -E:q : Quit on any errors, exit  (default when in batch mode)"
                    "\n     -E:c : Confirm with user on errors (default when interactive)"
                    "\n     -O:q : Quiet mode, just report on errors"
                    "\n     -O:n : Normal verbosity, a single line per saved file"
                    "\n     -O:v : Verbose mode, some file detail in 3 lines     (default)"
                    "\n     -O:x : Extreme verbose, include progress indicator 'dots'"
                    "\n     -O:m : Maximum verbose mode, include 'skipping' messages"
                    "\n     -p   : Append original path to specified destination (default)"
                    "\n     -p-  : Do NOT append original path to specified destination"
                    "\n     -r   : Recurse into subdirectories, tree recovery    (default)"
                    "\n     -r-  : Do NOT recurse into subdirectories (no DIR contents)"
                    "\n     -u   : Force unique names using hex sector number prefix\n");
         }
         else
         {
            BOOL error_strategy =  dfsa->eStrategy;
            TXLN      recFname  = {0}; // rename / full recovery filename
            TXLN      wildcard  = {0}; // wildcard string (modifyable!)

            if (TxaOptSet('r'))
            {
               dfsa->rParams.recurseDir   = TxaOption('r');
            }
            else if ((DFSBR_SnlistFlag & DFSBR_1ST_PARENT) == 0) // not from a single directory
            {
               dfsa->rParams.recurseDir   = FALSE; // no DIR recursion on 'random' lists
            }
            if (TxaOptSet('A'))
            {
               dfsa->rParams.noAllocCheck = TxaOptUnSet('A'); // FALSE if -A, TRUE if -A-
            }
            if (TxaOptSet('p'))
            {
               dfsa->rParams.nopath       = TxaOptUnSet('p'); // FALSE if -p, TRUE if -p-
            }
            if (TxaOptSet('u'))
            {
               dfsa->rParams.unique       = TxaOption('u');
            }
            if (TxaOptSet('8'))
            {
               dfsa->rParams.name83       = TxaOption('8');
            }
            strcpy( wildcard, c2);              // may be modified later by dialog
            dfsa->rParams.select   = wildcard;  // file selection, wildcard
            dfsa->rParams.newname  = NULL;      // no rename
            dfsa->rParams.recFname = recFname;

            if (cc > 1)
            {
               strcpy( dfsa->SavetoPath, c1);
            }
            #if defined (USEWINDOWING)
               if ((TxaOption('P')) || (cc == 1)) // prompting
               {
                  BOOL ignore_errors = (error_strategy == TXAE_IGNORE);

                  TxNamedMessage( !dfsa->batch, DFSC_RECOVER, " INFO: File recovery hints ", dfsRecoverHints);

                  if (dfsDirSelectDialog(
                        " Select recovery options and destination directory for file(s) ",
                          DFSC_RECOVER, dfsa->SavetoPath,
                         &ignore_errors,              "Ignore any errors, continue recovering",
                         &dfsa->rParams.recurseDir,   "Recurse into subdirectories, recover whole tree",
                         &dfsa->rParams.name83,       "Force 8.3 compatible (short) filenames and paths",
                         &dfsa->rParams.unique,       "Force unique filenames by prefixing a sector number",
                         &dfsa->rParams.nopath,       "Discard original file path, only use its base-filename",
                         &dfsa->rParams.noAllocCheck, "Do NOT perform an allocation check (is slightly faster)",
                         NULL, NULL))
                  {
                     if (ignore_errors)
                     {
                        dfsa->eStrategy = TXAE_IGNORE;
                     }
                  }
                  else
                  {
                     rc = DFS_NO_CHANGE;        // Abort without changes
                  }
               }
            #endif                              // USEWINDOWING
            if (rc == NO_ERROR)
            {
               h1 = dfsa->verbosity;            // save current setting
               if (!TxaOptSet('O'))             // unless explicit verbosity
               {                                // set other default value
                  dfsa->verbosity = TXAO_VERBOSE;
               }
               dfsInitMultiRecover();           // progress/status (recursive DIR)
               rc = dfsRecoverList( dfsa->SavetoPath, &dfsa->rParams);
               dfsa->verbosity = h1;            // restore
            }
            dfsa->eStrategy = error_strategy;   // restore global setting
            dfsa->rParams.recFname  = NULL;     // out of scope! Not allocated anymore
            dfsa->rParams.select    = NULL;     // out of scope! Not allocated anymore
         }
      }
      else if (strcasecmp( c0, "shiftl"  ) == 0)
      {
         if (TxaOption('?'))                    // explicit help request
         {
            TxPrint("\nShift data some bytes towards begin of sector (LEFT shift)\n");
            TxPrint("\n Usage:  %s  bytes  [-s:size]\n\n"
                    "   -F:sector : First sector of area to shift,       default 0\n",
                    "   -l:bytes  : Number of bytes to shift, 1 .. 511   default 2\n"
                    "   -s:size   : Size of area to shift, as mcs-number default 1\n\n", c0);
         }
         else
         {
            ULONG bytes = TxaOptNum('l', "LeftShift", 2);            //- shift distance
            dfsOptionSizeSectors(   's', DFSTORE, 's', 1, &sectors); //- size
            dfsOptionSizeSectors(   'F', DFSTORE, 's', 0, &sn);      //- from

            sprintf( s1, "Shift LEFT %u bytes at 0x0%llx, size: ", bytes, sn);
            dfstrSz64( s1, "", sectors, "");
            if ((dfsa->batch) || (TxConfirm( 5668,
               "SHIFT bytes towards the beginning (LEFT) in sectors on:\n%s\n%s\n\n%s\n\n"
               "Are you sure you want to SHIFT-LEFT this object ? [Y/N] : ",
                dfstStoreDesc1( DFSTORE), dfstStoreDesc2( DFSTORE), s1)))
            {
               rc = dfsByteShiftLeftSector( bytes, sn, sectors);
            }
            else
            {
               rc = DFS_NO_CHANGE;
            }
         }
      }
      else if (strcasecmp( c0, "shiftr"  ) == 0)
      {
         if (TxaOption('?'))                    // explicit help request
         {
            TxPrint("\nShift data some bytes towards end of sector (RIGHT shift)\n");
            TxPrint("\n Usage:  %s  bytes  [-s:size]\n\n"
                    "   -F:sector : First sector of area to shift,       default 0\n",
                    "   -l:bytes  : Number of bytes to shift, 1 .. 511   default 2\n"
                    "   -s:size   : Size of area to shift, as mcs-number default 1\n\n", c0);
         }
         else
         {
            ULONG bytes = TxaOptNum('l', "RightShift", 2);           //- shift distance
            dfsOptionSizeSectors(   's', DFSTORE, 's', 1, &sectors); //- size
            dfsOptionSizeSectors(   'F', DFSTORE, 's', 0, &sn);      //- from

            sprintf( s1, "Shift RIGHT %u bytes at 0x0%llx, size: ", bytes, sn);
            dfstrSz64( s1, "", sectors, "");
            if ((dfsa->batch) || (TxConfirm( 5668,
               "SHIFT bytes towards the end (RIGHT) in sectors on:\n%s\n%s\n\n%s\n\n"
               "Are you sure you want to SHIFT-RIGHT this object ? [Y/N] : ",
                dfstStoreDesc1( DFSTORE), dfstStoreDesc2( DFSTORE), s1)))
            {
               rc = dfsByteShiftRightSector( bytes, sn, sectors);
            }
            else
            {
               rc = DFS_NO_CHANGE;
            }
         }
      }
      else if (strcasecmp(c0, "sort"   ) == 0)
      {
         if (TxaOption('?'))                    // explicit help request
         {
            TxPrint("\nSort the Sectorlist (as shown by 'list')\n");
            TxPrint("\n Usage:  %s  [-d] [-u]\n"
                    "\n   -d   = descending sort, high to low numbers\n"
                      "   -u   = make unique, remove duplicate entries\n", c0);
         }
         else
         {
            strcpy( s1, (TxaOption('d')) ? "-" : "");
            strcat( s1, (TxaOption('u')) ? "u" : "");
            dfsSortSectorList(dfsa->snlist, s1);
            TxPrint("%u sector numbers in sorted list\n", dfsa->snlist[0]);
         }
      }
      else if (strncasecmp(c0, "wipel", 5 ) == 0)
      {
         if (TxaOption('?'))                    // explicit help request
         {
            TxPrint("\nWipe (selected) sectors in the Sectorlist\n");
            TxPrint("\n Usage:  %s  [sel] [pat]\n\n"
                      "   sel  = File selection parameters, in the form:\n\n"
                      "          wildcard%%threshold  (sometimes shown as [w%%t])\n\n"
                      "            wildcard   =  wildcard selection for display/recover\n\n"
                      "            threshold  =  +nn = Minimum allocation-OK percentage\n"
                      "                      or  -nn = Maximum allocation-OK percentage\n\n"
                      "          default: *%-100 ==> all except 100%% recoverable ones\n\n"
                      "          To wipe ALL sectors, any type, specify '!'\n\n"
                      "   pat  = pattern to wipe with, default zeroes\n", c0);
         }
         else
         {
            if ((ml = dfsGetMixedString( c2, "wipe-pattern", FALSE, FALSE,
                                             (char *) &ms[0])) == 0)
            {
               ml = 16;
               memset( ms, 0, ml);              // default pattern, zeroes
            }
            TxPrint( "\nWipe uses the repeated pattern with"
                     " length %u, and value:\n", ml);
            TxDisplHexDump( ms, ml);
            strcpy( s1, (cc <= 1) ? "*%-100" : c1);
            if ((dfsa->batch) ||                // forced or confirmed
                (TxConfirm( 5012,
               "Are you sure you want to wipe sectors from the Sectorlist "
               "matching %s%s%s with the shown pattern ? [Y/N]: ", CBC, s1, CNN)))
            {
               if (DFSTORE_WRITE_ALLOWED)
               {
                  rc = dfsDelWipeList( dfsa->snlist, ms, ml, s1);
               }
               else
               {
                  rc = DFS_READ_ONLY;
               }
            }
            else
            {
               rc = DFS_NO_CHANGE;
            }
         }
      }
      else if (strcasecmp( c0, "wipesecure"  ) == 0)
      {
         if (TxaOption('?'))                    // explicit help request
         {
            TxPrint("\nWipe ALL or just FREESPACE sectors in the currently selected\n"
                    "object using a write, write-inverted, write-random and verify\n"
                    "to make sure the data can never be recovered or undeleted.\n");
            TxPrint("\n Usage:  %s  [-f]\n\n"
                    "   Options:  -f   = Wipe freespace sectors in filesystem only\n\n", c0);
         }
         else
         {
            if (DFSTORE_WRITE_ALLOWED)
            {
               strcpy( s0, "wipe -B");
               if (TxaOption('f'))              // wipe freespace only
               {
                  strcat( s0, " -f");
                  strcpy( s1, "freespace (unused) sectors only");
               }
               else
               {
                  strcpy( s1, "ALL sectors");
               }
               sprintf( msgText, "DoD-5220.22/NIST-SP800-88 military spec SECURE and UNRECOVERABLE\n"
                                 "multi-pass inverting plus random pattern wipe data sectors on:\n"
                                 "%s\n%s\n\nWiping %s.\n", dfstStoreDesc1( DFSTORE),
                                                           dfstStoreDesc2( DFSTORE), s1);
               TxPrint( "\n%s", msgText);

               if ((dfsa->batch) || (TxConfirm( 5666, "%s.\n"
                         "Are you sure you want to SECURE-WIPE this object ? [Y/N] : ", msgText)))
               {
                  dfsProgressItemsTodo( 3, "Pass");
                  TxPrint( "\nFirst pass, writing fixed pattern ...\n");
                  sprintf( s1, "%s -hex:'6aa6'", s0);
                  if ((rc = dfsMultiCommand( s1, 0, TRUE, FALSE, FALSE)) == NO_ERROR)
                  {
                     TxPrint( "\nSecond pass, writing inverted pattern ...\n");
                     sprintf( s1, "%s -hex:'a66a'", s0);
                     if ((rc = dfsMultiCommand( s1, 0, TRUE, FALSE, FALSE)) == NO_ERROR)
                     {
                        TxPrint( "\nFinal pass, writing random pattern and verify ...\n");
                        sprintf( s1, "%s -Verify -random'", s0);
                        if ((rc = dfsMultiCommand( s1, 0, TRUE, FALSE, TRUE)) == NO_ERROR)
                        {
                           TxPrint( "\nDoD 5220.22 SECURE wipe %s completed.\n",
                                    (TxaOption('f')) ? "freespace" : "ALL sectors");
                        }
                     }
                  }
               }
               else
               {
                  rc = DFS_NO_CHANGE;
               }
            }
            else
            {
               rc = DFS_READ_ONLY;
            }
         }
      }
      else if ((strcasecmp(c0, "export" ) == 0) ||
               (strcasecmp(c0, "import" ) == 0)  )
      {
         if ((cc == 2) && (!TxaOption('?')))    // filename given, no help
         {
            strcpy( s1, c1);
            TxFnameExtension( s1, "lsn");
            if (TxaOption('d'))                 // data file too ?
            {
               char   *ext;                     // lsn file extension
               TXTS    bin;                     // binary file extension

               strcpy(    s0, s1);
               TxStripExtension( s0);           // temporary, default for s0

               ext = s1 + strlen( s0) +1;       // start of original fname ext

               if ((strlen( ext) == 3) && (isdigit( ext[2])))
               {
                  if (isdigit( ext[1]))         // 2 digit sequence nr
                  {
                     sprintf( bin, "b%s", ext +1);
                  }
                  else                          // 1 digit sequence nr
                  {
                     sprintf( bin, "bn%s", ext +2);
                  }
               }
               else
               {
                  strcpy( bin, "bin");          // default extension
               }
               TxFnameExtension( s0, bin);
            }
            else                                // no data file
            {
               strcpy( s0, "");
            }
            if (strcasecmp(c0, "export" ) == 0)
            {
               rc = dfsExportSectorList( s1, s0);
            }
            else
            {
               rc = dfsImportSectorList( s1, s0);
               //- to be refined, warn if different mode ?
            }
         }
         else
         {
            TxPrint("%s the Sectorlist, optionaly including the data\n", c0);
            TxPrint("\n Usage:  %s  fname  [-d]\n"
                    "\n   fname = filename with (ASCII) sectornumbers\n"
                      "   -d    = %s sector-contents to/from 'fname.bin'\n", c0, c0);
            if (!TxaOption('?'))
            {
               TxPrint( "\nNo %s file specified (.lsn), present ones are:\n", c0);
               sprintf( s0, "%s *.lsn", FS_DIR_CMD);
               TxExternalCommand( s0);          // show lsn files
            }
         }
      }
      else if (strcasecmp(c0, "listset") == 0)  // and new command name
      {
         if (TxaOption('?'))                    // explicit help request
         {
            TxPrint("\nAdd one or more sectornumbers to the list\n");
            TxPrint("\n Usage:  %s  [-c[:n]]  [sn  [size]]\n", c0);
            TxPrint("\n   -c[:n]   = make list empty, or clip at index n\n"
                      "   sn       = mcs-number for first sector to add\n"
                      "   size     = number of sn-values to be set, 0..n\n");
         }
         else
         {
            switch (c1[0])
            {
               case 'U': sn = nav.up;                           break;
               case 'D': sn = nav.down;                         break;
               case 'X': sn = nav.xtra;                         break;
               default : sn = dfsGetSymbolicSN( c1, nav.this);  break;
            }
            size = dfsGetSymbolicSize( c2, (cc >= 2) ? 1 : 0, DFSTORE, sn);
            if (TxaOptSet('c'))                 // clear list to size value
            {
               dfsInitList( TxaOptNum('c', NULL, 0), "-w", "-d");
            }
            for (ei = 0; (ei < size) && DFSNL_ROOM; ei++)
            {
               dfsAdd2SectorList( sn + ei);     // add to list
            }
            if (dfsa->snlist[0] == 0)
            {
               TxPrint( "Sectorlist cleared: empty\n");
            }
            else
            {
               if (dfsa->snlist[0] <= 120)      // unless empty or too large
               {
                  strcpy(  dc, "list -r");      // display in range format
                  TxaReParseCommand( dc);
                  rc = DFS_PENDING;             // handle translated command
               }
            }
         }
      }
      else if (strcasecmp(c0, "scl"   ) == 0)
      {
         if (cc > 1)
         {
            dfstSetClusterSize( DFSTORE, (USHORT) atoi(c1));
         }
         TxPrint( "\nClustersize       : %hu = 0x%hx\n",
               dfsGetClusterSize(), dfsGetClusterSize());
      }
      else if (strcasecmp(c0, "bps"   ) == 0)
      {
         if (cc > 1)
         {
            if (dfstStoreType( DFSTORE) == DFST_MEMORY)
            {
               TxPrint( "\nChanging the sectorsize is NOT supported on memory disks!\n");
            }
            else
            {
               dfstSetSectorSize( DFSTORE, (USHORT) atoi(c1));
            }
         }
         TxPrint( "\nBytes per sector  : %hu = 0x%hx\n",
            dfstGetSectorSize( DFSTORE),
            dfstGetSectorSize( DFSTORE));
      }
      else if (strcasecmp(c0, "chs"      ) == 0)
      {
         if ((cc >1) && (!TxaOption('?')))      // one parem, no help
         {
            rc = dfsReadDisplayPhysical( dfstChs2Psn( DFSTORE,
                                         dfsGetMcsNumber( c1, 0),
                                         dfsGetMcsNumber( c2, 0),
                                         dfsGetMcsNumber( c3, 1)), &st);
         }
         else
         {
            TxPrint("\nDisplay sector specified by CHS address\n");
            TxPrint("\n Usage:  %s  C  [H  [S]]\n", c0);
            TxPrint("\n   C   = Cylinder number, 0 .. #cylinders -1\n"
                      "   H   = Head number, 0 .. 255, default 0\n"
                      "   S   = Sector, 1 .. 63, default 1\n");
         }
      }
      else if ((strcasecmp(c0, "pritems" ) == 0))
      {
         dfsProgressItemsTodo( dfsGetMcsNumber( c1, 0), c2);
      }
      else if (strncasecmp(c0, "gb", 2   ) == 0)
      {
         sn = dfsGetSymbolicSN( c1, 1);         // default 1 sector
         rc = dfsReadAnDisplay( (nav.this > sn) ? nav.this - sn : 0, 0, &st);
      }
      else if (strncasecmp(c0, "gf", 2   ) == 0)
      {
         sn = dfsGetSymbolicSN( c1, 1);         // default 1 sector
         rc = dfsReadAnDisplay( nav.this + sn, 0, &st);
      }
      else if (strncasecmp(c0, "geo", 3  ) == 0)
      {
         if (TxaOption('?'))
         {
            TxPrint("\nDisplay or set the geometry (L-Geo) for the currently opened object\n");
            TxPrint("\n Usage:  %s  ?|c h s f b [options] (all parameters are optional)\n\n", c0);
            TxPrint("        ?     = Recalculate number of cylinders from existing size\n"
                    "        c     = new number of cylinders, 0 to keep value\n"
                    "        h     = new number of heads,     0 to keep value\n"
                    "        s     = new number of sectors,   0 to keep value\n"
                    "        f     = optional first, or base-sector number\n"
                    "        b     = optional bytes-per-sector value\n\n");
            TxPrint("       -b:bps : optional bytes-per-sector value\n"
                    "       -d:nr  : optional disk number, to be selected first\n"
                    "       -S     : Synchronize size and system-geometry (S-Geo) too\n"
                    "       -!     : present a disk-geometry show/set dialog popup (default)\n"
                    "       -!-    : do NOT present a disk-geometry show/set dialog popup\n");
         }
         else
         {
            if (TxaOptSet('d'))                 // select a disk first
            {
               index = TxaOptNum('d', NULL, (SINF->disknr) ? SINF->disknr : 1);
               dfsSelectDisk( index, FALSE, FALSE); // quietly select disk
            }
            #if defined (USEWINDOWING)
               if (TxaOption('!'))
               {
                  dfsGeoDialog();
               }
               else
            #endif
            {
               DFSGEODISK *dskgeo = dfsGetDiskGeoInfo( SINF->disknr);

               if (dskgeo)
               {
                  strcpy( dskgeo->Reason, "GEO command");
               }
               h1 = (c1[0] == '?') ? L32_NULL    : dfsGetMcsNumber( c1, 0);
               h2 =                                dfsGetMcsNumber( c2, 0);
               h3 =                                dfsGetMcsNumber( c3, 0);

               //- JvW 28-03-2010, let base default to CURRENT base, not 0!
               sn =                                dfsGetMcsNumber( c4, dfstLSN2Psn( DFSTORE, 0));
               ml = (USHORT) TxaOptNum('b', "bps", dfsGetMcsNumber( c5, 0));
               rc = dfstDiskGeometry( DFSTORE,
                                      sn, h1, h2, h3, ml,
                                      TxaOption('S'), // Sync S-Geo as well ?
                                      TRUE);
                                                // make sure cSC values are OK
               dfsReadDiskInfo( FDSK_QUIET);    // re-read all partition info
            }
         }
      }
      else if ((strcasecmp(c0, "lsn"      ) == 0))
      {
         if (TxaOptSet('e'))                    // e option specified ?
         {
            sninfo = DFSSNIGET(TxaOptNum('e', "Entry", nav.this_sninfo)) | DFSSNINFO;
         }
         sn = dfsGetSymbolicSN( c1, 0);         // default lsn 0
         rc = dfsReadAnDisplay( sn, sninfo, &st);
      }
      else if ((strcasecmp(c0, "psn"      ) == 0))
      {
         sn = dfsGetSymbolicSN( c1, 0);         // default psn 0
         rc = dfsReadDisplayPhysical( sn, &st);
      }
      else if (strncasecmp(c0, "trans", 5) == 0)
      {
         if (cc > 1)                            // at least 1 parameters
         {
            if      (TxaOption('p'))            // PSN input
            {
               sl = dfsGetSymbolicSN( c1, 0);   // default psn 0
               sn = dfstPsn2LSN( DFSTORE, sl);
            }
            else if (TxaOption('i'))            // BIOS/Int13 input
            {
               sscanf(c1, "%x", &h1);           // Head
               sscanf(c2, "%x", &h2);           // HiByte  sec-cyl
               sscanf(c3, "%x", &h3);           // LowByte sec-cyl
               h4 = ((h3 << 8) | h2);           // combine hi-lo bytes
               sl = dfstChs2Psn( DFSTORE,
                                 DFSC2CYLIND(  h4),  //- Cylinder
                                               h1,   //- Head
                                 DFSC2SECTOR(  h4)); //- Sector
               sn = dfstPsn2LSN( DFSTORE, sl);
            }
            else if (TxaOption('c'))            // CHS input
            {
               h1 = dfsGetMcsNumber( c1, 0);
               h2 = dfsGetMcsNumber( c2, 0);
               h3 = dfsGetMcsNumber( c3, 0);
               if (h2 >= dfstGeoHeads( DFSTORE))
               {
                  TxPrint( "WARNING, head-nr  : %u, invalid for current "
                           "geometry! Cyl/Head  recalculated\n", h2);
               }
               if (h3 >= dfstGeoSectors( DFSTORE))
               {
                  TxPrint( "WARNING, sector   : %u, invalid for current "
                           "geometry! Head/Sect recalculated\n", h3);
               }
               sl = dfstChs2Psn( DFSTORE, h1, h2, h3);
               sn = dfstPsn2LSN( DFSTORE, sl);
            }
            else                                // assume LSN input
            {
               sn = dfsGetSymbolicSN( c1, 0);   // default lsn 0
            }
            dfsDisplayTransLsn( "Translations, LSN : ", sn);

         }
         else
         {
            dfsDisplayTransLsn( "Translated 'this' : ", nav.this);
         }
      }
      else if (strcasecmp(c0, "al" ) == 0)      // show/set allocation status for LSN/cluster
      {
         if ((cc > 1) || TxaOption('c'))
         {
            ULONG   alStatus = TxaOptNum('w', "AllocBit", 1);

            sn = dfsGetSymbolicSN( c1, nav.this);   //- get LSN or Cluster
            nr = dfsGetSymbolicSize( c2, 1, DFSTORE, sn);

            strcpy( s1, (TxaOption('w')) ? "Write" : "Read ");

            TxPrint("\n");
            for (ei = 0; (ei < (ULONG) nr) && !TxAbort(); ei++)
            {
               sl = TxaOption('c') ? dfsCl2Sn(sn + ei) : sn + ei;

               if (TxaOption('w'))              // set allocation status
               {
                  if ((dfsa->Fsi != NULL) && (dfsa->FsLsnSetAlloc != NULL))
                  {
                     rc = DFSFNCALL( dfsa->FsLsnSetAlloc, sl, 0, (alStatus) ? "s" : NULL, NULL);
                  }
                  else
                  {
                     TxPrint("'%s -w' command (write) not supported for filesystem %s%s%s\n",
                               c0, CBM, SINF->afsys, CNN);
                     break;
                  }
               }
               else                             // get/display allocation
               {
                  if ((dfsa->Fsi != NULL) && (dfsa->FsLsnAllocated != NULL))
                  {
                     alStatus = DFSFNCALL( dfsa->FsLsnAllocated, sl, 0, NULL, NULL);
                  }
                  else
                  {
                     TxPrint("'%s' command (display) not supported for filesystem %s%s%s\n",
                               c0, CBM, SINF->afsys, CNN);
                     break;
                  }
               }
               if (rc == NO_ERROR)
               {
                  TxPrint( " %s alloc-bit for cl/block: 0x0%llx = LSN: 0x0%llX,  Status: %s\n",
                             s1, dfsSn2Cl( sl), sl, (alStatus != 0) ? "Allocated" : "Free");
               }
            }
         }
         else
         {
            TxPrint("\nDisplay or write allocation bit for (range of) sectors/clusters/blocks\n");
            TxPrint("\nUsage '%s'  [location  [size]]  [-c]  [-w:0|1]\n"
                    "\n     location = location, sector or cluster/block number"
                    "\n     size     = size, either in sectors or clusters/blocks\n"
                    "\n     -w:0|1   = write allocation: 0=free, 1=allocated"
                    "\n     -c       = loc and size are clusters/blocks\n", c0);
         }
      }
      else if ((strcasecmp(c0, "query"   ) == 0))
      {
         strcpy( s1, (TxaOption('?')) ? "?" : c1);
         if ((strlen(s1) == 0) || (isdigit(s1[0]))) // numeric or no param
         {
            char amode = TxGetAnsiMode();
            TxSetAnsiMode( A_OFF);           // avoid ANSI in output

            dfsa->regconfirm = FALSE;           // no nagging on exit
            TxaGetArgString( TXA_CUR, 1, TXA_OPT, TXMAXLN, s1);
            sprintf( dc, "part -C -m %s", s1);  // part -C -m + query params
            rc = dfsMultiCommand( dc, 0, FALSE, FALSE, TRUE);

            TxSetAnsiMode( amode);
         }
         else                                   // query individual value
         {
            rc = dfsQueryValue( s1, c2);
         }
         dfsa->explain = FALSE;
      }
      else if ((strcasecmp(c0, "disk"    ) == 0) ||
               (strcasecmp(c0, "walk"    ) == 0)  )
      {
         rc = dfsFdskCommand( 0, 0, dc, NULL);
      }
      else if ((strcasecmp(c0, "findpath") == 0) || // generic find path command
               (c0[0]) == '\\')
      {
         if (TxaOption('?'))                    // explicit help request
         {
            TxPrint("\nFind filesystem information on specified path\n");
            TxPrint("\n Usage:  FINDPATH [%c]path-from-root\n"
                      "    or:  \\path-from-root\n\n"
                      "   path-from-root = Complete path starting from ROOT directory\n"
                      "                    components separated with either '\\' or '/'\n"
                    "\n   The path can be for a directory, or a specific file.\n", FS_PATH_SEP);
         }
         else
         {
            if (c0[0] == '\\')
            {
               pp = dc+1;                       // relative to ROOT
            }
            else
            {
               if ((c1[0] == FS_PATH_SEP) ||    // relative to ROOT, find
                   (c1[0] == FS_PALT_SEP )  )   // (either DOS-like or UNIX)
               {
                  pp++;
               }
            }
            strcpy( s1, pp);                    // copy path from ROOT
            TxRepl( s1, FS_PALT_SEP, FS_PATH_SEP); // fixup ALT separators

            if ((rc = DFSFNCALL(dfsa->FsFindPath, 1, 0, s1, &params)) == NO_ERROR)
            {
               TRACES(("FS LSN: 0x%llx, info: 0x%llx\n", params.Lsn, params.Number));
               if (params.Flag == TRUE)         // Size from directory is available
               {
                  strcpy( s0, "");
                  dfstr64XiB( s0, "", params.byteSize, "");
                  if (params.byteSize > 999)
                  {
                     dfstrUllDot20( s0, " = ", params.byteSize, "");
                  }
                  TxPrint( "Size  (DIR-entry) :   %s bytes\n", s0);
               }
               rc = dfsReadAnDisplay( params.Lsn, params.Number, &st);
            }
            else if (rc == DFS_PENDING)
            {
               TxPrint( "\n%s not implemented for %s%s%s yet\n", c0, CBM, SINF->afsys, CNN);
               rc = NO_ERROR;
            }
         }
      }
      else if (strcasecmp(c0, "findspace"  ) == 0) // show findSpace alloc, if any
      {
         if (dfsa->findSpace.space != NULL)
         {
            TxPrint( "Allocation for the filefind/delfind find-SPACE (DIR/MFT/FNODES/INODES:\n");
            dfsSspaceDisplay( SD_SUMMARY | SD_TOPLINE | SD_BOTLINE | SD_LIMIT32,
                              dfsa->findSpace.chunks, dfsa->findSpace.space);
         }
         else
         {
            TxPrint( "There is no 'findSPACE info available yet, for current filesystem/mode!\n");
         }
      }
      else if (strncasecmp(c0, "find", 4) == 0)    // generic sector find
      {
         rc = dfsFindSectorData( pp, (cc > 1) ? "1" : "*");
      }
      else if ((strcasecmp(c0, "fixhs"  ) == 0) || // for backward compatibility
               (strcasecmp(c0, "fixpbr" ) == 0)  )
      {
         if (TxaOption('?'))                    // explicit help request
         {
            TxPrint("\nFix HiddenSectors and GEO fields in the bootsector to match partition tables.\n");
            TxPrint("\n Usage:  %s  [-g-] [-h-]\n\n"
                      "  -g     =  Update the geometry head/sector values  (default)"
                      "  -g-    =  Do NOT update the geometry head and sector values"
                      "  -h     =  Update the hidden sectors value         (default)"
                      "  -h-    =  Do NOT update the bootsector hidden sectors value", c0);
         }
         else
         {
            if ((rc = dfsFixPbrHsGeoFields(
                    (!TxaOptUnSet('g') && (strcasecmp(c0, "fixpbr" ) == 0)),
                     !TxaOptUnSet('h'), FALSE,
                      TxaOption(  '!'))) == NO_ERROR)
            {
               TxPrint( "PBR contents now  :\n\n");
               sprintf( dc, "part -r -q %hu#d", SINF->partid);
               rc = dfsMultiCommand( dc, 0, FALSE, FALSE, TRUE);
            }
         }
      }
      else if (strcasecmp(c0, "serial"  ) == 0)
      {
         if (TxaOption('?'))                    // explicit help request
         {
            TxPrint("\nSet volume-serial number in partition bootsector.\n");
            TxPrint("\n Usage:  %s  [serial]  [-!-]\n\n"
                    "   serial : new volume serial number\n\n"
                    "   -!     = Prompt for new value (default)\n"
                    "   -!-    = do not prompt for new value\n", c0);
         }
         else
         {
            rc = dfsSetVolumeSerialNr( !TxaOptUnSet('!'), c1);
         }
      }
      else if (strcasecmp(c0, "fi"     ) == 0)  // list fileinfo for SN
      {
         if (TxaOptSet('e'))                    // e option specified ?
         {
            sninfo = DFSSNIGET(TxaOptNum('e', "Entry", nav.this_sninfo)) | DFSSNINFO;
         }
         sn = dfsGetSymbolicSN( c1, nav.this);  // default current sector
         sprintf( s0, "SectorNr %12.12llx : ", sn);
         rc = DFSFNCALL(dfsa->FsFileInformation, sn, sninfo, "*", s0);
         if (rc == DFS_PENDING)
         {
            dfsDisplaySnType( s0, sn, 0, dfsReadIdentifySector(sn));
            rc = NO_ERROR;
         }
      }
      else if (strcasecmp(c0, "format" ) == 0)     // format current object
      {
         TxPrint("%s command not supported for filesystem %s%s%s\n",
                 c0, CBM, SINF->afsys, CNN);
      }
      else if (strcasecmp(c0, "list"   ) == 0)
      {
         TxPrint("\nSN list contains  : %s%llu%s entries ", CBC, dfsa->snlist[0], CNN);

         if (dfsa->brlist != 0)                 // Browse 'this' info available
         {
            strcpy( s0, "");                    // no screen output, just return path
            strcpy( s1, "*");                   // and allow any filename
            if (DFSFNCALL(dfsa->FsFileInformation, dfsa->brlist, dfsa->brinfo, s1, s0) == NO_ERROR)
            {
               TxPrint( "from path %s%s%s\n", CBY, s0, CNN);

               //- TxPrint( "\ns1: '%s'\ns2: '%s\n", s1, s0);
            }
         }
         else
         {
            TxPrint( "\n");
         }
         if (TxaOption('?'))
         {
            TxShowTxt( cmd_list);               // give usage
         }
         else if (!TxaOption('n'))              // just show number of entries
         {
            dfsDisplaySectorList( pp);
         }
      }
      else if (strcasecmp(c0, "arealist"   ) == 0)
      {
         if (TxaOption('?'))
         {
            TxPrint( "\nDisplay contents of the sector-area list (bad sector areas)\n");
         }
         else
         {
            dfsDisplayAreaList( dfsa->salist);
         }
      }
      else if (strcasecmp(c0, "logfile" ) == 0)
      {
         //- to be refined, integrate into TX log command itself ?

         #if defined (USEWINDOWING)
         if (txwIsWindow( TXHWND_DESKTOP))
         {
            dfsLogDialog( (cc > 1) ? c1 : (dfsa->logDate) ? "dfsee-~" :
                                          (dfsa->logAuto) ? "dfsee^"  :
                                                            "dfswork",
                          DFSC_LOGF, TxaOption('r'), TxaOptStr( 'm', "Message", ""));
         }
         else
         #endif
         {
            sprintf( dc, "log %s %s", (cc > 1) ? c1 : (dfsa->logDate) ? "dfsee-~" :
                                                      (dfsa->logAuto) ? "dfsee^"  :
                                                                        "dfswork",
                         (TxaOption('r')) ? " -r" : "");
            rc = dfsMultiCommand( dc, 0, TRUE, FALSE, TRUE);
         }
      }
      #if defined (REGISTRATION)
      else if (strcasecmp(c0, "patchkey") == 0)
      {
         rc = dfsRegistrationPatch();           // patch regkey value into current sector
         if (rc == NO_ERROR)                    // must be a key already to succeed
         {
            TxPrint( "\nActual registration key inserted into current sector.\n");
         }
         else
         {
            TxPrint( "\nRegistration key replacement failed!\n");
         }
      }
      #endif
      else if (strcasecmp(c0, "part" ) == 0)
      {
         TXTM  text;
         BOOL  ok = TRUE;

         strcpy( text, "if a partition is specified!");

         ok &= TxaOptMutEx((cc > 1), "Vw",  text, NULL);
         ok &= TxaOptMutEx((cc > 1), "c",   text, NULL);
         ok &= TxaOptMutEx((cc > 1), "e",   text, NULL);
         ok &= TxaOptMutEx((cc > 1), "f",   text, NULL);
         ok &= TxaOptMutEx((cc > 1), "n",   text, NULL);
         ok &= TxaOptMutEx((cc > 1), "p",   text, NULL);
         ok &= TxaOptMutEx((cc > 1), "t",   text, NULL);

         if ((!ok) || TxaOption('?'))
         {
            if (ok)
            {
               TxShowTxt( cmd_part);            // give usage
            }
            TxPrint( "\n");
         }
         else
         {
            BOOL refresh = TxaOptSet('r') ||    // force re-read
                           TxaOptSet('F') ||    // First EXT implies -r
                           TxaOptSet('m') ||    // Show MBR, implies -r
                           TxaOptSet('g') ||    // show geo, implies -r
                           TxaOptSet('i') ||    // ind. std, implies -r
                           TxaOptSet('i') ||    // ind. std, implies -r
                           TxaOptSet('L');      // specials, implies -r

            #if defined (REGISTRATION)
               dfsRegistrationInfo( DFSRG_WHEN_NOT_REGISTRED, DFS_V, dfsa->regconfirm);
            #endif
            ml = dfsGetDiskNumber( 'd', "*", NULL, refresh);  // one or all, +reread
            if ((cc > 1) || TxaOption('P'))     // some partition spec, or prompt
            {
               if (TxaOption('P'))              // explicit prompting
               {
                  #if defined (USEWINDOWING)
                  ULONG  listcode = TXDID_OK;
                  TXRECT where = {18,0,0,0};    // fixed position

                  listcode = txwListBox( TXHWND_DESKTOP, TXHWND_DESKTOP, &where,
                           " Select partition to work with ", "",
                             5403, TXLB_MOVEABLE,
                             cMenuTextStand, cMenuBorder_top, // same color as menu
                             dfsa->slPartOne);

                  if (listcode > TXDID_MAX)
                  {
                     index = listcode - TXDID_MAX; // calculate pid
                  }
                  else                          // most likely canceled
                  {
                     rc = DFS_USER_ABORT;
                  }
                  #else
                     index = 1;                 // just take first one
                  #endif
               }
               else if (isdigit(c1[0]))         // partition nr specified
               {
                  if ((index = (USHORT) atoi(c1)) != 0)
                  {                             // support relative PID's
                     DFSDISKINFO *d = dfsGetDiskInfo( dfsGetDiskNumber( 'd', ".", "", refresh));

                     if ((d) && (d->firstpart) && strstr( c1, ",r"))
                     {
                        index += d->firstpart -1;  // calculate absolute PID
                     }
                  }
               }
               else if (isalpha(c1[0]))         // drive letter
               {
                  if ((c1[1] == ':') || (c1[1] == 0))
                  {
                     index = dfsDrive2PartId( c1);
                     if (index == 0)            // not found
                     {
                        TxPrint("Partition for drive '%c': not found\n", c1[0]);
                     }
                  }
                  else                          // some text, assume auto/new
                  {
                     index = dfsa->autoPid;     // use automatic PID
                  }
               }
               else if (c1[0] == '.')           // use current (or none)
               {
                  if ((index = SINF->partid) == 0)
                  {
                     if ((index = dfsa->autoPid) == 0)
                     {
                        rc = DFS_NO_DEVICE;
                     }
                  }
               }
               if ((index != 0) && (rc == NO_ERROR)) // partition specified
               {
                  //- MUST be closed before opening something else, or flushing
                  //- from a FS-specific cache may go to the wrong object!
                  rc = DFSFNCALL(dfsa->FsClose,0,0,NULL,NULL); // Close filesystem

                  nav.index = 0;
                  rc = dfsSelectPartition( index);
                  if (rc == NO_ERROR)
                  {
                     SINF->object = DFSO_PART;  // partition
                     SINF->p      = dfsGetPartInfo( index);
                     SINF->disknr = SINF->p->disknr;
                     SINF->partid = index;
                     strcpy(  SINF->drive, SINF->p->drive);
                     strcpy(  SINF->afsys, SINF->p->fsform);
                     TRACES(("SINF->p->scluster : %hu\n", SINF->p->scluster));

                     if (dfsDid2DiskType( SINF->disknr) == DFSD_IMZD)
                     {
                        dfstSetReadOnly( DFSTORE, FALSE, TRUE);
                     }
                     else                       // other types, reset R/O
                     {
                        if (SINF->partid != index) // different partition
                        {
                           dfstSetReadOnly( DFSTORE, FALSE, FALSE);
                        }
                     }
                     dfsVerboseShowPart( SINF->p);
                     rc = dfsInitFileSystem();
                  }
                  if (rc == NO_ERROR)
                  {
                     if (TxaOptSet('R'))        // local readonly ?
                     {
                        dfstSetReadOnly( DFSTORE, TRUE, TxaOption('R'));
                     }
                     else if (dfsa->ReadOnly)   // global readonly ?
                     {
                        dfstSetReadOnly( DFSTORE, TRUE, TRUE);
                     }
                  }
               }
               else if (rc == NO_ERROR)
               {
                  TxPrint( "\nInvalid partition specification: '%s'\n", c1);
                  rc = DFS_VALUE_ERROR;
               }
            }
            else                                // partition list or table
            {
               if ((ml == FDSK_ANY) || (ml <= dfsGetDiskCount( FALSE)))
               {
                  dfsa->number = dfsShowPartitions( ml);
               }
               else
               {
                  TxPrint( "\nDisknr is invalid : %u\n", ml);
                  rc = DFS_VALUE_ERROR;
               }
            }
         }
      }
      else if (strcasecmp(c0, "raw2imz" ) == 0)
      {
         if (TxaOption('?') || (cc == 1))
         {
            TxPrint(
              "\nCreate a compressed imagefile (IMZ) from a RAW image or other file\n"
              "\n Usage:  %s   RAW-filename  [IMZ-filename]\n"
              "\n   RAW-filename = Filename of uncompressed inputfile"
              "\n   IMZ-filename = Filename of compressed outputfile\n"
              "\n   -m[:mcs]     = Maximum size per imagefile as mcs-nr, default 650 MiB"
              "\n   -M           = Media change prompt after each file (on removables)"
              "\n   -!           = Use dialog to check or specify more options\n", c0);
         }
         else
         {
            strcpy( s0, c1);
            TxFnameExtension( s0, "img");       // default extension
            if (TxFileExists( s0))
            {
               TXTT    moption;

               if (cc > 2)                      // 2nd parameter, outputfile
               {
                  strcpy( s1, c2);
               }
               else
               {
                  strcpy( s1, c1);              // copy input filename
                  TxStripExtension( s1);        // remove extension
               }
               if (TxaOption('m'))
               {
                  dfsOptionSizeSectors( 'm', DFSTORE, 'm', DFSECT_CD80, &size);
                  sprintf( moption, " -m:%u", DFSECT2MIB( size, dfsGetSectorSize()));
               }
               else
               {
                  strcpy( moption, "");
               }
               sprintf( dc, "image '%s' -I:'%s' -z%s%s%s", s1, s0,
                                      (TxaOption('!')) ? " -!" : "",
                                      (TxaOption('M')) ? " -M" : "", moption);
               rc = dfsMultiCommand( dc, 0, TRUE, FALSE, TRUE);
            }
            else
            {
               TxPrint( "\nRAW image file '%s' not found ...\n", s0);
               rc = DFS_NOT_FOUND;
            }
         }
      }
      else if (strcasecmp(c0, "imz2raw" ) == 0)
      {
         if (TxaOption('?') || (cc == 1))
         {
            TxPrint(
              "\nCreate uncompressed (RAW) imagefile from an IMZ compressed one\n"
              "\n Usage:  %s   IMZ-filename  [RAW-filename]\n"
              "\n   IMZ-filename  = Filename of compressed inputfile"
              "\n   RAW-filename  = Filename of uncompressed outputfile\n"
              "\n   -!            = Use dialog to check or specify more options\n", c0);
         }
         else
         {
            strcpy( s0, c1);
            TxFnameExtension( s0, "imz");       // default extension
            if ((TxaOption('!')) || (TxFileExists( s0)))
            {
               if (cc > 2)                      // 2nd parameter, outputfile
               {
                  strcpy( s1, c2);
               }
               else
               {
                  strcpy( s1, c1);              // copy input filename
                  TxStripExtension( s1);        // remove extension
               }
               TxFnameExtension( s1, "img");    // default extension
               if (TxCreateEmptyFile( s1, !dfsa->batch) == NO_ERROR)
               {
                  sprintf( dc, "restore '%s' -I:'%s' -S%s", s0, s1,
                                      (TxaOption('!')) ? " -!" : "");
                  rc = dfsMultiCommand( dc, 0, TRUE, FALSE, TRUE);
                  dfsMultiCommand( "close", 0, TRUE, FALSE, TRUE); // close image!
               }
            }
            else
            {
               TxPrint( "\nCompressed image file '%s' not found ...\n", s0);
               rc = DFS_NOT_FOUND;
            }
         }
      }
      else if (strcasecmp(c0, "imzinfo" ) == 0)
      {
         if (TxaOption('?') || (cc == 1))
         {
            TxPrint(
              "\nDisplay information from an (IMZ compressed) imagefile\n"
              "\n Usage:  %s   filename\n"
              "\n   filename  = Filename of inputfile\n", c0);
         }
         else
         {
            strcpy( s0, c1);
            TxFnameExtension( s0, "imz");       // default extension
            if (TxFileExists( s0))
            {
               rc = dfsImageInfoDisplay( s0);
            }
            else
            {
               TxPrint( "\nCompressed image file '%s' not found ...\n", s0);
               rc = DFS_NOT_FOUND;
            }
         }
      }
      else if ((strcasecmp(c0, "sim" ) == 0) || (strcasecmp(c0, "image"  ) == 0))
      {
         rc = dfsImageRestoreCommand( cc, c1, c2, c3, TRUE);
      }
      else if ((strcasecmp(c0, "wrim") == 0) || (strcasecmp(c0, "restore") == 0))
      {
         rc = dfsImageRestoreCommand( cc, c1, c2, c3, FALSE);
      }
      else if (strcasecmp(c0, "wipe") == 0)
      {
         rc = dfsWipeCommand( cc, c1, c2, c3);
      }
      else if (strcasecmp(c0, "slt" ) == 0)
      {
         if (TxaOption('?'))
         {
            TxShowTxt( cmd_slt);                // give usage
         }
         else if (TxaOption('x'))
         {
            dfsSlTableReset();                  // quit building SLT
            TxPrint( "\nSLT build stopped\n");
         }
         else
         {
            static BOOL  slt_verbose = TRUE;    // default SLT format
            static BOOL  slt_oneline = TRUE;
            BOOL         verbose = slt_verbose;
            BOOL         oneline = slt_oneline;

            if (TxaOptSet('v')) verbose = TxaOption('v'); // overruled
            if (TxaOptSet('1')) oneline = TxaOption('1'); // overruled
            if (!TxaOptUnSet('d'))
            {
               slt_verbose = verbose;           // keep as new defaults
               slt_oneline = oneline;
            }
            if (dfsa->Fsi != NULL)
            {
               TRACES(( "DirtyStatus now: %u\n", dfsa->FsDirtyStatus));
               if (TxaOption('r') || TxaOption('c')) // refresh existing SLT
               {
                  dfsSlTableReset();            // force rebuild of SLT
               }
               #if defined (HAVETHREADS)
               if ((rc = dfsSlTableStatus(&sltSize)) != SLT_READY)
               {
                  dfsa->sltProgress = TRUE;
                  TxPrint("\nBuilding SLT in the background, use <Esc> "
                            "to cancel waiting (but keep building)\n");
                  if ((rc = dfsSlTableStatus(&sltSize)) == SLT_EMPTY)
                  {
                     if (TxBeginThread( dfsSlTableCreate, 0xf000, dfsa->Fsi) == -1)
                     {
                        //- Thread failed to start, reason unknown ...
                        TxPrint( "\nFailed to start the SLT thread in the background, aborting ...\n");
                        TxSetPendingAbort();    // don't wait for <Esc> key, abort
                     }
                  }
                  else                          // restarting (check progress)
                  {
                     dfsProgressResume("");     // rebuild progress BAR
                  }
                  if (!TxaOption('P'))
                  {
                     do
                     {
                        TxSleep( 500);
                        if (TxAbort())
                        {
                           dfsProgressSuspend();
                           TxPrint( "Waiting for SLT canceled. "
                                    "Continue building in background ...\n");
                           break;
                        }
                     } while ((rc = dfsSlTableStatus(&sltSize)) != SLT_READY);

                     dfsa->sltProgress = FALSE; // back to DFS commandline
                     TxSleep( 200); //- make sure any SLT thread has chance to run
                     TxPrint("\n");
                  }
               }
               #else
                  rc = dfsSlTableStatus(&sltSize);
                  if (rc != SLT_READY)
                  {
                     dfsa->sltProgress = TRUE;
                     dfsSlTableCreate(dfsa->Fsi); // synchronious build SLT
                     rc = dfsSlTableStatus(&sltSize);
                  }
               #endif
               #if defined (USEWINDOWING)
               if (TxaOption('P'))
               {
                  if ((dfsSltDisplayDialog()  == DFS_NO_CHANGE) &&
                      (dfsSlTableStatus(NULL) != SLT_READY)) //- canceled dlg
                  {                                          //- while building
                     #if defined (HAVETHREADS)
                        if (TxConfirm( 5557,
                            "        The Sector Lookup Table is not ready yet.\n\n"
                            "You can choose to continue building it in the background\n"
                            "while working on the same object (partition / volume),\n"
                            "or you can cancel the build now, discarding results.\n\n"
                            "        Continue SLT build in the background ? [Y/N]: ", dfsa->sltProgress))
                        {
                           TxPrint( "\nWaiting for SLT canceled. "
                                    "Continue building in background ...\n");
                           dfsProgressSuspend();
                        }
                        else
                        {
                           dfsSlTableReset();   // quit building SLT
                           TxPrint( "\nWaiting for SLT canceled. "
                                    "Building is stopped.\n");
                        }
                     #else
                        rc = DFS_NO_CHANGE;
                     #endif
                  }
               }
               else
               #endif
               if (rc == SLT_READY)             // screen display modus
               {
                  TXTS          stype;

                  TRARGS(("SLT size: %u\n", sltSize));
                  strcpy( stype, "*");          // default all types
                  TxaOptAsString( 't', TXMAXTS, stype);

                  #if defined (USEWINDOWING)
                     ml2 = TxScreenRows() -12;  // one 'screen' full
                  #else
                     ml2 = 13;                  // assume 80x25
                  #endif
                  nr = ml2;

                  if (TxaOptSet('a'))           // around sectornumber
                  {
                     sn = TxaOptNum('a', "",   nav.this);
                     if (dfsSlTableFind( sn, NULL, NULL, NULL, &nav.index))
                     {
                        if (nav.index > (ml2 / 3))
                        {
                           nav.index -= (ml2 / 3);
                        }
                        else
                        {
                           nav.index  = 0;
                        }
                     }
                     else
                     {
                        TxPrint( "\nSpecified sector 0x0%llx not found in SLT!\n", sn);
                        nav.index = sltSize - ml2;
                        nr        = ml2;        // SN probably too large, display END
                     }
                  }
                  if (TxaOptSet('m'))           // display (some) errors only
                  {
                     ef = TxaOptNum('m', NULL, 0xffffffff);
                  }
                  else
                  {
                     ef = 0;                    // display non-errors too
                  }
                  if (cc > 1)
                  {
                     if (cc > 2)
                     {
                        if (c2[0] == '*')
                        {
                           nr = sltSize;
                        }
                        else
                        {
                           if ((nr = atol(c2)) == 0)
                           {
                              nr = ml2;
                           }
                        }
                     }
                     switch (c1[0])
                     {
                        case '-':
                        case '+':
                           nav.index += atol(c1) - ml2;
                           break;

                        case 'A':               // ALL
                        case 'a':
                           nav.index = 0;
                           nr        = sltSize;
                           break;

                        case 'E':               // from END
                        case 'e':
                           nav.index = sltSize - nr;
                           break;

                        case '.':
                           break;

                        default:
                           nav.index = atol(c1);
                           break;
                     }
                  }
                  if (TxaOptSet('e'))           // display end of table
                  {
                     nr = TxaOptNum('e', "", nr);
                     nav.index = sltSize - nr;
                  }
                  if (nav.index > sltSize)         // wrapped outside boundary
                  {
                     nav.index = 0;
                  }
                  TRARGS(("start: %u, size: %u\n", nav.index, nr));
                  nav.index = dfsSlTableDisplay( nav.index, nr, stype[0],
                                                 ef, verbose, oneline, &ei);
                  if (nav.index != 0)
                  {
                     nav.down = L64_NULL;       // force <Enter> to SLT cmd
                  }
                  else                          // wrapped back to beginning
                  {
                     dfsa->number = (ULONG) ei; // remember error count
                     nav.down = nav.this;       // back to last position
                  }
               }
               else
               {
                  TxPrint("SLT is not ready yet, no info available.\n");
               }
               rc = NO_ERROR;
            }
            else
            {
               TxPrint("%s command not supported for filesystem %s%s%s\n",
                       c0, CBM, SINF->afsys, CNN);
            }
         }
      }
      else if (strcasecmp(c0, "id"    ) == 0)
      {
         if (TxaOption('?'))
         {
            TxPrint("\nIdentify specified or current sector using the SLT\n");
            TxPrint("\n Usage:  %s  [ssn  |  -L]\n\n", c0);
            TxPrint("   ssn      = symbolic sector number to identify\n"
                    "   -L       = Use LSN of LAST used sector in FS\n"
                    "   -v       = Verbose, display of contents for sector (default)\n"
                    "   -v-      = Suppress display of contents for sector\n\n");
         }
         else
         {
            if (dfsSlTableStatus( NULL) != SLT_READY)
            {
               TxPrint( "\nBuilding sector lookup table, please wait ...\n");
               dfsMultiCommand( "slt 0 1 -v-", 0, TRUE, FALSE, FALSE);
            }
            if (TxaOption('L'))                 // last used sector/cluster
            {
               if (dfsa->FsLastInUse == 0)      // unknown last sector
               {
                  dfsMultiCommand( "alloc -l:1", 0, TRUE, FALSE, TRUE);
               }
               sn = dfsa->FsLastInUse;
            }
            else
            {
               sn = dfsGetSymbolicSN( c1, nav.this); // default current LSN
            }
            sl = L64_NULL;
            rc = dfsDisplaySltIdInfo( sn, &sl);
         }
      }
      else if (strcasecmp(c0, "smart" ) == 0)
      {
         if (TxaOption('?'))
         {
            TxPrint("\nShow allocation status (FDISK mode)\n");
            TxPrint("\n Usage:  %s  [ssn]\n\n", c0);
            TxPrint("   ssn      = symbolic sector number to test\n");
         }
         else
         {
            char       at = 0;
            BOOL       al;

            sn = dfsGetSymbolicSN( c1, nav.this); // default current LSN
            al = DFSFNCALL( dfsa->FsLsnAllocated, sn, 0, &at, NULL);
            TxPrint( "Allocation area   : '%c'   LSN: 0x0%llx status: %s\n",
                      at, sn, (al) ? "Allocated" : "Free");
         }
      }
      else if (strcasecmp(c0, "slt2list"    ) == 0)
      {
         if (TxaOption('?'))
         {
            TxPrint("\nAdd (selected) sectors from the SLT to the Sectorlist\n");
            TxPrint("\n Usage:  %s  [cl] [-m:mask] [-s:size] [-F:from]\n\n", c0);
            TxPrint("   cl       = Nr of sectors added per area, default 1\n"
                    "   -a       = Add to existing Sectorlist\n"
                    "   -F:from  = Include range start sector,   default 0\n"
                    "   -m:mask  = Error filtering mask, default 0xfffffffe\n"
                    "              resulting in just 'lost clusters' added.\n"
                    "   -m:0     = Add EVERY SLT area to the Sectorlist,\n"
                    "              not just the ones with reported errors\n"
                    "   -r         Add REFERENCE sectors to the list only\n"
                    "   -s:size  = Size of the 'include' range,  default ALL\n");
         }
         else
         {
            BOOL   ok    = TRUE;
            ULN64  lsize = dfsGetLogicalSize();
            ULONG  mask  = TxaOptNum('m', NULL, 0xfffffffe);
            TXTS           stype;

            strcpy( stype, "*");                // default all types
            TxaOptAsString( 't', TXMAXTS, stype);

            if (dfsSlTableStatus( NULL) != SLT_READY)
            {
               TxPrint( "\nBuilding sector lookup table, please wait ...\n");
               dfsMultiCommand( "slt 0 1 -v-", 0, TRUE, FALSE, FALSE);
            }
            ok &= dfsOptionSizeSectors( 'F', DFSTORE, 's', 0, &sn);       //- from
            if (sn < lsize)
            {
               lsize -= sn;                     // set size to remainder of object
            }
            ok &= dfsOptionSizeSectors( 's', DFSTORE, 's', lsize, &size); //- size

            if (ok)
            {
               if (!TxaOption('a'))             // add to existing list
               {
                  dfsInitList( 0, "-w", "-d");  // init, optimal list options
               }
               sectors = dfsGetMcsNumber( c1, 1);
               rc = dfsSltRange2List( sectors, sn, size, mask, TxaOption('r'), stype[0]);
            }
         }
      }
      else if (strcasecmp(c0, "bsfind"     ) == 0)
      {
         if (TxaOption('?'))                    // explicit help request
         {
            TxPrint("\nFind MBR, EBR, LVM, BOOT and SUPERBLOCK sectors on a disk\n");
            TxPrint("\n Usage:  %s  [types]  [All | Cyl | MiB | Both\n\n", c0);
            TxPrint("   types    = types of sectors to find, default 'brelL[s]' = all\n"
                    "              b = bootsectors            (BOOT)\n"
                    "              r = master boot records     (MBR)\n"
                    "              e = extended boot records   (EBR)\n"
                    "              l = LVM info sectors       (DLAT)\n"
                    "              L = LVM signature sectors   (BBR)\n"
                    "              s = FS superblocks (in non FDISK)\n"
                    "   A(ll)    = search ALL sectors, not just cyl.  (slowest)\n"
                    "   C(yl)    = search on cylinder boundaries only (fastest)\n"
                    "   M(iB)    = search on 1-MiB boundaries only\n"
                    "   B(oth)   = search on cylinder and 1-MiB boundaries\n\n"
                    "   -f:[pos] = set start position mcn-number, defaults to sectors\n");
         }
         else
         {
            dfstDiskGeometry( DFSTORE, 0, 0, 0, 0, 0, FALSE, TRUE);
            switch (toupper(c2[0]))
            {
               case 'A':
                  strcpy( dc, "find -first -o:*%! -t:");
                  strcpy( s1, "in every sector (quite slow!)\n");
                  break;

               case 'B':
                  sprintf( dc, "find -first -o:*%%!~ -i:0x%x -t:", ((ULONG) 0x100000 / dfsGetSectorSize()));
                  strcpy( s1, "on Cyl. and 1-MiB boundaries\n");
                  break;

               case 'C':
               default:
                  strcpy( dc, "find -first -o:*%!~ -t:");
                  strcpy( s1, "on cylinder boundaries (fast)\n");
                  break;

               case 'M':
                  sprintf( dc, "find -first -o:*%%! -i:0x%x -t:", ((ULONG) 0x100000 / dfsGetSectorSize()));
                  strcpy( s1, "on MiB boundaries only\n");
                  break;
            }
            TxPrint( "Search sect types : ");
            if ((cc > 1) && (c1[0] != '*'))     // specified types
            {
               TxPrint( "'%s' %s", c1, s1);
               strcat( dc, c1);
            }
            else
            {
               TxPrint( "MBR/EBR, BOOT, LVM + SUPERBLOCKS %s", s1);
               strcat( dc, "brelLs");
            }
            //- set start position to search FROM (default is MBR)
            dfsOptionSizeSectors( 'f', DFSTORE, 's', 0, &nav.this);

            rc = dfsMultiCommand( dc, 0, TRUE, FALSE, TRUE);
            strcpy( dfsa->sncompact, "-s");     // overrule default find
            strcpy( dfsa->snverbose, "-b");     // optimal list options

            //- cancel a possible abort, so DFSDISK procedure will continue
            //- with the detailed display and generation of other files
            TxCancelAbort();                    // might have aborted
         }
      }
      else if (strcasecmp(c0, "cslist"     ) == 0)
      {
         if (TxaOption('?'))                    // explicit help request
         {
            TxPrint("\nPut Cylinder aligned special Sectors into sector-list\n");
            TxPrint("\n Usage:  %s  [-s] [-m]\n\n", c0);
            TxPrint("   option    -m = Include MiB boundaries (0x800 multiples)\n");
            TxPrint("             -s = Include potential 'superblock' sectors\n\n");
         }
         else
         {
            dfstDiskGeometry( DFSTORE, 0, 0, 0, 0, 0, FALSE, TRUE);

            rc = dfsCylinderSectorList();

            TxPrint("\nSN list contains  : %s%u%s special sector(s)\n",
                       CBY, dfsa->snlist[0], CNN);
         }
      }
      else if (strncasecmp(c0, "dfscheck", 7 ) == 0) // run DFSxxxx check for FS
      {
         if ((!strchr( c1, '?')) && !TxaOption('?'))
         {
            strcpy( s1, (cc == 1) ? "*" : c1);
            switch ((index = dfsParsePartSpec( s1, dfsGetDiskNumber( 'd', ".", "", FALSE), &p)))
            {
               case FDSK_ANY:                   // all partitions
                  if (cc <= 2)
                  {
                     ml = dfsPartitions();
                     for (index = 1; index <= ml; index++)
                     {
                        TxPrint("\nStarting analysis : CHKPART for partition: %hu (of %u)\n", index, ml);
                        rc = dfsRunChkPartAnalysis( index, "");
                        TxCancelAbort();        // might have aborted script
                     }
                  }
                  else
                  {
                     TxPrint( "Specifying a filesystem ('%s') invalid with '%s'\n", c2, s1);
                     rc = DFS_VALUE_ERROR;
                  }
                  break;

               case 0:                          // invalid partition
                  TxPrint( "Partition '%s' not found\n", s1);
                  rc = DFS_VALUE_ERROR;
                  break;

               default:                         // specific partition
                  rc = dfsRunChkPartAnalysis( index, c2);
                  break;
            }
            if (rc == NO_ERROR)
            {
               sprintf( s0, "\nRegistered users may call for support on analysing the results, and\n"
                       "when possible recovery of damaged filesystems, by sending the (zipped)\n"
                       DFSDISKCHECK "*.* files plus any other useful info on your problem to:\n\n"
                       "                          %ssupport@dfsee.com%s\n\n"
                       "Note: A DFSDISK run on the same disk(s) will often be VERY helpful!\n", CBG, CNN);
               TxNamedMessage( !TxaOptUnSet( TXA_O_SCREEN), 0, " NOTE: Filesystem analysis finished ",
                               "%21.21sDFSCHECK procedure finished.\n%s", "", s0);
            }
         }
         else
         {
            TxPrint("\nCollect information to aid in filesystem and other recovery situations.\n"
                    "Uses a specific 'dfsXXXX' script for filesystems: FAT, HPFS, NTFS, JFS.\n");
            TxPrint("Output will be in " DFSDISKCHECK "XXXX.?nn files (overwriting old ones).\n");
            TxPrint("\nUsage: %s  *   | part  [fsys]\n\n", c0);
            TxPrint("    *      = Analyse ALL partitions on all disks\n");
            TxPrint("    part   = Partition-ID for one partition to analyse\n");
            TxPrint("    fsys   = Filesystem, only valid with explicit 'part', any of:\n\n"
                    "               FAT    =  For any FAT16 or FAT32 partition\n"
                    "               NTFS   =  For the Windows NTFS filesystem\n"
                    "               HPFS   =  For OS/2 and eCS HPFS[386] filesystem\n"
                    "               JFS    =  For OS/2 and eCS Journaled filesystem\n");
         }
      }
      else if (strncasecmp(c0, "dfsdisk", 7 ) == 0)   // create analysis files
      {
         index = (cc > 1) ? dfsParseDiskSpec(c1, NULL) : FDSK_ANY;

         #if defined (USEWINDOWING)
         if (((cc == 1)        || TxaOption('!')) &&  //- no dfsdproc param, no help
              (dfsa->dialogs) && !TxaOption('?'))     //- and dialogs are allowed
         {
            rc = dfsDfsDiskDialog( index, TxaOptStr( 'b', "baseName", DFSDISK_BASE));
         }
         else                                   // run the DFSDISK command
         #endif
         {
            if (!TxaOption('?'))
            {
               USHORT        first;
               USHORT        last;

               first = dfsParseDiskSpec( c1, &last); // request as a range
               if (last != 0)                   // valid single disk or range
               {
                  for (index = first; (index <= last) && (rc == NO_ERROR); index++)
                  {
                     if (dfsDiskAccessible( index))
                     {
                        TxPrint("\nStarting analysis : DFSDISK for disk: %hu (of %u)\n", index, last - first + 1);
                        rc = dfsRunDfsDiskAnalysis( index, c2, c3, c4, c5,
                                         TxaOptStr( 's', "super-FS", "ANY"),
                                         TxaOptStr( 'b', "baseName", ""));
                     }
                     else
                     {
                        TxPrint("\nSkipping analysis : DFSDISK for disk: %hu (of %hu) Disk is inaccessible\n", index, last - first + 1);
                     }
                     TxCancelAbort();           // might have aborted script
                  }
               }
               else
               {
                  TxPrint( "Disknr is invalid : %u\n", first);
                  rc = DFS_VALUE_ERROR;
               }
               if (rc == NO_ERROR)
               {
                  sprintf( s0, "\nRegistered users may call for support on analysing the results, and\n"
                          "when possible recovery of lost partitions or other damage, by sending\n"
                          "the (zipped) files plus any other useful info on your problem to:\n\n"
                          "                          %ssupport@dfsee.com%s\n", CBG, CNN);
                  TxNamedMessage( !TxaOptUnSet( TXA_O_SCREEN), 0, " NOTE: Full disk analysis finished ",
                                  "%21.21sDFSDISK procedure finished.\n%s", "", s0);
               }
            }
            else
            {
               TxPrint("\nCollect information to aid in UNFDISK and other recovery situations.\n"
                       "Includes a backup of partitioning information and a generic overview.\n");
               TxPrint("Output will be in " DFSDISK_BASE ".* files (overwriting old ones).\n");
               TxPrint("\nUsage: %s  disk  [A | C | M | B  [? | #cyl [#head [#sect]]]] [-b:file] [-s:FS]\n\n", c0);
               TxPrint("    disk   = Number for one disk, '*' for all disks\n");
               TxPrint("    A(ll)  = search ALL sectors, not just cyl.  (slowest)\n"
                       "    B(oth) = search on cylinder and 1-MiB boundaries\n"
                       "    C(yl)  = search on cylinder boundaries only (fastest)\n"
                       "    M(iB)  = search on 1-MiB boundaries only\n");
               TxPrint("     ?     = calculate #cyl for the specified #head and #sect\n"
                       "    #cyl   = Explicit number of cylinders for GEO, or '.' for current\n");
               TxPrint("    #head  = Explicit number of heads     for GEO, or '.' for current\n"
                       "    #sect  = Explicit number of sectors   for GEO, or '.' for current\n");
               TxPrint("   -!      = Force interactive dialog for DFSDISK parameters and options.\n");
               TxPrint("   -b:Name = Base filename for output, may include a full path component.\n");
               TxPrint("   -s:FS   = Filesystem for which superblocks are to be found, any of:\n\n"
                       "                 FDISK  =  NO filesystem superblocks, just partitioning\n"
                       "                 ANY    =  ANY type of FS superblock supported (default)\n\n"
                       "                 APFS   =  Only macOS APFS native filesystem\n"
                       "                 EXT    =  Only Linux EXT2, EXT3 or EXT4\n"
                       "                 HFS    =  Only macOS HFS+ native filesystem\n"
                       "                 HPFS   =  Only OS/2 or ArcaOS/eCS HPFS[386]\n"
                       "                 JFS    =  Only OS/2 or ArcaOS/eCS Journaled FS\n"
                       "                 NTFS   =  Only Windows NT/XP/Vista/7/8/10\n"
                       "                 REISER =  Only Linux ReiserFS\n"
                       "                 XFS    =  Only Linux XFS filesystem\n\n");
            }
         }
      }
      else if (strncasecmp(c0, "dfsfast", 7 ) == 0)   // create analysis files, fast
      {
         index = (cc > 1) ? dfsParseDiskSpec(c1, NULL) : FDSK_ANY;

         #if defined (USEWINDOWING)
         if (((cc == 1)        || TxaOption('!')) &&  //- no dfsdfast param, no help
              (dfsa->dialogs) && !TxaOption('?'))     //- and dialogs are allowed
         {
            rc = dfsDfsFastDialog( index, TxaOptStr( 'b', "baseName", DFSFAST_BASE));
         }
         else                                   // run the DFSFAST command
         #endif
         {
            if (!TxaOption('?'))
            {
               USHORT        first;
               USHORT        last;

               first = dfsParseDiskSpec( c1, &last); // request as a range
               if (last != 0)                   // valid single disk or range
               {
                  for (index = first; (index <= last) && (rc == NO_ERROR); index++)
                  {
                     if (dfsDiskAccessible( index))
                     {
                        TxPrint("\nStarting analysis : DFSFAST for disk: %hu (of %u)\n", index, last - first + 1);
                        rc = dfsRunDfsFastAnalysis( index, TxaOptStr( 'b', "baseName", ""));
                     }
                     else
                     {
                        TxPrint("\nSkipping analysis : DFSFAST for disk: %hu (of %hu) Disk is inaccessible\n", index, last - first + 1);
                     }
                     TxCancelAbort();           // might have aborted script
                  }
               }
               else
               {
                  TxPrint( "Disknr is invalid : %u\n", first);
                  rc = DFS_VALUE_ERROR;
               }
               if (rc == NO_ERROR)
               {
                  sprintf( s0, "\nRegistered users may call for support on analysing the results,\n"
                          "by sending the (zipped) files plus any other info on your problem to:\n\n"
                          "                          %ssupport@dfsee.com%s\n", CBG, CNN);
                  TxNamedMessage( !TxaOptUnSet( TXA_O_SCREEN), 0, " NOTE: Fast disk analysis finished ",
                                  "%21.21sDFSFAST procedure finished.\n%s", "", s0);
               }
            }
            else
            {
               TxPrint("\nCollect information to aid in disk partitioning and other recovery situations.\n"
                         "Includes a backup of partitioning information and a generic overview.\n");
               TxPrint("Output will be in " DFSFAST_BASE ".* files (overwriting old ones).\n");
               TxPrint("\nUsage: %s  disk  [-b:file]\n\n", c0);
               TxPrint("    disk   = Number for one disk, '*' for all disks\n");
               TxPrint("   -!      = Force interactive dialog for DFSFAST parameters and options.\n");
               TxPrint("   -b:Name = Base filename for output, may include a full path component.\n");
            }
         }
      }
      else if (strncasecmp(c0, "memstat", 7 ) == 0) // Memory-Disks status
      {
         dfsShowMdiskStatus(0);                 // for all Memory disks
      }
      else if (strcasecmp( c0, "clone" ) == 0)  // clone from other store
      {
         rc = dfsCloneCompCommand( cc, c1, c2, TRUE);
      }
      else if (strncasecmp(c0, "comp", 4) == 0) // compare to other store
      {
         rc = dfsCloneCompCommand( cc, c1, c2, FALSE);
      }
      else if (strncasecmp(c0, "close", 5) == 0)
      {
         DFSFNCALL(dfsa->FsClose,0,0,NULL,NULL);  // FS specific handler
         rc = dfstClose( DFSTORE);                // Close default store
         SINF->disknr  = 0;
         SINF->partid  = 0;
         strcpy(SINF->drive, "--");
         strcpy(SINF->afsys, "--none--");
         #if !defined (OEM_BRANDED)
            TxPrint( "Current object closed, status values reset to 0 0 -- --none--\n");
         #endif
      }
      else if (strcasecmp(c0, "edit"   ) == 0)  // Edit sector contents
      {
         if (TxaOption('?'))                    // explicit help request
         {
            TxShowTxt( cmd_edit);
         }
         else
         {
            size  = 0;                          // default (mark) size, NO-MARK!
            index = (USHORT) TxaOptNum('p', " Position  ", 0);
            if (TxaOptMutEx( TRUE, "ahuP", NULL, 0)) // NO hex/ascii/unicode string
            {
               TXLN    p1;                      // copy first param (can be empty)

               strcpy( p1, c1);                 // filename, sectornumber, dir or empty

               #if defined (USEWINDOWING)
               if (TxaOptSet( TXA_O_DIR))       // directory as string-opt or param
               {
                  if (txwIsWindow( TXHWND_DESKTOP)) // does not work non-windowed
                  {
                     if (strlen( p1) == TxStrWcnt( p1)) // no wildcard ?
                     {
                        strcpy( s0, TxaOptStr( TXA_O_DIR, NULL, p1)); // p1 is default value
                        strcpy( s1, FS_WILDCARD);
                     }
                     else                       // param is a wildcard, use for OpenDialog
                     {
                        strcpy( s0, TxaOptStr( TXA_O_DIR, NULL, "")); // current dir is default
                        strcpy( s1, p1);        // file selection wildcard
                     }
                     if (txwOpenFileDialog( s1, (strlen( s0)) ? s0 : NULL, NULL, DFSC_SHFILE, NULL, NULL,
                         " Select a file to be HEX/ASCII edited ", p1))
                     {
                        TxPrint( "Selected file for edit: '%s'\n", p1);
                     }
                     else
                     {
                        rc = DFS_USER_ABORT;
                     }
                  }
                  else
                  {
                     TxPrint("\nHex edit using a file-dialog only works in windowed mode (use 'mode -w')\n");
                  }
               }
               #endif
               if (rc == NO_ERROR)              // not canceled (filedialog)
               {
                  #if defined (USEWINDOWING)
                     if (TxaOptMutEx((strlen(p1) > 1), "F", "if start-sector or filename specified!", NULL))
                     {
                        ULN64   orgSize = 0;    // original size binary file
                        ULN64   newSize = 0;    // new      size binary file
                        BOOL    binFile = FALSE;

                        if (TxaOptSet('F'))
                        {
                           dfsOptionSizeSectors( 'F', DFSTORE, 's', nav.this, &sn);
                        }
                        else
                        {
                           if ((p1[0] != '.') && (((binFile = TxFileExists( p1)) != FALSE) ||
                                  (strcspn( p1, ":." FS_PATH_STR) < strlen( p1)) ))
                           {
                              //- Open the image RO when -A-, RW otherwise (append option)
                              sprintf( dc, "im %s -f- -RAW '%s'", (TxaOptUnSet('A')) ? "-A-" : "-A", p1);
                              if ((rc = dfsMultiCommand(dc, 0, FALSE, FALSE, TRUE)) == NO_ERROR)
                              {
                                 binFile = TRUE;
                                 orgSize = dfstGetByteSize( DFSTORE);
                              }
                              else
                              {
                                 binFile = FALSE;
                                 TxNamedMessage( !dfsa->batch, 0, " ERROR: Binary file edit ", "File '%s' can not be opened!", p1);
                              }
                              sn = 0;           // start at start of binary file
                           }
                           else
                           {
                              sn = dfsGetSymbolicSN( p1, nav.this);
                              if (TxaOptSet('e')) // e option specified ?
                              {
                                 sninfo = TxaOptNum('e', "Entry", nav.this_sninfo);
                                 size  = sizeof(S_FATDIR);
                                 index = sninfo * size;
                              }
                           }
                        }
                        if (rc == NO_ERROR)
                        {
                           if (sn < dfsGetLogicalSize())
                           {
                              TxPrint("\n");    // create one empty separator line
                              dfsHexObjectEditor( sn,
                                                  TxaOptNum('s', NULL, dfsGetClusterSize()),
                                                  TxaOptNum('M', NULL, (ULONG) index),
                                                  TxaOptNum('m', NULL,          size),
                                            NULL, TxaOptNum('v', NULL, DFSH_HEX_VIEW),
                                           index, TxaOptNum('r', NULL, 0),
                                                  TxaOptNum('c', NULL, 0));

                              newSize = dfstGetByteSize( DFSTORE);
                              if (binFile)
                              {
                                 if (newSize < orgSize)
                                 {
                                    if ((!dfsa->batch) && (!TxConfirm( 5517,
                                       "The size of the file has been reduced "
                                       "to %llu bytes, (%llu bytes deleted).\n\n"
                                       "Update the file information on the disk to "
                                       "reflect this smaller size ? [Y/N]: ",
                                        newSize, (orgSize - newSize))))
                                    {
                                       newSize = orgSize; // revert size change
                                       dfstSetByteSize( DFSTORE, newSize);
                                    }
                                 }
                                 TxPrint( "File size at exit : ");
                                 if      (newSize < orgSize)
                                 {
                                    TxPrint( "%llu bytes, %llu bytes deleted\n",
                                              newSize, orgSize - newSize);
                                 }
                                 else if (newSize > orgSize)
                                 {
                                    TxPrint( "%llu bytes, %llu bytes added\n",
                                              newSize, newSize - orgSize);
                                 }
                                 else
                                 {
                                    TxPrint( "%llu bytes, size was not changed\n\n",
                                              newSize);
                                 }
                                 dfstClose( DFSTORE); // close the imagefile
                              }
                           }
                           else
                           {
                              TxNamedMessage( !dfsa->batch, 0, " ERROR: Invalid Sectornumber ", "LSN out of range: 0x%0llx", sn);
                           }
                        }
                     }
                  #else
                     TxPrint( "\nInteractive HEX editor not supported in this version.\n");
                  #endif
               }
            }
            else                                // hex/ascii/unicode string
            {
               if (TxaOptMutEx((cc > 1), "p", "if position parameter is specified!", NULL))
               {
                  TxPrint( "\nSector editor\n");
                  index = (USHORT) dfsGetMcsNumber( c1, index);
                  if (((cc == 1) && !TxaOptSet('p')) || TxaOption('P'))
                  {
                     sprintf( s1, "0x%hx", index);

                     TxPrompt( 5501, 7, s1, "Byte position/offset of area to edit, "
                               "current value: 0x%hx hex = %hu decimal", index, index);

                     index = (USHORT) dfsGetMcsNumber( s1, index);
                  }

                  TxPrint("\nData in disk-buffer at offset 0x%2.2X:\n", index);
                  TxDisplayHex( "",  rbuf + index, (ml | 0xf) + 1,      index);

                  TxaGetArgString(TXA_CUR, 2, 0, TXMAXLN, s0); // args, starting with 2nd
                  ml = dfsGetMixedString( s0, "replacement", TRUE, FALSE, (char *) &ms[0]);

                  if (ml > 0)                   // data available
                  {
                     memcpy( rbuf + index, &ms[0], ml);

                     TxPrint("\nReplaced by data of length %u = 0x%2.2X:\n", ml, ml);
                     TxDisplayHex( "", rbuf + index, ml, index);

                     if ( TxaOption('w') ||     // -w:1 or -w+ option used
                         !TxaOptSet('w'))       // or not set at all
                     {
                        sprintf( dc, "write . -a"); // to same sector, prompt

                        rc = dfsMultiCommand(dc, 0, FALSE, FALSE, TRUE);
                     }
                     else                       // -w- no direct write yet
                     {
                        TxPrint("\nUse 'show' command to display, 'write' "
                                "command to write it back (commit)\n");
                     }
                  }
                  else
                  {
                     TxPrint("No valid data specified\n");
                  }
               }
            }
         }
      }
      else if (strcasecmp(c0, "write"  ) == 0)     // Write-back 'this' sector
      {
         if (TxaOption('?'))                    // explicit help request
         {
            TxPrint("\nWrite back the current sector (after editing :-)\n");
            TxPrint("\n Usage:  %s  [ .  | sn  | -at[:sn] ]\n", c0);
         }
         else
         {
            if (DFSTORE_WRITE_ALLOWED)
            {
               sn = dfsGetSymbolicSN( c1, nav.this); // current sector is default

               if (TxaOption('a'))
               {
                  if ((sl = TxaOptNum('a', NULL, DFS_MAX_PSN)) != DFS_MAX_PSN)
                  {
                     sn = sl;
                  }
                  if (!dfsa->batch)
                  {
                     sprintf(  s1, "0x0%llx", sn);
                     TxPrompt( 5511, 11, s1, "Sectornumber to write to (mcs-number).\n"
                                             "Original sectornumber was: 0x0%llx", nav.this);
                     sn = dfsGetSymbolicSN( s1, nav.this);
                  }
               }
               if ((dfsa->batch) ||             // forced or confirmed
                   (TxConfirm( 5516,
                  "Write edited sector to sectornumber %8.8llx ? [Y/N]: ", sn)))
               {
                  if ((rc = dfsWrite(sn, 1, rbuf)) == NO_ERROR)
                  {
                     TxPrint("\nCurrent sector contents written to %llX\n", sn);
                  }
                  else
                  {
                     TxPrint("\nWrite failed, LSN : %llX\n", sn);
                  }
               }
            }
            else
            {
               rc = DFS_READ_ONLY;
            }
         }
      }
      else if (strcasecmp(c0, "im"     ) == 0)
      {
         strcpy( s1, TxaOptStr('e', "extension", "imz"));
         if (TxaOption('?'))                    // explicit help request
         {
            TxPrint("\nOpen specified imagefile, or present available ones\n");
            TxPrint("\n Usage:  %s  [fname | dir] [-R-] [-f-] [-A-] [-e:ext]\n\n", c0);
            TxPrint("   fname    = Name of a RAW or Compressed image to open\n"
                    "   dir      = Open-dialog for directory (must end in '%c')\n\n"
                    "   -A       = Allow append or delete characters in'RAW'   (default)\n"
                    "   -A-      = Do NOT allow appending/delete in 'RAW' image\n"
                    "   -e:'ext' = Overrule default 'imz' extension with 'ext'\n"
                    "   -f       = Initialise for detected Mode=xxx filesystem (default)\n"
                    "   -f-      = Do NOT initialise for any Mode=xxx filesystem\n"
                    "   -fs:'fs' = Initialise for specified filesystem 'fs'\n"
                    "   -R       = Open in Read-only mode, no writes           (default)\n"
                    "   -R-      = Do NOT open in Read-only mode, allow writes\n"
                    "   -RAW     = Force open in RAW mode, even for IMZ/VDI etc\n", FS_PATH_SEP);
         }
         else
         {
            strcpy( s0, c1);                    // filename
            TxRepl( s0, FS_PALT_SEP, FS_PATH_SEP); // fixup ALT separators
            if ((cc > 1) && (strlen(s0)    == TxStrWcnt(s0)) && // not a wildcard
                (         s0[strlen(s0)-1] != FS_PATH_SEP)   && // not a directory
                ((FS_PATH_SEP ==    s0[0]) || isalnum(s0[0]) || // and looks like
                 (strchr("_-!#$%",  s0[0]) != NULL)          )) // valid filename
            {
               SINF->p      = NULL;
               SINF->disknr = 0;
               SINF->partid = 0;

               if (!TxFileExists( s0))          // as given (perhaps no ext)
               {
                  TxFnameExtension( s0, s1);    // add default extension
               }
               rc = dfstOpenImageFile( DFSTORE, s0, !TxaOptUnSet('A'), TxaOption(TXA_O_RAW), TRUE, &h1);
               if (rc == NO_ERROR)
               {
                  if (h1 > 1)                   // multiple files present
                  {
                     TxPrint( "\nMulti-file image  : %u files in total\n", h1);
                  }
                  strcpy( SINF->drive, s0);

                  if (!TxaOptUnSet('f'))        // no -fs-, suppress FS-init ?
                  {
                     rc = DFSFNCALL(dfsa->FsClose,0,0,NULL,NULL); // Close FS
                     strcpy( SINF->afsys, "");
                     rc = dfsInitFileSystem();
                     if ((rc == NO_ERROR) && (!TxaOptUnSet('R')))
                     {
                        dfstSetReadOnly( DFSTORE, TRUE, !TxaOptUnSet('R'));
                     }
                  }
               }
               else
               {
                  TxPrint("Image file '%s' not found\n", s0);
                  strcpy( SINF->drive, "--");
               }
            }
            else                                // Present dialog, re-execute IM
            {
               #if defined (USEWINDOWING)
                  dfsImageSelectDialog( c1, DFSD_IRAW, "*");
               #else
                  TxPrint( "\nAvailable image files in current directory are:");
                  sprintf( s0, "%s *.%s", FS_DIR_CMD, s1);
                  TxExternalCommand( s0);       // show image files
               #endif
            }
         }
      }
      else if (strcasecmp(c0, "imz") == 0)
      {
         strcpy( s1, TxaOptStr('e', "extension", "imz"));
         if (TxaOption('?'))                    // explicit help request
         {
            TxPrint("\nOpen IMZ compressed partition/volume/disk image, for Browsing\n");
            TxPrint("\n Usage:  %s  [fname | dir] [-e:ext]\n\n", c0);
            TxPrint("   fname    = Name of a Compressed (IMZ) image to open\n"
                    "   dir      = Open-dialog for directory (must end in '%c')\n\n"
                    "   -e:'ext' = Overrule the default 'imz' extension with 'ext'\n\n"
                    "   -p:[rP]  = On partitioned-disk IMZ, Browse relative part # rP, default FIRST\n"
                    "              without -p, a partition-selecion list is presented for the disk\n", FS_PATH_SEP);
         }
         else
         {
            strcpy( s0, c1);                    // filename
            TxRepl( s0, FS_PALT_SEP, FS_PATH_SEP); // fixup ALT separators
            if ((cc > 1) && (strlen(s0)    == TxStrWcnt(s0)) && // not a wildcard
                (         s0[strlen(s0)-1] != FS_PATH_SEP)   && // not a directory
                ((FS_PATH_SEP    == s0[0]) || isalnum(s0[0]) || // and looks like
                 (strchr("_-!#$%",  s0[0]) != NULL)          )) // valid filename
            {
               if (!TxFileExists( s0))          // as given (perhaps no ext)
               {
                  TxFnameExtension( s0, s1);    // add default extension
               }

               //- determine type of image, partitioned whole-disk or anything else
               dfsImageTypeIsImz( s0, TRUE, NULL, NULL, NULL, NULL, NULL, NULL, &st);

               if (st == ST_MASTR)              // whole disk, mount -i, browse
               {
                  sprintf( s1, "attach -P- -i:'%s'", s0);
                  dfsMultiCommand( s1, 0, TRUE, FALSE, TRUE); // new virtual disk will be current disk

               #if defined (USEWINDOWING)
                  if (!TxaOptSet( 'p'))
                  {
                     ULONG     listcode  = TXDID_OK;

                     TXSELIST *dskPartList = dfsPartSelist( FALSE, SINF->disknr, '*');
                     TXRECT    where  = {1,0,0,0}; // fixed position

                     if (dskPartList != NULL)
                     {
                        if (dskPartList->count < (TxScreenRows() - 6))
                        {
                           where.top = (TxScreenRows() - dskPartList->count - 6) / 2;            //- center vertically
                        }
                        if (strlen( dskPartList->items[0]->text) < (TxScreenCols() - 4))
                        {
                           where.left = (TxScreenCols() - strlen( dskPartList->items[0]->text) - 4) / 2; // horizontally
                        }
                        listcode = txwListBox( TXHWND_DESKTOP, TXHWND_DESKTOP, &where,
                                               " Select IMZ partition to browse ", "", 0, TXLB_MOVEABLE,
                                               cSchemeColor, cSchemeColor, // std list colors
                                               dskPartList);
                        if (listcode > TXDID_MAX)      //- return list selection
                        {                              //- plus current item base value
                           sprintf( dc, "browse %u", listcode - TXDID_MAX); // browse selected
                           TxaReParseCommand( dc);
                           rc = DFS_PENDING;    // handle translated command
                        }
                        else                    // most likely canceled
                        {
                           rc = DFS_USER_ABORT;
                        }
                        txSelDestroy( &(dskPartList)); // free selection list
                     }
                  }
                  else                           // Browse pre-selected partition, default 1
               #endif
                  {
                     sprintf( dc, "browse %llu,r", TxaOptNum( 'p', NULL, 1)); // default FIRST
                     TxaReParseCommand( dc);
                     rc = DFS_PENDING;          // handle translated command
                  }
               }
               else                             // not a whole disk, 'im', browse root of FS
               {
                  sprintf( s1, "im '%s'",   s0);
                  dfsMultiCommand( s1, 0, TRUE, FALSE, TRUE);
                  sprintf( dc, "browse %c", FS_PATH_SEP);
                  TxaReParseCommand( dc);
                  rc = DFS_PENDING;             // handle translated command
               }
            }
            else                                // Present dialog, re-execute IMZ
            {
               #if defined (USEWINDOWING)
                  dfsImageSelectDialog( s0, DFSD_IMZD, s1);
               #else
                  TxPrint( "\nAvailable image files in (given) directory are:");
                  sprintf( s0, "%s *.%s", FS_DIR_CMD, s1);
                  TxExternalCommand( s0);       // show image files
               #endif
            }
         }
      }
      else if (strcasecmp(c0, "vdi") == 0)
      {
         strcpy( s1, TxaOptStr('e', "extension", "vdi"));
         if (TxaOption('?'))                    // explicit help request
         {
            TxPrint("\nOpen VDI, static/dynamic/snapshot VirtualBox disk image\n");
            TxPrint("\n Usage:  %s  [fname | dir] [-e:ext]\n\n", c0);
            TxPrint("   fname    = Name of a VirtualBox disk image to open/add\n"
                    "   dir      = Open-dialog for directory (must end in '%c')\n\n"
                    "   -e:'ext' = Overrule default 'vdi' extension with 'ext'\n", FS_PATH_SEP);
         }
         else
         {
            strcpy( s0, c1);                    // filename
            TxRepl( s0, FS_PALT_SEP, FS_PATH_SEP); // fixup ALT separators
            if ((cc > 1) && (strlen(s0)    == TxStrWcnt(s0)) && // not a wildcard
                (         s0[strlen(s0)-1] != FS_PATH_SEP)   && // not a directory
                ((FS_PATH_SEP ==    s0[0]) || isalnum(s0[0]) || // and looks like
                 (strchr("_-!#$%",  s0[0]) != NULL)          )) // valid filename
            {
               if (!TxFileExists( s0))          // as given (perhaps no ext)
               {
                  TxFnameExtension( s0, s1);    // add default extension
               }
               sprintf( s1, "attach -P- -i:'%s'", s0);
               dfsMultiCommand( s1, 0, TRUE, FALSE, TRUE);
            }
            else                                // Present dialog, re-execute VDI
            {
               #if defined (USEWINDOWING)
                  sprintf( s0, "*.%s",  s1);
                  dfsImageSelectDialog( s0, DFSD_VBOX, "vdi");
               #else
                  TxPrint( "\nAvailable IMZ image files in current directory are:");
                  sprintf( s0, "%s *.%s", FS_DIR_CMD, s1);
                  TxExternalCommand( s0);       // show image files
               #endif
            }
         }
      }
      else if (strcasecmp(c0, "media"   ) == 0)
      {
         if (TxaOption('?'))                    // explicit help request
         {
            TxPrint("\nDisplay of partitionable media (DFSee disk list), optional Update\n\n");
            TxPrint(" Usage:  %s  [-update  [-d:count | -d-]]  [-v]\n\n", c0);
            TxPrint("   -update  = Detach all, re-scan physical disks and attach them\n"
                    "   -d:count = Only attach the first 'count' physical disks\n"
                    "   -d       = Attach all detected media to disk list  (default)\n"
                    "   -d-      = Don't attach any, same as '-d:0', empty disk list\n"
                    "   -v       = Verbose, include a 'map' display of updated list\n\n");

         }
         else
         {
            if (TxaOption( TXA_O_UPDATE))       // update/refresh list
            {
               DFSFNCALL(dfsa->FsClose,0,0,NULL,NULL); // FS specific handler, close
               dfstInitStoreInfo();             // close all open stores
               dfsmDetachMedium( DFSM_UNMOUNT_ALL, FALSE); // quietly detach

               dfsInitMediaMap( TxaOptNum( 'd', NULL, DFS_MAX_DISK));

               TxPrint("\n");
               dfsGetDiskCount( TRUE);          // display number of disks, verbose

               dfsReadDiskInfo( FDSK_QUIET);    // re-read all partition info
               dfsFdskInit();                   // Well defined startup mode
            }
            if (TxaOption( 'v'))                // verbose
            {
               dfsShowMap( 0, '-');
            }
            dfsShowMediaMap( atoi( c1), atoi(c2)); // show partitionable media map
         }
      }
      else if ((strcasecmp(c0, "vol"    ) == 0) ||
               (strcasecmp(c0, "dev"    ) == 0))
      {
         if (TxaOption('?'))                    // explicit help request
         {
            TxPrint("\nOpen specified device/volume, or list available ones\n\n");
            #if defined (UNIX)
               TxPrint(" Usage:  %s  [device name] [-R]\n\n", c0);
               TxPrint("    device name : full name or basename (no leading '/dev/')\n\n");
            #else
               TxPrint(" Usage:  %s  [volume-driveletter [-R]] |  [opts]\n\n", c0);
               TxPrint("     -c   = Include CDROM drives in volume list (default)\n"
                       "     -c-  = Exclude CDROM drives from volume list\n");
               TxPrint("     -f   = Include floppy-disks in volume list (default)\n"
                       "     -f-  = Exclude floppy-disks from volume list\n");
            #endif
            #if defined (DEV32)
               TxPrint("     -l   = Include LAN drives in volume list   (default)"
                       "     -l-  = Exclude LAN drives from volume list");
            #endif
            TxPrint(   "     -R   = open device/volume in read-only mode\n\n");
         }
         else
         {
            if (cc > 1)                         // no syntax checking here ...
            {
               SINF->p      = NULL;
               SINF->disknr = 0;
               SINF->partid = 0;
               #if defined (UNIX)
                  if (toupper(c1[0]) == 'A')    // A:, translate to device
                  {
                     strcpy( s0, "/dev/fd0");   // most likely (when mounted)
                  }
                  else if (c1[0] == '/')        // full file/device name
                  {
                     strcpy( s0, c1);
                  }
                  else                          // default /dev/...
                  {
                     sprintf( s0, "/dev/%s", c1);
                  }
               #else
                  sprintf( s0, "%c:", toupper(c1[0]));
               #endif

               //- MUST be closed before opening something else, or flushing
               //- from a FS-specific cache may go to the wrong object!
               rc = DFSFNCALL(dfsa->FsClose,0,0,NULL,NULL); // Close FS

               if (dfsa->verbosity > TXAO_QUIET)
               {
                  TxPrint( "Attempt to open   : %s\n", s0);
               }
               rc = dfstOpenVolume( DFSTORE, s0, s1, TRUE);
               if (rc == NO_ERROR)
               {
                  strcpy( SINF->drive, s0);
                  strcpy(SINF->afsys, s1);
                  TxFsShow( "", s0);
                  rc = dfsInitFileSystem();
                  if ((rc == NO_ERROR) && (TxaOptSet('R')))
                  {
                     dfstSetReadOnly( DFSTORE, TRUE, TxaOption('R'));
                  }
               }
               else
               {
                  TxPrint("Error opening device or volume '%s'\n", s0);
               }
            }
            else                                // List Device/Volumes
            {
               ULONG   select = TXFSV_HD | TXFSV_CD | TXFSV_LAN;

               if (!TxaOptUnSet('f'))
               {
                  select |= TXFSV_FLOP;
               }
               if (TxaOptUnSet('l'))
               {
                  select &= ~TXFSV_LAN;
               }
               if (TxaOptUnSet('c'))
               {
                  select &= ~TXFSV_CD;
               }
               #if   defined (LINUX)
                  TxPrint( "\n'df' info plus mounted filesystems/devices in /proc/mounts:\n");
                  dfsMultiCommand( "cmd 'df -h -l -m'", 0, TRUE, FALSE, TRUE);
                  TxPrint("\n");
               #elif defined (DARWIN)
                  TxPrint( "\n'df' info for mounted filesystems/devices in / and /Volumes:\n");
                  dfsMultiCommand( "cmd 'df -h -l -m'", 0, TRUE, FALSE, TRUE);
                  TxPrint("\n");
               #else
                  TxPrint( "\nVolumes (all driveletters) for the currently active Operating System:\n\n");
               #endif
               TxFsDrivemap( "OS ", select);
            }
         }
      }
      else if (strncasecmp(c0, "sto", 3 ) == 0)    // Store selection & status
      {
         pp = &c1[0];
         switch (*pp)                           // first parameter char
         {
            case '\0': case '?':
            case 'h':  case 'H':
               TxPrint( "\nTo select a store : '%s   a|A|1  |  b|B|2   [-R | -R-]' (Read-Only)"
                        "\nAvailable stores:\n", c0);
               dfstShowStoreInfo( 1);
               break;

            case '0':
            case 's':
            case 'S':
               TxPrint( "Store S or 0 is reserved for DFSee internal use ...\n");
               dfstShowStoreInfo( DFST_SYSTEM);
               break;

            default:
               switch (*pp)
               {
                  case 'd':
                  case 'D':
                     index = 0; // Select the 'DFSee' system store (undocumented :-)
                     break;

                  case '1':
                  case 'a':
                  case 'A':
                     index = 1;                 // First user store
                     break;

                  case '2':
                  case 'b':
                  case 'B':
                     index = 2;                 // Second user store
                     break;

                  default:
                     index = (USHORT) DFSTORE;  // CURRENT store
                     break;
               }
               if (index != (USHORT) DFSTORE)   // not current one?
               {
                  rc = dfstSetDefaultStore( index);
                  if (rc == NO_ERROR)
                  {
                     if (TxaOptSet('R'))
                     {
                        dfstSetReadOnly( DFSTORE, TRUE, TxaOption('R'));
                     }
                     if (dfstQueryStoreType( index, NULL, s0) != DFST_UNUSED)
                     {
                        TxPrint("\nSwitch to store %s%s%s : "
                                "re-initialize mode (FS) ...\n",
                                CBG, (index) ? (index == 1) ? "A" : "B" : "S", CNN);
                        rc = dfsInitFileSystem();
                     }
                     TxPrint( "%s\n\n", s0);    // query result, description
                  }
               }
               else
               {
                  if (TxaOptSet('R'))
                  {
                     dfstSetReadOnly( DFSTORE, TRUE, TxaOption('R'));
                     dfstQueryStoreType( index, NULL, s0);
                     TxPrint( "%s\n\n", s0);    // query result, description
                  }
               }
               break;
         }
      }
      else if (strcasecmp(c0, "check") == 0)
      {
         if (TxaOption('?'))                    // explicit help request
         {
            TxPrint("\nCheck current filesystem for problems, using the SLT command\n");
            TxPrint("\n Usage:  %s  [-r]\n", c0);
            TxPrint("\n   -M        = Force creation of 'magic' recovery names (JFS only)\n");
            TxPrint("\n   -r        = Force rebuilt of an already present SLT\n\n");
         }
         else
         {
            dfsSelectPartVol( c1, "-O:q");      // optional selection
            strcpy( dc, "slt 0 * -m -1- -d-");
            if (TxaOption('r')) strcat( dc, " -r");
            if (TxaOption('M')) strcat( dc, " -M");
            dfsMultiCommand( dc, 0, TRUE, FALSE, TRUE);
            dfsa->explain = FALSE;              // avoid garbage explanation
            rc = dfsa->number;                  // number of errors
            if ((rc == DFS_CMD_UNKNOWN) ||
                (rc == DFS_CMD_UNKNOWN)  )
            {                                   // avoid infinite looping
               rc = DFS_BAD_STRUCTURE;          // or external execution
               rc++;
            }
         }
      }
      else if (strcasecmp(c0, "cl") == 0)       // show cluster info
      {
         if (cc > 1)
         {
            sn = dfsGetSymbolicSN( c1, 2);      // default cluster 2
            if (sn <= dfsSn2Cl(DFS_MAX_PSN))
            {
               if (cc > 2)
               {
                  sprintf( dc, "%s ", c2);
               }
               else
               {
                  strcpy(  dc, "");
               }
               sprintf( s1, "0x0%llX", dfsCl2Sn( sn));
               strcat(  dc, s1);
               TxaReParseCommand( dc);
               rc = DFS_PENDING;                // handle translated command
            }
            else
            {
               dfsX10( "Invalid cluster nr: ", sn, CBR, "\n");
            }
         }
         else
         {
            dfsX10("Current cluster nr: ", dfsSn2Cl( nav.this), CBM, "\n");
         }
      }
      else if (strcasecmp(c0, "cl2s"    ) == 0)    // show lsn for cluster value
      {
         sn = dfsGetSymbolicSN( c1, 2);         // default cluster 2
         if (sn <= dfsSn2Cl( DFS_MAX_PSN))
         {
            TxPrint( "LSN for ClusterNr : % 8llX = %8.8llX\n", sn, dfsCl2Sn( sn));
         }
         else
         {
            dfsX10( "Invalid cluster nr: ", sn, CBR, "\n");
         }
      }
      else if (strcasecmp(c0, "s2cl"    ) == 0)    // show cluster for lsn value
      {
         sn = dfsGetSymbolicSN( c1, nav.this);  // default current sector
         TxPrint( "ClusterNr for LSN : %8.8llX = % 8llX\n", sn, dfsSn2Cl( sn));
      }
      else if (strcasecmp(c0, "delshow"  ) == 0)
      {
         TxPrint("Show deleted files from list ...\n");
         strcpy( dc, "list -f ");
         if (cc > 1)
         {
            strcat( dc, c1);
            if (strchr(c1, '%') == NULL)
            {
               strcat( dc, "%100");
            }
         }
         else
         {
            strcat( dc, "*%100");
         }
         TxaReParseCommand( dc);
         rc = DFS_PENDING;                      // handle translated command
      }
      else if (strcasecmp(c0, "dirfind") == 0)  // find file/dirs in directory
      {
         if (TxaOption('?'))                    // explicit help request
         {
            TxPrint("\nFind files/directories by walking the directory tree\n");
            TxPrint("\n Usage:  %s  [levels] [path] [-t:type] [-l[:wc]\n", c0);
            TxPrint("   levels    = number of levels to include, 's' for all\n"
                    "   path      = path from root to start, default root itself\n"
                    "   -l[:wc]   : list found files, optional wildcard filter\n"
                    "   -t:type   : Type D = Directories only, f = files only\n"
                    "               Type A = Dirs+files, incl deleted (HPFS)\n");
         }
         else
         {
            dfsInitList(0, "-f -P", "-d");      // optimal for menu file-recovery
            params.Func = dfsAdd2List;          // Iterator callback function
            if (cc > 1)                         // mode, and possible path
            {
               pp += strlen(c1);
               while ((*pp == ' ') || (*pp == FS_PATH_SEP) || (*pp == FS_PALT_SEP))
               {
                  pp++;                         // skip white and leading '\' or '/'
               }
            }
            ei = dfsGetMcsNumber( c1, (ULONG) -1); // nr of levels
            rc = DFSFNCALL(dfsa->FsFindPath, 0, 0, pp, &params);
            if (rc == NO_ERROR)
            {
               strcpy( s1, TxaOptStr('t', NULL, "*"));
               TxPrint("Search for '%s' in : '%s'   please wait ...\n", s1, c2);
               rc = DFSFNCALL(dfsa->FsDirIterator, ei, s1[0], pp, &params);
               if (rc == NO_ERROR)
               {
                  TxPrint("Nr of items Found : %u\n", dfsa->snlist[0]);
                  if (dfsa->snlist[0] != 0)
                  {
                     if (TxaOption('l'))        // automatic list
                     {
                        sprintf( dc, "list -f %s", TxaOptStr('l', NULL, "*"));
                        TxaReParseCommand( dc);
                        rc = DFS_PENDING;       // execute the list command
                     }
                     else
                     {
                        TxPrint("\nYou can use 'list -f', 'recover' or other list-compatible\n"
                                  "commands for further processing of the results.\n");
                     }
                  }
               }
            }
            if (rc == DFS_PENDING)              // FindPath/Iterator not there
            {
               TxPrint( "\n%s not implemented for %s%s%s yet\n", c0, CBM, SINF->afsys, CNN);
               rc = NO_ERROR;
            }
         }
      }
      else if (strcasecmp(c0, "browse") == 0)   // generic directory BROWSE command
      {
         if (TxaOption('?'))                    // explicit help request
         {
            TxPrint("\nBrowse contents of one directory, using the Sectorlist\n");
            TxPrint("\n Usage:  %s [[%c]path-from-root | .]] [sorting] [filter]\n\n"
                      "   path-from-root = Complete DIR path starting from ROOT directory\n"
                      "                    components separated with either '\\' or '/'\n"
                      "          .       = Use current Sectorlist contents, if any\n",
                                           c0, FS_PATH_SEP);
            TxPrint("\n Options are\n"
                      "  -P              = Force prompt dialog to specify starting path\n"
                      "  -R              = Start in recovery mode, recoverability percentage\n"
                      "  -debug          = Show Inode, Parent and path+filename on Name-cache build\n"
                      "  -reset          = Reset/rebuild Name-cache with Parent and path+filename\n"
                      "  -tree           = Build Name-cache from root-directory tree (non-damaged!)\n"
                      "  -s:sortorder    = Initial sort: 3=Percentage 4=Date 5=Size 6=Path 7=ext\n"
                      "  -f:[w%%t%%m%%M%%S%%d]    = File selection filter string, containing:\n\n"
                      "      w = wildcard    : wildcard selection on full path string\n"
                      "      t = percentage  : minimum file-allocation-OK percentage\n"
                      "      m = minsize     : minimum filesize as MCS number like 1,k\n"
                      "      M = maxsize     : maximum filesize as MCS number like 999,k\n"
                      "      S = filter type : D=Directories F=Files B=Browse-mode N=Normal (all)\n"
                      "      d = modifydate  : Restrict to files modified AFTER date or #days\n\n"
                      "  Default filter : '%%0%%0%%*%%B%%*' => any percentage/size/date, BROWSE\n"
                      "  Default sort   : 6            => Path+Name, ascending\n\n");
         }
         else if ((dfsa->FsMakeBrowseList != NULL) || (c1[0] == '.')) // allow browse from list if '.'
         {
            #if defined (USEWINDOWING)

            TXLN       selectString;
            USHORT     defaultSort = (USHORT) TxaOptNum('s', "SortOrder", DFSBR_SORT_PATH);

            strcpy( selectString, TxaOptStr( 'f', "Select",  "%0%0,s%*%B%*"));
            strcpy( s1,           TxaOptStr( 'o', "Options", ""));
            if (TxaOption('8'))
            {
               strcat( s1, "~");                // Add explicit 8.3 option
            }
            if ( TxaOption('P') || ((dfsa->snlist[0] == 0) && (cc == 1)))
            {
               static BOOL promptSelect = FALSE; // persistent, and default DO NOT present the filter dialog

               strcpy( s0, (dfsa->snlist[0] == 0) ? FS_PATH_STR : ".");
               rc = dfsPromptOptDialog(
                      " Specify absolute DIRECTORY path or '.'",
                      "Specify the full path for the DIRECTORY you want to browse\n"
                      "starting from the ROOT directory for THIS filesystem\n"
                      #if !defined (UNIX)
                         "(without the leading driveletter/colon pair)\n"
                      #endif
                      "\nOR you can use '.' to browse the current Sectorlist\n",
                      5207, TXMAXLN, s0,
                      &promptSelect,           "Specify custom file filtering criteria",
                      &dfsa->browseShowHidden, "Show hidden directories and files",
                      NULL, NULL, NULL, NULL, NULL, NULL);

               if ((rc == NO_ERROR) && promptSelect)
               {
                  dfsFileListDialog( " Specify selection critera for files to be BROWSED ",
                                      *(dfsa->snlist), TRUE, selectString);
               }
            }
            else
            #endif
            if (cc > 1)                         // path specified on command
            {
               strcpy( s0, c1);                 // path, or '.'
            }
            else
            {
               strcpy( s0, ".");                // default to snlist, or ROOT
            }
            if (rc != DFS_NO_CHANGE)
            {
               rc = dfsSltPrepareNameLookup();
               if ((rc == NO_ERROR) && (strcmp( s0, ".") != 0)) // need to fill the snlist first
               {
                  pp = s0;
                  if ((s0[0] == FS_PATH_SEP) || // relative to ROOT, find
                      (s0[0] == FS_PALT_SEP ) ) // (either DOS-like or UNIX)
                  {
                     pp++;
                  }
                  TxRepl( pp, FS_PALT_SEP, FS_PATH_SEP); // fixup ALT separators

                  sprintf( dfsa->brdescript, "BROWSE: %c%s", FS_PATH_SEP, pp);

                  if ((rc = DFSFNCALL(dfsa->FsFindPath, 0, 0, pp, &params)) == NO_ERROR)
                  {
                     TRACES(("FS LSN: 0x0%llx, info: 0x%llx\n", params.Lsn, params.Number));
                     rc = DFSFNCALL(dfsa->FsMakeBrowseList, params.Lsn, params.Number, s1, NULL);
                     if (rc == NO_ERROR)
                     {
                        dfsa->brlist = params.Lsn; // can be used with fileinfo() to get path back
                        dfsa->brinfo = params.Number;
                        TxPrint( "Listed files/dirs : %10u in ", dfsa->snlist[0]);
                        if (*pp)
                        {
                           TxPrint( "%s%c%s%c%s\n", CBY, FS_PATH_SEP, pp, FS_PATH_SEP, CNN);
                        }
                        else
                        {
                           TxPrint( "%sROOT%s\n", CBY, CNN);
                        }
                     }
                  }
                  if (rc == DFS_PENDING)        // FindPath/Browse not supported
                  {
                     TxPrint( "BROWSE not implemented for 'Mode=%s' yet\n", SINF->afsys);
                     rc = NO_ERROR;             // avoid command loop, allow empty list
                  }
               }
               if (rc == NO_ERROR)              // may have aborted, or no BrowseList created
               {
                  #if defined (USEWINDOWING)
                     ULONG browseFlags = DFSBR_MOVEABLE;

                     if (TxaOption( 'R'))
                     {
                        browseFlags |= DFSBR_RECOVERY; // start in recovery mode
                     }
                     rc = dfsBrowseSectorList( selectString, browseFlags, defaultSort);
                  #else
                     //- browse the snlist here, for plain text mode
                     sprintf( dc, "list -S");   // 8.3 ~ option NOT possible with string output!
                     TxaReParseCommand( dc);
                     rc = DFS_PENDING;
                  #endif
               }
            }
         }
         else                                   // browsing not supported
         {
            TxNamedMessage( TRUE, 0, " INFO: Not implemented ", "BROWSE not implemented for 'Mode=%s' yet\n", SINF->afsys);
         }
      }
      else if (strcasecmp(c0, "undelete") == 0) // generic undelete BROWSE command
      {
         if (TxaOption('?'))                    // explicit help request
         {
            TxPrint("\nFind deleted files and Browse the resulting Sectorlist\n");
            TxPrint("\n Usage:  %s [partial-name] [sorting] [filter]\n\n"
                      "   partial-name = part of filename wanted, not a true wildcard,\n"
                      "                  but may start and end in a '*' character\n", c0);
            TxPrint("\n Options are\n"
                      "  -s:sortorder    = Initial sort: 3=Percentage 4=Date 5=Size 6=Path 7=ext\n"
                      "  -f:[w%%t%%m%%M%%S]  = File selection filter string, containing:\n\n"
                      "      w = wildcard    : wildcard selection on full path string\n"
                      "      t = percentage  : minimum file-allocation-OK percentage\n"
                      "      m = minsize     : minimum filesize as MCS number like 1,k\n"
                      "      M = maxsize     : maximum filesize as MCS number like 999,k\n"
                      "      S = filter type : D=Directories F=Files B=Browse-mode\n\n"
                      "Default filter : '%%100%%1,s%%*%%F' => non-empty 100%% files only\n"
                      "Default sort   : 4              => Date descending\n\n");
         }
         else if (dfsa->FsDirFileSaveAs != NULL)
         {
            TxPrint( "\nSearching candidates for UNDELETE, may take quite a long time!\n");
            #if defined (USEWINDOWING)
            {
               TXLN       selectString;
               USHORT     defaultSort = (USHORT) TxaOptNum('s', "SortOrder", DFSBR_SORT_DATE);

               strcpy( selectString, TxaOptStr( 'f', "Select",  "%100%1,s%*%F"));

               rc = dfsSltPrepareNameLookup();
               sprintf( dc, "delfind %s#browse . -s:%hu -f:'%s'", c1, defaultSort, selectString);
            }
            #else
               sprintf( dc, "delfind %s#say#say Use 'list -f' and 'recover' to copy the found files", c1);
            #endif

            if (rc == NO_ERROR)                 // may have aborted
            {
               rc = dfsMultiCommand( dc, 0, TRUE, FALSE, TRUE);
            }
         }
         else                                   // browsing not supported
         {
            TxNamedMessage( TRUE, 0, " INFO: Not implemented ", "UNDELETE not implemented for 'Mode=%s' yet\n", SINF->afsys);
         }
      }
      else if (strcasecmp(c0, "alloc"    ) == 0)
      {
         if (TxaOption('?'))                    // explicit help request
         {
            TxPrint("\nShow disk/filesystem data distribution, RESIZE limits\n");
            TxPrint("\n Usage:  %s  [~ | - | + | * | $ | spc | -l:x]\n\n", c0);
            TxPrint("    @      = Lowest detail      exactly 1 line\n"
                    "    ~      = Lower  detail      2 to    4 lines\n"
                    "    -      = Low detail         8 to   16 lines\n"
                    "    +      = Medium detail     32 to   64 lines\n"
                    "    *      = High detail      128 to  256 lines\n"
                    "    $      = Extreme detail,  512 to 1024 lines\n\n"
                    "    spc    = size per character as an mcs-number,\n"
                    "             in Clusters, Kib, Mib, or Gib\n\n"
                    "    -l:x   = Map with specified number of lines\n"
                    "    -c:x   = Map with specified number of characters\n"
                    "    -f     = Map for FAT-area, not bitmap            (EFAT only)\n"
                    "    -m:x   = Use bitmap 'x': 0=IPool 1=Main 2=Tier2  (APFS only)\n");
         }
         else
         {
            rc = DFSFNCALL( dfsa->FsAllocDisplay, 0, 0, c1, NULL);
            if (rc == DFS_PENDING)
            {
               TxPrint( "\n%s not implemented for %s%s%s yet\n", c0, CBM, SINF->afsys, CNN);
               rc = NO_ERROR;
            }
         }
      }
      else if (strcasecmp(c0, "type"     ) == 0)   // show system type(s)
      {
         BOOL          hidden;

         TxPrint("\n");
         index = (USHORT) TxaParseNumber( c1, DFSRADIX_TYPE, NULL);
         if (cc == 2)
         {
            hidden = dfsPartTypeDescription((BYTE) index, s1);

            TxPrint("Partition type %-3hu: 0x%2.2hx = %s%s  %s%s%s\n",
               index, index, CBC, s1, CNC, (hidden) ? "(Hidden)" : "", CNN);
         }
         else
         {
            TXLN          tline;                // line with types
            USHORT        lim;                  // limit value (last type)
            USHORT        li1;                  // limit value 1st column
            USHORT        st1;                  // step value 1st column
            USHORT        c;                    // column counter

            ml = (USHORT) TxaParseNumber( c2, DFSRADIX_TYPE, NULL);
            lim = (ml) ? (ml & 0xff) : 0xff;    // limit = clipped last value
            if (lim < index)                    // backward order given
            {
               ml    = lim;
               lim   = index;
               index = ml;
            }
            st1 = ((lim - index) / 4) +1;       // step value  col 1
            li1 = index + st1;                  // limit value col 1
            while(index < li1)
            {
               nr = 0;
               strcpy( tline, "");
               for ( c = 0,      ml  = 0;
                    (c < 4) && ((ml + index) <= lim) && !TxAbort();
                     c++,        ml += st1)
               {
                  BOOL unknown;

                  hidden  = dfsPartTypeDescription((BYTE) (index +ml), s1);
                  unknown = (strncasecmp( s1, "Unknown", 7) != 0);

                  sprintf( s0, " %2.2hx = %s%s%s%s%s ", (USHORT)(index + ml), CNC,
                     (hidden)  ? "H-" : "  ", CNN,
                     (unknown) ?  s1  : "         ",
                     (ml < 3)  ? "  " : "");
                  strcat( tline, s0);
                  nr++;
               }
               if (nr > 0)
               {
                  TxPrint("%s\n", tline);
               }
               index++;
            }
         }
      }
      else if (strcasecmp(c0, "pt") == 0)          // show partition-table for PID
      {
         if ((c1[0] != '?') && (!TxaOption('?')))
         {
            if (cc > 1)
            {
               TxaGetArgString( TXA_CUR, 1, TXA_OPT, TXMAXLN, s0); // args + options
               sprintf( dc, "fdisk pt %s", s0);
            }
            else                                // current partition, or 1
            {
               sprintf( dc, "fdisk pt %d", (SINF->partid) ? SINF->partid : 1);
            }
         }
         else
         {
            sprintf( dc, "fdisk pt -?");
         }
         TxaReParseCommand( dc);
         rc = DFS_PENDING;
      }
      #if defined (USEWINDOWING)
      else if (strncasecmp(c0, "pte", 3) == 0)     // PTEdit
      {
         if ((c1[0] != '?') && !TxaOption('?') && dfsa->dialogs)
         {
            FDSK_CB_INFO  cbi;                  // callback parameter struct
            ULONG         changes = 0;          // changes made
            DFST_HANDLE   cst = dfstGetDefaultStore();
            ULONG         listcode = TXDID_OK;
            BOOL          useGptStyle = FALSE;  // remember which style is used

            dfsPartitionableDisks();            // read diskinfo if not present

            memset( &cbi, 0, sizeof(cbi));      // initial callback info
            cbi.sn        = dfsGetSymbolicSN( c1, nav.this); // current is default
            cbi.number    = 1;                  // 1 sector
            cbi.disknr    = TxaOptNum('d', NULL, (SINF->disknr) ? SINF->disknr : 1);
            cbi.cbOption1 = !TxaOptUnSet('s');  // no smart updating ?
            cbi.cbOption2 = TxaOption('f');     // force MBR/EBR signature ?

            TRACES(( "PTE for '%s %s' cbi.sn: 0x%llx\n", c1, c2, cbi.sn));

            dfstSetDefaultStore( DFST_SYSTEM);  // select system store

            if ((isdigit( c1[0])) &&            // numeric
                ((atol(c1) == 0)  ||            // but zero
                 (atol(c1) > dfsPartitions()))) // or no valid partition-id
            {
               if (cc > 2)
               {
                  cbi.partnr = atol( c2) -1;    // entry number in table
               }
            }
            else
            {
               TXRECT where = {18,0,0,0};       // fixed position

               if (TxaOption('P'))              // explicit prompting
               {
                  listcode = txwListBox( TXHWND_DESKTOP, TXHWND_DESKTOP, &where,
                           " Select partition to start PTE with ", "",
                             5401, TXLB_MOVEABLE,
                             cMenuTextStand, cMenuBorder_top, // same color as menu
                             dfsa->slPartOne);

                  strcpy( s1, "*");
                  if (listcode > TXDID_MAX)     // return list selection
                  {                             // plus current item base value
                     sprintf( s1, "%u", listcode - TXDID_MAX); // calculate pid
                  }
               }
               else
               {
                  strcpy( s1, c1);
               }

               //- select partition by number or driveletter, quiet & fast (no alloc)
               if (dfsSelectPartVol( s1, "-O:q -a- -Q -B"))
               {
                  if (SINF->p == NULL)          // no partition selected
                  {                             // refresh values after select
                     cbi.sn     = dfsGetSymbolicSN( c1, nav.this);
                     cbi.disknr = 0;
                  }
               }
               if (SINF->p != NULL)             // partition selected
               {
                  if (SINF->p->tablenr == GPT_STYLE)
                  {
                     useGptStyle = TRUE;
                     cbi.thisSn  = SINF->p->partPsn; // Table sectornumber
                     cbi.sn      = SINF->p->basePsn; // Start of partition
                  }
                  else
                  {
                     cbi.sn      = SINF->p->partPsn; // Table sn, for ExecOnSectors
                  }
                  cbi.disknr = SINF->p->disknr;
                  cbi.partnr = SINF->p->partnr; // entry number in table
               }
            }
            if (listcode != TXDID_CANCEL)
            {
               cbi.cbOption4 = TRUE;            // Initialize FOCUS field to default one
               dfsSelectDisk( cbi.disknr, FALSE, FALSE); // quietly select disk (not partition)
               do
               {
                  dfsGetPartDesc( dfsGetPartInfo( dfsDiskPsn2PartId( cbi.disknr, cbi.sn)), s0);

                  if (useGptStyle)
                  {
                     TxPrint("PTEdit GPT entry nr. %03hu on  %s\n", cbi.partnr, dfstStoreDesc2( DFSTORE));
                     TxPrint("PTEdit GPT at 0x0%llx for %s\n", cbi.sn, s0);
                     rc = dfsFdskPtEditGpt( &cbi);
                     if ((rc == NO_ERROR) || (rc == DFS_NO_CHANGE))
                     {
                        if (rc != DFS_NO_CHANGE) // a change has been made
                        {
                           changes++;           // remember for refresh
                        }
                        switch (cbi.cbOptNum1)
                        {
                           case DFSDLG_NEXT:    // edit next used table entry
                              dfsGptNextEntry( &cbi);
                              break;

                           case DFSDLG_PREV:    // edit previous used table entry
                              dfsGptPrevEntry( &cbi);
                              break;

                           case DFSDLG_EDIT:    // slot-number is in partnr
                              index = (cbi.partnr % (dfsGetSectorSize() / sizeof( S_GPTENTRY)));
                              sn    = (cbi.partnr / (dfsGetSectorSize() / sizeof( S_GPTENTRY))) + cbi.thisSn;

                              dfsHexObjectEditor( sn, 0, // start sector and itemsize in sectors
                                                      index * sizeof( S_GPTENTRY), sizeof( S_GPTENTRY), // mark
                                 NULL, DFSH_HEX_VIEW, index * sizeof( S_GPTENTRY), 0, 0); // cursor position

                              //- Read from disk, to have changes show up in the GPT PT-editor
                              dfsReadDiskInfo( FDSK_QUIET); // re-read all partition info
                              break;

                           case DFSDLG_REQ5:    // First sector of partition
                           case DFSDLG_REQ6:
                           case DFSDLG_REQ7:    // Last  sector of partition
                           case DFSDLG_REQ8:
                              dfsHexObjectEditor( cbi.cbOptNum2, 0, 0, 0, NULL, DFSH_AUTOVIEW, 0, 0, 0);
                              break;

                           default:
                              break;
                        }
                     }
                     else
                     {
                        break;                  // out of loop, keep RC
                     }
                  }
                  else                          // MBR style partition
                  {
                     TxPrint("PTEdit %sBR table slot# %hu on  %s\n",
                        (cbi.sn == 0) ? "M" : "E", cbi.partnr, dfstStoreDesc2( DFSTORE));
                     if (cbi.sn == 0)           // MBR sector, primaries
                     {
                        strcpy( s0, "primary partitions and extended container.");
                     }
                     TxPrint("PTEdit %sBR at 0x0%llx for %s\n", (cbi.sn == 0) ? "M" : "E", cbi.sn, s0);
                     rc = dfsExecOnSectors( dfsFdskPtEditMbr, &cbi);
                     if ((rc == NO_ERROR) || (rc == DFS_NO_CHANGE))
                     {
                        if (rc != DFS_NO_CHANGE) // a change has been made
                        {
                           changes++;           // remember for refresh
                        }
                        switch (cbi.cbOptNum1)
                        {
                           case DFSDLG_NEXT:    // edit next part-table
                              cbi.sn = cbi.cbOptNum2; // passed from dialog ...
                              cbi.cbOptNum2 = 0; // focus on first entry
                              break;

                           case DFSDLG_PREV:    // edit previous part-table
                              cbi.sn = dfsPreceedingTable( cbi.sn);
                              cbi.cbOptNum2 = 0;
                              break;

                           case DFSDLG_EDIT:    // slot-number is in partnr
                              dfsHexObjectEditor( cbi.cbOptNum2, 1, 0x1be, 64, NULL, DFSH_HEX_VIEW,
                                                  cbi.partnr * 16  + 0x1c2, 0, 0);
                              break;

                           case DFSDLG_REQ5:    // First sector of partition
                           case DFSDLG_REQ6:
                           case DFSDLG_REQ7:    // Last  sector of partition
                           case DFSDLG_REQ8:
                              dfsHexObjectEditor( cbi.cbOptNum2, 0, 0, 0, NULL, DFSH_HEX_VIEW, 0, 0, 0);
                              break;

                           default:
                              break;
                        }
                     }
                     else
                     {
                        break;                  // out of loop, keep RC
                     }
                  }
               } while (cbi.cbOptNum1 != DFSDLG_NONE);

               TxPrint("\nPTeditor  changes : %u (MBR/EBR secors, or PTA entries)\n", changes);
               if (changes != 0)
               {
                  dfsReadDiskInfo( FDSK_QUIET); // re-read all partition info
               }
            }
            dfstRestoreDefaultStore( cst);      // reselect current store
            TxCancelAbort();                    // might have aborted ...
         }
         else
         {
            if (!dfsa->dialogs)
            {
               TxPrint( "\nThe %s command is supported only in the "
                        "windowed environment of DFSee\n"
                        "when batch-mode is not activated.", c0);
            }
            else
            {
               TxShowTxt( cmd_pte);             // give usage
            }
         }
      }
      #endif                                    // USEWINDOWING

      #if defined (HAVE_CMD_GRAB)
      else if (strncasecmp(c0, "svexpand", 8) == 0) // Expand SVISTA image (RAW)
      {
         if ((cc > 2) && (c1[0] != '?') && !TxaOption('?'))
         {
            sprintf( dc, "im -fs- -e:'hdd' %s", c1); // open, no FS-init .HDD
            rc = dfsMultiCommand( dc, 0, TRUE, FALSE, TRUE);
            if (rc == NO_ERROR)
            {
               FDSK_NUMVAL size     = FDSK_NUMZERO; // size param info
               ULONG       geoHeads = dfstGeoHeads(       DFSTORE);
               ULONG       geoSects = dfstGeoSectors(     DFSTORE);
               ULN64       oldSize  = dfstGetLogicalSize( DFSTORE);
               ULN64       newSize;
               ULONG       altHeads = dfsGetMcsNumber( c3, 0);
               ULONG       altSects = dfsGetMcsNumber( c4, 0);
               DFSGEODISK *dskgeo   = dfsGetDiskGeoInfo( SINF->disknr);

               if (dskgeo)
               {
                  strcpy( dskgeo->Reason, "SVISTA expand");
               }

               if (altHeads != 0)
               {
                  geoHeads = altHeads;
               }
               if (altSects != 0)
               {
                  geoSects = altSects;
               }
               dfsParseCylinderSize( c2, geoHeads, geoSects, SECTORSIZE, &size);
               newSize = (ULN64) size.value * geoHeads * geoSects;
               if (newSize > oldSize)
               {
                  rc = dfstDiskGeometry( DFSTORE, 0, size.value, altHeads, altSects, 0, FALSE, TRUE);
                  dfsSz64( "\nExpanding image to: ", newSize, "  Please wait ...\n\n");

                  if (TxaOption('t') || TxaOption(TXA_O_TEST))
                  {
                     TxPrint( "Just testing!\n");
                     TxSleep( 2000);
                  }
                  else
                  {
                     ml = 16;
                     memset( ms, 0, ml);        // default pattern, zeroes
                     if (TxaOptUnSet('z'))      // -z- just wipe last sector
                     {
                        rc = dfsWipeArea( newSize - 1, 1, DFSP_NONE, NULL, FALSE, ms, ml); // no progress
                     }
                     else                       // ZERO whole new area
                     {
                        rc = dfsWipeArea( oldSize, newSize - oldSize, DFSP_BARS, "Expanding SVISTA", FALSE, ms, ml); // show progress
                     }
                  }
                  TxPrint( "\nFinished.\n");
               }
               else
               {
                  rc = DFS_VALUE_ERROR;         // new size too small
               }
            }
         }
         else
         {
            TxPrint("\nExpand SVISTA standard (RAW) imagefile to a larger size\n\n"
                      " Usage: %s  filename  newSize  [geoHead  [geoSect]]\n\n"
                      "  newSize = desired size in MiB, GiB or Cylinders (mcs-number)\n"
                      "            default is MiB; use n,g for GiB, n,c for cylinders\n\n"
                      "  geoHead = nr of heads   to use for geometry, default 255\n"
                      "  geoSect = nr of sectors to use for geometry, default 63\n\n"
                      "  -t      : test modus, no real expand of the imagefile\n"
                      "  -z-     : write last-sector only (leaves garbage in DOS!)\n", c0);
            TxPrint("\n Use the '-q -G' switches to suppress normal output, and enable\n"
                      " error messages and a percentage progress on STDOUT, to drive\n"
                      " a GUI frontend progress indicator.  Example invocation:\n\n"
                      " %s -q -G %s x:%svmfile.hdd 3,g\n", TxaExeArgv(0), c0, FS_PATH_STR);
         }
      }
      else if ((strncasecmp(c0, "svgrab" , 6) == 0) || // grab partitions to image
               (strncasecmp(c0, "imgrab" , 6) == 0)  ) // grab partitions to image
      {
         rc = dfsvGrabPartitions();
      }
      #endif

      else if (strncasecmp(c0, "move", 4)   == 0)  // move towards begin or end
      {
         rc = dfsMovePartition((cc > 1) ? c1 : NULL);
      }
      else if (strncasecmp(c0, "resize", 6) == 0)  // resize, making part smaller
      {
         rc = dfsResizePartition((cc > 1) ? c1 : NULL);
      }
      else if (strncasecmp(c0, "expand", 6) == 0)  // resize, making part larger
      {
         rc = dfsExpandPartition((cc > 1) ? c1 : NULL);
      }
      else if ((strcasecmp(c0, "wipefree"    ) == 0) ||
               (strcasecmp(c0, "zerofree"    ) == 0)  )
      {
         if ((c1[0] != '?') && !TxaOption('?'))
         {
            if (dfsSelectPartVol( c1, "-O:q"))  // optional select quiet
            {
               sprintf( dc, "wipe %s -f %s", TxaOption('t') ? "-t" : "", c2);
               TxaReParseCommand( dc);
               rc = DFS_PENDING;
            }
            else if ((dfsa->FsModeId != DFS_FS_FDISK) && (dfsa->FsLsnAllocated))
            {
               //- current object seems to be a filesystem with alloc info
               strcpy( dc, "wipe -f");
               TxaReParseCommand( dc);
               rc = DFS_PENDING;
            }
            else
            {
               TxPrint( "\nThere is no partition, volume or other filesystem object selected ...\n");
            }
         }
         else
         {
            TxPrint("\nWipe all freespace inside the filesystem to ZERO\n\n"
                      "  Usage: %s  [options]  [part | vol]\n\n"
                      "      part  = partition, by number or letter\n"
                      "      vol   = volume, driveletter\n\n"
                      "   -i:img   = image filename\n\n"
                      "   -p:part  = partition, number or letter\n"
                      "   -O:v     = verbose, extra progress info\n"
                      "   -O:q     = quiet,   no progress info\n"
                      "   -test    = no wipe, just show\n"
                      "   -v:vol   = volume, driveletter\n\n"
                      "  use 'wipe -f' directly for more options\n\n", c0);
         }
      }
      else if ((strcasecmp( c0, "fs"    ) == 0) || // check on "isFdskCmd" must
               (strcasecmp( c0, "fdisk" ) == 0) || // be done last to avoid a
               (dfsFdskIsFdskCmd( c0, FALSE)) ) // switch for "subword" cmds
      {                                         // like "cl" <> "cleanup"
         //- MUST be closed before opening something else, or flushing
         //- from a FS-specific cache may go to the wrong object!
         rc = DFSFNCALL( dfsa->FsClose,0,0,NULL,NULL); // Close current FS

         strcpy( s1, c1);                       // remember FS-name for 'fs'
         if ((strcasecmp( c0, "fs") != 0) ||
             (cc == 1))
         {
            strcpy( s0, SINF->afsys);           // remember for messages
            strcpy( s1, "FDISK");               // default to fdisk FS
            if (SINF->disknr == 0)              // no disk open
            {
               dfsSelectDisk( 1, FALSE, FALSE); // quietly select disk 1
            }
         }
         strcpy( SINF->afsys, TxStrToUpper(s1));
         rc = dfsInitFileSystem();
         if (rc == NO_ERROR)
         {
            if (strcasecmp( c0, "fs") != 0)
            {
               if (SINF->partid != 0)
               {
                  dfsa->autoPid = SINF->partid; // remember as 'last' ...
                  SINF->partid  = 0;
               }
               strcpy( SINF->drive, "--");
            }
            strcpy( SINF->afsys, TxStrToUpper(s1)); // possibly destroyed by Init

            TRACES(("afsys just before execute: '%s'\n", SINF->afsys));
            if (dfsFdskIsFdskCmd( c0, FALSE))   // execute, stay in FDISK mode
            {
               if (strcasecmp( s0, "FDISK") != 0)  // avoid FDISK->FDISK message
               {
                  TxNamedMessage( ((!dfsa->batch) && (!TxaOptUnSet('P'))), 5015,
                     " NOTE: Switching to FDISK mode ",
                     "The '%s' command switches to FDISK mode, unselecting the "
                     "partition, volume or image with %s specific "
                     "commands.\n\n", c0, s0);
               }
               rc = DFSFNCALL(dfsa->FsCommand, 0, 0, NULL, NULL);
            }
            else if (strcasecmp(c0, "fdisk") == 0)
            {
               if (cc > 1)                      // there are args ...
               {                                // log FDISK cmd for auto-log
                  TxaGetArgString( TXA_CUR, 1, TXA_OPT, TXMAXLN, dc); // args + options
                  TxPrint("DFS FDISK command : %s\n", dc);
                  TxaReParseCommand( dc);
                  rc = DFSFNCALL(dfsa->FsCommand, 0, 0, NULL, NULL);
                  if ((rc == DFS_PENDING) ||    // cmd modified, pending
                      (rc == DFS_CMD_UNKNOWN))  // cmd unknown, not handled
                  {                             // only execute an autoshow
                     dfsReadDiskInfo(   FDSK_ANY);
                     dfsShowPartitions( FDSK_ANY);
                     rc = NO_ERROR;
                  }
               }
               else                             // starting interactive FDISK
               {
                  if ( TxaOption('r') ||        // -r:1 or -r+ option used
                      !TxaOptSet('r'))          // or not set at all
                  {                             // (otherwise '-r-')
                     dfsReadDiskInfo( FDSK_ANY);
                  }
                  dfsShowPartitions(  FDSK_ANY);
               }
            }
         }
      }
      else if ((strcasecmp(c0, "down") == 0) && // show DOWN sector
               (strcasecmp(c0, "next") == 0)  )
      {
         sprintf( dc, "d %s", c1);
         TxaReParseCommand( dc);
         rc = DFS_PENDING;
      }
      else if (strcasecmp(c0, "this") == 0)     // show THIS sector
      {
         sprintf( dc, "t %s", c1);
         TxaReParseCommand( dc);
         rc = DFS_PENDING;
      }
      else if (strcasecmp(c0, "up") == 0)       // show UP sector
      {
         sprintf( dc, "u %s", c1);
         TxaReParseCommand( dc);
         rc = DFS_PENDING;
      }
      else if (strcasecmp(c0, "xtra") == 0)     // show XTRA sector
      {
         sprintf( dc, "x %s", c1);
         TxaReParseCommand( dc);
         rc = DFS_PENDING;
      }
      else
      {
         rc = DFS_CMD_UNKNOWN;
      }
   }
   else                                         // single char, or .NNN +xxx etc
   {
      TRACES(("dc:'%s' c0:'%s' sn=0x%llx sectors=0x%llx\n", dc, c0, sn, sectors));
      switch (c0[0])
      {
         case '?':
            TxShowTxt( dfsGenericHelp);
            TxPrint(  " %s: %s %s\n", DFS_N, DFS_V, DFS_C);
            break;

         case 'q':
         case 'Q':
            dfsa->explain = FALSE;              // no explain of this RC
            rc = DFS_QUIT;
            break;

         case 't':
            sl = nav.down;
            sn = nav.this;
            st = c1[0];
            dfsa->defDispSize = (isupper(st)) ? dfsGetSectorSize() : SECTORSIZE/2;
            rc = dfsReadAnDisplay( nav.this, nav.this_sninfo, &st);
            nav.xtra = nav.down;
            nav.down = sl;                      // restore original 'next'
            break;

         case '\\':                             // display root-directory
            if ((rc = DFSFNCALL(dfsa->FsFindPath, 1, 0, "", &params)) == NO_ERROR)
            {
               rc = dfsReadAnDisplay( params.Lsn, params.Number, &st);
            }
            else if (rc == DFS_PENDING)
            {
               TxPrint( "\n%s not implemented for %s%s%s yet\n", c0, CBM, SINF->afsys, CNN);
               rc = NO_ERROR;
            }
            break;

      #if defined (USEWINDOWING)
         case '/':                              // search dialog
            txwSbViewFindRequest( TXWS_REQ_FINDDIALOG, "");
            break;
      #endif

         case '+': nav.this += 1; break;        // special case, to mean forward/backward
         case '-': nav.this -= 1; break;        // by ONE sector, same as '+1' resp '-1'

         case 'H':
         case 'h':
         case 'A':
         case 'a':
            if ((cc > 1) || (TxaOption('?')))                         // sn param mandatory
            {
               if (TxaOption('?'))
               {
                  TxPrint("\nDisplay sector(s) as Hexadecimal dump or Ascii text\n\n");
                  TxPrint("\n   Usage:  %s  lsn | . [size [S]]\n", c0);
                  TxPrint("   lsn        = LSN of sector to dump, default is current\n"
                          "    .         = Placeholder for current sector LSN 'this'\n\n"
                          "   size       = size to dump:  1..63   specifies sectors\n"
                          "                              64..xxx  specifies bytes\n"
                          "                Default is 512 bytes for the 'H' command\n"
                          "                       and 256 bytes for the 'h' command\n"
                          "\n"
                          "   S          = Force size parameter to be sectors, not bytes\n");
                  if (toupper( c0[0]) == 'H')
                  {
                     TxPrint("\n"
                             "   -4         = Output 32-bit intel-reversed HEX values, in C/C++\n"
                             "                syntax like: '0x12345678, 0x87654321, // xV4.!Ce.\n");
                  }
               }
               else
               {
                  sn = dfsGetSymbolicSN( c1, L64_NULL);
                  if ((nr = dfsGetSymbolicSize( c2, 0, DFSTORE, sn)) != 0)
                  {
                     if ((nr <= DFS_MAX_DISPSECTOR) || (toupper(c3[0]) == 'S'))
                     {
                        nr *= dfsGetSectorSize(); // translate sectors to bytes ...
                     }
                  }
                  else
                  {
                     dfsa->defDispSize = (isupper(c0[0])) ? dfsGetSectorSize() : SECTORSIZE/2;
                     nr = dfsa->defDispSize;
                  }
                  switch (c0[0])
                  {
                     case 'h':
                     case 'H':
                        dfsSectorAsHex(  sn, nr, TxaOption( '4'));
                        break;

                     default:
                        dfsSectorAsAscii(sn, nr);
                        break;
                  }
                  nav.this = sn;
               }
               break;                           // handled, in other cases use
            }                                   // fall through (HEX number)
         default:                               // using std ReadAnDisplay()
            TRACES(("single char command: '%s' = 0x%2.2hx\n", c0, (USHORT) c0[0]));
            switch (c0[0])
            {
               case '.':
                  if (sscanf( c0 +1, "%u", &ei) == 1)
                  {
                     if (ei < dfsa->snlist[0])
                     {
                        sn     = dfsa->snlist[ei+1];  //- LSN itself
                        sninfo = dfsa->sninfo[ei+1];  //- Additional info, optional
                        nav.this_sninfo = sninfo;     //- remember for saveto ...
                     }
                     else
                     {
                        sn   = L64_NULL;
                        TxPrint( "\nSector-list index : %s is invalid, the ", c0);
                        if (dfsa->snlist[0])
                        {
                           TxPrint( "last one is .%u\n", dfsa->snlist[0] -1);
                        }
                        else
                        {
                           TxPrint( "Sectorlist is empty!\n");
                        }
                        rc = DFS_VALUE_ERROR;
                     }
                  }
                  else
                  {
                     TxPrint( "\nNo Sector-list index specified\n");
                     sn = L64_NULL;
                     rc = DFS_VALUE_ERROR;
                  }
                  break;

               case ' ':
               case '\0':
                  sn = nav.down;                      //- next one on ENTER
                  sninfo = nav.down_sninfo;           //- with AUTO sninfo
                  nav.this_sninfo = sninfo;           //- move info to THIS
                  nav.down_sninfo = 0;                //- reset down sninfo
                  dfsa->defDispSize = SECTORSIZE/2;   //- Just half HEX sect
                  break;


               case 'u':  sn = nav.up;    st = c1[0];           break;
               case 'x':  sn = nav.xtra;  st = c1[0];           break;
               case 'd':                        // down
               case 'n':                        // next
                  if (c0[1] == 0)               // single (hex) digit
                  {
                     sn = nav.down;  st = c1[0];
                     sninfo = nav.down_sninfo;  // with AUTO sninfo
                     nav.this_sninfo = sninfo;  // move info to THIS
                     nav.down_sninfo = 0;       // reset down sninfo
                     break;                     // handled, otherwise use
                  }                             // fallthrough (HEX / MCS)
               default:
                  if ((strlen(c0) > 1) ||       // numbers like -10, +3a6 etc
                      (isxdigit(c0[0]) ))       // or a hexadecimal number
                  {
                     sn = dfsGetSymbolicSN( c0, L64_NULL);
                  }
                  else                          // single digit, non-hex
                  {
                     sn = L64_NULL;
                     rc = DFS_CMD_UNKNOWN;      // will result in external!
                  }
                  break;
            }
            if (TxaOptSet('e'))                 // e option specified ?
            {
               sninfo = DFSSNIGET(TxaOptNum('e', "Entry", nav.this_sninfo)) | DFSSNINFO;
            }
            TRACES(("dc:'%s' c0:'%s' sn=0x%llx sninfo=%4.4hx sectors=0x0%llx\n", dc, c0, sn, sninfo, sectors));
            if (sn != L64_NULL)
            {
               rc = dfsReadAnDisplay( sn, sninfo, &st);
            }
            else if (rc != DFS_VALUE_ERROR)
            {
               rc = DFS_CMD_UNKNOWN;            // will result in external!
            }
            break;
      }
   }
   RETURN (rc);
}                                               // end 'dfsGenericCommand'
/*---------------------------------------------------------------------------*/

