//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// FDISK dump & display Analysis functions
//
// Author: J. van Wijk
//
// JvW  28-02-2021 Added OS2 ArcaOS Type 1 partition to CR help on GPT symbolic types
// JvW  26-11-2002 Added dfsFdskIsFdskCmd() to check for valid commands (modeless)
// JvW  21-11-2001 Used new command parser for all commands and options
// JvW  22-02-2001 Made "DELETE" cmd parsing better avoiding DELFIND confusion
// JvW  29-06-2000 Split-off dfscfdsk with callback and command implementations
// JvW  21-04-2000 Added drive-letter change (generic) in DFS part info
// JvW  17-04-2000 Added NT registry diskkey generation and display
// JvW  17-02-2000 Removed "multiple primary" warning on set visible on LOGICAL
// JvW  26-01-2000 Added PCLEAR cmd to clear Parttables, LVM and/or boot-recs
// JvW  16-12-99   Added FAT2OS with OS2, NT, IBMDOS and MSDOS targets
// JvW  21-02-99   Initial version, cloned from DFSANTFS
//

#include <txlib.h>                              // TX library interface

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFSee navigation and defs
#include <dfsmdisk.h>                           // Memory disk interface
#include <dfsimage.h>                           // CMD image create restore
#include <dfsutil.h>                            // DFSee utility functions
#include <dfsspace.h>                           // DFS file-space interface
#include <dfshexed.h>                           // DFS  HEX editor interface
#include <dfstable.h>                           // SLT utility functions
#include <dfsupart.h>                           // FDISK partition functions
#include <dfsafdsk.h>                           // FDISK display & analysis
#include <dfscfdsk.h>                           // FDISK callback and commands
#include <dfscflvm.h>                           // FDISK LVM callback and commands
#include <dfsdembr.h>                           // FDISK PTE MBR dialog functions
#include <dfsdfmbr.h>                           // FDISK MBR dialog functions
#include <dfsdfgpt.h>                           // FDISK GPT dialog functions
#include <dfsufdsk.h>                           // FDISK utility functions
#include <dfsufgpt.h>                           // FDISK GPT utility functions
#include <dfscfgpt.h>                           // FDISK GPT command functions
#include <dfslfdsk.h>                           // FDISK SLT functions
#include <dfsarea.h>                            // FDISK Area/Alloc functions
#include <dfsbfat.h>                            // FAT bootsector functions
#include <dfsetbt.h>                            // DFSee setboot function
#include <dfswin.h>                             // windowing functions
#include <dfsver.h>                             // version and naming info
#include <dfseoem.h>                            // eCS OEM functions

#if defined (WIN32)
   #include <dfsntreg.h>                        // DFSee NT registry functions
#endif

#define DFSFDSK_AUTOLOG "dfsfdisk.log"          // FDISK auto-log filename

char dfsFdskSectorTypes[] =                     // ONLY recognized in FDISK mode!
{
   0                                            //     string terminating ZERO
};


static DFSAFDSK    fdsk_anchor   =
{
   NULL,
   FALSE,                                       // autolog indicator
   FALSE,                                       // change indicator for show
   FALSE,                                       // executing AutoShow
   ""                                           // default cmd after changes
};

       DFSAFDSK   *fdsk = &fdsk_anchor;

//- commands that should cause an automatic mode-change to FDISK mode
static char *fdsk_commands =
   " ATTACH BMP BMFIX CLEANUP CREATE DELETE DETACH DFSIBMGR DISK FAT2OS FIXCHS"
   " DISKLIST FIXEXT GENPART LVMSHOW LVMSET LVMREDO NEWMBR NTSIGN"
   " PLIST PCLEAR PSAVE PRESTORE SETACCESS SETBOOT SETLET SETNAME"
   " SETTYPE STARTABLE VCU VIRTUAL VREMOVE W2KBM WALK";

//- commands that are a substring of one of the above, but could be external
static char *fdsk_invalids =
   " DEL SET START ";

//- commands that WILL cause switching to FDISK mode, but not autoquit
static char *fdsk_no_quits =
   " VIRT VREMOVE MOUNT UNMOUNT DISK ";


static  char       *fdsk_txt[] =
{
   "",
   "Active filesystem : FDISK / IBM-BMGR,  specific commands are:",
   "",
   " APMAP                      [-v] = Display Apple-Part-MAP in first disk track",
   " ATTACH [opt] Fname | size [H S] = Attach a partitionable medium (virtual / image)",
   " BMFIX     [disk               ] = Fix IBM BMGR bootsector W2K & CHS issues",
   " BMP       [disk  | -d:disknr  ] = Show BMGR primary-names sector for a disk",
   " CLEANUP   [disk|.|*           ] = Cleanup partitiontables, fix several errors",
   " CR        [parameters][options] = Create a new partition, use 'CR' for usage",
   " DELETE     pid [disk   [clean]] = Delete one or more partitions, clean=zero",
   " DETACH     disk | * | -a        = Detach removable medium (virtual / image)",
   " DISK      [disk]           [-r] = Select disk, show MBR; <Enter> = next EBR",
   " DISKLIST                        = Show compact list if all disks, including size",
   " DISKINFO   disk|.|*             = Show disk name/serial info for disk(s), if any",
   " DFSIBMGR   disk [imagefilename] = Create new IBM BootManager image for 'create'",
   " FAT2OS     pid [disk [boot-OS]] = Set FAT-BR bootcode to OS2|NT|IBMDOS|MSDOS",
   " FIXCHS     disk|.|*             = Fix CHS values to match LBA/current geometry",
   " FIXEXT     disk|.|* [to [from]] = Fix ext-partition types to standard value",
   " GENPART    disk|.|* [fn] [opts] = Generate DFSee partitioning script for disk(s)",
   " GPT      [ disk|.|* ]    [-fix] = Display or fixup GPT partitioning informtion",
   " LVM       [pid]       [options] = Update name, letter and BM-menu for partition",
   " LVMCLEAN  [disk|.|*]       [-a] = Clean obsolete LVM DLAT sectors in MBR track",
   " LVMREDO    disk|.|*        [-C] = Refresh LVM info sectors (DLAT) for disk(s)",
   " LVMSHOW   [pid |  *]            = Show related LVM-info for 1 or all partitions",
   " NEWMBR    [disk|.|*]   [-clean] = Refresh MBR bootcode, [-clean] = delete all",
   " NTSIGN    [disk    [signature]] = Set NT-signature value (32 bits) in MBR",
   " PLIST      p|f|e|m|b|w|l [d[r]] = List: part|free|ebr|mbr|boot|walk|lvm [opt]",
   " PCLEAR     disk [P][Lvm[S]][Br] = Clear info in P-tables, LVM and/or Boot-rec",
   " PSAVE      disk|.|*  Fn [descr] = Save Partition-table and LVM info in a file",
   " PRESTORE   disk|.|*  Fn [types] = Write info from file into P-tables and LVM",
   " PT        [pid|drive|*]         = Display partitioning sectors for partition(s)",
   " SETACCESS  pid [hide|vis|multi] = Make partition 'pid' accessible or hidden",
   " SETBOOT           opt[:options] = Bootmanager setup and reboot => 'setboot ?'",
   " SETLET     pid  letter | - [ns] = Set driveletter for DFSee; [ns] = no sync (NT)",
   " SETNAME    pid      [BMGR-name] = Set or reset (non-LVM) BMGR name for partition",
   " SETTYPE    pid  disk new  [old] = Change partition-type from old to new value",
   " STARTABLE  pid  [multi | clear] = Make partition 'pid' startable ([not]ACTIVE)",
   " TYPE        [type | first last] = Show system type description (for 00..ff)",
   " UUIDGEN         [-i:pid | disk] = Create and display UUID (GUID) value for ID",
   " VCU        disk | . | * -d[:nr] = Clear existing LVM-info, create new defaults",
   " VSTATUS                         = Show R/W and usage status of virtual disks",
   " WALK      [disk | .]       [-r] = Select disk, show the MBR, walk EBR chain",
   "",
   NULL
};

static  char       *cmd_create[] =
{
   "",
   " CR    [positional-params] [options] = Create a new partition",
   "",
   "    pos. params  : pri|log|gpt  [type  [size  [loc  [pos  [BMGR-name]]]]]]",
   "",
   "   -a:abs-pos    : Position from start of disk (mcs-number like 1,c or 0x3e,s)",
   "   -A:pid|letter : create AS another partition; using the same type and size",
   "   -align-       : Allow UNALIGNED partition start, and size. No alignment",
   "   -attrib:'hex' : GPT style partition flags, in a 16 hex-digit STRING",
   "   -b:BMGR-name  : partition name for (IBM, non-LVM) Bootmanager menu",
   "   -c:[0|1|2|?]  : CHS dummy style: 1=PQmagic, 2=MS, default 0=IBM/DFSee",
   "   -C (any type) : Clear bootsector with 0xF6 pattern  (FDISK behaviour)",
   "   -C (guard EE) : On GPT 0xEE create: Clear existing PTA (no recovery!)",
   "   -d:disk-nr    : location, use first fitting area on specified Disk",
   "   -e:end-pos    : Relative pos from end of freespace (mcs-number like 50,c)",
   "   -F            : Flag as bootable in partition-table status    (active)",
   "   -f:id         : location, as Freespace id     (1st column from 'part')",
   "   -G:n          : GAP value between table and bootsect, 1 .. tracksize",
   "   -guid:'36-ch' : Explicit partition GUID value for a new GPT partition",
   "   -i:i13-pos    : Relative pos before Int-13 limit   (mcs-number like 10,c)",
   "   -I[:image]    : Imagefile to be written to new partition",
   "   -L            : automatic 'lvm -V' on new partition",
   "   -L:'lvm opts' : automatic 'lvm' using specified options",
   "   -M            : Allow multiple visible primaries (no automatic HIDE)",
   "   -N            : When an MBR is created, leave code-area/NT-sig empty",
   "   -name:Pname   : Partition name string for a GPT partition    (max 36)",
   "   -o            : open the new partition after create",
   "   -p, -l or -g  : Make Primary, Logical or GPT part/guard.  (mandatory)",
   "   -r:rel-pos    : Relative pos from start freespace  (mcs-number like 99,c)",
   "   -s:size       : Size, will be rounded UP to next cylinder boundary",
   "   -S:0|1|2|3    : prefered table entry for the partition, default 0",
   "   -t:number /sym: MBR partition Type, numeric HEX, decimal or a symbol",
   "   -t:guidstr/sym: GPT partition Type, GUID-str without {}, or a symbol",
   "   -x:e13-pos    : Relative pos after Int-13 limit    (mcs-number like 12,c)",
   "   -X:0|1|2|3    : prefered MBR entry extended container,  default 3 (last)",
   "   -Y:0|1|2|3    : prefered EBR entry extended containers, default 1 (2nd)",
   "",
   " mcs-number: [0x]nnnn[,g|m|k|c|t|s] = HEX/decimal, GiB, MiB, Cyls, Sectors",
   "",
   NULL
};

static  char       *mbr_symbol[] =
{
   "",
   " Some symbolic names + HEX value for MBR-style partitiong:",
   " FAT   = 01/04/06/0b/0c automatic FAT    BMGR = 0a  OS/2 Bootmanager",
   " FAT32 = 0b/0c 32-bit automatic FAT32    JFS  = 35  Journaled FS (LVM)",
   " HPFS  = 07  OS/2 native filesystem      SWAP = 82  Linux SWAP partition",
   " NTFS  = 07  Windows-NT filesystem       EXT2 = 83  Linux EXT2 filesystem",
   " GUARD = ee  GPT guard partition defined in the MBR for a GPT disk",
   "",
   NULL
};

static  char       *gpt_symbol[] =
{
   "",
   " Some symbolic names + HEX value for GPT-style partitiong:",
   " EFI                             = ef / ef00   UEFI System partition",
   " BIOS                            = bb / bb00   BIOS Boot partition",
   " WIN, FAT, FAT32, EFAT, NTFS     = 07 / 0700   Windows Basic Data",
   " REC                             = 27 / 2700   Windows Recovery",
   " SWAP                            = 82 / 8200   Linux Swap",
   " LIN, LINUX, EXT2, EXT3, EXT4    = 83 / 8300   Linux Data",
   " HOME                            =      8302   Linux Home",
   " ROOT32                          =      8303   Linux Root x86-32",
   " ROOT64                          =      8304   Linux Root x86-64",
   " SRV                             =      8306   Linux Server Data",
   " DMCR                            =      8308   Linux Plain DM-Crypt",
   " LUKS                            =      8309   Linux LUKS encrypted",
   " RAID                            = fd / fd00   Linux Raid",
   " APPLE                           = ab / ab00   Apple Boot",
   " HFS                             = af / af00   Apple MAC OSX HFS+",
   " APFS                            = 73 / af73   Apple APFS Container",
   " AOT1                            = ad / ad00   OS2 ArcaOS Type 1",
   "",
   NULL
};

static  char       *cmd_lvm[] =
{
   "",
   "Set one or more LVM related values for selected partition(s), or disk(s)",
   "",
   " Usage: LVM [pid | *] -d:nr -v:volume -p:part -l:drive -m:menu",
   "",
   "   pid          : partition id as shown by DFSee part list, or '*'. OPTIONAL",
   "",
   "   -c           : DISK: Update CRC values for LVM sectors on selected disk(s)",
   "   -C           : Delete existing LVM signature sector (BBR) for non LVM",
   "                  type 0x35 partitions, forces COMPATIBILITY mode to avoid",
   "                  'partition corrupt' errors from LVM.EXE or eCS installer",
   "                  Can be used with, or without specifying a specific PID",
   "   -d:nr        : disk number, restrict updates to specified disk only",
   "   -d           : restrict updates to the current disk only",
   "   -D           : Delete existing LVM info, and for an LVM-type (0x35) try",
   "                  to recover the values from an existing LVM-signature area",
   "                  Without recovery, default values will be substituted.",
   "   -D -C        : Delete existing LVM info including the signature area,",
   "                  and force default values (resets LVM to compatibility)",
   "   -G           : DISK: Update LVM geometry to match the DFSee L-Geo for disk",
   "   -i    or -i- : set installable status for volume,  or remove it",
   "   -l:drive     : set prefered driveletter for the volume",
   "   -l:\"\" or -l- : remove driveletter for the volume (hide)",
   "   -list        : List LVM structures only, no update, implies -P-",
   "   -m    or -m- : set volume on the IBM BootManager menu, or remove it",
   "   -n[:name]    : DISK: set new disk name for selected disk(s), optional prompt",
   "   -n[:name]    : With an explicit PID specified (partition update), will set",
   "                  the disk name to use, when no LVM diskname is known yet",
   "   -p:part      : set partition name for this partition",
   "   -p:\"\" or -p- : remove partitionname. LVM will use a default.",
   "   -P-          : Do not prompt for any values with a dialog-window",
   "   -r-          : Do NOT use existing LVM-signature data in LVM recovery",
   "   -s           : force synchronization of start/size with p-tables",
   "   -v:volume    : set volumename (= IBM BootManager name if on menu)",
   "   -v:\"\" or -v- : remove volumename. This deletes the volume for LVM",
   "   -V           : Automatically assign a volume-name for a new partition",
   "",
   "                  The options marked with 'DISK:' work at the disk level",
   "                  and do not require a 'pid' to be specified.",
   "",
   NULL
};


static  char       *cmd_walk[] =
{
   "",
   "Select specified disk and display MBR contents, 'walk' shows EBR's too",
   "",
   " Usage: DISK | WALK  [options]  [disk-nr]",
   "",
   "   disk-nr      : disk number to open/show, default is current or first",
   "",
   "   -f:'fsname'  : Set mode to filesystem 'fsname' on large-floppy",
   "   -F           : Force use of First EXT-container  (reread too)",
   "   -g           : Show disk geometry too            (reread too)",
   "   -std         : Industry std letter assignment    (reread too)",
   "   -L[:0|1|2]   : Show preferred: none (0), LVM (1), Windows (2)",
   "   -m           : Identify MBR, and show I13 limit  (reread too)",
   "   -q           : Quiet, select only, no display",
   "   -r           : Refresh partition tables from disk    (reread)",
   "                : and RESET previously forced geometry",
   "   -R           : open the disk in Read-only  mode",
   "   -R-          : force the disk in Read-write mode",
   "   -q           : quiet, select disk, minimal display detail",
   "",
   NULL
};



// Close FDISK filesystem for analysis and free any resources
static ULONG dfsFdskClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
);


// Display all partitioning related sectors for the specified partition
static ULONG dfsFdskPtableRelated
(
   DFSPARTINFO        *p                        // IN    partition to display
);

// FDISK filesystem, supply sector-type description string
static ULONG dfsFdskStype
(
   ULN64               di,                      // IN    sector type
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // OUT   type description
   void               *data                     // INOUT dummy
);

// FDISK filesystem, display sector-contents based on type
static ULONG dfsFdskDispl
(
   ULN64               di,                      // IN    sector type
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // IN    sector contents
);


// Display possible MAC DPM sectors in first track
ULONG dfsFdskMacPartMap                         // RET   result
(
   BOOL                verbose                  // IN    output style
);

// Display MAC disk-partition-map sector, 1-line compact
static ULONG dfsFdskMacDPMline
(
   ULONG               psn,                     // IN    sector number
   S_MCDPM            *sd                       // IN    Sector data as MCDPM
);


// Issue a warning about using this command on Windows-2000 or XP (P# 139)
static BOOL dfsFdskWarningW2KXP
(
   char               *cmd                      // IN    name of command
);


/*****************************************************************************/
// Initialize FDISK filesystem analysis
/*****************************************************************************/
ULONG dfsFdskInit
(
   void
)
{
   ULONG               rc = NO_ERROR;
   FILE               *fp;

   ENTER();

   dfsa->FsCommand          = dfsFdskCommand;
   dfsa->FsClose            = dfsFdskClose;
   dfsa->FsIdentifySector   = dfsFdskIdent;
   dfsa->FsShowType         = dfsFdskStype;
   dfsa->FsDisplaySector    = dfsFdskDispl;
   dfsa->FsFileInformation  = NULL;             // FS file alloc/path/size/date
   dfsa->FsGetAllocSpace    = NULL;             // FS Get file allocation, SPACE
   dfsa->FsWriteMetaSpace   = NULL;
   dfsa->FsFindPath         = NULL;
   dfsa->FsLsnAllocated     = dfsFdskAreaAllocated;
   dfsa->FsLsnSetAlloc      = NULL;
   dfsa->FsSltBuild         = dfsFdskSltBuild;
   dfsa->FsNpBuild          = NULL;
   dfsa->FsDisplayError     = dfsFdskDispError;
   dfsa->FsDisplayLsnInfo   = NULL;
   dfsa->FsDirIterator      = NULL;
   dfsa->FsDirFileSaveAs    = NULL;
   dfsa->FsAllocDisplay     = dfsFdskAllocMap;  // show partition map
   dfsa->FsCl2Lsn           = NULL;             // dummies, one to one
   dfsa->FsLsn2Cl           = NULL;
   dfsa->Fsi                = fdsk;
   dfsa->FsSectorTypes      = dfsFdskSectorTypes;
   dfsa->FsCmdHelp          = fdsk_txt;         // FS specific cmdhelp text
   dfsa->FsSltRecSize       = 0;
   dfsa->FsModeId           = DFS_FS_FDISK;

   dfsa->FsEntry = 0;
   nav.down = LSN_BOOTR;

   if (strlen(SINF->drive) <= 2)                // when not an image file ...
   {
      if ((fp = fopen( DFSFDSK_AUTOLOG, "rb")) != NULL) // existing file
      {
         fclose( fp);
         TxAppendToLogFile( DFSFDSK_AUTOLOG, TRUE);
         fdsk->AutoLog = TRUE;
      }
      if (dfsGetDiskCount(FALSE) != 0)          // if any media present
      {
         if (strcasecmp(SINF->afsys, "FDISK") == 0) // not for BMGR etc!
         {
            //- quietly (re)select disk, ignore RC, set base to whole disk!
            dfsSelectDisk( SINF->disknr, TRUE, FALSE);
            if ((SINF->disknr) && !TxaOptUnSet('r')) // if not -r-
            {
               dfsReadDiskInfo( FDSK_QUIET);
            }
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsFdskInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Close FDISK filesystem for analysis and free any resources
/*****************************************************************************/
static ULONG dfsFdskClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   dfsSlTableReset();                           // stop SLT thread and reset
   dfsCloseFileSystem();
   if (fdsk->AutoLog)
   {
      TxAppendToLogFile("", TRUE);
   }
   if (fdsk->crcTable != NULL)
   {
      free( fdsk->crcTable);
      fdsk->crcTable = NULL;
   }
   RETURN (rc);
}                                               // end 'dfsFdskClose'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Test for commands that should cause an automatic mode-change to FDISK mode
/*****************************************************************************/
BOOL dfsFdskIsFdskCmd                           // RET   is an FDISK command
(
   char               *cmd,                     // IN    command to test
   BOOL                autoquit                 // IN    autoquit, not switch
)
{
   BOOL                rc = FALSE;              // function return
   TXTM                candidate;

   ENTER();

   if (strlen(cmd) < (TXMAXTM -2))
   {
      sprintf(       candidate, " %s", cmd);
      TxStrToUpper(  candidate);                // make it all uppercase

      if (strstr( fdsk_commands, candidate) != NULL)
      {
         strcat( candidate, " ");
         if (strstr( fdsk_invalids, candidate) == NULL)
         {
            if (autoquit)                       // exclude the noquit ones
            {
               rc = (strstr( fdsk_no_quits, candidate) == NULL);
            }
            else
            {
               rc = TRUE;                       // consider as FDISK cmd
            }
         }
      }
   }
   BRETURN (rc);
}                                               // end 'dfsFdskIsFdskCmd'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Interpret and execute specific FDISK command;
/*****************************************************************************/
ULONG dfsFdskCommand
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *none,                    // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;
   ULN64               sn = 0;                  // sector number input
   BYTE                st = 0;                  // sector type wanted
   LONG                nr = 0;
   TXLN                dc;                      // DFS command
   int                 cc;                      // command string count
   char               *c0, *c1, *c2, *c3;       // parsed command parts
   char               *c4, *c5, *c6;            // parsed command parts
   TXLN                s0, s1;                  // big temporary string space
   char               *pp;                      // parameter pointer
   char               *s;
   DFSPARTINFO        *p  = NULL;
   DFSDISKINFO        *d  = NULL;
   USHORT              ml = 0;                  // 16-bit value
   USHORT              index = 0;               // disk/partition index
   FDSK_CB_INFO        cbi;                     // callback parameter struct

   ENTER();

   pp = TxaGetArgString( TXA_CUR, 1, 0, TXMAXLN, dc); // dc => cmd from arg 1
   cc = TxaArgCount( );                         // number of parameters
   c0 = TxaArgValue(0);
   c1 = TxaArgValue(1);
   c2 = TxaArgValue(2);
   c3 = TxaArgValue(3);
   c4 = TxaArgValue(4);
   c5 = TxaArgValue(5);
   c6 = TxaArgValue(6);

   sn = nav.this;                               // default at current sector

   if (strstr( fdsk->ShowCmd, FDSK_SINGLE_SHOW) != NULL)
   {
      //- make sure any -d: portion is stripped off, showing ALL disks by default
      //- Selected commands will add the -d:n back using dfsFdskSingleAutoShow

      if ((s = strstr( fdsk->ShowCmd,  " -d:")) != NULL)
      {
         *s = 0;
      }
   }
   else if (strlen(fdsk->ShowCmd) == 0)         // not set yet ...
   {
      if (TxaExeSwitch('m'))                    // switch specified ?
      {
         strcpy( fdsk->ShowCmd, TxaExeSwitchStr( 'm', NULL, "map -r"));
      }
      else                                      // use hardcoded default
      {
         strcpy( fdsk->ShowCmd, "part -r");
      }
   }

   TRACES(("ZERO the cbi structure at: 0x%p\n", &cbi));
   memset( &cbi, 0, sizeof(cbi));               // initial callback info
   cbi.disknr  = FDSK_ANY;                      // to their default values

   if (strcmp(c0, "?") == 0)
   {
      TxShowTxt( fdsk_txt);
   }
   else if (strcasecmp(c0, "autoshow"  ) == 0)  // modify AutoShow cmd
   {
      if (cc > 1)                               // Changing it ...
      {
         strcpy( fdsk->ShowCmd, (strcmp(c1, "-") == 0) ? "" : pp);
      }
      TxPrint("\nAutoShow command  : '%s%s%s'\n",
                 CBC, fdsk->ShowCmd, CNN);
   }
   else if (strncasecmp(c0, "apmap", 3) == 0)
   {
      if (!TxaOption('?'))
      {
         dfsFdskMacPartMap( (dfsa->verbosity >= TXAO_VERBOSE)); // display partition map
      }
      else
      {
         TxPrint("\nDisplay possible Apple-Partition-MAP records in the first disk track\n");
         TxPrint("\n Usage: %s  [-v]\n\n"
                 "         -v      = Display each record (partition) in detail\n", c0);
      }
   }
   else if (strncasecmp(c0, "uuidgen", 7) == 0)
   {
      if (!TxaOption('?'))
      {
         DFS_GUID      guid;                    // generated
         DFS_GUID      guidConverted;           // converted to string, and back

         dfsGptMkGuid( TxaOptNum('i', NULL, 1), guid);

         TxPrint("Generated GUID is : %s\n", dfsFsUuidValueString( guid));
         if (dfsUidStringToBinary( dfsFsUuidValueString( guid), guidConverted, TRUE) == NO_ERROR)
         {
            TxPrint("Double conversion : %s\n", dfsFsUuidValueString( guidConverted));
         }
         else
         {
            TxPrint( "\nConversion to string, then back to binary failed!\n");
         }
      }
      else
      {
         TxPrint("\nCreate and display a unique GUID value\n");
         TxPrint("\n Usage: %s  [-i:objectid]\n", c0);
      }
   }
   else if (strncasecmp(c0, "bmp", 3) == 0)
   {
      if (!TxaOption('?'))
      {
         if (dfsa->bmgrDisk != 0)               // disk number for BM
         {
            ml = dfsGetDiskNumber( 'd', ".", c1, FALSE); // current is default

            if (((d = dfsGetDiskInfo( ml)) == NULL) ||
                 (d->bmgrPsn               == L64_NULL))
            {
               TxPrint( "\nThere is no BMGR on disk %hu, show default BM\n", ml);
               ml = dfsa->bmgrDisk;             // show default BM disk
            }
            TRACES(( "Disk:%hu, d:%p  bmgrPsn:0x%llX\n", ml, d, d->bmgrPsn));
            if ((d = dfsGetDiskInfo( ml)) != NULL)
            {
               USHORT pdisk = SINF->disknr;     // remember current disk

               TxPrint( "\nDisplay disk-%hu BMGR sector with primary partition "
                        "BMGR names:\n\n", ml);

               dfsSelectDisk( ml, FALSE, FALSE); // select BM-disk
               dfsSectorAsHex( d->bmgrPsn +3, dfsGetSectorSize()*5/8, FALSE);
               dfsSelectDisk( pdisk, FALSE, FALSE); // quietly re-select disk
            }
         }
         else
         {
            TxPrint("\nNo IBM BootManager present on any disk\n");
         }
      }
      else
      {
         TxPrint("\nShow BMGR recorded names for primary partitions\n");
         TxPrint("\n Usage: %s  [disknr]\n", c0);
      }
   }
   else if (strcasecmp(c0, "browse" ) == 0)
   {
      if (!TxaOption('?'))
      {
         ULONG  listcode  = TXDID_OK;

      #if defined (USEWINDOWING)
         TXRECT where  = {18,0,0,0};            // fixed position

         if (TxaOption('P') || TxaOption('!') || (cc == 1))
         {
            listcode = txwListBox( TXHWND_DESKTOP, TXHWND_DESKTOP, &where,
                     " Select partition to start BROWSING on ", "",
                       0, TXLB_MOVEABLE,
                       cMenuTextStand, cMenuBorder_top, // same color as menu
                       dfsa->slPartOne);

            strcpy( s1, "1");
            if (listcode > TXDID_MAX)           // return list selection
            {                                   // plus current item base value
               sprintf( s1, "%u", listcode - TXDID_MAX); // calculate pid
            }
         }
         else
      #endif                                    // USEWINDOWING
         {
            strcpy( s1, c1);
         }

         if (listcode != TXDID_CANCEL)          // not escaped ...
         {
            index = dfsParsePartSpec( s1, dfsGetDiskNumber( 'd', ".", "", FALSE), &p);
            switch (index)
            {
               case FDSK_ANY:                   // all partitions
                  TxPrint( "Multiple partitions (%s) not supported!\n", c1);
                  break;

               case 0:                          // invalid partition
                  TxPrint( "Partition '%s' not found\n", c1);
                  break;

               default:                         // specific partition
                  sprintf( dc, "part -q %hu", index);
                  TxPrint( "Open for browsing : %s%s%s\n", CBG, dc, CNN);
                  dfsMultiCommand( dc, 0, FALSE, FALSE, TRUE);
                  if (cc > 2)
                  {
                     sprintf( dc, "browse %s", c2);
                  }
                  else                          // no starting DIR, prompt with ROOT
                  {
                     sprintf( dc, "browse %c -P", FS_PATH_SEP);
                  }
                  TxaReParseCommand( dc);
                  rc = DFS_PENDING;
                  break;
            }
         }
      }
      else
      {
         TxPrint("\nStart a file/directory browse from a partition to be selected\n");
         TxPrint("\n Usage: %s  [pid  [path]]\n\n"
                 "         pid   = DFSee ID for partition to browse, default 1\n"
                 "         path  = Optional PATH from root, to start browse with\n\n"
                 "         -P    = Force partition selection dialog\n", c0);
      }
   }
   else if (strcasecmp(c0, "dfsibmgr"  ) == 0)  // Create a DFSIBMGR.IMG
   {                                            // for auto BMGR install
      if (cc > 1)                               // disk number
      {
         if ((d = dfsGetDiskInfo((USHORT) atoi( c1))) != NULL)
         {
            if (d->bmgrPsn != L64_NULL)         // BMGR present on disk
            {
               USHORT pdisk = SINF->disknr;

               dfsSelectDisk( (USHORT) atoi( c1), FALSE, FALSE); // select BM-disk

               strcpy( s0, (cc > 2) ? c2 : DFSFDSK_BMGRIMG);
               TxFnameExtension( s0, "img");
               TxPrint( "\nCreate image '%s' for BMGR on disk-%hu ...\n\n",
                           s0, SINF->disknr);

               rc = dfsCreateImage( d->bmgrPsn, DFSFDSK_BMGRSEC, s0, DFSI_RAWIMAGE, 0);

               dfsSelectDisk( pdisk, FALSE, FALSE); // quietly re-select disk
            }
            else
            {
               TxPrint("Disk %s does not contain a valid IBM BootManager\n", c1);
            }
         }
         else
         {
            TxPrint("Disk %s is not present\n", c1);
         }
      }
      else
      {
         TxPrint("\nCreate IBM BootManager image '%s' for auto-install, or\n"
                 "a backup image named by the 2nd parameter\n\n", DFSFDSK_BMGRIMG);
         TxPrint("\n Usage: %s  disknr-to-get-image-from  [image-filename]\n", c0);
      }
   }
   else if ((strncasecmp(c0, "bmfix", 5 ) == 0) ||  // Fix BM bootsector
            (strncasecmp(c0, "w2kbm", 5 ) == 0)  )  // Make W2K BMGR (deprecated)
   {
      if (!TxaOption('?'))
      {
         rc = dfsFdskBmgrFixBootSec((USHORT) atoi( c1), FALSE);
         if (rc == NO_ERROR)
         {
            dfsReadDiskInfo( FDSK_QUIET);       // re-read all partition info
         }                                      // to remove warnings
      }
      else
      {
         TxPrint("\nProtect IBM BootManager from Win2000 CHKDSK, fix CHS values and I13X\n");
         TxPrint("\n Usage: %s  [disknr-with-BM]\n\n"
                 "         -I13X   = Require I13X capable MBR"
                                   " to boot beyond cyl 1024\n", c0);
      }
   }
   else if (strncasecmp(c0, "cr", 2    ) == 0)  // create new partition
   {
     BOOL             ok = TRUE;               // arg/option combinations OK
     TXTT             a1, a2, a3, a4, a5, a6;  // resulting argument values
     TXA_OPTION      *opt;

     #if defined (USEWINDOWING)
        TXRECT where  = {18,0,0,0};            // fixed position
        ULONG  listcode  = TXDID_OK;

        if (TxaOption('P') || TxaOption('!'))  // explicit prompting
        {
           //- enable all freespace of minimal 1/2 Cyl size, any kind
           dfsa->nrFreeCCr = dfsFreeSlEnable( dfsa->slFreeCCr, L64_NULL, TRUE, 0, '*');
           if (dfsa->nrFreeCCr != 0)
           {
              listcode = txwListBox( TXHWND_DESKTOP, TXHWND_DESKTOP, &where,
                       " Select freespace area for new partition ", "",
                         5402, TXLB_MOVEABLE,
                         cMenuTextStand, cMenuBorder_top, // same color as menu
                         dfsa->slFreeCCr);

              if ((listcode > TXDID_MAX) && (listcode != TXDID_CANCEL)) // valid and not escaped
              {
                 DFSPARTINFO        *freeSpace = dfsGetFreeInfo( listcode - TXDID_MAX);
                 BOOL                haveCommand;

                 if (freeSpace->partent.PartitionType > DFS_FSP_GPT_STYLE)
                 {
                    haveCommand = txwCreateMbrPartDialog( freeSpace, dc);
                 }
                 else
                 {
                    haveCommand = txwCreateGptPartDialog( freeSpace, dc);
                 }
                 if (haveCommand)
                 {
                    rc = dfsMultiCommand(dc, 0, TRUE, FALSE, TRUE);
                 }
              }
           }
           else
           {
              TxNamedMessage( !dfsa->batch, 0, " ERROR: Not enough freespace ",
                  "There is no freespace area large enough to create a new partition!");
           }
        }
        else
     #endif                                    // USEWINDOWING
     {
        char          targetKind;              // Gpt, Primary, Logical

        // Non-dialog, analyse and execute the actual create command
        ok &= TxaOptMutEx((cc > 1), "lpg",   "if any argument is specified!", 0);
        ok &= TxaOptMutEx((cc > 2), "t",     "if 'type' is specified!",       0);
        ok &= TxaOptMutEx((cc > 3), "s",     "if 'size' is specified!",       0);
        ok &= TxaOptMutEx((cc > 4), "df",    "if 'location' is specified!",   0);
        ok &= TxaOptMutEx((cc > 5), "areix", "if 'position' is specified!",   0);
        ok &= TxaOptMutEx((cc > 6), "b",     "if 'BMGR name' is specified!",  0);

        a1[0] = a2[0] = a3[0] = a4[0] = a5[0] = a6[0] = 0;

        cbi.cbOption1 = TxaOption('F');        // force 'ACTIVE' status flag
        cbi.cbOption2 = TxaOption('M');        // multiple visible primaries
        if ((opt = TxaOptValue('A')) != NULL)  // create AS
        {
           DFSPARTINFO  *p = NULL;

           switch (opt->type)
           {
              case TXA_NUMBER: p = dfsGetPartInfo( (USHORT) opt->value.number);          break;
              case TXA_STRING: p = dfsGetPartInfo( dfsDrive2PartId( opt->value.string)); break;
           }
           if (p != NULL)                      // referenced partition
           {
              sprintf( a3, "%.0lf", TXSMIB((p->sectors), p->bpsector) - (double) 0.5);
              if (p->tablenr == GPT_STYLE)
              {
                 strcpy(  a1, "gpt");
                 strcpy(  a2, dfsFsUuidValueString( p->typeGuid));
                 TRACES(( "Create GPT-style As: %s %s %s\n", a1, a2, a3));
              }
              else
              {
                 BYTE       partType = p->partent.PartitionType;

                 if ((cbi.cbOption2) && dfsPartTypeHidable( partType)) // option 'M'
                 {
                    partType &= ~DFS_P_PHIDDEN; // force visible type
                 }
                 strcpy(  a1,                 (p->primary) ? "pri" : "log");
                 sprintf( a2, "0x%2.2hhX",     partType);
                 cbi.cbOptNum2 =               p->partnr; // prefered slot number
                 TRACES(( "Create MBR-style As: %s %s %s using prefered entry: %llu\n",
                                     a1, a2, a3, cbi.cbOptNum2));
              }
           }
        }                                      // explicit -S: overrules ...
        cbi.cbOptNum2 = TxaOptNum('S', NULL, cbi.cbOptNum2); // prefered slot number
        if (cc > 1)                            // explicit pri/log
        {
           TxCopy( a1, c1, TXMAXTT);
        }
        else
        {
           if      (TxaOption('p')) strcpy( a1, "pri");
           else if (TxaOption('l')) strcpy( a1, "log");
           else if (TxaOption('g')) strcpy( a1, "gpt");
           else if (strlen(a1) == 0)
           {
              TxPrint( "\n* Specify '-p' primary, '-l' logical or '-g' gpt option!");
              ok = FALSE;
           }
        }
        targetKind = toupper(a1[0]);           // freespace kind needed

        if (cc > 2)                            // explicit type
        {
           if ((targetKind == 'G') &&          // GPT style partition
               (strlen( c2) >= 36)  )          // and 'long' type spec
           {
              sprintf( a2, "{%.36s}", c2);
           }
           else                                // symbolic, or MBR hex
           {
              TxCopy( a2, c2, TXMAXTT);
           }
        }
        else
        {
           if ((opt = TxaOptValue('t')) != NULL)
           {
              switch (opt->type)
              {
                 case TXA_STRING:
                    if ((targetKind == 'G') && // GPT style partition
                        (strlen( opt->value.string) >= 36))
                    {
                       sprintf( a2, "{%.36s}", opt->value.string);
                    }
                    else                       // symbolic, or MBR hex
                    {
                       TxCopy( a2, opt->value.string, TXMAXTT);
                    }
                    break;

                 default:
                    if (targetKind == 'G')
                    {
                       sprintf(a2, "%4.4llX", opt->value.number);
                    }
                    else
                    {
                       sprintf(a2, "0x%2.2llX", opt->value.number);
                    }
                    break;

              }
           }
           else if (strlen(a2) == 0)           // no arg and no option ...
           {
              if (strncasecmp(a1, "gpt", 3) == 0)  // GPT, initial create (cr pri guard)
              {
                 strcpy( a1, "pri");
                 strcpy( a2, "guard");
                 targetKind = 'P';             // MBR-style! cr pri guard
              }
              else
              {
                 TxPrint( "\n* A system Type must be specified!");
                 ok = FALSE;
              }
           }
        }
        if (cc > 3)                       // rest of parameters/options is optional; just copy ...
        {
           TxCopy( a3, c3, TXMAXTT);
           if (cc > 4)
           {
              TxCopy( a4, c4, TXMAXTT);
              if (cc > 5)
              {
                 TxCopy( a5, c5, TXMAXTT);
                 if (cc > 6)
                 {
                    TxCopy( a6, c6, TXMAXTT);
                 }
              }
           }
        }
        if ((opt = TxaOptValue('s')) != NULL)   // number including unit
        {
           switch (opt->type)
           {
              case TXA_STRING: TxCopy( a3, opt->value.string, TXMAXTT); break;
              default:
                 sprintf( a3, "%llu,%c", opt->value.number, opt->unit);
                 break;
           }
        }
        if (strlen(a4) == 0)                    // no disk parameter
        {
           if (!TxaOptUnSet('d'))               // -d- means all disks
           {
              sprintf( a4,  "%llu", TxaOptNum('d', NULL, 0));
           }
        }

        p= NULL;                                // no freespace info
        if (TxaOptSet('f'))                     // location, freespace-id
        {
           index = TxaOptNum('f', "FreeSpace  ", 0);
           if (index)
           {
              p = dfsGetFreeInfo( index);       // likely freespace info
           }
           sprintf( a4, "@%hu", index);
        }
        if ((opt = TxaOptValue('a')) != NULL)   // Absolute-pos, nr incl unit
        {
           switch (opt->type)
           {
              case TXA_STRING:
                 TxPrint( "\n Absolute position '-a' must be a number!");
                 ok = FALSE;
                 break;

              default:
                 sprintf( a5, "@%llu,%c", opt->value.number, opt->unit);
                 break;
           }
        }
        if ((opt = TxaOptValue('r')) != NULL)   // Relative-pos, nr incl unit
        {
           switch (opt->type)
           {
              case TXA_STRING:
                 TxPrint( "\n Relative position '-r' must be a number!");
                 ok = FALSE;
                 break;

              default:
                 sprintf( a5, "+%llu,%c", opt->value.number, opt->unit);
                 break;
           }
        }
        if ((opt = TxaOptValue('e')) != NULL)   // from-End-pos, nr incl unit
        {
           switch (opt->type)
           {
              case TXA_STRING:
                 TxPrint( "\n from-End position '-e' must be a number!");
                 ok = FALSE;
                 break;

              default:
                 sprintf( a5, "-%llu,%c", opt->value.number, opt->unit);
                 break;
           }
        }

        if (targetKind == 'G')                 // GPT, check valid type
        {
           if (dfsParseGptType( a2) == FALSE)
           {
              TxPrint( "\n GPT partition type '%s' is unknown/invalid!", a2);
              ok = FALSE;
           }

           if ((opt = TxaOptValue( DFS_O_GUID)) != NULL)
           {
              switch (opt->type)
              {
                 case TXA_STRING:
                    if (strlen( opt->value.string) >= 36)
                    {
                       sprintf( s0, "{%.36s}", opt->value.string);
                       if (dfsUidStringIsValid( s0)) // correct syntax
                       {
                          dfsUidStringToBinary( s0, cbi.partGuid, TRUE);
                          break;
                       }
                    }
                 default:
                    TxPrint( "\n Partition GUID '-guid' must be a 36 character string");
                    ok = FALSE;
                    break;
              }
           }
        }
        else                                   // options for MBR style only
        {
           if ((opt = TxaOptValue('i')) != NULL) // before-Int13, nr incl unit
           {
              switch (opt->type)
              {
                 case TXA_STRING:
                    TxPrint( "\n before-Int13 pos  '-i' must be a number!");
                    ok = FALSE;
                    break;

                 default:
                    sprintf( a5, "{%llu,%c", opt->value.number, opt->unit);
                    break;
              }
           }
           if ((opt = TxaOptValue('x')) != NULL) // after-Int13, nr incl unit
           {
              switch (opt->type)
              {
                 case TXA_STRING:
                    TxPrint( "\n after-Int13  pos  '-x' must be a number!");
                    ok = FALSE;
                    break;

                 default:
                    sprintf( a5, "}%llu,%c", opt->value.number, opt->unit);
                    break;
              }
           }
           if (TxaOptSet('b'))                 // bootmanager name, string
           {
              TxCopy( a6,  TxaOptStr('b', "BMGR name  ", ""), TXMAXTT);
           }
        }
        if ((ok) && (!TxaOption('?')))         // parameters OK sofar, no help
        {
           FDSK_NUMVAL   size = FDSK_NUMZERO;  // size param info
           FDSK_NUMVAL   pos  = FDSK_NUMZERO;  // pos  param info
           BYTE          pType = 0;

           strcpy( s0, a3);                    // size parameter

           cbi.cbOption3 = TxaOption('C');     // clear bootsector ?
           cbi.flag      = FALSE;              // init to logical partition

           switch (targetKind)
           {
              case 'G':                        // GPT style
                 dfsUidStringToBinary( a2, cbi.typeGuid, TRUE);
                 sprintf( a6, "New: %s", dfsGptGuidDescription( cbi.typeGuid));
                 TxStrip( a6, a6, 0, ' ');     // strip trailing spaces

                 if ((atoi( s0) == 0) &&        // no size or 0 specified on EFI partition
                     (GptGuidMatch( cbi.typeGuid, dfsGuid_EFI_SYSTEM)))
                 {
                    strcpy( s0, "128");        // default to 128 MiB
                 }
                 dfsParseNewSize( s0, targetKind, p, NULL, &size);

                 sprintf( cbi.string, "CREATE %s '%s' %s %s %s\n\n", a1, a6 + 5, a3, a4, a5);
                 break;

              case 'P':                        // MBR style, primary
                 cbi.flag = TRUE;
              case 'L':                        // MBR style, logical
                 pType = dfsParsePartType( a2, NULL);
                 if      (pType == DFS_P_BOOTMGR)
                 {
                    if (atoi( s0) == 0)        // no size or 0 specified
                    {
                       strcpy( s0, "1,c");     // default to 1 cylinder
                    }
                    if (cbi.flag == FALSE)
                    {
                       TxPrint("WARNING: Bootmanager needs a PRIMARY partition to work!\n");
                    }
                 }
                 else if (pType == DFS_P_EFI_GPT)
                 {
                    TxPrint( "Forcing CR option : '-align-' to ensure full disksize is used for the GPT guard.\n");
                    TxaOptSetItem( "-align-"); // Use FULL disk (no align)
                 }
                 dfsParseNewSize( s0, targetKind, p, NULL, &size);
                 sprintf( cbi.string, "CREATE %s %s %s %s %s %s\n\n", a1, a2, a3, a4, a5, a6);
                 break;

              default:
                 TxPrint("Partition style '%s' is invalid, specify primary, logical or gpt\n", a1);
                 rc = DFS_VALUE_ERROR;
                 break;
           }
           TRACES(("cbi.string: '%s'", cbi.string));
           if (rc == NO_ERROR)
           {
              //- following code is Gpt/Mbr generic
              if ((p = dfsParseLocation( a4, a5, targetKind, &size)) != NULL)
              {
                 ULN64         fs = p->sectors;

                 //- make sure the right disk is opened (translations!)
                 rc = dfsSelectDisk( p->disknr, FALSE, FALSE);

                 sprintf( s1, "Freespace ID %2.2u   : %8.1lf MiB disk %hu\n",
                               p->id, TXSMIB( fs, p->bpsector), p->disknr);
                 strcat( cbi.string, s1);      // add to pop-up description

                 sprintf( s1, "Found fsp-area %s%2u%s : ", CBM, p->id, CNN);
                 dfsSz64Bps( s1, fs, p->bpsector, "     ");
                 TxPrint("type: %s\n", p->descr);

                 cbi.disknr = p->disknr;
                 cbi.cbOptNum1 = TxaOptNum('G', NULL, (pType == DFS_P_EFI_GPT) ? 1 // no GAP on guard
                                          : dfsDefaultGapSectors( targetKind, p)); // GAP based on alignment
                 strcpy( pos.prefix,  "+-*@{}");
                 strcpy( pos.format,  "dxoDXO");
                 dfsParseNumValue( a5, &pos);  // RAW parse of position

                 strcpy( size.prefix, "");
                 strcpy( size.format, "dxoDXO");
                 strcpy( size.type,   "mkgtchs");
                 dfsParseNewSize( s0, targetKind, p, &pos, &size);

                 pos.value = 0;
                 strcpy( pos.prefix,  "+-*@{}");
                 strcpy( pos.format,  "dxoDXO");
                 strcpy( pos.type,    "mkgtchs");
                 rc = dfsParsePosition( a5, targetKind, cbi.cbOptNum1, p, &size, &pos);
                 if (rc == NO_ERROR)
                 {
                    USHORT pid = p->id;        // remember freespace pid for post processing
                    USHORT dsk = p->disknr;    // remember disknr

                    cbi.more    = p;           // attach freespace info
                    cbi.size    = &size;       // attach to callback info
                    cbi.pos     = &pos;
                    cbi.disknr  = p->disknr;
                    cbi.number  = size.value;
                    cbi.sn      = p->basePsn + pos.value;
                    cbi.ntype   = DFS_P_EMPTY; // avoid post processing errors

                    if (targetKind == 'G')     // GPT style freespace/partition
                    {
                       dfsUidStringToBinary( a2, cbi.typeGuid, TRUE);
                       sprintf( a6, "New: %s", dfsGptGuidDescription( cbi.typeGuid));
                       TxStrip( a6, a6, 0, ' '); // strip trailing spaces
                       TxCopy(  cbi.pname,     TxaOptStr( TXA_O_NAME,   NULL, a6), GPT_PNAME_LEN +1);
                       TxCopy(  cbi.cbOptStr1, TxaOptStr( DFS_O_ATTRIB, NULL, "0"), 17);

                       sprintf( s0, "%-18.18s: %8.1lf MiB named '%s'\n\n", a6 +5,
                                TXSMIB(cbi.number, p->bpsector), cbi.pname);
                       strcat(  cbi.string, s0); // add to popup

                       // TxPrint( "\nAbout to create a GPT style partition: %s", cbi.string);
                       rc = dfsGptCreatePartition( &cbi);
                       if (rc == NO_ERROR)     // do some post-processing
                       {
                          //- to be refined, perhaps restore default images (EFI)
                       }
                    }
                    else                       // MBR style freespace/partition
                    {
                       //- Note: cbi.sn  is the PSN of the partition bootsector
                       //-       EBR-PSN is in p->partPsn (by ParsePosition)

                       if (TxaOption('N'))      // No-code ?
                       {
                          cbi.creMbr = FDSK_CR_MBR_EMPTY; // allow MBR create, NO code
                       }
                       else
                       {
                          cbi.creMbr = FDSK_CR_MBR_STD; // allow MBR create, STD code
                       }
                       cbi.creEbr = TRUE;      // allow EBR creation
                       cbi.ntype  = dfsParsePartType( a2, &cbi);
                       dfsPartTypeDescription( cbi.ntype, s1);

                       sprintf( s0, "%s  = %2.2x : %8.1lf MiB %s\n\n",
                                 s1, cbi.ntype,
                          TXSMIB(cbi.number, p->bpsector),
                                (cbi.flag) ? "Primary" : "Logical");

                       strcat(cbi.string, s0); // add to popup

                       strcpy( cbi.bname, "");
                       if (strlen(a6) != 0)    // BM-name specified
                       {
                          memset( cbi.bname, ' ', BT_BM_L);
                          memcpy( cbi.bname, a6,  strlen(a6));
                          cbi.bname[ BT_BM_L] = '\0';
                          TxPrint(" as %s%s%s", CBC, cbi.bname, CNN);
                       }
                       TxPrint( "\n");

                       TxPrint("In freespace area : %2.2u  ", p->id);
                       dfsX10( "at rel pos: ", pos.value,  CNC, "\n");
                       dfsGeoDispTransPsn( "Part-table offset :", p->geoHeads, p->geoSecs, p->partPsn);
                       dfsGeoDispTransPsn( "First part sector :", p->geoHeads, p->geoSecs, cbi.sn);
                       dfsGeoDispTransPsn( "Last  part sector :", p->geoHeads, p->geoSecs, cbi.sn -1 + cbi.number);

                       if (cbi.ntype != 0)
                       {
                          nav.down   = p->partPsn;  //- make it easy to refer to
                          rc = dfsExecOnBootRec( p, cbi.disknr, dfsFdskCreate, &cbi);
                       }
                       else
                       {
                          TxNamedMessage( !dfsa->batch, 5024, " ERROR: Invalid partition type ",
                                     "You can NOT create a new partition with partition type 00 = FREESPACE!");
                          rc = DFS_CMD_FAILED;
                       }
                    }
                    if (rc == NO_ERROR)        // GPT/MBR generic some post-processing
                    {
                       TRACES(( "f *%p, f->id:%hu; pid:%hu on disk:%hu (%hu)\n",
                                 p, p->id, pid, p->disknr, dsk));

                       if (pid > dfsPartitions()) // at end of disk
                       {
                          DFSPARTINFO *q;

                          for (pid = dfsPartitions(); pid; pid--)
                          {
                             q = dfsGetPartInfo( pid);
                             TRACES(( "Check fsp *%p to pid:%hu on disk %hu\n", q, q->id, q->disknr));
                             if (q->disknr <= dsk)
                             {
                                break;         // found adjacent
                             }
                          }
                          pid++;               // this will be our new one ...
                       }
                       dfsa->autoPid = pid;
                       TRACES(("CR : autoPid set to %2.2hu\n", pid));

                       dfsReadDiskInfo( FDSK_QUIET); // update in-memory, freespace etc!

                       if ((targetKind != 'G') && (TxaOption('L'))) // create default LVM info
                       {
                          sprintf( dc, "lvm %2.2hu %s", pid, TxaOptStr('L', NULL, "-V"));
                          if (cbi.cbOption3)   // bootsector cleared ?
                          {
                             strcat(dc, " -D"); // Clear/Init DLAT sector
                             if (cbi.ntype != DFS_P_WARP_LVM)
                             {
                                strcat(dc, " -C"); // Clear BBR sector
                             }
                          }
                          TxPrint( "Add default LVM   : %s%s%s\n\n", CBG, dc, CNN);
                          rc = dfsMultiCommand(dc, 0, TRUE, FALSE, TRUE);
                          if (rc == DFS_NO_CHANGE) // Canceled
                          {
                             rc = NO_ERROR;    // keep OK create RC
                          }
                       }

                       TxPrint("\nCreate partition  : %s\n", (rc) ? "not completed" : "successful");
                       if (TxaOption('o'))     // open partition
                       {
                          sprintf( dc, "part -r -q -2- %2.2hu", pid);
                          TxPrint( "Automatic open as : %s%s%s\n", CBG, dc, CNN);
                          TxaReParseCommand( dc);
                          rc = DFS_PENDING;
                       }
                       else                    // inform of autoPid
                       {
                          if (cbi.ntype == DFS_P_EFI_GPT) // New GPT guard
                          {
                             TxPrint( "An empty GPT-guard partition has been created, you may want to create an\n"
                                      "EFI-System partition next, either using the menu or the command 'cr gpt efi'\n");
                          }
                          else
                          {
                             TxPrint( "New selectable as : %2.2hu and set as default for 'part .'\n\n", pid);
                          }
                       }
                    }
                 }
              }
              else
              {
                 rc = DFS_NOT_FOUND;
              }
           }
        }
        else
        {
           TxPrint("\n");
           if (TxaOption('?'))
           {
              TxShowTxt( cmd_create);
           }
           else
           {
              TxPrint( "\nTo get verbose usage information, use '%s  -?'\n", c0);
           }
           switch (targetKind)
           {
              case 'P':
              case 'L':
                 TxShowTxt( mbr_symbol);
                 if (strcasecmp( a2, "guard") != 0)
                 {
                    break;                     // no GPT type unless 'guard'
                 }
              case 'G':
                 TxShowTxt( gpt_symbol);
                 break;

              default:
                 TxShowTxt( mbr_symbol);
                 TxShowTxt( gpt_symbol);
                 break;
           }
        }
     }
   }
   else if (strncasecmp(c0, "delete", 6) == 0)     // delete partition
   {
      if (!TxaOption('?'))
      {
         cbi.option  = !TxaOptUnSet('c');       // clear full entry unless -c-
         cbi.disknr  = dfsGetDiskNumber( 'd', "*", c2, FALSE); // allow any disk
         cbi.verbose = TRUE;
         cbi.related = TRUE;                    // include related Ext-container
         switch (dfsParsePartSpec( c1, dfsGetDiskNumber( 'd', ".", c2, FALSE), &p))
         {
            case FDSK_ANY:                      // all partitions ...
               if (cbi.disknr != FDSK_ANY)      // but only for ONE disk!
               {
                  d = dfsGetDiskInfo( cbi.disknr);

                  if ((d->gptHeader != NULL) && (d->gptArray != NULL))
                  {
                     rc = dfsGptDeleteAll( d);
                  }
                  else                          // MBR style disk
                  {
                     //- Note that #partitions and the PARTINFO structures are
                     //- changed after every partition that is deleted!
                     //- Delete from highest PID to avoid skipping any

                     cbi.cbOption2 = TRUE;      // request descending iterate
                     rc = dfsIteratePmbr( dfsFdskDelete, &cbi);
                  }
               }
               else
               {
                  TxPrint( "Deleting ALL partitions only supported for a single disk!\n");
               }
               break;

            case 0:                             // invalid partition
               TxPrint( "Partition '%s' not found\n", c1);
               break;

            default:                            // specific partition
               if (p->tablenr == GPT_STYLE)
               {
                  rc = dfsGptDeletePartition( p, cbi.disknr);
               }
               else
               {
                  rc = dfsExecOnBootRec( p, cbi.disknr, dfsFdskDelete, &cbi);
               }
               break;
         }
      }
      else
      {
         TxPrint("\nDelete one partition from the partition-tables\n");
         TxPrint("\n Usage: %s  pid  [disk]  [options]\n\n"
                 "  pid    = ID of partition to delete, * to delete ALL for 1 disk\n"
                 "  disk   = optional disknumber for extra verification\n\n"
                 "  -c-     MBR-style: No full clear, reset system-type to 0x00\n"
                 "  -d      Partition to delete must be on current disk\n"
                 "  -d:[nr] Partition to delete must be on specified disk\n"
                 "  -L-     MBR-style: Do NOT delete related LVM DLAT for a primary,\n"
                 "          as this allows recreation in same place (resizing)\n"
                 "  -p:x    Partition ID (PID) to delete (same as 'pid' parameter)\n"
                 "  -p:x,r  Relative partition ID on specified or current disk\n", c0);
      }
   }
   else if (strcasecmp(c0, "edit"   ) == 0)     // Edit sector contents
   {
      //- make sure the extra (marking) options are ONLY set when editing the current object
      if (TxaOptMutEx( TRUE, "ahuP", NULL, 0))  // NO hex/ascii/unicode string
      {
         strcpy( dc, c0);
         if ((!TxaOption('?')) && (!TxaOption(TXA_O_DIR)) && ((cc == 1) || (strcmp( c1, ".") == 0)))
         {
            if (TxaOptSet('F'))                 // convert -F to default param
            {
               dfsOptionSizeSectors( 'F', DFSTORE, 's', nav.this, &nav.this);
            }
            dfsRead( nav.this, 1, rbuf);        // make sure nav.this loaded
            st = dfsIdentifySector( nav.this, nav.this_sninfo, rbuf);
            if ((st == ST_EXTBR) || (st == ST_MASTR))
            {
               if ((st == ST_EXTBR) || ((rbuf[0] != DFS_M_OPCO_1) && (rbuf[0] != DFS_M_OPCO_2)))
               {
                  ml = TxaOptNum('p', NULL, 0); // unless overruled
                  sprintf( s0, "-p:%u", (ml) ? ml : 0x1c2);
                  TxaOptSetItem( s0);           // position at start tables
               }
               if (!TxaOptSet('M') && (!TxaOptSet('m')))
               {
                  TxaOptSetItem("-M:0x1be");    // mark at start part-table
                  TxaOptSetItem("-m:64");       // size 4 entries
                  if (!TxaOptSet('v'))          // unless overruled, use HEX
                  {
                     TxaOptSetItem("-v:0");     // and stay in HEX view mode
                  }
               }
            }
         }
      }
      rc = DFS_PENDING;
   }
   else if (strncasecmp(c0, "setlet", 6) == 0)     // set drive letter
   {
      if ((cc > 1) && !TxaOption('?'))
      {
         if (dfsa->lvmPresent)                  // give some guidance
         {
            TxNamedMessage( !dfsa->batch, 5017, " WARNING: Command information ",
               "Using '%s' on an LVM system "
               #if defined (WIN32)
                  "will change the driveletter within Windows only! "
               #else
                  "is not very useful! "
               #endif
               "To change the volume driveletter or other LVM info for "
               "the OS/2 or eCS system, use the LVM command instead.", c0);
         }
         if (dfsFdskWarningW2KXP( c0))
         {
            switch (dfsParsePartSpec( c1, dfsGetDiskNumber( 'd', ".", "", FALSE), &p))
            {
               case FDSK_ANY:                   // all partitions
                  TxPrint( "Multiple partitions (%s) not supported!\n", c1);
                  break;

               case 0:                          // invalid partition
                  TxPrint( "Partition '%s' not found\n", c1);
                  break;

               default:                         // specific partition
                  rc = dfsSetDriveLetter( p, c2);
                  #if defined (WIN32)
                     if ((rc == NO_ERROR) && !TxaOption('n'))
                     {
                        rc = ntregSyncDevMapDiskKey();
                        ntregShowDriveLetters();
                     }
                  #endif
                  break;
            }
         }
      }
      else
      {
         TxPrint("\nSet internal DFSee driveletter\n");
         TxPrint("\n Usage: %s  pid  letter | -  [-nosync]\n", c0);
      }
   }
   else if ((strcasecmp(c0, "fat2os" ) == 0) ||  // Change FAT-BR DOS -> boot-OS
            (strcasecmp(c0, "dos2os" ) == 0) )   // (for backward compatibility)
   {
      if ((cc > 1) && !TxaOption('?'))
      {
         if (cc > 3)                            // 3rd param
         {
            strcpy( cbi.string, TxStrToUpper(c3));
         }
         else                                   // default to OS/2 target
         {
            strcpy( cbi.string, "OS/2");
         }
         cbi.disknr  = dfsGetDiskNumber( 'd', "*", c2, FALSE); // allow any disk
         cbi.verbose = TRUE;
         cbi.usebase = TRUE;                    // work on bootsectors
         switch (dfsParsePartSpec( c1, dfsGetDiskNumber( 'd', ".", c2, FALSE), &p))
         {
            case FDSK_ANY:                      // all partitions
               rc = dfsIteratePart( dfsFdskFat2Boot, &cbi);
               break;

            case 0:                             // invalid partition
               TxPrint( "Partition '%s' not found\n", c1);
               break;

            default:                            // specific partition
               rc = dfsExecOnBootRec( p, cbi.disknr, dfsFdskFat2Boot, &cbi);
               break;
         }
         TxPrint("\nChange to bootable FAT for operating system '%s' %s\n",
                    cbi.string, (rc) ? "failed" : "successful");
      }
      else
      {
         TxPrint("\nUpdate bootcode in FAT boot record for specific OS\n");
         TxPrint("\n Usage: %s  pid  [disk [boot-OS]]\n\n",  c0);
         TxPrint("  pid       = ID of partition to delete\n");
         TxPrint("  disk      = disknumber for extra verification\n");
         TxPrint("  boot-OS   = OS2 | NT | IBMDOS | PCDOS | MSDOS\n");
      }
   }
   else if (strncasecmp(c0, "sett", 4  ) == 0)  // change partition type
   {
      if ((cc > 2) && !TxaOption('?'))
      {
         TXTT    a3 = {0};                      // type specifier string

         switch (dfsParsePartSpec( c1, dfsGetDiskNumber( 'd', ".", c2, FALSE), &p))
         {
            case FDSK_ANY:                      // all partitions
               TxPrint( "Only a single partition can be changed at a time!\n");
               break;

            case 0:                             // invalid partition
               TxPrint( "Partition '%s' not found\n", c1);
               break;

            default:                            // specific partition
               if (cc > 3)                      // explicit type
               {
                  if ((p->tablenr == GPT_STYLE) && (strlen( c3) >= 36))
                  {
                     sprintf( a3, "{%.36s}", c3);
                  }
                  else                          // symbolic, or MBR hex
                  {
                     TxCopy( a3, c3, TXMAXTT);
                  }
               }
               else
               {
                  TXA_OPTION      *opt;

                  if ((opt = TxaOptValue('t')) != NULL)
                  {
                     switch (opt->type)
                     {
                        case TXA_STRING:
                           if ((p->tablenr == GPT_STYLE) && (strlen( opt->value.string) >= 36))
                           {
                              sprintf( a3, "{%.36s}", opt->value.string);
                           }
                           else                 // symbolic, or MBR hex
                           {
                              TxCopy( a3, opt->value.string, TXMAXTT);
                           }
                           break;

                        default:
                           if (p->tablenr == GPT_STYLE)
                           {
                              sprintf(a3, "%4.4llX", opt->value.number);
                           }
                           else
                           {
                              sprintf(a3, "0x%2.2llX", opt->value.number);
                           }
                           break;
                     }
                  }
               }
               cbi.verbose = TRUE;
               cbi.disknr  = dfsGetDiskNumber( 'd', "*", c2, FALSE);
               if (p->tablenr != GPT_STYLE)     // MBR partition
               {
                  if ((cbi.ntype = dfsParsePartType( a3, NULL)) != 0)
                  {
                     cbi.stype   = dfsParsePartType( c4, NULL);
                     if ((TxaOption('e')) ||    // allow extended types ?
                         ((dfsIsExtendedType(cbi.stype) == FALSE) &&
                          (dfsIsExtendedType(cbi.ntype) == FALSE) ))
                     {
                        strcpy( cbi.string, c3); // original 'new' type spec
                        rc = dfsExecOnBootRec( p, cbi.disknr, dfsFdskSetType, &cbi);
                     }
                     else
                     {
                        TxPrint("\nNew or old partition-type should not specify an extended container (05/0f).\n"
                                  "Use the 'fixext' command to change them instead, or use the '-e' option to\n"
                                  "force this non-standard value to be set for a regular partition\n");
                        rc = DFS_VALUE_ERROR;
                     }
                  }
                  else
                  {
                     TxPrint("New MBR partition-type '%s' is not valid\n", c3);
                     rc = DFS_VALUE_ERROR;
                  }
               }
               else                             // GPT partition
               {
                  if (dfsParseGptType( a3) == TRUE)
                  {
                     cbi.partnr = p->partnr;    // entry number in GPT
                     dfsUidStringToBinary( a3, cbi.typeGuid, TRUE);
                     rc = dfsGptSetType( p, &cbi);
                  }
                  else
                  {
                     TxPrint( "\n GPT partition type '%s' is unknown/invalid!", a3);
                     rc = DFS_VALUE_ERROR;
                  }
               }
               break;
         }
      }
      else
      {
         TxPrint( "Change type of a primary/logical MBR partition, or a GPT one\n\n");
         TxPrint( " Usage: %s pid  disk | *   newtype [oldtype]  |  -t:type  [-e]\n\n"
                  "  pid            = ID of a single partition to change the type for\n"
                  "  disk           = disknumber or '*' for all disks\n"
                  "  newtype        = new MBR or GPTtype for partition, HEX or symbolic\n"
                  "  oldtype        = current MBR type, for verification only.\n"
                  "  -t:number /sym = new MBR partition Type, numeric HEX, decimal or symbolic\n"
                  "  -t:guidstr/sym = new GPT partition Type, GUID-str without {}, or symbolic\n"
                  "  -e             = Force change to an extended container type.\n\n", c0);
         TxPrint( "Symbolic values are the same as for the 'cr' command: FAT, NTFS ...\n\n");
         TxPrint( "To change the type of an MBR extended container (0x05 / 0x0f),\n"
                  "use the 'fixext' command, or force it using the '-e' option.\n");
      }
   }
   else if (strncasecmp(c0, "fixchs", 6  ) == 0)   // make CHS match LBA Geo
   {
      if ((cc > 1) && !TxaOption('?'))
      {
         cbi.disknr = dfsParseDiskSpec( c1, NULL);
         cbi.flag   = TRUE;                     // visit all MBR/EBR sectors
         rc = dfsIteratePlvm( dfsFdskFixChs, &cbi);
      }
      else
      {
         TxPrint("\nFix all CHS values on a disk to acceptable values\n");
         TxPrint("\n Usage: %s  disknr | . | *  [-c:style]\n\n", c0);
         TxPrint("   disknr    = disk-number, '.' for current or '*' for all disks\n"
                 "   -c:style  = CHS-dummy style to use,\n"
                 "               0 or 'IBM' = IBM/DFSee, this is the default\n"
                 "               1 or 'PQ'  = PowerQuest: P-Magic, DriveImage\n"
                 "               2 or 'MS'  = Microsoft FDISK classic style\n"
                 "               ! or  ?    = Use CHS-style selection dialog\n\n");
         TxPrint("   Note:     CHS values in every partition-table entry\n"
                 "             will be adjusted to match the linear (LBA)\n"
                 "             values using the current logical geometry\n"
                 "             and default or selected CHS-dummy style\n");
      }
   }
   else if (strncasecmp(c0, "pl", 2    ) == 0)  // show fdisk-structure info
   {
      BOOL             ok = TRUE;               // arg/option combinations OK

      ok &= TxaOptMutEx((cc > 2), "d", "if a 2nd (disk) parameter is specified!", 0);

      if ((cc > 1) && ok && !TxaOption('?'))
      {
         DFST_HANDLE  cst = dfstGetDefaultStore();

         dfsInitList( 0, "-w", "-b");           // with optimal list options
         if (cc > 2)                            // use disk specified as param
         {
            cbi.disknr = dfsParseDiskSpec( c2, NULL); // parse disk BEFORE setting store!
         }
         else if (TxaOption('d'))               // use -d: value or current
         {
            cbi.disknr = TxaOptNum('d', NULL, (SINF->disknr) ? SINF->disknr  : 1);
         }
         else                                   // all disks
         {
            cbi.disknr = FDSK_ANY;
         }

         TRACES(( "cbi.disknr: %hu  SINF->disknr: %hu\n", cbi.disknr, SINF->disknr));

         dfstSetDefaultStore( DFST_SYSTEM);     // select system store

         cbi.related = TxaOption('r');          // include any related sectors
         cbi.nowrite = TRUE;                    // never write back ...
         switch (c1[0])
         {
            case 'f':                           // freespace area's
            case 'F':                           // freespace area's
               strcpy( cbi.string, "Freespace");
               rc = dfsIterateFree( dfsFdskListPinfo, &cbi);
               break;

            case 'e':                           // EBR chain entries
            case 'E':                           // EBR chain entries
               strcpy( cbi.string, "EBR  sect");
               rc = dfsIterateExtContainers( dfsFdskListPinfo, &cbi);
               break;

            case 'm':                           // MBR entries
            case 'M':                           // MBR entries
               strcpy( cbi.string, "MBR  sect");
               rc = dfsIteratePmbr( dfsFdskListPinfo, &cbi);
               break;

            case 'b':                           // bootsectors
            case 'B':                           // bootsectors
               cbi.usebase = TRUE;
               strcpy( cbi.string, "Boot-sect");
               rc = dfsIteratePart( dfsFdskListPinfo, &cbi);
               break;

            case 'w':                           // walk, starting with MBR
            case 'W':                           // walk, starting with MBR
               cbi.flag    = TRUE;              // flag indicates MBR/EBR
            case 'l':
            case 'L':
               cbi.option  = TRUE;              // include signature sects
               rc = dfsIteratePlvm( dfsFdskLvmWalk, &cbi);
               break;

            default:
               strcpy( cbi.string, "Partition");
               rc = dfsIteratePart( dfsFdskListPinfo, &cbi);
               break;
         }
         dfstRestoreDefaultStore( cst);         // reselect current store
         TxCancelAbort();                       // might have aborted ...
      }
      else
      {
         TxPrint("\n Usage: %s  part|free|ebr|mbr|boot|walk|lvm "
                 "[disk] [-related]\n\n", c0);
         TxPrint("  -r         will combine MBR/EBR with related LVM sectors\n"
                 "             and bootsectors with related EBR sectors.\n");
         TxPrint("  -d:nr      disk number to list, default is ALL disks\n\n");
         TxPrint("   Note:     sector numbers will be stored in the DFSee\n"
                 "             SN-list too, for use with LIST and EXPORT.\n");
      }
   }
   else if (strncasecmp(c0, "pcl", 3 ) == 0)    // clear partition info
   {
      if ((cc > 2) && !TxaOption('?'))
      {
         sprintf(s1, "%s%s%s", c2, c3, c4);     // allow multiple type params
         TxStrToUpper( s1);                     // convert to uppercase
         strcpy( cbi.string, "");               // action string

         cbi.disknr  = dfsParseDiskSpec( c1, NULL);
         cbi.nowrite = FALSE;                   // allow write back ...
         if (strchr( s1, 'L') != 0)
         {
            cbi.flag = FALSE;                   // Clear LVM sectors
            strcpy( cbi.string, "LVM-info (DLAT)");
            if (strchr( s1, 'S') != 0)          // LVM signature too ?
            {
               cbi.option  = TRUE;              // include signature sects
               strcat( cbi.string, " and signature (BBR)");
            }
            if (strchr( s1, 'P') != 0)          // and P-tables
            {
               cbi.related = TRUE;              // Clear related sectors too
               strcat( cbi.string, " and Partition-tables");
            }
         }
         else if (strchr( s1, 'P') != 0)        // no LVM, check P-table
         {
            cbi.flag = TRUE;                    // Clear MBR/EBR + tables
            strcpy( cbi.string, "Partition-table");
         }
         if (strlen(cbi.string) != 0)           // Plvm action required ?
         {
            rc = dfsIteratePlvm( dfsFdskClearSect, &cbi);
            if (rc == NO_ERROR)
            {
               if (strchr( s1, 'L') != 0)       // LVM-info
               {
                  if ((!TxAbort()) &&           // when not aborted yet
                      (cbi.confirmed == TRUE))  // and at least 1 confirmed
                  {
                     //- clear OTHER possible LVM DLAT sectors in MBR track
                     rc = dfsFdskClearLvmSectors( cbi.disknr, TRUE);
                  }
               }
            }
         }
         if (strchr( s1, 'B') != 0)             // boot records
         {
            cbi.related = FALSE;                // Don't visit EBR sectors
            cbi.usebase = TRUE;                 // select boot records
            rc = dfsIteratePart( dfsFdskClearBoot, &cbi);
         }
      }
      else                                      // no parameters, give usage
      {
         TxPrint("\nClear contents of all partition-table sectors, LVM information sectors\n");
         TxPrint(  "and/or partition boot records. (Multiple selections are possible)\n");
         TxPrint("\n Usage: %s  disknr|*  [Part-tables][Lvm-info [lvm-Sign]][Boot-rec]\n", c0);
      }
   }
   else if (strncasecmp(c0, "psa", 3 ) == 0)       // save partition info
   {
      if ((cc > 2) && !TxaOption('?'))
      {
         USHORT        first;
         USHORT        last;

         first = dfsParseDiskSpec( c1, &last);  // request as a range
         if (last != 0)                         // valid single disk or range
         {
            for (index = first; index <= last; index++)
            {
               if (dfsDiskAccessible( index))
               {
                  rc = dfsSavePartInfo( index, c2, pp +strlen(c1) +1);
               }
            }
         }
         else
         {
            TxPrint( "Disknr is invalid : %u\n", first);
            rc = DFS_VALUE_ERROR;
         }
      }
      else
      {
         TxPrint("\nSave contents of all partition-table "
                 "and LVM sectors to one or more files\n");
         TxPrint("\n Usage: %s  disknr | . | *  filename  [descr]\n\n", c0);
         TxPrint("   disknr    = disk-number, '.' for current or '*' for all disks\n"
                 "   filename  = base name for psave file(s) to be saved\n"
                 "   descr     = one line description, to be added to the files\n");
      }
   }
   else if (strncasecmp(c0, "pre", 3 ) == 0)       // restore partition info
   {
      if ((cc > 2) && !TxaOption('?'))
      {
         USHORT        first;
         USHORT        last;

         if (((s = strrchr( c2, '.')) == NULL) || // no file extension
                  (strstr(  c2, ".*") != NULL)  ) // or explicit wildcard
         {
            first = dfsParseDiskSpec( c1, &last); // request as a range
            if (last != 0)                      // valid single disk or range
            {
               for (index = first; (index <= last) && !TxAbort(); index++)
               {
                  if (dfsDiskAccessible( index))
                  {
                     rc = dfsRestorePartInfo( index, c2, c3);
                  }
               }
            }
            else
            {
               index = first;                   // the invalid disk number
               rc = DFS_PENDING;
            }
         }
         else                                   // restore to SAME disk
         {
            switch (c1[0])
            {
               case '.':                        // to current disk
                  index = (SINF->disknr) ? SINF->disknr : 1; // default current disk or 1
                  break;

               case '*':                        // to SAME disk
                  index = atoi( s+3);           // convert the x in .PDx
                  break;

               default:                         // to specified disk
                  index = (USHORT) atoi( c1);
                  break;
            }
            rc = DFS_PENDING;
         }
         if (rc == DFS_PENDING)                 // restore to one disk
         {
            if ((index != 0) && (index <= dfsPartitionableDisks()))
            {
               strcpy( s0, // prepare warning message for interactive single-disk restore
                 "This command will NOT prompt for each individual sector to restore,\n"
                 "so make REALLY sure this backup-set matches the current disk layout!\n\n"
                 "All partition tables, LVM info and ALL bootsectors (filesystem FORMAT)\n"
                 "will be restored to the exact situation when the backup was created ...\n");

               if (dfsDid2DiskType( index) != DFSD_VIRT) // don't mention when it IS a virtual
               {
                  strcat( s0, "\nYou may want to restore to a VIRTUAL disk first, to verify the layout.\n");
               }
               TxNamedMessage( ((!dfsa->batch) && TxaOptUnSet('c')), DFSC_RESTPD, " INFO: Restore partition info ", s0);

               rc = dfsRestorePartInfo( index, c2, c3);
            }
            else
            {
               TxPrint( "Disknr is invalid : %u\n", index);
               rc = DFS_VALUE_ERROR;
            }
         }
      }
      else
      {
         TxPrint("\nRestore file with partition-table "
                 "and LVM sectors info from specified file(s)\n");
         TxPrint("\n Usage: %s  disknr | . | *  filename  [types]\n\n", c0);
         TxPrint("   disknr    = disk-number, or symbolic value:\n"
                 "                    .  = Current disk\n"
                 "                    *  = ALL disks when no file extension present\n"
                 "                    *  = SAME disk as file with a file extension\n"
                 "   filename  = name for psave file(s) to be restored\n"
                 "   types     = only restore sectors of specified types:\n"
                 "                    r  = Master boot record\n"
                 "                    e  = Extended boot records\n"
                 "                    b  = Filesystem boot records\n"
                 "                    l  = LVM information sectors\n"
                 "                    s  = LVM signature   sectors\n\n");
         TxPrint("   options:  -c- = confirm file only, not each sector\n"
                 "             -l  = list sectors only (no write back)\n"
                 "             -v  = verbose display and prompting\n");
         TxPrint("\n To restore a file to a different disk, specify the NEW TARGET as disknr,\n"
                   " and the ORIGINAL SOURCE as file extention. Example (disk 1 file to disk 3):\n\n"
                   "         prestore 3 test.PD1\n");
      }
   }
   else if (strcasecmp(c0, "genpart") == 0)     // GENerate PARTion CR script
   {
      rc = dfseGenPartScript();
   }
   else if (strcasecmp(c0, "gpt") == 0)         // GPT specific command
   {
      if (!TxaOption('?'))
      {
         if (TxaOption( TXA_O_NAME))            // set name for specific partition
         {
            TxCopy(  cbi.pname, TxaOptStr( TXA_O_NAME, NULL, c2), GPT_PNAME_LEN +1);
            TRACES(("cbi.pname set to: '%s'\n", cbi.pname));
            switch (dfsParsePartSpec( c1, dfsGetDiskNumber( 'd', ".", "", FALSE), &p))
            {
               case FDSK_ANY:                   // all partitions
                  TxPrint("'%s -name' on all partitions not supported\n", c0);
                  break;

               case 0:                          // invalid partition
                  TxPrint( "Partition '%s' not found\n", c1);
                  break;

               default:                         // specific partition
                  if (p->tablenr == GPT_STYLE)
                  {
                     if (strlen( cbi.pname) > 0) // PID and name to set are OK
                     {
                        cbi.number    = p->id;
                        cbi.confirmed = dfsa->batch;
                        cbi.partnr    = p->partnr;
                        cbi.disknr    = p->disknr;
                        TRACES(("cbi.pname Now Is: '%s'\n", cbi.pname));
                        TRACES(("PID: %llu  partnr: %hu  name: '%s'\n", cbi.number, cbi.partnr, cbi.pname));
                        rc = dfsFdskGptSetName( &cbi);
                     }
                     else
                     {
                        TxPrint( "You MUST specify a name value!\n");
                        rc = DFS_VALUE_ERROR;
                     }
                  }
                  else
                  {
                     TxPrint( "Partition %s is NOT a GPT style partition\n", c1);
                     rc = DFS_VALUE_ERROR;
                  }
                  break;
            }
         }
         else                                   // disk-level operations
         {
            USHORT        first;
            USHORT        last;

            first = dfsParseDiskSpec( c1, &last); // request as a range
            if (last != 0)                      // valid single disk or range
            {
               for (index = first; index <= last; index++)
               {
                  if (dfsDiskAccessible( index))
                  {
                     rc = dfsGptExcecute( index, TxaOption( TXA_O_FIX));
                  }
               }
            }
            else
            {
               TxPrint( "Disknr is invalid : %u\n", first);
               rc = DFS_VALUE_ERROR;
            }
         }
      }
      else
      {
         TxPrint( "\nDisplay GPT header and partition table array, or fix a damaged GPT setup\n\n");
         TxPrint( " Usage: %s  [[disknr | . | *] [-fix]] | [PID  -name:Pname]\n\n"
                  "  disknr      = disk-number, '.' for current or '*' for all disks; default is '.'\n"
                  "  -fix        = Fix on-disk GPT structures by writing back the retrieved\n"
                  "                in-memory ones, entries sorted on starting sector number\n"
                  "                and with all CRC values recalculated.\n\n"
                  "  PID         = Partition-ID for GPT partition, to set the name for\n"
                  "  -name:Pname = New name for the GPT-partition (quote if any spaces)\n", c0);
      }
   }
   else if (strcasecmp(c0, "gpt2mbr") == 0)     // Convert GPT to MBR style, prompted
   {
      if (!TxaOption('?'))
      {
         ml = dfsPartitionableDisks();
         if (((index = dfsParseDiskSpec(c1, NULL)) != 0) && (index != FDSK_ANY) && (index <= ml))
         {
            rc = dfsSelectDisk( index, FALSE, FALSE); // Select disk quietly
            if (rc == NO_ERROR)
            {
               if (!TxaOptUnSet('p'))
               {
                  if ((dfsa->batch) || TxConfirm( 5038,
                       "Convert disk %hu to MBR-style, "
                       "removing ALL current GPT partitions ? [Y/N] : ", index))
                  {
                     if (!TxaOptUnSet('m'))
                     {
                        sprintf( dc, "newmbr -B -c");
                        if (TxaOption('N'))
                        {
                           strcat( dc, " -N");
                        }
                        rc = dfsMultiCommand( dc, 0, FALSE, FALSE, TRUE);
                     }
                     if (rc == NO_ERROR)
                     {
                        sprintf( dc, "wipe -B z 1 1");
                        rc = dfsMultiCommand( dc, 0, FALSE, FALSE, TRUE);
                     }
                  }
                  else
                  {
                     rc = DFS_USER_ABORT;
                  }
               }
               if ((!TxaOptUnSet('a')) && (rc == NO_ERROR))
               {
                  if ((dfsa->batch && TxaOption('a')) || TxConfirm( 5039,
                       "Clearing the alternate GPT-header at the END of the disk\n"
                       "will make recovery of GPT partitions almost impossible\n\n"
                       "Clear the alternate GPT-header now too ? [Y/N] : "))
                  {
                     sn = dfsGetLogicalSize() - 1;

                     sprintf( dc, "wipe -B z %llu 1", sn);
                     rc = dfsMultiCommand( dc, 0, FALSE, FALSE, TRUE);
                  }
               }
            }
            if (rc == NO_ERROR)                 // display resulting state
            {
               sprintf( dc, "part -r -d");
               rc = dfsMultiCommand( dc, 0, FALSE, FALSE, TRUE);
            }
         }
         else
         {
            TxPrint( "Disknr is invalid : %u\n", index);
            rc = DFS_VALUE_ERROR;
         }
      }
      else
      {
         TxPrint( "\nConvert a (possibly) GPT style disk to the MBR style, removing GPT partitions\n\n");
         TxPrint( " Usage: %s  [disknr | .] [options]\n\n"
                  "  disknr    = disk-number, '.' for current disk; default is '.'\n"
                  "  -a        = Clear the alternate GPT-header at the END of the disk too\n"
                  "              (will make automatic recovery to GPT much harder!)\n"
                  "  -a-       = Do NOT clear the alternate GPT header at end of disk (default)\n"
                  "  -p-       = Do NOT clear the primary-GPT header and the MBR partitions\n"
                  "  -m-       = Do NOT clear the MBR partitions, just the primary-GPT header\n"
                  "  -N        = When (re)creating the MBR, do NOT insert any code, table only\n", c0);
      }
   }
   else if (strcasecmp(c0, "mbr2gpt") == 0)     // Convert MBR to GPT style, prompted
   {
      if (!TxaOption('?'))
      {
         ml = dfsPartitionableDisks();
         if (((index = dfsParseDiskSpec(c1, NULL)) != 0) && (index != FDSK_ANY) && (index <= ml))
         {
            rc = dfsSelectDisk( index, FALSE, FALSE); // Select disk quietly
            if (rc == NO_ERROR)
            {
               if ((dfsa->batch) || TxConfirm( 5038,
                    "Converting a disk to GPT may recover existing GPT partitions too!\n\n"
                    "Convert disk %hu to GPT and remove ALL current MBR partitions ? [Y/N] : ", index))
               {
                  sprintf( dc, "newmbr -B -c");
                  if (TxaOption('N'))
                  {
                     strcat( dc, " -N");
                  }
                  rc = dfsMultiCommand( dc, 0, FALSE, FALSE, TRUE);
                  if (rc == NO_ERROR)
                  {
                     sprintf( dc, "set gpt off#part -r -d#cr -B gpt#set gpt on#part -r -d");
                     rc = dfsMultiCommand( dc, 0, FALSE, FALSE, TRUE);
                  }
               }
               else
               {
                  rc = DFS_USER_ABORT;
               }
            }
         }
         else
         {
            TxPrint( "Disknr is invalid : %u\n", index);
            rc = DFS_VALUE_ERROR;
         }
      }
      else
      {
         TxPrint( "\nConvert a (possibly) MBR style disk to the GPT style, removing MBR partitions\n\n");
         TxPrint( " Usage: %s  [disknr | .] [options]\n\n"
                  "  disknr    = disk-number, '.' for current disk; default is '.'\n\n"
                  "  -N        = When (re)creating the MBR, do NOT insert any code, table only\n", c0);
      }
   }
   else if (strncasecmp(c0, "lvmclean", 8 ) == 0) // Clean obsolete LVM DLAT sectors
   {
      if ((TxaOption('?')) || (cc == 1))
      {
         TxPrint( "\nClean obsolete LVM DLAT sectors in MBR track area (1..254)\n");
         TxPrint("\n Usage: %s  disknr | . | *  [-a]\n\n", c0);
         TxPrint("   disknr = disk-number, '.' for current or '*' for all disks\n"
                 "   -a     = ALL, do NOT exclude DLAT for current disk geometry\n", c0);
      }
      else
      {
         rc = dfsFdskClearLvmSectors( dfsParseDiskSpec( c1, NULL), (TxaOption('a')) ? FALSE : TRUE);
      }
   }
   else if (strcasecmp(c0, "lvmredo") == 0)        // Revive LVM DLAT info on disk
   {
      rc = dfseLvmRedoDlat();
   }
   else if (strcasecmp(c0, "lvmshow") == 0)        // show LVM info for part
   {
      if (TxaOption('?'))
      {
         TxPrint( "\nShow LVM information for one or all partitions\n\n");
         TxPrint( " Usage: %s  pid | letter | *\n\n"
                  "  pid       = partition id as shown in PART 1st column\n"
                  "  letter    = driveletter assigned to a  partition\n"
                  "  * or 0    = show for ALL partitions\n", c0);
      }
      else
      {
         if ((c1[0] == '*') || (c1[0] == '0'))  // All partitions
         {
            rc = dfsMultiCommand( "plist lvm", 0, TRUE, FALSE, TRUE);
         }
         else                                   // list current/specified only
         {
            sprintf( dc, "lvm -list -R %s", c1);
            rc = dfsMultiCommand( dc, 0, FALSE, FALSE, TRUE);
         }
      }
   }
   else if (strncasecmp(c0, "lvm", 3) == 0)        // set/show LVM information
   {
      BOOL          ok = TRUE;
      TXA_OPTION   *opt;

      ok &= TxaOptMutEx(((cc > 1)),         "cG", "if a partition is specified!", 0);
      ok &= TxaOptMutEx(((TxaOption('C'))), "cG", "if the -C option is specified!", 0);

      if ((!TxaOption('?')) && (ok))            // no help, or param error ?
      {
         DFST_HANDLE  cst = dfstGetDefaultStore();
         BOOL         dlg = dfsa->dialogs;

         if (TxaOption( TXA_O_LIST))            // list values only (lvmshow)
         {
            TxScreenState( DEVICE_OFF);         // no regular output wanted
            dfsa->dialogs = FALSE;              // suppress dialog popups
         }
         ml         = dfsGetDiskNumber( 'd', ".", "", FALSE); // disknrs, current/ANY,
         cbi.disknr = dfsGetDiskNumber( 'd', "*", "", FALSE); // parse BEFORE set store!
         TRACES(( "1st disknr-set as: %hu\n", cbi.disknr));

         dfstSetDefaultStore( DFST_SYSTEM);     // select system store

         strcpy( s1, c1);                       // param 1 is partition
         if ( TxaOption('c') ||                 // CRC update         whole disk (only)
              TxaOption('G') ||                 // GEO update         whole disk (only)
            ((TxaOption('C') ||                 // BBR erase non 0x35 whole disk/1-partition
              TxaOption('n')) && (cc <= 1)))    // Name update        whole disk/1-partition
         {
            if (TxaOption('n'))                 // set disk name -n
            {
               if ((opt = TxaOptValue('n')) != NULL) // disk name
               {
                  if ((opt->type == TXA_STRING)  &&
                      (strlen(opt->value.string) != 0))
                  {
                     strncpy( cbi.vname, opt->value.string, BMSYS_LONG);
                     cbi.vname[ BMSYS_LONG -1] = 0;
                  }
                  else                          // request prompt for name
                  {
                     strcpy( cbi.vname, "-");   // signal name change
                     strcpy( cbi.pname, "-");   // and request prompting
                     cbi.cbOptNum2 = 0;         // last disk prompted

                     TRACES(("LVM disknr %hu prompt %llu %s %s\n",
                              cbi.disknr, cbi.cbOptNum2, cbi.pname, cbi.vname));
                  }
               }
            }
            if (TxaOption('c') || TxaOption('n') || TxaOption('C'))
            {
               cbi.option = TRUE;               // include signature sectors
            }
            if (TxaOption('G'))                 // Sync LVM GEO with L-Geo
            {
               cbi.cbOptNum1 = 1;               // signal GEO update wanted
            }
            else if (TxaOption('C'))            // Clear LVM-signature sector
            {
               cbi.doAllBBR = TRUE;             // include non-LVM (0x35) type
            }
            rc = dfsIteratePlvm( dfsFdskLvmSetInfo, &cbi);

            dfsReadDiskInfo( FDSK_QUIET);       // re-read all partition info
         }
         else                                   // actions on partitions
         {
            ULONG  listcode  = TXDID_OK;
            BOOL   multiple_partitions  = FALSE;   //- auto loop over partitions
            USHORT lastPrimaryLvmDiskNr = 0;       //- last disknr handled

            #if defined (USEWINDOWING)
               TXRECT where  = {18,0,0,0};      // fixed position

               if (TxaOption('P') || TxaOption('!')) // explicit prompting
               {
                  listcode = txwListBox( TXHWND_DESKTOP, TXHWND_DESKTOP, &where,
                           " Select partition to start LVM-edit with ", "",
                             5401, TXLB_MOVEABLE,
                             cMenuTextStand, cMenuBorder_top, // same color as menu
                             dfsa->slPartOne);

                  strcpy( s1, "*");
                  if (listcode > TXDID_MAX)     // return list selection
                  {                             // plus current item base value
                     sprintf( s1, "%u", listcode - TXDID_MAX); // calculate pid
                  }
               }
            #endif                              // USEWINDOWING

            strcpy( dc, "");                    // auto post-lvm command empty
            if (listcode != TXDID_CANCEL)       // not escaped ...
            {
               if (TxaOption('d') && (strlen(s1) == 0)) // default all partitions
               {
                  strcpy( s1, "*");
               }
               switch (dfsParsePartSpec( s1, ml, &p))
               {
                  case 0:                       // invalid partition
                     TxPrint( "\nPartition '%s' not found\n", s1);
                     rc = DFS_NOT_FOUND;
                     break;

                  case FDSK_ANY:                // all partitions, start with 1
                     multiple_partitions = TRUE;
                     if (TxaOption('d'))        // disk number specified
                     {
                        p = NULL;               // start with NO partition!
                        if ((d = dfsGetDiskInfo( cbi.disknr)) != NULL)
                        {
                           if (d->firstpart)
                           {
                              p = dfsGetPartInfo( d->firstpart);
                           }
                        }
                     }
                     if (p == NULL)             // no partitions at all!
                     {                          // so NOTHING to be done!
                        break;
                     }
                  default:                      // specific partition or '*'
                     do                         // allow partition looping
                     {
                        d = dfsGetDiskInfo( p->disknr);
                        if ((d != NULL) && (d->pStyle[0] != DFSP_MBR)) // allow MBR-style only!
                        {
                           TxPrint( "\nPartition %02.2hu is located on a '%s' style disk, not 'MBR', "
                                    "incompatible with OS/2 LVM!\n", p->id, d->pStyle);
                        }
                        else if (d != NULL)     // MBR style disk, perform LVM operation
                        {
                           if ((opt = TxaOptValue('v')) != NULL) // volume name
                           {
                              if ((opt->type == TXA_STRING)  &&
                                  (strlen(opt->value.string) != 0))
                              {
                                 strncpy( cbi.vname, opt->value.string, BMSYS_LONG);
                                 cbi.vname[ BMSYS_LONG -1] = 0;
                              }
                              else if (opt->value.number == 0) // -v- or -v:0
                              {
                                 strcpy( cbi.vname, "-"); // remove volume name
                              }
                           }
                           if ((opt = TxaOptValue('p')) != NULL) // partition name
                           {
                              if ((opt->type == TXA_STRING)  &&
                                  (strlen(opt->value.string) != 0))
                              {
                                 strncpy( cbi.pname, opt->value.string, BMSYS_LONG);
                                 cbi.pname[ BMSYS_LONG -1] = 0;
                              }
                              else if (opt->value.number == 0) // -p- or -p:0
                              {
                                 strcpy( cbi.pname, "-"); // remove partition name
                              }
                           }
                           if ((opt = TxaOptValue('l')) != NULL) // drive letter
                           {
                              if ((opt->type == TXA_STRING)  &&
                                  (strlen(opt->value.string) != 0))
                              {
                                 cbi.bname[0] = (char) toupper( opt->value.string[0]);
                              }
                              else if (opt->value.number == 0) // -l- or -l:0
                              {
                                 cbi.bname[0] = '-'; // remove drive letter
                              }
                              cbi.bname[1] = 0;
                           }
                           cbi.ntype = (BYTE) 0xff;
                           if (TxaOptSet('m'))
                           {
                              cbi.ntype = (BYTE) TxaOption('m');
                           }
                           if (TxaOptSet(TXA_O_MENU)) // -menu as alias for -m
                           {
                              cbi.ntype = (BYTE) TxaOption(TXA_O_MENU);
                           }
                           if (TxaOptSet('i'))
                           {
                              cbi.stype = (BYTE) TxaOption('i');
                           }
                           else
                           {
                              cbi.stype = (BYTE) 0xff;
                           }
                           cbi.cbOption1  = TxaOption('s');  //- sync size with partition
                           cbi.cbOption2  = TxaOption('D');  //- Delete DLAT-entry first
                           cbi.cbOption3  = TxaOption('C');  //- Clear LVM-signature sector
                                                             //- forcing Compatibility mode
                           cbi.doAllBBR   = cbi.cbOption3;   //- and allow for non 0x35 type
                           if (ok)
                           {
                              cbi.more    = p;  // attach partition info
                              cbi.disknr  = p->disknr; // only on same disk
                              cbi.abortNc = TRUE; // stop on NO_CHANGE
                              cbi.option  = TRUE; // signature sectors too

                              TxaGetArgString( TXA_CUR, TXA_OPT, TXA_OPT, TXMAXLN, cbi.string);

                              rc = dfsProcessPlvm( dfsFdskLvmSetInfo, &cbi);
                              if (rc == NO_ERROR)
                              {
                                 TxPrint( "\nLVM info (DLAT) for partition %02.2hu ", p->id);
                                 if (cbi.result) // successful update
                                 {
                                    TxPrint( "successfully updated\n");

                                    #if defined (DEV32)
                                       TRACES(("cbi.lvmSync now : %s  syncName:'%s' lvm.letter: %d\n",
                                               (cbi.lvmSync) ? "TRUE" : "FALSE", cbi.syncName, cbi.syncLetter));
                                       if (cbi.lvmSync)
                                       {
                                          dfsLvmSyncEngine( TRUE, FALSE, cbi.syncName, cbi.syncLetter);
                                       }
                                    #endif

                                    //- prepare automatic creation primary LVM DLAT (by specifying a diskname)
                                    if ((!TxaOption( TXA_O_LIST)) &&
                                        (cbi.disknr != lastPrimaryLvmDiskNr))
                                    {
                                       if ((d = dfsGetDiskInfo( cbi.disknr)) != NULL)
                                       {
                                          if (d->lvmPartitions == 0) // no LVM info present yet
                                          {
                                             if (strlen( dc) != 0)
                                             {
                                                strcat( dc, "#"); // need a separator
                                             }
                                             sprintf( s0, "lvm -d:%hu -n", cbi.disknr);
                                             strcat(  dc, s0);
                                          }
                                       }
                                       lastPrimaryLvmDiskNr = cbi.disknr;
                                    }
                                 }
                                 else           // LVM info for part not found
                                 {
                                    TxPrint( "not found!\n");
                                 }
                              }
                              nav.down = p->partPsn + p->geoSecs -1; // LVM-info sector
                              nav.offset = (p->partent.PartitionType == DFS_P_WARP_LVM);
                           }
                           else
                           {
                              TxShowTxt( cmd_lvm);
                              rc = DFS_CMD_FAILED;
                           }
                           if ((nr = cbi.cbOptNum1) != DFSD_NONE)
                           {
                              memset( &cbi, 0, sizeof(cbi)); // reset callback info
                              cbi.cbOptNum1 = nr; // but keep looping info
                           }
                           else if (multiple_partitions)
                           {
                              nr = DFSDLG_NEXT; // default on <Enter>
                              memset( &cbi, 0, sizeof(cbi)); // reset callback info
                              cbi.cbOptNum1 = nr; // but keep looping info
                           }
                           switch (nr)
                           {
                              case DFSDLG_NEXT: // next partition
                                 if ((d == NULL) || (p->id < d->lastpart))
                                 {
                                    p = dfsGetPartInfo( p->id +1);
                                 }
                                 else
                                 {
                                    p = NULL;
                                 }
                                 break;

                              case DFSDLG_PREV: // previous partition
                                 if ((d == NULL) || (p->id > d->firstpart))
                                 {
                                    p = dfsGetPartInfo( p->id -1);
                                 }
                                 else
                                 {
                                    p = NULL;
                                 }
                                 break;

                           #if defined (USEWINDOWING)
                              case DFSDLG_EDIT: // Edit LVM-info sector
                                 dfsHexObjectEditor( p->partPsn + p->geoSecs -1, 0,  0, 0, NULL, DFSH_HEX_VIEW, 0, 0, 0);
                                 break;

                              case DFSDLG_REQ5: // Edit partition bootsector
                              case DFSDLG_REQ6: // First sector of partition
                                 dfsHexObjectEditor( p->basePsn, 0,  0, 0, NULL, DFSH_HEX_VIEW, 0, 0, 0);
                                 break;

                              case DFSDLG_REQ7: // Edit LVM-BBR sector
                              case DFSDLG_REQ8: // Last sector of partition
                                 dfsHexObjectEditor( p->lastPsn, 0,  0, 0, NULL, DFSH_HEX_VIEW, 0, 0, 0);
                                 break;
                           #endif

                              default:
                                 break;
                           }
                           if (p == NULL)       // beyond last or first now
                           {
                              cbi.cbOptNum1 = 0; // quit auto looping
                           }
                        }
                     } while ((cbi.cbOptNum1 != 0) && // looping over partitions
                              (!TxAbort())         && // not Canceled
                             ((rc == NO_ERROR)     || // and RC still acceptable
                              (rc == DFS_NO_CHANGE)));

                     if (strlen( dc) != 0)      // post-LVM command created?
                     {
                        TxPrint( "Automatic creation of MBR DLAT using: %s%s%s\n", CBG, dc, CNN);
                        TxaReParseCommand( dc);
                        rc = DFS_PENDING;
                     }
                     break;
               }
            }
         }
         if ((rc == NO_ERROR) && (cbi.result))  // successful update
         {
            dfsFdskSingleAutoShow( cbi.disknr);
         }
         if (TxaOption( TXA_O_LIST))            // list values only (lvmshow)
         {
            TxScreenState( DEVICE_ON);          // restore regular output
            if ((rc == NO_ERROR) || (rc == DFS_NO_CHANGE))
            {                                   // LVM-info + BBR or just info
               rc = dfsMultiCommand((nav.offset) ? "d#u" : "d", 0, FALSE, FALSE, TRUE);
            }
            dfsa->dialogs = dlg;                // restore dialog popup state
         }
         dfstRestoreDefaultStore( cst);         // reselect current store
         TxCancelAbort();                       // might have aborted ...
      }
      else
      {
         TxShowTxt( cmd_lvm);
      }
   }
   else if ((strncasecmp( c0, "attach", 6 ) == 0) || // Attach partitionable medium
            (strncasecmp( c0, "virt",   4 ) == 0)  ) // for backward compatibility
   {
      if (!TxaOption('?') && TxaOptMutEx(  FALSE,   "piv", "", NULL)
                          && TxaOptMutEx( (cc > 1), "pi",  "", NULL))
      {
         if      (TxaOption('p'))               // Physical disk
         {
#if defined (UNIX)
            TxaOptAsString( 'p', TXMAXLN, s1);
            if (dfsUxDeviceExists( s1, TRUE))
            {
               rc = dfsAddMediaMapping( DFSD_PHYS, 0, 0, s1, c1, TRUE);
            }
            else
            {
               TxPrint( "\nDevice '%s' does not exist or is not accessible!\n", s1);
            }
#else
            rc = dfsAddMediaMapping( DFSD_PHYS, TxaOptNum('p', "PhysDisk", 1), 0, "", c1, TRUE);
#endif
         }
         else if (TxaOption('i'))               // Raw/Compressed/Vbox Image file
         {
            TxaOptAsString( 'i', TXMAXLN, s1);
            rc = dfsmCreateImageDisk(  s1);
         }
         else                                   // Memory disk, is default
         {
            rc = dfsmCreateMemoryDisk( cc, c1, c2, c3, c4);
         }
         if (rc == NO_ERROR)
         {
            dfsReadDiskInfo( FDSK_QUIET);       // re-read diskinfo

            if (!TxaOptUnSet('o'))              // if not -o-
            {
               index = dfsGetDiskCount( FALSE); // show & open LAST disk
               sprintf( dc, "part -g -d:%hu#disk -q -w- %s %hu",         index,
                       (TxaOptUnSet(DFS_O_GEOCALC)) ? " -geocalc-" : "", index);
               dfsMultiCommand( dc, 0, FALSE, FALSE, TRUE);
            }
            else
            {
               dfsFdskSingleAutoShow( index);
            }
         }
      }
      else
      {
         TxPrint( "\nAttach a new partitionable medium (DFSee 'disk' object)\n\n");
         TxPrint( " Usage: %s  [option  [descr]] | PDxFile | size  [Heads  [Sects [Bps]]]\n\n"
                  "  option    = Type and specification for medium to attach:\n\n"
                  "              -i:'file' : Image file, RAW, Compressed (.IMZ), or .VDI\n"
                  "              -m        : Memory disk (DFSee internal)\n"
#if defined (UNIX)
                  "              -p:device : Disk or other devicename\n"
#else
                  "              -p:number : Physical disk number, 1..n\n"
#endif
                "\n              -o-       : Do NOT open the newly attached disk\n\n", c0);

         TxPrint( "  descr     = Description, valid with the -p option only)\n"
                  "  PDxFile   = PDx filename to be used as template for memory disk\n"
                  "  size      = Memory disk size in MiB, or in cylinders for nn,c\n"
                  "  Heads     = Number of heads   to use for geometry, default 255\n"
                  "  Sects     = Number of sectors to use for geometry, default 63\n"
                  "  Bps       = Bytes per sector, 512 * (power of 2),  default 512\n");
      }
   }
   else if (strncasecmp(c0, "detach", 6 ) == 0)       // Detach mapped medium
   {
      if (((cc > 1) || (TxaOption('l')))  && !TxaOption('?'))
      {
         USHORT disk = (TxaOption('l')) ? dfsGetDiskCount(FALSE) : atoi(c1);

         TxPrint( "\n");
         rc = dfsmDetachMedium( disk, TRUE);
         if (rc == NO_ERROR)
         {
            dfsReadDiskInfo( FDSK_QUIET);       // re-read diskinfo
         }
      }
      else
      {
         TxPrint( "Detach mapped partitionable medium\n\n");
         TxPrint( " Usage: %s  [disk | * | -l]\n\n"
                  "  disk = DFSee id of DISK to detach\n"
                  "  *    = Detach ALL disks\n"
                  "  -l   = Detach LAST disk\n", c0);
      }
   }
   else if (strncasecmp(c0, "vcu", 3 ) == 0)    // Volume Conversion, the way
   {                                            // it was supposed to be :-)
      USHORT           disk;
      BOOL             ok = TRUE;

      ok &= TxaOptMutEx((cc > 1), "d", "if a (disk) parameter is specified!", 0);

      if (ok && (c1[0] != '?') && !TxaOption('?'))
      {
         if (cc > 1)
         {
            disk = dfsParseDiskSpec( c1, NULL);
         }
         else if (TxaOption('d'))
         {
            disk = TxaOptNum('d', NULL, (SINF->disknr) ? SINF->disknr : 1);
         }
         else
         {
            disk = FDSK_ANY;
         }
         if (disk == FDSK_ANY)
         {
            strcpy( s1, "*");
         }
         else
         {
            d = dfsGetDiskInfo( disk);          // keep disk info available
            sprintf( s1, "%hu", disk);
         }
         if ((dfsa->batch) || TxConfirm( 5021,
              "Delete existing and create new default LVM "
              "information for %s %s ? [Y/N] : ",
               (disk == FDSK_ANY) ? "ALL"   : "disk",
               (disk == FDSK_ANY) ? "disks" :  s1))
         {
            S_LVINIT  *saved = dfsLvmVcuPreservation();

            strcpy(  s0, (dfsa->verbosity >= TXAO_VERBOSE) ? "" : " -Q"); // -Q unless verbose
            sprintf( dc, "pclear -Q -B %s lvm#lvm -B%s -V -d:%s *#lvm -B%s -n -d:%s",
                                       s1,          s0,      s1,        s0,      s1);

            strcpy( s1,  fdsk->ShowCmd);        // save autoshow-cmd
            strcpy( fdsk->ShowCmd, "-");        // and switch it off

            TxPrint( "\nDelete old and create new default LVM info ...\n");

            if ((rc = dfsMultiCommand( dc, 0, TRUE, FALSE, TRUE)) == NO_ERROR)
            {
               if (saved != NULL)               // any saved partition info?
               {
                  USHORT  pi;
                  USHORT  first = (disk != FDSK_ANY) ? d->firstpart : 1;
                  USHORT  last  = (disk != FDSK_ANY) ? d->lastpart  :
                                                       dfsPartitions();

                  TxPrint( "\nApply preserved driveletters and names ...\n");

                  for (pi = first; (pi <= last) && !TxAbort(); pi++)
                  {
                     if (saved[pi].letter     ||
                         saved[pi].OnBmMenu   ||
                         saved[pi].VoluName[0] )
                     {
                        sprintf( dc, "lvm -B %s %2.2hu", s0, pi);
                        if (saved[pi].letter)
                        {
                           sprintf( s0, " -letter:%c", saved[pi].letter);
                           strcat(  dc, s0);
                        }
                        if (saved[pi].VoluName[0] != 0)
                        {
                           sprintf( s0, " -v:'%s'", saved[pi].VoluName);
                           strcat(  dc, s0);
                        }
                        if (saved[pi].OnBmMenu)
                        {
                           strcat(  dc, " -menu");
                        }
                        dfsMultiCommand( dc, 0, TRUE, FALSE, TRUE);
                     }
                  }
               }
            }
            strcpy( fdsk->ShowCmd, s1);         // restore autoshow cmd
            fdsk->AutoShow = TRUE;              // indicate changes, all disks

            TxFreeMem( saved);                  // free temporary info
         }
         else
         {
            rc = DFS_NO_CHANGE;
         }
      }
      else
      {
         TxPrint( "Perform 'Volume Conversion' adding LVM-information to partitions\n\n");
         TxPrint( " Usage: %s disk | . | * | -d[:nr]  [-v]\n\n"
                  "  disk    = disk-number, '.' for current or '*' for all disks\n"
                  "  -d[:nr] = disk-number\n"
                  "  -v      = verbose operation\n", c0);
      }
   }
   else if (strcasecmp(c0, "ntsign"   ) == 0)
   {
      if (!TxaOption('?'))
      {
         USHORT        first;
         USHORT        last;

         first = dfsParseDiskSpec( c1, &last);  // request as a range
         if (last != 0)                         // valid single disk or range
         {
            for (index = first; index <= last; index++)
            {
               if (dfsDiskAccessible( index))
               {
                  rc = dfsFdskNtSignature( index, c2);
               }
            }
         }
         else
         {
            TxPrint( "Disknr is invalid : %u\n", first);
            rc = DFS_VALUE_ERROR;
         }
      }
      else
      {
         TxPrint( "Set a disk signature used by windows-NT/XP for identification\n\n");
         TxPrint( " Usage: %s disk | . | * |  [sigvalue] [-query][-v-]\n\n"
                  "  disk      = disk-number, '.' for current or '*' for all disks\n"
                  "  sigvalue  = signature value, hexadecimal string length 8\n"
                  "  -query    = display only, no update\n"
                  "  -v-       = query output raw HEX number only\n", c0);
      }
   }
   else if (strncasecmp(c0, "nt", 2 ) == 0)     // Any other command with 'nt'
   {                                            // is an NT-4 specific command
   #if defined (WIN32)                          // any Windows NT-like version
      if (dfsFdskWarningW2KXP( c0))             // warn on non NT-4 systems
      {
         if (strcasecmp(c0, "ntcdr") == 0)      // NT Set CDROM letter
         {
            if (cc > 2)                         // two parameters required!
            {
               rc = ntregModifyCDRomLetter( (USHORT) (atoi(c1) -1),
                                              TxStrToUpper(c2));
               if (rc == NO_ERROR)
               {
                  TxPrint( "\nDriveletter for CDROM %s set to '%s%c:%s'\n",
                              c1, CBG, toupper(c2[0]), CNN);
                  ntregShowDriveLetters();
               }
            }
            else
            {
               TxPrint("\n Usage: %s  CDROM-nr  letter | -\n", c0);
            }
         }
         else if (strcasecmp(c0, "ntmap") == 0)    // NT disk registry query
         {
            switch (toupper(c1[0]))
            {
               case 'C':                        // create registry diskkey
               case 'S':                        // and sync DosDevice mapping
                  rc = ntregSyncDevMapDiskKey();
                  break;

               default:                         // display registry key
                  switch (toupper(c1[0]))
                  {
                     case 'V': rc = ntregShowDiskkey();                  break;
                     default:  rc = ntregShowDriveLetters();             break;
                  }
                  break;
            }
         }
         else if (strcasecmp(c0, "ntdev") == 0)    // NT Dos device query
         {
            ntregShowDosDevice( c1, c2);
         }
         else if (strcasecmp(c0, "ntddd") == 0)    // DefineDosDevice test
         {
            if (cc > 2)
            {
               if (DefineDosDevice( (DWORD) atol(c1), c2, c3))
               {
                  TxPrint( "\nSuccess!\n");
               }
               else
               {
                  TxPrint( "\nFailed, error : %s\n", txNtLastError());
               }
            }
            else
            {
               TxPrint("\n Usage: %s  flags device target\n", c0);
            }
         }
      }
   #else
      TxPrint( "%s command not available in this DFSee version\n", c0);
   #endif
   }
   else if (strcasecmp(c0, "disklist" ) == 0)   // descriptive list, to STDOUT too
   {
      if (TxaOption('?'))                       // explicit help request
      {
         TxPrint("\nDisplay list of disks present with size and vendor description\n\n");
         TxPrint(" Usage:  %s  [-d:count | -d-]\n\n", c0);
         TxPrint("   -d:count = Only attach the first 'count' physical disks\n"
                 "   -d-      = Don't attach any, same as '-d:0', empty disk list\n\n"
                 " Note: Includes a 'media -update' prior to displaying the list\n\n");
      }
      else
      {
         DFSFNCALL(dfsa->FsClose,0,0,NULL,NULL); // FS specific handler, close
         dfstInitStoreInfo();                   // close all open stores
         dfsmDetachMedium( DFSM_UNMOUNT_ALL, FALSE); // quietly detach
         dfsInitMediaMap( TxaOptNum( 'd', NULL, DFS_MAX_DISK));
         dfsReadDiskInfo( FDSK_QUIET);          // re-read all partition info
         dfsFdskInit();                         // Well defined startup mode

         ml = dfsPartitionableDisks();

         dfsGuiStdMessage( "\nNr   Style    Unix-device  Disk-name/description Disk size           Status   ");
         dfsGuiStdMessage(   "==   ======== ============ ===================== ==================  =========");

         for (index = 1; index <= ml; index++)
         {
            if (dfsDiskAccessible( index))
            {
               if ((d = dfsGetDiskInfo( index)) != NULL)
               {
                  double  sizeGiB = ((double) DFSECT2MIB( d->sectors, d->bpsector)) / 1024.0;

                  sprintf( s0, "%2hu = %s disk %-12.12s %-21.21s Size: %8.2lf GiB  %s",
                           index, d->pStyle, d->UnixDeviceDisk, d->DiskName, sizeGiB,
                                                               (d->Removable) ? "Removable" : "");
                  dfsGuiStdMessage( s0);

                  dfsGetDiskIdentification( index, d->UnixDeviceDisk, 0, s1);
                  TxStrip( s1, s1, ' ', ' ');   // strip leading/trailing spaces
                  if (strlen( s1) != 0)
                  {
                     sprintf( s0, "     %-42.42s", s1); // indented 2nd clipped description line
                     dfsGuiStdMessage( s0);
                  }
               }
            }
         }
      }
   }
   else if ((strcasecmp(c0, "disk"     ) == 0) ||
            (strcasecmp(c0, "walk"     ) == 0) )
   {
      if (TxaOption('?'))
      {
         TxShowTxt( cmd_walk);                  // give usage
      }
      else
      {
         index = (USHORT) atoi(c1);             // 0 if not specified or '.'
         if ((index == 0) && (c1[0] != '0'))
         {
            index = (SINF->disknr) ? SINF->disknr : 1;
         }
         //- if (dfsa->FsModeId != DFS_FS_FDISK)    // need to switch to FDISK mode ?
         {
            rc = DFSFNCALL(dfsa->FsClose,0,0,NULL,NULL); // Close filesystem
            SINF->p      = NULL;
            SINF->disknr = index;
            SINF->partid = 0;
            strcpy(SINF->drive, "--");
            strcpy(SINF->afsys, "FDISK");
            rc = dfsFdskInit();
         }
         if (rc == NO_ERROR)
         {
            rc = dfsSelectDisk( index, TRUE, TxaOption('r')); // Select disk, reset
            if (rc == NO_ERROR)
            {
               if (TxaOptSet('R'))
               {
                  dfstSetReadOnly( DFSTORE, TRUE, TxaOption('R'));
               }
               if (strcasecmp(c0, "disk"  ) == 0)
               {
                  DFSDISKINFO *d;

                  if ((d = dfsGetDiskInfo( index)) != NULL)
                  {
                     dfsShowDiskWarnings( d, DFS_F_M_SHOWALL); // consider all MAJOR
                     if (d->flags & DFS_F_FSYSTEMONLY)
                     {
                        strcpy(SINF->afsys, ""); // no default preference
                        rc = dfsInitFileSystem();
                     }
                     else                       // regular disk select/display
                     {
                        if (dfsa->verbosity > TXAO_QUIET)
                        {
                           //- Geo most likely displayed by SelectDisk already!
                           dfstDiskGeometry( DFSTORE, 0, 0, 0, 0, 0, FALSE, TRUE);
                           rc = dfsReadAnDisplay(0, 0, &st); // start with the MBR at 0
                        }
                     }
                  }
               }
               else                             // walk MBR/EBR chain
               {
                  ULN64    backlink = 0;

                  if (st == ST_MCDDM)           // Macintosh driver record
                  {
                     st = ST_MASTR;             // allow walk PC-style stuff
                  }
                  rc = dfsReadAnDisplay(0, 0, &st); // start with the MBR at 0
                  while ((rc == NO_ERROR) && (nav.down != 0) && !TxAbort())
                  {
                     if ((st == ST_MASTR) || (st == ST_EXTBR)) // Last MBR/EBR
                     {
                        dfsX10("\nLink to next  EBR : ", nav.down, CBG, "\n");

                        if (nav.down <= nav.this)
                        {
                           if (nav.down == nav.this) // looping EBR
                           {
                              TxPrint( "\nERROR: links to same EBR sector, forming illegal logical loop!\n");
                              rc = DFS_BAD_STRUCTURE;
                           }
                           else                 // backward linking!
                           {
                              if (nav.down == backlink) // same link 2nd time
                              {
                                 TxPrint( "\nERROR: links BACK to earlier EBR, forming illegal logical loop!\n");
                                 rc = DFS_BAD_STRUCTURE;
                              }
                              else
                              {
                                 TxPrint( "\nWARNING: links BACK to an earlier EBR sector, illegal order!\n");
                                 if (backlink != 0)
                                 {
                                    if ((dfsa->batch) || (TxConfirm( 5016,
                                        "Next EBR at 0x0%llX is a BACKWARD link "
                                        "from the current EBR 0x0%llX\n\n"
                                        "Do you want to follow the link anyway, "
                                        "risking a logical loop ? [Y/N] : ",
                                         nav.down, nav.this)))
                                    {
                                       rc = DFS_BAD_STRUCTURE;
                                    }
                                 }
                                 else           // remember the first backlink
                                 {
                                    backlink = nav.down;
                                 }
                              }
                           }
                        }
                        if (rc == NO_ERROR)     // fine sofar, display next
                        {
                           st = 0;              // use autodetect
                           rc = dfsReadAnDisplay( nav.down, 0, &st);
                        }
                     }

                     if ((rc == NO_ERROR) &&    // check next EBR type
                         (st != ST_MASTR) &&
                         (st != ST_EXTBR))
                     {
                        TxPrint("\nShown sector type : %s", CBR);
                        dfsShowSectorType(st);
                        TxPrint(" is unexpected, should be MBR or EBR!\n");
                        rc = DFS_ST_MISMATCH;
                     }
                  }
               }
            }
         }
         dfsa->number = (ULONG) dfsGetDiskCount( FALSE); // for SCRIPT usage
      }
   }
   else if (strcasecmp(c0, "fixext"   ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint( "\nChange type of extended partition containers\n\n");
         TxPrint( " Usage: %s disk | . | *  [newtype  [oldtype]] [-a] [-e]\n\n"
                  "  disk      = disk-number, '.' for current or '*' for all disks\n"
                  "  newtype   = new type for extended container, default 0x05\n"
                  "  oldtype   = current type, for verification.\n\n"
                  "  -a        = Change ALL 0x0F to 0x05, not just the first one\n"
                  "  -e        = Force change to a regular partition type\n\n", c0);
         TxPrint( "To change the type of a regular primary or logical partition\n"
                  "use the 'settype' command, or force it being changed to a\n"
                  "regular partition type by using the '-e' option.\n");
      }
      else
      {
         BYTE       to = dfsParsePartType( c2, NULL);
         BYTE       fr = dfsParsePartType( c3, NULL);

         if (to == 0)                           // not specified
         {
            to = DFS_P_EXTENDED;                // default to-type (0x05)
         }
         if ((TxaOption('e')) || dfsIsExtendedType(to))
         {
            USHORT        first;
            USHORT        last;

            first = dfsParseDiskSpec( c1, &last); // request as a range
            if (last != 0)                      // valid single disk or range
            {
               for (index = first; index <= last; index++)
               {
                  if (dfsDiskAccessible( index))
                  {
                     rc = dfsFixExtType( index, fr, to);
                  }
               }
            }
            else
            {
               TxPrint( "Disknr is invalid : %u\n", first);
            }
         }
         else                                   // non-std type for extended
         {
            TxPrint("New extended-type should only specify an extended container "
                    "(05 / 0f).\nUse the 'settype' command to change regular "
                    "partitions\nor use the '-e' option to force this "
                    "non-standard value to be set.\n");
         }
      }
   }
   else if (strcasecmp(c0, "alignext"   ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint( "\nAlign start of Extended Container for a disk, or specified logical partition\n\n");
         TxPrint( " Usage: %s disk | . | * | -p:PID\n\n"
                  "  disk      = disk-number, '.' for current or '*' for all disks\n"
                  "  -p:PID    = ID for a logical partition to align the container for\n\n", c0);
         TxPrint( "This deletes, then recreates the (first) logical partition at the\n"
                  "proper cylinder and track boundaries, fixing possible damage done\n"
                  "by technically challenged Windows or Linux partitioning tools.\n");
      }
      else
      {
         BOOL             ok = TRUE;            // arg/option combinations OK

         ok &= TxaOptMutEx((cc > 1), "p", "if 'disk' is specified!", NULL);

         if (ok)
         {
            TxPrint( "\n");
            if (TxaOptSet('p'))
            {
               index  = TxaOptNum( 'p', NULL, 0);
               if ((p = dfsGetPartInfo( index)) != NULL)
               {
                  rc = dfsAlignPartContainer( p);
               }
               else
               {
                  TxPrint( "Invalid partition ID specified: %hu\n", index);
               }
            }
            else
            {
               USHORT        first;
               USHORT        last;

               first = dfsParseDiskSpec( c1, &last); // request as a range
               if (last != 0)                   // valid single disk or range
               {
                  for (index = first; (index <= last) && !TxAbort(); index++)
                  {
                     if (dfsDiskAccessible( index))
                     {
                        rc = dfsAlignDiskContainer( index);
                     }
                  }
               }
               else
               {
                  TxPrint( "Disknr is invalid : %u\n", first);
               }
            }
         }
         else
         {
            TxPrint( "\n");
         }
      }
   }
   else if (strncasecmp(c0, "cleanup", 7) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint( "\nCleanup partitiontable flags and chain of extended boot records\n\n");
         TxPrint( " Usage: %s disk | . | *   [-c:style]\n\n"
                  "  disk      = disk-number, '.' for current or '*' for all disks\n"
                  "  -c:style  = CHS-dummy style to use,\n"
                  "              0 or 'IBM' = IBM/DFSee, this is the default\n"
                  "              1 or 'PQ'  = PowerQuest: P-Magic, DriveImage\n"
                  "              2 or 'MS'  = Microsoft FDISK classic style\n"
                  "              ! or  ?    = Use CHS-style selection dialog\n\n", c0);
      }
      else
      {
         USHORT        first;
         USHORT        last;

         first = dfsParseDiskSpec( c1, &last);  // request as a range
         if (last != 0)                         // valid single disk or range
         {
            for (index = first; index <= last; index++)
            {
               if (dfsDiskAccessible( index))
               {
                  rc = dfsCleanupExtChain( index, FALSE);
               }
            }
         }
         else
         {
            TxPrint( "Disknr is invalid : %u\n", first);
         }
      }
   }
   else if (strcasecmp(c0, "newmbr"   ) == 0)
   {
      BOOL             clean = TxaOption('c') || (c2[0] == 'c');

      if (TxaOption('?'))
      {
         TxPrint( "\nWrite NEW MBR bootcode to the MBR sector (sector 0)\n\n");
         TxPrint( " Usage: %s disk | . | * | -d[:disk]  [-clean]  [-I[:image]]\n\n"
                  "  disk      = disk-number, '.' for current or '*' for all disks\n"
                  "  -c[lean]  = clear the partition tables too, resulting\n"
                  "              in an empty disk, with valid MBR code\n"
                  "  -d:disk   = apply to specified disk or current disk\n"
                  "  -f:disk   = Clone MBR-code from the specified disk to this one\n"
                  "  -I[:img]  = Use specified or default (newmbr.img) imagefile\n"
                  "              for the bootcode instead of the builtin code\n"
                  "  -n[ocode] = clear the code area, upto the Win-NT signature value\n"
                  "  -N[ocode] = clear the code area, including the Win-NT signature\n", c0);
      }
      else
      {
         BOOL          ok     = TRUE;           // arg/option combinations OK

         ok &= TxaOptMutEx((cc > 1), "d", "if a (disk) parameter is specified!", 0);
         ok &= TxaOptMutEx( FALSE, "If", "", NULL);
         if (ok)
         {
            ml = dfsPartitionableDisks();
            if ((index = dfsGetDiskNumber( 'd', ".", c1, FALSE)) == FDSK_ANY)
            {
               for (index = 1; (index <= ml) && !TxAbort() && (rc == NO_ERROR); index++)
               {
                  if (dfsDiskAccessible( index))
                  {
                     rc = dfsFdskNewMbr( index, clean);
                  }
               }
               if ((rc == NO_ERROR) && clean)
               {
                  fdsk->AutoShow = TRUE;        // indicate changes
               }
            }
            else if ((index != 0) && (index <= ml))
            {
               rc = dfsFdskNewMbr( index, clean);
               if ((rc == NO_ERROR) && clean)
               {
                  dfsFdskSingleAutoShow( index);
               }
            }
            else
            {
               TxPrint( "Disknr is invalid : %u\n", index);
               rc = DFS_VALUE_ERROR;
            }
         }
      }
   }
   else if (strcasecmp(c0, "pt") == 0)             // show partition-table for PID
   {
      if ((cc > 1) && (!TxaOption('?')))
      {
         switch (dfsParsePartSpec( c1, ml, &p))
         {
            case FDSK_ANY:                      // all partitions
               if (TxaOption('d') || (cc > 2))
               {
                  ml = dfsGetDiskNumber( 'd', ".", c2, TxaOption('r'));
               }
               else
               {
                  ml = FDSK_ANY;
               }
               for (index = 1; (index <= dfsPartitions()) && (!TxAbort()); index++)
               {
                  if ((p = dfsGetPartInfo( index)) != NULL)
                  {
                     if ((ml == FDSK_ANY) || (ml == p->disknr))
                     {
                        rc = dfsFdskPtableRelated( p);
                     }
                  }
               }
               break;

            case 0:                             // invalid partition
               TxPrint( "Partition '%s' not found\n", c1);
               break;

            default:                            // specific partition
               rc = dfsFdskPtableRelated( p);
               break;
         }
      }
      else
      {
         TxPrint("\nDisplay partition table sector, and optional bootsector plus"
                 "\nrelated LVM-information, for the specified partition(s)\n"
                 "\n Usage: %s  pid | driveletter | *  [-related]  [-d[:nr]]\n", c0);
         TxPrint("\n From there, quick navigation between some related sectors"
                 "\n can be done using single-letter commands as shown below:\n");
         TxPrint("\n    Ŀ       Ŀ       Ŀ"
                 "\n    Partition tablex>LVM informationu>LVM signature"
                 "\n ĳsector for pid <xĳsector (DLAT)  <dĳsector       "
                 "\n                 "
                 "\n                             ^                           "
                 "\n                                                        ");
         TxPrint("\n d                 u           x         d                  u"
                 "\n                                                        "
                 "\n              Ŀ                    "
                 "\n >Next          >Boot sector for<                  "
                 "\n    Ptable           the partition  <"
                 "\n               ");
      }
   }
   else if ((strncasecmp(c0, "seta",  4 ) == 0) ||
            (strncasecmp(c0, "start", 5 ) == 0) )
   {
      if (cc > 1)
      {
         switch (dfsParsePartSpec( c1, dfsGetDiskNumber( 'd', ".", "", FALSE), &p))
         {
            case FDSK_ANY:                      // all partitions
               TxPrint("%s on all partitions not supported\n", c0);
               break;

            case 0:                             // invalid partition
               TxPrint( "Partition '%s' not found\n", c1);
               break;

            default:                            // specific partition
               cbi.flag = (strncasecmp(c0, "sta", 3) == 0);
               if ((p->primary) || (cbi.flag == FALSE) || // it is not the 'startable' command
                             (toupper(c2[0]) == 'C')   || // or we just want to clear the flag
                             (toupper(c2[0]) == 'F')    ) // or we want to FORCE a logical active
               {
                  cbi.verbose = TRUE;
                  cbi.more    = (toupper(c2[0]) == 'M') ? p : NULL;
                  if (cbi.flag)
                  {
                     cbi.result = (toupper(c2[0]) == 'C'); // CLEAR startable
                     rc = dfsExecOnBootRec( p, p->disknr, dfsFdskSetStatus, &cbi);
                  }
                  else                          // setaccess
                  {
                     if (dfsPartTypeHidable(p->partent.PartitionType))
                     {
                        cbi.option = (toupper(c2[0]) == 'H');
                        rc = dfsExecOnBootRec( p, p->disknr, dfsFdskSetStatus, &cbi);
                     }
                     else
                     {
                        dfsPartTypeDescription( p->partent.PartitionType, s1);
                        TxPrint("%s not supported on type %2.2x = %s\n",
                                 c0, p->partent.PartitionType, s1);
                     }
                  }
               }
               else                             // startable on logical
               {
                  TxPrint("\nSetting a logical partition STARTABLE (ACTIVE) needs the FORCE option!\n");
               }
               break;
         }
      }
      else
      {
         TxPrint("\n Usage: %s  part [%s | multiple-primaries]\n", c0,
            (strncasecmp(c0, "sta", 3)) ? "hidden | visible" : "clear | force-logical");
      }
   }
   else if (strncasecmp(c0, "setb", 4   ) == 0) // setboot, BM setup
   {
      rc = setbCommand();

      TRACES(("Setting retc to: %u = 0x%X\n", rc, rc));
      //- to be refined, may be able to to without a separate retc variable now
      dfsa->retc = rc;                          // set overall return-code
      if (dfsa->autoquit)                       // from cmdline, with args
      {
         if (!TxaOptUnSet('Q'))                 // no -Q-
         {
            rc |= DFS_QUIT;                     // automagical quit
         }
      }
   }
   else if (strncasecmp(c0, "setn", 4   ) == 0) // set BM name
   {
      if (cc > 1)                               // some parameters required!
      {
         cbi.verbose = TRUE;
         switch (dfsParsePartSpec( c1, dfsGetDiskNumber( 'd', ".", "", FALSE), &p))
         {
            case FDSK_ANY:                      // all partitions
               TxPrint("%s on all partitions not supported\n", c0);
               break;

            case 0:                             // invalid partition
               TxPrint( "Partition '%s' not found\n", c1);
               break;

            default:                            // specific partition
               if (p->primary)
               {
                  rc = dfsFdskSetPriBm( p, c2, FALSE);
               }
               else
               {
                  rc = dfsFdskSetLogBm( p, c2, FALSE);
               }
               break;
         }
      }
      else
      {
         TxPrint("\n Usage: %s  pid   [BMGR name]\n", c0);
      }
   }
   else if ((strcasecmp(c0, "delfind"  ) == 0) ||  // Filesystem specific ones
            (strcasecmp(c0, "filefind" ) == 0) ||  // give a hint on why they
            (strcasecmp(c0, "fixboot"  ) == 0) ||  // do not work here :-)
            (strcasecmp(c0, "bootini"  ) == 0) ||
            (strcasecmp(c0, "dirty"    ) == 0) ||
            (strcasecmp(c0, "path"     ) == 0) ||
            (strcasecmp(c0, "ca"       ) == 0) ||
            (strcasecmp(c0, "cl"       ) == 0) ||
            (strcasecmp(c0, "bl"       ) == 0) ||
            (strcasecmp(c0, "super"    ) == 0)  )
   {
      TxPrint( "\nThe '%s' command is NOT supported in FDISK mode.\n\n"
               "It is specific to one or more of the supported filesystems\n"
               "like HPFS, NTFS, FAT etc. Opening a partition or volume with\n"
               "one of these filesystems present will automatically activate\n"
               "an FS-specific mode, and make these commands available.\n", c0);
   }
   else
   {
      rc = DFS_CMD_UNKNOWN;                     // cmd not recognized
   }
   if ((rc == NO_ERROR) && (fdsk->AutoShow) &&
       (strlen(fdsk->ShowCmd)) && (fdsk->ShowCmd[0] != '-'))
   {
      TxPrint( "Execute autoshow  : '%s'\n",
                fdsk->ShowCmd);
      if (fdsk->InAutoShow == FALSE)            // avoid recursion
      {
         TxCancelAbort();                       // might have aborted ...
         fdsk->InAutoShow = TRUE;
         dfsMultiCommand( fdsk->ShowCmd, 0, FALSE, FALSE, TRUE);
         fdsk->InAutoShow = FALSE;
      }
      else                                      // AutoShow causes AutoShow
      {
         TxPrint( "Warning: '%s%s%s' is not suitable as AutoShow command!\n",
                   CBR, c0, CNN);
      }
   }
   fdsk->AutoShow = FALSE;                      // reset indicator
   RETURN (rc);
}                                               // end 'dfsFdskCommand'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display all partitioning related sectors for the specified partition
/*****************************************************************************/
static ULONG dfsFdskPtableRelated
(
   DFSPARTINFO        *p                        // IN    partition to display
)
{
   ULONG               rc = NO_ERROR;
   ULONG               sn;
   BYTE                st = 0;                  // sector type wanted
   TXLN                dc;                      // generated command

   ENTER();

   if (p != NULL)
   {
      TxPrint( "\nPartTable for PID : %hu on disk %hu\n", p->id, p->disknr);
      dfsSelectDisk( p->disknr, FALSE, FALSE);  // quietly select disk
      st = 0;                                   // use autodetect for MBR/EBR
      rc = dfsReadAnDisplay( p->partPsn, 0, &st);
      if (rc == NO_ERROR)
      {
         if (TxaOption('r'))                    // related sectors too
         {
            TxPrint( "\nRelated bootsector\n");
            sn = nav.down;                      // remember next-EBR sector
            st = 0;                             // use autodetect for PBR
            rc = dfsReadAnDisplay( p->basePsn, 0, &st);
            if (dfsa->lvmPresent)
            {
               strcpy( dc, "x ;related LVM-info (DLAT) sector");
               if (p->partent.PartitionType == DFS_P_WARP_LVM)
               {
                  strcat( dc, dfsa->cmdSeparator);
                  strcat( dc, "u ;related LVM signature (BBR) sector");
               }
               dfsMultiCommand( dc, 0, TRUE, FALSE, TRUE);
            }
            nav.down = sn;                      // restore next sector reference
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsFdskPtableRelated'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// FDISK filesystem, identify specified sector
/*****************************************************************************/
ULONG dfsFdskIdent
(
   ULN64               lsn,                     // IN    LSN for sector
   ULN64               d2,                      // IN    dummy
   char               *st,                      // OUT   sector type
   void               *sec                      // IN    sector contents
)
{
   ULONG               rc = NO_ERROR;
   BYTE                tp = ST_UDATA;

   ENTER();

   TRDUMP(70, "FDISK ident, start of sector:\n", sec, 64, 0);

   switch (tp)
   {
      case ST_UDATA:
         rc = DFS_PENDING;
         break;

      default:
         break;
   }
   *st = tp;
   RETURN (rc);
}                                               // end 'dfsFdskIdent'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// FDISK filesystem, supply sector-type description string
/*****************************************************************************/
static ULONG dfsFdskStype
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *st,                      // IN    sector type
   void               *data                     // OUT   type description
)
{
   ULONG               rc  = NO_ERROR;

   //- Note: most/all sectortypes moved to generic DFSUDISP so they
   //-       are properly recognized and displayed in any mode

   switch (*st)                                 // searchable types
   {
      default:         rc = DFS_PENDING;
   }
   return (rc);
}                                               // end 'dfsFdskStype'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// FDISK filesystem, display sector-contents based on type
/*****************************************************************************/
static ULONG dfsFdskDispl
(
   ULN64               psn,                     // IN    base psn for sector
   ULN64               d2,                      // IN    dummy
   char               *type,                    // IN    type of sector
   void               *data                     // IN    sector contents
)
{
   ULONG               rc = NO_ERROR;
   BYTE                st = (BYTE) *type;

   ENTER();

   switch (st)
   {
      default:
         rc = DFS_PENDING;
         break;
   }
   RETURN (rc);
}                                               // end 'dfsFdskDispl'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display MAC driver-descriptor-record, in MBR sector, can be combined!
/*****************************************************************************/
ULONG dfsFdskMacDDM
(
   BYTE               *sector                   // IN    Sector data
)
{
   ULONG               rc = 0;                  // rc, sector match
   S_MCDDM            *sd = (S_MCDDM *) sector;
   USHORT              bps;
   ULONG               ulv;
   USHORT              ent;
   USHORT              i;

   ENTER();

   ent = txSwapUS( sd->entries);
   bps = txSwapUS( sd->bpsector);
   ulv = txSwapUL( sd->sectors);                            //- read as Big-Endian (as specced)
   if (((ULN64) (ulv / 2)) > dfstGetLogicalSize( DFSTORE))  //- must be too big, try little endian
   {                                                        //- Seen on AdobeFlash disk-image 19MB
      ulv = sd->sectors;                                    //- 20170920 JvW: Field seems Little-endian
      TxPrint( "Warning: size interpreted as little-endian, due to its value, may be incorrect ...\n");
   }

   dfsDec4(    "Block size, bytes : ",     (ULONG) bps, "\n");
   TxPrint(    "Disk blocks, size : %10u dec = ", ulv);
   dfsSizeBps( "", ulv, bps, "\n");

   if ((ent > 60) || (bps > 2048))
   {
      TxPrint("DDM key values seem to be invalid, restrict display to 60 entries!\n");
      ent = 60;
   }
   for (i = 0; (i < ent) && (!TxAbort()); i++)
   {
      ulv = txSwapUL( sd->driver[i].start);
      TxPrint( "  Driver #%hu start : 0x%8.8X = %10u dec  Type : 0x%4.4hx\n",
                          i+1, ulv, ulv, txSwapUS( sd->driver[i].type));
      dfsSizeBps("  size in sectors : ", txSwapUS( sd->driver[i].size), bps, "\n");
   }

   dfsFdskMacPartMap( (dfsa->verbosity >= TXAO_VERBOSE)); // display partition map

   if (sd->pcSignature == SV_BOOTR)             // PC-style signature too ?
   {
      dfsPartitionRecord( sector, 'r', 0);      // display as MBR
   }
   RETURN (rc);
}                                               // end 'dfsFdskMacDDM'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display possible MAC DPM sectors in first track, until #DPM sectors reached
/*****************************************************************************/
ULONG dfsFdskMacPartMap                         // RET   result
(
   BOOL                verbose                  // IN    output style
)
{
   ULONG               rc  = NO_ERROR;          // function return
   BYTE               *sec = NULL;              // sector buffer

   ENTER();

   if ((sec = (TxAlloc( 1, dfsGetSectorSize()))) != NULL)
   {
      ULONG            last = dfstGeoSectors( DFSTORE);
      ULONG            dpms;
      ULONG            macs = 0;                // nr of MAC partitions seen
      S_MCDPM         *sd = (S_MCDPM *) sec;    // as MCDPM sector

      for (dpms = 1; dpms <= last; dpms++)
      {
         rc = dfstReadPsn( DFSTORE, dpms, 1, sec);
         if (rc == NO_ERROR)
         {
            if (dfsIdentifySector( dpms, 0, sec) == ST_MCDPM)
            {

               macs++;
               if (verbose)
               {
                  TxPrint( "\nDisk sector %8.8X for MAC partition %u\n", dpms, macs);
                  dfsFdskMacDPM( sec);
               }
               else
               {
                  if (dpms == 1)                // header on first line
                  {
                     TxPrint("\n");
                     TxPrint("nr Part-start   (hex)  Part-size    (hex) "
                             "              Type:Name  \n");
                     TxPrint("== ==========-======== ==========-==========---=========== ====================================\n");
                  }
                  dfsFdskMacDPMline( macs, sd);
               }
               if (macs >= txSwapUL( sd->mapSectors)) // reached nr of partitions listed
               {
                  break;
               }
            }
         }
      }
      if ((!verbose) && (macs > 0))
      {
         TxPrint("== ==========-======== ==========-==========---=========== ====================================\n");
      }
      TxPrint("\n#of MAC partitions: %u\n\n", macs);
      TxFreeMem( sec);                          // free the memory
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsFdskMacPartMap'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display MAC disk-partition-map sector, 1-line compact
/*****************************************************************************/
static ULONG dfsFdskMacDPMline
(
   ULONG               nr,                      // IN    MAC part index 1..n
   S_MCDPM            *sd                       // IN    Sector data as MCDPM
)
{
   ULONG               rc = 0;                  // rc, sector match
   ULONG               base;
   ULONG               size;

   ENTER();

   base = txSwapUL( (ULONG) sd->basePsn);
   size = txSwapUL( (ULONG) sd->sectors);

   TxPrint( "%2u %-10u %8.8X|%-10u", nr, base, base, size);
   dfsSizeBps(" ", size, dfsGetSectorSize(), "|");

   TxPrint( "%-.32s:%s%-.32s%s\n", sd->partType, CBG, sd->partName, CNN);

   RETURN (rc);
}                                               // end 'dfsFdskMacDPMline'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display MAC disk-partition-map sector, usually at sector 1 .. #partitions
/*****************************************************************************/
ULONG dfsFdskMacDPM
(
   BYTE               *sector                   // IN    Sector data
)
{
   ULONG               rc = 0;                  // rc, sector match
   S_MCDPM            *sd = (S_MCDPM *) sector; // as MCDPM sector
   ULONG               base;
   ULONG               size;

   ENTER();

   base = txSwapUL( sd->basePsn);
   size = txSwapUL( sd->sectors);

   TxPrint( "Nr of DPM sectors : %2u         Decimal start block :%10u  size %10u\n",
                                 txSwapUL( sd->mapSectors), base, size);

   dfsGeoDispTransPsn( "First part sector :",  dfstGeoHeads( DFSTORE), dfstGeoSectors( DFSTORE), base);
   dfsGeoDispTransPsn( "Last  part sector :",  dfstGeoHeads( DFSTORE), dfstGeoSectors( DFSTORE), base + size -1);
   dfsSizeBps(         "Part blocks, size : ", size, dfsGetSectorSize(), "\n");

   TxPrint( "Part type & name  : %-32.32s %s%-32.32s%s\n", sd->partType, CBG, sd->partName, CNN);

   //- to be refined, add rest of DPM fields to display

   RETURN (rc);
}                                               // end 'dfsFdskMacDPM'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Issue a warning about using this command on Windows-2000 or XP (P# 139)
/*****************************************************************************/
static BOOL dfsFdskWarningW2KXP
(
   char               *cmd                      // IN    name of command
)
{
   BOOL                rc = TRUE;               // default allow

   ENTER();

   #if defined (WIN32)
      {
         OSVERSIONINFO ver;

         ver.dwOSVersionInfoSize = sizeof( OSVERSIONINFO);
         ver.dwMajorVersion      = 0;
         ver.dwMinorVersion      = 0;
         ver.dwBuildNumber       = 0;

         GetVersionEx( &ver);                   // try to get real version

         if ((ver.dwMajorVersion != 4) && (!dfsa->batch) && (!TxConfirm( 5018,
            "WARNING: The '%s' command might cause driveletter assignment "
            "problems when used on Windows-2000, XP or later versions.\n\n"
            "You are running Windows version %u.%u, build %u\n\n"
            "Are you sure you want to run it ? [Y/N] : ", cmd,
             ver.dwMajorVersion, ver.dwMinorVersion, ver.dwBuildNumber)))
         {
            rc = FALSE;
         }
      }
   #endif
   BRETURN (rc);
}                                               // end 'dfsFdskWarningW2KXP'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set next autoshow command to use specified disk only
/*****************************************************************************/
void dfsFdskSingleAutoShow
(
   USHORT              disknr                   // IN    single disk to show
)
{
   ENTER();

   if (dfsa->verbosity > TXAO_QUIET)
   {
      if ((strstr( fdsk->ShowCmd, " -r") != NULL) &&
          (strstr( fdsk->ShowCmd, " -d") == NULL)  )
      {
         TXTM  diskopt;                         // specific disknr to show

         sprintf( diskopt, " -d:%hu %s", disknr, FDSK_SINGLE_SHOW);
         strcat( fdsk->ShowCmd, diskopt);
      }
      fdsk->AutoShow = TRUE;                    // indicate changes
   }
   VRETURN ();
}                                               // end 'dfsFdskSingleAutoShow'
/*---------------------------------------------------------------------------*/

