//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// FAT dump & display Analysis functions
//
// Author: J. van Wijk
//
// JvW  21-05-97   Initial version
// JvW  05-01-2000 Work-around for AllocChain bug (size 0)
// JvW  18-11-2001 Used TXA as command parser, reducing stack usage
// JvW  20-08-2018 Allow recovery of contiguous files if FAT emptied after FORMAT (photo cards)
// JvW  08-04-2021 Use rc CMD_WARNING on FileSaveAs alloc errors, considered OK
//
// Author: J. van Wijk
//

#include <txlib.h>                              // TX library interface

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfswin.h>                             // DFS help ids
#include <dfswipe.h>                            // wipe functions
#include <dfsutil.h>                            // DFS utility functions
#include <dfsspace.h>                           // DFS  file-space interface
#include <dfstable.h>                           // SLT utility functions

#include <dfsupart.h>                           // FDISK partition functions
#include <dfsafdsk.h>                           // FDISK display & analysis
#include <dfsufdsk.h>                           // FDISK utility functions
#include <dfscfdsk.h>                           // FDISK util, FAT sectors

#include <dfsafat.h>                            // FAT display & analysis
#include <dfsffat.h>                            // FAT formatting and defs
#include <dfsbfat.h>                            // FAT bootsector functions
#include <dfsdfat.h>                            // FAT dialogs and definitions
#include <dfsufat.h>                            // FAT utility functions
#include <dfslfat.h>                            // FAT SLT functions
#include <dfsosapi.h>                           // OS specific stuff

#define  FAT_EADATA_FILE  "EA DATA. SF"         // OS/2 FAT EA-data file

char dfsFatSectorTypes[] =
{
   ST_FAT_1,                                    //  I  1st FAT area
   ST_FAT_2,                                    //  I  2nd FAT area
   ST_FAT12,                                    //  I  FAT 12 1st sector
   ST_FAT16,                                    //  I  FAT 16 1st sector
   ST_FAT32,                                    //  I  FAT 32 1st sector
   ST_ROOTD,                                    //  I  Root directory area
   ST_SUBDR,                                    //     Sub-directory, non-ROOT
   ST_SUBRT,                                    //     Sub-directory from ROOT
   ST_DIREC,                                    //     directory area
   ST_FATEA,                                    //     FAT EAdata cluster
   ST_EADAT,                                    //     FAT EAdata header
   0                                            //     string terminating ZERO
};


static DFSAFAT    fat_anchor =
{
   0,                                           // LSN of 1st Fat
   0,                                           // LSN of 2nd Fat
   0,                                           // LSN Temp.  Fat area in resize
   0,                                           // LSN of cluster-2
   0,                                           // LSN of Root directory
   0,                                           // nr of sectors in Root dir
   0,                                           // nr of FATs (usually 2)
   0,                                           // Nr of sectors per FAT
   0,                                           // Number of sectors
   0,                                           // #clusters, from #sectors
   0,                                           // #clusters, from #fatSectors
   0,                                           // First free cluster, trunc-point
   0,                                           // Nr of clusters still free
   0,                                           // valid cluster size
   0,                                           // Maximum size for expand, resize
   0,                                           // Maximum size for current FAT area
   0,                                           // Size of each FAT entry
   FALSE,                                       // FAT32 spare sect 6&7 used
   FALSE,                                       // FAT values faked, no bootsec
   TRUE,                                        // allocation cached cluster
   L32_NULL,                                    // cached cluster number
   0,                                           // FAT in-use, LSN
   {0, NULL, 0, FALSE, 'A'},                    // 1st cache, A
   {0, NULL, 0, FALSE, 'B'},                    // 2nd cache, B
   {0, NULL, 0, NULL, NULL, 0},                 // EA data admin and cache
};

       DFSAFAT   *fat = &fat_anchor;

       char sg_fat12[SG_FAT12] = {SV_FAT12};
       char sg_fat16[SG_FAT16] = {SV_FAT16};
       char sg_fat32[SG_FAT32] = {SV_FAT32};

static  char       *fat_txt[] =
{
   "",
   "Active filesystem : FAT, specific commands are:",
   "",
   " \\[path-spec]    = find and show ROOT or file/directory relative to root",
   " BOOTINI  [part] = Find the (first) boot.ini file present in the filesystem",
   " BSCLEAR         = Reset bad-sector/cluster administration to ZERO bad sectors",
   " DELFIND  [name] = Find deleted files with DIR-entry containing (partial) name",
   " DIRTY   [d|c|#] = Display/set DIRTY, CLEAN  status or numeric flag value #",
   " FATSHOW  [*|nr] = Display contents of the FAT from memory, [nr]= # of entries",
   " FATSELECT   [f] = Select FAT area to be used (cached), f = 1 or 2; default is 1",
   " FATSYNC     [f] = Synchronize FAT-areas from 'f' to other, default is from 1 to 2",
   " FATSIM  img [f] = Save FAT table contents for FAT 'f' (1 or 2) to an imagefile",
   " FATWRIM img [f] = Restore an imagefile with the FAT table to FAT 'f' (1 or 2)",
   " FILEFIND [name] = Find normal files with DIR-entry containing (partial) name",
   " FINDROOT        = Find and list possible FAT32 root directory clusters/sectors",
   " FIRSTFREE       = Find first FREE cluster in FAT, FAT32: optional write back",
   " FIXBOOT  [os|2] = Fix FAT(32) bootsector from P-tables or spares for OS 'os'",
   " FORMAT   [opts] = Format current object with a FAT12/16 or FAT32 filesystem",
   " SETROOT [s|.nn] = Set FAT32 root directory LSN to sector [s] or listvalue [.nn]",
   " SPACE   [clust] = Show allocation for specified cluster or current LSN 'this'",
   " SUBFIND  [opts] = Find subdirectories (.. entry) from start or current lsn",
   "",
   NULL
};


static  char       *cmd_format[] =
{
   "",
   "Format current object with the FAT filesystem",
   "",
   "usage:  FORMAT  [options]",
   "",
   "   -a:offset = Sector offset for first FAT area from bootsector",
   "   -c:spc    = Clustersize in 512 byte units: power of 2, 1 .. 64",
   "   -f:bits   = Number of bits used for FAT entries: 12, 16 or 32",
   "   -l        = Long format, clear all data-area sectors to ZERO",
   "   -o:letter = Operating system identification letter, boot code:",
   "",
   "                 I or P   IBM/PC-DOS with IBMBIO/IBMDOS.COM",
   "                 M        MSDOS      with IO/MSDOS.SYS",
   "                 O        OS/2       with OS2BOOT",
   "                 N        Win-NT     with NTLDR       (FAT16)",
   "                 9        Win-9x     with WINBOOT.SYS (FAT32)",
   "",
   "   -r:size   = number of entries in root directory    (FAT12/16)",
   "   -s:vsn    = volume serial number to be used  (use 0x for hex)",
   "   -v:label  = ASCII label string for filesystem identification",
   "   -!        = force use of option dialog to specify parameters",
   "",
   NULL
};


// Close FAT filesystem for analysis and free any resources
static ULONG dfsFatClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
);

// Close FAT filesystem for Area analysis and free any resources
static ULONG dfsFatAreaClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
);

// Search and list possible Root directory clusters for a FAT32 partition
static ULONG dfsFatFindFat32Roots               // RET   nr of candidates found
(
   void
);

// Interpret and execute specific FAT command;
static ULONG dfsFatCommand
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *none,                    // IN    dummy
   void               *data                     // INOUT dummy
);

// FAT filesystem, identify specified sector
static ULONG dfsFatIdent
(
   ULN64               di,                      // IN    LSN for sector
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // OUT   sector type
   void               *data                     // IN    sector contents
);

// FAT filesystem, supply sector-type description string
static ULONG dfsFatStype
(
   ULN64               di,                      // IN    sector type
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // OUT   type description
   void               *data                     // INOUT dummy
);

// FAT filesystem, display sector-contents based on type
static ULONG dfsFatSectorContents
(
   ULN64               psn,                     // IN    base psn for sector
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // IN    sector contents
);

// FAT filesystem, display special LSN info
static ULONG dfsFatDisplayLsnInfo
(
   ULN64               lsn,                     // IN    possible special lsn
   ULN64               info,                    // IN    possible DIR entry nr
   char               *dc,                      // IN    dummy
   void               *data                     // IN    dummy
);

// Display FAT Root- or sub-directory sector(s)
static ULONG dfsFatDirDisplay
(
   ULONG               psn,                     // IN    base psn for sector
   BYTE                type                     // IN    type of sector
);

// Find LSN* for specified path, starting at Root directory LSN
static ULONG dfsFatFindPath
(
   ULN64               loud,                    // IN    Show progress
   ULN64               d2,                      // IN    dummy
   char               *path,                    // IN    path specification
   void               *vp                       // OUT   found dir/file FNODE
);

// Find specific name in a (sub) directory SPACE structure
static BOOL dfsFatDirSpace2Entry                // RET   name found
(
   char               *name,                    // IN    subdir or filename
   ULONG               chunks,                  // IN    nr of space entries
   S_SPACE            *space,                   // IN    space allocation
   ULN64              *lsn,                     // OUT   Dirsector found
   ULONG              *info,                    // OUT   Dir entry-nr found
   S_FATDIR           *fatdir                   // OUT   Directory entry
);

// DFS FAT write-file to disk (SaveTo fnode to file)
static ULONG dfsFatFileSaveAs
(
   ULN64               dse,                     // IN    dirsec/entry LSN
   ULN64               info,                    // IN    info, dir entry
   char               *path,                    // IN    destination path
   void               *ren                      // IN    new name (like 8.3)
);

// Make system/FAT usage map dump on TxPrint output
static ULONG dfsFatAllocMap
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *options,                 // IN    display options
   void               *data                     // OUT   dummy
);

// Truncate or expand the filesystem size, keeping all data intact
static ULONG dfsFatResizeFS
(
   ULN64               newsz64,                 // IN    new size in sectors
   ULN64               d2,                      // IN    dummy
   char               *string,                  // IN    dummy
   void               *data                     // IN    partition info
);

/*****************************************************************************/
// Initialize FAT filesystem analysis
/*****************************************************************************/
ULONG dfsFatInit
(
   char               *fs                       // forced filesystem type
)
{
   ULONG               rc = NO_ERROR;
   TXTM                label;                   // FAT Root label
   TXTM                tbuf;
   TXLN                text;
   FAT32B2            *fat32 = (FAT32B2 *) &dfsa->boot[1]; // 2nd bootrec
   USHORT              bits  = 0;
   BOOL                Check2ndFAT = !TxaOptUnSet('2');
   BYTE                FsysValue = 0;              // display value

   ENTER();

   dfsa->FsCommand          = dfsFatCommand;
   dfsa->FsClose            = dfsFatClose;
   dfsa->FsIdentifySector   = dfsFatIdent;
   dfsa->FsShowType         = dfsFatStype;
   dfsa->FsDisplaySector    = dfsFatSectorContents;
   dfsa->FsFileInformation  = dfsFatFileInfo;
   dfsa->FsGetAllocSpace    = dfsFatGetAllocSpace;
   dfsa->FsWriteMetaSpace   = NULL;
   dfsa->FsUpdateFileName   = NULL;
   dfsa->FsSetAltBrecLabel  = dfsFatSetAltBrecLabel;
   dfsa->FsMakeBrowseList   = dfsFatMakeBrowseList;
   dfsa->FsFindPath         = dfsFatFindPath;
   dfsa->FsLsnAllocated     = dfsFatAllocated;
   dfsa->FsLsnSetAlloc      = NULL; // not really implemented: dfsFatSetAlloc;
   dfsa->FsSltBuild         = dfsFatSltBuild;
   dfsa->FsNpBuild          = NULL;
   dfsa->FsDisplayError     = dfsFatDispError;
   dfsa->FsDisplayLsnInfo   = dfsFatDisplayLsnInfo;
   dfsa->FsDirIterator      = dfsFatIterator;
   dfsa->FsDirFileSaveAs    = dfsFatFileSaveAs;
   dfsa->FsTruncateSize     = dfsFatResizeFS;
   dfsa->FsAllocDisplay     = dfsFatAllocMap;
   dfsa->FsCl2Lsn           = dfsFatCl2Lsn;
   dfsa->FsLsn2Cl           = dfsFatLsn2Cl;
   dfsa->FsCmdHelp          = fat_txt;          // FS specific cmdhelp text

   dfsa->FsModeId           = DFS_FS_FAT;

   dfsa->Fsi                = fat;
   dfsa->FsSectorTypes      = dfsFatSectorTypes;

   TRACES(("sizeof S_BOOTR        = % 3u\n", sizeof(S_BOOTR)));
   TRACES(("sizeof S_FAT12        = % 3u\n", sizeof(S_FAT12)));
   TRACES(("sizeof S_FAT16        = % 3u\n", sizeof(S_FAT16)));
   TRACES(("sizeof S_FAT32        = % 3u\n", sizeof(S_FAT32)));
   TRACES(("sizeof S_DATIM        = % 3u\n", sizeof(S_DATIM)));
   TRACES(("sizeof S_DATIM        = % 3u\n", sizeof(S_DATIM)));
   TRACES(("sizeof S_FATDIR       = % 3u\n", sizeof(S_FATDIR)));
   TRACES(("sizeof S_VFSLOT       = % 3u\n", sizeof(S_VFSLOT)));
   TRACES(("sizeof S_EACLUST      = % 3u\n", sizeof(S_EACLUST)));
   TRACES(("sizeof S_EAINDEX      = % 3u\n", sizeof(S_EAINDEX)));

   if (fs && (strlen(fs) > 3) && (isdigit(fs[3]))) // forced FATnn filesystem
   {
      bits = (USHORT) atoi( fs +3);
      TxPrint( "FAT entry size of : %2.2hu  Bits   (%s) ", bits, fs);
      switch (bits)
      {
         case 12:
         case 16:
         case 32:
            TxPrint( "will be forced.\n");
            break;

         default:
            TxPrint( "is invalid and will be ignored\n");
            bits = 0;
            break;
      }
   }
   if ((bits == 0) && (strncasecmp( dfsa->boot->f2.Type, "FAT32", 5) == 0))
   {
      bits = 32;                                // assume 32-bit from bootrec
   }

   //- For FatSectors validation, note that a resized FAT16 can become slightly
   //- larger than the maximum of 256 sectors due to alignment to clustersize
   //- so the real maximum is 256 + #sectors/cluster! (bug upto version 13.0)

   if ((dfsa->boot                == NULL    ) ||       // invalid boot record
       (dfsa->boot->Signature     != SV_BOOTR) ||
       (dfsa->boot->eb.ClustSize  == 0       ) ||
       (dfsa->boot->eb.ClustSize   > 128     ) ||
       (dfsa->boot->eb.FatSectors  > (256 + dfsa->boot->eb.ClustSize)))
   {
      TxPrint( "Bootsector seems %sinvalid%s, trying (FAT32) spare "
               "sectors 6 & 7\n\n", CBR, CNN);
      if ((rc = dfsRead(LSN_FAT32SPARE, 2, brec)) == NO_ERROR)
      {
         nav.down   = LSN_FAT32SPARE;
         dfsa->boot = (S_BOOTR *) brec;
         switch (dfsIdentifySector(0, 0, brec))
         {
            case ST_BOOTX:
               TxPrint( "Non-standard boot sector detected ...\n");
            case ST_BOOTR:
               if (SINF->p == NULL)             // No partition info yet
               {
                  dfstSetSectorSize( DFSTORE, (USHORT) dfsa->boot->eb.SectSize);
                  dfstSetClusterSize(DFSTORE, (USHORT) dfsa->boot->eb.ClustSize);
               }
               TxPrint("Bootsec. checksum : %8.8X  Spare copy looks %sOK%s!\n",
                        TxHpfsCheckSum((char *) brec), CBG, CNN);
               TxPrint("Use 'fixboot' to replace the "
                       "bootsector (0) by this spare ...\n");
               fat->Fat32Spare = TRUE;
               break;

            default:
               TxPrint("Sector %u is not a valid filesystem BOOT record either!\n",
                       LSN_FAT32SPARE);
               dfsRead( LSN_BOOTR, 2, brec);          // re-read real bootsector(s)
               break;
         }
      }
      else
      {
         TxPrint( "Reading FAT32 spare-sectors failed!\n");
         dfsRead( LSN_BOOTR, 2, brec);          // re-read real bootsector(s)
      }
   }

   if ((dfsa->boot                != NULL    ) && // could be valid boot record
       (dfsa->boot->Signature     == SV_BOOTR) &&
       (dfsa->boot->eb.ClustSize  >= 1       ) &&
       (dfsa->boot->eb.ClustSize  <= 128     ) &&
       (dfsa->boot->eb.FatSectors <= (256 + dfsa->boot->eb.ClustSize)) &&
      ((dfsa->boot->f2.FatSectors != 0) || (bits != 32))) // must have for FAT32
   {
      fat->Sect  = max(dfsa->boot->eb.BigSectors, (ULONG) dfsa->boot->eb.Sectors);

      /* disabled, not always desirable to do this automatically (RAW2IMZ) !
      if (dfstStoreType( DFSTORE) == DFST_RAWIMAGE)
      {
         if (fat->Sect > dfsGetLogicalSize()) // limits set to low!
         {
            ULONG      base = dfstLSN2Psn( DFSTORE, 0);

            dfstLogicalLimits( DFSTORE, base, base + fat->Sect -1);
         }
      }
      */
      fat->ClustSize     = (USHORT) dfsa->boot->eb.ClustSize;
      fat->NrOfFats      = (USHORT) dfsa->boot->eb.NrOfFats;
      fat->Fat1          =          dfsa->boot->eb.FatOffset;

      TRACES(("FAT ClustSize: %4.4hu, Offset: %8.8x\n, fat->ClustSize, fat->Fat1"));

      if (bits == 32)                           // forced or recognized FAT32
      {
         fat->FatSectors = dfsa->boot->f2.FatSectors;
         if (fat->NrOfFats > 1)
         {
            fat->Fat2  = fat->Fat1 + fat->FatSectors;
         }
         else
         {
            fat->Fat2  = fat->Fat1;
         }
         fat->Roots =             fat->ClustSize; // minimum size
         fat->ClTwo = fat->Fat1 + (fat->FatSectors * fat->NrOfFats);

         fat->Root  = dfsFatClust2LSN( dfsa->boot->f2.RootCluster);
      }
      else
      {
         fat->FatSectors = dfsa->boot->eb.FatSectors;
         if (fat->NrOfFats > 1)
         {
            fat->Fat2  = fat->Fat1 + fat->FatSectors;
         }
         else
         {
            fat->Fat2  = fat->Fat1;
         }
         fat->Root  = fat->Fat1 + (fat->FatSectors * fat->NrOfFats);
         fat->Roots = (dfsa->boot->eb.RootEntries * sizeof(S_FATDIR)) / dfsGetSectorSize();
         fat->ClTwo = fat->Root + fat->Roots;
      }

      if (fat->ClTwo < fat->Sect)
      {
         fat->RawClust = ((fat->Sect - fat->ClTwo) / fat->ClustSize) + 2;
      }
      else
      {
         fat->RawClust = (fat->Sect - 1) / fat->ClustSize;
         TxPrint("\nWarning: Number of sectors and Fat/Root size are inconsistent!\n"
                   "         Calculated number of clusters is not reliable\n");
      }

      if (bits == 0)                            // not known yet
      {
         if      (fat->RawClust  > (FAT_MAXCLUST & 0xffff)) bits = 32;
         else if (fat->RawClust  > (FAT_MAXCLUST & 0xfff )) bits = 16;
         else if (fat->ClustSize > 8)                       bits = 16; // resized ?
         else                                               bits = 12;
      }
      fat->FatBits  = bits;
      fat->MapClust = fat->FatSectors * (dfsGetSectorSize() * 8 / bits); // mapped clusters

      if ((dfsFatDirtyStatusBootrec( FALSE, 0) == 0) &&
          (dfsFatDirtyStatusFatarea( FALSE, 0) == 0)  )
      {
         dfsa->FsDirtyStatus = DFSTAT_CLEAN;
      }
      else
      {
         dfsa->FsDirtyStatus = DFSTAT_DIRTY;
      }
   }
   else                                         // set sane values
   {
      BYTE             ClustSize;
      ULONG            FatSectors;
      TXTM             Signature;               // FAT signature buffer

      Signature[0]   = (char) 0xf8;             // assume harddisk :-)

      TxPrint( "Bootsector seems invalid, using most likely values ...\n");
      if (SINF->p != NULL)                      // partition info avaliable
      {
         TRACES(("PartitionType: %2.2hx\n", SINF->p->partent.PartitionType));

         if ((SINF->p->partent.PartitionType == DFS_P_FAT32)   ||
             (SINF->p->partent.PartitionType == DFS_P_FAT32X)  ||
             (SINF->p->partent.PartitionType == DFS_P_ACRONISSZ))
         {
            bits = 32;                          // force 32 based on type
         }
         fat->Sect  = SINF->p->sectors;
      }
      else                                      // use current store size
      {
         fat->Sect  = dfsGetLogicalSize();
      }
      TxPrint( "Searching 1st FAT area ... ");

      if (( dfsa->boot       != NULL) &&        // fat32 ptr valid
          ((bits             == 32  ) ||        // FAT32 part type or forced
          ((fat32->Signatur1 == SV_BOOT21) &&   // or valid FAT32 signature
           (fat32->Signatur2 == SV_BOOT22))))
      {
         TRACES(("dfsa->boot: %8.8x  f32-sig1:%8.8x  f32-sig2:%8.8x\n",
                  dfsa->boot, fat32->Signatur1, fat32->Signatur2));

         memcpy( &(Signature[1]), sg_fat32, SG_FAT32);
         if ((fat->Fat1 = (dfsLocateSignatureSector(
                              1, 0x1000, Signature, SG_FAT32 +1, 0))) == 0)
         {
            Check2ndFAT = FALSE;
            fat->Fat1   = 0x20;
            TxPrint( "not found, using default 0x20 (LSN 20)\n"
                     "FIXBOOT will NOT be reliable!\n");
         }
         else
         {
            TxPrint( "found at 0x%4.4x (LSN % lu)\n", fat->Fat1, fat->Fat1);
         }
         TxPrint( "Assuming 4 KiB cluster size ...\n");
         ClustSize    = 8;                      // assume 8 sector clusters
         fat->FatBits = 32;
         fat->Roots   = ClustSize;              // minimum is 1 cluster ...
         FatSectors   = 131000;                 // 64 GB part, 4KB clusters

         dfsa->boot->f2.RootCluster = 2;
         TxPrint( "\nRoot directory now set at default cluster 2, if this was\n"
                  "a working and formatted FAT32 partition, you might need to\n"
                  "use 'findroot', 'setroot' and 'fixboot' to get the right\n"
                  "reference to the root directory (RECOVERY)\n\n");
      }
      else                                      // FAT16
      {
         ULONG         Clust = fat->Sect;

         memcpy( &(Signature[1]), sg_fat16, SG_FAT16);
         if ((fat->Fat1 = (dfsLocateSignatureSector(
                              1, 100, Signature, SG_FAT16 +1, 0))) == 0)
         {
            Check2ndFAT = FALSE;
            fat->Fat1   = 0x01;
            TxPrint( "not found, using default 0x01 (LSN 1)\n"
                     "FIXBOOT will NOT be reliable!\n");
         }
         else
         {
            TxPrint( "found at 0x%4.4x (LSN % lu)\n", fat->Fat1, fat->Fat1);
         }
         fat->FatBits     = 16;
         for ( ClustSize  = 1;
              (ClustSize <= 64) && (Clust > 65520L);
               ClustSize *= 2)                  // final value can be 128!
         {
            Clust /= 2;                         // divided until in range ...
         }
         fat->Roots   = 0x20;                   // usually 512 entries ...
         FatSectors   = 256;                    // maximum for FAT16
      }
      fat->RawClust   = fat->Sect / ClustSize;
      fat->ClustSize  = ClustSize;

      fat->Fat2 = fat->Fat1 + 9;                // default floppy sized FAT

      if (Check2ndFAT == TRUE)                  // OK to check
      {
         TxPrint( "Searching 2nd FAT area ... ");
         if ((fat->Fat2 = (dfsLocateDuplicateSector(fat->Fat1, FatSectors,
                                                    dfsGetSectorSize()))) != 0)
         {
            TxPrint( "found at 0x%4.4x (LSN % lu)\n", fat->Fat2, fat->Fat2);
         }
         else
         {
            TxPrint( "not found, using default (floppy sized)\n"
                     "\nFIXBOOT will not be reliable!\n");
         }
      }
      FatSectors = fat->Fat2 - fat->Fat1;       // real value

      TRACES(("FatSectors calculated as: %u\n", FatSectors));

      fat->Root  = fat->Fat2 + FatSectors;
      if (fat->FatBits == 32)
      {
         fat->ClTwo = fat->Root;
      }
      else                                      // FAT16
      {
         fat->ClTwo = fat->Root + fat->Roots;
      }
      fat->FatSectors = FatSectors;
      fat->MapClust   = fat->RawClust;
      nav.down        = fat->Root;              // next sector to visit

      TxPrint( "\nRestricted functionality, CHECK and SLT functions disabled!\n"
               "\nWhen it has been created new, you only need to FORMAT it!\n"
               "(and ignore most other warnings given here :-)\n\n");
      fat->FatFaked = TRUE;                     // FAT values faked, no bootsec
   }
   dfstSetClusterSize(DFSTORE, fat->ClustSize); // make sure it is set!

   dfsa->FsEntry = fat->Root;                   // entry-point in filesystem

   fat->TruncCluster  = fat->RawClust;          // First free cluster, initial
                                                // until calculated in ALLOC
   if ((fat->FatBits != 12) && (dfsa->warnLevel > 1))
   {
      ULONG         mfs;                        // minimum fat sectors

      mfs = ((fat->RawClust -1) / ((fat->FatBits == 32) ? 128 : 256)) +1;
      if (mfs < fat->FatSectors)
      {
         TxPrint("\nWarning: Number of FAT-sectors allocated is excessive!\n");
         TxPrint( "Needed FAT sectors: %8.8x  ", mfs);
         dfsSiz8( "Wasted disk space : ", (2 * (fat->FatSectors - mfs)), "");
      }
   }

   sprintf( text, "Size of FAT entry : %u  Bits", fat->FatBits);
   if (dfsFatRootLabel(dfsa->boot, LSN_BOOTR, label))
   {
      sprintf( tbuf, "%s%s%s", CBC, label, CNN);
   }
   else
   {
      sprintf( tbuf, "--none--");
   }
   TxPrint( "%s    FAT label in Root : %s\n", text, tbuf);

   fat->MappedSize = (fat->MapClust - 2) * fat->ClustSize + fat->ClTwo;
   if      (fat->FatBits == 12)                 // FAT12, no real expand implemented
   {
      fat->ExpandSize = fat->MappedSize;
   }
   else if (fat->FatBits == 16)                 // FAT16, max clusters, same CL size
   {
      fat->ExpandSize = (FAT_MAXCLF16  - 2) * fat->ClustSize + fat->ClTwo;
   }
   else                                         // FAT32, no limit cl#, same CL size!
   {
      if (((DFS32MAX - fat->ClTwo) / fat->ClustSize) > (FAT_MAXCLUST - 2)) // no overflow ?
      {
         fat->ExpandSize = (FAT_MAXCLUST  - 2) * fat->ClustSize + fat->ClTwo;
      }
      else                                      // overflow, limit to 32-bit sectornumbers
      {
         fat->ExpandSize = DFS32MAX;
      }
   }

   if (dfsa->verbosity > TXAO_QUIET)
   {
      sprintf(   text, "Size, in clusters : 0x%8.8X  ", fat->RawClust);
      dfstrSiz8( text, "Number of sectors : ",          fat->Sect, "");
      TxPrint( "%s\n", text);

      sprintf(   text, "Cl. mapped by FAT : 0x%8.8X  ", fat->MapClust);
      dfstrSiz8( text, "Maximum FSys size : ",          fat->ExpandSize, "");
      TxPrint( "%s\n", text);

      strcpy(    text, "");
      dfstrX10(  text, "1st FAT start LSN : ", fat->Fat1,  CBC, "  ");
      dfstrX10(  text, "2nd FAT start LSN : ", fat->Fat2,  CNC, "");
      TxPrint( "%s\n", text);

      strcpy(    text, "");
      dfstrX10(  text, "RootDirectory LSN : ", fat->Root,  CBY, "  ");
      if (fat->FatBits == 32)
      {
         TRACES(( "FAT32: dfsa->boot: %8.8x  fat32: %8.8x\n", dfsa->boot, fat32));

         dfstrX10(  text, "Root-Dir  Cluster : ",
                           dfsa->boot->f2.RootCluster, CBM, "\n");
         TxPrint(   text);
         TxPrint( "FreespaceInfo LSN : 0x%8.8hX  ", dfsa->boot->f2.FsInfoLsn);
         TxPrint( "Spare bootsec LSN : 0x%8.8hX\n", dfsa->boot->f2.SpBootLsn);

         sprintf(   text, "FAT32 free clust. : 0x%8.8X  ", fat32->FreeClusters);
         dfstrSiz8( text, "Freespace sectors : ",         (fat32->FreeClusters *
                                                           fat->ClustSize), "");
         TxPrint( "%s\n", text);
      }
      else
      {
         dfstrSiz8( text, "Root-size sectors : ", fat->ClTwo - fat->Root, "\n");
         TxPrint(   text);
      }

      strcpy(   text, "");
      dfstrX10( text, "1st DataClust LSN : ", fat->ClTwo,  CNG, "");
      TxPrint( "%s\n", text);
   }

   TRACES(("Allocate memory for %u fat-entries\n", (USHORT) DFSFAT_CACHE));
   if (((fat->CacheA.Value = TxAlloc(DFSFAT_CACHE, sizeof(ULONG))) != NULL) &&
       ((fat->CacheB.Value = TxAlloc(DFSFAT_CACHE, sizeof(ULONG))) != NULL)  )
   {
      fat->FatInUse = fat->Fat1;
      if ((rc = dfsFatGetFAT( 0, &fat->CacheA)) == NO_ERROR)
      {
         ULONG         crc1 = 0;
         ULONG         crc2 = 0;
         ULONG         size;
         DFSPARTINFO  *p;


         rc = dfsFatEaInit( &fat->EaData);      // Initialize EA data info
         if ((rc == NO_ERROR) && (dfsa->verbosity > TXAO_QUIET))
         {
            sprintf(   text, "EA data fragments : %-8u    ", fat->EaData.Chunks);
            dfstrSiz8( text, "EA header sectors : ",         fat->EaData.Sectors, "");
            TxPrint( "%s\n", text);

            TRHEXS( 70, fat->EaData.Map, 0x40, "fat->EaData.Map");
         }
         else if (rc == DFS_BAD_STRUCTURE)
         {
            TxPrint( "OS/2 EA data file : %s seems corrupted!\n", FAT_EADATA_FILE);
         }
         dfsa->FsSltRecSize = fat->RawClust +1000; // max records in SLT

         //- Report the FAT dirty status here, from Bootrec AND the Fatareas
         FsysValue  = dfsFatDirtyStatusBootrec( FALSE, 0); // just get BOOTREC value
         FsysValue |= dfsFatDirtyStatusFatarea( FALSE, 0); // and merge FATarea value
         TxPrint("FAT DIRTY status  : 0x%2.2hx = ", FsysValue);
         if (FsysValue == 0)
         {
            TxPrint( "CLEAN\n");
         }
         else if (FsysValue <= 0x03)
         {
            TxPrint( "DIRTY, surface-SCAN/CHKDSK required (MS)\n");
         }
         else
         {
            TxPrint( "DIRTY, non-standard value or read/write error occured\n");
         }

         if (((fat->Fat2 - fat->Fat1) > 0) && (Check2ndFAT == TRUE))
         {
            #if defined (USEWINDOWING)
               if (txwIsWindow( TXHWND_DESKTOP))
               {
                  sprintf( text, " Comparing first and second FAT area ... ");
                  txwSendMsg( dfsa->sbwindow, TXWM_STATUS, (TXWMPARAM) text, (TXWMPARAM) cSchemeColor);
               }
            #endif

            crc2 = dfsFatCalculateFatCRC( fat->Fat2); // FAT2 before 1, to avoid
            crc1 = dfsFatCalculateFatCRC( fat->Fat1); // it staying the active!
            TxPrint( "FAT1 CRC-32 value : 0x%8.8x  "
                     "FAT2 CRC-32 value : 0x%s%8.8x%s = %s\n", crc1,
                                  (crc1 == crc2) ? CBG  : CBR,  crc2, CNN,
                                  (crc1 == crc2) ? "OK" : "out of sync!",  crc2, CNN);

            if (TxaOption('a'))                 // show NO allocation by default
            {
               dfsFatAllocMap( 0, 0, "@", NULL);   // get/show Min/Max size
               if ((p = SINF->p) != NULL)       // we have a real partition
               {
                  if ((size = p->expandSize) != 0) // freespace to expand
                  {                             // but perhaps too small FATs!
                     size = min( p->expandSize, fat->ExpandSize);
                  }
                  if (size > p->cSC)            // prevent rounding up for cyl boundary
                  {
                     ULONG   EntPerClust;
                     ULONG   newRawClust;
                     ULONG   kmgAlign = dfsAlignmentSectors((p->tablenr == GPT_STYLE) ? 'G' : 'P', p);

                     dfsa->FsExpandSize = size - ((p->basePsn + size) % kmgAlign);

                     // Approximate nr of clusters available in maximum size
                     EntPerClust  = (fat->ClustSize * dfsGetSectorSize() * 8) / fat->FatBits;
                     newRawClust  = (dfsa->FsExpandSize - fat->Fat1) / fat->ClustSize;
                     newRawClust -= ((newRawClust / EntPerClust) * 2);

                     dfsSz64( "Max. size / Clust : ", dfsa->FsExpandSize, " ");
                     TxPrint( "/ 0x%8.8x Cl   (for Expanding)\n", newRawClust);
                  }
               }
            }
            if (crc1 != crc2)
            {
               TxPrint("\nWarning: The first and second FAT areas are NOT identical!\n"
                       "This could be the result of a crash, virus, or a damaged disk.\n"
                       "You could use the operating systems CHKDSK or SCANDISK, or simply\n"
                       "use the DFSee FATSYNC command directly (see docs in DFSFAT.TXT).\n");
            }
         }
      }
      nav.up   = fat->Fat1;
      nav.xtra = fat->Root;

      if (fat->MapClust < fat->RawClust)        // not all mapped ?
      {
         TxPrint("\nWarning: Number of sectors in the bootsector is LARGER than accounted for\n"
                 "         by the number of FAT-sectors, there are %s%u unmapped clusters!%s\n",
                           CBR, fat->RawClust - fat->MapClust, CNN);
         TxPrint("         May cause crashes with some operating systems/drivers (FAT32.IFS)\n");

         if (SINF->p != NULL)                   // we have a real partition
         {
            TxPrint("         Can be fixed by a DFSee partition resize to a size of : %s%u%s MiB\n",
                              CBG, (fat->MappedSize - dfsCylinderSize()) / 2048, CNN);
         }
      }
   }
   else
   {
      TxFreeMem( fat->CacheA.Value);
      TxFreeMem( fat->CacheB.Value);
      rc = DFS_ALLOC_ERROR;
   }
   TRACES(( "Fat1:%llx Fat2:%llx InUse:%llx\n", fat->Fat1, fat->Fat2, fat->FatInUse));
   TRACES(( "Sectors: %llx  RawClust: %8.8x  ClTwo: %llx  ClustSize: %hu\n",
             fat->Sect,  fat->RawClust,    fat->ClTwo,    fat->ClustSize));
   RETURN (rc);                                 // when needed
}                                               // end 'dfsFatInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Close FAT filesystem for analysis and free any resources
/*****************************************************************************/
static ULONG dfsFatClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   dfsSlTableReset();                           // stop SLT thread and reset
   if ((fat->CacheA.Dirty) || (fat->CacheB.Dirty))
   {
      TxPrint( "\nWriting pending changes to FAT-areas on previously opened FS ...\n");
      if (fat->CacheA.Dirty)
      {
         rc = dfsFatFlushFAT( &fat->CacheA);
      }
      if (fat->CacheB.Dirty)
      {
         rc = dfsFatFlushFAT( &fat->CacheB);
      }
   }
   TxFreeMem(fat->CacheA.Value);
   TxFreeMem(fat->CacheB.Value);
   dfsFatEaTerminate( &fat->EaData);            // Remove any existing info
   TxFreeMem( dfsa->findSpace.space);           // free the ALLDIRS SPACE
   dfsCloseFileSystem();
   RETURN (rc);
}                                               // end 'dfsFatClose'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Initialize FAT filesystem analysis for Area (FS info in FDISK mode)
/*****************************************************************************/
ULONG dfsFatAreaInit
(
   void
)
{
   ULONG               rc = NO_ERROR;
   USHORT              bits  = 0;

   ENTER();

   dfsa->FsAreaClose        = dfsFatAreaClose;
   dfsa->FsAreaLsnAllocated = dfsFatAllocated;

   dfsa->boot = (S_BOOTR *) brec;
   if ((rc = dfsRead( dfstAreaP2Disk( DFSTORE, LSN_BOOTR), 1, brec)) == NO_ERROR)
   {
      if ((dfsa->boot                != NULL    ) && // could be valid boot record
          (dfsa->boot->Signature     == SV_BOOTR) &&
          (dfsa->boot->eb.ClustSize  >= 1       ) &&
          (dfsa->boot->eb.ClustSize  <= 128     ) &&
          (dfsa->boot->eb.FatSectors <= 256     )  ) // is zero for FAT32
      {
         fat->Sect      =     max( dfsa->boot->eb.BigSectors,
                           (ULONG) dfsa->boot->eb.Sectors);
         fat->ClustSize = (USHORT) dfsa->boot->eb.ClustSize;
         fat->Fat1      =          dfsa->boot->eb.FatOffset;

         if ((bits == 0) && (strncasecmp( dfsa->boot->f2.Type, "FAT32", 5) == 0))
         {
            bits = 32;                          // assume 32-bit from bootrec
         }
         if (bits == 32)                        // forced or recognized FAT32
         {
            fat->FatSectors = dfsa->boot->f2.FatSectors;
            fat->Fat2  = fat->Fat1 + fat->FatSectors;

            fat->Roots =             fat->ClustSize; // minimum size
            fat->ClTwo = fat->Fat2 + fat->FatSectors;

            fat->Root  = dfsFatClust2LSN( dfsa->boot->f2.RootCluster);
         }
         else
         {
            fat->FatSectors = dfsa->boot->eb.FatSectors;
            fat->Fat2  = fat->Fat1 + fat->FatSectors;
            fat->Root  = fat->Fat2 + fat->FatSectors;
            fat->Roots = (dfsa->boot->eb.RootEntries * sizeof(S_FATDIR)) / dfsGetSectorSize();
            fat->ClTwo = fat->Root + fat->Roots;
         }

         if (fat->ClTwo < fat->Sect)
         {
            fat->RawClust = ((fat->Sect - fat->ClTwo) / fat->ClustSize) +2;
         }
         else
         {
            fat->RawClust = (fat->Sect - 1) / fat->ClustSize;
         }
         if (bits == 0)                         // not known yet
         {
            if      (fat->RawClust  > (FAT_MAXCLUST & 0xffff)) bits = 32;
            else if (fat->RawClust  > (FAT_MAXCLUST & 0xfff )) bits = 16;
            else if (fat->ClustSize > 8)                       bits = 16; // resized ?
            else                                               bits = 12;
         }
         fat->FatBits = bits;
         fat->MapClust = fat->FatSectors * (dfsGetSectorSize() * 8 / bits); // mapped clusters
      }
      TRACES(("Allocate memory for %u fat-entries\n", (USHORT) DFSFAT_CACHE));
      if (((fat->CacheA.Value = TxAlloc(DFSFAT_CACHE, sizeof(ULONG))) != NULL) &&
          ((fat->CacheB.Value = TxAlloc(DFSFAT_CACHE, sizeof(ULONG))) != NULL)  )
      {
         fat->FatInUse = fat->Fat1;
         rc = dfsFatGetFAT( 0, &fat->CacheA);
      }
      else
      {
         TxFreeMem( fat->CacheA.Value);
         TxFreeMem( fat->CacheB.Value);
         rc = DFS_ALLOC_ERROR;
      }
   }
   TRACES(( "Fat1:%llx Fat2:%llx InUse:%llx\n", fat->Fat1, fat->Fat2, fat->FatInUse));
   TRACES(( "Sectors: %llx  RawClust: %8.8x  ClTwo: %llx  ClustSize: %hu\n",
             fat->Sect,  fat->RawClust,    fat->ClTwo,    fat->ClustSize));
   RETURN (rc);                                 // when needed
}                                               // end 'dfsFatAreaInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Close FAT filesystem for Area analysis and free any resources
/*****************************************************************************/
static ULONG dfsFatAreaClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   dfsa->FsAreaClose        = NULL;
   dfsa->FsAreaLsnAllocated = NULL;

   if (fat->CacheA.Dirty)                       // should not happen in area ...
   {
      rc = dfsFatFlushFAT( &fat->CacheA);
   }
   if (fat->CacheB.Dirty)
   {
      rc = dfsFatFlushFAT( &fat->CacheB);
   }
   TxFreeMem(fat->CacheA.Value);
   TxFreeMem(fat->CacheB.Value);
   RETURN (rc);
}                                               // end 'dfsFatAreaClose'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// (Re)Initialize EA data information and caching information
/*****************************************************************************/
ULONG dfsFatEaInit
(
   DFSFATEA           *EaData                   // INOUT EA data info structure
)
{
   ULONG               rc = NO_ERROR;
   ULONG               chunks;                  // chunks in space structure
   S_SPACE            *space;                   // space structure
   S_FATDIR            eadata;                  // EAdata directory entry
   ULONG               dummy;

   ENTER();

   dfsFatEaTerminate( EaData);                  // Remove any existing info

   if (dfsDirLsn2Space(fat->Root, &chunks, &space) == NO_ERROR)
   {
      if (dfsFatDirSpace2Entry( FAT_EADATA_FILE, chunks, space, NULL, NULL, &eadata))
      {
         dfsFatAllocChain( dfsFatDir2Clust( &eadata), // cluster of EA data
                           eadata.fsize,
                           &dummy,
                           &fat->EaData.Chunks, // save allocation info
                           &fat->EaData.Space);
         if (fat->EaData.Chunks > 0)
         {
            S_EAINDEX      eai;

            if (dfsRead(fat->EaData.Space[0].start, 1, (BYTE *) &eai) == NO_ERROR)
            {
               TRACES(("eai.signature: 0x%4.4hx\n", eai.signature));
               if (eai.signature == SV_EAINDEX) // seems valid
               {
                  fat->EaData.Sectors = eai.base[0] * fat->ClustSize;
               }
               else                             // Corrupted EA's ?
               {
                  fat->EaData.Sectors = max( 2, fat->ClustSize);
               }
            }
            fat->EaData.MapSize = (fat->EaData.Sectors -1) *
                                   dfsGetSectorSize() / sizeof(USHORT);

            if (((fat->EaData.Index = TxAlloc( fat->EaData.Sectors,
                                               dfsGetSectorSize())) != NULL))
            {
               rc = dfsSspaceReadFilePart( fat->EaData.Chunks,
                                           fat->EaData.Space,
                                           0,   // Index starts at 1st sector
                                           fat->EaData.Sectors,
                                  (BYTE *) fat->EaData.Index);
               fat->EaData.Map = &fat->EaData.Index->mapsector[0];
               TRACES(("EA data: Chunks: %u, MapSize: %u, Sectors: %u\n",
                          fat->EaData.Chunks,
                          fat->EaData.MapSize,
                          fat->EaData.Sectors));
               TRHEXS(70, fat->EaData.Space, 0x20, "fat->EaData.Space")
               TRHEXS(70, fat->EaData.Index, 0x40, "fat->EaData.Index")
               TRHEXS(70, fat->EaData.Map  , 0x20, "fat->EaData.Map  ")

               TRACES(("eai.signature: 0x%4.4hx\n", eai.signature));
               if (eai.signature != SV_EAINDEX) // overrule on invalid sig
               {
                  rc = DFS_BAD_STRUCTURE;
               }
            }
            else
            {
               rc = DFS_ALLOC_ERROR;
            }
         }
      }
      free( space);
   }
   RETURN (rc);
}                                               // end 'dfsFatEaInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Terminate EA data information and caching information
/*****************************************************************************/
void dfsFatEaTerminate
(
   DFSFATEA           *EaData                   // INOUT EA data info structure
)
{
   ENTER();

   TRACES(("EA data: Chunks: %u, MapSize: %u, Sectors: %u\n",
              fat->EaData.Chunks,
              fat->EaData.MapSize,
              fat->EaData.Sectors));
   TRHEXS(70, fat->EaData.Space, 0x20, "fat->EaData.Space")
   TRHEXS(70, fat->EaData.Map  , 0x20, "fat->EaData.Map  ")
   EaData->Chunks  = 0;
   EaData->MapSize = 0;
   EaData->Sectors = 0;
   if (EaData->Space != NULL)
   {
      free( EaData->Space);
      EaData->Space = NULL;
   }
   TRACES(("Space freed\n"));
   TxFreeMem( EaData->Index);
   TRACES(("Map   freed\n"));
   VRETURN();
}                                               // end 'dfsFatEaTerminate'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Search and list possible Root directory clusters for a FAT32 partition
/*****************************************************************************/
static ULONG dfsFatFindFat32Roots               // RET   nr of candidates found
(
   void
)
{
   ULONG               rc = 0;                  // function return
   ULONG               cBase;                   // lowest cl in cache
   ULONG               cOffs;                   // offset cl in cache
   ULONG               cFirst;                  // first offset to check
   ULONG               cLast;                   // last offset to check
   ULONG               this;                    // current cluster number
   ULONG               value;                   // cluster value
   ULONG               continuation = 0;        // last continuation cluster

   ENTER();

   fat->Root    = 0;                            // define dummy root to avoid
   fat->Roots   = 0;                            // false findings

   TxPrint( "\nSearching for possible rootdirectory clusters ...\n\n");
   dfsInitList( 0, "-w", "-f -P");              // with optimal list options

   fat->CacheB.First  = FAT_NO_CLUST;           // invalidate B cache

   for ( cBase  = 0;                            // Big steps, whole cache ...
        (cBase <= fat->MapClust) && !TxAbort();
         cBase += DFSFAT_CACHE)
   {
      dfsFatGetFAT( cBase, &fat->CacheA);       // cache needed part of FAT

      cFirst = (cBase == 0) ? 2 : 0;            // first valid cl is 2
      if ((cBase + DFSFAT_CACHE) > fat->MapClust)
      {
         cLast = fat->MapClust % DFSFAT_CACHE;     // last valid offset
      }
      else
      {
         cLast = DFSFAT_CACHE -1;               // last offset within cache
      }
      for (cOffs = cFirst; (cOffs <= cLast) &&  !TxAbort(); cOffs++)
      {
         this  = cBase + cOffs;                 // this cluster nr
         value = fat->CacheA.Value[cOffs];          // cluster value (next/free)

         //- to be refined, use Sectorlist to cache more continuation clusters
         //- this is a speed optimization only (hardly needed with FAT32 spare)

         if ((this  != continuation) &&         // not next in a chain
             (value != FAT_FREECLUS) &&         // not a free cluster
             (value != FAT_BADCLUST)  )         // and not a BAD cluster
         {
            if (dfsFatRootCandidate( this))     // DIR sector without '..'
            {
               if (dfsFatFindReference( this) == 0) // not a DIR continuation
               {
                  ULN64  lsn = dfsFatClust2LSN( this);

                  dfsAdd2SectorList( lsn);

                  TxPrint( "Possible root DIR : %8.8x = LSN: 0x0%llx\n", this, lsn);
                  rc++;
               }
            }
         }
         continuation = value;
      }
   }
   RETURN (rc);
}                                               // end 'dfsFatFindFat32Roots'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Interpret and execute specific FAT command;
/*****************************************************************************/
static ULONG dfsFatCommand
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *none,                    // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;
   ULN64               sn = 0;                  // sector number input
   LONG                nr = 0;
   TXLN                dc;                      // DFS command
   int                 cc;                      // command string count
   char               *c0, *c1, *c2;            // parsed command parts
   char               *s;
   TXLN                s1;                      // big temporary string space
   TXLN                s2;                      // and one more ...
   ULONG               size  = 1;               // default display size
   time_t              stime;

   ENTER();

   time( &stime);                               // timestamp value

   TxaGetArgString( TXA_CUR, 0, 0, TXMAXLN, dc); // dc => cmd from arg 0
   cc = TxaArgCount( );                         // number of parameters
   c0 = TxaArgValue(0);
   c1 = TxaArgValue(1);
   c2 = TxaArgValue(2);

   sn = nav.this;                               // default at current sector

   if ((strcmp(c0, "/?") == 0) ||
       (strcmp(c0,  "?") == 0)  )
   {
      TxShowTxt( fat_txt);
   }
   else if (strcasecmp(c0, "bootini"  ) == 0)   // find (first) boot.ini
   {
      if (TxaOption('?'))
      {
         TxPrint("\nDisplay and optionally fix the partition-index to boot for\n"
                   "the default line in the Windows-NT/W2K/XP BOOT.INI file.\n");
         TxPrint("\n Usage:  %s  [* | index]   [-2] [-c]\n\n"
                   "   *     = use DFSee calculated value  as new partition-index\n"
                   "   index = use specified numeric value as new partition-index\n\n"
                   "   -2    = update/synchronize index in 2nd BOOT.INI line too\n\n"
                   "   -c    = work on current sector, do not search BOOT.INI\n", c0);
      }
      else
      {
         if (TxaOption('c'))                    // force current sector ?
         {
            sn = nav.this;                      // use current, no search
            TRACES(("-c = Assume bootini is in current sector: %llx\n", sn));
         }
         else
         {
            nav.this = min( fat->ClTwo, fat->Root);
            TxPrint( "\nSearch for the (first) boot.ini file in the filesystem ...\n");
            strcpy( dc, "find -t:Dd -o:$%]Ff -a:'boot    ini'");
            rc = dfsMultiCommand( dc, 0, TRUE, FALSE, TRUE);
            if (rc == NO_ERROR)                 // boot.ini seems to be there
            {
               sn = nav.down;                   // LSN for 1st data cluster
            }
         }
         if (rc == NO_ERROR)                    // boot.ini seems to be there
         {
            rc = dfsCheckFixNtBootIni( sn, c1, TxaOption('2'));
         }
         else                                   // no boot.ini found, return
         {                                      // calculated boot.ini id
            if (SINF->p != NULL)                // do we have a partition id ?
            {
               rc = SINF->p->diskPart;          // ARC style partition index
               dfsa->explain = FALSE;           // don't give incorrect
            }                                   // explanations about this :-)
         }
      }
   }
   else if (strcasecmp(c0, "fatselect"   ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nSelect to use First or Second FAT area\n");
         TxPrint("\n Usage:  %s  1 | 2\n\n", c0);
      }
      else
      {
         nr = dfsGetMcsNumber( c1, 1);
         dfsFatSetFatInUse((ULN64) (nr != 2) ? fat->Fat1 : fat->Fat2);
         rc = dfsFatGetFAT(  0, &fat->CacheA);
         if (rc == NO_ERROR)
         {
            dfsX10("\nSuccessfully read FAT sectors from LSN : ", fat->FatInUse, CBG, "\n");
         }
      }
   }
   else if (strcasecmp(c0, "fatshow"     ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nShow contents of (part of) the FAT area\n");
         TxPrint("\n Usage:  %s  [*|size] [-c:start]\n\n", c0);
         TxPrint("   size      = Number of cluster values to show, '*' ia ALL\n");
         TxPrint("   -c:start  = Start cluster number to show, default is 0\n\n");
      }
      else
      {
         size = 160;
         if (cc > 1)
         {
            if (c1[0] == '*')
            {
               size = fat->MapClust +1;
            }
            else
            {
               size = min((ULONG) atol(c1), fat->MapClust +1);
            }
         }
         dfsFatShowFAT( size, TxaOptNum('c', NULL, 0), c1);
      }
   }
   else if (strcasecmp(c0, "fatset" ) == 0)
   {
      ULONG         cluster;
      ULONG         size;

      if ((cc > 1) && !TxaOption('?'))
      {
         if (c1[0] == '.')
         {
            ULONG      entry;

            if (sscanf(&(c1[1]),"%d", &nr) != 1)
            {
               nr = 0;
            }
            sn    = dfsa->snlist[ nr + 1];
            entry = dfsa->sninfo[ nr + 1];

            cluster = dfsLsnInf2Cluster(  sn, entry);
            size    = dfsLsnInf2FileSize( sn, entry);
         }
         else
         {
            sscanf( c1,"%x", &cluster);

            size = TxaOptNum( TXA_O_SIZE, NULL, 1); // default to 1-byte file (one cluster)
         }

         //- silently ignore directries, empty files and invalid cluster values
         if ((size != 0) && (cluster >= 2) && (cluster <= fat->MapClust))
         {
            //- Set the actual allocation-chain using FatSetCluster() in a loop

            nr = (size + (fat->ClustSize * dfsGetSectorSize()) - 1) /
                         (fat->ClustSize * dfsGetSectorSize());

            if (!TxaOptUnSet('v'))              // unless supressed with -v-
            {
               TxPrint("Set FAT alloc chain starting at Cluster: 0x%x for %d clusters (%u bytes)\n", cluster, nr, size);
            }
            while ((nr > 0) && (rc == NO_ERROR))
            {
               rc = dfsFatSetCluster( cluster, (nr > 1) ? cluster + 1 : FAT_MAXEOFCL);
               if (rc == DFS_NO_CHANGE)
               {
                  rc = NO_ERROR;                // ignore overwriting with same value
               }
               cluster++;                       // next cluster in chain
               nr--;                            // remaining clusters to do
            }
         }
      }
      else
      {
         TxPrint( "\nSet FAT-allocation chain for specified cluster/size combination\n"
                    "(assuming the file allocation is contiguous, no fragmentation!)\n");
         TxPrint( "\nUsage: %s   cluster [-size:filesize] | .NNNNN\n"
                  "\n  cluster         hex cluster number, 1st cluster used by file/dir"
                  "\n  -size:filesize  filesize in bytes, decimal, default one cluster\n"
                  "\n  .NNNNN          sectorlist index, to get the LSN+entry value to"
                  "\n                  retrieve the 1st cluster and size from DIR-entry"
                  "\n                  (All directories and empty files are ignored!)\n",
                  "\n  -v-             Non-verbose operation, process silently\n", c0);
      }
   }
   else if (strcasecmp(c0, "fatsync"   ) == 0)
   {
      if (!TxaOption('?'))
      {
         if ((fat->Fat2 - fat->Fat1) > 0)
         {
            nr = 1;
            if (cc > 1)
            {
               nr = atol(c1);
            }
            if ((dfsa->batch) || (TxConfirm( 5123, "Synchronize both FAT areas by copying FAT area %hu "
                                                   "to FAT area %hu ? [Y/N] : ", nr, (nr == 2) ? 1 : 2)))
            {
               ULN64   fromLsn = (nr == 2) ? fat->Fat2 : fat->Fat1;
               ULN64   toLsn   = (nr == 2) ? fat->Fat1 : fat->Fat2;

               dfsProgressInit( fromLsn, fat->FatSectors, 0, "FAT Sector:", "synchronized", DFSP_BARS, 0);
               for (nr = 0; (nr < fat->FatSectors) && (rc == NO_ERROR); nr++, fromLsn++, toLsn++)
               {
                  if ((rc = dfsRead( fromLsn, 1, rbuf)) == NO_ERROR)
                  {
                     rc = dfsWrite( toLsn, 1, rbuf); // explicit CLONE sectors from one FAT to the other

                     dfsProgressShow( fromLsn + nr, 0, 0, NULL);
                  }
               }
               dfsProgressTerm();

               if (rc == NO_ERROR)
               {
                  TxPrint( "The FAT areas are successfully synchronized\n\n");
                  TxPrint( "You may want to run a filesystem check by the operating system "
                  #if   defined (DEV32)
                     "(CHKDSK /f:2)"
                  #elif defined (UNIX)
                     "like FSCK"
                  #elif defined (DOS32)
                     "like CHKDSK"
                  #else
                     "like SCANDISK"
                  #endif
                     "\nas reassurance and to make sure any other inconsistensies are fixed too\n");
               }
            }
            else
            {
               rc = DFS_USER_ABORT;
            }
         }
         else
         {
            TxPrint( "\nThere is just a single FAT area! Synchronize not possible!\n");
         }
      }
      else
      {
         TxPrint( "\nSynchronise both FAT areas, by copying one over the other\n");
         TxPrint( "\nUsage: %s  [1 | 2]\n"
                  "\n   1 | 2       FAT area to read from, 1st or 2nd, default 1\n"
                  "\n   The contents of the selected FAT is read sector by sector,"
                  "\n   then written to the other FAT area, to make them identical\n"
                  "\n   You may determine which area (1 or 2) to copy by selecting"
                  "\n   each one with FATSELECT, then run a CHECK command to find"
                  "\n   inconsistensies compared to the directory structure info\n\n", c0);
      }
   }
   else if (strcasecmp(c0, "fatsim"   ) == 0)
   {
      if ((cc > 1) && !TxaOption('?'))
      {
         nr = 1;
         if (cc > 2)
         {
            nr = atol(c2);
         }
         sprintf( dc, "image %s 0x0%llx 0x0%x", c1, (nr == 1) ? fat->Fat1 : fat->Fat2, fat->FatSectors);
         TxaReParseCommand( dc);
         if (TxaOption('z'))
         {
            TxaOptSetItem( "-z");               // set for image command too
         }
         rc = DFS_PENDING;
      }
      else
      {
         TxPrint( "\nSave all sectors for the FAT table to a file\n");
         TxPrint( "\nUsage: %s  filename  [1 | 2]  [-z]\n"
                  "\n   filename    name of the imagefile, raw .IMG or compressed .IMZ"
                  "\n   1 | 2       FAT area to read from, 1st or 2nd, default 1"
                  "\n   -z          Write the image in compressed format (IMZ)\n", c0);
      }
   }
   else if (strcasecmp(c0, "fatwrim"   ) == 0)
   {
      if ((cc > 1) && !TxaOption('?'))
      {
         if (cc > 2)
         {
            nr = atol(c2);                      // use specified FAT nr
         }
         else
         {
            nr = (fat->NrOfFats > 1) ? 2 : 1;   // default to 2nd FAT, if there
         }
         sprintf( dc, "wrim %s 0x0%llx 0x0%x", c1, (nr == 1) ? fat->Fat1 : fat->Fat2, fat->FatSectors);
         TxaReParseCommand( dc);
         rc = DFS_PENDING;
      }
      else
      {
         TxPrint( "\nRestore all sectors for the FAT table from a file\n");
         TxPrint( "\nUsage: %s  filename  [1 | 2]\n"
                  "\n   filename    name of the imagefile, raw .IMG or compressed .IMZ"
                  "\n   1 | 2       FAT area to write to, 1st or 2nd, default 2\n", c0);
         TxPrint( "\nMake sure to restore to the OTHER FAT-area to synchronize!"
                  "\nso, after FATSIM from 1, use a FATWRIM to 2 ...\n");
      }
   }
   else if (strcasecmp(c0, "findroot"       ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nFind possible root-directory on a FAT32 filesystem, to fixup bootsector link\n");
         TxPrint("\n Usage:  %s\n\n", c0);
      }
      else
      {
         if (fat->FatBits == 32)
         {
            ULONG         Roots = dfsFatFindFat32Roots();

            if (Roots != 0)
            {
               TxPrint( "\n%u possible Rootdirectories found, you can use the "
                        "'list -f' command\nto view them, 'export' and 'import'"
                        "to save and restore them and\nfinally 'setroot' and "
                        "'fixboot' to select one for further use.\n", Roots);
            }
            else
            {
               TxPrint( "Warning: No Root directory candidates found!\n");
            }
         }
         else
         {
            TxNamedMessage( !dfsa->batch, 0, " WARNING: Not a FAT32 filesystem ", "FINDROOT only useful on FAT32 filesystems!");
         }
      }
   }
   else if (strcasecmp(c0, "setroot"       ) == 0)
   {
      if ((cc == 1) || (TxaOption('?')))
      {
         TxPrint( "\nSet rootdirectory sector (LSN) for FAT32 bootsector\n");
         TxPrint( "\nUsage: %s   hex-sectornumber or .nnnn listnumber\n", c0);
      }
      else
      {
         if (fat->FatBits == 32)
         {
            ULONG         cluster;

            if (cc > 1)
            {
               if (c1[0] == '.')
               {
                  if (sscanf(&(c1[1]),"%d", &nr) != 1)
                  {
                     nr = 0;
                  }
                  sn = dfsa->snlist[ nr + 1];
               }
               else
               {
                  sscanf( c1,"%llx", &sn);
               }
               cluster = (ULONG) dfsFatLSN2Clust(sn); // invalids => cluster 2!

               fat->Root = dfsFatClust2LSN( cluster);
               TxPrint( "\nRoot directory at : %8.8x = cluster: %8.8x\n\n", fat->Root, cluster);
               TxPrint( "\nUse 'fixboot 9' to write this value to the bootsector\n");
            }
         }
         else
         {
            TxNamedMessage( !dfsa->batch, 0, " WARNING: Not a FAT32 filesystem ", "SETROOT is only useful on FAT32 filesystems!");
         }
      }
   }
   else if (strcasecmp(c0, "firstfree" ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nFind first free cluster, then if FAT32, set it as 'next cluster'\n");
         TxPrint("\n Usage:  %s\n\n", c0);
      }
      else
      {
         ULONG         cluster;

         TxPrint( "\nSearching first free cluster in the 1st FAT ...  ");

         for (cluster = 2; cluster < fat->MapClust; cluster++)
         {
            if (dfsFatValue( cluster) == 0)     // cluster is free
            {
               break;
            }
         }
         if (cluster < fat->MapClust)
         {
            TxPrint( "found at CL-number 0x%8.8x\n\n", cluster);
            if (fat->FatBits == 32)
            {
               if ((dfsa->batch) || (TxConfirm( 5125, "Do you want to set the FAT32 "
                                    "'next cluster' to 0x%8.8x ? [Y/N] : ", cluster)))
               {
                  if (DFSTORE_WRITE_ALLOWED)
                  {
                     rc = dfsRead( LSN_BOOTR + 1, 1, brec);
                     if (rc == NO_ERROR)
                     {
                        FAT32B2 *fat32 = (FAT32B2 *) brec;

                        fat32->NextSearchCl = cluster;

                        rc = dfsWrite( LSN_BOOTR + 1, 1, brec);
                        if (rc == NO_ERROR)
                        {
                           TxPrint("FAT32 2nd boot record updated '1st Cl to use' to 0x%8.8x\n", cluster);
                        }
                     }
                  }
                  else
                  {
                     rc = DFS_READ_ONLY;
                  }
               }
               else
               {
                  rc = DFS_NO_CHANGE;
               }
            }
         }
         else
         {
            TxPrint( "\n\nNo free cluster found, FAT volume seems to be FULL!\n");
         }
      }
   }
   else if (strcasecmp(c0, "path"     ) == 0)
   {
      ULONG    entry = 0;

      if (TxaOption('?'))
      {
         TxPrint("\nFind path to root, from specified, or current sector number\n");
         TxPrint("\n Usage:  %s  [ -e:entry | lsn  [entry]]\n\n", c0);
         TxPrint("   lsn     = sectornumber for a FAT directory sector, default current sector\n"
                 "   entry   = entry index in the FAT sector, range 0 .. 15, default 0\n"
                 "             (must be the entry for the SHORT filename, not lfn!)\n\n");
      }
      else
      {
         if ((dfsSlTableStatus(NULL)) != SLT_READY)
         {
            TxPrint( "\nBuilding SLT to resolve parent links (.. entries)\n");
            dfsa->sltProgress = TRUE;
            dfsSlTableCreate(dfsa->Fsi);        // build SLT now
            TxPrint( "\n");                     // terminate progress bars
         }
         if (TxaOptSet('e'))                    // e option specified ?
         {
            entry = DFSSNIGET(TxaOptNum('e', "Entry", 0)) | DFSSNINFO;
         }
         else
         {
            sscanf( c2, "%u", &entry);
         }
         sn = dfsGetSymbolicSN( c1, nav.this);  // default current LSN
         if (dfsFatLsnInfo2Path( sn, entry | DFSSNINFO, s1) == NO_ERROR)
         {
            TxPrint("\nFound path to root: '%s%s%s'\n", CBY, s1, CNN);
         }
         else
         {
            TxPrint("\nNo FAT path to Root found, %s\n",
                   (dfsSlTableStatus(NULL) == SLT_READY) ? "not a directory sector?" :
                                                           "need to build SLT first!");
            rc = DFS_NOT_FOUND;
         }
      }
   }
   else if (strcasecmp(c0, "dirty"    ) == 0)
   {
      BYTE             FsysValue = 0;           // display value
      BYTE             FsysDirty = 0;           // update value

      //- Note JvW: Two areas exist for the dirty flags, a byte in the bootsector
      //-           and two bits in the FAT-entry 1, the last one is documented best
      //-           eCS uses the bootsector-byte for FAT16 and eCS the FAT for FAT32
      //- For maximum effect, set BOTH, and merge results on read

      if (cc > 1)
      {
         switch (c1[0])
         {
            case 's': case 'S': FsysDirty = 2;                 break;
            case 'd': case 'D': FsysDirty = 1;                 break;
            case 'c': case 'C': FsysDirty = 0;                 break;
            default:            FsysDirty = (BYTE) atol( c1);  break;
         }
         FsysValue = dfsFatDirtyStatusBootrec( TRUE, FsysDirty); // update value
         if (FsysValue == FsysDirty)
         {
            FsysValue = dfsFatDirtyStatusFatarea( TRUE, FsysDirty); // update value
         }
      }
      else
      {
         FsysValue  = dfsFatDirtyStatusBootrec( FALSE, 0); // just get BOOTREC value
         FsysValue |= dfsFatDirtyStatusFatarea( FALSE, 0); // and merge FATarea value
         TxPrint("\nUsage: %s  clean | dirty | scan | status-value\n", c0);
         TxPrint("\n      0 = FS clean");
         TxPrint("\n      1 = CHKDSK required");
         TxPrint("\n      2 = Surface-SCAN required\n");
      }
      TxPrint("\nFAT DIRTY status  : 0x%2.2hx = ", FsysValue);
      if (FsysValue == 0)
      {
         TxPrint( "CLEAN\n");
      }
      else if (FsysValue <= 0x03)
      {
         TxPrint( "DIRTY, surface-SCAN/CHKDSK required (MS)\n");
      }
      else
      {
         TxPrint( "DIRTY, non-standard value or read/write error occured\n");
         rc = FsysValue;
      }
   }
   else if (strcasecmp(c0, "fixboot"  ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nFix the bootsector for a FAT16 or FAT32 partition\n");
         TxPrint("\n Usage:  %s  [os] [-s]\n\n", c0);
         TxPrint("   os        = Operating system to be booted from this FAT partition:\n\n");
         TxPrint("                 I or P   PC-DOS with IBMBIO/IBMDOS.COM\n"
                 "                 M        MSDOS  with IO/MSDOS.SYS\n"
                 "                 O        OS/2   with OS2BOOT\n"
                 "                 V        OS/2   with OS2LDR.SEK    (Veit Kannegieser)\n");
         TxPrint("                 N        Win-NT with NTLDR         (FAT16)\n"
                 "                 W        Win-9x with IO/MSDOS.SYS  (FAT16)\n"
                 "                 9        Win-9x with WINBOOT.SYS   (FAT32)\n\n");
         TxPrint("   -s        = Force a copy of the SPARE bootsector (FAT32)\n\n");
         TxPrint("   -!        = Present a selection dialog for the types above\n\n");
      }
      else
      {
         #if defined (USEWINDOWING)
         if (TxaOption('!'))                    // present interactive dialog
         {
            dfsFatFixBootDialog((fat->FatBits == 32) ? '2' : 'M');
         }
         else                                   // execute the FIXBOOT
         #endif
         {
            if ((fat->FatBits == 32) && ((TxaOption('s')) ||
                        (c1[0] == '1') || (c1[0] == '2'))  )
            {
               if ((rc = dfsRead( LSN_FAT32SPARE, 2, brec)) == NO_ERROR)
               {
                  dfsa->boot = (S_BOOTR *) brec;
                  switch (dfsIdentifySector(0, 0, brec))
                  {
                     case ST_BOOTR:
                        if (SINF->p == NULL)    // No partition info yet
                        {
                           dfstSetSectorSize( DFSTORE, (USHORT) dfsa->boot->eb.SectSize);
                           dfstSetClusterSize(DFSTORE, (USHORT) dfsa->boot->eb.ClustSize);
                        }
                        break;

                     default:
                        TxPrint("Sector %u is not a valid FAT32 BOOT record!\n",
                                 LSN_FAT32SPARE);
                        break;
                  }
                  fat->Fat32Spare = TRUE;       // force in all cases
               }
               else
               {
                  TxPrint( "Reading FAT32 spare-sectors failed!\n");
               }
            }
            if ((fat->FatBits == 32) &&         // FAT32 FS
                (fat->Fat32Spare)    &&         // Spare sector(s) available
                ((cc <= 1) ||                   // no params, or
                  strchr( "12", c1[0]) != NULL)) // 1 or 2 spare-sector copy
            {
               if (c1[0] != '2')                // copy both
               {
                  sprintf( s1, " (0) with the spare copy from sector 6");
               }
               else
               {
                  sprintf( s1, "s (0/1) with the spare copies from sector 6/7");
               }
               if ((dfsa->batch) || (TxConfirm( 5128, "Do you want to replace "
                                    "the FAT32 bootsector%s ? [Y/N] : ", s1)))
               {
                  if (DFSTORE_WRITE_ALLOWED)
                  {
                     rc = dfsWrite( LSN_BOOTR, (c1[0] == '2') ? 2 : 1, brec);
                     if (rc == NO_ERROR)
                     {
                        TxPrint("\nFAT32 boot record updated from spare ...\n");
                        if (SINF->partid != 0)  // was it a partition ?
                        {
                           TxPrint( "Resulting partition after the update:\n\n");
                           sprintf( dc, "part -r %hu", SINF->partid);
                           TxaReParseCommand( dc);
                           rc = DFS_PENDING;    // reread diskinfo & select
                        }
                     }
                  }
                  else
                  {
                     rc = DFS_READ_ONLY;
                  }
               }
               else
               {
                  rc = DFS_NO_CHANGE;
               }
            }
            else                                // create new Bootsector ...
            {                                   // with default values
               sn = dfstLSN2Psn( DFSTORE, 0);
               if ((rc = dfsFatMkBootRec( c1[0], TRUE, 0,
                                (ULONG) ( sn + stime), NULL)) == NO_ERROR)
               {
                  TRACES(("FatMkBoot OK, SINF->partid == %hu\n", SINF->partid));
                  if (SINF->partid != 0)        // was it a partition ?
                  {
                     TxPrint( "Resulting partition after fixing the bootsector:\n\n");
                     sprintf( dc, "part -r -q %hu#d", SINF->partid);
                     rc = dfsMultiCommand( dc, 0, FALSE, FALSE, TRUE);
                  }
               }
               else
               {
                  TRACES(("FatMkBoot failed with rc = %u\n", rc));
               }
            }
            TRACES(("fixboot rc = %u   dc = '%s'\n", rc, dc));
         }
      }
   }
   else if (strcasecmp(c0, "format" ) == 0)     // format current object
   {
      if (TxaOption('?'))
      {
         TxShowTxt( cmd_format);                // give usage
      }
      else                                      // calculate limits + defaults
      {                                         // based on size and fat->info
         FAT_FMT_INFO  fi;                      // format parameter structure
         ULONG         stdSects;                // sector count for std SECTORSIZE

         memset( &fi, 0, sizeof( fi));          // initialize to all zero

         fi.sectors      = dfsGetLogicalSize();
         fi.minimumCs_32 = dfsGetSectorSize() / SECTORSIZE; // minimal 1 'real' sector
         fi.allowedBits  = 32;                  // FAT32 always OK

         stdSects = fi.sectors * fi.minimumCs_32; // normalized sectorcount

         TRACES(( "Standard sector count (512 BPS): 0x%8.8x  actual: 0x%8.8x\n", stdSects, fi.sectors));

         if      (stdSects < FAT32_01LIM) fi.defaultCsize =  1;
         else if (stdSects < FAT32_02LIM) fi.defaultCsize =  2;
         else if (stdSects < FAT32_04LIM) fi.defaultCsize =  4;
         else if (stdSects < FAT32_08LIM) fi.defaultCsize =  8;
         else if (stdSects < FAT32_16LIM) fi.defaultCsize = 16;
         else if (stdSects < FAT32_32LIM) fi.defaultCsize = 32;
         else                             fi.defaultCsize = 64;

         if (fi.sectors < FAT12_LIMIT)
         {
            fi.allowedBits += 12;
            fi.minimumCs_12 = ((fi.sectors -1) / (FAT_MAXCLUST & 0xfff)) +1;
         }
         if (fi.sectors < FAT16_LIMIT)
         {
            fi.allowedBits += 16;
            fi.minimumCs_16 = ((fi.sectors -1) / (FAT_MAXCLUST & 0xffff)) +1;
         }

         fi.desiredBits  = TxaOptNum('f', NULL, 0);
         if ((fi.desiredBits & fi.allowedBits) == 0) // illegal desire :-)
         {
            fi.desiredBits  = (fi.allowedBits & 12) ? 12 :
                              (fi.allowedBits & 16) ? 16 : 32;
         }
         sn = dfstLSN2Psn( DFSTORE, 0);
         fi.longFormat   = TxaOption('l');
         fi.desiredRsize = TxaOptNum('r', NULL, (fi.allowedBits & 12) ? 224 : 512);
         fi.desiredCsize = TxaOptNum('c', NULL, 0);
         fi.serial       = TxaOptNum('s', NULL, (ULONG) (sn + stime));
         fi.reservedSect = TxaOptNum('a', NULL, (fi.desiredBits == 32) ? 32 : 1);
         sprintf( s1, "DFS%8.8x", fi.serial);  // default label
         strcpy( fi.label, TxaOptStr('v', NULL, s1));
         TxPClip(fi.label, FAT_N, ' ');         // pad spaces to 11 positions
         strcpy( fi.os   , TxaOptStr('o', NULL, "M"));
         TxStrToUpper( fi.os);

         fi.desiredRsize = ((((fi.desiredRsize -1) & 0xffff) / 16) +1) * 16;

         TRACES(( "FFI sectors: %8.8x  serial: %8.8x\n",    fi.sectors,      fi.serial));
         TRACES(( "FFI allowed:     %4.4hx  desire: %hu\n", fi.allowedBits,  fi.desiredBits));
         TRACES(( "FFI minClSize-12:  %2hu  minC16: %hu\n", fi.minimumCs_12, fi.minimumCs_16));
         TRACES(( "FFI desired-size:  %2hu default: %hu\n", fi.desiredCsize, fi.defaultCsize));
         TRACES(( "FFI Rsize:%hu long:%s label:'%s' os:'%s'\n", fi.desiredRsize,
                                                               (fi.longFormat) ? "YES" : "NO",
                                                                fi.label, fi.os));
         #if defined (USEWINDOWING)
         if (TxaOption('!'))                    // present interactive dialog
         {
            dfsFatFormatDialog( &fi);
         }
         else                                   // execute the FORMAT
         #endif
         {
            TXTS         osname;
            DFSPARTINFO *p = SINF->p;           // partition info

            if ((p == NULL) && (fi.desiredBits == 32)  && // large floppy FAT32
                (dfstStoreType( DFSTORE) == DFST_PHDISK))
            {
               sprintf( s2, "WARNING: FAT32 on whole disk is NOT supported on OS/2!"
               #if defined (DEV32)
                            "\nTo use FAT32, you need to create a partition first."
               #endif
                            "\n\n");
            }
            else
            {
               strcpy( s2, "");
            }

            dfsMakeFatOsString( fi.os[0], osname);

            fi.desiredCsize = dfsFatFmtClusterSize( &fi); // valid clustersize

            if ((dfsa->batch) || (TxConfirm( 5131,
                "      FORMAT confirmation for current object:\n\n%s\n\n"
                "To be formatted as FAT%hu with %hu sectors per cluster\n"
                "using bootcode based on '%s'\n\n%s"
                "All data on this volume/partition/disk will be lost!\n"
                "            Proceed with FORMAT ? [Y/N] : ",
                 dfsDriveDescription( s1), fi.desiredBits, fi.desiredCsize, osname, s2)))
            {
               BOOL        ok = TRUE;           // writing allowed/confirmed

               #if defined (DEV32)
                  char      letter = 0;         // driveletter to restore
                  TXTT      volume;             // LVM volumename

                  //- note that SyncDesired checks for NULL pointer too!
                  if ((dfsLvmSyncDesired( p))   && // partition, sync desired
                      (p->lvm.VoluName[0] != 0) && // and a volumename
                      (p->lvm.letter      != 0)  ) // and driveletter
                  {
                     letter = p->lvm.letter;
                     TxCopy( volume, p->lvm.VoluName, LVM_NAME_L);
                     dfsLvmSyncEngine( TRUE, FALSE, volume, 0); // hide volume
                  }
                  else                          // use locking mechanism
                  {
                     ok = dfstCheckWriteAccess( DFSTORE, TRUE);
                  }
               #else
                  ok = dfstCheckWriteAccess( DFSTORE, TRUE);
               #endif

               if (ok)                          // writing allowed/locked
               {
                  rc = dfsFatFormat( &fi);      // perform actual formatting

                  if (p != NULL)                // if it was a partition
                  {
                     if (rc == NO_ERROR)        // touch up
                     {
                        sprintf( s1, "FAT%2.2hu", fi.desiredBits);

                        if (p->tablenr != GPT_STYLE)
                        {
                           //- Update type in partitiontable, MBR-style only!
                           dfsSetPartTypeForFS( p, s1);
                        }
                        strcpy( p->fsform, s1); // update in internal tables

                        dfsReadDiskInfo( FDSK_QUIET); // re-read all partition info
                     }
                     //- auto reselect partition to refresh info, no screen output
                     dfsMultiCommand( "part . -a-", 0, FALSE, FALSE, FALSE);
                  }

                  #if defined (DEV32)           // need to re-assign LVM letter
                  if ((letter != 0) && (p->tablenr != GPT_STYLE) &&
                      (dfsLvmSyncEngine( TRUE, FALSE, volume, letter) == NO_ERROR))
                  {
                     TxPrint( "Formatted volume  : %c: is available "
                              "for use now, no reboot required.\n", letter);
                  }
                  else
                  #endif
                  {
                     dfstLocking( DFSTORE, DFS_UNLOCK, FALSE,  TRUE);
                     TxPrint( "A reboot may be required before "
                              "the volume can be accessed ...\n");
                  }
               }
               else
               {
                  TxPrint( "Formatting not possible, object is not writable ...\n");
               }
            }
         }
      }
   }
   else if ((strcasecmp(c0, "bsclear"   ) == 0) || // almost like 'bsimport -clear'
            (strcasecmp(c0, "nobads"    ) == 0)  ) // deprecated
   {
      if (TxaOption('?'))
      {
         TxPrint("\nReset bad-sector administration in both FAT areas\n");
         TxPrint("\n Usage:  %s\n\n", c0);
      }
      else
      {
         rc = dfsFatResetBadClus();
      }
   }
   else if (strcasecmp(c0, "space"       ) == 0)
   {
      sn = dfsGetSymbolicSN( c1, dfsFatLSN2Clust(nav.this));
      dfsFatShowAlloc( sn);                     // note: sn is a CLUSTER here!
   }
   else if (strcasecmp(c0, "filefind"  ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nFind normal files, directory entries containing [name]\n");
         TxPrint("\n Usage:  %s [-c] [-D] [-d-] [-v]  [name]\n\n", c0);
         TxPrint("   name      = part or whole filename wanted, NOT a true wildcard,\n"
                 "               do not use any '?' characters, and only use '*' to\n"
                 "               search for extensions as in '*.doc' or '*.c*'\n"
                 "   -c        = start from current sector, not start of volume\n"
                 "   -D        = search directories only, not files\n"
                 "   -d-       = search outside EXISTING directories too (SLOW!)\n"
                 "   -v        = verbose search, list files while found  (SLOW!)\n");
      }
      else
      {
         if (dfsSlTableStatus( NULL) != SLT_READY)
         {
            TxPrint( "\nBuilding SLT for filename lookup, please wait ..."
                     "   (or abort that using <Esc>)\n");
            dfsMultiCommand( "slt 0 1 -v- -d-", 0, TRUE, FALSE, FALSE);
            TxCancelAbort();                    // might have aborted
         }
         if (dfsa->findSpace.space == NULL)     // no DIR areas determined yet
         {
            //- Create the ALLDIRS SPACE structure (for 'find -o:S' etc)
            if (dfsFatAllDirs2Space( FALSE, &dfsa->findSpace.chunks, &dfsa->findSpace.space) == NO_ERROR)
            {
               sprintf(   s2, "Total # DIR areas : %-8u    ", dfsa->findSpace.chunks);
               dfstrSz64( s2, "Size in DIR areas : ",
                 dfsSspaceSectors( TRUE, dfsa->findSpace.chunks, dfsa->findSpace.space), "");
               TxPrint( "%s\n", s2);
            }
         }
         sprintf( dfsa->brdescript, "FAT %s %s", c0, c1);
         if (((c1[0] == '*') && (c1[1] == '.')) || // *.xxx extension
              (c1[0] == '.')                     ) // just .xxx extension
         {
            if (c1[0] == '*')
            {
               strcpy( s1, c1 + 2);
            }
            else
            {
               strcpy( s1, c1 + 1);
            }
            if ((s = strchr( s1, '*')) != NULL)
            {
               *s = 0;                          // terminate ext wildcard
            }
            else                                // make exact ext match
            {
               strcat( s1, "   ");
               s1[3] = 0;
            }
            sprintf( dc, "find -t:Dd -o:*$%%]%sMFe!%s",
                   ( TxaOptUnSet('d')) ? "" : "S", // DIR only (-o:S spaced)
                   ( TxaOption( 'v')) ? "" : "Q"); // verbose or quiet, FAST
            if (strlen( s1))
            {
               strcat( dc, " -a:'");
               strcat( dc,  s1);
               strcat( dc, "'");
            }
            TxPrint("Find files by EXT : %s%s%s\n", CBC, dc, CNN);
         }
         else                                   // generic find by name
         {
            if (strchr( c1, '.') != NULL)       // convert to FAT-DIR format
            {
               S_FATDIR   de;

               dfsFatName2Entry( c1, &de);
               de.FatAttrib = 0;                // zero terminate name :-)
               strcpy( s1, de.Name);
            }
            else
            {
               strcpy( s1, c1);                 // direct copy
            }
            sprintf( dc, "find -t:Dd -o:*$%%]%sMF%s!%s",
                   ( TxaOptUnSet('d')) ? ""  : "S",   //- DIR-areas only (-o:S spaced)
                   ( TxaOption(  'D')) ? "D" : "f",   //- find DIRs only
                   ( TxaOption(  'v')) ? ""  : "Q");  //- verbose or quiet, FAST
            if (strlen( s1))
            {
               strcat( dc, " -a:'");
               strcat( dc,  s1);
               strcat( dc, "'");
            }
            TxPrint("Find file (entry) : %s%s%s\n", CBC, dc, CNN);
         }

         if (!TxaOption('c'))                   // if not 'from current'
         {
            nav.this = fat->Fat2 + fat->FatSectors; // search from end of FAT
         }
         TxaReParseCommand( dc);
         rc = DFS_PENDING;                      // handle translated command
      }
   }
   else if (strcasecmp(c0, "delfind"  ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nFind deleted files, directory entries containing [name]\n");
         TxPrint("\n Usage:  %s [-c] [-d-] [-v]  [name]\n\n", c0);
         TxPrint("   name      = part or whole filename wanted, not a true wildcard,\n"
                 "               do not use any '*' or '?' characters\n\n"
                 "   -c        = start from current sector, not start of volume\n"
                 "   -d-       = search outside EXISTING directories too (SLOW!)\n"
                 "   -v        = verbose search, list files while found  (SLOW!)\n");
      }
      else
      {
         if (dfsSlTableStatus( NULL) != SLT_READY)
         {
            TxPrint( "\nBuilding SLT for filename lookup, please wait ..."
                     "   (or abort that using <Esc>)\n");
            dfsMultiCommand( "slt 0 1 -v- -d-", 0, TRUE, FALSE, FALSE);
            TxCancelAbort();                    // might have aborted
         }
         if (dfsa->findSpace.space == NULL)     // no DIR areas determined yet
         {
            //- Create the ALLDIRS SPACE structure (for 'find -o:S' etc)
            if (dfsFatAllDirs2Space( FALSE, &dfsa->findSpace.chunks, &dfsa->findSpace.space) == NO_ERROR)
            {
               sprintf(   s2, "Total # DIR areas : %-8u    ", dfsa->findSpace.chunks);
               dfstrSz64( s2, "Size in DIR areas : ",
                 dfsSspaceSectors( TRUE, dfsa->findSpace.chunks, dfsa->findSpace.space), "");
               TxPrint( "%s\n", s2);
            }
         }
         if (strchr( c1, '.') != NULL)          // convert to FAT-DIR format
         {
            S_FATDIR   de;

            dfsFatName2Entry( c1, &de);
            de.FatAttrib = 0;                   // zero terminate name :-)
            strcpy( s1, de.Name +1);            // skip first (DIRDEL) char
         }
         else
         {
            strcpy( s1, c1);                    // direct copy
         }
         sprintf( dfsa->brdescript, "FAT %s %s", c0, c1);
         sprintf( dc, "find -t:Dd -o:*$%%%sMFd!%s \"%s\"",
                   ( TxaOptUnSet('d')) ? "" : "S", // DIR only (-o:S spaced)
                   ( TxaOption(  'v')) ? "" : "Q", (cc > 1) ? s1 : "");
         TxPrint("Find deleted file : %s%s%s\n", CBC, dc, CNN);

         if (!TxaOption('c'))                   // if not 'from current'
         {
            nav.this = fat->Fat2 + fat->FatSectors; // search from end of FAT
         }
         TxaReParseCommand( dc);
         rc = DFS_PENDING;                      // handle translated command
      }
   }
   else if (strcasecmp(c0, "subfind"  ) == 0)
   {
      sprintf( dfsa->brdescript, "FAT %s %s", c0, c1);
      sprintf( dc, "find -t:");
      switch (c1[0])
      {
         case 's':
            strcat( dc, "s");
            TxPrint("Find subdirectory : from ROOT only : %s%s%s\n", CBC, dc, CNN);
            break;

         case 'd':
            strcat( dc, "d");
            TxPrint("Find subdirectory : non-ROOT only  : %s%s%s\n", CBC, dc, CNN);
            break;

         default:
            strcat( dc, "sd");
            TxPrint("Find subdirectory : ROOT & non-ROOT: %s%s%s\n", CBC, dc, CNN);
            break;
      }
      if (!TxaOption('c'))                      // if not 'from current'
      {
         nav.this = dfsFatClust2LSN(2);         // search from first possible
      }
      TxaReParseCommand( dc);
      rc = DFS_PENDING;                         // handle translated command
   }
   else
   {
      rc = DFS_PENDING;                         // cmd not recognized
   }
   RETURN (rc);
}                                               // end 'dfsFatCommand'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// FAT filesystem, identify specified sector
/*****************************************************************************/
static ULONG dfsFatIdent
(
   ULN64               lsn,                     // IN    LSN for sector
   ULN64               d2,                      // IN    dummy
   char               *st,                      // OUT   sector type
   void               *sec                      // IN    sector contents
)
{
   ULONG               dr = NO_ERROR;
   BYTE                rc = ST_UDATA;

   ENTER();

   if ((((S_FAT16 *)sec)->MediaType == dfsa->boot->eb.MediaType) &&
       ((lsn == fat->Fat1) || (lsn == fat->Fat2)))
   {
      switch (fat->FatBits)
      {
         case 12: rc=ST_FAT12; break;
         case 32: rc=ST_FAT32; break;
         default: rc=ST_FAT16; break;
      }
   }
   else if ((lsn >= fat->Root) && (lsn != 0) &&
            (lsn < (fat->Root + fat->Roots)))
   {                                            // Root-sector reported as
      rc = ST_DIREC;                            // ST_DIREC for searching!
   }
   else if ((rc = dfsFatDirClusterType( lsn)) == ST_UDATA)
   {
      if      ((((S_EACLUST *)sec)->signature == SV_EACLUST) &&
               (((S_EACLUST *)sec)->reserv1   == 0         )  )
      {
         rc = ST_FATEA;
      }
      else if ((((S_EAINDEX *)sec)->signature == SV_EAINDEX) &&
               (((S_EAINDEX *)sec)->base[EAI_SIZE -1] ==
                ((S_EAINDEX *)sec)->base[EAI_SIZE -2]))
      {
         rc = ST_EADAT;
      }
      else                                      // future expansion :-)
      {
      }
   }
   switch (rc)
   {
      case ST_UDATA:
         dr = DFS_PENDING;
         break;

      default:
         *st = rc;
         break;
   }
   RETURN (dr);
}                                               // end 'dfsFatIdent'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// FAT filesystem, supply sector-type description string
/*****************************************************************************/
static ULONG dfsFatStype
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *st,                      // IN    sector type
   void               *data                     // OUT   type description
)
{
   ULONG               dr  = NO_ERROR;
   char               *buf = (char *) data;
   BYTE                tp  = st[0];

   switch (tp)                                  // searchable types
   {
      case ST_FAT12: sprintf(buf,"Start 12-Bits FAT"); break;
      case ST_FAT16: sprintf(buf,"Start 16-Bits FAT"); break;
      case ST_FAT32: sprintf(buf,"Start 32-Bits FAT"); break;
      case ST_SUBDR: sprintf(buf,"Subdir, non-ROOT "); break;
      case ST_SUBRT: sprintf(buf,"Subdir from ROOT "); break;
      case ST_DIREC: sprintf(buf,"Directory data   "); break;
      case ST_FATEA: sprintf(buf,"FAT1x, EA cluster"); break;
      case ST_EADAT: sprintf(buf,"FAT1x EA data. sf"); break;
      default:
         switch (tp | ST__INFO)                 // non-searchable ones
         {
            case ST_FAT_1: sprintf(buf,"First  FAT area  "); break;
            case ST_FAT_2: sprintf(buf,"Second FAT area  "); break;
            case ST_ROOTD: sprintf(buf,"Root directory   "); break;
            default:       dr = DFS_PENDING;
         }
         break;
   }
   return (dr);
}                                               // end 'dfsFatStype'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// FAT filesystem, display sector-contents based on type
/*****************************************************************************/
static ULONG dfsFatSectorContents
(
   ULN64               psn,                     // IN    base psn for sector
   ULN64               d2,                      // IN    dummy
   char               *type,                    // IN    type of sector
   void               *data                     // IN    sector contents
)
{
   ULONG               dr = NO_ERROR;
   ULONG               sn32 = (ULONG) psn;
   BYTE                st = (BYTE) *type;

   ENTER();

   switch (st)
   {
      case ST_ROOTD:
      case ST_SUBDR:
      case ST_SUBRT:
      case ST_DIREC:
         dr = dfsFatDirDisplay( sn32, st);
         break;

      case ST_FATEA:
         dr = dfsFatEAdataDisplay( dfstPsn2LSN( DFSTORE, sn32));
         break;

      case ST_EADAT:
         dr = dfsFatDispEAindex((S_EAINDEX *) data, dfsGetSectorSize());
         break;

      default:
         dr = DFS_PENDING;
         break;
   }
   RETURN (dr);
}                                               // end 'dfsFatSectorContents'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// FAT filesystem, display special LSN info
/*****************************************************************************/
static ULONG dfsFatDisplayLsnInfo
(
   ULN64               lsn,                     // IN    possible special lsn
   ULN64               info,                    // IN    possible DIR entry nr
   char               *dc,                      // IN    dummy
   void               *data                     // IN    dummy
)
{
   ULONG               rc = DFS_PENDING;
   TXLN                path;

   ENTER();

   if (info & DFSSNINFO)                        // extra info present
   {
      if (dfsFatLsnInfo2Path( lsn, info, path) == NO_ERROR)
      {
         TxPrint("\nFound path to root: '%s%s%s'\n", CBY, path, CNN);
      }
      nav.xtra = lsn;                           // keep reference around
      rc = dfsFatShowDirectoryEntry( lsn, DFSSNIGET(info));
   }
   RETURN (rc);
}                                               // end 'dfsFatDisplayLsnInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display FAT Root- or sub-directory sector(s)
/*****************************************************************************/
static ULONG dfsFatDirDisplay
(
   ULONG               psn,                     // IN    base psn for sector
   BYTE                type                     // IN    type of sector
)
{
   ULONG               rc  = NO_ERROR;
   ULONG               lsn = dfstPsn2LSN( DFSTORE, psn);
   ULONG               chunks;                  // nr of space entries
   S_SPACE            *space;                   // space allocation
   ULONG               ef = 0;                  // Initialize error flag!

   ENTER();

   if ((type == ST_SUBDR) ||                    // sub-directory, non-ROOT
       (type == ST_SUBRT)  )                    // sub-directory from ROOT
   {
      ULONG    clust = dfsFatLSN2Clust(lsn);    // cluster nr for subdir

      if (clust > 1)                            // valid cluster number
      {
         TxPrint("Subdir cluster nr : %s%8.8X%s\n", CBC, clust, CNN);
         if (dfsFatAllocChain( clust, 0, &ef, &chunks, &space) == 0)
         {
            if (ef != 0)                        // inconsistent FAT values
            {
               dfsFatDispError( ef, 0, "\nWarning: ", NULL);

               chunks = 1;                      // just return first cluster
               if ((space = (S_SPACE *) calloc((size_t) chunks, sizeof(S_SPACE))) != NULL)
               {
                  space->start = lsn;           // start of cluster
                  space->size  = fat->ClustSize; // size of chunk
               }
               else
               {
                  rc = DFS_ALLOC_ERROR;
               }
            }
            else
            {
               rc = DFS_ALLOC_ERROR;
            }
         }
      }
      else
      {
         TxPrint("Invalid cluster   : %s0x%8.8X%s for directory at LSN 0x%8.8X\n",
                  CBR, clust, CNN, lsn);
         TxPrint("This could be caused by wrong bootsector values or an\n"
                 "incorrect starting sector set for the partition.\n\n");
         rc = DFS_BAD_STRUCTURE;
      }
   }
   else                                         // display one chunk
   {
      rc = dfsDirLsn2Space( lsn, &chunks, &space);
   }
   if (rc == NO_ERROR)
   {
      dfsFatDirectory( chunks, space);
      TxFreeMem( space);
   }
   RETURN (rc);
}                                               // end 'dfsFatDirDisplay'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display FAT directory-block
/*****************************************************************************/
ULONG dfsFatDirectory                           // RET   nr of entries
(
   ULONG               chunks,                  // IN    nr of space entries
   S_SPACE            *space                    // IN    space allocation
)
{
   ULONG               rc;
   ULONG               used = 0;                // nr of used entries
   ULONG               chunk;                   // index in space-area
   ULONG               sect;                    // sectors to handle
   ULONG               entry;                   // index in fat directory
   S_FATDIR           *fe;                      // Fat directory entry
   USHORT              ei = 0;                  // entry index
   int                 cl = TXaNWnZ;            // LSN color
   int                 ac;                      // alternate color fg+bg
   ULN64               data;                    // 1st data sector
   ULN64               dirsec;                  // current dir sector LSN
   TXLN                lfn;                     // VFAT long filename
   ULONG               dirholes = 0;            // empty / free entries
   ULONG               deleted  = 0;            // nr of deleted ones
   USHORT              bps = dfsGetSectorSize();
   USHORT              entries = (bps / sizeof(S_VFAT));
   BYTE               *dirbuf;
   S_VFAT             *fatdir;                  // Fat directory sector

   ENTER();

   if ((dirbuf = TxAlloc( 1, bps)) != NULL)     // allocate one sector
   {
      fatdir = (S_VFAT *) dirbuf;

      TxPrint("\n");
      dfsFatDirHeader( "Nr      Sector  ", 0);      // display header

      TRACES(("Sizeof S_VFAT: %u\n", (USHORT) sizeof(S_VFAT)));

      if (!TxaOptUnSet('l'))                    // create a new list for DIR
      {
         dfsInitList(0, "-f -P", "-d");         // optimal for menu file-recovery
      }
      strcpy( lfn, "");                         // init first to not-present
      for (chunk = 0; (chunk < chunks)  && !TxAbort(); chunk++)
      {
         for (sect = 0; (sect < space[chunk].size)  && !TxAbort(); sect++)
         {
            dirsec  = space[chunk].start + sect;
            if ((rc = dfsRead(dirsec, 1, dirbuf)) == NO_ERROR)
            {
               for (entry = 0; (entry < entries) && !TxAbort(); entry++)
               {
                  fe = &(fatdir[entry].d);
                  switch (fe->Name[0])
                  {
                     case FAT_DIRFREE:          // free entry
                        strcpy( lfn, "");       // reset LFN to not-present
                        dirholes++;             // and count them
                        break;

                     case FAT_DIRDEL:           // deleted entry, do list them!
                        strcpy( lfn, "");       // but reset LFN to not-present
                        deleted++;
                        if (!TxaOption('D'))    // suppress deleted ones, unless -D
                        {
                           used++;              // count used slots
                           break;               // but do not display
                        }
                     default:                   // used
                        if ((fe->FatAttrib == VFAT_ATTRIB) &&
                            (fe->clust     == VFAT_CLUSTR) )
                        {
                           TRACES(("Lfn slot Id: %u\n", fatdir[entry].v.SlotId));
                           dfsFatAddLfnSlot( &fatdir[entry].v, lfn);
                        }
                        else
                        {
                           if (dirholes != 0)   // zeroed entries, invalid!
                           {
                              TxPrint( "\n%sWARNING: %u Invalid ZERO entries, regular 'dir'"
                                       " will not see rest of directory!\n\n", CNN, dirholes);
                              dirholes = 0;
                           }
                           data = dfsFatClust2LSN( dfsFatDir2Clust(fe));

                           if (ei % 2)
                           {
                              ac = TXaBWnC;
                           }
                           else
                           {
                              ac = TXaNWnZ;
                           }
                           switch (ei)
                           {
                              case 0:  cl = TXaBGnZ; nav.down = data; break;
                              case 1:  cl = TXaBYnC; nav.xtra = data; break;
                              default: cl = ac;                       break;
                           }
                           TxPrint("%s.%05.5u", ansi[ac], ei);
                           dfsX10( " ", data,   ansi[cl], ansi[ac]);

                           dfsFatShowDirEntry( fe, Ccbg(ac));
                           TxPrint("%s\n", CGE);

                           dfsFatShowVfdEntry(fe, lfn, Ccbg(ac));
                           TxPrint("%s", CNN);

                           strcpy( lfn, "");    // init next to not-present

                           if (!TxaOptUnSet('l'))
                           {
                              dfsAddSI2List( dirsec, entry);
                           }
                           ei++;
                           used++;
                        }
                        break;
                  }
               }
            }
         }
      }
      if (ei > 12)
      {
         dfsFatDirHeader( "Nr      Sector  ", ei); // display footer
      }
      if (deleted != 0)
      {
         TxPrint("\n%u entries are for deleted files, use 't -D' to display..\n", deleted);
      }
      TxFreeMem( dirbuf);
   }
   RETURN (used);
}                                               // end 'dfsFatDirectory'
/*---------------------------------------------------------------------------*/

//- to be refined: extra parameter to allow finding 'deleted' files (only ?)
/*****************************************************************************/
// Find LSN* for specified path, starting at Root directory LSN
/*****************************************************************************/
static ULONG dfsFatFindPath
(
   ULN64               loud,                    // IN    Show progress
   ULN64               d2,                      // IN    dummy
   char               *path,                    // IN    path specification
   void               *vp                       // OUT   found dir/file + fat entry
)
{
   BOOL                rc  = NO_ERROR;
   ULN64               fs;                      // Current FNODE LSN
   ULN64               ds  = 0;                 // Current DIR LSN
   TXLN                part;
   char               *p   = path;
   int                 l;
   S_FATDIR            de;                      // directory entry (next)
   ULONG               chunks;                  // chunks in space structure
   S_SPACE            *space;                   // space structure
   DFS_PARAMS         *parm = (DFS_PARAMS *) vp;
   ULONG               entry;

   ENTER();
   fs = fat->Root;                              // start LSN
   if (loud)
   {
      dfsX10( "RootDirectory LSN : ", fs, CNN, "   ");
      TxPrint("find path: '%s'\n", path);
   }
   parm->Lsn    = fs;
   parm->Number = strlen(p) ? DFSSNINFO : 0;    // no info for ROOT!
   while ((rc == NO_ERROR) && strlen(p) && !TxAbort())
   {
      if ((l = strcspn(p, FS_PATH_STR)) != 0)
      {
         strncpy(part, p, l);                   // isolate part
         part[l] = '\0';
         p += l;                                // skip part
         if (*p == FS_PATH_SEP)
         {
            p++;                                // skip '\'
         }
      }
      if (strlen(part))
      {
         if (dfsDirLsn2Space(fs, &chunks, &space) == NO_ERROR)
         {
            if (loud)
            {
               dfsX10( "Base DirBlock LSN : ", fs, CNN, TREOLN);
            }
            if (dfsFatDirSpace2Entry(part, chunks, space, &ds, &entry, &de))
            {
               if (loud)
               {
                  dfsX10( " - Lsn ", ds, CNN, ", ");
                  TxPrint("entry %2u for '%s'\n", entry, part);
               }
               if (*p == '\0')                        //- end of string, found!
               {
                  parm->Lsn      = ds;                  //- DIR sector number
                  parm->Number   = entry | DFSSNINFO;   //- entry 0..15 in that sector
                  parm->Flag     = TRUE;                //- Size is from DIR-entry
                  parm->byteSize = de.fsize;
               }
               else
               {
                  if ((de.FatAttrib & FATTR_DIRECTORY) == 0)
                  {
                     if (loud)
                     {
                        TxPrint("Object at cluster : %8.8X is not a directory\n",
                                                           dfsFatDir2Clust(&de));
                     }
                     rc = ERROR_PATH_NOT_FOUND;
                  }
               }
               fs = dfsFatClust2LSN( dfsFatDir2Clust(&de)); // next LSN
            }
            else
            {
               if (loud)
               {
                  TxPrint(" - Search failed   for '%s'\n", part);
               }
               rc = ERROR_PATH_NOT_FOUND;
            }
            free( space);                       // free the memory !
         }
      }
      else
      {
         rc = ERROR_PATH_NOT_FOUND;
      }
   }
   RETURN(rc);
}                                               // end 'dfsFatFindPath'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// Find specific name in a (sub) directory SPACE structure
/*****************************************************************************/
static BOOL dfsFatDirSpace2Entry                // RET   name found
(
   char               *name,                    // IN    subdir or filename
   ULONG               chunks,                  // IN    nr of space entries
   S_SPACE            *space,                   // IN    space allocation
   ULN64              *lsn,                     // OUT   Dir-sector LSN found
   ULONG              *info,                    // OUT   Dir entry-nr found
   S_FATDIR           *dirent                   // OUT   Directory entry
)
{
   BOOL                found = FALSE;
   ULONG               chunk;                   // index in space-area
   ULONG               sect;                    // sectors to handle
   ULONG               entry;                   // index in fat directory
   ULN64               dirsec;                  // current dir sector LSN
   TXLN                lfn;                     // VFAT long filename
   S_FATDIR           *fe;                      // Fat directory entry
   S_FATDIR            target;                  // name in FAT DIR format
   USHORT              bps = dfsGetSectorSize();
   USHORT              entries = (bps / sizeof(S_FATDIR));
   BYTE               *dirbuf;
   S_VFAT             *fatdir;                  // Fat directory sector

   ENTER();
   TRARGS(("Name to find : '%s'\n", name));

   if ((dirbuf = TxAlloc( 1, bps)) != NULL)     // allocate one sector
   {
      fatdir = (S_VFAT *) dirbuf;

      dfsFatName2Entry( name, &target);         // create formatted name

      strcpy( lfn, "");                         // init first to not-present
      for ( chunk = 0;
           (chunk < chunks) && !TxAbort() && !found && (space);
            chunk++)
      {
         for ( sect = 0;
              (sect < space[chunk].size)  && !TxAbort() && !found;
               sect++)
         {
            dirsec  = space[chunk].start + sect;
            if (dfsRead(dirsec, 1, dirbuf) == NO_ERROR)
            {
               for ( entry = 0;
                    (entry < entries) && !TxAbort() && !found;
                     entry++)
               {
                  fe = &(fatdir[entry].d);
                  TRACES(("de:%3u cl:%4.4X name[0]: %2.2X, name: %*.*s\n", entry,
                      dfsFatDir2Clust( fe), fe->Name[0], FAT_N, FAT_N, fe->Name));

                  switch (fe->Name[0])
                  {
                     case FAT_DIRFREE:          // free entry
                     case FAT_DIRDEL:           // deleted entry
                        strcpy( lfn, "");       // reset LFN to not-present
                        break;

                     default:                   // used, LFN or short-name entry
                        if ((fe->FatAttrib == VFAT_ATTRIB) &&
                            (fe->clust     == VFAT_CLUSTR) &&
                            (fatdir[entry].v.Vzero     == 0))
                        {
                           TRACES(("Lfn slot Id: %u\n", fatdir[entry].v.SlotId));
                           dfsFatAddLfnSlot( &fatdir[entry].v, lfn); //- just add to lfn
                        }
                        else  //- regular (short) entry, but lfn may hold the long name!
                        {
                           if (((fe->FatAttrib & FATTR_LABEL) == 0) && //- no label and short-match
                               ((strncasecmp(fe->Name, target.Name, FAT_NSIZE + FAT_ESIZE) == 0) ||
                                (strcasecmp( lfn, name) == 0) ))       //- or a long-filename match
                           {
                              if (lsn)          // note that even when found through
                              {                 // the lfn compare, it is still the
                                 *lsn = dirsec; // short-name DIR entry info that is
                              }                 // returned as the search result!
                              if (info)
                              {
                                 *info = entry;
                              }
                              if (dirent)
                              {
                                 *dirent = *fe;
                              }
                              found = TRUE;
                           }
                           strcpy( lfn, "");    // init next to not-present
                        }
                        break;
                  }
               }
            }
         }
      }
      TxFreeMem( dirbuf);
   }
   BRETURN(found);
}                                               // end 'dfsFatDirSpace2Entry'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// DFS FAT write-file to disk (SaveTo to file)
/*****************************************************************************/
static ULONG dfsFatFileSaveAs
(
   ULN64               dse,                     // IN    dirsec/entry LSN
   ULN64               info,                    // IN    info, dir entry
   char               *path,                    // IN    destination path
   void               *recp                     // INOUT recovery parameters
)
{
   ULONG               rc  = NO_ERROR;
   S_RECOVER_PARAM    *param = (S_RECOVER_PARAM *) recp;
   S_FATDIR           *de;                      // Fat dir-entry ptr
   DFSISPACE           isp;                     // allocation SPACE info
   ULONG               ef = 0;                  // allocation error flag
   USHORT              bps = dfsGetSectorSize();
   BYTE               *dirbuf;
   S_FATDIR           *fatdir;                  // Fat directory sector
   ULONG               cluster;

   ENTER();
   TRARGS(("Dirsec LSN : %llX, entry:%u to path: '%s'\n", dse, DFSSNIGET(info), path));

   memset( &isp, 0, sizeof( DFSISPACE));

   if ((dirbuf = TxAlloc( 3, bps)) != NULL)     // allocate previous sectors (complete LFN)
   {
      fatdir = (S_FATDIR *) (dirbuf + (2 *bps));

      rc = dfsRead(dse -2, 3, dirbuf);
      if (rc == NO_ERROR)
      {
         de = &(fatdir[DFSSNIGET(info)]);
         cluster = dfsFatDir2Clust(de);
         if ((dfsFatAllocChain( cluster, de->fsize, &ef, &isp.chunks, &isp.space) != 0) ||
                                       ((de->fsize == 0) && ((de->FatAttrib & FATTR_DIRECTORY) == 0)))
         {
            TRACES(("Using regular FAT allocation chain\n"));
         }
         else if ((cluster >= 2) && (cluster < fat->MapClust) && (de->fsize != 0)) // regular, non-empty file
         {
            //- assume contiguous allocation, to allow SaveAs after FORMAT, for example Photo-cards
            isp.chunks = 1;                     // make available to SpaceFileSaveAs
            isp.space  = TxAlloc( 1, sizeof(S_SPACE));

            isp.space->start = dfsFatClust2LSN( cluster);
            isp.space->size  = (de->fsize + dfsGetSectorSize() - 1) / dfsGetSectorSize();

            param->noAllocCheck = TRUE;         // force no-alloc-check on save, avoiding errors

            TRACES(("Contiguous cluster: 0x%8.8x = LSN: 0x%llx size %u = %u sectors\n",
                                cluster, isp.space->start, de->fsize, isp.space->size));
         }
         else
         {
            if (ef != 0)                        // inconsistent FAT values
            {
               dfsFatDispError( ef, 0, "\nWarning: ", NULL);
               rc = DFS_BAD_STRUCTURE;
            }
            else
            {
               rc = DFS_ALLOC_ERROR;
            }
         }
         if (rc == NO_ERROR)                    // we have an S_SPACE allocation
         {
            TXLN       fname;                   // destination base filename
            TXLN       fullfn;                  // unique filename prefix

            #if defined (DEV32)
               ULONG       ea2Size;
               S_FEA2LIST *ea2List = NULL;

               if ((de->OS2EA) && (fat->FatBits != 32))
               {
                  dfsFatReadEaIndex( de->OS2EA, &ea2Size, &ea2List);

                  if (ea2List  != NULL)
                  {
                     TRACES(("Attach xatts ea2List: %8.8x\n", ea2List));
                     isp.xattrs = ea2List;
                     isp.xasize = ea2Size;
                     isp.xatype = DFSXA_FEA2;
                  }
               }
            #endif
            if ((param->newname == NULL) || (*param->newname == 0)) // no newname present
            {
               if ((TxaOptUnSet( DFS_O_LONGFN)) ||              //- if -lfn- option used
                   (dfsFatExtractLfn( dirbuf, de, fname) == 0)) //- or no lfn present
               {
                  dfsFatEntry2Name( de, fname);  // use the short-name directly
               }
               TRACES(("Recover to fname: '%s'\n", fname));
            }
            else                                // rename specified
            {
               strcpy( fname, param->newname);
            }
            if (param->unique)                  // force unique naming
            {                                   // on PATH and FILE components
               sprintf(fullfn, "%8.8llX_%s", dse, fname);
               strcpy( fname, fullfn);          // and move back to fname
            }
            if (param->recFname)
            {
               strcpy( param->recFname, fname); // return full base-filename
            }

            isp.byteSize = de->fsize;
            rc = dfsSspaceFileSaveAs( &isp, (de->FatAttrib & FATTR_DIRECTORY),
                                            (de->Name[0]  == FAT_DIRDEL),
                                             param->noAllocCheck, param->name83, path, fname, fullfn);
            if ((rc == NO_ERROR)       ||       // set original timestamps
                (rc == DFS_CMD_WARNING) )       // even when allocation errors present
            {
               time_t      mod = txOS2FileTime2t( &de->date.u, &de->time.u);
               time_t      cre = mod;
               time_t      acc = mod;           // default others to MOD time

               if ((de->Vcdate.u != 0) && (de->Vcdate.u != 0x21)) // create-date ?
               {
                  //- to be refined: ( + fe->Vctime10ms / 100) for second resolution
                  //- would need to add the Time10ms parameter to FileTime2t()

                  cre = txOS2FileTime2t( &de->Vcdate.u, &de->Vctime.u);
               }
               if ((de->Vadate.u != 0) && (de->Vadate.u != 0x21)) // access-date ?
               {
                  USHORT   zero = 0;

                  acc = txOS2FileTime2t( &de->Vadate.u, &zero);
               }
               TxSetFileTime( fullfn, &cre, &acc, &mod); // set time on full-fn from SaveAs

               if (param->recFname)
               {
                  strcpy( param->recFname, fullfn);
               }
            }
            #if defined (DEV32)
               TxFreeMem( ea2List);
            #endif
            free( isp.space);
         }
      }
      TxFreeMem( dirbuf);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN(rc);
}                                               // end 'dfsFatFileSaveAs'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// Make system/FAT usage map dump on TxPrint output
/*****************************************************************************/
static ULONG dfsFatAllocMap
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *options,                 // IN    display options
   void               *data                     // OUT   dummy
)
{
   ULONG               i;                       // index in data-area
   ULONG               b;                       // byte-index for one mapchar
   ULONG               cClus = 0;               // Clusters in current char
   LLONG               lClus = 0;               // Clusters in current line
   ULONG               mClus = 0;               // Clusters in current map
   ULONG               bClus = 0;               // Clusters marked as bad
   ULONG               uClus = 1;               // Last cluster that is in use
   ULONG               xClus = 0;               // Clusters that are NOT mapped
   ULONG               perc  = 0;               // percentage
   ULONG               l;                       // line number, 0 based
   ULONG               a;                       // index in display-array
   TX1K                ascii;                   // display-array
   char                mChar;                   // single character to display
   ULONG               spe;                     // sectors per entry
   ULONG               acl;                     // ascii alloc chars per line
   ULONG               cpc;                     // clusters per display-char
   ULONG               spc;                     // sectors per display-char
   ULONG               size;                    // nr of clusters to map
   ULONG               value;                   // FAT entry value
   ULONG               nonClust = fat->NrOfFats * fat->FatSectors +1; // FATs + bootsec
   BOOL                verbose  = (*options != '@');
   ULONG               bsSmart;
   ULN64               unallocSmart = 0;        // Size of THIS unallocated area
   BOOL                oneLineProgress = FALSE; // alloc in one line, output per character

   ENTER();

   dfsa->FsUnallocSmart = 0;                    // will be recalculated now
   bsSmart = dfsGetBufferSize( DFSOPTIMALBUF, DFSMAXBUFSIZE); // for SMART prediction

   spe   = fat->ClustSize;
   size  = fat->RawClust - 2;                   // #clusters minus first two

   acl   = dfsAllocCharsPerLine( 'c');
   cpc   = dfsAllocItemsPerChar( 'l', options, size, acl);
   if (cpc >= (size / acl))                     // will be a single line
   {
      oneLineProgress = TRUE;                   // one char at a time, no ascii string used
   }
   spc   = cpc * spe;
   dfsProgressInit( 0, size * spe, 0, "Get ALLOCATION info, at Sector:", dfstStoreDpid( DFSTORE), DFSP_STAT, 0);

   if (verbose)
   {
      dfsSz64("Size for one line : ", (ULN64) acl * spc, "  ");
      TxPrint("with %3u characters,", acl);
      dfsSiz8(" size : ", spc, "\n");
      TxPrint(" %16.16s = Allocation grade, empty to full.     ", mapchar);
      dfsSiz8(" SmartBuf size : ", bsSmart, "\n");
      if (fat->MapClust < fat->RawClust)
      {
         TxPrint( " %-16.16s = Filesystem area NOT mapped by the FAT!\n", "X");
      }
   }
   TxPrint(   "          %s%*.*s%s\n", CBC, acl, acl, BLIN, CNN);

   TxPrint("%s0x00000000%s%s%s", CNZ, CBC, CnB, "B");
   if (fat->FatBits == 32)
   {
      if (fat->NrOfFats == 1)
      {
         TxPrint("%s%*.*s         Fat area 32-bit entries, size:%8u KiB            %s%s",
                     CnM, (acl - 64), (acl - 64), "", (ULONG) (fat->FatSectors / 2), CBC, CNN);
      }
      else
      {
         TxPrint("%s%*.*s Fat1 32-bit, size:%7u KiB  %s%*.*s Fat2 32-bit, size:%7u KiB %s%s",
                     CnM, ((acl - 64) / 2), ((acl - 64) / 2), "", (ULONG) (fat->FatSectors / 2),
                     CnG, ((acl - 64) / 2), ((acl - 64) / 2), "", (ULONG) (fat->FatSectors / 2), CBC, CNN);
      }
   }
   else if ((fat->FatSectors +fat->Roots) != 0) // invalid, avoid divide by 0
   {
      S_SPACE          root;                    // fixed allocation chunk
      ULONG            half  = (acl - 64) / 2;
      ULONG            third = (acl - 64) / 3;
      ULONG            rest  = (acl - 64) - (2 * third);

      if ((fat->FatSectors < 4) || (fat->Roots < 4))
      {
         if (fat->NrOfFats == 1)
         {
            TxPrint("%s%*.*s Fat area, size:%5u bytes  %s%*.*s Root directory, size:%5u bytes %s",
                     CnM, half,             half,             "", fat->FatSectors * 512,
                     CnY, half,             half,             "", fat->Roots * 512, CBC);
         }
         else
         {
            TxPrint("%s%*.*s Fat1, size:%5u b  %s%*.*s Fat2, size:%5u b  %s%*.*s Root, size:%5u b  %s",
                     CnM, third,            third,            "", fat->FatSectors * 512,
                     CnG, third,            third,            "", fat->FatSectors * 512,
                     CnY, rest,             rest,             "", fat->Roots * 512, CBC);
         }
      }
      else
      {
         if (fat->NrOfFats == 1)
         {
            TxPrint("%s%*.*s Fat area, size:%3u KiB      %s%*.*s Root directory, size:%3u KiB      %s",
                     CnM, half,             half,             "", fat->FatSectors /2,
                     CnY, half,             half,             "", fat->Roots /2, CBC);
         }
         else
         {
            TxPrint("%s%*.*s Fat1, size:%3u KiB  %s%*.*s Fat2, size:%3u KiB  %s%*.*s Root, size:%3u KiB  %s",
                     CnM, third,            third,            "", fat->FatSectors /2,
                     CnG, third,            third,            "", fat->FatSectors /2,
                     CnY, rest,             rest,             "", fat->Roots /2, CBC);
         }
      }

      nonClust  += fat->Roots;                  // additional non-cluster space
      root.start = fat->Root;                   // first LSN for Root
      root.size  = fat->Roots;
      if (dfsa->boot->eb.RootEntries != 0)
      {
         perc = ((100 * dfsFatDirUsed(1, &root)) / dfsa->boot->eb.RootEntries);
      }
      TxPrint("%s% 3u%s%%", (perc > 90) ? CBR : CNY, perc, CNN);
   }
   else
   {
      TxPrint( "%s       FAT + ROOTDIR number of sectors is ZERO (invalid)"
                 "       %s%s%s", CnR, CNN, CBC, CNN);
   }
   TxPrint("\n");
   dfsX10( CNZ, dfsFatClust2LSN(2), CNZ, "");   // display address for first block
   TxPrint("%s%s", CBC, CNC);

   for (i=0, a=0, l=0; (i < size) && (!TxAbort()) && (fat->CacheA.Value);)
   {
      if (a == 0)
      {
         memset(ascii, 0, TXMAXLN);
         lClus = 0;
      }
      for (cClus=0, b=0; (b < cpc) && (i < size); b++, i++)
      {
         if (i < fat->MapClust)                 // mapped in the FAT area ?
         {
            value = dfsFatValue(i+2);           // one FAT entry value
            if (value > FAT_MAXCLUST)
            {
               cClus++;                         // cluster in use or bad
               if (value < FAT_MINEOFCL)
               {
                  bClus++;                      // bad sectors
               }
               else                             // EOF cluster
               {
                  uClus = i+2;                  // last in-use one sofar
               }
            }
            else
            {
               if (value != 0)
               {
                  if (unallocSmart != 0)        // first in-use after unallocated?
                  {
                     if (unallocSmart >= bsSmart)                  //- if at least ONE smart buffer
                     {
                        unallocSmart -= (bsSmart / 2);             //- compensate for alignment issues
                        unallocSmart -= (unallocSmart % bsSmart);  //- clip to whole smart buffer size

                        dfsa->FsUnallocSmart += unallocSmart;      //- add this area to total
                     }
                     unallocSmart = 0;          // reset for next area to come
                  }
                  cClus++;                      // count cluster in use
                  uClus = i+2;                  // last in-use one sofar
               }
               else                             // unallocated, Smart predict
               {
                  unallocSmart += spe;          // add to size THIS area
               }
            }
         }
         else
         {
            xClus++;
         }
      }
      lClus += cClus;

      dfsProgressShow( (i * spe) + spc, 0, 0, NULL);

      TRACES(("lClus 2nd Loop i: %u, b:%u = %u\n", i, b, lClus));
      if (i < fat->MapClust)
      {
         mChar = (char) ((cClus == 0) ? ' ' : mapchar[cClus * 8 / cpc]);
      }
      else
      {
         mChar = 'X';                           // non mapped clusters!
      }
      if (oneLineProgress == TRUE)
      {
         TxPrint( "%c", mChar);                 // slow, acts as progress bar
      }
      else
      {
         ascii[a] = mChar;
      }
      a++;
      if ((i && ((i%(acl*cpc)) == 0)) || (i >= size))
      {
         if (oneLineProgress == FALSE)
         {
            TxPrint( "%s", ascii);              // display accumulated chars (fast)
         }
         perc  = (ULONG)(100*lClus / (a*cpc));
         if (a == acl)
         {
            TxPrint("%s%s% 3u%%\n", CBC, CNN, perc);
         }
         else
         {
            TxPrint("%s%.*s%s% 3u%%\n", CBC, (int) (acl-a-1), BLIN, CNN, perc);
         }
         if (i < size)
         {
            dfsX10( CNZ, dfsFatClust2LSN(i+2), CNZ, "");
            TxPrint("%s%s", CBC, CNC);
            a = 0;                              // keep a value on last line
         }
         mClus += lClus;
         l++;
      }
   }
   if (unallocSmart != 0)                           //- unalloc area pending?
   {
      if (unallocSmart >= bsSmart)                  //- if at least ONE smart buffer
      {
         unallocSmart -= (bsSmart / 2);             //- compensate for alignment issues
         unallocSmart -= (unallocSmart % bsSmart);  //- clip to whole smart buffer size

         dfsa->FsUnallocSmart += unallocSmart;      //- add this area to total
      }
   }
   TxPrint("          %s%.*s%s\n", CBC, (USHORT) a, BLIN, CNN);

   if (TxAbort())
   {
      dfsa->FsUnallocSmart = 0;                 // signal info NOT available!
      TxPrint( "\nAllocation check and display aborted! Used/Free info not available.\n");
   }
   else
   {
      if (size != 0)                            // avoid devide by 0
      {
         ULONG            total = size  * spe + nonClust +1; //- include bootsec
         ULONG            used  = mClus * spe + nonClust -1; //- used incl FAT areas
         ULN64            free;

         fat->FreeClusters   = size - mClus;                 //- NET  frees (for resize)
         dfsa->FsUnallocated = fat->FreeClusters * spe;      //- free space (optimize hint)

         dfsSz64("Unallocated sects : ", dfsa->FsUnallocated, " ");
         dfsSz64("SmartUse ", dfsa->FsUnallocSmart, " ");
         TxPrint("=% 5.1lf%%\n", (double) (100.0 * ((double) dfsa->FsUnallocSmart) / (double) total));

         dfsSz64("Allocated sectors : ", used,  " ");
         dfsSz64("of total ",            total, " ");
         TxPrint("=% 5.1lf%%\n",    (double) (100.0 * ((double) used) / (double) total));

         free = ((ULN64) dfsa->FsUnallocated) * dfsGetSectorSize();
         dfsSz64( "Unallocated space : ", dfsa->FsUnallocated, " ");
         dfsUllDot20(   " = ", free, " bytes free\n");

         if (bClus != 0)
         {
            dfsSiz8("Bad sectors       : ", bClus * spe, " =");
            TxPrint("% 8.5lf%%\n", (double) (100 * ((double) bClus) / fat->MapClust));
         }
      }

      fat->TruncCluster  = uClus + 1;           // First free cluster for truncate
      dfsa->FsTruncPoint = dfsFatClust2LSN( fat->TruncCluster);
      dfsa->FsLastInUse  = dfsa->FsTruncPoint -1;
      dfsSz64( "Min. size / Clust : ", dfsa->FsTruncPoint, " ");
      TxPrint( "/ 0x%8.8x Cl   (for  Resizing)\n", fat->TruncCluster);

      if (xClus)
      {
         dfsSiz8( "NOT mapped by FAT : ", xClus * fat->ClustSize, " (wasted space!)\n");
      }
   }
   dfsProgressTerm();
   RETURN (NO_ERROR);
}                                               // end 'dfsFatAllocMap'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Truncate or expand the filesystem size, keeping all data intact; (2 sect br)
/*****************************************************************************/
static ULONG dfsFatResizeFS
(
   ULN64               newsz64,                 // IN    new size in sectors
   ULN64               d2,                      // IN    dummy
   char               *string,                  // IN    dummy
   void               *data                     // IN    partition info
)
{
   ULONG               rc  = NO_ERROR;
   ULONG               newsize = (ULONG) newsz64;
   DFSPARTINFO        *p   = (DFSPARTINFO *) data;
   TXTM                text;

   ENTER();
   TRARGS(("New size: %8.8x\n", newsize));

   strcpy( text, "");
   dfstrSizeBps( text, "to a new size of: ", newsize, (USHORT) p->bpsector, "");

   TRACES(("Resize partinfo: 0x%8.8x, bpsector:%hu\n", p, p->bpsector));

   //- confirmation skipped when expanding (runs in batch automatically!)
   if ((dfsa->batch) || (TxConfirm( 5130, "Resize this FAT%2.2hu filesystem "
                                          "%s ? [Y/N] : ", fat->FatBits, text)))
   {
      S_BOOTR         *br = (S_BOOTR *)  brec;
      FAT32B2         *b2 = (FAT32B2 *) (brec + p->bpsector);
      ULONG            oldsize;                 // current in bootrec
      ULONG            maxsize;                 // max supported by the FAT
      ULONG            extraFatSizeCl = 0;      // growth for 1 FAT area, in clusters
      ULONG            extraFatsSize  = 0;      // growth all FAT areas, in sectors

      oldsize = max( br->eb.BigSectors, (ULONG) br->eb.Sectors);

      //- calculate max size using CURRENT FAT-areas unchanged (simple truncate)
      //- this may actually allow a slighly LARGER filesystem by using up slack
      maxsize = ((fat->FatSectors * 4096 / fat->FatBits) -2) * fat->ClustSize + fat->ClTwo;

      TRACES(( "newsize: %8.8x  oldsize: %8.8x  max truncate size: %8.8x\n",
                newsize,        oldsize,        maxsize));

      if (newsize > maxsize)                   // simple truncate is NOT possible
      {
         ULONG     fePS = (dfsGetSectorSize() * 8 / fat->FatBits); // entries per sector
         ULONG     oldFatSectors = fat->FatSectors;
         BYTE      zeroPattern   = 0;           // small wipe pattern for FAT area

         TRACES(( "fePS: %u  oldFatSectors: 0x%x  ClustSize: %hu\n",
                   fePS,     oldFatSectors,  fat->ClustSize));

         //- Step (1) calculate extra size (in clusters) for single FAT area, round up
         extraFatSizeCl = (((newsize - oldsize -1) / fat->ClustSize) / (fePS * fat->ClustSize)) +1;
         extraFatsSize  = extraFatSizeCl * fat->ClustSize * fat->NrOfFats;
         dfsSizeXiB( "\nFAT areas need ", extraFatsSize, dfsGetSectorSize(),
                     " extra space for new size.\n");
         TRACES(( "extraFatSizeCl: 0x%x, total %u\n", extraFatSizeCl, extraFatsSize));

         //- FAT location and size is now in-flux, FS as a whole NOT-CONSISTENT!
         fat->FatSectors += (extraFatSizeCl * fat->ClustSize);
         fat->MapClust    = fat->FatSectors * (dfsGetSectorSize() * 8 / fat->FatBits);

         //- Step (2) select between in-place and copied-FAT type of expanding
         if ((extraFatSizeCl * fat->ClustSize) < oldFatSectors)
         {
            //- FAT growth for single FAT fits in current FAT2 area, expand in-place
            fat->FatResize   = fat->Fat1;
            fat->Fat2        = 0;               // disables FAT flushing to FAT2

            TxPrint("Expand FAT area in-place at sector: 0x0%llx\n", fat->FatResize);
         }
         else
         {
            //- FAT growth for single FAT larger than FAT2 area, copy to end of new FS size
            fat->FatResize   = newsize - fat->FatSectors;
            TxPrint("Expand FAT area to temp space at sector: 0x0%llx.\n", fat->FatResize);

            rc = dfsCopySectorData( fat->Fat1, fat->FatResize, oldFatSectors);
         }

         if (rc == NO_ERROR)
         {
            //- Wipe the new (still free) part of the enlarged FAT-area used while resizing
            TxPrint("Clearing new part of the FAT-area to unallocated ...\n");
            rc = dfsWipeArea( fat->FatResize  + oldFatSectors, // after (copied/in-place) FAT
                              fat->FatSectors - oldFatSectors, // for the added FAT size
                              DFSP_STAT, "Resizing FAT", FALSE, &zeroPattern, sizeof(BYTE));
            if (rc == NO_ERROR)
            {
               S_SPACE         *ads = NULL;
               ULONG            adc = 0;

               TxPrint( "Locating ALL directory sectors in the filesystem ...\n");
               if (dfsFatAllDirs2Space( TRUE, &adc, &ads) == NO_ERROR)
               {
                  TxPrint( "Updating cluster-number in ALL directory entries ...\n");
                  rc = dfsFatFixDirs4Resize( extraFatSizeCl, adc, ads);
                  TxFreeMem( ads);              // free the SPACE

                  if (rc == NO_ERROR)           // now update the new FAT area too
                  {
                     TxPrint( "Updating all cluster-numbers in resized FAT-area ...\n");
                     dfsFatSetFatInUse( fat->FatResize);
                     rc = dfsFatFixRfat4Resize( extraFatSizeCl);

                     if (rc == NO_ERROR)        // Move data-clusters 2..EFSC-1 to current end
                     {
                        dfsSizeXiB( "Move ", extraFatsSize, dfsGetSectorSize(),
                                    " of data-cluster contents upward ...\n");
                        rc = dfsCopySectorData( fat->ClTwo,
                                                dfsFatClust2LSN( fat->TruncCluster),
                                                extraFatsSize);
                     }
                  }
               }
            }
         }
         if (rc == NO_ERROR)                    // finish by updating Root+FATs to new location
         {
            if (fat->FatBits == 32)             // FAT32, update ROOT cluster in bootsector
            {
               //- Update FAT32 root-cluster value based on movements done
               dfsa->boot->f2.RootCluster = dfsFatResizedValue( dfsa->boot->f2.RootCluster,
                                                                extraFatSizeCl);
               fat->Root  = dfsFatClust2LSN( dfsa->boot->f2.RootCluster);
               TxPrint( "FAT32 Root directory cluster updated to: 0x%8.8x.\n", dfsa->boot->f2.RootCluster);
            }
            else                                // FAT16, copy ROOT DIR to new location
            {
               //- Root directory needs to move UP by #FATS * EFSC, and can overlap!
               rc = dfsCopySectorData( fat->Root, fat->Root + extraFatsSize, fat->Roots);
               fat->Root += extraFatsSize;
               TxPrint( "FAT16 Root directory moved to sector  :  0x0%llx.\n", fat->Root);
            }
            if (rc == NO_ERROR)
            {
               int     fnr;
               ULONG   fsn;

               //- Copy resized FAT to new FAT1 .. FATn locations
               for ( fnr = (fat->FatResize == fat->Fat1) ? 1 : 0;
                    (fnr <  fat->NrOfFats) && (rc == NO_ERROR);
                     fnr++)
               {
                  fsn = fat->Fat1 + (fnr * fat->FatSectors);
                  TxPrint( "Copy %6u KiB new FAT-area to FAT%u at: 0x%8.8x.\n",
                            fat->FatSectors / 2, fnr + 1, fsn);
                  rc = dfsCopySectorData( fat->FatResize, fsn, fat->FatSectors);
                  if (rc == NO_ERROR)
                  {
                     if (fnr == 1)              // 2nd FAT, update its starting LSN
                     {
                        fat->Fat2 = fsn;
                     }
                  }
               }
               if (rc == NO_ERROR)
               {
                  //- Update FAT size in bootsector, at std or F32 location
                  if (fat->FatBits == 32)
                  {
                     br->f2.FatSectors = (USHORT) fat->FatSectors;
                  }
                  else
                  {
                     br->eb.FatSectors = (USHORT) fat->FatSectors;
                  }
               }
            }
         }
      }

      if (rc == NO_ERROR)
      {
         //- Generic processing for smaller as well as larger sizes, update BR and spares
         if (newsize > 0xffff)
         {
            br->eb.Sectors    = 0;
            br->eb.BigSectors = newsize;
         }
         else
         {
            br->eb.Sectors    = (USHORT) newsize;
            br->eb.BigSectors = 0;
         }
         TRACES(("Freespace check, br:0x%8.8x  b2:0x%8.8x\n, br, b2"));
         TRHEXS( 70, br, 0x400, "FAT32 br and b2 sectors");
         if ((fat->FatBits   == 32)        &&   // fat32
             (b2->Signatur1  == SV_BOOT21) &&   // and freespace BR is there
             (b2->Signatur2  == SV_BOOT22)  )
         {
            if (fat->FreeClusters != 0)         // known from DFSee ALLOC, so use that
            {
               b2->FreeClusters = fat->FreeClusters;
            }
            TRACES(("Freespace update from %u clusters, clustersize: %hu\n", b2->FreeClusters, fat->ClustSize));
            if (fat->ClustSize)                 // avoid devide by zero
            {
               ULONG      cl;

               if (newsize > oldsize)
               {
                  cl = ((newsize - oldsize) / fat->ClustSize) - (extraFatSizeCl * fat->NrOfFats);
                  b2->FreeClusters += cl;       // adjust nr of free clusters UP
                  TRACES(("Freespace increased with %u clusters to %u\n", cl, b2->FreeClusters));
               }
               else
               {
                  cl = ((oldsize - newsize) / fat->ClustSize);
                  b2->FreeClusters -= cl;       // adjust nr of free clusters DOWN
                  TRACES(("Freespace decreased with %u clusters to %u\n", cl, b2->FreeClusters));
               }
            }
         }
         else
         {
            TRACES(("No Freespace update, no FAT32 or incorrect signatures!\n"));
         }
         if (DFSTORE_WRITE_ALLOWED)
         {
            rc = dfsWrite( LSN_BOOTR, (fat->FatBits == 32) ? 2 : 1, brec); // 1 for FAT16!
            if (rc == NO_ERROR)
            {
               if ((dfsRead( LSN_FAT32SPARE, 2, rbuf)) == NO_ERROR) // read 2 sect to check
               {
                  b2 = (FAT32B2 *) (rbuf + p->bpsector);

                  if ((fat->FatBits   == 32)        && // fat32
                      (b2->Signatur1  == SV_BOOT21) &&
                      (b2->Signatur2  == SV_BOOT22)  ) // and current spare is present
                  {
                     dfsWrite( LSN_FAT32SPARE, 2, brec); // copy brec to spare lsn
                  }
                  else
                  {
                     TRACES(("No F32SPARE update, no FAT32 or incorrect signatures!\n"));
                  }
               }
               TxPrint("\nFAT%2.2hu filesystem resized %s ...\n", fat->FatBits, text);
            }
         }
         else
         {
            rc = DFS_READ_ONLY;
         }
      }
   }
   else
   {
      rc = DFS_NO_CHANGE;
   }
   RETURN (rc);
}                                               // end 'dfsFatResizeFS'
/*---------------------------------------------------------------------------*/


