//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// DFS display and analysis, EXT2/3 specific functions and definitions
//
// Author: J. van Wijk
//
// JvW  29-03-2004 Initial version, derived from JFS
//
// Author: J. van Wijk

#ifndef    DFSAEXT_H
   #define DFSAEXT_H

#include <dfsext.h>                             // EXT  structure defs

// EXT specific sector types
                                                // S   used on searching
                                                //  I  identification only
#define ST_EXTSUP  's'                          // S   EXT  super-block         (512 BPS)
#define ST_EXTBSU  'S'                          //  I  EXT  boot+superblock    (4096 BPS)
#define ST_EXTDIR  'd'                          // S   EXT  Directory block
#define ST_EXTINO  'f'                          //  I  EXT  File Inode (in-use, sector)
#define ST_EXTDIN  'D'                          //  I  EXT  Dir  Inode (in-use, sector)
#define ST_EXTEAB  'E'                          // S   EXT  EA   data block
#define ST_EXTBUS ('B' + ST__INFO)              //  I  EXT  BACKUP Superblock
#define ST_EXTGRP ('K' + ST__INFO)              //  I  EXT  Group Descriptor Table
#define ST_EXTBGR ('c' + ST__INFO)              //  I  EXT  BACKUP GDT
#define ST_EXTGDR ('k' + ST__INFO)              //  I  EXT  GDT-reserved (resize inode 7)
#define ST_EXTBBM ('M' + ST__INFO)              //  I  EXT  Block Bitmap sector
#define ST_EXTIBM ('m' + ST__INFO)              //  I  EXT  Inode Bitmap sector
#define ST_EXTITB ('i' + ST__INFO)              //  I  EXT  Inode Table sector
#define ST_EXTBI1 ('1' + ST__INFO)              //  I  EXT  Block-Indirect level-1 block
#define ST_EXTBI2 ('2' + ST__INFO)              //  I  EXT  Block-Indirect level-2 block
#define ST_EXTBI3 ('3' + ST__INFO)              //  I  EXT  Block-Indirect level-3 block
#define ST_EXTAIX ('A' + ST__INFO)              //  I  EXT  Alloc-Extent   index-node
#define ST_EXTALF ('a' + ST__INFO)              //  I  EXT  Alloc-Extent   leaf-node
#define ST_EXTJRN ('j' + ST__INFO)              //  I  EXT  Journal-File   (system)
#define ST_EXTRES ('z' + ST__INFO)              //  I  EXT  Resize-Info    (system)


#define DFS_EDIR_THEADER   0x0001               // add directory header at top
#define DFS_EDIR_BHEADER   0x0002               // optional header at bottom
#define DFS_EDIR_LISTADD   0x0010               // add entries to the sectorlist
#define DFS_EDIR_USEINODE  0x0200               // use inode for filesize/mode

#define DFS_EDIR_STD  (DFS_EDIR_THEADER | DFS_EDIR_BHEADER | DFS_EDIR_USEINODE)

#define DFSEXT_NOGROUP ((ULONG) 0xffffffff)
typedef struct S_EXT_BMCACHE                    // bitmap cache structure
{
   BOOL                Dirty;                   // cache contents modified
   ULONG               Group;                   // Group nr 0 .. Groups -1
   ULONG               Sects;                   // Cache size in sectors
   BYTE               *Cache;                   // bitmap itself
} S_EXT_BMCACHE;                                // end of struct "S_EXT_BMCACHE"


typedef struct s_ext_inamecache                 // inode2name caching element
{
   ULONG               Parent;                  // Parent Inode number
   char               *Name;                    // related name (allocated)
} S_EXT_INAMECACHE;                             // end of struct "dfsextiname"


typedef struct dfsaext                          // EXT analysis information
{
   ULN64               Sect;                    // Number of sectors
   ULN64               Block;                   // Number of blocks
   ULONG               BlockSize;               // Block size in bytes
   ULONG               SectorsPerBlock;         // sectors per block
   S_EXT_SUPER        *sup;                     // ptr to superblock
   ULONG               GroupDescrSize;          // Group desc size in bytes
   ULONG               Groups;                  // Number of block groups
   ULONG               GdtBlocks;               // Group Descriptor Table, size in blocks
   S_EXT_GROUPDESCR   *GroupDescTable;          // Array of group descriptors
   S_EXT_INAMECACHE   *Ic;                      // Inode to name/parent cache
   S_EXT_BMCACHE       Im;                      // Inode bitmap cache
   S_EXT_BMCACHE       Bm;                      // Block bitmap cache
   ULONG               SsCount;                 // Sparse-Super BG table count
   ULONG              *SsTable;                 // Sparse-Super kBlockgroup table
   ULONG               GdtReservedBlocks;       // GDT reserved blocks (FS expand)
   ULONG               InodeTableBlocks;        // Inode Table size in blocks
   ULONG               InodeMax;                // Maximum Inode number
   ULONG               InodeUsed;               // Number of Inodes used (from GDT)
   ULN64               FreeSect;                // Number of free sectors (from BG)
   DFSISPACE           DirCache;                // Cached alloc for last DIR inode
} DFSAEXT;                                      // end of struct "dfsaext"

// Note:
// The sum of GdtReservedBlocks and BmPlusTableBlocks is the 'allocated' number
// of blocks at the start of a blockgroup, when it is unused (BLOCK_UNINIT)

extern DFSAEXT        *ext;

// Initialize EXT filesystem analysis
ULONG dfsExtInit
(
   char               *fs                       // forced filesystem type
);


// Initialize EXT filesystem analysis for Area (FS info in FDISK mode)
ULONG dfsExtAreaInit
(
   void
);

// EXT  filesystem, identify specified sector as a valid superblock
BOOL dfsExtIsSuperBlock                         // RET   sector is a valid sb
(
   BYTE               *sec                      // IN    sector contents
);

// EXT  filesystem, identify specified sector as (start of) a directory block
BOOL dfsExtIsDirectoryBlock                     // RET   sector is a valid sb
(
   BYTE               *sec                      // IN    sector contents
);

// EXT  filesystem, check if given BG has backups for superblock + BG-descr
BOOL dfsExtIsBackupSuperBG                      // RET   BG# has super backups
(
   ULONG               group                    // IN    blockgroup number
);

// EXT filesystem, display sector constents as a superblock
void dfsExtDisplaySuperBlock
(
   BYTE               *sec                      // IN    sector contents
);

// Display one Inode Record from supplied sector data, based on the index
ULONG dfsExtShowInodeRecord
(
   BYTE               *sector,                  // IN    Inode sector data
   ULN64               lsn,                     // IN    LSN for Inode sector
   BYTE                index                    // IN    index of the Inode
);

// EXT filesystem, identify specified sector
ULONG dfsExtIdent
(
   ULN64               lsn,                     // IN    LSN        (ignored!)
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // OUT   sector type
   void               *data                     // IN    sector contents
);


#endif
