//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// EXT2/3 dump & display Analysis functions
//
// Author: J. van Wijk
//
// JvW  29-03-2004 Initial version, derived from JFS
// JvW  24-08-2016 Major update, adding file-level allocation and directory support
// JvW  08-04-2021 Use rc CMD_WARNING on FileSaveAs alloc errors, considered OK
//
// Author: J. van Wijk

#include <txlib.h>                              // TX library interface

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsver.h>                             // DFS version information
#include <dfsutil.h>                            // DFS utility functions
#include <dfsposix.h>                           // POSIX utility functions
#include <dfsspace.h>                           // DFS file-space interface
#include <dfstable.h>                           // SLT utility functions
#include <dfsdgen.h>                            // Generic dialogs (UUID edit)

#include <dfsupart.h>                           // prereq. for UFDSK
#include <dfsufdsk.h>                           // FDISK generic Bootsector display

#include <dfsaext.h>                            // EXT display & analysis
#include <dfsuext.h>                            // EXT utility functions
#include <dfslext.h>                            // EXT SLT functions


char dfsExtSectorTypes[] =
{
   ST_EXTSUP,                                   // EXT  super-block         (512 BPS)
   ST_EXTBSU,                                   // EXT  boot+superblock    (4096 BPS)
   ST_EXTDIR,                                   // EXT  Directory block
   ST_EXTINO,                                   // EXT  File Inode (in-use, sector)
   ST_EXTDIN,                                   // EXT  Dir  Inode (in-use, sector)
   ST_EXTEAB,                                   // EXT  EA   data block
   ST_EXTBUS,                                   // EXT  BACKUP Superblock
   ST_EXTGRP,                                   // Group Descriptor Table
   ST_EXTBGR,                                   // EXT  BACKUP GDT
   ST_EXTGDR,                                   // GDT-reserved area (resize inode 7)
   ST_EXTBBM,                                   // Block Bitmap sector
   ST_EXTIBM,                                   // Inode Bitmap sector
   ST_EXTITB,                                   // Inode Table sector
   ST_EXTBI1,                                   // EXT  Block-Indirect level-1 block
   ST_EXTBI2,                                   // EXT  Block-Indirect level-2 block
   ST_EXTBI3,                                   // EXT  Block-Indirect level-3 block
   ST_EXTAIX,                                   // EXT  Alloc-Extent   index-node
   ST_EXTALF,                                   // EXT  Alloc-Extent   leaf-node
   ST_EXTJRN,                                   // EXT  Journal-File   (system)
   ST_EXTRES,                                   // EXT  Resize-info    (system)
   0                                            // string terminating ZERO
};


//- calculated by taking powers of 3, 5 and 7
#define DFSEXT_S_S_COUNT    37         // #sparse-super groups in table
static ULONG      ext_sstable[DFSEXT_S_S_COUNT] =
{
          1,                                    // n^0
          3,                                    // 3^1
          5,                                    // 5^1
          7,                                    // 7^1
          9,                                    // 3^2
         25,                                    // 5^2
         27,                                    // 3^3
         49,                                    // 7^2
         81,                                    // 3^4
        125,                                    // 5^3
        243,                                    // 3^5
        343,                                    // 7^3
        625,                                    // 5^4
        729,                                    // 3^6
       2187,                                    // 3^7
       2401,                                    // 7^4
       3125,                                    // 5^5
       6561,                                    // 3^8
      15625,                                    // 5^6
      16807,                                    // 7^5
      19683,                                    // 3^9
      59049,                                    // 3^10
      78125,                                    // 5^7
     117649,                                    // 7^6
     177147,                                    // 3^11
     390625,                                    // 5^8
     531441,                                    // 3^12
     823543,                                    // 7^7
    1594323,                                    // 3^13
    1953125,                                    // 5^9
    4782969,                                    // 3^14
    5764801,                                    // 7^8
    9765625,                                    // 5^9
   14348907,                                    // 3^15
   40353607,                                    // 7^9
   43046721,                                    // 3^16
   48828125,                                    // 5^10
};


static DFSAEXT    ext_anchor =                  // EXT specific global info
{
   0, 0,                                        // zero size
   4096,                                        // Blocksize in bytes
   8,                                           // default 8 sectors/block
   NULL,                                        // superblock pointer
   32, 0, 0, NULL,                              // GDT size, count, blocks and array
   NULL,                                        // Inode Name and 1st-parent cache
   {FALSE, DFSEXT_NOGROUP, 0, NULL},            // Inode Bitmap cache, for 1 BG
   {FALSE, DFSEXT_NOGROUP, 0, NULL},            // Block Bitmap cache, for 1 BG
   DFSEXT_S_S_COUNT,                            // Built-in sparse-super table size
   ext_sstable,                                 // Sparse-Super BG table + count
   1, 1, 0xFFFFFFFF,                            // min GdtReserved, Itable size, max-Inode#
   0,                                           // Free sectors in filesystem
   0,                                           // Number of Inodes used
   {0, NULL, 0, 0, 0, 0}                        // Cached alloc for last DIR Inode
};

       DFSAEXT   *ext = &ext_anchor;

static  char       *ext_txt[] =
{
   "",
   "Active filesystem : EXT2/3/4, specific commands are:",
   "",
   " \\path-spec       = find and show file/directory relative to root (FINDPATH)",
   " ABG              = Display filesystem ALLOC map, with one line per blockgroup",
   " BG group | *     = Display group info and block-bitmap for one or all groups",
   " BL               = Translate and display 'this' LSN as a block number",
   " BL block  [cmd]  = Translate specified block-number to LSN, display using 'cmd'",
   " BSL  [-f]        = Display Sparse-Super blockgroup# table, location of backup",
   " FILEFIND [name]  = Find normal files, with name-info matching (partial) name",
   " INO  [-v][ino#]  = Calculate sector+index for Inode, display contents [verbose]",
   " LABEL   [label]  = Display/edit 16-char volume label in the superblock",
   " SUPER            = Display the filesystem SUPERBLOCK sector/block",
   " UUID [[-u:]uuid] = Display or set a UUID string, dialog if no id specified",
   "",
   NULL
};




// Close EXT filesystem for analysis and free any resources
static ULONG dfsExtClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
);

// Close EXT filesystem for Area analysis and free any resources
static ULONG dfsExtAreaClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
);

// Interpret and execute specific EXT command;
static ULONG dfsExtCommand
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *none,                    // IN    dummy
   void               *data                     // INOUT dummy
);

// EXT filesystem, supply sector-type description string
static ULONG dfsExtStype
(
   ULN64               di,                      // IN    sector type
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // OUT   type description
   void               *data                     // INOUT dummy
);

// EXT filesystem, display sector-contents based on type
static ULONG dfsExtDispl
(
   ULN64               psn,                     // IN    base psn for sector
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // IN    sector contents
);

// EXT filesystem, display special LSN info: Inode embedded in sector
static ULONG dfsExtDisplayLsnInfo
(
   ULN64               lsn,                     // IN    possible special lsn
   ULN64               info,                    // IN    possible INODE index
   char               *dc,                      // IN    dummy
   void               *data                     // IN    dummy
);

// Display EXT  super-block
static ULONG dfsExtSuperBlock                   // RET   rc = 0 if type match
(
   BYTE               *sector,                  // IN    Drive specification
   BOOL                navigation               // IN    update nav values
);

// Display value of superblock status value
static void dfsExtDisplayStatus
(
   char               *lead,                    // IN    leading text
   ULONG               st                       // IN    EXT status value
);


// Display one Inode Record, contents including allocation information
static ULONG dfsExtInodeRecord
(
   S_EXT_INODE        *inode,                   // IN    Inode record
   ULONG               inoLsn,                  // IN    LSN for Inode sector
   USHORT              inoIdx                   // IN    index of the Inode
);

// Display and List one (or ALL, from cached Inode) directory blocks as a DIR
static ULONG dfsExtShowDirectory
(
   BYTE               *sector,                  // IN    Dirblock sector data
   ULONG               lsn                      // IN    lsn of this sector
);                                              //       (for re-reading more)

// Display and List one single block of directory information
static ULONG dfsExtShowDirBlock
(
   BYTE               *block,                   // IN    Block of DIR info
   ULONG               dirFlags,                // IN    presentation flags
   ULONG              *entries                  // INOUT total entries done
);

// Convert Linux (Ext?) Filetype value to string (from directory entries)
static char *dfsExtFileType2String              // RET   filetype string[4]
(
   BYTE                fType                    // IN    file type
);

// Make filesystem usage map dump on TxPrint output
static ULONG dfsExtAllocMap
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *options,                 // IN    display options
   void               *data                     // OUT   dummy
);

// DFS EXT write-file to disk (SaveTo inode to file)
static ULONG dfsExtFileSaveAs
(
   ULN64               inoLsn,                  // IN    Inode sectornumber
   ULN64               inoIdx,                  // IN    Inode index in sector
   char               *path,                    // IN    destination path
   void               *ren                      // IN    new name (like 8.3)
);


static char       extDirHeaderText[] =
 " Nr    InoLsn+index Creation/LastAccess Attr Modified / Filename             Filesize    Inode EA";

static char       extDirHeaderLine[] =
 " ===== ==========-= =================== ==== =================== ==================== ======== ==";


/*****************************************************************************/
// Initialize EXT filesystem analysis
/*****************************************************************************/
ULONG dfsExtInit
(
   char               *fs                       // forced filesystem type
)
{
   ULONG               rc = NO_ERROR;
   ULN64               allocDiff = 0;           // groups versus bitmap alloc
   ULONG               group;

   ENTER();

   TRACES(("sizeof S_EXT_SUPER : %u\n", sizeof( S_EXT_SUPER)));
   TRACES(("sizeof S_EXT_INODE : %u\n", sizeof( S_EXT_INODE)));
   TRACES(("sizeof U_EXT_ALLOC : %u\n", sizeof( U_EXT_ALLOC)));

   dfsa->FsCommand          = dfsExtCommand;
   dfsa->FsClose            = dfsExtClose;
   dfsa->FsIdentifySector   = dfsExtIdent;
   dfsa->FsShowType         = dfsExtStype;
   dfsa->FsDisplaySector    = dfsExtDispl;
   dfsa->FsFileInformation  = dfsExtFileInfo;
   dfsa->FsGetAllocSpace    = dfsExtGetAllocSpace;
   dfsa->FsWriteMetaSpace   = NULL;
   dfsa->FsUpdateFileName   = NULL;
   dfsa->FsSetAltBrecLabel  = NULL;
   dfsa->FsMakeBrowseList   = dfsExtMakeBrowseList;
   dfsa->FsFindPath         = dfsExtFindPath;
   dfsa->FsLsnAllocated     = dfsExtAllocated;
   dfsa->FsLsnSetAlloc      = NULL;
   dfsa->FsSltBuild         = dfsExtSltBuild;
   dfsa->FsNpBuild          = dfsExtNpBuild;
   dfsa->FsDisplayError     = dfsExtDispError;
   dfsa->FsDisplayLsnInfo   = dfsExtDisplayLsnInfo;
   dfsa->FsDirIterator      = NULL;
   dfsa->FsDirFileSaveAs    = dfsExtFileSaveAs;
   dfsa->FsAllocDisplay     = dfsExtAllocMap;
   dfsa->FsCl2Lsn           = dfsExtCl2Lsn;
   dfsa->FsLsn2Cl           = dfsExtLsn2Cl;
   dfsa->FsCmdHelp          = ext_txt;          // FS specific cmdhelp text

   dfsa->FsModeId           = DFS_FS_EXT;       // common for EXT2/3/4

   dfsa->Fsi                = ext;
   dfsa->FsSectorTypes      = dfsExtSectorTypes;

   dfsa->FsEntry            = LSN_EXTSUP;       // entry-point in filesystem

   if (((ext->sup = dfsExtGetSuperBlock( NULL)) != NULL)   &&
       (dfsIdentifySector( LSN_EXTSUP, 0, (BYTE *) ext->sup)  == ST_EXTSUP))
   {
      ext->BlockSize       = 1 << (10 + ext->sup->LogBlockSize);
      ext->SectorsPerBlock = ext->BlockSize / dfsGetSectorSize();
      ext->Block           = ext->sup->Blocks;
      ext->Sect            = ext->Block * ext->SectorsPerBlock;
      ext->GroupDescrSize  = (ext->sup->FeatureInComp & EXT_FIN_64BIT) ?
                              ext->sup->DescSize : 32; // older EXT2/EXT3, fixed 32

      TRACES(("BlockSize: %u  SectorsPerBlock: %u   GroupDescrSize: %u\n",
               ext->BlockSize, ext->SectorsPerBlock,  ext->GroupDescrSize));

      if (dfsExtGroupDescriptors( dfsa->verbosity) == NO_ERROR)
      {
         if (ext->sup->FeatureCompat & EXT_FCO_SPARSE_SUPER2)
         {
            ext->SsCount = 2;                   // just 2 backups, listed in superblock
            ext->SsTable = ext->sup->BackupSuperBgs;
         }
         TxPrint( "Group Descr. size : %2u bytes, in %u blocks, for %u groups\n",
                   ext->GroupDescrSize, ext->GdtBlocks, ext->Groups);

         if ((rc = dfsExtBlkBitMapInit()) == NO_ERROR)
         {
            if (TxaOption('a'))                 // show NO allocation by default
            {
               dfsExtAllocMap( 0, 0, "@", NULL); // not really needed (no resize)
            }

            if (ext->sup->State == EXT_FS_CLEAN)
            {
               for (group = 0; group < ext->Groups; group++) // silent check groups
               {
                  dfsExtShowBgBitmap( group, TXAO_SILENT, NULL, NULL, &allocDiff);
               }
               if (allocDiff == 0)
               {
                  TxPrint( "Filesystem CLEAN  : No Groups/Bitmap alloc difference and status OK\n");
                  dfsa->FsDirtyStatus = DFSTAT_CLEAN; // clean  AND no alloc diff
               }
               else                             // clean, but alloc diff, DIRTY
               {
                  TxPrint( "Filesystem %sDIRTY%s  : Groups/Bitmap alloc difference: %llu blocks ",  CBR, CNN, allocDiff);
                  dfsSizeXiB( "(", allocDiff * ext->SectorsPerBlock, dfsGetSectorSize(), ")\n");
                  dfsa->FsDirtyStatus = DFSTAT_DIRTY;
               }
            }
            else                                // dirty status, OR groups/bitmap alloc diff
            {
               TxPrint( "Filesystem %sDirty%s  : As set in the superblock status: %hu\n",  CBR, CNN, ext->sup->State);
               dfsa->FsDirtyStatus = DFSTAT_DIRTY;
            }
            rc = dfsExtInoBitMapInit();
         }
      }
      else
      {
         TxPrint("Error reading Group descriptors in 1st group\n");
      }
   }
   else
   {
      TxPrint("Error reading 1st EXT Super-block at byte offset 1024, defaults used!\n");

      ext->BlockSize       = 4096;
      ext->SectorsPerBlock = 4096 / dfsGetSectorSize();
      ext->Sect            = dfsGetLogicalSize();
      ext->Block           = ext->Sect  / ext->SectorsPerBlock;
      ext->Groups          = ext->Block / EXT_USUAL_BLOCKSPERGROUP + 1;
      //- to be refined, may copy BlockPerGroup into ext anchor, and use that

   }
   ext->InodeMax = ext->Groups * ext->sup->InodesPerGroup - 1;
   dfstSetClusterSize( DFSTORE, (USHORT) ext->SectorsPerBlock); // make sure!

   RETURN (rc);                                 // when needed
}                                               // end 'dfsExtInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Close EXT filesystem for analysis and free any resources
/*****************************************************************************/
static ULONG dfsExtClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;
   int                 i;

   ENTER();

   dfsSlTableReset();                           // stop SLT thread and reset

   dfsExtInoBitmapFlush( FALSE);                // (R/O), and terminate cache
   dfsExtBlkBitmapFlush( TRUE);                 // flush, and terminate cache

   if (ext->Ic != NULL)
   {
      for (i = 0; i <= ext->InodeMax; i++)
      {
         TxFreeMem( ext->Ic[i].Name);           // free cached Inode names
      }
      TxFreeMem( ext->Ic);                      // free name/parent cache
   }

   TxFreeMem( ext->sup);
   TxFreeMem( ext->GroupDescTable);

   dfsCloseFileSystem();
   RETURN (rc);
}                                               // end 'dfsExtClose'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Initialize EXT filesystem analysis for Area (FS info in FDISK mode)
/*****************************************************************************/
ULONG dfsExtAreaInit
(
   void
)
{
   ULONG               rc = NO_ERROR;
   char                st = 0;

   ENTER();

   dfsa->FsAreaClose        = dfsExtAreaClose;
   dfsa->FsAreaLsnAllocated = dfsExtAllocated;

   if ((ext->sup = dfsExtGetSuperBlock( NULL)) != NULL)
   {
      //- will identify on either real superblock, or super-part of EXTBSU
      dfsExtIdent( 0, 0, &st, ext->sup);        // dfsIdentify not available!
      if (st == ST_EXTSUP)
      {
         ext->BlockSize       = 1 << (10 + ext->sup->LogBlockSize);
         ext->SectorsPerBlock = ext->BlockSize / dfsGetSectorSize();
         ext->Block           = ext->sup->Blocks;
         ext->Sect            = ext->Block * ext->SectorsPerBlock;
         ext->GroupDescrSize  = (ext->sup->FeatureInComp & EXT_FIN_64BIT) ?
                                 ext->sup->DescSize : 32; // older EXT2/EXT3, fixed 32

         if (dfsExtGroupDescriptors( TXAO_QUIET) == NO_ERROR)
         {
            rc = dfsExtBlkBitMapInit();
         }
         ext->InodeMax = ext->Groups * ext->sup->InodesPerGroup - 1;
      }
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsExtAreaInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Close EXT filesystem for Area analysis and free any resources
/*****************************************************************************/
static ULONG dfsExtAreaClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   dfsa->FsAreaClose        = NULL;
   dfsa->FsAreaLsnAllocated = NULL;

   rc = dfsExtBlkBitmapFlush( TRUE);            // flush, and terminate cache

   TxFreeMem( ext->sup);
   TxFreeMem( ext->GroupDescTable);

   RETURN (rc);
}                                               // end 'dfsExtAreaClose'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Interpret and execute specific EXT command;
/*****************************************************************************/
static ULONG dfsExtCommand
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *none,                    // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;
   ULN64               sn = 0;                  // sector number input
   LONG                nr = 0;
   TXLN                dc;                      // DFS command
   TXLN                s0;                      // big temporary string space
   TXLN                s1;                      // big temporary string space
   int                 cc;                      // command string count
   char               *c0, *c1, *c2;            // parsed command parts
   char               *pp;                      // parameter pointer

   ENTER();

   pp = TxaGetArgString( TXA_CUR, 1, 0, TXMAXLN, dc); // dc => cmd from arg 1
   cc = TxaArgCount( );                         // number of parameters
   c0 = TxaArgValue(0);
   c1 = TxaArgValue(1);
   c2 = TxaArgValue(2);

   sn = nav.this;                               // default at current sector

   if ((strcmp(c0, "/?") == 0) ||
       (strcmp(c0,  "?") == 0)  )
   {
      TxShowTxt( ext_txt);
   }
   else if (strcasecmp(c0, "super"    ) == 0)
   {
      TxPrint( "\nEXT2/3/4 superblock displayed as HEX dump and formatted ...\n");
      sprintf( dc, "H 0x0%llx%st s", LSN_EXTSUP, dfsa->cmdSeparator);
      rc = dfsMultiCommand( dc, 0, TRUE, FALSE, TRUE);
   }
   else if ((strcasecmp(c0, "cl"       ) == 0) ||
            (strcasecmp(c0, "bl"       ) == 0)  )
   {
      if (cc > 1)
      {
         sn = dfsGetSymbolicSN( c1, 0);
         if (sn <= ext->Block)
         {
            if (cc > 2)
            {
               sprintf( dc, "%s ", c2);
            }
            else
            {
               strcpy(  dc, "");
            }
            sprintf( s1, "0x0%llX", dfsExtBlock2Lsn(sn));
            strcat(  dc, s1);
            TxaReParseCommand( dc);
            rc = DFS_PENDING;                   // handle translated command
         }
         else
         {
            dfsX10( "Invalid block   nr: ", sn, CBR, "\n");
         }
      }
      else
      {
         dfsX10("Current block   nr: ", dfsExtLsn2Block(nav.this), CBM, "\n");
      }
   }
   else if (strcasecmp(c0, "abg"       ) == 0)
   {
      if ((cc == 1) && !TxaOption('?'))
      {
         nr = (dfsGetDisplayMargin() > 140) ? 128 : 64;
         sprintf( dc, "alloc %u -c:%d", ext->sup->BlocksPerGroup / nr, nr);
         TxaReParseCommand( dc);
         rc = DFS_PENDING;                      // handle translated command
      }
      else
      {
         TxPrint("\nDisplay ALLOC map, with exactly one line per block-group\n");
         TxPrint("\nUsage: %s\n", c0);
      }
   }
   else if (strcasecmp(c0, "bg"       ) == 0)
   {
      if ((cc > 1) && !TxaOption('?'))
      {
         if (c1[0] == '*')
         {
            ULONG      group;
            ULN64      usedBitmap = 0;
            ULN64      usedDescr  = 0;
            ULN64      difference = 0;          // absolute difference BM/Desc

            if (dfsa->verbosity == TXAO_NORMAL)       // not whe verbose!
            {
               TxPrint("\nBG Flags: %sS%s=SuperBackup  %sB%s=BLOCK_UNINIT"
                       "  I=INODE_UNINIT  z=InodeTable zeroed\n\n", CBC, CNN, CBY, CNN);
            }
            for (group = 0; ((group < ext->Groups) && !TxAbort()); group++)
            {
               dfsExtShowBgBitmap( group, dfsa->verbosity, &usedBitmap, &usedDescr, &difference);
            }

            if (difference > 0)
            {
               TxPrint( "\nBitmap allocation differs by %d blocks ",  difference);
               dfsSizeXiB( "(", difference * ext->SectorsPerBlock, dfsGetSectorSize(), ")");
               TxPrint(" from BG-desciptors (%sdirty!%s)\n", CBR, CNN);

               TxPrint("\nBG Used blocks: 0x%10.10llx = %12llu, from BG-descr. tables;   Usage % 4.1lf%%",
                           usedDescr, usedDescr, (double) (100 * ((double) usedDescr)  / ext->Block));
            }
            TxPrint("\nBM Used blocks: 0x%10.10llx = %12llu of %10.10llx = %12llu; Usage % 4.1lf%%\n",
                                  usedBitmap, usedBitmap,   ext->Block, ext->Block,
                    (double) (100.0 * ((double) usedBitmap) / (double) ext->Block));
         }
         else
         {
            dfsExtShowBgBitmap( atoi( c1), dfsa->verbosity, NULL, NULL, NULL);
         }
      }
      else
      {
         TxPrint("\nDisplay block-group free/used from desc/bitmap, optional display bitmap\n");
         TxPrint("\nUsage: %s  group | * [-O:v]\n"
                 "\n     group : Group number, range 0 - %u, decimal or hex"
                 "\n         * : Iterate over all groups, displaying each one, then"
                 "\n             showing totals, with Descriptor/Bitmap differences\n"
                 "\n      -O:v : Verbose, display a bitmap allocation graphic\n",
                                                           c0, ext->Groups -1);
      }
   }
   else if (strcasecmp(c0, "bsl"       ) == 0)
   {
      if ((cc == 1) && !TxaOption('?'))
      {
         ULONG         i;
         ULONG         bg;
         LLONG         block;
         ULONG         bpg = (ext->sup->BlocksPerGroup) ? ext->sup->BlocksPerGroup
                                                        : EXT_USUAL_BLOCKSPERGROUP;
         for (i = 0; i < ext->SsCount; i++)
         {
            bg     = ext->SsTable[ i];
            block  = ((LLONG) bg) * bpg;
            if ((block > ((LLONG) ext->Block)) && (!TxaOption('f')))
            {
               break;
            }
            TxPrint( "Backup#:%3u at BG:%9lu  "
                     "Superblock at bl:0x%10.11llx, sect:0x%11.11llx\n",
                        i + 1, bg, block, block * ext->SectorsPerBlock);
         }
      }
      else
      {
         TxPrint("\nDisplay Sparse-Super blockgroup# table, location of backup\n");
         TxPrint("\nUsage: %s  [-f]\n"
                 "\n       -f : Full table, do not limit to current FS size\n", c0);
      }
   }
   else if (strcasecmp(c0, "filefind" ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint( "\nFind files by direct search of in-use file-inodes in the Inode tables\n");
         TxPrint( "\n Usage: %s  [name]  [-D | -f] [-f:first] [-v]\n\n", c0);
         TxPrint( "   name      = partial or whole filename wanted (NOT a wildcard!)\n"
                  "               matches on filename only, not the path component!\n"
                  "   -D        = search  directories  only, not files+directories\n"
                  "   -f        = search regular files only, not files+directories\n"
                  "   -F:first  = start at given Inode nr, not known first user one\n"
                  "   -v        = verbose search, list files while found (SLOW!)\n");
      }
      else                                      // search Inodes
      {
         TXTS          findtype;

         if      (TxaOption('D'))               // Directories only
         {
            strcpy( findtype, TxaOption('D') ? "D" : "fD");
         }
         else if (TxaOption('f'))               // Files only
         {
            strcpy( findtype, TxaOption('D') ? "D" : "fD");
         }
         else                                   // Files + Directories
         {
            strcpy( findtype, "fD");
         }

         if (dfsSlTableStatus( NULL) != SLT_READY)
         {
            TxPrint( "\nBuilding SLT for filename lookup, please wait ..."
                     "   (or abort that using <Esc>)\n");
            dfsMultiCommand( "slt 0 1 -v- -d-", 0, TRUE, FALSE, FALSE);
            TxCancelAbort();                    // might have aborted
         }
         sn = TxaOptNum('F', "",  ext->sup->FirstInode); // First Inode nr todo
         TxPrint( "\nQuick find '%s' Inodes matching %s%s%s\n",
              findtype, (cc > 1) ? CBG : "", (cc > 1) ? c1 : "any name", CNN);
         sprintf( dfsa->brdescript, "EXTn Quick %s %s", c0, c1);
         rc = dfsExtQuickFindInodes( TxaOption('v'), sn, findtype, c1);
         if ((rc == NO_ERROR) && (dfsa->snlist[0] != 0))
         {
            TxPrint( "\nYou can use the 'list -f' or 'browse' command (key <F9>),\n"
                     "to display/browse the found files, and copy/recover them.\n");
         }
      }
   }
   else if (strcasecmp(c0, "ino"      ) == 0)   // calculate LSN+index for INODE
   {
      USHORT           index;

      if (cc > 1)
      {
         nr = dfsGetSymbolicSN( c1, 2);         // default INODE 2 (root)
         sn = dfsExtInode2LsnIndex( nr, &index);
         if      (sn == L64_NULL)
         {
            dfsX10(  "\nInvalid INODE  nr : ", nr, CBR, "\n");
            dfsDec8( "Max  INODE number : ", ext->InodeMax, "\n");
         }
         else if (sn == 0)                      // to be refined for EXT!
         {
            dfsX10(  "\nINODE extent for  : ", nr, CBM, " is not allocated yet (not in use)\n");
         }
         else
         {
            sprintf( dc, "0x0%llX -e:%hu", sn, index);
            TxaReParseKeepOpts( dc);            // reparse but keep option values
            rc = DFS_PENDING;                   // handle translated command
         }
      }
   }
   else if (strcasecmp(c0, "label"    ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nSet volume label in superblock (16-character, Linux FS-name)\n");
         TxPrint("\nUsage: %s  [label]\n\n"
                 "   label  : new volume label, maximum 16 characters\n"
                 "   -!-    = do not prompt for new value, just set it\n", c0);
      }
      else
      {
         if (cc > 1)
         {
            strcpy( s1, c1);                    // use specified parameter value
         }
         else if (SINF->p)                      // it is a partition
         {
            strcpy( s1, SINF->p->plabel);       // use label determined by readdiskinfo
         }
         else
         {
            strcpy( s1, "");                    // start with empty label
         }
         s1[ EXT_LEN_LBL] = 0;                  // truncate to allowed length

         if (dfsa->batch)                       // s1 has new label value
         {
            memcpy( ext->sup->VolumeName, s1, EXT_LEN_LBL);
            rc = dfsExtWriteSuperBlocks();
         }
         else
         {
            #if defined (USEWINDOWING)          // Allow interactive update
            if (!TxaOptUnSet('!') && txwIsWindow( TXHWND_DESKTOP))
            {
               TXLN      prompt;

               sprintf( prompt, "%s\n\nVolume label in superblock: '%s'\n\n"
                                "Specify new %d-character volume label\n",
                                 dfstStoreDesc1( DFSTORE) + 10, s1, EXT_LEN_LBL);

               if (txwPromptBox( TXHWND_DESKTOP, TXHWND_DESKTOP, NULL,
                      prompt, " Set volume label in superblock ",
                      5144, TXPB_MOVEABLE | TXPB_HCENTER | TXPB_VCENTER,
                      EXT_LEN_LBL, s1) != TXDID_CANCEL)
               {
                  TxStrip( s1, s1, ' ', ' ');   // strip leading/trailing spaces
                  memcpy( ext->sup->VolumeName, s1, EXT_LEN_LBL);
                  rc = dfsExtWriteSuperBlocks();
               }
            }
            else
            #endif
            {
               if (TxConfirm( 5144, "Write Volume LABEL '%s' to superblock ? [Y/N] : ", s1))
               {
                  memcpy( ext->sup->VolumeName, s1, EXT_LEN_LBL);
                  rc = dfsExtWriteSuperBlocks();
               }
            }
         }
      }
   }
   else if (strcasecmp(c0, "uuid"    ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nUsage: %s  [UUID] | [-u:UUID]\n"
                 "\n      UUID : UUID string value, 5 groups separated by hyphens, no brackets"
                 "\n   -u:UUID : 36 character UUID value: 12345678-abcd-0123-4567-0123456789ab\n", c0);
      }
      else
      {
         if (TxaOptMutEx((cc > 1), "u", "if a UUID parameter is specified!", NULL))
         {
            if (cc > 1)
            {
               strcpy( s1, c1);                 // use specified parameter value
            }
            else
            {
               strcpy( s1, TxaOptStr( 'u', NULL, ""));
            }
            sprintf( s0, "{%s}", s1);           // Make fully-formed UUID string

            if (((strlen( s1) == 0) && (!dfsa->batch)) || dfsUidStringIsValid( s0))
            {
               if (dfsa->batch)                 // s0 has valid new UUID
               {
                  dfsUidStringToBinary( s0, ext->sup->Uuid, TRUE);
                  rc = dfsExtWriteSuperBlocks();
               }
               else
               {
                  #if defined (USEWINDOWING)    // Allow interactive update
                  if (txwIsWindow( TXHWND_DESKTOP))
                  {
                     if (dfsGuidUuidEditor( "Update UUID value in EXT Superblock",
                                            "Write Back", SINF->partid,
                                             5143, s0, TRUE, ext->sup->Uuid) == NO_ERROR)
                     {
                        rc = dfsExtWriteSuperBlocks();
                     }
                  }
                  else
                  #endif
                  {
                     if (TxConfirm( 5143, "Write UUID %s to superblock ? [Y/N] : ", s0))
                     {
                        dfsUidStringToBinary( s0, ext->sup->Uuid, TRUE);
                        rc = dfsExtWriteSuperBlocks();
                     }
                  }
               }
            }
            else
            {
               TxPrint( "\nInvalid UUID value: '%s'\n", s1);
            }
         }
      }
      TxPrint( "\nVolume UUID now   : %s\n", dfsFsUuidValueString( ext->sup->Uuid));
   }
   else
   {
      rc = DFS_CMD_UNKNOWN;                     // cmd not recognized
   }
   RETURN (rc);
}                                               // end 'dfsExtCommand'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// EXT  filesystem, identify specified sector as a valid superblock
/*****************************************************************************/
BOOL dfsExtIsSuperBlock                         // RET   sector is a valid sb
(
   BYTE               *sec                      // IN    sector contents
)
{
   BOOL                rc = FALSE;
   S_EXT_SUPER        *sp = (S_EXT_SUPER*) sec;

   ENTER();

   if ((sp->Signature    == SV_EXTSUP)       &&
       (sp->State        <= EXT_FS_MAXSTATE) &&
       (sp->LogBlockSize <= EXT_BLOCKS_4K  )  )
   {
      rc = TRUE;
   }
   BRETURN(rc);
}                                               // end 'dfsExtIsSuperBlock'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// EXT  filesystem, identify specified sector as (start of) a directory block
/*****************************************************************************/
BOOL dfsExtIsDirectoryBlock                     // RET   sector is a valid sb
(
   BYTE               *sec                      // IN    sector contents
)
{
   BOOL                rc   = FALSE;
   S_EXT_DIRENTRY     *d0   = (S_EXT_DIRENTRY *)           sec;
   S_EXT_DIRENTRY     *d1   = (S_EXT_DIRENTRY *) ((BYTE *) sec + 12);
   USHORT              eBlk = ext->BlockSize - 12;  //- 2nd goes to end-block EMPTY
                                                    //- DIR, or hidden INDEX slots.
   ENTER();

   TRHEXS( 70, d0, 16, "S_EXT_DIRENTRY 0");
   TRHEXS( 70, d1, 16, "S_EXT_DIRENTRY 1");

   if ((d0->NameLen == 1) && (d0->FileType == EXT_FT_DIR) &&  (d0->RecLen == 12)    &&
       (d1->NameLen == 2) && (d1->FileType == EXT_FT_DIR) && ((d1->RecLen == 12)    ||
                                                              (d1->RecLen == eBlk)) &&
       (strcmp( d0->Name, "." ) == 0)   &&
       (strcmp( d1->Name, "..") == 0)    )      // DOT and DOTDOT in each DIR
   {
      rc = TRUE;
   }
   BRETURN(rc);
}                                               // end 'dfsExtIsDirectoryBlock'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// EXT  filesystem, check if given BG has backups for superblock + BG-descr
/*****************************************************************************/
BOOL dfsExtIsBackupSuperBG                      // RET   BG# has super backups
(
   ULONG               group                    // IN    blockgroup number
)
{
   BOOL                rc = FALSE;

   ULONG               i;
   ULONG               bg;

   if (group != 0)                              // search table for group > 0
   {
      for (i = 0; i < ext->SsCount; i++)
      {
         bg = ext->SsTable[ i];
         if (group == bg)                       // exact match
         {
            rc = TRUE;
         }
         if (group <= bg)                       // match, or smaller than
         {                                      // all following ones
            break;
         }
      }
   }
   else                                         // group 0 always has SUPER
   {
      rc = TRUE;
   }
   return (rc);
}                                               // end 'dfsExtIsBackupSuperBG'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// EXT filesystem, display sector constents as a superblock
/*****************************************************************************/
void dfsExtDisplaySuperBlock
(
   BYTE               *sec                      // IN    sector contents
)
{
   dfsExtSuperBlock( sec, FALSE);
}                                               // end 'dfsExtDisplaySuperBlock'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// EXT filesystem, identify specified sector
/*****************************************************************************/
ULONG dfsExtIdent
(
   ULN64               lsn,                     // IN    LSN  (possible Inode)
   ULN64               sninfo,                  // IN    INFO (possible Inode)
   char               *st,                      // OUT   sector type
   void               *sec                      // IN    sector contents
)
{
   ULONG               dr = NO_ERROR;
   char                rc = ST_UDATA;

   ENTER();
   TRACES(("lsn:%8.8x sninfo:%8.8x\n", lsn, sninfo));

   if      (dfsExtIsDirectoryBlock( (BYTE *) sec))
   {
      rc = ST_EXTDIR;
   }
   else if (dfsExtIsSuperBlock( (BYTE *) sec))
   {
      rc = ST_EXTSUP;
   }
   else if ((dfsGetSectorSize() >= 2048) && (dfsExtIsSuperBlock( (BYTE *) sec + 1024)))
   {
      rc = ST_EXTBSU;                           // combined Boot+Superblock sector
   }
   else if (sninfo & ~DFSSNINFO)                // there IS an Inode index
   {
      if (dfsExtLsnIndex2Inode( lsn, (USHORT) (sninfo & ~DFSSNINFO)) != 0) // valid Inode
      {
         S_EXT_INODE  *inode = (S_EXT_INODE *) sec; // treat as array of Inodes

         if (S_ISDIR( inode[sninfo & ~DFSSNINFO].Mode))
         {
            rc = ST_EXTDIN;                     // Dir  Inode (type 'D')
         }
         else
         {
            rc = ST_EXTINO;                     // File Inode (type 'f')
         }
      }
   }
   switch (rc)
   {
      case ST_UDATA:
         dr = DFS_PENDING;
         break;

      default:
         break;
   }
   *st = rc;
   RETURN (dr);
}                                               // end 'dfsExtIdent'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// EXT filesystem, supply sector-type description string
/*****************************************************************************/
static ULONG dfsExtStype
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *st,                      // IN    sector type
   void               *data                     // OUT   type description
)
{
   ULONG               dr  = NO_ERROR;
   char               *buf = (char *) data;
   BYTE                tp  = st[0];

   switch (tp)                                  // searchable types
   {
      case ST_EXTSUP: sprintf(buf,"EXT superblock   "); break;
      case ST_EXTBSU: sprintf(buf,"EXT boot+superblk"); break;
      case ST_EXTDIR: sprintf(buf,"Directory block  "); break;
      case ST_EXTINO: sprintf(buf,"File Inode       "); break;
      case ST_EXTDIN: sprintf(buf,"Directory Inode  "); break;
      case ST_EXTEAB: sprintf(buf,"Ext-Attrib block "); break;
      default:
         switch (tp | ST__INFO)                 // non-searchable ones
         {
            case ST_EXTBUS: sprintf(buf,"BACKUP Superblock"); break;
            case ST_EXTBGR: sprintf(buf,"BACKUP GD-Table  "); break;
            case ST_EXTGRP: sprintf(buf,"Group Descr-Table"); break;
            case ST_EXTGDR: sprintf(buf,"GDT Reserved area"); break;
            case ST_EXTBBM: sprintf(buf,"BlockAlloc Bitmap"); break;
            case ST_EXTIBM: sprintf(buf,"InodeAlloc Bitmap"); break;
            case ST_EXTITB: sprintf(buf,"Inode Table      "); break;
            case ST_EXTBI1: sprintf(buf,"BlockIndirectLvl1"); break;
            case ST_EXTBI2: sprintf(buf,"BlockIndirectLvl2"); break;
            case ST_EXTBI3: sprintf(buf,"BlockIndirectLvl3"); break;
            case ST_EXTAIX: sprintf(buf,"AllocExtent Index"); break;
            case ST_EXTALF: sprintf(buf,"AllocExtent Leaf "); break;
            case ST_EXTJRN: sprintf(buf,"Journaling File  "); break;
            case ST_EXTRES: sprintf(buf,"ResizeInformation"); break;
            default:       dr = DFS_PENDING;
               break;
         }
         break;
   }
   return (dr);
}                                               // end 'dfsExtStype'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// EXT filesystem, display sector-contents based on type
/*****************************************************************************/
static ULONG dfsExtDispl
(
   ULN64               psn,                     // IN    base psn for sector
   ULN64               d2,                      // IN    dummy
   char               *type,                    // IN    type of sector
   void               *data                     // IN    sector contents
)
{
   ULONG               dr   = NO_ERROR;
   BYTE                st   = (BYTE) *type;
   BYTE               *sect = (BYTE *) data;
   ULN64               lsn  = dfstPsn2LSN( DFSTORE, psn);

   ENTER();

   switch (st)
   {
      case ST_EXTSUP:
         (void) dfsExtSuperBlock( sect, TRUE);
         break;

      case ST_EXTBSU:
         if (dfsGetSectorSize() >= 2048)        // safety check, avoid traps
         {
            if (((S_BOOTR*)sect)->Signature == SV_BOOTR)
            {
               TxPrint( "\nEXTn Combined Boot+Superblock, with boot signature, BOOTRECORD contents:\n\n");
               (void) dfsBootSector( sect);
            }
            else                                // start does NOT have bootcode
            {
               TxPrint( "\nEXTn Combined Boot+Superblock, WITHOUT bootcode present, superblock only.\n\n");
            }
            (void) dfsExtSuperBlock( sect + 1024, TRUE);
         }
         else                                   // should not happen (for real EXTBSU)
         {
            dr = DFS_PENDING;
         }
         break;

      case ST_EXTDIR:
         dr = dfsExtShowDirectory( sect, lsn);
         break;

      default:
         dr = DFS_PENDING;
         break;
   }
   RETURN (dr);
}                                               // end 'dfsExtDispl'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// EXT filesystem, display special LSN info: Inode embedded in sector
/*****************************************************************************/
static ULONG dfsExtDisplayLsnInfo
(
   ULN64               lsn,                     // IN    possible special lsn
   ULN64               info,                    // IN    possible INODE index
   char               *dc,                      // IN    dummy
   void               *data                     // IN    dummy
)
{
   ULONG               dr = DFS_PENDING;

   /*
   TXLN                path;
   */

   ENTER();

   TRACES(("lsn:%llx  info:%8.8x\n", lsn, info));

   if (info & DFSSNINFO)                        // extra info present
   {
      /* to be refined
      if (dfsExtLsnInfo2Path( lsn, info, path) == NO_ERROR)
      {
         TxPrint("\nFound path to root: '%s%s%s'\n", CBY, path, CNN);
      }
      */
      nav.xtra = lsn;                           // keep reference around
      dr = dfsExtReadDisplayInode( lsn, (BYTE) DFSSNIGET(info));
   }
   RETURN (dr);
}                                               // end 'dfsExtDisplayLsnInfo'
/*---------------------------------------------------------------------------*/


#define EXT_SCF(fl,str) if (sd->FeatureCompat & fl) TxPrint("CompatibleFeature : 0x%8.8x = %s\n", fl, str)
#define EXT_SRF(fl,str) if (sd->FeatureRoComp & fl) TxPrint("Read-Only Feature : 0x%8.8x = %s\n", fl, str)
#define EXT_SIF(fl,str) if (sd->FeatureInComp & fl) TxPrint("InCompat. Feature : 0x%8.8x = %s\n", fl, str)
/*****************************************************************************/
// Display EXT  super-block (can be called WITHOUT Init being run!)
/*****************************************************************************/
static ULONG dfsExtSuperBlock                   // RET   rc = 0 if type match
(
   BYTE               *sector,                  // IN    Superblock data
   BOOL                navigation               // IN    update nav values
)
{
   ULONG               rc = 0;                  // rc, sector match
   S_EXT_SUPER        *sd = (S_EXT_SUPER *) sector;
   TXTM                text;

   ENTER();

   sprintf( text, "%u.%hu", sd->ExtRevision, sd->ExtMinorRev);
   TxPrint("Superbl signature : 0x%s%4.4hx%s       in BlockGroup number: %hu\n",
           (sd->Signature == SV_EXTSUP) ? CBG : CBR, sd->Signature, CNN, sd->BlockGroupNr);

   TxPrint("Formatted by  EXT : %-10.10s   Creator OS: 0x%4.4hx = ", text, sd->CreatorOS);
   switch (sd->CreatorOS)
   {
      case EXT_OS_LINUX:       TxPrint( "Linux\n");     break;
      case EXT_OS_HURD:        TxPrint( "Hurd\n");      break;
      case EXT_OS_MASIX:       TxPrint( "Masix\n");     break;
      case EXT_OS_FREEBSD:     TxPrint( "FreeBSD\n");   break;
      case EXT_OS_LITES4:      TxPrint( "Lites4\n");    break;
      default:                 TxPrint( "Unknown\n");   break;
   }
   dfsExtDisplayStatus( "Filesystem status : ",  sd->State);
   TxPrint("FS mounted since  : %-24.24s, mount %hu of %hu before FSCK\n",
                                (sd->Mtime) ? ctime((time_t *) &sd->Mtime)
                              : "Never", sd->Mounts, sd->MaxMounts);
   TxPrint("Last check done   : %-24.24s, error mask: 0x%4.4hx\n",
               ctime((time_t *) &sd->LastCheck), sd->Errors);
   TxPrint("Write timestamp   : %-24.24s, Uid: 0x%4.4hx Gid: 0x%4.4hx\n",
               ctime((time_t *) &sd->Wtime), sd->UidReserved, sd->GidReserved);
   TxPrint("Flags, Compatible : 0x%8.8x, Read-Only: 0x%8.8x, InCompatible: 0x%8.8x\n",
             sd->FeatureCompat, sd->FeatureRoComp, sd->FeatureInComp);

   EXT_SCF( EXT_FCO_IMAGIC_INODES    , "Magic Inodes");
   EXT_SCF( EXT_FCO_HAS_JOURNAL      , "Journaling present                 (EXT3/EXT4)");
   EXT_SCF( EXT_FCO_EXT_ATTR         , "Extended attributes");
   EXT_SCF( EXT_FCO_RESIZE_INODE     , "Has GDT-reserved blocks       (RESIZE_INODE 7)");
   EXT_SCF( EXT_FCO_DIR_INDEX        , "Directory indexes present          (EXT3/EXT4)");
   EXT_SCF( EXT_FCO_SPARSE_SUPER2    , "Sparse Super 2         (Backup in 2 BGs, EXT4)");
   if (sd->FeatureCompat & ~EXT_FCO_KNOWN_FLAGS)
   {
      TxPrint( "CompatibleFeature : 0x%s%8.8x%s = Unknown  COMPATIBLE  feature(s) for "
               "DFSee %s\n", CBC, (sd->FeatureCompat & ~EXT_FCO_KNOWN_FLAGS), CNN, DFS_VN);
   }
   EXT_SRF( EXT_FRO_SPARSE_SUPER     , "Sparse Super            (BG 0, 1, 3^n,5^n,7^n)");
   EXT_SRF( EXT_FRO_LARGE_FILE       , "Large file support, > 2Gb is OK");
   EXT_SRF( EXT_FRO_HUGE_FILE        , "Huge files, size in blocks              (EXT4)");
   EXT_SRF( EXT_FRO_GDT_CSUM         , "Group descriptors have checksums");
   EXT_SRF( EXT_FRO_DIR_NLINK        , "Huge directories, more than 65k         (EXT4)");
   EXT_SRF( EXT_FRO_EXTRA_ISIZE      , "Large Inodes exist in FS                (EXT4)");
   EXT_SRF( EXT_FRO_BTREE_DIR        , "Btree directories");
   EXT_SRF( EXT_FRO_BIGALLOC         , "BIGALLOC, bitmap represents Clusters    (EXT4)");
   EXT_SRF( EXT_FRO_METADATA_CSUM    , "FS uses Metadata CSUM, implies GDT_CSUM (EXT4)");
   EXT_SRF( EXT_FRO_REPLICA          , "FS supports Replicas                    (EXT4)");
   EXT_SRF( EXT_FRO_READONLY         , "FS must be mounted as RO                (EXT4)");
   EXT_SRF( EXT_FRO_PROJECT          , "FS tracks project quota                 (EXT4)");
   if (sd->FeatureCompat & ~EXT_FRO_KNOWN_FLAGS)
   {
      TxPrint( "Read-Only Feature : 0x%s%8.8x%s = Unknown  READ-ONLY   feature(s) for "
               "DFSee %s\n", CBY, (sd->FeatureCompat & ~EXT_FRO_KNOWN_FLAGS), CNN, DFS_VN);
   }
   EXT_SIF( EXT_FIN_COMPRESSION      , "Compression");
   EXT_SIF( EXT_FIN_FILETYPE         , "Filetype on DIR entries");
   EXT_SIF( EXT_FIN_RECOVER          , "Recover");
   EXT_SIF( EXT_FIN_JOURNAL_DEV      , "Journal device");
   EXT_SIF( EXT_FIN_META_BG          , "Meta Block Groups, disables DFSee SMART (EXT4)");
   EXT_SIF( EXT_FIN_EXTENTS          , "File alloc uses extents                 (EXT4)");
   EXT_SIF( EXT_FIN_64BIT            , "Block numbers are 64 bits               (EXT4)");
   EXT_SIF( EXT_FIN_MMP              , "Multiple Mount Protection               (EXT4)");
   EXT_SIF( EXT_FIN_FLEX_BG          , "Flexible Block Groups used              (EXT4)");
   EXT_SIF( EXT_FIN_EA_INODE         , "Inode may be used for large EA values   (EXT4)");
   EXT_SIF( EXT_FIN_DIRDATA          , "Data in Dir-entry (Not Implemented?)    (EXT4)");
   EXT_SIF( EXT_FIN_CSUM_SEED        , "Metadata checksum seed is in superblock (EXT4)");
   EXT_SIF( EXT_FIN_LARGEDIR         , "Directory > 2GB or 3-levels htree       (EXT4)");
   EXT_SIF( EXT_FIN_INLINE_DATA      , "Data may be present inside an Inode     (EXT4)");
   EXT_SIF( EXT_FIN_ENCRYPT          , "Encrypted Inodes are present            (EXT4)");
   if (sd->FeatureInComp & ~EXT_FIN_KNOWN_FLAGS)
   {
      TxPrint( "InCompat. Feature : 0x%s%8.8x%s = Unknown INCOMPATIBLE feature(s) for "
               "DFSee %s\n", CBR, (sd->FeatureCompat & ~EXT_FIN_KNOWN_FLAGS), CNN, DFS_VN);
   }
   if (sd->FeatureInComp & EXT_FIN_FLEX_BG)
   {
      TxPrint("#BG per FlexGroup : %10u   "
              "from 2^logBgPerFlex : %10u\n", (1 << sd->LogBgPerFlex),  sd->LogBgPerFlex);
   }
   if (sd->FeatureRoComp & EXT_FRO_BIGALLOC)
   {
      TxPrint("Blocks  / Cluster : %10u   "
              "from 2^logClustSize : %10u\n", (1 << sd->LogClustSize),  sd->LogClustSize);
   }
   TxPrint("Clusters  / Group : %10u   Nr. of block groups : %10u = 0x%8.8x\n",
                                            sd->ClustsPerGroup, ext->Groups,   ext->Groups);
   TxPrint("Free Inodes count : %10u   Allocated Inodes    : %10u = 0x%8.8x\n",
                                            sd->FreeInodes,     sd->Inodes,    sd->Inodes);
   TxPrint("Inodes, per Group : %10u   Inode Size in bytes : %10hu = 0x%8.8x\n",
                                            sd->InodesPerGroup, sd->InodeSize, sd->InodeSize);
   dfsSz64("Block Group  size : ", (ULN64)  sd->BlocksPerGroup * ext->SectorsPerBlock, "");
   TxPrint("  blocks : %10u = 0x%8.8x\n",   sd->BlocksPerGroup, sd->BlocksPerGroup);
   dfsSz64("FS     total size : ",          ext->Sect,  "");
   TxPrint("  blocks : %10u = 0x%8.8x\n",   sd->Blocks, sd->Blocks);
   dfsSiz8("Sect/Block   size : ",          ext->SectorsPerBlock, "\n");
   if (sd->VolumeName[0] == 0)
   {
      strcpy( text, "-none-");
   }
   else
   {
      TxCopy(  text, sd->VolumeName, EXT_LEN_LBL + 1);
   }
   TxPrint("Volume label      : %s%s%s\n", CBG, text, CNN);
   TxPrint("Volume UUID       : %s\n", dfsFsUuidValueString( sd->Uuid));

   if (navigation)                              // set auto display on ENTER to the root Inode
   {
      nav.down         = dfsExtInode2LsnIndex( EXT_I_ROOT, &nav.down_sninfo);
      nav.down_sninfo |= DFSSNINFO;
   }
   RETURN (rc);
}                                               // end 'dfsExtSuperBlock'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display value of superblock status value
/*****************************************************************************/
static void dfsExtDisplayStatus
(
   char               *lead,                    // IN    leading text
   ULONG               st                       // IN    EXT status value
)
{
   TxPrint("%s%u = ", lead, st);
   if      (st == EXT_FS_VIRGIN) TxPrint( "%sNever mounted, virgin. ", CBY);
   else if (st == EXT_FS_CLEAN)  TxPrint( "%sUnmounted and clean.   ", CBG);
   else
   {
      TxPrint( "%s", CBR);
      if (st & EXT_FS_ERROR)     TxPrint( "Errors present; ");
      if (st & EXT_FS_ORPHAN)    TxPrint( "Orphans being recovered; ");
   }
   if (st > EXT_FS_MAXSTATE)
   {
      TxPrint( "%sInvalid state value%s", CBR, CNN);
   }
   TxPrint( "%s\n", CNN);                       // back to neutral color
}                                               // end 'dfsExtDisplayStatus'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display one Inode Record from supplied sector data, based on the index
/*****************************************************************************/
ULONG dfsExtShowInodeRecord
(
   BYTE               *sector,                  // IN    Inode sector data
   ULN64               lsn,                     // IN    LSN for Inode sector
   BYTE                index                    // IN    index of the Inode
)
{
   ULONG               rc = NO_ERROR;
   TX1K                zero;

   ENTER();
   TRACES(("index: 0x%2.2hhx\n", index));

   if (index < (dfsGetSectorSize() / ext->sup->InodeSize))
   {
      BYTE *inodeRecord = sector + (index * ext->sup->InodeSize);

      memset( zero, 0, sizeof(zero));

      if (memcmp( inodeRecord, zero, ext->sup->InodeSize) != 0)
      {
         rc = dfsExtInodeRecord( (S_EXT_INODE *) inodeRecord, lsn, index);
      }
      else
      {
         TxPrint( "\nThe Inode contents is all ZERO, unused inode record ...\n");
      }
   }
   else
   {
      rc = DFS_VALUE_ERROR;
   }
   RETURN( rc);
}                                               // end 'dfsExtShowInodeRecord'
/*---------------------------------------------------------------------------*/


#define EXT_INF(fl,str) if (inode->Flags & fl) TxPrint("Inode Flag value  : 0x%8.8x = %s\n", fl, str)
/*****************************************************************************/
// Display one Inode Record, contents including allocation information
/*****************************************************************************/
static ULONG dfsExtInodeRecord
(
   S_EXT_INODE        *inode,                   // IN    Inode record
   ULONG               inoLsn,                  // IN    LSN for Inode sector
   USHORT              inoIdx                   // IN    index of the Inode
)
{
   ULONG               rc = NO_ERROR;
   TXLN                text;                    // ASCII buffer, large
   ULN64               fileSize;                // file size in bytes
   ULN64               alBlocks;                // allocated size in bytes
   ULN64               fsBlocks;                // allocated size in blocks
   ULONG               AtExtra = 0;             // Access time, extra bits
   ULONG               ItExtra = 0;             // Inode Change time, extra bits
   ULONG               MtExtra = 0;             // Modification time, extra bits
   BOOL                isDir   = (S_ISDIR( inode->Mode));
   DFSISPACE           fdata;                   // file data allocation
   DFSISPACE           meta1;                   // meta data L1/leaf
   DFSISPACE           meta2;                   // meta data L2/index
   ULONG               parent;                  // parent Inode#

   ENTER();

   memset( &fdata, 0, sizeof(DFSISPACE));       // initialize to all zero
   memset( &meta1, 0, sizeof(DFSISPACE));
   memset( &meta2, 0, sizeof(DFSISPACE));

   TxPrint( "Hard Links Count  : %10u            Generation : %u\n",
                         inode->Links,     inode->Generation);

   dfsPosixUidGidPrint( (inode->UidHigh << 16) + inode->UidLow,
                        (inode->GidHigh << 16) + inode->GidLow);
   TxPrint( "File Mode   (%03o) : %s\n", (inode->Mode & 0777),
                      dfsPosixMode2String( inode->Mode, TXMAXLN, text));

   if (dfsExtResolveInodeName( inoLsn, inoIdx, inode, &parent, text))
   {
      if (parent != L32_NULL)                   // parent resolved too ?
      {
         TxPrint( "Parent   Inode nr : 0x%8.8x\n", parent);
      }
      if (strlen( text))
      {
         TxPrint( "1st link Filename : %s%s%s\n", CBY, text, CNN);
      }
      if (dfsExtFindRootInode( dfsExtLsnIndex2Inode( inoLsn, inoIdx), text, &parent))
      {
         TxPrint( "Filename fullpath : %s%s%s\n", CBY, text, CNN);
      }
   }

   if (inode->Flags)
   {
      TxPrint("Inode Flag value  : 0x%8.8x\n", inode->Flags);

      EXT_INF( EXT_FL_SECRM           , "Requires secure delete       (not impl)");
      EXT_INF( EXT_FL_UNRM            , "Preserve on undelete         (not impl)");
      EXT_INF( EXT_FL_COMPR           , "File is compressed           (not impl)");
      EXT_INF( EXT_FL_SYNC            , "All writes must be synchronous");
      EXT_INF( EXT_FL_IMMUTABLE       , "File is immutable");
      EXT_INF( EXT_FL_APPEND          , "File can only be appended");
      EXT_INF( EXT_FL_NODUMP          , "do not dump file (dumputil)");
      EXT_INF( EXT_FL_NOATIME         , "do not update access time");
      EXT_INF( EXT_FL_DIRTY           , "Dirty compressed file        (not impl)");
      EXT_INF( EXT_FL_COMPRBLK        , "File has compressed clusters (not impl)");
      EXT_INF( EXT_FL_NOCOMPR         , "Don't compress this file");
      EXT_INF( EXT_FL_ENCRYPT         , "Inode is encrypted");
      EXT_INF( EXT_FL_INDEX           , "Directory has hash-indexes");
      EXT_INF( EXT_FL_IMAGIC          , "AFS magic directory");
      EXT_INF( EXT_FL_JRNL_DATA       , "File data must be be journaled");
      EXT_INF( EXT_FL_NO_TAIL         , "File tail should not be merged");
      EXT_INF( EXT_FL_DIR_SYNC        , "Dir data must be written synchronously");
      EXT_INF( EXT_FL_TOP_DIR         , "Top of directory hierarchy (root)");
      EXT_INF( EXT_FL_HUGE_FILE       , "Huge file,");
      EXT_INF( EXT_FL_EXTENTS         , "Inode uses extents for allocation");
      EXT_INF( EXT_FL_LARGE_EA        , "Inode for a large extended attribute");
      EXT_INF( EXT_FL_EOFBLOCKS       , "File has blocks past EOF (deprecated)");
      EXT_INF( EXT_FL_SNAPSHOT        , "Inode is a Snapshot");
      EXT_INF( EXT_FL_SNAP_DEL        , "Snapshot is being deleted");
      EXT_INF( EXT_FL_SNAPSHRUNK      , "Snapshot shrink has completed");
      EXT_INF( EXT_FL_INLINEDATA      , "Inode has small INLINE data");
      EXT_INF( EXT_FL_PROJ_INHRT      , "Create childeren with same project");
      EXT_INF( EXT_FL_RESERVED        , "Reserved for EXT4 library");
   }

   fileSize = TXmku64( inode->SizeLo,    inode->SizeHi);
   alBlocks = TXmku64( inode->BlocksLo, (ext->sup->FeatureRoComp & EXT_FRO_HUGE_FILE) ? inode->BlocksHi : 0);

   if (ext->sup->FeatureRoComp & EXT_FRO_HUGE_FILE)
   {
      if (inode->Flags & EXT_FL_HUGE_FILE)      // alloc is 48-bit #blocks
      {
         alBlocks *= ext->BlockSize;
      }
      else                                      // alloc is 48-bit #512-byte
      {
         alBlocks *= 512;
      }
   }
   else                                         // alloc is 32-bit #512-byte
   {
      alBlocks *= 512;
   }
   fsBlocks = alBlocks / ext->BlockSize;    // allocated filesystem blocks

   text[0] = 0;
   dfstrSz64Byte( text, "File size (exact) : ", fileSize, " = ");
   dfsUllDot20(   text,                         fileSize,  " bytes\n");

   text[0] = 0;
   dfstrSz64Byte( text, "Allocated  size   : ", alBlocks, " = ");
   dfsUllDot20(   text,                         alBlocks,  " bytes\n");

   if (ext->sup->InodeSize > 128)               // extra fields are available
   {
      TxPrint("Inode extra size  : 0x%s%4.4hx%s     = %hu for a total of %hu bytes\n",
              (inode->extraSize <= (ext->sup->InodeSize - 128)) ? CBG : CBR,
               inode->extraSize, CNN, inode->extraSize,   128 + inode->extraSize);

      AtExtra = inode->AtExtra;
      ItExtra = inode->ItExtra;
      MtExtra = inode->MtExtra;
      TxPrint( "File Create  time : %s\n", dfsExtTime2str( inode->Ctime, inode->CtExtra, text));
   }
   TxPrint(    "Last Access  time : %s",   dfsExtTime2str( inode->Atime, AtExtra, text));
   TxPrint(   " Inode Last Change : %s\n", dfsExtTime2str( inode->Itime, ItExtra, text));
   TxPrint(    "Last Modify  time : %s",   dfsExtTime2str( inode->Mtime, MtExtra, text));
   if (inode->Dtime != 0)
   {
      TxPrint(" File deleted time : %s",   dfsExtTime2str( inode->Dtime, 0      , text));
   }
   TxPrint("\nAllocation using  : %s\n", (!(inode->Flags & EXT_FL_EXTENTS)) ? "BLOCK-INDIRECT-MAP" :
                                              (inode->Alloc.ex.hdr.ehDepth) ? "EXTENT-TREE INDEX"  :
                                                                              "EXTENT-TREE LEAF");
   if (dfsa->verbosity >= TXAO_VERBOSE)
   {
      TxDisplHexDump( (BYTE *) &inode->Alloc, 60);
   }

   if (fsBlocks != 0)                           // display allocated blocks
   {
      if ((rc = dfsExtInodeData2Space( inode, &fdata, &meta1, &meta2)) == NO_ERROR)
      {
         ULONG      sdo = SD_BLOCKS | SD_TOPLINE;
         ULN64      totals   = 0;
         ULN64      invalids = 0;

         if (dfsa->verbosity >= TXAO_VERBOSE)
         {
            if (meta1.chunks)
            {
               TxPrint( "First-level/leaf meta-blocks used for allocation:\n");
               dfsSspaceDisplay( sdo, meta1.chunks, meta1.space);
            }
            if (meta2.chunks)
            {
               TxPrint( "Second-level/index meta-blocks used for allocation:\n");
               dfsSspaceDisplay( sdo, meta2.chunks, meta2.space);
            }
         }

         rc = dfsExtSpaceCheckAlloc( &fdata, FALSE, "", &totals, &invalids, NULL);
         TxPrint( "\n Allocation Check : ");
         switch (rc)
         {
            case DFS_BAD_STRUCTURE:
               TxPrint("%sFAIL%s     ", CBR, CNN);
               dfsSz64("failing sectors : ",  invalids, "\n");
               break;

            case DFS_PSN_LIMIT:
               TxPrint("%sERROR%s, sector nrs are too large!\n", CBR, CNN);
               break;

            default:
               TxPrint("%sOK%s%s\n", CBG, CNN, (isDir) ? "" :
                "         Saveto/Recover/Undelete of filedata is possible");
               break;
         }
         if (fdata.chunks > 1)
         {
            sdo |= SD_SUMMARY;
         }
         if (fdata.chunks > 32)                 // multi-page is likely
         {
            if (dfsa->verbosity >= TXAO_VERBOSE)
            {
               sdo |= SD_BOTLINE;
            }
            else
            {
               sdo |= SD_LIMIT32;               // just first 32 + summary
            }
         }
         if (!TxaOptUnSet('X'))                 // unless -X- specified
         {
            sdo |= SD_NAVDOWN;                  // next is data sector
         }
         if (TxaOption('X'))
         {
            sdo |= SD_ALLINFO;                  // incl Index and Info
         }
         dfsSspaceDisplay( sdo, fdata.chunks, fdata.space);
      }
      else
      {
         TxPrint( "Data allocation   : type or structure contents are inconsistent!\n");
      }
   }
   else                                         // empty, or inlined/symlink
   {
      TxPrint( "Data allocation   : ");
      if (fileSize == 0)                        // empty
      {
         TxPrint( "None, and ZERO size (empty file)\n");
      }
      else if ((fileSize < 60) && ((inode->Mode & _S_IFLNK) == _S_IFLNK))
      {
         TXTM          linkname;

         TxCopy( linkname, (char *) &inode->Alloc, 60);
         TxPrint( "SYMLINK  : '%s%s%s'\n", CBG, linkname, CNN);
      }
      else                                      // Unknown, perhaps INLINED
      {
         if (fileSize < 60)                  // possibly INLINED
         {
            TxPrint( "None, with a small filesize, possibly INLINED data:\n");
         }
         else
         {
            TxPrint( "None, with a significant filesize! Allocation-area:\n");
         }
         TxDisplHexDump( (BYTE *) &inode->Alloc, 60);
      }
   }

   if (isDir) //- directory bit is NOT reliable for deleted directories?
   {
      ext->DirCache.chunks = 0;                 // signal: no allocation
      TxFreeMem( ext->DirCache.space);          // free previous DIR alloc
      ext->DirCache = fdata;                    // cache space structure
   }

   TxFreeMem( meta1.space);                     // free the meta allocation
   TxFreeMem( meta2.space);
   RETURN( rc);
}                                               // end 'dfsExtInodeRecord'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display and List one (or ALL, from cached Inode) directory blocks as a DIR
/*****************************************************************************/
static ULONG dfsExtShowDirectory
(
   BYTE               *sector,                  // IN    Dirblock sector data
   ULONG               lsn                      // IN    lsn of this sector
)                                               //       (for re-reading more)
{
   ULONG               rc = NO_ERROR;
   ULONG               totalDirSects = ext->SectorsPerBlock; // default 1 block
   ULONG               done = 0;                // sectors handled
   BYTE               *dirBlock;                // one block of DIR data
   ULONG               format = DFS_EDIR_STD;   // display options
   ULONG               linesDone = 0;           // total lines done

   ENTER();
   TRACES(("lsn: 0x%llx\n", lsn));

   if (!TxaOptUnSet('l'))                       // allow create Sectorlist ?
   {
      format |= DFS_EDIR_LISTADD;
   }
   rc = dfsExtAllocReadBlk( dfsExtLsn2Block( lsn), 1, &dirBlock);
   if (rc == NO_ERROR)
   {
      S_EXT_DXROOT     *ir = (S_EXT_DXROOT *) dirBlock; // block viewed as INDEX-root

      if ((ir->dotEntry.RecLen == 12) && (ir->ddtEntry.RecLen == (ext->BlockSize - 12)) &&
          (ir->fakeInode == 0) && (ir->infoLength == 8) && (ir->unusedFlags == 0))
      {
         TxPrint("\n INDEX ROOT with %hhu leaf-nodes (DIR-blocks) "
                    "plus %hhu interior NODE levels, hash: %hhu\n",
                 ir->usedEntries, ir->indirectLevels, ir->hashVersion);
      }

      //- get total size of the directory from inode cached allocation info, if matching
      if ((ext->DirCache.chunks != 0) && (ext->DirCache.space[0].start == lsn))
      {
         totalDirSects = dfsSspaceSectors( TRUE, ext->DirCache.chunks, ext->DirCache.space);
      }
      if (format & DFS_EDIR_THEADER)
      {
         TxPrint( "\n%s\n%s", extDirHeaderText, extDirHeaderLine);
      }
      TxPrint( "\n");
      if (format & DFS_EDIR_LISTADD)            // create a new list for DIR
      {
         dfsInitList(0, "-f -P", "-d");         // optimal for menu file-recovery
      }

      do
      {
         rc = dfsExtShowDirBlock( dirBlock, format, &linesDone);

         done += ext->SectorsPerBlock;          // number of sectors handled now

         if (done < totalDirSects)              // more sectors in the cache
         {
            rc = dfsSspaceReadFilePart( ext->DirCache.chunks, ext->DirCache.space,
                                        done, ext->SectorsPerBlock, dirBlock);
         }
      } while ((rc == NO_ERROR) && (done < totalDirSects) && !TxAbort());

      if (rc == DFS_NOT_FOUND)                  // DIR traversal ended on empty slot
      {
         rc = NO_ERROR;
      }
      if ((format & DFS_EDIR_BHEADER) && (linesDone > 12)) // reversed header at end
      {
         TxPrint( "%s\n%s\n", extDirHeaderLine, extDirHeaderText);
      }
      TxFreeMem( dirBlock);
   }
   RETURN( rc);
}                                               // end 'dfsExtShowDirectory'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display and List one single block of directory information
/*****************************************************************************/
static ULONG dfsExtShowDirBlock
(
   BYTE               *block,                   // IN    Block of DIR info
   ULONG               dirFlags,                // IN    presentation flags
   ULONG              *entries                  // INOUT total entries done
)
{
   ULONG               rc = NO_ERROR;
   S_EXT_DIRENTRY     *de;                      // single directory slot
   ULONG               offset = 0;              // byte offset into block
   ULONG               ei = *entries;           // entry index, continue
   int                 cl;                      // Inode LSN color
   int                 ci;                      // Inode index color
   int                 ac;                      // alternate color fg+bg
   int                 gc;                      // grey FG, alternate BG
   TX1K                text;                    // text buffer, MAX_PATH + line
   TXLN                tbuf;                    // temporary text buffer
   S_EXT_INODE        *inode;                   // Inode sector data
   ULN64               inoLsn;                  // Inode LSN
   USHORT              inoIdx;                  // Inode index
   ULONG               AtExtra = 0;             // Access time, extra bits
   ULONG               ItExtra = 0;             // Inode Change time, extra bits
   ULONG               MtExtra = 0;             // Modification time, extra bits
   ULONG               CtExtra = 0;             // File Create  time, extra bits

   ENTER();

   if ((inode = TxAlloc( 1, ext->sup->InodeSize)) != NULL)
   {
      while ((rc == NO_ERROR) && (offset < ext->BlockSize) && !TxAbort())
      {
         de = (S_EXT_DIRENTRY *) ((BYTE *) block + offset);

         if (de->Inode == 0)                    // unused entry, end of DIR
         {
            if (de->RecLen == 0)
            {
               //- should not happen if all RecLen values are correct
               TRACES(("Empty entry nr:%u signals end of DIR ???\n", ei));
               rc = DFS_NOT_FOUND;
               break;                           // end the loop
            }
         }
         else                                   // normal entry, display it
         {
            inoLsn  = dfsExtInode2LsnIndex( de->Inode, &inoIdx);
            if ((rc = dfsExtGetInodeRecord( inoLsn, inoIdx, inode)) == NO_ERROR)
            {
               if (ei % 2)
               {
                  ac = TXaBWnC;
                  gc = TXaBZnC;
                  ci = TXaBCnC;
               }
               else
               {
                  ac = TXaNWnZ;
                  gc = TXaBZnZ;
                  ci = TXaBCnZ;
               }
               if (ei == 1)                     // Set DOTDOT as down <Enter>
               {                                // in case there are no more
                  if (inoLsn)
                  {
                     nav.down        = inoLsn;  // Inode is target for <Enter>
                     nav.down_sninfo = inoIdx | DFSSNINFO;
                  }
               }
               if (ei == 2)                     // first real entry in the list
               {                                // after DOT and DOTDOT ones
                  cl = TXaBGnZ;
                  if (inoLsn)
                  {
                     nav.down        = inoLsn;  // Inode is target for <Enter>
                     nav.down_sninfo = inoIdx | DFSSNINFO;
                  }
               }
               else
               {
                  cl = ac;
               }
               sprintf(  text, "%s.%5.5u",   ansi[ac], ei);
               dfstrX10( text, "",          inoLsn,   ansi[cl], ansi[ac]);
               sprintf(  tbuf, "^%s%hx%s ",  ansi[ci], inoIdx,   ansi[ac]);
               strcat(   text, tbuf);

               if (ext->sup->InodeSize > 128)   // extra fields are available
               {
                  AtExtra = inode->AtExtra;
                  ItExtra = inode->ItExtra;
                  MtExtra = inode->MtExtra;
                  CtExtra = inode->CtExtra;
                  strcat(   text, dfsExtTime2str( inode->Ctime, CtExtra, tbuf));
               }
               else
               {
                  strcat(   text, dfsExtTime2str( inode->Itime, ItExtra, tbuf));
               }

               sprintf(  tbuf, " %s%s%s", ansi[ci], dfsExtFileType2String(de->FileType), ansi[ac]);
               strcat(   text, tbuf);

               if ((MtExtra != CtExtra) || (inode->Mtime != inode->Ctime))
               {
                  strcat( text, " ");
                  strcat( text, dfsExtTime2str( inode->Mtime, MtExtra, tbuf));
               }
               else
               {
                  strcat( text, "                    ");
               }

               if ((inode->SizeLo == 0) && (inode->SizeHi == 0))
               {
                  strcat( text, "                      ");
               }
               else
               {
                  dfstrUllDot20( text, " ", TXmku64( inode->SizeLo,    inode->SizeHi), " ");
               }
               sprintf( tbuf, "%8x%s%s", de->Inode, CGE, ansi[ac]);
               strcat(  text, tbuf);

               if ((inode->EaBlockHi != 0) || (inode->EaBlockLo != 0))
               {
                  //- to be refined, different for INLINE EA?
                  strcpy( tbuf, "*\n");
               }
               else
               {
                  strcpy( tbuf, "\n");
               }
               strcat(  text, tbuf);

               if ((AtExtra != CtExtra) || (inode->Mtime != inode->Ctime))
               {
                  sprintf( tbuf, "  %saccessed  %s%s ", ansi[gc], CNN, ansi[ac]);
                  strcat(  text, tbuf);
                  strcat(  text, dfsExtTime2str( inode->Atime, AtExtra, tbuf));
               }
               else
               {
                  sprintf( tbuf, "  %30.30s", "");
                  strcat(  text, tbuf);
               }
               strcat( text, (S_ISDIR( inode->Mode)) ? " D" : "  ");
               if (inode->Mode & 0777)
               {
                  sprintf( tbuf, "%3ho ", (USHORT) (inode->Mode & 0777));
                  strcat(  text, tbuf);
               }
               else
               {
                  strcat(  text, "    ");
               }
               strcat(  text, ansi[Ccol((CcY | CcI), Ccbg(ac))]); // Yellow
               TxCopy(  tbuf, de->Name, de->NameLen + 1); // null-terminated name
               strcat(  text, tbuf);
               sprintf( tbuf, "%s%s%s\n", ansi[Ccol( CcW, Ccbg(ac))], CGE, CNN);
               strcat(  text, tbuf);

               TxPrint( "%s", text);
            }
            else                                // Inode can not be read
            {
               //- to be refined, stripped down display
               TxCopy( tbuf, de->Name, de->NameLen + 1); // null-terminated name
               TxPrint("offset: %3u namelen:%3hu type:%d Inode:%8.8x name:'%s'\n",
                        offset, de->NameLen, de->FileType, de->Inode, tbuf);
            }
            ei++;
            if (dirFlags & DFS_EDIR_LISTADD)    // create a new list for DIR
            {
               dfsAddSI2List( inoLsn, inoIdx);
            }
         }
         offset += de->RecLen;                  // to next DIR entry, or end
      }
      TxFreeMem( inode);
      *entries = ei;                            // update #lines done
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN( rc);
}                                               // end 'dfsExtShowDirBlock'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Convert Linux (Ext?) Filetype value to string (from directory entries)
/*****************************************************************************/
static char *dfsExtFileType2String              // RET   filetype string[4]
(
   BYTE                fType                    // IN    file type
)
{
   switch (fType)
   {
      case EXT_FT_REG_FILE : return( "file"); break;
      case EXT_FT_DIR      : return( " dir"); break;
      case EXT_FT_CHRDEV   : return( "chrd"); break;
      case EXT_FT_BLKDEV   : return( "blkd"); break;
      case EXT_FT_FIFO     : return( "fifo"); break;
      case EXT_FT_SOCK     : return( "sock"); break;
      case EXT_FT_SYMLINK  : return( "SYML"); break;
   }
   return( "ZERO");
}                                               // end 'dfsExtFileType2String'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Make filesystem usage map dump on TxPrint output
/*****************************************************************************/
static ULONG dfsExtAllocMap
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *options,                 // IN    display options
   void               *data                     // OUT   dummy
)
{
   ULN64               i;                       // index in data-area
   ULONG               b;                       // byte-index for one mapchar
   ULN64               cClus = 0;               // Clusters in current char
   ULN64               lClus = 0;               // Clusters in current line
   ULN64               mClus = 0;               // Clusters in current map
   ULN64               uClus = 1;               // Last cluster that is in use
   ULONG               l;                       // line number, 0 based
   ULONG               a;                       // index in display-array
   ULN64               perc;                    // percentage
   TX1K                ascii;                   // display-array
   ULONG               spe;                     // sectors per entry
   ULONG               acl;                     // ascii alloc chars per line
   ULONG               cpc;                     // clusters per display-char
   ULONG               spc;                     // sectors per display-char
   ULN64               size;                    // nr of clusters to map
   BOOL                bgLine  = FALSE;         // include linenumber as BG#
   BOOL                verbose = (*options != '@');
   ULONG               bsSmart;
   ULN64               unallocSmart = 0;        // Size of THIS unallocated area
   BOOL                oneLineProgress = FALSE; // alloc in one line, output per character

   ENTER();

   dfsa->FsUnallocSmart = 0;                    // will be recalculated now
   bsSmart = dfsGetBufferSize( DFSOPTIMALBUF, DFSMAXBUFSIZE); // for SMART prediction

   spe   = ext->SectorsPerBlock;
   size  = ext->Block;                          // Block 0 .. last-valid

   acl   = dfsAllocCharsPerLine( 'c');
   cpc   = dfsAllocItemsPerChar( 'l', options, size, acl);
   if (cpc >= (size / acl))                     // will be a single line
   {
      oneLineProgress = TRUE;                   // one char at a time, no ascii string used
   }
   spc   = cpc * spe;
   dfsProgressInit( 0, size * spe, 0, "Get ALLOCATION info, at Sector:", dfstStoreDpid( DFSTORE), DFSP_STAT, 0);

   if ((ext->sup != NULL) && ((cpc * acl) == ext->sup->BlocksPerGroup))  // we have exactly 1 BG / line
   {
      bgLine = TRUE;
   }

   if (verbose)
   {
      dfsSz64("Size for one line : ", (ULN64) acl * spc, "  ");
      TxPrint("with %3u characters,", acl);
      dfsSiz8(" size : ", spc, "\n");
      TxPrint(" %16.16s = Allocation grade, empty to full.     ", mapchar);
      dfsSiz8(" SmartBuf size : ", bsSmart, "\n");
   }
   TxPrint(   "          %s%*.*s%s\n", CBC, acl, acl, BLIN, CNN);
   dfsX10( CNZ, dfsExtBlock2Lsn(0), CNZ, "");   // display address for first block
   TxPrint("%s%s", CBC, CNC);

   for (i=0, a=0, l=0; (i < size) && (!TxAbort());)
   {
      if (a == 0)
      {
         memset(ascii, 0, TXMAX1K);
         lClus = 0;
      }
      for (cClus=0, b=0; (b < cpc) && (i < size); b++, i++)
      {
         if (dfsExtBlkBitmapCache(i, NULL))     // block in use ?
         {
            if (unallocSmart != 0)              // first in-use after unallocated?
            {
               if (unallocSmart >= bsSmart)                  //- if at least ONE smart buffer
               {
                  unallocSmart -= (bsSmart / 2);             //- compensate for alignment issues
                  unallocSmart -= (unallocSmart % bsSmart);  //- clip to whole smart buffer size

                  dfsa->FsUnallocSmart += unallocSmart;      //- add this area to total
               }
               unallocSmart = 0;                // reset for next area to come
            }
            cClus++;
            uClus = i;                          // last in-use one sofar
         }
         else                                   // unallocated, Smart predict
         {
            unallocSmart += spe;                // add to size THIS area
         }
      }
      lClus += cClus;

      dfsProgressShow( (i * spe) + spc, 0, 0, NULL);

      TRACES(("lClus 2nd Loop i: %llu, b:%u = %llu\n", i, b, lClus));
      if (oneLineProgress == TRUE)
      {
         TxPrint( "%c", (char) ((cClus == 0) ? ' ' : mapchar[cClus * 8 / cpc])); // slow, acts as progress bar
      }
      else
      {
         ascii[a] = (char) ((cClus == 0) ? ' ' : mapchar[cClus * 8 / cpc]);
      }
      a++;
      if ((i && ((i%(acl*cpc)) == 0)) || (i >= size))
      {
         if (oneLineProgress == FALSE)
         {
            TxPrint( "%s", ascii);              // display accumulated chars (fast)
         }
         perc  = (ULN64)(100*lClus / (a*cpc));
         if (a == acl)
         {
            TxPrint("%s%s% 3u%%", CBC, CNN, perc);
         }
         else
         {
            TxPrint("%s%.*s%s% 3u%%", CBC, (int) (acl-a-1), BLIN, CNN, perc);
         }
         if (bgLine)
         {
            TxPrint( " BG#%4u", l);            // add blockgroup number
         }
         TxPrint( "\n");

         if (i < size)
         {
            dfsX10( CNZ, dfsExtBlock2Lsn(i), CNZ, "");
            TxPrint("%s%s", CBC, CNC);
            a = 0;                              // only reset when more todo
         }
         mClus += lClus;
         l++;
      }
   }
   if (unallocSmart != 0)                           //- unalloc area pending?
   {
      if (unallocSmart >= bsSmart)                  //- if at least ONE smart buffer
      {
         unallocSmart -= (bsSmart / 2);             //- compensate for alignment issues
         unallocSmart -= (unallocSmart % bsSmart);  //- clip to whole smart buffer size

         dfsa->FsUnallocSmart += unallocSmart;      //- add this area to total
      }
   }
   TxPrint("          %s%.*s%s\n", CBC, (USHORT) a, BLIN, CNN);

   if (TxAbort())
   {
      dfsa->FsUnallocSmart = 0;                 // signal info NOT available!
      TxPrint( "\nAllocation check and display aborted! Used/Free info not available.\n");
   }
   else
   {
      if (size != 0)                            // avoid devide by 0
      {
         ULN64            used = mClus * spe;

         dfsa->FsUnallocated   = size * spe - used; // free space (optimize hint)

         dfsSz64("Unallocated sects : ", dfsa->FsUnallocated, " ");
         dfsSz64("SmartUse ", dfsa->FsUnallocSmart, " ");
         TxPrint("=% 5.1lf%%\n", (double) (100.0 * ((double) dfsa->FsUnallocSmart) / (double) (size * spe)));

         dfsSz64("Allocated sectors : ", used, " ");
         dfsSz64("of total ", size * spe, " ");
         TxPrint("=% 5.1lf%%\n", (double) (100.0 * ((double) mClus) / (double) size));
      }
      dfsa->FsTruncPoint = dfsExtBlock2Lsn( uClus +1); // first free (truncate)
      dfsa->FsLastInUse  = dfsa->FsTruncPoint -1;

      #if defined (EXT_RESIZING)
         dfsSz64( "Minimum vol. size : ", dfsa->FsTruncPoint, " (for  Resizing)\n");
      #endif
   }
   dfsProgressTerm();
   RETURN (NO_ERROR);
}                                               // end 'dfsExtAllocMap'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// DFS EXT write-file to disk (SaveTo to file)
/*****************************************************************************/
static ULONG dfsExtFileSaveAs
(
   ULN64               inoLsn,                  // IN    Inode sectornumber
   ULN64               inoIdx,                  // IN    Inode index in sector
   char               *path,                    // IN    destination path
   void               *recp                     // INOUT recovery parameters
)
{
   ULONG               rc = NO_ERROR;
   S_RECOVER_PARAM    *param = (S_RECOVER_PARAM *) recp;
   S_EXT_INODE        *inode;
   DFSISPACE           isp;                     // allocation SPACE info
   TXLN                uniq;                    // unique filename prefix
   TXLN                fname;                   // filename component
   TXLN                safname;                 // resulting SaveAs name
   ULN64               fileSize;                // file size in bytes

   ENTER();
   TRARGS(("Inode LSN: 0x0%llX - index %llu, to path: '%s'\n", inoLsn, inoIdx, path));

   if ((inode = TxAlloc( 1, ext->sup->InodeSize)) != NULL)
   {
      if ((rc = dfsExtGetInodeRecord( inoLsn, inoIdx & ~DFSSNINFO, inode)) == NO_ERROR)
      {
         fileSize = TXmku64( inode->SizeLo,    inode->SizeHi);

         memset( &isp, 0, sizeof( DFSISPACE));
         if (!S_ISDIR( inode->Mode))
         {
            rc = dfsExtInodeData2Space( inode, &isp, NULL, NULL);
            if (rc == NO_ERROR)
            {
               if (isp.space == NULL)
               {
                  if (fileSize < 60)            // possibly INLINED or symlink
                  {
                     isp.chunks = 0;            // make available to SpaceFileSaveAs
                     isp.space  = TxAlloc( 1, fileSize);
                  }
               }
            }
         }
         if ((param->newname == NULL) || (*param->newname == 0)) // no newname present
         {
            dfsExtResolveInodeName( inoLsn, inoIdx, inode, NULL, fname);
         }
         else                                   // rename specified
         {
            strcpy( fname, param->newname);
         }
         if (param->unique)                     // force unique naming
         {                                      // on PATH and FILE components
            sprintf(uniq, "%12.12llX_%llX_%s", inoLsn, inoIdx, fname);
            strcpy( fname, uniq);               // and move back to fname
         }
         if (param->recFname)
         {
            strcpy( param->recFname, fname);    // return full base-filename
         }

         isp.byteSize = fileSize;
         rc = dfsSspaceFileSaveAs( &isp, (S_ISDIR( inode->Mode)),   //- directory ?
                                         (inode->Links == 0),       //- deleted ?
                                          param->noAllocCheck, param->name83, path, fname, safname);
         if ((rc == NO_ERROR)       ||          // set original timestamps
             (rc == DFS_CMD_WARNING) )          // even when allocation errors present
         {
            ULONG   CreateTime = (ext->sup->InodeSize > 128) ? inode->Ctime : inode->Itime;
            TxSetFileTime( safname, (time_t *) &CreateTime,
                                    (time_t *) &inode->Atime,
                                    (time_t *) &inode->Mtime);
            if (param->recFname)
            {
               strcpy( param->recFname, safname);
            }
         }
         TxFreeMem( isp.space);                 // free SPACE, if any
         TxFreeMem( inode);
      }
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN( rc);
}                                               // end 'dfsExtFileSaveAs'
/*---------------------------------------------------------------------------*/
