//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// EXFAT dump & display Analysis functions
//
// Author: J. van Wijk
//
// JvW  19-06-2014   Initial version, derived from DFSAFAT
//

#include <txlib.h>                              // TX library interface

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfswin.h>                             // DFS help ids
#include <dfswipe.h>                            // wipe functions
#include <dfsutil.h>                            // DFS utility functions
#include <dfsspace.h>                           // DFS  file-space interface
#include <dfstable.h>                           // SLT utility functions

#include <dfsupart.h>                           // FDISK partition functions
#include <dfsafdsk.h>                           // FDISK display & analysis
#include <dfsufdsk.h>                           // FDISK utility functions
#include <dfscfdsk.h>                           // FDISK util, FAT sectors

#include <dfsaexft.h>                           // EXFT display & analysis
#include <dfsuexft.h>                           // EXFT utility functions
#include <dfslexft.h>                           // EXFT SLT functions
#include <dfsosapi.h>                           // OS specific stuff


char dfsExftSectorTypes[] =
{
   ST_FAT_1,                                    //  I  1st FAT area
   ST_FAT_2,                                    //  I  2nd FAT area
   ST_EXFAT,                                    //  I  EXFAT 1st sector
   ST_ROOTD,                                    //  I  Root directory area
   ST_DIREC,                                    //     directory area
   0                                            //     string terminating ZERO
};


static DFSAEXFT    exft_anchor =
{
   0,                                           // LSN of 1st Fat
   0,                                           // LSN of 2nd Fat
   0,                                           // LSN of cluster-heap (data)
   0,                                           // LSN of Root directory
   0,                                           // nr of sectors in Root dir
   0,                                           // nr of FATs (usually 2)
   0,                                           // Offset to first FAT
   0,                                           // Nr of sectors per FAT
   0,                                           // Number of sectors
   0,                                           // #clusters, from #sectors
   0,                                           // #clusters, from #fatSectors
   0,                                           // Nr of clusters still free
   0,                                           // valid cluster size
   0,                                           // Maximum size for expand, resize
   0,                                           // Maximum size for current FAT area
   FALSE,                                       // EXFAT values faked, no bootsec
   TRUE,                                        // allocation cached cluster
   LSN_NULL,                                    // cached cluster number
   0,                                           // FAT in-use, LSN
   {0, NULL, 0, FALSE, 'A'},                    // 1st cache, A
   {0, NULL, 0, FALSE, 'B'},                    // 2nd cache, B
   {0, NULL, 0, 0, 0, FALSE, NULL},             // Bitmap cache structure
   0, 0,                                        // BM cached cluster and mapsize
   0, 0, NULL                                   // DP cache sizes and array
};

       DFSAEXFT   *exft = &exft_anchor;

       char sg_exfat[SG_EXFAT] = {SV_EXFAT};

static  char       *exft_txt[] =
{
   "",
   "Active filesystem : E-FAT, specific commands are:",
   "",
   " \\[path-spec]     = find and show ROOT or file/directory relative to root",
// " ALLDIRS          = Show disk allocation for all directories on the volume",
   " BOOTSYNC [a][-s] = Read, calculate CRC and synchronise primary/spare boot areas",
   " BSCLEAR          = Reset bad-sector/cluster administration to ZERO bad sectors",
   " DELFIND   [name] = Find deleted files with DIR-entry containing (partial) name",
   " DIRTY      [d|c] = Display/set DIRTY, CLEAN  status for the E-FAT filesystem",
   " FATSELECT  [fat] = Select fat to be used (cached), fat = 1 or 2; default is 1",
   " FATSHOW   [*|nr] = Display contents of the FAT from memory, [nr]= # of entries",
   " FATSIM   img [f] = Save FAT table contents for FAT 'f' (1 or 2) to an imagefile",
   " FATWRIM  img [f] = Restore an imagefile with the FAT table to FAT 'f' (1 or 2)",
   " FILEFIND  [name] = Find normal files with DIR-entry containing (partial) name",
   " FINDROOT         = Find and list possible E-FAT root directory clusters/sectors",
   " FIXBOOT [1|*|-s] = Fix E-FAT bootsector area (0..11) from spare (12..23)",
// " FORMAT    [opts] = Format current object with a FAT12/16 or FAT32 filesystem",
// " SETROOT  [s|.nn] = Set E-FAT root directory LSN to sector [s] or listvalue [.nn]",
// " SPACE    [clust] = Show allocation for specified cluster or current LSN 'this'",
// " SUBFIND   [opts] = Find subdirectories (.. entry) from start or current lsn",
   "",
   NULL
};

#if defined (NEVER)
static  char       *cmd_format[] =
{
   "",
   "Format current object with the E-FAT filesystem",
   "",
   "usage:  FORMAT  [options]",
   "",
   "   -a:offset = Sector offset for first FAT area from bootsector",
   "   -c:spc    = Number of sector per cluster, power of 2, 1 .. 65536",
   "   -l        = Long format, clear all data-area sectors to ZERO",
   "",
   "   -s:vsn    = volume serial number to be used  (use 0x for hex)",
   "   -v:label  = ASCII label string for filesystem identification",
   "   -!        = force use of option dialog to specify parameters",
   "",
   NULL
};
#endif


// Close FAT filesystem for analysis and free any resources
static ULONG dfsExftClose
(
   ULONG               di,                      // IN    dummy
   ULONG               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
);

// Close FAT filesystem for Area analysis and free any resources
static ULONG dfsExftAreaClose
(
   ULONG               di,                      // IN    dummy
   ULONG               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
);

// Search and list possible Root directory clusters for an E-FAT partition
static ULONG dfsExftFindExfatRoots              // RET   nr of candidates found
(
   void
);

// Extract BitMap allocation S_SPACE structure and initialize the BitMap cache
static ULONG dfsExftBitMapInit                  // RET   rc = 0 if type match
(
   ULONG               firstCluster,            // IN    first bitmap cluster
   ULONG               bmByteSize               // IN    size BM file in bytes
);

// Interpret and execute specific E-FAT command;
static ULONG dfsExftCommand
(
   ULONG               di,                      // IN    dummy
   ULONG               d2,                      // IN    dummy
   char               *none,                    // IN    dummy
   void               *data                     // INOUT dummy
);

// FAT filesystem, identify specified sector
static ULONG dfsExftIdent
(
   ULONG               di,                      // IN    LSN for sector
   ULONG               d2,                      // IN    dummy
   char               *dc,                      // OUT   sector type
   void               *data                     // IN    sector contents
);

// FAT filesystem, supply sector-type description string
static ULONG dfsExftStype
(
   ULONG               di,                      // IN    sector type
   ULONG               d2,                      // IN    dummy
   char               *dc,                      // OUT   type description
   void               *data                     // INOUT dummy
);

// FAT filesystem, display sector-contents based on type
static ULONG dfsExftDispl
(
   ULONG               psn,                     // IN    base psn for sector
   ULONG               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // IN    sector contents
);

// FAT filesystem, display special LSN info
static ULONG dfsExftDisplayLsnInfo
(
   ULONG               lsn,                     // IN    possible special lsn
   ULONG               info,                    // IN    possible DIR entry nr
   char               *dc,                      // IN    dummy
   void               *data                     // IN    dummy
);

// Display FAT Root- or sub-directory sector(s)
static ULONG dfsExftDirDisplay
(
   ULONG               psn,                     // IN    base psn for sector
   BYTE                type                     // IN    type of sector
);

// Find LSN* for specified path, starting at Root directory LSN
static ULONG dfsExftFindPath
(
   ULONG               dummy,
   ULONG               d2,                      // IN    dummy
   char               *path,                    // IN    path specification
   void               *vp                       // OUT   found dir/file Lsn+entry
);

// Find specific name in a (sub) directory SPACE structure, return file+stream
static BOOL dfsExftDirSpace2Entry                // RET   name found
(
   char               *name,                    // IN    subdir or filename
   ULONG               chunks,                  // IN    nr of space entries
   S_SPACE            *space,                   // IN    space allocation
   LSN                *lsn,                     // OUT   Dirsector found
   ULONG              *info,                    // OUT   Dir entry-nr found
   S_EXFTDIR          *efile,                   // OUT   Directory entry, FILE
   S_EXFTDIR          *estream                  // OUT   Directory entry, STREAM
);

// DFS FAT write-file to disk (SaveTo fnode to file)
static ULONG dfsExftFileSaveAs
(
   ULONG               dse,                     // IN    dirsec/entry LSN
   ULONG               info,                    // IN    info, dir entry
   char               *path,                    // IN    destination path
   void               *ren                      // IN    new name (like 8.3)
);

// Make E-FAT usage map based on the bitmap, or the first FAT area
static ULONG dfsExftAllocMap
(
   ULONG               di,                      // IN    dummy
   ULONG               d2,                      // IN    dummy
   char               *options,                 // IN    display options
   void               *data                     // OUT   dummy
);

// Make E-FAT usage map based on the E-FAT bitmap
static ULONG dfsExftAllocBm
(
   char               *options                  // IN    display options
);

// Make E-FAT usage map based on the FAT area
static ULONG dfsExftAllocFat
(
   char               *options                  // IN    display options
);

/*****************************************************************************/
// Initialize FAT filesystem analysis
/*****************************************************************************/
ULONG dfsExftInit
(
   char               *fs                       // forced filesystem type
)
{
   ULONG               rc = NO_ERROR;
   TXTM                label;                   // FAT Root label
   TXTM                tbuf;
   TXLN                text;
   BOOL                Check2ndFAT = !TxaOptUnSet('2');
   BYTE                FsysValue = 0;              // display value

   ENTER();

   dfsa->FsCommand          = dfsExftCommand;
   dfsa->FsClose            = dfsExftClose;
   dfsa->FsIdentifySector   = dfsExftIdent;
   dfsa->FsShowType         = dfsExftStype;
   dfsa->FsDisplaySector    = dfsExftDispl;
   dfsa->FsFileInformation  = dfsExftFileInfo;
   dfsa->FsGetAllocSpace    = dfsExftGetAllocSpace;
   dfsa->FsWriteMetaSpace   = NULL;
   dfsa->FsFindPath         = dfsExftFindPath;
   dfsa->FsUpdateFileName   = NULL;
   dfsa->FsMakeBrowseList   = dfsExftMakeBrowseList;
   dfsa->FsLsnAllocated     = dfsExftAllocated;
   dfsa->FsLsnSetAlloc      = dfsExftSetAlloc;
   dfsa->FsSltBuild         = dfsExftSltBuild;
   dfsa->FsNpBuild          = NULL;
   dfsa->FsDisplayError     = dfsExftDispError;
   dfsa->FsDisplayLsnInfo   = dfsExftDisplayLsnInfo;
   dfsa->FsDirIterator      = dfsExftIterator;
   dfsa->FsDirFileSaveAs    = dfsExftFileSaveAs;
   dfsa->FsTruncateSize     = NULL;
   dfsa->FsAllocDisplay     = dfsExftAllocMap;
   dfsa->FsCl2Lsn           = dfsExftCl2Lsn;
   dfsa->FsLsn2Cl           = dfsExftLsn2Cl;
   dfsa->FsCmdHelp          = exft_txt;         // FS specific cmdhelp text

   dfsa->FsModeId           = DFS_FS_E_FAT;

   dfsa->Fsi                = exft;
   dfsa->FsSectorTypes      = dfsExftSectorTypes;

   TRACES(("sizeof S_BOOTR        = % 3lu\n", sizeof(S_BOOTR)));
   TRACES(("sizeof S_EXFAT        = % 3lu\n", sizeof(S_EXFAT)));
   TRACES(("sizeof DFSEXINFO      = % 3lu\n", sizeof(DFSEXINFO)));
   TRACES(("sizeof DFSF2INFO      = % 3lu\n", sizeof(DFSF2INFO)));

   if ((dfsRead( LSN_BOOTR, EXFAT_BOOTSIZE, brec)) == NO_ERROR)
   {
      ULONG            calculatedChecksum  = 0;
      ULONG            storedChecksum      = 0;
      BOOL             bootAreaChecksumsOK = FALSE;

      calculatedChecksum = dfsExftBootCheckSum(    brec, EXFAT_BOOTSIZE * dfsGetSectorSize());
      TxPrint("Calculated BR Crc : 0x%8.8lX  ",    calculatedChecksum);
      if ((storedChecksum = dfsExftStoredChecksum( brec, EXFAT_BOOTSIZE * dfsGetSectorSize())) != 0)
      {
         if (storedChecksum == calculatedChecksum)
         {
            TxPrint("Match to every CS value in reference sector\n");
            bootAreaChecksumsOK = TRUE;
         }
         else
         {
            TxPrint("NO match to values in reference sector: %8.8lX\n", storedChecksum);
         }
      }
      else
      {
         TxPrint("But reference sector has inconsistent contents!\n");
      }
      dfsa->boot = (S_BOOTR *) brec;

      if ((bootAreaChecksumsOK       == FALSE   ) || // checksum inconsistency
          (dfsa->boot->Signature     != SV_BOOTR) || // invalid bootrecord
          (dfsa->boot->ex.BpsShift   <  9       ) ||
          (dfsa->boot->ex.BpsShift   > 12       ) ||
          (dfsa->boot->ex.SpcShift   > 25       ) ||
          (dfsa->boot->ex.NrOfFats   >  2       ) ||
          (dfsa->boot->ex.FatLength  == 0       )  )
      {
         TxPrint( "Bootsector seems %sinvalid%s, trying (E-FAT) spare "
                  "sectors 12..23\n\n", CBR, CNN);
         if ((rc = dfsRead(EXFAT_BACKUPBOOT, EXFAT_BOOTSIZE, brec)) == NO_ERROR)
         {
            nav.down   = EXFAT_BACKUPBOOT;
            switch (dfsIdentifySector(0, 0, brec))
            {
               case ST_BOOTX:
                  TxPrint( "Non-standard boot sector detected ...\n");
               case ST_BOOTR:
                  if (SINF->p == NULL)          // No partition info yet
                  {
                     dfstSetSectorSize( DFSTORE, (USHORT) (1 << dfsa->boot->ex.BpsShift));
                     dfstSetClusterSize(DFSTORE, (USHORT) (1 << dfsa->boot->ex.SpcShift));
                  }

                  bootAreaChecksumsOK = FALSE;
                  calculatedChecksum  = dfsExftBootCheckSum(   brec, EXFAT_BOOTSIZE * dfsGetSectorSize());
                  TxPrint("Calculated BR Crc : 0x%8.8lX  ",    calculatedChecksum);
                  if ((storedChecksum = dfsExftStoredChecksum( brec, EXFAT_BOOTSIZE * dfsGetSectorSize())) != 0)
                  {
                     if (storedChecksum == calculatedChecksum)
                     {
                        TxPrint("Match to every CS value in reference sector\n");
                        TxPrint("You may use use 'bootsync -s' to replace the primary bootarea by the spare ...\n");
                        bootAreaChecksumsOK = TRUE;
                     }
                     else
                     {
                        TxPrint("NO match to values in reference sector: %8.8lX\n", storedChecksum);
                     }
                  }
                  else
                  {
                     TxPrint("But reference sector has inconsistent contents!\n");
                  }
                  break;

               default:
                  TxPrint("Sector %u is not a valid filesystem BOOT record either!\n", EXFAT_BACKUPBOOT);
                  dfsRead( LSN_BOOTR, EXFAT_BOOTSIZE, brec); // re-read real bootsector(s)
                  break;
            }
         }
         else
         {
            TxPrint( "Reading E-FAT spare-sectors failed!\n");
            dfsRead( LSN_BOOTR, EXFAT_BOOTSIZE, brec); // re-read real bootsector(s)
         }
      }

      if ((bootAreaChecksumsOK       == TRUE    ) && // checksums consistent
          (dfsa->boot->Signature     == SV_BOOTR) && // could be valid boot record
         ((dfsa->boot->ex.BpsShift   >= 9       ) &&
          (dfsa->boot->ex.BpsShift   <= 12     )) &&
          (dfsa->boot->ex.SpcShift   <= 25      ) &&
          (dfsa->boot->ex.NrOfFats   <= 2       ) &&
          (dfsa->boot->ex.FatLength  != 0       )  )
      {
         exft->Sect      = dfsa->boot->ex.VolumeLength; // to be refined (LLONG?)
         exft->ClustSize = 1 << dfsa->boot->ex.SpcShift;
         exft->NrOfFats  =      dfsa->boot->ex.NrOfFats;
         exft->Fat1      =      dfsa->boot->ex.FatOffset;

         TRACES(("FAT ClustSize: %4.4hu, Offset: %8.8lx\n, exft->ClustSize, exft->Fat1"));

         exft->FatOffset     = dfsa->boot->ex.FatOffset;
         exft->FatSectors    = dfsa->boot->ex.FatLength;
         if (exft->NrOfFats > 1)
         {
            exft->Fat2  = exft->Fat1 + dfsa->boot->ex.FatLength;
         }
         else
         {
            exft->Fat2  = exft->Fat1;
         }
         exft->Roots    = exft->ClustSize;      // minimum size
         exft->ClHeap   = dfsa->boot->ex.ClHeapOffset;
         exft->RawClust = dfsa->boot->ex.ClusterCount;

         exft->Root     = dfsExftClust2LSN( dfsa->boot->ex.RootCluster);

         exft->MapClust = exft->FatSectors * (dfsGetSectorSize() * 8 / 32); // mapped clusters

         if (dfsExftDirtyStatusBootrec( FALSE, FALSE) == FALSE)
         {
            dfsa->FsDirtyStatus = DFSTAT_CLEAN;
         }
         else
         {
            dfsa->FsDirtyStatus = DFSTAT_DIRTY;
         }
      }
      else                                      // set sane values
      {
         //- to be refined, may use some values from boot-areas?
         //- and find root directory automatically (for bitmap etc)

         BYTE             ClustSize;
         ULONG            FatSectors;
         TXTM             Signature;            // FAT signature buffer

         Signature[0]   = (char) 0xf8;          // assume harddisk :-)

         TxPrint( "Bootsector seems invalid, using most likely values ...\n");
         if (SINF->p != NULL)                   // partition info avaliable
         {
            TRACES(("PartitionType: %2.2hx\n", SINF->p->partent.PartitionType));

            exft->Sect  = SINF->p->sectors;
         }
         else                                   // use current store size
         {
            exft->Sect  = dfsGetLogicalSize();
         }
         TxPrint( "Searching 1st FAT area ... ");

         TRACES(("dfsa->boot: %8.8lx\n", dfsa->boot));

         memcpy( &(Signature[1]), sg_exfat, SG_EXFAT);
         if ((exft->Fat1 = (dfsLocateSignatureSector(
                              1, 0x1000, Signature, SG_EXFAT +1, 0))) == 0)
         {
            Check2ndFAT = FALSE;
            exft->Fat1  = 0x800;
            TxPrint( "not found, using default 0x800\n"
                     "FIXBOOT will NOT be reliable!\n");
         }
         else
         {
            TxPrint( "found at 0x%4.4lx (LSN % lu)\n", exft->Fat1, exft->Fat1);
         }

         //- to be refined, could do a signature search for UPCASE file (cl 3)
         //- and Root directory (cl 4) to determine the clustersize.
         //- sanity check, must be power of 2 etc



         TxPrint( "Assuming 4 KiB cluster size ...\n");
         ClustSize    = 8;                      // assume 8 sector clusters
         exft->Roots  = ClustSize;              // minimum is 1 cluster ...
         FatSectors   = 131000;                 // 64 GB part, 4KB clusters

         dfsa->boot->ex.RootCluster = 4;
         TxPrint( "\nRoot directory now set at default cluster 4, if this was\n"
                  "a working and formatted E-FAT partition, you might need to\n"
                  "use 'findroot', 'setroot' and 'fixboot' to get the right\n"
                  "reference to the root directory (RECOVERY)\n\n");

         exft->RawClust   = exft->Sect / ClustSize;
         exft->ClustSize  = ClustSize;

         exft->Fat2  = exft->Fat1;              // single FAT

         exft->Root       = exft->Fat2 + FatSectors;
         exft->ClHeap     = exft->Root;
         exft->FatSectors = FatSectors;
         exft->MapClust   = exft->RawClust;
         nav.down         = exft->Root;         // next sector to visit

         TxPrint( "\nRestricted functionality, CHECK and SLT functions disabled!\n"
                  "\nWhen it has been created new, you only need to FORMAT it!\n"
                  "(and ignore most other warnings given here :-)\n\n");
         exft->ExftFaked = TRUE;                // FAT values faked, no bootsec
      }
      dfstSetClusterSize(DFSTORE, exft->ClustSize); // make sure it is set!
      dfsa->FsEntry = exft->Root;               // entry-point in filesystem

      exft->TruncCluster  = exft->RawClust;     // First free cluster, initial
                                                // until calculated in ALLOC

      sprintf( text, "Size of FAT entry : 32  Bits");
      if (dfsExftRootLabel(dfsa->boot, LSN_BOOTR, label))
      {
         sprintf( tbuf, "%s%s%s", CBC, label, CNN);
      }
      else
      {
         sprintf( tbuf, "--none--");
      }
      TxPrint( "%s    E-FAT label  Root : %s\n", text, tbuf);

      exft->MappedSize = (exft->MapClust - 2) * exft->ClustSize + exft->ClHeap;

      if (!TxaOption('q'))
      {
         sprintf(   text, "Size, in clusters : 0x%8.8lX  ", exft->RawClust);
         dfstrSiz8( text, "Number of sectors : ",           exft->Sect, "");
         TxPrint( "%s\n", text);

         sprintf(   text, "Cl. mapped by FAT : 0x%8.8lX  ", exft->MapClust);
         TxPrint( "%s\n", text);

         strcpy(    text, "");
         dfstrLsx(  text, "1st FAT start LSN : ", exft->Fat1,   CBC, "  ");
         if (exft->Fat2 != exft->Fat1)
         {
            dfstrLsx(  text, "2nd FAT start LSN : ", exft->Fat2,   CNC, "");
         }
         TxPrint( "%s\n", text);

         strcpy(    text, "");
         dfstrLsx(  text, "RootDirectory LSN : ", exft->Root,   CBY, "  ");

         strcpy(   text,  "");
         dfstrLsx( text,  "1st DataClust LSN : ", exft->ClHeap, CNG, "");
         TxPrint( "%s\n", text);
      }

      TRACES(("Allocate memory for %u fat-entries\n", (USHORT) DFSEXFT_CACHE));
      if (((exft->CacheA.Value = TxAlloc(DFSEXFT_CACHE, sizeof(ULONG))) != NULL) &&
          ((exft->CacheB.Value = TxAlloc(DFSEXFT_CACHE, sizeof(ULONG))) != NULL)  )
      {
         exft->FatInUse = exft->Fat1;
         if ((rc = dfsExftGetFAT( 0, &exft->CacheA)) == NO_ERROR)
         {
            ULONG         crc1 = 0;
            ULONG         crc2 = 0;

            dfsa->FsSltRecSize = exft->RawClust +1000; // max records in SLT

            //- Report the FAT dirty status here, from Bootrec
            TxPrint("FAT DIRTY status  : 0x%2.2hx = ", FsysValue);
            if (dfsExftDirtyStatusBootrec( FALSE, FALSE) == FALSE)
            {
               TxPrint( "CLEAN\n");
            }
            else
            {
               TxPrint( "DIRTY, surface-SCAN/CHKDSK required (MS)\n");
            }

            if (((exft->Fat2 - exft->Fat1) > 0) && (Check2ndFAT == TRUE))
            {
               #if defined (USEWINDOWING)
                  if (txwIsWindow( TXHWND_DESKTOP))
                  {
                     sprintf( text, " Comparing first and second FAT area ... ");
                     txwSendMsg( dfsa->sbwindow, TXWM_STATUS, (ULONG) text, cSchemeColor);
                  }
               #endif

               crc2 = dfsExftCalculateFatCRC( exft->Fat2); // FAT2 before 1, to avoid
               crc1 = dfsExftCalculateFatCRC( exft->Fat1); // it staying the active!
               TxPrint( "FAT1 CRC-32 value : 0x%8.8lx  "
                        "FAT2 CRC-32 value : 0x%s%8.8lx%s = %s\n", crc1,
                                     (crc1 == crc2) ? CBG  : CBR,  crc2, CNN,
                                     (crc1 == crc2) ? "OK" : "out of sync!",  crc2, CNN);

               if (!TxaOptUnSet('a'))           // show allocation by default
               {
                  dfsExftAllocMap( 0, 0, "@", NULL); // get/show Min/Max size
               }
               if (crc1 != crc2)
               {
                  TxPrint("\nWarning: The first and second FAT areas are NOT identical!\n"
                          "This could be the result of a crash, virus, or a damaged disk.\n");
               }
            }
         }
         nav.up   = exft->Fat1;
         nav.xtra = exft->Root;

         if (exft->MapClust < exft->RawClust)   // not all mapped ?
         {
            TxPrint("\nWarning: Number of sectors in the bootsector is LARGER than accounted for\n"
                    "         by the number of FAT-sectors, there are %s%lu unmapped clusters!%s\n",
                              CBR, exft->RawClust - exft->MapClust, CNN);
         }

         if (dfsExftRootGetBmCluster( dfsa->boot, LSN_BOOTR, EXFAT_1ST_BITMAP,
                                     &exft->BmCluster, &exft->BmByteSize))
         {
            //- to be refined, for TexFAT also need to consider 2nd bitmap (active if a flag in boot?)
            rc = dfsExftBitMapInit(   exft->BmCluster,  exft->BmByteSize); // init Bitmap cache
            if (rc == NO_ERROR)
            {
               rc = dfsExftInitDpCache();       // cache parents for each subdirectory

               if (!TxaOptUnSet('a'))           // show allocation by default
               {
                  dfsExftAllocMap( 0, 0, "@", NULL); // get/show Minimum size
               }
            }
         }
         else
         {
            rc = DFS_BAD_STRUCTURE;
         }

      }
      else
      {
         TxFreeMem( exft->CacheA.Value);
         TxFreeMem( exft->CacheB.Value);
         rc = DFS_ALLOC_ERROR;
      }
      TRACES(( "Fat1:%8.8lx Fat2:%8.8lx InUse:%8.8lx\n", exft->Fat1, exft->Fat2, exft->FatInUse));
      TRACES(( "Sectors: %8.8lx  RawClust: %8.8lx  ClHeap: %8.8lx  ClustSize: %hu\n",
                exft->Sect,  exft->RawClust,    exft->ClHeap,    exft->ClustSize));
   }
   else
   {
      TxPrint("Reading E-FAT primary boot sectors failed!\n");
   }
   RETURN (rc);                                 // when needed
}                                               // end 'dfsExftInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Close FAT filesystem for analysis and free any resources
/*****************************************************************************/
static ULONG dfsExftClose
(
   ULONG               di,                      // IN    dummy
   ULONG               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   dfsSlTableReset();                           // stop SLT thread and reset
   if (exft->CacheA.Dirty)
   {
      rc = dfsExftFlushFAT( &exft->CacheA);
   }
   if (exft->CacheB.Dirty)
   {
      rc = dfsExftFlushFAT( &exft->CacheB);
   }
   TxFreeMem(exft->CacheA.Value);
   TxFreeMem(exft->CacheB.Value);

   rc = dfsExftBitmapFlush( TRUE);              // flush, and terminate cache
   TxFreeMem( dfsa->findSpace.space);           // free the ALLDIRS SPACE
   if (exft->Dp)
   {
      free( exft->Dp);                          // free DirParent cache (realloc!)
   }
   dfsCloseFileSystem();
   RETURN (rc);
}                                               // end 'dfsExftClose'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Initialize FAT filesystem analysis for Area (FS info in FDISK mode)
/*****************************************************************************/
ULONG dfsExftAreaInit
(
   void
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   dfsa->FsAreaClose        = dfsExftAreaClose;
   dfsa->FsAreaLsnAllocated = dfsExftAllocated;

   dfsa->boot = (S_BOOTR *) brec;
   if ((rc = dfsRead( dfstAreaSN2LSN( DFSTORE, LSN_BOOTR), 1, brec)) == NO_ERROR)
   {
      if ((dfsa->boot                != NULL    ) && // could be valid boot record
          (dfsa->boot->Signature     == SV_BOOTR) &&
         ((dfsa->boot->ex.BpsShift   >= 9       ) &&
          (dfsa->boot->ex.BpsShift   <= 12     )) &&
          (dfsa->boot->ex.SpcShift   <= 25      ) &&
          (dfsa->boot->ex.NrOfFats   <= 2       ) &&
          (dfsa->boot->ex.FatLength  != 0       )  )
      {
         exft->Sect  = (ULONG) dfsa->boot->ex.VolumeLength; // to be refined (LLONG?)

         exft->ClustSize     = 1 << dfsa->boot->ex.SpcShift;
         exft->NrOfFats      =      dfsa->boot->ex.NrOfFats;
         exft->Fat1          =      dfsa->boot->ex.FatOffset;

         TRACES(("FAT ClustSize: %4.4hu, Offset: %8.8lx\n, exft->ClustSize, exft->Fat1"));

         exft->FatSectors    = dfsa->boot->ex.FatLength;
         if (exft->NrOfFats > 1)
         {
            exft->Fat2  = exft->Fat1 + dfsa->boot->ex.FatLength;
         }
         else
         {
            exft->Fat2  = exft->Fat1;
         }
         exft->Roots    = exft->ClustSize;      // minimum size
         exft->ClHeap   = dfsa->boot->ex.ClHeapOffset;
         exft->RawClust = dfsa->boot->ex.ClusterCount;

         exft->Root     = dfsExftClust2LSN( dfsa->boot->ex.RootCluster);

         exft->MapClust = exft->FatSectors * (dfsGetSectorSize() * 8 / 32); // mapped clusters
      }
      TRACES(("Allocate memory for %u fat-entries\n", (USHORT) DFSEXFT_CACHE));
      if (((exft->CacheA.Value = TxAlloc(DFSEXFT_CACHE, sizeof(ULONG))) != NULL) &&
          ((exft->CacheB.Value = TxAlloc(DFSEXFT_CACHE, sizeof(ULONG))) != NULL)  )
      {
         exft->FatInUse = exft->Fat1;
         rc = dfsExftGetFAT( 0, &exft->CacheA);
         if (rc == NO_ERROR)
         {
            if (dfsExftRootGetBmCluster( dfsa->boot, LSN_BOOTR, EXFAT_1ST_BITMAP,
                                        &exft->BmCluster, &exft->BmByteSize))
            {
               //- to be refined, for TexFAT also need to consider 2nd bitmap (active is a flag in boot?)
               rc = dfsExftBitMapInit(   exft->BmCluster,  exft->BmByteSize); // init Bitmap cache
            }
            else
            {
               rc = DFS_BAD_STRUCTURE;
            }
         }
      }
      else
      {
         TxFreeMem( exft->CacheA.Value);
         TxFreeMem( exft->CacheB.Value);
         rc = DFS_ALLOC_ERROR;
      }
   }
   TRACES(( "Fat1:%8.8lx Fat2:%8.8lx InUse:%8.8lx\n", exft->Fat1, exft->Fat2, exft->FatInUse));
   TRACES(( "Sectors: %8.8lx  RawClust: %8.8lx  ClHeap: %8.8lx  ClustSize: %hu\n",
             exft->Sect,  exft->RawClust,    exft->ClHeap,    exft->ClustSize));
   RETURN (rc);                                 // when needed
}                                               // end 'dfsExftAreaInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Close FAT filesystem for Area analysis and free any resources
/*****************************************************************************/
static ULONG dfsExftAreaClose
(
   ULONG               di,                      // IN    dummy
   ULONG               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   dfsa->FsAreaClose        = NULL;
   dfsa->FsAreaLsnAllocated = NULL;

   if (exft->CacheA.Dirty)                       // should not happen in area ...
   {
      rc = dfsExftFlushFAT( &exft->CacheA);
   }
   if (exft->CacheB.Dirty)
   {
      rc = dfsExftFlushFAT( &exft->CacheB);
   }
   TxFreeMem(exft->CacheA.Value);
   TxFreeMem(exft->CacheB.Value);

   rc = dfsExftBitmapFlush( TRUE);              // flush, and terminate cache
   RETURN (rc);
}                                               // end 'dfsExftAreaClose'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Search and list possible Root directory clusters for a FAT32 partition
/*****************************************************************************/
static ULONG dfsExftFindExfatRoots               // RET   nr of candidates found
(
   void
)
{
   ULONG               rc = 0;                  // function return
   ENTER();

   TxPrint( "\nSearching for possible rootdirectory clusters, (NOT IMPLEMENTED YET)\n\n");

   #if defined (NEVER)

   //- could replace this with a sector search for the bitmap/upcase entries?
   //- these should be recognizable, but are not alwaus in the same position
   //- depending on a label being present or not.
   //- perhaps look for DIR sector that has label/bitmap/upcase in 1st entry?

   ULONG               cBase;                   // lowest cl in cache
   ULONG               cOffs;                   // offset cl in cache
   ULONG               cFirst;                  // first offset to check
   ULONG               cLast;                   // last offset to check
   ULONG               this;                    // current cluster number
   ULONG               value;                   // cluster value
   ULONG               continuation = 0;        // last continuation cluster
   ULONG               fRoot  = exft->Root;
   ULONG               fRoots = exft->Roots;     // current Root LSN/size

   exft->Root    = 0;                            // define dummy root to avoid
   exft->Roots   = 0;                            // false findings

   TxPrint( "\nSearching for possible rootdirectory clusters ...\n\n");
   dfsInitList( 0, "-w", "-f -P");              // with optimal list options

   exft->CacheB.First  = EXFT_NO_CLUST;         // invalidate B cache

   for ( cBase  = 0;                            // Big steps, whole cache ...
        (cBase <= exft->MapClust) && !TxAbort();
         cBase += DFSEXFT_CACHE)
   {
      dfsExftGetFAT( cBase, &exft->CacheA);     // cache needed part of FAT

      cFirst = (cBase == 0) ? 2 : 0;            // first valid cl is 2
      if ((cBase + DFSEXFT_CACHE) > exft->MapClust)
      {
         cLast = exft->MapClust % DFSEXFT_CACHE;     // last valid offset
      }
      else
      {
         cLast = DFSEXFT_CACHE -1;               // last offset within cache
      }
      for (cOffs = cFirst; (cOffs <= cLast) &&  !TxAbort(); cOffs++)
      {
         this  = cBase + cOffs;                 // this cluster nr
         value = exft->CacheA.Value[cOffs];          // cluster value (next/free)

         //- to be refined, use sector list to cache more continuation clusters
         //- this is a speed optimization only (hardly needed with FAT32 spare)

         if ((this  != continuation) &&         // not next in a chain
             (value != FAT_FREECLUS) &&         // not a free cluster
             (value != FAT_BADCLUST)  )         // and not a BAD cluster
         {
            if (dfsExftRootCandidate( this))     // DIR sector without '..'
            {
               if (dfsExftFindReference( this) == 0) // not a DIR continuation
               {
                  ULONG  lsn = dfsExftClust2LSN( this);

                  dfsAdd2SectorList( lsn);

                  TxPrint( "Possible root DIR : %8.8lx = LSN: %8.8lx\n",
                            this, lsn);
                  rc++;
               }
            }
         }
         continuation = value;
      }
   }
   exft->Root    = fRoot;                        // restore previous values
   exft->Roots   = fRoots;
   #endif
   RETURN (rc);
}                                               // end 'dfsExftFindExfatRoots'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Extract BitMap allocation S_SPACE structure and initialize the BitMap cache
// Note: Area aware to allow usage from FDISK mode too
/*****************************************************************************/
static ULONG dfsExftBitMapInit                  // RET   rc = 0 if type match
(
   ULONG               firstCluster,            // IN    first bitmap cluster
   ULONG               bmByteSize               // IN    size BM file in bytes
)
{
   ULONG               rc = 0;                  // rc, sector match
   ULONG               ef = 0;                  // Initialize error flag!

   ENTER();

   exft->Bm.Size = (ULONG) max(exft->ClustSize, (ULONG) 8);
   if ((exft->Bm.Buffer = (BYTE *) TxAlloc( exft->Bm.Size,
                                            dfsGetSectorSize())) != NULL)
   {
      rc = dfsExftAllocChain( firstCluster, 0, &ef,
                             &exft->Bm.Extents,
                             &exft->Bm.Space);
      if (rc != 0)
      {
         //- LimitLsn is the last addressable LSN in the bitmap, +1
         //- and the first bit represents cluster number 2
         exft->Bm.LimitLsn = ((bmByteSize * 8 +2) *   //- highest CLUSTER in bitmap
                               exft->ClustSize)   +   //- convert to highest SECTOR
                               exft->ClHeap;          //- add offset of 1st cluster (2)
         exft->Bm.Dirty    = FALSE;
         exft->Bm.First    = 0;
         rc = dfsSspaceReadFilePart( exft->Bm.Extents,
                                     exft->Bm.Space,
                                     exft->Bm.First,
                                     exft->Bm.Size,
                            (BYTE *) exft->Bm.Buffer);

         #if defined (DUMP)
            if (exft->Bm.Buffer != NULL)
            {
               TRHEXS( 70, exft->Bm.Buffer, 0xc0, "exft->Bm.Buffer");
            }
         #endif
      }
      else
      {
         TxPrint("\nError interpreting $Bitmap allocation, no Bitmap info available!\n");
         exft->Bm.Size = 0;                     // signal no BitMaps present
      }
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   TRACES(( "Bm.Size: %lu\n", exft->Bm.Size));
   RETURN (rc);
}                                               // end 'dfsExftBitMapInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Interpret and execute specific E-FAT command;
/*****************************************************************************/
static ULONG dfsExftCommand
(
   ULONG               di,                      // IN    dummy
   ULONG               d2,                      // IN    dummy
   char               *none,                    // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;
   LSN                 sn = 0;                  // sector number input
   LONG                nr = 0;
   TXLN                dc;                      // DFS command
   int                 cc;                      // command string count
   char               *c0, *c1, *c2;            // parsed command parts
   TXLN                s1;                      // big temporary string space
   TXLN                s2;                      // big temporary string space
   ULONG               size  = 1;               // default display size
   time_t              stime;

   ENTER();

   time( &stime);                               // timestamp value

   TxaGetArgString( TXA_CUR, 0, 0, TXMAXLN, dc); // dc => cmd from arg 0
   cc = TxaArgCount( );                         // number of parameters
   c0 = TxaArgValue(0);
   c1 = TxaArgValue(1);
   c2 = TxaArgValue(2);

   sn = nav.this;                               // default at current sector

   if ((strcmp(c0, "/?") == 0) ||
       (strcmp(c0,  "?") == 0)  )
   {
      TxShowTxt( exft_txt);
   }
#if defined (NEVER)
   //- to be refined, does not work for E-FAT yet
   else if (strcasecmp(c0, "alldirs"  ) == 0)   // test ALL Dir alloc
   {
      S_SPACE         *ads = NULL;
      ULONG            adc = 0;

      TxPrint( "\nLocating ALL directory sectors in the filesystem ...\n");
      if (dfsExftAllDirs2Space( FALSE, &adc, &ads) == NO_ERROR)
      {
         TxPrint( "Used sectors for ALL directories:\n");
         dfsSspaceDisplay( SD_SUMMARY | SD_TOPLINE | SD_BOTLINE, adc, ads);
         TxFreeMem( ads);                       // free the SPACE
      }
   }
#endif
   else if (strcasecmp(c0, "bootsync"   ) == 0)
   {
      nr = dfsGetMcsNumber( c1, 0);
      if (TxaOption('?') || (nr > 2))
      {
         TxPrint("\nRead bootarea, calculate CRC, and write back to one or both areas\n");
         TxPrint("\n Usage:  %s   [area=0|1|2] [-s]\n\n", c0);
         TxPrint("   area      = Boot area(s) to WRITE, '0' is BOTH areas (default)\n");
         TxPrint("               '1' is PRIMARY (LSN 0), '2' is SPARE area (LSN 12)\n\n");
         TxPrint("   -s        = Read from the SPARE boot-area instead of primary\n\n");
      }
      else
      {
         rc = dfsExftCrcSyncBootAreas((LSN) (TxaOption('s') ? EXFAT_BACKUPBOOT : LSN_BOOTR), nr);
         if (rc == NO_ERROR)
         {
            TxPrint("\nCalculated CRC and synchronised %s boot area to %s\n",
                      (TxaOption('s')) ? "SPARE" :             "PRIMARY",
                      (nr == 2)        ? "SPARE" : (nr == 1) ? "PRIMARY" : "PRIMARY+SPARE");
         }
      }
   }
   else if (strcasecmp(c0, "fatselect"   ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nSelect to use First or Second FAT area\n");
         TxPrint("\n Usage:  %s  1 | 2\n\n", c0);
      }
      else
      {
         nr = dfsGetMcsNumber( c1, 1);
         dfsExftSetFatInUse((LSN) (nr != 2) ? exft->Fat1 : exft->Fat2);
         rc = dfsExftGetFAT(  0, &exft->CacheA);
         if (rc == NO_ERROR)
         {
            dfsLsx("\nSuccessfully read FAT sectors from LSN : ", exft->FatInUse, CBG, "\n");
         }
      }
   }
   else if (strcasecmp(c0, "fatshow"     ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nShow contents of (part of) the FAT area\n");
         TxPrint("\n Usage:  %s  [*|size] [-c:start]\n\n", c0);
         TxPrint("   size      = Number of cluster values to show, '*' ia ALL\n");
         TxPrint("   -c:start  = Start cluster number to show, default is 0\n\n");
      }
      else
      {
         size = 160;
         if (cc > 1)
         {
            if (c1[0] == '*')
            {
               size = exft->MapClust +1;
            }
            else
            {
               size = min((ULONG) atol(c1), exft->MapClust +1);
            }
         }
         dfsExftShowFAT( size, TxaOptNum('c', NULL, 0), c1);
      }
   }
   else if (strcasecmp(c0, "fatsim"   ) == 0)
   {
      if (cc > 1)
      {
         BOOL          compress = TxaOption('z');

         nr = 1;
         if (cc > 2)
         {
            nr = atol(c2);
         }
         sprintf( dc, "image %s 0x0%lx 0x0%lx",
                  c1, (nr == 1) ? exft->Fat1 : exft->Fat2, exft->FatSectors);
         TxaReParseCommand( dc);
         if (compress)
         {
            TxaOptSetItem( "-z");               // set for SIM command too
         }
         rc = DFS_PENDING;
      }
      else
      {
         TxPrint( "\nSave all sectors for the FAT table to a file\n");
         TxPrint( "\nUsage: %s  [-z]  filename  [1 | 2]\n", c0);
      }
   }
   else if (strcasecmp(c0, "fatwrim"   ) == 0)
   {
      if (cc > 1)
      {
         nr = 1;
         if (cc > 2)
         {
            nr = atol(c2);
         }
         sprintf( dc, "wrim %s 0x0%lx 0x0%lx",
                  c1, (nr == 1) ? exft->Fat1 : exft->Fat2, exft->FatSectors);
         TxaReParseCommand( dc);
         rc = DFS_PENDING;
      }
      else
      {
         TxPrint( "\nRestore all sectors for the FAT table from a file\n");
         TxPrint( "\nUsage: %s  filename  [1 | 2]\n", c0);
      }
   }
   else if (strcasecmp(c0, "findroot"       ) == 0)
   {
      ULONG         Roots = dfsExftFindExfatRoots();

      if (Roots != 0)
      {
         TxPrint( "\n%lu possible Rootdirectories found, you can use the "
                  "'list -f' command\nto view them, 'export' and 'import'"
                  "to save and restore them and\nfinally 'setroot' and "
                  "'fixboot' to select one for further use.\n", Roots);
      }
      else
      {
         TxPrint( "Warning: No Root directory candidates found!\n");
      }
   }
   else if (strcasecmp(c0, "setroot"       ) == 0)
   {
      TxPrint("Not implemented for E-FAT yet\n");
   }
   else if (strcasecmp(c0, "path"     ) == 0)
   {
      ULONG    entry = 0;

      if (TxaOptSet('e'))                       // e option specified ?
      {
         entry = DFSSNIGET(TxaOptNum('e', "Entry", 0)) | DFSSNINFO;
      }
      else
      {
         sscanf( c2, "%lu", &entry);
      }
      sn = dfsGetSymbolicSN( c1, nav.this);     // default current LSN
      if (dfsExftLsnInfo2Path( sn, entry, s1) == NO_ERROR)
      {
         TxPrint("\nFound path to root: '%s%s%s'\n", CBY, s1, CNN);
      }
      else
      {
         TxPrint("\nNo FAT path to Root found, %s\n",
                (dfsSlTableStatus(NULL) == SLT_READY) ? "not a directory sector?" :
                                                        "need to build SLT first!");
         rc = DFS_NOT_FOUND;
      }
   }
   else if (strcasecmp(c0, "dirty"    ) == 0)
   {
      BOOL             FsysValue = 0;           // display value
      BOOL             FsysDirty = 0;           // update value

      if (cc > 1)
      {
         switch (c1[0])
         {
            case 'c': case 'C': FsysDirty = FALSE;             break;
            case 'd': case 'D':
            default:            FsysDirty = TRUE;              break;
         }
         FsysValue = dfsExftDirtyStatusBootrec( TRUE, FsysDirty); // update value
      }
      else
      {
         FsysValue  = dfsExftDirtyStatusBootrec( FALSE, FALSE); // just get BOOTREC value
         TxPrint("\nUsage: %s  clean | dirty\n", c0);
         TxPrint("\n      0 = FS clean");
         TxPrint("\n      1 = FS dirty, CHKDSK required");
      }
      TxPrint("\n\nE-FAT status now  : ");
      if (FsysValue == FALSE)
      {
         TxPrint( "CLEAN\n");
      }
      else
      {
         TxPrint( "DIRTY, surface-SCAN/CHKDSK required (MS)\n");
      }
   }
   else if (strcasecmp(c0, "fixboot"  ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nFix the bootsector for an E-FAT filesystem\n");
         TxPrint("\n Usage:  %s  [1 | * | -s]\n\n", c0);
         TxPrint("\n    1    = Force a copy of the SPARE bootsector only (LSN 12 -> 0)");
         TxPrint("\n    *    = Force a copy of the whole SPARE bootsector area to LSN 0");
         TxPrint("\n   -s    = Force a copy of the SPARE bootsector area (same as '*')\n\n");
      }
      else
      {
         if ((TxaOption('s') || (c1[0] == '1') || (c1[0] == '*')))
         {
            if ((rc = dfsRead( EXFAT_BACKUPBOOT, EXFAT_BOOTSIZE, brec)) == NO_ERROR)
            {
               dfsa->boot = (S_BOOTR *) brec;
               switch (dfsIdentifySector(0, 0, brec))
               {
                  case ST_BOOTR:
                     if (SINF->p == NULL)       // No partition info yet
                     {
                        dfstSetSectorSize( DFSTORE, (USHORT) (1 << dfsa->boot->ex.BpsShift));
                        dfstSetClusterSize(DFSTORE, (USHORT) (1 << dfsa->boot->ex.SpcShift));
                     }
                     break;

                  default:
                     TxPrint("Sector %u is not a valid E-FAT BOOT record!\n", EXFAT_BACKUPBOOT);
                     break;
               }
            }
            else
            {
               TxPrint( "Reading E-FAT spare-sectors failed!\n");
            }
         }
         if (((cc <= 1) ||                      // no params, or
               strchr( "1*", c1[0]) != NULL))   // 1 or ALL spare-sector copy
         {
            if (c1[0] == '*')                   // copy whole boot-area
            {
               sprintf( s1, "s (0..11) with the spare copies from sectors 12..23");
            }
            else
            {
               sprintf( s1, " (0) with the spare copy from sector 12");
            }
            if ((dfsa->batch) || (TxConfirm( 5128, "Do you want to replace "
                                 "the E-FAT bootsector%s ? [Y/N] : ", s1)))
            {
               if (DFSTORE_WRITE_ALLOWED)
               {
                  rc = dfsWrite( LSN_BOOTR, (c1[0] == '*') ? EXFAT_BOOTSIZE : 1, brec);
                  if (rc == NO_ERROR)
                  {
                     TxPrint("\nE-FAT boot record updated from spare ...\n");
                     if (SINF->partid != 0)     // was it a partition ?
                     {
                        TxPrint( "Resulting partition after the update:\n\n");
                        sprintf( dc, "part -r %hu", SINF->partid);
                        TxaReParseCommand( dc);
                        rc = DFS_PENDING;       // reread diskinfo & select
                     }
                  }
               }
            }
            else
            {
               rc = DFS_NO_CHANGE;
            }
         }
         else                                   // create new Bootsector ...
         {                                      // with default values
            sn = dfstLSN2Psn( DFSTORE, 0);
            if ((rc = dfsExftMkBootRec( TRUE, (ULONG) ( sn + stime))) == NO_ERROR)
            {
               //- to be refined, need to update the checksums and SPARE,
               //- and sectors 1..10 should be created/written too?

               TRACES(("ExftMkBoot OK, SINF->partid == %hu\n", SINF->partid));
               if (SINF->partid != 0)           // was it a partition ?
               {
                  TxPrint( "Resulting partition after fixing the bootsector:\n\n");
                  sprintf( dc, "part -r -q %hu#d", SINF->partid);
                  rc = dfsMultiCommand( dc, 0, FALSE, FALSE, TRUE);
               }
            }
            else
            {
               TRACES(("ExftMkBoot failed with rc = %lu\n", rc));
            }
         }
         TRACES(("fixboot rc = %lu   dc = '%s'\n", rc, dc));
      }
   }
#if defined (NEVER)
   else if (strcasecmp(c0, "format" ) == 0)     // format current object
   {
      if (TxaOption('?'))
      {
         TxShowTxt( cmd_format);                // give usage
      }
      else                                      // calculate limits + defaults
      {                                         // based on size and exft->info
         FAT_FMT_INFO  fi;                      // format parameter structure

         memset( &fi, 0, sizeof( fi));          // initialize to all zero

         fi.sectors      = dfsGetLogicalSize();
         fi.minimumCs_32 = 1;
         fi.allowedBits  = 32;                  // FAT32 always OK

         if (fi.sectors < FAT12_LIMIT)
         {
            fi.allowedBits += 12;
            fi.minimumCs_12 = ((fi.sectors -1) / (FAT_MAXCLUST & 0xfff)) +1;
         }
         if (fi.sectors < FAT16_LIMIT)
         {
            fi.allowedBits += 16;
            fi.minimumCs_16 = ((fi.sectors -1) / (FAT_MAXCLUST & 0xffff)) +1;
         }
         if      (fi.sectors < FAT32_01LIM) fi.defaultCsize =  1;
         else if (fi.sectors < FAT32_02LIM) fi.defaultCsize =  2;
         else if (fi.sectors < FAT32_04LIM) fi.defaultCsize =  4;
         else if (fi.sectors < FAT32_08LIM) fi.defaultCsize =  8;
         else if (fi.sectors < FAT32_16LIM) fi.defaultCsize = 16;
         else if (fi.sectors < FAT32_32LIM) fi.defaultCsize = 32;
         else                               fi.defaultCsize = 64;

         fi.desiredBits  = TxaOptNum('f', NULL, 0);
         if ((fi.desiredBits & fi.allowedBits) == 0) // illegal desire :-)
         {
            fi.desiredBits  = (fi.allowedBits & 12) ? 12 :
                              (fi.allowedBits & 16) ? 16 : 32;
         }
         sn = dfstLSN2Psn( DFSTORE, 0);
         fi.longFormat   = TxaOption('l');
         fi.desiredRsize = TxaOptNum('r', NULL, (fi.allowedBits & 12) ? 224 : 512);
         fi.desiredCsize = TxaOptNum('c', NULL, 0);
         fi.serial       = TxaOptNum('s', NULL, (ULONG) (sn + stime));
         fi.reservedSect = TxaOptNum('a', NULL, (fi.desiredBits == 32) ? 32 : 1);
         sprintf( s1, "DFS%8.8lx", fi.serial);  // default label
         strcpy( fi.label, TxaOptStr('v', NULL, s1));
         TxPClip(fi.label, FAT_N, ' ');         // pad spaces to 11 positions
         strcpy( fi.os   , TxaOptStr('o', NULL, "M"));
         TxStrToUpper( fi.os);

         fi.desiredRsize = ((((fi.desiredRsize -1) & 0xffff) / 16) +1) * 16;

         TRACES(( "FFI sectors: %8.8lx  serial: %8.8lx\n",  fi.sectors,      fi.serial));
         TRACES(( "FFI allowed:     %4.4hx  desire: %hu\n", fi.allowedBits,  fi.desiredBits));
         TRACES(( "FFI minClSize-12:  %2hu  minC16: %hu\n", fi.minimumCs_12, fi.minimumCs_16));
         TRACES(( "FFI desired-size:  %2hu default: %hu\n", fi.desiredCsize, fi.defaultCsize));
         TRACES(( "FFI Rsize:%hu long:%s label:'%s' os:'%s'\n", fi.desiredRsize,
                                                               (fi.longFormat) ? "YES" : "NO",
                                                                fi.label, fi.os));
         #if defined (USEWINDOWING)
         if (TxaOption('!'))                    // present interactive dialog
         {
            dfsExftFormatDialog( &fi);
         }
         else                                   // execute the FORMAT
         #endif
         {
            TXTS         osname;
            DFSPARTINFO *p = SINF->p;           // partition info

            if ((p == NULL) && (fi.desiredBits == 32)  && // large floppy FAT32
                (dfstStoreType( DFSTORE) == DFST_PHDISK))
            {
               sprintf( s2, "WARNING: FAT32 on whole disk is NOT supported on OS/2!"
               #if defined (DEV32)
                            "\nTo use FAT32, you need to create a partition first."
               #endif
                            "\n\n");
            }
            else
            {
               strcpy( s2, "");
            }

            dfsMakeFatOsString( fi.os[0], osname);

            fi.desiredCsize = dfsExftFmtClusterSize( &fi); // valid clustersize

            if ((dfsa->batch) || (TxConfirm( 5131,
                "      FORMAT confirmation for current object:\n\n%s\n\n"
                "To be formatted as FAT%hu with %hu sectors per cluster\n"
                "using bootcode based on '%s'\n\n%s"
                "All data on this volume/partition/disk will be lost!\n"
                "            Proceed with FORMAT ? [Y/N] : ",
                 dfsDriveDescription( s1), fi.desiredBits, fi.desiredCsize, osname, s2)))
            {
               BOOL        ok = TRUE;           // writing allowed/confirmed

               #if defined (DEV32)
                  char      letter = 0;         // driveletter to restore
                  TXTT      volume;             // LVM volumename

                  //- note that SyncDesired checks for NULL pointer too!
                  if ((dfsLvmSyncDesired( p))   && // partition, sync desired
                      (p->lvm.VoluName[0] != 0) && // and a volumename
                      (p->lvm.letter      != 0)  ) // and driveletter
                  {
                     letter = p->lvm.letter;
                     TxCopy( volume, p->lvm.VoluName, LVM_NAME_L);
                     dfsLvmSyncEngine( TRUE, FALSE, volume, 0); // hide volume
                  }
                  else                          // use locking mechanism
                  {
                     ok = dfstCheckWriteAccess( DFSTORE, TRUE);
                  }
               #else
                  ok = dfstCheckWriteAccess( DFSTORE, TRUE);
               #endif

               if (ok)                          // writing allowed/locked
               {
                  rc = dfsExftFormat( &fi);      // perform actual formatting

                  if (p != NULL)                // if it was a partition
                  {
                     if (rc == NO_ERROR)        // touch up
                     {
                        sprintf( s1, "FAT%2.2hu", fi.desiredBits);
                        dfsSetPartTypeForFS( p, s1); // update in bootsector itself
                        strcpy( p->fsform, s1); // update in internal tables

                        dfsReadDiskInfo( FDSK_QUIET); // re-read all partition info
                     }
                     //- auto reselect partition to refresh info, no screen output
                     dfsMultiCommand( "part .", 0, FALSE, FALSE, FALSE);
                  }

                  #if defined (DEV32)
                  if ((letter != 0) &&          // need to re-assign LVM letter
                      (dfsLvmSyncEngine( TRUE, FALSE, volume, letter) == NO_ERROR))
                  {
                     TxPrint( "Formatted volume  : %c: is available "
                              "for use now, no reboot required.\n", letter);
                  }
                  else
                  #endif
                  {
                     dfstLocking( DFSTORE, DFS_UNLOCK, FALSE,  TRUE);
                     TxPrint( "A reboot may be required before "
                              "the volume can be accessed ...\n");
                  }
               }
            }
         }
      }
   }
#endif
   else if ((strcasecmp(c0, "bsclear"   ) == 0) || // almost like 'bsimport -clear'
            (strcasecmp(c0, "nobads"    ) == 0)  ) // deprecated
   {
      if (TxaOption('?'))
      {
         TxPrint("\nReset bad-sector administration in both FAT areas\n");
         TxPrint("\n Usage:  %s\n\n", c0);
      }
      else
      {
         rc = dfsExftResetBadClus();
      }
   }
#if defined (NEVER)
   else if (strcasecmp(c0, "space"       ) == 0)
   {
      sn = dfsGetSymbolicSN( c1, dfsExftLSN2Clust(nav.this));
      dfsExftShowAlloc( xxxx);                  // note: would need a STREAM dir entry
   }
#endif
   else if (strcasecmp(c0, "filefind"  ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nFind normal (not deleted/renamed) files or directories, by name fragment\n");
         TxPrint("\n Usage:  %s  [name]  [-c] [-d-] [-v]\n\n", c0);
         TxPrint("   name      = part or whole filename wanted, NOT a true wildcard,\n"
                 "               do not use any '?' or '*' characters. Limit the name to\n"
                 "               (a part of) any of 15-character fragments of the full name\n"
                 "   -c        = start from current sector, not start of volume\n"
                 "   -D        = search directories only, not files\n"
                 "   -d-       = search outside EXISTING directories too (SLOW!)\n"
                 "   -v        = verbose search, list files while found (SLOW!)\n");
      }
      else
      {
         if (dfsSlTableStatus( NULL) != SLT_READY)
         {
            TxPrint( "\nBuilding SLT for filename lookup, please wait ..."
                     "   (or abort that using <Esc>)\n");
            dfsMultiCommand( "slt 0 1 -v- -d-", 0, TRUE, FALSE, FALSE);
            TxCancelAbort();                    // might have aborted
         }
         if (dfsa->findSpace.space == NULL)     // no DIR areas determined yet
         {
            //- Create the ALLDIRS SPACE structure (for 'find -o:S' etc)
            if (dfsExftAllDirs2Space( FALSE, &dfsa->findSpace.chunks, &dfsa->findSpace.space) == NO_ERROR)
            {
               sprintf(   s2, "Total # DIR areas : %-8lu    ", dfsa->findSpace.chunks);
               dfstrSiz8( s2, "Size in DIR areas : ",
                 dfsSspaceSectors( TRUE, dfsa->findSpace.chunks, dfsa->findSpace.space), "");
               TxPrint( "%s\n", s2);
            }
         }
         strcpy( s1, c1);                       // direct copy
         sprintf( dfsa->brdescript, "E-FAT %s %s", c0, c1);
         sprintf( dc, "find -t:Dd -o:*$%%]%sMF%s!%s",
                  ( TxaOptUnSet('d')) ? ""  : "S",   //- DIR-areas only (-o:S spaced)
                  ( TxaOption(  'D')) ? "D" : "f",   //- find DIRs only
                  ( TxaOption(  'v')) ? ""  : "Q");  //- verbose or quiet, FAST
         if (strlen( s1))
         {
            strcat( dc, " -u:'");
            strcat( dc,  s1);
            strcat( dc, "'");
         }
         TxPrint("Find file (entry) : %s%s%s\n", CBC, dc, CNN);

         if (!TxaOption('c'))                   // if not 'from current'
         {
            nav.this = exft->Fat2 + exft->FatSectors; // search from end of FAT
         }
         TxaReParseCommand( dc);
         rc = DFS_PENDING;                      // handle translated command
      }
   }
   else if (strcasecmp(c0, "delfind"  ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nFind deleted or renamed) files or directories, by name fragment\n");
         TxPrint("\n Usage:  %s  [name]  [-c] [-d-] [-v]\n\n", c0);
         TxPrint("   name      = part or whole filename wanted, NOT a true wildcard,\n"
                 "               do not use any '?' or '*' characters. Limit the name to\n"
                 "               (a part of) any of 15-character fragments of the full name\n"
                 "   -c        = start from current sector, not start of volume\n"
                 "   -d-       = search outside EXISTING directories too (SLOW!)\n"
                 "   -v        = verbose search, list files while found (SLOW!)\n");
      }
      else
      {
         if (dfsSlTableStatus( NULL) != SLT_READY)
         {
            TxPrint( "\nBuilding SLT for filename lookup, please wait ..."
                     "   (or abort that using <Esc>)\n");
            dfsMultiCommand( "slt 0 1 -v- -d-", 0, TRUE, FALSE, FALSE);
            TxCancelAbort();                    // might have aborted
         }
         if (dfsa->findSpace.space == NULL)     // no DIR areas determined yet
         {
            //- Create the ALLDIRS SPACE structure (for 'find -o:S' etc)
            if (dfsExftAllDirs2Space( FALSE, &dfsa->findSpace.chunks, &dfsa->findSpace.space) == NO_ERROR)
            {
               sprintf(   s2, "Total # DIR areas : %-8lu    ", dfsa->findSpace.chunks);
               dfstrSiz8( s2, "Size in DIR areas : ",
                 dfsSspaceSectors( TRUE, dfsa->findSpace.chunks, dfsa->findSpace.space), "");
               TxPrint( "%s\n", s2);
            }
         }
         strcpy( s1, c1);                       // direct copy
         sprintf( dfsa->brdescript, "E-FAT %s %s", c0, c1);
         sprintf( dc, "find -t:Dd -o:*$%%]%sMFx!%s",
                ( TxaOptUnSet('d')) ? "" : "S", // DIR only (-o:S spaced)
                ( TxaOption('v')) ? "" : "Q");  // verbose or quiet, FAST
         if (strlen( s1))
         {
            strcat( dc, " -u:'");
            strcat( dc,  s1);
            strcat( dc, "'");
         }
         TxPrint("Find deleted files: %s%s%s\n", CBC, dc, CNN);

         if (!TxaOption('c'))                   // if not 'from current'
         {
            nav.this = exft->Fat2 + exft->FatSectors; // search from end of FAT
         }
         TxaReParseCommand( dc);
         rc = DFS_PENDING;                      // handle translated command
      }
   }
#if defined (NEVER)
   else if (strcasecmp(c0, "subfind"  ) == 0)
   {
      sprintf( dfsa->brdescript, "E-FAT %s %s", c0, c1);
      sprintf( dc, "find -t:");
      switch (c1[0])
      {
         case 's':
            strcat( dc, "s");
            TxPrint("Find subdirectory : from ROOT only : %s%s%s\n", CBC, dc, CNN);
            break;

         case 'd':
            strcat( dc, "d");
            TxPrint("Find subdirectory : non-ROOT only  : %s%s%s\n", CBC, dc, CNN);
            break;

         default:
            strcat( dc, "sd");
            TxPrint("Find subdirectory : ROOT & non-ROOT: %s%s%s\n", CBC, dc, CNN);
            break;
      }
      if (!TxaOption('c'))                      // if not 'from current'
      {
         nav.this = dfsExftClust2LSN(2);         // search from first possible
      }
      TxaReParseCommand( dc);
      rc = DFS_PENDING;                         // handle translated command
   }
#endif
   else if (strcasecmp(c0, "serial"  ) == 0)
   {
      if (TxaOption('?'))                       // explicit help request
      {
         TxPrint("\nSet volume-serial number in partition boot areas.\n");
         TxPrint("\n Usage:  %s  [serial]  [-!-]\n\n"
                 "   serial : new volume serial number\n"
                 "   -!-    = do not prompt for new value\n", c0);
      }
      else
      {
         if ((rc = dfsSetVolumeSerialNr( FALSE, !TxaOptUnSet('!'), c1)) == NO_ERROR)
         {
            TxCancelAbort();                    // might have canceled the dialog

            dfsExftCrcSyncBootAreas( LSN_BOOTR, 0); // sync to BOTH areas

            TxPrint( "Written to PRIMARY and SPARE boot area. PBR contents now:\n\n");
            sprintf( dc, "0");
            rc = dfsMultiCommand( dc, 0, FALSE, FALSE, TRUE);
         }
      }
   }
   else
   {
      rc = DFS_PENDING;                         // cmd not recognized
   }
   RETURN (rc);
}                                               // end 'dfsExftCommand'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// FAT filesystem, identify specified sector
/*****************************************************************************/
static ULONG dfsExftIdent
(
   ULONG               lsn,                     // IN    LSN for sector
   ULONG               d2,                      // IN    dummy
   char               *st,                      // OUT   sector type
   void               *sec                      // IN    sector contents
)
{
   ULONG               dr = NO_ERROR;
   BYTE                rc = ST_UDATA;

   ENTER();

   if ((((S_EXFAT *)sec)->MediaType == 0xF8) &&
       ((lsn == exft->Fat1) || (lsn == exft->Fat2)))
   {
      rc = ST_EXFAT;
   }
   else if ((lsn >= exft->Root) && (lsn != 0) &&
            (lsn < (exft->Root + exft->Roots)))
   {                                            // Root-sector reported as
      rc = ST_DIREC;                            // ST_DIREC for searching!
   }
   else if ((rc = dfsExftDirClusterType( lsn)) == ST_UDATA)
   {
      //- future expansion :-)
   }
   switch (rc)
   {
      case ST_UDATA:
         dr = DFS_PENDING;
         break;

      default:
         *st = rc;
         break;
   }
   RETURN (dr);
}                                               // end 'dfsExftIdent'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// FAT filesystem, supply sector-type description string
/*****************************************************************************/
static ULONG dfsExftStype
(
   ULONG               di,                      // IN    dummy
   ULONG               d2,                      // IN    dummy
   char               *st,                      // IN    sector type
   void               *data                     // OUT   type description
)
{
   ULONG               dr  = NO_ERROR;
   char               *buf = (char *) data;
   BYTE                tp  = st[0];

   switch (tp)                                  // searchable types
   {
      case ST_EXFAT: sprintf(buf,"Start E-FAT area "); break;
      case ST_DIREC: sprintf(buf,"Directory data   "); break;
      default:
         switch (tp | ST__INFO)                 // non-searchable ones
         {
            case ST_FAT_1: sprintf(buf,"First  FAT area  "); break;
            case ST_FAT_2: sprintf(buf,"Second FAT area  "); break;
            case ST_ROOTD: sprintf(buf,"Root directory   "); break;
            default:       dr = DFS_PENDING;
         }
         break;
   }
   return (dr);
}                                               // end 'dfsExftStype'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// FAT filesystem, display sector-contents based on type
/*****************************************************************************/
static ULONG dfsExftDispl
(
   ULONG               psn,                     // IN    base psn for sector
   ULONG               d2,                      // IN    dummy
   char               *type,                    // IN    type of sector
   void               *data                     // IN    sector contents
)
{
   ULONG               dr = NO_ERROR;
   BYTE                st = (BYTE) *type;

   ENTER();

   switch (st)
   {
      case ST_ROOTD:
      case ST_DIREC:
         dr = dfsExftDirDisplay( psn, st);
         break;

      default:
         dr = DFS_PENDING;
         break;
   }
   RETURN (dr);
}                                               // end 'dfsExftDispl'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// FAT filesystem, display special LSN info
/*****************************************************************************/
static ULONG dfsExftDisplayLsnInfo
(
   ULONG               lsn,                     // IN    possible special lsn
   ULONG               info,                    // IN    possible DIR entry nr
   char               *dc,                      // IN    dummy
   void               *data                     // IN    dummy
)
{
   ULONG               dr = DFS_PENDING;
   TXLN                path;

   ENTER();
   TRACES(("lsn:%8.8lx info:%8.8lx\n", lsn, info));

   if (info & DFSSNINFO)                        // extra info present
   {
      if (dfsExftLsnInfo2Path( lsn, info, path) == NO_ERROR)
      {
         TxPrint("\nFound path to root: '%s%s%s'\n", CBY, path, CNN);
      }
      nav.xtra = lsn;                           // keep reference arround
      dr = dfsExftShowDirectoryEntry( lsn, DFSSNIGET(info));
   }
   RETURN (dr);
}                                               // end 'dfsExftDisplayLsnInfo'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// Display E-FAT directory sector(s)
/*****************************************************************************/
static ULONG dfsExftDirDisplay
(
   ULONG               psn,                     // IN    base psn for sector
   BYTE                type                     // IN    type of sector
)
{
   ULONG               rc  = NO_ERROR;
   ULONG               lsn = dfstPsn2LSN( DFSTORE, psn);
   ULONG               chunks;                  // nr of space entries
   S_SPACE            *space;                   // space allocation

   ENTER();

   rc = dfsExftDirLsn2Space( lsn, &chunks, &space);
   if (rc == NO_ERROR)
   {
      dfsExftDirectory( chunks, space);
      free( space);                             // alloc'ed no TxFreeMem !
   }
   RETURN (rc);
}                                               // end 'dfsExftDirDisplay'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display E-FAT directory-block
/*****************************************************************************/
ULONG dfsExftDirectory                          // RET   nr of entries
(
   ULONG               chunks,                  // IN    nr of space entries
   S_SPACE            *space                    // IN    space allocation
)
{
   ULONG               rc;
   ULONG               usedEntries = 0;         // nr of used entries
   ULONG               delrEntries = 0;         // deleted/renamed entries
   ULONG               usedFileDir = 0;         // Files or Dirs present
   ULONG               delrFileDir = 0;         // deleted/renamed files
   ULONG               chunk;                   // index in space-area
   ULONG               sect;                    // sectors to handle
   ULONG               entry;                   // index in fat directory
   S_EXFTDIR           prevEntryBuffer;         // Previous entry, contents
   S_EXFTDIR          *prev;                    // Previous entry, pointer
   S_EXFTDIR          *this;                    // Fat directory entry
   int                 links = 0;               // down/xtra links given
   int                 ei = -1;                 // entry index (pre-increment)
   int                 cl = TXaNWnZ;            // LSN color
   int                 ac;                      // alternate color fg+bg
   LSN                 data;                    // 1st data sector
   LSN                 dirsec;                  // current dir sector LSN
   LSN                 prevsec;                 // previous dir sector LSN
   TXLN                lfn;                     // Long filename
   ULONG               dirholes = 0;            // empty / free entries
   USHORT              bps = dfsGetSectorSize();
   USHORT              entries = (bps / sizeof(S_EXFTDIR));
   BYTE                pendingName = 0;
   USHORT              fragmentSize;            // Filename fragment size
   BYTE               *dirbuf;
   S_EXFTDIR          *fatdir;                  // Fat directory sector

   ENTER();

   if ((dirbuf = TxAlloc( 1, bps)) != NULL)     // allocate one sector
   {
      fatdir = (S_EXFTDIR *) dirbuf;
      prev   = &prevEntryBuffer;

      TxPrint("\n");
      dfsExftDirHeader( "Nr  Sector  ", TRUE);   // display header

      TRACES(("Sizeof S_EXFTDIR: %u\n", (USHORT) sizeof(S_EXFTDIR)));

      if (!TxaOptUnSet('l'))                    // create a new list for DIR
      {
         dfsInitList(0, "-f -P", "-d");         // optimal for menu file-recovery
      }
      strcpy( lfn, "");                         // init first to not-present
      prevsec = space[0].start;                 // safe start value for prev

      for (chunk = 0; (chunk < chunks)  && !TxAbort(); chunk++)
      {
         for (sect = 0; (sect < space[chunk].size)  && !TxAbort(); sect++)
         {
            dirsec  = space[chunk].start + sect;
            if ((rc = dfsRead(dirsec, 1, dirbuf)) == NO_ERROR)
            {
               for (entry = 0; (entry < entries) && !TxAbort(); entry++)
               {
                  this = &(fatdir[entry]);

                  if (this->EntryType & EXFAT_DIRBIT_IN_USE)
                  {
                     if (dirholes != 0)         // zeroed entries before this, invalid!
                     {
                        TxPrint( "\nWARNING: %lu Invalid ZERO entries, regular 'dir'"
                                 " will not see rest of directory!\n\n", dirholes);
                        dirholes = 0;
                     }
                     usedEntries++;
                  }
                  else if (this->EntryType != EXFAT_DIR_EODIR)
                  {
                     delrEntries++;
                  }
                  switch (this->EntryType & ~EXFAT_DIRBIT_IN_USE) // normal + deleted
                  {
                     case EXFAT_DEC_VLABEL:
                        if ((this->EntryType & EXFAT_DIRBIT_IN_USE) || TxaOption('D'))
                        {
                           ei++;                // to first/next entry-index
                           ac = (ei % 2) ? TXaBWnC : TXaNWnZ;
                           TxPrint("%s.%03.3u", ansi[ac], ei);
                           TxUnic2Ascii( this->lb.Name, lfn, EXFAT_LABEL_SIZE);
                           TxPrint( "%32.32s Label: ", "");
                           TxPrint( "%s%s%s%s\n", ansi[Ccol((CcG | CcI), Ccbg(ac))], lfn,
                                                  ansi[Ccol( CcW,        Ccbg(ac))], CGE);

                           if (!TxaOptUnSet('l'))
                           {
                              dfsAdd2List( dirsec, entry | DFSSNINFO, NULL, NULL);
                           }
                        }
                        break;

                     case EXFAT_DEC_BITMAP:
                        if ((this->EntryType & EXFAT_DIRBIT_IN_USE) || TxaOption('D'))
                        {
                           ei++;                // to first/next entry-index
                           data = dfsExftClust2LSN( dfsExftDir2Clust(this));
                           ac = (ei % 2) ? TXaBWnC : TXaNWnZ;
                           TxPrint("%s.%03.3u", ansi[ac], ei);
                           dfsLsn( " ", data,   ansi[ac], ansi[ac]);
                           TxPrint( "%23.23s Sfile: alloc-bitmap %13llu %8lX%s\n",
                                    "", this->bm.Length, this->bm.Cluster, CGE);

                           if (!TxaOptUnSet('l'))
                           {
                              dfsAdd2List( dirsec, entry | DFSSNINFO, NULL, NULL);
                           }
                        }
                        break;

                     case EXFAT_DEC_UPCASE:
                        if ((this->EntryType & EXFAT_DIRBIT_IN_USE) || TxaOption('D'))
                        {
                           ei++;                // to first/next entry-index
                           data = dfsExftClust2LSN( dfsExftDir2Clust(this));
                           ac = (ei % 2) ? TXaBWnC : TXaNWnZ;
                           TxPrint("%s.%03.3u", ansi[ac], ei);
                           dfsLsn( " ", data,   ansi[ac], ansi[ac]);
                           TxPrint( "%23.23s Sfile: upcase-table %13llu %8lX%s\n",
                                    "", this->up.Length, this->up.Cluster, CGE);

                           if (!TxaOptUnSet('l'))
                           {
                              dfsAdd2List( dirsec, entry | DFSSNINFO, NULL, NULL);
                           }
                        }
                        break;

                     case EXFAT_DEC_FILE:
                        //- no print action, handled by STREAM using prev entry
                        break;

                     case EXFAT_ANY_STREAM:
                        if ((prev->EntryType & ~EXFAT_DIRBIT_IN_USE) == EXFAT_DEC_FILE)
                        {
                           if (this->EntryType & EXFAT_DIRBIT_IN_USE) // FILE+STREAM seen
                           {
                              usedFileDir++;
                           }
                           else
                           {
                              delrFileDir++;
                           }
                           strcpy( lfn, "");
                           pendingName = this->st.NameLength;
                           if ((this->EntryType & EXFAT_DIRBIT_IN_USE) || TxaOption('D'))
                           {
                              ei++;             // to first/next entry-index
                              data = dfsExftClust2LSN( dfsExftDir2Clust(this));
                              ac = (ei % 2) ? TXaBWnC : TXaNWnZ;
                              TxPrint("%s.%03.3u", ansi[ac], ei);
                              switch (links++)
                              {
                                 case 0:  cl = Ccol((CcG | CcI), Ccbg(ac)); nav.down = data; break;
                                 case 1:  cl = Ccol((CcY | CcI), Ccbg(ac)); nav.xtra = data; break;
                                 default: cl = ac;                                           break;
                              }
                              dfsLsn( " ", data,   ansi[cl], ansi[ac]);
                              dfsExftShowDirEntry( prev, this, ac);

                              if (!TxaOptUnSet('l'))
                              {
                                 if (entry == 0) // STREAM is first in sector, FILE is last in PREV
                                 {
                                    dfsAdd2List( prevsec, entries -1 | DFSSNINFO, NULL, NULL);
                                 }
                                 else
                                 {
                                    dfsAdd2List( dirsec, entry -1 | DFSSNINFO, NULL, NULL);
                                 }
                              }
                           }
                        }
                        else
                        {
                           TxPrint( "E-FAT stream, unexpected at this position\n");
                        }
                        break;

                     case EXFAT_ANY_FILENAME:
                        if (pendingName > 0)
                        {
                           if (pendingName > EXFAT_NAME_SIZE)
                           {
                              fragmentSize = EXFAT_NAME_SIZE;
                              pendingName -= EXFAT_NAME_SIZE;
                           }
                           else                 // no more name fragments expected
                           {
                              fragmentSize = pendingName;
                              pendingName  = 0;
                           }
                           if ((this->EntryType & EXFAT_DIRBIT_IN_USE) || TxaOption('D'))
                           {
                              TxUnicAppend( this->nm.Name, lfn, fragmentSize);
                              if (pendingName == 0)
                              {
                                 TxPrint( "%s%s%s%s\n", ansi[Ccol((CcG | CcI), Ccbg(ac))], lfn,
                                                        ansi[Ccol( CcW,        Ccbg(ac))], CGE);
                              }
                           }
                        }
                        else
                        {
                           //- Stray name fragments DO happen with file delete/rename (OSX trashcan etc)
                           //- strcpy( lfn, "");
                           //- TxUnicAppend( this->nm.Name, lfn, EXFAT_NAME_SIZE);
                           //- TxPrint( "E-FAT name fragment '%s', unexpected at this position\n", lfn);
                        }
                        break;

                     case EXFAT_DIR_EODIR:      // free entry
                        dirholes++;             // and count them
                        break;

                     default:                   // others (deleted/renamed etc)
                        break;
                  }
                  prevEntryBuffer = *this;      // make a persistent copy!
                  prevsec = dirsec;             // and remember its LSN (STREAM to FILE backtracking)
               }
            }
         }
      }
      TxPrint( "%s", CNN);
      if (ei > 12)
      {
         dfsExftDirHeader( "Nr  Sector  ", FALSE); // display footer
      }
      if (usedEntries != 0)
      {
         TxPrint("\n%6lu entries are for %5lu files or subdirectories.\n",    usedEntries, usedFileDir);
      }
      if (delrEntries != 0)
      {
         TxPrint("%6lu entries are for %5lu %sdel%seted (or renamed) files, use 't -D' to display.\n",
                 delrEntries, delrFileDir, (TxaOption('D')) ? CBM : CNN, CNN);
      }
      TxFreeMem( dirbuf);
   }
   RETURN (usedEntries);
}                                               // end 'dfsExftDirectory'
/*---------------------------------------------------------------------------*/

//- to be refined: extra parameter to allow finding 'deleted' files (only ?)
/*****************************************************************************/
// Find LSN* for specified path, starting at Root directory LSN
/*****************************************************************************/
static ULONG dfsExftFindPath
(
   ULONG               loud,                    // IN    Show progress
   ULONG               d2,                      // IN    dummy
   char               *path,                    // IN    path specification
   void               *vp                       // OUT   found dir/file + fat entry
)
{
   BOOL                rc  = NO_ERROR;
   LSN                 fs;                      // Current FNODE LSN
   LSN                 ds  = 0;                 // Current DIR LSN
   TXLN                part;
   char               *p   = path;
   int                 l;
   S_EXFTDIR           de;                      // directory entry, FILE
   S_EXFTDIR           stream;                  // directory entry, STREAM
   ULONG               chunks;                  // chunks in space structure
   S_SPACE            *space;                   // space structure
   DFS_PARAMS         *parm = (DFS_PARAMS *) vp;
   ULONG               entry;

   ENTER();
   fs = exft->Root;                             // start LSN
   if (loud)
   {
      TxPrint("RootDirectory LSN : %08.8lX   find path: '%s'\n", fs, path);
   }

   parm->Lsn    = fs;
   parm->Number = strlen(p) ? DFSSNINFO : 0;    // no info for ROOT!
   while ((rc == NO_ERROR) && strlen(p) && !TxAbort())
   {
      if ((l = strcspn(p, FS_PATH_STR)) != 0)
      {
         strncpy(part, p, l);                   // isolate part
         part[l] = '\0';
         p += l;                                // skip part
         if (*p == FS_PATH_SEP)
         {
            p++;                                // skip '\'
         }
      }
      if (strlen(part))
      {
         if (dfsExftDirLsn2Space(fs, &chunks, &space) == NO_ERROR)
         {
            if (loud)
            {
               TxPrint("Base DirBlock LSN : %08.8lX%s", fs, TREOLN);
            }
            if (dfsExftDirSpace2Entry(part, chunks, space, &ds, &entry, &de, &stream))
            {
               if (loud)
               {
                  TxPrint(" - Lsn %08.8lX, entry %2lu for '%s'\n", ds, entry, part);
               }
               if (*p == '\0')                          //- end of string, found!
               {
                  parm->Lsn      = ds;                  //- DIR sector number
                  parm->Number   = entry | DFSSNINFO;   //- entry 0..15 in that sector
                  parm->Flag     = TRUE;                //- Size is from DIR-entry
                  parm->byteSize = stream.st.Length;
               }
               else
               {
                  if ((de.fl.Attrib & FATTR_DIRECTORY) == 0)
                  {
                     if (loud)
                     {
                        TxPrint("Object at sector  : 0x%8.8X, entry 0x%2lx is not a directory\n", ds, entry);
                     }
                     rc = ERROR_PATH_NOT_FOUND;
                  }
               }
               fs = dfsExftClust2LSN( dfsExftDir2Clust(&stream)); // LSN for next DIR cluster in path
            }
            else
            {
               if (loud)
               {
                  TxPrint(" - Search failed   for '%s'\n", part);
               }
               rc = ERROR_PATH_NOT_FOUND;
            }
            free( space);                       // free the memory !
         }
      }
      else
      {
         rc = ERROR_PATH_NOT_FOUND;
      }
   }
   RETURN(rc);
}                                               // end 'dfsExftFindPath'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// Find specific name in a (sub) directory SPACE structure, return file+stream
/*****************************************************************************/
static BOOL dfsExftDirSpace2Entry               // RET   name found
(
   char               *name,                    // IN    subdir or filename
   ULONG               chunks,                  // IN    nr of space entries
   S_SPACE            *space,                   // IN    space allocation
   LSN                *lsn,                     // OUT   Dir-sector LSN found
   ULONG              *info,                    // OUT   Dir entry-nr found
   S_EXFTDIR          *efile,                   // OUT   Directory entry, FILE
   S_EXFTDIR          *estream                  // OUT   Directory entry, STREAM
)
{
   BOOL                found = FALSE;
   ULONG               chunk;                   // index in space-area
   ULONG               sect;                    // sectors to handle
   ULONG               entry;                   // index in fat directory
   LSN                 dirsec;                  // current dir sector LSN
   TXLN                lfn;                     // VFAT long filename
   S_EXFTDIR          *fe;                      // Fat directory entry
   USHORT              bps = dfsGetSectorSize();
   USHORT              entries = (bps / sizeof(S_EXFTDIR));
   BYTE               *dirbuf;
   S_EXFTDIR          *fatdir;                  // Fat directory sector

   ENTER();
   TRARGS(("Name to find : '%s'\n", name));

   if ((dirbuf = TxAlloc( 3, bps)) != NULL)     // allocate sectors
   {
      fatdir = (S_EXFTDIR *) dirbuf;

      for ( chunk = 0;
           (chunk < chunks) && !TxAbort() && !found && (space);
            chunk++)
      {
         for ( sect = 0;
              (sect < space[chunk].size)  && !TxAbort() && !found;
               sect++)
         {
            dirsec  = space[chunk].start + sect;
            if (dfsRead(dirsec, 3, dirbuf) == NO_ERROR) // read 3, any name entries
            {
               for ( entry = 0;
                    (entry < entries) && !TxAbort() && !found;
                     entry++)
               {
                  fe = &(fatdir[entry]);

                  switch (fe->EntryType & ~EXFAT_DIRBIT_IN_USE) // normal + deleted
                  {
                     case EXFAT_DEC_VLABEL:
                     case EXFAT_DEC_BITMAP:
                     case EXFAT_DEC_UPCASE:
                     case EXFAT_DEC_FILE:
                        dfsExftEntry2Name( fe, lfn); // get real or faked file name
                        if (strcasecmp( lfn, name) == 0)
                        {
                           if (lsn)
                           {
                              *lsn = dirsec;
                           }
                           if (info)
                           {
                              *info = entry;
                           }
                           if (efile)
                           {
                              *efile = *fe;
                           }
                           if (estream)
                           {
                              *estream = *(fe +1); // stream is next entry
                           }
                           found = TRUE;
                        }
                        break;

                     default:
                        break;
                  }
               }
            }
         }
      }
      TxFreeMem( dirbuf);
   }
   BRETURN(found);
}                                               // end 'dfsExftDirSpace2Entry'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// DFS FAT write-file to disk (SaveTo to file)
/*****************************************************************************/
static ULONG dfsExftFileSaveAs
(
   ULONG               dse,                     // IN    dirsec/entry LSN
   ULONG               info,                    // IN    info, dir entry
   char               *path,                    // IN    destination path
   void               *recp                     // INOUT recovery parameters
)
{
   ULONG               rc  = NO_ERROR;
   S_RECOVER_PARAM    *param = (S_RECOVER_PARAM *) recp;
   S_EXFTDIR          *efile;                   // Fat dir-entry ptr, FILE
   S_EXFTDIR          *stream;                  // STREAM (or bitmap/upcase)
   DFSISPACE           isp;                     // allocation SPACE info
   ULONG               ef = 0;                  // allocation error flag
   BYTE               *sb;
   S_EXFTDIR          *fatdir;                  // ExFat directory sector
   BOOL                isDir = FALSE;           // regular entry for a directory
   int                 sanity;

   ENTER();
   TRARGS(("Dirsec LSN : %8.8lX, entry:%lu to path: '%s'\n", dse, DFSSNIGET(info), path));

   if ((sb = TxAlloc(5, dfsGetSectorSize())) != NULL) // 2 extra sectors BEFORE and AFTER
   {
      if ((rc = dfsRead( dse - 2, 5, sb)) == NO_ERROR)
      {
         fatdir = (S_EXFTDIR *) (sb + (2 * dfsGetSectorSize()));
         efile  = &(fatdir[DFSSNIGET(info)]);
         stream = efile;                        // stream is same entry for bitmap/upcase
         TRACES(("fatdir:%8.8lx efile:%8.8lx for entry %lu Type: 0x%2.2hx\n",
                  fatdir,       efile, DFSSNIGET(info), stream->EntryType));
         switch (efile->EntryType & ~EXFAT_DIRBIT_IN_USE)
         {
            case EXFAT_ANY_STREAM:
            case EXFAT_ANY_FILENAME:            // need to search backwards to FILE entry
               sanity = 2 * dfsGetSectorSize() / sizeof(S_EXFTDIR);
               while (((efile->EntryType & ~EXFAT_DIRBIT_IN_USE) != EXFAT_DEC_FILE) && (sanity--))
               {
                  efile--;                      // move backwards over 1 entry
               }
               if ((efile->EntryType & ~EXFAT_DIRBIT_IN_USE) != EXFAT_DEC_FILE)
               {
                  TRACES(("No FILE entry found (backward from stream/filename)\n"));
                  rc = DFS_BAD_STRUCTURE;
                  break;                        // no FILE entry found, fail
               }
            case EXFAT_DEC_FILE:
               stream = efile +1;               // advance to actual stream entry
               isDir  = ((efile->fl.Attrib & FATTR_DIRECTORY) != 0);
            case EXFAT_DEC_BITMAP:
            case EXFAT_DEC_UPCASE:
               if (dfsExftAllocStream( stream, &ef, &isp.chunks, &isp.space) != 0)
               {
                  TXLN       fname;             // destination base filename
                  TXLN       fullfn;            // unique filename prefix

                  if ((param->newname == NULL) || (*param->newname == 0)) // no newname present
                  {
                     dfsExftEntry2Name( efile, fname);
                  }
                  else                          // rename specified
                  {
                     strcpy( fname, param->newname);
                  }
                  if (param->unique)            // force unique naming
                  {                             // on PATH and FILE components
                     sprintf(fullfn, "%8.8lX_%s", dse, fname);
                     strcpy( fname, fullfn);    // and move back to fname
                  }
                  if (param->recFname)
                  {
                     strcpy( param->recFname, fname); // return full base-filename
                  }

                  isp.byteSize = stream->st.Length;
                  rc = dfsSspaceFileSaveAs( &isp, isDir,
                                          ((efile->EntryType & EXFAT_DIRBIT_IN_USE) == 0),
                                            param->name83, path, fname, fullfn);
                  if (rc == NO_ERROR)
                  {
                     time_t      mod = txOS2FileTime2t( &efile->fl.DtModify.u, &efile->fl.TmModify.u);
                     time_t      cre = txOS2FileTime2t( &efile->fl.DtCreate.u, &efile->fl.TmCreate.u);
                     time_t      acc = txOS2FileTime2t( &efile->fl.DtAccess.u, &efile->fl.TmAccess.u);

                     TRACES(("Set original date/time for: '%s'\n", fullfn));

                     TxSetFileTime( fullfn, &cre, &acc, &mod); // set time on full-fn from SaveAs

                     if (param->recFname)
                     {
                        strcpy( param->recFname, fullfn);
                     }
                  }
                  free( isp.space);             // use free (realloc/calloc)
               }
               else
               {
                  if (ef != 0)                  // inconsistent FAT values
                  {
                     dfsExftDispError( ef, 0, "\nWarning: ", NULL);
                     rc = DFS_BAD_STRUCTURE;
                  }
                  else
                  {
                     rc = DFS_ALLOC_ERROR;
                  }
               }
               break;

            default:
               TRACES(("Invalid Entrytype\n"));
               rc = DFS_BAD_STRUCTURE;
               break;
         }
      }
      TxFreeMem( sb);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN(rc);
}                                               // end 'dfsExftFileSaveAs'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Make E-FAT usage map based on the bitmap, or the first FAT area
/*****************************************************************************/
static ULONG dfsExftAllocMap
(
   ULONG               di,                      // IN    dummy
   ULONG               d2,                      // IN    dummy
   char               *options,                 // IN    display options
   void               *data                     // OUT   dummy
)
{
   if (TxaOption( 'f'))                         // base map display on FAT
   {
      return dfsExftAllocFat( options);
   }
   else                                         // base map display on bitmap
   {
      return dfsExftAllocBm( options);
   }
}                                               // end 'dfsExftAllocMap'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Make E-FAT usage map based on the E-FAT bitmap
/*****************************************************************************/
static ULONG dfsExftAllocBm
(
   char               *options                  // IN    display options
)
{
   ULONG               i;                       // index in data-area
   ULONG               b;                       // byte-index for one mapchar
   ULONG               cClus = 0;               // Clusters in current char
   LLONG               lClus = 0;               // Clusters in current line
   ULONG               mClus = 0;               // Clusters in current map
   ULONG               uClus = 1;               // Last cluster that is in use
   ULONG               l;                       // line number, 0 based
   ULONG               a;                       // index in display-array
   ULONG               perc;                    // percentage
   TXLN                ascii;                   // display-array
   LSN                 lbase = 0;               // current line base LSN
   ULONG               spe;                     // sectors per entry
   ULONG               cpc;                     // clusters per display-char
   ULONG               spc;                     // sectors per display-char
   ULONG               size;                    // nr of clusters to map
   BOOL                verbose = (*options != '@');

   ENTER();

   spe   = exft->ClustSize;
   size  = exft->RawClust;                      // Cluster 2 .. last-valid

   cpc   = dfsAllocItemsPerChar( 'l', options, size);
   spc   = cpc * spe;
   dfsProgressInit( 0, size * spe, "Sector:", "alloc checked", DFSP_STAT, DFSP_ETA_FACTOR);

   if (verbose)
   {
      dfsSiz8("Sectors per char  : ", spc, "\n");
      TxPrint(" %16.16s = Allocation grade, empty upto fully occupied\n", mapchar);
   }
   TxPrint(   "        %s%s%s\n", CBC, BLIN, CNN);

   TxPrint("00000000%s%sB%s %5lu KiB", CBC, CnB, CnY, exft->FatOffset / 2);

   if (exft->NrOfFats == 1)
   {
      TxPrint("%s     Fat area 32-bit entries, size:%8lu KiB      %s%s",
               CnM,           exft->FatSectors / 2,             CBC, CNN);
   }
   else
   {
      TxPrint("%s Fat1 32-bit, %7lu KiB  %s Fat2 32-bit, %7lu KiB %s%s",
         CnM, exft->FatSectors / 2, CnG, exft->FatSectors / 2, CBC, CNN);
   }
   TxPrint("\n");

   for (i=0, a=0, l=0; (i < size) && (!TxAbort());)
   {
      if (a == 0)
      {
         memset(ascii, 0, TXMAXTM);
         lClus = 0;
         lbase = dfsExftClust2LSN(i + 2);       // first used clusternr is 2!
      }
      for (cClus=0, b=0; (b < cpc) && (i < size); b++, i++)
      {
         if (dfsExftBitmapCache(i + 2, NULL))   // cluster in use ?
         {
            cClus++;
            uClus = i + 2;                          // last in-use one sofar
         }
      }
      lClus += cClus;

      dfsProgressShow( (i * spe) + spc, 0, NULL);

      TRACES(("lClus 2nd Loop i: %lu, b:%lu = %lu\n", i, b, lClus));
      ascii[a] = (char) ((cClus == 0) ? ' ' : mapchar[cClus * 8 / cpc]);
      TRACES(("Character for pos %lu = '%c'\n", a, ascii[a]));
      a++;
      if ((i && ((i%(64*cpc)) == 0)) || (i >= size))
      {
         perc  = (ULONG)(100*lClus / (a*cpc));
         TRACES(("Percentage 2nd Loop i: %lu = %lu\n", i, perc));
         TxPrint("%08.08lX%s%s%s%s", lbase, CBC, CNC, ascii, CNN);
         if (a == 64)
         {
            TxPrint("%s%s% 3lu%%\n", CBC, CNN, perc);
         }
         else
         {
            TxPrint("%s%.*s%s% 3lu%%\n",
                    CBC, (int) (64-a-1), BLIN, CNN, perc);
         }
         if (i < size)
         {
            a = 0;                              // keep a value on last line
         }
         mClus += lClus;
         l++;
      }
   }
   TxPrint("        %s%.*s%s\n", CBC, (USHORT) a, BLIN, CNN);

   if (TxAbort())
   {
      TxPrint( "\nAllocation check and display aborted! Used/Free info not available.\n");
   }
   else
   {
      if (verbose)
      {
         dfsSiz8("Size for one line : ", 64*cpc*spe, "\n");
      }

      dfsSiz8("Total / Used size : ", size*spe, " ");
      if (size != 0)                            // avoid devide by 0
      {
         ULONG            used = mClus * spe;

         dfsa->FsUnallocated   = size * spe - used; // free space (optimize hint)

         dfsSiz8("/ ", used, " ");
         TxPrint("% 5.1lf%%\n", (double) (100 * ((double) mClus) / size));
      }

      dfsa->FsTruncPoint = dfsExftClust2LSN( uClus +1); // first free (truncate)
      dfsa->FsLastInUse  = dfsa->FsTruncPoint -1;
      //- dfsSiz8( "Minimum vol. size : ", dfsa->FsTruncPoint, " (for  Resizing)\n");
   }
   dfsProgressTerm();
   RETURN (NO_ERROR);
}                                               // end 'dfsExftAllocBm'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Make E-FAT usage map based on the FAT area (informational only?)
/*****************************************************************************/
static ULONG dfsExftAllocFat
(
   char               *options                  // IN    display options
)
{
   ULONG               i;                       // index in data-area
   ULONG               b;                       // byte-index for one mapchar
   ULONG               cClus = 0;               // Clusters in current char
   LLONG               lClus = 0;               // Clusters in current line
   ULONG               mClus = 0;               // Clusters in current map
   ULONG               bClus = 0;               // Clusters marked as bad
   ULONG               uClus = 1;               // Last cluster that is in use
   ULONG               xClus = 0;               // Clusters that are NOT mapped
   LSN                 lbase = 0;               // current line base LSN
   ULONG               perc  = 0;               // percentage
   ULONG               l;                       // line number, 0 based
   ULONG               a;                       // index in display-array
   TXLN                ascii;                   // display-array
   ULONG               spe;                     // sectors per entry
   ULONG               cpc;                     // clusters per display-char
   ULONG               spc;                     // sectors per display-char
   ULONG               size;                    // nr of clusters to map
   ULONG               value;                   // FAT entry value
   ULONG               nonClust = exft->ClHeap;
   BOOL                verbose  = (*options != '@');

   ENTER();

   spe   = exft->ClustSize;
   size  = exft->RawClust - 2;                   // #clusters minus first two

   cpc   = dfsAllocItemsPerChar( 'l', options, size);

   spc = cpc * spe;
   dfsProgressInit( 0, size * spe, "Sector:", "alloc checked", DFSP_STAT, DFSP_ETA_FACTOR);

   if (verbose)
   {
      dfsSiz8("Sectors per char  : ", spc, "\n");
      TxPrint(" %-16.16s = Allocation grade, empty upto fully occupied\n", mapchar);
      if (exft->MapClust < exft->RawClust)
      {
         TxPrint( " %-16.16s = Filesystem area NOT mapped by the FAT!\n", "X");
      }
   }
   TxPrint(   "        %s%s%s\n", CBC, BLIN, CNN);

   TxPrint("00000000%s%sB%s %5lu KiB", CBC, CnB, CnY, exft->FatOffset / 2);

   if (exft->NrOfFats == 1)
   {
      TxPrint("%s     Fat area 32-bit entries, size:%8lu KiB      %s%s",
               CnM,           exft->FatSectors / 2,             CBC, CNN);
   }
   else
   {
      TxPrint("%s Fat1 32-bit, %7lu KiB  %s Fat2 32-bit, %7lu KiB %s%s",
         CnM, exft->FatSectors / 2, CnG, exft->FatSectors / 2, CBC, CNN);
   }
   TxPrint("\n");

   for (i=0, a=0, l=0; (i < size) && (!TxAbort()) && (exft->CacheA.Value);)
   {
      if (a == 0)
      {
         memset(ascii, 0, TXMAXTM);
         lClus = 0;
         lbase = dfsExftClust2LSN(i+2);
      }
      for (cClus=0, b=0; (b < cpc) && (i < size); b++, i++)
      {
         if (i < exft->MapClust)                // mapped in the FAT area ?
         {
            value = dfsExftValue(i+2);          // one FAT entry value
            if (value > EXFAT_MAXCLUST)
            {
               cClus++;                         // cluster in use or bad
               if (value < EXFAT_MEDIA_CL)
               {
                  bClus++;                      // bad sectors
               }
               else                             // EOF cluster
               {
                  uClus = i+2;                  // last in-use one sofar
               }
            }
            else
            {
               if (value != 0)
               {
                  cClus++;                      // count cluster in use
                  uClus = i+2;                  // last in-use one sofar
               }
            }
         }
         else
         {
            xClus++;
         }
      }
      lClus += cClus;

      dfsProgressShow( (i * spe) + spc, 0, NULL);

      TRACES(("lClus 2nd Loop i: %lu, b:%lu = %lu\n", i, b, lClus));
      if (i < exft->MapClust)
      {
         ascii[a] = (char) ((cClus == 0) ? ' ' : mapchar[cClus * 8 / cpc]);
      }
      else
      {
         ascii[a] = 'X';                        // non mapped clusters!
      }
      TRACES(("Character for pos %lu = '%c'\n", a, ascii[a]));
      a++;
      if ((i && ((i%(64*cpc)) == 0)) || (i >= size))
      {
         perc  = (ULONG)(100*lClus / (a*cpc));
         TRACES(("Percentage 2nd Loop i: %lu = %lu\n", i, perc));
         TxPrint("%08.08lX%s%s%s%s", lbase, CBC, CNC, ascii, CNN);
         if (a == 64)
         {
            TxPrint("%s%s% 3lu%%\n", CBC, CNN, perc);
         }
         else
         {
            TxPrint("%s%.*s%s% 3lu%%\n",
                    CBC, (int) (64-a-1), BLIN, CNN, perc);
         }
         if (i < size)
         {
            a = 0;                              // keep a value on last line
         }
         mClus += lClus;
         l++;
      }
   }
   TxPrint("        %s%.*s%s\n", CBC, (USHORT) a, BLIN, CNN);

   if (TxAbort())
   {
      TxPrint( "\nAllocation check and display aborted! Used/Free info not available.\n");
   }
   else
   {
      if (verbose)
      {
         dfsSiz8("Size for one line : ", 64 * spc, "\n");
      }

      if (size != 0)                            // avoid devide by 0
      {
         ULONG            total = size  * spe + nonClust +1; //- include bootsec
         ULONG            used  = mClus * spe + nonClust -1; //- used incl FAT areas
         ULN64            free;

         exft->FreeClusters  = size - mClus;                 //- NETT frees (for resize)
         dfsa->FsUnallocated = exft->FreeClusters * spe;     //- free space (optimize hint)

         dfsSiz8( "Total / Used size : ", total, " ");
         dfsSiz8( "/ ",                   used,  " ");
         TxPrint( "% 5.1lf%%\n",    (double) (100 * ((double) used) / total));

         free = ((ULN64) dfsa->FsUnallocated) * dfsGetSectorSize();
         dfsSiz8( "Freespace / Clust : ", dfsa->FsUnallocated, " ");
         if (free > 999999999LL)
         {
            TxPrint( "/ 0x%8.8lx Cl = %12llu KB\n", exft->FreeClusters, (free / 1000));
         }
         else
         {
            TxPrint( "/ 0x%8.8lx Cl = %9llu bytes\n", exft->FreeClusters, free);
         }

         if (bClus != 0)
         {
            dfsSiz8("Bad sectors       : ", bClus * spe, " =");
            TxPrint("% 8.5lf%%\n", (double) (100 * ((double) bClus) / exft->MapClust));
         }
      }

      if (xClus)
      {
         dfsSiz8( "NOT mapped by FAT : ", xClus * exft->ClustSize, " (wasted space!)\n");
      }
   }
   dfsProgressTerm();
   RETURN (NO_ERROR);
}                                               // end 'dfsExftAllocFat'
/*---------------------------------------------------------------------------*/

