//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// EFAT dump & display Analysis functions
//
// Author: J. van Wijk
//
// JvW  19-06-2014 Initial version, derived from DFSAFAT
// JvW  22-06-2020 Allow recovery of contiguous files if FAT emptied after FORMAT (photo cards)
// JvW  08-04-2021 Use rc CMD_WARNING on FileSaveAs alloc errors, considered OK
//

#include <txlib.h>                              // TX library interface

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfswin.h>                             // DFS help ids
#include <dfswipe.h>                            // wipe functions
#include <dfsutil.h>                            // DFS utility functions
#include <dfsspace.h>                           // DFS  file-space interface
#include <dfstable.h>                           // SLT utility functions

#include <dfsupart.h>                           // FDISK partition functions
#include <dfsafdsk.h>                           // FDISK display & analysis
#include <dfsufdsk.h>                           // FDISK utility functions
#include <dfscfdsk.h>                           // FDISK util, FAT sectors

#include <dfsaefat.h>                           // EFAT display & analysis
#include <dfsuefat.h>                           // EFAT utility functions
#include <dfslefat.h>                           // EFAT SLT functions
#include <dfsosapi.h>                           // OS specific stuff


char dfsEfatSectorTypes[] =
{
   ST_FAT_1,                                    //  I  1st FAT area
   ST_FAT_2,                                    //  I  2nd FAT area
   ST_EFAT,                                     //  I  EFAT 1st sector
   ST_ROOTD,                                    //  I  Root directory area
   ST_DIREC,                                    //     directory area
   0                                            //     string terminating ZERO
};


static DFSAEFAT    efat_anchor =
{
   0,                                           // LSN of 1st Fat
   0,                                           // LSN of 2nd Fat
   0,                                           // LSN of cluster-heap (data)
   0,                                           // LSN of Root directory
   0,                                           // nr of sectors in Root dir
   0,                                           // nr of FATs (usually 2)
   0,                                           // Offset to first FAT
   0,                                           // Nr of sectors per FAT
   0,                                           // Number of sectors
   0,                                           // #clusters, from #sectors
   0,                                           // #clusters, from #fatSectors
   0,                                           // Nr of clusters still free
   0,                                           // valid cluster size
   0,                                           // Maximum size for expand, resize
   0,                                           // Maximum size for current FAT area
   FALSE,                                       // EFAT values faked, no bootsec
// TRUE,                                        // allocation cached cluster
// L32_NULL,                                    // cached cluster number
   0,                                           // FAT in-use, LSN
   {0, NULL, 0, FALSE, 'A'},                    // 1st cache, A
   {0, NULL, 0, FALSE, 'B'},                    // 2nd cache, B
   {0, NULL, 0, 0, 0, FALSE, NULL},             // Bitmap cache structure
   0, 0,                                        // BM cached cluster and mapsize
   0, 0, NULL,                                  // DP cache sizes and array
   {0}                                          // Empty directory entry (cache)
};

       DFSAEFAT   *efat = &efat_anchor;

       char sg_efat[SG_EFAT] = {SV_EFAT};

static  char       *efat_txt[] =
{
   "",
   "Active filesystem : EFAT, specific commands are:",
   "",
   " \\[path-spec]     = find and show ROOT or file/directory relative to root",
   " BOOTSYNC [a][-s] = Read, calculate CRC and synchronise primary/spare boot areas",
   " BSCLEAR          = Reset bad-sector/cluster administration to ZERO bad sectors",
   " DELFIND   [name] = Find deleted files with DIR-entry containing (partial) name",
   " DIRTY      [d|c] = Display/set DIRTY, CLEAN  status for the EFAT filesystem",
   " FATSELECT  [fat] = Select fat to be used (cached), fat = 1 or 2; default is 1",
   " FATSHOW   [*|nr] = Display contents of the FAT from memory, [nr]= # of entries",
   " FATSIM   img [f] = Save FAT table contents for FAT 'f' (1 or 2) to an imagefile",
   " FATWRIM  img [f] = Restore an imagefile with the FAT table to FAT 'f' (1 or 2)",
   " FILEFIND  [name] = Find normal files with DIR-entry containing (partial) name",
   " FINDROOT         = Find and list possible EFAT root directory clusters/sectors",
   " FIXBOOT [1|*|-s] = Fix EFAT bootsector area (0..11) from spare (12..23)",
   " LABEL    [label] = Display/edit 11-char volume label in the ROOT directory",
   "",
   NULL
};

// Close FAT filesystem for analysis and free any resources
static ULONG dfsEfatClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
);

// Close FAT filesystem for Area analysis and free any resources
static ULONG dfsEfatAreaClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
);

// Search and list possible Root directory clusters for an EFAT partition
static ULONG dfsEfatFindEfatRoots               // RET   nr of candidates found
(
   void
);

// Extract BitMap allocation S_SPACE structure and initialize the BitMap cache
static ULONG dfsEfatBitMapInit                  // RET   rc = 0 if type match
(
   ULONG               firstCluster,            // IN    first bitmap cluster
   ULONG               bmByteSize               // IN    size BM file in bytes
);

// Interpret and execute specific EFAT command;
static ULONG dfsEfatCommand
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *none,                    // IN    dummy
   void               *data                     // INOUT dummy
);

// FAT filesystem, identify specified sector
static ULONG dfsEfatIdent
(
   ULN64               di,                      // IN    LSN for sector
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // OUT   sector type
   void               *data                     // IN    sector contents
);

// FAT filesystem, supply sector-type description string
static ULONG dfsEfatStype
(
   ULN64               di,                      // IN    sector type
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // OUT   type description
   void               *data                     // INOUT dummy
);

// FAT filesystem, display sector-contents based on type
static ULONG dfsEfatDispl
(
   ULN64               psn,                     // IN    base psn for sector
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // IN    sector contents
);

// FAT filesystem, display special LSN info
static ULONG dfsEfatDisplayLsnInfo
(
   ULN64               lsn,                     // IN    possible special lsn
   ULN64               info,                    // IN    possible DIR entry nr
   char               *dc,                      // IN    dummy
   void               *data                     // IN    dummy
);

// Display FAT Root- or sub-directory sector(s)
static ULONG dfsEfatDirDisplay
(
   ULONG               psn,                     // IN    base psn for sector
   BYTE                type                     // IN    type of sector
);

// Find LSN* for specified path, starting at Root directory LSN
static ULONG dfsEfatFindPath
(
   ULN64               dummy,
   ULN64               d2,                      // IN    dummy
   char               *path,                    // IN    path specification
   void               *vp                       // OUT   found dir/file Lsn+entry
);

// Find specific name in a (sub) directory SPACE structure, return file+stream
static BOOL dfsEfatDirSpace2Entry                // RET   name found
(
   char               *name,                    // IN    subdir or filename
   ULONG               chunks,                  // IN    nr of space entries
   S_SPACE            *space,                   // IN    space allocation
   ULN64              *lsn,                     // OUT   Dirsector found
   ULONG              *info,                    // OUT   Dir entry-nr found
   S_EFATDIR          *efile,                   // OUT   Directory entry, FILE
   S_EFATDIR          *estream                  // OUT   Directory entry, STREAM
);

// DFS FAT write-file to disk (SaveTo fnode to file)
static ULONG dfsEfatFileSaveAs
(
   ULN64               dse,                     // IN    dirsec/entry LSN
   ULN64               info,                    // IN    info, dir entry
   char               *path,                    // IN    destination path
   void               *ren                      // IN    new name (like 8.3)
);

// Make EFAT usage map based on the bitmap, or the first FAT area
static ULONG dfsEfatAllocMap
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *options,                 // IN    display options
   void               *data                     // OUT   dummy
);

// Make EFAT usage map based on the EFAT bitmap
static ULONG dfsEfatAllocBm
(
   char               *options                  // IN    display options
);

// Make EFAT usage map based on the FAT area
static ULONG dfsEfatAllocFat
(
   char               *options                  // IN    display options
);

/*****************************************************************************/
// Initialize FAT filesystem analysis
/*****************************************************************************/
ULONG dfsEfatInit
(
   char               *fs                       // forced filesystem type
)
{
   ULONG               rc = NO_ERROR;
   TXTM                label;                   // FAT Root label
   TXTM                tbuf;
   TXLN                text;
   BOOL                Check2ndFAT = !TxaOptUnSet('2');
   BYTE                FsysValue = 0;              // display value

   ENTER();

   dfsa->FsCommand          = dfsEfatCommand;
   dfsa->FsClose            = dfsEfatClose;
   dfsa->FsIdentifySector   = dfsEfatIdent;
   dfsa->FsShowType         = dfsEfatStype;
   dfsa->FsDisplaySector    = dfsEfatDispl;
   dfsa->FsFileInformation  = dfsEfatFileInfo;
   dfsa->FsGetAllocSpace    = dfsEfatGetAllocSpace;
   dfsa->FsWriteMetaSpace   = NULL;
   dfsa->FsFindPath         = dfsEfatFindPath;
   dfsa->FsUpdateFileName   = NULL;
   dfsa->FsSetAltBrecLabel  = NULL;
   dfsa->FsMakeBrowseList   = dfsEfatMakeBrowseList;
   dfsa->FsLsnAllocated     = dfsEfatAllocated;
   dfsa->FsLsnSetAlloc      = dfsEfatSetAlloc;
   dfsa->FsSltBuild         = dfsEfatSltBuild;
   dfsa->FsNpBuild          = dfsEfatInitDpCache;
   dfsa->FsDisplayError     = dfsEfatDispError;
   dfsa->FsDisplayLsnInfo   = dfsEfatDisplayLsnInfo;
   dfsa->FsDirIterator      = dfsEfatIterator;
   dfsa->FsDirFileSaveAs    = dfsEfatFileSaveAs;
   dfsa->FsTruncateSize     = NULL;
   dfsa->FsAllocDisplay     = dfsEfatAllocMap;
   dfsa->FsCl2Lsn           = dfsEfatCl2Lsn;
   dfsa->FsLsn2Cl           = dfsEfatLsn2Cl;
   dfsa->FsCmdHelp          = efat_txt;         // FS specific cmdhelp text

   dfsa->FsModeId           = DFS_FS_E_FAT;

   dfsa->Fsi                = efat;
   dfsa->FsSectorTypes      = dfsEfatSectorTypes;

   TRACES(("sizeof S_BOOTR        = % 3lu\n", sizeof(S_BOOTR)));
   TRACES(("sizeof S_EFAT        = % 3lu\n", sizeof(S_EFAT)));
   TRACES(("sizeof DFSEXINFO      = % 3lu\n", sizeof(DFSEXINFO)));
   TRACES(("sizeof DFSF2INFO      = % 3lu\n", sizeof(DFSF2INFO)));

   if ((dfsRead( LSN_BOOTR, EFAT_BOOTSIZE, brec)) == NO_ERROR)
   {
      ULONG            calculatedChecksum  = 0;
      ULONG            storedChecksum      = 0;
      BOOL             bootAreaChecksumsOK = FALSE;

      calculatedChecksum = dfsEfatBootCheckSum(    brec, EFAT_BOOTSIZE * dfsGetSectorSize());
      TxPrint("Calculated BR Crc : 0x%8.8X  ",    calculatedChecksum);
      if ((storedChecksum = dfsEfatStoredChecksum( brec, EFAT_BOOTSIZE * dfsGetSectorSize())) != 0)
      {
         if (storedChecksum == calculatedChecksum)
         {
            TxPrint("Match to every CS value in reference sector\n");
            bootAreaChecksumsOK = TRUE;
         }
         else
         {
            TxPrint("NO match to values in reference sector: %8.8X\n", storedChecksum);
         }
      }
      else
      {
         TxPrint("But reference sector has inconsistent contents!\n");
      }
      dfsa->boot = (S_BOOTR *) brec;

      if ((bootAreaChecksumsOK       == FALSE   ) || // checksum inconsistency
          (dfsa->boot->Signature     != SV_BOOTR) || // invalid bootrecord
          (dfsa->boot->ex.BpsShift   <  9       ) ||
          (dfsa->boot->ex.BpsShift   > 12       ) ||
          (dfsa->boot->ex.SpcShift   > 25       ) ||
          (dfsa->boot->ex.NrOfFats   >  2       ) ||
          (dfsa->boot->ex.FatLength  == 0       )  )
      {
         TxPrint( "Bootsector seems %sinvalid%s, trying (EFAT) spare "
                  "sectors 12..23\n\n", CBR, CNN);
         if ((rc = dfsRead(EFAT_BACKUPBOOT, EFAT_BOOTSIZE, brec)) == NO_ERROR)
         {
            nav.down   = EFAT_BACKUPBOOT;
            switch (dfsIdentifySector(0, 0, brec))
            {
               case ST_BOOTX:
                  TxPrint( "Non-standard boot sector detected ...\n");
               case ST_BOOTR:
                  if (SINF->p == NULL)          // No partition info yet
                  {
                     dfstSetSectorSize( DFSTORE, (USHORT) (1 << dfsa->boot->ex.BpsShift));
                     dfstSetClusterSize(DFSTORE, (USHORT) (1 << dfsa->boot->ex.SpcShift));
                  }

                  bootAreaChecksumsOK = FALSE;
                  calculatedChecksum  = dfsEfatBootCheckSum(   brec, EFAT_BOOTSIZE * dfsGetSectorSize());
                  TxPrint("Calculated BR Crc : 0x%8.8X  ",    calculatedChecksum);
                  if ((storedChecksum = dfsEfatStoredChecksum( brec, EFAT_BOOTSIZE * dfsGetSectorSize())) != 0)
                  {
                     if (storedChecksum == calculatedChecksum)
                     {
                        TxPrint("Match to every CS value in reference sector\n");
                        TxPrint("You may use use 'bootsync -s' to replace the primary bootarea by the spare ...\n");
                        bootAreaChecksumsOK = TRUE;
                     }
                     else
                     {
                        TxPrint("NO match to values in reference sector: %8.8X\n", storedChecksum);
                     }
                  }
                  else
                  {
                     TxPrint("But reference sector has inconsistent contents!\n");
                  }
                  break;

               default:
                  TxPrint("Sector %u is not a valid filesystem BOOT record either!\n", EFAT_BACKUPBOOT);
                  dfsRead( LSN_BOOTR, EFAT_BOOTSIZE, brec); // re-read real bootsector(s)
                  break;
            }
         }
         else
         {
            TxPrint( "Reading EFAT spare-sectors failed!\n");
            dfsRead( LSN_BOOTR, EFAT_BOOTSIZE, brec); // re-read real bootsector(s)
         }
      }

      if ((bootAreaChecksumsOK       == TRUE    ) && // checksums consistent
          (dfsa->boot->Signature     == SV_BOOTR) && // could be valid boot record
         ((dfsa->boot->ex.BpsShift   >= 9       ) &&
          (dfsa->boot->ex.BpsShift   <= 12     )) &&
          (dfsa->boot->ex.SpcShift   <= 25      ) &&
          (dfsa->boot->ex.NrOfFats   <= 2       ) &&
          (dfsa->boot->ex.FatLength  != 0       )  )
      {
         efat->Sect      = dfsa->boot->ex.VolumeLength;
         efat->ClustSize = 1 << dfsa->boot->ex.SpcShift;
         efat->NrOfFats  =      dfsa->boot->ex.NrOfFats;
         efat->Fat1      =      dfsa->boot->ex.FatOffset;

         TRACES(("FAT ClustSize: %4.4hu, Offset: %8.8x\n, efat->ClustSize, efat->Fat1"));

         efat->FatOffset     = dfsa->boot->ex.FatOffset;
         efat->FatSectors    = dfsa->boot->ex.FatLength;
         if (efat->NrOfFats > 1)
         {
            efat->Fat2  = efat->Fat1 + dfsa->boot->ex.FatLength;
         }
         else
         {
            efat->Fat2  = efat->Fat1;
         }
         efat->Roots    = efat->ClustSize;      // minimum size
         efat->ClHeap   = dfsa->boot->ex.ClHeapOffset;
         efat->RawClust = dfsa->boot->ex.ClusterCount;

         efat->Root     = dfsEfatClust2LSN( dfsa->boot->ex.RootCluster);

         efat->MapClust = (ULONG) efat->FatSectors * (dfsGetSectorSize() * 8 / 32); // mapped clusters

         if (dfsEfatDirtyStatusBootrec( FALSE, FALSE) == FALSE)
         {
            dfsa->FsDirtyStatus = DFSTAT_CLEAN;
         }
         else
         {
            dfsa->FsDirtyStatus = DFSTAT_DIRTY;
         }
      }
      else                                      // set sane values
      {
         //- to be refined, may use some values from boot-areas?
         //- and find root directory automatically (for bitmap etc)

         BYTE             ClustSize;
         ULONG            FatSectors;
         TXTM             Signature;            // FAT signature buffer

         Signature[0]   = (char) 0xf8;          // assume harddisk :-)

         TxPrint( "Bootsector seems invalid, using most likely values ...\n");
         if (SINF->p != NULL)                   // partition info avaliable
         {
            TRACES(("PartitionType: %2.2hx\n", SINF->p->partent.PartitionType));

            efat->Sect  = SINF->p->sectors;
         }
         else                                   // use current store size
         {
            efat->Sect  = dfsGetLogicalSize();
         }
         TxPrint( "Searching 1st FAT area ... ");

         TRACES(("dfsa->boot: %8.8x\n", dfsa->boot));

         memcpy( &(Signature[1]), sg_efat, SG_EFAT);
         if ((efat->Fat1 = (dfsLocateSignatureSector(
                              1, 0x1000, Signature, SG_EFAT +1, 0))) == 0)
         {
            Check2ndFAT = FALSE;
            efat->Fat1  = 0x800;
            TxPrint( "not found, using default 0x800\nFIXBOOT will NOT be reliable!\n");
         }
         else
         {
            TxPrint( "found at 0x%4.4lx (LSN % lu)\n", efat->Fat1, efat->Fat1);
         }

         //- to be refined, could do a signature search for UPCASE file (cl 3)
         //- and Root directory (cl 4) to determine the clustersize.
         //- sanity check, must be power of 2 etc



         TxPrint( "Assuming 4 KiB cluster size ...\n");
         ClustSize    = 8;                      // assume 8 sector clusters
         efat->Roots  = ClustSize;              // minimum is 1 cluster ...
         FatSectors   = 131000;                 // 64 GB part, 4KB clusters

         dfsa->boot->ex.RootCluster = 4;
         TxPrint( "\nRoot directory now set at default cluster 4, if this was\n"
                  "a working and formatted EFAT partition, you might need to\n"
                  "use 'findroot', 'setroot' and 'fixboot' to get the right\n"
                  "reference to the root directory (RECOVERY)\n\n");

         efat->RawClust   = (ULONG) (efat->Sect / ClustSize);
         efat->ClustSize  = ClustSize;

         efat->Fat2  = efat->Fat1;              // single FAT

         efat->Root       = efat->Fat2 + FatSectors;
         efat->ClHeap     = efat->Root;
         efat->FatSectors = FatSectors;
         efat->MapClust   = efat->RawClust;
         nav.down         = efat->Root;         // next sector to visit

         TxPrint( "\nRestricted functionality, CHECK and SLT functions disabled!\n"
                  "\nWhen it has been created new, you only need to FORMAT it!\n"
                  "(and ignore most other warnings given here :-)\n\n");
         efat->EfatFaked = TRUE;                // FAT values faked, no bootsec
      }
      dfstSetClusterSize(DFSTORE, efat->ClustSize); // make sure it is set!
      dfsa->FsEntry = efat->Root;               // entry-point in filesystem

      efat->TruncCluster  = efat->RawClust;     // First free cluster, initial
                                                // until calculated in ALLOC

      sprintf( text, "Size of FAT entry : 32  Bits");
      if (dfsEfatRootLabel(dfsa->boot, LSN_BOOTR, label))
      {
         sprintf( tbuf, "%s%s%s", CBC, label, CNN);
      }
      else
      {
         sprintf( tbuf, "--none--");
      }
      TxPrint( "%s    EFAT label,  Root : %s\n", text, tbuf);

      efat->MappedSize = (ULN64) (efat->MapClust - 2) * efat->ClustSize + efat->ClHeap;

      if (dfsa->verbosity > TXAO_QUIET)
      {
         sprintf(   text, "Size, in clusters : 0x%8.8X  ", efat->RawClust);
         dfstrSz64( text, "Number of sectors : ",              efat->Sect, "");
         TxPrint( "%s\n", text);

         sprintf(   text, "Cl. mapped by FAT : 0x%8.8X",   efat->MapClust);
         TxPrint( "%s\n", text);

         strcpy(    text, "");
         dfstrX10(  text, "1st FAT start LSN : ", efat->Fat1,   CBC, "  ");
         if (efat->Fat2 != efat->Fat1)
         {
            dfstrX10(  text, "2nd FAT start LSN : ", efat->Fat2,   CNC, "");
         }
         TxPrint( "%s\n", text);

         strcpy(    text, "");
         dfstrX10(  text, "RootDirectory LSN : ", efat->Root,   CBY, "  ");

         strcpy(   text,  "");
         dfstrX10( text,  "1st DataClust LSN : ", efat->ClHeap, CNG, "");
         TxPrint( "%s\n", text);
      }

      TRACES(("Allocate memory for %u fat-entries\n", (USHORT) DFSEFAT_CACHE));
      if (((efat->CacheA.Value = TxAlloc(DFSEFAT_CACHE, sizeof(ULONG))) != NULL) &&
          ((efat->CacheB.Value = TxAlloc(DFSEFAT_CACHE, sizeof(ULONG))) != NULL)  )
      {
         efat->FatInUse = efat->Fat1;
         if ((rc = dfsEfatGetFAT( 0, &efat->CacheA)) == NO_ERROR)
         {
            ULONG         crc1 = 0;
            ULONG         crc2 = 0;

            dfsa->FsSltRecSize = efat->RawClust +1000; // max records in SLT

            //- Report the FAT dirty status here, from Bootrec
            TxPrint("FAT DIRTY status  : 0x%2.2hx = ", FsysValue);
            if (dfsEfatDirtyStatusBootrec( FALSE, FALSE) == FALSE)
            {
               TxPrint( "CLEAN\n");
            }
            else
            {
               TxPrint( "DIRTY, surface-SCAN/CHKDSK required (MS)\n");
            }

            if (((efat->Fat2 - efat->Fat1) > 0) && (Check2ndFAT == TRUE))
            {
               #if defined (USEWINDOWING)
                  if (txwIsWindow( TXHWND_DESKTOP))
                  {
                     sprintf( text, " Comparing first and second FAT area ... ");
                     txwSendMsg( dfsa->sbwindow, TXWM_STATUS, (TXWMPARAM) text, (TXWMPARAM) cSchemeColor);
                  }
               #endif

               crc2 = dfsEfatCalculateFatCRC( efat->Fat2); // FAT2 before 1, to avoid
               crc1 = dfsEfatCalculateFatCRC( efat->Fat1); // it staying the active!
               TxPrint( "FAT1 CRC-32 value : 0x%8.8x  "
                        "FAT2 CRC-32 value : 0x%s%8.8x%s = %s\n", crc1,
                                     (crc1 == crc2) ? CBG  : CBR,  crc2, CNN,
                                     (crc1 == crc2) ? "OK" : "out of sync!",  crc2, CNN);

               if (TxaOption('a'))              // show NO allocation by default
               {
                  dfsEfatAllocMap( 0, 0, "@", NULL); // get/show Min/Max size
               }
               if (crc1 != crc2)
               {
                  TxPrint("\nWarning: The first and second FAT areas are NOT identical!\n"
                          "This could be the result of a crash, virus, or a damaged disk.\n");
               }
            }
         }
         nav.up   = efat->Fat1;
         nav.xtra = efat->Root;

         if (efat->MapClust < efat->RawClust)   // not all mapped ?
         {
            TxPrint("\nWarning: Number of sectors in the bootsector is LARGER than accounted for\n"
                    "         by the number of FAT-sectors, there are %s%u unmapped clusters!%s\n",
                              CBR, efat->RawClust - efat->MapClust, CNN);
         }

         if (dfsEfatRootGetBmCluster( dfsa->boot, LSN_BOOTR, EFAT_1ST_BITMAP,
                                     &efat->BmCluster, &efat->BmByteSize))
         {
            //- to be refined, for TeFAT also need to consider 2nd bitmap (active if a flag in boot?)
            rc = dfsEfatBitMapInit(   efat->BmCluster,  efat->BmByteSize); // init Bitmap cache
            if (rc == NO_ERROR)
            {
               if (TxaOption('a'))              // show NO allocation by default
               {
                  dfsEfatAllocMap( 0, 0, "@", NULL); // get/show Minimum size
               }
            }
         }

      }
      else
      {
         TxFreeMem( efat->CacheA.Value);
         TxFreeMem( efat->CacheB.Value);
         rc = DFS_ALLOC_ERROR;
      }
      TRACES(( "Fat1:%llx Fat2:%llx InUse:%llx\n", efat->Fat1, efat->Fat2, efat->FatInUse));
      TRACES(( "Sectors: %llx  RawClust: %x  ClHeap: %8.8x  ClustSize: %hu\n",
                efat->Sect,  efat->RawClust,  efat->ClHeap,   efat->ClustSize));
   }
   else
   {
      TxPrint("Reading EFAT primary boot sectors failed!\n");
   }
   RETURN (rc);                                 // when needed
}                                               // end 'dfsEfatInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Close FAT filesystem for analysis and free any resources
/*****************************************************************************/
static ULONG dfsEfatClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   dfsSlTableReset();                           // stop SLT thread and reset
   if (efat->CacheA.Dirty)
   {
      rc = dfsEfatFlushFAT( &efat->CacheA);
   }
   if (efat->CacheB.Dirty)
   {
      rc = dfsEfatFlushFAT( &efat->CacheB);
   }
   TxFreeMem(efat->CacheA.Value);
   TxFreeMem(efat->CacheB.Value);

   rc = dfsEfatBitmapFlush( TRUE);              // flush, and terminate cache
   TxFreeMem( dfsa->findSpace.space);           // free the ALLDIRS SPACE
   efat->dpCacheAlloc = 0;                      // make sure next Init will
   efat->dpCacheSize  = 0;                      // start with an EMPTY cache
   TxFreeMem( efat->Dp);                        // free DirParent cache
   dfsCloseFileSystem();
   RETURN (rc);
}                                               // end 'dfsEfatClose'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Initialize FAT filesystem analysis for Area (FS info in FDISK mode)
/*****************************************************************************/
ULONG dfsEfatAreaInit
(
   void
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   dfsa->FsAreaClose        = dfsEfatAreaClose;
   dfsa->FsAreaLsnAllocated = dfsEfatAllocated;

   dfsa->boot = (S_BOOTR *) brec;
   if ((rc = dfsRead( dfstAreaP2Disk( DFSTORE, LSN_BOOTR), 1, brec)) == NO_ERROR)
   {
      if ((dfsa->boot                != NULL    ) && // could be valid boot record
          (dfsa->boot->Signature     == SV_BOOTR) &&
         ((dfsa->boot->ex.BpsShift   >= 9       ) &&
          (dfsa->boot->ex.BpsShift   <= 12     )) &&
          (dfsa->boot->ex.SpcShift   <= 25      ) &&
          (dfsa->boot->ex.NrOfFats   <= 2       ) &&
          (dfsa->boot->ex.FatLength  != 0       )  )
      {
         efat->Sect  = (ULONG) dfsa->boot->ex.VolumeLength;

         efat->ClustSize     = 1 << dfsa->boot->ex.SpcShift;
         efat->NrOfFats      =      dfsa->boot->ex.NrOfFats;
         efat->Fat1          =      dfsa->boot->ex.FatOffset;

         TRACES(("FAT ClustSize: %4.4hu, Offset: %llx\n, efat->ClustSize, efat->Fat1"));

         efat->FatSectors    = dfsa->boot->ex.FatLength;
         if (efat->NrOfFats > 1)
         {
            efat->Fat2  = efat->Fat1 + dfsa->boot->ex.FatLength;
         }
         else
         {
            efat->Fat2  = efat->Fat1;
         }
         efat->Roots    = efat->ClustSize;      // minimum size
         efat->ClHeap   = dfsa->boot->ex.ClHeapOffset;
         efat->RawClust = dfsa->boot->ex.ClusterCount;

         efat->Root     = dfsEfatClust2LSN( dfsa->boot->ex.RootCluster);

         efat->MapClust = (ULONG) efat->FatSectors * (dfsGetSectorSize() * 8 / 32); // mapped clusters
      }
      TRACES(("Allocate memory for %u fat-entries\n", (USHORT) DFSEFAT_CACHE));
      if (((efat->CacheA.Value = TxAlloc(DFSEFAT_CACHE, sizeof(ULONG))) != NULL) &&
          ((efat->CacheB.Value = TxAlloc(DFSEFAT_CACHE, sizeof(ULONG))) != NULL)  )
      {
         efat->FatInUse = efat->Fat1;
         rc = dfsEfatGetFAT( 0, &efat->CacheA);
         if (rc == NO_ERROR)
         {
            if (dfsEfatRootGetBmCluster( dfsa->boot, dfstAreaP2Disk( DFSTORE, LSN_BOOTR), EFAT_1ST_BITMAP,
                                        &efat->BmCluster, &efat->BmByteSize))
            {
               //- to be refined, for TeFAT also need to consider 2nd bitmap (active is a flag in boot?)
               rc = dfsEfatBitMapInit(   efat->BmCluster,  efat->BmByteSize); // init Bitmap cache
            }
            else
            {
               rc = DFS_BAD_STRUCTURE;
            }
         }
      }
      else
      {
         TxFreeMem( efat->CacheA.Value);
         TxFreeMem( efat->CacheB.Value);
         rc = DFS_ALLOC_ERROR;
      }
   }
   TRACES(( "Fat1:%llx Fat2:%llx InUse:%llx\n", efat->Fat1, efat->Fat2, efat->FatInUse));
   TRACES(( "Sectors: %llx  RawClust: %x  ClHeap: %8.8x  ClustSize: %hu\n",
             efat->Sect,  efat->RawClust,  efat->ClHeap,   efat->ClustSize));
   RETURN (rc);                                 // when needed
}                                               // end 'dfsEfatAreaInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Close FAT filesystem for Area analysis and free any resources
/*****************************************************************************/
static ULONG dfsEfatAreaClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   dfsa->FsAreaClose        = NULL;
   dfsa->FsAreaLsnAllocated = NULL;

   if (efat->CacheA.Dirty)                       // should not happen in area ...
   {
      rc = dfsEfatFlushFAT( &efat->CacheA);
   }
   if (efat->CacheB.Dirty)
   {
      rc = dfsEfatFlushFAT( &efat->CacheB);
   }
   TxFreeMem(efat->CacheA.Value);
   TxFreeMem(efat->CacheB.Value);

   rc = dfsEfatBitmapFlush( TRUE);              // flush, and terminate cache
   RETURN (rc);
}                                               // end 'dfsEfatAreaClose'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Search and list possible Root directory clusters for a FAT32 partition
/*****************************************************************************/
static ULONG dfsEfatFindEfatRoots               // RET   nr of candidates found
(
   void
)
{
   ULONG               rc = 0;                  // function return
   ENTER();

   TxPrint( "\nSearching for possible rootdirectory clusters, (NOT IMPLEMENTED YET)\n\n");

   //- for example code see dfsafat.c

   RETURN (rc);
}                                               // end 'dfsEfatFindEfatRoots'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Extract BitMap allocation S_SPACE structure and initialize the BitMap cache
// Note: Area aware to allow usage from FDISK mode too
/*****************************************************************************/
static ULONG dfsEfatBitMapInit                  // RET   rc = 0 if type match
(
   ULONG               firstCluster,            // IN    first bitmap cluster
   ULONG               bmByteSize               // IN    size BM file in bytes
)
{
   ULONG               rc = 0;                  // rc, sector match
   ULONG               ef = 0;                  // Initialize error flag!

   ENTER();

   efat->Bm.Size = (ULONG) max(efat->ClustSize, (ULONG) 8);
   if ((efat->Bm.Buffer = (BYTE *) TxAlloc( efat->Bm.Size, dfsGetSectorSize())) != NULL)
   {
      rc = dfsEfatAllocChain( firstCluster, 0, &ef, &efat->Bm.Extents, &efat->Bm.Space);
      if (rc != 0)
      {
         //- LimitLsn is the last addressable LSN in the bitmap, +1
         //- and the first bit represents cluster number 2
         efat->Bm.LimitLsn = ((bmByteSize * 8 +2) *   //- highest CLUSTER in bitmap
                               efat->ClustSize)   +   //- convert to highest SECTOR
                               efat->ClHeap;          //- add offset of 1st cluster (2)
         efat->Bm.Dirty    = FALSE;
         efat->Bm.First    = 0;
         rc = dfsSspaceReadFilePart( efat->Bm.Extents,
                                     efat->Bm.Space,
                                     efat->Bm.First,
                                     efat->Bm.Size,
                            (BYTE *) efat->Bm.Buffer);

         #if defined (DUMP)
            if (efat->Bm.Buffer != NULL)
            {
               TRHEXS( 70, efat->Bm.Buffer, 0xc0, "efat->Bm.Buffer");
            }
         #endif
      }
      else
      {
         TxPrint("\nError interpreting $Bitmap allocation, no Bitmap info available!\n");
         efat->Bm.Size = 0;                     // signal no BitMaps present
      }
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   TRACES(( "Bm.Size: %u\n", efat->Bm.Size));
   RETURN (rc);
}                                               // end 'dfsEfatBitMapInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Interpret and execute specific EFAT command;
/*****************************************************************************/
static ULONG dfsEfatCommand
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *none,                    // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;
   ULN64               sn = 0;                  // sector number input
   LONG                nr = 0;
   TXLN                dc;                      // DFS command
   int                 cc;                      // command string count
   char               *c0, *c1, *c2;            // parsed command parts
   TXLN                s1;                      // big temporary string space
   TXLN                s2;                      // big temporary string space
   ULONG               size  = 1;               // default display size
   time_t              stime;

   ENTER();

   time( &stime);                               // timestamp value

   TxaGetArgString( TXA_CUR, 0, 0, TXMAXLN, dc); // dc => cmd from arg 0
   cc = TxaArgCount( );                         // number of parameters
   c0 = TxaArgValue(0);
   c1 = TxaArgValue(1);
   c2 = TxaArgValue(2);

   sn = nav.this;                               // default at current sector

   if ((strcmp(c0, "/?") == 0) ||
       (strcmp(c0,  "?") == 0)  )
   {
      TxShowTxt( efat_txt);
   }
#if defined (NEVER)
   //- to be refined, does not work for EFAT yet
   else if (strcasecmp(c0, "alldirs"  ) == 0)   // test ALL Dir alloc
   {
      S_SPACE         *ads = NULL;
      ULONG            adc = 0;

      TxPrint( "\nLocating ALL directory sectors in the filesystem ...\n");
      if (dfsEfatAllDirs2Space( FALSE, &adc, &ads) == NO_ERROR)
      {
         TxPrint( "Used sectors for ALL directories:\n");
         dfsSspaceDisplay( SD_SUMMARY | SD_TOPLINE | SD_BOTLINE, adc, ads);
         TxFreeMem( ads);                       // free the SPACE
      }
   }
#endif
   else if (strcasecmp(c0, "bootsync"   ) == 0)
   {
      nr = dfsGetMcsNumber( c1, 0);
      if (TxaOption('?') || (nr > 2))
      {
         TxPrint("\nRead bootarea, calculate CRC, and write back to one or both areas\n");
         TxPrint("\n Usage:  %s   [area=0|1|2] [-s]\n\n", c0);
         TxPrint("   area      = Boot area(s) to WRITE, '0' is BOTH areas (default)\n");
         TxPrint("               '1' is PRIMARY (LSN 0), '2' is SPARE area (LSN 12)\n\n");
         TxPrint("   -s        = Read from the SPARE boot-area instead of primary\n\n");
      }
      else
      {
         rc = dfsEfatCrcSyncBootAreas((ULN64) (TxaOption('s') ? EFAT_BACKUPBOOT : LSN_BOOTR), nr);
         if (rc == NO_ERROR)
         {
            TxPrint("\nCalculated CRC and synchronised %s boot area to %s\n",
                      (TxaOption('s')) ? "SPARE" :             "PRIMARY",
                      (nr == 2)        ? "SPARE" : (nr == 1) ? "PRIMARY" : "PRIMARY+SPARE");
         }
      }
   }
   else if (strcasecmp(c0, "fatselect"   ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nSelect to use First or Second FAT area\n");
         TxPrint("\n Usage:  %s  1 | 2\n\n", c0);
      }
      else
      {
         nr = dfsGetMcsNumber( c1, 1);
         dfsEfatSetFatInUse((ULN64) (nr != 2) ? efat->Fat1 : efat->Fat2);
         rc = dfsEfatGetFAT(  0, &efat->CacheA);
         if (rc == NO_ERROR)
         {
            dfsX10("\nSuccessfully read FAT sectors from LSN : ", efat->FatInUse, CBG, "\n");
         }
      }
   }
   else if (strcasecmp(c0, "fatshow"     ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nShow contents of (part of) the FAT area\n");
         TxPrint("\n Usage:  %s  [*|size] [-c:start]\n\n", c0);
         TxPrint("   size      = Number of cluster values to show, '*' ia ALL\n");
         TxPrint("   -c:start  = Start cluster number to show, default is 0\n\n");
      }
      else
      {
         size = 160;
         if (cc > 1)
         {
            if (c1[0] == '*')
            {
               size = efat->MapClust +1;
            }
            else
            {
               size = min((ULONG) atol(c1), efat->MapClust +1);
            }
         }
         dfsEfatShowFAT( size, TxaOptNum('c', NULL, 0), c1);
      }
   }
   else if (strcasecmp(c0, "fatsim"   ) == 0)
   {
      if (cc > 1)
      {
         BOOL          compress = TxaOption('z');

         nr = 1;
         if (cc > 2)
         {
            nr = atol(c2);
         }
         sprintf( dc, "image %s 0x0%llx 0x0%x",
                  c1, (nr == 1) ? efat->Fat1 : efat->Fat2, efat->FatSectors);
         TxaReParseCommand( dc);
         if (compress)
         {
            TxaOptSetItem( "-z");               // set for SIM command too
         }
         rc = DFS_PENDING;
      }
      else
      {
         TxPrint( "\nSave all sectors for the FAT table to a file\n");
         TxPrint( "\nUsage: %s  [-z]  filename  [1 | 2]\n", c0);
      }
   }
   else if (strcasecmp(c0, "fatwrim"   ) == 0)
   {
      if (cc > 1)
      {
         nr = 1;
         if (cc > 2)
         {
            nr = atol(c2);
         }
         sprintf( dc, "wrim %s 0x0%llx 0x0%x",
                  c1, (nr == 1) ? efat->Fat1 : efat->Fat2, efat->FatSectors);
         TxaReParseCommand( dc);
         rc = DFS_PENDING;
      }
      else
      {
         TxPrint( "\nRestore all sectors for the FAT table from a file\n");
         TxPrint( "\nUsage: %s  filename  [1 | 2]\n", c0);
      }
   }
   else if (strcasecmp(c0, "findroot"       ) == 0)
   {
      ULONG         Roots = dfsEfatFindEfatRoots();

      if (Roots != 0)
      {
         TxPrint( "\n%u possible Rootdirectories found, you can use the "
                  "'list -f' command\nto view them, 'export' and 'import'"
                  "to save and restore them and\nfinally 'setroot' and "
                  "'fixboot' to select one for further use.\n", Roots);
      }
      else
      {
         TxPrint( "Warning: No Root directory candidates found!\n");
      }
   }
   else if (strcasecmp(c0, "label"    ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nSet volume label in ROOT-directory\n");
         TxPrint("\nUsage: %s  [label]\n\n"
                 "   label  : new volume label, maximum 11 characters\n"
                 "   -!-    = do not prompt for new value, just set it\n", c0);
      }
      else
      {
         if (cc > 1)
         {
            strcpy( s1, c1);                    // use specified parameter value
         }
         else if (SINF->p)                      // it is a partition
         {
            strcpy( s1, SINF->p->plabel);       // use label determined by readdiskinfo
         }
         else
         {
            strcpy( s1, "");                    // start with empty label
         }
         s1[ EFAT_LABEL_SIZE] = 0;              // truncate to allowed length

         if (dfsa->batch)                       // s1 has new label value
         {
            rc = dfsEfatSetRootLabel( s1);
         }
         else
         {
            #if defined (USEWINDOWING)          // Allow interactive update
            if (!TxaOptUnSet('!') && txwIsWindow( TXHWND_DESKTOP))
            {
               TXLN      prompt;

               sprintf( prompt, "%s\n\nVolume label in ROOT directory: '%s'\n\n"
                                "Specify new %d-character volume label\n",
                                 dfstStoreDesc1( DFSTORE) + 10, s1, EFAT_LABEL_SIZE);

               if (txwPromptBox( TXHWND_DESKTOP, TXHWND_DESKTOP, NULL,
                      prompt, " Set volume label in ROOT directory ",
                      5163, TXPB_MOVEABLE | TXPB_HCENTER | TXPB_VCENTER,
                      EFAT_LABEL_SIZE, s1) != TXDID_CANCEL)
               {
                  TxStrip( s1, s1, ' ', ' ');   // strip leading/trailing spaces
                  rc = dfsEfatSetRootLabel( s1);
               }
            }
            else
            #endif
            {
               if (TxConfirm( 5163, "Write Volume LABEL '%s' to ROOT directory ? [Y/N] : ", s1))
               {
                  rc = dfsEfatSetRootLabel( s1);
               }
            }
         }
      }
   }
   else if (strcasecmp(c0, "setroot"       ) == 0)
   {
      TxPrint("Not implemented for EFAT yet\n");
   }
   else if (strcasecmp(c0, "path"     ) == 0)
   {
      ULONG    entry = 0;

      if (TxaOptSet('e'))                       // e option specified ?
      {
         entry = DFSSNIGET(TxaOptNum('e', "Entry", 0)) | DFSSNINFO;
      }
      else
      {
         sscanf( c2, "%u", &entry);
      }
      sn = dfsGetSymbolicSN( c1, nav.this);     // default current LSN
      if (dfsEfatLsnInfo2Path( sn, entry, s1) == NO_ERROR)
      {
         TxPrint("\nFound path to root: '%s%s%s'\n", CBY, s1, CNN);
      }
      else
      {
         TxPrint("\nNo FAT path to Root found, %s\n",
                (dfsSlTableStatus(NULL) == SLT_READY) ? "not a directory sector?" :
                                                        "need to build SLT first!");
         rc = DFS_NOT_FOUND;
      }
   }
   else if (strcasecmp(c0, "dirty"    ) == 0)
   {
      BOOL             FsysValue = 0;           // display value
      BOOL             FsysDirty = 0;           // update value

      if (cc > 1)
      {
         switch (c1[0])
         {
            case 'c': case 'C': FsysDirty = FALSE;             break;
            case 'd': case 'D':
            default:            FsysDirty = TRUE;              break;
         }
         FsysValue = dfsEfatDirtyStatusBootrec( TRUE, FsysDirty); // update value
      }
      else
      {
         FsysValue  = dfsEfatDirtyStatusBootrec( FALSE, FALSE); // just get BOOTREC value
         TxPrint("\nUsage: %s  clean | dirty\n", c0);
         TxPrint("\n      0 = FS clean");
         TxPrint("\n      1 = FS dirty, CHKDSK required");
      }
      TxPrint("\n\nEFAT status now  : ");
      if (FsysValue == FALSE)
      {
         TxPrint( "CLEAN\n");
      }
      else
      {
         TxPrint( "DIRTY, surface-SCAN/CHKDSK required (MS)\n");
      }
   }
   else if (strcasecmp(c0, "fixboot"  ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nFix the bootsector for an EFAT filesystem\n");
         TxPrint("\n Usage:  %s  [1 | * | -s]\n\n", c0);
         TxPrint("\n    1    = Force a copy of the SPARE bootsector only (LSN 12 -> 0)");
         TxPrint("\n    *    = Force a copy of the whole SPARE bootsector area to LSN 0");
         TxPrint("\n   -s    = Force a copy of the SPARE bootsector area (same as '*')\n\n");
      }
      else
      {
         if ((TxaOption('s') || (c1[0] == '1') || (c1[0] == '*')))
         {
            if ((rc = dfsRead( EFAT_BACKUPBOOT, EFAT_BOOTSIZE, brec)) == NO_ERROR)
            {
               dfsa->boot = (S_BOOTR *) brec;
               switch (dfsIdentifySector(0, 0, brec))
               {
                  case ST_BOOTR:
                     if (SINF->p == NULL)       // No partition info yet
                     {
                        dfstSetSectorSize( DFSTORE, (USHORT) (1 << dfsa->boot->ex.BpsShift));
                        dfstSetClusterSize(DFSTORE, (USHORT) (1 << dfsa->boot->ex.SpcShift));
                     }
                     break;

                  default:
                     TxPrint("Sector %u is not a valid EFAT BOOT record!\n", EFAT_BACKUPBOOT);
                     break;
               }
            }
            else
            {
               TxPrint( "Reading EFAT spare-sectors failed!\n");
            }
         }
         if (((cc <= 1) ||                      // no params, or
               strchr( "1*", c1[0]) != NULL))   // 1 or ALL spare-sector copy
         {
            if (c1[0] == '*')                   // copy whole boot-area
            {
               sprintf( s1, "s (0..11) with the spare copies from sectors 12..23");
            }
            else
            {
               sprintf( s1, " (0) with the spare copy from sector 12");
            }
            if ((dfsa->batch) || (TxConfirm( 5128, "Do you want to replace "
                                 "the EFAT bootsector%s ? [Y/N] : ", s1)))
            {
               if (DFSTORE_WRITE_ALLOWED)
               {
                  rc = dfsWrite( LSN_BOOTR, (c1[0] == '*') ? EFAT_BOOTSIZE : 1, brec);
                  if (rc == NO_ERROR)
                  {
                     TxPrint("\nEFAT boot record updated from spare ...\n");
                     if (SINF->partid != 0)     // was it a partition ?
                     {
                        TxPrint( "Resulting partition after the update:\n\n");
                        sprintf( dc, "part -r %hu", SINF->partid);
                        TxaReParseCommand( dc);
                        rc = DFS_PENDING;       // reread diskinfo & select
                     }
                  }
               }
               else
               {
                  rc = DFS_READ_ONLY;
               }
            }
            else
            {
               rc = DFS_NO_CHANGE;
            }
         }
         else                                   // create new Bootsector ...
         {                                      // with default values
            sn = dfstLSN2Psn( DFSTORE, 0);
            if ((rc = dfsEfatMkBootRec( TRUE, (ULONG) ( sn + stime))) == NO_ERROR)
            {
               //- to be refined, need to update the checksums and SPARE,
               //- and sectors 1..10 should be created/written too?

               TRACES(("EfatMkBoot OK, SINF->partid == %hu\n", SINF->partid));
               if (SINF->partid != 0)           // was it a partition ?
               {
                  TxPrint( "Resulting partition after fixing the bootsector:\n\n");
                  sprintf( dc, "part -r -q %hu#d", SINF->partid);
                  rc = dfsMultiCommand( dc, 0, FALSE, FALSE, TRUE);
               }
            }
            else
            {
               TRACES(("EfatMkBoot failed with rc = %u\n", rc));
            }
         }
         TRACES(("fixboot rc = %u   dc = '%s'\n", rc, dc));
      }
   }
#if defined (NEVER)
   else if (strcasecmp(c0, "format" ) == 0)     // format current object
   {
      if (TxaOption('?'))
      {
         TxShowTxt( cmd_format);                // give usage
      }
      else                                      // calculate limits + defaults
      {                                         // based on size and efat->info
         FAT_FMT_INFO  fi;                      // format parameter structure
         ULONG         stdSects;                // sector count for std SECTORSIZE

         memset( &fi, 0, sizeof( fi));          // initialize to all zero

         fi.sectors      = dfsGetLogicalSize();
         fi.minimumCs_32 = dfsGetSectorSize() / SECTORSIZE; // minimal 1 'real' sector
         fi.allowedBits  = 32;                  // FAT32 always OK

         stdSects = fi.sectors * fi.minimumCs_32; // normalized sectorcount

         if      (stdSects < FAT32_01LIM) fi.defaultCsize =  1;
         else if (stdSects < FAT32_02LIM) fi.defaultCsize =  2;
         else if (stdSects < FAT32_04LIM) fi.defaultCsize =  4;
         else if (stdSects < FAT32_08LIM) fi.defaultCsize =  8;
         else if (stdSects < FAT32_16LIM) fi.defaultCsize = 16;
         else if (stdSects < FAT32_32LIM) fi.defaultCsize = 32;
         else                             fi.defaultCsize = 64;

         if (fi.sectors < FAT12_LIMIT)
         {
            fi.allowedBits += 12;
            fi.minimumCs_12 = ((fi.sectors -1) / (FAT_MAXCLUST & 0xfff)) +1;
         }
         if (fi.sectors < FAT16_LIMIT)
         {
            fi.allowedBits += 16;
            fi.minimumCs_16 = ((fi.sectors -1) / (FAT_MAXCLUST & 0xffff)) +1;
         }

         fi.desiredBits  = TxaOptNum('f', NULL, 0);
         if ((fi.desiredBits & fi.allowedBits) == 0) // illegal desire :-)
         {
            fi.desiredBits  = (fi.allowedBits & 12) ? 12 :
                              (fi.allowedBits & 16) ? 16 : 32;
         }
         sn = dfstLSN2Psn( DFSTORE, 0);
         fi.longFormat   = TxaOption('l');
         fi.desiredRsize = TxaOptNum('r', NULL, (fi.allowedBits & 12) ? 224 : 512);
         fi.desiredCsize = TxaOptNum('c', NULL, 0);
         fi.serial       = TxaOptNum('s', NULL, (ULONG) (sn + stime));
         fi.reservedSect = TxaOptNum('a', NULL, (fi.desiredBits == 32) ? 32 : 1);
         sprintf( s1, "DFS%8.8x", fi.serial);  // default label
         strcpy( fi.label, TxaOptStr('v', NULL, s1));
         TxPClip(fi.label, FAT_N, ' ');         // pad spaces to 11 positions
         strcpy( fi.os   , TxaOptStr('o', NULL, "M"));
         TxStrToUpper( fi.os);

         fi.desiredRsize = ((((fi.desiredRsize -1) & 0xffff) / 16) +1) * 16;

         TRACES(( "FFI sectors: %8.8x  serial: %8.8x\n",    fi.sectors,      fi.serial));
         TRACES(( "FFI allowed:     %4.4hx  desire: %hu\n", fi.allowedBits,  fi.desiredBits));
         TRACES(( "FFI minClSize-12:  %2hu  minC16: %hu\n", fi.minimumCs_12, fi.minimumCs_16));
         TRACES(( "FFI desired-size:  %2hu default: %hu\n", fi.desiredCsize, fi.defaultCsize));
         TRACES(( "FFI Rsize:%hu long:%s label:'%s' os:'%s'\n", fi.desiredRsize,
                                                               (fi.longFormat) ? "YES" : "NO",
                                                                fi.label, fi.os));
         #if defined (USEWINDOWING)
         if (TxaOption('!'))                    // present interactive dialog
         {
            dfsEfatFormatDialog( &fi);
         }
         else                                   // execute the FORMAT
         #endif
         {
            TXTS         osname;
            DFSPARTINFO *p = SINF->p;           // partition info

            if ((p == NULL) && (fi.desiredBits == 32)  && // large floppy FAT32
                (dfstStoreType( DFSTORE) == DFST_PHDISK))
            {
               sprintf( s2, "WARNING: FAT32 on whole disk is NOT supported on OS/2!"
               #if defined (DEV32)
                            "\nTo use FAT32, you need to create a partition first."
               #endif
                            "\n\n");
            }
            else
            {
               strcpy( s2, "");
            }

            dfsMakeFatOsString( fi.os[0], osname);

            fi.desiredCsize = dfsEfatFmtClusterSize( &fi); // valid clustersize

            if ((dfsa->batch) || (TxConfirm( 5131,
                "      FORMAT confirmation for current object:\n\n%s\n\n"
                "To be formatted as FAT%hu with %hu sectors per cluster\n"
                "using bootcode based on '%s'\n\n%s"
                "All data on this volume/partition/disk will be lost!\n"
                "            Proceed with FORMAT ? [Y/N] : ",
                 dfsDriveDescription( s1), fi.desiredBits, fi.desiredCsize, osname, s2)))
            {
               BOOL        ok = TRUE;           // writing allowed/confirmed

               #if defined (DEV32)
                  char      letter = 0;         // driveletter to restore
                  TXTT      volume;             // LVM volumename

                  //- note that SyncDesired checks for NULL pointer too!
                  if ((dfsLvmSyncDesired( p))   && // partition, sync desired
                      (p->lvm.VoluName[0] != 0) && // and a volumename
                      (p->lvm.letter      != 0)  ) // and driveletter
                  {
                     letter = p->lvm.letter;
                     TxCopy( volume, p->lvm.VoluName, LVM_NAME_L);
                     dfsLvmSyncEngine( TRUE, FALSE, volume, 0); // hide volume
                  }
                  else                          // use locking mechanism
                  {
                     ok = dfstCheckWriteAccess( DFSTORE, TRUE);
                  }
               #else
                  ok = dfstCheckWriteAccess( DFSTORE, TRUE);
               #endif

               if (ok)                          // writing allowed/locked
               {
                  rc = dfsEfatFormat( &fi);      // perform actual formatting

                  if (p != NULL)                // if it was a partition
                  {
                     if (rc == NO_ERROR)        // touch up
                     {
                        sprintf( s1, "FAT%2.2hu", fi.desiredBits);
                        dfsSetPartTypeForFS( p, s1); // update in bootsector itself
                        strcpy( p->fsform, s1); // update in internal tables

                        dfsReadDiskInfo( FDSK_QUIET); // re-read all partition info
                     }
                     //- auto reselect partition to refresh info, no screen output
                     dfsMultiCommand( "part .", 0, FALSE, FALSE, FALSE);
                  }

                  #if defined (DEV32)
                  if ((letter != 0) &&          // need to re-assign LVM letter
                      (dfsLvmSyncEngine( TRUE, FALSE, volume, letter) == NO_ERROR))
                  {
                     TxPrint( "Formatted volume  : %c: is available "
                              "for use now, no reboot required.\n", letter);
                  }
                  else
                  #endif
                  {
                     dfstLocking( DFSTORE, DFS_UNLOCK, FALSE,  TRUE);
                     TxPrint( "A reboot may be required before "
                              "the volume can be accessed ...\n");
                  }
               }
               else
               {
                  TxPrint( "Formatting not possible, object is not writable ...\n");
               }
            }
         }
      }
   }
#endif
   else if ((strcasecmp(c0, "bsclear"   ) == 0) || // almost like 'bsimport -clear'
            (strcasecmp(c0, "nobads"    ) == 0)  ) // deprecated
   {
      if (TxaOption('?'))
      {
         TxPrint("\nReset bad-sector administration in both FAT areas\n");
         TxPrint("\n Usage:  %s\n\n", c0);
      }
      else
      {
         rc = dfsEfatResetBadClus();
      }
   }
#if defined (NEVER)
   else if (strcasecmp(c0, "space"       ) == 0)
   {
      sn = dfsGetSymbolicSN( c1, dfsEfatLSN2Clust(nav.this));
      dfsEfatShowAlloc( xxxx);                  // note: would need a STREAM dir entry
                                                // could come from DirEntryCache perhaps
   }
#endif
   else if (strcasecmp(c0, "filefind"  ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nFind normal (not deleted/renamed) files or directories, by name fragment\n");
         TxPrint("\n Usage:  %s  [name]  [-c] [-d-] [-v]\n\n", c0);
         TxPrint("   name      = part or whole filename wanted, NOT a true wildcard,\n"
                 "               do not use any '?' or '*' characters. Limit the name to\n"
                 "               (a part of) any of 15-character fragments of the full name\n"
                 "   -c        = start from current sector, not start of volume\n"
                 "   -D        = search directories only, not files\n"
                 "   -d-       = search outside EXISTING directories too (SLOW!)\n"
                 "   -v        = verbose search, list files while found (SLOW!)\n");
      }
      else
      {
         if (dfsSlTableStatus( NULL) != SLT_READY)
         {
            TxPrint( "\nBuilding SLT for filename lookup, please wait ..."
                     "   (or abort that using <Esc>)\n");
            dfsMultiCommand( "slt 0 1 -v- -d-", 0, TRUE, FALSE, FALSE);
            TxCancelAbort();                    // might have aborted
         }
         if (dfsa->findSpace.space == NULL)     // no DIR areas determined yet
         {
            //- Create the ALLDIRS SPACE structure (for 'find -o:S' etc)
            if (dfsEfatAllDirs2Space( FALSE, &dfsa->findSpace.chunks, &dfsa->findSpace.space) == NO_ERROR)
            {
               sprintf(   s2, "Total # DIR areas : %-8u    ", dfsa->findSpace.chunks);
               dfstrSz64( s2, "Size in DIR areas : ",
                 dfsSspaceSectors( TRUE, dfsa->findSpace.chunks, dfsa->findSpace.space), "");
               TxPrint( "%s\n", s2);
            }
         }
         strcpy( s1, c1);                       // direct copy
         sprintf( dfsa->brdescript, "EFAT %s %s", c0, c1);
         sprintf( dc, "find -t:Dd -o:*$%%]%sMF%s!%s",
                  ( TxaOptUnSet('d')) ? ""  : "S",   //- DIR-areas only (-o:S spaced)
                  ( TxaOption(  'D')) ? "D" : "f",   //- find DIRs only
                  ( TxaOption(  'v')) ? ""  : "Q");  //- verbose or quiet, FAST
         if (strlen( s1))
         {
            strcat( dc, " -u:'");
            strcat( dc,  s1);
            strcat( dc, "'");
         }
         TxPrint("Find file (entry) : %s%s%s\n", CBC, dc, CNN);

         if (!TxaOption('c'))                   // if not 'from current'
         {
            nav.this = efat->Fat2 + efat->FatSectors; // search from end of FAT
         }
         TxaReParseCommand( dc);
         rc = DFS_PENDING;                      // handle translated command
      }
   }
   else if (strcasecmp(c0, "delfind"  ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nFind deleted or renamed) files or directories, by name fragment\n");
         TxPrint("\n Usage:  %s  [name]  [-c] [-d-] [-v]\n\n", c0);
         TxPrint("   name      = part or whole filename wanted, NOT a true wildcard,\n"
                 "               do not use any '?' or '*' characters. Limit the name to\n"
                 "               (a part of) any of 15-character fragments of the full name\n"
                 "   -c        = start from current sector, not start of volume\n"
                 "   -d-       = search outside EXISTING directories too (SLOW!)\n"
                 "   -v        = verbose search, list files while found  (SLOW!)\n");
      }
      else
      {
         if (dfsSlTableStatus( NULL) != SLT_READY)
         {
            TxPrint( "\nBuilding SLT for filename lookup, please wait ..."
                     "   (or abort that using <Esc>)\n");
            dfsMultiCommand( "slt 0 1 -v- -d-", 0, TRUE, FALSE, FALSE);
            TxCancelAbort();                    // might have aborted
         }
         if (dfsa->findSpace.space == NULL)     // no DIR areas determined yet
         {
            //- Create the ALLDIRS SPACE structure (for 'find -o:S' etc)
            if (dfsEfatAllDirs2Space( FALSE, &dfsa->findSpace.chunks, &dfsa->findSpace.space) == NO_ERROR)
            {
               sprintf(   s2, "Total # DIR areas : %-8u    ", dfsa->findSpace.chunks);
               dfstrSz64( s2, "Size in DIR areas : ",
                 dfsSspaceSectors( TRUE, dfsa->findSpace.chunks, dfsa->findSpace.space), "");
               TxPrint( "%s\n", s2);
            }
         }
         strcpy( s1, c1);                       // direct copy
         sprintf( dfsa->brdescript, "EFAT %s %s", c0, c1);
         sprintf( dc, "find -t:Dd -o:*$%%]%sMFx!%s",
                ( TxaOptUnSet('d')) ? "" : "S", // DIR only (-o:S spaced)
                ( TxaOption('v')) ? "" : "Q");  // verbose or quiet, FAST
         if (strlen( s1))
         {
            strcat( dc, " -u:'");
            strcat( dc,  s1);
            strcat( dc, "'");
         }
         TxPrint("Find deleted files: %s%s%s\n", CBC, dc, CNN);

         if (!TxaOption('c'))                   // if not 'from current'
         {
            nav.this = efat->Fat2 + efat->FatSectors; // search from end of FAT
         }
         TxaReParseCommand( dc);
         rc = DFS_PENDING;                      // handle translated command
      }
   }
#if defined (NEVER)
   else if (strcasecmp(c0, "subfind"  ) == 0)
   {
      sprintf( dfsa->brdescript, "EFAT %s %s", c0, c1);
      sprintf( dc, "find -t:");
      switch (c1[0])
      {
         case 's':
            strcat( dc, "s");
            TxPrint("Find subdirectory : from ROOT only : %s%s%s\n", CBC, dc, CNN);
            break;

         case 'd':
            strcat( dc, "d");
            TxPrint("Find subdirectory : non-ROOT only  : %s%s%s\n", CBC, dc, CNN);
            break;

         default:
            strcat( dc, "sd");
            TxPrint("Find subdirectory : ROOT & non-ROOT: %s%s%s\n", CBC, dc, CNN);
            break;
      }
      if (!TxaOption('c'))                      // if not 'from current'
      {
         nav.this = dfsEfatClust2LSN(2);         // search from first possible
      }
      TxaReParseCommand( dc);
      rc = DFS_PENDING;                         // handle translated command
   }
#endif
   else if (strcasecmp(c0, "serial"  ) == 0)
   {
      if (TxaOption('?'))                       // explicit help request
      {
         TxPrint("\nSet volume-serial number in partition boot areas.\n");
         TxPrint("\n Usage:  %s  [serial]  [-!-]\n\n"
                 "   serial : new volume serial number\n"
                 "   -!-    = do not prompt for new value\n", c0);
      }
      else
      {
         if ((rc = dfsSetVolumeSerialNr( !TxaOptUnSet('!'), c1)) == NO_ERROR)
         {
            TxCancelAbort();                    // might have canceled the dialog

            dfsEfatCrcSyncBootAreas( LSN_BOOTR, 0); // sync to BOTH areas

            TxPrint( "Written to PRIMARY and SPARE boot area. PBR contents now:\n\n");
            sprintf( dc, "0");
            rc = dfsMultiCommand( dc, 0, FALSE, FALSE, TRUE);
         }
      }
   }
   else
   {
      rc = DFS_PENDING;                         // cmd not recognized
   }
   RETURN (rc);
}                                               // end 'dfsEfatCommand'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// FAT filesystem, identify specified sector
/*****************************************************************************/
static ULONG dfsEfatIdent
(
   ULN64               lsn,                     // IN    LSN for sector
   ULN64               d2,                      // IN    dummy
   char               *st,                      // OUT   sector type
   void               *sec                      // IN    sector contents
)
{
   ULONG               dr = NO_ERROR;
   BYTE                rc = ST_UDATA;

   ENTER();

   if ((((S_EFAT *)sec)->MediaType == 0xF8) &&
       ((lsn == efat->Fat1) || (lsn == efat->Fat2)))
   {
      rc = ST_EFAT;
   }
   else if ((lsn >= efat->Root) && (lsn != 0) &&
            (lsn < (efat->Root + efat->Roots)))
   {                                            // Root-sector reported as
      rc = ST_DIREC;                            // ST_DIREC for searching!
   }
   else if ((rc = dfsEfatDirClusterType( lsn)) == ST_UDATA)
   {
      //- future expansion :-)
   }
   switch (rc)
   {
      case ST_UDATA:
         dr = DFS_PENDING;
         break;

      default:
         *st = rc;
         break;
   }
   RETURN (dr);
}                                               // end 'dfsEfatIdent'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// FAT filesystem, supply sector-type description string
/*****************************************************************************/
static ULONG dfsEfatStype
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *st,                      // IN    sector type
   void               *data                     // OUT   type description
)
{
   ULONG               dr  = NO_ERROR;
   char               *buf = (char *) data;
   BYTE                tp  = st[0];

   switch (tp)                                  // searchable types
   {
      case ST_EFAT: sprintf(buf,"Start EFAT area "); break;
      case ST_DIREC: sprintf(buf,"Directory data   "); break;
      default:
         switch (tp | ST__INFO)                 // non-searchable ones
         {
            case ST_FAT_1: sprintf(buf,"First  FAT area  "); break;
            case ST_FAT_2: sprintf(buf,"Second FAT area  "); break;
            case ST_ROOTD: sprintf(buf,"Root directory   "); break;
            default:       dr = DFS_PENDING;
         }
         break;
   }
   return (dr);
}                                               // end 'dfsEfatStype'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// FAT filesystem, display sector-contents based on type
/*****************************************************************************/
static ULONG dfsEfatDispl
(
   ULN64               psn,                     // IN    base psn for sector
   ULN64               d2,                      // IN    dummy
   char               *type,                    // IN    type of sector
   void               *data                     // IN    sector contents
)
{
   ULONG               dr = NO_ERROR;
   BYTE                st = (BYTE) *type;

   ENTER();

   switch (st)
   {
      case ST_ROOTD:
      case ST_DIREC:
         dr = dfsEfatDirDisplay( psn, st);
         break;

      default:
         dr = DFS_PENDING;
         break;
   }
   RETURN (dr);
}                                               // end 'dfsEfatDispl'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// FAT filesystem, display special LSN info
/*****************************************************************************/
static ULONG dfsEfatDisplayLsnInfo
(
   ULN64               lsn,                     // IN    possible special lsn
   ULN64               info,                    // IN    possible DIR entry nr
   char               *dc,                      // IN    dummy
   void               *data                     // IN    dummy
)
{
   ULONG               dr = DFS_PENDING;
   TXLN                path;

   ENTER();
   TRACES(("lsn:%8.8x info:%8.8x\n", lsn, info));

   if (info & DFSSNINFO)                        // extra info present
   {
      if (dfsEfatLsnInfo2Path( lsn, info, path) == NO_ERROR)
      {
         TxPrint("\nFound path to root: '%s%s%s'\n", CBY, path, CNN);
      }
      nav.xtra = lsn;                           // keep reference around
      dr = dfsEfatShowDirectoryEntry( lsn, DFSSNIGET(info));
   }
   RETURN (dr);
}                                               // end 'dfsEfatDisplayLsnInfo'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// Display EFAT directory sector(s)
/*****************************************************************************/
static ULONG dfsEfatDirDisplay
(
   ULONG               psn,                     // IN    base psn for sector
   BYTE                type                     // IN    type of sector
)
{
   ULONG               rc  = NO_ERROR;
   ULONG               lsn = dfstPsn2LSN( DFSTORE, psn);
   ULONG               chunks;                  // nr of space entries
   S_SPACE            *space;                   // space allocation

   ENTER();

   rc = dfsEfatDirLsn2Space( lsn, &chunks, &space);
   if (rc == NO_ERROR)
   {
      dfsEfatDirectory( chunks, space);
      TxFreeMem( space);
   }
   RETURN (rc);
}                                               // end 'dfsEfatDirDisplay'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display EFAT directory-block
/*****************************************************************************/
ULONG dfsEfatDirectory                          // RET   nr of entries
(
   ULONG               chunks,                  // IN    nr of space entries
   S_SPACE            *space                    // IN    space allocation
)
{
   ULONG               rc;
   ULONG               usedEntries = 0;         // nr of used entries
   ULONG               delrEntries = 0;         // deleted/renamed entries
   ULONG               usedFileDir = 0;         // Files or Dirs present
   ULONG               delrFileDir = 0;         // deleted/renamed files
   ULONG               chunk;                   // index in space-area
   ULONG               sect;                    // sectors to handle
   ULONG               entry;                   // index in fat directory
   S_EFATDIR           prevEntryBuffer;         // Previous entry, contents
   S_EFATDIR          *prev;                    // Previous entry, pointer
   S_EFATDIR          *this;                    // Fat directory entry
   int                 links = 0;               // down/xtra links given
   int                 ei = -1;                 // entry index (pre-increment)
   int                 cl = TXaNWnZ;            // LSN color
   int                 ac = 0;                  // alternate color fg+bg
   ULN64               data;                    // 1st data sector
   ULN64               dirsec;                  // current dir sector LSN
   ULN64               prevsec;                 // previous dir sector LSN
   TXLN                lfn;                     // Long filename
   ULONG               dirholes = 0;            // empty / free entries
   USHORT              bps = dfsGetSectorSize();
   USHORT              entries = (bps / sizeof(S_EFATDIR));
   BYTE                pendingName = 0;
   USHORT              fragmentSize;            // Filename fragment size
   BYTE               *dirbuf;
   S_EFATDIR          *fatdir;                  // Fat directory sector

   ENTER();

   if ((dirbuf = TxAlloc( 1, bps)) != NULL)     // allocate one sector
   {
      fatdir = (S_EFATDIR *) dirbuf;
      prev   = &prevEntryBuffer;

      TxPrint("\n");
      dfsEfatDirHeader( "Nr    Sector-LSN", 0);   // display TOP header

      TRACES(("Chunks:%u  Sizeof S_EFATDIR: %u\n", chunks, sizeof(S_EFATDIR)));

      if (!TxaOptUnSet('l'))                    // create a new list for DIR
      {
         dfsInitList(0, "-f -P", "-d");         // optimal for menu file-recovery
      }
      strcpy( lfn, "");                         // init first to not-present
      prevsec = space[0].start;                 // safe start value for prev

      for (chunk = 0; (chunk < chunks)  && !TxAbort(); chunk++)
      {
         for (sect = 0; (sect < space[chunk].size)  && !TxAbort(); sect++)
         {
            dirsec  = space[chunk].start + sect;
            if ((rc = dfsRead(dirsec, 1, dirbuf)) == NO_ERROR)
            {
               for (entry = 0; (entry < entries) && !TxAbort(); entry++)
               {
                  this = &(fatdir[entry]);

                  TRLEVX(700,("chunk:%5u  sect:%5u  entry:%5u  EntryType:0x%2.2hhx\n", chunk, sect, entry, this->EntryType));
                  if (this->EntryType & EFAT_DIRBIT_IN_USE)
                  {
                     if (dirholes != 0)         // zeroed entries before this, invalid!
                     {
                        TRACES(("chunk:%5u  sect:%5u  entry:%5u  EntryType:0x%2.2hhx\n", chunk, sect, entry, this->EntryType));
                        TxPrint( "\n%sWARNING: %u Invalid ZERO entries, regular 'dir'"
                                 " will not see rest of directory!\n\n", CNN, dirholes);
                        dirholes = 0;
                     }
                     usedEntries++;
                  }
                  else if (this->EntryType != EFAT_DIR_EODIR)
                  {
                     delrEntries++;
                  }
                  switch (this->EntryType & ~EFAT_DIRBIT_IN_USE) // normal + deleted
                  {
                     case EFAT_DEC_VLABEL:
                        if ((this->EntryType & EFAT_DIRBIT_IN_USE) || TxaOption('D'))
                        {
                           ei++;                // to first/next entry-index
                           ac = (ei % 2) ? TXaBWnC : TXaNWnZ;
                           TxPrint("%s.%05.5u", ansi[ac], ei);
                           TxUnic2Ascii( this->lb.Name, lfn, EFAT_LABEL_SIZE);
                           TxPrint( "%32.32s Label: ", "");
                           TxPrint( "%s%s%s%s\n", ansi[Ccol((CcG | CcI), Ccbg(ac))], lfn,
                                                  ansi[Ccol( CcW,        Ccbg(ac))], CGE);

                           if (!TxaOptUnSet('l'))
                           {
                              dfsAddSI2List( dirsec, entry);
                           }
                        }
                        break;

                     case EFAT_DEC_BITMAP:
                        if ((this->EntryType & EFAT_DIRBIT_IN_USE) || TxaOption('D'))
                        {
                           ei++;                // to first/next entry-index
                           data = dfsEfatClust2LSN( dfsEfatDir2Clust(this));
                           ac = (ei % 2) ? TXaBWnC : TXaNWnZ;
                           TxPrint("%s.%05.5u", ansi[ac], ei);
                           dfsX10( " ", data,   ansi[ac], ansi[ac]);
                           dfsUllDot20( "                        Sfile: alloc-bitmap ", this->bm.FileLength, " ");
                           TxPrint( "%8X%s\n", this->bm.Cluster, CGE);

                           if (!TxaOptUnSet('l'))
                           {
                              dfsAddSI2List( dirsec, entry);
                           }
                        }
                        break;

                     case EFAT_DEC_UPCASE:
                        if ((this->EntryType & EFAT_DIRBIT_IN_USE) || TxaOption('D'))
                        {
                           ei++;                // to first/next entry-index
                           data = dfsEfatClust2LSN( dfsEfatDir2Clust(this));
                           ac = (ei % 2) ? TXaBWnC : TXaNWnZ;
                           TxPrint("%s.%05.5u", ansi[ac], ei);
                           dfsX10( " ", data,   ansi[ac], ansi[ac]);
                           dfsUllDot20( "                        Sfile: upcase-table ", this->up.FileLength, " ");
                           TxPrint( "%8X%s\n", this->up.Cluster, CGE);

                           if (!TxaOptUnSet('l'))
                           {
                              dfsAddSI2List( dirsec, entry);
                           }
                        }
                        break;

                     case EFAT_DEC_FILE:
                        //- no print action, handled by STREAM using prev entry
                        break;

                     case EFAT_ANY_STREAM:
                        if ((prev->EntryType & ~EFAT_DIRBIT_IN_USE) == EFAT_DEC_FILE)
                        {
                           if (this->EntryType & EFAT_DIRBIT_IN_USE) // FILE+STREAM seen
                           {
                              usedFileDir++;
                           }
                           else
                           {
                              delrFileDir++;
                           }
                           strcpy( lfn, "");
                           pendingName = this->st.NameLength;
                           if ((this->EntryType & EFAT_DIRBIT_IN_USE) || TxaOption('D'))
                           {
                              ei++;             // to first/next entry-index
                              data = dfsEfatClust2LSN( dfsEfatDir2Clust(this));
                              ac = (ei % 2) ? TXaBWnC : TXaNWnZ;
                              TxPrint("%s.%05.5u", ansi[ac], ei);
                              switch (links++)
                              {
                                 case 0:  cl = Ccol((CcG | CcI), Ccbg(ac)); nav.down = data; break;
                                 case 1:  cl = Ccol((CcY | CcI), Ccbg(ac)); nav.xtra = data; break;
                                 default: cl = ac;                                           break;
                              }
                              dfsX10( " ", data,   ansi[cl], ansi[ac]);
                              dfsEfatShowDirEntry( prev, this, ac);

                              if (!TxaOptUnSet('l'))
                              {
                                 if (entry == 0) // STREAM is first in sector, FILE is last in PREV
                                 {
                                    dfsAddSI2List( prevsec, entries -1);
                                 }
                                 else
                                 {
                                    dfsAddSI2List( dirsec, entry -1);
                                 }
                              }
                           }
                        }
                        else
                        {
                           TxPrint( "EFAT stream, unexpected at this position\n");
                        }
                        break;

                     case EFAT_ANY_FILENAME:
                        if (pendingName > 0)
                        {
                           if (pendingName > EFAT_NAME_SIZE)
                           {
                              fragmentSize = EFAT_NAME_SIZE;
                              pendingName -= EFAT_NAME_SIZE;
                           }
                           else                 // no more name fragments expected
                           {
                              fragmentSize = pendingName;
                              pendingName  = 0;
                           }
                           if ((this->EntryType & EFAT_DIRBIT_IN_USE) || TxaOption('D'))
                           {
                              TxUnicAppend( this->nm.Name, lfn, fragmentSize);
                              if (pendingName == 0)
                              {
                                 TxPrint( "%s%s%s%s\n", ansi[Ccol((CcY | CcI), Ccbg(ac))], lfn,
                                                        ansi[Ccol( CcW,        Ccbg(ac))], CGE);
                              }
                           }
                        }
                        else
                        {
                           //- Stray name fragments DO happen with file delete/rename (OSX trashcan etc)
                           //- strcpy( lfn, "");
                           //- TxUnicAppend( this->nm.Name, lfn, EFAT_NAME_SIZE);
                           //- TxPrint( "EFAT name fragment '%s', unexpected at this position\n", lfn);
                        }
                        break;

                     case EFAT_DIR_EODIR:       // free entry
                        dirholes++;             // and count them
                        TRLEVX(700,("dirholes now:%u\n", dirholes));
                        break;

                     default:                   // others (deleted/renamed etc)
                        break;
                  }
                  prevEntryBuffer = *this;      // make a persistent copy!
                  prevsec = dirsec;             // and remember its LSN (STREAM to FILE backtracking)
               }
            }
         }
      }
      TxPrint( "%s", CNN);
      if (ei > 12)
      {
         dfsEfatDirHeader( "Nr    Sector-LSN", ei); // display BOTTOM footer
      }
      if (usedEntries != 0)
      {
         TxPrint("\n%6u entries are for %5u files or subdirectories.\n",    usedEntries, usedFileDir);
      }
      if (delrEntries != 0)
      {
         TxPrint("%6u entries are for %5u %sdel%seted (or renamed) files, use 't -D' to display.\n",
                 delrEntries, delrFileDir, (TxaOption('D')) ? CBM : CNN, CNN);
      }
      TxFreeMem( dirbuf);
   }
   RETURN (usedEntries);
}                                               // end 'dfsEfatDirectory'
/*---------------------------------------------------------------------------*/

//- to be refined: extra parameter to allow finding 'deleted' files (only ?)
/*****************************************************************************/
// Find LSN* for specified path, starting at Root directory LSN
/*****************************************************************************/
static ULONG dfsEfatFindPath
(
   ULN64               loud,                    // IN    Show progress
   ULN64               d2,                      // IN    dummy
   char               *path,                    // IN    path specification
   void               *vp                       // OUT   found dir/file + fat entry
)
{
   BOOL                rc  = NO_ERROR;
   ULN64               fs;                      // Current FNODE LSN
   ULN64               ds  = 0;                 // Current DIR LSN
   TXLN                part;
   char               *p   = path;
   int                 l;
   S_EFATDIR           de;                      // directory entry, FILE
   S_EFATDIR           stream;                  // directory entry, STREAM
   ULONG               chunks;                  // chunks in space structure
   S_SPACE            *space;                   // space structure
   DFS_PARAMS         *parm  = (DFS_PARAMS *) vp;
   ULONG               entry = 0;

   ENTER();
   TRACES(("loud:%s  path:'%s'  vp:%p\n", (loud) ? "YES" : "NO", path, vp));

   fs = efat->Root;                             // start LSN
   if (loud)
   {
      dfsX10( "RootDirectory LSN : ", fs,  "", "");
      TxPrint("   find path: '%s'\n", path);
   }

   parm->Lsn    = fs;
   parm->Number = strlen(p) ? DFSSNINFO : 0;    // no info for ROOT!
   while ((rc == NO_ERROR) && strlen(p) && !TxAbort())
   {
      if ((l = strcspn(p, FS_PATH_STR)) != 0)
      {
         strncpy(part, p, l);                   // isolate part
         part[l] = '\0';
         p += l;                                // skip part
         if (*p == FS_PATH_SEP)
         {
            p++;                                // skip '\'
         }
      }
      if (strlen(part))
      {
         if (dfsEfatDirLsn2Space(fs, &chunks, &space) == NO_ERROR)
         {
            if (loud)
            {
               dfsX10( "Base DirBlock LSN : ", fs, "", TREOLN);
            }
            if (dfsEfatDirSpace2Entry(part, chunks, space, &ds, &entry, &de, &stream))
            {
               TRACES(("Found entry: %u in sector:%llx\n", entry, ds));
               if (loud)
               {
                  dfsX10( " - Lsn ", ds, "", "");
                  TxPrint(", entry %2u for '%s'\n", entry, part);
               }
               if (*p == '\0')                          //- end of string, found!
               {
                  parm->Lsn      = ds;                  //- DIR sector number
                  parm->Number   = entry | DFSSNINFO;   //- entry 0..15 in that sector
                  parm->Flag     = TRUE;                //- Size is from DIR-entry
                  parm->byteSize = stream.st.FileLength;
               }
               else
               {
                  if ((de.fl.Attrib & FATTR_DIRECTORY) == 0)
                  {
                     if (loud)
                     {
                        dfsX10( "Object at sector  : ", ds, "", "");
                        TxPrint(", entry 0x%2lx is not a directory\n", entry);
                     }
                     rc = ERROR_PATH_NOT_FOUND;
                  }
               }
               //- Write stream-DIR-entry for new directory in the Directory-cache, so invalid-chain is known
               efat->CachedDirEntry = stream;   // complete DIR-entry structure copy!

               fs = dfsEfatClust2LSN( dfsEfatDir2Clust(&stream)); // LSN for next DIR cluster in path
            }
            else
            {
               if (loud)
               {
                  TxPrint(" - Search failed   for '%s'\n", part);
               }
               rc = ERROR_PATH_NOT_FOUND;
            }
            free( space);                       // free the memory !
         }
      }
      else
      {
         rc = ERROR_PATH_NOT_FOUND;
      }
   }
   RETURN(rc);
}                                               // end 'dfsEfatFindPath'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// Find specific name in a (sub) directory SPACE structure, return file+stream
/*****************************************************************************/
static BOOL dfsEfatDirSpace2Entry               // RET   name found
(
   char               *name,                    // IN    subdir or filename
   ULONG               chunks,                  // IN    nr of space entries
   S_SPACE            *space,                   // IN    space allocation
   ULN64              *lsn,                     // OUT   Dir-sector LSN found
   ULONG              *info,                    // OUT   Dir entry-nr found
   S_EFATDIR          *efile,                   // OUT   Directory entry, FILE
   S_EFATDIR          *estream                  // OUT   Directory entry, STREAM
)
{
   BOOL                found = FALSE;
   ULONG               chunk;                   // index in space-area
   ULONG               sect;                    // sectors to handle
   ULONG               entry;                   // index in fat directory
   ULN64               dirsec;                  // current dir sector LSN
   TXLN                lfn;                     // VFAT long filename
   S_EFATDIR          *fe;                      // Fat directory entry
   USHORT              bps = dfsGetSectorSize();
   USHORT              entries = (bps / sizeof(S_EFATDIR));
   BYTE               *dirbuf;
   S_EFATDIR          *fatdir;                  // Fat directory sector

   ENTER();
   TRARGS(("Name to find : '%s'\n", name));

   if ((dirbuf = TxAlloc( 3, bps)) != NULL)     // allocate sectors
   {
      fatdir = (S_EFATDIR *) dirbuf;

      for ( chunk = 0;
           (chunk < chunks) && !TxAbort() && !found && (space);
            chunk++)
      {
         for ( sect = 0;
              (sect < space[chunk].size)  && !TxAbort() && !found;
               sect++)
         {
            dirsec  = space[chunk].start + sect;
            if (dfsRead(dirsec, 3, dirbuf) == NO_ERROR) // read 3, any name entries
            {
               for ( entry = 0;
                    (entry < entries) && !TxAbort() && !found;
                     entry++)
               {
                  fe = &(fatdir[entry]);

                  switch (fe->EntryType & ~EFAT_DIRBIT_IN_USE) // normal + deleted
                  {
                     case EFAT_DEC_VLABEL:
                     case EFAT_DEC_BITMAP:
                     case EFAT_DEC_UPCASE:
                     case EFAT_DEC_FILE:
                        dfsEfatEntry2Name( fe, lfn); // get real or faked file name
                        if (strcasecmp( lfn, name) == 0)
                        {
                           if (lsn)
                           {
                              *lsn = dirsec;
                           }
                           if (info)
                           {
                              *info = entry;
                           }
                           if (efile)
                           {
                              *efile = *fe;
                           }
                           if (estream)
                           {
                              *estream = *(fe +1); // stream is next entry
                           }
                           found = TRUE;
                        }
                        break;

                     default:
                        break;
                  }
               }
            }
         }
      }
      TxFreeMem( dirbuf);
   }
   BRETURN(found);
}                                               // end 'dfsEfatDirSpace2Entry'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// DFS FAT write-file to disk (SaveTo to file)
/*****************************************************************************/
static ULONG dfsEfatFileSaveAs
(
   ULN64               dse,                     // IN    dirsec/entry LSN
   ULN64               info,                    // IN    info, dir entry
   char               *path,                    // IN    destination path
   void               *recp                     // INOUT recovery parameters
)
{
   ULONG               rc  = NO_ERROR;
   S_RECOVER_PARAM    *param = (S_RECOVER_PARAM *) recp;
   S_EFATDIR          *efile;                   // Fat dir-entry ptr, FILE
   S_EFATDIR          *stream;                  // STREAM (or bitmap/upcase)
   DFSISPACE           isp;                     // allocation SPACE info
   ULONG               ef = 0;                  // allocation error flag
   BYTE               *sb;
   S_EFATDIR          *fatdir;                  // Efat directory sector
   BOOL                isDir = FALSE;           // regular entry for a directory
   int                 sanity;

   ENTER();
   TRARGS(("Dirsec LSN : %8.8X, entry:%u to path: '%s'\n", dse, DFSSNIGET(info), path));

   if ((sb = TxAlloc(5, dfsGetSectorSize())) != NULL) // 2 extra sectors BEFORE and AFTER
   {
      if ((rc = dfsRead( dse - 2, 5, sb)) == NO_ERROR)
      {
         fatdir = (S_EFATDIR *) (sb + (2 * dfsGetSectorSize()));
         efile  = &(fatdir[DFSSNIGET(info)]);
         stream = efile;                        // stream is same entry for bitmap/upcase
         TRACES(("fatdir:%8.8x efile:%8.8x for entry %u Type: 0x%2.2hx\n",
                  fatdir,       efile, DFSSNIGET(info), stream->EntryType));
         switch (efile->EntryType & ~EFAT_DIRBIT_IN_USE)
         {
            case EFAT_ANY_STREAM:
            case EFAT_ANY_FILENAME:            // need to search backwards to FILE entry
               sanity = 2 * dfsGetSectorSize() / sizeof(S_EFATDIR);
               while (((efile->EntryType & ~EFAT_DIRBIT_IN_USE) != EFAT_DEC_FILE) && (sanity--))
               {
                  efile--;                      // move backwards over 1 entry
               }
               if ((efile->EntryType & ~EFAT_DIRBIT_IN_USE) != EFAT_DEC_FILE)
               {
                  TRACES(("No FILE entry found (backward from stream/filename)\n"));
                  rc = DFS_BAD_STRUCTURE;
                  break;                        // no FILE entry found, fail
               }
            case EFAT_DEC_FILE:
               stream = efile +1;               // advance to actual stream entry
               isDir  = ((efile->fl.Attrib & FATTR_DIRECTORY) != 0);
            case EFAT_DEC_BITMAP:
            case EFAT_DEC_UPCASE:
               if (dfsEfatAllocStream( stream, &ef, &isp.chunks, &isp.space) != 0)
               {
                  TRACES(("Using regular ExFAT allocation info\n"));
               }
               else if ((stream->st.Cluster    >= 2)  && (stream->st.Cluster < efat->MapClust) &&
                        (stream->st.FileLength != 0)) // regular, non-empty file
               {
                  //- assume contiguous allocation, to allow SaveAs after FORMAT, for example Photo-cards
                  isp.chunks = 1;               // make available to SpaceFileSaveAs
                  isp.space  = TxAlloc( 1, sizeof(S_SPACE));

                  isp.space->start = dfsEfatClust2LSN( stream->st.Cluster);
                  isp.space->size  = (stream->st.FileLength + dfsGetSectorSize() - 1) / dfsGetSectorSize();

                  param->noAllocCheck = TRUE;   // force no-alloc-check on save, avoiding errors

                  TRACES(("ASSUME Contiguous cluster: 0x%8.8x = LSN: 0x%llx size %llu = %u sectors\n",
                                      stream->st.Cluster, isp.space->start, stream->st.FileLength , isp.space->size));

                  TxPrint( " ASSUME Contiguous!"); // prefix final warning text with extra hint
                  rc = DFS_CMD_WARNING;         // trigger a WARNING fragment in output line
               }
               else
               {
                  if (ef != 0)                  // inconsistent FAT values
                  {
                     dfsEfatDispError( ef, 0, "\nWarning: ", NULL);
                     rc = DFS_BAD_STRUCTURE;
                  }
                  else
                  {
                     rc = DFS_ALLOC_ERROR;
                  }
               }
               if ((rc == NO_ERROR)       ||    // we have an S_SPACE allocation
                   (rc == DFS_CMD_WARNING) )    // even when allocation errors present
               {
                  TXLN       fname;             // destination base filename
                  TXLN       fullfn;            // unique filename prefix

                  if ((param->newname == NULL) || (*param->newname == 0)) // no newname present
                  {
                     dfsEfatEntry2Name( efile, fname);
                  }
                  else                          // rename specified
                  {
                     strcpy( fname, param->newname);
                  }
                  if (param->unique)            // force unique naming
                  {                             // on PATH and FILE components
                     sprintf(fullfn, "%8.8llX_%2.2llX_%s", dse, info, fname);
                     strcpy( fname, fullfn);    // and move back to fname
                  }
                  if (param->recFname)
                  {
                     strcpy( param->recFname, fname); // return full base-filename
                  }

                  isp.byteSize = stream->st.FileLength;
                  rc = dfsSspaceFileSaveAs( &isp, isDir,
                                          ((efile->EntryType & EFAT_DIRBIT_IN_USE) == 0),
                                            param->noAllocCheck, param->name83, path, fname, fullfn);
                  if ((rc == NO_ERROR)       || // set original timestamps
                      (rc == DFS_CMD_WARNING) ) // even when allocation errors present
                  {
                     time_t      mod = txOS2FileTime2t( &efile->fl.DtModify.u, &efile->fl.TmModify.u);
                     time_t      cre = txOS2FileTime2t( &efile->fl.DtCreate.u, &efile->fl.TmCreate.u);
                     time_t      acc = txOS2FileTime2t( &efile->fl.DtAccess.u, &efile->fl.TmAccess.u);

                     TRACES(("Set original date/time for: '%s'\n", fullfn));

                     TxSetFileTime( fullfn, &cre, &acc, &mod); // set time on full-fn from SaveAs

                     if (param->recFname)
                     {
                        strcpy( param->recFname, fullfn);
                     }
                  }
                  TxFreeMem( isp.space);
               }
               break;

            default:
               TRACES(("Invalid Entrytype\n"));
               rc = DFS_BAD_STRUCTURE;
               break;
         }
      }
      TxFreeMem( sb);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN(rc);
}                                               // end 'dfsEfatFileSaveAs'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Make EFAT usage map based on the bitmap, or the first FAT area
/*****************************************************************************/
static ULONG dfsEfatAllocMap
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *options,                 // IN    display options
   void               *data                     // OUT   dummy
)
{
   if (TxaOption( 'f'))                         // base map display on FAT
   {
      return dfsEfatAllocFat( options);
   }
   else                                         // base map display on bitmap
   {
      return dfsEfatAllocBm( options);
   }
}                                               // end 'dfsEfatAllocMap'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Make EFAT usage map based on the EFAT bitmap
/*****************************************************************************/
static ULONG dfsEfatAllocBm
(
   char               *options                  // IN    display options
)
{
   ULONG               i;                       // index in data-area
   ULONG               b;                       // byte-index for one mapchar
   ULONG               cClus = 0;               // Clusters in current char
   ULONG               lClus = 0;               // Clusters in current line
   ULONG               mClus = 0;               // Clusters in current map
   ULONG               uClus = 1;               // Last cluster that is in use
   ULONG               l;                       // line number, 0 based
   ULONG               a;                       // index in display-array
   ULN64               perc;                    // percentage
   TX1K                ascii;                   // display-array
   ULN64               lbase = 0;               // current line base LSN
   ULONG               spe;                     // sectors per entry
   ULONG               acl;                     // ascii alloc chars per line
   ULONG               cpc;                     // clusters per display-char
   ULONG               spc;                     // sectors per display-char
   ULONG               size;                    // nr of clusters to map
   BOOL                verbose = (*options != '@');
   ULONG               bsSmart;
   ULN64               unallocSmart = 0;        // Size of THIS unallocated area

   ENTER();

   dfsa->FsUnallocSmart = 0;                    // will be recalculated now
   bsSmart = dfsGetBufferSize( DFSOPTIMALBUF, DFSMAXBUFSIZE); // for SMART prediction

   spe   = efat->ClustSize;
   size  = efat->RawClust;                      // Cluster 2 .. last-valid

   acl   = dfsAllocCharsPerLine( 'c');
   cpc   = dfsAllocItemsPerChar( 'l', options, size, acl);
   spc   = cpc * spe;
   dfsProgressInit( 0, size * spe, 0, "Get ALLOCATION info, at Sector:", dfstStoreDpid( DFSTORE), DFSP_STAT, 0);

   if (verbose)
   {
      dfsSz64("Size for one line : ", (ULN64) acl * spc, "  ");
      TxPrint("with %3lu characters,", acl);
      dfsSiz8(" size : ", spc, "\n");
      TxPrint(" %16.16s = Allocation grade, empty to full.     ", mapchar);
      dfsSiz8(" SmartBuf size : ", bsSmart, "\n");
   }
   TxPrint(   "          %s%*.*s%s\n", CBC, acl, acl, BLIN, CNN);

   TxPrint("%s0x00000000%s%sB%s %5u KiB  ", CNZ, CBC, CnB, CnY, efat->FatOffset / 2);

   if (efat->NrOfFats == 1)
   {
      TxPrint("%s%*.*s Fat area 32-bit entries, size:%8u KiB        %s%s",
                  CnM, (acl - 64), (acl - 64), "", (ULONG) (efat->FatSectors / 2), CBC, CNN);
   }
   else
   {
      TxPrint("%s%*.*s Fat1 32-bit %7u KiB  %s%*.*s Fat2 32-bit %7u KiB %s%s",
                  CnM, ((acl - 64) / 2), ((acl - 64) / 2), "", (ULONG) (efat->FatSectors / 2),
                  CnG, ((acl - 64) / 2), ((acl - 64) / 2), "", (ULONG) (efat->FatSectors / 2), CBC, CNN);
   }
   TxPrint("\n");

   for (i=0, a=0, l=0; (i < size) && (!TxAbort());)
   {
      if (a == 0)
      {
         memset(ascii, 0, TXMAXLN);
         lClus = 0;
         lbase = dfsEfatClust2LSN(i + 2);       // first used clusternr is 2!
      }
      for (cClus=0, b=0; (b < cpc) && (i < size); b++, i++)
      {
         if (dfsEfatBitmapCache(i + 2, NULL))   // cluster in use ?
         {
            if (unallocSmart != 0)              // first in-use after unallocated?
            {
               if (unallocSmart >= bsSmart)                  //- if at least ONE smart buffer
               {
                  unallocSmart -= (bsSmart / 2);             //- compensate for alignment issues
                  unallocSmart -= (unallocSmart % bsSmart);  //- clip to whole smart buffer size

                  dfsa->FsUnallocSmart += unallocSmart;      //- add this area to total
               }
               unallocSmart = 0;                // reset for next area to come
            }
            cClus++;
            uClus = i + 2;                          // last in-use one sofar
         }
         else                                   // unallocated, Smart predict
         {
            unallocSmart += spe;                // add to size THIS area
         }
      }
      lClus += cClus;

      dfsProgressShow( (i * spe) + spc, 0, 0, NULL);

      TRACES(("lClus 2nd Loop i: %u, b:%u = %u\n", i, b, lClus));
      ascii[a] = (char) ((cClus == 0) ? ' ' : mapchar[cClus * 8 / cpc]);
      TRACES(("Character for pos %u = '%c'\n", a, ascii[a]));
      a++;
      if ((i && ((i%(acl*cpc)) == 0)) || (i >= size))
      {
         perc  = (ULN64)(100*lClus / (a*cpc));
         TRACES(("Percentage 2nd Loop i: %u = %llu\n", i, perc));
         dfsX10( CNZ, lbase, CNZ, "");
         TxPrint("%s%s%s%s", CBC, CNC, ascii, CNN);
         if (a == acl)
         {
            TxPrint("%s%s% 3llu%%\n", CBC, CNN, perc);
         }
         else
         {
            TxPrint("%s%.*s%s% 3llu%%\n", CBC, (int) (acl-a-1), BLIN, CNN, perc);
         }
         if (i < size)
         {
            a = 0;                              // keep a value on last line
         }
         mClus += lClus;
         l++;
      }
   }
   if (unallocSmart != 0)                           //- unalloc area pending?
   {
      if (unallocSmart >= bsSmart)                  //- if at least ONE smart buffer
      {
         unallocSmart -= (bsSmart / 2);             //- compensate for alignment issues
         unallocSmart -= (unallocSmart % bsSmart);  //- clip to whole smart buffer size

         dfsa->FsUnallocSmart += unallocSmart;      //- add this area to total
      }
   }
   TxPrint("          %s%.*s%s\n", CBC, (USHORT) a, BLIN, CNN);

   if (TxAbort())
   {
      dfsa->FsUnallocSmart = 0;                 // signal info NOT available!
      TxPrint( "\nAllocation check and display aborted! Used/Free info not available.\n");
   }
   else
   {
      if (size != 0)                            // avoid devide by 0
      {
         ULN64            used = (ULN64) mClus * spe;

         dfsa->FsUnallocated   = size * spe - used; // free space (optimize hint)

         dfsSz64("Unallocated sects : ", dfsa->FsUnallocated, " ");
         dfsSz64("SmartUse ", dfsa->FsUnallocSmart, " ");
         TxPrint("=% 5.1lf%%\n", (double) (100.0 * ((double) dfsa->FsUnallocSmart) / (double) (size * spe)));

         dfsSz64("Allocated sectors : ", used, " ");
         dfsSz64("of total ",      size * spe, " ");
         TxPrint("=% 5.1lf%%\n", (double) (100.0 * ((double) mClus) / (double) size));
      }
      dfsa->FsTruncPoint = dfsEfatClust2LSN( uClus +1); // first free (truncate)
      dfsa->FsLastInUse  = dfsa->FsTruncPoint -1;
      //- dfsSz64( "Minimum vol. size : ", dfsa->FsTruncPoint, " (for  Resizing)\n");
   }
   dfsProgressTerm();
   RETURN (NO_ERROR);
}                                               // end 'dfsEfatAllocBm'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Make EFAT usage map based on the FAT area (informational only?)
/*****************************************************************************/
static ULONG dfsEfatAllocFat
(
   char               *options                  // IN    display options
)
{
   ULONG               i;                       // index in data-area
   ULONG               b;                       // byte-index for one mapchar
   ULONG               cClus = 0;               // Clusters in current char
   ULONG               lClus = 0;               // Clusters in current line
   ULONG               mClus = 0;               // Clusters in current map
   ULONG               bClus = 0;               // Clusters marked as bad
   ULONG               uClus = 1;               // Last cluster that is in use
   ULONG               xClus = 0;               // Clusters that are NOT mapped
   ULN64               perc  = 0;               // percentage
   ULONG               l;                       // line number, 0 based
   ULONG               a;                       // index in display-array
   TX1K                ascii;                   // display-array
   char                mChar;                   // single character to display
   ULONG               spe;                     // sectors per entry
   ULONG               acl;                     // ascii alloc chars per line
   ULONG               cpc;                     // clusters per display-char
   ULONG               spc;                     // sectors per display-char
   ULONG               size;                    // nr of clusters to map
   ULONG               value;                   // FAT entry value
   ULONG               nonClust = efat->ClHeap;
   BOOL                verbose  = (*options != '@');
   ULONG               bsSmart;
   ULN64               unallocSmart = 0;        // Size of THIS unallocated area
   BOOL                oneLineProgress = FALSE; // alloc in one line, output per character

   ENTER();

   dfsa->FsUnallocSmart = 0;                    // will be recalculated now
   bsSmart = dfsGetBufferSize( DFSOPTIMALBUF, DFSMAXBUFSIZE); // for SMART prediction

   spe   = efat->ClustSize;
   size  = efat->RawClust - 2;                   // #clusters minus first two

   acl   = dfsAllocCharsPerLine( 'c');
   cpc   = dfsAllocItemsPerChar( 'l', options, size, acl);
   if (cpc >= (size / acl))                     // will be a single line
   {
      oneLineProgress = TRUE;                   // one char at a time, no ascii string used
   }
   spc = cpc * spe;
   dfsProgressInit( 0, size * spe, 0, "Get ALLOCATION info, at Sector:", dfstStoreDpid( DFSTORE), DFSP_STAT, 0);

   if (verbose)
   {
      dfsSz64("Size for one line : ", (ULN64) acl * spc, "  ");
      TxPrint("with %3lu characters,", acl);
      dfsSiz8(" size : ", spc, "\n");
      TxPrint(" %16.16s = Allocation grade, empty to full.     ", mapchar);
      dfsSiz8(" SmartBuf size : ", bsSmart, "\n");
      if (efat->MapClust < efat->RawClust)
      {
         TxPrint( " %-16.16s = Filesystem area NOT mapped by the FAT!\n", "X");
      }
   }
   TxPrint(   "          %s%*.*s%s\n", CBC, acl, acl, BLIN, CNN);

   TxPrint("%s0000000000%s%sB%s %5u KiB", CNZ, CBC, CnB, CnY, efat->FatOffset / 2);

   if (efat->NrOfFats == 1)
   {
      TxPrint("%s%*.*s         Fat area 32-bit entries, size:%8u KiB            %s%s",
                  CnM, (acl - 64), (acl - 64), "", (ULONG) (efat->FatSectors / 2), CBC, CNN);
   }
   else
   {
      TxPrint("%s%*.*s Fat1 32-bit, size:%7u KiB  %s%*.*s Fat2 32-bit, size:%7u KiB %s%s",
                  CnM, ((acl - 64) / 2), ((acl - 64) / 2), "", (ULONG) (efat->FatSectors / 2),
                  CnG, ((acl - 64) / 2), ((acl - 64) / 2), "", (ULONG) (efat->FatSectors / 2), CBC, CNN);
   }
   TxPrint("\n");
   dfsX10( CNZ, dfsEfatClust2LSN(2), CNZ, "");  // display address for first block
   TxPrint("%s%s", CBC, CNC);

   for (i=0, a=0, l=0; (i < size) && (!TxAbort()) && (efat->CacheA.Value);)
   {
      if (a == 0)
      {
         memset(ascii, 0, TXMAXLN);
         lClus = 0;
      }
      for (cClus=0, b=0; (b < cpc) && (i < size); b++, i++)
      {
         if (i < efat->MapClust)                // mapped in the FAT area ?
         {
            value = dfsEfatValue(i+2);          // one FAT entry value
            if (value > EFAT_MAXCLUST)
            {
               cClus++;                         // cluster in use or bad
               if (value < EFAT_MEDIA_CL)
               {
                  bClus++;                      // bad sectors
               }
               else                             // EOF cluster
               {
                  uClus = i+2;                  // last in-use one sofar
               }
            }
            else
            {
               if (value != 0)
               {
                  if (unallocSmart != 0)        // first in-use after unallocated?
                  {
                     if (unallocSmart >= bsSmart)                  //- if at least ONE smart buffer
                     {
                        unallocSmart -= (bsSmart / 2);             //- compensate for alignment issues
                        unallocSmart -= (unallocSmart % bsSmart);  //- clip to whole smart buffer size

                        dfsa->FsUnallocSmart += unallocSmart;      //- add this area to total
                     }
                     unallocSmart = 0;          // reset for next area to come
                  }
                  cClus++;                      // count cluster in use
                  uClus = i+2;                  // last in-use one sofar
               }
               else                             // unallocated, Smart predict
               {
                  unallocSmart += spe;          // add to size THIS area
               }
            }
         }
         else
         {
            xClus++;
         }
      }
      lClus += cClus;

      dfsProgressShow( (i * spe) + spc, 0, 0, NULL);

      TRACES(("lClus 2nd Loop i: %u, b:%u = %u\n", i, b, lClus));
      if (i < efat->MapClust)
      {
         mChar = (char) ((cClus == 0) ? ' ' : mapchar[cClus * 8 / cpc]);
      }
      else
      {
         mChar = 'X';                           // non mapped clusters!
      }
      if (oneLineProgress == TRUE)
      {
         TxPrint( "%c", mChar);                 // slow, acts as progress bar
      }
      else
      {
         ascii[a] = mChar;
      }
      a++;
      if ((i && ((i%(acl*cpc)) == 0)) || (i >= size))
      {
         if (oneLineProgress == FALSE)
         {
            TxPrint( "%s", ascii);              // display accumulated chars (fast)
         }
         perc  = (ULN64)(100*lClus / (a*cpc));
         if (a == acl)
         {
            TxPrint("%s%s% 3llu%%\n", CBC, CNN, perc);
         }
         else
         {
            TxPrint("%s%.*s%s% 3llu%%\n", CBC, (int) (acl-a-1), BLIN, CNN, perc);
         }
         if (i < size)
         {
            dfsX10( CNZ, dfsEfatClust2LSN(i+2), CNZ, "");
            TxPrint("%s%s", CBC, CNC);
            a = 0;                              // keep a value on last line
         }
         mClus += lClus;
         l++;
      }
   }
   if (unallocSmart != 0)                           //- unalloc area pending?
   {
      if (unallocSmart >= bsSmart)                  //- if at least ONE smart buffer
      {
         unallocSmart -= (bsSmart / 2);             //- compensate for alignment issues
         unallocSmart -= (unallocSmart % bsSmart);  //- clip to whole smart buffer size

         dfsa->FsUnallocSmart += unallocSmart;      //- add this area to total
      }
   }
   TxPrint("          %s%.*s%s\n", CBC, (USHORT) a, BLIN, CNN);

   if (TxAbort())
   {
      dfsa->FsUnallocSmart = 0;                 // signal info NOT available!
      TxPrint( "\nAllocation check and display aborted! Used/Free info not available.\n");
   }
   else
   {
      if (size != 0)                            // avoid devide by 0
      {
         ULN64            total = (ULN64) size  * spe + nonClust +1; //- include bootsec
         ULN64            used  = (ULN64) mClus * spe + nonClust -1; //- used incl FAT areas
         ULN64            free;

         efat->FreeClusters  = size - mClus;                     //- NET frees  (for resize)
         dfsa->FsUnallocated = (ULN64) efat->FreeClusters * spe; //- free space (optimize hint)

         dfsSz64("Unallocated sects : ", dfsa->FsUnallocated, " ");
         dfsSz64("SmartUse ", dfsa->FsUnallocSmart, " ");
         TxPrint("=% 5.1lf%%\n", (double) (100.0 * ((double) dfsa->FsUnallocSmart) / (double) total));

         dfsSz64("Allocated sectors : ", used,  " ");
         dfsSz64("of total ",            total, " ");
         TxPrint("=% 5.1lf%%\n",    (double) (100 * ((double) used) / (double) total));

         free = ((ULN64) dfsa->FsUnallocated) * dfsGetSectorSize();
         dfsSz64( "Unallocated space : ", dfsa->FsUnallocated, " ");
         dfsUllDot20(   " = ", free, " bytes free\n");

         if (bClus != 0)
         {
            dfsSz64("Bad sectors       : ", (ULN64) bClus * spe, " =");
            TxPrint("% 8.5lf%%\n", (double) (100 * ((double) bClus) / efat->MapClust));
         }
      }

      if (xClus)
      {
         dfsSz64( "NOT mapped by FAT : ", (ULN64) xClus * efat->ClustSize, " (wasted space!)\n");
      }
   }
   dfsProgressTerm();
   RETURN (NO_ERROR);
}                                               // end 'dfsEfatAllocFat'
/*---------------------------------------------------------------------------*/

