//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// OS/2 DUMPFS dump & display Analysis functions
//
// Author: J. van Wijk
//
// JvW  03-06-2014 Initial version, derived from SWAP

#include <txlib.h>                              // TX library interface

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions

#include <dfsadump.h>                           // DUMP display & analysis

#define DUMP_STARTSEC  0L                       // DUMP start sector
#define DUMP_SUPER     0L                       // DUMP info  sector (same!)
#define DUMP_SSECTS    1L                       // one sector

char dfsDumpSectorTypes[] =
{
   ST_DUMPFS,                                   //     DUMP super-block
   0
};

static DFSADUMPFS dfsdumpfs_anchor =
{
   NULL,
   0
};

DFSADUMPFS   *dumpfs = &dfsdumpfs_anchor;

static  char       *dump_txt[] =
{
   "",
   "Active filesystem : OS/2 DUMPFS, specific commands are:",
   "",
   " SUPER           = Display the filesystem SUPERBLOCK (first sector)",
   "",
   NULL
};

// Close DUMP filesystem for analysis and free any resources
static ULONG dfsDumpClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
);

// Close DUMP filesystem for Area analysis and free any resources
static ULONG dfsDumpAreaClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
);

// Interpret and execute specific DUMP command;
static ULONG dfsDumpCommand
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *none,                    // IN    dummy
   void               *data                     // INOUT dummy
);

// DUMP filesystem, identify specified sector
static ULONG dfsDumpIdent
(
   ULN64               lsn,                     // IN    LSN for sector
   ULN64               d2,                      // IN    dummy
   char               *st,                      // OUT   sector type
   void               *sec                      // IN    sector contents
);

// DUMP filesystem, supply sector-type description string
static ULONG dfsDumpStype
(
   ULN64               di,                      // IN    sector type
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // OUT   type description
   void               *data                     // INOUT dummy
);

// DUMP filesystem, display sector-contents based on type
static ULONG dfsDumpDispl
(
   ULN64               di,                      // IN    sector type
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // IN    sector contents
);

// Display DUMP  super-block
static ULONG dfsDumpSuperBlock                  // RET   rc = 0 if type match
(
   BYTE              *sector                    // IN    Sector contents
);

// Determine allocation-bit for specified LSN, FREE beyond first block!
static ULONG dfsDumpAllocated                   // RET   LSN is allocated
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
);


// Display the allocation bitmaps based on presence and size of DUMPDATA file
static ULONG dfsDumpAllocMap
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *options,                 // IN    dummy
   void               *data                     // OUT   dummy
);


/*****************************************************************************/
// Check if passed sector is a valid DUMPFS info sector, optional label return
/*****************************************************************************/
BOOL dfsBootsecIsDumpFS                         // RET   bootsector is DUMPFS
(
   BYTE               *brec,                    // IN    bootsector contents
   char               *label                    // OUT   buffer for label, or NULL
)
{
   BOOL                rc = FALSE;
   char                st = ST_UDATA;

   ENTER();

   if (((dfsDumpIdent( 0, 0, &st, brec)) == NO_ERROR) && (st == ST_DUMPFS))
   {
      if (label != NULL)
      {
         DUMPFSINFO         *sd = (DUMPFSINFO *) brec;

         strcpy( label, (sd->DumpSize1 != 0) ? "DUMPpresent" : "DUMP empty");
      }
      rc = TRUE;
   }
   BRETURN (rc);
}                                               // end 'dfsBootsecIsDumpFS'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Initialize DUMP filesystem analysis
/*****************************************************************************/
ULONG dfsDumpInit
(
   char               *fs                       // forced filesystem type
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   dfsa->FsCommand          = dfsDumpCommand;
   dfsa->FsClose            = dfsDumpClose;
   dfsa->FsIdentifySector   = dfsDumpIdent;
   dfsa->FsShowType         = dfsDumpStype;
   dfsa->FsDisplaySector    = dfsDumpDispl;
   dfsa->FsFileInformation  = NULL;
   dfsa->FsGetAllocSpace    = NULL;
   dfsa->FsWriteMetaSpace   = NULL;
   dfsa->FsFindPath         = NULL;
   dfsa->FsLsnAllocated     = dfsDumpAllocated;
   dfsa->FsLsnSetAlloc      = NULL;
   dfsa->FsSltBuild         = NULL;
   dfsa->FsNpBuild          = NULL;
   dfsa->FsDisplayError     = NULL;
   dfsa->FsDisplayLsnInfo   = NULL;
   dfsa->FsDirIterator      = NULL;
   dfsa->FsDirFileSaveAs    = NULL;
   dfsa->FsAllocDisplay     = dfsDumpAllocMap;
   dfsa->FsCl2Lsn           = NULL;             // dummies, one to one
   dfsa->FsLsn2Cl           = NULL;
   dfsa->FsCmdHelp          = dump_txt;         // FS specific cmdhelp text

   dfsa->FsModeId           = DFS_FS_DUMP;      // common for DUMP

   dfsa->Fsi                = NULL;
   dfsa->FsSectorTypes      = dfsDumpSectorTypes;

   dfsa->FsEntry            = DUMP_STARTSEC;    // entry-point in filesystem

   if ((dumpfs->header = TxAlloc( DUMP_SSECTS, dfsGetSectorSize())) != NULL)
   {
      if (dfsRead( 0, DUMP_SSECTS, (BYTE   *) dumpfs->header) != NO_ERROR)
      {
         TxPrint("Error reading the DUMP header sectors from disk, possible bad-sectors!\n");
      }
      else
      {
         TRDUMP(70, "DUMPFS map, start of sector:\n", dumpfs->header, 64, 0);

         if (!TxaOptUnSet('a'))                 // show allocation by default
         {
            dfsDumpAllocMap( 0, 0, NULL, NULL); // include single line alloc
         }
      }
      if (dumpfs->header->DumpSize1 != 0)
      {
         dumpfs->filesectors = ((dumpfs->header->DumpSize1 - 1) / dfsGetSectorSize()) + 1;
      }
      else
      {
         dumpfs->filesectors = DUMP_SSECTS;     // minimal allocation when formatted
      }
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);                                 // when needed
}                                               // end 'dfsDumpInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Close DUMP filesystem for analysis and free any resources
/*****************************************************************************/
static ULONG dfsDumpClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   TxFreeMem( dumpfs->header);

   dfsCloseFileSystem();
   RETURN (rc);
}                                               // end 'dfsDumpClose'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Initialize DUMP filesystem analysis for Area (FS info in FDISK mode)
/*****************************************************************************/
ULONG dfsDumpAreaInit
(
   void
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   if ((dumpfs->header = TxAlloc( DUMP_SSECTS, dfsGetSectorSize())) != NULL)
   {
      if ((rc = dfsRead( 0, DUMP_SSECTS, (BYTE   *) dumpfs->header)) == NO_ERROR)
      {
         if (dumpfs->header->DumpSize1 != 0)
         {
            dumpfs->filesectors = ((dumpfs->header->DumpSize1 - 1) / dfsGetSectorSize()) + 1;
         }
         else
         {
            dumpfs->filesectors = DUMP_SSECTS;  // minimal allocation when formatted
         }
      }
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   dfsa->FsAreaClose        = dfsDumpAreaClose;
   dfsa->FsAreaLsnAllocated = dfsDumpAllocated;

   RETURN (rc);
}                                               // end 'dfsDumpAreaInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Close DUMP filesystem for Area analysis and free any resources
/*****************************************************************************/
static ULONG dfsDumpAreaClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   dfsa->FsAreaClose        = NULL;
   dfsa->FsAreaLsnAllocated = NULL;

   TxFreeMem( dumpfs->header);

   RETURN (rc);
}                                               // end 'dfsDumpAreaClose'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Interpret and execute specific DUMP command;
/*****************************************************************************/
static ULONG dfsDumpCommand
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *none,                    // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;
   ULN64               sn = 0;                  // sector number input
   TXLN                dc;                      // DFS command
   int                 cc;                      // command string count
   char               *c0;                      // parsed command parts
   char               *pp;                      // parameter pointer

   ENTER();

   pp = TxaGetArgString( TXA_CUR, 1, 0, TXMAXLN, dc); // dc => cmd from arg 1
   cc = TxaArgCount( );                         // number of parameters
   c0 = TxaArgValue(0);

   sn = nav.this;                               // default at current sector

   if ((strcmp(c0, "/?") == 0) ||
       (strcmp(c0,  "?") == 0)  )
   {
      TxShowTxt( dump_txt);
   }
   else if (strcasecmp(c0, "super"    ) == 0)
   {
      sprintf( dc, "0x%lx", DUMP_SUPER);        // guaranteed to show as super
      TxaReParseCommand( dc);
      rc = DFS_PENDING;                         // display the sector
   }
   else
   {
      rc = DFS_CMD_UNKNOWN;                     // cmd not recognized
   }
   RETURN (rc);
}                                               // end 'dfsDumpCommand'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// DUMP filesystem, identify specified sector
/*****************************************************************************/
static ULONG dfsDumpIdent
(
   ULN64               lsn,                     // IN    LSN for sector
   ULN64               d2,                      // IN    dummy
   char               *st,                      // OUT   sector type
   void               *sec                      // IN    sector contents
)
{
   ULONG               dr = NO_ERROR;
   BYTE                rc = ST_UDATA;
   DUMPFSINFO         *sd = (DUMPFSINFO *) sec;

   ENTER();

   if (lsn < DUMP_SSECTS)
   {
      if ((sd->Signatur1 == SV_DUMPS1) && (sd->Signatur2 == SV_DUMPS2) &&
          (sd->Signatur3 == SV_DUMPS3) && (sd->Signatur4 == SV_DUMPS4) &&
          (sd->PartSects != 0))
      {
         rc = ST_DUMPFS;
      }
   }
   switch (rc)
   {
      case ST_UDATA:
         dr = DFS_PENDING;
         break;

      default:
         break;
   }
   *st = rc;
   RETURN (dr);
}                                               // end 'dfsDumpIdent'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// DUMP filesystem, supply sector-type description string
/*****************************************************************************/
static ULONG dfsDumpStype
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *st,                      // IN    sector type
   void               *data                     // OUT   type description
)
{
   ULONG               dr  = NO_ERROR;
   char               *buf = (char *) data;
   BYTE                tp  = st[0];

   switch (tp)                                  // searchable types
   {
      case ST_DUMPFS: sprintf(buf,"DUMPFS superblock"); break;
      default:
         switch (tp | ST__INFO)                 // non-searchable ones
         {
            default:       dr = DFS_PENDING;
               break;
         }
         break;
   }
   return (dr);
}                                               // end 'dfsDumpStype'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// DUMP filesystem, display sector-contents based on type
/*****************************************************************************/
static ULONG dfsDumpDispl
(
   ULN64               psn,                     // IN    base psn for sector
   ULN64               d2,                      // IN    dummy
   char               *type,                    // IN    type of sector
   void               *data                     // IN    sector contents
)
{
   ULONG               dr = NO_ERROR;
   BYTE                st = (BYTE) *type;

   ENTER();

   switch (st)
   {
      case ST_DUMPFS:                           // read total DUMP super
         dfsRead( DUMP_STARTSEC, DUMP_SSECTS, rbuf);
         (void) dfsDumpSuperBlock(rbuf);
         break;

      default:
         dr = DFS_PENDING;
         break;
   }
   RETURN (dr);
}                                               // end 'dfsDumpDispl'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display DUMP  super-block
/*****************************************************************************/
static ULONG dfsDumpSuperBlock                  // RET   rc = 0 if type match
(
   BYTE               *sector                   // IN    sector buffer
)
{
   ULONG               rc   = 0;                // rc, sector match
   DUMPFSINFO         *sd   = (DUMPFSINFO *) sector;

   ENTER();

   dfsSiz8("\nMax  DUMP sectors : ", sd->PartSects, "\n");

   if (sd->DumpSize1 != sd->DumpSize2)
   {
      TxPrint("\nWARNING: Dumpfile size values are inconsistent!\n");
   }
   if (sd->DumpSize1 != 0)
   {
      dfsSiz8("Dump sectors used : ", ((sd->DumpSize1 - 1) / dfsGetSectorSize()) + 1, "\n");
      dfsDec8("DUMPDATA.001 size : ", sd->DumpSize1, " bytes\n");

      TxPrint("Dump completed at : %2.2hx%2.2hx-%2.2hx-%2.2hx   %2.2hx:%2.2hx:%2.2hx\n",
                                   sd->dtCentury, sd->dtYears, sd->dtMonth, sd->dtDay,
                                   sd->tmHours, sd->tmMinutes, sd->tmSeconds);
   }
   else
   {
      // to be refined, may detect 'stale' dump still there after format
      TxPrint("\nThe DUMP filesystem is currently EMPTY, no dump present\n");
   }
   dfsDumpAllocMap( 0, 0, NULL, NULL);          // include single line alloc
   RETURN (rc);
}                                               // end 'dfsDumpSuperBlock'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Determine allocation-bit for specified LSN, FREE beyond first block!
/*****************************************************************************/
static ULONG dfsDumpAllocated                   // RET   LSN is allocated
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc  = 1;                 // allocated
   ULONG               asn = dfstAreaD2Part( DFSTORE, lsn); // Area aware sn

   if (asn >= DUMP_SSECTS)                       // within valid block area
   {
      if (asn >= dumpfs->filesectors)
      {
         rc = NO_ERROR;                         // not allocated
      }
   }
   return (rc);
}                                               // end 'dfsDumpAllocated'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display the allocation bitmaps based on presence and size of DUMPDATA file
/*****************************************************************************/
static ULONG dfsDumpAllocMap
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *options,                 // IN    dummy
   void               *data                     // OUT   dummy
)
{
   ULONG               a;                       // index in display-array
   ULONG               perc;                    // percentage
   TX1K                ascii;                   // display-array
   ULONG               acl;                     // ascii alloc chars per line
   DUMPFSINFO         *sd;
   ULONG               allocatedCount;

   ENTER();

   if ((sd = dumpfs->header) != NULL)
   {
      TRDUMP(70, "DUMPFS map, start of sector:\n", sd, 64, 0);

      TRACES(("Part: %8.8lx  file: %8.8lx  Cyl: %8.8lx\n",
               sd->PartSects, dumpfs->filesectors, dfsCylinderSize()));


      acl   = dfsAllocCharsPerLine( 'c');

      if ((sd->PartSects - dumpfs->filesectors) > dfsCylinderSize()) // significant freespace
      {
         allocatedCount = dumpfs->filesectors / (sd->PartSects / acl);
         perc           = dumpfs->filesectors / (sd->PartSects / 100);
      }
      else
      {
         allocatedCount = acl;                  // show as fully allocated
         perc           = 100;
      }

      TRACES(("allocatedCount: %lu  perc:%lu\n", allocatedCount, perc));

      TxPrint(   "          %s%*.*s%s\n", CBC, acl, acl, BLIN, CNN);

      memset(ascii, 0, TXMAXLN);
      ascii[0] = mapchar[5];                    // first sector always in use

      for (a = 1; a < acl; a++)
      {
         ascii[a] = (char) ((a >= allocatedCount) ? ' ' : mapchar[8]);
      }
      dfsX10( CNZ, 0, CNZ, "");
      TxPrint("%s%s%s%s", CBC, CNC, ascii, CNN);
      TxPrint("%s%s% 3lu%%\n", CBC, CNN, perc);

      TxPrint("          %s%*.*s%s\n", CBC, acl, acl, BLIN, CNN);

      if (sd->PartSects != 0)                   // avoid devide by 0
      {
         dfsSz64("Allocated sectors : ", dumpfs->filesectors +1, " ");
         dfsSz64("of ", sd->PartSects, " ");
         TxPrint("=% 5.1lf%%\n", (double) (100 * ((double)  dumpfs->filesectors +1) / sd->PartSects));
      }
   }
   RETURN (NO_ERROR);
}                                               // end 'dfsDumpAllocMap'
/*---------------------------------------------------------------------------*/


