//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// APFS (container) dump & display Analysis functions
//
// Author: J. van Wijk
//
// JvW  24-01-2018 Initial version, derived from High Performance File System
//
#ifndef    DFSAAPFS_H
   #define DFSAAPFS_H

#include <dfsapfs.h>                            // APFS structure defs

// APFS specific sector types
                                                // S   used on searching
                                                //  I  identification only
#define ST_C_SUP   's'                          //     APFS container super-block
#define ST_BTREE   'B'                          //     APFS B-Tree (root)
#define ST_ANODE   'N'                          //     APFS node block
#define ST_SPMGR   'M'                          //     APFS Space Manager
#define ST_SPCAB   'S'                          //     APFS Space Manager Address
#define ST_SPCIB   'i'                          //     APFS Space Manager Info
#define ST_SPBMP   'm'                          //     APFS Space Manager Bitmap
#define ST_OBMAP   'O'                          //     APFS Object Map
#define ST_CHKPT   'c'                          //     APFS Checkpoint
#define ST_V_SUP   'D'                          //     APFS volume    super-block
#define ST_REAPR   'Y'                          //     APFS Reaper object
#define ST_REAPL   'y'                          //     APFS Reaper list
#define ST_EFIBT   'E'                          //     APFS EFI jumpstart boot
#define ST_FUWBC   'F'                          //     APFS Fusion WriteBack Cache
#define ST_FUWBL   'f'                          //     APFS Fusion WriteBack List
#define ST_ENCRS   'x'                          //     APFS Encryption Rolling State
#define ST_GBMAP   'z'                          //     APFS General Bitmap Object
#define ST_GBMBL   'Z'                          //     APFS General Bitmap Block

#define APFS_BLOCKSIZE                  4096

// There appears to be a seperate bitmap for the two devices, MAIN and TIER2 (Fusion drives)
// as well as a differently structured Internal-Pool. They all use structures like CAB/CIB.
// On single drive, non-fusion, the MAIN bitmap seems the best for general use, since the
// Internal-Pool is smaller, less well understood, and seems older (often lower Xid's)
// The CACHE structures below support all tree of them, but using the MAIN one will be
// the one implemented and tested first (to be refined)
// Eventually it should be possible to use each one, for example with the 'alloc' command

typedef union dfsbmtoplevel                     // pointer to op level, either CIB or CAB
{
   S_SP_CIB_BLOCK     *cib;
   S_SP_CAB_BLOCK     *cab;
} DFSBMTOPLEVEL;                                // end of union "dfsbmtoplevel"

#define BMB_NONE   0xffffaaaacccceeeeULL        // no BM block cached yet

// Cache info for a single BLOCK cache for one of the 2 bitmaps, MAIN or TIER2
typedef struct dfsapfsbmcache                   // Block bitmap cache for single area
{
   ULONG               cacheId;                 // IPOOL, MAIN or TIER2  (using SFQ_ values)
   ULN64               blockCount;              // number of blocks managed by this bitmap
   ULONG               bmType;                  // top level CAB or CIB allocated in SpMgr
   ULN64               bmSparse;                // relative bitmap block cached as being ZERO (FREE)
   ULN64               bmBlock;                 // relative bitmap block cached in buffer 0..n
   ULONG               bmCount;                 // number of top level CIB/CAB pointers
   DFSBMTOPLEVEL      *bmTopLevel;              // allocated array of top level blocks
   BYTE               *Buffer;                  // current cached bitmap block
} DFSAPFSBMCACHE;                               // end of struct "dfsapfsbmcache"

typedef struct dfsapfsleafcache
{
   TXTS                vTreeName;               // Short tree description (align: max 8 characters!)
   ULN64               vNodeCount;              // Total number of nodes in tree (change detect)
   ULN64               vLastIndex;              // Last index retrieved (new startpoint)
   ULN64               vNlCount;                // Number of Next-Leaf blocknumbers cached
   ULN64              *vNlCache;                // Array  of Next-Leaf blocknumbers (virt or phys)
} DFSAPFSLEAFCACHE;                             // end of struct "dfsapfsleafcache"

#define APFS_MOUNT_TR_LIMIT             50ULL   // nr of transactions considered still 'stable'

typedef struct dfsaapfs                         // APFS analysis information
{
   ULN64               Sect;                    // Number of sectors
   ULN64               Block;                   // Number of blocks
   ULONG               BlockSize;               // Block size in bytes
   ULONG               SectorsPerBlock;         // sectors per block
   S_C_SUPER          *sup;                     // Master superblock, block 0, 512 bytes!
   ULONG               cpCount;                 // Number of checkpoints
   ULN64               cpLsn;                   // LSN of current C-superblock
   S_CHECKPOINT_MAP   *cpMap;                   // checkpoint Map for current C-SB
   S_C_SUPER          *cpSup;                   // current C-superblock        (SB)
   S_BT_NODE          *cOmap;                   // Container Object-Map Btree root node
   ULN64               cTransactions;           // Container #transactions since last (un)mount
   ULN64               vTransactions;           // Volume    #transactions since last (un)mount
   BOOL                vMountWarning;           // Mount warning already given (on make browse)
   ULONG               vCount;                  // Number of volumes    in superblock
   ULONG               vIndex;                  // Index current Volume in superblock
   ULN64               vsLsn;                   // LSN of current V-superblock
   S_V_SUPER          *vsBlk;                   // current V-superblock        (VSB)
   S_BT_NODE          *vOmap;                   // Volume Object-Map Btree root node
   S_SPACEMANAGER     *SpMgr;                   // Spacemanager block
   DFSAPFSLEAFCACHE    vCatTree;                // Catalog-tree next-Leaf blocknumber cache
   DFSAPFSLEAFCACHE    vExtTree;                // Extent-tree next-Leaf blocknumber cache
   DFSAPFSLEAFCACHE    vSnapTree;               // Snapshot-tree next-Leaf blocknumber cache
   DFSAPFSBMCACHE      Bm[ SPD_COUNT];          // Bitmap cache for each device
} DFSAAPFS;                                     // end of struct "dfsaapfs"

extern DFSAAPFS       *apfs;

// Initialize APFS filesystem analysis
ULONG dfsApfsInit
(

   char               *fs                       // forced filesystem type
);

// Initialize APFS filesystem analysis for Area (FS info in FDISK mode)
ULONG dfsApfsAreaInit
(
   void
);

// APFS filesystem, identify specified sector as a valid superblock
BOOL dfsApfsIsSuperBlock                        // RET   sector is a valid sb
(
   BYTE               *sec                      // IN    sector contents
);

// APFS filesystem, display sector contents as a superblock
void dfsApfsDisplaySuperBlock
(
   BYTE               *sec                      // IN    sector contents
);

// Display APFS Block, Volume super-block
ULONG dfsApfsBlockVsuper                        // RET   rc = 0 if type match
(
   BYTE               *block                    // IN    Block contents
);

// Make filesystem usage map dump on TxPrint output, MAIN and TIER2 (if any)
ULONG dfsApfsAllocMap
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *options,                 // IN    display options
   void               *data                     // OUT   dummy
);


#endif
