//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// APFS (container) dump & display Analysis functions
//
// Author: J. van Wijk
//
// JvW  08-04-2021 Use rc CMD_WARNING on FileSaveAs alloc errors, considered OK
// JvW  24-01-2018 Initial version, derived from High Performance File System
//
// Author: J. van Wijk
//

#include <txlib.h>                              // TX library interface

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsfind.h>                            // CMD find sector data
#include <dfswin.h>                             // HELP item definitions
#include <dfsutil.h>                            // DFS utility functions
#include <dfsposix.h>                           // POSIX utility functions
#include <dfsspace.h>                           // DFS file-space interface
#include <dfstable.h>                           // SLT utility functions
#include <dfsaapfs.h>                           // APFS display & analysis
#include <dfsuapfs.h>                           // APFS utility functions
#include <dfslapfs.h>                           // APFS SLT functions


char dfsApfsSectorTypes[] =
{
   ST_C_SUP,                                    // APFS container super-block
   ST_BTREE,                                    // APFS B-Tree (root)
   ST_ANODE,                                    // APFS node block
   ST_SPMGR,                                    // APFS Space Manager
   ST_SPCAB,                                    // APFS Space Manager Address
   ST_SPCIB,                                    // APFS Space Manager Info
   ST_SPBMP,                                    // APFS Space Manager Bitmap
   ST_OBMAP,                                    // APFS Object Map
   ST_CHKPT,                                    // APFS Checkpoint
   ST_V_SUP,                                    // APFS volume    super-block
   ST_REAPR,                                    // APFS Reaper object
   ST_REAPL,                                    // APFS Reaper list
   ST_EFIBT,                                    // APFS EFI jumpstart boot
   ST_FUWBC,                                    // APFS Fusion WriteBack Cache
   ST_FUWBL,                                    // APFS Fusion WriteBack List
   ST_ENCRS,                                    // APFS Encryption Rolling State
   ST_GBMAP,                                    // APFS General Bitmap Object
   ST_GBMBL,                                    // APFS General Bitmap Block
   0                                            // string terminating ZERO
};


static DFSAAPFS    apfs_anchor =
{
   0,                                           // Number of sectors
   0,                                           // Number of blocks
   APFS_BLOCKSIZE,                              // Block size in bytes
   8,                                           // sectors per block    (512)
   NULL,                                        // master superblock    (512)
   1,                                           // number of checkpoints
   0,                                           // LSN for current C-superblock
   NULL,                                        // checkpoint for current SB
   NULL,                                        // current superblock    (SB)
   NULL,                                        // current Container Object-Map (tree)
   0, 0, FALSE,                                 // c/v #transact since last (un)mount
   1,                                           // Number of volumes    in superblock
   0,                                           // Index current Volume in superblock
   0,                                           // LSN for current V-superblock
   NULL,                                        // current V-superblock (VSB)
   NULL,                                        // current Volume Object-Map (tree)
   NULL,                                        // current SpaceManager object
   {"Catalog ", 0, 0, 0, NULL},                 // FS-tree next-Leaf blocknumber cache
   {"Extents ", 0, 0, 0, NULL},                 // Extent-tree next-Leaf blocknumber cache
   {"SnapShot", 0, 0, 0, NULL},                 // Snapshot-tree next-Leaf blocknumber cache
   {{SPD_MAIN , 0,BT_NO_TYPE,0,0,0,NULL,NULL},  // empty MAIN  bitmap cache
    {SPD_TIER2, 0,BT_NO_TYPE,0,0,0,NULL,NULL}}  // empty TIER2 bitmap cache
};

       DFSAAPFS   *apfs = &apfs_anchor;

static char sg_c_efi_j[SG_C_EFI_J] = {SV_C_EFI_J};
static char sg_c_super[SG_C_SUPER] = {SV_C_SUPER};
static char sg_v_super[SG_V_SUPER] = {SV_V_SUPER};

static  char       *apfs_txt[] =
{
   "",
   "Active filesystem : APFS container, specific commands are:",
   "",
   " BL              = Translate and display 'this' LSN as a block number",
   " BL block  [cmd] = Translate specified block-number to LSN, display using 'cmd'",
   " CATS [id][name] = Search FS-tree for INO and optional file/folder name",
   " COLlissions     = Find filename duplicates (HASH collisions) in APFS volume",
   " CP      [index] = Refresh to latest or specified checkpoint (history)",
   " DESC    [f [l]] = Describe object TYPE/SUBTYPE, and FS/XFIELD RECORD types",
   " DUMP   [params] = Dump recognized APFS blocks to display, with some filtering",
   " FOLDER [dirIno] = Display Catalog directory data for specified folder INO nr",
   " LEAVES [params] = Display contents of various APFS filesystem tree leaf-nodes",
   " LEAF       [id] = Determine NEXT-leaf ID for given/last leaf ID, 0 = FIRST",
   " SUPER           = Display the selected container SUPERBLOCK structure",
   " VI  [vol-index] = Select a volume from a list, or specified volume index",
   " VIRT   o x [-c] = Find blocknr for objec/transaction ID, volume or container",
   "",
   NULL
};

// Close APFS filesystem for analysis and free any resources
static ULONG dfsApfsClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
);

// Close APFS filesystem for Area analysis and free any resources
static ULONG dfsApfsAreaClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
);

// Interpret and execute specific APFS command;
static ULONG dfsApfsCommand
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *none,                    // IN    dummy
   void               *data                     // INOUT dummy
);

// APFS filesystem, identify specified sector
static ULONG dfsApfsIdent
(
   ULN64               sn64,                    // IN    LSN for sector
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // OUT   sector type
   void               *data                     // IN    sector contents
);

// APFS filesystem, supply sector-type description string
static ULONG dfsApfsStype
(
   ULN64               di,                      // IN    sector type
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // OUT   type description
   void               *data                     // INOUT dummy
);

// APFS filesystem, display APFS block contents at PSN based on type
static ULONG dfsApfsBlockAtPsn
(
   ULN64               psn,                     // IN    base psn for sector
   ULN64               d2,                      // IN    dummy
   char               *type,                    // IN    type of sector
   void               *dummy                    // IN    sector contents (rbuf)
);

// APFS filesystem, display Block-contents from given block buffer
static ULONG dfsApfsBlockContents
(
   BYTE                blockType,               // IN    type of block (st)
   BYTE               *blockBuffer              // IN    buffer with one block
);

// APFS filesystem, display special LSN info (Node lsn + record-index value)
static ULONG dfsApfsDisplayLsnInfo
(
   ULN64               lsn,                     // IN    possible special lsn
   ULN64               info,                    // IN    possible REC index
   char               *dc,                      // IN    dummy
   void               *data                     // IN    dummy
);

// Display all available info about a Node Record, in an FS-Tree Catalog Node
static ULONG dfsApfsCatalogRecord
(
   ULN64               nodeLsn,                 // IN    lsn of Catalog Node
   USHORT              index                    // IN    index of record 0..N
);

// Display APFS Block, Container super-block (or sector :)
static ULONG dfsApfsBlockCsuper                 // RET   rc = 0 if type match
(
   BYTE               *buffer,                  // IN    Block contents
   BOOL                singleSector             // IN    buffer is just one sector
);

// Display APFS Block, Generic BTree node
static ULONG dfsApfsBtreeNode                   // RET   rc = 0 if type match
(
   BYTE               *block                    // IN    Block contents
);

// Display APFS Block, Space Manager
static ULONG dfsApfsSpaceManager                // RET   rc = 0 if type match
(
   BYTE               *block                    // IN    Block contents
);

// Display APFS Space Manager DEVICE structure with CAB/CIB blk# array
static void dfsApfsSpaceMgrDev
(
   BYTE               *block,                   // IN    Block contents SPMGR
   S_SPMGR_DEVICE     *dev,                     // IN    Device in this SPMGR
   char               *devName                  // IN    Device name, length 5
);

// Display APFS Space Manager FREE-QUEUE structure for Ipool/Main/Tier2
static void dfsApfsSpaceMgrSfq
(
   BYTE               *block,                   // IN    Block contents SPMGR
   S_SPMGR_FREEQ      *que,                     // IN    Queue  in this SPMGR
   char               *queName                  // IN    Queue  name, length 5
);

// Display APFS Block, Space Manager Chunk-Address-Block
static ULONG dfsApfsSpaceMgrCab                 // RET   rc = 0 if type match
(
   BYTE               *block                    // IN    Block contents
);

// Display APFS Block, Space Manager CIB
static ULONG dfsApfsSpaceMgrCib                 // RET   rc = 0 if type match
(
   BYTE               *block                    // IN    Block contents
);

// Display APFS Block, Space Manager BMAP
static ULONG dfsApfsSpaceMgrBmap                // RET   rc = 0 if type match
(
   BYTE               *block                    // IN    Block contents
);

// Display APFS Block, Object Map
static ULONG dfsApfsObjectMap                   // RET   rc = 0 if type match
(
   BYTE               *block                    // IN    Block contents
);

// Display APFS Block, Check Point
static ULONG dfsApfsCheckPointMap               // RET   rc = 0 if type match
(
   BYTE               *block                    // IN    Block contents
);

// Display APFS Block, type 0x11 Reaper object
static ULONG dfsApfsBlockReaper                 // RET   rc = 0 if type match
(
   BYTE               *block                    // IN    Block contents
);

// Display APFS Block, Reaper List
static ULONG dfsApfsReaperList                  // RET   rc = 0 if type match
(
   BYTE               *block                    // IN    Block contents
);


// Display APFS Block, EFI JumpStart (boot)
static ULONG dfsApfsEfiJumpStart                // RET   rc = 0 if type match
(
   BYTE               *block                    // IN    Block contents
);

// Display APFS Block, Fusion Write-Back Cache
static ULONG dfsApfsFusionWbCache               // RET   rc = 0 if type match
(
   BYTE               *block                    // IN    Block contents
);

// Display APFS Block, Fusion Write-Back List
static ULONG dfsApfsFusionWbList                // RET   rc = 0 if type match
(
   BYTE               *block                    // IN    Block contents
);

// Display APFS Block, Encryption Rolling State
static ULONG dfsApfsEncrRollState               // RET   rc = 0 if type match
(
   BYTE               *block                    // IN    Block contents
);

// Display APFS Block, General BitMap object
static ULONG dfsApfsGeneralBitMap               // RET   rc = 0 if type match
(
   BYTE               *block                    // IN    Block contents
);

// Display APFS Block, General BitMap Block
static ULONG dfsApfsGenBmapBlock                // RET   rc = 0 if type match
(
   BYTE               *block                    // IN    Block contents
);

// Display APFS block header structure, in a generic way
static ULONG dfsApfsBlockHeader                 // RET   rc = 0 if contents OK
(
   BYTE               *buffer,                  // IN    sector/block buffer
   BOOL                singleSector             // IN    buffer is just one sector
);

// Make filesystem usage map dump on TxPrint output, one of three map types
static ULONG dfsApfsAnyAllocMap
(
   ULONG               mapType,                 // IN    Kind of BM, MAIN, TIER2, IPOOL
   char               *options                  // IN    display options
);

// DFS APFS write-file to disk (SaveTo to file)
static ULONG dfsApfsFileSaveAs
(
   ULN64               leafLsn,                 // IN    Leaf-node sectornumber
   ULN64               leafIdx,                 // IN    Leaf-node record index
   char               *path,                    // IN    destination path
   void               *recp                     // IN    recovery params (rename)
);

/*****************************************************************************/
// Initialize APFS filesystem analysis
/*****************************************************************************/
ULONG dfsApfsInit
(
   char               *fs                       // forced filesystem type
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   dfsa->FsCommand          = dfsApfsCommand;
   dfsa->FsClose            = dfsApfsClose;
   dfsa->FsIdentifySector   = dfsApfsIdent;
   dfsa->FsShowType         = dfsApfsStype;
   dfsa->FsDisplaySector    = dfsApfsBlockAtPsn;
   dfsa->FsFileInformation  = dfsApfsFileInfo;
   dfsa->FsGetAllocSpace    = dfsApfsGetAllocSpace;
   dfsa->FsWriteMetaSpace   = NULL;
   dfsa->FsFindPath         = dfsApfsFindPath;
   dfsa->FsUpdateFileName   = NULL;
   dfsa->FsSetAltBrecLabel  = NULL;
   dfsa->FsMakeBrowseList   = dfsApfsMakeBrowseList;
   dfsa->FsLsnAllocated     = dfsApfsAllocated;
   dfsa->FsLsnSetAlloc      = NULL;
   dfsa->FsSltBuild         = dfsApfsSltBuild;
   dfsa->FsNpBuild          = NULL;
   dfsa->FsDisplayError     = dfsApfsDispError;
   dfsa->FsDisplayLsnInfo   = dfsApfsDisplayLsnInfo;
   dfsa->FsDirIterator      = NULL;
   dfsa->FsDirFileSaveAs    = dfsApfsFileSaveAs;
   dfsa->FsTruncateSize     = NULL;
   dfsa->FsAllocDisplay     = dfsApfsAllocMap;
   dfsa->FsCl2Lsn           = dfsApfsCl2Lsn;
   dfsa->FsLsn2Cl           = dfsApfsLsn2Cl;
   dfsa->Fsi                = apfs;
   dfsa->FsSectorTypes      = dfsApfsSectorTypes;
   dfsa->FsCmdHelp          = apfs_txt;         // FS specific cmdhelp text

   dfsa->FsModeId           = DFS_FS_APFS;

   dfsa->FsEntry            = LSN_C_SUPER;      // default entry-point in filesystem

   TRACES(("sizeof S_C_SUPER      = % 3u\n", sizeof(S_C_SUPER)));
   TRACES(("sizeof S_BT_NODE      = % 3u\n", sizeof(S_BT_NODE)));

   //- allocate and analyse Container-superblock, complete block
   if (((apfs->sup = TxAlloc(1, APFS_BLOCKSIZE)) != NULL) )
   {
      rc = dfsRead( LSN_C_SUPER, (APFS_BLOCKSIZE / dfsGetSectorSize()), (BYTE *) apfs->sup);
      TRHEXS( 70,  apfs->sup,  0x100, "apfs superblock");

      if ((rc == NO_ERROR) && (dfsIdentifySector( LSN_C_SUPER, 0, (BYTE *) apfs->sup) == ST_C_SUP))
      {
         ULONG         blocksPerCheckPoint = (apfs->sup->csCpDescLength != 0) ? apfs->sup->csCpDescLength :  2;

         apfs->BlockSize       = apfs->sup->csBlockSize;
         apfs->SectorsPerBlock = apfs->sup->csBlockSize / dfsGetSectorSize();
         apfs->Block           = apfs->sup->csTotalBlocks;
         apfs->Sect            = apfs->Block * apfs->SectorsPerBlock;
         apfs->cpCount         = apfs->sup->csSbDescBlockCount / blocksPerCheckPoint;
         if (dfsa->verbosity > TXAO_QUIET)
         {
            TxPrint("Checkpoint Count  : %4u   with %u blocks each\n", apfs->cpCount, blocksPerCheckPoint);
         }
      }
      else
      {
         TxPrint("Error reading APFS container Super-block at block 0, defaults used!\n");

         apfs->BlockSize       = 4096;
         apfs->SectorsPerBlock = 4096 / dfsGetSectorSize();
         apfs->Sect            = dfsGetLogicalSize();
         apfs->Block           = apfs->Sect  / apfs->SectorsPerBlock;
         apfs->cpCount         = 0;
      }
      dfstSetClusterSize( DFSTORE, apfs->SectorsPerBlock);

      TRACES(("BlockSize:%6u  SectorsPerBlock:%3u  Sectors: 0x0%llx\n",
               apfs->BlockSize, apfs->SectorsPerBlock, apfs->Sect));

      //- allocate complete (1-block) Container-superblock, Checkpoint Map and Volume Super
      if (((apfs->cpSup = TxAlloc(1, apfs->BlockSize)) != NULL) &&
          ((apfs->cpMap = TxAlloc(1, apfs->BlockSize)) != NULL) &&
          ((apfs->vsBlk = TxAlloc(1, apfs->BlockSize)) != NULL)  )
      {
         apfs->vMountWarning = FALSE;           // mountwarning not given yet
         rc = dfsApfsFindCheckpoint( 0, dfsa->verbosity); // find latest checkpoint/superblock

         if ((rc == NO_ERROR) && (dfsa->verbosity > TXAO_QUIET))
         {
            dfsMultiCommand( "folder 2 -refresh-", 0, TRUE, FALSE, TRUE); // include command echo
         }
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }

   #if defined (USEWINDOWING)
      if (dfsa->FsModeSelist == NULL)           // always have some relevant info there ...
      {
         dfsa->FsModeSelist = TxSelEmptyList( "- No Volumes present in APFS container -",
               "No Volumes found, container is empty or this is not an APFS container at all", FALSE);

      }
   #endif
   TRHEXS( 70,  apfs,  sizeof(DFSAAPFS), "apfs global info");
   RETURN (rc);
}                                               // end 'dfsApfsInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Close APFS filesystem for analysis and free any resources
/*****************************************************************************/
static ULONG dfsApfsClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   dfsSlTableReset();                           // stop SLT thread and reset
   TxFreeMem( apfs->sup);
   TxFreeMem( apfs->cpMap);
   TxFreeMem( apfs->cpSup);
   TxFreeMem( apfs->vsBlk);

   dfsApfsFreeObjectMap( &apfs->cOmap);
   dfsApfsFreeObjectMap( &apfs->vOmap);

   dfsApfsLeafCacheFree( &apfs->vCatTree);      // Free possible next-leaf caches
   dfsApfsLeafCacheFree( &apfs->vExtTree);
   dfsApfsLeafCacheFree( &apfs->vSnapTree);

   dfsApfsBitMapFree( SPD_MAIN);                // Free possible bitmap caches
   dfsApfsBitMapFree( SPD_TIER2);

   apfs->vIndex = 0;

   dfsCloseFileSystem();
   RETURN (rc);
}                                               // end 'dfsApfsClose'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Initialize APFS filesystem analysis for Area (FS info in FDISK mode)
/*****************************************************************************/
ULONG dfsApfsAreaInit
(
   void
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   dfsa->FsAreaClose        = dfsApfsAreaClose;
   dfsa->FsAreaLsnAllocated = NULL;

   if (((apfs->sup = TxAlloc(1, dfstGetSectorSize(DFSTORE))) != NULL) )
   {
      rc = dfsRead( dfstAreaP2Disk( DFSTORE, LSN_C_SUPER), 1, (BYTE   *) apfs->sup);
      if ((rc == NO_ERROR) &&
          (dfsIdentifySector( LSN_C_SUPER, 0, (BYTE *) apfs->sup)  == ST_C_SUP))
      {
         apfs->BlockSize       = apfs->sup->csBlockSize;
         apfs->SectorsPerBlock = apfs->BlockSize / dfsGetSectorSize();
         apfs->Block           = apfs->sup->csTotalBlocks;
         apfs->Sect            = apfs->Block * apfs->SectorsPerBlock;

         TRACES(("BlockSize:%6u  SectorsPerBlock:%3u  Sectors: 0x0%llx\n",
                  apfs->BlockSize, apfs->SectorsPerBlock, apfs->Sect));
      }
      else
      {
         TxPrint("Error reading APFS container Super-block at block 0, defaults used!\n");

         apfs->BlockSize       = 4096;
         apfs->SectorsPerBlock = 4096 / dfsGetSectorSize();
         apfs->Sect            = dfsGetLogicalSize();
         apfs->Block           = apfs->Sect  / apfs->SectorsPerBlock;
      }
      dfstSetClusterSize( DFSTORE, apfs->SectorsPerBlock);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsApfsAreaInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Close APFS filesystem for Area analysis and free any resources
/*****************************************************************************/
static ULONG dfsApfsAreaClose
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   dfsa->FsAreaClose        = NULL;
   dfsa->FsAreaLsnAllocated = NULL;

   TxFreeMem(apfs->sup);

   RETURN (rc);
}                                               // end 'dfsApfsAreaClose'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Interpret and execute specific APFS command;
/*****************************************************************************/
static ULONG dfsApfsCommand
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *none,                    // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               rc = NO_ERROR;
   ULN64               sn = 0;                  // sector number input
   ULN64               sl = 0;                  // sector LSN
   BYTE                st = 0;                  // sector type wanted
   ULONG               nr = 0;
   TXLN                dc;                      // DFS command
   int                 cc;                      // command string count
   char               *c0, *c1, *c2, *c3;       // parsed command parts
   char               *pp;                      // parameter pointer
   TXLN                s1;                      // big temporary string space
   ULONG               size  = 1;               // default display size
   ULN64               id64  = 0;               // (virtual) node number / fsId
   USHORT              index = 0;
   int                 rType;

   ENTER();

   pp = TxaGetArgString( TXA_CUR, 1, 0, TXMAXLN, dc); // dc => cmd from arg 1
   cc = TxaArgCount( );                         // number of parameters
   c0 = TxaArgValue(0);
   c1 = TxaArgValue(1);
   c2 = TxaArgValue(2);
   c3 = TxaArgValue(3);

   sn = (ULONG) nav.this;                       // default at current sector

   if ((strcmp(c0, "/?") == 0) ||
       (strcmp(c0,  "?") == 0)  )
   {
      TxShowTxt( apfs_txt);
   }
   else if ((strcasecmp(c0, "cl"       ) == 0) ||
            (strcasecmp(c0, "bl"       ) == 0)  )
   {
      if (TxaOption('?'))
      {
         TxPrint("\nShow BLOCK-number for current sector, or display contents for specified blocknumber\n");
         TxPrint("\n Usage:  %s  [block] [-O:q | -O:v]\n\n"
                 "   block = Blocknumber for block contents to be displayed\n\n"
                 "   -O:q  = Display block in a compact format, using less lines (quiet)\n"
                 "   -O:v  = Display block with more detail,    using more lines (verbose)\n", c0);
      }
      else
      {
         if (cc > 1)
         {
            sn = dfsGetSymbolicSN( c1, 0);
            if (sn <= apfs->Block)
            {
               if (cc > 2)
               {
                  sprintf( dc, "%s ", c2);
               }
               else
               {
                  strcpy(  dc, "");
               }
               sprintf( s1, "0x0%llX", dfsApfsBlock2Lsn(sn));
               strcat(  dc, s1);
               TxaReParseKeepOpts( dc);         // reparse but keep option values
               rc = DFS_PENDING;                // handle translated command
            }
            else
            {
               dfsX10( "Invalid block  nr : ", sn, CBR, "\n");
            }
         }
         else
         {
            dfsX10("Current block  nr : ", dfsApfsLsn2Block(nav.this), CBM, "\n");
         }
      }
   }
   else if (strcasecmp(c0, "cp"     ) == 0)     // refresh/set current checkpoint
   {
      if (TxaOption('?'))
      {
         TxPrint("\nDetermine latest valid checkpoint, optional offset by checkpoint index\n");
         TxPrint("\n Usage:  %s  [index] [-O:s | -O:q | -O:v]\n\n", c0);
         TxPrint("   index       = Number of checkpoints to go BACK in history, default 0\n"
                 "                 valid range is 0 .. %u  (equals Checkpoint-Count - 1)\n\n", apfs->cpCount - 1);
         TxPrint("   -O:s          = NO display of superblocks, no output                (silent)\n"
                 "   -O:q          = Display superblocks in a compact format, less lines (quiet)\n"
                 "   -O:v          = Display superblocks with more detail,    more lines (verbose)\n\n");
      }
      else
      {
         nr = atol( c1);

         if (dfsa->verbosity > TXAO_SILENT)
         {
            TxPrint( "\n");
         }
         if ((rc = dfsApfsFindCheckpoint( nr, dfsa->verbosity)) == NO_ERROR) // find latest, use index nr
         {
            nav.down = apfs->cpLsn;             // next sector to visit, checkpoint superblock
         }
      }
   }
   else if ((strcasecmp(c0, "cats"  ) == 0))    // CATalog Search for various FS-Tree records
   {
      if (TxaOption('?') || (cc < 2))
      {
         TxPrint("\nSearch FS-tree for fsId or folder-ID and optional file/folder name, then display result\n");
         TxPrint("\n Usage:  %s  Id  [Key] [-r:N]\n\n"
                 "    Id  = Primary key: Parent-INO for DIR; INO for other (attribute) records\n"
                 "    Key = Additional key value: Name for DIR or XATTR; sibling-ID for sibling link\n"
                 "          Without a Key, it will search for an INODE record  for INO 'Id'\n\n", c0);
         TxPrint("   -r:N = Select type of record to search, default INO, or DIR when a name is given\n"
                 "          1=SMET 2=PXT 3=INO 4=XAT 5=LNK 6=DSTR 8=FXT 9=DIR 10=STAT 11=SNAP 12=LM\n");
      }
      else if (apfs->vsBlk != NULL)
      {
         S_NODE_KEY   key;                      // universal key pointer (union)
         key.data =   s1;                       // storage large enough for a full filename

         memset( s1, 0, TXMAXLN);               // start out all ZERO for key fields

         rType = TxaOptNum( 'r', NULL, (cc > 2) ? 9 : 3);   //- record type to search for
         id64  = dfsGetSymbolicSN( c1, 2);                  //- primary key value

         key.dh->fsId = APFSmkFsId( id64, rType); // hybrid primary key value
         if (cc > 2)
         {
            switch (rType)                      // fill tertiary key values from 2nd param
            {
               case APFST_DIR_RECORD:
                  key.dh->drLengthHash = dfsApfsNameHash( (BYTE *) c2, strlen( c2) + 1);
                  strcpy((char *) key.dh->drName, c2); // copy name itself too (hash-collission-detect)
                  break;

               case APFST_XATTR:
                  key.xa->xaNameLength  = strlen( c2) + 1;
                  strcpy((char *) key.xa->xaName, c2); // copy XATTR name
                  break;

               case APFST_FILE_EXTENT:
                  key.fe->feLogAddress = dfsGetSymbolicSN( c2, 0); // relative first position in extent
                  break;

               case APFST_SIBLING_LINK:
                  key.sl->slSibId = dfsGetSymbolicSN( c2, 0); // sibling ID
                  break;

               default:
                  break;
            }
         }

         if ((id64 = dfsApfsSearchCatalog( key, &index, NULL)) == 0)
         {
            dfsX10(  "\nBad / unused fsID : ", key.dh->fsId, CBR, "\n");
         }
         else if (id64 == L64_NULL)
         {
            if (rType == APFST_DIR_RECORD)
            {
               TxPrint( "\nFile/Folder  name : '%s' not found in folder fsId %x\n", c2, key.dh->fsId);
            }
            else
            {
               TxPrint( "\nNo %s%s%s RECORD found for fsId 0x0%llx\n", CBC, dfsApfsFsRecType( rType), CNN, key.dh->fsId);
            }
         }
         else
         {
            TxPrint( "\n%s%s%s record found in node 0x0%llx index:%3hx\n", CBC, dfsApfsFsRecType( rType), CNN, id64, index);

            /*
            sn = dfsHfsNode2Lsn( id64);
            hfs->preferedFolderId = 0;          // default ANY folder (to be refined)
            sprintf( dc, "0x0%llX -e:%hu", sn, index); // display as SN+INFO
            TxaReParseKeepOpts( dc);            // reparse but keep option values
            rc = DFS_PENDING;                   // handle translated command
            */
         }
      }
   }
   else if (strncasecmp(c0, "col", 3 ) == 0)   // Iterate leaf nodes, show name-HASH duplicates (collisions)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nFind any filename (HASH) duplicates (HASH collisions) in an APFS volume\n");
         TxPrint("\n Usage:  %s                                 (use <Escape> to abort ...)\n\n"
                 "  -test:[mask] = Ignore DIR, optional mask some HASH bits (default 0xffffff)\n"
                 "  -refresh-    = Do NOT synchronize to latest volume checkpoint automatically\n", c0);
      }
      else
      {
         DFSAPFSLEAFCACHE *leafTree = &apfs->vCatTree; // default to catalog tree (FS info)
         ULN64             leafId   = 0;
         BOOL              testMode = TxaOption( TXA_O_TEST); // ignore directory parent inode
         ULONG             testMask = TxaOptNum( TXA_O_TEST, NULL, 0xffffff);

         if ((TxaOption( TXA_O_REFR)) || ((apfs->vTransactions > APFS_MOUNT_TR_LIMIT) && (!TxaOptUnSet( TXA_O_REFR))))
         {
            ULONG             savedVerbosity = dfsa->verbosity;
            TxPrint( "\nRefresh/synchronize to latest volume checkpoint ...\n");
            dfsa->verbosity = TXAO_SILENT;      // needed for the SB display functions!
            dfsApfsFindCheckpoint( 0, TXAO_SILENT); // silently sync to latest checkpoint
            dfsa->verbosity = savedVerbosity;
         }

         leafId = dfsApfsNextLeafLink( 0, leafTree); // 0 means first leaf present ...

         if (dfsApfsVirt2Phys( leafId, APFS_ANY_XID, apfs->vOmap) != 0) // first leafId is valid
         {
            dfsProgressInit( 0, leafTree->vNlCount - sl - 1, 0, "LeafNode:", "displayed", DFSP_STAT | DFSP_VDEC, 0);
            dfsInitList(0, "-f -P", "-d");      // optimal for file-display

            TxPrint( "\nStart search for filename-HASH collisions ...\n\n");

            for ( sl = 0;
                 (sl < leafTree->vNlCount) && (leafId != 0) && (rc == NO_ERROR) && !TxAbort();
                  sl++,                        leafId  = dfsApfsNextLeafLink( leafId, leafTree))
            {
               nav.this = dfsApfsBlock2Lsn( dfsApfsVirt2Phys( leafId, APFS_ANY_XID, apfs->vOmap));
               rc = dfsRead( nav.this, apfs->SectorsPerBlock, rbuf);
               if (rc == NO_ERROR)
               {
                  if (dfsApfsValidBlock( rbuf, apfs->BlockSize))
                  {
                     ULN64         lastPrim = 0; // last values, to detect a duplicate (collision)
                     ULONG         lastHash = 0;
                     S_BT_NODE    *sd = (S_BT_NODE *) rbuf;
                     ULONG         keyOffs   = sizeof( S_B_HEADER) + sizeof( S_BTREE_NODE) + sd->btNode.btToc.Offset + sd->btNode.btToc.Length;
                     ULONG         entry;
                     S_BT_ENTRY   *varToc  = (S_BT_ENTRY *) (sd->btData + sd->btNode.btToc.Offset);
                     S_NODE_KEY    nodeKey;
                     ULN64         rPrim;       // Primary key part of fsId (parent inode-nr)
                     ULONG         nameHash;
                     TXLN          rName;

                     for (entry = 0; (entry < sd->btNode.btCount) && (!TxAbort()); entry++)
                     {
                        nodeKey.data = (rbuf + keyOffs + varToc[ entry].nKoffset);

                        rPrim = APFSidPrim( nodeKey.in->fsId); // primary key part (parent or ino)
                        rType = APFSidType( nodeKey.in->fsId); // record type

                        if (rType == APFST_DIR_RECORD)
                        {
                           if (varToc[ entry].nKlength != nodeKey.dr->drNameLength + 10)
                           {
                              strcpy( rName, (char *) nodeKey.dh->drName);
                              nameHash = DRHash( nodeKey.dh->drLengthHash);

                              if (testMode)
                              {
                                 nameHash = nameHash & testMask; // compare a few bits only, more hits :)
                              }
                              if ((nameHash == lastHash) && ((rPrim == lastPrim) || testMode))
                              {
                                 dfsAddSI2List( nav.this, entry); // add to list
                                 TxPrint( ".%07.7llu %sR%s%3x %s%s%s DIR:%16.16llx Hash:%6.6x %s file %s%s%s\n",
                                    dfsa->snlist[0] - 1, CBM, CNN, entry, CBC, dfsApfsFsRecType( rType), CNN,
                                         rPrim, nameHash, (testMode) ? "TEST" : "COLLISION", CBY, rName, CNN);
                              }
                              lastPrim = rPrim;
                              lastHash = nameHash;
                           }
                        }
                     }
                  }
               }
               dfsProgressShow( sl + 1, 0, 0, NULL); // Progress EVERY block (could be many records)
            }
            dfsProgressTerm();

            TxPrint( "\nLeaves/nodes done : %llu of %llu  ", sl, leafTree->vNlCount - 1);
            TxPrint( "with %llu COLLISIONS added to sector list", dfsa->snlist[0]);
            if (testMode)
            {
               TxPrint( "  (mask:0x%0x)", testMask);
            }
            else
            {
            }
            TxPrint( "\n");
         }
         else
         {
            TxPrint( "\nLeaf number %llu = 0x0%llx is not a valid LEAF for tree: '%s'\n",
                        leafId, leafId, leafTree->vTreeName);
         }
      }
   }
   else if (strcasecmp(c0, "dump"     ) == 0)   // dump all blocks with valid checksum
   {
      if (TxaOption('?'))
      {
         TxPrint("\nDump recognized and valid APFS blocks to display, with some filtering\n");
         TxPrint("\n Usage:  %s  [* | blocks  [start  [types]]] [selection-options]\n\n", c0);
         TxPrint("   *           = dump until end of partition, or <Esc> key\n"
                 "   blocks      = # recognized blocks to dump,  default 2000\n"
                 "   start       = blocknumber to start dumping, default 0\n"
                 "   types       = sector-types to be included,  default all     ('?\?\?' or 'desc')\n\n");
         TxPrint("   -dir:P      = On tree/node objects, limit to parent-INO  P (use on FS-Tree only)\n"
                 "   -r:N        = On tree/node objects, limit to record type N (use on Snap/FS-Tree)\n"
                 "                 1=SMET 2=PXT 3=INO 4=XAT 5=LNK 6=DSTR 8=FXT 9=DIR 10=STAT 11=SNAP 12=LM\n"
                 "   -header-    = Suppress display of header info, show blocknumber and contents only\n"
                 "   -list       = On tree/node objects, add each found (DIR) record to the sector-list\n"
                 "   -name:wildc = On tree/node objects, display DIR/XATTR only if wildcard name match\n"
                 "   -node       = On tree/node objects, display NODEs only (tree) no LEAF (contents)\n"
                 "   -node-      = On tree/node objects, display LEAFs only (contents) no NODE (tree)\n"
                 "   -raw        = On tree/node objects, NO record interpretation, dump RAW key/value\n"
                 "   -tree[:st]  = Limit to blocks with (tree) subtype 'st' default is 14 for FS-Tree\n"
                 "                 14=FS-Tree  15=Extents-Tree  16=Snapshot-Tree  21=Fusion-Tree\n");
         TxPrint("   -X:Xid      = Limit to blocks with Xid same/smaller as given transaction-ID 'Xid'\n"
                 "   -x:subType  = Exclude blocks with object subtype given    (list with  'desc')\n"
                 "   -x          = Exclude blocks with object subtype 9        (SpcMan Free-Queue)\n"
                 "   -y:subType  = Exclude blocks with object subtype given    (list with  'desc')\n"
                 "   -y          = Exclude blocks with object subtype 11              (Object Map)\n"
                 "   -z:subType  = Exclude blocks with object subtype given    (list with  'desc')\n"
                 "   -z          = Exclude blocks with object subtype 15  (Block-Extent-reference)\n\n");
      }
      else
      {
         ULONG         tSubType = BT_NO_TYPE;   // any subtype
         ULONG         xSubType = 0xdeadface;   // exclude none
         ULONG         ySubType = 0xdeadface;   // exclude none
         ULONG         zSubType = 0xdeadface;   // exclude none
         ULN64         limitXid = TxaOptNum( 'X', NULL, apfs->cpSup->blockHdr.transactId); // default ALL

         if (TxaOptSet( DFS_O_TREE))
         {
            tSubType = TxaOptNum( DFS_O_TREE, NULL, BT_V_FS_TREE);
         }
         if (TxaOptSet( 'x'))
         {
            xSubType = TxaOptNum( 'x', NULL, BT_SPACEMANFREE);
         }
         if (TxaOptSet( 'y'))
         {
            ySubType = TxaOptNum( 'y', NULL, BT_OBJECT_MAP);
         }
         if (TxaOptSet( 'z'))
         {
            zSubType = TxaOptNum( 'z', NULL, BT_V_BLOCKREFTREE);
         }
         if (c1[0] == '*')
         {
            size = (ULONG) apfs->Block;
         }
         else
         {
            if ((size = atol(c1)) == 0)
            {
               size = 2000;                     // default display 2000 blocks
            }
         }
         sl = dfsGetMcsNumber( c2, 0);          // start block, default 0

         dfsProgressInit( 0, apfs->Block - sl, 0, "Block:", "analysed", DFSP_STAT, 0);
         if (TxaOption( TXA_O_LIST))            // create a list, allowing selective access
         {
            dfsInitList(0, "-f -P", "-d");      // optimal for menu file-recovery
         }

         nr = sl;
         for (sl = 0; (rc == NO_ERROR) && (sl < size) && !TxAbort(); nr++)
         {
            nav.this = dfsApfsBlock2Lsn( nr);
            rc = dfsRead( nav.this, apfs->SectorsPerBlock, rbuf);
            if (rc == NO_ERROR)
            {
               if (dfsApfsValidBlock( rbuf, apfs->BlockSize))
               {
                  S_B_HEADER  *hdr = (S_B_HEADER *) rbuf;

                  if ((limitXid >= hdr->transactId) && // same or smaller, default last checkpoint
                      (xSubType != hdr->objSubType) &&
                      (ySubType != hdr->objSubType) && // exclusion selections
                      (zSubType != hdr->objSubType) &&
                     ((tSubType == hdr->objSubType) || (tSubType == BT_NO_TYPE))) // tree selection
                  {
                     st = dfsApfsBlock2SecType( hdr->objectType);

                     if ((cc <= 3) || (strchr( c3, st) != 0)) // all, or selected types
                     {
                        BOOL showBlock = TRUE; // show this kind of block (NODE/LEAF in tree)

                        if ( (TxaOptSet( TXA_O_NODE)) &&
                            ((hdr->objectType == BT_BTREE_ROOT) || //- we don't want both
                             (hdr->objectType == BT_BTREE_NODE) )) //- NODE and LEAF blocks
                        {
                           S_BT_NODE *treeBlock = (S_BT_NODE *) rbuf;

                           if (TxaOption( TXA_O_NODE))  //- want just the NODE ones
                           {
                              showBlock = ((treeBlock->btNode.btFlags & BNF_LEAF) == 0);
                           }
                           else                         //- want just the LEAF ones
                           {
                              showBlock = ((treeBlock->btNode.btFlags & BNF_LEAF) == BNF_LEAF);
                           }
                        }
                        if (showBlock)
                        {
                           if (!TxaOptUnSet( TXA_O_HEADER))
                           {
                              dfsSectorTypeAsAscii(  st, s1);
                              TxPrint( "\nRecognized Block  : 0x%8.8x = sector-LSN: 0x%16.16llx = %s%s%s\n",
                                                                     nr,            nav.this, CBM, s1, CNN);
                           }
                           if (dfsApfsBlockContents( st, rbuf) == DFS_PENDING)
                           {
                              if (!TxaOptUnSet( TXA_O_HEADER))
                              {
                                 dfsApfsBlockHeader( rbuf, FALSE); // show header info
                                 TxPrint("\nStart of unrecognized block contents:\n");
                                 TxDisplHexDump( rbuf, 0x180);
                              }
                           }
                           sl++;                // one more done ...
                        }
                     }
                  }
               }
            }
            dfsProgressShow( nr, 0, sl, "Displayed:"); // Progress EVERY block (could be many records)
         }
         TxPrint( "\nRecognized blocks dumped: %u, last block analysed: 0x0%x of 0x0%llx\n", sl, nr, apfs->Block);
         dfsProgressTerm();

         if (rc == DFS_PSN_LIMIT)               // expected at end of object ...
         {
            rc = NO_ERROR;
         }
      }
   }
   else if ((strcasecmp(c0, "folder"  ) == 0))  // display folder contents (aka 'leaves' in DIR format)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nDisplay contents of catalog (leaf nodes) for specified folder-ID in directory format\n");
         TxPrint("\n Usage:  %s  [folderId]\n\n"
                   "   folderId = Hexadecimal ID for folder to display, often refered to as CnID\n\n"
                   "   -l-      = Do NOT create a new sector list from the folder contents\n", c0);
      }
      else if (apfs->vsBlk != NULL)
      {
         ULN64        leafId   = 0;
         S_NODE_KEY   key;                      // universal key pointer (union)
         key.data =   s1;                       // storage large enough for a full filename

         if ((TxaOption( TXA_O_REFR)) || ((apfs->vTransactions > APFS_MOUNT_TR_LIMIT) && (!TxaOptUnSet( TXA_O_REFR))))
         {
            ULONG             savedVerbosity = dfsa->verbosity;
            TxPrint( "\nRefresh/synchronize to latest volume checkpoint ...\n");
            dfsa->verbosity = TXAO_SILENT;      // needed for the SB display functions!
            dfsApfsFindCheckpoint( 0, TXAO_SILENT); // silently sync to latest checkpoint
            dfsa->verbosity = savedVerbosity;
         }

         memset( s1, 0, TXMAXLN);               // start out all ZERO for key fields
         id64  = dfsGetSymbolicSN( c1, 2);                   //- primary key value
         key.dh->fsId = APFSmkFsId( id64, APFST_DIR_RECORD); // hybrid primary key value
         leafId = dfsApfsSearchCatalog( key, &index, NULL);

         if ((leafId == 0) || (leafId == L64_NULL))
         {
            dfsX10(  "\nInvalid folder ID : ", id64, CBR, "\n");
         }
         else
         {
            TxPrint( "\nList entries for directory INO 0x%llx starting at leafId 0x%llx\n\n", id64, leafId);
            rc = dfsShowFolderContents( id64, leafId);
         }
      }
   }
   else if (strcasecmp(c0, "hash"     ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nCalculate combined CRC32C based Hash+length value for filenames in DIR-RECORD key\n");
         TxPrint("\n Usage:  %s  [filename]\n\n"
                 "   filename = UTF-8 (or ASCII) name of a file to calculate hash for\n", c0);
      }
      else
      {
         nr = dfsApfsNameHash( (BYTE *) c1, strlen( c1) + 1);

         TxPrint( "\nHash-Length value : 0x%8.8x for filename '%s'\n", nr, c1);
      }
   }
   else if (strcasecmp(c0, "leaves"   ) == 0)   // Iterate over tree leaf nodes, and display them
   {
      if (TxaOption('?'))
      {
         TxPrint("\nDisplay contents of various APFS filesystem trees (leafs) , with some filtering\n");
         TxPrint("\n Usage:  %s  [* | leafs  [start]] [selection-options]\n\n", c0);
         TxPrint("   *           = display until end of cached tree LEAF nodes reached, or <Esc> key\n"
                 "   leafs       = # recognized LEAF nodes to display,  default 2000\n"
                 "   start       = Leaf ID to start display, default 0\n"
                 "   -refresh-   = Do NOT synchronize to the latest volume checkpoint automatically\n"
                 "   -debug      = Only list the LEAF (virtual) node ID values, no contents display\n");
         TxPrint("   -dir:P      = On tree/node objects, limit to parent-INO  P (use on FS-Tree only)\n"
                 "   -r:N        = On tree/node objects, limit to record type N (use on Snap/FS-Tree)\n"
                 "                 1=SMET 2=PXT 3=INO 4=XAT 5=LNK 6=DSTR 8=FXT 9=DIR 10=STAT 11=SNAP 12=LM\n"
                 "   -header-    = Suppress display of header info, show blocknumber and contents only\n"
                 "   -list       = On tree/node objects, add each found (DIR) record to the sector-list\n"
                 "   -name:wildc = On tree/node objects, display DIR/XATTR only if wildcard name match\n"
                 "   -raw        = On tree/node objects, NO record interpretation, dump RAW key/value\n"
                 "   -tree[:st]  = Use prepared LEAF-cache for selected tree, default is 14 for FS-Tree\n"
                 "                 14=FS-Tree  15=Extents-Tree  16=Snapshot-Tree\n");
         TxPrint("   -X:Xid      = Limit to blocks with Xid same/smaller as given transaction-ID 'Xid'\n"
                 "   -x:subType  = Exclude blocks with object subtype given    (list with  'desc')\n"
                 "   -x          = Exclude blocks with object subtype 9        (SpcMan Free-Queue)\n"
                 "   -y:subType  = Exclude blocks with object subtype given    (list with  'desc')\n"
                 "   -y          = Exclude blocks with object subtype 11              (Object Map)\n"
                 "   -z:subType  = Exclude blocks with object subtype given    (list with  'desc')\n"
                 "   -z          = Exclude blocks with object subtype 15  (Block-Extent-reference)\n\n");
      }
      else
      {
         DFSAPFSLEAFCACHE *leafTree = &apfs->vCatTree; // default to catalog tree (FS info)
         ULONG         tSubType = BT_V_FS_TREE;
         ULONG         xSubType = 0xdeadface; // exclude none
         ULONG         ySubType = 0xdeadface; // exclude none
         ULONG         zSubType = 0xdeadface; // exclude none
         ULN64         limitXid = TxaOptNum( 'X', NULL, apfs->cpSup->blockHdr.transactId); // default ALL
         ULN64         leafId   = 0;

         if ((TxaOption( TXA_O_REFR)) || ((apfs->vTransactions > APFS_MOUNT_TR_LIMIT) && (!TxaOptUnSet( TXA_O_REFR))))
         {
            ULONG             savedVerbosity = dfsa->verbosity;
            TxPrint( "\nRefresh/synchronize to latest volume checkpoint ...\n");
            dfsa->verbosity = TXAO_SILENT;      // needed for the SB display functions!
            dfsApfsFindCheckpoint( 0, TXAO_SILENT); // silently sync to latest checkpoint
            dfsa->verbosity = savedVerbosity;
         }

         if (TxaOptSet( DFS_O_TREE))
         {
            tSubType = TxaOptNum( DFS_O_TREE, NULL, BT_V_FS_TREE);
         }
         if (TxaOptSet( 'x'))
         {
            xSubType = TxaOptNum( 'x', NULL, BT_SPACEMANFREE);
         }
         if (TxaOptSet( 'y'))
         {
            ySubType = TxaOptNum( 'y', NULL, BT_OBJECT_MAP);
         }
         if (TxaOptSet( 'z'))
         {
            zSubType = TxaOptNum( 'z', NULL, BT_V_BLOCKREFTREE);
         }
         if (c1[0] == '*')
         {
            size = (ULONG) leafTree->vNlCount;
         }
         else
         {
            if ((size = atol(c1)) == 0)
            {
               size = 2000;                     // default display 2000 blocks
            }
         }
         leafId = dfsGetMcsNumber( c2, 0);      // explicit leaf to start, or 0
         if (leafId == 0)
         {
            leafId = dfsApfsNextLeafLink( 0, leafTree);  // 0 means first leaf present ...
         }

         if (dfsApfsVirt2Phys( leafId, APFS_ANY_XID, apfs->vOmap) != 0) // first leafId is valid
         {
            switch (tSubType)
            {
               default:
               case BT_V_FS_TREE      : leafTree = &apfs->vCatTree;     break;
               case BT_EXTENT_TREE    : leafTree = &apfs->vExtTree;     break;
               case BT_V_SNAPMETATREE : leafTree = &apfs->vSnapTree;    break;
            }

            dfsProgressInit( 0, leafTree->vNlCount - sl - 1, 0, "LeafNode:", "displayed", DFSP_STAT | DFSP_VDEC, 0);
            if (TxaOption( TXA_O_LIST))         // create a list, allowing selective access
            {
               dfsInitList(0, "-f -P", "-d");   // optimal for menu file-recovery
            }

            for ( sl = 0;
                 (sl < size) && (leafId != 0) && (rc == NO_ERROR) && !TxAbort();
                  sl++,          leafId  = dfsApfsNextLeafLink( leafId, leafTree))
            {
               if (TxaOption( TXA_O_DEBUG))     // just list the LEAF node ID's (debug)
               {
                  dfsX10((sl % 10) ? " " : "\n", leafId, CNN, "");
               }
               else                             // regular leaf contents display
               {
                  if (tSubType == BT_V_FS_TREE) // need to resolve virtual ID to a block number
                  {
                     nav.this = dfsApfsBlock2Lsn( dfsApfsVirt2Phys( leafId, APFS_ANY_XID, apfs->vOmap));
                  }
                  else
                  {
                     nav.this = dfsApfsBlock2Lsn( leafId);
                  }
                  rc = dfsRead( nav.this, apfs->SectorsPerBlock, rbuf);
                  if (rc == NO_ERROR)
                  {
                     if (dfsApfsValidBlock( rbuf, apfs->BlockSize))
                     {
                        S_B_HEADER  *hdr = (S_B_HEADER *) rbuf;

                        if ((limitXid >= hdr->transactId) && // same or smaller, default last checkpoint
                            (xSubType != hdr->objSubType) &&
                            (ySubType != hdr->objSubType) && // exclusion selections
                            (zSubType != hdr->objSubType)  )
                        {
                           st = dfsApfsBlock2SecType( hdr->objectType);

                           if (!TxaOptUnSet( TXA_O_HEADER))
                           {
                              dfsSectorTypeAsAscii(  st, s1);
                              TxPrint( "\nRecognized Leaf Id: 0x%16.16llx = sector-LSN: 0x%16.16llx = %s%s%s\n",
                                                              leafId,            nav.this, CBM, s1, CNN);
                           }
                           dfsApfsBtreeNode( rbuf); // it can only be a tree node :)
                        }
                     }
                  }
               }
               dfsProgressShow( sl + 1, 0, 0, NULL); // Progress EVERY block (could be many records)
            }
            dfsProgressTerm();

            TxPrint( "\nLeaves/nodes done : %llu of %llu  ", sl, leafTree->vNlCount - 1);
            if (TxaOption( TXA_O_LIST))         // create a list, allowing selective access
            {
               TxPrint( "with %llu records added to sector list", dfsa->snlist[0]);
            }
            TxPrint("\n");
         }
         else
         {
            TxPrint( "\nLeaf number %llu = 0x0%llx is not a valid LEAF for tree: '%s'\n",
                        leafId, leafId, leafTree->vTreeName);
         }
      }
   }
   else if (strcasecmp(c0, "leaf" ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nLocate specified or NEXT LEAF node in a tree, and display its contents\n");
         TxPrint("\n Usage:  %s  [virt/phys ID]\n\n", c0);
         TxPrint("  virt/phys Id = LEAF node ID to find/display (virtual or physical block-number)\n"
                 "                 when not specified, will find the NEXT LEAF node in the tree\n\n");
         TxPrint("  -tree[:st]   = Use prepared LEAF-cache for selected tree, default is 14 for FS-Tree\n"
                 "                 14=FS-Tree  15=Extents-Tree  16=Snapshot-Tree\n");
      }
      else
      {
         DFSAPFSLEAFCACHE *leafTree = &apfs->vCatTree;     //- default to catalog tree (FS info)
         ULONG             tSubType = BT_V_FS_TREE;
         static ULN64      leafId   = 0;                   //- starting LEAF ID (first)

         leafId = dfsGetMcsNumber( c1, (ULONG) leafId);    //- start leaf, default last one found

         if ((cc == 1) || (leafId == 0))        // need to find NEXT leaf, not specified one
         {
            switch (tSubType)
            {
               default:
               case BT_V_FS_TREE      : leafTree = &apfs->vCatTree;     break;
               case BT_EXTENT_TREE    : leafTree = &apfs->vExtTree;     break;
               case BT_V_SNAPMETATREE : leafTree = &apfs->vSnapTree;    break;
            }
            leafId   = dfsApfsNextLeafLink( leafId, leafTree); // search next leaf node ID in cache
         }
         nav.this = dfsApfsBlock2Lsn( dfsApfsVirt2Phys( leafId, APFS_ANY_XID, apfs->vOmap));

         if (leafId != 0)
         {
            TxPrint( "\n(next)LEAF nodeID : 0x0%llx, in tree: '%s'\n", leafId, leafTree->vTreeName);

            sprintf( dc, "0x%llx", nav.this);   // auto display the node
            TxaReParseKeepOpts( dc);            // reparse but keep option values
            rc = DFS_PENDING;                   // handle translated command
         }
         else                                   // reached last leaf in the tree
         {
            TxPrint( "\nReached last LEAF node in the tree, repeat '%s' to restart with first ...\n", c0);
         }
      }
   }
   else if (strcasecmp(c0, "super"    ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nDisplay the current CONTAINER superblock structure\n");
         TxPrint("\n Usage:  %s  [-O:q | -O:v]\n\n"
                 "   -O:q    = Display superblock in a compact format, using less lines (quiet)\n"
                 "   -O:v    = Display superblock with more detail,    using more lines (verbose)\n\n"
                 " The UP link (command 'u') is set to the corresponding Checkpoint MAP\n", c0);
      }
      else
      {
         nav.up = apfs->cpLsn - apfs->SectorsPerBlock; // easy way to navigate to CheckPoint Map
         sprintf( dc, "0x%llx", apfs->cpLsn);
         TxaReParseKeepOpts( dc);               // reparse but keep option values
         rc = DFS_PENDING;                      // handle translated command
      }
   }
   else if (strcasecmp(c0, "desc"     ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint("\nDescribe object TYPE/SUBTYPE with select char, and FS/XFIELD RECORD types\n");
         TxPrint("\n Usage:  %s  [first [last]]\n\n"
                 "   first = First numeric value to show the type description for\n"
                 "   last  = Last numeric value to show, default same value as 'first'\n\n"
                 "  Without specified values, all defined types are listed, and the\n"
                 "  Btree FS/XFIELD RECORD types are listed with a short description as well.\n", c0);
      }
      else
      {
         if (cc > 1)
         {
            sn = atol( c1);
            sl = sn;
            if (cc > 2)
            {
               sl = atol( c1);
            }
         }
         else
         {
            sn = 1;
            sl = BT_GENERIC_BITMAP_BLK;
         }
         TxPrint( "\nAPFS block/object types/subtypes (see 'dump -?', 'leaves -?', 'find' and '?\?\?' commands)\n\n");
         for (nr = sn; nr < sl; nr++)
         {
            TxPrint( "(sub) Type number :%5u = 0x%4.4X : %s%-25.25s%s", nr, nr, CBY, dfsApfsBlockType( nr), CNN);
            if ((st = dfsApfsBlock2SecType( nr)) != ST_UDATA)
            {
               TxPrint( "    '%s%c%s' = Block/Sector selection char", CBM, st, CNN);
            }
            TxPrint( "\n");
         }
         if (cc == 1)                           // no params, show RECORDs too
         {
            TxPrint( "\n");
            for (nr = 1; nr < max( APFST_COUNT, XFT_COUNT); nr++)
            {
               if (nr < APFST_COUNT)            // in range for SF record type
               {
                  TxPrint( "Btree RECORD type :%5u = 0x%4.4X : %s%-14.14s%s", nr, nr, CBC, dfsApfsFsRecType( nr), CNN);
               }
               else
               {
                  TxPrint( "%*.*s", 50, 50, "");
               }
               if (nr < XFT_COUNT)              // in range for XF record type
               {
                  TxPrint( "Xfield type :%5u = 0x%4.4X : %s%-14.14s%s", nr, nr, CNC, dfsApfsXfieldType( nr, FALSE), CNN);
               }
               TxPrint( "\n");
            }
         }
      }
   }
   else if (strcasecmp(c0, "vi"      ) == 0)
   {
      strcpy( s1, "");                          // init to no value
      if (!TxaOption('?'))                      // No explicit help request
      {
         ULONG  listcode  = TXDID_OK;

         #if defined (USEWINDOWING)
            TXRECT where  = {18,0,0,0};         // fixed position

            if ((dfsa->FsModeSelist != NULL) && (TxaOption('P')))
            {
               dfsa->FsModeSelist->selected = apfs->vIndex; // default to CURRENT volume

               listcode = txwListBox( TXHWND_DESKTOP, TXHWND_DESKTOP, &where,
                        " Select a Volume to work with ", "",
                          DFSC_APFSVI, TXLB_MOVEABLE,
                          cMenuTextStand, cMenuBorder_top, // same color as menu
                          dfsa->FsModeSelist);

               if (listcode >= TXDID_MAX)       // return list selection
               {                                // plus current item base value
                  sprintf( s1, "%u", listcode - TXDID_MAX); // calculate volume index
               }
            }
         #endif                                 // USEWINDOWING

         if (listcode != TXDID_CANCEL)          // not escaped ...
         {
            if ((cc > 1) && (strlen(s1) == 0))  // No selection from list, use param
            {
               strcpy(  s1, c1);
            }
            if (strlen( s1))                    // We have some volume specification
            {
               apfs->vIndex = atol( s1);
               apfs->vMountWarning = FALSE;     // mountwarning not given yet

               rc = dfsApfsSelectVolume( apfs->vIndex, dfsa->verbosity); //- activate the volume
               if (rc == NO_ERROR)
               {
                  dfsInitList(0, "-f -P", "-d"); // clear any old (invalid) list
               }
            }
         }
      }
      else
      {
         TxPrint( "Select a specific Volume to be used with associated directory-tree\n\n");
         TxPrint( " Usage: %s  [index | -P] [-O:q | -O:v]\n\n"
                  "  index = Volume index, 0 .. %u                      (see 'super' display)\n\n"
                  "  -O:q  = Display superblock in a compact format, using less lines (quiet)\n"
                  "  -O:v  = Display superblock with more detail,    using more lines (verbose)\n"
                  "  -P    = Select volume by NAME, from list of available ones\n", c0, apfs->vCount);
      }
   }
   else if (strcasecmp(c0, "virt"   ) == 0)     // Find oid/xid, in volume or container ObjectMap
   {                                            // and display the block when found
      if ((TxaOption('?')) || (cc < 2))
      {
         TxPrint("\nFind Object/Transaction-ID pair, in Volume/Container Object-Map, and display\n");
         TxPrint("\n Usage:  %s  oid  [xid]  [-c] [-O:q | -O:v]\n\n"
                 "   oid   = Virtual Object-ID to find, from superblocks and related structures\n"
                 "   xid   = Transaction-ID to match (max), matches ANY Xid when not specified\n\n"
                 "   -c    = Search in overall CONTAINER Object-Map, not the selected-VOLUME one\n"
                 "   -O:q  = Display object in a compact format, using less lines (quiet)\n"
                 "   -O:v  = Display object with more detail,    using more lines (verbose)\n", c0);
      }
      else
      {
         ULN64         blocknr;
         ULN64         oid;
         ULN64         xid;
         S_BT_NODE    *omap = (TxaOption('c')) ? apfs->cOmap : apfs->vOmap;

         oid = dfsGetSymbolicSN( c1, 0);             //- Object-ID, no default    (0 is invalid)
         xid = dfsGetSymbolicSN( c2, APFS_ANY_XID);  //- Transaction-ID, default ANY (max value)

         if ((blocknr = dfsApfsVirt2Phys( oid, xid, omap)) != 0) // find latest, verbose, use index nr
         {
            dfsX10("\nPhysical block nr : ", blocknr, CBG, "\n");
            sprintf( dc, "0x0%llX", dfsApfsBlock2Lsn( blocknr));
            TxaReParseKeepOpts( dc);            // reparse but keep option values
            rc = DFS_PENDING;                   // handle translated command
         }
         else
         {
            TxPrint( "\nVirtual object ID : 0x%16.16llx, Transaction-ID 0x%16.16llx, not found!\n", oid, xid);
         }
      }
   }
   else
   {
      rc = DFS_CMD_UNKNOWN;                     // cmd not recognized
   }
   RETURN (rc);
}                                               // end 'dfsApfsCommand'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// APFS filesystem, identify specified sector as a valid superblock
/*****************************************************************************/
BOOL dfsApfsIsSuperBlock                        // RET   sector is a valid sb
(
   BYTE               *sec                      // IN    sector contents
)
{
   return (!memcmp(((S_C_SUPER*)sec)->Signature,sg_c_super,SG_C_SUPER));
}                                               // end 'dfsApfsIsSuperBlock'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// APFS filesystem, display sector contents as a superblock
/*****************************************************************************/
void dfsApfsDisplaySuperBlock
(
   BYTE               *sec                      // IN    sector contents
)
{
   dfsApfsBlockCsuper( sec, TRUE);              // signal just a single sector!
}                                               // end 'dfsApfsDisplaySuperBlock'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// APFS filesystem, identify specified sector
/*****************************************************************************/
static ULONG dfsApfsIdent
(
   ULN64               sn64,                    // IN    LSN for sector
   ULN64               d2,                      // IN    dummy
   char               *st,                      // OUT   sector type
   void               *sec                      // IN    sector contents
)                                               //       usually rbuf ...
{
   ULONG               dr  = NO_ERROR;
   BYTE                rc  = ST_UDATA;

   ENTER();

   //- perform the quick detections first (for faster searching etc)
   if      (!memcmp(((S_C_SUPER*)sec)->Signature,sg_c_super,SG_C_SUPER)) rc=ST_C_SUP;
   else if (!memcmp(((S_V_SUPER*)sec)->Signature,sg_v_super,SG_V_SUPER)) rc=ST_V_SUP;
   else if (!memcmp(((S_B_EFI_J*)sec)->Signature,sg_c_efi_j,SG_C_EFI_J)) rc=ST_EFIBT;

   if (rc == ST_UDATA)
   {
      S_B_HEADER     *hdr = (S_B_HEADER *) sec;

      if ((hdr->checksum   != 0) &&
          (hdr->checksum   != DFS64MAX) &&
          (hdr->transactId != 0))               // possible APFS block with header
      {
         switch (APFShdrType( hdr))
         {
            case BT_C_SUPERBLOCK        :
            case BT_BTREE_ROOT          :
            case BT_BTREE_NODE          :
            case BT_SPACEMANAGER        :
            case BT_SPACEMAN_CAB        :
            case BT_SPACEMAN_CIB        :
            case BT_SPACEMANBMAP        :
            case BT_SPACEMANFREE        :
            case BT_OBJECT_MAP          :
            case BT_CHECKP_MAP          :
            case BT_V_SUPERBLOCK        :
            case BT_REAPER              :
            case BT_REAP_LIST           :
            case BT_EFI_JUMPSTART       :
            case BT_FUSION_WRBACK_CACHE :
            case BT_FUSION_WRBACK_LIST  :
            case BT_ENCR_ROLLING_STATE  :
            case BT_GENERIC_BITMAP      :
            case BT_GENERIC_BITMAP_BLK  :
            case BT_UNIVERSAL_TEST_TYPE :
               if (sec == rbuf)                 // sec in rbuf, read whole block
               {
                  if ((dfsRead( sn64, apfs->SectorsPerBlock, rbuf)) == NO_ERROR)
                  {
                     if (dfsApfsValidBlock( rbuf, apfs->BlockSize))
                     {
                        rc = dfsApfsBlock2SecType( hdr->objectType);
                     }
                  }
               }
               break;

            default:
               break;
         }
      }
   }
   switch (rc)
   {
      case ST_UDATA:
         dr = DFS_PENDING;
         break;

      default:
         break;
   }
   *st = rc;
   RETURN (dr);
}                                               // end 'dfsApfsIdent'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// APFS filesystem, supply sector-type description string
/*****************************************************************************/
static ULONG dfsApfsStype
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *st,                      // IN    sector type
   void               *data                     // OUT   type description
)
{
   ULONG               rc  = NO_ERROR;
   char               *buf = (char *) data;

   switch (*st)                                 // searchable types
   {
      case ST_C_SUP: sprintf(buf,"APFS C-superblock"); break;
      case ST_BTREE: sprintf(buf,"APFS Btree (root)"); break;
      case ST_ANODE: sprintf(buf,"APFS generic node"); break;
      case ST_SPMGR: sprintf(buf,"APFS SpaceManager"); break;
      case ST_SPCAB: sprintf(buf,"Sp CAB lvl2 Index"); break;
      case ST_SPCIB: sprintf(buf,"Sp CIB lvl1 Index"); break;
      case ST_SPBMP: sprintf(buf,"Sp  BitMap  Block"); break;
      case ST_OBMAP: sprintf(buf,"Object Map  Block"); break;
      case ST_CHKPT: sprintf(buf,"CheckPoint Map   "); break;
      case ST_V_SUP: sprintf(buf,"Volume Superblock"); break;
      case ST_REAPR: sprintf(buf,"Container Reaper "); break;
      case ST_REAPL: sprintf(buf,"Cont. Reaper List"); break;
      case ST_EFIBT: sprintf(buf,"EFIboot JumpStart"); break;
      case ST_FUWBC: sprintf(buf,"Fusion WrBk Cache"); break;
      case ST_FUWBL: sprintf(buf,"Fusion WrBk List "); break;
      case ST_ENCRS: sprintf(buf,"Encr RollingState"); break;
      case ST_GBMAP: sprintf(buf,"General Bitmap   "); break;
      case ST_GBMBL: sprintf(buf,"Gen BitMap List  "); break;
      default:
         switch ((*st) | ST__INFO)              // non-searchable ones
         {
            default:       rc = DFS_PENDING;
         }
         break;
   }
   return (rc);
}                                               // end 'dfsApfsStype'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// APFS filesystem, display APFS block contents at PSN based on type
/*****************************************************************************/
static ULONG dfsApfsBlockAtPsn
(
   ULN64               psn,                     // IN    base psn for sector
   ULN64               d2,                      // IN    dummy
   char               *type,                    // IN    type of sector
   void               *dummy                    // IN    sector contents (rbuf)
)
{
   ULONG               rc = NO_ERROR;
   BYTE                st = (BYTE) *type;

   ENTER();

   //- First re-read this sector, now in full block-size
   rc = dfstReadPsn( DFSTORE, psn, apfs->SectorsPerBlock, rbuf);
   if (rc == NO_ERROR)
   {
      //- to be refined, may add calculated LSN as param, so function can read more
      rc = dfsApfsBlockContents( st, rbuf);
   }
   RETURN (rc);
}                                               // end 'dfsApfsBlockAtPsn'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// APFS filesystem, display Block-contents from given block buffer
/*****************************************************************************/
static ULONG dfsApfsBlockContents
(
   BYTE                blockType,               // IN    type of block (st)
   BYTE               *blockBuffer              // IN    buffer with one block
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   switch (blockType)
   {
      case ST_C_SUP: (void) dfsApfsBlockCsuper(   blockBuffer, FALSE);   break;
      case ST_BTREE:
      case ST_ANODE: (void) dfsApfsBtreeNode(     blockBuffer);          break;
      case ST_SPMGR: (void) dfsApfsSpaceManager(  blockBuffer);          break;
      case ST_SPCAB: (void) dfsApfsSpaceMgrCab(   blockBuffer);          break;
      case ST_SPCIB: (void) dfsApfsSpaceMgrCib(   blockBuffer);          break;
      case ST_SPBMP: (void) dfsApfsSpaceMgrBmap(  blockBuffer);          break;
      case ST_OBMAP: (void) dfsApfsObjectMap(     blockBuffer);          break;
      case ST_CHKPT: (void) dfsApfsCheckPointMap( blockBuffer);          break;
      case ST_V_SUP: (void) dfsApfsBlockVsuper(   blockBuffer);          break;
      case ST_REAPR: (void) dfsApfsBlockReaper(   blockBuffer);          break;
      case ST_REAPL: (void) dfsApfsReaperList(    blockBuffer);          break;
      case ST_EFIBT: (void) dfsApfsEfiJumpStart(  blockBuffer);          break;
      case ST_FUWBC: (void) dfsApfsFusionWbCache( blockBuffer);          break;
      case ST_FUWBL: (void) dfsApfsFusionWbList(  blockBuffer);          break;
      case ST_ENCRS: (void) dfsApfsEncrRollState( blockBuffer);          break;
      case ST_GBMAP: (void) dfsApfsGeneralBitMap( blockBuffer);          break;
      case ST_GBMBL: (void) dfsApfsGenBmapBlock(  blockBuffer);          break;

      default:
         rc = DFS_PENDING;
         break;
   }
   RETURN (rc);
}                                               // end 'dfsApfsBlockContents'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// APFS filesystem, display special LSN info (Node lsn + record-index value)
/*****************************************************************************/
static ULONG dfsApfsDisplayLsnInfo
(
   ULN64               lsn,                     // IN    possible special lsn
   ULN64               info,                    // IN    possible REC index
   char               *dc,                      // IN    dummy
   void               *data                     // IN    dummy
)
{
   ULONG               rc = DFS_PENDING;
   TXLN                path;
   int                 recType;

   ENTER();

   if (info & DFSSNINFO)                        // extra info present
   {
      TxPrint( "\n");
      if ((dfsApfsLsnInfo2Path( lsn, DFSSNIGET( info), &recType, path) == NO_ERROR) && (strlen( path)))
      {
         TxPrint("FsTree %s%s%s record for: '%s%s%s'\n\n", CBC, dfsApfsFsRecType( recType), CNN, CBY, path, CNN);
      }
      nav.xtra = lsn;                           // keep reference arround
      rc = dfsApfsCatalogRecord( lsn, DFSSNIGET( info));
   }
   RETURN (rc);
}                                               // end 'dfsApfsDisplayLsnInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display all available info about a Node Record, in an FS-Tree Catalog Node
/*****************************************************************************/
static ULONG dfsApfsCatalogRecord
(
   ULN64               nodeLsn,                 // IN    lsn of Catalog Node
   USHORT              index                    // IN    index of record 0..N
)
{
   ULONG               rc = 0;
   char                recKind = 0;             // record: (f)ile, (D)ir or (S)ymlink
   ULN64               ino = 0;
   DFSISPACE           isp;                     // integrated alloc SPACE info

   ENTER();
   TRACES(("nodeLsn: 0x%llx, entry: 0x%4.4hx\n", nodeLsn, index));

   dfsApfsDirHeader( " Node-sector - Index", 0);

   dfsX10(  " ", nodeLsn,     CBC, "      ");
   TxPrint( "%s0x%s%2.2hx%s", CNZ, CNC,  index, CNN);
   ino = dfsApfsShowCatRecord( nodeLsn, index, CcZ, &recKind);

   //- Note: recKind LINK already has its value displayed by the ShowCatRecord
   if (recKind == 'f')                          // show file allocation
   {
      if ((rc = dfsApfsGetAllocSpace( nodeLsn, index, NULL, &isp)) == NO_ERROR)
      {
         dfsUllDot20( "\nData stream  size : ", isp.byteSize, " bytes,   ");
         dfsSz64("Sectors : ", dfsSspaceSectors( TRUE, isp.chunks, isp.space), "\n\n");
         dfsSspaceDisplay( SD_BLOCKS | SD_BOTLINE | SD_NAVDOWN, isp.chunks, isp.space);
         TxFreeMem( isp.space);
      }
   }
   else if (recKind == 'D')                     // auto display folder contents
   {
      TXTM             command;

      sprintf(         command, "folder 0x%llx", ino);
      dfsMultiCommand( command, 0, TRUE, FALSE, TRUE); // include command echo
   }
   RETURN( rc);
}                                               // end 'dfsApfsCatalogRecord'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display APFS Block, Container super-block
/*****************************************************************************/
static ULONG dfsApfsBlockCsuper                 // RET   rc = 0 if type match
(
   BYTE               *buffer,                  // IN    Block contents
   BOOL                singleSector             // IN    buffer is just one sector
)
{
   ULONG               rc = 0;                  // rc, sector match
   S_C_SUPER          *sd = (S_C_SUPER *) buffer;
   ULONG               i;
   BOOL                navMap = FALSE;          // nav.down leads to ObjectMap

   ENTER();

   if (dfsa->verbosity > TXAO_SILENT)
   {
      dfsApfsBlockHeader( buffer, singleSector); // show header info
   }

   if (dfsa->verbosity > TXAO_QUIET)
   {
      TxPrint("Next transact Xid : 0x%16.16llx        Next Oid : 0x%16.16llx\n",
                                   sd->csNextTransactId, sd->csNextPhysBlkOid);

      if (singleSector == FALSE)                // display info beyond 1st sector
      {
         TxPrint("Counter#   SET CS : 0x%16.16llx        Failed CS: 0x%16.16llx\n",
                  sd->csCounters[ SB_CNT_CHECKSUM_SET],  sd->csCounters[ SB_CNT_CHECKSUM_FAIL]);

         if (sd->csEfiJumpStart != 0)
         {
            nav.xtra = dfsApfsBlock2Lsn( sd->csEfiJumpStart);
            TxPrint("EFI JumpStart Blk : 0x%16.16llx        BOOT  Lsn: 0x%s%16.16llx%s\n",
                                             sd->csEfiJumpStart, CBY, nav.xtra, CNN);
         }
      }

      TxPrint("FS extra Features : 0x%16.16llx        %s", sd->csFeatures,      CBG);
      if (sd->csFeatures     &  CFF_DEFRAG)           TxPrint("Defrag  ");
      if (sd->csFeatures     &  CFF_LCFD)             TxPrint("Low-Cap-Fusion-Mode  ");
      if (sd->csFeatures     & ~CFI_VALIDMASK)        TxPrint("%sUnknown!",     CBR);
      TxPrint("%s\n", CNN);

      TxPrint("ReadOnly Features : 0x%16.16llx        %s", sd->csRoCompatible,  CBY);
      if (sd->csRoCompatible & ~CFI_VALIDMASK)        TxPrint("%sUnknown!",     CBR);
      TxPrint("%s\n", CNN);

      TxPrint("Incompat Features : 0x%16.16llx        %s", sd->csInCompatible,  CBC);
      if (sd->csInCompatible &  CFI_VERSION1)         TxPrint("APFS-version-1  ");
      if (sd->csInCompatible &  CFI_VERSION2)         TxPrint("APFS-version-2  ");
      if (sd->csInCompatible &  CFI_FUSION)           TxPrint("Fusion-Drives  ");
      if (sd->csInCompatible & ~CFI_VALIDMASK)        TxPrint("%sUnknown!",     CBR);
      TxPrint("%s\n", CNN);
   }

   if (dfsa->verbosity > TXAO_SILENT)
   {
      dfsSz64("FS Container size : ",              apfs->Sect,  "");
      TxPrint("  # blocks : 0x%16.16llx = %llu\n", sd->csTotalBlocks, sd->csTotalBlocks);
   }
   if (dfsa->verbosity > TXAO_QUIET)
   {
      dfsSiz8("Sect/Block   size : ",           apfs->SectorsPerBlock, "\n");
   }
   if (dfsa->verbosity >= TXAO_VERBOSE)
   {
      TxPrint("SbDESC BlockCount : 0x%8.8x Next: 0x%8.8x BaseBlk: 0x%16.16llx\n",
               sd->csSbDescBlockCount, sd->csNextSbDescBlock, sd->csSbDescBaseBlock);

      nav.up   = dfsApfsBlock2Lsn( sd->csSbDescBaseBlock + sd->csCpFirstDescBlock);
      TxPrint("Chkpnt desc index : 0x%8.8x Len : 0x%8.8x At  LSN: 0x%s%16.16llx%s\n",
               sd->csCpFirstDescBlock, sd->csCpDescLength, CBC, nav.up,   CNN);

      TxPrint("SbDATA BlockCount : 0x%8.8x Next: 0x%8.8x BaseBlk: 0x%16.16llx\n",
               sd->csSpDataBlockCount, sd->csNextSbDataBlock, sd->csSbDataBaseBlock);

      TxPrint("Chkpnt data index : 0x%8.8x Len : 0x%8.8x At  LSN: 0x%16.16llx\n",
               sd->csCpFirstDataBlock, sd->csCpDataLength,
                     dfsApfsBlock2Lsn( sd->csSbDataBaseBlock + sd->csCpFirstDataBlock));
   }

   nav.xtra = dfsApfsBlock2Lsn( sd->csObjectMapBlock); // 'x' goes to the Object Map
   nav.down = dfsApfsBlock2Lsn( dfsApfsVirt2Phys( sd->csVolumeOid[ apfs->vIndex], APFS_ANY_XID, apfs->cOmap));
   if (nav.down == 0)                           // No volume LSN, incorrect virtual ID (non-active superblock?)
   {
      nav.down = nav.xtra;                      // <ENTER> goes to Object Map too ...
      navMap = TRUE;
   }

   if (dfsa->verbosity > TXAO_QUIET)
   {
      TxPrint("Object-Map  block : 0x%16.16llx",       sd->csObjectMapBlock);
      TxPrint( "          At  LSN: 0x%s%16.16llx%s      Signature : %s%4.4s%s\n",
             (navMap) ? CBG : CNN, dfsApfsBlock2Lsn(   sd->csObjectMapBlock), CNN, CBM, sd->Signature, CNN);
      TxPrint("Oid Ckpt SpaceMgr : 0x%16.16llx\n",     sd->csCpSpManObjId);
      TxPrint("Oid Reaper object : 0x%16.16llx\n",     sd->csReaperObjId);
   }

   for (i = 0; i < sd->csMaxVolCount; i++)
   {
      if ( (dfsa->verbosity > TXAO_SILENT) && (sd->csVolumeOid[ i] != 0) &&
          ((dfsa->verbosity > TXAO_QUIET)  || (i == apfs->vIndex)))
      {
         TxPrint( "Oid for Volume %2u : 0x%s%16.16llx%s\n", i,
                ((navMap == FALSE) && (i == apfs->vIndex)) ? CBG : CNN, sd->csVolumeOid[ i], CNN);
      }
      if (singleSector == FALSE)                // display rest of info beyond 1st sector
      {
         if (i >= 40)                           // only 40 volumes fit in 1st sector
         {
            break;
         }
      }
   }
   if (dfsa->verbosity > TXAO_SILENT)
   {
      TxPrint("FS Container UUID : %s\n", dfsFsUuidValueNoSwap( sd->csContainerGuid));
   }
   RETURN (rc);
}                                               // end 'dfsApfsBlockCsuper'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display APFS Block, Volume super-block
/*****************************************************************************/
ULONG dfsApfsBlockVsuper                        // RET   rc = 0 if type match
(
   BYTE               *block                    // IN    Block contents
)
{
   ULONG               rc = 0;                  // rc, sector match
   S_V_SUPER          *sd = (S_V_SUPER *) block;
   TXLN                text;
   TXTM                timestr;
   int                 i;
   BOOL                navMap = FALSE;          // nav.down leads to ObjectMap

   ENTER();

   if (dfsa->verbosity > TXAO_SILENT)
   {
      dfsApfsBlockHeader( block, FALSE);        // show header info
   }

   if (dfsa->verbosity > TXAO_QUIET)
   {
      TxPrint("FS extra Features : 0x%16.16llx        %s", sd->vsFeatures,          CBG);
      if (sd->vsFeatures     &  VFF_OLDDEFRAG     )   TxPrint("OLD Defrag! ");
      if (sd->vsFeatures     &  VFF_HARDLINKS     )   TxPrint("Hardlinks  ");
      if (sd->vsFeatures     &  VFF_DEFRAG        )   TxPrint("Defrag  ");
      if (sd->vsFeatures     & ~VFF_VALIDMASK     )   TxPrint("%sUnknown!",         CBR);
      TxPrint("%s\n", CNN);

      if (dfsa->verbosity >= TXAO_VERBOSE)
      {
         TxPrint("ReadOnly Features : 0x%16.16llx     %s", sd->vsRoCompatible,      CBY);
         if (sd->vsRoCompatible & ~VFI_VALIDMASK  )   TxPrint("%sUnknown!",         CBR);
         TxPrint("%s\n", CNN);
      }
      TxPrint("Incompat Features : 0x%16.16llx        %s", sd->vsInCompatible,      CBC);
      if (sd->vsInCompatible &  VFI_CASE_INSENS   )   TxPrint("Case-insensitive  ");
      if (sd->vsInCompatible &  VFI_DATALESS_SNAPS)   TxPrint("Empty Snapshots  ");
      if (sd->vsInCompatible &  VFI_ENCRYPT_ROLLED)   TxPrint("Encryption-Rolled  ");
      if (sd->vsInCompatible &  VFI_NORMALI_INSENS)   TxPrint("Normalization-insensitive  ");
      if (sd->vsInCompatible & ~VFI_VALIDMASK     )   TxPrint("%sUnknown!",         CBR);
      TxPrint("%s\n", CNN);

      TxPrint("Volume Flags      : 0x%16.16llx        %s", sd->vsVolumeFlags,       CBM);
      if (sd->vsVolumeFlags  &  VFL_UNENCRYPTED   )   TxPrint("Unencrypted  ");
      if (sd->vsVolumeFlags  &  VFL_EFFACEABLE    )   TxPrint("Effaceable  ");
      if (sd->vsVolumeFlags  &  VFL_RESERVED_4    )   TxPrint("Reserved-flag!  ");
      if (sd->vsVolumeFlags  &  VFL_ONE_KEY       )   TxPrint("OneKeyEncr  ");
      if (sd->vsVolumeFlags  &  VFL_SPILLEDOVER   )   TxPrint("SpilledOver!  ");
      if (sd->vsVolumeFlags  &  VFL_SPOVERCLEAN   )   TxPrint("SpillCleanNeeded  ");
      if (sd->vsVolumeFlags  &  VFL_CHECKEXTENTREF)   TxPrint("CheckExtentRef  ");
      if (sd->vsVolumeFlags  & ~VFL_VALIDMASK     )   TxPrint("%sUnknown-flags!",   CBR);
      TxPrint("%s\n", CNN);
   }

   //- Try to go to the ROOT of the filesystem-tree next, works for active volume superblock ...
   nav.down = dfsApfsBlock2Lsn( dfsApfsVirt2Phys( sd->vsRootTreeOid, APFS_ANY_XID, apfs->vOmap));
   if (nav.down == 0)                           // incorrect virtual ID (non-active superblock?)
   {
      nav.down = dfsApfsBlock2Lsn( sd->vsObjectMapOid); // go to the Object Map instead
      navMap = TRUE;
   }
   if (dfsa->verbosity > TXAO_SILENT)
   {
      TxPrint("FS-Tree  virt Oid : 0x%s%16.16llx%s        ", (navMap) ? CNN : CBG, sd->vsRootTreeOid, CNN);
      TxPrint(  "Obj:%s%s%s : 0x%8.8x = %s%s%s\n", CBC,
                         (sd->vsRootTreeType & OID_TYPE_PHYS) ? "PHYS" :
                         (sd->vsRootTreeType & OID_TYPE_EPHM) ? "EPHM" : "VIRT",                      CNN,
                          sd->vsRootTreeType, CBY, dfsApfsBlockType( sd->vsRootTreeType),             CNN);
   }
   if (dfsa->verbosity > TXAO_QUIET)
   {
      TxPrint("ExtentsTree  @Blk : 0x%16.16llx        ", sd->vsExtentsTreeOid);
      TxPrint(  "Obj:%s%s%s : 0x%8.8x = %s%s%s\n", CBC,
                         (sd->vsExtentTreeType & OID_TYPE_PHYS) ? "PHYS" :
                         (sd->vsExtentTreeType & OID_TYPE_EPHM) ? "EPHM" : "VIRT",                 CNN,
                          sd->vsExtentTreeType, CBY, dfsApfsBlockType( sd->vsExtentTreeType),      CNN);

      TxPrint("Object-Map   @Blk : 0x%16.16llx",         sd->vsObjectMapOid);
      TxPrint( "        At   LSN : 0x%s%16.16llx%s      Signature : %s%4.4s%s\n",
                                          (navMap) ? CBG : CNN, nav.down, CNN, CBM, sd->Signature, CNN);

      if (dfsa->verbosity >= TXAO_VERBOSE)
      {
         TxPrint("SnapshotTree @Blk : 0x%16.16llx        ", sd->vsSnapMetaTreeOid);
         TxPrint(  "Obj:%s%s%s : 0x%8.8x = %s%s%s\n", CBC,
                            (sd->vsSnapMtTreeType & OID_TYPE_PHYS) ? "PHYS" :
                            (sd->vsSnapMtTreeType & OID_TYPE_EPHM) ? "EPHM" : "VIRT",              CNN,
                             sd->vsSnapMtTreeType, CBY, dfsApfsBlockType( sd->vsSnapMtTreeType),   CNN);

         if (sd->vsRootToXid != 0)
         {
            TxPrint("Snapshot ROOT Xid : 0x%16.16llx\n", sd->vsRootToXid);
         }
         if (sd->vsErStateOid != 0)
         {
            TxPrint("Encrypt state Oid : 0x%16.16llx\n", sd->vsErStateOid);
         }
         if (sd->vsUnknownValue != 0)
         {
            TxPrint("Unknown 64b Value : 0x%16.16llx\n", sd->vsUnknownValue);
         }
      }
      TxPrint("Next Document  ID : 0x%8.8lx\n",       sd->vsNextDocumentId);
   }
   apfs->vTransactions = sd->blockHdr.transactId - sd->vsLastMountXid;
   if (dfsa->verbosity > TXAO_SILENT)
   {
      TxPrint("Last mounted   on : %s   Xid: 0x%16.16llx %s%3lld%s transactions since mount\n",
         dfsApfsTime2str( sd->vsLastMountTime, 3, timestr), sd->vsLastMountXid, (apfs->vTransactions == 0) ? CBG :
                       (apfs->vTransactions <= APFS_MOUNT_TR_LIMIT) ? CBY : CBR, apfs->vTransactions,        CNN);
   }
   if (dfsa->verbosity >= TXAO_VERBOSE)
   {
      for (i = 0; i < V_I_NUM; i++)
      {
         if (sd->vsModHistory[ i].vsIdTimestamp != 0)
         {
            TxCopy(  text, sd->vsModHistory[ i].vsIdString, V_I_LEN + 1);
            TxPrint("Mount/Modified on : %s   Xid: 0x%16.16llx By: %s\n",
                     dfsApfsTime2str( sd->vsModHistory[ i].vsIdTimestamp, 3, timestr),
                                      sd->vsModHistory[ i].vsIdVersion,      text);
         }
      }
   }
   if (dfsa->verbosity > TXAO_QUIET)
   {
      TxCopy(  text, sd->vsFormattedBy.vsIdString, V_I_LEN + 1);
      TxPrint("Volume  Format on : %s   Xid: 0x%16.16llx By: %s\n",
               dfsApfsTime2str( sd->vsFormattedBy.vsIdTimestamp, 3, timestr),
                                sd->vsFormattedBy.vsIdVersion,      text);
   }
   if (dfsa->verbosity > TXAO_SILENT)
   {
      TxCopy(  text, sd->vsVolumeName, V_N_LEN + 1);
      TxPrint("Volume - FS  Name : %s%s%s\n", CBG, text, CNN);
      TxPrint("Volume - FS  UUID : %s       ", dfsFsUuidValueNoSwap( sd->vsVolumeGuid));
      TxPrint("V-Role : 0x%4.4hx = %s", sd->vsVolumeRole,                       CBY);
      if (sd->vsVolumeRole    ==  V_ROLE_NONE      )   TxPrint("No-defined-role  ");
      if (sd->vsVolumeRole     &  V_ROLE_SYSTEM    )   TxPrint("System-ROOT '/'  ");
      if (sd->vsVolumeRole     &  V_ROLE_USER      )   TxPrint("Users-HOME-DIRs  ");
      if (sd->vsVolumeRole     &  V_ROLE_RECOVERY  )   TxPrint("Recovery-system  ");
      if (sd->vsVolumeRole     &  V_ROLE_VM        )   TxPrint("VirtMemory-SWAP  ");
      if (sd->vsVolumeRole     &  V_ROLE_PREBOOT   )   TxPrint("PreBoot-Encrypt  ");
      if (sd->vsVolumeRole     &  V_ROLE_INSTALLER )   TxPrint("Installer-Usage  ");
      if (sd->vsVolumeRole     &  V_ROLE_DATA      )   TxPrint("iOS-User-Data  ");
      if (sd->vsVolumeRole     &  V_ROLE_BASEBAND  )   TxPrint("iOS-BaseBand  ");
      if (sd->vsVolumeRole     &  V_ROLE_RESERVED  )   TxPrint("Apple-Reserved  ");
      TxPrint("%s\n", CNN);
   }
   RETURN (rc);
}                                               // end 'dfsApfsBlockVsuper'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display APFS Block, Generic BTree node, ROOT, INDEX, LEAF or ROOT+LEAF
/*****************************************************************************/
static ULONG dfsApfsBtreeNode                   // RET   rc = 0 if type match
(
   BYTE               *block                    // IN    Block contents
)
{
   ULONG               rc = 0;                  // rc, sector match
   S_BT_NODE          *sd = (S_BT_NODE *) block;
   ULONG               keyOffs   = sizeof( S_B_HEADER) + sizeof( S_BTREE_NODE) + sd->btNode.btToc.Offset + sd->btNode.btToc.Length;
   ULONG               valOffs   = APFS_BLOCKSIZE;
   ULONG               entry;
   ULN64              *p64;                     // pointer to 64-bit key or value
   TXLN                rName;                   // recordname modifyable copy
   TXLN                tmpstr;
   int                 i;

   ENTER();

   if (sd->btNode.btFlags & BNF_LEAF)           // Set NEXT LEAF as default <ENTER> location
   {
      ULN64 nextLeaf = dfsApfsNextLeafLink( sd->blockHdr.objectId, &apfs->vCatTree);
      if (nextLeaf == 0)                        // was the LAST LEAF, wrap to FIRST
      {
         nextLeaf = dfsApfsNextLeafLink( 0, &apfs->vCatTree);
      }
      nav.down = dfsApfsBlock2Lsn( dfsApfsVirt2Phys( nextLeaf, APFS_ANY_XID, apfs->vOmap));
   }

   if (!TxaOptUnSet( TXA_O_HEADER))             // unless header suppressed
   {
      dfsApfsBlockHeader( block, FALSE);        // show header info, full block

      if (APFSobjType( sd) == BT_BTREE_ROOT)
      {
         if (sd->btInfo.bfNodeSize == APFS_BLOCKSIZE)
         {
            TxPrint("Tree level  flags : 0x%8.8x                %s", sd->btInfo.bfFlags, CBM);
            if (sd->btInfo.bfFlags  &  BTF_UINT64_KEYS     ) TxPrint("64bit-keys  ");
            if (sd->btInfo.bfFlags  &  BTF_SEQ_INSERT      ) TxPrint("SeqInsert  ");
            if (sd->btInfo.bfFlags  &  BTF_ALLOW_GHOSTS    ) TxPrint("AllowGhosts  ");
            if (sd->btInfo.bfFlags  &  BTF_EPHEMERAL       ) TxPrint("Epehemeral  ");
            if (sd->btInfo.bfFlags  &  BTF_PHYSICAL        ) TxPrint("Physical  ");
            if (sd->btInfo.bfFlags  &  BTF_NONPERSISTENT   ) TxPrint("NonPeristent  ");
            if (sd->btInfo.bfFlags  &  BTF_KV_NONALIGNED   ) TxPrint("NonAligned  ");
            if (sd->btInfo.bfFlags  & ~BTF_VALIDMASK       ) TxPrint("%sUnknown-flags!", CBR);
            TxPrint("%s\n", CNN);

            TxPrint("Key / Value sizes :%4u /%5u                Longest Key / Value :%4u /%5u\n",
                sd->btInfo.bfMinKeySize, sd->btInfo.bfMinValSize, sd->btInfo.bfMaxKeySize, sd->btInfo.bfMaxValSize);

            dfsX10( "Tree # of Entries : ", sd->btInfo.bfEntryCount, CNN, " =");
            TxPrint("%11llu   ",            sd->btInfo.bfEntryCount);
            dfsX10( "Nodes: ",              sd->btInfo.bfNodeCount,  CNN, "   = ");
            TxPrint("%3llu\n",              sd->btInfo.bfNodeCount);
         }
         else                                   // field not present on other sizes (garbage)
         {
            TxPrint("Node Size  @4096  : 0x%8.8x is non-standard, multiple block nodes?\n", sd->btInfo.bfNodeSize);
         }
      }
      TxPrint("Node # of Entries : 0x%8.8lx = %10u   Node level in Btree : %3hu\n",
                     sd->btNode.btCount, sd->btNode.btCount, sd->btNode.btLevel);

      TxPrint("Node level  flags : 0x%4.4hx                    %s", sd->btNode.btFlags, CBC);
      if (sd->btNode.btFlags  &  BNF_ROOT            ) TxPrint("ROOT  ");
      if (sd->btNode.btFlags  &  BNF_LEAF            ) TxPrint("LEAF  ");
      if (sd->btNode.btFlags  &  BNF_FIXED_SIZE      ) TxPrint("FixedSize  ");
      if (sd->btNode.btFlags  &  BNF_KOFF_INVAL      ) TxPrint("OffsetInvalid!  ");
      if (sd->btNode.btFlags  & ~BNF_VALIDMASK       ) TxPrint("%sUnknown-flags!",      CBR);
      TxPrint("%s\n", CNN);
   }

   if (APFSobjType( sd) == BT_BTREE_ROOT)
   {
      valOffs -= sizeof( S_BTREE_INFO);         // values start just before the INFO struct
   }

   if (dfsa->verbosity > TXAO_QUIET)            // list entries too, unless in quiet mode
   {
      if (sd->btNode.btFlags  &  BNF_FIXED_SIZE) // TOC only has offsets, no sizes
      {
         ULONG            keySize = 16;         // most likely size
         ULONG            valSize = 16;         // most likely size
         S_BT_FIXED      *fixToc  = (S_BT_FIXED *) (sd->btData + sd->btNode.btToc.Offset);

         if (APFSobjType( sd) == BT_BTREE_ROOT)
         {
            keySize  = sd->btInfo.bfMinKeySize;
            valSize  = sd->btInfo.bfMinValSize;
         }
         else                                   // try to derive key and value size
         {
            for (entry = 0; (entry < sd->btNode.btCount); entry++) // examine the TOC values, keys
            {
               if ((fixToc[ entry].nKoffset & 0x0008) == 0x0008) // there is a non-16byte aligned offset
               {
                  keySize = 8;                  // so assume 8 byte values
                  break;
               }
            }
            if (sd->btNode.btFlags & BNF_LEAF)  // Only LEAF has different sizes
            {
               for (entry = 0; entry < sd->btNode.btCount; entry++) // examine the TOC values, values
               {
                  if ((fixToc[ entry].nVoffset & 0x0008) == 0x0008) // there is a non-16byte aligned offset
                  {
                     valSize = 8;               // so assume 8 byte values
                     break;
                  }
               }
            }
            else
            {
               valSize = 8;                     // non-LEAF nodes are fixed 8 byte size
            }
            TxPrint("Key / Value sizes :%4u /%5u                (calculated from TOC values)\n", keySize, valSize);
         }
         TRACES(("key size:%4u offset: 0x%3x  value size:%4u offset: 0x%3x  1st Key: 0x%16.16llx\n",
                  keySize, keyOffs, valSize, valOffs,  *((ULN64 *) (block + keyOffs))));

         //- list the key and value fields, formatted depending on known sizes
         for (entry = 0; (entry < sd->btNode.btCount) && (!TxAbort()); entry++)
         {
            p64 = (ULN64 *) (block + keyOffs + fixToc[ entry].nKoffset);
            TxPrint( "    Entry%4u Key : %16.16llx", entry, *p64);
            if (keySize > 8)
            {
               p64++;
               TxPrint( " %16.16llx ", *p64);
            }
            p64 = (ULN64 *) (block + valOffs - fixToc[ entry].nVoffset);
            if ((entry == 0) && (valSize <= 8))
            {
               TxPrint( "  Value : %s%16.16llx%s", CBG, *p64, CNN);
               nav.down = dfsApfsBlock2Lsn( *p64);
            }
            else
            {
               TxPrint( "  Value : %16.16llx", *p64);
            }
            if (valSize > 8)                    // for LEAF nodes only!
            {
               p64++;
               if (entry == 0)
               {
                  TxPrint( " %s%16.16llx%s", CBG, *p64, CNN); // indicate nav.down value set
                  nav.down = dfsApfsBlock2Lsn( *p64);
               }
               else
               {
                  TxPrint( "  2nd: %16.16llx", *p64);
               }
            }
            if (valSize > 16)                   // show rest as auto hex/ascii stream
            {                                   // (most likely not occuring in APFS)
               p64++;
               TxPrint( "  Rest:%s%s%s", CNC, TxData2AutoHexAscii( (BYTE *) p64,
               min( valSize - 16, ((DFS_SCROLL_W - 80) * 10) / 32), TXMAXLN, tmpstr), CNN);
            }
            TxPrint( "\n");
         }
      }
      else                                      // TOC has both offset and size info
      {
         S_BT_ENTRY   *varToc  = (S_BT_ENTRY *) (sd->btData + sd->btNode.btToc.Offset);
         S_NODE_KEY    nodeKey;
         S_NODE_VAL    nodeVal;                 // pointer union to various known types
         int           rSelect = TxaOptNum( 'r', NULL, 0); // record type select, default ALL
         int           rType;                   // FS record type
         ULN64         rPrim;                   // Primary key part of fsId
         ULN64         pSelect = TxaOptNum( TXA_O_DIR, NULL, 0); // parent-ID select, default ALL
         BOOL          nameWildCardReject;      // record rejected based on DIR/XATTR name
         TXTM          nameWildCard;            // value of the wildcard string, or empty
         ULONG         nameHash;
         BOOL          useRawFormat = TxaOptSet( TXA_O_RAW);

         strcpy( nameWildCard, TxaOptStr( TXA_O_NAME, NULL, "")); // default no wildcard (display all)

         for (entry = 0; (entry < sd->btNode.btCount) && (!TxAbort()); entry++)
         {
            nodeKey.data = (block + keyOffs + varToc[ entry].nKoffset);
            nodeVal.data = (block + valOffs - varToc[ entry].nVoffset);

            if ((sd->blockHdr.objSubType == BT_V_SNAPMETATREE) ||
                (sd->blockHdr.objSubType == BT_V_FS_TREE)      ||
                (sd->blockHdr.objSubType == BT_EXTENT_TREE)     )
            {
               useRawFormat = TxaOptSet( TXA_O_RAW);
            }
            else                                // always use RAW for other records
            {
               useRawFormat = TRUE;
            }
            rPrim = APFSidPrim( nodeKey.in->fsId); // primary key part (parent or ino)
            rType = APFSidType( nodeKey.in->fsId); // record type

            nameWildCardReject = FALSE;         // unless rejected ...

            if ((rType == APFST_DIR_RECORD) ||
                (rType == APFST_INODE)      ||
                (rType == APFST_XATTR)      ||
                (rType == APFST_SNAP_NAME)   )
            {
               switch (rType)                   // Evaluate selection by name wildcard
               {
                  case APFST_DIR_RECORD:
                     if (varToc[ entry].nKlength == nodeKey.dr->drNameLength + 10)
                     {
                        strcpy( rName, (char *) nodeKey.dr->drName);
                        TxRepl( rName, 0x0d, ' '); // prevent misalignments in display
                     }
                     else                       // HASH+LENGTH type DIR-record
                     {
                        strcpy( rName, (char *) nodeKey.dh->drName);
                        nameHash = dfsApfsNameHash( (BYTE *) rName, strlen( rName) + 1);
                        TxRepl( rName, 0x0d, ' '); // prevent misalignments in display
                     }
                     break;

                  case APFST_INODE:             // get dir/filename from Inode Xfield
                     if (varToc[ entry].nVlength > sizeof( S_INODE_VAL)) // Xfields present
                     {
                        S_XF_BLOB *xfb  = (S_XF_BLOB *) (nodeVal.in->inXfields);
                        BYTE      *data = ((BYTE *) xfb->xfData) + (xfb->xfExtents * sizeof( S_XF_FIELD));

                        if (xfb->xfData[ 0].xfType == XFT_INO_FILENAME)
                        {
                           strcpy( rName, (char *) data);
                        }
                     }
                     break;

                  case APFST_XATTR:
                     strcpy(  rName, (char *) nodeKey.xa->xaName);
                     break;

                  default:
                  case APFST_SNAP_NAME:
                     strcpy(  rName, (char *) nodeKey.sn->snName);
                     break;
               }
               if (           (strlen( nameWildCard) != 0) && // wildcard is there,
                   (TxStrWicmp( rName, nameWildCard) <  0)  ) // and does not match
               {
                  nameWildCardReject = TRUE;    // reject this record for display
               }
            }

            if ((nameWildCardReject == FALSE)          && // record passes selection criteria
                ((rSelect == 0) || (rSelect == rType)) &&
                ((pSelect == 0) || (pSelect == rPrim))  )
            {
               if (useRawFormat)                // RAW or other subtypes, use generic display
               {
                  p64 = (ULN64 *) nodeKey.data;
                  TxPrint( "Entry%4x KeyOffs :%4hx  Len:%4hu = %16.16llx", entry,
                            keyOffs + varToc[ entry].nKoffset, varToc[ entry].nKlength, *p64);

                  p64 = (ULN64 *) nodeVal.data;
                  TxPrint( "   ValOffs :%4hx  Len:%4hu", valOffs - varToc[ entry].nVoffset, varToc[ entry].nVlength);
                  if (varToc[ entry].nVlength == 8)
                  {
                     TxPrint( " = %16.16llx", *p64);
                  }
                  TxPrint( "   Data:%s%s%s\n", CNC, TxData2AutoHexAscii( (BYTE *) p64,
                               min( varToc[ entry].nVlength, ((DFS_SCROLL_W - 90) * 10) / 32), TXMAXLN, tmpstr), CNN);
               }
               else                             // Well-known record-tytpes, formatted display
               {
                  if (TxaOption( TXA_O_LIST))   // create a list, allowing selective access
                  {
                     dfsAddSI2List( nav.this, entry); // add to list
                     TxPrint( ".%07.7llu %sR%s%3x %s%s%s ", dfsa->snlist[0] - 1,
                                CBM, CNN, entry, CBC, dfsApfsFsRecType( rType), CNN);
                  }
                  else                          // no list, show blocknr when no header; and rec-nr
                  {
                     if (TxaOptUnSet( TXA_O_HEADER)) // unless header suppressed
                     {
                        TxPrint( "%sB%s", CBM, CNN);
                        dfsX10(":", dfsApfsLsn2Block( nav.this), CNN, " ");
                        TxPrint( "%sR%s%3x : %s%s%s ", CBM, CNN, entry, CBC, dfsApfsFsRecType( rType), CNN);
                     }
                     else
                     {
                        TxPrint( "R %3x : %s%s%s: ", entry, CBC, dfsApfsFsRecType( rType), CNN);
                     }
                  }
                  if (rPrim != APFSI_MASK)
                  {
                     dfsX10(  "", rPrim, CNN, " "); // Primary key value
                  }
                  else
                  {
                     TxPrint( "-no-value- ");
                  }

                  switch (rType)                // (secondary) KEY section display
                  {
                     case APFST_DIR_RECORD:
                        if (varToc[ entry].nKlength == nodeKey.dr->drNameLength + 10)
                        {
                           TxPrint( "0x%4.4hx %s%-60s%s", nodeKey.dr->drNameLength,
                                                          nodeKey.dr->drNameLength, CBY, rName, CNN);
                        }
                        else               // HASH+LENGTH type DIR-record (hash RED when incorrect)
                        {
                           if ((dfsa->verbosity >= TXAO_VERBOSE) && (nodeKey.dh->drLengthHash != nameHash))
                           {
                              TxPrint( "%8.8x ", nameHash); // include FULL calculated namehash value too
                           }
                           TxPrint( "%s%6.6x %s%-60s%s", (nodeKey.dh->drLengthHash == nameHash) ? CNN : CBR,
                                                         (nodeKey.dh->drLengthHash >> 8), CBY, rName,   CNN);
                        }
                        break;

                     case APFST_XATTR:
                        TxPrint( "0x%4.4hx %s%-60s%s", nodeKey.xa->xaNameLength, CNC, rName, CNN);
                        break;

                     case APFST_FILE_EXTENT:
                        dfsX10("Offset:", nodeKey.fe->feLogAddress, CNN, " at ");
                        break;

                     case APFST_SIBLING_LINK:
                        dfsX10("SiblId:", nodeKey.sl->slSibId, CNN, " ");
                        break;

                     case APFST_SNAP_NAME:
                        TxPrint( "       %s%-45s%s", CBM, nodeKey.sn->snName, CNN);
                        break;

                     default:                   // rest has no secondary keys
                        break;
                  }

                  if (sd->btNode.btFlags & BNF_LEAF)
                  {
                     switch (rType)             // display VALUE section for specific record types
                     {
                        case APFST_DIR_RECORD:
                           TxPrint( "%s%s%s ", CBM, dfsApfsDirRecType( nodeVal.dr->drFlags), CNN);
                           dfsX10(  "Ino:",                APFSidPrim( nodeVal.dr->fileId),  CNN, "  ");
                           TxPrint( "Cre:%s",         dfsApfsTime2str( nodeVal.dr->drDateAdded, 0, tmpstr));

                           if ((dfsa->verbosity > TXAO_QUIET) && // list Xfields too, unless in quiet mode
                               (varToc[ entry].nVlength > sizeof( S_DREC_VAL))) // there are Xfields
                           {
                              S_XF_BLOB *xfb  = (S_XF_BLOB *) (nodeVal.dr->drXfields);
                              BYTE      *data = ((BYTE *) xfb->xfData) + (xfb->xfExtents * sizeof( S_XF_FIELD));

                              if      (TxaOption( TXA_O_LIST))     TxPrint( "\n         %sx%s ",     CBM, CNN);
                              else if (TxaOptUnSet( TXA_O_HEADER)) TxPrint( "\n             %sx%s ", CBM, CNN);
                              else                                 TxPrint( "\n x ");

                              for (i = 0; i < xfb->xfExtents; i++)
                              {
                                 TxPrint( "%2d%s %s%s%s%s ",      i, (TxaOption( TXA_O_LIST)) ? "" : " :",
                                          CNC, dfsApfsXfieldType( xfb->xfData[ i].xfType, TRUE), CNN,
                                    ((TxaOptUnSet( TXA_O_HEADER)) || (TxaOption( TXA_O_LIST))) ? "" : ":");

                                 TRACES(( "Xfield:%d size: %hu at:%p offset:0x%x\n", i, xfb->xfData[ i].xfSize, data, data - block));

                                 switch (xfb->xfData[ i].xfType)
                                 {
                                    case XFT_DREC_SIBLING_ID:   TxPrint( "0x%16.16llx ", (ULN64 *) data);      break;

                                    default:
                                       TxPrint( "= 0x%hhx Flg:0x%hhx Size:0x%4.4hx ", xfb->xfData[ i].xfType,
                                                             xfb->xfData[ i].xfFlags, xfb->xfData[ i].xfSize);
                                       break;
                                 }
                                 data += ((xfb->xfData[ i].xfSize + 7) & 0xFFF8); //  next Xfield (8 byte size align)
                              }
                           }
                           TxPrint( "\n");
                           break;

                        case APFST_INODE:
                           if (dfsa->verbosity >= TXAO_VERBOSE)
                           {
                              dfsX10(  "Dstr:",       APFSidPrim( nodeVal.in->inObjectId),  CNN, " ");
                           }
                           dfsX10(  "Parent:",        APFSidPrim( nodeVal.in->inParentId),  CNN, " ");
                           TxPrint( "%s%s%s ",             CBM, ((nodeVal.in->inMode & S_IFLNK) == S_IFLNK) ? "Link" :
                                                         S_ISDIR( nodeVal.in->inMode)  ? "Dir " : "file", CNN);
                           TxPrint( "Mode (%03o) %s ",          ( nodeVal.in->inMode & 0777),
                                             dfsPosixMode2String( nodeVal.in->inMode,   12, tmpstr));
                           TxPrint( "Gid:%-7u Uid:%-7u ",         nodeVal.in->inGid, nodeVal.in->inUid);
                           TxPrint( "Bsd:0x%8.8x  ",              nodeVal.in->inBsdFlags);
                           TxPrint( "Mod:%s ",   dfsApfsTime2str( nodeVal.in->inModTime, 0, tmpstr));
                           TxPrint( "Acc:%s ",   dfsApfsTime2str( nodeVal.in->inAccTime, 0, tmpstr));
                           TxPrint( "Chg:%s ",   dfsApfsTime2str( nodeVal.in->inChgTime, 0, tmpstr));
                           TxPrint( "Cre:%s  ",  dfsApfsTime2str( nodeVal.in->inCreTime, 0, tmpstr));
                           TxPrint( "RefCount: %llu",             nodeVal.in->inRefCount);

                           if ((dfsa->verbosity > TXAO_QUIET) && // list Xfields too, unless in quiet mode
                               (varToc[ entry].nVlength > sizeof( S_INODE_VAL))) // there are Xfields
                           {
                              S_XF_BLOB *xfb  = (S_XF_BLOB *) (nodeVal.in->inXfields);
                              BYTE      *data = ((BYTE *) xfb->xfData) + (xfb->xfExtents * sizeof( S_XF_FIELD));

                              if      (TxaOption( TXA_O_LIST))     TxPrint( "\n         %sx%s ",     CBM, CNN);
                              else if (TxaOptUnSet( TXA_O_HEADER)) TxPrint( "\n             %sx%s ", CBM, CNN);
                              else                                 TxPrint( "\n x ");

                              for (i = 0; i < xfb->xfExtents; i++)
                              {
                                 TxPrint( "%2d%s %s%s%s%s ",      i, (TxaOption( TXA_O_LIST)) ? "" : " :",
                                          CNC, dfsApfsXfieldType( xfb->xfData[ i].xfType, FALSE), CNN,
                                    ((TxaOptUnSet( TXA_O_HEADER)) || (TxaOption( TXA_O_LIST))) ? "" : ":");

                                 TRACES(( "Xfield:%d size: %hu at:%p offset:0x%x\n", i, xfb->xfData[ i].xfSize, data, data - block));

                                 switch (xfb->xfData[ i].xfType)
                                 {
                                    case XFT_INO_SNAPSHOT_ID:
                                    case XFT_INO_DELTATREEID:
                                    case XFT_INO_DIRSTAT_KEY:
                                    case XFT_INO_SPARSEBYTES:
                                    case XFT_INO_PREV_FSIZE:  TxPrint( "0x%16.16llx ",    (ULN64 *) data     );  break;
                                    case XFT_INO_RDEVICE_ID:
                                    case XFT_INO_DOCUMENT_ID: TxPrint( "0x%8.8x ",        (ULONG *) data     );  break;
                                    case XFT_INO_FILENAME:    TxPrint( "%s%-43s%s",        CBY,     data, CNN);  break;
                                    case XFT_INO_FINDERINFO:  TxPrint( "-Opaque- "                           );  break;
                                    case XFT_INO_FS_UUID:     TxPrint( "%s ", dfsFsUuidValueNoSwap( data)    );  break;
                                    case XFT_INO_DATASTREAM:
                                       dfsUllDot20( "",          ((S_DSTREAM *) data)->dsByteSize,          " byte " );
                                       dfsSz64XiB(  "in ",       ((S_DSTREAM *) data)->dsAllocSize, 1,  "     Bytes ");
                                       dfsUllDot20( "Written:",  ((S_DSTREAM *) data)->dsBytesWritten,        "     ");
                                       dfsUllDot20( "Rd:",       ((S_DSTREAM *) data)->dsBytesRTead,       " Crypto:");
                                       if (((S_DSTREAM *) data)->dsCryptoId == 0)
                                       {
                                          TxPrint( " None ");
                                       }
                                       else
                                       {
                                          TxPrint( "%16.16llx ", ((S_DSTREAM *) data)->dsCryptoId);
                                       }
                                       break;

                                    default:
                                       TxPrint( "= 0x%hhx Flg:0x%hhx Size:0x%4.4hx ", xfb->xfData[ i].xfType,
                                                             xfb->xfData[ i].xfFlags, xfb->xfData[ i].xfSize);
                                       break;
                                 }
                                 data += ((xfb->xfData[ i].xfSize + 7) & 0xFFF8); //  next Xfield (8 byte size align)
                              }
                           }
                           TxPrint( "\n");
                           break;

                        case APFST_XATTR:
                           TxPrint( "  Len:0x%4.4hx ", nodeVal.xa->xaLength);
                           if (nodeVal.xa->xaFlags & XAF_EMBEDDED)
                           {
                              TxPrint( "Data%s%s%s\n", CNC, TxData2AutoHexAscii( nodeVal.xa->xaData,
                                  min( nodeVal.xa->xaLength, ((DFS_SCROLL_W - 126) * 10) / 32), TXMAXLN, tmpstr), CNN);
                           }
                           else                 // data is ULN64 data-stream fsId
                           {
                              dfsX10(  "Dstr:", *((ULN64 *) nodeVal.xa->xaData),  CNN, "\n");
                           }
                           break;

                        case APFST_FILE_EXTENT:
                           dfsX10(  "BlckNr:",            nodeVal.fe->feBlockNr,       CNN, " ");
                           TxPrint( "Flgs:0x%2.2llx ",   (nodeVal.fe->feLengthFlags >> APFSFK_SHIFT));
                           dfsSz64XiB( "Size:",          (nodeVal.fe->feLengthFlags  & APFSFL_MASK), 1, " ");
                           if (nodeVal.fe->feCryptoId == 0)
                           {
                              TxPrint( "Crypt:none\n");
                           }
                           else                 // crypto tweak value, or crypto-object-ID
                           {
                              TxPrint( "CrId:0x%16.16llx\n", nodeVal.fe->feCryptoId);
                           }
                           break;

                        case APFST_PHYS_EXTENT:
                           dfsX10(  "OwnOid:",            nodeVal.pe->peOwningOid,    CNN, " ");
                           TxPrint( "Kind:%1.1llx ",     (nodeVal.pe->peLengthKind >> APFSPK_SHIFT));
                           dfsSz64XiB( "Size:",          (nodeVal.pe->peLengthKind  & APFSPL_MASK), apfs->BlockSize, " ");
                           TxPrint( "RefCnt: %u\n",       nodeVal.pe->peRefCount);
                           break;

                        case APFST_DSTREAM_ID:
                           TxPrint( "RefCnt:%10u\n",      nodeVal.di->diRefCount);
                           break;

                        case APFST_SIBLING_LINK:
                           dfsX10(  "Parent:",            nodeVal.sl->slParentIno, CNN, " ");
                           TxPrint( "%s%s%s\n", CNY,      nodeVal.sl->slName,      CNN);
                           break;

                        case APFST_SIBLING_MAP:
                           dfsX10("Linked Inode : ",      nodeVal.sp->spFileId, CNN, "\n");
                           break;

                        case APFST_SNAP_NAME:
                           dfsX10(  "Xid:",               nodeVal.sn->snXid,  CNN, "\n");
                           break;

                        case APFST_SNAP_METADATA:
                           TxPrint( "%s %s%-45s%s",      (nodeVal.sm->smFlags & SMF_SNAP_PENDING_DATALESS) ? "NoData" : "In-Use",
                                                                       CBM, nodeVal.sm->smName, CNN);
                           dfsX10(  "Vsb:",                                 nodeVal.sm->smVolSuper,  CNN, " ");
                           TxPrint( "Mod:%s ",             dfsApfsTime2str( nodeVal.sm->smChangeTime, 0, tmpstr));
                           TxPrint( "Cre:%s  ",            dfsApfsTime2str( nodeVal.sm->smCreateTime, 0, tmpstr));
                           dfsX10(  "ExTree:",                              nodeVal.sm->smExtTreeOid,  CNN, " ");
                           TxPrint("Obj:%s%s%s : 0x%8.8x = %s%s%s\n", CBC, (nodeVal.sm->smExtTreeType & OID_TYPE_PHYS) ? "PHYS" :
                                                                           (nodeVal.sm->smExtTreeType & OID_TYPE_EPHM) ? "EPHM" : "VIRT", CNN,
                                                                            nodeVal.sm->smExtTreeType, CBY, dfsApfsBlockType( nodeVal.sm->smExtTreeType), CNN);
                           //- to be refined, extent-tree info to be added ?
                           break;

                        default:
                           TxPrint( "Undecoded Data:%s%s%s\n", CNC, TxData2AutoHexAscii( (BYTE *) &nodeVal.data,
                                     min( varToc[ entry].nVlength, ((DFS_SCROLL_W - 40) * 10) / 32), TXMAXLN, tmpstr), CNN);
                           break;
                     }
                  }
                  else                          // Nodes only have a NODE link as data!
                  {
                     switch (rType)           // display EMPTY (secondary) KEY section where needed
                     {
                        case APFST_DIR_RECORD:
                        case APFST_XATTR:  // already have an actual secondary key, length 47 chars
                           break;

                        default:          // rest has no secondary keys, fill up to align node link
                           TxPrint( "%67.67s", "");
                           break;
                     }
                     p64 = (ULN64 *) nodeVal.data;
                     dfsX10(  "  Node:", *p64,  (entry == 0) ? CBG : CNN, "\n");
                     if (entry == 0)
                     {
                        nav.down = dfsApfsBlock2Lsn( dfsApfsVirt2Phys( *p64, APFS_ANY_XID, apfs->vOmap));
                     }
                  }
               }
            }
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsApfsBtreeNode'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display APFS Block, Space Manager
/*****************************************************************************/
static ULONG dfsApfsSpaceManager                // RET   rc = 0 if type match
(
   BYTE               *block                    // IN    Block contents
)
{
   ULONG               rc = 0;                  // rc, sector match
   S_SPACEMANAGER     *sd = (S_SPACEMANAGER *) block;
   ULONG               i;

   ENTER();

   dfsApfsBlockHeader( block, FALSE);           // show header info, full block

   if (dfsa->verbosity >= TXAO_VERBOSE)
   {
      TxPrint( "SpaceMg blocksize : 0x%8.8x =%6u   bytes\n",          sd->spBlockSize,      sd->spBlockSize);
      TxPrint( "Structure version : 0x%8.8x =%6u        Structure base size : 0x%8.8x = %6u\n",
                sd->spVersion, sd->spVersion,           sd->spBaseSize, sd->spBaseSize);
   }
   TxPrint("#Blocks per chunk : 0x%8.8x =%6u ",                   sd->spBlocksPerChunk, sd->spBlocksPerChunk);
   TxPrint("  with a maximum of %u chunks/CIB and %u CIBs/CAB\n", sd->spChunksPerCib,   sd->spCibsPerCab);

   dfsApfsSpaceMgrDev( block, &sd->spDev[ SPD_MAIN], "MAIN ");
   if ((sd->spDev[ SPD_TIER2].spdBlockCount != 0) || (dfsa->verbosity >= TXAO_VERBOSE))
   {
      dfsApfsSpaceMgrDev( block, &sd->spDev[ SPD_TIER2], "TIER2");
   }
   TxPrint("Reaper Object Blk : 0x%16.16llx        (tentative, not in APPLE spec!)\n",     sd->spReaperObjectBlock);

   TxPrint("I-Pool TxMultiply : 0x%8.8x =%6u        SpaceManager Flag : 0x%8.8x  %s",
       sd->spIpTxMultiplier, sd->spIpTxMultiplier, sd->spFlags, CBC);
   if (sd->spFlags  &  SPF_VERSIONED       ) TxPrint("Versioned  ");
   if (sd->spFlags  & ~SPF_VALIDMASK       ) TxPrint("%sUnknown-flags!", CBR);
   TxPrint("%s\n", CNN);

   //- Note that layout/order of fields is DIFFERENT from APPLE spec since ReaperBlock is missing from that
   TxPrint("I-Pool BM CxB Blk : 0x%16.16llx        I-Pool block  Cnt : 0x%8.8x = %6u\n", sd->spIpBaseBlock,   sd->spIpBlockCount ,   sd->spIpBlockCount);
   TxPrint("I-Pool Bitmap Blk : 0x%16.16llx        I-Pool bitmap Cnt : 0x%8.8x = %6u\n", sd->spIpBmBaseBlock, sd->spIpBmBlockCount , sd->spIpBmBlockCount);

   dfsSz64( "Reserve Nr Blocks : ", sd->spFsReserveBlockCount * apfs->SectorsPerBlock, "  ");
   dfsSz64( "Reserve Allocated : ", sd->spFsReserveAllocCount * apfs->SectorsPerBlock, "\n");

   if ((sd->spFreeQ[ SFQ_IPOOL].sfqTreeOid != 0) || (dfsa->verbosity >= TXAO_VERBOSE))
   {
      dfsApfsSpaceMgrSfq( block, &sd->spFreeQ[ SFQ_IPOOL], "IPOOL");
   }
   if ((sd->spFreeQ[ SFQ_MAIN ].sfqTreeOid != 0) || (dfsa->verbosity >= TXAO_VERBOSE))
   {
      dfsApfsSpaceMgrSfq( block, &sd->spFreeQ[ SFQ_MAIN ], "MAIN ");
   }
   if ((sd->spFreeQ[ SFQ_TIER2].sfqTreeOid != 0) || (dfsa->verbosity >= TXAO_VERBOSE))
   {
      dfsApfsSpaceMgrSfq( block, &sd->spFreeQ[ SFQ_TIER2], "TIER2");
   }

   if (dfsa->verbosity >= TXAO_VERBOSE)
   {
      TxPrint( "BMfree Head index : 0x%4.4hx = %9hu        BMfree Tail index : 0x%4.4hx = %9hu\n",
               sd->spIpBmFreeHead,        sd->spIpBmFreeHead,       sd->spIpBmFreeTail,  sd->spIpBmFreeTail);

      TxPrint( "IP Bitmap   Xid @ : 0x%8.8x =%6u        I-Pool Bitmap Xid : 0x%16.16llx\n",
                sd->spIpBmXidOffset, sd->spIpBmXidOffset,           *((ULN64 *) (block + sd->spIpBmXidOffset)));

      TxPrint( "IP relat Bitmap @ : 0x%8.8x =%6u        I-Pool rel Bitmap : 0x%16.16llx\n",
                sd->spIpBitMapOffset,     sd->spIpBitMapOffset,     *((ULN64 *) (block + sd->spIpBitMapOffset)));

      TxPrint( "IP Bitmap index @ : 0x%8.8x =%6u        I-Pool bitmap index value array:",
                sd->spIpBmFreeNextOffset, sd->spIpBmFreeNextOffset);

      for (i = 0; i < sd->spIpBmBlockCount; i++)
      {
         if ((i % 8) == 0)                      // split to 8 values per line
         {
            TxPrint( "\n       index  %3u :", i);
         }
         TxPrint( " 0x%4.4hx ", *((USHORT *) (block + sd->spIpBmFreeNextOffset + i + i)));
      }
      TxPrint( "\n");
   }
   RETURN (rc);
}                                               // end 'dfsApfsSpaceManager'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display APFS Space Manager DEVICE structure with CAB/CIB blk# array
/*****************************************************************************/
static void dfsApfsSpaceMgrDev
(
   BYTE               *block,                   // IN    Block contents SPMGR
   S_SPMGR_DEVICE     *dev,                     // IN    Device in this SPMGR
   char               *devName                  // IN    Device name, length 5
)
{
   ULONG               i;
   ULN64              *indexBlock;              // array of CAB/CIB block numbers
   TXTT                lead;

   ENTER();

   if ((dev != NULL) && (dev->spdIndexOffset < APFS_BLOCKSIZE))
   {
      sprintf( lead, "%s device #blk : ", devName);
      dfsSz64( lead,                dev->spdBlockCount *  apfs->SectorsPerBlock, "  ");
      dfsSz64("Free : ",            dev->spdFreeCount  *  apfs->SectorsPerBlock, "  ");
      TxPrint("Index @: 0x%4.4x\n", dev->spdIndexOffset);
      TxPrint("      bitmap #blk : 0x%16.16llx = %llu indexed by %u leaf-level (CIB) and %u node-level (CAB)\n",
                     dev->spdChunkCount, dev->spdChunkCount, dev->spdCibCount, dev->spdCabCount);

      indexBlock = (ULN64 *) (block + dev->spdIndexOffset);
      if (dev->spdCabCount != 0)                // Links are NODE / CAB structure  (huge TiB containers)
      {
         for (i = 0; i < dev->spdCabCount; i++)
         {
            if ((i % 4) == 0) TxPrint( " CAB block nrs %2d : 0x", i);
                              TxPrint( "%s%16.16llx%s   ", (i == 0) ? CBG : CNN, indexBlock[ i], CNN);
            if (((i % 4) == 3) || (i == (dev->spdCabCount - 1))) TxPrint( "\n");

            if (i == 0)
            {
               nav.down = dfsApfsBlock2Lsn( indexBlock[ i]);
            }
         }
      }
      else                                      // Links are LEAF / CIB structure (small/medium containers)
      {
         for (i = 0; i < dev->spdCibCount; i++)
         {
            if ((i % 4) == 0) TxPrint( " CIB block nrs %2d : 0x", i);
                              TxPrint( "%s%16.16llx%s   ", (i == 0) ? CBG : CNN, indexBlock[ i], CNN);
            if (((i % 4) == 3) || (i == (dev->spdCibCount - 1))) TxPrint( "\n");

            if (i == 0)
            {
               nav.down = dfsApfsBlock2Lsn( indexBlock[ i]);
            }
         }
      }
   }
   VRETURN();
}                                               // end 'dfsApfsSpaceMgrDev'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display APFS Space Manager FREE-QUEUE structure for Ipool/Main/Tier2
/*****************************************************************************/
static void dfsApfsSpaceMgrSfq
(
   BYTE               *block,                   // IN    Block contents SPMGR
   S_SPMGR_FREEQ      *que,                     // IN    Queue  in this SPMGR
   char               *queName                  // IN    Queue  name, length 5
)
{
   TXTT                lead;

   ENTER();

   if ((que != NULL) && (que->sfqTreeOid != 0))
   {
      sprintf( lead, "%s Queue  #blk : ", queName);
      dfsSz64( lead,                                      que->sfqCount * apfs->SectorsPerBlock, "  ");
      TxPrint( "Tree   Node Limit : 0x%4.4hx\n",          que->sfqNodeLimit, que->sfqNodeLimit);
      TxPrint( "Oldest tree   Xid : 0x%16.16llx        ", que->sfqOldestXid);
      TxPrint( "Queue  tree   Oid : 0x%16.16llx\n",       que->sfqTreeOid);
   }
   VRETURN();
}                                               // end 'dfsApfsSpaceMgrSfq'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display APFS Block, Space Manager CAB
/*****************************************************************************/
static ULONG dfsApfsSpaceMgrCab                 // RET   rc = 0 if type match
(
   BYTE               *block                    // IN    Block contents
)
{
   ULONG               rc = 0;                  // rc, sector match
   ULONG               i;
   S_SP_CAB_BLOCK     *sd = (S_SP_CAB_BLOCK *) block;

   ENTER();

   dfsApfsBlockHeader( block, FALSE);           // show header info, full block

   TxPrint("Nr. of CIB blocks : 0x%8.8x = 6%u        ", sd->cabCibCount, sd->cabCibCount);
   TxPrint(         "CabIndex : 0x%8.8x = %u\n",        sd->cabIndex,    sd->cabIndex);

   for (i = 0; i < sd->cabCibCount; i++)
   {
      if ((i % 4) == 0) TxPrint( " CIB block #  %3d : 0x", i);
                        TxPrint( "%s%16.16llx%s   ", (i == 0) ? CBG : CNN, sd->cabCibBlock[ i], CNN);
      if (((i % 4) == 3) || (i == (sd->cabCibCount - 1))) TxPrint( "\n");

      if (i == 0)
      {
         nav.down = dfsApfsBlock2Lsn( sd->cabCibBlock[ i]);
      }
   }
   RETURN (rc);
}                                               // end 'dfsApfsSpaceMgrCab'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display APFS Block, Space Manager Chunk-Info-Block
/*****************************************************************************/
static ULONG dfsApfsSpaceMgrCib               // RET   rc = 0 if type match
(
   BYTE               *block                    // IN    Block contents
)
{
   ULONG               rc = 0;                  // rc, sector match
   ULONG               i;
   S_SP_CIB_BLOCK     *sd = (S_SP_CIB_BLOCK *) block;
   S_SP_CIB_ENTRY     *ce;

   ENTER();

   dfsApfsBlockHeader( block, FALSE);           // show header info, full block

   TxPrint("Nr. of BM entries : 0x%8.8x = %5u        ", sd->cibEntryCount, sd->cibEntryCount);
   TxPrint(         "CibIndex : 0x%8.8x = %u\n",        sd->cibIndex,      sd->cibIndex);

   for (i = 0; i < sd->cibEntryCount; i++)
   {
      ce = &sd->cibEntry[ i];
      if ((ce->cieBmAddr != 0) || (dfsa->verbosity >= TXAO_VERBOSE))
      {
         TxPrint( "BitMap entry %4d : ", i);
         dfsX10(  "Xid=", ce->cieXid,  CNN, " ");
         dfsX10(  "Blk:", ce->cieAddr, CNN, " ");
         TxPrint( "Blocks:%4.4x Free:%4.4x BM: 0x%s%16.16llx%s\n", ce->cieBlockCount, ce->cieFreeCount,
                                             (i == 0) ? CBG : CNN, ce->cieBmAddr, CNN);
         if (i == 0)
         {
            nav.down = dfsApfsBlock2Lsn( ce->cieBmAddr);
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsApfsSpaceMgrCib'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display APFS Block, Space Manager BMAP
/*****************************************************************************/
static ULONG dfsApfsSpaceMgrBmap                // RET   rc = 0 if type match
(
   BYTE               *block                    // IN    Block contents
)
{
   ULONG               rc = 0;                  // rc, sector match

   ENTER();

   dfsApfsBlockHeader( block, FALSE);           // show header info, full block

   TxPrint("\nBlock contents including header:\n");
   TxDisplHexDump( block, 0x100);


   RETURN (rc);
}                                               // end 'dfsApfsSpaceMgrBmap'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display APFS Block, Object Map
/*****************************************************************************/
static ULONG dfsApfsObjectMap                   // RET   rc = 0 if type match
(
   BYTE               *block                    // IN    Block contents
)
{
   ULONG               rc = 0;                  // rc, sector match
   S_OBJECT_MAP       *sd = (S_OBJECT_MAP *) block;

   ENTER();

   dfsApfsBlockHeader( block, FALSE);           // show header info, full block

   TxPrint("Object Tree   Oid : %s0x%16.16llx%s        ", CBG, sd->omTreeOid,    CNN);
   TxPrint("Obj:%s%s%s : 0x%8.8x = %s%s%s\n", CBC, (sd->omTreeType & OID_TYPE_PHYS) ? "PHYS" :
                                                   (sd->omTreeType & OID_TYPE_EPHM) ? "EPHM" : "VIRT",      CNN,
                                                    sd->omTreeType, CBY, dfsApfsBlockType( sd->omTreeType), CNN);
   nav.down = dfsApfsBlock2Lsn( sd->omTreeOid); // next block to view, the tree ...

   TxPrint("Object Map  Flags : 0x%8.8x                %s", sd->omFlags,         CBM);
   if (sd->omFlags  &  OMF_MANUALLY_MANAGED    )   TxPrint("ManualManaged  ");
   if (sd->omFlags  &  OMF_ENCRYPTING          )   TxPrint("Encrypting  ");
   if (sd->omFlags  &  OMF_DECRYPTING          )   TxPrint("Decrypting  ");
   if (sd->omFlags  &  OMF_KEYROLLING          )   TxPrint("KeyRolling  ");
   if (sd->omFlags  &  OMF_CRYPTO_GEN          )   TxPrint("CryptoGen!  ");
   if (sd->omFlags  & ~OMF_VALIDMASK           )   TxPrint("%sUnknown-flags!",   CBR);
   TxPrint("%s\n", CNN);

   if ((sd->omSnapCount != 0) || (sd->omSnapTreeOid != 0))
   {
      TxPrint("Snapshot Tree Oid : 0x%16.16llx        ", sd->omSnapTreeOid);
      TxPrint("Obj:%s%s%s : 0x%8.8x = %s%s%s\n", CBC, (sd->omSnapTreeType & OID_TYPE_PHYS) ? "PHYS" :
                                                      (sd->omSnapTreeType & OID_TYPE_EPHM) ? "EPHM" : "VIRT",      CNN,
                                                       sd->omSnapTreeType, CBY, dfsApfsBlockType( sd->omSnapTreeType), CNN);
      TxPrint("Snapshot Last Xid : 0x%16.16llx\n",     sd->omSnapLastXid);
   }
   if ((sd->omPendingRevertMin != 0) || (sd->omPendingRevertMax != 0))
   {
      TxPrint("PndRevert MIN Xid : 0x%16.16llx        MAX Xid : 0x%16.16llx\n", sd->omPendingRevertMin, sd->omPendingRevertMax);
   }
   RETURN (rc);
}                                               // end 'dfsApfsObjectMap'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display APFS Block, Check Point
/*****************************************************************************/
static ULONG dfsApfsCheckPointMap               // RET   rc = 0 if type match
(
   BYTE               *block                    // IN    Block contents
)
{
   ULONG               rc = 0;                  // rc, sector match
   S_CHECKPOINT_MAP   *sd = (S_CHECKPOINT_MAP *) block;
   ULONG               mapping;

   ENTER();

   dfsApfsBlockHeader( block, FALSE);           // show header info, full block

   TxPrint("Checkpt map flags : 0x%8.8lx (1=LAST)       Map Count: %u\n", sd->cpm_flags,  sd->cpm_count);

   //- to be refined, may add an outer loop to handle multiple blocks
   //- and reading the next block until the flags == LAST
   for (mapping = 0; mapping < sd->cpm_count; mapping++)
   {
      TxPrint("\nChkPtMap#%3u Size : %12u bytes        ", mapping, sd->cpm_map[mapping].cpm_size);
      TxPrint(  "Obj:%s%s%s : 0x%8.8x = %s%s%s", CBC,
                         (sd->cpm_map[mapping].cpm_type & OID_TYPE_PHYS) ? "PHYS" :
                         (sd->cpm_map[mapping].cpm_type & OID_TYPE_EPHM) ? "EPHM" : "VIRT",    CNN,
                          sd->cpm_map[mapping].cpm_type, CBY, dfsApfsBlockType( sd->cpm_map[mapping].cpm_type), CNN);
      if (sd->cpm_map[mapping].cpm_subtype != 0)
      {
         TxPrint( " (%s)", dfsApfsBlockType( sd->cpm_map[mapping].cpm_subtype));
      }
      TxPrint( "\nVirtual Volume ID : 0x%16.16llx        objectId : 0x%16.16llx\n", sd->cpm_map[mapping].cpm_fs_oid, sd->cpm_map[mapping].cpm_oid);
      if (mapping == 0)
      {
         nav.down = dfsApfsBlock2Lsn( sd->cpm_map[mapping].cpm_paddr);
         TxPrint(   "Data Area BlockNr : %s0x%16.16llx%s\n", CBG, sd->cpm_map[mapping].cpm_paddr, CNN);
      }
      else
      {
         TxPrint(   "Data Area BlockNr : 0x%16.16llx\n", sd->cpm_map[mapping].cpm_paddr);
      }
      if ((dfsa->verbosity < TXAO_VERBOSE) == 0)      // only list first one (SpaceMgr) unless verbose
      {
         break;
      }
   }
   nav.xtra = nav.this + apfs->SectorsPerBlock; // 'x' goes to related C-super
   RETURN (rc);
}                                               // end 'dfsApfsCheckPointMap'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display APFS Block, Reaper object
/*****************************************************************************/
static ULONG dfsApfsBlockReaper                 // RET   rc = 0 if type match
(
   BYTE               *block                    // IN    Block contents
)
{
   ULONG               rc = 0;                  // rc, sector match

   ENTER();

   dfsApfsBlockHeader( block, FALSE);           // show header info, full block

   TxPrint("\nBlock contents including header:\n");
   TxDisplHexDump( block, 0x100);


   RETURN (rc);
}                                               // end 'dfsApfsBlockReaper'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display APFS Block, Reaper List
/*****************************************************************************/
static ULONG dfsApfsReaperList                  // RET   rc = 0 if type match
(
   BYTE               *block                    // IN    Block contents
)
{
   ULONG               rc = 0;                  // rc, sector match

   ENTER();

   dfsApfsBlockHeader( block, FALSE);           // show header info, full block

   TxPrint("\nBlock contents including header:\n");
   TxDisplHexDump( block, 0x100);


   RETURN (rc);
}                                               // end 'dfsApfsReaperList'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display APFS Block, EFI JumpStart (boot)
/*****************************************************************************/
static ULONG dfsApfsEfiJumpStart                // RET   rc = 0 if type match
(
   BYTE               *block                    // IN    Block contents
)
{
   ULONG               rc = 0;                  // rc, sector match
   S_B_EFI_J          *sd = (S_B_EFI_J *) block;
   ULONG               extent;

   ENTER();

   dfsApfsBlockHeader( block, FALSE);           // show header info, full block

   dfsUlDot13("EFI drv file size :", sd->ejFileLength, " Bytes");
   TxPrint( "        Version  : 0x%8.8x              Signature : %s%4.4s%s\n",
                            sd->ejVersion, CBM, sd->Signature, CNN);

   for (extent = 0; extent < sd->ejExtentCount; extent++)
   {
      TxPrint("EFI drv extent %2u : 0x%16.16llx        BlockCnt :%6u   At LSN: 0x%s%16.16llx%s\n",
                       extent,  sd->ejExtents[ extent].block, sd->ejExtents[ extent].count,
                                              (extent == 0) ? CBG :  CNN,
                    dfsApfsBlock2Lsn( sd->ejExtents[ extent].block), CNN);
      if (extent == 0)
      {
         nav.down = dfsApfsBlock2Lsn( sd->ejExtents[ extent].block);
      }
   }
   RETURN (rc);
}                                               // end 'dfsApfsEfiJumpStart'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display APFS Block, Fusion Write-Back Cache
/*****************************************************************************/
static ULONG dfsApfsFusionWbCache               // RET   rc = 0 if type match
(
   BYTE               *block                    // IN    Block contents
)
{
   ULONG               rc = 0;                  // rc, sector match

   ENTER();

   dfsApfsBlockHeader( block, FALSE);           // show header info, full block

   TxPrint("\nBlock contents including header:\n");
   TxDisplHexDump( block, 0x100);


   RETURN (rc);
}                                               // end 'dfsApfsFusionWbCache'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display APFS Block, Fusion Write-Back List
/*****************************************************************************/
static ULONG dfsApfsFusionWbList                // RET   rc = 0 if type match
(
   BYTE               *block                    // IN    Block contents
)
{
   ULONG               rc = 0;                  // rc, sector match

   ENTER();

   dfsApfsBlockHeader( block, FALSE);           // show header info, full block

   TxPrint("\nBlock contents including header:\n");
   TxDisplHexDump( block, 0x100);


   RETURN (rc);
}                                               // end 'dfsApfsFusionWbList'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display APFS Block, Encryption Rolling State
/*****************************************************************************/
static ULONG dfsApfsEncrRollState               // RET   rc = 0 if type match
(
   BYTE               *block                    // IN    Block contents
)
{
   ULONG               rc = 0;                  // rc, sector match

   ENTER();

   dfsApfsBlockHeader( block, FALSE);           // show header info, full block

   TxPrint("\nBlock contents including header:\n");
   TxDisplHexDump( block, 0x100);


   RETURN (rc);
}                                               // end 'dfsApfsEncrRollState'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display APFS Block, General BitMap object
/*****************************************************************************/
static ULONG dfsApfsGeneralBitMap               // RET   rc = 0 if type match
(
   BYTE               *block                    // IN    Block contents
)
{
   ULONG               rc = 0;                  // rc, sector match

   ENTER();

   dfsApfsBlockHeader( block, FALSE);           // show header info, full block

   TxPrint("\nBlock contents including header:\n");
   TxDisplHexDump( block, 0x100);


   RETURN (rc);
}                                               // end 'dfsApfsGeneralBitMap'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display APFS Block, General BitMap Block
/*****************************************************************************/
static ULONG dfsApfsGenBmapBlock                // RET   rc = 0 if type match
(
   BYTE               *block                    // IN    Block contents
)
{
   ULONG               rc = 0;                  // rc, sector match

   ENTER();

   dfsApfsBlockHeader( block, FALSE);           // show header info, full block

   TxPrint("\nBlock contents including header:\n");
   TxDisplHexDump( block, 0x100);


   RETURN (rc);
}                                               // end 'dfsApfsGenBmapBlock'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display APFS block header structure, in a generic way
/*****************************************************************************/
static ULONG dfsApfsBlockHeader                 // RET   rc = 0 if contents OK
(
   BYTE               *buffer,                  // IN    sector/block buffer
   BOOL                singleSector             // IN    buffer is just one sector
)
{
   ULONG               rc = 0;                  // rc, sector match
   S_B_HEADER         *sd = (S_B_HEADER *) buffer;
   BOOL                csOk = TRUE;

   ENTER();

   if (singleSector == FALSE)                   // whole block, validate checksum
   {
      csOk =  dfsApfsValidBlock( buffer, apfs->BlockSize);
   }
   TxPrint("CS:%s%s%s  Object-Id : 0x%16.16llx        ", (csOk) ? CBG : CBR, (csOk) ? "OK " : "BAD", CNN, sd->objectId);
   TxPrint("Obj:%s%s%s : 0x%8.8x = %s%s%s\n",      CBC, (sd->objectType & OID_TYPE_PHYS) ? "PHYS" :
                                                        (sd->objectType & OID_TYPE_EPHM) ? "EPHM" : "VIRT",      CNN,
                                                         sd->objectType, CBY, dfsApfsBlockType( sd->objectType), CNN);
   TxPrint(   "Transaction - Xid : 0x%16.16llx        ", sd->transactId);
   if (sd->objSubType != BT_NO_TYPE)
   {
      TxPrint(  "Sub-Type : 0x%8.8x = %s%s%s",           sd->objSubType, CNY, dfsApfsBlockType( sd->objSubType), CNN);
   }
   TxPrint( "\n");
   RETURN (rc);
}                                               // end 'dfsApfsBlockHeader'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Make filesystem usage map dump on TxPrint output, try both MAIN and TIER2
/*****************************************************************************/
ULONG dfsApfsAllocMap
(
   ULN64               di,                      // IN    dummy
   ULN64               d2,                      // IN    dummy
   char               *options,                 // IN    display options
   void               *data                     // OUT   dummy
)
{
   if (TxaOptSet( 'm'))                          // specific bitmap requested
   {
      switch ((int) TxaOptNum( 'm', NULL, 0))
      {
         case SPD_TIER2 : dfsApfsAnyAllocMap( SPD_TIER2, options); break;
         default:         dfsApfsAnyAllocMap( SPD_MAIN,  options); break;
      }
   }
   else                                         // default, show main/tier2 when present
   {
      if (apfs->Bm[ SPD_MAIN ].bmCount != 0)
      {
         dfsApfsAnyAllocMap( SPD_MAIN, options);
      }
      if (apfs->Bm[ SPD_TIER2].bmCount != 0)
      {
         dfsApfsAnyAllocMap( SPD_TIER2, options);
      }
   }
   RETURN (NO_ERROR);
}                                               // end 'dfsApfsAllocMap'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Make filesystem usage map dump on TxPrint output, one of three map types
/*****************************************************************************/
static ULONG dfsApfsAnyAllocMap
(
   ULONG               bmId,                    // IN    Bitmap ID, MAIN, TIER2, IPOOL
   char               *options                  // IN    display options
)
{
   ULN64               i;                       // index in data-area
   ULONG               b;                       // byte-index for one mapchar
   ULN64               cClus = 0;               // Clusters in current char
   ULN64               lClus = 0;               // Clusters in current line
   ULN64               mClus = 0;               // Clusters in current map
   ULN64               uClus = 1;               // Last cluster that is in use
   ULONG               l;                       // line number, 0 based
   ULONG               a;                       // index in display-array
   ULN64               perc;                    // percentage
   TX1K                ascii;                   // display-array
   ULONG               spe;                     // sectors per entry
   ULONG               acl;                     // ascii alloc chars per line
   ULONG               cpc;                     // clusters per display-char
   ULONG               spc;                     // sectors per display-char
   ULN64               size;                    // nr of blocks to map
   BOOL                verbose = (*options != '@');
   ULONG               bsSmart;
   ULN64               unallocSmart = 0;        // Size of THIS unallocated area
   BOOL                oneLineProgress = FALSE; // alloc in one line, output per character
   DFSAPFSBMCACHE     *cache = &apfs->Bm[ bmId];

   ENTER();

   if (cache->bmCount != 0)
   {
      dfsa->FsUnallocSmart = 0;                 // will be recalculated now
      bsSmart = dfsGetBufferSize( DFSOPTIMALBUF, DFSMAXBUFSIZE); // for SMART prediction

      spe   = apfs->SectorsPerBlock;
      size  = (cache != NULL) ? cache->blockCount : apfs->Block;

      acl   = dfsAllocCharsPerLine( 'c');
      cpc   = dfsAllocItemsPerChar( 'l', options, size, acl);
      if (cpc >= (size / acl))                  // will be a single line
      {
         oneLineProgress = TRUE;                // one char at a time, no ascii string used
      }
      spc   = cpc * spe;
      dfsProgressInit( 0, size * spe, 0, "Get ALLOCATION info, at Sector:", dfstStoreDpid( DFSTORE), DFSP_STAT, 0);

      if (verbose)
      {
         dfsSz64( "Size for one line : ", (ULN64) acl * spc, "  ");
         TxPrint( "in %3lu chars,", acl);
         dfsSiz8( " size : ", spc, "\n");
         TxPrint( "BitMap - %s : %10.10s = Allocation grade.   ", dfsApfsDeviceBmType( bmId), mapchar);
         dfsSiz8( " SMART size : ", bsSmart, "\n");
      }
      TxPrint(    "          %s%*.*s%s\n", CBC, acl, acl, BLIN, CNN);
      dfsX10( CNZ, dfsApfsBlock2Lsn(0), CNZ, ""); // display address for first block
      TxPrint("%s%s", CBC, CNC);

      for (i=0, a=0, l=0; (i < size) && (!TxAbort());)
      {
         if (a == 0)
         {
            memset(ascii, 0, TXMAXLN);
            lClus = 0;
         }
         for (cClus=0, b=0; (b < cpc) && (i < size); b++, i++)
         {
            if (dfsApfsBitmapCache( bmId, i))   // block in use ?
            {
               if (unallocSmart != 0)           // first in-use after unallocated?
               {
                  if (unallocSmart >= bsSmart)                  //- if at least ONE smart buffer
                  {
                     unallocSmart -= (bsSmart / 2);             //- compensate for alignment issues
                     unallocSmart -= (unallocSmart % bsSmart);  //- clip to whole smart buffer size

                     dfsa->FsUnallocSmart += unallocSmart;      //- add this area to total
                  }
                  unallocSmart = 0;             // reset for next area to come
               }
               cClus++;
               uClus = i;                       // last in-use one sofar
            }
            else                                // unallocated, Smart predict
            {
               unallocSmart += spe;             // add to size THIS area
            }
         }
         lClus += cClus;

         dfsProgressShow( (i * spe) + spc, 0, 0, NULL);

         TRACES(("lClus 2nd Loop i: %llu, b:%u = %llu\n", i, b, lClus));
         if (oneLineProgress == TRUE)
         {
            TxPrint( "%c", (char) ((cClus == 0) ? ' ' : mapchar[cClus * 8 / cpc])); // slow, acts as progress bar
         }
         else
         {
            ascii[a] = (char) ((cClus == 0) ? ' ' : mapchar[cClus * 8 / cpc]);
         }
         a++;
         if ((i && ((i%(acl*cpc)) == 0)) || (i >= size))
         {
            if (oneLineProgress == FALSE)
            {
               TxPrint( "%s", ascii);           // display accumulated chars (fast)
            }
            perc  = (ULN64)((1000ULL * lClus / (a*cpc)) + 5) / 10; // percentage rounded
            if (a == acl)
            {
               TxPrint("%s%s% 3llu%%\n", CBC, CNN, perc);
            }
            else
            {
               TxPrint("%s%.*s%s% 3llu%%\n", CBC, (int) (acl-a-1), BLIN, CNN, perc);
            }
            if (i < size)
            {
               dfsX10( CNZ, dfsApfsBlock2Lsn(i), CNZ, "");
               TxPrint("%s%s", CBC, CNC);
               a = 0;                           // keep a value on last line
            }
            mClus += lClus;
            l++;
         }
      }
      if (unallocSmart != 0)                           //- unalloc area pending?
      {
         if (unallocSmart >= bsSmart)                  //- if at least ONE smart buffer
         {
            unallocSmart -= (bsSmart / 2);             //- compensate for alignment issues
            unallocSmart -= (unallocSmart % bsSmart);  //- clip to whole smart buffer size

            dfsa->FsUnallocSmart += unallocSmart;      //- add this area to total
         }
      }
      TxPrint("          %s%.*s%s\n", CBC, (USHORT) a, BLIN, CNN);

      if (TxAbort())
      {
         dfsa->FsUnallocSmart = 0;              // signal info NOT available!
         TxPrint( "\nAllocation check and display aborted! Used/Free info not available.\n");
      }
      else
      {
         if (size != 0)                         // avoid devide by 0
         {
            ULN64          used = mClus * spe;

            dfsa->FsUnallocated = size * spe - used; // free space (optimize hint)

            dfsSz64("Unallocated sects : ", dfsa->FsUnallocated, " ");
            dfsSz64("SmartUse ", dfsa->FsUnallocSmart, " ");
            TxPrint("=% 5.1lf%%\n", (double) (100.0 * ((double) dfsa->FsUnallocSmart) / (double) (size * spe)));

            dfsSz64("Allocated sectors : ", used, " ");
            dfsSz64("of total ", size * spe, " ");
            TxPrint("=% 5.1lf%%\n", (double) (100.0 * ((double) mClus) / (double) size));
         }
         dfsa->FsLastInUse  = dfsApfsBlock2Lsn( uClus +1) -1;
      }
      dfsProgressTerm();
   }
   else
   {
      TxPrint( "\nBitMap - %s : not present\n", dfsApfsDeviceBmType( bmId));
   }

   RETURN (NO_ERROR);
}                                               // end 'dfsApfsAnyAllocMap'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// DFS APFS write-file to disk (SaveTo to file)
/*****************************************************************************/
static ULONG dfsApfsFileSaveAs
(
   ULN64               leafLsn,                 // IN    Leaf-node sectornumber
   ULN64               leafIdx,                 // IN    Leaf-node record index
   char               *path,                    // IN    destination path
   void               *recp                     // IN    recovery params (rename)
)
{
   ULONG               rc  = NO_ERROR;
   S_RECOVER_PARAM    *param = (S_RECOVER_PARAM *) recp;
   DFSISPACE           isp;                     // integrated alloc SPACE info
   BOOL                isDir = FALSE;
   TXLN                fname;                   // destination base filename
   TXLN                fullfn;                  // unique filename prefix
   ULN64               parent;                  // parent INO (unused)
   S_BT_NODE          *node = NULL;

   ENTER();
   TRARGS(("Leaf node LSN : 0x%llX, record index: %lld, to path: '%s'\n", leafLsn, leafIdx, path));

   //- to be refined for files with Resource, but no data-fork ?
   //- may pass a 'request-code' in the xattr-type, one being 'auto'
   if ((rc = dfsApfsGetAllocSpace( leafLsn, leafIdx, NULL, &isp)) == NO_ERROR)
   {
      if ((param->newname == NULL) || (*param->newname == 0)) // no newname present
      {
         if (dfsApfsIno2Parent( dfsApfsLsnInfo2Ino( leafLsn, DFSSNIGET( leafIdx), NULL),
                                                    &parent, fname, &isDir) != NO_ERROR)
         {
            sprintf( fname, "Node_%12.12llX_%4.4hX", leafLsn, DFSSNIGET( leafIdx));
         }
      }
      else                                      // rename specified
      {
         strcpy( fname, param->newname);
      }
      if (param->unique)                        // force unique naming
      {                                         // on PATH and FILE components
         sprintf( fullfn, "%12.12llX_%4.4llX_%s", leafLsn, leafIdx, fname);
         strcpy(  fname, fullfn);               // and move back to fname
      }

      rc = dfsSspaceFileSaveAs( &isp, isDir, FALSE, param->noAllocCheck, param->name83, path, fname, fullfn);
      if (rc == NO_ERROR)
      {
         if ((leafIdx & DFSSNINFO) && (dfsApfsReadChkCatNode( leafLsn, &node) == NO_ERROR) && (node->btNode.btFlags & BNF_LEAF))
         {
            S_INODE_KEY      inoKey;            // key to search for the INODE
            S_NODE_KEY       nodeKey;           // generic record key   (pointer)
            S_NODE_VAL       nodeVal;           // generic record value (pointer)
            ULN64            ino = 0;
            ULN64            leafId;            // virtual leaf ID for found leaf
            USHORT           index;             // record index for INODE in found leaf
            int              recType;

            recType = dfsApfsNodeIdx2Rec( node, DFSSNIGET( leafIdx), NULL, &nodeKey, NULL, &nodeVal);
            switch (recType)
            {
               case APFST_DIR_RECORD:
               case APFST_DIR_STATS:
               case APFST_FILE_EXTENT:
               case APFST_DSTREAM_ID:
               case APFST_XATTR:
               case APFST_SIBLING_LINK:
                  if (recType == APFST_DIR_RECORD) // INO number is in the record VALUE
                  {
                     ino = nodeVal.dr->fileId;
                  }
                  else                          // INO number is in the record KEY
                  {                             // Inode record must be searched/read!
                     ino = APFSidPrim( nodeKey.ds->fsId);
                  }
                  TxFreeMem( node);             // free existing node record
                  nodeKey.in = &inoKey;         // create the key for Btree search
                  inoKey.fsId = APFSmkFsId( ino, APFST_INODE); // hybrid primary/secondary key value
                  leafId = dfsApfsSearchCatalog( nodeKey, &index, &node);
                  if ((leafId == 0) || (leafId == L64_NULL) ||
                      (dfsApfsNodeIdx2Rec( node, index, NULL, &nodeKey, NULL, &nodeVal) != APFST_INODE))
                  {
                     TRACES(("Failed to locate INODE record!\n"));
                     rc = DFS_BAD_STRUCTURE;
                  }
                  break;

               case APFST_INODE:
                  ino = APFSidPrim( nodeKey.in->fsId); // make INO value available for search EXTENTS
                  break;

               default:
                  rc = DFS_ST_MISMATCH;         // SN/index not for a valid record-type
                  break;
            }
            if (rc == NO_ERROR)                 // get byte-size from Inode Xfield
            {
               time_t      mod = dfsApfsTime2Tm( nodeVal.in->inModTime);
               time_t      cre = dfsApfsTime2Tm( nodeVal.in->inCreTime);
               time_t      acc = dfsApfsTime2Tm( nodeVal.in->inAccTime);

               TRACES(("Set original date/time for: '%s'\n", fullfn));

               TxSetFileTime( fullfn, &cre, &acc, &mod); // set time on full-fn from SaveAs

               if (param->recFname)             // return full recovered path+filename
               {
                  strcpy( param->recFname, fullfn);
               }
            }
            TxFreeMem( node);
         }
      }
      TxFreeMem( isp.space);
   }
   RETURN(rc);
}                                               // end 'dfsApfsFileSaveAs'
/*---------------------------------------------------------------------------*/
