//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// DFS navigation and general function interface
//
// Author: J. van Wijk
//
// JvW  12-01-97   Initial version, split off from DFS.C
//
#ifndef    DFS_H
   #define DFS_H

#include <dfsspace.h>

#if defined (OEMSB)
   #define DFS_PHYSDISK_ONLY
#endif

#if defined (OEMSV)
   #define OEM_BRANDED
   #define HAVE_CMD_GRAB
#elif defined (OEMAN)
   #define OEM_BRANDED
#else
   #define HAVE_CMD_GRAB
#endif


#define DFS_FS_UNKNOWN    ((ULONG)   1)         // FsIdentify default

#define DFS_BAD_STRUCTURE TX_INVALID_DATA       // 0xCF inconsistent structure
#define DFS_ALLOC_ERROR   TX_ALLOC_ERROR        // 0xD0 memory allocation error
#define DFS_PENDING       TX_PENDING            // 0xD9 cmd or function pending
#define DFS_READ_ONLY     TX_WRITE_PROTECT      // 0xDB Store is read-only
#define DFS_CMD_UNKNOWN   TX_CMD_UNKNOWN        // 0xDE cmd unknown, not handled
#define DFS_VALUE_ERROR   ((ULONG) 231)         // 0xE7 Value not acceptable
#define DFS_NOT_FOUND     ((ULONG) 232)         // 0xE8 search-item not found
#define DFS_LIST_EMPTY    ((ULONG) 233)         // 0xE9 (sector) list is empty
#define DFS_NO_SCRIPT     ((ULONG) 234)         // 0xEA script file not found
#define DFS_NO_DEVICE     ((ULONG) 244)         // 0xF4 no device opened
#define DFS_PSN_LIMIT     ((ULONG) 245)         // 0xF5 SN exceeds limit (seek err)
#define DFS_ST_MISMATCH   ((ULONG) 246)         // 0xF6 sector-type mismatch
#define DFS_REBOOT_REQ    ((ULONG) 248)         // 0xF8 RC OK, reboot required
#define DFS_USER_ABORT    ((ULONG) 249)         // 0xF9 RC OK, abort by user
#define DFS_NO_CHANGE     ((ULONG) 250)         // 0xFA RC OK, no changes pending
#define DFS_CMD_FAILED    ((ULONG) 252)         // 0xFC Generic cmd failure code
#define DFS_CMD_WARNING   ((ULONG) 254)         // 0xFE Cmd completed with warnings

#define DFS_QUIT          TX_APP_QUIT           // 0x100000 quit DFSee (RC flag)

#define DFS_O_LVMSIZE     TXA_O_APP0            // LVM size use for disksize/geo
#define DFS_O_LVMSNP      TXA_O_APP1            // LVM partition Serial#
#define DFS_O_LVMSNV      TXA_O_APP2            // LVM volume    Serial#
#define DFS_O_LVMSND      TXA_O_APP3            // LVM thisdisk  Serial#
#define DFS_O_LVMSNB      TXA_O_APP4            // LVM bootdisk  Serial#
#define DFS_O_SYNCLA      TXA_O_APP5            // LVM automatic sync
#define DFS_O_SKIPFS      TXA_O_APP6            // Skip FS's on correlate
#define DFS_O_SKBADS      TXA_O_APP7            // Skip bad areas on clone
#define DFS_O_GEOCALC     TXA_O_APP8            // Calculate geo from contents
#define DFS_O_I13X        TXA_O_APP9            // Require I13X signatures
#define DFS_O_FILESYS     TXA_O_AP10            // Filesystem specification
#define DFS_O_GPTAUTO     TXA_O_AP11            // Automatic GPT interpretation
#define DFS_O_LOGAUTO     TXA_O_AP12            // Automatic logfile numbering
#define DFS_O_LOGDATE     TXA_O_AP13            // Automatic logfile DATE postfix
#define DFS_O_LONGFN      TXA_O_AP14            // Long Filename
#define DFS_O_EXPERT      TXA_O_AP15            // Expert UI mode
#define DFS_O_ATTRIB      TXA_O_AP16            // Attribute value
#define DFS_O_ALIGN       TXA_O_AP17            // Alignment value or enable/disable
#define DFS_O_GUID        TXA_O_AP18            // GUID value
#define DFS_O_FLOPPY      TXA_O_AP19            // Include floppy in volume lists
#define DFS_O_STD         TXA_O_AP20            // Standard behaviour requested
#define DFS_O_TREE        TXA_O_AP21            // Tree iterate
#define DFS_O_PROFILE     TXA_O_AP22            // User specified profile

#if   defined (DOS32)
   #define DFS_SCROLL_L      1000               // total 0.3 Mb   (for width  132)
#elif defined (UNIX)
   #define DFS_SCROLL_L     30000               // total  48 Mb   (for width  800)
#else
   #define DFS_SCROLL_L     30000               // total  18 Mb   (for width  300)
#endif
#define DFS_SCROLL_W      TXW_MAX_COLS          // max scroll-buffer width

//- generic option flags
#define DFSOPT_TEST       0x00000001
#define DFSOPT_VERBOSE    0x00000002
#define DFSOPT_COMPARE    0x00000004

//- Imaging and cloning options
#define DFSI_STANDARD     0x00000000            // note: 01..08 reserved!
#define DFSI_COMPRESS     0x00000010            // compress the imagefile
#define DFSI_SMARTUSE     0x00000020            // use smart-usage alloc info
#define DFSI_RAWIMAGE     0x00000040            // do not add a header
#define DFSI_REMOVABL     0x00000080            // prompt for media change
#define DFSI_APPENDTO     0x00000100            // append to existing file
#define DFSI_NOPRITEM     0x00000200            // not a progress todo item
#define DFSI_NOCRCUSE     0x00000400            // no CRC (6.xx/5.xx compat)
#define DFSI_CR_INDEX     0x00000800            // Create index for IMZ too
#define DFSI_MERGEBAD     0x00001000            // Merge bad-sectors on clone
#define DFSI_HIGH2LOW     0x00002000            // Direction High to Low sects
#define DFSI_OVERLAPD     0x00004000            // Abort would loose partition
#define DFSI_NOFULBUF     0x00010000            // No compact (2-byte) RLE/Smart


//- lower limit of filesize, after which progress-reporting on Copy is used
#define DFS_COPY_THRESHOLD_SECTORS    2048      // about 1Mb of filesize

#define DFS_USER_PROFILE       "profile.dfs"
#define DFS_PROFILE_TEMPLATE   "proftemp.dfs"

//- name and location for dfspuppy images, when running from bootable USB stick
#define DFS_PUPPY_STICK_IMZ    "/mnt/home/dfsee/dfspuppy.imz"
#define DFS_PUPPY_LOCAL_IMZ    "dfspuppy.imz"
#define DFS_PUPPY_ISIZE_IMZ     29600
#define DFS_PUPPY_TRACKSIZE     32

#define DFS_PUP64_STICK_IMZ    "/mnt/home/dfsee/dfspup64.imz"
#define DFS_PUP64_LOCAL_IMZ    "dfspup64.imz"
#define DFS_PUP64_ISIZE_IMZ     60000
#define DFS_PUP64_TRACKSIZE     64

#define DFS_PUP32_STICK_IMZ    "/mnt/home/dfsee/dfspup32.imz"
#define DFS_PUP32_LOCAL_IMZ    "dfspup32.imz"
#define DFS_PUP32_ISIZE_IMZ     29501
#define DFS_PUP32_TRACKSIZE     64

#if defined (DOS32)
//- single-line, length 256 characters for wide screens
#define BLIN "" \
             "" \
             "" \
             ""
#else
//- single-line, length 768 characters for very wide screens (12 x 64)
#define BLIN "" \
             "" \
             "" \
             "" \
             "" \
             "" \
             "" \
             "" \
             "" \
             "" \
             "" \
             ""
#endif

typedef struct s_navigate
{
   ULN64               up;                      // up in hierarchie
   ULN64               down;                    // down in hierarchie
   ULN64               this;                    // this (current)
   ULN64               xtra;                    // Extra, alternative
   ULONG               index;                   // current index in LST
   ULONG               offset;                  // offset in sector (find)
   USHORT              this_sninfo;             // SN info for THIS sn (recover)
   USHORT              down_sninfo;             // SN info for NEXT sn (auto ..)
} S_NAVIGATE;                                   // end of struct "s_navigate"


typedef struct s_areaelem
{
   ULN64               start;                   // start of area (LSN)
   ULN64               size;                    // size of area (sectors)
   struct s_areaelem  *next;
} S_AREAELEM;                                   // end of struct "s_areaelem"

typedef struct s_arealist
{
   S_AREAELEM         *head;                    // start of list or NULL
   S_AREAELEM         *tail;                    // last element or NULL
   TXTM                descr;                   // contents description
} S_AREALIST;                                   // end of struct "s_arealist"

// Note: The intdata/intinfo pair often represents a single file or directory
//       with the intdata element being an LSN, containing the file info
//       and the  intinfo additional information for further identification:
//       For FAT  filesystems: Index of FAT-directory ENTRY within a sector
//       For EXT  filesystems: Index of one single INODE within a sector
//       For NTFS filesystems: The desired parent MFT number to select the
//                             correct up-link with multiple hard links

typedef ULONG (* DFS_FUNCTION)                  // RET   function result (rc)
(
   ULN64               intdata,                 // IN    integer, usually an LSN
   ULN64               intinfo,                 // IN    meta info with LSN
   char               *strdata,                 // INOUT string data
   void               *calldata                 // INOUT function data
);

typedef ULONG  (* DFS_UL_FUNCT)                 // RET   function result (data)
(
   ULN64               intdata,                 // IN    integer, usually an LSN
   ULN64               intinfo,                 // IN    meta info with LSN
   char               *strdata,                 // INOUT string data
   void               *calldata                 // INOUT function data
);

typedef ULN64  (* DFS_U64FUNCT)                 // RET   function result (data)
(
   ULN64               intdata,                 // IN    integer, usually an LSN
   ULN64               intinfo,                 // IN    meta info with LSN
   char               *strdata,                 // INOUT string data
   void               *calldata                 // INOUT function data
);

typedef struct dfs_params                       // generic parameter structure
{
   ULN64               Lsn;                     // Pure LSN (used to be lsn+info)
   ULN64               Number;                  // Often the split off 'info'
   BOOL                Flag;
   char               *String;
   ULN64               byteSize;                // size in bytes, usualy filesize
   ULONG               Count;                   // Another numeric (SPACE chunks)
   void               *Misc;                    // Misc pointer    (SPACE data)
   DFS_FUNCTION        Func;
} DFS_PARAMS;                                   // end of struct "dfs_params"

// macros to call function, if pointer defined
#define DFSFNCALL(f,a,b,c,d) ((f) ? (f)(a,b,c,d) : DFS_PENDING)
#define DFSFNCAUL(f,a,b,c,d) ((f) ? (f)(a,b,c,d) : a)

//- Shortcut defines for Cluster-to-LSN and LSN-to-Cluster translations
#define dfsSn2Cl(sn) DFSFNCAUL(dfsa->FsLsn2Cl,(sn),0,NULL,NULL)
#define dfsCl2Sn(cl) DFSFNCAUL(dfsa->FsCl2Lsn,(cl),0,NULL,NULL)

//- macro's to use the same 64-bit field for either a disk-location or memory-address
//- with FLAGGING to determine what the current value is. Requires the disk-location
//- and virtual-addresses to be 56 bits or less (works for Intel IA64/AMD64 architecture)
//- supporting sign-extended memory addresses, where the upper 16 bits are either 0 or 1
//- Can be used to store B-trees in memory instead of disk (APFS)
#define I64FLAGGED_VALUE  0xA000000000000000LL
#define I64FLAGVAL_MASK   0xF000000000000000LL
#define I64FLAGSGN_MASK   0x0100000000000000LL
#define I64Flagged(v64)   ((ULN64)  (((v64) &  I64FLAGVAL_MASK) == I64FLAGGED_VALUE))
#define I64FlagSet(v64)   ((ULN64)  (((ULN64)((v64)) & ~I64FLAGVAL_MASK) |  I64FLAGGED_VALUE))
#define I64FlagVal(v64)   ((ULN64)  ( (v64) & ~I64FLAGVAL_MASK))
#define I64FlagAdr(v64)   ((void *) (((v64) &  I64FLAGSGN_MASK) ? ((v64) | I64FLAGVAL_MASK) : ((v64) & ~I64FLAGVAL_MASK)))


// macro's to code/decode FAT DIR entry or EXT Inode-index values in an info value
#define DFSSNINFO          0x8000
#define DFSSNIGET(inf)     ((USHORT)(inf & 0x7fff))

// Flag location and values for SNLIST, to be stored in first entry of sninfo array
#define DFSBR_SnlistFlag  (dfsa->sninfo[0])
#define DFSBR_1ST_PARENT   0x0001               // first snlist entry is parent

// generic Iterator function codes
#define DFS_ITER_NEXT     ((ULN64)  0)          // get next value
#define DFS_ITER_FIRST    ((ULN64) -1)          // return 1st LSN
#define DFS_ITER_INIT     ((ULN64) -2)          // initialize iterator
#define DFS_ITER_TERM     ((ULN64) -3)          // terminate iterator

// chs Style values
#define DFS_CHSTYLE_IBM     0                   // IBM FDISK/LVM/DFSee
#define DFS_CHSTYLE_PQ      1                   // PowerQuest PMagic etc
#define DFS_CHSTYLE_MS      2                   // old Microsoft style
#define DFS_CHSTYLE_NONE    0xffffffff          // no style set yet

// Default HEX/DEC Radix mask bits to use with DFSee
#define DFSRADIX_SECT (TX_RADIX_APP_CLASS * 1)  // sector values
#define DFSRADIX_SIZE (TX_RADIX_APP_CLASS * 2)  // size values
#define DFSRADIX_TYPE (TX_RADIX_APP_CLASS * 4)  // type values

#define DFS_SEC_RADIX (TX_RADIX_UN_S_CLASS + DFSRADIX_SECT + DFSRADIX_TYPE)
#define DFS_STD_RADIX  TX_RADIX_DEC_CLASS

// DFSee partition create alignment, special values
#define DFS_CR_ALIGN_AUTO  0
#define DFS_CR_ALIGN_CYL   1
#define DFS_CR_ALIGN_MIB   0x800

// 64-bit ULN64 sectornumber '10-hex-digit' max clipping mask for displaying
#define DFS_SN10  ((ULN64) 0xffffffffff)

// DFSee standard Dirty/Clean status values
#define DFSTAT_CLEAN       0
#define DFSTAT_UNKNOWN     1
#define DFSTAT_DIRTY       2

#define DFSI13X_NOTTHERE   0
#define DFSI13X_OPTIONAL   1
#define DFSI13X_REQUIRED   2
\
//- shorthand macro to address fields in DFSISTORE (previously dfsa == DFSINF)
#define SINF           dfsa->is

//- Note: all FS specific fields could be moved to store-level too, then
//-       each store would have its own 'mode' ... (mode is global now!)

typedef enum   dfsmode
{
   DFS_FS_AUX,                                  // default (0)
   DFS_FS_FDISK,
   DFS_FS_EXT,
   DFS_FS_FAT,
   DFS_FS_HPFS,
   DFS_FS_APFS,
   DFS_FS_HFS,
   DFS_FS_JFS,
   DFS_FS_NTFS,
   DFS_FS_RSR,
   DFS_FS_XFS,
   DFS_FS_SWAP,
   DFS_FS_DUMP,
   DFS_FS_E_FAT,
   DFS_FS_ISO
} DFSMODE;                                      // end of enum "dfsmode"


typedef struct s_recover_param
{
   BOOL                unique;                  // make unique by prepending HEX
   BOOL                nopath;                  // discard original path
   BOOL                name83;                  // force 8.3 compatible name
   BOOL                noAllocCheck;            // skip allocation check on recover
   BOOL                recurseDir;              // recurse into directories
   char               *select;                  // RO, wildcard select str or NULL
   char               *newname;                 // RO, rename name/empty,  or NULL
   char               *recFname;                // RW, recovery full name, or NULL
} S_RECOVER_PARAM;                              // end of struct "s_recover_param"


typedef struct dfsinf                           // information anchor block
{
   DFS_FUNCTION        FsCommand;               // FS specific cmd handler
   DFS_FUNCTION        FsClose;                 // Filesystem deregister/close
   DFS_FUNCTION        FsIdentifySector;        // FS identify a sector
   DFS_FUNCTION        FsShowType;              // FS display ascii sector type
   DFS_FUNCTION        FsDisplaySector;         // FS display sector contents
   DFS_FUNCTION        FsLsnAllocated;          // FS test if LSN is allocated
   DFS_FUNCTION        FsLsnSetAlloc;           // FS set Alloc bit to value
   DFS_FUNCTION        FsFileInformation;       // FS file alloc/path/size/date
   DFS_FUNCTION        FsGetAllocSpace;         // FS Get file allocation, SPACE
   DFS_FUNCTION        FsWriteMetaSpace;        // FS Write Meta area, from SPACE
   DFS_FUNCTION        FsFindPath;              // FS find LSN for given path
   DFS_FUNCTION        FsUpdateFileName;        // FS rename file, same length!
   DFS_FUNCTION        FsSetAltBrecLabel;       // FS set LABEL in alternate location
   DFS_FUNCTION        FsMakeBrowseList;        // FS get 1 whole DIR in sn-list
   DFS_FUNCTION        FsSltBuild;              // FS build SLT, add contents
   DFS_FUNCTION        FsNpBuild;               // FS build name-parent cache
   DFS_FUNCTION        FsDisplayError;          // FS display flag-error text
   DFS_FUNCTION        FsDisplayLsnInfo;        // FS LSN special info display
   DFS_FUNCTION        FsDirIterator;           // FS directory iterator
   DFS_FUNCTION        FsDirFileSaveAs;         // FS dir and file save (undel)
   DFS_FUNCTION        FsTruncateSize;          // FS truncate filesystem size
   DFS_FUNCTION        FsAllocDisplay;          // FS graphic alloc display
   DFS_U64FUNCT        FsCl2Lsn;                // FS cluster to LSN convert
   DFS_U64FUNCT        FsLsn2Cl;                // FS LSN to cluster convert
   DFS_U64FUNCT        FsLsnIterator;           // FS LSN iterate function
   DFS_FUNCTION        FsAreaClose;             // FS Area deregister/close
   DFS_FUNCTION        FsAreaLsnAllocated;      // FS Area test LSN allocated
   ULONG               FsSltRecSize;            // FS recommended SLT size
   ULN64               FsEntry;                 // FS logical entry point
   ULN64               FsTruncPoint;            // FS LSN to truncate volume
   ULN64               FsLastInUse;             // FS LSN last sector in-use
   ULN64               FsExpandSize;            // FS maximum size for expand
   ULN64               FsUnallocated;           // FS freespace, unallocated
   ULN64               FsUnallocSmart;          // FS unalloc, smart candidate
   ULONG               FsDirtyStatus;           // FS dirty/clean status
   BOOL                FsToggleOption1;         // FS specific menu toggle 1
   BOOL                FsToggleOption2;         // FS specific menu toggle 2
   BOOL                FsToggleOption3;         // FS specific menu toggle 3
   DFSMODE             FsModeId;                // FS identifier
   TXSELIST           *FsModeSelist;            // FS specific selection list
   char              **FsCmdHelp;               // FS specific cmd help text
   void               *Fsi;                     // FS information structure
   char               *FsSectorTypes;           // FS string of sector types
   char               *BsSectorTypes;           // Base string of sector types
   S_BOOTR            *boot;                    // ptr to boot record
   ULONG               snsize;                  // allocated snlist size
   ULN64              *snlist;                  // large sector number list
   USHORT             *sninfo;                  // large sector number info
   ULN64               brlist;                  // Browse, snlist 'this' sn
   ULN64               brinfo;                  // Browse, snlist 'this' info
   TXLN                brdescript;              // list/browse description string
   TXTS                sncompact;               // optimal show list values
   TXTS                snverbose;               // optimal show list contents
   TXTS                snifsmode;               // list initial FS mode
   TXTM                snifsmcmd;               // FS specific command after select
   TXLN                SavetoPath;              // default saveto/recover path
   S_RECOVER_PARAM     rParams;                 // persistent recovery params
   DFSISPACE           findSpace;               // SPACE to be searched by 'find -o:S'
   S_AREALIST         *salist;                  // sector-area list
   DFSISTORE          *is;                      // info, store dependend
#if defined (DEV32)
   char               *BldLevel;                // satisfy compiler reference
   TXF_OS2DLLAPI      *os2api;                  // OS2 large file API's
   BOOL                lvmSyncApi;              // use automatic sync of LVM
#endif
   TXTM                skipFs;                  // FS's to skip on correlate
   BOOL                lvmPresent;              // valid LVM info is present
   ULONG               lvmBootId;               // lvm phys boot disk id
   USHORT              winCandidates;           // Nr of possible WIN partitions
   BOOL                snlact;                  // list input currently active
   ULONG               number;                  // value to pass to rexx
   USHORT              bmgrDisk;                // physical disk active BMGR
   USHORT              bmgrTimeout;             // current timeout value BMGR
   ULONG               bmgrI13X;                // I13X none/optional/needed
   BOOL                mbr1I13X;                // I13X capable MBR on disk 1
   USHORT              autoPid;                 // partition-id for 'part 0'
   BOOL                batch;                   // batch-mode active
   BOOL                dialogs;                 // dialogs will be used
   ULONG               verbosity;               // output verbosity
   int                 eStrategy;               // error strategy
   int                 readAttempts;            // nr of sector read attempts
   ULONG               chsStyle;                // default CHS-dummy style
   ULONG               retc;                    // overall return-code
   BOOL                explain;                 // RETC is explainable
   USHORT              sbWidth;                 // visible scroll-buffer width
   USHORT              sbLength;                // visible scroll-buffer length
   BOOL                OS2FsPresent;            // OS/2 filesystems likely present
   BOOL                LockIgn;                 // Ignore lock errors
   BOOL                autoquit;                // auto quit after fdisk/setb
   BOOL                regconfirm;              // registration conf required
   char                regTypeCh;               // reg type: U, T, S=Standard P=Pro F=Forensic
   BOOL                ReadOnly;                // default store mode R/O
   BOOL                sltProgress;             // SLT build can show progress
   BOOL                nowindow;                // running in classic mode
   BOOL                browseShowHidden;        // Show hidden files/dir in Browse
   ULONG               warnLevel;               // minimal 0 .. maximum 2
   TXTIMER             statusTimer;             // Timer for next statusline update
   ULONG               stdOutInterval;          // secs between STDOUT progress
   BOOL                stdOutProgBars;          // force progress bars to STDOUT
   BOOL                hexSNS;                  // Hex sector-nr and size fields
   BOOL                hexCHS;                  // Hex CHS-value fields
   BOOL                geoCalc;                 // Calculate geo from contents
   BOOL                gptAuto;                 // Auto-show and interpret GPT tables
   BOOL                volFloppy;               // Include FLOPPY in volumelist (slow)!
   BOOL                logAuto;                 // Automatic logfile numbering    from DLG
   BOOL                logDate;                 // Automatic logfile DATE postfix from DLG
   BOOL                veResolve;               // var/expr ${ resolve on cmds
   BOOL                pt0Show;                 // show part-type 0x00 as partitions
   TXWHANDLE           sbwindow;                // scroll-buffer window
   TXWHANDLE           menuOwner;               // menu handling window
   TXWHANDLE           cmdwindow;               // commandline window, history
   BOOL                cmdprompt;               // prompt after all manual commands
   BOOL                expertui;                // EXPERT menu/dialog UI (not BASIC)
   BOOL                automenu;                // automatic menu activation
   BOOL                autodrop;                // automatic pulldown drop
   BOOL                delaydrop;               // drop only on MOVE, not activation
   BOOL                rightmove;               // Move Right on right-arrow, don't open submenu
   ULONG               menuopen;                // default drop-down menu
   ULONG               worklevel;               // when 0, activate menu
   ULONG               alignment;               // 0=auto MiB/Cyl, other is sectors
   ULONG               defDispSize;             // default (hex) display size
   USHORT              lastFindLen;             // length last Find argument
   int                 maxPath;                 // maximum path+filename length
   TXSELIST           *slVolumes;               // sel-list, 1 volume
   TXSELIST           *slReMedia;               // sel-list, 1 removable
   TXSELIST           *slDiskOne;               // sel-list, 1 disk
   TXSELIST           *slDiskAll;               // sel-list, 1 disk or All
   TXSELIST           *slDiskSam;               // sel-list, 1 disk or SAME (Prestore)
   TXSELIST           *slDiskMnt;               // sel-list, mounted disks
   TXSELIST           *slPartOne;               // sel-list, 1 partition
   TXSELIST           *slPartLog;               // sel-list, logicals enabled only
   TXSELIST           *slPartAll;               // sel-list, 1 part or All
   TXSELIST           *slFreeCCr;               // sel-list, part COPY, disk CREATE
   ULONG               nrFreeCCr;               // #enabled  CCr
   TXSELIST           *slFreeMBm;               // sel-list, part MOVE, disk BMinstall
   ULONG               nrFreeMBm;               // #enabled  MBm
   TXSELIST           *slFreeGPr;               // sel-list, GPT + Primary, no LOG-only
   ULONG               nrFreeGPr;               // #enabled  GPr
   TXSELIST           *slFreeLog;               // sel-list, LOG, no PRI-only, no GPT
   ULONG               nrFreeLog;               // #enabled  Log
   TXSELIST           *slFreeAll;               // sel-list, All freespace
   TXSELIST           *slBootabl;               // sel-list, bootables
   TXSELIST           *slSchemes;               // sel-list, Color schemes
   TXSELIST           *slPalette;               // sel-list, Color palette
   short               margin;                  // default 'max' width of text in SB
   #if defined (DOS32)
      BOOL             win9x;                   // Windows-9x DosBox detected
   #endif
   char               *cmdSeparator;            // command separator string
   char               *dfsExeDir;               // executable directory, empty if unknown
   char               *dfsExeName;              // executable name
} DFSINF;                                       // end of struct "dfsinf"

#define DFS_MAX_DISPSECTOR 64                   // display # interpreted as sectors
                                                // above this wil be seen as bytes

//- Buffer-sector sizes, compatible with DOS DPMI memory allocation
#if defined (DOS32)
   #define BBUFS             32                 // max bootrec sectors  (16  KiB,  4 * 4096 BPS)
#else
   #define BBUFS            128                 // max bootrec sectors  (64  KiB, 16 * 4096 BPS)
#endif

#define BBUFBYTES ((ULONG) BBUFS * SECTORSIZE)  // boot-buffer bytes
#define BBUFSECTORS   (BBUFBYTES / dfsGetSectorSize())

#define RBUFS           (2 * DFS_MAX_SECT + 2)  // read-buffer sectors  (2 tracks +2 at 512 BPS)
#define RBUFBYTES ((ULONG) RBUFS * SECTORSIZE)  // read-buffer bytes    (256 KiB, 640Kb limited)
#define RBUFSECTORS   (RBUFBYTES / dfsGetSectorSize())

//- Default/optimal buffersize for NON-smart (wipe, scan, copy etc) specific to OS used
#if defined (DEV32) || defined (DOS32)          // DOS and OS/2 are track oriented
   #define DFSOPTIMALBUF   (dfstGeoSectors( DFSTORE))
#else                                           // will still fit in preallocated buffer
   #define DFSOPTIMALBUF   RBUFSECTORS
#endif


#if defined (DOS32)                             // need to use smaller DPMI buffer

   //- Maximum 256 KiB for single buffer (cloning, wiping etc), same as default SMART buffer
   #define DFSMAXBUFSIZE    RBUFSECTORS

   #define DFSMAXBUFIMZ   DFSMAXBUFSIZE
   #define LSIZE         ((ULONG)  55000)       // total appr. 0.5  Mb in size!

#else

   //- Maximum 64 MiB for single buffer (cloning, wiping etc) but NOT optimal for SMART-use
   #define DFSMAXBUFSIZE (((ULONG) 0x4000000) / dfsGetSectorSize())

   //- Maximum buffersize for 'large-buffer' IMZ is 2 MiB, so the required offset
   //- values for 8 Compr-Units in the index-file fit in 3+21 = 24 bits = 3 bytes
   #define DFSMAXBUFIMZ  (((ULONG) 0x200000) / dfsGetSectorSize())

   //- default size of the sectorlist
   #define LSIZE         ((ULONG) 4000000)      // total appr. 36 Mb in size!

#endif



//- Allow/prevent adding values to snlist
#define DFSNL_ACTIVE(bool) (dfsa->snlact = bool)
#define DFSNL_ROOM         (dfsa->snlist[0] < (LSIZE-1))

//- set actual size of snlist
#define DFSNL_SIZE(index)                   \
            if ((dfsa->snlist != NULL) &&   \
                (dfsa->snlact))             \
            {                               \
               dfsa->snlist[0]     = index; \
            }


extern  S_NAVIGATE  nav;                        // navigation info
extern  BYTE       *rbuf;                       // read buffer, generic use
extern  BYTE       *brec;                       // read buffer, boot recs
extern  DFSINF     *dfsa;                       // DFS anchor block

extern  char       *DFSeeSwitchhelp[];
extern  char       *hostvarhelp[];
extern  char       *partypehelp[];

#if defined (DOS32)
#define XBUFS       64                          // 64 sectors of DPMI buffer
extern  BYTE       *xpar;                       // DPMI parameter  ptr,  512
extern  BYTE       *xbuf;                       // DPMI big buffer ptr, 32Kb
extern  short       xparSelector;
extern  short       xbufSelector;
extern  short       rbufSelector;
extern  short       brecSelector;
#endif

// scroll to end (output), cancel-abort and execute a new command
#define dfsExecEnd(c) txwSendMsg( dfsa->sbwindow, TXWM_CHAR, 0, (TXWMPARAM) TXc_END); \
                      TxCancelAbort();                                                \
                      txwAdd2History( dfsa->cmdwindow, (c));                          \
                      rc = dfsMultiCommand((c), 0, TRUE, dfsa->cmdprompt, TRUE)

// cancel-abort and execute a new command
#define dfsExecCmd(c) TxCancelAbort();                                       \
                      txwAdd2History( dfsa->cmdwindow, (c));                 \
                      rc = dfsMultiCommand((c), 0, TRUE, dfsa->cmdprompt, TRUE)

// cancel-abort and execute a new command, NO echo or prompting
#define dfsExecSilent(c) TxCancelAbort();                                    \
                         txwAdd2History( dfsa->cmdwindow, (c));              \
                         rc = dfsMultiCommand((c), 0, FALSE, FALSE, TRUE)

//- WARNING: Never execute two dfsExec... from the menu directly, that will
//-          cause the menu to restart after the 1st, causing hangs or traps
//-          Use a real "multiple command" in one go instead, or wrap
//-          the menu execution code in a BEGINWORK/ENDWORK pair


#define dfsBEGINWORK() {                                                         \
                          TRACES(("Begin, worklevel: %lu\n", dfsa->worklevel));  \
                          dfsa->worklevel++;                                     \
                       }
#define dfsENDWORK()   {                                                         \
                          if (dfsa->worklevel)                                   \
                          {                                                      \
                             if ((--(dfsa->worklevel) == 0        ) &&           \
                                   ((dfsa->menuOwner  != TXWH_NULL) ||           \
                                    (dfsa->menuopen   != 0        ) ))           \
                             {                                                   \
                                if (dfsa->menuopen != 0)                         \
                                {                                                \
                                   txwPostMsg( dfsa->menuOwner,  TXWM_COMMAND,   \
                                                 (TXWMPARAM) DFSM_DEFAULT, 0);   \
                                }                                                \
                                else                                             \
                                {                                                \
                                   txwPostMsg( dfsa->menuOwner,  TXWM_COMMAND,   \
                                                 (TXWMPARAM) DFSM_AUTODELAY, 0); \
                                }                                                \
                             }                                                   \
                          }                                                      \
                          TRACES(("End,   worklevel: %lu\n", dfsa->worklevel));  \
                       }


// Print DFSee logo+status, do startup checks, run startup commands + profile
ULONG dfsStartupLogo                            // RET   Checks and firstcmd RC
(
   char               *firstcmd                 // IN    initial command
);


// Execute multiple dfs-commands separated by # characters
ULONG dfsMultiCommand
(                                               //       err text returned if lnr
   char               *cmdstring,               // INOUT multi command / err text
   ULONG               lnr,                     // IN    linenumber or 0
   BOOL                echo,                    // IN    Echo  before execute
   BOOL                prompt,                  // IN    prompt after execute
   BOOL                output                   // IN    screen-on during cmd
);


// Initialize datastructures for analysis of current filesystem
ULONG dfsInitFileSystem
(
   void
);


// Reset general datastructures on close of current filesystem
ULONG dfsCloseFileSystem
(
   void
);

#endif
