//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// Filesystem dump & display utility
//
// Author: J. van Wijk
//

#include <txlib.h>                              // TX library interface

#include <dfsver.h>                             // DFS version info
#include <dfsrgkey.h>                           // Registration interface
#include <dfsscvar.h>                           // Subcommand/script handling
#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfswin.h>                             // windowed entry point
#include <dfstore.h>                            // Store and sector I/O
#include <dfsmdisk.h>                           // Memory disk interface
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfsspace.h>                           // DFS file-space interface
#include <dfstable.h>                           // SLT utility functions
#include <dfsupart.h>                           // FDISK partition functions
#include <dfsauxil.h>                           // Auxiliary analysis module
#include <dfsagen.h>                            // Generic command handling
#include <dfsaext.h>                            // EXT    interface (for Init)
#include <dfsafat.h>                            // FAT    interface (for Init)
#include <dfsaefat.h>                           // EFAT   interface (for Init)
#include <dfsahfs.h>                            // HFS+   interface (for Init)
#include <dfsaapfs.h>                           // APFS   interface (for Init)
#include <dfsahpfs.h>                           // HPFS   interface (for Init)
#include <dfsajfs.h>                            // JFS    interface (for Init)
#include <dfsantfs.h>                           // NTFS   interface (for Init)
#include <dfsarsr.h>                            // REISER interface (for Init)
#include <dfsaxfs.h>                            // XFS    interface (for Init)
#include <dfsaiso.h>                            // ISO    interface (for Init)
#include <dfsaswap.h>                           // SWAP   interface (for Init)
#include <dfsadump.h>                           // DUMPFS interface (for Init)
#include <dfsafdsk.h>                           // FDISK  interface (for Init)
#include <dfsufdsk.h>                           // FDISK utility functions
#include <dfsufgpt.h>                           // FDISK GPT utility functions
#include <dfscfdsk.h>                           // FDISK callback functions
#include <dfsdgen.h>                            // RUN script dialogs
#include <dfsbrows.h>                           // DFS Sectorlist browsing
#include <dfsdegpt.h>                           // FDISK PTE GPT dialog functions
#include <dfsdembr.h>                           // FDISK PTE MBR dialog functions
#include <dfsdfmbr.h>                           // FDISK MBR dialog functions
#include <dfsdfgpt.h>                           // FDISK GPT dialog functions
#include <dfsdflvm.h>                           // FDISK LVM dialog functions

#if defined (DUMP)
#define DFS_TRACE        "DFSEETRACE"           // Trace startup values
#endif

#if defined (OEM_BRANDED)
   #define DFS_STARTCMD  "about"
   #define DFS_PRODUCT   " OEM Engine"
#else
   #define DFS_STARTCMD  "map -M -m#fdisk -r- -w-#part -t-"
   #define DFS_PRODUCT   " Retail"
#endif

#if defined (DEV32)
static char DfsBldLevel[] = "@#Fsys Software:" DFS_VN "#@##1##" __DATE__ "  " __TIME__
                            "    Excallibur::EN::::@@" DFS_REG_NAME DFS_PRODUCT "; " DFS_C;
#endif

static TX1K DfsStartupCmdLine = "";
static TXLN DfsExecutableDir  = "";
static TXLN DfsExecutableName = "";

char dfsBaseSectorTypes[] =
{
   ST_FDATA,                                    //     file data extent
   ST_UDATA,                                    //     unidentified data sector
   ST_ULZ32,                                    //  I  DFSee compressed 32-bit
   ST_HDATA,                                    //     Hex   format half sector
   ST_TDATA,                                    //     Text  format data sector
   ST_ULZ16,                                    //  I  DFSee compressed 16-bit
   ST_LVINF,                                    //     LVM DLAT info sector
   ST_LVSIG,                                    //     LVM BBR  signature sector
   ST_LVREM,                                    //     LVM BBR deleted volume
   ST_LVDLF,                                    //     LVM drive-linking feature
   ST_LVDLT,                                    //     LVM drive-linking table
   ST_LVBBF,                                    //     LVM bad-block rel feature
   ST_LVBBT,                                    //     LVM bad-block rel table
   ST_LUKSH,                                    //     LUKS Crypto Header
   ST_CORES,                                    //     Apple Core Storage Header
   ST_MASTR,                                    //     Master Boot record
   ST_EXTBR,                                    //     Extended Boot record
   ST_BOOTR,                                    //     Filesystem Boot sector
   ST_BOOTX,                                    //     non standard boot rec
   ST_BOOT2,                                    //     FAT32 2nd bootsector
   ST_BOOT3,                                    //     FAT32 3rd bootsector
   ST_GPTHD,                                    //     GPT header sector
   ST_GPTAS,                                    //     GPT table  sector
   ST_MCDDM,                                    //     MAC driver description
   ST_MCDPM,                                    //     MAC disk partition map
   ST_EMPTY,                                    //     ZERO  cleared sector 00
   ST_BADFE,                                    //     Marked as BAD sector FE
   ST_FDISK,                                    //     FDISK cleared sector F6
   ST_RESRV,                                    //  I  reserved area
   ST_KNOWN,                                    // S   any known type
   ST_WILDC,                                    // SI  any type
   ST_MINIM,                                    //  I  info only, no contents
   ST_FREES,                                    //  I  Free space
   ST_BADSC,                                    //  I  Bad sector
   ST_FINAL,                                    //  I  End-of-Partition marker
   0                                            //     string terminating ZERO
};


DFSINF      dfs_anchor =
{
   NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
   NULL, NULL, NULL, NULL, NULL, NULL, NULL,
   NULL, NULL, NULL, NULL, NULL, NULL, NULL,    // FS specific functions
   0,                                           // FS specific SLT size
   1,                                           // FS specific entry point
   0,                                           // FS specific truncation point
   0,                                           // FS last sector in-use
   0,                                           // FS maximum size for expand
   0,                                           // FS freespace, unallocated
   0,                                           // FS unallocated, Smart candidate
   DFSTAT_UNKNOWN,                              // FS dirty status unknown
   FALSE, FALSE, FALSE,                         // FS specific menu toggle options
   0,                                           // FS mode id (need some Init())
   NULL,                                        // FS specific selection list
   NULL,                                        // FS specific cmdhelp text
   NULL,                                        // FS specific info
   NULL,                                        // FS specific sector types
   dfsBaseSectorTypes,                          // BASE sector type string
   NULL,                                        // ptr to boot record
   0,                                           // default snlist size
   NULL,                                        // default sn list
   NULL,                                        // default sn info
   0, 0,                                        // Browse, 'this'   sn+info
   "Unknown",                                   // list/browse description
   "-w",                                        // default value    list-show
   "-d",                                        // default contents list-show
   "",                                          // initial list FS mode
   "",                                          // FS specific command after select
   ".",                                         // default SaveTo path (current dir)
   {FALSE, FALSE, FALSE, FALSE, TRUE,           // persistent recovery flags
                        "", "", NULL},          // persistent recovery strings
   {0,NULL,0,0,0,0},                            // SPACE to be searched by 'find -o:S'
   NULL,                                        // sector area list
   NULL,                                        // info, store dependend
#if defined (DEV32)
   DfsBldLevel,
   NULL,                                        // OS/2 dynamic entrypoint structure
   TRUE,                                        // use automatic sync of LVM
#endif
#if defined (OEM_BRANDED)
   "FAT32 ",                                    // avoid FAT32 open-volume bug in OEM
#else
   "",                                          // default no FS skip on correlate
#endif
   FALSE,                                       // no LVM info yet
   0,                                           // default boot-disk id
   0,                                           // Nr of possible WIN partitions
   TRUE,                                        // list active
   0,                                           // rexx value
   0,                                           // physical disk active BMGR
   15,                                          // current timeout value BMGR
   DFSI13X_NOTTHERE,                            // I13X none/optional/required
   FALSE,                                       // I13X capable MBR on disk 1
   1,                                           // partition-id for 'part 0'
   FALSE,                                       // batch-mode disabled
   TRUE,                                        // dialogs will be used
   TXAO_NORMAL,                                 // normal verbosity
   TXAE_CONFIRM,                                // confirmed quit on errors
   1,                                           // nr of sector read attempts
   DFS_CHSTYLE_IBM,                             // default CHS-dummy style IBM
   0,                                           // overall return-code
   TRUE,                                        // retc is explainable
   80,                                          // visible screen area characters
   22,                                          // visible screen area lines
   FALSE,                                       // OS/2 filesystems likely present
   FALSE,                                       // Ignore lock errors
   FALSE,                                       // auto quit on fdisk/setboot
   TRUE,                                        // reg confirm required
   'S',                                         // reg type: U, T, S=Standard P=Pro F=Forensic
   FALSE,                                       // default store mode R/O
   TRUE,                                        // SLT build can show progress
   FALSE,                                       // running in classic mode
   FALSE,                                       // Show hidden files/dir in Browse
   FALSE,                                       // show every warning and more
   0,                                           // timestamp next status update
   0,                                           // secs between STDOUT progress
   FALSE,                                       // force (just) progress-bar to STDOUT
   TRUE,                                        // use Hex sector-nr fields
   FALSE,                                       // use Hex CHS-value fields
#if defined (OEM_BRANDED)
   FALSE,                                       // geoCalc default OFF (Vista)
   FALSE,                                       // gptAuto default OFF
#else
   TRUE,                                        // geoCalc default ON
   TRUE,                                        // gptAuto default ON
#endif
   FALSE,                                       // volFloppy default OFF
   FALSE,                                       // logAuto default OFF
   FALSE,                                       // logDate default OFF
   TRUE,                                        // var/expr ${ resolve on cmds
   FALSE,                                       // show part-type 0x00 as partitions
   0,                                           // scroll-buffer window
   0,                                           // command handling window
   0,                                           // commandline window, history
   TRUE,                                        // prompt after every manual command
   FALSE,                                       // BASIC menu/dialog active (not EXPERT)
   FALSE,                                       // no automatic menu at start
   TRUE,                                        // automatic pulldown drop
   FALSE,                                       // delayed autodrop active
   FALSE,                                       // Move Right on right-arrow, don't open submenu
   0,                                           // default drop-down menu
   0,                                           // worklevel
   0,                                           // alignment (automatic Cyl/Mib)
   SECTORSIZE /2,                               // default (hex) display size
   0,                                           // length last Find argument
   TXMAXLN,                                     // hard-coded maximum
   NULL,                                        // sel-list, 1 volume
   NULL,                                        // sel-list, 1 removable
   NULL,                                        // sel-list, 1 disk
   NULL,                                        // sel-list, 1 disk or All
   NULL,                                        // sel-list, 1 disk or SAME (Prestore)
   NULL,                                        // sel-list, mounted disks
   NULL,                                        // sel-list, 1 partition
   NULL,                                        // sel-list, logicals enabled only
   NULL,                                        // sel-list, 1 part or All
   NULL,                                        // sel-list, part COPY, disk CREATE
   0,                                           // #enabled  CCr
   NULL,                                        // sel-list, part MOVE, disk BMinstall
   0,                                           // #enabled  MBm
   NULL,                                        // sel-list, GPT + Primary, no LOG-only
   0,                                           // #enabled  GPr
   NULL,                                        // sel-list, LOG, no PRI-only, no GPT
   0,                                           // #enabled  Log
   NULL,                                        // sel-list, All freespace
   NULL,                                        // sel-list, bootables
   NULL,                                        // sel-list, Color schemes
   NULL,                                        // sel-list, Color palette
   0,                                           // use ScreenCols, full width
   #if defined (DOS32)
      FALSE,                                    // Windows-9x DosBox detected
   #endif
   "#",                                         // command separator string
   NULL                                         // DFSee executable directory
};

DFSINF       *dfsa  = &dfs_anchor;              // DFS anchor block


S_NAVIGATE  nav =                               // navigation info
{
   0, 0, 0, 0,                                  // default LSN's
   0,                                           // default index
   0,                                           // default offset
   0,                                           // this sninfo (recover etc)
   0                                            // down sninfo (auto display)
};

BYTE       *rbuf = NULL;                        // read buffer ptr, generic
BYTE       *brec = NULL;                        // read buffer ptr, bootrec

#if defined (DOS32)
BYTE       *xpar = NULL;                        // DPMI parameter  ptr,  512
BYTE       *xbuf = NULL;                        // DPMI big buffer ptr, 16KB
   short    xparSelector = 0;
   short    xbufSelector = 0;
   short    rbufSelector = 0;
   short    brecSelector = 0;
#endif


char *partypehelp[] =
{
   "",
   "GPT style partition types",
   "=========================",
   "",
   " With GPT style, the actual 'type' of the partition is recorded",
   " in a 16-byte unique identifier (UUID), you could enter these",
   " manually with the DFSee 'cr' command, or in the GPT editor (PTE)",
   " but it is more convenient to use one of the symbolic values below.",
   "",
   " The HEX values are compatbile with the Linux Gparted disk tool",
   "",
   " Some symbolic names + HEX value for GPT-style partitiong:",
   " EFI                             = ef / ef00   UEFI System partition",
   " BIOS                            = bb / bb00   BIOS Boot partition",
   " WIN, FAT, FAT32, EFAT, NTFS     = 07 / 0700   Windows Basic Data",
   " REC                             = 27 / 2700   Windows Recovery",
   " SWAP                            = 82 / 8200   Linux Swap",
   " LIN, LINUX, EXT2, EXT3, EXT4    = 83 / 8300   Linux Data",
   " HOME                            =      8302   Linux Home",
   " ROOT32                          =      8303   Linux Root x86-32",
   " ROOT64                          =      8304   Linux Root x86-64",
   " SRV                             =      8306   Linux Server Data",
   " DMCR                            =      8308   Linux Plain DM-Crypt",
   " LUKS                            =      8309   Linux LUKS encrypted",
   " RAID                            = fd / fd00   Linux Raid",
   " APPLE                           = ab / ab00   Apple Boot",
   " HFS                             = af / af00   Apple MAC OSX HFS+",
   " APFS                            = 73 / af73   Apple APFS Container",
   "",
   "",
   "MBR style partition types",
   "=========================",
   "",
   "The partition type is a numeric value or a symbolic-name",
   "displayed or used as parameter in many commands in DFSee.",
   "",
   "Numeric values are DECIMAL by default, prefix with 0x for HEX",
   "The most used system-types are:",
   "",
   "0x01   FAT12  =  FAT12",
   "0x04   FAT16  =  FAT16 < 32 MiB",
   "0x05   EXT    =  Extended partition (chain)",
   "0x06   FAT    =  FAT16 > 32 MiB",
   "0x07   IFS    =  Installable File system",
   "0x07   HPFS   =  IFS, HPFS                 (alias)",
   "0x07   NTFS   =  IFS, NTFS                 (alias)",
   "0x07   BJFS   =  IFS, Bootable JFS (eCS)   (alias)",
   "0x0a   BMGR   =  IBM BootManager",
   "0x0b   FAT32  =  FAT32 small and below cyl 1024",
   "0x0c   FAT32X =  FAT32 large or beyond cyl 1024",
   "0x0f   BIGX   =  Large extended (Windows, PQMagic)",
   "0x35   JFS    =  Journaled Filesystem (OS/2 4.5x)",
   "0x35   LVM    =  LVM container, usually with JFS",
   "0x82   SWAP   =  Linux swap partition",
   "0x83   LINUX  =  Linux native fs (EXT2/3 etc)",
   "0x83   EXT2   =  Linux EXT2 data partition (alias)",
   "0x83   EXT3   =  Linux EXT3 data partition (alias)",
   "0xfe   PS2S   =  PS/2 system partition",
   "",
   " For these values (upto 0x0c), corresponding values exist that",
   " are used to 'HIDE' the partitions from older operating systems",
   " like DOS, Win3x and OS/2, with values 'non-hidden' type + 0x10",
   "",
   "",
   "Complete overview of MBR types known by DFSee",
   "=============================================",
   "",
   "00 =   Deleted!     40 =   Venix-286  80 =   Minix1.3-  c0 =   CTOS-fsys",
   "01 =   FAT12        41 =   Wind-PREP  81 =   Minix1.5+  c1 =   DrDosSF12",
   "02 =   XenixRoot    42 =   W2K/xpLDM  82 =   SunS/SWAP  c2 =",
   "03 =   XenixUser    43 =   LinuxDrSw  83 =   LinuxNatv  c3 =",
   "04 =   FAT16-32M    44 =   GoBack-DR  84 =   Hibernate  c4 =   DrDosSF16",
   "05 =   Extended     45 =              85 =   ExtLinux   c5 =",
   "06 =   FAT16        46 =              86 =   FAT16Vset  c6 =   FAT-badVS",
   "07 =   Inst-FSys    47 =              87 =   MirrorIFS  c7 =   DisablIFS",
   "08 =   AIX boot     48 =              88 =              c8 =",
   "09 =   AIX data     49 =              89 =              c9 =",
   "0a =   IBM-BMGR     4a =              8a =   LinuxKrnl  ca =",
   "0b =   FAT32        4b =              8b =              cb =",
   "0c =   FAT32-Ext    4c =              8c =              cc =",
   "0d =                4d =   QNXv4-1st  8d =              cd =   CTOSmDump",
   "0e =   FAT16-Ext    4e =   QNXv4-2nd  8e =   Linux-LVM  ce =",
   "0f =   ExtendBig    4f =   QNXv4-3rd  8f =   ExtLinBig  cf =",
   "10 =                50 =   OnTrack-R  90 =              d0 =",
   "11 = H-FAT12        51 =   OnTrackRW  91 =              d1 =",
   "12 =   Recovery     52 =   CP/M SysV  92 =              d2 =",
   "13 =                53 =   OnTrack-6  93 =   AmoebaSys  d3 =",
   "14 = H-FAT16-32M    54 =   OnTrackDO  94 =   Amoeba-BB  d4 =",
   "15 =   ExtendHid    55 =   EZ-drive   95 =              d5 =",
   "16 = H-FAT16        56 =   V-feature  96 =              d6 =",
   "17 = H-Inst-FSys    57 =              97 =              d7 =   DfsNoWarn",
   "18 =   AstWinSwp    58 =              98 =              d8 =   CPM/86",
   "19 =                59 =              99 =   MylexBigX  d9 =",
   "1a =                5a =              9a =              da =   LinxNonFS",
   "1b = H-FAT32        5b =              9b =              db =   CPMreserv",
   "1c = H-FAT32-Ext    5c =   Edisk-mgr  9c =              dc =",
   "1d =                5d =              9d =              dd =",
   "1e = H-FAT16-Ext    5e =              9e =              de =   DELL-Util",
   "1f =                5f =              9f =              df =   DFSeeType",
   "20 =                60 =              a0 =   Save2Disk  e0 =",
   "21 =                61 =   Speedstor  a1 =   Save2-NEC  e1 =   Speedst12",
   "22 =   OxygenExt    62 =              a2 =              e2 =",
   "23 =                63 =   SCO/SystV  a3 =              e3 =   DosAccess",
   "24 =   Nec-Dos3x    64 =   Novell-R2  a4 =              e4 =   Speedst16",
   "25 =                65 =   Novell3/4  a5 =   FreeBSD    e5 =",
   "26 =                66 =   Novell386  a6 =   OpenBSD    e6 =",
   "27 =   IFShidden    67 =   Novell-x1  a7 =   NextStep   e7 =",
   "28 =                68 =   Novell-x2  a8 =   macOSdata  e8 =   LinuxLUKS",
   "29 =                69 =   Novell-SS  a9 =   NetBSD-98  e9 =",
   "2a =                6a =              aa =   OlService  ea =   Efi-Any-P",
   "2b =                6b =              ab =   macOSboot  eb =   BeOS-Fsys",
   "2c =                6c =              ac =              ec =",
   "2d =                6d =              ad =              ed =",
   "2e =                6e =              ae =              ee =   GPT-guard",
   "2f =                6f =              af =   macOS HFS  ef =   Efi-Sys-P",
   "30 =                70 =              b0 =              f0 =",
   "31 =                71 =              b1 =              f1 =   SdMaxtor1",
   "32 =                72 =              b2 =              f2 =   DosSecond",
   "33 =                73 =   APFS Cont  b3 =              f3 =",
   "34 =                74 =              b4 =              f4 =   SdMaxtor2",
   "35 =   Warp-LVM     75 =   PC/IX      b5 =              f5 =",
   "36 =                76 =              b6 =              f6 =",
   "37 =                77 =              b7 =   BSDI-fsys  f7 =",
   "38 =   TheosV3.2    78 =   XOSL-bmgr  b8 =   BSDI-swap  f8 =",
   "39 =   TheosV4sp    79 =              b9 =              f9 =",
   "3a =   TheosV4      7a =              ba =              fa =   BochsVirt",
   "3b =   TheosV4Ex    7b =              bb =   AcrHidden  fb =   VmWareSys",
   "3c =   PQMrecovr    7c =   VRAIDpart  bc =   AcronisSZ  fc =   VmWarSwap",
   "3d =                7d =              bd =              fd =   LinuxRaid",
   "3e =                7e =              be =   SunS-Boot  fe =   PS/2-syst",
   "3f =                7f =              bf =   Solaris-Z  ff =   Xenix-Bbl",
   "",
   NULL
};


char *hostvarhelp[] =
{
   "",
   "All variable names start with the '$' character. To embed a variable or",
   "expression in a command, enclose it in DOUBLE curly braces: {{expr}}",
   "",
   "Naming of user variables is free, except for names with the '$_' prefix",
   "which are reserved for system variables (like DFSee host variables) and:",
   "",
   "  $0 .. $9 which are reserved for argument passing into scripts.",
   "",
   "  $_rc     special host-variable, set automatically after every command,",
   "           when executed from a script, but can also be set manually with",
   "           a regular assignment like '$_rc = 500'",
   "",
   "           The return value from a script, either running to the end, or",
   "           with a RETURN statement will be the value of this $_rc variable.",
   "",
   "  $_retc   Always set by DFSee itself after executing a command, so it",
   "           can be used from the commandline too (outside a script).",
   "",
   "           Displaying these variables from a script is best done using",
   "           a 'PRINT' statement since a 'say' command resets them to 0.",
   "",
   "           For reserved rc values (constant definitions), see further below.",
   "",
   "Host variables available are either scalar, meaning a single value known",
   "within DFSee, or they are related to a Disk, Partition, Freespace area",
   "the Sectorlist, or to all sectors on the current opened object.",
   "",
   "In this case they are indexed arrays of values and you specify the disk,",
   "partition, freespace, list or sector number to get the desired value.",
   "",
   "- SCALAR, single value variables",
   "",
   "  $_base                PSN for first sector of opened object",
   "  $_batch               Boolean: script is running in batch mode",
   "  $_cmdsep              Command separator character, default is '#'",
   "  $_columns             Number of display columns on the screen window",
   "  $_disk                Current disk number if one is opened, or 0",
   "  $_disks               Number of disks mounted in DFSee",
   "  $_down                Sector number selected on <Enter> or 'd' command",
   "  $_drive               Driveletter string for current object, like 'D:'",
   "  $_drive               Driveletter string for current object, like 'D:'",
   "  $_dirsep              String with directory/path separator, one character",
   "  $_exedir              String with executable-directory (path), no separator",
   "  $_exename             String with executable-name only",
   "  $_fsdirty             Current opened filesystem is marked as 'dirty'",
   "  $_fsfree              Nr of 'free sectors' in filesystem   (when known)",
   "  $_fsmaxsize           Max size of FS in sectors for resize (when known)",
   "  $_fsminsize           Min size of FS in sectors for resize (when known)",
   "  $_fsys                Current filesystem name, like 'NTFS'  (mode=...)",
   "  $_geobps              Logical geometry, bytes per sector",
   "  $_geocyls             Logical geometry, number of cylinders",
   "  $_geohead             Logical geometry, number of heads",
   "  $_geosect             Logical geometry, sectors per track",
   "  $_lines               Number of display lines on the screen window",
   "  $_listed              Number of sectors listed in the Sectorlist",
   "  $_lvm                 LVM information is present on at least one disk",
   "  $_magic               DFSee version eyecatcher, use 'say {i2hex($_magic)}'",
   "  $_object              type of the current opened DFSee object (see below)",
   "  $_parts               Total number of partitions (on all disks)",
   "  $_pid                 PID for current opened object, or 0",
   "  $_platform            Operating system now running DFSee, like 'OS/2'",
   "  $_program             Program name running the script, like 'DFSee'",
   "  $_readonly            Boolean: Object is opened in read-only mode",
   "  $_retc                Return-code for last executed command, 0 is OK",
   "  $_secdesc             Current sector description like 'Master Boot Rec'",
   "  $_sectype             Single character sector-type, like 'r' for MBR",
   "  $_size                Size of the current object in sectors",
   "  $_store               Store number currently active, 0..2",
   "  $_this                Sector number for the CURRENT sector",
   "  $_up                  Sector number for 'parent' sector; 'u' command",
   "  $_version             DFSee version number *  100 ('1490'  for 14.9)",
   "                        Beyond 14.9: version * 1000 ('15040' for 15.4)",
   "  $_xtra                Sector number for 'related' sector; 'x' command",
   "",
   "   DFSee object types:  0 = no object or unknown",
   "   (value of $_object)  1 = partitioned medium, disk",
   "                        2 = partition on a disk",
   "                        3 = Volume or Device",
   "                        4 = Image File",
   "                        5 = Freespace area",
   "                        6 = Any object,  base != 0",
   "",
   "   DFSee sector types, as kept in the $_sectype host variable",
   "   can be displayed from the help menu using:",
   "",
   "                        Help -> Available sector types",
   "",
   "",
   "- DISK-LEVEL, where 'DSK' is the DFSee disk number 1..n",
   "",
   "  $_d_access[DSK]     Boolean: disk accessible (1) or not-accessible (0)",
   "  $_d_cylsize[DSK]    Cylindersize of the disk in sectors",
   "  $_d_disknr[DSK]     DFSee disk-id for this disk (same as 'DSK' :-)",
   "  $_d_firstpart[DSK]  DFSee PID for the first partition on this disk",
   "  $_d_flags[DSK]      Error/warning flags for the disk",
   "  $_d_frareas[DSK]    Number of freespace areas on the disk",
   "  $_d_frsects[DSK]    Nr of sectors in freespace areas on the disk",
   "  $_d_geobps[DSK]     Logical geometry, bytes per sector",
   "  $_d_geocyls[DSK]    Logical geometry, number of cylinders",
   "  $_d_geohead[DSK]    Logical geometry, number of heads",
   "  $_d_geosect[DSK]    Logical geometry, sectors per track",
   "  $_d_largeflop[DSK]  Boolean: disk is a large-floppy    (no partitions)",
   "  $_d_cryptdisk[DSK]  Boolean: disk is an encrypted disk (no partitions)",
   "  $_d_lastpart[DSK]   DFSee PID for the last partition on this disk",
   "  $_d_creatr[DSK]     Creator identification string, FSO or CRP pStyle",
   "  $_d_drive[DSK]      Driveletter for the disk, when FSO or CRP pStyle",
   "  $_d_fsform[DSK]     Filesystem format string, when FSO or CRP pStyle",
   "  $_d_label[DSK]      Filesystem label, string, when FSO or CRP pStyle",
   "  $_d_lvmname[DSK]    LVM identification string for this whole disk",
   "  $_d_lvmparts[DSK]   Number of partitions that have valid LVM information",
   "  $_d_ntsig[DSK]      Value of the (32-bit) NT/Windows disk signature",
   "  $_d_parts[DSK]      Number of partitions on the disk",
   "  $_d_primaries[DSK]  Number of primary partitions on the disk",
   "  $_d_pstyle[DSK]     Partition style string for disk (MBR, GPT, CRP, FSO)",
   "  $_d_removable[DSK]  Boolean: Disk appears to be a removable device",
   "  $_d_size[DSK]       Disk-size for the disk logical geometry, in sectors",
   "  $_p_unixdev[PID]    Unix devicename for the whole disk, like /dev/hda",
   "",
   "",
   "- PARTITION-LEVEL, where 'PID' is the DFSee partition-id 1..n",
   "",
   "  $_p_active[PID]     Boolean: Partition 'Active flag' is set (startable)",
   "  $_p_basepsn[PID]    Sector-number for the first sector (bootsector)",
   "  $_p_bmname[PID]     IBM bootmanager name, when using a non-LVM OS/2",
   "  $_p_checksum[PID]   DFSee checksum for the partition bootsector",
   "  $_p_cluster[PID]    Nr of sectors per cluster (or block) for filesystem",
   "  $_p_creatr[PID]     Creator identification string for the filesystem",
   "  $_p_cylsize[PID]    Cylindersize of the partition in sectors",
   "  $_p_descr[PID]      Partition type description, like 'Inst-Fsys'",
   "  $_p_disknr[PID]     Disk number for this partition",
   "  $_p_diskpart[PID]   Relative partition-number on this disk",
   "  $_p_drive[PID]      Driveletter for the partition, as known in DFSee",
   "  $_p_flags[PID]      Error/warning flags for the partition",
   "  $_p_fsform[PID]     Filesystem format for the partition (HPFS, FAT etc)",
   "  $_p_geobps[PID]     Logical geometry, bytes per sector",
   "  $_p_geocyls[PID]    Logical geometry, number of cylinders",
   "  $_p_geohead[PID]    Logical geometry, number of heads",
   "  $_p_geosect[PID]    Logical geometry, sectors per track",
   "  $_p_label[PID]      Filesystem label for the partition",
   "  $_p_lastpsn[PID]    Sector-number for the last sector in the partition",
   "  $_p_unixdev[PID]    Unix devicename for the partition, like /dev/hda8",
   "  $_p_unixnr[PID]     Unix device number for this partition",
   "  $_p_lvminfo[PID]    Boolean: Partition has LVM information defined",
   "  $_p_lvminstal[PID]  Boolean: Partition marked as 'installable' in LVM",
   "  $_p_lvmletter[PID]  LVM driveletter assigned to this partition",
   "  $_p_lvmonmenu[PID]  Boolean: Partition is on the IBM BootManager-menu",
   "  $_p_lvmpart[PID]    LVM partition name assigned to the partition",
   "  $_p_lvmvolume[PID]  LVM volume name assigned to the partition",
   "  $_p_partpsn[PID]    Sector-number where the partition-table is located",
   "  $_p_primary[PID]    Boolean: Partition is a primary",
   "  $_p_size[PID]       Partition-size, in sectors",
   "  $_p_tableslot[PID]  Slot-number used in the partition-table (0..3)",
   "  $_p_type[PID]       Partition-type value, 1..255",
   "",
   "",
   "- FREESPACE-LEVEL, where 'FID' is the DFSee freespace-id 1..n",
   "",
   "  $_f_basepsn[FID]    Sector-number for the first sector",
   "  $_f_cylsize[FID]    Cylindersize of the freespace area in sectors",
   "  $_f_descr[FID]      Freespace type description, like 'FreeSpace Pri/Log'",
   "  $_f_disknr[FID]     Disk number for this freespace area",
   "  $_f_geobps[FID]     Logical geometry, bytes per sector",
   "  $_f_geocyls[FID]    Logical geometry, number of cylinders",
   "  $_f_geohead[FID]    Logical geometry, number of heads",
   "  $_f_geosect[FID]    Logical geometry, sectors per track",
   "  $_f_lastpsn[FID]    Sector-number for the last sector in the area",
   "  $_f_size[FID]       Freespace area size, in sectors",
   "  $_f_type[FID]       Freespace type value, 1..255",
   "",
   "",
   "- SECTOR-LIST",
   "",
   "  $_s_list[#]         Sector number at specified index in the list",
   "",
   "",
   "- SECTOR-CONTENTS",
   "",
   "  $_b_sector[#]       Sector contents as 512 (binary) bytes",
   "  $_h_sector[#]       Sector contents as 512 * 2 hex digits",
   "",
   "",
   " Constant values defined with TxLib and DFSee:",
   "",
   " true                    1          Logical values",
   " false                   0",
   "",
   " rc_ok                   0          $_rc and $_retc 'OK'",
   "",
   " rc_file_not_found       2          Generic OS RC values",
   " rc_path_not_found       3",
   " rc_too_many_files       4",
   " rc_access_denied        5",
   " rc_invalid_handle       6",
   " rc_no_more_files       18",
   " rc_write_protect       19",
   " rc_not_ready           21",
   " rc_crc                 23",
   " rc_seek                25",
   " rc_sector_not_found    27",
   " rc_write_fault         29",
   " rc_read_fault          30",
   " rc_gen_failure         31",
   " rc_file_sharing        32",
   " rc_lock_violation      33",
   " rc_wrong_disk          34",
   "",
   " rc_error              200          TxLib specific RC values",
   " rc_invalid_file       202",
   " rc_invalid_path       203",
   " rc_access_denied      205",
   " rc_invalid_handle     206",
   " rc_invalid_data       207",
   " rc_alloc_error        208",
   " rc_syntax_error       210",
   " rc_invalid_drive      215",
   " rc_pending            217",
   " rc_failed             218",
   " rc_write_protect      219",
   " rc_cmd_unknown        222",
   " rc_no_compress        223",
   " rc_no_initialize      224",
   " rc_aborted            225",
   " rc_bad_option_char    226",
   " rc_too_many_args      227",
   " rc_display_change     228",
   " rc_app_quit           229",
   "",
   " rc_value_error        231          DFSee specific RC values",
   " rc_not_found          232",
   " rc_list_empty         233",
   " rc_no_script          234",
   " rc_no_device          244",
   " rc_psn_limit          245",
   " rc_st_mismatch        246",
   " rc_reboot_req         248",
   " rc_user_abort         249",
   " rc_no_change          250",
   " rc_cmd_failed         252",
   " rc_cmd_warning        254",
   " rc_not_a_cmd         1041          External command not found",
   NULL
};

static char *usagehelp[] =
{
   "  [global-dfsee-switches]  [DFSee multi-command]",
   "",
   "  Switch character for dfsee switches is '-' or '/'. All single letter",
   "  switches are case-sensitive, long switchnames like 'query' are not.",
   "",
   NULL
};

char *DFSeeSwitchhelp[] =
{
   "",
   "DFSee program specific switches:",
   "================================",
   "",
   " -?            = help on DFSee commandline switches (this text)",
   " -A:nr         = number of sector read attempts, default 1 (no retries)",
   " -align:#sect  = partition alignment on CR, default 0=auto-Cyl/MiB, 1=MiB",
   " -b            = batch option, automatic 'batch on' command at startup",
   " -b-           = interactive option, 'batch off' command at startup (default)",
   " -B            = Show the version/copyright banner on DFSee startup (default)",
   " -B-           = Suppress the version/copyright banner on DFSee startup",
   " -c:[1|2]      = CHS dummy style: 1=PQmagic, 2=MS, default 0=IBM/DFSee",
   " -d            = Add the physical disks to devicemap at startup     (default)",
   " -d-           = do NOT add any physical disks to the devicemap at startup",
   " -d:[max]      = Add at most 'max' physical disks to devicemap at startup",
   " -D:letter     = linux device letter, default SCSI/SATA, use 'h' for IDE",
   " -echo         = Echo each command to output-window and logfile     (default)",
   " -echo-        = Suppress command echo before each command being executed",
   " -expert       = Set UI menus and dialogs to EXPERT mode on startup",
   " -expert-      = Set UI menus and dialogs to BASIC  mode on startup (default)",
   " -floppy       = Include floppies A: and B: in volume lists (menus/dialogs)",
   " -g            = Display specific information to STDOUT only (Engine mode)",
   " -G[:1..20]    = Progress percentage to STDOUT only (> 10 is single line)",
   " -G:0          = Display progress BAR to STDOUT, no percentage, no texts",
   " -geocalc      = Calculate disk geometry from the MBR/LVM contents  (default)",
   " -geocalc-     = Do NOT calculate disk geometry from contents, use OS geo",
   " -gptauto      = Interpret GPT info when present show GPT tables    (default)",
   " -gptauto-     = Do NOT interpret GPT info, but show protective MBR instead",
   " -h:[0|1|2|3]  = PTE use hex SN+Size fields (1), CHS fields (2) or both (3)",
   " -H-           = Set default input radix for SECTOR and TYPE values to DECIMAL",
   " -H:mask       = Set HEX input radix for selected classes (see dfscmds.txt)",
#if defined (DOS32)
   " -I            = use std-Int13 when #cyl is below 1024  (default,   DOS only)",
   " -I-           = use standard-Int13, avoid ext-int13 check call    (DOS only)",
#endif
#if defined (DEV32)
   " -i            = ignore lock-errors, no pop-up warnings  (default, OS/2 only)",
   " -i-           = do not ignore lock-errors, with extra pop-up,    (OS/2 only)",
   " -L            = use large-file for > 2 GiB file support (default, OS/2 only)",
   " -L-           = do not use large-file for > 2 GiB file support   (OS/2 only)",
#endif
   " -m[:autoshow] = use 'map' or user defined cmd for 'autoshow' in FDISK mode",
   " -p:[0|1|2]    = warning level, 0 = minimal (default), 2 = maximum  (pedantic)",
   " -P            = Use prompting and input dialog windows where needed (default)",
   " -P-           = Don't use prompting and input dialog windows (aka batch mode)",
   " -profile-     = Do NOT execute DFSee standard profile script 'profile.dfs'",
   " -profile:name = Use specific profile, name or full path, including extension",
   " -Q            = quit DFSee automatically after last command has been executed",
   " -Q-           = do NOT quit automatically after last command        (default)",
   " -q            = quiet option, automatic 'screen off' command at startup",
   " -query        = execute a query command with logfile, quit automatically",
#if defined (DARWIN)
   " -raw-         = Use block devices 'diskN', not the RAW 'rdiskN' (slower!)",
#endif
#if defined (REGISTRATION)
   " -r:regfile    = registration filename with optional path and extension",
#endif
   " -R            = start in Read-only operation, no write back to disk/volumes",
   " -S            = shell mode, do not allow quit from DFSee (use as OS shell)",
   " -s:separator  = specify a command separator character, default is '#'",
#if defined (DEV32)
   " -synclvm      = Try to synchronize the LVM Engine on driveletter changes",
#endif
#if defined (DUMP)
   " -simulate:nnn = simulate a BAD-SECTOR error on every modulo 'nnn' PSN read",
#endif
   " -skipfs:name  = skip filesystems of type 'name' when matching driveletters",
   " -t            = include timestamp line with date/time before each command",
   " -t-           = do not include timestamp lines before each command   (default)",
#if defined (USEWINDOWING)
   " -V            = Refresh the volumes list when starting a file-dialog (default)",
   " -V-           = Do NOT refresh volume list for each file-dialog, to save time",
   " -w            = Use windowing user-interface, with menus and dialogs (default)",
   " -w-           = Use a command-line drive user-interface, no menus or dialogs",
#endif
   "",
   " For help use the '?' and '?\?' commands, use the <F1> key or read DFSCMDS.TXT",
   "",
   NULL
};


// Interactive DFS mode
static ULONG dfsInteractive
(
   char               *cmdstring                // IN    initial command
);

// Interpret and execute DFSee command;
static ULONG dfsSingleCommand
(
   char               *dfscmd,                  // IN    DFS command
   BOOL                echo,                    // IN    Echo command
   BOOL                prompt,                  // IN    prompt after execute
   BOOL                output                   // IN    screen-on during cmd
);


#define DFS_RX_ENV   ((PSZ) "DFS")              // name of DFS REXX env
// Set extra long names usable as switch or command option
static void dfsSetLongSwitchNames
(
   void
);


int main (int argc, char *argv[]);

/*****************************************************************************/
/* Main function of the program, handle commandline-arguments                */
/*****************************************************************************/
int main (int argc, char *argv[])
{
   ULONG               rc = NO_ERROR;           // function return
   TXLN                stringBuffer;
   TXLN                fileName;
   int                 i;
   char               *s;
   TXA_OPTION         *opt;

   strcpy( DfsStartupCmdLine, argv[0]);         // name of executable, usually with path
   if (DfsStartupCmdLine[0] == '.')             // relative path
   {
      getcwd( DfsExecutableDir, TXMAXLN);
      strcat( DfsExecutableDir, DfsStartupCmdLine + 1);
   }
   else
   {
      strcpy( DfsExecutableDir, DfsStartupCmdLine);
   }
   for (i = 1; i < argc; i++)                   // add parameters to startup cmd
   {
      strcat( DfsStartupCmdLine, " ");
      strcat( DfsStartupCmdLine, argv[i]);
   }
   if ((s = strrchr( DfsExecutableDir, FS_PATH_SEP)) != NULL)
   {
      *s = 0;                                   // strip EXE-name, incl path separator
      strcpy( DfsExecutableName, s + 1);        // save  EXE-name itself too
   }
   else
   {
      strcpy( DfsExecutableName, DfsExecutableDir);  //- param-0 is EXE-name itself
      strcpy( DfsExecutableDir, "");                 //- EXE dir not really known
   }
   dfsa->dfsExeDir = DfsExecutableDir;
   dfsa->dfsExeName= DfsExecutableName;

   TxINITmain( DFS_TRACE, "DFS", TxaExeSwitch('q'), FALSE, dfsSetLongSwitchNames);

   TRACES(("sizeof(int): %zu  sizeof(ULONG): %zu\n", sizeof(int), sizeof( ULONG)));

   strcpy( dfsa->SavetoPath, TxaExeSwitchStr( TXA_O_DIR, NULL, ".")); // default to . (current DIR)

   if (((opt = TxaOptValue('l')) != NULL) && (opt->type == TXA_STRING)) // start named logfile now ?
   {
      TxaExeSwAsString( 'l', TXMAXLN, stringBuffer);
      TxRepl( stringBuffer, FS_PALT_SEP, FS_PATH_SEP); // fixup ALT separators
      if ( (stringBuffer[0] == FS_PATH_SEP) ||  // absolute path
          ((stringBuffer[2] == FS_PATH_SEP) &&
           (stringBuffer[1] == ':')         ))
      {
         strcpy(  fileName, stringBuffer);
      }
      else                                      // relative path
      {
         sprintf( fileName, "%s%c%s", dfsa->SavetoPath, FS_PATH_SEP, stringBuffer);
      }
      TxAppendToLogFile( fileName, TRUE);
   }
   if (TxaExeSwitch('?'))                       // switch help requested
   {
      TxPrint( "\nUsage: %s ", TxaExeArgv(0));
      TxShowTxt( usagehelp);                    // program usage, generic
      TxShowTxt( TxGetSwitchhelp());            // Library specific switches
      TxShowTxt( DFSeeSwitchhelp);              // DFSee specific switches
   }
   else
   {
      TXLN          ar;                         // arguments

      txsInitVariablePool( dfsGetVariable);     // var and expr resolving
      #if defined (DEV32)
         if ((TxaExeSwitchSet('L') == FALSE) || // not specified
             (TxaExeSwitch(   'L')))            // or requested use
         {
            TxInitDllApiOS2(&(dfsa->os2api));   // init entrypoints + PM-morph
         }
         dfsa->lvmSyncApi = TxaExeSwitchSet(DFS_O_SYNCLA); // -synclvm ?
      #endif
      if (TxaExeSwitchSet(DFS_O_SKIPFS))        // when default overruled
      {
         strcpy( dfsa->skipFs, TxaExeSwitchStr( DFS_O_SKIPFS, "Fs-skip", ""));
         strcat( dfsa->skipFs,  " ");           // always a space terminator!
         TxStrToUpper( dfsa->skipFs);           // and make uppercase
      }
      if (TxaExeSwitchSet(DFS_O_FLOPPY))        // when default overruled
      {
         dfsa->geoCalc = TxaExeSwitch(DFS_O_FLOPPY);
      }
      if (TxaExeSwitchSet(DFS_O_GEOCALC))       // when default overruled
      {
         dfsa->geoCalc = TxaExeSwitch(DFS_O_GEOCALC);
      }
      if (TxaExeSwitchSet(DFS_O_GPTAUTO))       // when default overruled
      {
         dfsa->gptAuto = TxaExeSwitch(DFS_O_GPTAUTO);
      }
      if (TxaExeSwitchSet(DFS_O_LOGDATE))       // when default overruled
      {
         dfsa->logDate = TxaExeSwitch(DFS_O_LOGDATE);
      }
      else if (TxaExeSwitchSet(DFS_O_LOGAUTO))  // when default overruled
      {
         dfsa->logAuto = TxaExeSwitch(DFS_O_LOGAUTO);
      }
      if (TxaExeSwitchSet(DFS_O_EXPERT))        // when default overruled
      {
         dfsa->expertui = TxaExeSwitch(DFS_O_EXPERT);
      }
      if (TxaExeSwitchSet(DFS_O_ALIGN))         // when default overruled
      {
         dfsa->alignment = TxaExeSwitchNum( DFS_O_ALIGN, NULL, 0);
      }
      if (TxaExeSwitchSet('$'))                 // when default overruled
      {
         dfsa->veResolve = TxaExeSwitch('$');
      }
      if (TxaExeSwitch(TXA_O_QUERY))            // -query long switch
      {
         if (!TxaExeSwitchSet('l'))             // no explicit logfile
         {
            TxAppendToLogFile( "dfsquery", TRUE);
         }
         dfsa->regconfirm = FALSE;              // no reg-nagging for now ...
         if (!TxaExeSwitchUnSet('Q'))           // no -Q-
         {
            dfsa->nowindow = TRUE;              // don't use windowing
            dfsa->autoquit = TRUE;              // auto quit after cmd
         }
         strcpy( ar,  "query");                 // default command
      }
      else
      {
         strcpy( ar,  DFS_STARTCMD);            // default command
      }
      if (TxaExeSwitchSet('G'))                 // 0 if not set at all
      {
         ULONG stdOutMode = TxaExeSwitchNum( 'G', NULL, 1);

         if (stdOutMode == 0)
         {
            dfsa->stdOutInterval = 0;
            dfsa->stdOutProgBars = TRUE;
         }
         else
         {
            dfsa->stdOutInterval = stdOutMode;
            dfsa->stdOutProgBars = FALSE;
         }
      }
      dfstInitStoreInfo();                      // Working stores
      if (TxaExeSwitch('R'))
      {
         dfsa->ReadOnly = TRUE;                 // set stores to R/O by default
      }
      #if defined (USEWINDOWING)
         if (TxaExeSwitch('Q'))                 // or -Q,  request autoquit
      #else
         if (!TxaExeSwitchUnSet('Q'))           // no -Q-, suppress autoquit
      #endif
      {
         dfsa->autoquit = TRUE;                 // auto quit after cmd
         dfsa->nowindow = TRUE;                 // default no win on autoquit
      }
      dfsa->warnLevel = TxaExeSwitchNum( 'p', NULL, 0);
      if (TxaExeSwitchSet('h'))                 // hex-fields as default
      {
         ULONG         hex = TxaExeSwitchNum( 'h', NULL, 1);

         dfsa->hexSNS = (hex & 0x01) ? TRUE : FALSE; // sector and size values
         dfsa->hexCHS = (hex & 0x02) ? TRUE : FALSE; // CHS values
      }
      if (TxaExeSwitchSet('H'))                 // sectors as HEX, or specific mask
      {                                         // note that -H- sets to ZERO (DECIMAL)
         TxSetNumberRadix( TxaExeSwitchNum( 'H', NULL, DFS_SEC_RADIX));
      }
      else                                      // standard (all decimal)
      {
         TxSetNumberRadix( DFS_SEC_RADIX);
      }

      if ((TxaExeSwitchSet('i')) &&             // specified
          (TxaExeSwitch(   'i') == FALSE) )     // and NO ignore
      {
         dfsa->LockIgn = FALSE;                 // require lock warnings
      }
      else
      {
         dfsa->LockIgn = TRUE;                  // default ignore
      }

      if (TxaExeSwitch('b'))
      {
         dfsa->batch     = TRUE;                // set global force on
         dfsa->LockIgn   = TRUE;                // global Ignore status
      }
      dfsa->defDispSize  = (TxScreenRows() >= 40) ? dfsGetSectorSize() : SECTORSIZE/2;
      dfsa->cmdSeparator = TxaExeSwitchStr(    's', "SeparatorCh", "#");
      dfsa->eStrategy    = TxaErrorStrategy(   'E', dfsa->batch);
      dfsa->verbosity    = TxaOutputVerbosity( 'O');
      dfsa->readAttempts = TxaExeSwitchNum(    'A', NULL, 1);
      dfsa->chsStyle     = dfsGetChsStyle(     'c', TXA_1ST, TRUE);
      if (_MAX_PATH < dfsa->maxPath)
      {
         dfsa->maxPath = _MAX_PATH;             // use smaller platform limit
      }

      TxSetPriority('H');                       // default high priority

      TRACES(("Allocating generic read and parameter buffers ...\n"));

      #if defined (DOS32)
         xpar = txDpmiAlloc(     1, SECTORSIZE, &xparSelector);
         xbuf = txDpmiAlloc( XBUFS, SECTORSIZE, &xbufSelector);
         rbuf = txDpmiAlloc( RBUFS, SECTORSIZE, &rbufSelector);
         brec = txDpmiAlloc( BBUFS, SECTORSIZE, &brecSelector);
      #else
         rbuf = TxAlloc( RBUFS, SECTORSIZE);
         brec = TxAlloc( BBUFS, SECTORSIZE);
      #endif
      dfsa->snsize = LSIZE;
      dfsa->snlist = TxAlloc( dfsa->snsize, sizeof(ULN64));
      if (dfsa->snlist == NULL)
      {
         TRACES(( "Allocate %u entry snlist failed, retry with /16 ...\n", LSIZE));
         dfsa->snsize = LSIZE / 16;
         dfsa->snlist = TxAlloc( dfsa->snsize, sizeof(ULN64));
      }
      dfsa->sninfo = TxAlloc( dfsa->snsize, sizeof(USHORT));

      TRACES(("Allocated %3hu * 512 readbuffer at: %p\n",        RBUFS, rbuf));
      TRACES(("Allocated %3hu * 512 bootbuffer at: %p\n",        BBUFS, brec));
      TRACES(("Allocated %8u entries of SNlist at: %p + %p\n",   dfsa->snsize, dfsa->snlist, dfsa->sninfo));

      if ( (rbuf != NULL)         && (brec != NULL) &&
         #if defined (DOS32)
           (xpar != NULL)         && (xbuf != NULL) &&
         #endif
           (dfsa->snlist != NULL) &&  (dfsa->sninfo != NULL))
      {
         #if defined (REGISTRATION)
            if (dfsRegistrationTest( DFS_REG_NAME, // product name
                    TxaExeSwitchStr( 'r', "regFilename", ""),
                                     DFS_VN,      // major version for registration
                                     DFS_INTROD,  // introduction/expiry date
                                     DFS_COMPILE, // compile date of THIS
                                     DFS_EXPIRE)) // valid days from compile
            {
               //- more startup checks here ?
            }
            else
            {
               TXTM       text;
               time_t     tt = time( &tt);      // current date/time

               strftime( text, TXMAXTM, "%A %d %B %Y", localtime( &tt));
               TxPrint( "\nThe evaluation period to use DFSee without a "
                          "registration key has expired,\n"
                          "the registration file is invalid, "
                          "or we are living in mysterious times :-)\n"
                          "The system date indicates it is %s today ...\n\n"
                          "Please register the program for further use.\n\n", text);
               #if defined (REGISTRATION)
                  dfsRegistrationInfo( DFSRG_CONFIRM_ONLY, DFS_V, TRUE); // force confirmation
               #endif
               rc = DFS_CMD_FAILED;
            }
         #endif
      }
      else
      {
         TxPrint( "\nError allocating minimum buffer memory\n");
         rc = DFS_ALLOC_ERROR;
      }
      if (rc == NO_ERROR)
      {
         //- Init Media map for specified or maximum number of disks
         dfsInitMediaMap( TxaExeSwitchNum( 'd', NULL, DFS_MAX_DISK));
         dfsFdskInit();                         // Well defined startup mode
         if (TxaExeArgc() > 1)                  // arguments given
         {
            char    qstring[3] = "?q";

            qstring[0] = dfsa->cmdSeparator[0]; // use current separator

            TxaGetArgString( TXA_1ST, 1, TXA_ALL, TXMAXLN, ar);
            if (strstr( ar, qstring) != NULL)
            {
               dfsa->nowindow = TRUE;           // don't use windowing
            }
            else if (((strncasecmp( TxaExeArgv(1), "dfsdisk",  7) == 0) &&
                               ((TxaExeArgc() > 2)      ||
                                 TxaExeSwitchUnSet('w') ))           ||
                      (strncasecmp( TxaExeArgv(1), "fdisk",    5) == 0) ||
                      (strncasecmp( TxaExeArgv(1), "dfscheck", 8) == 0) ||
                      (strncasecmp( TxaExeArgv(1), "query",    5) == 0) ||
               dfsFdskIsFdskCmd( TxaExeArgv(1),  TRUE))
            {
               if (strncasecmp( TxaExeArgv(1), "query", 5) == 0)
               {
                  dfsa->regconfirm = FALSE;     // no reg-nagging for now ...
                  // TxaExeSetItem("-B-");      // and no copyright logo
               }
               if (!TxaExeSwitchUnSet('Q'))     // no -Q-
               {
                  dfsa->nowindow = TRUE;        // don't use windowing
                  dfsa->autoquit = TRUE;        // auto quit after cmd
               }
            }
         }

         #if defined (USEWINDOWING)
            dfsFdskLvmInitHelp();
            dfsCreateMbrInitHelp();
            dfsCreateGptInitHelp();
            dfsFdskPteMbrInitHelp();
            dfsFdskPteGptInitHelp();
            dfsBrowseInitHelp();                // make BROWSE help available now
            if (TxaExeSwitchSet('w'))           // explicit windowing setting
            {
               if (TxaOption('w'))              // set windowing on
               {
                  dfsa->nowindow = FALSE;       // use windowing
               }
               else
               {
                  dfsa->nowindow = TRUE;        // don't use windowing
               }
            }
            do
            {
               if (dfsa->nowindow)
               {
                  dfsa->dialogs = FALSE;        // dialogs not supported
                  dfsInteractive( ar);          // non windowed interface
               }
               else
               {
                  dfsa->dialogs = !TxaExeSwitchUnSet('P');
                  dfsWindowed(    ar);          // windowed interface
               }
               strcpy( ar, "map#display");      // new initial command

               #if defined (DOS32)              // might need to probe ANSI ...
                  if (dfsa->retc == TX_DISPLAY_CHANGE)
                  {
                     TxPrintInit( TxaExeSwitch('a'), TxaExeSwitch('7'));
                  }
               #endif
            } while (dfsa->retc == TX_DISPLAY_CHANGE);
         #else
            dfsa->nowindow = TRUE;              // don't use windowing
            dfsa->dialogs  = FALSE;             // dialogs not supported
            dfsInteractive( ar);                // non windowed interface
         #endif

         DFSFNCALL(dfsa->FsClose,0,0,NULL,NULL);  // FS specific handler, close
         dfstInitStoreInfo();                   // close all open stores
         dfsmDetachMedium( DFSM_UNMOUNT_ALL, FALSE); // quietly detach
         rc = dfsa->retc;
         TRACES(("Program RC from retc: %u = 0x%X\n", rc, rc));
      }
      #if defined (USEWINDOWING)

         //- Signal DFSee has terminated in OS terminal/window title,
         //- requesting restore of original when available ...
         sprintf( stringBuffer, "%s %s%s", DFS_N, DFS_VN, TX_APPWIN_TRESTORE);
         dfsSetAppWindowTitle( stringBuffer);

         txSelDestroy( &(dfsa->slBootabl));
         txSelDestroy( &(dfsa->slFreeCCr));
         txSelDestroy( &(dfsa->slFreeMBm));
         txSelDestroy( &(dfsa->slFreeGPr));
         txSelDestroy( &(dfsa->slFreeLog));
         txSelDestroy( &(dfsa->slFreeAll));
         txSelDestroy( &(dfsa->slPartAll));
         txSelDestroy( &(dfsa->slPartOne));     // remove list contents
         txSelDestroy( &(dfsa->slPartLog));
         txSelDestroy( &(dfsa->slDiskOne));
         txSelDestroy( &(dfsa->slDiskMnt));
         txSelDestroy( &(dfsa->slDiskAll));
         txSelDestroy( &(dfsa->slDiskSam));
         txSelDestroy( &(dfsa->slVolumes));     // remove volume list
         txSelDestroy( &(dfsa->slReMedia));     // remove Removable media
         txSelDestroy( &(dfsa->slSchemes));
       #if defined (UNIX)
         txSelDestroy( &(dfsa->slPalette));
       #endif
      #endif                                    // USEWINDOWING

      txsTermVariablePool();                    // Free any txsvar resources
      dfsFreeDiskInfo();                        // free disk-partition info
      TxFreeMem( dfsa->snlist);
      TxFreeMem( dfsa->sninfo);                 // free Sectorlist and info

      #if defined (DOS32)
         txDpmiFree(xparSelector);
         txDpmiFree(xbufSelector);
         txDpmiFree(rbufSelector);
         txDpmiFree(brecSelector);
      #else
         TxFreeMem( brec);
         TxFreeMem( rbuf);
      #endif
   }
   TxEXITmain(rc);                              // TX Exit code, incl tracing
}                                               // end 'main'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set extra long names usable as switch or command option
/*****************************************************************************/
static void dfsSetLongSwitchNames
(
   void
)
{
   TxaOptionLongName(DFS_O_LVMSIZE, "lvmsize");  // LVM size use for disksize/geo
   TxaOptionLongName(DFS_O_LVMSNP,  "lvmsnp");   // LVM partition Serial#
   TxaOptionLongName(DFS_O_LVMSNV,  "lvmsnv");   // LVM volume    Serial#
   TxaOptionLongName(DFS_O_LVMSND,  "lvmsnd");   // LVM thisdisk  Serial#
   TxaOptionLongName(DFS_O_LVMSNB,  "lvmsnb");   // LVM bootdisk  Serial#
   TxaOptionLongName(DFS_O_SKBADS,  "skipbads"); // Skip bad areas on clone
   TxaOptionLongName(DFS_O_SKIPFS,  "skipfs");   // Skip FS's on correlate
   TxaOptionLongName(DFS_O_GEOCALC, "geocalc");  // Calculate geo from contents
   TxaOptionLongName(DFS_O_I13X,    "I13X");     // Require I13X signatures
   TxaOptionLongName(DFS_O_FILESYS, "fs");       // Filesystem specification
   TxaOptionLongName(DFS_O_GPTAUTO, "gptauto");  // Automatic GPT interpretation
   TxaOptionLongName(DFS_O_LOGAUTO, "logauto");  // Automatic logfile numbering
   TxaOptionLongName(DFS_O_LOGDATE, "logdate");  // Automatic logfile DATE postfix
   TxaOptionLongName(DFS_O_LONGFN,  "lfn");      // Long Filename
   TxaOptionLongName(DFS_O_EXPERT,  "expert");   // Expert UI menu mode
   TxaOptionLongName(DFS_O_ATTRIB,  "attrib");   // Attribute value
   TxaOptionLongName(DFS_O_ALIGN,   "align");    // Attribute value
   TxaOptionLongName(DFS_O_GUID,    "guid");     // GUID value
   TxaOptionLongName(DFS_O_FLOPPY,  "floppy");   // Include floppies in lists
   TxaOptionLongName(DFS_O_STD,     "std");      // STD behaviour wanted
   TxaOptionLongName(DFS_O_TREE,    "tree");     // TREE iterate
   TxaOptionLongName(DFS_O_PROFILE, "profile");  // PROFILE specification

   #if defined (DEV32)
      TxaOptionLongName( DFS_O_SYNCLA, "synclvm"); // LVM automatic sync
   #endif
}                                               // end 'dfsSetLongSwitchNames'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Print DFSee logo+status, do startup checks, run startup commands + profile
/*****************************************************************************/
ULONG dfsStartupLogo                            // RET   Checks and firstcmd RC
(
   char               *firstcmd                 // IN    initial command
)
{
   ULONG               rc = NO_ERROR;
   FILE               *fp;
   TXLN                osd;                     // OS description string
   TXLN                msg;
   BOOL                dologo = (!TxaExeSwitchUnSet('B'));

   ENTER();
   TRACES(("Startup command is '%s'\n", firstcmd));

   if (dologo)
   {
      TXTS             alead;

      strcpy( alead, " ");
      #if defined (USEWINDOWING)
         if (txwIsWindow( TXHWND_DESKTOP))
         {
            strcat( alead, "    ");
         }
      #endif

      TxPrint("\n%s     DFSee version %s  %s\n", alead, DFS_V, DFS_C);
      TxPrint(  "%s[ www.dfsee.com ]"
                  "\n", alead);

      if (dfsa->ReadOnly || TxaExeSwitch('S'))
      {
         if (dfsa->ReadOnly)                    // Readonly operation ?
         {
            TxPrint(  "%sRun read-only %s-R%s, normal writing is disabled\n",
                      alead, CBR, CNN);
         }
         if (TxaExeSwitch('S'))                 // Shell mode ?
         {
            TxPrint(  "%sRun as SHELL  %s-S%s, quit and <F3> are disabled\n",
                      alead, CBC, CNN);
         }
         TxPrint( "\n");
      }
   }
   TRACES(( "Retrieve OS version info ...\n"));
   (void) TxOsVersion( osd);                    // get/check OS version info
   TRACES(( "OS version info '%s'\n", osd));
   strcpy( msg, "");
   #if   defined (WIN32)
      if (strstr( osd, "Windows-9") != NULL)    // Win-9x detected
      {
         strcpy( msg, "Windows-9x is not supported by this "
                      "version of DFSee!  Use DFSDOS.EXE after "
                      "'Restart in MS-DOS mode' or use a DOS bootdisk.");
      }
   #elif defined (DOS32)
      if (strstr( osd, "DosBox") != NULL)       // Win or OS/2 DosBox (VDM)
      {
         if (strstr( osd, "Windows-9") != NULL) // Win-9x detected
         {
            dfsa->win9x = TRUE;

            strcpy( msg, "Running DFSDOS.EXE in a Win9x DosBox might "
                         "give unpredictable results!  Use DFSDOS.EXE after "
                         "'Restart in MS-DOS mode' or use a DOS bootdisk.");
         }
         else
         {
            sprintf( msg, "Running DFSDOS.EXE in a virtual DosBox "
                          "might give unpredictable results! Please use "
                          "the native version DFS%s.EXE when possible.",
                           (strstr( osd, "Win")) ? "WIN" : "OS2");
         }
      }
      else if (strstr( osd, "MemMgr") != NULL)  // EMM386 or other manager
      {
         strcpy( msg, "Running DFSDOS.EXE under a Memory Manager like "
                      "EMM386 or QEMM might give unpredictable results! "
                      "For maximum reliability, run without any external "
                      "memory manager (check autoexec.bat and config.sys).");
      }
   #elif defined (DARWIN)
      if (!TxaExeSwitchUnSet('d'))
      {
         dfsUxDeviceExists( "/dev/disk0", TRUE); // repeat SIP disk0 hint (log, window)
      }
   #endif

   #if defined (USEWINDOWING)
      dfsBEGINWORK();
   #endif                                       // USEWINDOWING
   if (strlen( msg) != 0)                       // OS versus DFSee mismatch
   {
      if (dfsa->batch)                          // just show to screen/log
      {
         strcat( msg, "\n\n");
         TxNamedMessage( FALSE, 5004, " WARNING: OS versus DFSee mismatch ", msg);
      }
      else                                      // interactive, confirm
      {
         strcat( msg, "\n\nDo you want to take the risk and continue with the program ? [Y/N]: ");
         if (!TxConfirm( 5004, msg))
         {
            rc = DFS_QUIT;
         }
      }
   }

   if (rc == NO_ERROR)                          // not aborted yet ...
   {
      if (!TxaExeSwitchUnSet( DFS_O_PROFILE))   // unless profile disabled
      {
         TRACES(( "Check for a DFSee profile ...\n"));
         strcpy( osd, DFS_USER_PROFILE);        // default profile name

         if ((fp = TxFindAndOpenFile( TxaExeSwAsString( DFS_O_PROFILE, TXMAXLN, osd), "PATH", msg)) == NULL)
         {
            fp = TxFindAndOpenFile( DFS_PROFILE_TEMPLATE, "PATH", msg);
         }
         if (fp != NULL)                        // user or template profile
         {
            TXLN             command;

            fclose( fp);
            sprintf( command, "run '%s'", msg);
            dfsMultiCommand( command, 0, FALSE, FALSE, TRUE); // exec profile
         }
      }

      if (strlen( firstcmd))
      {
         #if !defined (OEM_BRANDED)
            static BOOL initialLogDialogDone = FALSE;

            TRACES(( "Automatic start of logging, default log dialog; ONCE using -l\n"));
            if ((initialLogDialogDone == FALSE) &&       //- present dialog only once (not on resize)
                (TxQueryLogFile(NULL, NULL) == NULL) &&  //- if not logging/tracing yet
                (!TxaExeSwitchSet(  'b')))               //- and no -b  switch
            {
               TXA_OPTION *opt;

               if (((opt = TxaOptValue('l')) != NULL) && (opt->type == TXA_NO_VAL)) // -l without a value
               {
                  dfsMultiCommand( "logfile", 0, FALSE, FALSE, TRUE);
                  TxCancelAbort();              // reset pending abort status
               }
               initialLogDialogDone = TRUE;
            }
         #endif

         TxPrint("\nDFSee  invocation : %s%s%s\n", CBC, DfsStartupCmdLine, CNN);
         getcwd( msg, TXMAXLN);                 // get current drive/directory
         TxPrint(  "Working directory : %s%s%s\n", CBG, msg, CNN);

         TRACES(( "Execute initial command '%s'\n", firstcmd));
         if ((dologo) && (strstr( firstcmd, "about") == NULL)) // no about in cmd
         {
            dfsMultiCommand( "about -c- -P-", 0, FALSE, FALSE, TRUE);
         }
         #if defined (USEWINDOWING)
            txwAdd2History( dfsa->cmdwindow, firstcmd);
            rc = dfsMultiCommand(  firstcmd, 0,         TRUE,   //- execute command, with echo, and
                (strcmp( firstcmd, DFS_STARTCMD) != 0), TRUE);  //- status-prompt for non-default cmd
         #else
            //- keep no-status-prompt for non-windowd, to keep OEM-engine backward compatible
            rc = dfsMultiCommand( firstcmd, 0, TRUE, FALSE, TRUE); // execute command
         #endif
      }
      else
      {
         TxPrint("\nDFSee  invocation : %s%s%s\n", CBC, DfsStartupCmdLine, CNN);
      }
   }
   #if defined (USEWINDOWING)
      txwInvalidateAll();                       // make sure screen is complete
                                                // after possible popup dialog
      dfsENDWORK();
   #endif                                       // USEWINDOWING
   RETURN (rc);
}                                               // end 'dfsStartupLogo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Use classic text mode and accept interactive display commands
/*****************************************************************************/
static ULONG dfsInteractive
(
   char               *cmdstring                // IN    initial command
)
{
   ULONG               rc;
   TXLN                command;                 // dfs command

   ENTER();

   #if defined (UNIX)                           // no 8-bit ASCII allowed
      TxSetAscii7Mode( TRUE);                   // on standard terminals
   #endif

   #if defined (USEWINDOWING)
      dfsa->sbWidth  = TxScreenCols() -1;       // can use screen -1 (wrap)
      dfsa->sbLength = TxScreenRows();          // if windowing off
   #endif

   rc = dfsStartupLogo( cmdstring);

   #if defined (USEWINDOWING)
      if (!dfsa->autoquit && (dfsa->warnLevel > 1))
      {
         TxPrint("For the windowed user-interface use the 'mode -w' command.\n\n");
      }
   #endif
   while (((rc & DFS_QUIT) != DFS_QUIT) && !dfsa->autoquit)
   {
      TxPrint("%s;  Cmd : ",  DFS_N);
      fflush( stdout);
      memset( command, 0, TXMAXLN);
      fgets(  command, TXMAXLN, stdin);         // get next command
      TxRepl( command, '\n', 0);
      TxCancelAbort();                          // reset pending abort status
      rc = dfsMultiCommand( command, 0, TRUE, TRUE, TRUE);
   }
   RETURN ( dfsa->retc);
}                                               // end 'dfsInteractive'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Execute multiple dfs-commands separated by # characters
/*****************************************************************************/
ULONG dfsMultiCommand
(                                               //       err text returned if lnr
   char               *cmdstring,               // INOUT multi command / err text
   ULONG               lnr,                     // IN    linenumber or 0
   BOOL                echo,                    // IN    Echo  before execute
   BOOL                prompt,                  // IN    prompt after execute LAST cmd
   BOOL                output                   // IN    screen-on during cmd
)
{
   ULONG               rc = NO_ERROR;
   char               *resolved;                // parse copy, expandable
   TXTM                smalltext;               // text up to 84 chars
   char               *nextcmd;                 // next of multiple commands
   char               *command;                 // dfs command
   int                 ln;                      // length of cmd snippet

   ENTER();
   TRARGS(("Echo:%s Prompt:%s Output:%s  cmd:'%s'\n",
           (echo) ? "Y" : "N", (prompt) ? "Y" : "N", (output) ? "Y" : "N", cmdstring));

   #if defined (USEWINDOWING)
      dfsBEGINWORK();
   #endif                                       // USEWINDOWING
   dfsa->explain = TRUE;                        // can explain the RC
   if (strlen( cmdstring) != 0)                 // replace empty by space
   {
      if ((resolved = TxAlloc( 1, TXMAXLN)) != NULL)
      {
         for ( command  = cmdstring;            // abort from long multi
              (command != NULL) && !TxAbort() && // cmds must be possible!
             ((rc == NO_ERROR) || (dfsa->eStrategy != TXAE_QUIT));
               command  = nextcmd)
         {
            if ((nextcmd = strchr( command, dfsa->cmdSeparator[0])) != NULL)
            {
               ln = nextcmd - command;
               nextcmd++;                       // skip the separator
            }
            else                                // use whole string
            {
               ln = strlen( command);
            }
            strcpy( resolved, "");              // start empty
            strncat(resolved, command, ln);     // concat this command
            TRACES(("Command: '%s'  Nextcmd: '%s'\n", command, (nextcmd) ? nextcmd : ""));

            if (dfsa->veResolve)                // allow full resolving
            {
               rc = txsResolveExpressions( resolved - (command - cmdstring), lnr, FALSE,
                                           resolved,   TXMAXLN,  smalltext);
            }
            else                                // suppress '$' variable resolving
            {
               strcpy( resolved, command);      // use plain copy
               rc = TX_PENDING;
            }
            if (rc == TX_PENDING)
            {
               if (strlen(resolved))            // unless now empty
               {
                  TxCancelAbort();              // reset pending abort status
                  rc = dfsSingleCommand( resolved,
                                         echo,               //- echo each when requested
                          (prompt && (nextcmd == NULL))  &&  //- prompt only on LAST cmd
                          (((rc & DFS_QUIT) != DFS_QUIT) &&
                          !((nextcmd == NULL) && dfsa->autoquit)), output);
                  if ((rc & ~TX_RCFLAG_MASK) != NO_ERROR)
                  {
                     if (lnr != 0)
                     {
                        dfsGetRcString( rc, cmdstring);
                     }
                  }
               }
            }
            else if ((rc & ~TX_RCFLAG_MASK) != NO_ERROR)
            {
               TxPrint("\n%s\n", smalltext);
               if (lnr != 0)
               {
                  strcpy( cmdstring, smalltext);
               }
            }
         }
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   else
   {
      rc = dfsSingleCommand( " ", TRUE, prompt, TRUE);  // default cmd, <ENTER>
   }
   #if defined (USEWINDOWING)
      dfsENDWORK();

      sprintf( smalltext, "%s %s %s", DFS_N, DFS_VN, dfstStoreDesc1( DFSTORE) + 10);
      dfsSetAppWindowTitle( smalltext);
   #endif                                       // USEWINDOWING
   if (((rc & DFS_QUIT) == DFS_QUIT) && (TxaExeSwitch('S')))
   {
      TxPrint( "\nDFSee is running in SHELL mode, quit is not allowed ...\n");
      rc = NO_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsMultiCommand'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Interpret and execute DFSee command;
/*****************************************************************************/
static ULONG dfsSingleCommand
(
   char               *dfscmd,                  // IN    DFS command
   BOOL                echo,                    // IN    Echo command
   BOOL                prompt,                  // IN    prompt after execute
   BOOL                output                   // IN    screen-on during cmd
)                                               //       locally forced by -Q
{
   ULONG               rc;
   TXLN                dc;
   int                 cc = 0;                  // command string count
   char               *c0, *c1, *c2;            // parsed command parts
   TXLN                s0, s1;                  // temporary string space
   BOOL                l_batch   = FALSE;       // local batch-mode used
   BOOL                v_batch   = FALSE;       // saved batch-mode
   BOOL                v_ignst   = FALSE;       // saved ignore-state
   BOOL                l_dialogs = FALSE;       // local dialogs set
   BOOL                v_dialogs = FALSE;       // saved dialogs copy
   BOOL                l_errst   = FALSE;       // local error strategy
   int                 v_errst   = FALSE;       // saved copy
   BOOL                l_verbo   = FALSE;       // local verbosity
   ULONG               v_verbo   = FALSE;       // saved copy
   BOOL                l_trace   = FALSE;       // local trace-level set
   ULONG               v_trace   = FALSE;       // saved trace-level
   BOOL                l_echo    = echo;
   DEVICE_STATE        v_screen  = TxScreenState(  DEVICE_TEST); // saved copy
   DEVICE_STATE        v_logfile = TxLogfileState( DEVICE_TEST); // saved copy
   DEVICE_STATE        l_logfile = FALSE;       // local logfile status set
   int                 sanity    = 0;           // command loop sanity count
   BOOL                skip_double_prompt;
   static ULONG        cmdLevel  = 0;

   ENTER();
   cmdLevel++;
   TRARGS(("Echo:%s Prompt:%s Output:%s  cmd:'%s'\n",
           (echo) ? "Y" : "N", (prompt) ? "Y" : "N", (output) ? "Y" : "N", dfscmd));

   strcpy( dc, dfscmd);
   if (((dc[0] == 0) ||
        (dc[0] == ' ')  ||                      // empty command, <Enter>
        (strcmp(dc, "d") == 0)) &&              // or "d", special handling
       (nav.down == L64_NULL)    )              // if SLT page is showing
   {
      strcpy( dc, "slt");
   }

   TxaParseCommandString( dc, TRUE, NULL);      // parse, free-format
   cc = TxaArgCount( );                         // number of parameters
   c0 = TxaArgValue(0);
   c1 = TxaArgValue(1);
   c2 = TxaArgValue(2);

   if (TxaOption('-'))                          // option --
   {
      TxPrint("Command  : '%s'\n", dc);
      TxaShowParsedCommand( TRUE);              // option diagnostic
   }
   if (TxaOption('Q'))                          // locally forced quiet
   {
      ULONG quiet = TxaOptNum( 'Q', NULL, 1);   // default just the screen

      if (quiet & 0x02)                         // set logfile off
      {
         TxLogfileState( DEVICE_OFF);
         l_logfile = TRUE;                      // remember to set back :-)
      }
      if (quiet & 0x01)                         // set logfile off
      {
         output = FALSE;                        // force input parameter
      }                                         // to set screen off
   }
   if (TxaOptSet('B'))                          // local batch mode
   {
      if (TxaOption('B'))                       // set to batch-mode ON
      {
         v_ignst = dfstIgnoreLockErr( DFSTORE, TRUE); // ignore lock-errors too
      }
      else                                      // on -B-, return lock-status
      {                                         // to system default
         v_ignst = dfsa->LockIgn;
      }
      v_batch     = dfsa->batch;                // save old value
      l_batch     = TRUE;                       // use local force
      dfsa->batch = TxaOption('B');             // and set local value
   }
   if ((dfsa->batch) || (TxaOptSet('P')))       // batch or local -P option
   {
      v_dialogs    = dfsa->dialogs;             // save old value
      l_dialogs    = TRUE;                      // and signal restore

      dfsa->dialogs = (TxaOption('P') != 0);
   }
   if (TxaOptSet('E'))                          // local error strategy
   {
      v_errst = dfsa->eStrategy;
      l_errst = TRUE;
      dfsa->eStrategy = TxaErrorStrategy(   'E', dfsa->batch);
   }
   if (TxaOptSet('O'))                          // local output verbosity
   {
      v_verbo = dfsa->verbosity;
      l_verbo = TRUE;
      dfsa->verbosity = TxaOutputVerbosity( 'O');
   }
   #if defined (OEM_BRANDED)
   else
   {
      if (TxaOption('q'))                       // -q overrules standard -O: setting
      {                                         // for backward compatibility OEM engine
         v_verbo = dfsa->verbosity;
         l_verbo = TRUE;
         dfsa->verbosity = TXAO_QUIET;
      }
   }
   #endif
   TRACES(("batch: %u   eStrategy: %u    Verbosity: %u\n",
      dfsa->batch, dfsa->eStrategy, dfsa->verbosity));

   if (TxaOptSet(TXA_O_TRACE))                  // local change in tracelevel
   {
      v_trace   = TxTrLevel;
      l_trace   = TRUE;
      if (TxaOption(TXA_O_TRACE))
      {
         TxTrLevel = TxaOptNum(TXA_O_TRACE, NULL, 100);
         TRACES(( "Trace %3lu for '%s'\n", TxTrLevel, dfscmd));
      }
      else
      {
         TRACES(( "Trace OFF for '%s'\n", dfscmd));
         TxTrLevel = 0;
      }
   }

   dfsa->number = 0;

   if ((           c0[0] == '/')       ||       // search cmd, NO echo!
       (strcasecmp(c0, "screen") == 0) ||
       (strcasecmp(c0, "say"   ) == 0) ||
       (strcasecmp(c0, "ver"   ) == 0) ||
       (strcasecmp(c0, "chcp"  ) == 0))
   {
      l_echo             = FALSE;               // do not echo for these
      skip_double_prompt = TRUE;                // and no prompting either
   }
   else
   {
      skip_double_prompt = FALSE;
   }
   if (l_echo)
   {
      if ((!TxaExeSwitchUnSet( TXA_O_ECHO)) &&  // no suppress echo EXE switch ?
          (!TxaOptUnSet(       TXA_O_ECHO))  )  // and no suppress echo option ?
      {
         TxPrint("\n%s  %-5.5s: %sxecuting: %s%s%s\n", DFS_N, DFS_VN,
            #if defined (REGISTRATION)
                  (dfsRegistrationValid(NULL,NULL)) ? "E" :
            #endif
                                                      "e",
                   CBG, (cc == 0) ? "<Enter>"  : dfscmd, CNN);
         if ( (TxaExeSwitch('t') ||                // timestamping requested
               TxaExeSwitch('q') ||
               TxQueryLogFile(NULL, NULL)) &&      // or currently logging to file
             (!TxaExeSwitchUnSet('t')))            // and no suppress timestamp ?
         {
            time_t           tt = time( &tt);      // current date/time

            strftime( s0, TXMAXLN, "%A %Y-%m-%d %H:%M:%S", localtime( &tt));
            TxPrint( "Command timestamp : %s\n", s0);
         }
      }
   }
   if (!output)
   {
      TxScreenState( DEVICE_OFF);
   }

   do
   {
      if ((strcmp(c0, "/?") == 0) || (strcmp(c0,  "?") == 0)  )
      {
         TxPrint( "\nFor help on other (generic) commands, also use "
                  "'?\?', ?\?\?' and 'help'\n\n");
      }
      #if defined (USEWINDOWING)
         if (dfsa->sbwindow)
         {
            sprintf( s0, "%-5.5s executing: %s", SINF->afsys, dfscmd);
            txwSendMsg( dfsa->sbwindow, TXWM_STATUS, (TXWMPARAM) s0, (TXWMPARAM) cSchemeColor);
         }
      #endif                                       // USEWINDOWING
      TRACES(( "dfsa->FsCommand:%8.8x  dc:'%s'\n", dfsa->FsCommand, dc));
      rc = DFSFNCALL(dfsa->FsCommand,0,0, dc, NULL); // FS specific handler

      if ((rc == DFS_PENDING) ||                // cmd modified, pending
          (rc == DFS_CMD_UNKNOWN))              // cmd unknown, not handled
      {
         rc = TxStdCommand();                   // handle the std stuff

         TRACES(( "Command = '%s'  s0: '%s'\n", c0, s0));

         if (strcasecmp(c0, "log") == 0)        // quick hack, handle log
         {                                      // here to insert Reg info
            getcwd( s1, TXMAXLN);               // get current drive/directory
            TxPrint(  "Working directory : %s%s%s\n", CBG, s1, CNN);

            #if defined (REGISTRATION)
               dfsRegistrationInfo( DFSRG_DISPLAY_ONLY, DFS_V, dfsa->regconfirm);
            #endif
         }
      }
      //- to be refined, local options like -B should be interpreted here too!

      TRACES(( "rc = %u, sanity:%u\n", rc, sanity));

      if ((rc == DFS_PENDING) ||                // cmd modified, pending
          (rc == DFS_CMD_UNKNOWN))              // cmd unknown, not handled
      {
         if (++sanity > 10)
         {
            TxPrint( "\nCommand interpretation loop! Aborted after 10 attempts ...\n");
            rc = DFS_CMD_FAILED;
         }
         else
         {
            TRACES(( "About to call 'dfsGenericCommand()'\n"))
            rc = dfsGenericCommand();           // handle the generic stuff
         }
      }                                         // multiple passes possible!
   } while ((rc == DFS_PENDING) && !TxAbort());

   if (rc == DFS_CMD_UNKNOWN)                   // not handled yet
   {
      cc = TxaArgCount( );                      // number of parameters
      c0 = TxaArgValue(0);
      c1 = TxaArgValue(1);
      c2 = TxaArgValue(2);

      rc = NO_ERROR;                            // default result

      TRACES(("cc: %u c0:'%s' c1:'%s' c2:'%s'\n",
               cc,    c0,     c1,     c2));

      if ((strncasecmp(c0, "seta", 4  ) == 0) ||   // specific help info
          (strncasecmp(c0, "setn", 4  ) == 0) ||   // FDISK seta*
          (strncasecmp(c0, "sett", 4  ) == 0) ||   // FDISK sett*
          (strncasecmp(c0, "dele", 4  ) == 0) ||   // FDISK delete
          (strncasecmp(c0, "cr",   2  ) == 0) ||   // FDISK create
          (strncasecmp(c0, "sta",  3  ) == 0) ||   // FDISK startable
          (strncasecmp(c0, "newm", 4  ) == 0) ||   // FDISK newmbr
          (strncasecmp(c0, "fixb", 4  ) == 0)  )   // HPFS/NTFS fixboot
      {
         TxPrint("'%s' is not a valid command ", c0);
         if (strncasecmp(SINF->afsys, "--", 2) == 0)
         {
            TxPrint("at this moment, use the 'disk', 'part',\n'vol', 'fs' "
                    "or 'fdisk' command to activate a specific mode.\n");
         }
         else
         {
            TxPrint("for filesystem %s%s%s\n", CBM, SINF->afsys, CNN);
         }
      }
      else if ((strcasecmp(c0, "exit"  ) == 0) ||
               (strcasecmp(c0, "quit"  ) == 0) )
      {
         rc = DFS_QUIT;                         // set QUIT flag, with RC 0
      }
      else if ((strcasecmp(c0, "batch"   ) == 0))
      {
         if (TxaOption('?') ||
             (c1[0] == '?') ||                  // explicit BATCH help request
             TxaOption('h'))                    // script help
         {
            if (c1[0] != '?')                   // unless just a status request
            {
               TxPrint("\nSet GEOcalc mode, and RESET any force geometries\n\n"
                       " Usage: %s on | off\n\n"
                       "        ON  = Batch mode, NO questions asked ...\n"
                       "        OFF = Normal mode, with confirmations.\n\n", c0);
            }
         }
         else
         {
            if ((cc == 1) || (strcasecmp(c1, "on") == 0))
            {
               dfsa->batch   = TRUE;            // set global force on
               dfsa->LockIgn = TRUE;
            }
            else
            {
               dfsa->batch = FALSE;             // reset global force
               if (TxaExeSwitch('i') == FALSE)
               {
                  dfsa->LockIgn = FALSE;
               }
            }
            l_batch = FALSE;                    // reset local force
         }
         TxPrint( "BATCH mode is now : '%s'\n", (dfsa->batch) ? "ON" : "OFF");
      }
      else if (strcasecmp(c0, "run"   ) == 0)   // RUN frontend, prompt for
      {                                         // name & params if needed
         if (TxaOption('?') ||
             (c1[0] == '?') ||                  // explicit RUN help request
             TxaOption('h'))                    // script help
         {
            if (TxaOption('?') || (c1[0] == '?'))
            {
               rc = TxsNativeRun( NULL, NULL);  // get usage for RUN
            }
            else
            {
               TxShowTxt( hostvarhelp);
            }
         }
         else
         {
            #if defined (USEWINDOWING)
            if (((cc == 1) || TxaOption('P')) && // filedialog based
                (txwIsWindow( TXHWND_DESKTOP) )) // only when windowed ...
            {
               // Get combined scriptname + parameters from dialogs
               dfsRunScriptDialog( c1, s0);
            }
            else                                // parameter scriptname ?
            #endif                              // USEWINDOWING
            {
               // No file dialog, scriptname possibly specified
               strcpy( s0, c1);
               strcpy( s1, "");
               if ((cc == 1) || TxaOption('P')) // prompt for script + params
               {
                  TxPrompt( DFSC_RUNS, 40, s0, "Specify script to run plus parameters ...");
               }
               else
               {
                  if (strlen(s0) > 0)           // rebuild with name & params
                  {
                     TxaGetArgString( TXA_CUR, 2, TXA_OPT, TXMAXLN, s1);
                     strcat( s0, " ");
                     strcat( s0, s1);
                  }
               }
            }
            if (strlen(s0) > 0)                 // rebuild with name & params
            {
               sprintf( dc,   "runscript %s", s0);
               rc = dfsMultiCommand( dc, 0, TRUE, FALSE, TRUE);
               if (TxaOption('q'))
               {
                  rc |= DFS_QUIT;               // add QUIT flag to the RC
               }
            }
         }
      }
      else if ((strcasecmp(c0, "runscript") == 0)) // RUN, stage 2, reparsed
      {
         BOOL       isRexxScript = FALSE;       // safe default
         TXLN       scriptname;

         strcpy( s0, c1);                       // scriptname mandatory here!
         TxFnameExtension( s0, "dfs");
         if (TxsValidateScript( s0, &isRexxScript, NULL, scriptname))
         {
            DEVICE_STATE screen = TxScreenState( DEVICE_TEST);

            if (isRexxScript)
            {
               #if defined (DEV32)
                  TxaGetArgString( TXA_CUR, 2, 0, TXMAXLN, dc); // arguments, from 2nd

                  rc = TxsRexxRun( scriptname, dc, DFS_RX_ENV, dfs_subcom);
               #else
                  TxPrint( "\nREXX scripts are supported in the OS/2 version of DFSee only!\n");
               #endif
            }
            else                                // TXS native script
            {
               // Takes options and parameters from the Txa structures!
               rc = TxsNativeRun( scriptname, dfsMultiCommand);
               if (TxaOptNum('p', NULL, 3) & 0x02)
               {
                  // skip_double_prompt = TRUE;    // avoid double prompt
               }
            }
            TxScreenState( screen);             // restore initial state
         }
         else
         {
            TxPrint( "DFSee script file : '%s' not found\n", s0);
            rc = DFS_NO_SCRIPT;
         }
      }
      else if ((strcasecmp(c0, "repeat") == 0)) // REPEAT a (multi) command
      {
         if (TxaOption('?') || (cc == 1))       // help request
         {
            TxPrint("\n Usage:  %s  [options] command\n\n"
                      "   command = String with multiple commands (% separators)\n"
                      "   -E:i    = Ignore, do not quit on command errors\n"
                      "   -r:nr   = Repetition count, FOREVER when not specified\n"
                      "   -v      = Verbose, announce each iteration\n"
                      "   -v-     = No command echo, no promptline\n", c0);
            TxPrint("\n   Command string can use substitution for:\n"
                      "   $0      = iteration count 1..n\n\n"
                      "   Abort repeating using the <Esc> key ...\n");
         }
         else
         {
            ULONG      iterations = TxaOptNum( 'r', NULL, 0xffffffff);
            ULONG      i;                       // current iteration
            TXTS       iterstring;              // iterations as string
            char      *subst[1];                // array of substitutions
            BOOL       echoPrompt = !TxaOptUnSet('v');

            subst[0] = iterstring;              // array with one element

            TxaGetArgString( TXA_CUR, 1, 0, TXMAXLN, dc); // arguments, from 1st
            TxRepl( dc, '%', dfsa->cmdSeparator[0]);

            for ( i = 1;
                 (i <= iterations) &&
                       !TxAbort()   &&
                       ((rc == NO_ERROR) || (dfsa->eStrategy == TXAE_IGNORE));
                  i++)
            {
               sprintf( iterstring, "%u", i);   // iteration count string
               txsSubstituteParams( dc, subst, 1, TXMAXLN, s1);

               if (dfsa->verbosity >= TXAO_VERBOSE)
               {
                  TxPrint( "Repeat iteration : %u\n", i);
               }
               TRACES(( "Repeat i:%u cmd:'%s'\n", i, s1));

               rc = dfsMultiCommand( s1, 0, echoPrompt, echoPrompt, TRUE);
            }
         }
      }
      else if ((strcasecmp(c0, "set"      ) == 0))
      {
         if (cc > 1)
         {
            if      (strncasecmp(c1, "sep", 3) == 0)
            {
               if (cc > 2)
               {
                  dfsa->cmdSeparator[0] = c2[0];
               }
               TxPrint("Set command separactor character to be used\n\n"
                       " Usage: %s %s  character\n\n"
                       "Current separator : '%s'\n", c0, c1, dfsa->cmdSeparator);
            }
            else if (strncasecmp(c1, "warn", 4) == 0)
            {
               if (cc > 2)
               {
                  dfsa->warnLevel = atol( c2);

                  if (dfsa->warnLevel > 2)
                  {
                     dfsa->warnLevel = 2;
                  }
               }
               else
               {
                  TxPrint("\nSet WARNING level\n\n" " Usage: %s %s  0 | 1 | 2\n\n"
                          "    0 = default, significant warnings only\n"
                          "    1 = include more trivial or harmless warnings\n"
                          "    2 = pedantic, any warning and extra details\n\n", c0, c1);
               }
               TxPrint( "Warning level now : %u\n", dfsa->warnLevel);
            }
            else if (strncasecmp(c1, "pt0", 3) == 0)
            {
               if (cc > 2)
               {
                  if ((strcasecmp(c2, "on") == 0) || (c2[0] == '1'))
                  {
                     dfsa->pt0Show = TRUE;
                  }
                  else
                  {
                     dfsa->pt0Show = FALSE;
                  }
               }
               else
               {
                  TxPrint("\nSet Partition-Type 0x00 SHOW (deleted)\n\n" " Usage: %s %s on | off\n\n", c0, c1);
               }
               TxPrint( "PT0 Show mode now : '%s'\n", (dfsa->pt0Show) ? "ON" : "OFF");
            }
            else if (strncasecmp(c1, "floppy", 6) == 0)
            {
               if (cc > 2)
               {
                  if ((strcasecmp(c2, "on") == 0) || (c2[0] == '1'))
                  {
                     dfsa->volFloppy = TRUE;
                  }
                  else
                  {
                     dfsa->volFloppy = FALSE;
                  }
               }
               else
               {
                  TxPrint("\nSet use FLOPPY (A:/B:)\n\n" " Usage: %s %s on | off\n\n", c0, c1);
               }
               TxPrint( "Use Floppy (A:/B:) now : '%s'\n", (dfsa->volFloppy) ? "ON" : "OFF");
            }
            else if (strncasecmp(c1, "align", 5) == 0)
            {
               if (cc > 2)
               {
                  BYTE          unit;           // unit for size (k,m,g,c,h,s)

                  switch (c2[0])
                  {
                     case 'a':
                     case 'A':
                        dfsa->alignment = DFS_CR_ALIGN_AUTO;
                        break;

                     case 'c':
                     case 'C':
                        dfsa->alignment = DFS_CR_ALIGN_CYL;
                        break;

                     case 'm':
                     case 'M':
                        dfsa->alignment = DFS_CR_ALIGN_MIB;
                        break;

                     default:
                        dfsa->alignment = TxaParseNumber( c2, DFSRADIX_SIZE, &unit);
                        dfsa->alignment = dfsApplyNumberUnit( dfsa->alignment,
                                                       (unit == TXA_DFUNIT) ? 's' : unit, DFSTORE);
                        break;
                  }
               }
               else
               {
                  TxPrint("\nSet partition alignment on CREATE\n\n"
                          " Usage: %s %s Auto | Cyl | MiB | #sectors\n\n"
                          "    AUTO     = Cyl. alignment on MBR-type, MiB on GPT ones (default)\n"
                          "    CYL      = Align on Cylinder boundaries using disk geometry\n"
                          "    MIB      = Align on 1 MiB (0x800 512 byte sectors) boundaries\n"
                          "    #sectors = Align on multiples of the specified number of sectors\n\n"
                          "    To disable alignment for a new partition, use '-align-' on the CR\n"
                          "    command, or check the 'Do NOT align' checkbox in the create dialog.\n\n", c0, c1);
               }
               TxPrint( "Align setting now : %u = 0x%x %s\n", dfsa->alignment , dfsa->alignment,
                          (dfsa->alignment == DFS_CR_ALIGN_AUTO) ? "  (AUTO)" :
                          (dfsa->alignment == DFS_CR_ALIGN_CYL ) ? "  (CYL)"  :
                          (dfsa->alignment == DFS_CR_ALIGN_MIB ) ? "  (MIB)"  : "  (#sectors)");
            }
            else if (strncasecmp(c1, "error", 5) == 0)
            {
               if (cc > 2)
               {
                  switch (c2[0])
                  {
                     case 'c':
                     case 'C':
                        dfsa->eStrategy = TXAE_CONFIRM;
                        break;

                     case 'i':
                     case 'I':
                        dfsa->eStrategy = TXAE_IGNORE;
                        break;

                     default:
                        dfsa->eStrategy = TXAE_QUIT;
                        break;
                  }
               }
               else
               {
                  TxPrint("\nSet error handling to specified strategy\n\n"
                          " Usage: %s %s confirm | ignore | quit\n\n"
                          "    CONFIRM = Ask user confirmation to quit, or ignore the error\n"
                          "    IGNORE  = Ignore the error, set returncode ($_rc) to zero (0)\n"
                          "    QUIT    = Quit current operation, keep non-zero returncode\n\n", c0, c1);
               }
               TxPrint( "Error Strategy now: '%s'\n", (dfsa->eStrategy == TXAE_CONFIRM) ? "CONFIRM" :
                                                      (dfsa->eStrategy == TXAE_IGNORE ) ? "IGNORE"  :
                                                                                          "QUIT");
            }
            else if (strncasecmp(c1, "exp", 3) == 0)
            {
               if (cc > 2)
               {
                  if ((strcasecmp(c2, "toggle") == 0)  || (c2[0] == 'T'))
                  {
                     dfsa->expertui = !(dfsa->expertui);
                  }
                  else if ((strcasecmp(c2, "on") == 0) || (c2[0] == '1'))
                  {
                     dfsa->expertui = TRUE;
                  }
                  else
                  {
                     dfsa->expertui = FALSE;
                  }
               }
               else
               {
                  TxPrint("\nSet EXPERT UI-mode (versus BASIC mode)\n\n"
                          " Usage: %s %s on | off | toggle\n\n", c0, c1);
               }
               TxPrint( "Expert UImode now : '%s'\n", (dfsa->expertui) ? "ON   (expert)" : "OFF (basic)");
            }
            else if (strncasecmp(c1, "var", 3) == 0)
            {
               if (cc > 2)
               {
                  if ((strcasecmp(c2, "on") == 0) || (c2[0] == '1'))
                  {
                     dfsa->veResolve = TRUE;
                  }
                  else
                  {
                     dfsa->veResolve = FALSE;
                  }
               }
               else
               {
                  TxPrint("\nSet cmd variable/expression resolving\n\n"
                          " Usage: %s %s on | off\n\n"
                          "        ON  = Resolve $var and {expr} in all commands\n"
                          "        OFF = Do not resolve variables and expressions\n\n", c0, c1);
               }
               TxPrint( "$var resolve  now : '%s'\n", (dfsa->veResolve) ? "ON" : "OFF");
            }
            else if (strncasecmp(c1, "geo", 3) == 0)
            {
               if (cc > 2)
               {
                  if ((strcasecmp(c2, "on") == 0) || (c2[0] == '1'))
                  {
                     dfsa->geoCalc = TRUE;
                  }
                  else
                  {
                     dfsa->geoCalc = FALSE;
                  }
               }
               else
               {
                  TxPrint("\nSet GEOcalc mode, and RESET any force geometries\n\n"
                          " Usage: %s %s on | off\n\n"
                          "        ON  = based on disk contents, partition tables and LVM\n"
                          "        OFF = based on operating system info only\n\n", c0, c1);
               }
               TxPrint( "GEO-calc mode now : '%s'\n", (dfsa->geoCalc) ? "ON" : "OFF");
               if (cc > 2)
               {
                  USHORT                diskid;

                  for (diskid = 1; diskid <=  dfsPartitionableDisks(); diskid++)
                  {
                     dfsResetForcedGeo( diskid);
                  }
                  TxPrint( "Resulting geometry for all disks:\n");
                  dfsMultiCommand( "part -g -t-", 0, TRUE, FALSE, TRUE);
               }
            }
            else if (strncasecmp(c1, "log", 3) == 0)
            {
               if (cc > 2)
               {
                  if      ((strncasecmp( c2, "num", 3) == 0) || (strcasecmp(c2, "on") == 0) || (c2[0] == '1'))
                  {
                     dfsa->logDate = FALSE;
                     dfsa->logAuto = TRUE;
                  }
                  else
                  {
                     dfsa->logAuto = FALSE;
                     if ((strncasecmp( c2, "date", 4) == 0) || (c2[0] == '2'))
                     {
                        dfsa->logDate = TRUE;
                     }
                     else
                     {
                        dfsa->logDate = FALSE;
                     }
                  }
               }
               else
               {
                  TxPrint("\nSet LOG auto postfix by number or date\n\n"
                          " Usage: %s %s on | off | num | date\n\n"
                          "   ON | NUM  = Use automatic log NUMbering 001..999 from dialog\n"
                          "        DATE = Use DATE postfix from dialog, append when existing\n"
                          "        OFF  = Use logfilename 'as is', append if existing\n\n", c0, c1);
               }
               TxPrint( "LOG auto numbering now : '%s'\n", (dfsa->logAuto) ? "ON" : "OFF");
               TxPrint( "LOG auto DATE postfix  : '%s'\n", (dfsa->logDate) ? "ON" : "OFF");
            }
            else if (strncasecmp(c1, "gpt", 3) == 0)
            {
               if (cc > 2)
               {
                  if ((strcasecmp(c2, "on") == 0) || (c2[0] == '1'))
                  {
                     dfsa->gptAuto = TRUE;
                  }
                  else
                  {
                     dfsa->gptAuto = FALSE;
                  }
               }
               else
               {
                  TxPrint("\nSet GPT automatic mode\n\n"
                          " Usage: %s %s on | off\n\n"
                          "        ON  = Automatically interpret GPT partition tables\n"
                          "        OFF = Ignore GPT partitions, show protective 0xEE MBR\n\n", c0, c1);
               }
               TxPrint( "GPT auto mode now : '%s'\n", (dfsa->gptAuto) ? "ON" : "OFF");
            }
            else if (strncasecmp(c1, "margin", 6) == 0)
            {
               if (cc > 2)
               {
                  int     margin = atoi( c2);

                  if ((margin == 0) || ((margin >= 79) && (margin < TxScreenCols())))
                  {
                     dfsa->margin = margin;
                  }
                  else
                  {
                     TxPrint( "\nMargin value %d is out of the valid range 79 .. screenwidth\n", margin);
                  }
               }
               else
               {
                  TxPrint("\nSet display margin to limit number of characters per line\n\n"
                          " Usage: %s %s  0 | 79 .. screenwidth\n\n"
                          "    0      = automatic, using width of the output window\n"
                          "    79 ... = desired maximum number of characters per line\n\n"
                          " Note: works for many, but not all DFSee commands\n", c0, c1);
               }
               TxPrint( "\nMARGIN, number of characters per line: ");
               if (dfsa->margin)
               {
                  TxPrint( "SET MANUAL (%d)\n", dfsa->margin);
               }
               else
               {
                  TxPrint( "AUTOMATIC  (%d)\n", dfsa->sbWidth);
               }
            }
            else if (strncasecmp(c1, "menu", 4) == 0)
            {
               if (cc > 2)
               {
                  switch (c2[0])
                  {
                     case 'r':
                     case 'R':
                        dfsa->rightmove = TRUE;
                        break;

                     case '2':
                     case '3':
                     case 'n':
                     case 'N':
                        dfsa->autodrop  = FALSE;
                        dfsa->delaydrop = FALSE;
                        break;

                     case '4':
                     case '5':
                     case 'd':
                     case 'D':
                        dfsa->autodrop  = TRUE;
                        dfsa->delaydrop = TRUE;
                        break;

                     default:
                        dfsa->autodrop  = TRUE;
                        dfsa->delaydrop = FALSE;
                        break;
                  }
                  if (isdigit(c2[0]))
                  {
                     dfsa->rightmove = ((atoi(c2) & 1) == 1) ? TRUE : FALSE;
                  }
               }
               else
               {
                  TxPrint("\nSet main-MENU behaviour style\n\n"
                          " Usage: %s %s Auto | RightMove | NoDrop | Delayed | 0 .. 5\n\n"
                          "  0 | Auto      = Automatic dropdown, on moving left/right and activation (<F10>)\n"
                          "  1 | RightMove = Right-arrow moves right to next dropdown, doesn't open submenu\n"
                          "  2 | NoDrop    = No automatic dropdown, requires <Enter> to open pull-down\n"
                          "  4 | Delayed   = Delayed dropdown, on moving left/right only, not on activation\n\n", c0, c1);
               }
               TxPrint( "Menu style is now : %s\n", (dfsa->autodrop)  ? (dfsa->delaydrop)
                                               ? "4 = Delayed dropdown, on moving left/right only"
                                               : "0 = Auto dropdown on moving and activation"
                                               : "2 = No auto dropdown, requires <Enter>");
               if (dfsa->rightmove)
               {
                  TxPrint( "             plus : %s\n", "1 = Move right to next dropdown, don't open submenu");
               }
            }
            else if (strncasecmp(c1, "showhid", 7) == 0) // show hidden dirs/files, ON/OFF
            {
               if (cc > 2)
               {
                  if ((strcasecmp(c2, "on") == 0) || (c2[0] == '1'))
                  {
                     dfsa->browseShowHidden = TRUE;
                  }
                  else
                  {
                     dfsa->browseShowHidden = FALSE;
                  }
               }
               else
               {
                  TxPrint("\nSet SHOWHIDDEN directories/files in Browse\n\n"
                          " Usage: %s %s on | off\n\n", c0, c1);
               }
               TxPrint( "\nShow hidden files : %s\n", (dfsa->browseShowHidden)
                         ? "ON,  SHOW hidden files or directories in the browser"
                         : "OFF, HIDE hidden files or directories in the browser");
            }
            else if (strncasecmp(c1, "synclvm", 7) == 0)
            {
               #if defined (DEV32)
               if (cc > 2)
               {
                  if ((strcasecmp(c2, "on") == 0) || (c2[0] == '1'))
                  {
                     dfsa->lvmSyncApi = TRUE;
                  }
                  else
                  {
                     dfsa->lvmSyncApi = FALSE;
                  }
               }
               else
               {
                  TxPrint("\nSet SYNCLVM mode\n\n"
                          " Usage: %s %s on | off\n\n", c0, c1);
               }
               TxPrint( "Synclvm  mode now : '%s'\n", (dfsa->lvmSyncApi) ? "ON" : "OFF");
               #else
               TxPrint( "Synclvm  mode     : OFF\n");
               #endif
            }
            else if (strncasecmp(c1, "skipfs", 6) == 0)
            {
               if (cc > 2)
               {
                  sprintf( dfsa->skipFs, "%s %s %s %s ", c2, TxaArgValue(3), TxaArgValue(4), TxaArgValue(5));
                  TxStrToUpper( dfsa->skipFs);  // must be uppercase
               }
               else
               {
                  TxPrint("\nSet filesystems to SKIP on auto-correlate driveletters\n\n"
                          " Usage: %s %s space-separated-string\n\n"
                          "        'FAT32'    : Avoids sporadic program exit on OS/2 (FAT32.IFS bug)\n\n", c0, c1);
               }
               TxPrint( "SKIPFS filesystems now : '%s'\n", dfsa->skipFs);
            }
            else if (strncasecmp(c1, "radix", 5) == 0)
            {
               if (cc <= 2)
               {
                  TxPrint("\nDFSee number classes for %s %s (HEX when set):\n\n"
                          "     1 = standard / many other numbers\n"
                          "     2 = postfixed ,s values (sectors, like 123,s)\n"
                          "     4 = postfixed, other units       (like 456,m)\n"
                          "     8 = DFSee sector values like start and end location\n"
                          "    16 = DFSee size   values like a wipe or image size\n"
                          "    32 = DFSee partition type values\n", c0, c1);
               }
            }
            else
            {
               TxPrint("SET property name : '%s' unknown\n", c1);
            }
         }
         else
         {
            TxPrint(          "%s SET properties :  (capital part required as keyword only)\n", DFS_N);
            TxPrint( "  ALIGNment on CR : auto   |   cyl     |  mib    | #sectors\n");
            TxPrint( "  EXPert UI mode  : on     |   off\n");
            TxPrint( "  ERROR strategy  : quit   |   ignore  |  confirm\n");
            TxPrint( "  GEOcalc  mode   : on     |   off\n");
            TxPrint( "  FLOPPY usable   : on     |   off\n");
            TxPrint( "  GPT auto mode   : on     |   off\n");
            TxPrint( "  LOG  postfix    : on     |   off     |  num    | date\n");
            TxPrint( "  MENU drop down  : auto   |   right   |  nodrop | delayed   | numeric 0 .. 5\n");
            TxPrint( "  PT0show PT 0x00 : on     |   off\n");
         #if defined (DEV32)
            TxPrint( "  SYNCLVM  mode   : on     |   off\n");
         #endif
            TxPrint( "  VARiable subst  : on     |   off\n");
            TxPrint( "  WARNing level   : 0      |   1       |  2\n");
            TxPrint( "  MARGIN display  : numeric-value, range 80 .. screen-width  | 0  (no margin)\n");
            TxPrint( "  SEParator       : character\n");
            TxPrint( "  SHOWHIDden      : on     |   off          (show hidden items in Browse)\n");
            TxPrint( "  SKIPFSystems    : space separated string  (with FS names like FAT32)\n");
         }
      }
      else if (strcasecmp(c0, "rem") == 0)      // remark in a script
      {
      }
      else if (strcasecmp(c0, "win") == 0)      // avoid calling Windows :-)
      {
      }
      else if (strncasecmp(c0, "dfs", 3)  == 0) // no nested script/executable calling!
      {
         if (strstr( c0, ".dfs") != NULL)       // Seems to be a DFSee native script
         {
            TxPrint( "\nA native DFSee script like '%s' should be executed using the 'run' command\n"
                     "from the DFSee commandline, or the 'Scripts -> Run DFSee script' menu item.\n\n", c0);
         }
         else                                   // nested OS-level script ?
         {
            TxPrint( "\nThe '%s' script/program should be called from the operating\n"
                     "system command line, not from DFSee itself!\n\n"
                     "Quit DFSee and re-issue the command.\n", c0);
         }
      }
      else
      {
         if (TxaExeSwitch('S'))                 // shell mode ?
         {
            if (l_echo) TxPrint("\n'%s' is not a DFSee command, ignored ...\n", c0);
            rc = DFS_NO_CHANGE;
         }
         else
         {
            if (l_echo) TxPrint( "(executed by operating system)\n\n");
            rc = TxExternalCommand( dc);        // execute inputstring as cmd
         }
      }
   }
   TRACES(("dfs_number: %u, rc:  %u = 0x%X\n", dfsa->number, rc, rc));
   //- to be refined, may be able to to without a separate retc variable now
   dfsa->retc = rc & ~DFS_QUIT;                 // set overall return-code
   if (rc == TX_DISPLAY_CHANGE)                 // display mode has changed
   {
      if (TxaOptUnSet('w'))                     // non-windowed -w- ?
      {
         dfsa->nowindow = TRUE;
      }
      else                                      // default windowed
      {
         dfsa->nowindow = FALSE;
      }
      rc = DFS_QUIT;                            // quit command loop, restart
   }
   TRACES(("explain for rc:%u / retc:%u is %s\n", dfsa->retc, rc, (dfsa->explain) ? "YES" : "NO"));
   if (dfsa->explain)                           // unless filtered
   {
      dfsExplainRC( dfsa->retc);
      if ((dfsa->retc & ~DFS_QUIT) != NO_ERROR)
      {
         TRACES(("no more explaining!"));
         dfsa->explain = FALSE;                 // explain same RC just once
      }
   }
   TxaDropParsedCommand( FALSE);                // drop 1 level of parsed info
   if (l_batch)                                 // did we apply local force ?
   {
      dfstIgnoreLockErr( DFSTORE, v_ignst);     // restore ignore state
      dfsa->batch   = v_batch;                  // reset global batch-state
   }
   if (l_dialogs)                               // local strategy used ?
   {
      dfsa->dialogs = v_dialogs;                // reset dialogs strategy
   }
   if (l_errst)                                 // local strategy used ?
   {
      dfsa->eStrategy = v_errst;                // reset error strategy
   }
   if (l_verbo)                                 // local strategy used ?
   {
      dfsa->verbosity = v_verbo;                // reset verbosity
   }
   if (l_logfile)                               // local/overruled logfile state
   {
      TxLogfileState( v_logfile);               // reset explicit logfile status
   }
   if (l_trace)
   {
      TxTrLevel = v_trace;                      // restore global tracelevel
   }
   TRACES(("cmdLevel:%u  prompt:%s  skip_double_prompt:%s\n", cmdLevel,
                                                (prompt) ? "Y" : "N",
                                    (skip_double_prompt) ? "Y" : "N"));
   if (!skip_double_prompt)                     // avoid any prompting on say
   {
      if (prompt)                               // interactive prompt lines
      {
         dfsTxPrintStatusLine( rc & ~TX_RCFLAG_MASK); // print the status info
      }
   }
   #if !defined (OEM_BRANDED)
   if (!skip_double_prompt)
   {
      DEVICE_STATE screen = TxScreenState( DEVICE_TEST);

      if (!prompt)
      {
         TxScreenState( DEVICE_OFF);            // not to screen
      }
      TxPrint( "%s%s%-*.*s%s %u\n", CBZ, (prompt) ? "" : "\n",
                                     dfsa->sbWidth, dfsa->sbWidth,
                                     TX_SEPLINE TX_SEPLINE TX_SEPLINE,
                                     CNN, cmdLevel);
      if (!prompt)
      {
         TxScreenState( screen);                // restore initial state
      }
   }
   #endif
   if (!output)                                 // local/overruled screen state
   {
      TxScreenState( v_screen);                 // reset explicit quiet
   }
   #if defined (USEWINDOWING)
      if (dfsa->sbwindow)                       // Signal command completion
      {                                         // removed on first keypress
         sprintf( s0, "Done : %s", dfscmd);
         txwSendMsg( dfsa->sbwindow, TXWM_STATUS, (TXWMPARAM) s0, (TXWMPARAM) cSchemeColor);
      }
   #endif                                       // USEWINDOWING
   cmdLevel--;
   RETURN (rc);                                 // host variable for scripting
}                                               // end 'dfsSingleCommand'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Initialize datastructures for analysis of current filesystem
/*****************************************************************************/
ULONG dfsInitFileSystem
(
   void
)
{
   ULONG               rc = NO_ERROR;
   TXTM                fs;                      // filesystem name in bootrec
   BYTE                st = 0;
   BYTE                pt = DFS_P_EMPTY;
   DFSPARTINFO        *p  = SINF->p;            // open partition, or NULL
   ULONG               sects = BBUFSECTORS;

   ENTER();
   TRACES(("afsys at start init: '%s'\n", SINF->afsys));

   dfsa->boot =   (S_BOOTR *) brec;             // needed on failure too!
   rc = dfsRead( LSN_BOOTR, sects, brec);       // read whole buffer to include superblocks
   if (rc != NO_ERROR)
   {
      sects = dfstGeoSectors( DFSTORE);
      rc = dfsRead(LSN_BOOTR, sects, brec);     // retry with 1 track size
   }
   if (rc == NO_ERROR)                          // read whole buffer to include superblocks
   {
      if (dfsIsoFsIsPresent( brec, sects * dfsGetSectorSize()))
      {
         strcpy( SINF->afsys, "ISO");           // Has ISO9660 volume descriptors
      }
      else                                      // check for bootsector contents etc
      {
         nav.xtra        = dfsGetLogicalSize() -1;
         nav.down        = LSN_BOOTR;
         nav.this        = 0;                   // start from LSN 0 (edit .)
         nav.this_sninfo = 0;
         switch (st = dfsIdentifySector(0, 0, brec))
         {
            case ST_BOOTX:
               TxPrint( "Non-standard boot sector detected ...\n");
            case ST_BOOTR:
               if (p == NULL)                   // No partition info yet
               {
                  dfstSetSectorSize( DFSTORE, (USHORT) dfsa->boot->eb.SectSize);
                  dfstSetClusterSize(DFSTORE, (USHORT) dfsa->boot->eb.ClustSize);
               }
               if (strncasecmp((char *) &(dfsa->boot->Instruction[3]), "MSWIN4", 6) == 0)
               {
                  strcpy( fs, "FAT");           // maybe FAT32, wrong sys-type
               }
               else if (strncasecmp((char *) &(dfsa->boot->Instruction[2]), "LILO", 4) == 0)
               {
                  strcpy( fs, "EXT2");          // just a guess :-)
               }
               else
               {
                  if (strncmp(dfsa->boot->f2.Type, "FAT32", 5) == 0)
                  {
                     strcpy( fs, "FAT32");      // probably a FAT32
                  }
                  else
                  {
                     memcpy( fs, dfsa->boot->os.Type, BT_TYPE_L);
                     fs[BT_TYPE_L] = '\0';
                     if (fs[0] == '\0')         // no FS-type (NTFS hack!)
                     {
                        memcpy( fs, dfsa->boot->OpSys, BT_SYST_L);
                        fs[BT_SYST_L] = '\0';
                     }
                     else if (strcmp(dfsa->boot->os.Type +1, "3м") == 0)
                     {
                        strcpy( fs, "FAT");     // probably DOS 3.3 FAT
                     }
                  }
               }
               if (TxaOptSet( DFS_O_FILESYS))   // override filesystem to use (mode)
               {
                  strncpy( SINF->afsys, TxaOptStr( DFS_O_FILESYS, "Fsys", ""), TXMAXTS);
               }
               if (strncasecmp( fs, "EXFA", 4) == 0)
               {
                  strcpy( fs, "EFAT");          // Always use our internal FS name!
               }

               TRACES(("afsys: '%s'   fs:'%s'\n", SINF->afsys, fs));
               if ((!isalpha(SINF->afsys[0])) || // FS not known yet
                   (strlen(     SINF->afsys)                == 0)   ||
                   (strncasecmp(SINF->afsys,   "none",   4) == 0)   ||
                   (strncasecmp(SINF->afsys, "unknown",  7) == 0)   ||
                   (strncasecmp(SINF->afsys, "--none--", 8) == 0))
               {
                  strcpy( SINF->afsys, fs);     // copy from bootrec info
               }
               else if (dfsa->warnLevel > 1)    // pedantic, verify existing
               {
                  if (strncasecmp(SINF->afsys, fs, strlen( SINF->afsys)) != 0)
                  {
                     TxPrint("Warning: FS-type '%s' does not match "
                             "bootrecord info: '%s'\n", SINF->afsys, fs);
                  }
               }
               break;

            case ST_ULZ16:
               TxPrint("First sector (0) identified as DFSee small-buffer (16-bit) compressed data\n");
               if (!isalpha(SINF->afsys[0]))    // FS not known yet
               {
                  strcpy( SINF->afsys, "DFSI16");
               }
               break;

            case ST_ULZ32:
               TxPrint("First sector (0) identified as DFSee large-buffer (32-bit) compressed data\n");
               if (!isalpha(SINF->afsys[0]))    // FS not known yet
               {
                  strcpy( SINF->afsys, "DFSI32");
               }
               break;

            case ST_EXTBR:
            case ST_MASTR:
            case ST_MCDDM:
               if (dfstLSN2Psn( DFSTORE, 0) == 0) // correct MBR sector number
               {
                  if (!isalpha(SINF->afsys[0]) || (strcmp(SINF->afsys, "unknown") == 0))
                  {
                     switch (st)
                     {
                        case ST_EXTBR:
                           TxPrint("WARNING: EBR unexpected at this position ...\n");
                           break;

                        case ST_MCDDM:
                           TxPrint("WARNING: MAC driver description record ...\n");
                           break;

                        default:
                           break;
                     }
                     TxPrint("Enabling FS /mode : FDISK ...\n");
                     strcpy( SINF->afsys, "FDISK");
                  }
                  break;
               }                                // fallthrough to unrecognized
            case ST_EMPTY:
            case ST_BADFE:
            case ST_FDISK:
            default:
               if (p)                           // it is a partition
               {
                  switch (pt = p->partent.PartitionType)
                  {
                     case DFS_P_LINUXNATV:      // valid bootsector, Lilo/Grub
                     case DFS_P_LIN_NONFS:
                     case DFS_P_LINUXRAID:
                     case DFS_P_LINUX_LVM:
                     case DFS_P_SWAPSOLAR:      // some kind of SWAP partition
                     case DFS_P_BSDI_SWAP:
                     case DFS_P_VMWARSWAP:
                     case DFS_P_APFS_CONT:      // MAC HFS+
                     case DFS_P_MACXHFSP:       // MAC HFS+
                        break;

                     default:
                        pt = DFS_P_EMPTY;       // non-Linux, should have bootsec
                        break;
                  }
               }
               if ((dfsBootsecIsDumpFS( brec, NULL) == FALSE) && // exclude DUMPFS and
                       (pt == DFS_P_EMPTY))     // known Linux or MAC partitions
               {                                // to avoid unneeded warning msg
                  TxPrint("\nFirst sector (0) is not a valid filesystem BOOT record!\n");
                  if (st == ST_FDISK)
                  {
                     TxPrint("The first sector seems to be cleared to 0xF6 by FDISK or LVM!\n");
                  }
                  else if (st == ST_EMPTY)
                  {
                     TxPrint("The first sector seems to be cleared to 0x00, it is EMPTY!\n");
                  }
                  else if (st == ST_BADFE)
                  {
                     TxPrint("The first sector seems to be marked with 0xFE, BAD-sector!\n");
                  }
               }
               if ((!isalpha(SINF->afsys[0])) || // FS not known yet
                   (strncasecmp(SINF->afsys,   "none",   4) == 0)   ||
                   (strncasecmp(SINF->afsys, "unknown",  7) == 0)   ||
                   (strncasecmp(SINF->afsys, "--none--", 8) == 0))
               {
                  pt = DFS_P_EMPTY;             // re-init, to satisfy old logic
                  if (p)                        // it is a partition
                  {
                     pt = p->partent.PartitionType;
                     if (dfsBootsecIsDumpFS( brec, NULL)) // recognize this in ANY partition type!
                     {
                        strcpy( SINF->afsys, "DUMPFS");
                     }
                     else if (dfsBootAreaIsJFS( brec, BBUFSECTORS, NULL)) // Likely JFS superblock
                     {
                        strcpy( SINF->afsys, "JFS");
                     }
                     else
                     {
                        switch (pt)
                        {
                           case DFS_P_FAT12     :
                           case DFS_P_FAT16     :
                           case DFS_P_FAT16X    :
                           case DFS_P_BIGDOS    :
                              strcpy( SINF->afsys, "FAT");
                              break;

                           case DFS_P_FAT32     :
                           case DFS_P_FAT32X    :
                           case DFS_P_ACRONISSZ :
                              strcpy( SINF->afsys, "FAT32");
                              break;

                           case DFS_P_INST_FS   :
                           case DFS_P_IFSHIDDEN :
                              strcpy( SINF->afsys, "InsFS"); // will be an AUX-mode alias
                              break;

                           case DFS_P_WARP_LVM:
                              strcpy( SINF->afsys, "JFS");
                              break;

                           case DFS_P_LINUXNATV: // valid bootsector, Lilo/Grub
                           case DFS_P_LIN_NONFS:
                           case DFS_P_LINUXRAID:
                           case DFS_P_LINUX_LVM:
                              strcpy( SINF->afsys, "EXT2");
                              break;

                           case DFS_P_SWAPSOLAR: // some kind of SWAP partition
                           case DFS_P_BSDI_SWAP:
                           case DFS_P_VMWARSWAP:
                              strcpy( SINF->afsys, "SWAP");
                              break;

                           default:
                              break;
                        }
                     }
                  }
                  if (pt == DFS_P_EMPTY)
                  {
                     if (dfsDetectUnixFileSystem( 0, brec, NULL, NULL, SINF->afsys) == FALSE)
                     {
                        strcpy( SINF->afsys, "Binary");
                        TxPrint("\nNo filesystem known, using %s mode.\n", SINF->afsys);
                     }
                  }
               }
               memset( brec, 0, SECTORSIZE);    // wipe in-memory bootrec
               break;                           // avoiding FIXBOOT errors
         }
      }
   }

   //- Init FS even if no valid boot record found to allow analysis of
   //- corrupted volumes (using part table info or forced "FS xxxx")

   strcpy( fs, SINF->afsys);                    // fs to be used on init

   if ((strncasecmp( fs, "FDISK", 5) == 0) || (strncasecmp( fs, "BMGR", 4) == 0))
   {
      rc = dfsFdskInit();
      strcpy(  dfsa->brdescript, "Unknown"); // no directory contents
   }
   else
   {
      sprintf( dfsa->brdescript, "%s directory contents", fs); // most likely

      if      (strncasecmp( fs, "EXT",     3) == 0) rc = dfsExtInit(  fs);
      else if (strncasecmp( fs, "FAT",     3) == 0) rc = dfsFatInit(  fs);
      else if (strncasecmp( fs, "EFAT",    4) == 0) rc = dfsEfatInit( fs);
      else if (strncasecmp( fs, "HFS",     3) == 0) rc = dfsHfsInit(  fs);
      else if (strncasecmp( fs, "APFS",    4) == 0) rc = dfsApfsInit( fs);
      else if (strncasecmp( fs, "HPFS",    4) == 0) rc = dfsHpfsInit( fs);
      else if (strncasecmp( fs, "JFS",     3) == 0) rc = dfsJfsInit(  fs);
      else if (strncasecmp( fs, "NTFS",    4) == 0) rc = dfsNtfsInit( fs);
      else if (strncasecmp( fs, "REISER",  6) == 0) rc = dfsRsrInit(  fs);
      else if (strncasecmp( fs, "SWAP",    4) == 0) rc = dfsSwapInit( fs);
      else if (strncasecmp( fs, "DUMPFS",  6) == 0) rc = dfsDumpInit( fs);
      else if (strncasecmp( fs, "VFAT",    4) == 0) rc = dfsFatInit(  fs);
      else if (strncasecmp( fs, "XFS",     3) == 0) rc = dfsXfsInit(  fs);
      else if (strncasecmp( fs, "ISO",     3) == 0) rc = dfsIsoInit(  fs);
      else
      {
         if      (strcasecmp( fs, "InsFS") == 0)
         {
            TxPrint("\nInstallable-Filesystem partition seems not to be formatted (yet)");
         }
         else if (strcasecmp( fs, "AUX") != 0) // other filesystems
         {
            TxPrint("\nNo dedicated analysis module for filesystem: %s%s%s", CBM, fs, CNN);
            strcpy( SINF->afsys, "AUX");
         }
         rc = dfsAuxInit(  fs);
      }

      if ((rc == NO_ERROR) && (p != NULL)) // it is a partition
      {
         if (dfsa->FsExpandSize == 0)           // not set by FS Init yet, use maximum
         {
            ULONG   kmgAlign = dfsAlignmentSectors((p->tablenr == GPT_STYLE) ? 'G' : 'P', p);

            if ((p->expandSize != 0) &&         // fspAfter, at least one alignment unit
                (p->expandSize - p->sectors >= kmgAlign))
            {
               dfsa->FsExpandSize = p->expandSize; // set/show Maximum size

               dfsSz64( "Maximum vol. size : ", dfsa->FsExpandSize, " (for Expanding)\n");
            }
         }
         TRACES(( "DirtyStatus now: %u\n", dfsa->FsDirtyStatus));
         if (((st == ST_BOOTR) || (st == ST_BOOTX)) && // with valid bootsector
             ((dfsa->FsModeId  == DFS_FS_HPFS) || // and using an EBPB structure
              (dfsa->FsModeId  == DFS_FS_NTFS) || // with the real geometry ...
              (dfsa->FsModeId  == DFS_FS_FAT ) ))
         {                                      // check hiddenSectors/offset
            if ((dfsa->boot->eb.HiddenSectors != p->partent.BootSectorOffset) ||
                (dfsa->boot->eb.LogGeoHead != p->geoHeads) ||
                (dfsa->boot->eb.LogGeoSect != p->geoSecs )  )
            {
               if (dfsa->boot->eb.HiddenSectors != p->partent.BootSectorOffset)
               {
                  TxPrint("\nWarning: The HiddenSectors field 0x%s%8.8x%s in the bootsector does"
                          "\n         NOT match the offset in the partition-table 0x%s%8.8x%s!",
                           CBR, dfsa->boot->eb.HiddenSectors, CNN,
                           CBG, p->partent.BootSectorOffset, CNN);
               }
               if ((dfsa->boot->eb.LogGeoHead != p->geoHeads) ||
                   (dfsa->boot->eb.LogGeoSect != p->geoSecs )  )
               {
                  TxPrint("\nWarning: The GEO Head/Sectors fields %s%hu/%hu%s in the bootsector"
                          "\n         do NOT match the current disk geometry values %s%u/%u%s!",
                           CBR, dfsa->boot->eb.LogGeoHead, dfsa->boot->eb.LogGeoSect, CNN,
                           CBG, p->geoHeads,         p->geoSecs,          CNN);
               }
               TxPrint("\nThis may prevent using the partition from some operating systems like"
                       "\nOS/2, eCS or windows that check this for consistency. The most likely"
                       "\ncause for this is MOVING or CLONING the partition between different"
                       "\nprimary locations, or between a primary and logical location. Use the"
                       "\nFIXPBR command or 'Fix HiddenSectors/Geo value' menu to correct this.\n");
            }
         }
      }

      //- Check if (imported) list command is defined for this mode
      if ((strlen(     dfsa->snifsmcmd)) &&
          (strcasecmp( dfsa->snifsmode, SINF->afsys) == 0))
      {
         TxPrint( "\nList FS init  cmd : %s%s%s    (auto executed now)\n",
                     CBG, dfsa->snifsmcmd, CNN);

         dfsMultiCommand( dfsa->snifsmcmd, 0, FALSE, FALSE, FALSE);
      }
   }
   #if defined (USEWINDOWING)
      dfsAttachFsPulldown();
   #endif
   RETURN (rc);
}                                               // end 'dfsInitFileSystem'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Reset general datastructures on close of current filesystem
/*****************************************************************************/
ULONG dfsCloseFileSystem
(
   void
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   dfsa->FsCommand          = NULL;             // FS specific cmd handler
   dfsa->FsClose            = NULL;             // Filesystem deregister/close
   dfsa->FsIdentifySector   = NULL;             // FS identify a sector
   dfsa->FsShowType         = NULL;             // FS display ascii sector type
   dfsa->FsDisplaySector    = NULL;             // FS display sector contents
   dfsa->FsLsnAllocated     = NULL;             // FS test if LSN is allocated
   dfsa->FsLsnSetAlloc      = NULL;             // FS set Alloc bit to value
   dfsa->FsFileInformation  = NULL;             // FS file alloc/path/size/date
   dfsa->FsGetAllocSpace    = NULL;             // FS Get file allocation, SPACE
   dfsa->FsWriteMetaSpace   = NULL;             // FS Write Meta area, from SPACE
   dfsa->FsFindPath         = NULL;             // FS find LSN for given path
   dfsa->FsUpdateFileName   = NULL;             // FS rename file, same length! t
   dfsa->FsSetAltBrecLabel  = NULL;             // FS set alternate location label
   dfsa->FsMakeBrowseList   = NULL;             // FS make single-dir browse list
   dfsa->FsDisplayError     = NULL;             // FS display flag-error text
   dfsa->FsSltBuild         = NULL;             // FS build SLT, add contents
   dfsa->FsNpBuild          = NULL;             // FS build name-parent cache
   dfsa->FsDisplayLsnInfo   = NULL;             // FS LSN special info display
   dfsa->FsDirIterator      = NULL;             // FS directory iterator
   dfsa->FsDirFileSaveAs    = NULL;             // FS dir and file save (undel)
   dfsa->FsTruncateSize     = NULL;             // FS truncate filesystem size
   dfsa->FsAllocDisplay     = NULL;             // FS graphic alloc display
   dfsa->FsCl2Lsn           = NULL;             // FS cluster to LSN convert
   dfsa->FsLsn2Cl           = NULL;             // FS LSN to cluster convert
   dfsa->FsLsnIterator      = NULL;             // FS LSN iterate function
   dfsa->FsAreaClose        = NULL;             // FS Area deregister/close
   dfsa->FsAreaLsnAllocated = NULL;             // FS Area test LSN allocated
   dfsa->FsSltRecSize       = 0;                // FS recommended SLT size
   dfsa->FsEntry            = 0;                // FS logical entry point
   dfsa->FsTruncPoint       = 0;                // FS LSN to truncate volume
   dfsa->FsLastInUse        = 0;                // FS LSN last sector in-use
   dfsa->FsExpandSize       = 0;                // FS maximum size for expand
   dfsa->FsUnallocated      = 0;                // FS freespace, unallocated
   dfsa->FsUnallocSmart     = 0;                // FS unallocated, Smart candidate
   dfsa->FsDirtyStatus      = DFSTAT_UNKNOWN;   // FS dirty status unknown
   dfsa->FsCmdHelp          = NULL;             // FS specific cmdhelp text
   dfsa->FsModeId           = DFS_FS_AUX;       // FS mode id
   dfsa->Fsi                = NULL;
   dfsa->FsSectorTypes      = NULL;             // FS specific type string

   #if defined (USEWINDOWING)
      txSelDestroy( &(dfsa->FsModeSelist));     // Free FS specific list, if any
   #endif

   if (dfsa->brlist != 0)                       // there is a BROWSE list
   {
      dfsInitList(0, "-f -P", "-d");            // remove, invalid in other mode!
      dfsa->brlist = 0;
      dfsa->brinfo = 0;                         // make sure list-info is gone
   }
   RETURN (rc);
}                                               // end 'dfsCloseFileSystem'
/*---------------------------------------------------------------------------*/

